package se.antimon.monomeclock;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.text.FieldPosition;
import java.text.NumberFormat;

public class OscCommunication {
	private DatagramSocket socket = null;
	private int sendPort = -1;
	public OscCommunication(int sendPort) throws SocketException {
		this.sendPort = sendPort;
		socket = new DatagramSocket();
	}

	/**
	 * memAlign increases the memory to allow 1-4 zeroes may be added so that
	 * the resulting mem value is dividable by 4.
	 * @param unalignedMem
	 * @return
	 */
	private int memAlign(int unalignedMem) {
		return unalignedMem + 4 - (unalignedMem % 4);
	}
	private void storeInt(int value, byte[] mem, int memOffset) {
		mem[memOffset + 3] = (byte)(value & 0xff);
		mem[memOffset + 2] = (byte)((value & 0xff00) << 8);
		mem[memOffset + 1] = (byte)((value & 0xff0000) << 16);
		mem[memOffset + 0] = (byte)((value & 0xff000000) << 24);
	}
	private byte[] createOscBuffer(String command, Object[] arguments) {
		int commandMem = memAlign(command.length());
		int argCountMem = arguments == null ? 4 : memAlign(arguments.length + 1);
		StringBuffer argumentString = new StringBuffer(",");
		int argumentMem = 0;
		for (int i = 0; arguments != null && i < arguments.length; i++) {
			if (arguments[i] instanceof Integer) {
				argumentString.append("i");
				argumentMem += 4;
			} else if (arguments[i] instanceof String) {
				argumentString.append("s");
				argumentMem += memAlign(((String)arguments[i]).length());
			} else {
				throw new IllegalArgumentException("OSC argument of type " + arguments[i].getClass().getName()
						+ " not supported");
			}
		}
		byte[] result = new byte[commandMem + argCountMem + argumentMem];
		byte[] commandBytes = command.getBytes();
		System.arraycopy(commandBytes, 0, result, 0, commandBytes.length);
		byte[] argCountBytes = argumentString.toString().getBytes();
		System.arraycopy(argCountBytes, 0, result, commandMem, argCountBytes.length);
		if (arguments != null) {
			int currentMemPos = commandMem + argCountMem;
			for (int i = 0; i < arguments.length; i++) {
				if (arguments[i] instanceof Integer) {
					storeInt(((Integer)arguments[i]).intValue(), result, currentMemPos);
					currentMemPos += 4;
				} else if (arguments[i] instanceof String) {
					byte[] argBytes = ((String)arguments[i]).getBytes();
					System.arraycopy(argBytes, 0, result, currentMemPos, argBytes.length);
					currentMemPos += memAlign(argBytes.length);
				}
			}
		}
		return result;
	}
	private void printBuffer(byte[] buffer) {
		NumberFormat nf = NumberFormat.getInstance();
		for (int row = 0; row < buffer.length; row+=4) {
			for (int column = 0; column < 4; column++) {
				if (buffer[row+column] > 32) {
					System.out.print((char)buffer[row+column]);
				} else {
					System.out.print("?");
				}
			}
			System.out.print("  ");
			FieldPosition fp = new FieldPosition(NumberFormat.INTEGER_FIELD);
			for (int column = 0; column < 4; column++) {
				StringBuffer outBuffer = new StringBuffer();
				nf.format((long)buffer[row+column], outBuffer, fp);
				for (int i = 0; i < 4 - fp.getEndIndex(); i++) {
					System.out.print(" ");
				}
				System.out.print(outBuffer);
			}
			System.out.println();
		}
	}

	/**
	 * Send an OSC command to the localhost port configured in this
	 * OscCommunication instance.
	 * 
	 * @param command
	 *            the OSC string (including prefixes) with out argument
	 *            specifiers (e.g. ",ii"), these will be added automatically
	 * @param arguments
	 *            instances of String or Integer. Will be used to compile an
	 *            argument string (e.g. ",iis") in the message
	 * @param print
	 *            if true, print the message in ascii and byte values
	 */
	public void sendOscCommand(String command, Object[] arguments, boolean print) {
		byte[] oscBuffer = createOscBuffer(command, arguments);
		if (print) {
			System.out.println("Sending:");
			printBuffer(oscBuffer);
		}
		try {
			DatagramPacket packet = new DatagramPacket(oscBuffer, oscBuffer.length, 
					InetAddress.getLocalHost(), sendPort);
			socket.send(packet);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

}
