# chess/chessgame.py

# THE CODE AND DOCUMENTATION IN THIS DIRECTORY AND BELOW ARE OPEN
# SOURCE ACCORDING TO THE TERMS OF THE FOLLOWING LICENSE. PLEASE READ:
# -----------------------------------------------------------------------------
# ESST (Embedded System Software Tools) License
# -----------------------------------------------------------------------------
# 
# SOFTWARE LICENSE
# 
# This software is provided subject to the following terms and conditions,
# which you should read carefully before using the software. Using this
# software indicates your acceptance of these terms and conditions. 
# If you do not agree with these terms and conditions, do not use the software.
# 
# Copyright (c) 2008 Dale E. Parson, Ph.D. (a.k.a. Acoustic Interloper)
# All rights reserved.
# 
# Redistribution and use in source or binary forms, with or without
# modifications, are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following Disclaimer in comments in the code
# as well as in the documentation and/or other materials provided with the
# distribution.
# * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following Disclaimer in the documentation
# and/or other materials provided with the distribution.
# * Neither the name of Dale E. Parson, nor the names of the contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# Disclaimer
# 
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# ANY USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE
# USER'S OWN RISK. IN NO EVENT SHALL Dale E. Parson OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, INCLUDING,
# BUT NOT LIMITED TO, CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -----------------------------------------------------------------------------

import atexit
import copy
from math import *
from os import access, R_OK, X_OK
import socket
import SocketServer
from threading import *
import sys
from chess.interpreter import interpreter, ifactory
from chess.ChessException import *
from chess.UdpFormat import udpbigfmt, udpsmallfmt
import Queue
# The following are for transport which we will hide behind an API later.
from binascii import a2b_hex, b2a_hex
import struct
import time

advance = {}
advance['white'] = 1    # direction of advance
advance['black'] = -1

# Coordinates are always X,Y and deltaX,deltaY

board = {}
lastmvcolor = None
recordfilename = None
recordfile = None

""" board is a mapping from (x, y) to pits piece object or None. """
lastmove = []   # record start and end coords of latest move, dest. at end
class piece(object):
    """ piece is a chess piece. """
    def __init__(self, color, kind, index, x, y):
        self.color = color
        self.kind = kind
        self.index = index
        self.x = x
        self.y = y
        self.moved = False
        self.lastx = None
        self.lasty = None
        self.lastmoved = False
        self.piecesaffected = set([])
        self.crossforcastle = set([])
        self.lock = RLock()
    def __copy__(self):     # for copy.copy and virtual board planning
        c = piece(self.color, self.kind, self.index, self.x, self.y)
        c.moved = self.moved
        return c
    def undomove(self):
        """
        Cleanup after a ChessException
        """
        for op in self.piecesaffected:
            op.undomove()
        self.piecesaffected = set([])
        self.move(self.lastx, self.lasty, specialpiece=None, undoing=True)
        self.crossforcastle = set([])
    def getCastleCrossings(self):
        return copy.copy(self.crossforcastle)
    def move(self, x, y, specialpiece=None, undoing=False):
        """
        Move this piece on the board, removing any piece occupying (x,y).
        This method does not check validity of the move. It is the caller's
        responsibility to invoke getAttackSupportMove() to determine
        valid moves. Optional parameter specialpiece when specified can
        be a reference to a pawn to capture en passant by this pawn, or
        a rook with which to castle by this king; move() correctly moves
        the en passant pawn off the board, or moves the rook, before moving
        this piece. It does not check the validity of these compound moves.

        The specialpiece parameter can also be one of 'Q' 'R' 'N' 'B'
        strings for promotion when a pawn moves to the final opposing rank.
        In this case move() removes the pawn and creates the new piece at
        the new (x, y) without error checking.

        Return value is a reference to the moved piece, which is the piece
        invoked with move() unless the moved piece is a pawn to be promoted,
        in which case a reference to the new piece returns.
        """
        global board
        global lastmove
        if ((self.x == None or self.y == None) and not undoing):
            raise ChessException, "piece is not on the board: " + str(self)
        self.lock.acquire()
        result = self
        before = (self.x, self.y)
        after = (x, y)
        self.piecesaffected = set([])
        self.crossforcastle = set([])
        if (undoing):
            if (self.x != None and self.y != None               \
                    and board[(self.x, self.y)] == self):
                board[(self.x, self.y)] = None
            self.x = x
            self.y = y
            self.lastx = None
            self.lasty = None
            self.moved = self.lastmoved
            before = (None, None)
            if (x != None and y != None):
                board[(x, y)] = self
        elif ((x != self.x or y != self.y)                      \
                and (board[(self.x,self.y)] == self)):
            board[(self.x,self.y)] = None
            if (isinstance(specialpiece, piece)):
                if (self.kind == 'P' and specialpiece.kind == 'P'):
                    specialpiece.move(None, None)
                    self.piecesaffected.add(specialpiece)
                elif (self.kind == 'K' and specialpiece.kind == 'R'):
                    if (specialpiece.x < x):
                        rookx = x + 1
                        for xx in range(self.x, x, -1):
                            crossing = (xx, self.y)
                            self.crossforcastle.add(crossing)
                    else:
                        rookx = x - 1
                        for xx in range(self.x, x):
                            crossing = (xx, self.y)
                            self.crossforcastle.add(crossing)
                    specialpiece.move(rookx, y)
                    self.piecesaffected.add(specialpiece)
                else:
                    self.lock.release()
                    raise ChessException,                               \
                        "invalid castle or en passant piece: "  \
                            + str(specialpiece)
            self.x = x
            self.y = y
            if (x != None and y != None):   # None is off the board
                if (board[(self.x,self.y)]):
                    self.piecesaffected.add(board[(self.x,self.y)])
                    board[(self.x,self.y)].move(None, None) # take off board
                if (self.kind == 'P' and (self.y == 0 or self.y == 7)):
                    if (type(specialpiece) == str                       \
                            and specialpiece in set(['Q', 'B', 'N', 'R'])):
                        newkind = specialpiece
                    else:
                        newkind = 'Q'
                    newpiece = piece(self.color, newkind, 0, self.x, self.y)
                    newpiece.moved = True
                    board[(self.x, self.y)] = newpiece
                    self.piecesaffected.add(newpiece)
                    self.x = None
                    self.y = None
                    result = newpiece
                else:
                    board[(self.x,self.y)] = self
                lastmove = [before, after]
                self.lastmoved = self.moved
                self.moved = True
        else:
            self.lock.release()
            raise ChessException, "invalid move of " + self.color + " "  \
                + self.kind + " " + str(self.index) + " to "        \
                + str((x,y))
        self.lastx = before[0]
        self.lasty = before[1]
        self.lock.release()
        return result
    def getAttackSupportMove(self):
        """
        Return 5 tuple of:
            set of pieces I can attack
            set of pieces I support (similar to attack but my color)
            coordinates I can move to (including spaces under attack)
            set of opposing pawns this pawn can capture via en passant
                This set can contain 1 2-tuple of the space onto which the
                current pawn can move, and a reference to the pawn to capture
                en passant. The set is empty on no en passant.
            set of available rooks with which this king can currently castle
                This set contains 1 or 2 2-tuples of the space onto which the
                current king can move, and a reference to the rook to move
                across this king on castling. The set is empty on no castling.
        """
        if (self.x == None or self.y == None):
            raise ChessException, "piece is not on the board: " + str(self)
        self.lock.acquire()
        attackees = set([])
        supportees = set([])
        destinations = set([])
        enpassant = set([])
        castle = set([])
        if (self.kind == 'Q' or self.kind == 'R'):
            for x in range(self.x+1, 8):
                if (board[(x,self.y)] == None):
                    destinations.add((x,self.y))
                elif (board[(x,self.y)].color == self.color):
                    supportees.add(board[(x,self.y)])
                    break
                else:
                    attackees.add(board[(x,self.y)])
                    destinations.add((x,self.y))
                    break
            for x in range(self.x-1, -1, -1):
                if (board[(x,self.y)] == None):
                    destinations.add((x,self.y))
                elif (board[(x,self.y)].color == self.color):
                    supportees.add(board[(x,self.y)])
                    break
                else:
                    attackees.add(board[(x,self.y)])
                    destinations.add((x,self.y))
                    break
            for y in range(self.y+1, 8):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                elif (board[(self.x,y)].color == self.color):
                    supportees.add(board[(self.x,y)])
                    break
                else:
                    attackees.add(board[(self.x,y)])
                    destinations.add((self.x,y))
                    break
            for y in range(self.y-1, -1, -1):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                elif (board[(self.x,y)].color == self.color):
                    supportees.add(board[(self.x,y)])
                    break
                else:
                    attackees.add(board[(self.x,y)])
                    destinations.add((self.x,y))
                    break
        if (self.kind == 'Q' or self.kind == 'B'):
            for z in range(1,8):
                x = self.x + z
                y = self.y + z
                if (x > 7 or y > 7):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x + z
                y = self.y - z
                if (x > 7 or y < 0):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x - z
                y = self.y + z
                if (x < 0 or y > 7):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x - z
                y = self.y - z
                if (x < 0 or y < 0):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
        if (self.kind == 'N'):
            for dx, dy in [(-1, 2), (1, 2), (-2, 1), (2, 1),            \
                    (-1, -2), (1, -2), (-2, -1), (2, -1)]:
                x = self.x + dx
                y = self.y + dy
                if (x >= 0 and x < 8 and y >= 0 and y < 8):
                    if (board[(x,y)] == None):
                        destinations.add((x,y))
                    elif (board[(x,y)].color == self.color):
                        supportees.add(board[(x,y)])
                    else:
                        attackees.add(board[(x,y)])
                        destinations.add((x,y))
        if (self.kind == 'K'):
            for dx, dy in [(0, 1), (1, 1), (1,0), (1, -1),              \
                    (0, -1), (-1, -1), (-1, 0), (-1, 1)]:
                x = self.x + dx
                y = self.y + dy
                if (x >= 0 and x < 8 and y >= 0 and y < 8):
                    if (board[(x,y)] == None):
                        destinations.add((x,y))
                    elif (board[(x,y)].color == self.color):
                        supportees.add(board[(x,y)])
                    else:
                        attackees.add(board[(x,y)])
                        destinations.add((x,y))
            if (not self.moved):
                # Can I castle?
                if (board[(0,self.y)] and (not board[(0,self.y)].moved) \
                        and board[(1,self.y)] == None                   \
                        and board[(2,self.y)] == None                   \
                        and board[(3,self.y)] == None):
                    destinations.add((2, self.y))
                    rook = board[(0,self.y)]
                    dest = (2, self.y)
                    castle.add((dest, rook))
                if (board[(7,self.y)] and (not board[(7,self.y)].moved) \
                        and board[(5,self.y)] == None                   \
                        and board[(6,self.y)] == None):
                    destinations.add((6, self.y))
                    rook = board[(7,self.y)]
                    dest = (6, self.y)
                    castle.add((dest, rook))
        if (self.kind == 'P'):
            lx = self.x - 1
            hx = self.x + 1
            y = self.y + advance[self.color]
            if (len(lastmove) == 0):
                # special checks for en passant
                enpstart = 0
            elif (self.y == 4 and self.color == 'white'):
                enpstart = 6
            elif (self.y == 3 and self.color == 'black'):
                enpstart = 1
            else:
                enpstart = 0
            if (y >= 0 and y < 8):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                    if (not self.moved):
                        bigy = y + advance[self.color]
                        if (board[(self.x,bigy)] == None):
                            destinations.add((self.x,bigy))
                if (lx >= 0 and board[(lx,y)]):
                    if (board[(lx,y)].color != self.color):
                        attackees.add(board[(lx,y)])
                        destinations.add((lx,y))
                    else:
                        supportees.add(board[(lx,y)])
                if (hx < 8 and board[(hx,y)]):
                    if (board[(hx,y)].color != self.color):
                        attackees.add(board[(hx,y)])
                        destinations.add((hx,y))
                    else:
                        supportees.add(board[(hx,y)])
                if (enpstart):
                    if (lx >= 0                                         \
                            and board[(lx,y)] == None and board[(lx,self.y)] \
                            and board[(lx,self.y)].kind == 'P'          \
                            and board[(lx,self.y)].color != self.color  \
                            and lastmove[0] == (lx, enpstart)            \
                            and lastmove[1] == (lx, self.y)):
                        attackees.add(board[(lx,self.y)])
                        destinations.add((lx,y))
                        enpassant.add(((lx,y),board[(lx,self.y)]))
                    elif (hx < 8                                        \
                            and board[(hx,y)] == None and board[(hx,self.y)] \
                            and board[(hx,self.y)].kind == 'P'          \
                            and board[(hx,self.y)].color != self.color  \
                            and lastmove[0] == (hx, enpstart)            \
                            and lastmove[1] == (hx, self.y)):
                        attackees.add(board[(hx,self.y)])
                        destinations.add((hx,y))
                        enpassant.add(((hx,y),board[(hx,self.y)]))
        self.lock.release()
        return((attackees, supportees, destinations, enpassant, castle))

    def __str__(self):
        return self.color + self.kind + str(self.index)

def initboard(isclear=False):
    global mv_from
    global mv_to
    global mv_kingpanic
    global mv_reachgraph
    global mv_movenumber
    global lastmvcolor
    global board
    global recordfile
    global recordclock
    if (recordfile):
        nowtime = time.time()
        recordfile.write("time.sleep(" + str(nowtime-recordclock) + ")\n")
        recordfile.write("initboard(isclear=" + repr(isclear) + ")\n")
        recordclock = nowtime
    lastmvcolor = None
    mv_from = None
    mv_to = None
    mv_kingpanic = None
    mv_reachgraph = None
    mv_movenumber = 0
    if (isclear):
        for x in range(0,8):
            for y in range(0,8):
                board[(x,y)] = None
    else:
        board[(0,0)] = piece('white','R',0,0,0)
        board[(7,0)] = piece('white','R',1,7,0)
        board[(0,7)] = piece('black','R',0,0,7)
        board[(7,7)] = piece('black','R',1,7,7)
        board[(1,0)] = piece('white','N',0,1,0)
        board[(6,0)] = piece('white','N',1,6,0)
        board[(1,7)] = piece('black','N',0,1,7)
        board[(6,7)] = piece('black','N',1,6,7)
        board[(2,0)] = piece('white','B',0,2,0)
        board[(5,0)] = piece('white','B',1,5,0)
        board[(2,7)] = piece('black','B',0,2,7)
        board[(5,7)] = piece('black','B',1,5,7)
        board[(3,0)] = piece('white','Q',0,3,0)
        board[(3,7)] = piece('black','Q',0,3,7)
        board[(4,0)] = piece('white','K',0,4,0)
        board[(4,7)] = piece('black','K',0,4,7)
        for x in range(0,8):
            board[(x,1)] = piece('white','P',x,x,1)
            board[(x,6)] = piece('black','P',x,x,6)
            for y in range(2,6):
                board[(x,y)] = None
initboard()

recordclock = 0.0

def start_record(fname):
    global recordfilename
    global recordfile
    global recordclock
    if (recordfilename):
        raise ChessException, "Cannot record to " + str(fname)  \
            + ", " + recordfilename + " is already open."
    recordfile = open(fname, "w")
    recordfilename = fname
    recordclock = time.time()


def stop_record():
    global recordfilename
    global recordfile
    global recordclock
    if (not recordfilename):
        raise ChessException, "Cannot stop recording, no recording."
    recordfile.close()
    recordfile = None
    recordfilename = None
    recordclock = 0.0

killplayback = True

def play_record(fname):
    global killplayback
    killplayback = False
    # print "DEBUG IN PB " + str(killplayback)
    if (not killplayback):
        recordfile = open(fname, "r")
        try:
            try:
                while (not killplayback):
                    linebuf = recordfile.next().strip()
                    if (linebuf):
                        print "DEBUG EXEC " + linebuf
                        exec(linebuf)
            except StopIteration:
                pass
            except Exception, estr:
                print "EXEC exception: " + str(estr)
        finally:
            recordfile.close()

def dumpboard():
    for y in range(7,-1,-1):
        display = ""
        for x in range(0,8):
            if ((x+y) % 2):
                eyeflag = '+'
            else:
                eyeflag = 'x'
            if (board[(x,y)]):
                if (board[(x,y)].x != x):
                    raise ChessException, "bad x inside " + str(board[(x,y)])
                if (board[(x,y)].y != y):
                    raise ChessException, "bad y inside " + str(board[(x,y)])
                display = display + str(board[(x,y)]) + "(" + str(x)    \
                    + "," + str(y) + ")" + eyeflag + " "
            else:
                display = display + "    (" + str(x) + "," + str(y) + ")"   \
                    + eyeflag + "    "
        print display + "\n"

udpGameServer = None

def fetchmusicmap(target=None):
    global musicStateSerialNumber
    if (udpGameServer):
        result = m.getMusicGenState()
        datagram = struct.pack(udpbigfmt, musicStateSerialNumber, 0, 0, 0,
            'fetchmusicmap', result, "")
        if (target):
            try:
                ipaddr = target
                udpGameServer.listeners[ipaddr].send(datagram)
            except Exception, estr:
                print "Fetch Music Map Dropping callback to " + str(ipaddr) \
                    + " because: " + str(estr)
                del udpGameServer.listeners[ipaddr]
        else:
            for ipaddr in udpGameServer.listeners.keys():
                try:
                    udpGameServer.listeners[ipaddr].send(datagram)
                except Exception, estr:
                    print "Fetch Music Map Dropping callsback to "  \
                        + str(ipaddr) + " because: " + str(estr)
                    del udpGameServer.listeners[ipaddr]
    return ""

def fetcherror(errormessage, target=None):
    global musicStateSerialNumber
    if (udpGameServer):
        datagram = struct.pack(udpsmallfmt, 0, 0, 0, 0, 'fetcherror',
            errormessage, "")
        if (target):
            try:
                ipaddr = target
                udpGameServer.listeners[ipaddr].send(datagram)
            except Exception, estr:
                print "Fetch Music Map Dropping callback to " + str(ipaddr) \
                    + " because: " + str(estr)
                del udpGameServer.listeners[ipaddr]
        else:
            for ipaddr in udpGameServer.listeners.keys():
                try:
                    udpGameServer.listeners[ipaddr].send(datagram)
                except Exception, estr:
                    print "Fetch Music Map Dropping callsback to "  \
                        + str(ipaddr) + " because: " + str(estr)
                    del udpGameServer.listeners[ipaddr]
    return ""

def fetchboard4UDP(target=None, inhibitreach=False):
    global mv_from
    global mv_to
    global mv_kingpanic
    global mv_reachdisplay
    global mv_reachgraph
    result = ""
    if (udpGameServer):
        for row in range(0,8):
            for col in range(0,8):
                if (board[(col,row)]):
                    result = result + str(board[(col,row)])
                else:
                    result = result + "EMPTY"
                if (not (row == 7 and col == 7)):
                    result = result + ","
        if (mv_from):
            myfrom = mv_from
        else:
            myfrom = (-1, -1)
        if (mv_to):
            myto = mv_to
        else:
            myto = (-1, -1)
        panicstr = ""
        if (mv_kingpanic):
            if (mv_kingpanic['white']):
                panicstr = "WHITE IS IN CHECK!"
                # print "DEBUG REPORTING " + panicstr
            if (mv_kingpanic['black']):
                panicstr = "BLACK IS IN CHECK!"
                # print "DEBUG REPORTING " + panicstr
        reachstr = None
        # print "DEBUG MVR MVR ", str(mv_reachdisplay), str(mv_reachgraph)
        if (not inhibitreach):
            reachstr = ""
            if (mv_reachdisplay and mv_reachgraph):
                for row in range(0,8):
                    for col in range(0,8):
                        isupport = mv_reachgraph[(row,col)][1]
                        iattack = mv_reachgraph[(row,col)][2]
                        supportme = mv_reachgraph[(row,col)][3]
                        attackme = mv_reachgraph[(row,col)][4]
                        for p in isupport:
                            reachstr = reachstr + str(row) + ","        \
                                + str(col) + "," \
                                + str(p.x) + "," + str(p.y) + ","
                        reachstr = reachstr + "!,"
                        for p in iattack:
                            reachstr = reachstr + str(row) + ","        \
                                + str(col) + "," \
                                + str(p.x) + "," + str(p.y) + ","
                        reachstr = reachstr + "!,"
                        for p in supportme:
                            reachstr = reachstr + str(row) + ","        \
                                + str(col) + "," \
                                + str(p.x) + "," + str(p.y) + ","
                        reachstr = reachstr + "!,"
                        for p in attackme:
                            reachstr = reachstr + str(row) + ","        \
                                + str(col) + "," \
                                + str(p.x) + "," + str(p.y) + ","
                        reachstr = reachstr + "!,"

        # print "DEBUG LEN fetchboard is " + str(len(result))
        datagram = struct.pack(udpbigfmt,
            myfrom[0], myfrom[1], myto[0], myto[1],
                'fetchboard', result, panicstr)
        if (reachstr != None):
            # print "DEBUG LEN fetchreach is " + str(len(reachstr))
            datagram2 = struct.pack(udpbigfmt, 0, 0, 0, 0, 'fetchreach',
                reachstr, "")
        else:
            datagram2 = ""

        if (target):
            try:
                ipaddr = target
                udpGameServer.listeners[ipaddr].send(datagram)
                if (reachstr != None):
                    udpGameServer.listeners[ipaddr].send(datagram2)
                    # print "DEBUG SENDING RG " + reachstr
            except Exception, estr:
                print "Fetch Board Dropping callback to " + str(ipaddr) \
                    + " because: " + str(estr)
                del udpGameServer.listeners[ipaddr]
        else:
            for ipaddr in udpGameServer.listeners.keys():
                try:
                    udpGameServer.listeners[ipaddr].send(datagram)
                    if (reachstr != None):
                        udpGameServer.listeners[ipaddr].send(datagram2)
                        # print "DEBUG SENDING RG " + reachstr
                except Exception, estr:
                    print "Fetch Board Dropping callsback to " + str(ipaddr) \
                        + " because: " + str(estr)
                    del udpGameServer.listeners[ipaddr]
    return ""

class clientcallback(object):
    def __init__(self, udphost, udpport):
        self.udphost = udphost
        self.udpport = udpport
        self.skt = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.lock = RLock()
    def send(self, data):
        self.lock.acquire()
        try:
            self.skt.sendto(data, (self.udphost, self.udpport))
        finally:
            # m.kill() # DEBUG
            self.lock.release()

DEBUGTIME = time.time()
class tonegen(object):
    def __init__(self, udphost, udpport):
        self.udphost = udphost
        self.udpport = udpport
        self.skt = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.lock = RLock()
    def send(self, tonelist):
        """
        Transform list of ints and floats in tonelist parameter
        into packed OSC values of corresponding types, put into an OSC
        packet prececed by 'list\0\0\0\0,' and send out via UDP.
        """
        if (len(tonelist) < 11):
            raise ChessException, "invalid tonelist parameter: "    \
                + str(tonelist)
        listhead = a2b_hex('6c697374000000002c') # 'list\0\0\0\0,'
        hdrlen = 9
        listtail = ''
        # print "DEBUG ABOUT TO SEND LIST: " + str(tonelist) + " at " + str(time.time()-DEBUGTIME)
        for val in tonelist:
            if (type(val) == int):
                listhead = listhead + a2b_hex('69')   # 'i'
                listtail = listtail + struct.pack('>i', val)
            elif (type(val) == float):
                listhead = listhead + a2b_hex('66')   # 'f'
                listtail = listtail + struct.pack('>f', val)
            else:
                raise ChessException, "invalid type for tonelist value " \
                    + str(val) + ": " + str(type(val))
            hdrlen = hdrlen + 1
        if ((hdrlen % 4) == 0):
            listhead = listhead + a2b_hex('00')
            hdrlen = hdrlen + 1
        while (hdrlen % 4):
            listhead = listhead + a2b_hex('00')
            hdrlen = hdrlen + 1
        self.lock.acquire()
        try:
            # print "DEBUG UDP: " + str(b2a_hex(listhead+listtail))
            self.skt.sendto(listhead+listtail, (self.udphost, self.udpport))
        finally:
            # m.kill() # DEBUG
            self.lock.release()

t = tonegen('localhost', 57120)

mv_from = None
mv_to = None
mv_kingpanic = None
mv_reachgraph = None
mv_reachdisplay = False
mv_movenumber = 0
def mv(x1,y1,x2,y2, testing=0, promoter='Q'):
    global lastmvcolor
    global mv_from
    global mv_to
    global mv_kingpanic
    global mv_reachgraph
    global mv_movenumber
    global recordfile
    global recordclock
    if (recordfile):
        nowtime = time.time()
        recordfile.write("time.sleep(" + str(nowtime-recordclock) + ")\n")
        recordfile.write("mv(" + str(x1) + "," + str(y1) + ","      \
            + str(x2) + "," + str(y2) + ")\n")
        recordclock = nowtime
    p = board[(x1,y1)]
    otherp = board[(x2, y2)]
    if (otherp):
        othermoved = otherp.moved
    else:
        othermoved = False
    if (p == None):
        raise ChessException, "No piece at (" + str(x1) + "," + str(y1) + ")."
    if (lastmvcolor and lastmvcolor == p.color):
        if (testing):
            print "WARNING, moving color " + p.color + " in consecutive moves."
        else:
            raise ChessException, "Cannot move " + p.color  \
                + " in consecutive moves."
    if (otherp and p.color == otherp.color and p != otherp):
        if (testing):
            print "WARNING, a piece is taking its own color."
        else:
            raise ChessException, "A piece is taking its own color."
    attackset, supportset, coordset, pawnset, rookset       \
        = p.getAttackSupportMove()
    op = promoter
    if (len(pawnset)):
        print "DEBUG ENPASSANT 1", str(pawnset)
        op2 = (list(pawnset))[0]
        if (op2[0] == (x2, y2)):
            coordset.add(op2[0])
            op = op2[1]
    elif (len(rookset)):
        for rook2 in rookset:
            if (rook2[0] == (x2, y2)):
                coordset.add(rook2[0])
                op = rook2[1]
                break
    if (not (x2, y2) in coordset):
        if (testing):
            print "WARNING, illegal move!"
        else:
            l1 = (x1, y1)
            l2 = (x2, y2)
            raise ChessException, "Illegal move from " + str(l1) + " to " \
                + str(l2)
    p2 = p.move(x2,y2,op)
    isselfcheck = False
    try:
        mv_movenumber = mv_movenumber + 1
        (mv_reachgraph, mv_kingpanic) = m.interpret(board, p2, mv_movenumber)
    except ChessCheckException, estr:
        isselfcheck = str(estr)
        if (testing):
            print "WARNING, " + p.color + " moves into or stays in check!"
        else:
            # Undo the move, dump board, then rethrow.
            p.undomove()
            mv_movenumber = mv_movenumber - 1
    mv_from = (x1, y1)
    mv_to = (x2, y2)
    print 'MOVED ' + str(p) + " FROM " + str(mv_from) + " TO " + str(mv_to)
    print
    dumpboard()
    fetchboard4UDP()
    print
    print
    if (isselfcheck):
        raise ChessCheckException, isselfcheck
    else:
        lastmvcolor = p.color

def dropInPiece(color, kind, x2, y2, testing=False):
    """
    A piece appears out of nowhere!
    """
    global lastmvcolor
    global mv_from
    global mv_to
    global mv_kingpanic
    global mv_reachgraph
    global mv_movenumber
    global board
    global recordfile
    global recordclock
    if (recordfile):
        nowtime = time.time()
        recordfile.write("time.sleep(" + str(nowtime-recordclock) + ")\n")
        recordfile.write("dropInPiece(" + repr(color) + "," + repr(kind) + "," \
            + str(x2) + "," + str(y2) + ")\n")
        recordclock = nowtime
    otherp = board[(x2, y2)]
    if (color and kind):
        isnew = True
        oldmoved = False
        if (otherp):
            raise ChessException, "Cannot create " + str(color) + " " \
                + str(kind) + " at " + str((x2, y2)) + ", occupied by " \
                + str(otherp)
        p = piece(color, kind, 0, x2, y2)
        board[(x2,y2)] = p
        mv_from = None
        mv_to = (x2, y2)
        if (kind == 'P'):
            p.moved = not ((color == 'white' and y2 == 1)       \
                            or (color == 'black' and y2 == 6))
        elif ((color == 'white' and y2 == 0) or (color == 'black' and y2 == 7)):
            p.moved = not ((kind == 'R' and (x2 == 0 or x2 == 7))        \
                or (kind == 'N' and (x2 == 1 or x2 == 6))           \
                or (kind == 'B' and (x2 == 2 or x2 == 5))           \
                or (kind == 'Q' and x2 == 3)                         \
                or (kind == 'K' and x2 == 4))
        else:
            p.moved = True
        print 'CREATED ' + str(p) + " AT " + str(mv_to)
    else:
        isnew = False
        if (not otherp):
            raise ChessException, "No piece to delete at " + str((x2, y2))
        p = otherp
        oldmoved = p.moved
        # Don't vacate board until after interpret().
        # board[(x2,y2)] = None
        # p.x = None
        # p.y = None
        mv_from = (x2, y2)
        mv_to = None
        print 'DELETED ' + str(p) + " AT " + str(mv_from)
    isselfcheck = False
    try:
        mv_movenumber = mv_movenumber + 1
        (mv_reachgraph, mv_kingpanic) = m.interpret(board, p, mv_movenumber)
        if (not isnew):
            board[(x2,y2)] = None
            p.x = None
            p.y = None
    except ChessCheckException, estr:
        if ((not isnew) and p.kind == 'K'):     # always allow king removal
            board[(x2,y2)] = None
            p.x = None
            p.y = None
        else:
            isselfcheck = str(estr)
            if (testing):
                print "WARNING, " + p.color + " moves into or stays in check!"
            else:
                # Undo the move, dump board, then rethrow.
                if (isnew):
                    p.x = None
                    p.y = None
                    p.moved = False
                    board[(x2,y2)] = None
                    mv_movenumber = mv_movenumber - 1
                else:
                    p.x = x2
                    p.y = y2
                    p.moved = oldmoved
                    board[(x2,y2)] = p
                    mv_movenumber = mv_movenumber - 1
    print
    dumpboard()
    fetchboard4UDP()
    print
    print
    if (isselfcheck):
        raise ChessCheckException, isselfcheck
    else:
        lastmvcolor = p.color

musicStateSerialNumber = 0 

def setMusicGenState(assignment):
    """ Top-level function for plugin parameter setter. """
    global recordfile
    global recordclock
    global musicStateSerialNumber
    if (recordfile):
        nowtime = time.time()
        recordfile.write("time.sleep(" + str(nowtime-recordclock) + ")\n")
        recordfile.write("setMusicGenState(assignment=" + repr(assignment)   \
            + ")\n")
        recordclock = nowtime
    musicStateSerialNumber = musicStateSerialNumber + 1
    return m.setMusicGenState(assignment)

def silence_tones():
    global recordfile
    global recordclock
    if (recordfile):
        nowtime = time.time()
        recordfile.write("time.sleep(" + str(nowtime-recordclock) + ")\n")
        recordfile.write("silence_tones()\n")
        recordclock = nowtime
    return m.silence()

print
print
dumpboard()
print
print

SERVERPORT = 4000
CLIENTPORT = 4001

class udpChessGameServer(object):
    def __init__(self):
        self.listeners = {}

    def subscribe(self, ipaddr):
        try:
            proxy = clientcallback(ipaddr, CLIENTPORT)
            self.listeners[ipaddr] = proxy
            fetchboard4UDP(ipaddr)
            fetchmusicmap(ipaddr)
        except Exception, estr:
            print "UDP Exception: on back connection: " + str(estr)

    def unsubscribe(self, ipaddr):
        if (ipaddr in self.listeners.keys()):
            del self.listeners[ipaddr]

    def move(self,x1,y1,x2,y2,promoter, ipaddr):
        try:
            mv(x1,y1,x2,y2,promoter=promoter)
        except ChessException, estr:
            print "MOVE ERROR: " + str(estr)
            fetcherror(str(estr), ipaddr)

    def createpiece(self,color,kind,x2,y2,ipaddr):
        """
        Create a piece of color-king at x2, y2, or when color and kind == "",
        remove the piece at that spot.
        """
        try:
            dropInPiece(color,kind,x2,y2)
        except ChessException, estr:
            print "CREATE PIECE ERROR: " + str(estr)
            fetcherror(str(estr), ipaddr)

    def newgame(self):
        initboard()
        fetchboard4UDP()

    def clearboard(self):
        initboard(True)
        fetchboard4UDP()

    def silence(self):
        silence_tones()

    def redisplay(self, ipaddr):
        fetchboard4UDP(target=ipaddr)
        fetchmusicmap(target=ipaddr)

    def displayreach(self,isdisplay):
        global mv_reachdisplay
        mv_reachdisplay = not mv_reachdisplay
        fetchboard4UDP(inhibitreach=False)

    def setMusicGenState(self, assignment, ipaddr):
        global mv_reachgraph
        global mv_kingpanic
        global board
        global mv_to
        result = setMusicGenState(assignment)
        if (result == None):
            result = ""
        fetchmusicmap()
        if (result == "" and mv_to):
            # Extract a new sequence, update display for shortened graph.
            (mv_reachgraph, mv_kingpanic) \
                = m.interpret(board,board[(mv_to[0],mv_to[1])],mv_movenumber)
            fetchboard4UDP(inhibitreach=False)
        elif (result):
            fetcherror(str(result), ipaddr)
        return result

udpGameServer = udpChessGameServer()

class DatagramInterpreter(SocketServer.BaseRequestHandler):
    def handle(self):
        global udpGameServer
        try:
            # print "DEBUG RAW UDP: ", str(self.request), "\n",     \
                # b2a_hex(self.request[0])
            try:
                datatuple = struct.unpack(udpsmallfmt, self.request[0])
            except struct.error:
                datatuple = struct.unpack(udpbigfmt, self.request[0])
            # print "DEBUG incoming UDP data = " + str(datatuple)
            kw = datatuple[4][0:datatuple[4].find(a2b_hex('00'))]
            args = datatuple[5][0:datatuple[5].find(a2b_hex('00'))]
            ipaddr = datatuple[6][0:datatuple[6].find(a2b_hex('00'))]
            int0 = int(datatuple[0])
            int1 = int(datatuple[1])
            int2 = int(datatuple[2])
            int3 = int(datatuple[3])
            if (kw == 'subscribe'):
                udpGameServer.subscribe(ipaddr)
            elif (kw == 'unsubscribe'):
                udpGameServer.unsubscribe(ipaddr)
            elif (kw == 'move'):
                udpGameServer.move(int0,int1,int2,int3,args,ipaddr)
            elif (kw == 'createpiece'):
                color_kind = args.split(':')
                udpGameServer.createpiece(color_kind[0],color_kind[1],
                    int0, int1, ipaddr)
            elif (kw == 'newgame'):
                udpGameServer.newgame()
            elif (kw == 'clearboard'):
                udpGameServer.clearboard()
            elif (kw == 'silence'):
                udpGameServer.silence()
            elif (kw == 'redisplay'):
                udpGameServer.redisplay(ipaddr)
            elif (kw == 'displayreach'):
                isdisplay = bool(args)
                udpGameServer.displayreach(isdisplay)
            elif (kw == 'setMusicGenState'):
                udpGameServer.setMusicGenState(args,ipaddr)
            else:
                # print "DEBUG FKEY: ", len(kw), kw
                raise ChessException, "Invalid UDP datagram: " + str(datatuple)
        except Exception, estr:
            print "UDP CHESS GAME RECV ERROR: " + str(estr)

class udpCnxnManager(SocketServer.BaseRequestHandler):
    def __init__(self, ipaddrs, ipport):
        self.serveraddr = (ipaddrs, ipport)
        self.server = SocketServer.ThreadingUDPServer(self.serveraddr,
            DatagramInterpreter)
        self.server.daemon_threads = True
        self.server.request_queue_size = 10
        cv = Condition()
        self.threadisrunning = False
        isok = False
        self.thread = Thread(target = self.runthread,name='UDPRPC',
            args=(cv,))
        self.thread.setDaemon(1)
        cv.acquire()
        self.thread.start()
        while (not isok):
            isok = self.threadisrunning
            if (not isok):
                cv.wait()
        cv.release()

    def runthread(self, cv):
        cv.acquire()
        self.threadisrunning = True
        cv.notifyAll()
        cv.release()
        while(1):
            try:
                self.server.serve_forever()
            except Exception, estr:
                print "UDPRPC error in service thread: " + str(estr)

myipaddr = socket.gethostbyname(socket.gethostname())
myserver = udpCnxnManager(myipaddr, SERVERPORT)
print "Listening from ipaddr = " + myipaddr + ", ipport = " + str(SERVERPORT)

m = None
deathq = Queue.Queue(0)

def start(classname, oscperside):
    global m
    m = ifactory(classname,t,oscperside)

def testa():
    start('simpleSineInterpreter',16)

def testb():
    start('polySineInterpreter',16)

def testc():
    start('shuffleSineInterpreter',16)

def testd():
    start('aheadSineInterpreter',16)

def teste():
    start('sustainSineInterpreter',32)

def testf(transforms='transformsmore'):
    start('echosustainSineInterpreter',(32, transforms))

def testg(transforms='transformsmanymore'):
    start('pannerSustainInterpreter',(32, transforms))

def testh(transforms='transformsmanymore'):
    start('stutterPannerSustainInterpreter',(32, transforms))

def exitfunc():
    global m
    if (m):
        silence_tones()

atexit.register(exitfunc)
