// manytones.ck, June, 2008, Dale Parson (Acoustic Interloper),
// acknowledgements to Les Hall (Inventor) and Kassen for getting
// me into my first ChucK program. Thanks!

// Launch as a sound generator for chessgame.py as of 
// 18 June 2008. This OSC type format is patch specific.
// There are 4 banks of ugens with 64 ugens each.
// Bank 0 generates sines and bank 1 triangles in the
// initial test. Incoming OSC fields are describd in PROTOCOL_2.txt.
// SEE PROTOCOL_2.txt for the June, 2008, 9-field protocol.

// Add some additional sounds to chesstones.ck


// The structure of these classes mirrors the structure of the Max/MSP patch.

class OscBlock {                // a single ramped oscillator
    Osc @ gen ;                // SinOsc or TriOsc, set after construction
    0.0 => float savefreq ;
    0.0 => float lastfreq ;
    0.0 => float slidefreq ;
    10::ms => static dur slidedelay ;
    20 => static int slidesamples ;
    0 => int isslide ;
    Envelope ramp ;
    Gain lgain ;                // left channel gain
    Gain rgain ;                // right channel gain
    UGen @ leftdest ;
    UGen @ rightdest ;
    Event oscevent ;
    int isdead ;                // set to 1 when oscillator is off
    1 => isdead ;
    0 => int savefx ;
    0::ms => dur lfosampleperiod ;
    0.0 => float lfophase ;
    0 => static int numlfo ;
    8 => static int maxlfo ;
    // lampl and rampl record levels for output scaling
    0.0 => float lampl ;
    0.0 => float rampl ;
    100::ms => static dur gaindelay ;  // duration of a change in amplitude
    ramp => lgain ;
    ramp => rgain ;
    
    // Invoke init() one time after constructing this object, passing its
    // generator and left and right outputs.
    fun void init(Osc newgen, UGen leftout, UGen rightout) {
        newgen @=> gen ;
        1.0 => gen.gain ;
        leftout @=> leftdest ;
        rightout @=> rightdest ;
        gen => ramp ;
        spork ~ osc_run_thread();     // Each oscillator gets a garbage collector.
    }
    // Set up the oscillator path but do not advance time.
    // Use fx to predicate time for FX addition to basic signal.
    // For notes where sustain is short, FX won't be heard, but for sustained
    // tones it will.
    fun void play(float freq, float phase, float left, float right,
            int fx, int piecetype, int opiecetype) {
        left => lampl ;
        right => rampl ;
        isdead => int wasdead ;
        if (left == 0.0 && right == 0.0) {
            0 => ramp.target ;
            1 => isdead ;
            if (savefx != 0) {
                numlfo-- ;
            }
            0 => savefx ;
            if (savefreq != 0.0) {
                savefreq => lastfreq ;
            }
            0 => isslide ;
            0.0 => savefreq ;
            if (! wasdead) {
                oscevent.signal();   // Schedule disconenction after ramp down.
            }
        } else if (freq > -1.0) {
            0 => int issignal ;
            if (savefreq != 0.0) {
                savefreq => lastfreq ;
            }
            freq => savefreq ;
            if (lastfreq != 0.0 && piecetype == 4 && lastfreq != freq) {
                lastfreq => gen.freq ;
                (freq - lastfreq) / slidesamples => slidefreq ;
                slidesamples => isslide ;
                1 => issignal ;
            } else {
                freq => gen.freq ;
                0 => isslide ;
            }
            phase => gen.phase ;
            left => lgain.gain ;
            right => rgain.gain ;
            1 => ramp.target ;
            0 => isdead ;
            if (wasdead) {
                lgain => leftdest ;
                rgain => rightdest ;
            }
            if (fx != 0 && numlfo < maxlfo) { // bishop or rook
                // <<< "play", piecetype, fx, opiecetype >>>;
                if (savefx == 0) {
                    // 10 seconds / inter_piece_distance / 45 discrete samples
                    // Math.max(freq, savefreq) => float maxfreq ;
                    // Hi frequencies generate a lot of noise, so make
                    // the lfo period longer for high frequencies.
                    // Baseline is 10 seconds for 440 A.
                    // (maxfreq * maxfreq) / (440.0 * 440.0) * 10000.0 => float scnds ;
                    // (maxfreq / 440.0) * 10000.0 => float scnds ;
                    10000.0 => float scnds ;
                    if (scnds < 10000.0) {
                        10000.0 => scnds ;
                    }
                    fx => savefx ;
                    ((scnds / fx) / 45.0)::ms => lfosampleperiod ;
                    0.0 => lfophase ;
                    1 => issignal ;
                    numlfo++ ;
                }
            } else {
                if (savefx != 0) {
                    numlfo-- ;
                }
                0 => savefx ;
                0.0::ms => lfosampleperiod ;
                0.0 => lfophase ;
            }
            if (issignal != 0) {
                oscevent.signal();
            }
        }
        gaindelay => ramp.duration ;
    }
    // Disconnect a dead oscillator after ramp time expires.
    // Each OscBlock object has a shred whose sole job it is to do this.
    fun void osc_run_thread() {
        while (1) {
            // Wait until awakened and then until ramp down is done.
            if (isdead != 0) {
                gaindelay => now ;  // RAMP DOWN COMPLETED
                lgain =< leftdest ;
                rgain =< rightdest ;
                if (savefx != 0) {
                    numlfo-- ;
                }
                0 => savefx ;
                0.0::ms => lfosampleperiod ;
                0.0 => lfophase ;
                oscevent => now ;        // AWAKENED
            } else if (isslide != 0) {
                slidedelay => now ;
                isslide-- ;
                lastfreq + slidefreq => lastfreq ;
                if (isslide == 0) {
                    savefreq => gen.freq ;
                    0.0 => lastfreq ;
                } else {
                    lastfreq => gen.freq ;
                }
            } else if (savefx != 0) {
                lfosampleperiod => now ;
                savefreq + (Math.sin(lfophase) * 1.5 * savefreq) => gen.freq ;
                lfophase + (pi / 11.25) => lfophase ;
            } else {
                oscevent => now ;
            }
        }
    }
}

64 => int OscillatorsPerBank ;
4 => int Banks ;
OscillatorsPerBank * Banks => int NumberOscillators ;

OscBlock blocks[NumberOscillators] ;
LPF leftlpf ;
LPF rightlpf ;
Dyno leftcompressor ;
Dyno rightcompressor ;
Gain leftscale => leftcompressor => dac.left ;
Gain rightscale => rightcompressor => dac.right ;
1.0 => leftscale.gain ;
1.0 => rightscale.gain ;
10000.0 => leftlpf.freq ;
10.0 => leftlpf.Q ;
10000.0 => rightlpf.freq ;
10.0 => rightlpf.Q ;
0.20 => leftcompressor.slopeAbove ;
1.0 => leftcompressor.slopeBelow ;
0.8 => leftcompressor.thresh ;
5::ms => leftcompressor.attackTime ;
300::ms => leftcompressor.releaseTime ;
0 => leftcompressor.externalSideInput ;
0.20 => rightcompressor.slopeAbove ;
1.0 => rightcompressor.slopeBelow ;
0.8 => rightcompressor.thresh ;
5::ms => rightcompressor.attackTime ;
300::ms => rightcompressor.releaseTime ;
0 => rightcompressor.externalSideInput ;
0.0 => float leftsum ;
0.0 => float rightsum ;
0 => int osccount ;

// Create and wire the OscBlocks to the gain scalers.
for (0 => int b ; b < Banks ; b++) {
    Osc @ nextgen ;
    for (0 => int o ; o < OscillatorsPerBank ; o++) {
        (b * OscillatorsPerBank) + o => int i ;
        if ((b & 1) == 0) {
            new SinOsc @=> nextgen ;
        } else {
            new TriOsc @=> nextgen ;
        }
        blocks[i].init(nextgen, leftscale, rightscale) ;
    }
}

// Create our OSC (Open Sound Control) receiver.
OscRecv recv;
// port 7401 used by the chess program
57120 => recv.port;
// start listening (launch thread)
recv.listen();

// create an address in the receiver, store in new variable
recv.event( "list, i i f f f f i i i i i i" ) @=> OscEvent @ oe;

second => now ;

// infinite event loop
while( true )
{
    // wait for event to arrive
    oe => now;

    // grab the next message from the queue. 
    while( oe.nextMsg() )
    { 
        int bank, osc, player, piecetype, relationship ;
        int otherplayer, otherpiecetype, movenumber ;
        int piecediff, attacker, attackee ;
        float freq, phase, leftampl, rightampl ;

        // getFloat fetches the expected float (as indicated by "i f")
        oe.getInt() => bank ;
        oe.getInt() => osc ;
        oe.getFloat() => freq ;
        oe.getFloat() => phase ;
        oe.getFloat() => leftampl ;
        oe.getFloat() => rightampl ;
        oe.getInt() => player ;
        oe.getInt() => piecetype ;
        oe.getInt() => relationship ;
        oe.getInt() => otherplayer ;
        oe.getInt() => otherpiecetype ;
        oe.getInt() => movenumber ;
        // <<< "OSC:", bank, osc, freq, phase, leftampl, rightampl, player, piecetype, relationship, otherplayer, otherpiecetype, movenumber >>>;

        (OscillatorsPerBank * bank) + osc => int oscix ;
        if (oscix >= NumberOscillators) {
            continue ;
        }
        if (! blocks[oscix].isdead) {
            blocks[oscix].lampl -=> leftsum ;
            blocks[oscix].rampl -=> rightsum ;
            osccount-- ;
        }
        // Trigger lfo fx for attack relationships.
        // attacker->attackee generates FX only when attackee is worth more.
        // Limit attacker to bishop or rook.
        if (relationship == 1
                && (piecetype == 2 || piecetype == 3)) {
            otherpiecetype - piecetype => piecediff ;
            piecetype => attacker ;
            otherpiecetype => attackee ;
        } else if (relationship == 3
                && (otherpiecetype == 2 || otherpiecetype == 3)) {
            piecetype - otherpiecetype => piecediff ;
            otherpiecetype => attacker ;
            piecetype => attackee ;
        } else {
            0 => piecediff ;
            piecetype => attacker ;
            otherpiecetype => attackee ;
        }
        if (piecediff < 1) {
            0 => piecediff ;
        }
        // if (piecediff != 0) {
            // <<< "call", attacker, relationship, attackee >>>;
        // }
        blocks[oscix].play(freq, phase, leftampl, rightampl, piecediff,
            attacker, attackee);
        if (! blocks[oscix].isdead) {
            blocks[oscix].lampl +=> leftsum ;
            blocks[oscix].rampl +=> rightsum ;
            osccount++ ;
        }
        // <<< "SUMS", leftsum, rightsum >>>;
        if (leftsum > 1.0) {
            (1.0 / leftsum) => leftscale.gain ;
        } else {
            1.0 => leftscale.gain ;
        }
        if (rightsum > 1.0) {
            (1.0 / rightsum) => rightscale.gain ;
        } else {
            1.0 => rightscale.gain ;
        }
    }
}
