# echosustainSineInterpreter.py
# This one starts off with the usual two threads but allows creation
# of two more, one for white and one for black, in order to support
# echoes, continuous delay, etc.

# THE CODE AND DOCUMENTATION IN THIS DIRECTORY AND BELOW ARE OPEN
# SOURCE ACCORDING TO THE TERMS OF THE FOLLOWING LICENSE. PLEASE READ:
# -----------------------------------------------------------------------------
# ESST (Embedded System Software Tools) License
# -----------------------------------------------------------------------------
# 
# SOFTWARE LICENSE
# 
# This software is provided subject to the following terms and conditions,
# which you should read carefully before using the software. Using this
# software indicates your acceptance of these terms and conditions. 
# If you do not agree with these terms and conditions, do not use the software.
# 
# Copyright (c) 2008 Dale E. Parson, Ph.D. (a.k.a. Acoustic Interloper)
# All rights reserved.
# 
# Redistribution and use in source or binary forms, with or without
# modifications, are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following Disclaimer in comments in the code
# as well as in the documentation and/or other materials provided with the
# distribution.
# * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following Disclaimer in the documentation
# and/or other materials provided with the distribution.
# * Neither the name of Dale E. Parson, nor the names of the contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# Disclaimer
# 
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# ANY USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE
# USER'S OWN RISK. IN NO EVENT SHALL Dale E. Parson OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, INCLUDING,
# BUT NOT LIMITED TO, CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -----------------------------------------------------------------------------


import copy
from math import *
from threading import *
import time
from chess.interpreter import interpreter
from chess.chessgame import start_record, stop_record, play_record
from chess.protocol_2 import protocol_2
from chess.ChessException import *
import types

tempo = {}

# This generator takes the timeOfTheMostRecentMove / averageTimePerMove for
# this player as a multiplier for tempogen.
# This "echosustainSineInterpreter.py" generator adds sustain to
# "aheadSineInterpreter.py."

tempofromuser = {'white' : 1.0, 'black' : 1.0}
avgtimefromuser = {'white' : 0.0, 'black' : 0.0}
eventsfromuser = {'white' : 0, 'black' : 0}
lastclock = {'white' : 0.0, 'black' : 0.0}

def updateUserTempo(ucolor):
    global tempofromuser
    global avgtimefromuser
    global eventsfromuser
    global lastclock
    if (ucolor):    # could be None for no tempo modulation
        clk = time.time()
        if (eventsfromuser[ucolor]):    # This is at least the second
            myinterval = clk - lastclock[ucolor]
            totaltime = avgtimefromuser[ucolor] * (eventsfromuser[ucolor]-1) \
                + myinterval
            avgtimefromuser[ucolor] = totaltime / eventsfromuser[ucolor]
            tempofromuser[ucolor] = myinterval / avgtimefromuser[ucolor]
        lastclock[ucolor] = clk
        eventsfromuser[ucolor] = eventsfromuser[ucolor] + 1
        print "DEBUG tempo for " + ucolor + " is " + str(tempofromuser[ucolor])

def tempogen(carrier, swingdiv, phase, steps, mycolor=None):
    global tempofromuser
    if (mycolor):
        userbias = tempofromuser[mycolor]
    else:
        userbias = 1.0
    swing = carrier / swingdiv
    sinoffset = phase     # ramp 0 thru 2pi
    while(1):
        result = carrier + (sin(sinoffset) * swing)
        sinoffset = sinoffset + (pi / steps)
        if (sinoffset > (2 * pi)):
            sinoffset = 0.0
        yield (result * userbias)

temporoot = 0.125
# One per thread for each of these.
tempodelay = {0 : 0.0, 1 : 0.0, 2 : 0.0, 3 : 0.0}
tempolfogain = {0 : 1.0, 1 : 1.0, 2 : 1.0, 3 : 1.0}
numberthreads = 2
numberrunning = numberthreads
playcolor = 'both'

reffreq = 55.0
# conflictfreq = pow(4,1/12.0)    # equal tempered semi-tone. yuck!!!
conflictfreq = 6.0 / 5.0
delay2alterseq = 15.0
octrange = 8    # must be a power of 2 for 'permute' to hit all combinations.
lookaheadlimit = 6
inversionLambda = "lambda x : 1.0 / x"
inversionCode = compile(inversionLambda, '', 'eval')
inversionFunc = eval(inversionCode)
sustainmultiplier = 0
sustainacrossmove = 0
sustainthreshold = 5
sustainrelationships = [1, 3]
permute = {}
# permute runs octaves at play time rather than sequence time, so keep
# pawn interesting since it is used so often
permute['K'] = 3
permute['Q'] = 5
permute['P'] = 3
permute['R'] = 7
permute['N'] = 5
permute['B'] = 1
piecefreq = {}
piecefreq['K'] = 1.0            # King          root
piecefreq['Q'] = 2.0            # Queen         octave
piecefreq['P'] = 1.5            # Pawn          just 5th
piecefreq['R'] = 4.0 / 3.0      # Rook          just 4th
piecefreq['N'] = 5.0 / 4.0      # kNight        just major 3rd
piecefreq['B'] = 27.0 / 16.0    # Bishop        just 6th
# ADD SOME SPECIAL FREQUENCIES FOR NON-CENTRAL PAWNS
piecefreq['P0'] = piecefreq['R']
piecefreq['P7'] = piecefreq['R']
piecefreq['P1'] = piecefreq['N']
piecefreq['P6'] = piecefreq['N']
piecefreq['P2'] = piecefreq['B']
piecefreq['P5'] = piecefreq['B']
# just minor third is 6.0 / 5.0, just major third is 5.0 / 4.0
# just major 2nd is 9.0 / 8.0
conflictamplitude = 1.0
amplgain = 1.0
freqoffset = {}
freqoffset['white'] = 1.0
freqoffset['black'] = 1.0 #  5.0 / 4.0

othercolor = {}
othercolor['white'] = 'black'
othercolor['black'] = 'white'

# accents are patterns of full amplitude, 1/2 or 0 per moved piece
# second set of accents are for when a piece is taken off the board
accents = {}
accents['Q'] = ([1.0, 1.0, 0.0, 1.0, 0.0, 0.0, 1.0, 1.0], []) # 3-32 son clave
accents['R'] = ([0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 0.0, 0.0], []) # bars 1 & 2
accents['K'] = ([1.0, 0.5, 0.5, 1.0, 0.5, 0.5, 1.0, 0.0],[]) # Dale's pattern
accents['B'] = ([1.0, 0.0, 1.0, 0.5], [0.5, 0.0, 0.50, 0.0]) # frailing banjo
accents['N'] = ([0.0, 1.0, 0.5], [0.0, 0.5, 0.25])
accents['P'] = ([1.0, 0.0], [1.0, 0.0])
accents['Q'] = ([1.0, 1.0, 0.25, 1.0, 0.25, 0.25, 1.0, 1.0], []) # 3-32 son clave
accents['R'] = ([0.25, 0.25, 1.0, 1.0, 1.0, 1.0, 0.25, 0.25], []) # bars 1 & 2
accents['K'] = ([1.0, 0.5, 0.5, 1.0, 0.5, 0.5, 1.0, 0.25],[]) # Dale's pattern
accents['B'] = ([1.0, 0.25, 1.0, 0.5], [0.5, 0.25, 0.50, 0.25]) # frailing banjo
accents['N'] = ([0.25, 1.0, 0.5], [0.25, 0.5, 0.25])
accents['P'] = ([1.0, 0.25], [1.0, 0.25])

# OLD STUFF ABOVE, MODIFY TO TRY TO GET POLYRHYTHMS
accents['Q'] = ([1.0, 0.25, 0.25, 1.0, 0.25, 0.25, 1.0, 0.0], [])
accents['R'] = ([1.0, 0.25, 0.5,  0.0, 0.5, 0.0, 1.0, 0.25], [])
accents['K'] = ([1.0, 0.5, 0.5, 1.0, 0.5, 0.5, 1.0, 0.25], [])
accents['B'] = ([1.0, 0.5, 0.25, 0.0, 0.0, 0.25, 0.5, 1.0], [])
accents['N'] = ([1.0, 0.5, 0.25, 0.0, 0.25, 0.5], [0.0, 0.25, 0.5, 1.0, 0.5, 0.25])
accents['P'] = ([1.0, 0.0, 1.0, 0.5], [0.5, 0.0, 0.5, 0.25])

weight = {} # value of piece
weight['Q'] = 9.0
weight['R'] = 5.0
weight['K'] = 10.0
weight['B'] = 3.0
weight['N'] = 2.5
weight['P'] = 1.0

advance = {}
advance['white'] = 1    # direction of advance
advance['black'] = -1

# FIELDS ADDED JUNE 21 for PROTOCOL_2.
piecetype = {
    'P' : 0,
    'N' : 1,
    'B' : 2,
    'R' : 3,
    'Q' : 4,
    'K' : 5 }

player = { 'white' : 0, 'black' : 1}

recordfile = ""

# Coordinates are always X,Y and deltaX,deltaY

class echosustainSineInterpreter(protocol_2):
    def __init__(self, tonegenobj, consparam):
        protocol_2.__init__(self, tonegenobj, consparam)
        exec("from " + consparam[1] + " import transforms")
        self.transforms = transforms
        self.privateshufflegenerator = {}
        self.privateshufflegenerator['white'] = self.privateshufflegen('white')
        self.privateshufflegenerator['black'] = self.privateshufflegen('black')
        self.boardcopy = None   # Used for lookahead on a 'fake board'
        self.graphcopy = None
        self.movercopy = None
        self.sustainedmove = 0
        self.wtthread2 = None
        self.bkthread2 = None
        self.wtthread.start()
        self.bkthread.start()

    def protectedThreadChange(self):
        global numberthreads
        global numberrunning
        self.gate.acquire()
        try:
            if (numberrunning == 2 and numberthreads == 4):
                self.wtthread2 = Thread(target=self.protectedWorkloop, \
                    name="interpreter", args=['white', 2])
                self.wtthread2.setDaemon(True)
                self.bkthread2 = Thread(target=self.protectedWorkloop, \
                    name="interpreter", args=['black', 3])
                self.bkthread2.setDaemon(True)
                numberrunning = 4
                self.wtthread2.start()
                self.bkthread2.start()
            elif (numberrunning == 4 and numberthreads == 2):
                numberrunning = 2
        finally:
            self.gate.release()

    def presend(self, tonelist, sineoscfree, sineoscbusy, sineoscsustain,
                sqoscfree, sqoscbusy, sqoscsustain, sustainsteps):
        """
        Every send counts down all sustained notes and releases those due.
        NOTE: presend() can create side effects by changing freelist, busyset,
        sustainedset and sustainsteps, so if a call to presend occurs in a
        loop over one of these containers, use a copy.copy() for the loop!
        """
        if (len(tonelist) > 10 and tonelist[10]):
            moveno = tonelist[10]
        else:
            moveno = self.sustainedmove
        for osc in copy.copy(sineoscsustain):
            if (sustainsteps[osc] > 0):
                sustainsteps[osc] = sustainsteps[osc] - 1
            elif (sustainsteps[osc] < 0):
                sustainsteps[osc] = 0
            if ((sustainacrossmove == 0 and moveno != self.sustainedmove)   \
                    or (sustainsteps[osc] == 0)):
                self.send([osc, -1.0, -2.0, 0.0, 0.0])
                sustainsteps[osc] = 0
                try:
                    sineoscsustain.remove(osc)
                except KeyError:
                    pass    # presend or all loop calling it may do a remove()
                try:
                    sineoscbusy.remove(osc)
                except KeyError:
                    pass    # presend or all loop calling it may do a remove()
                sineoscfree.append(osc)
        for osc in copy.copy(sqoscsustain):
            if (sustainsteps[osc] > 0):
                sustainsteps[osc] = sustainsteps[osc] - 1
            elif (sustainsteps[osc] < 0):
                sustainsteps[osc] = 0
            if ((sustainacrossmove == 0 and moveno != self.sustainedmove)   \
                    or (sustainsteps[osc] == 0)):
                self.send([osc, -1.0, -2.0, 0.0, 0.0])
                sustainsteps[osc] = 0
                try:
                    sqoscsustain.remove(osc)
                except KeyError:
                    pass    # presend or all loop calling it may do a remove()
                try:
                    sqoscbusy.remove(osc)
                except KeyError:
                    pass    # presend or all loop calling it may do a remove()
                sqoscfree.append(osc)
        self.send(tonelist)
        self.sustainedmove = moveno

    def protectedBuildSequence(self, boardref, mover, reachgraph, notegraph,
            kingpanic, newoctave, isalreadylocked, lastpiece, lastdest,
            movenumber):
        if (not isalreadylocked):
            updateUserTempo(mover.color)    # This user has just moved.
            self.gate.acquire()
        self.boardcopy = copy.copy(boardref)
        self.graphcopy = copy.copy(reachgraph)
        self.movercopy = copy.copy(mover)
        if (not isalreadylocked):
            self.gate.release()
        # 2. Build seqs, using each piece at most once, out from current loc
        newseq = {}
        newseq['white'] = []
        newseq['black'] = []
        norelate = -2
        isupport = 0
        iattack = 1
        supportme = 2
        attackme = 3
        lookahead = 0
        # lookahead general stops before second color in corder is reached,
        # unless it is extended beyond the first color's graph
        if (mover.color == 'white'):
            corder = ['white', 'black']
        else:
            corder = ['black', 'white']
        for col in corder:
            lookahead = 0   # BUILD & MERGE TWO SEQUENCES -- SEE MOVE BELOW
            if (kingpanic[col]):
                print "CHECK ON", col
                # Turn on all notes & their discord for color under 'CHECK!'
                sounds = []
                newseq[col].append(sounds)  # First turn them all off.
                for pp in ['Q', 'K', 'R', 'B', 'N', 'P']:
                    for poct in range(0,5):
                        freq = reffreq * piecefreq[pp] * float(poct)    \
                            * freqoffset[col]
                        # Sound should be 1/30, but let's overdrive
                        od = 25.0
                        noteinfo = [-1, freq, 0.0, 1.0/od, 1.0/od,
                            player[col], piecetype['K'], -2, ((player[col]+1)&1),
                                piecetype['R'], movenumber, 1]
                        sounds.append(noteinfo)
                        freq = freq * conflictfreq
                        noteinfo = [-1, freq, 0.0, 1.0/od, 1.0/od,
                            player[col], piecetype['K'], -2,    \
                                ((player[col]+1)&1),
                                piecetype['R'], movenumber, 1]
                        sounds.append(noteinfo)
                newseq[col].append(sounds)
                continue
            if (lastdest[col] == None or lastpiece[col] == None):
                continue
            leftbias = (mover.color == col)
            visited = set([lastpiece[col]])
            inrvisited = set([])
            nextvisit = [(lastpiece[col], lastdest[col])]
            nextcolor = [col]
            relate = norelate   # first trip thru upcoming loop
            nextactivepiece = [piecetype[lastpiece[col].kind]]
            # nextcolor positionally holds color of the piece thru which
            # its nextvisit piece was reached via the graph,
            # and nextactivepiece holds the piecetype of that piece
            if (lastpiece[col] and lastpiece[col].x != None \
                    and lastpiece[col].y != None):
                notegraph[(lastpiece[col].x, lastpiece[col].y)]     \
                    = reachgraph[(lastpiece[col].x, lastpiece[col].y)]
            while (len(nextvisit)):
                lookahead = lookahead + 1
                if (lookahead > lookaheadlimit):
                    break
                thisvisit = nextvisit
                nextvisit = []
                thiscolor = nextcolor
                nextcolor = []
                activePieceList = nextactivepiece
                nextactivepiece = []
                ampl = amplgain * (0.50 / float(len(thisvisit)))        \
                    / (numberrunning / 2.0)
                sounds = []
                newoctave = ((len(thisvisit) + newoctave) % octrange) + 1
                allreached = 0
                alloccupied = 0
                for p_and_c in thisvisit:
                    pce = p_and_c[0]
                    crd = p_and_c[1]
                    thiscol = thiscolor[0]
                    thiscolor = thiscolor[1:]
                    active_piece = activePieceList[0]
                    activePieceList = activePieceList[1:]
                    pnix = pce.kind + str(pce.index)
                    pncix = pce.kind + str(pce.index) + "(" \
                        + pce.color[0].upper() + ")"
                    if (piecefreq.has_key(pncix)):
                        pfreq = piecefreq[pncix]
                    elif (piecefreq.has_key(pnix)):
                        pfreq = piecefreq[pnix]
                    else:
                        pfreq = piecefreq[pce.kind]
                    targetplayer = player[pce.color] # DEBUG
                    targetpiecetype = piecetype[pce.kind] # DEBUG
                    if (pce.x != None and pce.y != None):
                        if (pce in visited):
                            freqoff = pfreq
                            if (relate >= 0):   # not first trip thru loop
                                relate = isupport
                        else:
                            freqoff = inversionFunc(pfreq)
                            # This inversion occurs if a piece is encountered
                            # as an incoming supporter or attacker before out.
                            if (relate >= 0):   # not first trip thru loop
                                relate = supportme
                        if (pce.color == thiscol):
                            freq = reffreq * freqoff * freqoffset[col]
                            # Don't apply self.octave until playing thread.
                            isconflict = 0
                        else:
                            freq = reffreq * freqoff * freqoffset[col]  \
                                * conflictfreq
                            ampl = ampl * conflictamplitude
                            if (relate >= 0):
                                relate = relate | 1
                            isconflict = 1
                            # Don't apply self.octave until playing thread.
                        if (relate < 0):
                            targetplayer = -1
                            targetpiecetype = -1
                        else:
                            targetplayer = player[pce.color]
                            targetpiecetype = piecetype[pce.kind]
                        if (leftbias):
                            noteinfo = [-1, freq, 0.0, ampl * 0.75, ampl * .25,
                                player[thiscol], active_piece, relate,
                                    targetplayer, targetpiecetype, movenumber,
                                        isconflict]
                        else:
                            noteinfo = [-1, freq, 0.5, ampl * .25, ampl * 0.75,
                                player[thiscol], active_piece, relate,
                                    targetplayer, targetpiecetype, movenumber,
                                        isconflict]
                        # print "DEBUG SHIPPED " + str(player[thiscol]) + str(active_piece) + " " +  str(relate) + " " + str(targetplayer) + str(targetpiecetype)
                        # Current there is a bug where only white or black
                        # gets notes for a typical lookaheadlimit, so
                        # alternate the left and right bias.
                        leftbias = not leftbias
                        relate = isupport
                    else:
                        noteinfo = [-1, 0.0, 0.0, 0.0, 0.0, \
                            player[thiscol], active_piece, 3,
                                player[pce.color], piecetype[pce.kind],
                                    movenumber, 1]
                        # rest after capture
                    sounds.append(noteinfo)
                    reachout = reachgraph[crd][1] | reachgraph[crd][2]
                    reachin = reachgraph[crd][3] | reachgraph[crd][4]
                    reachp = reachout | reachin
                    reachs = reachgraph[crd][5]
                    allreached = allreached + len(reachs)
                    alloccupied = alloccupied + len(reachp)
                    # visit the outgoing at most once, but visit the incoming
                    # a second time if they have not yet been outgoing
                    for p in reachout:
                        if (not p in visited):
                            nextvisit.append((p,(p.x,p.y)))
                            nextcolor.append(pce.color)
                            nextactivepiece.append(piecetype[pce.kind])
                            visited.add(p)
                            notegraph[crd][0] = reachgraph[crd][0]
                            notegraph[crd][1] = reachgraph[crd][1]
                            notegraph[crd][2] = reachgraph[crd][2]
                    for p in reachin:
                        if (not p in visited):
                            if (p in inrvisited):   # don't come here 2d time
                                visited.add(p)
                            else:
                                nextvisit.append((p,(p.x,p.y)))
                                nextcolor.append(pce.color)
                                nextactivepiece.append(piecetype[pce.kind])
                                notegraph[crd][0] = reachgraph[crd][0]
                                notegraph[crd][3] = reachgraph[crd][3]
                                notegraph[crd][4] = reachgraph[crd][4]
                                inrvisited.add(p)
                            # NO! visited.add(p)
                newseq[col].append(sounds)
        # CONCURRENT THREAD PLAYING IS NOISY< SO DUMP THEM ALL TO ONE THREAD.
        ocolor = othercolor[mover.color]
        newseq[mover.color] = newseq[mover.color] + newseq[ocolor]
        newseq[ocolor] = []
        return((newseq, newoctave))

    def protectedWorkloop(self, color, threadnumber):
        global lastclock
        global delay2alterseq
        myaccents = [[], []]
        tempoincr = {'white' : 1.5, 'black' : 1.2}
        if (color == 'white'):
            oscbase = 0
        else:
            oscbase = self.oscperside
        if (threadnumber > 1):
            oscbase = oscbase + (2 * self.oscperbank)
        oscsqoffset = self.oscperbank
        sineoscfree = []        # use these in FIFO order
        sineoscbusy = set([])
        sineoscsustain = set([])
        sqoscfree = []
        sqoscbusy = set([])
        sqoscsustain = set([])
        sustainsteps = {}
        nosteps = {}
        for i in range(oscbase, oscbase + self.oscperside):
            sineoscfree.append(i)
            sqoscfree.append(i + oscsqoffset)
            sustainsteps[i] = 0
            sustainsteps[i + oscsqoffset] = 0
            nosteps[i] = 0
            nosteps[i + oscsqoffset] = 0
        sustainedmove = 0
        amplmul = tempogen(0.6, 1.5, pi / 2.0, 16, color)
        tempo = tempogen(temporoot, 2.0, tempoincr[color] * pi, 32.0, color)
        playoctave = 1
        # number of second to wait before becoming impatient
        basetickstodelay = 8 # Avoids calling clock too often.
        tickstodelay = basetickstodelay
        lastmyclk = time.time()
        self.gate.acquire()
        lastpiece = self.lastpiece
        lastdest = self.lastdest
        iswarble = 0
        warblegen = tempogen(1.0, 1.0, 0.0, 8, color)
        warbwiggle = 0.25    # maxmimum tremolo
        susosclimit = 4 # never sustains more than totalosc / susosclimit
        susstop = len(sineoscfree) / susosclimit
        oldnotecolor = color
        oldnumberrunning = numberrunning
        try:
            while (self.sequence[color] != None         \
                    and numberrunning > threadnumber):
                    # kill() terminates via None
                if (playcolor == 'white' or playcolor == 'black'):
                    if (len(self.sequence[playcolor])):
                        tmplaycolor = playcolor
                    else:
                        tmplaycolor = othercolor[playcolor]
                else:
                    tmplaycolor = playcolor
                if (tmplaycolor == 'white'):
                    notecolor = 'white'
                elif (tmplaycolor == 'black'):
                    notecolor = 'black'
                else:
                    notecolor = color
                ocolor = othercolor[notecolor]
                if (notecolor != oldnotecolor               \
                        or numberrunning != oldnumberrunning):
                    # Aligning or unaligning threads w.r.t. notes and timing,
                    # resync their timing and amplitude generators
                    amplmul = tempogen(0.6, 1.5, pi / 2.0, 16, notecolor)
                    tempo = tempogen(temporoot, 2.0,            \
                        tempoincr[notecolor] * pi, 32.0, notecolor)
                    oldnotecolor = notecolor
                    oldnumberrunning = numberrunning
                while (self.sequence[notecolor] == []):
                    for o in copy.copy(sineoscbusy):
                        if (o in sineoscsustain):
                            try:
                                sineoscsustain.remove(o)
                            except KeyError:
                                pass
                            sustainsteps[o] = 0
                        sineoscfree.append(o)
                        self.presend([o, 0.0, 0.0, 0.0, 0.0], [], set([]),
                            set([]), [], set([]), set([]), nosteps)
                    sineoscbusy.clear()
                    for o in copy.copy(sqoscbusy):
                        if (o in sqoscsustain):
                            try:
                                sqoscsustain.remove(o)
                            except KeyError:
                                pass
                            sustainsteps[o] = 0
                        sqoscfree.append(o)
                        self.presend([o, 0.0, 0.0, 0.0, 0.0], [], set([]),
                            set([]), [], set([]), set([]), nosteps)
                    sqoscbusy.clear()
                    self.gate.wait()
                if (self.sequence[notecolor] == None):
                    continue
                # accents for some captured pieces specify silence as []
                if (lastpiece[notecolor] == None                           \
                        or (lastpiece[notecolor].x == None                 \
                            and accents[lastpiece[notecolor].kind][1] == [])):
                    self.sequence[notecolor] = []
                    continue
                curseq = self.sequence[notecolor]
                playoctave = self.octave
                myaccents[1] = myaccents[0]
                if (lastpiece[notecolor].x == None):   # rhythm for captured
                    myaccents[0] = accents[lastpiece[notecolor].kind][1]
#                elif (lastpiece[ocolor] == None                        \
#                        or weight[lastpiece[notecolor].kind]               \
#                            >= weight[lastpiece[ocolor].kind]          \
#                        or lastpiece[ocolor].x == None):
                else:
                    myaccents[0] = accents[lastpiece[notecolor].kind][0]
#                else:
#                    myaccents[0] = accents[lastpiece[ocolor].kind][0]
                curaccents = copy.copy(myaccents[0])
                curaccents.extend(myaccents[1])
                accentix = 0
                # amplmul = tempogen(0.5, 1.0, pi / 2.0, 16)
                amplscale = amplmul.next()
                while (curseq is self.sequence[notecolor]):
                # interpret() changes self.sequence[notecolor]
                    for notelist in curseq:
                        if (len(notelist) == 0):     # rest, turn off everything
                            # Shut off in-use ones and use new ones
                            for o in copy.copy(sineoscbusy):
                                if (o in sineoscsustain):
                                    sineoscsustain.remove(o)
                                    sustainsteps[o] = 0
                                sineoscfree.append(o)
                                self.presend([o, -1.0, -3.0, 0.0, 0.0],
                                    sineoscfree, sineoscbusy,
                                    sineoscsustain,
                                    sqoscfree, sqoscbusy,
                                    sqoscsustain,
                                    sustainsteps)
                            sineoscbusy = copy.copy(sineoscsustain)
                            for o in copy.copy(sqoscbusy):
                                if (o in sqoscsustain):
                                    sqoscsustain.remove(o)
                                    sustainsteps[o] = 0
                                sqoscfree.append(o)
                                self.presend([o, -1.0, -3.0, 0.0, 0.0],
                                    sineoscfree, sineoscbusy,
                                    sineoscsustain,
                                    sqoscfree, sqoscbusy,
                                    sqoscsustain, sustainsteps)
                            sqoscbusy = copy.copy(sqoscsustain)
                        else:
                            mylimit = len(notelist)
                            if (mylimit < 2):
                                octavelimit = 3
                            elif (mylimit < (octrange / 2)):
                                octavelimit = 2 * mylimit
                            else:
                                octavelimit = octrange
                            # Shut off in-use ones and use new ones
                            for o in copy.copy(sineoscbusy):
                                if (not o in sineoscsustain):
                                    sineoscfree.append(o)
                                    self.presend([o, -1.0, -4.0, 0.0, 0.0],
                                        sineoscfree, sineoscbusy,
                                        sineoscsustain,
                                        sqoscfree, sqoscbusy,
                                        sqoscsustain,
                                        sustainsteps)
                            sineoscbusy = copy.copy(sineoscsustain)
                            for o in copy.copy(sqoscbusy):
                                if (not o in sqoscsustain):
                                    sqoscfree.append(o)
                                    self.presend([o, -1.0, -4.0, 0.0, 0.0],
                                        sineoscfree, sineoscbusy,
                                        sineoscsustain,
                                        sqoscfree, sqoscbusy,
                                        sqoscsustain, sustainsteps)
                            sqoscbusy = copy.copy(sqoscsustain)
                            if (iswarble):
                                warbnext = warblegen.next()
                                warbfactor = (warbnext - 1.0)   \
                                        * warbwiggle
                            else:
                                warbfactor = 0.0
                            for note in notelist:
                                mynote = copy.copy(note[0:11])
                                isconflict = False
                                if (len(note) > 11 and note[11]):
                                    isconflict = True
                                    if (len(sqoscfree)):
                                        newosc = sqoscfree[0]
                                        sqoscfree = sqoscfree[1:]
                                    else:
                                        minsteps = 1000000
                                        newosc = -1
                                        for osc in sqoscbusy:
                                            if (sustainsteps[osc] < minsteps):
                                                minsteps = sustainsteps[osc]
                                                newosc = osc
                                        sustainsteps[newosc] = 0
                                        try:
                                            sqoscbusy.remove(newosc)
                                        except KeyError:
                                            pass
                                        if newosc in sqoscsustain:
                                            sqoscsustain.remove(newosc)
                                        self.presend([newosc, -1.0, -5.0, \
                                            0.0, 0.0],
                                                sineoscfree, sineoscbusy,
                                                sineoscsustain,
                                                sqoscfree, sqoscbusy,
                                                sqoscsustain, sustainsteps)
                                        # DEBUG KEYERROR ALSO CHUCK OOB
                                    sqoscbusy.add(newosc)
                                else:
                                    # support
                                    if (len(sineoscfree)):
                                        newosc = sineoscfree[0]
                                        sineoscfree = sineoscfree[1:]
                                    else:
                                        minsteps = 1000000
                                        newosc = -1
                                        for osc in sineoscbusy:
                                            if (sustainsteps[osc] < minsteps):
                                                minsteps = sustainsteps[osc]
                                                newosc = osc
                                        sustainsteps[newosc] = 0
                                        try:
                                            sineoscbusy.remove(newosc)
                                        except KeyError:
                                            pass
                                        if newosc in sineoscsustain:
                                            sineoscsustain.remove(newosc)
                                        self.presend([newosc, -1.0, -5.0, \
                                            0.0, 0.0],
                                                sineoscfree, sineoscbusy,
                                                sineoscsustain,
                                                sqoscfree, sqoscbusy,
                                                sqoscsustain,
                                                sustainsteps)
                                    sineoscbusy.add(newosc)
                                mynote[0] = newosc
                                mynote[1] = mynote[1] * playoctave
                                if (iswarble):
                                    mynote[1] = mynote[1] \
                                        + warbfactor * mynote[1]
                                playoctave = ((playoctave                   \
                                    + permute[lastpiece[notecolor].kind]   \
                                    - 1) % octavelimit) + 1
                                mynote[3] = mynote[3] * amplscale   \
                                    * curaccents[accentix]
                                mynote[4] = mynote[4] * amplscale   \
                                    * curaccents[accentix]
                                # Add sustain proportional to piece diff.
                                # subject, object, relationship
                                if (mynote[6] < 0 or mynote[9] < 0 \
                                        or mynote[7] < 0    \
                                        or (not mynote[7] in    \
                                            sustainrelationships)):
                                    diff = 0
                                else:
                                    diff = mynote[6] - mynote[9]
                                if (mynote[7] == 0 or mynote[7] == 1):
                                    # object determines priority
                                    diff = - diff
                                sustainval = 0
                                if (diff >= sustainthreshold):
                                    sustainval = diff
                                sustainsteps[newosc] = sustainval   \
                                    * sustainmultiplier
                                if (sustainsteps[newosc]):
                                    if (isconflict):
                                        if (len(sqoscsustain) >= susstop):
                                            sustainsteps[newosc] = 0
                                        else :
                                            sqoscsustain.add(newosc)
                                    else:
                                        if (len(sineoscsustain) >= susstop):
                                            sustainsteps[newosc] = 0
                                        else :
                                            sineoscsustain.add(newosc)
                                self.presend(mynote,
                                    sineoscfree, sineoscbusy,
                                    sineoscsustain,
                                    sqoscfree, sqoscbusy,
                                    sqoscsustain, sustainsteps)
                            accentix = (accentix + 1) % len(curaccents)
                        self.gate.wait((tempolfogain[threadnumber]       \
                            * tempo.next())                      \
                            + (tempodelay[threadnumber] * temporoot))
                        if (not curseq is self.sequence[notecolor]):
                            # Sequence changed due to player.
                            tickstodelay = basetickstodelay
                            lastpiece = self.lastpiece
                            lastdest = self.lastdest
                            break
                        else:
                            # Are we impatient yet?
                            tickstodelay = tickstodelay - 1
                            if (tickstodelay <= 0):
                                tickstodelay = basetickstodelay
                                clk = time.time()
                                secsgoneby = clk - lastmyclk
                                if (secsgoneby > delay2alterseq):
                                    lastmyclk = clk
                                    (lastpiece, lastdest, iswarble) = \
                                        self.privateShuffle(notecolor,  \
                                            lastpiece, lastdest)
                                    break
                    amplscale = amplmul.next()
            for o in copy.copy(sineoscbusy):
                self.presend([o, 0.0, 0.0, 0.0, 0.0], [], set([]), set([]),
                    [], set([]), set([]), nosteps)
            for o in copy.copy(sqoscbusy):
                self.presend([o, 0.0, 0.0, 0.0, 0.0], [], set([]), set([]),
                    [], set([]), set([]), nosteps)
        finally:
            self.gate.release()

    def privateshufflegen(self, color):
        # Make sure the transforms are reversible transforms at the end and
        # also at the halfway point. these also can go up or down an octave.
        if (color == 'white'):
            offset = 0
        else:
            offset = len(self.transforms) / 2
        while (1):
            yield self.transforms[offset]
            offset = (offset + 1 ) % len(self.transforms)

    def privateShuffle(self, color, oldlastpiece, oldlastdest):
        genner = self.privateshufflegenerator[color]
        todo = genner.next()
        print "DEBUG " + color + " shuffle " + todo
        iswarble = 0
        if (todo == 'retro'):
            self.sequence[color].reverse()
        elif (todo == 'warble'):
            iswarble = 1
        elif (todo == 'ahead'):
            fakeboard = {}
            newlastpiece = copy.copy(oldlastpiece)
            newlastdest = copy.copy(oldlastdest)
            for coords in self.boardcopy.keys():
                # Not enough to copy the board; we must copy the pieces
                # in case they are moving in their virtual world.
                fakeboard[coords] = copy.copy(self.boardcopy[coords])
                if (fakeboard[coords] and \
                        oldlastpiece[fakeboard[coords].color] \
                            is self.boardcopy[coords]):
                    newlastpiece[fakeboard[coords].color] = fakeboard[coords]
                    newlastdest[fakeboard[coords].color] = coords
            for coords in fakeboard.keys():
                fakep = fakeboard[coords]
                if (fakep):
                    fakep.x = coords[0]
                    fakep.y = coords[1]
            # Move the opposite side from the side most recently moved.
            movepiece = None
            movespace = None
            value = 100
            # Propose taking mover with a piece with the least weight.
            movespace = (self.movercopy.x, self.movercopy.y)
            for opponent in \
                    self.graphcopy[(self.movercopy.x,self.movercopy.y)][4]:
                # There could be a race; graphcopy has object refs!
                # Its should be safe if somewhat dated>
                # If a player moves while we do this, we project future from
                # a copy of the past.
                if (weight[opponent.kind] < value):
                    movepiece = opponent
                    value = weight[opponent.kind]
            if (movepiece):
                movepiece = copy.copy(movepiece)
                print "DEBUG LOOKAHEAD " + str(movepiece) + " to "  \
                    + str(movespace)
                fakeboard[(movepiece.x, movepiece.y)] = None
                fakeboard[movespace] = movepiece
                movepiece.x = movespace[0]
                movepiece.y = movespace[1]
                saveseq = copy.copy(self.sequence)
                try:
                    newlastpiece[movepiece.color] = movepiece
                    newlastdest[movepiece.color] = copy.copy(movespace)
                    self.protectedOuterInterpret(fakeboard, movepiece,  \
                        newlastpiece, newlastdest, 0)
                    if (not (len(self.sequence['white'])        \
                            or len(self.sequence['black']))):
                        self.sequence = saveseq
                    elif (len(self.sequence[color]) == 0):
                        swapseq = {'white' : self.sequence['black'],    \
                            'black' : self.sequence['white']}
                        self.sequence = swapseq
                except ChessCheckException:
                    self.sequence = saveseq
                return ((newlastpiece, newlastdest, iswarble))
            # No pieces attack me, take the one I attack with greatest value.
            value = 0
            for opponent in \
                    self.graphcopy[(self.movercopy.x,self.movercopy.y)][2]:
                if (weight[opponent.kind] > value):
                    movepiece = opponent
                    movespace = (opponent.x, opponent.y)
                    value = weight[opponent.kind]
            if (movepiece):
                # movepiece is going off the board, replace it
                movepiece = copy.copy(self.movercopy)
                print "DEBUG LOOKAHEAD " + str(movepiece) + " to "  \
                    + str(movespace)
                fakeboard[(movepiece.x, movepiece.y)] = None
                fakeboard[movespace] = movepiece
                movepiece.x = movespace[0]
                movepiece.y = movespace[1]
                saveseq = copy.copy(self.sequence)
                try:
                    newlastpiece[movepiece.color] = movepiece
                    newlastdest[movepiece.color] = movespace
                    self.protectedOuterInterpret(fakeboard, movepiece,  \
                        newlastpiece, newlastdest, 0)
                    if (not (len(self.sequence['white'])        \
                            or len(self.sequence['black']))):
                        self.sequence = saveseq
                    elif (len(self.sequence[color]) == 0):
                        swapseq = {'white' : self.sequence['black'],    \
                            'black' : self.sequence['white']}
                        self.sequence = swapseq
                except ChessCheckException:
                    self.sequence = saveseq
                return ((newlastpiece, newlastdest, iswarble))
            # If we get here, just make some arbitrary move.
            for movespace in    \
                    self.graphcopy[(self.movercopy.x,self.movercopy.y)][5]:
                movepiece = copy.copy(self.movercopy)
                print "DEBUG LOOKAHEAD " + str(movepiece) + " to "  \
                    + str(movespace)
                fakeboard[(movepiece.x, movepiece.y)] = None
                fakeboard[movespace] = movepiece
                movepiece.x = movespace[0]
                movepiece.y = movespace[1]
                saveseq = copy.copy(self.sequence)
                try:
                    newlastpiece[movepiece.color] = movepiece
                    newlastdest[movepiece.color] = movespace
                    self.protectedOuterInterpret(fakeboard, movepiece,  \
                        newlastpiece, newlastdest, 0)
                    if (not (len(self.sequence['white'])        \
                            or len(self.sequence['black']))):
                        self.sequence = saveseq
                    elif (len(self.sequence[color]) == 0):
                        swapseq = {'white' : self.sequence['black'],    \
                            'black' : self.sequence['white']}
                        self.sequence = swapseq
                except ChessCheckException:
                    self.sequence = saveseq
                return ((newlastpiece, newlastdest, iswarble))
            return((oldlastpiece, oldlastdest, iswarble))
        else:
            if (todo == 'up5th'):
                mult = 1.5
            elif (todo == 'down5th'):
                mult = 2.0 / 3.0
            elif (todo == 'up4th'):
                mult = 4.0 / 3.0
            elif (todo == 'down4th'):
                mult = 0.75
            elif (todo == 'up3rd'):
                mult = 5.0 / 4.0
            elif (todo == 'down3rd'):
                mult = 4.0 / 5.0
            elif (todo == 'upmin3rd'):
                mult = 6.0 / 5.0
            elif (todo == 'downmin3rd'):
                mult = 5.0 / 6.0
            elif (todo == 'up6th'):
                mult = 27.0 / 16.0
            elif (todo == 'down6th'):
                mult = 16.0 / 27.0
            elif (todo == 'upoctave'):
                mult = 2.0
            elif (todo == 'downoctave'):
                mult = 0.5
            else:
                # raise ChessException, "Invalid interval adjustment: " + todo
                print "ERROR, Invalid interval adjustment: " + todo
                return((oldlastpiece, oldlastdest, iswarble))
            for inotelist in range(0, len(self.sequence[color])):
                for inote in range(0, len(self.sequence[color][inotelist])):
                    self.sequence[color][inotelist][inote][1]       \
                        = self.sequence[color][inotelist][inote][1] * mult
        self.sequence[color] = copy.copy(self.sequence[color])
        return((oldlastpiece, oldlastdest, iswarble))

    def getMusicGenState(self):
        result =            \
            "lookaheadlimit = " + str(lookaheadlimit)   + "\n"      \
            "sustainmultiplier = " + str(sustainmultiplier)   + "\n"      \
            "sustainacrossmove = " + str(sustainacrossmove)   + "\n"      \
            "sustainthreshold = " + repr(sustainthreshold)   + "\n"      \
            "sustainrelationships = " + repr(sustainrelationships)   + "\n" \
            "playcolor = " + str(playcolor)   + "\n"    \
            "delay2alterseq = " + str(delay2alterseq)   + "\n"          \
            "numberthreads = " + repr(numberthreads)   + "\n" \
            "amplgain = " + str(amplgain)    + "\n"      \
            "recordfile = " + repr(recordfile)    + "\n"      \
            "reffreq = " + str(reffreq)   + "\n"                    \
            "conflictfreq = " + str(conflictfreq)   + "\n"          \
            "octrange = " + str(octrange)   + "\n"
        result = result + 'inversionLambda = ' + inversionLambda
        if (inversionLambda.find('#') == -1):
            result = result + \
                "   # a single-parameter numeric lambda expression"
        result = result + "\n"
        ixs = tempodelay.keys()
        ixs.sort()
        for ix in ixs:
            frac = repr(tempodelay[ix])
            result = result + "tempodelay[" + repr(ix) + "] = " + frac + "\n"
        ixs = tempolfogain.keys()
        ixs.sort()
        for ix in ixs:
            frac = repr(tempolfogain[ix])
            result = result + "tempolfogain[" + repr(ix) + "] = " + frac + "\n"
        ixs = piecefreq.keys()
        ixs.sort()
        for ix in ixs:
            if (piecefreq[ix] == 4.0 / 3.0):
                frac = "4.0 / 3.0              # (Just 4th)"
            elif (piecefreq[ix] == 1.5):
                frac = "3.0 / 2.0              # (Just 5th)"
            elif (piecefreq[ix] == 5.0 / 4.0):
                frac = "5.0 / 4.0              # (Just Major 3rd)"
            elif (piecefreq[ix] == 27.0 / 16.0):
                frac = "27.0 / 16.0            # (Just 6th)"
            elif (piecefreq[ix] == 6.0 / 5.0):
                frac = "5.0 / 4.0              # (Just Minor 3rd)"
            else:
                frac = repr(piecefreq[ix])
            result = result + "piecefreq[" + ix + "] = " + frac + "\n"
        ixs = accents.keys()
        ixs.sort()
        for ix in ixs:
            result = result + "accents[" + ix + "] = " + repr(accents[ix]) \
                + "\n"
        ixs = freqoffset.keys()
        ixs.sort()
        for ix in ixs:
            if (freqoffset[ix] == 4.0 / 3.0):
                frac = "4.0 / 3.0              # (Just 4th)"
            elif (freqoffset[ix] == 1.5):
                frac = "3.0 / 2.0              # (Just 5th)"
            elif (freqoffset[ix] == 5.0 / 4.0):
                frac = "5.0 / 4.0              # (Just Major 3rd)"
            elif (freqoffset[ix] == 27.0 / 16.0):
                frac = "27.0 / 16.0            # (Just 6th)"
            elif (freqoffset[ix] == 6.0 / 5.0):
                frac = "5.0 / 4.0              # (Just Minor 3rd)"
            else:
                frac = repr(freqoffset[ix])
            result = result + "freqoffset[" + ix + "] = " + frac + "\n"
        return result

    def setMusicGenState(self, assignment):
        global lookaheadlimit
        global reffreq
        global conflictfreq
        global delay2alterseq
        global octrange
        global piecefreq
        global accents
        global inversionLambda
        global inversionCode
        global inversionFunc
        global freqoffset
        global sustainmultiplier
        global sustainacrossmove
        global sustainthreshold
        global sustainrelationships
        global temodelay
        global tempolfogain
        global numberthreads
        global playcolor
        global amplgain
        global recordfile

        assign = assignment.split('=')
        if (len(assign) != 2):
            return "Invalid assignment format: " + assignment
        lexpr = assign[0].strip()
        lmap = lexpr.split('[')
        lvar = lmap[0].strip()
        if (len(lmap) > 1):
            mindex = lmap[1].split(']')[0].strip()
        else:
            mindex = ""
        rexpr = assign[1].strip()
        try:
            if (lvar == 'inversionLambda'):
                mylambda = ""
                code = compile(rexpr, '', 'eval')
                func = eval(code)
                if (not (type(func) == types.FunctionType       \
                        and func.func_code.co_argcount == 1)):
                    raise ChessException,   \
                        "inversionLambda is not a single-parameter lambda: " \
                            + assignment
                inversionLambda = rexpr
                inversionCode = code
                inversionFunc = func
            elif (lvar == 'lookaheadlimit'):
                if (mindex):
                    raise ChessException,   \
                        "lookaheadlimit does not take a subscript: " \
                            + assignment
                # print "DEBUG ABOUT TO TAKE THE INT OF " + str(rexpr)
                tmp = int(eval(rexpr))
                # print "DEBUG TOOK INT " + str(tmp)
                if (tmp < 0):
                    raise ChessException, "Invalid lookaheadlimit: "    \
                        + str(tmp)
                lookaheadlimit = tmp
                # print "DEBUG SAVED INT " + str(lookaheadlimit)
            elif (lvar == 'sustainmultiplier'):
                if (mindex):
                    raise ChessException,   \
                        "sustainmultiplier does not take a subscript: " \
                            + assignment
                # print "DEBUG ABOUT TO TAKE THE INT OF " + str(rexpr)
                tmp = int(eval(rexpr))
                # print "DEBUG TOOK INT " + str(tmp)
                if (tmp < 0):
                    raise ChessException, "Invalid sustainmultiplier: "    \
                        + str(tmp)
                sustainmultiplier = tmp
                # print "DEBUG SAVED INT " + str(sustainmultiplier)
            elif (lvar == 'sustainacrossmove'):
                if (mindex):
                    raise ChessException,   \
                        "sustainacrossmove does not take a subscript: " \
                            + assignment
                # print "DEBUG ABOUT TO TAKE THE INT OF " + str(rexpr)
                tmp = int(eval(rexpr))
                # print "DEBUG TOOK INT " + str(tmp)
                if (tmp < 0):
                    raise ChessException, "Invalid sustainacrossmove: "    \
                        + str(tmp)
                sustainacrossmove = tmp
                # print "DEBUG SAVED INT " + str(sustainacrossmove)
            elif (lvar == 'playcolor'):
                if (mindex):
                    raise ChessException,   \
                        "playcolor does not take a subscript: " \
                            + assignment
                tmp = str(rexpr)
                # print "DEBUG TOOK INT " + str(tmp)
                playcolor = tmp
                # print "DEBUG SAVED INT " + str(playcolor)
            elif (lvar == 'sustainthreshold'):
                if (mindex):
                    raise ChessException,   \
                        "sustainthreshold does not take a subscript: " \
                            + assignment
                # print "DEBUG ABOUT TO TAKE THE INT OF " + str(rexpr)
                tmp = int(eval(rexpr))
                # print "DEBUG TOOK INT " + str(tmp)
                if (tmp < 1 or tmp > 5):
                    raise ChessException, "Invalid sustainthreshold: "    \
                        + str(tmp)
                sustainthreshold = tmp
                # print "DEBUG SAVED INT " + str(sustainthreshold)
            elif (lvar == 'recordfile'):
                if (mindex):
                    raise ChessException,   \
                        "recordfile does not take a subscript: " \
                            + assignment
                tmp = str(rexpr).strip()
                if (tmp):
                    start_record(tmp)
                else:
                    stop_record()
                recordfile = tmp
            elif (lvar == 'sustainrelationships'):
                if (mindex):
                    raise ChessException,   \
                        "sustainrelationships does not take a subscript: " \
                            + assignment
                # print "DEBUG ABOUT TO TAKE THE LIST OF " + str(rexpr)
                tmp = list(eval(rexpr))
                # print "DEBUG TOOK LIST " + str(tmp)
                validset = set([0, 1, 2, 3])
                for v in tmp:
                    if (type(v) != int or not v in validset):
                        raise ChessException, "Invalid sustainrelationships: " \
                            + str(tmp)
                sustainrelationships = tmp
                # print "DEBUG SAVED INT " + str(sustainrelationships)
            elif (lvar == 'reffreq'):
                if (mindex):
                    raise ChessException,   \
                        "reffreq does not take a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp <= 0.0):
                    raise ChessException, "Invalid reffreq: " + str(tmp)
                reffreq = tmp
            elif (lvar == 'conflictfreq'):
                if (mindex):
                    raise ChessException,   \
                        "conflictfreq does not take a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp <= 0.0):
                    raise ChessException, "Invalid conflictfreq: " + str(tmp)
                conflictfreq = tmp
            elif (lvar == 'delay2alterseq'):
                if (mindex):
                    raise ChessException,   \
                        "delay2alterseq does not take a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp <= 0.0):
                    raise ChessException, "Invalid delay2alterseq: " + str(tmp)
                delay2alterseq = tmp
            elif (lvar == 'amplgain'):
                if (mindex):
                    raise ChessException,   \
                        "amplgain does not take a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp < 0.0):
                    raise ChessException, "Invalid amplgain: " + str(tmp)
                amplgain = tmp
            elif (lvar == 'numberthreads'):
                if (mindex):
                    raise ChessException,   \
                        "numberthreads does not take a subscript: " \
                            + assignment
                tmp = int(eval(rexpr))
                if (tmp != 2 and tmp != 4):
                    raise ChessException, "Invalid numberthreads: " + str(tmp)
                numberthreads = tmp
                self.protectedThreadChange()
            elif (lvar == 'octrange'):
                if (mindex):
                    raise ChessException,   \
                        "octrange does not take a subscript: " \
                            + assignment
                tmp = int(eval(rexpr))
                if (tmp < 1):
                    raise ChessException, "Invalid octrange: "    \
                        + str(tmp)
                tmp2 = tmp
                bitcount = 0
                while (tmp2):
                    if (tmp2 & 1):
                        bitcount = bitcount + 1
                    tmp2 = tmp2 >> 1
                if (bitcount != 1):
                    raise ChessException, "octrange must be a power of 2: "  \
                        + str(tmp)
                octrange = tmp
            elif (lvar == 'tempodelay'):
                if (not mindex):
                    intindex = -1
                else:
                    intindex = int(eval(mindex))
                    if (intindex < 0 or intindex > 3):
                        raise ChessException,   \
                            "invalid integer subscript: " \
                                + assignment
                tmp = float(eval(rexpr))
                if (tmp < 0.0 or tmp > 100.0):
                    raise ChessException,   \
                        "invalid float value: " \
                            + assignment
                if (intindex < 0):
                    tempodelay[0] = tmp
                    tempodelay[1] = tmp
                    tempodelay[2] = tmp
                    tempodelay[3] = tmp
                else:
                    tempodelay[intindex] = tmp
            elif (lvar == 'tempolfogain'):
                if (not mindex):
                    intindex = -1
                else:
                    intindex = int(eval(mindex))
                    if (intindex < 0 or intindex > 3):
                        raise ChessException,   \
                            "invalid integer subscript: " \
                                + assignment
                tmp = float(eval(rexpr))
                if (tmp < 0.0 or tmp > 100.0):
                    raise ChessException,   \
                        "invalid float value: " \
                            + assignment
                if (intindex < 0):
                    tempolfogain[0] = tmp
                    tempolfogain[1] = tmp
                    tempolfogain[2] = tmp
                    tempolfogain[3] = tmp
                else:
                    tempolfogain[intindex] = tmp
            elif (lvar == 'accents'):
                if (not mindex):
                    raise ChessException,   \
                        "accents requires a subscript: " \
                            + assignment
                ftuple = eval(rexpr)
                if (len(ftuple) != 2):
                    raise ChessException,   \
                        "accents must be a list of two lists of floats: "   \
                            + str(rexpr)
                reslist = []
                for flist in ftuple:
                    sublist = []
                    for e in flist:
                        val = float(e)
                        if (val < 0.0 or val > 1.0):
                            raise ChessException,   \
                                'accents values must range from 0.0 to 1.0: ' \
                                    + str(rexpr)
                        sublist.append(val)
                    reslist.append(sublist)
                # eval(lexpr + " = " + repr(tuple(reslist)))
                # eval(assignment)
                accents[mindex] = reslist
            elif (lvar == 'piecefreq'):
                if (not mindex):
                    raise ChessException,   \
                        "piecefreq requires a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp <= 0.0):
                    raise ChessException, "Invalid piecefreq: " + str(tmp)
                # eval(lexpr + " = " + repr(tmp))
                # eval(assignment)
                piecefreq[mindex] = tmp
            elif (lvar == 'freqoffset'):
                if (not mindex):
                    raise ChessException,   \
                        "freqoffset requires a subscript: " \
                            + assignment
                tmp = float(eval(rexpr))
                if (tmp <= 0.0):
                    raise ChessException, "Invalid freqoffset: " + str(tmp)
                # eval(lexpr + " = " + repr(tmp))
                # eval(assignment)
                freqoffset[mindex] = tmp
            else:
                raise ChessException, "Invalid music mapping variable: " \
                    + lvar
        except ValueError, verr:
            return "Value Format Error: " + str(verr)
        except Exception, xerr:
            return "Assignment error: " + str(xerr)
        # print "DEBUG UPDATE RETURNS SAFELY"
        return ""
