# chess/interpreter.py

# THE CODE AND DOCUMENTATION IN THIS DIRECTORY AND BELOW ARE OPEN
# SOURCE ACCORDING TO THE TERMS OF THE FOLLOWING LICENSE. PLEASE READ:
# -----------------------------------------------------------------------------
# ESST (Embedded System Software Tools) License
# -----------------------------------------------------------------------------
# 
# SOFTWARE LICENSE
# 
# This software is provided subject to the following terms and conditions,
# which you should read carefully before using the software. Using this
# software indicates your acceptance of these terms and conditions. 
# If you do not agree with these terms and conditions, do not use the software.
# 
# Copyright (c) 2008 Dale E. Parson, Ph.D. (a.k.a. Acoustic Interloper)
# All rights reserved.
# 
# Redistribution and use in source or binary forms, with or without
# modifications, are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following Disclaimer in comments in the code
# as well as in the documentation and/or other materials provided with the
# distribution.
# * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following Disclaimer in the documentation
# and/or other materials provided with the distribution.
# * Neither the name of Dale E. Parson, nor the names of the contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# Disclaimer
# 
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# ANY USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE
# USER'S OWN RISK. IN NO EVENT SHALL Dale E. Parson OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, INCLUDING,
# BUT NOT LIMITED TO, CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -----------------------------------------------------------------------------

import copy
from math import *
from os import access, R_OK, X_OK
# import socket
from threading import *
# import sys
from types import TypeType
from chess.ChessException import *

class interpreter(object):
    """
    class 'interpreter' analyzes a COPY of the board and sends notes off to
    the renderer. May, 2008: interpreter is now an abstract base class.
    A subclass must supply methods protectedBuildSequence to build a sequence
    and protectedWorkloop to output the sequence (with possible deltas)
    to the sound generator.

    Each concrete derived class constructor must invoke the following two
    lines as the last steps in order to start the two music generation threads:

        self.wtthread.start()
        self.bkthread.start()
    """
    def __init__(self, tonegenobj, consparam):
        """
        Constructor starts 2 service threads that wait for 'interpret'
        messages with a copy of the board. It uses a lock and is thread safe.
        The tonegenobj is the object of class tonegen.
        """
        self.tonegenobj = tonegenobj
        self.consparam = consparam
        self.wtthread = Thread(target=self.protectedWorkloop, \
            name="interpreter", args=['white', 0])
        self.wtthread.setDaemon(True)
        self.bkthread = Thread(target=self.protectedWorkloop, \
            name="interpreter", args=['black', 1])
        self.bkthread.setDaemon(True)
        self.sequence = {}
        self.sequence['white'] = []
        self.sequence['black'] = []
        self.lastdest = {}
        self.lastdest['white'] = None
        self.lastdest['black'] = None
        self.lastpiece = {}
        self.lastpiece['white'] = None
        self.lastpiece['black'] = None
        self.gate = Condition()
        self.octave = 1
        self.dumpreachfile = None
    def interpret(self, boardref, mover, movenumber):
        """
        A client invokes interpret with a reference to the board, which
        must not change during the invocation of interpret. Interpret
        extracts musical information from this board and passes it
        to a worker thread to send to the renderer via tonegen.send().
        When interpret() returns it is safe for the client to modify the
        board.

        The mover parameter is the piece that just completed the move.
        Neither it nor any piece must move until after interpret() returns.

        May, 2008: Raises ChessCheckException if a color has moved its own
        King into check, BUT it only raises this at the end, i.e., after
        completing all sequence construction for music generation. The caller
        can allow self-induced check by handling ChessCheckException.
        """
        self.lastpiece[mover.color] = mover
        self.lastdest[mover.color] = (mover.x, mover.y)
        return self.protectedInnerInterpret(boardref, mover, False, \
            self.lastpiece, self.lastdest, movenumber)

    def protectedOuterInterpret(self, boardref, mover, lastpiece, lastdest,
            movenumber):
        """ Helper method for recursive calls from protectedWorkloop(). """
        self.protectedInnerInterpret(boardref, mover, True, \
            lastpiece, lastdest, movenumber)
        return None

    def protectedInnerInterpret(self, boardref, mover, isalreadylocked, \
            lastpiece, lastdest, movenumber):
        """ Helper method for recursive calls from protectedWorkloop().
           isalreadylocked means a recursive call with a virtual move from
           protectedWorkloop(); don't treat it as permamnent state.
        """
        # 1. For each space on the board, while a list of tuples holding:
        #   0 reference to occupying piece or None
        #   1 (set I support)
        #   2 (set I attack)
        #   3 (set who support me)
        #   4 (set who attack me)
        #   5 spaces that I can reach (coordinates)
        minx = 8
        miny = 8
        maxx = -1
        maxy = -1
        kingpanic = {}
        kingpanic['white'] = False
        kingpanic['black'] = False
        # Use reachedby of coords to check for a king crossing check.
        reachedby = {}
        reachedby['white'] = set([])
        reachedby['black'] = set([])
        reachgraph = {}
        notegraph = {}
        ocolor = {'white' : 'black', 'black' : 'white'}
        newoctave = self.octave
        for x in range(0,8):
            for y in range(0,8):
                reachgraph[(x,y)]                                           \
                    = [None, set([]), set([]), set([]), set([]), set([])]
                notegraph[(x,y)]                                           \
                    = [None, set([]), set([]), set([]), set([]), set([])]
        for x in range(0,8):
            for y in range(0,8):
                if (boardref[(x,y)]):
                    if (x < minx):
                        minx = x
                    if (x > maxx):
                        maxx = x
                    if (y < miny):
                        miny = y
                    if (y > maxy):
                        maxy = y
                    p = boardref[(x,y)]
                    reachgraph[(x,y)][0] = p
                    attackset, supportset, coordset, pawnset, rookset       \
                        = p.getAttackSupportMove()
                    reachgraph[(x,y)][1] = supportset
                    reachgraph[(x,y)][2] = attackset
                    reachgraph[(x,y)][5] = coordset
                    reachedby[boardref[(x,y)].color].update(coordset)
                    for other in supportset:
                        ox = other.x
                        oy = other.y
                        if (ox != None and oy != None):
                            reachgraph[(ox,oy)][3].add(p)
                    for other in attackset:
                        ox = other.x
                        oy = other.y
                        if (ox != None and oy != None):
                            reachgraph[(ox,oy)][4].add(p)
        for x in range(0,8):
            for y in range(0,8):
                # print "DEBUG at", x, y, str(reachgraph[(x,y)])
                if (boardref[(x,y)]):
                    p = boardref[(x,y)]
                    if (p.kind == 'K' and len(reachgraph[(x,y)][4])):
                        kingpanic[p.color] = reachgraph[(x,y)][4]
                        print "\n\nKING PANIC", p.color, "\n\n"
        if (mover.kind == 'K'):
            castlecrossings = mover.getCastleCrossings().intersection(  \
                reachedby[ocolor[mover.color]])
            if (len(castlecrossings)):
                kingpanic[mover.color] = True
                print "\n\nKING PANIC ON CASTLING", mover.color, "\n\n"
        # 2. Build seqs, using each piece at most once, out from current loc
        (newseq, newoctave) = self.protectedBuildSequence(boardref,
            mover, reachgraph, notegraph, kingpanic, newoctave, isalreadylocked,
            lastpiece, lastdest, movenumber)
        if (not isalreadylocked):
            self.gate.acquire()
        if ((not isalreadylocked) or len(newseq) != 0):
            # Do not silence on virtual lookahead.
            self.sequence = newseq
        # print "DEBUG NEWSEQ " + str(newseq)
        self.octave = newoctave
        self.board = copy.copy(boardref)
        if (not isalreadylocked):
            self.gate.notifyAll()
            self.gate.release()
            # self.dumpReachGraph(notegraph, 'NOTEGRAPH')
            # self.dumpReachGraph(reachgraph, 'REACHGRAPH')
        if (kingpanic[mover.color]):
            raise ChessCheckException, mover.color \
                + " would enter or stay in check!"
        if (isalreadylocked and (kingpanic['white'] or kingpanic['black'])):
            # Virtual board lookahead is not allowed to cause a check!
            raise ChessCheckException, mover.color \
                + " would enter or stay in check!"
        return ((notegraph, kingpanic))

    def protectedWorkloop(self, color, threadnumber):
        raise Exception, "protectedWorkloop is an abstract method"

    def dumpReachGraph(self, graph, label):
        if (not self.dumpreachfile):
            self.dumpreachfile = open('dumpreachfile.txt','a')
        self.dumpreachfile.write("\nDUMPING " + label + "\n")
        for x in range(0,8):
             for y in range(0,8):
                if (graph[(x,y)] and graph[(x,y)][0]):
                    self.dumpreachfile.write("X = " + str(x) + ", Y = "    \
                        + str(y) + ": " + str(graph[(x,y)][0]) + "\n")
                    name = {1 : 'I SUPPORT', 2 : 'I ATTACK',
                        3 : 'SUPPORT ME', 4 : 'ATTACK ME'}
                    for rel in range(1,5):
                        if (graph[(x,y)][rel]):
                            self.dumpreachfile.write("\t"      \
                                + name[rel] + "\n")
                            for op in graph[(x,y)][rel]:
                                self.dumpreachfile.write("\t\t"    \
                                    + str(op) + "\n")
        self.dumpreachfile.flush()

    def silence(self):
        self.gate.acquire()
        self.sequence['white'] = []
        self.sequence['black'] = []
        self.gate.notifyAll()
        self.gate.release()

    def kill(self):
        self.gate.acquire()
        self.sequence['white'] = None
        self.sequence['black'] = None
        self.gate.notifyAll()
        self.gate.release()

    def getMusicGenState(self):
        """
        Get human-readable music generation statevars, '' if not supported.
        """
        return ''

    def setMusicGenState(self, assignment):
        """
        eval an assignment to a state variable, return True if not supported
        and an error string on error, else return ""
        """
        return True

def ifactory(classname, tonegenner, consparam):
    module = "chess." + classname
    filename = classname + ".py"
    if (not access(filename, R_OK)):
        raise Exception, "class not found: " + classname
    exec "from " + module + " import " + classname
    consptr = eval(classname)
    if (not (type(consptr) == TypeType)):
        raise Exception, classname + " is not a valid class"
    result = apply(consptr, (tonegenner,consparam), {})
    if (not isinstance(result, interpreter)):
         raise Exception, classname + " is not derived from interpreter"
    return result
