import threading

# Load the following from a config file at some point.
# Or possibly set some from a GUI.
tempo = 0.25    # number of seconds per beat
reffreq = 110.0
piecefreq = {}
piecefreq['K'] = 1.0            # King          root
piecefreq['Q'] = 2.0            # Queen         octave
piecefreq['P'] = 1.5            # Pawn          just 5th
piecefreq['R'] = 4.0 / 3.0      # Rook          just 4th
piecefreq['N'] = 9.0 / 8.0      # kNight        just 2nd
piecefreq['B'] = 27.0 / 16.0    # Bishop        just 6th
# conflictfreq = 6.0 / 5.0        # just minor third, offset for attack
conflictfreq = 10.0 / 9.0        # just minor third, offset for attack
# just major third is 5.0 / 4.0

columns = ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h'] # white's left to right
rows = range(1,9)   # white to black
boardlength = 8 # 8 spaces
# Internally we use 0..7 for rows and columns. 'a'..'h' and 1..8 are
# just used to report results.
advance = {}
advance['white'] = 1    # direction of advance
advance['black'] = -1

# Coordinates are always X,Y and deltaX,deltaY

board = {}
lastmove = []   # record start and end coords of latest move, dest. at end
class piece(object):
    def __init__(self, color, kind, index, x, y):
        self.color = color
        self.kind = kind
        self.index = index
        self.x = x
        self.y = y
        self.moved = False
        self.lock = threading.RLock()
    def move(self, x, y, specialpiece=None):
        """
        Move this piece on the board, removing any piece occupying (x,y).
        This method does not check validity of the move. It is the caller's
        responsibility to invoke getAttackSupportMove() to determine
        valid moves. Optional parameter specialpiece when specified can
        be a reference to a pawn to capture en passant by this pawn, or
        a rook with which to castle by this king; move() correctly moves
        the en passant pawn off the board, or moves the rook, before moving
        this piece. It does not check the validity of these compound moves.

        The specialpiece parameter can also be one of 'Q' 'R' 'N' 'B'
        strings for promotion when a pawn moves to the final opposing rank.
        In this case move() removes the pawn and creates the new piece at
        the new (x, y) without error checking.

        Return value is a reference to the moved piece, which is the piece
        invoked with move() unless the moved piece is a pawn to be promoted,
        in which case a reference to the new piece returns.
        """
        if (self.x == None or self.y == None):
            raise Exception, "piece is not on the board: " + str(self)
        self.lock.acquire()
        result = self
        before = (self.x, self.y)
        after = (x, y)
        if ((x != self.x or y != self.y) and (board[(self.x,self.y)] == self)):
            board[(self.x,self.y)] = None
            if (isinstance(specialpiece, piece)):
                if (self.kind == 'P' and specialpiece.kind == 'P'):
                    specialpiece.move(None, None)
                elif (self.kind == 'K' and specialpiece.kind == 'R'):
                    if (specialpiece.x < x):
                        rookx = x + 1
                    else:
                        rookx = x - 1
                    specialpiece.move(rookx, y)
                else:
                    self.lock.release()
                    raise Exception, "invalid castle or en passant piece: "  \
                        + str(specialpiece)
            self.x = x
            self.y = y
            if (x != None and y != None):   # None is off the board
                if (board[(self.x,self.y)]):
                    board[(self.x,self.y)].move(None, None) # take off board
                if (self.kind == 'P' and (self.y == 0 or self.y == 7)):
                    if (type(specialpiece) == str                       \
                            and str in set(['Q', 'B', 'N', 'R'])):
                        newkind = specialpiece
                    else:
                        newkind = 'Q'
                    newpiece = piece(self.color, newkind, 0, self.x, self.y)
                    newpiece.moved = True
                    board[(self.x, self.y)] = newpiece
                    self.x = None
                    self.y = None
                    result = newpiece
                else:
                    board[(self.x,self.y)] = self
                lastmove = [before, after]
                self.moved = True
        else:
            self.lock.release()
            raise Exception, "invalid move of " + self.color + " "  \
                + self.kind + " " + str(self.index) + " to "        \
                + str((x,y))
        self.lock.release()
        return result
    def getAttackSupportMove(self):
        """
        Return 5 tuple of:
            set of pieces I can attack
            set of pieces I support (similar to attack but my color)
            coordinates I can move to (including spaces under attack)
            set of opposing pawns this pawn can capture via en passant
                This set can contain 1 2-tuple of the space onto which the
                current pawn can move, and a reference to the pawn to capture
                en passant. The set is empty on no en passant.
            set of available rooks with which this king can currently castle
                This set contains 1 or 2 2-tuples of the space onto which the
                current king can move, and a reference to the rook to move
                across this king on castling. The set is empty on no castling.
        """
        if (self.x == None or self.y == None):
            raise Exception, "piece is not on the board: " + str(self)
        self.lock.acquire()
        attackees = set([])
        supportees = set([])
        destinations = set([])
        enpassant = set([])
        castle = set([])
        if (self.kind == 'Q' or self.kind == 'R'):
            for x in range(self.x+1, 8):
                if (board[(x,self.y)] == None):
                    destinations.add((x,self.y))
                elif (board[(x,self.y)].color == self.color):
                    supportees.add(board[(x,self.y)])
                    break
                else:
                    attackees.add(board[(x,self.y)])
                    destinations.add((x,self.y))
                    break
            for x in range(self.x-1, -1, -1):
                if (board[(x,self.y)] == None):
                    destinations.add((x,self.y))
                elif (board[(x,self.y)].color == self.color):
                    supportees.add(board[(x,self.y)])
                    break
                else:
                    attackees.add(board[(x,self.y)])
                    destinations.add((x,self.y))
                    break
            for y in range(self.y+1, 8):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                elif (board[(self.x,y)].color == self.color):
                    supportees.add(board[(self.x,y)])
                    break
                else:
                    attackees.add(board[(self.x,y)])
                    destinations.add((self.x,y))
                    break
            for y in range(self.y-1, -1, -1):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                elif (board[(self.x,y)].color == self.color):
                    supportees.add(board[(self.x,y)])
                    break
                else:
                    attackees.add(board[(self.x,y)])
                    destinations.add((self.x,y))
                    break
        if (self.kind == 'Q' or self.kind == 'B'):
            for z in range(1,8):
                x = self.x + z
                y = self.y + z
                if (x > 7 or y > 7):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x + z
                y = self.y - z
                if (x > 7 or y < 0):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x - z
                y = self.y + z
                if (x < 0 or y > 7):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
            for z in range(1,8):
                x = self.x - z
                y = self.y - z
                if (x < 0 or y < 0):
                    break
                if (board[(x,y)] == None):
                    destinations.add((x,y))
                elif (board[(x,y)].color == self.color):
                    supportees.add(board[(x,y)])
                    break
                else:
                    attackees.add(board[(x,y)])
                    destinations.add((x,y))
                    break
        if (self.kind == 'N'):
            for dx, dy in [(-1, 2), (1, 2), (-2, 1), (2, 1),            \
                    (-1, -2), (1, -2), (-2, -1), (2, -1)]:
                x = self.x + dx
                y = self.y + dy
                if (x >= 0 and x < 8 and y >= 0 and y < 8):
                    if (board[(x,y)] == None):
                        destinations.add((x,y))
                    elif (board[(x,y)].color == self.color):
                        supportees.add(board[(x,y)])
                    else:
                        attackees.add(board[(x,y)])
                        destinations.add((x,y))
        if (self.kind == 'K'):
            for dx, dy in [(0, 1), (1, 1), (1,0), (1, -1),              \
                    (0, -1), (-1, -1), (-1, 0), (-1, 1)]:
                x = self.x + dx
                y = self.y + dy
                if (x >= 0 and x < 8 and y >= 0 and y < 8):
                    if (board[(x,y)] == None):
                        destinations.add((x,y))
                    elif (board[(x,y)].color == self.color):
                        supportees.add(board[(x,y)])
                    else:
                        attackees.add(board[(x,y)])
                        destinations.add((x,y))
            if (not self.moved):
                # Can I castle?
                if (board[(0,self.y)] and (not board[(0,self.y)].moved) \
                        and board[(1,self.y)] == None                   \
                        and board[(2,self.y)] == None                   \
                        and board[(3,self.y)] == None):
                    destinations.add((2, self.y))
                    rook = board[(0,self.y)]
                    dest = (2, self.y)
                    castle.add((dest, rook))
                if (board[(7,self.y)] and (not board[(7,self.y)].moved) \
                        and board[(5,self.y)] == None                   \
                        and board[(6,self.y)] == None):
                    destinations.add((6, self.y))
                    rook = board[(7,self.y)]
                    dest = (6, self.y)
                    castle.add((dest, rook))
        if (self.kind == 'P'):
            lx = self.x - 1
            hx = self.x + 1
            y = self.y + advance[self.color]
            if (len(lastmove) == 0):
                # special checks for en passant
                enpstart = 0
            elif (self.y == 4 and self.color == 'white'):
                enpstart = 6
            elif (self.y == 3 and self.color == 'black'):
                enpstart = 1
            else:
                enpstart = 0
            if (y >= 0 and y < 8):
                if (board[(self.x,y)] == None):
                    destinations.add((self.x,y))
                    if (not self.moved):
                        bigy = y + advance[self.color]
                        if (board[(self.x,bigy)] == None):
                            destinations.add((self.x,bigy))
                if (lx >= 0 and board[(lx,y)]                           \
                        and board[(lx,y)].color != self.color):
                    attackees.add(board[(lx,y)])
                    destinations.add((lx,y))
                if (hx < 8 and board[(hx,y)]                            \
                        and board[(hx,y)].color != self.color):
                    attackees.add(board[(hx,y)])
                    destinations.add((hx,y))
                if (enpstart):
                    if (board[(lx,y)] == None and board[(lx,self.y)]    \
                            and board[(lx,self.y)].kind == 'P'          \
                            and board[(lx,self.y)].color != self.color  \
                            and lastmove[0] == (lx, enpstart)            \
                            and lastmove[1] == (lx, self.y)):
                        attackees.add(board[(lx,self.y)])
                        destinations.add((lx,y))
                        enpassant.add(((lx,y),board[(lx,self.y)]))
                    elif (board[(hx,y)] == None and board[(hx,self.y)]    \
                            and board[(hx,self.y)].kind == 'P'          \
                            and board[(hx,self.y)].color != self.color  \
                            and lastmove[0] == (hx, enpstart)            \
                            and lastmove[1] == (hx, self.y)):
                        attackees.add(board[(hx,self.y)])
                        destinations.add((hx,y))
                        enpassant.add(((hx,y),board[(hx,self.y)]))
        self.lock.release()
        return((attackees, supportees, destinations, enpassant, castle))

    def __str__(self):
        return self.color + self.kind + str(self.index)

def initboard():
    board[(0,0)] = piece('white','R',0,0,0)
    board[(7,0)] = piece('white','R',1,7,0)
    board[(0,7)] = piece('black','R',0,0,7)
    board[(7,7)] = piece('black','R',1,7,7)
    board[(1,0)] = piece('white','N',0,1,0)
    board[(6,0)] = piece('white','N',1,6,0)
    board[(1,7)] = piece('black','N',0,1,7)
    board[(6,7)] = piece('black','N',1,6,7)
    board[(2,0)] = piece('white','B',0,2,0)
    board[(5,0)] = piece('white','B',1,5,0)
    board[(2,7)] = piece('black','B',0,2,7)
    board[(5,7)] = piece('black','B',1,5,7)
    board[(3,0)] = piece('white','Q',0,3,0)
    board[(3,7)] = piece('black','Q',0,3,7)
    board[(4,0)] = piece('white','K',0,4,0)
    board[(4,7)] = piece('black','K',0,4,7)
    for x in range(0,8):
        board[(x,1)] = piece('white','P',x,x,1)
        board[(x,6)] = piece('black','P',x,x,6)
        for y in range(2,6):
            board[(x,y)] = None
initboard()


def dumpboard():
    for y in range(7,-1,-1):
        display = ""
        for x in range(0,8):
            if (board[(x,y)]):
                if (board[(x,y)].x != x):
                    raise Exception, "bad x inside " + str(board[(x,y)])
                if (board[(x,y)].y != y):
                    raise Exception, "bad y inside " + str(board[(x,y)])
                display = display + str(board[(x,y)]) + "(" + str(x)    \
                    + "," + str(y) + ") "
            else:
                display = display + "    (" + str(x) + "," + str(y) + ")    "
        print display

class interpreter(object):
    """
    class 'interpreter' analyzes a COPY of the board and sends notes off to
    the renderer.
    """
    def __init__(self, tonegenobj):
        """
        Constructor starts a service thread that waits for 'interpret'
        messages with a copy of the board. It uses a lock and is thread safe.
        The tonegenobj is the object of class tonegen.
        """
        self.tonegenobj = tonegenobj
        self.mythread = threading.Thread(target=interpreter.privateworkloop, \
            name="interpreter", args=[self])
        self.mythread.setDaemon(True)
        self.sequence = []
        self.gate = threading.Condition()
        self.mythread.start()
    def interpret(self, boardref, mover):
        """
        A client invokes interpret with a reference to the board, which
        must not change during the invocation of interpret. Interpret
        extracts musical information from this board and passes it
        to a worker thread to send to the renderer via tonegen.send().
        When interpret() returns it is safe for the client to modify the
        board.

        The mover parameter is the piece that just completed the move.
        Neither it nor any piece must move until after interpret() returns.
        """
        visitedallies = set([])
        visitedenemies = set([])
        sounds = []
        nextallies = [mover]
        visitedallies.add(mover)
        nextenemies = []
        sequence = []
        goodside = True
        while (len(nextallies) or len(nextenemies)):
            nowallies = nextallies
            nextallies = []
            nowenemies = nextenemies
            nextenemies = []
            goodlen = len(nowallies)
            badlen = len(nowenemies)
            alllen = goodlen + badlen
            allieamp = (float(goodlen) / float(alllen)) / float(alllen)
            enemamp = (float(badlen) / float(alllen)) / float(alllen)
            allieoct = (goodlen % 4) + 1
            enemoct = (badlen % 4) + 1
            allspaces = 0
            for n in nowallies:
                freq = reffreq * piecefreq[n.kind] * allieoct
                if (goodside):
                    noteinfo = [-1, freq, 0.0, allieamp, enemamp]
                else:
                    noteinfo = [-1, freq, 0.0, enemamp, allieamp]
                sounds.append(noteinfo)
                allieoct = allieoct + 1
                if (allieoct > 4):
                    allieoct = 1
                attackset, supportset, coordset, pawnset, rookset       \
                    = n.getAttackSupportMove()
                for na in supportset:
                    if (not na in visitedallies):
                        nextallies.append(na)
                        visitedallies.add(na)
                for na in attackset:
                    if (not na in visitedenemies):
                        nextenemies.append(na)
                        visitedenemies.add(na)
                allspaces = allspaces + len(coordset) - len(attackset)
            for n in nowenemies:
                freq = reffreq * piecefreq[n.kind] * enemoct * conflictfreq
                if (goodside):
                    noteinfo = [-1, freq, 0.0, enemamp, allieamp]
                else:
                    noteinfo = [-1, freq, 0.0, allieamp, enemamp]
                sounds.append(noteinfo)
                enemoct = enemoct + 1
                if (enemoct > 4):
                    enemoct = 1
                attackset, supportset, coordset, pawnset, rookset       \
                    = n.getAttackSupportMove()
                for na in supportset:
                    if (not na in visitedallies):
                        nextallies.append(na)
                        visitedallies.add(na)
                for na in attackset:
                    if (not na in visitedenemies):
                        nextenemies.append(na)
                        visitedenemies.add(na)
                allspaces = allspaces + len(coordset) - len(attackset)
            goodside = not goodside
            sequence.append(sounds)
            if (allspaces >= len(sounds)):
                sequence.append([])    # append a rest
        self.gate.acquire()
        self.sequence = sequence
        self.gate.notifyAll()
        self.gate.release()

    def privateworkloop(self):
        osccount = 0
        self.gate.acquire()
        try:
            while (self.sequence != None):  # kill() terminates via None
                while (self.sequence == []):
                    for o in range(0,osccount):
                        self.tonegenobj.send([o, 0.0, 0.0, 0.0, 0.0])
                    osccount = 0
                    self.gate.wait()
                if (self.sequence == None):
                    continue
                curseq = self.sequence
                while (curseq is self.sequence):    # interpret() changes it
                    for notelist in curseq:
                        if (len(notelist) == 0):     # rest, turn off everything
                            for o in range(0,osccount):
                                self.tonegenobj.send([o, 0.0, 0.0, 0.0, 0.0])
                            osccount = 0
                        else:
                            newosc = 0
                            for note in notelist:
                                note[0] = newosc
                                newosc = newosc + 1
                                self.tonegenobj.send(note)
                            for o in range(newosc, osccount):
                                self.tonegenobj.send([o, 0.0, 0.0, 0.0, 0.0])
                            osccount = newosc
                        self.gate.wait(tempo)
                        if (not curseq is self.sequence):
                            break
            for o in range(0,osccount):
                self.tonegenobj.send([o, 0.0, 0.0, 0.0, 0.0])
                                
        finally:
            self.gate.release()

    def kill(self):
        self.gate.acquire()
        self.sequence = None
        self.gate.notifyAll()
        self.gate.release()

# The following are for transport which we will hide behind an API later.
from binascii import a2b_hex, b2a_hex
from struct import pack
from socket import *

class tonegen(object):
    def __init__(self, udphost, udpport):
        self.udphost = udphost
        self.udpport = udpport
        self.skt = socket(AF_INET, SOCK_DGRAM)
    def send(self,tonelist):
        if (len(tonelist) == 0):
            raise Exception, "empty tonelist parameter"

        listhead = a2b_hex('6c697374000000002c') # 'list\0\0\0\0,'
        hdrlen = 9
        listtail = ''
        for val in tonelist:
            if (type(val) == int):
                listhead = listhead + a2b_hex('69')   # 'i'
                listtail = listtail + pack('>i', val)
            elif (type(val) == float):
                listhead = listhead + a2b_hex('66')   # 'f'
                listtail = listtail + pack('>f', val)
            else:
                raise Exception, "invalid type for tonelist value " \
                    + str(val) + ": " + str(type(val))
            hdrlen = hdrlen + 1
        while (hdrlen % 4):
            listhead = listhead + a2b_hex('00')
            hdrlen = hdrlen + 1
        self.skt.sendto(listhead+listtail, (self.udphost, self.udpport))

t = tonegen('localhost', 7401)
m = interpreter(t)

def mv(x1,y1,x2,y2):
    board[(x1,y1)].move(x2,y2)
    print
    print
    dumpboard()
    print
    print
    m.interpret(board,board[(x2,y2)])

print
print
dumpboard()
print
print
