# protocol_2.py

# THE CODE AND DOCUMENTATION IN THIS DIRECTORY AND BELOW ARE OPEN
# SOURCE ACCORDING TO THE TERMS OF THE FOLLOWING LICENSE. PLEASE READ:
# -----------------------------------------------------------------------------
# ESST (Embedded System Software Tools) License
# -----------------------------------------------------------------------------
# 
# SOFTWARE LICENSE
# 
# This software is provided subject to the following terms and conditions,
# which you should read carefully before using the software. Using this
# software indicates your acceptance of these terms and conditions. 
# If you do not agree with these terms and conditions, do not use the software.
# 
# Copyright (c) 2008 Dale E. Parson, Ph.D. (a.k.a. Acoustic Interloper)
# All rights reserved.
# 
# Redistribution and use in source or binary forms, with or without
# modifications, are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following Disclaimer in comments in the code
# as well as in the documentation and/or other materials provided with the
# distribution.
# * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following Disclaimer in the documentation
# and/or other materials provided with the distribution.
# * Neither the name of Dale E. Parson, nor the names of the contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# Disclaimer
# 
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# ANY USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE
# USER'S OWN RISK. IN NO EVENT SHALL Dale E. Parson OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, INCLUDING,
# BUT NOT LIMITED TO, CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -----------------------------------------------------------------------------


from chess.interpreter import interpreter
from chess.ChessException import *
import copy

class protocol_2(interpreter):
    """
    This abstract subclass of chess.interpreter simply defines method
    "send()" that takes a list of int and float values, invoked from subclass
    interpret() method invocations, and packages them according to the
    game protocol defined in PROTOCOL_2.txt. It will take old-style 5-valued
    lists of [oscillator(int), frequency(float), phase(float),
        leftamplitude(float), rightamplitude(float)] and reformat them as
    [bank(int), oscillator(int), frequency(float), phase(float),
        leftamplitude(float), rightamplitude(float),
        active_player(int), active_piecetype(int), relationship(int),
        receiving_player(int), receiving_piecetype(int), movenumber(int)]
    splitting an old style oscillator number into bank, oscillator, and
    appending zeroes for missing fields.
    A packet that arrives at send() fully formatted is not modified.

    Send then forwards its packet to self.tonegenobj.send() for transmission.
    """
    def __init__(self, tonegenobj, consparam):
        """
        The constructor requires the number of oscillators per bank
        as its consparam parameter, which it stores in self.oscperside.
        This consparam may either be an integer or a list/tuple whose
        initial element is an integer representing oscillators per side.

    Each concrete derived class constructor must invoke the following two
    lines as the last steps in order to start the two music generation threads:

        self.wtthread.start()
        self.bkthread.start()
        """
        interpreter.__init__(self, tonegenobj, consparam)
        if (type(consparam) == int):
            self.oscperside = consparam
        elif (type(consparam[0]) == int):
            self.oscperside = consparam[0]
        else:
            raise ChessException, "constructor parameter must include number" \
                + " of oscillators per side: " + str(consparam)
        if (self.oscperside < 1):
            raise ChessException, "oscperside must be > 0: " + str(consparam)
        self.oscperbank = self.oscperside * 2

    def send(self, tonelist):
        """
        Send optionally converts PROTOCOL_1 messages from subclass
        invocations of interpret() to PROTOCOL_2 messages, and then
        invokes self.tonegenobj.send() for transmission.
        See class comments above.
        """
        if (len(tonelist) < 5):
            raise ChessException, "invalid tonelist parameter: "    \
                + str(tonelist)
        if (len(tonelist) == 5 or type(tonelist[1]) != int):    # protocol_1
            tonebank = tonelist[0] / self.oscperbank
            toneslot = tonelist[0] % self.oscperbank
            tonelist = [tonebank, toneslot] + tonelist[1:]
        if (len(tonelist) < 12):
            tonelist = copy.copy(tonelist)
            ll = len(tonelist)
            while (ll < 12):
                if (ll == 6 or ll == 7 or ll == 9 or ll == 10):
                    tonelist.append(-1)
                elif (ll == 8):
                    tonelist.append(-2)
                else:
                    tonelist.append(0)
                ll = len(tonelist)
        # print "DEBUG sending PROTOCOL_2: " + str(tonelist)
        self.tonegenobj.send(tonelist)
