# simpleSineInterpreter.py

# THE CODE AND DOCUMENTATION IN THIS DIRECTORY AND BELOW ARE OPEN
# SOURCE ACCORDING TO THE TERMS OF THE FOLLOWING LICENSE. PLEASE READ:
# -----------------------------------------------------------------------------
# ESST (Embedded System Software Tools) License
# -----------------------------------------------------------------------------
# 
# SOFTWARE LICENSE
# 
# This software is provided subject to the following terms and conditions,
# which you should read carefully before using the software. Using this
# software indicates your acceptance of these terms and conditions. 
# If you do not agree with these terms and conditions, do not use the software.
# 
# Copyright (c) 2008 Dale E. Parson, Ph.D. (a.k.a. Acoustic Interloper)
# All rights reserved.
# 
# Redistribution and use in source or binary forms, with or without
# modifications, are permitted provided that the following conditions are met:
# 
# * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following Disclaimer in comments in the code
# as well as in the documentation and/or other materials provided with the
# distribution.
# * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following Disclaimer in the documentation
# and/or other materials provided with the distribution.
# * Neither the name of Dale E. Parson, nor the names of the contributors
# may be used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# Disclaimer
# 
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# ANY USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE
# USER'S OWN RISK. IN NO EVENT SHALL Dale E. Parson OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, INCLUDING,
# BUT NOT LIMITED TO, CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
# OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# -----------------------------------------------------------------------------

import copy
from math import *
from chess.interpreter import interpreter
from chess.protocol_2 import protocol_2

tempo = {}

def tempogen(carrier, swingdiv, phase, steps):
    swing = carrier / swingdiv
    sinoffset = phase     # ramp 0 thru 2pi
    while(1):
        result = carrier + (sin(sinoffset) * swing)
        sinoffset = sinoffset + (pi / steps)
        if (sinoffset > (2 * pi)):
            sinoffset = 0.0
        yield result

tempo['white'] = tempogen(0.125, 2.0, 1.5 * pi, 32.0)
tempo['black'] = tempogen(0.125, 2.0, 1.2 * pi, 36.0)
reffreq = 55.0
octrange = 8    # must be a power of 2 for 'permute' to hit all combinations.
lookaheadlimit = 6
permute = {}
# permute runs octaves at play time rather than sequence time, so keep
# pawn interesting since it is used so often
permute['K'] = 3
permute['Q'] = 5
permute['P'] = 3
permute['R'] = 7
permute['N'] = 5
permute['B'] = 1
piecefreq = {}
piecefreq['K'] = 1.0            # King          root
piecefreq['Q'] = 2.0            # Queen         octave
piecefreq['P'] = 1.5            # Pawn          just 5th
piecefreq['R'] = 4.0 / 3.0      # Rook          just 4th
piecefreq['N'] = 9.0 / 8.0      # kNight        just 2nd
piecefreq['B'] = 27.0 / 16.0    # Bishop        just 6th
# conflictfreq = pow(4,1/12.0)    # equal tempered semi-tone. yuck!!!
conflictfreq = 6.0 / 5.0
# just minor third is 6.0 / 5.0, just major third is 5.0 / 4.0
freqoffset = {}
freqoffset['white'] = 1.0
freqoffset['black'] = 1.0 #  5.0 / 4.0

othercolor = {}
othercolor['white'] = 'black'
othercolor['black'] = 'white'

# accents are patterns of full amplitude, 1/2 or 0 per moved piece
# second set of accents are for when a piece is taken off the board
accents = {}
accents['Q'] = ([1.0, 1.0, 0.0, 1.0, 0.0, 0.0, 1.0, 1.0], []) # 3-32 son clave
accents['R'] = ([0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 0.0, 0.0], []) # bars 1 & 2
accents['K'] = ([1.0, 0.5, 0.5, 1.0, 0.5, 0.5, 1.0, 0.0],[]) # Dale's pattern
accents['B'] = ([1.0, 0.0, 1.0, 0.5], [0.5, 0.0, 0.50, 0.0]) # frailing banjo
accents['N'] = ([0.0, 1.0, 0.5], [0.0, 0.5, 0.25])
accents['P'] = ([1.0, 0.0], [1.0, 0.0])
accents['Q'] = ([1.0, 1.0, 0.25, 1.0, 0.25, 0.25, 1.0, 1.0], []) # 3-32 son clave
accents['R'] = ([0.25, 0.25, 1.0, 1.0, 1.0, 1.0, 0.25, 0.25], []) # bars 1 & 2
accents['K'] = ([1.0, 0.5, 0.5, 1.0, 0.5, 0.5, 1.0, 0.25],[]) # Dale's pattern
accents['B'] = ([1.0, 0.25, 1.0, 0.5], [0.5, 0.25, 0.50, 0.25]) # frailing banjo
accents['N'] = ([0.25, 1.0, 0.5], [0.25, 0.5, 0.25])
accents['P'] = ([1.0, 0.25], [1.0, 0.25])

weight = {} # value of piece
weight['Q'] = 9.0
weight['R'] = 5.0
weight['K'] = 10.0
weight['B'] = 3.0
weight['N'] = 2.5
weight['P'] = 1.0

advance = {}
advance['white'] = 1    # direction of advance
advance['black'] = -1

# Coordinates are always X,Y and deltaX,deltaY

class simpleSineInterpreter(protocol_2):
    """
    class 'interpreter' analyzes a COPY of the board and sends notes off to
    the renderer.
    """
    def __init__(self, tonegenobj, consparam):
        protocol_2.__init__(self, tonegenobj, consparam)
        self.wtthread.start()
        self.bkthread.start()
    def protectedBuildSequence(self, boardref, mover, reachgraph, notegraph,
            kingpanic, newoctave, isalreadylocked, lastpiece, lastdest,
            movenumber):
        # 2. Build seqs, using each piece at most once, out from current loc
        newseq = {}
        newseq['white'] = []
        newseq['black'] = []
        lookahead = 0
        for col in ['white', 'black']:
            if (kingpanic[col]):
                print "CHECK ON", col
                # Turn on all notes & their discord for color under 'CHECK!'
                sounds = []
                for pp in ['Q', 'K', 'R', 'B', 'N', 'P']:
                    for poct in range(0,5):
                        freq = reffreq * piecefreq[pp] * float(poct)    \
                            * freqoffset[col]
                        # Sound should be 1/30, but let's overdrive
                        noteinfo = [-1, freq, 0.0, 1.0/30.0, 1.0/30.0]
                        sounds.append(noteinfo)
                        freq = freq * conflictfreq
                        noteinfo = [-1, freq, 0.0, 1.0/30.0, 1.0/30.0]
                        sounds.append(noteinfo)
                newseq[col].append(sounds)
                continue
            if (lastdest[col] == None or lastpiece[col] == None):
                continue
            leftbias = (mover.color == col)
            visited = set([lastpiece[col]])
            inrvisited = set([])
            nextvisit = [(lastpiece[col], lastdest[col])]
            nextcolor = [col]
            if (lastpiece[col] and lastpiece[col].x != None \
                    and lastpiece[col].y != None):
                notegraph[(lastpiece[col].x, lastpiece[col].y)]     \
                    = reachgraph[(lastpiece[col].x, lastpiece[col].y)]
            while (len(nextvisit)):
                lookahead = lookahead + 1
                if (lookahead > lookaheadlimit):
                    break
                thisvisit = nextvisit
                nextvisit = []
                thiscolor = nextcolor
                nextcolor = []
                ampl = 0.5 / float(len(thisvisit))
                sounds = []
                newoctave = ((len(thisvisit) + newoctave) % octrange) + 1
                allreached = 0
                alloccupied = 0
                for p_and_c in thisvisit:
                    pce = p_and_c[0]
                    crd = p_and_c[1]
                    thiscol = thiscolor[0]
                    thiscolor = thiscolor[1:]
                    if (pce.x != None and pce.y != None):
                        if (pce in visited):
                            freqoff = piecefreq[pce.kind]
                        else:
                            freqoff = 1.0 / piecefreq[pce.kind] # inversion
                            # This inversion coours if a piece is encountered
                            # as an incoming supporter or attacker before out.
                        if (pce.color == thiscol):
                            freq = reffreq * freqoff * freqoffset[col]
                            # Don't apply self.octave until playing thread.
                        else:
                            freq = reffreq * freqoff * freqoffset[col]  \
                                * conflictfreq
                            # Don't apply self.octave until playing thread.
                        if (leftbias):
                            noteinfo = [-1, freq, 0.0, ampl * .75, ampl * .25]
                        else:
                            noteinfo = [-1, freq, 0.5, ampl * .25, ampl * .75]
                    else:
                        noteinfo = [-1, 0.0, 0.0, 0.0, 0.0] # rest after capture
                    sounds.append(noteinfo)
                    reachout = reachgraph[crd][1] | reachgraph[crd][2]
                    reachin = reachgraph[crd][3] | reachgraph[crd][4]
                    reachp = reachout | reachin
                    reachs = reachgraph[crd][5]
                    allreached = allreached + len(reachs)
                    alloccupied = alloccupied + len(reachp)
                    # visit the outgoing at most once, but visit the incoming
                    # a second time if they have not yet been outgoing
                    for p in reachout:
                        if (not p in visited):
                            nextvisit.append((p,(p.x,p.y)))
                            nextcolor.append(pce.color)
                            visited.add(p)
                            notegraph[crd][0] = reachgraph[crd][0]
                            notegraph[crd][1] = reachgraph[crd][1]
                            notegraph[crd][2] = reachgraph[crd][2]
                    for p in reachin:
                        if (not p in visited):
                            if (p in inrvisited):   # don't come here 2d time
                                visited.add(p)
                            else:
                                nextvisit.append((p,(p.x,p.y)))
                                nextcolor.append(pce.color)
                                notegraph[crd][0] = reachgraph[crd][0]
                                notegraph[crd][3] = reachgraph[crd][3]
                                notegraph[crd][4] = reachgraph[crd][4]
                                inrvisited.add(p)
                            # NO! visited.add(p)
                newseq[col].append(sounds)
#                if (allreached >= (2 * alloccupied)) :
#                    newseq[col].append([])  # rest if a lot of free space
        return((newseq, newoctave))

    def protectedWorkloop(self, color, threadnumber):
        ocolor = othercolor[color]
        myaccents = [[], []]
        if (color == 'white'):
            oscbase = 0
        else:
            oscbase = self.oscperside
        osccount = 0
        oscstart = 0
        # another bank of oscillators for conflict timbre
        amplmul = tempogen(0.6, 1.5, pi / 2.0, 16)
        playoctave = 1
        self.gate.acquire()
        try:
            while (self.sequence[color] != None):  # kill() terminates via None
                while (self.sequence[color] == []):
                    for o in range(oscbase,oscbase+osccount):
                        self.send([o, 0.0, 0.0, 0.0, 0.0])
                    osccount = 0
                    self.gate.wait()
                if (self.sequence[color] == None):
                    continue
                # accents for some captured pieces specify silence as []
                if (self.lastpiece[color] == None                           \
                        or (self.lastpiece[color].x == None                 \
                            and accents[self.lastpiece[color].kind][1] == [])):
                    self.sequence[color] = []
                    continue
                curseq = self.sequence[color]
                playoctave = self.octave
                myaccents[1] = myaccents[0]
                if (self.lastpiece[color].x == None):   # rhythm for captured
                    myaccents[0] = accents[self.lastpiece[color].kind][1]
#                elif (self.lastpiece[ocolor] == None                        \
#                        or weight[self.lastpiece[color].kind]               \
#                            >= weight[self.lastpiece[ocolor].kind]          \
#                        or self.lastpiece[ocolor].x == None):
                else:
                    myaccents[0] = accents[self.lastpiece[color].kind][0]
#                else:
#                    myaccents[0] = accents[self.lastpiece[ocolor].kind][0]
                curaccents = copy.copy(myaccents[0])
                curaccents.extend(myaccents[1])
                accentix = 0
                # amplmul = tempogen(0.5, 1.0, pi / 2.0, 16)
                amplscale = amplmul.next()
                while (curseq is self.sequence[color]):
                # interpret() changes it
                    for notelist in curseq:
                        if (len(notelist) == 0):     # rest, turn off everything
                            # Shut off in-use ones and use new ones
                            for o in range(0,osccount):
                                oabs = ((oscstart+o)%self.oscperside)+oscbase
                                self.send([oabs,         \
                                    -1.0, -1.0, 0.0, 0.0])
                            oscstart = (oscstart + osccount) % self.oscperside
                            osccount = 0
                        else:
                            mylimit = len(notelist)
                            if (mylimit < 2):
                                octavelimit = 3
                            elif (mylimit < (octrange / 2)):
                                octavelimit = 2 * mylimit
                            else:
                                octavelimit = octrange
                            # Shut off in-use ones and use new ones
                            for o in range(0,osccount):
                                oabs = ((oscstart+o)%self.oscperside)+oscbase
                                self.send([oabs,         \
                                    -1.0, -1.0, 0.0, 0.0])
                            oscstart = (oscstart + osccount) % self.oscperside
                            osccount = 0
                            for note in notelist:
                                newosc = ((oscstart + osccount) % self.oscperside) + oscbase
                                osccount = (osccount + 1) % self.oscperside
                                mynote = copy.copy(note[0:5])
                                mynote[0] = newosc
                                mynote[1] = mynote[1] * playoctave
                                playoctave = ((playoctave                   \
                                    + permute[self.lastpiece[color].kind]   \
                                    - 1) % octavelimit) + 1
                                mynote[3] = mynote[3] * amplscale   \
                                    * curaccents[accentix]
                                mynote[4] = mynote[4] * amplscale   \
                                    * curaccents[accentix]
                                newosc = ((newosc + 1) % self.oscperside) + oscbase
                                fadenote = [newosc, 0.0, 0.0, 0.0, 0.0]
                                self.send(fadenote)
                                self.send(mynote)
                            accentix = (accentix + 1) % len(curaccents)
                        self.gate.wait(tempo[color].next())
                        if (not curseq is self.sequence[color]):
                            break
                    amplscale = amplmul.next()
            for o in range(oscbase,oscbase+self.oscperside):
                self.send([o, 0.0, 0.0, 0.0, 0.0])
        finally:
            self.gate.release()
