unit Globals;

//  ////////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2004 Jan Punter
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  ////////////////////////////////////////////////////////////////////////////
//
//    lupd : 2004-04-04
//
//  ////////////////////////////////////////////////////////////////////////////


interface

Uses

  Classes, SysUtils, Windows, Messages, StdCtrls, Forms, Dialogs;


{$DEFINE DEBUG}
{$UNDEF DEBUG}

Type

  EGlobals = Class( Exception);
  EEscape  = Class( EGlobals );

  TBytes = Array[ 0 ..MaxInt Div SizeOf( Byte) - 1] Of Byte;
  PBytes = ^TBytes;

Const

  WM_GRIDSELECTIONCHANGED = WM_USER + 0;

// Our name and path stuff

  Function ApplicationPath    : String;
  Function ApplicationPathURL : String;
  Function ApplicationFileName: String;
  Function IniFileName        : String;
  Function LogFileName        : String;
  Function FileNameToURL( Const aFileName: String): String;

Const

  strZeroTime  = '0:00';
  strBlankTime = '';
  strDotTime   = '..';

  Function  MinutesToStr    ( aValue      : Word   ): String;
  Function  MinutesToStrZero( aValue      : Word   ): String;
  Function  MinutesToStrDots( aValue      : Word   ): String;
  Function  StrToMinutes    ( Const aValue: String ): Integer;
  Function  WeekDayToStr    ( aDay        : Word   ): String;
  Function  BoolToStr       ( aValue      : Boolean): String;
  Function  ByteToHex       ( aValue: Cardinal): String;
  Function  WordToHex       ( aValue: Cardinal): String;
  Function  CellToHex       ( aValue: Cardinal): String;
  Function  LongToHex       ( aValue: Cardinal): String;
  Function  LongToBinary    ( aValue: Cardinal): String;
  Function  LongToBase      ( aValue: LongInt; aBase, aFieldLen: Byte; aFiller: Char): String;
  Function  HexToByte       ( Const aValue: String): Cardinal;
  Function  HexToWord       ( Const aValue: String): Cardinal;
  Function  HexToCell       ( Const aValue: String): Cardinal;
  Function  HexToLong       ( Const aValue: String): Cardinal;
  Function  ByteToAscii     ( aValue: Byte): Char;
  Function  Indent          ( anInd: Integer): String;
  Procedure WriteLnInd      ( Var aFile: TextFile; anInd: Integer; Const aMsg: String); Overload
  Procedure WriteLnInd      ( Var aFile: TextFile; anInd: Integer; Const aFormat: String; Const aData: Array Of Const); Overload
  Function  TrimStr         ( Const S: String; aLength: Integer): String;
  Function  ExtendStr       ( Const S: String; aLength: Integer): String;
  Function  FEscape         ( Const S: String; aSeparator: Char): String;

  Function  IsDottedIp( Const S: String): Boolean;


// Debug support

{$IFDEF DEBUG}
  Procedure ClearDebugTiming;
  Procedure WriteDebugStr( Const aString: String);
{$ENDIF}

// TNumber

Type

  TNumber = Class( TObject)
      Number : Integer;
    Constructor Create( aNumber : Integer);
  End;


// System stuff

  Function  KeyboardRepeatTime: Integer;
  Function  KeyBoardInitialDelay: Integer;
  Function  Maximum(A, B: Integer): Integer;
  Function  Minimum(A, B: Integer): Integer;
  Function  CleanFileName( Const aFileName : String): String;
  Function  CleanupFilePath( Const aPath: String): String;
  // Returns filename only, path and extension are stripped off.


// Miscellaneous

  Procedure NotImplemented( Const aMsg: String);
  Procedure Noop;
  Procedure ParseString( Const aCommandLine: String; aCommands: TStrings); // Overload;



implementation

// Our name and path stuff

Var

  strApplicationPath     : String;
  strApplicationPathURL  : String;
  strApplicationFileName : String;
  strIniFileName         : String;
  strLogFileName         : String;
  strIndent              : String;

  Function ApplicationPath: String;
  Begin
    Result := strApplicationPath;
  End;

  Function ApplicationPathURL : String;
  Begin
    Result := strApplicationPathURL;
  End;

  Function ApplicationFileName: String;
  Begin
    Result := strApplicationFileName;
  End;

  Function IniFileName: String;
  Begin
    Result := strIniFileName;
  End;

  Function LogFileName: String;
  Begin
    Result := strLogFileName;
  End;

  Function FileNameToURL( Const aFileName: String): String;
  Var
    i : Integer;
  Begin
    Result := 'file:///';
    For i := 1 To Length( aFileName) Do
      Case Result[ i] Of
        ':' : Result := Result + '|';
        '\' : Result := Result + '/';
        Else  Result := Result + aFileName[ i];
      End;
  End;

// Date time stuff

Const

  WeekDays : Array [ 0 .. 6] Of String = (
    'maandag',
    'dinsdag',
    'woensdag',
    'donderdag',
    'vrijdag',
    'zaterdag',
    'zondag'
  );

  Function  MinutesToStr( aValue: Word): String;
  Begin
    If aValue = 0
    Then Result := strBlankTime
    Else Result := Format( '%d:%.2d', [ aValue Div 60, aValue Mod 60]);
  End;

  Function MinutesToStrZero( aValue: Word): String;
  Begin
    If aValue = 0
    Then Result := strZeroTime
    Else Result := Format( '%d:%.2d', [ aValue Div 60, aValue Mod 60]);
  End;

  Function  MinutesToStrDots( aValue: Word): String;
  Begin
    If aValue = 0
    Then Result := strDotTime
    Else Result := Format( '%d:%.2d', [ aValue Div 60, aValue Mod 60]);
  End;

  Function  WeekDayToStr( aDay: Word): String;
  Begin
    If aDay In [ 0 .. 6]
    Then Result := WeekDays[ aDay]
    Else Result := 'ongeldige dag';
  End;

  Function  BoolToStr( aValue : Boolean): String;
  Begin
    Case aValue Of
      False : Result := 'False';
      True  : Result := 'True';
    End;
  End;

  Function  StrToMinutes( Const aValue: String): Integer;
  Var
    p      : Integer;
    S1, S2 : String;
    V1, V2 : Integer;
  Begin
    Result := 0;
    p := Pos( ':', aValue);
    If p > 0
    Then Begin
      S1 := Copy( aValue, 1, p -1);
      S2 := Copy( aValue, p + 1, 2);
      V1 := StrToIntDef( S1, 0);
      V2 := StrToIntDef( S2, 0);
      Result := V1 * 60 + V2;
    End;
  End;

  Function  ByteToHex( aValue: Cardinal): String;
  Begin
    Result := Format( '%0.2x', [ aValue And $00ff]);
  End;

  Function  WordToHex( aValue: Cardinal): String;
  Begin
    Result := Format( '%0.4x', [ aValue And $ffff]);
  End;

  Function  CellToHex( aValue: Cardinal): String;
  Begin
    Result := Format( '%0.2x.%0.4x', [ aValue And $00ff0000 Shr 16, aValue And $ffff]);
  End;

  Function  LongToHex( aValue: Cardinal): String;
  Begin
    Result := Format( '%0.4x.%0.4x', [ aValue And $ffff0000 Shr 16, aValue And $ffff]);
  End;

  Function  LongToBinary( aValue: Cardinal): String;
  Begin
    Result := '';
    While aValue <> 0 Do
    Begin
      If aValue And 1 = 1
      Then Result := '1' + Result
      Else Result := '0' + Result;
      aValue := aValue Shr 1;
    End;
    If Result = ''
    Then Result := '0';
  End;

  Function  LongToBase( aValue: LongInt; aBase, aFieldLen: Byte; aFiller: Char): String;

    Function ToDigit( aValue: Integer): String;
    Begin
      Assert( aValue < aBase, 'Digit conversion error, Value >= Base');
      If aBase <= 36
      Then
        If aValue >= 10
        Then Result := Char( aValue + Ord( '0') + $27)
        Else Result := Char( aValue + Ord( '0'))
      Else Result := Format( '<&%d>', [ aValue]);
    End;

  Begin
    Result := '';
    Repeat
      Result := ToDigit( aValue Mod aBase) + Result;
      aValue := aValue Div aBase;
    Until aValue = 0;
    While Length( Result) < aFieldLen Do
      Result := aFiller + Result;
  End;

  Function  HexToByte( Const aValue: String): Cardinal;
  Begin
    Result := StrToInt( Format( '$%s', [ aValue]));
  End;

  Function  HexToWord( Const aValue: String): Cardinal;
  Begin
    Result := StrToInt( Format( '$%s', [ aValue]));
  End;

  Function  HexToCell( Const aValue: String): Cardinal;
  Var
    S : String;
    p : Integer;
  Begin
    S := aValue;
    p := Pos( '.', S);
    While p > 0 Do
    Begin
      S := Copy( S, 1, p - 1) + Copy( S, p + 1, Length( S));
      p := Pos( '.', S);
    End;
    Result := StrToInt( Format( '$%s', [ S]));
  End;

  Function  HexToLong( Const aValue: String): Cardinal;
  Var
    S : String;
    p : Integer;
  Begin
    S := aValue;
    p := Pos( '.', S);
    While p > 0 Do
    Begin
      S := Copy( S, 1, p - 1) + Copy( S, p + 1, Length( S));
      p := Pos( '.', S);
    End;
    Result := StrToInt( Format( '$%s', [ S]));
  End;

  Function  ByteToAscii( aValue: Byte): Char;
  Begin
    If ( aValue < 32) Or ( aValue >= 127)
    Then Result := '.'
    Else Result := Char( aValue);
  End;

  Function  Indent( anInd: Integer): String;
  Begin
    Result := Copy( strIndent, 1, 2 * anInd);
  End;

  Procedure WriteLnInd( Var aFile: TextFile; anInd: Integer; Const aMsg: String); // Overload
  Begin
    WriteLn( aFile, Format( '%s%s', [ Indent( anInd), aMsg]));
  End;

  Procedure WriteLnInd( Var aFile: TextFile; anInd: Integer; Const aFormat: String; Const aData: Array Of Const); // Overload
  Begin
    WriteLnInd( aFile, anInd, Format( aFormat, aData));
  End;

  Function  TrimStr( Const S: String; aLength: Integer): String;
  Begin
    Result := Copy( ExtendStr( S, aLength), 1, aLength); // Procustrus trimming.
  End;

  Function  ExtendStr( Const S: String; aLength: Integer): String;
  Begin
    Result := S;
    While Length( Result) < aLength Do
      Result := Result + ' ';
  End;

  Function  FEscape( Const S: String; aSeparator: Char): String;
  Type

    TNumBase = (
      nbBinary,
      nbDecimal,
      nbHex
    );

    Procedure BinaryError( aNum: Integer);
    Begin
      Raise
        EEscape.
          CreateFmt(
            'Binary escape sequence \%%s : %%%s > %%11111111',
            [ LongToBinary( aNum), LongToBinary( aNum)]
        );
    End;

    Procedure DecimalError( aNum: Integer);
    Begin
      Raise
        EEscape.
          CreateFmt(
            'Error in decimal escape sequence \%d : %d > 255',
            [ aNum, aNum]
          );
    End;

    Procedure HexError( aNum: Integer);
    Begin
      Raise
        EEscape.
          CreateFmt(
            'Hex escape sequence \$%x : $%x > $ff',
            [ aNum, aNum]
        );
    End;

    Function Convert( aNum: Integer; aNumBase: TNumBase): Char;
    Begin
      Result := '.';
      If aNum < 256
      Then Result := Char( aNum)
      Else
        Case aNumBase Of
          nbBinary  : BinaryError ( aNum);
          nbDecimal : DecimalError( aNum);
          nbHex     : HexError    ( aNum);
        End;
    End;

  Var
    i : Integer;
    State : (
      stNone,
      stEscaped,
      stDecimal,
      stHex,
      stBinary
    );
    Num : Integer;
    Cnt : Integer;
  Begin
    Result := '';
    i      := 1;
    Num    := 0;
    Cnt    := 0;
    State  := stNone;
    While i <= Length( S) Do
    Begin
      Case State Of

        stNone :
          Case S[ i] Of
            '\' : State := stEscaped;
            Else  Result := Result + S[ i];
          End;

        stEscaped :
          Case S[ i] Of

            '\' :
              Begin
                Result := Result + '\';
                State := stNone;
              End;

            '$' :
              Begin
                Num := 0;
                Cnt := 0;
                State := stHex;
              End;

            '%' :
              Begin
                Num := 0;
                Cnt := 0;
                State := stBinary;
              End;

            '0' .. '9' :
              Begin
                Num   := Byte( S[ i]) - Byte( '0');
                Cnt   := 1;
                State := stDecimal;
              End;

            Else Begin
              If S[ i] = aSeparator
              Then Result := Result + aSeparator
              Else Result := Result + Char( Byte( UpCase( S[ i])) And Not $40);
              State := stNone
            End;
          End;

        stDecimal :
          Case S[ i] Of

            '0' .. '9' :
              Begin
                Num := 10 * Num + ( Byte( S[ i]) - Byte( '0'));
                Inc( Cnt);
                If Cnt = 3
                Then Begin
                  Result := Result + Convert( Num, nbDecimal);
                  State  := stNone;
                End;
              End;

            Else Begin
              Result := Result + Convert( Num, nbDecimal);
              State  := stNone;
            End;
          End;

        stHex :
          Case s[ i] Of

            '0' .. '9' :
              Begin
                Num := 16 * Num + ( Byte( S[ i]) - Byte( '0'));
                Inc( Cnt);
                If Cnt = 2
                Then Begin
                  Result := Result + Convert( Num, nbHex);
                  State  := stNone;
                End;
              End;

            'a' .. 'f' :
              Begin
                Num := 16 * Num + ( Byte( S[ i]) - Byte( 'a') + 10);
                Inc( Cnt);
                If Cnt = 2
                Then Begin
                  Result := Result + Convert( Num, nbHex);
                  State  := stNone;
                End;
              End;

            'A' .. 'F' :
              Begin
                Num := 16 * Num + ( Byte( S[ i]) - Byte( 'A') + 10);
                Inc( Cnt);
                If Cnt = 2
                Then Begin
                  Result := Result + Convert( Num, nbHex);
                  State  := stNone;
                End;
              End;


            Else Begin
              Result := Result + Convert( Num, nbHex);
              State  := stNone;
            End;
          End;

        stBinary :
          Case S[ i] Of
            '0', '1' :

              Begin
                Num := 2 * Num + ( Byte( S[ i]) - Byte( '0'));
                Inc( Cnt);
                If Cnt = 8
                Then Begin
                  Result := Result + Convert( Num, nbBinary);
                  State  := stNone;
                End;
              End;
            Else Begin
              Result := Result + Convert( Num, nbBinary);
              State  := stNone;
            End;
          End;
      End;
      Inc( i);
    End;
  End;


  Function  IsDottedIp( Const S: String): Boolean;

    Function IsValidInteger( Const N: String): Boolean;
    Var
      aCode : Integer;
      aNum  : Integer;
    Begin
      Val( N, aNum, aCode);
      Result := ( aCode = 0) And ( aNum >= 0) And ( aNum < 256);
    End;

  Const
    Digits : Set Of Char = [ '0' .. '9'];
  Var
    p      : Integer;
    N      : String;
    DotCnt : Integer;
  Begin
    p      := 1;
    N      := '';
    DotCnt := 0;
    Result := True;
    While ( p <= Length( S)) And Result Do
    Begin
      If S[ p] In Digits
      Then Begin
        N := N + S[ p];
        If Length( N) > 3
        Then Begin
          Result := False;
          Break;
        End;
      End
      Else If S[ p] = '.'
      Then Begin
        Result := IsValidInteger( N);
        Inc( DotCnt);
        N := '';
      End
      Else Begin
        Result := False;
        Break;
      End;
      Inc( p);
    End;
    If Result
    Then Result := ( DotCnt = 3) And IsValidInteger( N);
  End;

// Debug support

{$IFDEF DEBUG}
Var

  DebugFile : Text;
  DebugName : String;
  DebugTime : Integer;

  Procedure ClearDebugTiming;
  Begin
    DebugTime := GetTickCount;
  End;

  Function FormatDebugTime( aTime: Integer): String;
  Var
    day, hr, min, sec, ms : Integer;
  Begin
    ms    := aTime Mod 1000;
    aTime := aTime Div 1000;
    sec   := aTime Mod   60;
    aTime := aTime Div   60;
    min   := aTime Mod   60;
    aTime := aTime Div   60;
    hr    := aTime Mod   24;
    aTime := aTime Div   24;
    day   := aTime;
    Result := Format( '%.2d:%.2d:%.2d.%.', [ day, hr, min, sec, ms]);
  End;

  Procedure WriteDebugStr( Const aString: String);
  Begin
    Append( DebugFile);
    WriteLn(
      DebugFile,
      Format(
        '%s - %s', [ FormatDebugTime( GetTickCount - DebugTime), aString ]
      )
    );
    Close( DebugFile);
  End;
{$ENDIF}


  Procedure InitModule;
  Var
    S : String;
    p : Integer;
  Begin
    S := Application.ExeName;
    strApplicationFileName := ExtractFileName( S);
    strApplicationPath     := ExtractFilePath( S);
    strApplicationPathURL  := FileNameToURL( strApplicationPath);
    p := Pos( '.', strApplicationFileName);
    If p = 0
    Then Begin
      strIniFileName := strApplicationPath + strApplicationFileName + '.INI';
      strLogFileName := strApplicationPath + strApplicationFileName + '.LOG';
    End
    Else Begin
      strIniFileName := strApplicationPath + Copy( strApplicationFileName, 0, p - 1) + '.INI';
      strLogFileName := strApplicationPath + Copy( strApplicationFileName, 0, p - 1) + '.LOG';
    End;
    strIndent := ' ';
    While Length( strIndent) < 256 Do
      strIndent := strIndent + strIndent;

{$IFDEF DEBUG}
    DebugName := strApplicationPath + 'debug.txt';
    Assign( DebugFile, DebugName);
    Rewrite( DebugFile);
    Close( DebugFile);
    ClearDebugTiming;
    WriteDebugStr( 'Program start');
{$ENDIF}

  End;


// TNumber support

{ ========
  TNumber = Class( TObject)
      Number : Integer;
}

    Constructor TNumber.Create( aNumber : Integer);
    Begin
      Inherited Create;
      Number := aNumber;
    End;

// System stuff


{$WARN UNSAFE_CODE OFF}

  Function KeyboardRepeatTime: Integer;
  Begin
    If Not SystemParametersInfo( SPI_GETKEYBOARDSPEED, 0, @ Result, 0)
    Then Result := 2;
    If Result = 0
    Then Result := 1000
    Else Result := Round( 1000 / Result);
  End;

{$WARN UNSAFE_CODE ON}

  Function KeyBoardInitialDelay: Integer;
  Begin
    // @@@@ nog aanpassen & uitzoeken
    Result := 250;
  End;

  Function Maximum(A, B: Integer): Integer;
  Begin
    If A > B
    Then Result := A
    Else Result := B;
  End;

  Function Minimum(A, B: Integer): Integer;
  Begin
    If A < B
    Then Result := A
    Else Result := B;
  End;

  Function  CleanFileName( Const aFileName : String): String;
  Var
    i : Integer;
  Begin
    Result := ExtractFileName( aFileName);
    i := Pos( '.', Result);
    If i > 0
    Then Result := Copy( Result, 1, i - 1);
  End;

  Function  CleanupFilePath( Const aPath: String): String;
  Begin
    Result := ExpandFileName( aPath);  
  End;


// Miscellaneous

  Procedure NotImplemented( Const aMsg: String);
  Begin
    MessageDlg(
      Format(
        'Not implemented : %s',
        [ aMsg]
      ),
      mtInformation,
      [ mbOk, mbCancel],
      0
    );
  End;

  Procedure Noop;
  Begin
  End;


  Procedure ParseString( Const aCommandLine: String; aCommands: TStrings); // Overload;
  // Parse commandline assuming space characters as separators.
  // Honour quoted parts in this process.
  // Parse results will be stored into aCommands.
  Var
    i     : Integer;
    S     : String;
    State : (
      stNone,
      stInSingle,
      stInDouble,
      stSingleNest,
      stDoubleNest
    );

    Procedure SkipSpaces;
    Begin
      While ( i <= Length( aCommandLine)) And ( aCommandLine[ i] = ' ') Do
        Inc( i);
    End;

  Begin
    With aCommands Do
    Begin
      Clear;
      i := 1;
      SkipSpaces;
      State := stNone;
      S     := '';
      While i <= Length( aCommandLine) Do
      Begin
        Case State Of

          stNone :
            Case aCommandLine[ i] Of
              ' ' : Begin
                  If Length( S) > 0
                  Then Begin
                    Add( S);
                    S := '';
                    SkipSpaces;
                    Dec( i);
                  End;
                End;
              '''' : State := stInSingle;
              '"'  : State := stInDouble;
              Else S := S + aCommandLine[ i];
            End;

          stInSingle : Begin
              Case aCommandLine[ i] Of
                '''' : State := stSingleNest;
                Else S := S + aCommandLine[ i];
              End;
            End;

          stInDouble : Begin
              Case aCommandLine[ i] Of
                '"' : State := stDoubleNest;
                Else S := S + aCommandLine[ i];
              End;
            End;

          stSingleNest : Begin
              Case aCommandLine[ i] Of
                '''' : Begin
                    S := S + aCommandLine[ i];
                    SkipSpaces;
                    State := stInSingle;
                  End;
                Else Begin
                    If Length( S) > 0
                    Then Begin
                      Add( S);
                      S := '';
                    End;
                    State := stNone;
                  End;
              End;
            End;

          stDoubleNest : Begin
              Case aCommandLine[ i] Of
                '"' : Begin
                    S := S + aCommandLine[ i];
                    SkipSpaces;
                    State := stInSingle;
                  End;
                Else Begin
                    If Length( S) > 0
                    Then Begin
                      Add( S);
                      S := '';
                    End;
                    State := stNone;
                  End;
              End;
            End;

        End; // Case State Of
        Inc( i);
      End;
      If S <> ''
      Then Add( S);
    End;
  End;


// Initialisation:
//   get version info from executable
//   setup debug stuff
//   etc.

Initialization

  InitModule;

Finalization

end.


