Unit BitHose;

//  ////////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2004 Jan Punter
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  ////////////////////////////////////////////////////////////////////////////
//
//    lupd : 2004-04-04
//
//  ////////////////////////////////////////////////////////////////////////////


Interface

Uses

  Math;

Type

  TBitCell   = Type Cardinal;            // Cardinals used for bit cells
  TByteArray = Array Of Byte;

Const

  CellSize = 8 * SizeOf( TBitCell);      // Nr of bits in  each cell


  // In the following constant the number of bits used must equal CellSize.
  //
  AllOnes  = $ffffffff;

Type

  TBitIndex = 0 .. CellSize - 1;         // Range for bit indices
  TBitCount = 0 .. CellSize;             // Range for bit counts
  TBitValue = 0 .. 1;                    // Possible values for a single bit.

  TBitHose = Class
  Private
    FData     : Array Of TBitCell;
    FSize     : Integer;
    FWritePtr : Integer;
    FReadPtr  : Integer;
  Private
    Procedure   SetSize( aValue: Integer);
    Function    GetBits( anIndex: Integer; aBitCount: TBitCount): TBitCell;
    Procedure   SetBits( anIndex: Integer; aBitCount: TBitCount; aValue: TBitCell);
  Public
    Constructor Create;
    Destructor  Destroy;                                               Override;
    Function    Clone: TBitHose;
    Function    GetBitHose( anIndex: Integer; aBitCount: Integer): TBitHose;
    Procedure   AddBits( aBitCount: TBitCount; aValue: TBitCell);
    Function    ExtractBits( aBitCount: TBitCount): TBitCell;
    Procedure   ShiftInSeptets( Const aData: TByteArray);
    Procedure   ShiftInOctets ( Const aData: TByteArray);
    Procedure   AddSeptets( Const aData: TByteArray);
    Procedure   AddOctets ( Const aData: TByteArray);
    Procedure   Reset;
    Procedure   Clear;
    Procedure   Pad( aBitCount: TBitCount; aBitValue: TBitValue = 0);
    Function    GetOctets : TByteArray;
    Function    GetSeptets: TByteArray;
  Public
    Property    Size: Integer Read FSize Write SetSize;
    Property    Bits[ anBitIndex: Integer; aBitCount: TBitCount]: TBitCell Read GetBits Write SetBits;
    Property    ReadPointer: Integer Read FReadPtr Write FReadPtr;
  End;


Implementation


{ ========
  TBitHose = Class
  Private
    FData     : Array Of TBitCell;
    FSize     : Integer;
    FWritePtr : Integer;
    FReadPtr  : Integer;
  Public
    Property    Size: Integer Read FSize Write SetSize;
    Property    Bits[ anBitIndex: Integer; aBitCount: TBitCount]: TBitCell Read GetBits Write SetBits;
    Property    ReadPointer: Integer Read FReadPtr Write FReadPtr;
  Private
}

    Procedure   TBitHose.SetSize( aValue: Integer);
    Begin
      SetLength( FData, ( aValue + CellSize - 1 ) Div CellSize);
      FSize := aValue;
    End;

    Function    TBitHose.GetBits( anIndex: Integer; aBitCount: TBitCount): TBitCell;
    Var
      CrdPtr : Integer;
      BitPtr : TBitIndex;
      Filled : TBitCount;   // Nr of bits available in byte pointed to by CrdPtr
      Shift  : Integer;     // >= 0 -> right shift,  < 0 -> left shift
    Begin
      If aBitCount > 0
      Then Begin
        CrdPtr := anIndex Div CellSize;
        BitPtr := anIndex Mod CellSize;
        Filled := CellSize - BitPtr;
        Shift  := Filled - aBitCount;
        If Shift >= 0
        Then Result := ( FData[ CrdPtr] Shr Shift) And ( AllOnes Shr ( CellSize - aBitCount))
        Else
          Result :=
            (( FData[ CrdPtr] And ( AllOnes Shr BitPtr)) Shl -Shift) Or
            ( FData[ CrdPtr + 1] Shr ( CellSize + Shift));
      End
      Else Result := 0;
    End;

    Procedure   TBitHose.SetBits( anIndex: Integer; aBitCount: TBitCount; aValue: TBitCell);
    Var
      CrdPtr : Integer;
      BitPtr : TBitIndex;
      Free   : TBitCount;  // Free bits in cell pointed to by CrdPtr
      Shift  : Integer;    // >= 0 -> left shift, < 0 -> right shift
    Begin
      If aBitCount > 0
      Then Begin
        CrdPtr := anIndex Div CellSize;
        BitPtr := anIndex Mod CellSize;
        Size   := Max( anIndex + aBitCount, Size); // Ensure the new bits will fit.
        Free   := CellSize - BitPtr;               //   so we indeed always have 1 .. 32 free bits ...
        Shift  := Free - aBitCount;
        aValue := aValue And ( AllOnes Shr ( CellSize - aBitCount));
        If Shift >= 0
        Then FData[ CrdPtr] := ( FData[ CrdPtr] And ( AllOnes Shl Free)) Or ( aValue Shl Shift)
        Else Begin
          FData[ CrdPtr    ] := ( FData[ CrdPtr] And ( AllOnes Shl Free)) Or ( aValue Shr -Shift);
          FData[ CrdPtr + 1] := aValue Shl ( CellSize + Shift);
        End;
      End;
    End;

//   Public

    Constructor TBitHose.Create;
    Begin
      Inherited;
      Clear;
    End;

    Destructor  TBitHose.Destroy; // Override;
    Begin
      Clear;
      Inherited;
    End;

    Function    TBitHose.Clone: TBitHose;
    Begin
      Result := TBitHose.Create;
      Result.Size := Size;
      If Size > 0
      Then Move( FData, Result.FData, ( Size + CellSize - 1) Div CellSize);
    End;

    Function    TBitHose.GetBitHose( anIndex: Integer; aBitCount: Integer): TBitHose;
    Var
      i : Integer;
    Begin
      Result := TBitHose.Create;
      Result.Size := aBitCount;
      For i := 0 To aBitCount - 1 Do
        Result.AddBits( 1, Bits[ anIndex + i, 1]);
    End;

    Procedure   TBitHose.AddBits( aBitCount: TBitCount; aValue: TBitCell);
    Begin
      Bits[ FWritePtr, aBitCount] := aValue;
      Inc( FWritePtr, aBitCount);
    End;

    Function    TBitHose.ExtractBits( aBitCount: TBitCount): TBitCell;
    Begin
      Result := Bits[ FReadPtr, aBitCount];
      Inc( FReadPtr, aBitCount);
    End;

    Procedure   TBitHose.ShiftInSeptets( Const aData: TByteArray);
    Var
      i : Integer;
    Begin
      For i := 0 To Length( aData) - 1 Do
        AddBits( 7, aData[ i]);
    End;

    Procedure   TBitHose.ShiftInOctets( Const aData: TByteArray);
    Var
      i : Integer;
    Begin
      For i := 0 To Length( aData) - 1 Do
        AddBits( 8, aData[ i]);
    End;

    Procedure   TBitHose.AddSeptets( Const aData: TByteArray);
    Begin
      ShiftInSeptets( aData);
    End;

    Procedure   TBitHose.AddOctets ( Const aData: TByteArray);
    Begin
      ShiftInOctets( aData);
    End;

    Procedure   TBitHose.Reset;
    Begin
      FReadPtr  := 0;
      FWritePtr := 0;
    End;

    Procedure   TBitHose.Clear;
    Begin
      Reset;
      Size := 0;
    End;

    Procedure   TBitHose.Pad( aBitCount: TBitCount; aBitValue: TBitValue = 0);
    Begin
      While FWritePtr Mod aBitCount <> 0 Do
        AddBits( 1, aBitValue);
    End;

    Function    TBitHose.GetOctets: TByteArray;
    Var
      i : Integer;
    Begin
      SetLength( Result, Size Div 8);
      For i := 0 To Length( Result) - 1 Do
        Result[ i] := Bits[ i * 8, 8] And $ff;
    End;

    Function    TBitHose.GetSeptets: TByteArray;
    Var
      i : Integer;
    Begin
      SetLength( Result, ( Size + 6) Div 7);
      For i := 0 To Length( Result) - 1 Do
        Result[ i] := Bits[ i * 7, 7] And $7f;
    End;


End.
