Unit datadefs;

//  ////////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2004 Jan Punter
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//  ////////////////////////////////////////////////////////////////////////////
//
//    lupd : 2004-04-04
//
//  ////////////////////////////////////////////////////////////////////////////


Interface

Uses

  SysUtils, Math, BitHose, Globals;

{$WARN UNSAFE_TYPE OFF}
{$WARN UNSAFE_CODE OFF}

Const

  MidiBlockSize = 419; // The number of whole octets in  a full size MIDI packet

Type

  TByteFile      = File Of Byte;                                               // Binary file of byte
  TPatchDataType = ( pdtInvalid, pdtPatch, pdtPerformance);                    // patch type defs
  TString16      = String[ 16];                                                // MIDI String16
  TOnParseError  = Procedure( aSender: TObject; Const aMsg: String) Of Object; // 'Method' pointer
  TOnExecute     = Procedure( aSender: TObject; aData: TByteArray; Const IsLastData: Boolean) Of Object;
  EParseError    = Class( Exception);                                          // Parse exception

  TPatchData = Class
  Private
    FFileName      : String;          // Patch file name
    FPatchDataType : TPatchDataType;  // Patch type as an enumeration type
    FVersionString : String;          // Patch version as a string
    FTypeString    : String;          // Patch type    as a string
    FVersionInt    : Integer;         // Patch version as an integer
    FInfoString    : String;          // Build info string
    FData          : Array Of Byte;   // The binary contents of the patch as octets
    FDataSize      : Integer;         // Number of octets available in FData
    FConvertor     : TBitHose;        // Septet <-> Octet conversions
    FChecksum      : Integer;         // MIDI checksum septet
  Private
    FOnParseError  : TOnParseError;   // Error     handler (pointer)
    FOnExecute     : TOnExecute;      // Execution handler (pointer)
  Private
    Procedure   SetFileName( Const aValue: String);
    Function    GetFileExtension: String;
    Function    GetData ( anIndex: Integer): Byte;
    Procedure   SetPatchDataType( aValue: TPatchDataType);
  Private
    Procedure   AddOctets ( Const aData: Array Of Byte; aSize: Integer);
    Procedure   AddSeptets( Const aData: Array Of Byte; aSize: Integer);
    Procedure   Clear;
    Procedure   Error( Const aMsg: String);
    Procedure   ErrorFmt( Const aFormat: String; Const anArgs: Array Of Const);
    Procedure   AddMidiByte   ( Var aData: TByteArray; aByte: Byte);
    Procedure   AddMidiInt    ( Var aData: TByteArray; anInteger: Integer);
    Procedure   AddMidiName   ( Var aData: TByteArray);
    Procedure   AddMidiData   ( Var aData: TByteArray; Const AppendData: Array Of Byte);
    Function    CreateMidiBlock( aBlockNr, aBlockCount, anOffset, aSize: Integer): TByteArray;
  Private
    Procedure   ParsePatchHeader( Var aFile: TByteFile);
  Private
    Procedure   WritePatchByte  ( Var aFile: TByteFile; aByte: Byte);
    Procedure   WritePatchString( Var aFile: TByteFile; Const aValue : String);
    Procedure   WritePatchData  ( Var aFile: TBytefile);
  Private
    Function    ParseMidiConst  ( Var aFile: TByteFile; aValue: Byte): Byte;
    Function    ParseMidiSeptet ( Var aFile: TByteFile): Byte;
    Function    ParseMidiInt    ( Var aFile: TByteFile): Integer;
    Function    ParseMidiString ( Var aFile: TByteFile): TString16;
    Procedure   ParseMidiData   ( Var aFile: TByteFile; aSize: Integer);
  Private
    Procedure   LoadPatch( Var aFile: TByteFile);                      Overload;
    Procedure   SavePatch( Var aFile: TByteFile);                      Overload;
    Procedure   LoadMidi ( Var aFile: TByteFile);                      Overload;
    Procedure   SaveMidi ( Var aFile: TByteFile);                      Overload;
  Public
    Constructor Create;                                                 Virtual;
    Destructor  Destroy;                                               Override;
    Procedure   Execute;
  Public
    Procedure   LoadPatch( Const aFileName: String);                   Overload;
    Procedure   SavePatch( Const aFileName: String);                   Overload;
    Procedure   LoadMidi ( Const aFileName: String);                   Overload;
    Procedure   SaveMidi ( Const aFileName: String);                   Overload;
  Public
    Property    Data [ anIndex: Integer]: Byte Read GetData;
  Public
    Property    FileName       : String         Read FFileName      Write SetFileName;
    Property    FileExtension  : String         Read GetFileExtension;
    Property    DataSize       : Integer        Read FDataSize;
    Property    PatchDataType  : TPatchDataType Read FPatchDataType Write SetPatchDataType;
    Property    VersionString  : String         Read FVersionString;
    Property    TypeString     : String         Read FTypeString;
    Property    VersionInt     : Integer        Read FVersionInt;
    Property    InfoString     : String         Read FInfoString;
  Public
    Property    OnParseError   : TOnParseError  Read FOnParseError  Write FOnParseError;
    Property    OnExecute      : TOnExecute     Read FOnExecute     Write FOnExecute;
  End;



Implementation


{ ========
  TPatchData = Class
  Private
    FFileName      : String;          // Patch file name
    FPatchDataType : TPatchDataType;  // Patch type as an enumeration type
    FVersionString : String;          // Patch version as a string
    FTypeString    : String;          // Patch type    as a string
    FVersionInt    : Integer;         // Patch version as an integer
    FInfoString    : String;          // Build info string
    FData          : Array Of Byte;   // The binary contents of the patch as octets
    FDataSize      : Integer;         // Number of octets available in FData
    FConvertor     : TBitHose;        // Septet <-> Octet conversions
    FChecksum      : Integer;         // MIDI checksum septet
  Private
    FOnParseError  : TOnParseError;   // Error     handler (pointer)
    FOnExecute     : TOnExecute;      // Execution handler (pointer)
  Public
    Property    Data [ anIndex: Integer]: Byte Read GetData;
  Public
    Property    FileName       : String         Read FFileName      Write SetFileName;
    Property    FileExtension  : String         Read GetFileExtension;
    Property    DataSize       : Integer        Read FDataSize;
    Property    PatchDataType  : TPatchDataType Read FPatchDataType Write SetPatchDataType;
    Property    VersionString  : String         Read FVersionString;
    Property    TypeString     : String         Read FTypeString;
    Property    VersionInt     : Integer        Read FVersionInt;
    Property    InfoString     : String         Read FInfoString;
  Public
    Property    OnParseError   : TOnParseError  Read FOnParseError  Write FOnParseError;
    Property    OnExecute      : TOnExecute     Read FOnExecute     Write FOnExecute;
  Private
}

    Procedure   TPatchData.SetFileName( Const aValue: String);
    Begin
      FFileName := CleanFileName( aValue);
    End;

    Function    TPatchData.GetFileExtension: String;
    Begin
      Case PatchDatatype Of
        pdtPatch       : Result := '.pch2';
        pdtPerformance : Result := '.prf2';
        Else Error( 'Invalid patch file type, neither patch nor performance');
      End;
    End;

    Function    TPatchData.GetData( anIndex: Integer): Byte;
    Begin
      Result := FData[ anIndex];
    End;


    Procedure   TPatchData.SetPatchDataType( aValue: TPatchDataType);
    Begin
      FTypeString    := '';
      FPatchDatatype := aValue;
      Case FPatchDatatype Of
        pdtPatch       : FTypeString := 'Patch';
        pdtPerformance : FTypeString := 'Performance';
      End;
    End;

//  Private


    Procedure   TPatchData.AddOctets( Const aData: Array Of Byte; aSize: Integer);
    Var
      i       : Integer;
      NewSize : Integer;
    Begin
      NewSize := FDataSize + aSize;
      SetLength( FData, NewSize);
      For i := FDataSize To NewSize - 1 Do
        FData[ i] := aData[ i - FDataSize];
      FDataSize := NewSize;
    End;


    Procedure   TPatchData.AddSeptets( Const aData: Array Of Byte; aSize: Integer);
    Var
      LocData : TByteArray;
      i       : Integer;
    Begin
      SetLength( LocData, aSize);
      For i := 0 To aSize - 1 Do
      Begin
        LocData[ i] := aData[ i];
        FChecksum := ( FChecksum + LocData[ i]) And $7f;
      End;
      With FConvertor Do
      Begin
        Clear;
        AddSeptets( LocData);
        LocData := GetOctets;
      End;
      AddOctets( LocData, Length( LocData));
    End;


    Procedure   TPatchData.Clear;
    Begin
      SetLength( FData, 0);
      FDataSize      := 0;
      FVersionString := '';
      FTypeString    := '';
      FVersionInt    := 0;
      FInfoString    := '';
      PatchDataType  := pdtInvalid;
    End;


    Procedure   TPatchData.Error( Const aMsg: String);
    Begin
      If Assigned( FOnParseError)
      Then FOnParseError( Self, aMsg)
      Else Raise EParseError.Create( aMsg);
    End;


    Procedure   TPatchData.ErrorFmt( Const aFormat: String; Const anArgs: Array Of Const);
    Begin
      Error( Format( aFormat, anArgs));
    End;


    Procedure   TPatchData.AddMidiByte( Var aData: TByteArray; aByte: Byte);
    // Adds a MIDI byte to aData
    Begin
      SetLength( aData, Length( aData) + 1);
      aData[ Length( aData) - 1] := aByte;
      FChecksum := ( FChecksum + aByte) And $7f;
    End;

    Procedure   TPatchData.AddMidiInt( Var aData: TByteArray; anInteger: Integer);
    // Adds an integer (0 .. 16383) to aData
    Begin
      AddMidiByte( aData, ( anInteger Shr 7) And $7f); // Write high septet
      AddMidiByte( aData, anInteger And $7f);          // then  low  septet
    End;

    Procedure   TPatchData.AddMidiName( Var aData: TByteArray);
    // Adds the patch name as a String[ 16]\0 to aData
    Var
      i : Integer;
    Begin
      For i := 1 To 16 Do                            // Max 16 septets
        If i > Length( FFileName)
        Then AddMidiByte( aData, 0)
        Else AddMidiByte( aData, Byte( FFileName[ i]) And $7f);
      AddMidiByte( aData, $00);                      // and finalizing \0
    End;

    Procedure   TPatchData.AddMidiData( Var aData: TByteArray; Const AppendData: Array Of Byte);
    // Add AppendData to aData
    Var
      i : Integer;
    Begin
      For i := Low( AppendData) To High( AppendData) Do 
        AddMidiByte( aData, AppendData[ i]);
    End;

    Function    TPatchData.CreateMidiBlock( aBlockNr, aBlockCount, anOffset, aSize: Integer): TByteArray;
    // Writes one MIDI data block to Result
    Const
      Data : Array[ 0 .. 3] Of Byte = (              // MIDI header data
        $F0, // Sysex
        $33, // Clavia
        $7F, // ?
        $0A  // G2
      );
    Var
      i       : Integer;
      LocData : TByteArray;
      C       : Byte;
    Begin
      Setlength( result, 0);
      FChecksum := 0;                                // Cheksum starts at 0
      AddMidiData( result, Data);                    // Header
      Case PatchDataType Of                          // Dump type
        pdtPatch       : AddMidiByte( result, $20);  //   patch
        pdtPerformance : AddMidiByte( result, $28);  //   performance
        Else             Error( 'Invalid patch type, can''t write MIDI');
      End;
      AddMidiByte( Result, $00        );             // ?
      AddMidiByte( Result, $00        );             // ?
      AddMidiByte( Result, $00        );             // ?
      AddMidiInt ( Result, aBlockNr   );             // Block number
      AddMidiInt ( Result, aBlockCount);             // Block count
      AddMidiName( Result             );             // Patch name

      SetLength( LocData, aSize);                    // Collect data as septets
      For i := 0 To aSize - 1 Do
        LocData[ i] := FData[ anOffset + i];
      With FConvertor Do
      Begin
        Clear;
        AddOctets( LocData);                         // by adding octets
        LocData := GetSeptets;                       // and retrieving septets
      End;

      AddMidiInt( result, Length( LocData));         // Septet count
      For i := 0 To Length( LocData) - 1 Do          // Data septets
        AddMidiByte( result, LocData[ i]);

      C := FChecksum And $7f;
      AddMidiByte( Result, C);                       // Checksum
      AddMidiByte( Result, $f7);                     // End of sysex
    End;


//  Private


    Procedure   TPatchData.ParsePatchHeader( Var aFile: TByteFile);
    //
      Procedure ParseKVPair( Const S: String);
      //
        Procedure SetVersion( Const aValue: String);
        Var
          V : Integer;
        Begin
          V := StrToIntDef( avalue, 0);
          If V = 0
          Then FVersionString := aValue  // Non integer Value, probably string  version
          Else FVersionInt    := V;      //     integer Value, probably integer version
        End;
      //
        Procedure SetType( Const aValue: String);
        Begin
          FTypeString := aValue;
          If UpperCase( aValue) = 'PATCH'            // Patch is a patch
          Then PatchDataType := pdtPatch
          Else If UpperCase( aValue) = 'PERFORMANCE' // Patch is a performance
          Then PatchDataType := pdtPerformance
          Else Begin
            PatchDataType := pdtInvalid;             // patch is rubbish
            Error( 'Invalid patch file type, neither patch nor performance');
          End;
        End;
      //
      Var
        p     : Integer;
        Key   : String;
        Value : String;
      Begin                         // ParseKVPair
        p := Pos( '=', S);
        Key   := UpperCase( Copy( S, 1, p - 1));
        Value := Copy( S, p + 1, Length( S));
        If      Key = 'VERSION'
        Then SetVersion( Value)     // Two Version sections are present, string and int
        Else If Key = 'TYPE'
        Then SetType( Value)        // Always a string, must be checked though for its contents
        Else If Key = 'INFO'
        Then FInfoString := Value;  // Always a string (build info)
      End;                          // ParseKVPair
    //
    Var
      C : Byte;
      S : String;
    Begin                           // TPatchData.ParsePatchHeader
      S := '';
      While True Do
      Begin
        Read( aFile, C);
        Case( C) Of
          0   : Break;              // NOTE : Out of 'While True' loop !!
          $0d : ;                   // Skip CR
          $0a : Begin               // LF terminates the 'Key=Value' string
              ParseKVPair( S);      //   process the string
              S := '';              //   and then clear it.
            End;
          Else S := S + Char( C);   // Other characters belong to the string
        End;
      End;
    End;                            // TPatchData.ParsePatchHeader


//  Private


    Procedure   TPatchData.WritePatchByte( Var aFile: TByteFile; aByte: Byte);
    // Writes a byte to a patch file
    Begin
      Write( aFile, aByte);
    End;


    Procedure   TPatchData.WritePatchString( Var aFile: TByteFile; Const aValue : String);
    // Writes a string to a patch file
    Var
      i : Integer;
    Begin
      For i := 1 To Length( aValue) Do
        WritePatchByte( aFile, Byte( avalue[ i]));
      WritepatchByte( aFile, $0d);
      WritepatchByte( aFile, $0a);
    End;


    Procedure   TPatchData.WritePatchData( Var aFile: TBytefile);
    // Writes all octet data to a patch file
    Var
      i : Integer;
    Begin
      For i := 0 To DataSize - 1 Do
        WritePatchByte( aFile, Data[ i]);
    End;


//  Private


    Function    TPatchData.ParseMidiConst( Var aFile: TByteFile; aValue: Byte): Byte;
    // Reads a constant value from a MIDI file
    Var
      C : Byte;
    Begin
      Read( aFile, C);
      Result := C;
      If C <> aValue
      then ErrorFmt( 'Invalid MIDI dump, %.2x expected but %.2x read', [ aValue, C]);
      FChecksum := ( FChecksum + Result) And $7f;
    End;


    Function    TPatchData.ParseMidiSeptet( Var aFile: TByteFile): Byte;
    // Reads an arbitrary data value from a MIDI file
    Var
      C : Byte;
    Begin
      Read( aFile, C);
      If C > $7f
      Then ErrorFmt( 'Invalid MIDI dump, %.2x is not MIDI data', [ C]);
      Result := C;
      FChecksum := ( FChecksum + Result) And $7f;
    End;


    Function    TPatchData.ParseMidiInt( Var aFile: TByteFile): Integer;
    // Reads an Integer (0 .. 16383) value from a MIDI file
    Begin
      Result := 128 * ParseMidiSeptet( aFile);
      Inc( Result, ParseMidiSeptet( aFile));
    End;


    Function    TPatchData.ParseMidiString( Var aFile: TByteFile): TString16;
    // Reads a String[ 16]\0 from a MIDI file
    Var
      C : Byte;
      i : Integer;
    Begin
      Result := '';
      For i := 1 To 16 Do                    // Read the 16 characters
      Begin
        C := parseMidiSeptet( afile);
        If C <> 0                            // Don't add \0's to pascal string
        Then Result := Result + Char( C);
      End;
      ParseMidiConst( aFile, 0);             // Read the closing \0
    End;


    Procedure   TPatchData.ParseMidiData( Var aFile: TByteFile; aSize: Integer);
    // Reads in aSize data bytes from a MIDI file
    Const
      BlockSize = 1024;
    Var
      Data     : Array[ 0 .. BlockSize - 1] Of Byte;
      ReadSize : Integer;
      Actual   : Integer;
    Begin
      ReadSize := Min( BlockSize, aSize);
      BlockRead( aFile, Data, ReadSize, Actual);
      While Actual > 0 Do
      begin
        AddSeptets( Data, Actual);
        aSize    := aSize - Actual;
        ReadSize := Min( BlockSize, aSize);
        Blockread( aFile, Data, ReadSize, Actual);
      End;
    End;


//  Private


    Procedure   TPatchData.LoadPatch( Var aFile: TByteFile); // Overload;
    // Loads a patch or a performance file
    Const
      BlockLen = 8192;
    Var
      aData  : Array[ 0 .. BlockLen - 1] Of Byte;
      RCount : Integer;
    Begin
      ParsePatchHeader( aFile);                      // Read and parse header
      FDataSize := 1;                                // Add end of header byte
      SetLength( FData, 1);
      FData[ 0] := 0;
      BlockRead( aFile, aData, BlockLen, RCount);    // Get first data block
      AddOctets( aData, RCount);                     //   and save it
      While RCount = BlockLen Do
      Begin
        BlockRead( aFile, aData, BlockLen, RCount);  // Get next data block
        AddOctets( aData, RCount);                   //   and save it
      End;
    End;


    Procedure   TPatchData.SavePatch( Var aFile: TByteFile); // Overload;
    // Save data as a patch or a performance file
    Begin
      WritePatchString( aFile, Format( 'Version=%s', [ VersionString]));
      WritePatchString( aFile, Format( 'Type=%s'   , [ TypeString   ]));
      WritePatchString( aFile, Format( 'Version=%d', [ VersionInt   ]));
      WritePatchString( aFile, Format( 'Info=%s'   , [ InfoString   ]));
      WritepatchData  ( aFile);
    End;


    Procedure   TPatchData.LoadMidi( Var aFile: TByteFile); // Overload;
    // Reads all MIDI data blocks from a MIDI file
    Var
      MData        : Integer;
      FName        : String;
      BlockCount   : Integer;
      CurrentBlock : Integer;
      pdt          : TpatchDataType;
    Begin
      FileName       := '';
      BlockCount     := 0;
      CurrentBlock   := 0;
      FPatchDataType := pdtInvalid;
      pdt            := pdtInvalid;
      While Not Eof( aFile) Do
      Begin
        FChecksum := 0;
        ParseMidiConst( aFile, $f0);            // Start sysex
        ParseMidiConst( aFile, $33);            // Clavia
        ParseMidiConst( aFile, $7f);            // ?
        ParseMidiConst( aFile, $0a);            // G2
        Case ParseMidiSeptet( aFile) Of         // patch / performance selector
          $20 : pdt := pdtPatch;
          $28 : pdt := pdtPerformance;
          Else  Error( 'Invalid MIDI dump, neither patch nor performance');
        End;
        If PatchDataType = pdtInvalid
        Then PatchDataType := pdt
        Else
          If patchDataType <> pdt
          Then Error( 'Invalid MIDI, data type changed during load');
        ParseMidiConst( aFile, $00);            // ?
        ParseMidiConst( aFile, $00);            // ?
        ParseMidiConst( aFile, $00);            // ?
        MData := ParseMidiInt( aFile);          // Block Number
        If MData <> CurrentBlock
        Then ErrorFmt( 'Invalid MIDI, block %d expected but block %d seen', [ CurrentBlock, MData]);
        MData := ParseMidiInt( afile);          // Block count
        If BlockCount = 0
        Then BlockCount := MData
        Else
          If BlockCount <> MData
          Then ErrorFmt( 'Block count changed during MIDI load (%d -> %d)', [ BlockCount, MData]);
        FName := ParseMidiString( aFile);       // Patch name
        If FileName = ''
        Then FileName := FName
        Else
          If FileName <> FName
          Then
            ErrorFmt(
              'Filename changed during MIDI load (''%s'' -> ''%s'')',
              [ FileName, FName]
            );
        MData := ParseMidiInt( aFile);          // Septet count
        ParseMidiData( aFile, MData);           // Read and add data
        MData := FChecksum;
        If ParseMidiSeptet( afile) <> MData     // Checksum
        Then ErrorFmt( 'Invalid MIDI data, checksum error in block %d', [ CurrentBlock]);
        ParseMidiConst( aFile, $f7);            // End of sysex
        Inc( CurrentBlock);
      End;
      // For the time being the following is performed always,
      // the actual info should somehow be obtained from the MIDI file, later ...
      FVersionString := 'Nord Modular G2 File Format 1';
      FVersionInt    := 19;
      FInfoString    := 'BUILD 146';
    End;


    Procedure   TPatchData.SaveMidi( Var aFile: TByteFile); // Overload;
    // Saves all data as a set of MIDI blocks to a MIDI file
    Var
      Offset     : Integer;     // Current septet start offset
      BlockNr    : Integer;     // Current MIDI block number
      BlockCount : Integer;     // Total block count
      Remaining  : Integer;     // Total of remaining septets tp handle
      SaveSize   : Integer;     // Save size for current block, in septets
      Data       : TByteArray;
      i          : Integer;
      C          : Byte;
    Begin
      SetLength( Data, 0);
      BlockCount := ( DataSize + MidiBlockSize - 1) Div MidiBlockSize;
      BlockNr    :=  0;
      Offset     :=  0;
      Remaining  := DataSize;
      SaveSize   := Min( Remaining, MidiBlockSize);
      While Remaining > 0 Do
      Begin
        Data := CreateMidiBlock( BlockNr, BlockCount, Offset, SaveSize);
        For i := 0 To Length( Data) - 1 Do
        Begin
          C := Data[ i];
          Write( aFile, C);
        End;
        Inc( BlockNr);
        Inc( Offset, MidiBlockSize);
        Remaining := Remaining - SaveSize;
        SaveSize  := Min( Remaining, MidiBlockSize);
      End;
    End;


//  Public


    Constructor TPatchData.Create; // Virtual;
    // Creates and resets a TPatchData struct
    Begin
      Inherited Create;
      FConvertor := TBitHose.Create;
      Clear;
    End;


    Destructor  TPatchData.Destroy; // Override;
    // Get rid of selff
    Begin
      FreeAndNil( FConvertor);
      Clear;
      Inherited Destroy;
    End;


    Procedure   TPatchData.Execute;
    // Saves all data as a set of MIDI blocks to a MIDI file
    Var
      Offset     : Integer;     // Current septet start offset
      BlockNr    : Integer;     // Current MIDI block number
      BlockCount : Integer;     // Total block count
      Remaining  : Integer;     // Total of remaining septets tp handle
      SaveSize   : Integer;     // Save size for current block, in septets
      Data       : TByteArray;
    Begin
      If Assigned( FOnExecute)
      Then Begin
        SetLength( Data, 0);
        BlockCount := ( DataSize + MidiBlockSize - 1) Div MidiBlockSize;
        BlockNr    :=  0;
        Offset     :=  0;
        Remaining  := DataSize;
        SaveSize   := Min( Remaining, MidiBlockSize);
        While Remaining > 0 Do
        Begin
          Data := CreateMidiBlock( BlockNr, BlockCount, Offset, SaveSize);
          FOnExecute( Self, Data, Not( BlockNr < BlockCount - 1));
          Inc( BlockNr);
          Inc( Offset, MidiBlockSize);
          Remaining := Remaining - SaveSize;
          SaveSize  := Min( Remaining, MidiBlockSize);
        End;
      End;
    End;


//  Public


    Procedure   TPatchData.LoadPatch( Const aFileName: String); // Overload;
    // Loads a patch or a performance filee
    Var
      aFile : TByteFile;
    Begin
      AssignFile( aFile, aFileName);
      Reset( aFile);
      Clear;
      Try
        LoadPatch( aFile);
        FileName := aFileName;
      finally
        CloseFile( aFile);
      End;
    End;


    Procedure   TPatchData.SavePatch( Const aFileName: String); // Overload;
    // Saves self as a patch or a performance file
    Var
      aFile : TByteFile;
    Begin
      AssignFile( aFile, aFileName);
      Rewrite( aFile);
      Try
        SavePatch( aFile);
        FileName := aFileName;
      Finally
        CloseFile( aFile);
      End;
    End;


    Procedure   TPatchData.LoadMidi( Const aFileName: String); // Overload;
    // Loads a patch or performnance in MIDI format
    Var
      aFile : TByteFile;
    Begin
      AssignFile( aFile, aFileName);
      Reset( aFile);
      Clear;
      Try
        LoadMidi( aFile);
      Finally
        CloseFile( aFile);
      End;
    End;


    Procedure   TPatchData.SaveMidi( Const aFileName: String); // Overload;
    // Saves self as a patch or performance in MIDI format
    Var
      aFile : TByteFile;
    Begin
      AssignFile( aFile, aFileName);
      Rewrite( aFile);
      Try
        SaveMidi( aFile);
      finally
        CloseFile( aFile);
      End;
    End;


End.
