`timescale 1ns / 1ps
//////////////////////////////////////////////////////////////////////////////////
// Engineer: Scott R. Gravenhorst
// email: music.maker@gte.net
// Create Date:    09/10/2007
// Design Name:    SVF
// Module Name:    SVF
// Project Name:   State Variable Filter Array
// Description:    SVF with shared multiplier.
// 
// Using 35 bit signed arithmetic to expand Q
//
// Execution time = 5 clocks
//
// 32 filters, RAM-based
//
// uses 4 dedicated multipliers
//
// - Improved timing
// - Added reset to squelch input to RAMs.  This should allow silencing an unstable
//   filter.
//
//////////////////////////////////////////////////////////////////////////////////

//module SVF( clk, ena, sel, f, q, In, Out, DIAG_LED );
module SVF( clk, ena, sel, f, q, In, Out, reset );
  parameter DATAWIDTH = 35;             // 35 bit signed arithmetic
  parameter HIBIT = DATAWIDTH - 1;
  parameter MULT_OUT_HIBIT = (2*DATAWIDTH)-1;

  input clk;                            // system clock
  input ena;                            // Tell the filter to go
  input         [4:0]     sel;          // select one of 32 filters
  input signed  [HIBIT:0] f;            // f (not Hz, but usable to control frequency)
  input signed  [HIBIT:0] q;            // q (1/Q)
  input signed  [23:0]    In;
  output signed [HIBIT:0] Out;
  input                   reset;
  
  wire clk;
  wire ena;
  wire         [4:0]     sel;
  wire signed  [HIBIT:0] f;
  wire signed  [HIBIT:0] q;
  wire signed  [23:0]    In;
  wire signed  [HIBIT:0] Out;
  wire                   reset;

  reg signed  [23:0] InReg;
  wire signed [HIBIT:0] In35;                       // input sign extended to 35 bits from 24 bits
  assign In35 = {{11{InReg[23]}},InReg};            // sign extension from 24 to 35 bits

///////////////////////////////////////////////////////////////////////////////////////
// SVF RAM for feedback storage  
  reg WE = 0;                          // RAM write enable

  reg signed  [69:0] Z1_I;             // RAM input register
  wire signed [69:0] Z1_O;             // RAM output - wire vector
  wire signed [69:0] I1;
  assign I1 = ( reset ) ? 0 : Z1_I ;
  SVF_RAM SVF_Z1 ( .clk(clk), .A(sel), .I(I1), .O(Z1_O), .WE(WE) );

  reg signed  [69:0] Z2_I;             // RAM input register
  wire signed [69:0] Z2_O;             // RAM output - wire vector
  wire signed [69:0] I2;
  assign I2 = ( reset ) ? 0 : Z2_I ;
  SVF_RAM SVF_Z2 ( .clk(clk), .A(sel), .I(I2), .O(Z2_O), .WE(WE) );

///////////////////////////////////////////////////////////////////////////////////////  

  reg  signed [HIBIT:0] mA = 0;
  reg  signed [HIBIT:0] mB = 0;
  wire signed [MULT_OUT_HIBIT:0] mP;                // multiplier output P
  assign mP = mA * mB;                              // shared multiplier

  assign Out = (Z2_O >>> 35);                       // assign Out = Z2 >>> 18;

  reg signed [69:0] tmp;

// SVF state machine, shares a multiplier
  reg       run   = 1'b0;
  reg [2:0] state = 3'b0;

  always @ ( posedge clk )
    begin
    if ( ena == 1'b1 ) 
      begin
      run <= 1'b1;
      state <= 3'd0;
      InReg <= In;                                  // lock the input value this enable cycle
      mA <= f;
      mB <= (Z1_O >>> 34);
      end
    else
      begin
      if ( run == 1'b1 )
        begin
        state <= state + 1;

        case ( state )

          3'd0:
            begin
            tmp <= ((In35 << 35) - Z2_O);

            Z2_I <= mP + Z2_O;                             // mP = f * (Z1_O >>> 34)

            mA <= q;
            mB <= (Z1_O >>> 34);                           // prepare for q * z1
            end

          3'd1:
            begin
            mA <= f;
            mB <= ( tmp - mP ) >>> 34;
            end
          
          3'd2:
            begin
            Z1_I <= mP + Z1_O;                             // mP = f * (((In35 << 35) - mP - Z2_O) >>> 34) 
            WE <= 1;                                       // set write enable true
            end

          3'd3:
            begin
            WE <= 0;                                       // set write enable false
            run <= 1'b0;                                   // stop state machine
            end
            
        endcase

        end
      end
    end
 endmodule
