`timescale 1ns / 1ps
// Engineer: Scott R. Gravenhorst
// COPYRIGHT 2009 and in perpetuity by and for SCOTT R. GRAVENHORST, all rights reserved.
// email: music.maker@gte.net
// Date Started: 2009-01-16
// Design Name: PolyDrum
//
///////////////////////////////////////////////////////////////////////////////////////////////////
//
// ver_a - take from ver_o of SVF_drum project (MagicSmoke).  This is an experiment, hopefully
//         we can get 32 SVFs running for minimum of 8 voices (4 SVFs per voice)
//
// ver_b - reduce sample rate to 100 KHz, easiest way to make 32 filters available.  Should be OK.
//         Compiles with close timing, less than 1 nS slack.
//
// ver_c - First cut at polyphony.  8 voices total, 4 resonators per voice.
//       - Never got to the polyphony part.  Had to fix the problem with negative values sent to
//         the expo generator.  Sign extension wasn't working properly so all negative values sent
//         were working as very large positive numbers.
//
// ver_d - First cut at polyphony.  Requires change to PicoBlaze code.
//         This synth will have a structure as follows:
//         Each of the 32 SVFs provides a single voice.  Each voice will be tunable and assignable 
//         to a MIDI note number.  More than one resonator (SVF) can be assigned to the same MIDI
//         note number to facilitate more complex sounds.  If only one voice is assigned per note
//         number, the "rack" of drums can provide chromatic scale tones of almost 3 octaves.  If
//         a "chord" of select notes is needed, this can be implemented as well by how the voices
//         are assigned to MIDI note numbers.
//         With each note on message, the PicoBlaze MIDI controller will output the note number 
//         and set a "key pressed" flipflop.  Each DACena, the flipflop will be checked.  If set,
//         the strike system will be initialized and started for any resonator with a matching
//         note number setting.
//
// ver_e - ver_d is a functional design, ver_e will enable expo envelope.  
//         Pulse width needs to be calculated by the patch editor and sent as K/Fc (where K is an 
//         arbitrary constant).
//         reset signal now squelches output (in case of instability).
//
// ver_f - an more bits to the frequency value extending the range upward.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
module PolyDrum ( CLK_16MHZ, LED, PUSH_RESET, PUSH_A, DIGI1, DIGI2_3, USB_RS232_Rx );
  input  CLK_16MHZ;                       // 16 MHz soldered clock.
  output [3:0] LED;                       // 4 LEDs on the board.
  input  PUSH_RESET;                      // MCU reset
  input  PUSH_A;
  output [3:0] DIGI1;                     // J7 - I2S DAC
  input  DIGI2_3;                         // J6 - MIDI         
  input  USB_RS232_Rx;                    // TTY input via USB-RS232 bridge

  parameter MIDI_CHAN = 1;                // SET THE ACTUAL MIDI CHANNEL HERE
  parameter HARDWARE_CHAN = MIDI_CHAN-1;  // subtract 1 from MIDI_CHAN for hardware.

  wire CLK_16MHZ;  
  wire [3:0] LED;
  wire PUSH_RESET;
  wire PUSH_A;
  wire [3:0] DIGI1;
  wire DIGI2_3;
  wire USB_RS232_Rx;
  
///////////////////////////////////////////////////////////////////////////////////////////////////

  wire [3:0] rstd;                // POR delay
  wire reset;                     // POR/User reset
  
  wire interrupt;
  wire interrupt_ack;
  wire interrupt0;
  wire interrupt1;
  wire [9:0] address;             // wires to connect address lines from uC to ROM
  wire [17:0] instruction;        // uC data lines, need connection between uC and ROM
  wire [7:0] out_port;            //
  wire [7:0] in_port;             // 
  wire [7:0] port_id;
  reg [7:0] in_port_reg;          // hold data for mcu
  assign in_port = in_port_reg;

// MIDI & TTY UART receivers
  wire [7:0] rx0data;
  wire rx0ready_status;
  wire reset_rx0ready_status;
  
  wire [7:0] rx1data;
  wire reset_rx1ready_status;

  wire resetsignal;
// MCU
  wire read_strobe;
  wire write_strobe;

///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
// 38.4 MHz clock generator

  wire clk;                       // for 38.4 MHz clock

//  DCM_46_0000 DCM ( .CLKIN_IN( CLK_16MHZ ), .RST_IN( 1'b0 ), .CLKFX_OUT( clk ), .CLKIN_IBUFG_OUT(), .CLK0_OUT() );
  DCM_38_4 DCM ( .CLKIN_IN( CLK_16MHZ ), .RST_IN( 1'b0 ), .CLKFX_OUT( clk ), .CLKIN_IBUFG_OUT(), .CLK0_OUT() );
  
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
// POR delay FF chain - taken from Eric Brombaugh's code for the SPI DAC.
  FDCE rst_bit0 (.Q(rstd[0]), .C(clk), .CE(1'b1), .CLR(1'b0), .D(1'b1));
  FDCE rst_bit1 (.Q(rstd[1]), .C(clk), .CE(1'b1), .CLR(1'b0), .D(rstd[0]));
  FDCE rst_bit2 (.Q(rstd[2]), .C(clk), .CE(1'b1), .CLR(1'b0), .D(rstd[1]));
  FDCE rst_bit3 (.Q(rstd[3]), .C(clk), .CE(1'b1), .CLR(1'b0), .D(rstd[2]));
  assign reset = ~rstd[3] | PUSH_RESET;

///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////
// instantiate the uC (kcpsm3) with it's ROM

  kcpsm3 MCU ( .address(address), .instruction(instruction), .port_id(port_id), 
    .write_strobe(write_strobe), .out_port(out_port), .read_strobe(read_strobe), .in_port(in_port), 
    .interrupt(interrupt), .interrupt_ack(interrupt_ack), 
    .reset(reset), .clk(clk) );  

  midictrl PSMROM ( .address(address), .instruction(instruction), .clk(clk) );
  
  // MIDI receiver.  31,250 Baud
  assign resetsignal = write_strobe && ( port_id == 8'hFF );  // When port_id == FF with write strobe, reset the UARTs
  assign MIDI_In = DIGI2_3;

///////////////////////////////////////////////////////////////////////////////////////
// MIDI UART
  MIDIuartrx RX0 ( .dout(rx0data), .clk(clk), .reset(resetsignal), .rxd(MIDI_In), 
    .frame(), .overrun(), .ready(), .busy(), .CS(), 
    .interrupt(interrupt0), .interrupt_ack(interrupt_ack), 
    .rxready_status(rx0ready_status), 
    .reset_rxready_status(reset_rx0ready_status)
    );
  /////// VERY IMPORTANT HARDWARE /////////////////////////////////////////////
  // decode read port 01, send pulse to reset rxready flop
  // This allows the mcu to clear the rxready bit automatically just by reading rxdata.
  assign reset_rx0ready_status = (read_strobe == 1'b1) & (port_id[3:0] == 4'h1);

///////////////////////////////////////////////////////////////////////////////////////
// TTY UART, 115.2 or 19.2 kilobuad (baudrate configured in module)
  TTYuartrx RX1 ( .dout(rx1data), .clk(clk), .reset(resetsignal), .rxd(USB_RS232_Rx), 
    .frame(), .overrun(), .ready(), .busy(), .CS(), 
    .interrupt(interrupt1), .interrupt_ack(interrupt_ack), 
    .rxready_status(rx1ready_status), 
    .reset_rxready_status(reset_rx1ready_status)
    );
  /////// VERY IMPORTANT HARDWARE /////////////////////////////////////////////
  // decode read port 09, send pulse to reset rxready flop
  // This allows the mcu to clear the rxready bit automatically just by reading rxdata.
  assign reset_rx1ready_status = (read_strobe == 1'b1) & (port_id[3:0] == 4'h9);

// common
  assign interrupt = (interrupt0 | interrupt1);  // ISR gets to figure out which UART did it.

//////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////
// DAC - 200 KHz sample rate
// MCLK = clk = 38.4 MHz, so there are 192 clk clocks per DACena

  wire DACena;                            // DACena is generated by the I2S DAC hardware
  reg signed  [23:0] L_DACreg = 24'h0;    // LEFT  or WHITE
  reg signed  [23:0] R_DACreg = 24'h0;    // RIGHT or RED

  i2s_out DAC ( .clk( clk ), .reset( reset ), 
                .l_data( L_DACreg ), .r_data( R_DACreg ), 
                .sdout( DIGI1[3] ), .sclk( DIGI1[2] ), .lrclk( DIGI1[1] ), 
                .load( DACena) );

  assign DIGI1[0] = clk;   // clk serves as master clock (MCLK)

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////  
// Synth Registers

  reg       MCU_GATE = 1'b0;

  reg [7:0] SYSEX_ADDR_MSB = 8'h00;
  reg [6:0] MOD_WHL;
  reg [6:0] VEL;
  reg [6:0] MCU_NOTE_NUM;                        // register where MCU stores current note number
  reg [3:0] MIDI_CHANNEL = 0;                    // There are no switches on this board, set the MIDI channel...

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////  
// Trigger generator (derived from MCU_GATE), high pulse one clk time wide.

  wire KEYPRESS;

  wire SET_KEYPRESS;
  reg  CLR_KEYPRESS;

  reg  MCU_GATE_old = 0;

  always @ ( posedge clk ) MCU_GATE_old <= MCU_GATE;

  assign SET_KEYPRESS = ( MCU_GATE != MCU_GATE_old && MCU_GATE == 1'b1 );
  
// This flip flop sets whenever there is a keypress:
   FDCPE #( .INIT(1'b0) ) PRESS ( .Q(KEYPRESS), .C(1'b0), .CE(1'b0), .CLR(CLR_KEYPRESS) , .D(1'b0), .PRE(SET_KEYPRESS) );

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// SVF - shared digital state variable filter
  
  wire signed [23:0] SVFin;
  wire signed [34:0] SVFout;
  reg                SVFena;
  reg         [4:0]  SVFsel = 0;
  
  reg         [6:0]  qvalH [31:0];                    // High 7 bits
  reg         [6:0]  qvalM [31:0];                    // Mid  7 bits
  reg         [6:0]  qvalL [31:0];                    // Low  7 bits
  reg         [20:0] q_reg;                           // 21 bit
  wire signed [34:0] q_wire;
  assign q_wire = {1'b0,q_reg,13'b0_0000_0000_0000};  // this is what the filter actually sees for q (where q = 1/Q).

//  reg [1:0] OCT_SHIFT = 0;                                  // changes tuning by N octaves

  reg         [1:0]  FcH [31:0];
  reg         [6:0]  FcM [31:0];
  reg         [6:0]  FcL [31:0];
  reg  signed [20:0] Fc;                              // Fc register for selected SVF
  wire signed [20:0] Fc_21bits;                       // concatenation of the 2 sysex bytes for Fc
//  assign Fc_21bits = ( ({7'b0000000,FcH[SVFsel],FcL[SVFsel]}) << OCT_SHIFT ) ; // 21 bit signed version for calculation
  assign Fc_21bits = {5'b00000,FcH[SVFsel],FcM[SVFsel],FcL[SVFsel]} ; // 21 bit signed version for calculation
  
  wire signed [34:0] f_wire;
  assign f_wire = {Fc,14'b00_0000_0000_0000}; // this is what the filter actually sees for Fc

  SVF SVF (
    .clk( clk ), 
    .ena( SVFena ),                      // ena tells filter to "go"
    .sel( SVFsel ),
    .f( f_wire ),                        // Filter corner frequency (not linear, but close at the low end of the range)
    .q( q_wire ),                        // Filter 1/Q value   (35 bits signed) [qval is 21 bits unsigned]
    .In( SVFin ),                        // signed 24 bit input
    .Out( SVFout ),                      // signed 35 bit output 
    .reset( reset )
    );
  
  reg signed [6:0] VelPulse [31:0] ;
  assign SVFin = {1'b0,VelPulse[SVFsel],16'h0000} ;   // shift to correct amplitude

  reg [6:0] PulseWidth [31:0];
  reg [6:0] PulseWidth_cache;

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// exponential curve generator

  reg signed  [6:0]  exp_DEL_H [31:0];
  reg signed  [6:0]  exp_DEL_L [31:0];
  reg signed  [13:0] exp_DEL;
  reg signed  [6:0]  exp_I_H   [31:0];
  reg signed  [6:0]  exp_I_L   [31:0];
  reg signed  [13:0] exp_I;
  reg         [31:0] exp_init = 0;
  reg                exp_ena = 0;
  wire signed [17:0] exp_O;
  wire signed [17:0] exp_I_sext;

  assign exp_I_sext = {{3{exp_I[13]}},exp_I,1'b0} ;
    
  expogen EXP ( .clk( clk ), 
                .init( exp_init ),          // 32 bits
                .ena( exp_ena ), 
                .sel( SVFsel ),
                .DEL( {exp_DEL,4'h0} ), 
                .I( exp_I_sext ),           // 18 bits signed
                .O( exp_O ) );              // 18 bits signed

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////  
//
  reg [6:0] NOTE_NUM [31:0];                // RAM for note number mapping storage

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////  
// Synth State Machine
//reg [3:0] DIAG0 = 0;
//reg [3:0] DIAG1 = 0;
//reg DIAG = 0;

  reg               run = 0;
  reg        [3:0]  state = 0;

  reg [15:0] STRIKE_CNTR ;                             // cache
  reg [15:0] STRIKE_CNTR_RAM [31:0] ;                  // dist RAM

  reg [6:0] NOTE_NUM_cache;
  
  reg SYNCHED_KEYPRESS = 0;

// DAC window registers
  reg        [3:0] WINDOW_SHIFTS = 4'h4;               // holds the number of shifts for the DAC window.
  reg signed [29:0] L_sum;
//  reg signed [29:0] R_sum;

  always @ ( posedge clk )
    begin
    if ( reset )
      begin
      SVFena   <= 0;
      state    <= 0;
      run      <= 0;
      L_DACreg <= 0;
      R_DACreg <= 0;
      end
    else
      begin
      if ( DACena )
        begin
        case ( WINDOW_SHIFTS )             // implement DAC window
          default: L_DACreg <= L_sum[29:6];    // default case sets to lowest amplification factor.
          4'h0:    L_DACreg <= L_sum[29:6];
          4'h1:    L_DACreg <= L_sum[28:5];
          4'h2:    L_DACreg <= L_sum[27:4];
          4'h3:    L_DACreg <= L_sum[26:3];
          4'h4:    L_DACreg <= L_sum[25:2];
          4'h5:    L_DACreg <= L_sum[24:1];
          4'h6:    L_DACreg <= L_sum[23:0];
          4'h7:    L_DACreg <= {L_sum[22:0],1'b0};
          4'h8:    L_DACreg <= {L_sum[21:0],2'b00};
          4'h9:    L_DACreg <= {L_sum[20:0],3'b000};
//          4'hA:    L_DACreg <= {L_sum[19:0],4'b0000};
//          4'hB:    L_DACreg <= {L_sum[18:0],5'b00000};
//          4'hC:    L_DACreg <= {L_sum[17:0],6'b000000};
//          4'hD:    L_DACreg <= {L_sum[16:0],7'b0000000};
//          4'hE:    L_DACreg <= {L_sum[15:0],8'b00000000};
        endcase
       
        state    <= 0;
        run      <= 1;
        SVFsel   <= 5'h00;                          // select SVF #0 
        L_sum    <= 0;                              // zero out L_sum register

        if ( KEYPRESS )
          begin        
          SYNCHED_KEYPRESS <= 1;                    // synched to DACena
          CLR_KEYPRESS <= 1;                        // clear the KEYPRESS flipflop
          end
        end

      else      // if NOT DACena
        
        begin
        
        CLR_KEYPRESS <= 0;
        
        if ( run )
          begin
          case ( state )
          
          4'h0:
            begin  state <= state + 1;
            NOTE_NUM_cache   <= NOTE_NUM[SVFsel];        // get the response note number for this resonator
            STRIKE_CNTR      <= STRIKE_CNTR_RAM[SVFsel]; // load the cache in case it's not a keypress
            PulseWidth_cache <= PulseWidth[SVFsel];
            end

          4'h1:
            begin  state <= state + 1;
            exp_ena            <= 1;
            if ( SYNCHED_KEYPRESS == 1'b1 && NOTE_NUM_cache == MCU_NOTE_NUM )
              begin              // we have a keypress and a note number match
              STRIKE_CNTR      <= ( (PulseWidth_cache) << 4 ) ;        // initialize strike pulse width counter cache - it will be written to the RAM in a later state
              VelPulse[SVFsel] <= VEL;
              exp_DEL          <= {exp_DEL_H[SVFsel],exp_DEL_L[SVFsel]};
              exp_I            <= {exp_I_H[SVFsel],exp_I_L[SVFsel]};   // set peak value of the envelope
              exp_init[SVFsel] <= 1;
              end
            else
              begin
              if ( STRIKE_CNTR != 0 )
                begin
                STRIKE_CNTR <= STRIKE_CNTR - 1;  // decrement STRIKE_CNTR
                end
              else 
                begin
                VelPulse[SVFsel] <= 0;                                 // turn pulse off when STRIKE_CNTR is zero          
                end
              end              
            end

          4'h2:
            begin  state <= state + 1;
            STRIKE_CNTR_RAM[SVFsel] <= STRIKE_CNTR;  // save cache back into RAM
            exp_init <= 0;              // zero all exp_init bits (should be 32)
            exp_ena  <= 0;

            Fc       <= Fc_21bits + {{7{exp_O[17]}},exp_O[17:4]} ;
            q_reg    <= {qvalH[SVFsel],qvalM[SVFsel],qvalL[SVFsel]};  // q value for SVF
            SVFena   <= 1;                                            // fire SVF
            end
            
          4'h3:                                           // finished SVF state 0
            begin  state <= state + 1;
            SVFena <= 0;
            end

          4'h4:                                           // finished SVF state 1
            begin  state <= state + 1;
            end

          4'h5:                                           // finished SVF state 2
            begin  state <= state + 1;
            end

          4'h6:                                           // finished SVF state 3
            begin  state <= state + 1;
            end

          4'h7:                                           // finished SVF state 4
            begin
            L_sum <= L_sum + SVFout;                      // accumulate DAC output value
            SVFsel <= SVFsel + 1;                         // select next SVF

            if ( SVFsel == 5'b11111 )                     // test if we've processed all configured SVFs 
              begin
              run <= 0;                                   // stop the state machine
              SYNCHED_KEYPRESS <= 0;                      // reset this after all resonators are processed.
              end
            else  
              begin
              state <= 0;                                 // restart at state 0 for next SVF
              end
            end

///////////////////////////////////////////////////////////////////

          endcase
          end
        end
      end
    end

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////  
// Decode structures for hardware receiving data from the MCU

  always @ ( posedge clk )
  begin
    if ( write_strobe == 1'b1 )
    begin
// This case block contains selection logic for system level ports and for CC ports.
// Note that these ports all have the bit 7 or port_id set.
    case ( port_id )
    8'hF3:       MOD_WHL               <= out_port[6:0];  // modulation wheel, global
    8'hF4:       MCU_NOTE_NUM          <= out_port[6:0];  //
    8'hF8:       MCU_GATE              <= out_port[0];    // MCU_GATE signal
    8'hF9:       VEL                   <= out_port[6:0];  // port to set synth hardware velocity register
    8'hE0:       SYSEX_ADDR_MSB        <= out_port[7:0];
//    8'hFE: MCU_LED <= out_port;           // for diagnostics
//    8'hFF: reset both UARTs.  See area of instantiation of UARTs    
    endcase
//////////////////////////////////////////////////////////////////////////////////////////
// These if blocks contain case blocks for sysex populated parameters     
// below this, all port_id values have bit 7 set to zero
// Address space is 14 bits.

    if ( SYSEX_ADDR_MSB == 8'h00 )                 // PAGE 0 - qval
      begin
      casex ( port_id )
      8'b000xxxxx:       qvalH[port_id[4:0]]          <= out_port[6:0];
      8'b001xxxxx:       qvalM[port_id[4:0]]          <= out_port[6:0];
      8'b010xxxxx:       qvalL[port_id[4:0]]          <= out_port[6:0];
      endcase
      end

    if ( SYSEX_ADDR_MSB == 8'h01 )                 // PAGE 1 - Fc
      begin
      casex ( port_id )
      8'b000xxxxx:       FcH[port_id[4:0]]            <= out_port[1:0];
      8'b001xxxxx:       FcM[port_id[4:0]]            <= out_port[6:0];
      8'b010xxxxx:       FcL[port_id[4:0]]            <= out_port[6:0];
      endcase
      end       

    if ( SYSEX_ADDR_MSB == 8'h02 )                 // PAGE 2 - Expo curve generator BW values
      begin
      casex ( port_id )
      8'b000xxxxx:       exp_DEL_H[port_id[4:0]]       <= out_port[6:0];
      8'b001xxxxx:       exp_DEL_L[port_id[4:0]]       <= out_port[6:0];
      endcase
      end       

    if ( SYSEX_ADDR_MSB == 8'h03 )                 // PAGE 3 - Expo curve generator amplitude values
      begin
      casex ( port_id )
      8'b000xxxxx:       exp_I_H[port_id[4:0]]        <= out_port[6:0];
      8'b001xxxxx:       exp_I_L[port_id[4:0]]        <= out_port[6:0];
      endcase
      end       

    if ( SYSEX_ADDR_MSB == 8'h04 )                 // PAGE 4 - Note Number mapping
      begin
      casex ( port_id )
      8'b000xxxxx:       NOTE_NUM[port_id[4:0]]       <= out_port[6:0];
      endcase
      end       

    if ( SYSEX_ADDR_MSB == 8'h05 )                 // PAGE 5 - Pulse width
      begin
      casex ( port_id )
      8'b000xxxxx:       PulseWidth[port_id[4:0]]     <= out_port[6:0];
      endcase
      end       

    if ( SYSEX_ADDR_MSB == 8'h7F )                 // PAGE 7F - system and global values
      begin
      case ( port_id )
      8'h00:             WINDOW_SHIFTS                <= out_port[3:0];
//      8'h01:             OCT_SHIFT                    <= out_port[1:0];
      8'h02:             MIDI_CHANNEL                 <= out_port[3:0];
      endcase
      end
    end
  end

  always @ ( posedge clk ) 
    begin
    casex ( port_id[3:0] )                               // decode and transfer data to in_port_reg
    4'h0:    in_port_reg <= {rx1ready_status,rx0ready_status,6'b000000}; // UART1 & UART0 rxready bits
    4'h1:    in_port_reg <= rx0data;                     // MIDI UART rxdata
    4'h2:    in_port_reg <= {4'b0000,MIDI_CHANNEL};      // MCU gets the MIDI channel here
    4'h9:    in_port_reg <= rx1data;                     // TTY UART rxdata
    default: in_port_reg <= 8'bxxxxxxxx;
	  endcase
    end

/////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////
// LEDs

  assign LED = (PUSH_A) ? MIDI_CHANNEL : {2'b00,MCU_GATE,~MIDI_In} ;

/////////////////////////////////////////////////////////////////////////////////////////////

endmodule
