/*********************************************************************************** 
 * Mars Radio Synth V0.1
 *
 *
 ***********************************************************************************/

#include "main.h" // Template Framework
#include "Synth_MarsRadio.h"


s8  LFSR[0x2000];

int PARAM_MR_ALG = 0;
int PARAM_MR_PITCH = 4;
int PARAM_MR_LENGTH = 0x800;
int PARAM_MR_ITERATIONS = 1;

int PARAM_MR_CUTOFF = 2048;
int PARAM_MR_Q;


GUI_Slider  CMR_LENGTH;
GUI_Slider  CMR_PITCH;
GUI_Slider  CMR_ITERATIONS;
GUI_Button  CMR_RESET;

GUI_Slider  CMR_Cutoff;
GUI_Slider  CMR_Q;

GUI_Button  CMR_ALG1;
GUI_Button  CMR_ALG2;
GUI_Button  CMR_ALG3;
GUI_Button  CMR_ALG4;



void MarsRadio_ResetLFSR (void) {
	int i;
	for (i=0; i<0x2000; i++) LFSR[i]=i;
}



void MarsRadio_GUI_AlgButtons_Set (int Alg) {
	GUI_SetButton(&CMR_ALG1,false);
	GUI_SetButton(&CMR_ALG2,false);
	GUI_SetButton(&CMR_ALG3,false);
	GUI_SetButton(&CMR_ALG4,false);
	switch (Alg)
	{
		case 0:  GUI_SetButton(&CMR_ALG1,true); break;
		case 1:  GUI_SetButton(&CMR_ALG2,true); break;
		case 2:  GUI_SetButton(&CMR_ALG3,true); break;
		case 3:  GUI_SetButton(&CMR_ALG4,true); break;
		default: GUI_SetButton(&CMR_ALG1,true); break;
	}
}

int MarsRadio_GUI_AlgButtons_Check (void) {
	bool old;
	static int newalg;
	old = GUI_GetButton(&CMR_ALG1);
	if (GUI_CheckButton(&CMR_ALG1) != old) { newalg = 0; MarsRadio_GUI_AlgButtons_Set(newalg); } 
	old = GUI_GetButton(&CMR_ALG2);
	if (GUI_CheckButton(&CMR_ALG2) != old) { newalg = 1; MarsRadio_GUI_AlgButtons_Set(newalg); }
	old = GUI_GetButton(&CMR_ALG3);
	if (GUI_CheckButton(&CMR_ALG3) != old) { newalg = 2; MarsRadio_GUI_AlgButtons_Set(newalg); }
	old = GUI_GetButton(&CMR_ALG4);
	if (GUI_CheckButton(&CMR_ALG4) != old) { newalg = 3; MarsRadio_GUI_AlgButtons_Set(newalg); }
	return newalg;
}




/**********************************************************************************
 * MarsRadio Init
 **********************************************************************************/
bool MarsRadio_Init (void) {
	static bool init = false;

	MarsRadio_ResetLFSR();

	screenClear(SCREEN_TOP);
	screenClear(SCREEN_BOTTOM);
	
	SetScreen(SCREEN_TOP);
		
	iprintf("\x1b[1;9HMars Radio V0.1");
	
	SetScreen(SCREEN_BOTTOM);

	// Setup LFSR Parameter
	iprintf("\x1b[0;1HLFSR:");
	GUI_CreateSliderBarH(&CMR_LENGTH,     8, (2<<3)-3, 128, 12, BLUE);
	GUI_CreateSliderBarH(&CMR_ITERATIONS, 8, (4<<3)-3, 128, 12, BLUE);
	GUI_CreateSliderBarH(&CMR_PITCH,      8, (6<<3)-3, 128, 12, BLUE);
	
	// Setup LFSR Mod Buttons
	iprintf("\x1b[2;19HRST"); 
	GUI_CreateButton(&CMR_RESET, (18<<3), (2<<3)-3, (5<<3)-2, 12, BLUE);
		
	// Setup Filter GUI
	iprintf("\x1b[8;1HFilter:");
	iprintf("\x1b[10;18HCutOff");
	iprintf("\x1b[12;18HQ");
	GUI_CreateSliderBarH(&CMR_Cutoff,8,77,128,12,BLUE);
	GUI_CreateSliderBarH(&CMR_Q,     8,93,128,12,BLUE);

	// Setup Algorithm Selection Buttons
	iprintf("\x1b[23;1HALG 1");
	iprintf("\x1b[23;9HALG 2");
	iprintf("\x1b[23;17HALG 3");
	iprintf("\x1b[23;25HALG 4");
	GUI_CreateButton(&CMR_ALG1, 0,     180, (8<<3)-2, 10, BLUE);
	GUI_CreateButton(&CMR_ALG2, 8<<3,  180, (8<<3)-2, 10, BLUE);
	GUI_CreateButton(&CMR_ALG3, 16<<3, 180, (8<<3)-2, 10, BLUE);
	GUI_CreateButton(&CMR_ALG4, 24<<3, 180, (8<<3)-2, 10, BLUE);

	// Default Values
	GUI_SetSliderBarH(&CMR_PITCH, PARAM_MR_PITCH<<3);
	GUI_SetSliderBarH(&CMR_LENGTH, PARAM_MR_LENGTH>>5);
	GUI_SetSliderBarH(&CMR_ITERATIONS, PARAM_MR_ITERATIONS);
	GUI_SetSliderBarH(&CMR_Cutoff, PARAM_MR_CUTOFF>>4);
	MarsRadio_GUI_AlgButtons_Set(PARAM_MR_ALG);

	init = true;

	return true;
}


/**********************************************************************************
 * MarsRadio Close Task, free allocated memory
 **********************************************************************************/
void MarsRadio_Close (void) {

}

/**********************************************************************************
 * MarsRadio Task
 **********************************************************************************/
void MarsRadio_Task (bool runtask) {
	
	if (!runtask) { // close synth and free resources
		MarsRadio_Close();
		return;
	} 
	
	SetScreen(SCREEN_BOTTOM);

	PARAM_MR_PITCH = (GUI_CheckSliderBarH(&CMR_PITCH) >> 3) + 1;
	PARAM_MR_LENGTH = GUI_CheckSliderBarH(&CMR_LENGTH) << 5;
	PARAM_MR_ITERATIONS = GUI_CheckSliderBarH(&CMR_ITERATIONS) >> 2 ;
	if (PARAM_MR_LENGTH == 0) PARAM_MR_LENGTH = 1;
	if (PARAM_MR_ITERATIONS == 0) PARAM_MR_ITERATIONS = 1;
	
	if (GUI_CheckButton(&CMR_RESET)) {
		GUI_SetButton(&CMR_RESET,false);
		MarsRadio_ResetLFSR();
	}
	
	//
	// Check Algorithm selection buttons
	//
	PARAM_MR_ALG = MarsRadio_GUI_AlgButtons_Check();
	
	//
	// Check LowPass Filter Parameter
	//
	PARAM_MR_CUTOFF = GUI_CheckSliderBarH(&CMR_Cutoff) << 1; 		
	PARAM_MR_Q      = GUI_CheckSliderBarH(&CMR_Q) << 1;
	PARAM_MR_CUTOFF <<= 2;                    // CutOff 0..2048
	if (PARAM_MR_Q > 0xEF) PARAM_MR_Q = 0xEF; // Q      0..0xEF 

	
	//
	// DEBUG Info
	//
	SetScreen(SCREEN_TOP);
	
	iprintf("\x1b[6;8HAlgorithm = 0x%02X", PARAM_MR_ALG+1);				
	iprintf("\x1b[7;8HPitch     = 0x%02X", PARAM_MR_PITCH);				
	iprintf("\x1b[8;8HLength    = 0x%04X", PARAM_MR_LENGTH);
	iprintf("\x1b[9;8HIteration = 0x%02X", PARAM_MR_ITERATIONS);

	iprintf("\x1b[11;8HCutOff    = 0x%04X", PARAM_MR_CUTOFF);
	iprintf("\x1b[12;8HQ         = 0x%02X", PARAM_MR_Q);

	//iprintf("\x1b[22;20HTX=%2d  TY=%2d", TOUCH.px>>3, TOUCH.py>>3);
	//iprintf("\x1b[23;20HX=%3d  Y=%3d", TOUCH.px, TOUCH.py);
}


/***********************************************************************************
 * DSP Algorithm
 ***********************************************************************************/
s8 DSP_MarsRadio (void) {
	static int cnt = 1;
	static int lfsrpos = 0;
	static s8  sample = 0;
	s8 feedback = 0;
	int i;

	if (cnt-- != 0) return sample;
	cnt = PARAM_MR_PITCH;

	for (i=0; i<PARAM_MR_ITERATIONS; i++) {
		sample = LFSR[lfsrpos];
		if (lfsrpos++ >= PARAM_MR_LENGTH) lfsrpos -= PARAM_MR_LENGTH;
		
		switch (PARAM_MR_ALG) {
			case 0:
					feedback = sample ^ LFSR[lfsrpos];
					break;
			case 1:		
					feedback = sample ^ (LFSR[lfsrpos] | LFSR[(lfsrpos>>1)]);
					break;
			case 2:
					feedback = sample ^ (LFSR[lfsrpos+255]);
					break;					
			default:
					PARAM_MR_ALG = 0;
		}			
		LFSR[lfsrpos] = feedback; 
	}

	// calc 12dB LowPass filter
	sample = DSP_Filter (sample, PARAM_MR_CUTOFF, PARAM_MR_Q, FILTER_LP);

	return sample;
}