/*********************************************************************************** 
 * Graphic API V0.1
 *
 * Toolchain: 
 * - DevKitPro Release 24 
 * - libnds 1.3.1
 ***********************************************************************************/

#include "main.h"


/***********************************************************************************
 * Graphic functions
 *
 * 
 ***********************************************************************************/

void GFX_Init (void) {
	// set vram banks
	vramSetMainBanks(VRAM_A_MAIN_BG, VRAM_B_MAIN_SPRITE, VRAM_C_SUB_BG, VRAM_D_SUB_SPRITE );

	// set the mode for 2 text layers and two extended background layers
	videoSetMode   (MODE_5_2D | DISPLAY_SPR_ACTIVE | DISPLAY_BG3_ACTIVE );
	videoSetModeSub(MODE_5_2D | DISPLAY_SPR_ACTIVE | DISPLAY_BG3_ACTIVE );

	// Init print consoles
	ConsoleTop    = *consoleInit(0, 0, BgType_Text4bpp, BgSize_T_256x256, 0, 1, true);
	ConsoleBottom = *consoleInit(0, 0, BgType_Text4bpp, BgSize_T_256x256, 0, 1, false);

	// set up top bitmap background
	int bgTop = bgInit(3, BgType_Bmp16, BgSize_B16_256x256, 2 , 0);
	bgSet(bgTop, 0, 256, 256, 0, 0, 0, 0);
	bgSetPriority(bgTop, 3);
	bgUpdate(bgTop);
	
	// set up bottom bitmap background
	int bgBtm = bgInitSub(3, BgType_Bmp16, BgSize_B16_256x256, 2 , 0);
	bgSet(bgBtm, 0, 256, 256, 0, 0, 0, 0);
	bgSetPriority(bgBtm, 3);
	bgUpdate(bgBtm);

	// Set bitmap background RAM adress
	RAM_BitmapTop    = BG_GFX     + 0x4000;
	RAM_BitmapBottom = BG_GFX_SUB + 0x4000;
}

// set actual drawing screen
void SetScreen (bool screen) {
	SCREEN = screen;
	if (screen)	consoleSelect(&ConsoleBottom);
	else		consoleSelect(&ConsoleTop);
}

// clear bitmap background
void backgroundClear (void) {
	if (SCREEN)	dmaFillWords(0, RAM_BitmapBottom, 0x18000); // 256*192*2
	else		dmaFillWords(0, RAM_BitmapTop,    0x18000); // 256*192*2
}

// clear console and background
void screenClear(bool screen) {
	SetScreen(screen);
	consoleClear();
	backgroundClear();
}

// set pixel
void setpixel (int x, int y, int color) {
	if (SCREEN)	RAM_BitmapBottom[(y<<8)+x]  = color | 0x8000; //BIT(15);
	else		RAM_BitmapTop[(y<<8)+x] 	= color | 0x8000; //BIT(15);
}

// draw line (Bresenham's line drawing algorithm)
void line (int x1, int y1, int x2, int y2, int color) {
	int dy = y2 - y1;
	int dx = x2 - x1;
	int stepx, stepy;

	if (dy < 0) { dy = -dy;  stepy = -1; } else { stepy = 1; }
	if (dx < 0) { dx = -dx;  stepx = -1; } else { stepx = 1; }
	dy <<= 1;                                                  
	dx <<= 1;                                                  

	setpixel(x1, y1, color);
	if (dx > dy) {
		int fraction = dy - (dx >> 1);                         
		while (x1 != x2) {
			if (fraction >= 0) {
				y1 += stepy;
				fraction -= dx;                                
			}
			x1 += stepx;
			fraction += dy;                                    
			setpixel(x1, y1, color);
		}
	} 
	else {
		int fraction = dx - (dy >> 1);
		while (y1 != y2) {
			if (fraction >= 0) {
				x1 += stepx;
				fraction -= dy;
			}
			y1 += stepy;
			fraction += dx;
			setpixel(x1, y1, color);
		}
	}
}

// draw line horizontal
void lineh (int x, int y, int x2, int color) {
	int i;
	x2++;
	for (i=x;i<x2;i++) setpixel(i,y,color);
}

// draw line vertical
void linev (int x, int y, int y2, int color) {
	int i;
	y2++;
	for (i=y;i<y2;i++) setpixel(x,i,color);
}

// draw rectangle
void rect (int x1, int y1, int x2, int y2, int color) {
	int i;
	for (i=x1;i<x2+1;i++) {
		setpixel(i,y1,color);
		setpixel(i,y2,color);
	}
	for (i=y1;i<y2+1;i++) {
		setpixel(x1,i,color);
		setpixel(x2,i,color);
	}
}

// draw filled rectangle
void rectfill (int x1, int y1, int x2, int y2, int color, int colorfill) {
	int i;
	rect(x1,y1,x2,y2,color); // draw border
	for (i=y1+1;i<y2;i++) lineh(x1+1,i,x2-1,colorfill); // fill rectangle
}

// draw circle
void circle (int xCenter, int yCenter, int radius, int color) {
	int x, y, r2;

	r2 = radius * radius;
	setpixel(xCenter, yCenter + radius, color);
	setpixel(xCenter, yCenter - radius, color);
	setpixel(xCenter + radius, yCenter, color);
	setpixel(xCenter - radius, yCenter, color);

	y = radius;
	x = 1;
	y = (int) (swiSqrt(r2 - 1) + 0.5);
	while (x < y) {
		setpixel(xCenter + x, yCenter + y, color);
		setpixel(xCenter + x, yCenter - y, color);
		setpixel(xCenter - x, yCenter + y, color);
		setpixel(xCenter - x, yCenter - y, color);
		setpixel(xCenter + y, yCenter + x, color);
		setpixel(xCenter + y, yCenter - x, color);
		setpixel(xCenter - y, yCenter + x, color);
		setpixel(xCenter - y, yCenter - x, color);
		x += 1;
		y = (int) (swiSqrt(r2 - x*x) + 0.5);
	}
	if (x == y) {
		setpixel(xCenter + x, yCenter + y, color);
		setpixel(xCenter + x, yCenter - y, color);
		setpixel(xCenter - x, yCenter + y, color);
		setpixel(xCenter - x, yCenter - y, color);
	}
}



/***********************************************************************************
 * GUI Functions (uses Grafic functions)
 *
 * 
 ***********************************************************************************/

// Button
void GUI_DrawButton (GUI_Button *buttondata) {
	int color;
	if (buttondata->value)  color = buttondata->color;
	else				    color = BLACK;
	rectfill(buttondata->x, buttondata->y, buttondata->x+buttondata->w, buttondata->y+buttondata->h, WHITE, color);	
}

void GUI_CreateButton (GUI_Button *buttondata, int x, int y, int w, int h, int color) {
	buttondata->x = x;
	buttondata->y = y;
	buttondata->w = w;
	buttondata->h = h;
	buttondata->value = false;
	buttondata->lock = false;
	buttondata->color = color;
	GUI_DrawButton (buttondata);
}

void GUI_SetButton (GUI_Button *buttondata, bool newvalue) {
	buttondata->value = newvalue;
	GUI_DrawButton (buttondata);
}

bool GUI_CheckButton (GUI_Button *buttondata) {
	if ((TOUCH.px == 0) && (TOUCH.py == 0)) buttondata->lock = false;
	if (buttondata->lock) return buttondata->value;
	if ( (TOUCH.px > buttondata->x) && (TOUCH.py > buttondata->y) && (TOUCH.px < buttondata->x+buttondata->w) && (TOUCH.py < buttondata->y+buttondata->h) )
	{
		buttondata->lock = true;
		buttondata->value = !buttondata->value;
		GUI_DrawButton (buttondata);
	}
	return buttondata->value;
}

bool GUI_GetButton (GUI_Button *buttondata) {
	return buttondata->value;
}

// Slider Horizontal
void GUI_DrawSliderH (GUI_Slider *sliderdata) {
	rectfill(sliderdata->x, sliderdata->y, sliderdata->x+sliderdata->w+1, sliderdata->y+sliderdata->h+1, WHITE, BLACK);	
	linev(sliderdata->value+sliderdata->x+1, sliderdata->y+1, sliderdata->y+sliderdata->h, sliderdata->color);	
}

void GUI_CreateSliderH (GUI_Slider *sliderdata, int x, int y, int w, int h, int color) {
	sliderdata->x = x;
	sliderdata->y = y;
	sliderdata->w = w;
	sliderdata->h = h;
	sliderdata->value = 0;
	sliderdata->color = color;
	GUI_DrawSliderH (sliderdata);
}

void GUI_SetSliderH (GUI_Slider *sliderdata, bool newvalue) {
	sliderdata->value = newvalue;
	GUI_DrawSliderH (sliderdata);
}

int  GUI_CheckSliderH (GUI_Slider *sliderdata) {
	if ( (TOUCH.px > sliderdata->x) && (TOUCH.py > sliderdata->y) && (TOUCH.px <= sliderdata->x+sliderdata->w) && (TOUCH.py <= sliderdata->y+sliderdata->h) )
	{
		sliderdata->value = TOUCH.px - sliderdata->x - 1;
		GUI_DrawSliderH (sliderdata);
	}
	return sliderdata->value;
}

int  GUI_GetSliderH (GUI_Slider *sliderdata) {
	return sliderdata->value;
}

// Slider Vertical
void GUI_DrawSliderV (GUI_Slider *sliderdata) {
	rectfill(sliderdata->x, sliderdata->y, sliderdata->x+sliderdata->w+1, sliderdata->y+sliderdata->h+1, WHITE, BLACK);	
	lineh(sliderdata->x+1, sliderdata->value+sliderdata->y+1, sliderdata->x+sliderdata->w, sliderdata->color);
}

void GUI_CreateSliderV (GUI_Slider *sliderdata, int x, int y, int w, int h, int color) {
	sliderdata->x = x;
	sliderdata->y = y;
	sliderdata->w = w;
	sliderdata->h = h;
	sliderdata->value = 0;
	sliderdata->color = color;
	GUI_DrawSliderV (sliderdata);
}

void GUI_SetSliderV (GUI_Slider *sliderdata, bool newvalue) {
	sliderdata->value = newvalue;
	GUI_DrawSliderV (sliderdata);
}

int  GUI_CheckSliderV (GUI_Slider *sliderdata) {
	if ( (TOUCH.px > sliderdata->x) && (TOUCH.py > sliderdata->y) && (TOUCH.px <= sliderdata->x+sliderdata->w) && (TOUCH.py <= sliderdata->y+sliderdata->h) )
	{
		sliderdata->value = TOUCH.py - sliderdata->y - 1;
		GUI_DrawSliderV (sliderdata);
	}
	return sliderdata->value;
}

int  GUI_GetSliderV (GUI_Slider *sliderdata) {
	return sliderdata->value;
}

// SliderBar Horizontal
void GUI_DrawSliderBarH (GUI_Slider *sliderdata) {
	rectfill(sliderdata->x, sliderdata->y, sliderdata->x+sliderdata->w+1, sliderdata->y+sliderdata->h+1, WHITE, BLACK);	
	rectfill(sliderdata->x+1,sliderdata->y+1, sliderdata->value+sliderdata->x+1, sliderdata->y+sliderdata->h, GRAY, sliderdata->color);
}

void GUI_CreateSliderBarH (GUI_Slider *sliderdata, int x, int y, int w, int h, int color) {
	sliderdata->x = x;
	sliderdata->y = y;
	sliderdata->w = w;
	sliderdata->h = h;
	sliderdata->value = 0;
	sliderdata->color = color;
	GUI_DrawSliderBarH (sliderdata);
}

void GUI_SetSliderBarH (GUI_Slider *sliderdata, bool newvalue) {
	sliderdata->value = newvalue;
	GUI_DrawSliderBarH (sliderdata);
}

int  GUI_CheckSliderBarH (GUI_Slider *sliderdata) {
	if ( (TOUCH.px > sliderdata->x) && (TOUCH.py > sliderdata->y) && (TOUCH.px <= sliderdata->x+sliderdata->w) && (TOUCH.py <= sliderdata->y+sliderdata->h) )
	{
		sliderdata->value = TOUCH.px - sliderdata->x - 1;
		GUI_DrawSliderBarH (sliderdata);
	}
	return sliderdata->value;
}

int  GUI_GetSliderBarH (GUI_Slider *sliderdata) {
	return sliderdata->value;
}

// SliderBar Vertical
void GUI_DrawSliderBarV (GUI_Slider *sliderdata) {
	int valinv = sliderdata->h-sliderdata->value;
	rectfill(sliderdata->x, sliderdata->y, sliderdata->x+sliderdata->w+1, sliderdata->y+sliderdata->h+1, WHITE, BLACK);	
	rectfill(sliderdata->x+1, sliderdata->y+valinv, sliderdata->w+sliderdata->x, sliderdata->y+sliderdata->h, GRAY, sliderdata->color);
}

void GUI_CreateSliderBarV (GUI_Slider *sliderdata, int x, int y, int w, int h, int color) {
	sliderdata->x = x;
	sliderdata->y = y;
	sliderdata->w = w;
	sliderdata->h = h;
	sliderdata->value = 0;
	sliderdata->color = color;
	GUI_DrawSliderBarV (sliderdata);
}

void GUI_SetSliderBarV (GUI_Slider *sliderdata, bool newvalue) {
	sliderdata->value = newvalue;
	GUI_DrawSliderBarV (sliderdata);
}

int  GUI_CheckSliderBarV (GUI_Slider *sliderdata) {
	if ( (TOUCH.px > sliderdata->x) && (TOUCH.py > sliderdata->y) && (TOUCH.px <= sliderdata->x+sliderdata->w) && (TOUCH.py <= sliderdata->y+sliderdata->h) )
	{
		sliderdata->value = sliderdata->h - TOUCH.py - sliderdata->y;
		GUI_DrawSliderBarV (sliderdata);
	}
	return sliderdata->value;
}

int  GUI_GetSliderBarV (GUI_Slider *sliderdata) {
	return sliderdata->value;
}

// PadXY
void GUI_DrawPadXY (GUI_PadXY *padxydata) {
	rectfill(padxydata->x, padxydata->y, padxydata->x+padxydata->w+1, padxydata->y+padxydata->h+1, WHITE, BLACK);	
	lineh(padxydata->x+1, padxydata->valueY+padxydata->y+1, padxydata->x+padxydata->w, padxydata->color);
	linev(padxydata->valueX+padxydata->x+1, padxydata->y+1, padxydata->y+padxydata->h, padxydata->color);			
}

void GUI_CreatePadXY (GUI_PadXY *padxydata, int x, int y, int w, int h, int color) {
	padxydata->x = x;
	padxydata->y = y;
	padxydata->w = w;
	padxydata->h = h;
	padxydata->valueX = 0;
	padxydata->valueY = 0;
	padxydata->color = color;
	GUI_DrawPadXY (padxydata);
}

void GUI_SetPadXY (GUI_PadXY *padxydata, int newvalueX, int newvalueY) {
	padxydata->valueX = newvalueX;
	padxydata->valueY = newvalueY;
	GUI_DrawPadXY (padxydata);
}

void GUI_CheckPadXY (GUI_PadXY *padxydata) {
	if ( (TOUCH.px > padxydata->x) && (TOUCH.py > padxydata->y) && (TOUCH.px <= padxydata->x+padxydata->w) && (TOUCH.py <= padxydata->y+padxydata->h) )
	{
		padxydata->valueX = TOUCH.px - padxydata->x - 1;
		padxydata->valueY = TOUCH.py - padxydata->y - 1;
		GUI_DrawPadXY (padxydata);
	}
}

int  GUI_GetPadX (GUI_PadXY *padxydata) {
	return padxydata->valueX;
}

int  GUI_GetPadY (GUI_PadXY *padxydata) {
	return padxydata->valueY;
}

// Bitfield
void GUI_DrawBitfield (GUI_Bitfield *bitfielddata) {
	int i;
	int color;
	int temp = bitfielddata->value;
	int x = bitfielddata->x;
	for (i=0; i<8; i++)
	{
		if ((temp&0x80) == 0x80)	color = bitfielddata->color;
		else					  	color = BLACK;
		rectfill(x,bitfielddata->y,x+bitfielddata->w,bitfielddata->y+bitfielddata->h,WHITE,color);
		x += bitfielddata->w;
		temp <<= 1;
	}
}

void GUI_CreateBitfield (GUI_Bitfield *bitfielddata, int x, int y, int w, int h, int color) {
	bitfielddata->x = x;
	bitfielddata->y = y;
	bitfielddata->w = w;
	bitfielddata->h = h;
	bitfielddata->value = 0;
	bitfielddata->lock = false;
	bitfielddata->color = color;
	GUI_DrawBitfield (bitfielddata);
}

void GUI_SetBitfield (GUI_Bitfield *bitfielddata, bool newvalue) {
	bitfielddata->value = newvalue;
	GUI_DrawBitfield (bitfielddata);
}

int  GUI_CheckBitfield (GUI_Bitfield *bitfielddata) {
	int i;
	int mask = 0x80; 
	int x = bitfielddata->x;
	int value = bitfielddata->value;
	if ((TOUCH.px == 0) && (TOUCH.py == 0)) bitfielddata->lock = false;
	if (bitfielddata->lock) return bitfielddata->value;
	if ((TOUCH.py > bitfielddata->y) && (TOUCH.py < bitfielddata->y+bitfielddata->h))
	{
		bitfielddata->lock = true;
		for (i=0; i<8; i++)
		{
			if ((TOUCH.px > x) && (TOUCH.px < x+bitfielddata->w))
			{
				if ((value&mask)!=0) value -= mask;
				else				 value |= mask;	
			}
			mask >>= 1;
			x += bitfielddata->w;
		}
		bitfielddata->value = value;
		GUI_DrawBitfield (bitfielddata);
	}
	return bitfielddata->value;
}

int  GUI_GetBitfield (GUI_Bitfield *bitfielddata) {
	return bitfielddata->value;
}

