`timescale 1ns / 1ps
// Title: ADSR.v
// Author: Scott R. Gravenhorst
// Date: 2008-04-19
//
// attack - decay - sustain - release, retriggerable.
// A, D and R represent the Attack rate, Decay rate and Release rate
// respectively.  S is the sustain level input.
//
// This version is a RAM based multi ADSR, capable of servicing 32 tone generators.
//
// Requires 3 clocks
//
// 2009-04-04:
//  This is for ver_n, expo/linear selectable release phase.
//  When expo_R == 0, release is linear
//  when expo_R == 1, release is exponential
//
// 2009-04-18:
//  Convert ADSR code to 3 clock single enable using expanded state machine
//  Expand ADSR to support 32 ADSRs instead of 8
//
// 2011-03-05:
//  Modifications for additive flute synth (16 bit output):
//  -- Reduce output width to 16 bits
//  -- Reduce ADSR_CNT to 4
//  The flute instrument doesn't need really slow A D and R rates.
//////////////////////////////////////////////////////////////////////////////////

module ADSR( out, clk, ena, sel, GATE, A, D, S, R, expo_R, is_idle );

// SIZE is the word size of the calculations made for the ADSR output.  It
// is adjusted to get the best "feel" from both the low and high ends of
// the range of values for A, D and R.  

  parameter SIZE = 28;
  parameter HIBIT = SIZE-1;

  parameter ADSR_CNT = 4;
  parameter ADSR_MAX = ADSR_CNT-1;
  
  parameter IDLE    = 3'b000;
  parameter ATTACK  = 3'b001;
  parameter DECAY   = 3'b010;
  parameter SUSTAIN = 3'b011;
  parameter RELEASE = 3'b100;
    
  output [15:0] out;
  input clk;                            // 50 MHz
  input ena;
  input [1:0] sel;                      // This signal selects the ADSR, make sure the bit width accomodates the full range.
  input GATE;                           // GATE signal
  input [14:0] A;                       // attack rate
  input [14:0] D;                       // decay rate
  input [14:0] S;                       // sustain level
  input [14:0] R;                       // release rate
  input expo_R;                         // flag for exponential release, uses R for control 
  output is_idle;
  
  wire clk;                             // 50 MHz
  wire ena;
  wire [1:0] sel;                       ////////////////////////  MUST ACCOMODATE ADSR_CNT
  wire GATE;                            // GATE signal
  wire [14:0] A;
  wire [14:0] D;
  wire [14:0] S;
  wire [14:0] R;
  wire signed [15:0] out;
  wire expo_R;
  
  reg signed [HIBIT:0] OUTregRAM [ADSR_MAX:0];   // This is the accumulator/integrator for attack, decay and release
  reg signed [HIBIT:0] OUTreg;                 

  reg is_idle = 0;

  wire signed [HIBIT:0] dif0;          // OUTreg - D 
  wire signed [HIBIT:0] dif1;          // OUTreg - R
  wire signed [HIBIT:0] sum0;          // OUTreg + A
  
  wire signed [HIBIT:0] IIR;           // OUTreg * b1

//  assign out = OUTreg[HIBIT:HIBIT-15];       // 16 bits signed.
  assign out = OUTregRAM[sel][HIBIT:HIBIT-15];       // 16 bits signed.
  
  assign sum0 = OUTreg + {{9{1'b0}},A,4'b0000} ;   // 9 + 15 + 4 = 28
  assign dif0 = OUTreg - {{9{1'b0}},D,4'b0000} ;
  assign dif1 = OUTreg - {{9{1'b0}},R,4'b0000} ;
  
  reg signed [15:0] b1;              // for expo release, still works like a rate control, lowest is slowest
  wire signed [15:0] RATE;
  assign RATE = {1'b0,R} ;  // 1 + 15 = 16
  
  wire signed [15:0] mA;
  wire signed [31:0] PROD;
  assign mA = OUTreg[27:12];    // make signed 16 bit value for multiplier as upper 18 bits of OUTregRAM value
  assign PROD = mA * b1 ;       // multiply b1 x mA
  assign IIR = PROD[31:4];      // IIR value is 30 bits of PROD excluding bit 35 (bit 34 is also a "sign" bit)

////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
// This is a state machine of 5 states, IDLE, ATTACK, DECAY, SUSTAIN and RELEASE.

  reg [7:0] count = 0;                    // clicking because too slow?  Trying 8 bits wide - will need smaller ADSR numbers
  reg [7:0] countRAM [ADSR_MAX:0];
  
  reg run = 0;
  reg state = 0;                            // state machine state
  reg [2:0] ADSRstateRAM [ADSR_MAX:0];      // state register holds machine state
  reg [2:0] ADSRstate;                      // ADSR state cache

  always @ ( posedge clk )
    begin
    if ( ena )
      begin
      OUTreg         <= OUTregRAM[sel];            // get OUTregRAM[] data for selected ADSR
      count          <= countRAM[sel]; 
      ADSRstate      <= ADSRstateRAM[sel];              
      b1             <= (16'sh7FFF - RATE);
      run            <= 1;
      state          <= 0;
      end
    else
      begin
      if ( run )
        begin
        case ( state )
        1'h0:
          begin  state <= 1'h1;

          case ( ADSRstate )

          IDLE: 
            begin
            if ( GATE == 1'b1 ) ADSRstate <= ATTACK;            // escape IDLE state to ATTACK on GATE high
            else                ADSRstate <= IDLE;
            end

          ATTACK:
              begin                        
              if ( GATE == 1'b0 ) 
                begin
                ADSRstate     <= RELEASE;                           // if GATE is LOW... advance to state 4 (release)
                end
              else            // GATE is high
                begin
                if ( sum0 >= 0 )
                  begin
                  OUTreg      <= sum0;                       // increment out by A
                  ADSRstate   <= ATTACK;                     // remain in ATTACK
                  end
                else                                        // if increment DOES cause overflow...
                  begin                                     // increment would cause overflow
                  OUTreg      <= {1'b0,{27{1'b1}}};          // so set out = max
                  ADSRstate   <= DECAY;                          // advance to state 2 (decay)
                  end
                end
              end

          DECAY:
              begin
              if ( GATE == 1'b1 )
                begin
                if ( dif0 > {1'b0,S,{12{1'b0}}} )      // 1 + 15 + 12 = 28
                  begin
                  OUTreg         <= dif0;                           // decrement out by D
                  ADSRstate      <= DECAY;
                  end
                else                                        // if decrement DID cause underflow...
                  begin
                  OUTreg         <= {1'b0,S,{12{1'b0}}} ;
                  ADSRstate      <= SUSTAIN;
                  end
                end
              else 
                begin
                ADSRstate        <= RELEASE;                           // gate went low, move to state 4
                end
              end

          SUSTAIN:                
              begin
              if ( GATE == 1'b0 )   ADSRstate <= RELEASE;         // Go to state 4, RELEASE
              else                  ADSRstate <= SUSTAIN;         // stay in sustain
              end

          RELEASE:
              begin
              if ( GATE == 1'b1 ) 
                begin
                ADSRstate           <= ATTACK;
                end
              else  
                begin                ///////////// GATE is LOW
                if ( expo_R )
                  begin
                  if ( |IIR != 0 ) // if ( IIR != 30'b000000000000000000000000000000 )    // if decrement DID cause underflow...
                    begin
                    ADSRstate       <= RELEASE;
                    if ( count == 8'h00 ) OUTreg <= IIR ;
                    count <= count + 8'h01;
                    end
                  else 
                    begin
                    OUTreg          <= 0;                          // set out = 0
                    ADSRstate       <= IDLE;
                    end
                  end
                else
                  begin
                  if ( dif1 > 0 )                                  // if decrement DID cause underflow...
                    begin
                    ADSRstate       <= RELEASE;
                    OUTreg          <= dif1;                       // decrement out by R
                    end
                  else 
                    begin
                    OUTreg          <= 0;                          // set out = 0
                    ADSRstate       <= IDLE;
                    end
                  end
                end
              end
            endcase
            end

          1'h1:
            begin
            OUTregRAM[sel]     <= OUTreg;
            ADSRstateRAM[sel]  <= ADSRstate;
            countRAM[sel]      <= count;
            run                <= 0;

            is_idle <= ( ADSRstate == IDLE ) ? 1 : 0 ;
            end
        endcase
        end
      end
    end

endmodule
