`timescale 1ns / 1ps
//////////////////////////////////////////////////////////////////////////////////
// Engineer: Scott R. Gravenhorst
// email: music.maker@gte.net
// Create Date:    09/10/2007
// Design Name:    SVF
// Module Name:    SVF
// Project Name:   State Variable Filter
// Description:    SVF with shared multiplier.
// 
// Execution time = 4 clocks
//
// 2010-03-17: Convert to 32 bits
//
//////////////////////////////////////////////////////////////////////////////////

module SVF( 
  clk,                          // system clock
  ena,                          // Tell the filter to go
  f,                            // f (not Hz, but usable to control frequency)
  q,                            // q (1/Q)
  In,
  Out
  );

  input clk;  
  input ena;
  input signed  [31:0] f;
  input signed  [31:0] q;
  input signed  [31:0] In;
  output signed [31:0] Out;

  wire clk;
  wire ena;
  wire signed [31:0] f;
  wire signed [31:0] q;
  wire signed [31:0] In;
  reg signed  [31:0] InReg;
  wire signed [31:0] Out;
  
  reg signed [63:0] z1 = 0;                // feedback #1
  reg signed [63:0] z2 = 0;                // feedback #2

  reg signed [63:0] tmp = 0;

//  assign Out = z2 >>> 31;                  // lowpass output
  assign Out = z1 >>> 31;                  // bandpass output  

  reg signed [31:0] mA = 0;
  reg signed [31:0] mB = 0;
  wire signed [63:0] mP;
  assign mP = mA * mB;

// SVF state machine, shares a multiplier
  reg run = 1'b0;
  reg [1:0] state = 0;

  always @ ( posedge clk )
    begin
    if ( ena )
      begin
      run <= 1;
      state <= 0;
      InReg <= In;     // lock the input value this enable cycle
      mA <= z1 >>> 31;
      mB <= q;
      end
    else
      begin
      if ( run )
        begin
        state <= state + 1;

        case ( state )
        
          2'd0:
            begin
            mA <= f;
            mB <= z1 >>> 31;

            tmp <= mP;
            end
          
          2'd1:
            begin
            mA <= f;
            mB <= ((InReg << 32) - tmp - z2) >>> 31;

            z2 <= mP + z2;
            end
          
          2'd2:
            begin
            z1 <= mP + z1;

            run <= 1'b0;
            end

        endcase

        end
      end
    end

//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
// C source for working floating point SVF:

/*
while ( fgets( buf, BUFSIZE, stdin ) != NULL )
  {
  input = atof( buf );
    
  multq = fb1 * q;
      
  sum1 = input + (-multq) + (-output);        
  mult1 = f * sum1;

  sum2 = mult1 + fb1;      
  mult2 = f * fb1;

  sum3 = mult2 + fb2;
                  
  fb1 = sum2;
  fb2 = sum3;
                      
  output = sum3;
  printf( "%20.18lf\n", output );
  }
*/

endmodule
