unit HrastUnit;

// Copyright  2014 by HrastProgrammer. All rights reserved.
// This unit contains WREN synthesizer modules developed by HrastProgrammer.
// Can be distributed only as part of the complete WREN source package.

interface

uses Module_Defs, Module_Implementations;

const Pow2Size=8192;

type

  THrastOsc=class(TMod)
  protected
    z,Phase: TSignal;
    PrevSync: Boolean;
  public
    procedure DoTick; override;
    procedure GatherLights(const Prefix: string; const Callback: TLightsHandler); override;
    function InputName(Index: Integer): string; override;
    function OutputName(Index: Integer): string; override;
  end;

var Pow2Table: array [0..Pow2Size+1] of TSignal;

function CreateModule(ModuleType: Integer; Parent: TSynthPatch; const Name: string; InputCount: Integer): TMod;

implementation

uses Globals, Math, SysUtils;

const HrastModules: array [4001..4001] of TModuleClass=(THrastOsc);

function CreateModule;
begin
  if (ModuleType<Low(HrastModules)) or (ModuleType>High(HrastModules)) then
    Result:=nil
  else
    Result:=HrastModules[ModuleType].Create(Parent,Name,InputCount)
end;

function Pow2Freq(Freq: TSignal): TSignal;
var
  n: Integer;
  f: TSignal;
begin
  f:=Abs(Freq)*Pow2Size;
  if f>Pow2Size-1 then f:=Pow2Size-1;
  n:=Trunc(f); f:=f-n;
  Result:=Pow2Table[n];
  Result:=Result+f*(Pow2Table[n+1]-Result)
end;

const
  Freq=0;
  Morph=1;
  Shape=2;
  Bandlimit=3;
  PW=4;
  PWM=5;
  PWMInput=6;
  FM=7;
  FMInput=8;
  PM=9;
  PMInput=10;
  Vintage=11;
  SyncInput=12;
  Level=13;
  Mute=14;

procedure THrastOsc.GatherLights;
begin
  Callback(Self,Format('%s/%s/SyncInput/%f',[Prefix,Name,FInCache[SyncInput]],AppLocale))
end;

function THrastOsc.InputName;
begin
  case Index of
    Freq: Result:='Freq';
    Morph: Result:='Morph';
    Shape: Result:='Shape';
    Bandlimit: Result:='Bandlimit';
    PW: Result:='PW';
    PWM: Result:='PWM';
    PWMInput: Result:='PWMInput';
    FM: Result:='FM';
    FMInput: Result:='FMInput';
    PM: Result:='PM';
    PMInput: Result:='PMInput';
    Vintage: Result:='Vintage';
    SyncInput: Result:='SyncInput';
    Level: Result:='Level';
    Mute: Result:='Mute'
    else Result:=No_Input
  end
end;

function THrastOsc.OutputName;
begin
  case Index of
    0: Result:='Output';
    else Result:=No_Output
  end
end;

procedure THrastOsc.DoTick;
var DeltaPhase: TSignal;

  function PolyBLEP(t: TSignal): TSignal;
  var dt: TSignal;
  begin
    dt:=FInCache[Bandlimit]*DeltaPhase;
    if t<dt then
    begin
      t:=t/dt-1;
      Result:=-t*t
    end
    else
      if t>1-dt then
      begin
        t:=(t-1)/dt+1;
        Result:=t*t
      end
      else
        Result:=0
  end;

var
  n: Integer;
  t,y,ypw: TSignal;
begin
  if not PrevSync and SignalToLogic(FInCache[SyncInput]) then Phase:=0;
  PrevSync:=SignalToLogic(FInCache[SyncInput]);
  if SignalToLogic(FInCache[Mute]) then begin FOutCache[0]:=0; Exit end;
  DeltaPhase:=Pow2Freq(FInCache[Freq]+FInCache[FM]*FInCache[FMInput])/System_Rate;
  t:=Phase+FInCache[PM]*FInCache[PMInput]; n:=Trunc(t); t:=t-n; if t<0 then t:=t+1;
  Phase:=Phase+DeltaPhase; n:=Trunc(Phase); Phase:=Phase-n; if Phase<0 then Phase:=Phase+1;
  y:=2*t-1-PolyBlep(t);
  t:=t+FInCache[PW]+FInCache[PWM]*FInCache[PWMInput]; n:=Trunc(t); t:=t-n; if t<0 then t:=t+1;
  ypw:=2*t-1-PolyBlep(t);
  t:=FInCache[Shape];
  if FInCache[Vintage]<>0 then
  begin
    if t>0.999 then t:=0.999;
    y:=(y+1)*0.5; y:=y*y; y:=2*y-1;
    ypw:=(ypw+1)*0.5; ypw:=ypw*ypw; ypw:=2*ypw-1
  end;
  z:=y-ypw*FInCache[Morph]+t*z;
  if t>0.99 then t:=0.99;
  y:=z*(t-1);
  FOutCache[0]:=y*FInCache[Level]
end;

procedure Initialize;
var i: Integer;
begin
  for i:=0 to Pow2Size do Pow2Table[i]:=UnitsToFrequency(i/Pow2Size);
  Pow2Table[Pow2Size+1]:=Pow2Table[Pow2Size];
  RegisterModuleTypes([THrastOsc])
end;

initialization

  Initialize

end.
