unit Convolution;

{
  after : https://github.com/HiFi-LoFi/FFTConvolver
}

interface

uses

  System.SysUtils, System.Math, fftw_, fftwgen, KnobsUtils, KnobsConversions;


{$ifdef DEBUG}

  // ---------------------------------------------------------------------------------------------
  // Shorter endless arrays will stop the debugger crashes a bit ... but I may have a size mistake
  // that I wouldnt want to creep into a release version.
  // ---------------------------------------------------------------------------------------------

const

  DEBUG_SHORT = 128 - 1;  // should be larger than or equal to 2 times the actual block size used minus 1
                          // (actually: next power of two of specified block size, minus one).

type

  Tfftw_short_real_array    = array[ 0 .. DEBUG_SHORT] of Tfftw_real;
  Tfftw_short_complex_array = array[ 0 .. DEBUG_SHORT] of Tfftw_complex;
  TSignals                  = array[ 0 .. DEBUG_SHORT] of TSignal;

{$else}

type

  Tfftw_short_real_array    = Tfftw_real_array;
  Tfftw_short_complex_array = Tfftw_complex_array;
  TSignals                  = array[ 0 .. MaxInt div SizeOf( TSignal) - 1] of TSignal;
{$endif}


  Pfftw_short_real_array    = ^Tfftw_short_real_array;
  Pfftw_short_complex_array = ^Tfftw_short_complex_array;
  PSignals                  = ^TSignals;


  TAudioFFT = class
  private
    FSize         : Integer;
    FComplexSize  : Integer;
    FPlanForward  : fftw_plan;
    FPlanBackWard : fftw_plan;
    FData         : Pfftw_real;
    FComplex      : Pfftw_complex;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Init( aSize: Integer);
    procedure   fft ( const aData: PSignals; aRe, anIm: PSignals);
    procedure   ifft( aData: PSignals; const aRe, anIm: PSignals);
  end;


  TSampleBuffer = class
  private
    FSize : Integer;
    FData : PSignals;
  private
    procedure   SetSize( aValue: Integer);
    function    GetDataElt( const anIndex: Integer): TSignal;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   SetZero;
  public
    property    Size                            : Integer read FSize      write SetSize;
    property    DataElt[ const anIndex: Integer]: TSignal read GetDataElt              ; default;
  end;


  TSplitComplex = class
  private
    FSize : Integer;
    FRe   : TSampleBuffer;
    FIm   : TSampleBuffer;
  private
    procedure   SetSize( aValue: Integer);
  public
    constructor Create( aSize: Integer = 0);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   SetZero;
    function    Re: PSignals;
    function    Im: PSignals;
  public
    property    Size: Integer read FSize write SetSize;
  end;


  TFFTConvolver = class
  private
    FBlockSize       : Integer;
    FSegSize         : Integer;
    FSegCount        : Integer;
    FComplexSize     : Integer;
    FSegments        : array of TSplitComplex;
    FSegmentsIR      : array of TSplitComplex;
    FfftBuffer       : TSampleBuffer;
    Ffft             : TAudioFFT;
    FpreMultiplied   : TSplitComplex;
    FOverlap         : TSampleBuffer;
    FCurrent         : Integer;
    FInputBuffer     : TSampleBuffer;
    FInputBufferFill : Integer;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    function    Init( aBlockSize: Integer; anImpulseResponse: TSignalArray): Boolean;
    procedure   Process( anInput: TSignal; var anOutput: TSignal);
    procedure   Reset;
  end;



implementation



  function IsPowerOfTwo( aValue: Integer): Boolean;
  begin
    Result := ( aValue and ( aValue - 1)) = 0;
  end;


  function NextPowerOfTwo( aValue: Integer): Integer;
  begin
    Result := 1;

    while Result < aValue
    do Result := Result shl 1;
  end;


  function ComplexSize( aSize: Integer): Integer;
  begin
    Result := ( aSize div 2) + 1;
  end;


  procedure CopyAndPad( aSrc: Pointer; const aLength: Integer; aDst: TSampleBuffer);
  begin
    Assert( aDst.Size > aLength);
    Move( aSrc^, aDst.FData[ 0], aLength * SizeOf( TSignal));
    FillChar( aDst.FData[ aLength], ( aDst.Size - aLength) * SizeOf( TSignal), 0);
  end;


  procedure ComplexMultiplyAccumulate( Re, Im: PSignals; const Rea, Ima, Reb, Imb: PSignals; const aSize: Integer); overload;
  var
    i : Integer;
  begin
    for i := 0 to aSize - 1
    do begin
      Re^[ i] := Re^[ i] + Rea^[ i] * Reb^[ i] - Ima^[ i] * Imb^[ i];
      Im^[ i] := Im^[ i] + Rea^[ i] * Imb^[ i] + Ima^[ i] * Reb^[ i];
    end;
  end;


  procedure ComplexMultiplyAccumulate( aResult: TSplitComplex; const a, b: TSplitComplex); overload;
  begin
    ComplexMultiplyAccumulate( aResult.Re, aResult.Im, a.Re, a.Im, b.Re, b.Im, aResult.Size);
  end;


{ ========
  TAudioFFT = class
  private
    FSize         : Integer;
    FComplexSize  : Integer;
    FPlanForward  : fftw_plan;
    FPlanBackWard : fftw_plan;
    FData         : Pfftw_real;
    FComplex      : Pfftw_complex;
  public
}


    constructor TAudioFFT.Create;
    begin
      FSize         := 0;
      FComplexSize  := 0;
      FPlanForward  := nil;
      FPlanBackWard := nil;
      FData         := nil;
      FComplex      := nil;
    end;


    destructor  TAudioFFT.Destroy; // override;
    begin
    end;


    procedure   TAudioFFT.Init( aSize: Integer);
    begin
      Assert( IsPowerOfTwo( aSize));

      if aSize <> FSize
      then begin
        if FSize > 0
        then begin
          fftw_destroy_plan( FPlanForward );
          fftw_destroy_plan( FPlanBackWard);
          FPlanForward  := nil;
          FPlanBackWard := nil;
          FSize        := 0;
          FComplexSize := 0;

          if Assigned( FComplex)
          then begin
            fftw_free( FComplex);
            FComplex := nil;
          end;
        end;

        if aSize > 0
        then begin
          FSize         := aSize;
          FComplexSize  := ComplexSize( FSize);
          FData         := fftw_malloc( FSize        * SizeOf( Tfftw_real   ));
          FComplex      := fftw_malloc( FComplexSize * SizeOf( Tfftw_complex));
          FPlanForward  := fftw_plan_dft_r2c_1d( FSize, FData, FComplex, FFTW_MEASURE);
          FPlanBackWard := fftw_plan_dft_c2r_1d( FSize, FComplex, FData, FFTW_MEASURE);
        end;
      end;
    end;

    procedure   TAudioFFT.fft( const aData: PSignals; aRe, anIm: PSignals);
    var
      i : Integer;
    begin
      Move( aData^, FData^, FSize * SizeOf( TSignal));
      fftw_execute_dft_r2c( FPlanForward, FData, FComplex);

      for i := 0 to FComplexSize - 1
      do begin
        aRe ^[ i] := Pfftw_short_complex_array( FComplex)^[ i].re.v;
        anIm^[ i] := Pfftw_short_complex_array( FComplex)^[ i].im.v;
      end;
    end;


    procedure   TAudioFFT.ifft( aData: PSignals; const aRe, anIm: PSignals);
    var
      i : Integer;
    begin
      for i := 0 to FComplexSize - 1
      do begin
        Pfftw_short_complex_array( FComplex)^[ i].re.v := aRe ^[ i];
        Pfftw_short_complex_array( FComplex)^[ i].im.v := anIm^[ i];
      end;

      fftw_execute_dft_c2r( FPlanForward, FComplex, FData);
      Move( FData^, aData^, FSize * SizeOf( TSignal));
    end;


{ ========
  TSampleBuffer = class
  private
    FSize : Integer;
    FData : PSignals;
  public
    property    Size                            : Integer read FSize      write SetSize;
    property    DataElt[ const anIndex: Integer]: TSignal read GetDataElt              ; default;
  private
}


    procedure   TSampleBuffer.SetSize( aValue: Integer);
    begin
      if aValue <> FSize
      then begin
        if Assigned( FData)
        then begin
          FreeMem( FData, FSize * SizeOf( TSignal));
          FData := nil;
        end;

        FSize := aValue;

        if FSize > 0
        then FData := AllocMem( FSize * SizeOf( TSignal)); // Allocate and zero
      end;
    end;


    function    TSampleBuffer.GetDataElt( const anIndex: Integer): TSignal;
    begin
      Result := FData^[ anIndex];
    end;


//  public

    constructor TSampleBuffer.Create;
    begin
      inherited Create;
      FData := nil;
      Size  := 0;
    end;


    destructor  TSampleBuffer.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TSampleBuffer.Clear;
    begin
      Size := 0;
    end;


    procedure   TSampleBuffer.SetZero;
    begin
      if Size > 0
      then FillChar( FData^, Size * SizeOF( TSignal), 0);
    end;


{ ========
  TSplitComplex = class
  private
    FSize : Integer;
    FRe   : TSampleBuffer;
    FIm   : TSampleBuffer;
  public
    property    Size: Integer read FSize write SetSize;
  private
}

    procedure   TSplitComplex.SetSize( aValue: Integer);
    begin
      FSize    := aValue;
      FRe.Size := aValue;
      FIm.Size := aValue;
    end;


//  public

    constructor TSplitComplex.Create( aSize: Integer = 0);
    begin
      inherited Create;
      FRe  := TSampleBuffer.Create;
      FIm  := TSampleBuffer.Create;
      Size := aSize;
    end;


    destructor  TSplitComplex.Destroy; // override;
    begin
      FreeAndNil( FRe);
      FreeAndNil( FIm);
      inherited;
    end;


    procedure   TSplitComplex.Clear;
    begin
      FRe.Clear;
      FIm.Clear;
      FSize := 0;
    end;


    procedure   TSplitComplex.SetZero;
    begin
      FRe.SetZero;
      FIm.SetZero;
    end;


    function    TSplitComplex.Re: PSignals;
    begin
      Result := FRe.FData;
    end;


    function    TSplitComplex.Im: PSignals;
    begin
      Result := FIm.FData;
    end;


{ ========
  TFFTConvolver = class
  private
    FBlockSize       : Integer;
    FSegSize         : Integer;
    FSegCount        : Integer;
    FComplexSize     : Integer;
    FSegments        : array of TSplitComplex;
    FSegmentsIR      : array of TSplitComplex;
    FfftBuffer       : TSampleBuffer;
    Ffft             : TAudioFFT;
    FpreMultiplied   : TSplitComplex;
    FOverlap         : TSampleBuffer;
    FCurrent         : Integer;
    FInputBuffer     : TSampleBuffer;
    FInputBufferFill : Integer;
  public
}


    constructor TFFTConvolver.Create;
    begin
      inherited;
      FfftBuffer       := TSampleBuffer.Create;
      Ffft             := TAudioFFT.Create;
      FpreMultiplied   := TSplitComplex.Create;
      FOverlap         := TSampleBuffer.Create;
      FInputBuffer     := TSampleBuffer.Create;
    end;


    destructor  TFFTConvolver.Destroy; // override;
    var
      i : Integer;
    begin
      FreeAndNil( FInputBuffer);
      FreeAndNil( FOverlap);
      FreeAndNil( FpreMultiplied);
      FreeAndNil( Ffft);
      FreeAndNil( FfftBuffer);

      for i := 0 to Length( FSegments) - 1
      do FreeAndNil( FSegments[ i]);

      for i := 0 to Length( FSegmentsIR) - 1
      do FreeAndNil( FSegmentsIR[ i]);

      inherited;
    end;


    function    TFFTConvolver.Init( aBlockSize: Integer; anImpulseResponse: TSignalArray): Boolean;
    var
      i          : Integer;
      aSegment   : TSplitComplex;
      aRemaining : Integer;
      aCopySize  : Integer;
      anIrLen    : Integer;
    begin
      Reset;

      if aBlockSize <= 0
      then Exit( False);

      anIrLen := Length( anImpulseResponse);

      while ( anIrLen > 0) and ( Abs( anImpulseResponse[ anIrLen - 1]) < 1e-5 )
      do Dec( anIrLen);

      if anIrLen = 0
      then Exit( True);

      FBlockSize   := NextPowerOfTwo( aBlockSize);
      FSegSize     := 2 * FBlockSize;
      FSegCount    := Ceil( anIrLen / FBlockSize);
      FComplexSize := ComplexSize( FSegSize);
      Ffft.Init( FSegSize);
      FfftBuffer.Size := FSegSize;
      SetLength( FSegments  , FSegCount);
      SetLength( FSegmentsIR, FSegCount);

      for i := 0 to FSegCount - 1
      do begin
        FSegments[ i] := TSplitComplex.Create( FComplexSize);
        aSegment   := TSplitComplex.Create( FComplexSize);
        aRemaining := anIrLen - i * FBlockSize;
        aCopySize  := Min( FBlockSize, aRemaining);
        CopyAndPad( @ anImpulseResponse[ i * FBlockSize], aCopySize, FfftBuffer);
        Ffft.fft( FfftBuffer.FData, aSegment.FRe.FData, aSegment.FIm.FData);
        FSegmentsIR[ i] := aSegment;
      end;

      FpreMultiplied.Size := FComplexSize;
      FOverlap      .Size := FBlockSize;
      FInputBuffer  .Size := FBlockSize;
      FInputBufferFill    := 0;
      FCurrent            := 0;
      Result              := True;
    end;


    procedure   TFFTConvolver.Process( anInput: TSignal; var anOutput: TSignal);
    var
      i : Integer;
    begin
      if FSegCount = 0
      then anOutput := 0
      else begin
        FInputBuffer.FData^[ FInputBufferFill] := anInput;
        CopyAndPad( FInputBuffer.FData, FBlockSize, FfftBuffer);
        Ffft.fft( FfftBuffer.FData, FSegments[ FCurrent].Re, FSegments[ FCurrent].Im);

        if FInputBufferFill = 0
        then begin
          FpreMultiplied.SetZero;

          for i := 1 to FSegCount - 1
          do ComplexMultiplyAccumulate( FpreMultiplied, FSegmentsIR[ i], FSegments[ ( FCurrent + i) mod FSegCount]);

          ComplexMultiplyAccumulate( FpreMultiplied, FSegments[ FCurrent], FSegmentsIR[ 0]);
        end;

        Ffft.ifft( FfftBuffer.FData, FpreMultiplied.Re, FpreMultiplied.Im);
        anOutput := FfftBuffer.DataElt[ FInputBufferFill] + FOverlap.DataElt[ FInputBufferFill];
        Inc( FInputBufferFill);

        if FInputBufferFill = FBlockSize
        then begin
          FInputBufferFill := 0;
          Move( FfftBuffer.FData^[ FBlockSize], FOverlap.FData^, FBlockSize * SizeOf( TSignal));

          if FCurrent > 0
          then Dec( FCurrent)
          else FCurrent := FSegCount - 1;
        end;
      end;
    end;


    procedure   TFFTConvolver.Reset;
    var
      i : Integer;
    begin
      for i := 0 to Length( FSegments) - 1
      do FSegments[ i].DisposeOf;

      for i := 0 to Length( FSegmentsIR) - 1
      do FSegmentsIR[ i].DisposeOf;

      SetLength( FSegments  , 0);
      SetLength( FSegmentsIR, 0);
      FBlockSize       := 0;
      FsegSize         := 0;
      FSegCount        := 0;
      FComplexSize     := 0;
      Fcurrent         := 0;
      FinputBufferFill := 0;
      FfftBuffer    .Clear;
      FpreMultiplied.Clear;
      Foverlap      .Clear;
      FinputBuffer  .Clear;
      Ffft.Init( 0);
    end;


end.

