unit FormantControl;

interface

uses

  System.SysUtils, System.Math, System.Classes, System.StrUtils,

  KnobsUtils, KnobsConversions;


type

  Tstring40 = string[ 80];           // Use a short sting to avoid dynamic (re)allocation.

  TFormant = record                  // Represents a single formant peak
    Freg      : TSignal;             // formant frequency               , as read from file, in Hz
    DbLevel   : TSignal;             // formant level                   , as read from file, in dB
    BandWidth : TSignal;             // formant bandwidth               , as read from file, in Hz
    Q         : TSignal;             // Q factor calculated from Banswidth
    LinLevel  : TSignal;             // Linear volume calculated from DbLevel
  end;
  TFormants = array of TFormant;


  TVowel = record                    // Represents a vowel, consisting of a set of formant peaks and a description
    Name1    : TString40;            // First  descriptive text          , as read from file
    Name2    : TString40;            // Second descriptive text - unicode, as read from file
    Formants : TFormants;
  end;
  TVowels = array of TVowel;


  TVowelSet = record                 // Represents a vowel category, a named set of vowels
    Description : string;            // Set description as read rom file, a dynamic string is OK here, not used real time.
    Vowels      : TVowels;
  end;
  TVowelSets = array of TVowelSet;


  TFormantCollection = class         // Holds all vowel sets present in the system
  private
    FCurrentVowel : Integer;
    FVowelSets    : TVowelSets;
    FLastToken    : string;
    FOnLog        : TKnobsOnLog;
  private
    function    NextVowel                                        : string;
    function    GetSetCount                                      : Integer;
    function    GetVowelSet    ( aSet                  : Integer): TVowelSet;
    function    GetVowelCount  ( aSet                  : Integer): Integer;
    function    GetFormantCount( aSet, aVowel          : Integer): Integer;
    function    GetVowel       ( aSet, aVowel          : Integer): TVowel;
 // function    GetName1       ( aSet, aVowel          : Integer): TString40;             // ANSI issues with vowel names, but they are not being used anyway
 // function    GetName2       ( aSet, aVowel          : Integer): TString40;             // ANSI issues with vowel names, but they are not being used anyway
    function    GetFormant     ( aSet, aVowel, aFormant: Integer): TFormant;
    function    GetFrequency   ( aSet, aVowel, aFormant: Integer): TSignal;
    function    GetdBLevel     ( aSet, aVowel, aFormant: Integer): TSignal;
    function    GetQ           ( aSet, aVowel, aFormant: Integer): TSignal;
    function    GetLinLevel    ( aSet, aVowel, aFormant: Integer): TSignal;
  private
    procedure   CalculateSignals;
    procedure   StripComments( const aList: TStringList);
    function    GetErrorLocation( const aString: string; const anOffset: Integer): string;
    function    CheckOffset     ( const aString: string; const anOffset: integer                            ): Boolean;
    function    CheckChar       ( const aString: string; var   anOffset: Integer; const aChar: Char         ): Boolean;
    function    LookAhead       ( const aString: string; var   anOffset: Integer; const aChar: Char         ): Boolean;
    procedure   SkipWhiteSpace  ( const aString: string; var   anOffset: Integer);
    function    ParseLParen     ( const aString: string; var   anOffset: Integer                            ): Boolean;
    function    ParseRParen     ( const aString: string; var   anOffset: Integer                            ): Boolean;
    function    ParseComma      ( const aString: string; var   anOffset: Integer                            ): Boolean;
    function    ParseString     ( const aString: string; var   anOffset: Integer                            ): Boolean;
    function    ParseSetName    ( const aString: string; var   anOffset: Integer; var aSetName   : string   ): Boolean;
    function    ParseVowelName  ( const aString: string; var   anOffset: Integer; var aVowelName : string   ): Boolean;
    function    ParseInteger    ( const aString: string; var   anOffset: Integer; var aValue     : Integer  ): Boolean;
    function    ParseFormant    ( const aString: string; var   anOffset: Integer; var aFormant   : TFormant ): Boolean;
    function    ParseFormants   ( const aString: string; var   anOffset: Integer; var aFormants  : TFormants): Boolean;
    function    ParseVowel      ( const aString: string; var   anOffset: Integer; var aVowel     : TVowel   ): Boolean;
    function    ParseVowels     ( const aString: string; var   anOffset: Integer; var aVowels    : TVowels  ): Boolean;
    function    ParseVowelSet   ( const aString: string; var   anOffset: Integer; var aVowelSet  : TVowelSet): Boolean;
    function    ParseText       ( const aString: string): Boolean;
    function    Parse( const aList    : TStringList): Boolean;                                                 overload;
    function    Parse( const aFileName: string): Boolean;                                                      overload;
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   Error( const aMsg, aString: string; const anOffset: Integer);                                             overload;
    procedure   Error( const aFmt: string; const anArgs: array of const; const aString: string; const anOffset: Integer); overload;
  public
    constructor Create( const aFileName: string; const aLogger: TKnobsOnLog = nil);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Interpolate( aSet: Integer; anIndex: TSignal; var aVowel: TVowel);
    procedure   CollectSetNames( const aStringList: TStringList);
    procedure   Dump( const aStringList: TStringList);                                                         overload;
    procedure   Dump( const aFileName: string);                                                                overload;
   public
    property    SetCount                                       : Integer   read GetSetCount;
    property    VowelSet    [ aSet                  : Integer] : TVowelSet read GeTVowelSet;
    property    VowelCount  [ aSet                  : Integer] : Integer   read GetVowelCount;
    property    FormantCount[ aSet, aVowel          : Integer] : Integer   read GetFormantCount;
    property    Vowel       [ aSet, aVowel          : Integer] : TVowel    read GetVowel;
//  property    Name1       [ aSet, aVowel          : Integer] : TString40 read GetName1; // ANSI issues with vowel names, but they are not being used anyway
//  property    Name2       [ aSet, aVowel          : Integer] : TString40 read GetName2; // ANSI issues with vowel names, but they are not being used anyway
    property    Formant     [ aSet, aVowel, aFormant: Integer] : TFormant  read GetFormant;
    property    Frequency   [ aSet, aVowel, aFormant: Integer] : TSignal   read GetFrequency;
    property    dBLevel     [ aSet, aVowel, aFormant: Integer] : TSignal   read GetdBLevel;
    property    Q           [ aSet, aVowel, aFormant: Integer] : TSignal   read GetQ;
    property    LinLevel    [ aSet, aVowel, aFormant: Integer] : TSignal   read GetLinLevel;
  end;


  TFormantSet = record
    Description : string;
    Mode        : string;
    Formants    : TFormants;
  end;
  TFormantSets = array of TFormantSet;


  TModalCollection = class
  private
    FFormants  : TFormantSets;
    FLastToken : string;
    FOnLog     : TKnobsOnLog;
  private
    function    GetSetCount                              : Integer;
    function    GetFormantSet  ( aSet          : Integer): TFormantSet;
    function    GetFormantCount( aSet          : Integer): Integer;
    function    GetName        ( aSet          : Integer): string;
    function    GetMode        ( aSet          : Integer): string;
    function    GetFormant     ( aSet, aFormant: Integer): TFormant;
    function    GetFrequency   ( aSet, aFormant: Integer): TSignal;
    function    GetdBLevel     ( aSet, aFormant: Integer): TSignal;
    function    GetQ           ( aSet, aFormant: Integer): TSignal;
    function    GetLinLevel    ( aSet, aFormant: Integer): TSignal;
  private
    procedure   CalculateSignals;
    procedure   StripComments( const aList: TStringList);
    function    GetErrorLocation( const aString: string; const anOffset: Integer): string;
    function    CheckOffset     ( const aString: string; const anOffset: integer                              ): Boolean;
    function    CheckChar       ( const aString: string; var   anOffset: Integer; const aChar: Char           ): Boolean;
    function    LookAhead       ( const aString: string; var   anOffset: Integer; const aChar: Char           ): Boolean;
    procedure   SkipWhiteSpace  ( const aString: string; var   anOffset: Integer);
    function    ParseComma      ( const aString: string; var   anOffset: Integer                              ): Boolean;
    function    ParseSemi       ( const aString: string; var   anOffset: Integer                              ): Boolean;
    function    ParseString     ( const aString: string; var   anOffset: Integer                              ): Boolean;
    function    ParseSetName    ( const aString: string; var   anOffset: Integer; var aSetName   : string     ): Boolean;
    function    ParseMode       ( const aString: string; var   anOffset: Integer; var aMode      : string     ): Boolean;
    function    ParseFloat      ( const aString: string; var   anOffset: Integer; var aValue     : TSignal    ): Boolean;
    function    ParseFormant    ( const aString: string; var   anOffset: Integer; const aMode: string; var aFormant : TFormant ): Boolean;
    function    ParseFormants   ( const aString: string; var   anOffset: Integer; const aMode: string; var aFormants: TFormants): Boolean;
    function    ParseFormantSet ( const aString: string; var   anOffset: Integer; var aFormantSet: TFormantSet): Boolean;
    function    ParseText       ( const aString: string): Boolean;
    function    Parse           ( const aList    : TStringList): Boolean;                                      overload;
    function    Parse           ( const aFileName: string): Boolean;                                           overload;
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   Error( const aMsg, aString: string; const anOffset: Integer);                                             overload;
    procedure   Error( const aFmt: string; const anArgs: array of const; const aString: string; const anOffset: Integer); overload;
  public
    constructor Create( const aFileName: string; const aLogger: TKnobsOnLog = nil);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   CollectSetNames( const aStringList: TStringList);
    procedure   Dump( const aStringList: TStringList);                                                         overload;
    procedure   Dump( const aFileName: string);                                                                overload;
  public
    property    SetCount                               : Integer     read GetSetCount;
    property    FormantSet  [ aSet          : Integer] : TFormantSet read GetFormantSet;
    property    FormantCount[ aSet          : Integer] : Integer     read GetFormantCount;
    property    Name        [ aSet          : Integer] : string      read GetName;
    property    Mode        [ aSet          : Integer] : string      read GetMode;
    property    Formant     [ aSet, aFormant: Integer] : TFormant    read GetFormant;
    property    Frequency   [ aSet, aFormant: Integer] : TSignal     read GetFrequency;
    property    dBLevel     [ aSet, aFormant: Integer] : TSignal     read GetdBLevel;
    property    Q           [ aSet, aFormant: Integer] : TSignal     read GetQ;
    property    LinLevel    [ aSet, aFormant: Integer] : TSignal     read GetLinLevel;
  end;


var

  FormantCollection : TFormantCollection = nil;
  ModalCollection   : TModalCollection   = nil;

  procedure ReadFormantCollection( const aFileName: string; const aLogger: TKnobsOnLog);
  function  CreateFormantNames: TStringList;
  procedure ReadModalCollection( const aFileName: string; const aLogger: TKnobsOnLog);
  function  CreateModalNames: TStringList;


implementation


  procedure ReadFormantCollection( const aFileName: string; const aLogger: TKnobsOnLog);
  begin
    if Assigned( FormantCollection)
    then FreeAndNil( FormantCollection);

    FormantCollection := TFormantCollection.Create( aFileName, aLogger);
  end;


  function  CreateFormantNames: TStringList;
  begin
    Result := TStringList.Create;

    if Assigned( FormantCollection)
    then FormantCollection.CollectSetNames( Result);
  end;


  procedure ReadModalCollection( const aFileName: string; const aLogger: TKnobsOnLog);
  begin
    if Assigned( ModalCollection)
    then FreeAndNil( ModalCollection);

    ModalCollection := TModalCollection.Create( aFileName, aLogger);
  end;


  function  CreateModalNames: TStringList;
  begin
    Result := TStringList.Create;

    if Assigned( ModalCollection)
    then ModalCollection.CollectSetNames( Result);
  end;


const

  WhiteSpace   = [ #0 .. #32];
  Digits       = [ '0' .. '9', '-', '+'];
  HexDigits    = Digits + [ 'a' .. 'f', 'A' .. 'F'];
  AlphaNumeric = [ #33 .. #255] - [ '(', ')', ',', '[', ']'];


{ ========
  TFormantCollection = class
  private
    FCurrentVowel : Integer;
    FVowelSets    : TVowelSets;
    FLastToken    : string;
    FOnLog        : TKnobsOnLog;
   public
    property    SetCount                                       : Integer   read GetSetCount;
    property    VowelSet    [ aSet                  : Integer] : TVowelSet read GeTVowelSet;
    property    VowelCount  [ aSet                  : Integer] : Integer   read GetVowelCount;
    property    FormantCount[ aSet, aVowel          : Integer] : Integer   read GetFormantCount;
    property    Vowel       [ aSet, aVowel          : Integer] : TVowel    read GetVowel;
    property    Name1       [ aSet, aVowel          : Integer] : string    read GetName1;
    property    Name2       [ aSet, aVowel          : Integer] : string    read GetName2;
    property    Formant     [ aSet, aVowel, aFormant: Integer] : TFormant  read GetFormant;
    property    Frequency   [ aSet, aVowel, aFormant: Integer] : TSignal   read GetFrequency;
    property    dBLevel     [ aSet, aVowel, aFormant: Integer] : TSignal   read GetdBLevel;
    property    Q           [ aSet, aVowel, aFormant: Integer] : TSignal   read GetQ;
    property    LinLevel    [ aSet, aVowel, aFormant: Integer] : TSignal   read GetLinLevel;
  private
}

    function    TFormantCollection.NextVowel: string;
    begin
      Inc( FCurrentVowel);
      Result := Format( '%d', [ FCurrentVowel], AppLocale);
    end;


    function    TFormantCollection.GetSetCount: Integer;
    begin
      Result := Length( FVowelSets);
    end;


    function    TFormantCollection.GetVowelSet( aSet: Integer): TVowelSet;
    begin
      Result := FVowelSets[ aSet];
    end;


    function    TFormantCollection.GetVowelCount( aSet: Integer): Integer;
    begin
      Result := Length( VowelSet[ aSet].Vowels);
    end;


    function    TFormantCollection.GetFormantCount( aSet, aVowel : Integer): Integer;
    begin
      Result := Length( VowelSet[ aSet].Vowels[ aVowel].Formants);
    end;


    function    TFormantCollection.GetVowel( aSet, aVowel : Integer): TVowel;
    begin
      Result := VowelSet[ aSet].Vowels[ aVowel];
    end;


 // function    TFormantCollection.GetName1( aSet, aVowel : Integer): TString40; // ANSI issues with vowel names, but they are not being used anyway
 // begin
 //   Result := Vowel[ aSet, aVowel].Name1;
 // end;


 // function    TFormantCollection.GetName2( aSet, aVowel : Integer): TString40; // ANSI issues with vowel names, but they are not being used anyway
 // begin
 //   Result := Vowel[ aSet, aVowel].Name2;
 // end;


    function    TFormantCollection.GetFormant( aSet, aVowel, aFormant: Integer): TFormant;
    begin
      Result := Vowel[ aSet, aVowel].Formants[ aFormant];
    end;


    function    TFormantCollection.GetFrequency( aSet, aVowel, aFormant: Integer): TSignal;
    begin
      Result := Formant[ aSet, aVowel, aFormant].Freg;
    end;


    function    TFormantCollection.GetDbLevel( aSet, aVowel, aFormant: Integer): TSignal;
    begin
      Result := Formant[ aSet, aVowel, aFormant].dBLevel;
    end;


    function    TFormantCollection.GetQ( aSet, aVowel, aFormant: Integer): TSignal;
    begin
      Result := Formant[ aSet, aVowel, aFormant].Q;
    end;


    function    TFormantCollection.GetLinLevel( aSet, aVowel, aFormant: Integer): TSignal;
    begin
      Result := Formant[ aSet, aVowel, aFormant].LinLevel;
    end;


//  private

    procedure   TFormantCollection.CalculateSignals;
    var
      aSet     : Integer;
      aVowel   : Integer;
      aFormant : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do begin
        for aVowel := 0 to VowelCount[ aSet] - 1
        do begin
          for aFormant := 0 to FormantCount[ aSet, aVowel] - 1
          do begin
            FVowelSets[ aSet].Vowels[ aVowel].Formants[ aFormant].LinLevel :=
              dBToUnits( Formant[ aSet, aVowel, aFormant].DbLevel);

            if Formant[ aSet, aVowel, aFormant].BandWidth > 0
            then begin
              // Calcultate Q frome specified bandwidth
              FVowelSets[ aSet].Vowels[ aVowel].Formants[ aFormant].Q :=
                Formant[ aSet, aVowel, aFormant].Freg /
                Formant[ aSet, aVowel, aFormant].BandWidth;
            end
            else begin
              // Some arbitrary Q when bandwidth was not specified
              FVowelSets[ aSet].Vowels[ aVowel].Formants[ aFormant].Q := 10;
            end;
          end;
        end;
      end;
    end;


    procedure   TFormantCollection.StripComments( const aList: TStringList);
    var
      i      : Integer;
      aParts : TStringList;
    begin
      if Assigned( aList)
      then begin
        i := 0;

        while i < aList.Count
        do begin
          aParts := Explode( aList[ i], '//');

          try
            if ( aParts.Count = 1) and ( Trim( aParts[ 0]) = '')
            then aList[ i] := ''
            else if aParts.Count > 1
            then aList[ i] := aParts[ 0];
          finally
            FreeAndNil( aParts);
          end;

          aList[ i] := Trim( aList[ i]);

          if aList[ i] = ''
          then aList.Delete( i)
          else Inc( i);
        end;
      end;
    end;


    function    TFormantCollection.GetErrorLocation( const aString: string; const anOffset: Integer): string;
    var
      aBegin  : Integer;
      aLength : Integer;
    begin
      aBegin  := anOffset - 40;
      aLength := 80;

      if aBegin < Low( aString)
      then aBegin := Low( aString);

      if aBegin + aLength > High( aString)
      then aLength := High( aString) - aBegin;

      Result := Copy( aString, aBegin, aLength);
    end;


    function    TFormantCollection.CheckOffset( const aString: string; const anOffset: integer): Boolean;
    begin
      Result := ( anOffset >= Low( aString)) and  ( anOffset <= High( aString));
    end;


    function    TFormantCollection.CheckChar( const aString: string; var anOffset: Integer; const aChar: Char): Boolean;
    begin
      Result := LookAhead( aString, anOffset, aChar);

      if Result
      then Inc( anOffset);
    end;


    function    TFormantCollection.LookAhead( const aString: string; var anOffset: Integer; const aChar: Char): Boolean;
    begin
      SkipWhiteSpace( aString, anOffset);
      Result := CheckOffset( aString, anOffset) and ( aString[ anOffset] = aChar);
    end;


    procedure   TFormantCollection.SkipWhiteSpace( const aString: string; var anOffset: Integer);
    begin
      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], WhiteSpace)
      do Inc( anOffset);
    end;


    function    TFormantCollection.ParseLParen( const aString: string; var anOffset: Integer): Boolean;
    begin
      Result := CheckChar( aString, anOffset, '(');
    end;


    function    TFormantCollection.ParseRParen( const aString: string; var anOffset: Integer): Boolean;
    begin
      Result := CheckChar( aString, anOffset, ')');
    end;


    function    TFormantCollection.ParseComma( const aString: string; var anOffset: Integer): Boolean;
    begin
      Result := CheckChar( aString, anOffset, ',');
    end;


    function    TFormantCollection.ParseString( const aString: string; var anOffset: Integer): Boolean;
    begin
      FLastToken := '';

      SkipWhiteSpace( aString, anOffset);

      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], AlphaNumeric)
      do begin
        FLastToken := FLastToken + aString[ anOffset];
        Inc( anOffset);
      end;

      Result := FLastToken <> '';
    end;


    function    TFormantCollection.ParseSetName( const aString: string; var anOffset: Integer; var aSetName: string): Boolean;
    begin
      Result := ParseString( aString, anOffset);

      if Result
      then aSetName := AnsiReplaceText( FLastToken, '_', ' ')
      else aSetName := '';
    end;


    function    TFormantCollection.ParseVowelName( const aString: string; var anOffset: Integer; var aVowelName : string): Boolean;
    var
      aValue : Integer;
    begin
      Result := ParseString( aString, anOffset);

      if Result
      then begin
        aVowelName := FLastToken;

        if Pos( '\u', LowerCase( aVowelName)) = 1  // Check for Unicode code point
        then begin
          aValue := StrToIntDef( '$' + Copy( aVowelName, Low( aVowelName) + 3, Length( aVowelName)), -1);

          if aValue >= 0
          then aVowelName := Char( aValue);
        end;
      end
      else begin
        aVowelName := NextVowel;                   // If vowel could not be parsed ... "make something up"
        Result     := True;
      end;
    end;


    function    TFormantCollection.ParseInteger( const aString: string; var anOffset: Integer; var aValue: Integer): Boolean;
    begin
      Result     := False;
      aValue     := 0;
      FLastToken := '';
      SkipWhiteSpace( aString, anOffset);

      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], Digits)
      do begin
        FLastToken := FLastToken + aString[ anOffset];
        Inc( anOffset);
      end;

      if FLastToken <> ''
      then begin
        aValue := StrToIntDef( FLastToken, - MaxInt);

        if aValue = - MaxInt
        then begin
          aValue := 0;
          Error( 'Expected integer value, saw %s in ParseInteger', [ FLastToken], aString, anOffset);
        end
        else Result := True;
      end;
    end;


    function    TFormantCollection.ParseFormant( const aString: string; var anOffset: Integer; var aFormant: TFormant): Boolean;
    var
      aSavedOffset : Integer;
      aFrequency   : Integer;
      aLevel       : Integer;
      aBandwidth   : Integer;
    begin
      Result       := False;
      aSavedOffset := anOffset;

      if ParseInteger( aString, anOffset, aFrequency)
      then begin
        if ParseInteger( aString, anOffset, aLevel)
        then begin
          aSavedOffset := anOffset;

          if not ParseInteger( aString, anOffset, aBandwidth)
          then begin
            aBandwidth := 0;
            anOffset   := aSavedOffset;
          end;

          aFormant.Freg      := aFrequency;
          aFormant.dBLevel   := aLevel;
          aFormant.BandWidth := aBandwidth;
          Result             := True;
        end
        else Error( 'Expected level, in ParseFormant', aString, anOffset)
      end
      else begin
        anOffset := aSavedOffset;

        if not LookAhead( aString, anOffset, ')')
        then Error( 'Expected frequency or '')'', in ParseFormant', aString, anOffset);
      end;
    end;


    function    TFormantCollection.ParseFormants( const aString: string; var anOffset: Integer; var aFormants: TFormants): Boolean;
    var
      aFormant     : TFormant;
      aSavedOffset : Integer;
    begin
      Result := False;

      while CheckOffset( aString, anOffset) and ParseFormant( aString, anOffset, aFormant)
      do begin
        SetLength( aFormants, Length( aFormants) + 1);
        aFormants[ Length( aFormants) - 1] := aFormant;
        aSavedOffset := anOffset;

        if not ParseComma( aString, anOffset)
        then begin
          anOffset := aSavedOffset;
          Result   := True;
          Break;
        end;
      end;

      if not Result and not CheckOffset( aString, anOffset)
      then Error( 'Expected Formant-set, in ParseFormants', aString, anOffset);
    end;


    function    TFormantCollection.ParseVowel( const aString: string; var anOffset: Integer; var aVowel: TVowel): Boolean;
    var
      aVowelName1 : string;
      aVowelName2 : string;
      aFormants   : TFormants;
    begin
      Result := False;

      if ParseLParen( aString, anOffset)
      then begin
        if ParseVowelName( aString, anOffset, aVowelName1)
        then begin
          if ParseVowelName( aString, anOffset, aVowelName2)
          then begin
            if ParseComma( aString, anOffset)
            then begin
              SetLength( aFormants, 0);

              if ParseFormants( aString, anOffset, aFormants)
              then begin
                if ParseRParen( aString, anOffset)
                then begin
                  aVowel.Name1    := TString40( aVowelName1);
                  aVowel.Name2    := TString40( aVowelName2);
                  aVowel.Formants := aFormants;
                  Result := True;
                end
                else Error( 'Expected '')'', in ParseVowel', aString, anOffset);
              end
              else Error( 'Expected Formant-set, in ParseVowel', aString, anOffset);
            end
            else Error( 'Expected '','', in ParseVowel', aString, anOffset)
          end
          else Error( 'Expected VowelName2, in ParseVowel', aString, anOffset);
        end
        else Error( 'Expected VowelName1, in ParseVowel', aString, anOffset);
      end
      else begin
        if CheckOffset( aString, anOffset) and not ParseRParen( aString, anOffset)
        then Error( 'Expected new vowel starting with ''('' or end of vowel set indicated by '')'', in ParseVowel', aString, anOffset);
      end;
    end;


    function    TFormantCollection.ParseVowels( const aString: string; var anOffset: Integer; var aVowels: TVowels): Boolean;
    var
      aVowel : TVowel;
    begin
      Result := False;

      while CheckOffset( aString, anOffset) and ParseVowel( aString, anOffset, aVowel)
      do begin
        SetLength( aVowels, Length( aVowels) + 1);
        aVowels[ Length( aVowels) - 1] := aVowel;

        if LookAhead( aString, anOffset, ')')
        then begin
          Result := True;
          Break;
        end;
      end;

      if not Result
      then Error( 'Expected Vowel-set, in ParseVowels', aString, anOffset);
    end;


    function    TFormantCollection.ParseVowelSet( const aString: string; var anOffset: Integer; var aVowelSet: TVowelSet): Boolean;
    var
      aVowels : TVowels;
    begin
      Result        := False;
      FCurrentVowel := 0;

      if ParseSetName( aString, anOffset, aVowelSet.Description)
      then begin
        if ParseLParen( aString, anOffset)
        then begin
          SetLength( aVowels, 0);

          if ParseVowels( aString, anOffset, aVowels)
          then begin
            aVowelSet.Vowels := aVowels;

            if ParseRParen( aString, anOffset)
            then Result := True
            else Error( 'Expected '')'', in ParseVowelSet', aString, anOffset);
          end
          else Error( 'Expected vowels, in ParseVowelSet', aString, anOffset);
        end
        else begin
          if CheckOffset( aString, anOffset) and not LookAhead( aString, anOffset, ')')
          then Error( 'Expected new vowel starting with ''('' or end of vowel set indicated by '')'', in ParseVowelSet', aString, anOffset)
          else Result := True;
        end;
      end
      else begin
        if CheckOffset( aString, anOffset)                            // there is more input ... so must be an error
        then Error( 'Expected a set name, read "%s", in ParseVowelSet', [ FLastToken], aString, anOffset)
        else Result := True;                                          // No more input, we are done then
      end;
    end;


    function    TFormantCollection.ParseText( const aString: string): Boolean;
    var
      anOffset  : Integer;
      aVowelSet : TVowelSet;
    begin
      Result   := False;
      anOffset := Low( aString);

      while CheckOffset( aString, anOffset)
      do begin
        if ParseVowelSet( aString, anOffset, aVowelSet)
        then begin
          SetLength( FVowelSets, SetCount + 1);
          FVowelSets[ SetCount - 1] := aVowelSet;
        end
        else begin
          if CheckOffset( aString, anOffset)
          then Error( 'Excpected VowelSet, in ParseText', aString, anOffset)
          else Result := True;

          Break;
        end;
      end;
    end;


    function    TFormantCollection.Parse( const aList: TStringList): Boolean; // overload;
    begin
      Clear;
      StripComments( aList);
      Result := ParseText( Implode( aList, ' '));
    end;


    function    TFormantCollection.Parse( const aFileName: string): Boolean; // overload;
    var
      aStringList: TStringList;
    begin
      Result := False;

      if FileExists( aFileName)
      then begin
        aStringList := TStringList.Create;

        try
          aStringList.LoadFromFile( aFileName);
          Result := Parse( aStringList);

          if Result
          then LogFmt( 'File ''%s'' parsed OK', [ aFileName]);
        finally
          aStringList.DisposeOf;
        end;
      end
      else LogFmt( 'File ''%s'' does not exist', [ aFileName])
    end;


    procedure   TFormantCollection.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( Self, LC_FORMANT, aMsg);
    end;


    procedure   TFormantCollection.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs));
    end;


    procedure   TFormantCollection.Error( const aMsg, aString: string; const anOffset: Integer); // overload;
    begin
      LogFmt( 'Formant spec error: %s, context is "%s"', [ aMsg, GetErrorLocation( aString, anOffset)]);
      Clear;
    end;


    procedure   TFormantCollection.Error( const aFmt: string; const anArgs: array of const; const aString: string; const anOffset: Integer); // overload;
    begin
      Error( Format( aFmt, anArgs), aString, anOffset);
    end;


  // public

    constructor TFormantCollection.Create( const aFileName: string; const aLogger: TKnobsOnLog = nil);
    begin
      inherited Create;
      FOnLog := aLogger;
      Parse( aFileName);
      CalculateSignals;
    end;


    destructor  TFormantCollection.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TFormantCollection.Clear;
    var
      aSet   : Integer;
      aVowel : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do begin
        for aVowel := 0 to VowelCount[ aSet] - 1
        do SetLength( FVowelSets[ aSet].Vowels[ aVowel].Formants, 0);

        SetLength( FVowelSets[ aSet].Vowels, 0)
      end;

      SetLength( FVowelSets, 0);
    end;


    procedure   TFormantCollection.Interpolate( aSet: Integer; anIndex: TSignal; var aVowel: TVowel);
    var
      p           : Integer;
      q           : Integer;
      d           : TSignal;
      i           : Integer;
      aVowelCount : Integer;
    begin
      // FSignals[ p] + d * ( FSignals[ q] - FSignals[ p]);
      if Length( aVowel.Formants) = FormantCount[ aSet, 0]
      then begin
        aVowelCount  := VowelCount[ aSet];
        aVowel.Name1 := '';
        aVowel.Name2 := '';
        aSet         := Clip( aSet   , 0, SetCount - 1      );
        anIndex      := Clip( anIndex, 0, aVowelCount - 1e-3);       // Just under one more than the last vowel index .. treated cyclically
        p            := Trunc( anIndex);                             // so p < VowelCount
        d            := anIndex - p;
        q            := ( p + 1);

        if q > aVowelCount - 1
        then q := 0;

        for i := 0 to Length( aVowel.Formants) - 1
        do begin
          aVowel.Formants[ i].Freg     := Formant[ aSet, p, i].Freg     + d * ( Formant[ aSet, q, i].Freg     - Formant[ aSet, p, i].Freg    );
          aVowel.Formants[ i].Q        := Formant[ aSet, p, i].Q        + d * ( Formant[ aSet, q, i].Q        - Formant[ aSet, p, i].Q       );
          aVowel.Formants[ i].LinLevel := Formant[ aSet, p, i].LinLevel + d * ( Formant[ aSet, q, i].LinLevel - Formant[ aSet, p, i].LinLevel);
        end;
      end
      else Nop;
    end;


    procedure   TFormantCollection.CollectSetNames( const aStringList: TStringList);
    var
      i : Integer;
    begin
      if Assigned( aStringList)
      then begin
        aStringList.Clear;

        for i := 0 to SetCount - 1
        do aStringList.Add( VowelSet[ i].Description);
      end;
    end;


    procedure   TFormantCollection.Dump( const aStringList: TStringList); // overload;
    var
      aSet     : Integer;
      aVowel   : Integer;
      aFormant : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do begin
        aStringList.Add( Format(     'set : ''%s''', [ VowelSet[ aSet].Description]));

        for aVowel := 0 to VowelCount[ aSet] - 1
        do begin
          aStringList.Add( Format(   '  Name1 : ''%s''', [ Vowel[ aSet, aVowel].Name1]));
          aStringList.Add( Format(   '  Name2 : ''%s''', [ Vowel[ aSet, aVowel].Name2]));

          for aFormant := 0 to FormantCount[ aSet, aVowel] - 1
          do begin
            aStringList.Add( Format( '    Freq      : %f', [ Formant[ aSet, aVowel, aFormant].Freg     ]));
            aStringList.Add( Format( '    dBLevel   : %f', [ Formant[ aSet, aVowel, aFormant].dBLevel  ]));
            aStringList.Add( Format( '    BandWidth : %f', [ Formant[ aSet, aVowel, aFormant].BandWidth]));
            aStringList.Add( Format( '    Q         : %f', [ Formant[ aSet, aVowel, aFormant].Q        ]));
            aStringList.Add( Format( '    LinLevel  : %f', [ Formant[ aSet, aVowel, aFormant].LinLevel ]));
          end;
        end;
      end;
    end;


    procedure   TFormantCollection.Dump( const aFileName: string); // overload;
    var
      aStringList: TStringList;
    begin
      aStringList := TStringList.Create;

      try
        Dump( aStringList);
        aStringList.SaveToFile( aFileName);
      finally
        aStringList.DisposeOf;
      end;
    end;


{ ========
  TModalCollection = class
  private
    FFormants  : TFormantSets;
    FLastToken : string;
    FOnLog     : TKnobsOnLog;
  public
    property    SetCount                               : Integer     read GetSetCount;
    property    FormantSet  [ aSet          : Integer] : TFormantSet read GetFormantSet;
    property    FormantCount[ aSet          : Integer] : Integer     read GetFormantCount;
    property    Name        [ aSet          : Integer] : string      read GetName;
    property    Formant     [ aSet, aFormant: Integer] : TFormant    read GetFormant;
    property    Frequency   [ aSet, aFormant: Integer] : TSignal     read GetFrequency;
    property    dBLevel     [ aSet, aFormant: Integer] : TSignal     read GetdBLevel;
    property    Q           [ aSet, aFormant: Integer] : TSignal     read GetQ;
    property    LinLevel    [ aSet, aFormant: Integer] : TSignal     read GetLinLevel;
  private
}

    function    TModalCollection.GetSetCount: Integer;
    begin
      Result := Length( FFormants);
    end;


    function    TModalCollection.GetFormantSet( aSet: Integer): TFormantSet;
    begin
      Result := FFormants[ aSet];
    end;


    function    TModalCollection.GetFormantCount( aSet: Integer): Integer;
    begin
      Result := Length( FFormants[ aSet].Formants);
    end;


    function    TModalCollection.GetName( aSet: Integer): string;
    begin
      Result := FFormants[ aSet].Description;
    end;


    function    TModalCollection.GetMode( aSet: Integer): string;
    begin
      Result := FFormants[ aSet].Mode;
    end;


    function    TModalCollection.GetFormant( aSet, aFormant: Integer): TFormant;
    begin
      Result := FFormants[ aSet].Formants[ aFormant];
    end;


    function    TModalCollection.GetFrequency( aSet, aFormant: Integer): TSignal;
    begin
      Result := FFormants[ aSet].Formants[ aFormant].Freg;
    end;


    function    TModalCollection.GetdBLevel( aSet, aFormant: Integer): TSignal;
    begin
      Result := FFormants[ aSet].Formants[ aFormant].DbLevel;
    end;


    function    TModalCollection.GetQ( aSet, aFormant: Integer): TSignal;
    begin
      Result := FFormants[ aSet].Formants[ aFormant].Q;
    end;


    function    TModalCollection.GetLinLevel( aSet, aFormant: Integer): TSignal;
    begin
      Result := FFormants[ aSet].Formants[ aFormant].LinLevel;
    end;


//  private

    procedure   TModalCollection.CalculateSignals;
    var
      aSet     : Integer;
      aFormant : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do begin
        for aFormant := 0 to FormantCount[ aSet] - 1
        do begin
          FFormants[ aSet].Formants[ aFormant].LinLevel := dBToUnits( Formant[ aSet, aFormant].DbLevel);

          if Formant[ aSet, aFormant].BandWidth > 0
          then FFormants[ aSet].Formants[ aFormant].Q := Formant[ aSet, aFormant].Freg / Formant[ aSet, aFormant].BandWidth // Calcultate Q frome specified bandwidth
          else FFormants[ aSet].Formants[ aFormant].Q := 10;                                                                // Some arbitrary Q when bandwidth was not specified
        end;
      end;
    end;


    procedure   TModalCollection.StripComments( const aList: TStringList);
    var
      i      : Integer;
      aParts : TStringList;
    begin
      if Assigned( aList)
      then begin
        i := 0;

        while i < aList.Count
        do begin
          aParts := Explode( aList[ i], '//');

          try
            if ( aParts.Count = 1) and ( Trim( aParts[ 0]) = '')
            then aList[ i] := ''
            else if aParts.Count > 1
            then aList[ i] := aParts[ 0];
          finally
            FreeAndNil( aParts);
          end;

          aList[ i] := Trim( aList[ i]);

          if aList[ i] = ''
          then aList.Delete( i)
          else Inc( i);
        end;
      end;
    end;


    function    TModalCollection.GetErrorLocation( const aString: string; const anOffset: Integer): string;
    var
      aBegin  : Integer;
      aLength : Integer;
    begin
      aBegin  := anOffset - 40;
      aLength := 80;

      if aBegin < Low( aString)
      then aBegin := Low( aString);

      if aBegin + aLength > High( aString)
      then aLength := High( aString) - aBegin;

      Result := Copy( aString, aBegin, aLength);
    end;


    function    TModalCollection.CheckOffset( const aString: string; const anOffset: integer): Boolean;
    begin
      Result := ( anOffset >= Low( aString)) and  ( anOffset <= High( aString));
    end;


    function    TModalCollection.CheckChar( const aString: string; var anOffset: Integer; const aChar: Char): Boolean;
    begin
      Result := LookAhead( aString, anOffset, aChar);

      if Result
      then Inc( anOffset);
    end;


    function    TModalCollection.LookAhead( const aString: string; var anOffset: Integer; const aChar: Char): Boolean;
    begin
      SkipWhiteSpace( aString, anOffset);
      Result := CheckOffset( aString, anOffset) and ( aString[ anOffset] = aChar);
    end;


    procedure   TModalCollection.SkipWhiteSpace( const aString: string; var anOffset: Integer);
    begin
      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], WhiteSpace)
      do Inc( anOffset);
    end;


    function    TModalCollection.ParseComma( const aString: string; var anOffset: Integer): Boolean;
    begin
      Result := CheckChar( aString, anOffset, ',');
    end;


    function    TModalCollection.ParseSemi( const aString: string; var anOffset: Integer): Boolean;
    begin
      Result := CheckChar( aString, anOffset, ';');
    end;


    function    TModalCollection.ParseString( const aString: string; var anOffset: Integer): Boolean;
    begin
      FLastToken := '';

      SkipWhiteSpace( aString, anOffset);

      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], AlphaNumeric + [ ' ', '[', ']'])
      do begin
        FLastToken := FLastToken + aString[ anOffset];
        Inc( anOffset);
      end;

      FLastToken := Trim( FLastToken);
      Result     := FLastToken <> '';
    end;


    function    TModalCollection.ParseSetName( const aString: string; var anOffset: Integer; var aSetName: string): Boolean;
    begin
      Result := ParseString( aString, anOffset);

      if Result
      then aSetName := FLastToken
      else aSetName := '';
    end;


    function    TModalCollection.ParseMode( const aString: string; var anOffset: Integer; var aMode: string): Boolean;
    begin
      Result := ParseString( aString, anOffset);

      if   Result
      and  ( SameText( FLastToken, 'ratio') or SameText( FLastToken, 'hz'))
      then aMode := FLastToken
      else aMode := '';
    end;


    function    TModalCollection.ParseFloat( const aString: string; var   anOffset: Integer; var aValue: TSignal  ): Boolean;
    begin
      Result     := False;
      aValue     := 0.0;
      FLastToken := '';
      SkipWhiteSpace( aString, anOffset);

      while CheckOffset( aString, anOffset) and CharInSet( aString[ anOffset], Digits + [ '.'])
      do begin
        FLastToken := FLastToken + aString[ anOffset];
        Inc( anOffset);
      end;

      if FLastToken <> ''
      then begin
        aValue := StrToFloatDef( FLastToken, NaN);

        if IsNan( aValue)
        then begin
          aValue := 0.0;
          Error( 'Expected float value, saw %s in ParseFloat', [ FLastToken], aString, anOffset);
        end
        else Result := True;
      end;
    end;


    function    TModalCollection.ParseFormant( const aString: string; var anOffset: Integer; const aMode: string; var aFormant: TFormant): Boolean;
    var
      aSavedOffset : Integer;
      aValue       : TSignal;
    begin
      Result       := False;
      aSavedOffset := anOffset;

      if ParseFloat( aString, anOffset, aValue)
      then begin
        if SameText( aMode, 'ratio')
        then aFormant.Freg := 440.0 * aValue
        else aFormant.Freg :=         aValue;

        aFormant.dBLevel   := 0;
        aFormant.BandWidth := 0;

        if   not LookAhead( aString, anOffset, ',')
        and  not LookAhead( aString, anOffset, ';')
        then begin
          if ParseFloat( aString, anOffset, aValue)
          then begin
            aFormant.DbLevel := aValue;

            if   not LookAhead( aString, anOffset, ',')
            and  not LookAhead( aString, anOffset, ';')
            then begin
              if ParseFloat( aString, anOffset, aValue)
              then begin
                if aValue > 0
                then begin
                  aFormant.BandWidth := aFormant.Freg / aValue;
                  Result := True
                end
                else Error( 'dB value %g not > 0', [ aValue], aString, anOffset);
              end
              else Error( 'Expected a Q value in ParseFormant', aString, anOffset);
            end
            else Result := True
          end
          else Error( 'Expected a dB value in ParseFormant', aString, anOffset);
        end
        else Result := True
      end
      else begin
        anOffset := aSavedOffset;

        if not LookAhead( aString, anOffset, ',')
        then Error( 'Expected a frequency value in ParseFormant', aString, anOffset);
      end;
    end;


    function    TModalCollection.ParseFormants( const aString: string; var   anOffset: Integer; const aMode: string; var aFormants: TFormants  ): Boolean;
    var
      aFormant     : TFormant;
      aSavedOffset : Integer;
    begin
      Result := False;

      while CheckOffset( aString, anOffset) and ParseFormant( aString, anOffset, aMode, aFormant)
      do begin
        SetLength( aFormants, Length( aFormants) + 1);
        aFormants[ Length( aFormants) - 1] := aFormant;
        aSavedOffset := anOffset;

        if not ParseComma( aString, anOffset)
        then begin
          if ParseSemi( aString, anOffset)
          then begin
            Result := True;
            Break;
          end
          else begin                   // Accept missing semi-colon
            anOffset := aSavedOffset;
            Result   := True;
            Break;
          end;
        end;
      end;

      if not Result and not CheckOffset( aString, anOffset)
      then Error( 'Expected at least one Formant, in ParseFormants', aString, anOffset);
    end;


    function    TModalCollection.ParseFormantSet( const aString: string; var anOffset: Integer; var aFormantSet: TFormantSet): Boolean;
    var
      aFormants : TFormants;
      aMode     : string;
    begin
      Result := False;

      if ParseSetName( aString, anOffset, aFormantSet.Description)
      then begin
        if ParseComma( aString, anOffset)
        then begin
          aFormantSet.Mode := '';

          if ParseMode( aString, anOffset, aMode)
          then begin
            aFormantSet.Mode := aMode;

            if ParseComma( aString, anOffset)
            then begin
              if ParseFormants( aString, anOffset, aMode, aFormants)
              then begin
                aFormantSet.Formants := aFormants;
                Result := True;
              end
              else Error( 'Expected formants, in ParseFormantSet', aString, anOffset);
            end
            else begin
              if CheckOffset( aString, anOffset) and not LookAhead( aString, anOffset, ',')
              then Error( 'Expected a comma in ParseFormantSet', aString, anOffset)
              else Result := True;
            end;
          end
          else begin
            if CheckOffset( aString, anOffset) and not LookAhead( aString, anOffset, ',')
            then Error( 'Expected a mode specifier ParseFormantSet', aString, anOffset)
            else Result := True;
          end;
        end
        else begin
          if CheckOffset( aString, anOffset) and not LookAhead( aString, anOffset, ',')
          then Error( 'Expected a comma in ParseFormantSet', aString, anOffset)
          else Result := True;
        end;
      end
      else begin
        if CheckOffset( aString, anOffset)                            // there is more input ... so must be an error
        then Error( 'Expected a set name, read "%s", in ParseVowelSet', [ FLastToken], aString, anOffset)
        else Result := True;                                          // No more input, we are done then
      end;
    end;


    function    TModalCollection.ParseText( const aString: string): Boolean;
    var
      anOffset  : Integer;
      aFormants : TFormantSet;
    begin
      Result   := False;
      anOffset := Low( aString);

      while CheckOffset( aString, anOffset)
      do begin
        if ParseFormantSet( aString, anOffset, aFormants)
        then begin
          SetLength( FFormants, SetCount + 1);
          FFormants[ SetCount - 1] := aFormants;
        end
        else begin
          if CheckOffset( aString, anOffset)
          then Error( 'Excpected FormantSet, in ParseText', aString, anOffset)
          else Result := True;

          Break;
        end;
      end;
    end;


    function    TModalCollection.Parse( const aList: TStringList): Boolean; // overload;
    begin
      Clear;
      StripComments( aList);
      Result := ParseText( Implode( aList, ' '));
    end;


    function    TModalCollection.Parse( const aFileName: string): Boolean; // overload;
    var
      aStringList: TStringList;
    begin
      Result := False;

      if FileExists( aFileName)
      then begin
        aStringList := TStringList.Create;

        try
          aStringList.LoadFromFile( aFileName);
          Result := Parse( aStringList);

          if Result
          then LogFmt( 'File ''%s'' parsed OK', [ aFileName]);
        finally
          aStringList.DisposeOf;
        end;
      end
      else LogFmt( 'File ''%s'' does not exist', [ aFileName])
    end;


    procedure   TModalCollection.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( Self, LC_FORMANT, aMsg);
    end;


    procedure   TModalCollection.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs));
    end;


    procedure   TModalCollection.Error( const aMsg, aString: string; const anOffset: Integer); // overload;
    begin
      LogFmt( 'Formant spec error: %s, context is "%s"', [ aMsg, GetErrorLocation( aString, anOffset)]);
      Clear;
    end;


    procedure   TModalCollection.Error( const aFmt: string; const anArgs: array of const; const aString: string; const anOffset: Integer); // overload;
    begin
      Error( Format( aFmt, anArgs), aString, anOffset);
    end;


//  public

    constructor TModalCollection.Create( const aFileName: string; const aLogger: TKnobsOnLog = nil);
    begin
      inherited Create;
      FOnLog := aLogger;
      Parse( aFileName);
      CalculateSignals;
    end;


    destructor  TModalCollection.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TModalCollection.Clear;
    var
      aSet : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do SetLength( FFormants[ aSet].Formants, 0);

      SetLength( FFormants, 0);
    end;


    procedure   TModalCollection.CollectSetNames( const aStringList: TStringList);
    var
      i : Integer;
    begin
      if Assigned( aStringList)
      then begin
        aStringList.Clear;

        for i := 0 to SetCount - 1
        do aStringList.Add( Name[ i]);
      end;
    end;


    procedure   TModalCollection.Dump( const aStringList: TStringList); // overload;
    var
      aSet     : Integer;
      aFormant : Integer;
    begin
      for aSet := 0 to SetCount - 1
      do begin
        aStringList.Add( Format(   'set  : ''%s''', [ Name[ aSet]]));
        aStringList.Add( Format(   'mode : ''%s''', [ Mode[ aSet]]));

        for aFormant := 0 to FormantCount[ aSet] - 1
        do begin
          aStringList.Add( Format( '  #         : %d', [ aFormant                          ]));
          aStringList.Add( Format( '  Freq      : %f', [ Formant[ aSet, aFormant].Freg     ]));
          aStringList.Add( Format( '  dBLevel   : %f', [ Formant[ aSet, aFormant].dBLevel  ]));
          aStringList.Add( Format( '  BandWidth : %f', [ Formant[ aSet, aFormant].BandWidth]));
          aStringList.Add( Format( '  Q         : %f', [ Formant[ aSet, aFormant].Q        ]));
          aStringList.Add( Format( '  LinLevel  : %f', [ Formant[ aSet, aFormant].LinLevel ]));
        end;
      end;
    end;


    procedure   TModalCollection.Dump( const aFileName: string); // overload;
    var
      aStringList: TStringList;
    begin
      aStringList := TStringList.Create;

      try
        Dump( aStringList);
        aStringList.SaveToFile( aFileName);
      finally
        aStringList.DisposeOf;
      end;
    end;



initialization

finalization

  if Assigned( FormantCollection)
  then FreeAndNil( FormantCollection);

  if Assigned( ModalCollection)
  then FreeAndNil( ModalCollection);

end.

