unit FrmLastEdits;

{

   COPYRIGHT 2017 .. 2019 Blue Hell / Jan Punter

  Some parts are copyright :

     Author  : Neugls.
     Website : Http://www.neugls.info :: NOTE : as of 2017-06-11 website is all chinese now
     Email   : NeuglsWorkStudio@gmail.com

   these are marked with (*Neugls*)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  Winapi.Windows, Winapi.Messages,

  System.SysUtils, System.Variants, System.Classes, System.Rtti, System.StrUtils, System.IniFiles, System.Math,

  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls, Vcl.Grids, Vcl.ComCtrls,

  KnobsUtils, Knobs2013, KnobsConversions;


type

  TFormLastEdits = class(TForm)
    PanelTop: TPanel;
    PanelMain: TPanel;
    BitBtnClose: TBitBtn;
    BitBtnSetSelected: TBitBtn;
    ListView: TListView;
    BitBtnClear: TBitBtn;
    procedure BitBtnCloseClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure BitBtnSetSelectedClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListViewDblClick(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure BitBtnClearClick(Sender: TObject);
  private
    FEditor     : TKnobsWirePanel;
    FItems      : TStringList;
    FShown      : Boolean;
    FColWidths  : array[ 0 .. 4] of Integer;
  private
    procedure   SplitItem( anItemIndex: Integer; var Variation, Module, Control, PrevValue, CurrValue: string);
    procedure   ClearVisuals;
    procedure   AddLine( anItemIndex: Integer);
    procedure   UpdateVisuals;
    function    IsSelected( anItemIndex: Integer): Boolean;
    procedure   LockChanges;
    procedure   UnlockChanges;
    procedure   FillHistory;
    procedure   UndoItem( anIndex: Integer);
    procedure   UndoSelectedItems;
    procedure   ClearHistory;
  public
    procedure   Initialize( anEditor: TKnobsWirePanel);
    procedure   Clear;
    procedure   HistoryChanged;
    procedure   RemoveValuedControl( aControl: TKnobsValuedControl);
    procedure   SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    procedure   LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
  end;

var

  FormLastEdits: TFormLastEdits;

implementation

{$R *.dfm}

// User area

{ ========
  TFormLastEdits = class(TForm)
}

  //  private

    procedure   TFormLastEdits.SplitItem( anItemIndex: Integer; var Variation, Module, Control, PrevValue, CurrValue: string);
    var
      S      : string;
      aParts : TStringList;
      bParts : TStringList;
    begin
      Variation := '?';
      Module    := '';
      Control   := '';
      PrevValue := '';
      CurrValue := '';

      if ( anItemIndex >= 0) and ( anItemIndex < FItems.Count)
      then begin
        S      := FItems[ anItemIndex];
        Module := S;
        aParts := Explode( S, '|');

        try
          Variation := IntToStr( aParts.Count);

          if aParts.Count >= 4
          then begin
            TrimStrings( aParts);
            bParts := Explode( aParts[ 0], ' ');

            try
              Control := IntToStr( bParts.Count);

              if bParts.Count >= 4
              then begin
                TrimStrings( bParts);
                Variation := aParts[ 1];
                Module    := Format( '%s %s   %s', [ bParts[ 0], bParts[ 1], bParts[ 2]], AppLocale);
                Control   := bParts[ 3];
                PrevValue := aParts[ 2];
                CurrValue := aParts[ 3];
              end;
            finally
              bParts.DisposeOf;
            end;
          end;
        finally
          aParts.DisposeOf;
        end;
      end;
    end;


    procedure   TFormLastEdits.ClearVisuals;
    begin
      ListView.Clear;
    end;


    procedure   TFormLastEdits.AddLine( anItemIndex: Integer);
    var
      Variation : string;
      Module    : string;
      Control   : string;
      PrevValue : string;
      CurrValue : string;
      Item      : TListItem;
    begin
      if ( anItemIndex >= 0) and ( anItemIndex < FItems.Count)
      then begin
        SplitItem( anItemIndex, Variation, Module, Control, PrevValue, CurrValue);
        Item := ListView.Items.Add;
        Item.Caption    := Variation;
        Item.SubItems.Add( Module   );
        Item.SubItems.Add( Control  );
        Item.SubItems.Add( PrevValue);
        Item.SubItems.Add( CurrValue);
      end;
    end;


    procedure   TFormLastEdits.UpdateVisuals;
    var
      i : Integer;
    begin
      ClearVisuals;

      for i := 0 to FItems.Count - 1
      do AddLine( i);
    end;


    function    TFormLastEdits.IsSelected( anItemIndex: Integer): Boolean;
    begin
      if   ( anItemIndex >= 0)
      and  ( anItemIndex < FItems.Count)
      then begin
        if ListView.Items[ anItemIndex].Selected
        then Result := True
        else Result := False;
      end
      else Result := False;
    end;


    procedure   TFormLastEdits.LockChanges;
    begin
      if Assigned( FEditor)
      then FEditor.LockEditChanges;
    end;


    procedure   TFormLastEdits.UnlockChanges;
    begin
      if Assigned( FEditor)
      then FEditor.UnlockEditChanges;
    end;


    procedure   TFormLastEdits.FillHistory;
    begin
      if   Visible
      and  Assigned( FEditor)
      and  Assigned( FEditor.EditHistory)
      then begin
        FEditor.EditHistory.FillStrings( FItems, False, VIEWER_HISTORY_COUNT);
        UpdateVisuals;
      end;
    end;


    procedure   TFormLastEdits.UndoItem( anIndex: Integer);
    begin
      if   Assigned( FEditor)
      and  Assigned( FEditor.EditHistory)
      and  ( anIndex >= 0)
      and  ( anIndex < FItems.Count)
      then FEditor.EditHistory.UndoItem( anIndex);
    end;


    procedure   TFormLastEdits.UndoSelectedItems;
    var
      i : Integer;
    begin
      if   Assigned( FEditor)
      and  Assigned( FEditor.EditHistory)
      then begin
        FEditor.EditHistory.DeselectAll;

        for i := 0 to FItems.Count - 1
        do begin
          if IsSelected( i)
          then FEditor.EditHistory.SelectItem( i);
        end;

        LockChanges;

        try
          FEditor.EditHistory.UndoSelected;
        finally
          UnlockChanges;
        end;
      end;
    end;


    procedure   TFormLastEdits.ClearHistory;
    begin
      if   Assigned( FEditor)
      and  Assigned( FEditor.EditHistory)
      then begin
        FEditor.EditHistory.Clear;
        FillHistory;
      end;
    end;


//  public

    procedure   TFormLastEdits.Initialize( anEditor: TKnobsWirePanel);
    begin
      FEditor := anEditor;
    end;


    procedure   TFormLastEdits.Clear;
    begin
     FItems.Clear;
     UpdateVisuals;
    end;


    procedure   TFormLastEdits.HistoryChanged;
    begin
      FillHistory;
    end;


    procedure   TFormLastEdits.RemoveValuedControl( aControl: TKnobsValuedControl);
    begin
      if   Assigned( aControl)
      and  Assigned( FEditor)
      and  Assigned( FEditor.EditHistory)
      then begin
        FEditor.EditHistory.RemoveControl( aControl);
        FillHistory;
      end;
    end;


    procedure   TFormLastEdits.SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    var
      i : Integer;
    begin
      if Assigned( anIniFile)
      then begin
        with anIniFile
        do begin
          EraseSection( aSectionName);
          WriteInteger( aSectionName, 'Left'       , Left                 );
          WriteInteger( aSectionName, 'Top'        , Top                  );
          WriteInteger( aSectionName, 'Width'      , Width                );
          WriteInteger( aSectionName, 'Height'     , Height               );
          WriteInteger( aSectionName, 'WindowState', Integer( WindowState));

          if FShown
          then begin
            for i := 0 to 4
            do WriteInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), ListView.Columns[ i].Width);
          end
          else begin
            for i := 0 to 4
            do WriteInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), FColWidths[ i]);
          end;
        end;
      end;
    end;


    procedure   TFormLastEdits.LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
    var
      L, T, W, H : Integer;
      WinState   : TWindowState;
      i          : Integer;
    begin
      if Assigned( anIniFile)
      then begin
        with anIniFile
        do begin
          L        :=                ReadInteger( aSectionName, 'Left'       ,          Left        );
          T        :=                ReadInteger( aSectionName, 'Top'        ,          Top         );
          W        :=                ReadInteger( aSectionName, 'Width'      ,          Width       );
          H        :=                ReadInteger( aSectionName, 'Height'     ,          Height      );
          WinState := TWindowState ( ReadInteger( aSectionName, 'WindowState', Integer( WindowState)));

          if ( L < 0) or ( L > Screen.Width  - 20) then L := 0;
          if ( T < 0) or ( T > Screen.Height - 20) then T := 0;
          if W > Screen.Width                      then W := Screen.Width  - 20;
          if H > Screen.Height                     then H := Screen.Height - 20;

          SetBounds( L, T, W, H);

          if WinState = wsMaximized
          then WindowState := wsMaximized;

          for i := Low( FColWidths) to High( FColWidths)
          do FColWidths[ i] := ReadInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), ListView.Columns[ i].Width)
        end;
      end;
    end;



// Delphi area

procedure TFormLastEdits.ListViewDblClick(Sender: TObject);
begin
  UndoItem( ListView.Selected.Index);
end;

procedure TFormLastEdits.FormCreate(Sender: TObject);
begin
  FItems := TStringList.Create;
end;

procedure TFormLastEdits.FormDestroy(Sender: TObject);
begin
  FreeAndNil( FItems);
end;

procedure TFormLastEdits.FormHide(Sender: TObject);
var
  i : Integer;
begin
  for i := Low( FColWidths) to High( FColWidths)
  do FColWidths[ i] := ListView.Columns[ i].Width;
end;

procedure TFormLastEdits.FormShow(Sender: TObject);
var
  i : Integer;
begin
  FillHistory;

  for i := Low( FColWidths) to High( FColWidths)
  do ListView.Columns[ i].Width := FColWidths[ i];

  FShown := True;
end;

procedure TFormLastEdits.BitBtnSetSelectedClick(Sender: TObject);
begin
  UndoSelectedItems;
end;

procedure TFormLastEdits.BitBtnClearClick(Sender: TObject);
begin
  ClearHistory;
end;

procedure TFormLastEdits.BitBtnCloseClick(Sender: TObject);
begin
  Close;
end;

end.

