unit FrmRandomizer;

{

   COPYRIGHT 2016 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

{$ifdef DEBUG}
  {$define DEBUG_GENE}
{$endif}

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics, Vcl.Controls,
  Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.Buttons, Vcl.ExtCtrls, Vcl.CheckLst, System.IniFiles, Vcl.ComCtrls,

  knobs2013, KnobsUtils, KnobsConversions, FrmStore, KnobsWorms;

type

  TWormSelection = set of 0 .. 23;

  TOnRandomizerLog              = procedure( const aSender: TObject; const aMsg: string)                      of object;
  TOnRandomizerLiveMorphChanged = procedure( const aSender: TObject; aValue: TSignal)                         of object;
  TOnSendToEditor               = procedure( const Sender: TKnobsWormPanel; const aSelection: TWormSelection) of object;
  TOnLoadFromEditor             = procedure( const Sender: TKnobsWormPanel; const aSelection: TWormSelection) of object;
  TOnPatchRandomize             = procedure( const aSender: TObject)                                          of object;
  TOnPatchMutate                = procedure( const aSender: TObject)                                          of object;
  TOnPatchMate                  = procedure( const aSender: TObject)                                          of object;
  TOnPatchMorph                 = procedure( const aSender: TObject)                                          of object;
  TOnWormSelected               = procedure( const aSender: TObject; const aWorm: TKnobsWorm)                 of object;
  TOnWormDropped                = procedure( const aSender: TObject; const aWorm: TKnobsWorm)                 of object;


  TFormRandomizer = class(TForm)
    PageControlMain: TPageControl;
    TabSheetSettings: TTabSheet;
    PanelTypes: TPanel;
    PanelControlTypes: TPanel;
    CheckListBoxAllowedControlTypes: TCheckListBox;
    PanelControlTypesControl: TPanel;
    Label1: TLabel;
    BitBtnSelectAllControls: TBitBtn;
    BitBtnDeselectAllControls: TBitBtn;
    PanelModuleTypes: TPanel;
    BitBtnToggleAllControls: TBitBtn;
    PanelModuleStuff: TPanel;
    PanelModuleTypesControl: TPanel;
    Label2: TLabel;
    BitBtnSelectAllModules: TBitBtn;
    BitBtnDeselectAllModules: TBitBtn;
    BitBtnToggleAllModules: TBitBtn;
    CheckListBoxAllowedModuleTypes: TCheckListBox;
    PanelModuleCategories: TPanel;
    PanelModuleCategoriesControl: TPanel;
    Label3: TLabel;
    CheckListBoxAllowedModuleCategories: TCheckListBox;
    TabSheetBrreder: TTabSheet;
    PanelBreeder: TPanel;
    KnobsWormPanelBreeder: TKnobsWormPanel;
    KnobsKnobCrossoverProbability: TKnobsKnob;
    KnobsDisplayCrossoverProbability: TKnobsDisplay;
    Label5: TLabel;
    KnobsKnobMutationProbability: TKnobsKnob;
    KnobsDisplayMutationProbability: TKnobsDisplay;
    Label6: TLabel;
    KnobsKnobMutationRange: TKnobsKnob;
    KnobsDisplayMutationRange: TKnobsDisplay;
    Label7: TLabel;
    CheckBoxShowLines: TCheckBox;
    KnobsWormPanelScratch: TKnobsWormPanel;
    Label9: TLabel;
    Label10: TLabel;
    BitBtnRandomize: TBitBtn;
    BitBtnMutate: TBitBtn;
    BitBtnMate: TBitBtn;
    BitBtnMorph: TBitBtn;
    KnobsWormPatch: TKnobsWorm;
    Label11: TLabel;
    KnobsWormEraser: TKnobsWorm;
    Label8: TLabel;
    Label12: TLabel;
    KnobsKnobLiveMorph: TKnobsKnob;
    KnobsDisplayLiveMorph: TKnobsDisplay;
    CheckBoxShowSoapy: TCheckBox;
    BitBtnLoadFromEditor0: TBitBtn;
    BitBtnSendToEditor0: TBitBtn;
    BitBtnLoadFromEditor1: TBitBtn;
    BitBtnSendToEditor1: TBitBtn;
    BitBtnLoadFromEditor2: TBitBtn;
    BitBtnSendToEditor2: TBitBtn;
    BitBtnLoadFromEditor3: TBitBtn;
    BitBtnSendToEditor3: TBitBtn;
    BitBtnClear1: TBitBtn;
    BitBtnClear2: TBitBtn;
    BitBtnClear3: TBitBtn;
    Label13: TLabel;
    RndAmount: TKnobsKnob;
    DisplayRandomizationAmount: TKnobsDisplay;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure CheckListBoxAllowedControlTypesClickCheck(Sender: TObject);
    procedure CheckListBoxAllowedModuleTypesClickCheck(Sender: TObject);
    procedure BitBtnSelectAllControlsClick(Sender: TObject);
    procedure BitBtnDeselectAllControlsClick(Sender: TObject);
    procedure BitBtnSelectAllModulesClick(Sender: TObject);
    procedure BitBtnDeselectAllModulesClick(Sender: TObject);
    procedure BitBtnToggleAllControlsClick(Sender: TObject);
    procedure BitBtnToggleAllModulesClick(Sender: TObject);
    procedure CheckListBoxAllowedModuleCategoriesClickCheck(Sender: TObject);
    procedure RndAmount_ValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double; IsFinal,
      IsAutomation: Boolean);
    procedure KnobsKnobCrossoverProbabilityValueChanged(const aSender: TObject; const aPath, aControlType: string;
      aValue: Double; IsFinal, IsAutomation: Boolean);
    procedure KnobsKnobMutationProbabilityValueChanged(const aSender: TObject; const aPath, aControlType: string;
      aValue: Double; IsFinal, IsAutomation: Boolean);
    procedure KnobsKnobMutationRangeValueChanged(const aSender: TObject; const aPath, aControlType: string;
      aValue: Double; IsFinal, IsAutomation: Boolean);
    procedure CheckBoxShowLinesClick(Sender: TObject);
    procedure BitBtnRandomizeClick(Sender: TObject);
    procedure BitBtnMutateClick(Sender: TObject);
    procedure BitBtnMateClick(Sender: TObject);
    procedure BitBtnMorphClick(Sender: TObject);
    procedure KnobsKnobLiveMorphValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
      IsFinal, IsAutomation: Boolean);
    procedure CheckBoxShowSoapyClick(Sender: TObject);
    procedure KnobsWormPanelBreederSelected(Sender: TObject);
    procedure SendToEditorClick(Sender: TObject);
    procedure LoadFromEditorClick(Sender: TObject);
    procedure ClearWormsClick(Sender: TObject);
    procedure KnobsWormPanelScratchSelected(Sender: TObject);
    procedure KnobsWormPanelBreederWormDropped(const aSender: TObject; const aGene: TKnobsGene);
  private
    FPatchEditor                : TKnobsWirePanel;
    FControlTypes               : TKnobsConverters;
    FModuleTypes                : TKnobsModuleList;
    FOrigShowAllowRandomization : Boolean;
    FControlMode                : TDistanceMode;
    FWheelSupport               : Boolean;
    FWheelSensitivity           : Integer;
    FRndKnobPosition            : Integer;
    FRandomAmount               : TSignal;
    FLiveMorph                  : TSignal;
    FInitialized                : Boolean;
    FUpdateLock                 : Integer;
  private
    FCrossoverProbability       : TSignal;
    FMutationProbability        : TSignal;
    FMutationRange              : TSignal;
    FShowLines                  : Boolean;
    FShowSoapy                  : Boolean;
  private
    FLogger                     : TOnRandomizerLog;
    FOnLiveMorphChanged         : TOnRandomizerLiveMorphChanged;
    FOnSendToEditor             : TOnSendToEditor;
    FOnLoadFromEditor           : TOnLoadFromEditor;
    FOnPatchRandomize           : TOnPatchRandomize;
    FOnPatchMutate              : TOnPatchMutate;
    FOnPatchMate                : TOnPatchMate;
    FOnPatchMorph               : TOnPatchMorph;
    FOnWormSelected             : TOnWormSelected;
    FOnWormDropped              : TOnWormDropped;
  private
    procedure   SetControlMode      ( aValue: TDistanceMode);
    procedure   SetWheelSupport     ( aValue: Boolean      );
    procedure   SetWheelSensitivity ( aValue: Integer      );
    procedure   SetRndKnobPosition  ( aValue: Integer      );
  private
    procedure   SetLiveMorph        ( aValue: TSignal);
    procedure   SetShowLines        ( aValue: Boolean);
    procedure   SetShowSoapy        ( aValue: Boolean);
    procedure   SetPatchEditor      ( const aValue: TKnobsWirePanel);
  private
    procedure   DeselectAllControlTypes;
    procedure   DeselectAllModuleypes;
    procedure   SelectAllControlTypes;
    procedure   SelectAllModuleTypes;
    procedure   ToggleAllControlTypes;
    procedure   ToggleAllModuleTypes;
    procedure   SyncControlTypes;
    procedure   CollectControlTypes;
    procedure   SyncModuleTypes;
    procedure   CollectModuleTypes;
    procedure   CollectModuleCategories;
    procedure   UpdateControlType   ( anIndex: Integer);
    procedure   UpdateModuleType    ( anIndex: Integer);
    procedure   UpdateModuleCategory( anIndex: Integer);
    procedure   UpdateAllControlTypes;
    procedure   UpdateAllModuleTypes;
    procedure   CloseView;
    procedure   DoLog( const aSender: TObject; const aMsg: string);
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
  private
    procedure   SendToEditor  ( aSelection: Integer);
    procedure   LoadFromEditor( aSelection: Integer);
    procedure   ClearWorms    ( aSelection: Integer);
  private
    procedure   FixGeneSizes;
    procedure   BreederRandomize;
    procedure   BreederMutate;
    procedure   BreederMate;
    procedure   BreederMorph;
    procedure   BreederDropped;
    procedure   BreederSelected;
    procedure   ScratchSelected;
  private
    procedure   SaveControlTypes( const anIniFile: TMemIniFile; const aSection: string);
    procedure   SaveModuleTypes ( const anIniFile: TMemIniFile; const aSection: string);
    procedure   LoadControlTypes( const anIniFile: TMemIniFile; const aSection: string);
    procedure   LoadModuleTypes ( const anIniFile: TMemIniFile; const aSection: string);
  public
    procedure   SaveSettings    ( const anIniFile: TMemIniFile; const aSection: string);
    procedure   LoadSettings    ( const anIniFile: TMemIniFile; const aSection: string);
    procedure   View;
    procedure   PatchChanged( aVariation: Integer);
    procedure   PatchRecompiled;
  private
  public
    property    ControlMode          : TDistanceMode                 read FControlMode          write SetControlMode;
    property    WheelSupport         : Boolean                       read FWheelSupport         write SetWheelSupport;
    property    WheelSensitivity     : Integer                       read FWheelSensitivity     write SetWheelSensitivity;
    property    RndKnobPosition      : Integer                       read FRndKnobPosition      write SetRndKnobPosition;
    property    RandomAmount         : TSignal                       read FRandomAmount;
    property    LiveMorph            : TSignal                       read FLiveMorph            write SetLiveMorph;
    property    Logger               : TOnRandomizerLog              read FLogger               write FLogger;
    property    OnLiveMorphChanged   : TOnRandomizerLiveMorphChanged read FOnLiveMorphChanged   write FOnLiveMorphChanged;
    property    OnSendToEditor       : TOnSendToEditor               read FOnSendToEditor       write FOnSendToEditor;
    property    OnLoadFromEditor     : TOnLoadFromEditor             read FOnLoadFromEditor     write FOnLoadFromEditor;
    property    OnPatchRandomize     : TOnPatchRandomize             read FOnPatchRandomize     write FOnPatchRandomize;
    property    OnPatchMutate        : TOnPatchMutate                read FOnPatchMutate        write FOnPatchMutate;
    property    OnPatchMate          : TOnPatchMate                  read FOnPatchMate          write FOnPatchMate;
    property    OnPatchMorph         : TOnPatchMorph                 read FOnPatchMorph         write FOnPatchMorph;
    property    OnWormSelected       : TOnWormSelected               read FOnWormSelected       write FOnWormSelected;
    property    OnWormDropped        : TOnWormDropped                read FOnWormDropped        write FOnWormDropped;
  public
    property    CrossoverProbability : TSignal                       read FCrossoverProbability write FCrossoverProbability;
    property    MutationProbability  : TSignal                       read FMutationProbability  write FMutationProbability;
    property    MutationRange        : TSignal                       read FMutationRange        write FMutationRange;
    property    ShowLines            : Boolean                       read FShowLines            write SetShowLines;
    property    ShowSoapy            : Boolean                       read FShowSoapy            write SetShowSoapy;
    property    PatchEditor          : TKnobsWirePanel               read FPatchEditor          write SetPatchEditor;
  end;

var

  FormRandomizer: TFormRandomizer;



implementation



// User area

const

  sdfs_controltypes = '_ControlTypes';
  sdfs_moduletypes  = '_ModuleTypes';


//  private

    procedure   TFormRandomizer.SetControlMode( aValue: TDistanceMode);
    begin
      if aValue <> FControlMode
      then begin
        FControlMode := aValue;
        RndAmount                    .ControlMode := aValue;
        KnobsKnobCrossoverProbability.ControlMode := aValue;
        KnobsKnobMutationProbability .ControlMode := aValue;
        KnobsKnobMutationRange       .ControlMode := aValue;
        KnobsKnobLiveMorph           .ControlMode := aValue;
      end;
    end;


    procedure   TFormRandomizer.SetWheelSupport( aValue: Boolean);
    begin
      if aValue <> FWheelSupport
      then begin
        FWheelSupport := aValue;
        RndAmount                    .WheelSupport := aValue;
        KnobsKnobCrossoverProbability.WheelSupport := aValue;
        KnobsKnobMutationProbability .WheelSupport := aValue;
        KnobsKnobMutationRange       .WheelSupport := aValue;
        KnobsKnobLiveMorph           .WheelSupport := aValue;
      end
    end;


    procedure   TFormRandomizer.SetWheelSensitivity( aValue: Integer);
    begin
      if aValue <> FWheelSensitivity
      then begin
        FWheelSensitivity := aValue;
        RndAmount                    .WheelSensitivity := aValue;
        KnobsKnobCrossoverProbability.WheelSensitivity := aValue;
        KnobsKnobMutationProbability .WheelSensitivity := aValue;
        KnobsKnobMutationRange       .WheelSensitivity := aValue;
        KnobsKnobLiveMorph           .WheelSensitivity := aValue;
      end
    end;


    procedure   TFormRandomizer.SetRndKnobPosition( aValue: Integer);
    begin
      if aValue <> FRndKnobPosition
      then begin
        FRndKnobPosition       := aValue;
        RndAmount.KnobPosition := aValue;
        FRandomAmount          := RndAmount.AsValue;
      end;
    end;


//  private

    procedure   TFormRandomizer.SetLiveMorph( aValue: TSignal);
    begin
      aValue := Clip( aValue, 0.0, 1.0);

      if aValue <> FLiveMorph
      then begin
        {$ifdef DEBUG_GENE} LogFmt( 'TFormRandomizer.SetLiveMorph.Changed.Enter : %g', [ aValue]); {$endif}

        FLiveMorph := aValue;
        KnobsKnobLiveMorph.KnobPosition := Round( aValue * ( KnobsKnobLiveMorph.StepCount - 1));

        if Assigned( FOnLiveMorphChanged)
        then FOnLiveMorphChanged( Self, LiveMorph);

        {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.SetLiveMorph.Changed.Leave'); {$endif}
      end;
    end;


    procedure   TFormRandomizer.SetShowLines( aValue: Boolean);
    begin
      if aValue <> FShowLines
      then begin
        FShowLines                      := aValue;
        CheckBoxShowLines    .Checked   := aValue;
        KnobsWormPanelBreeder.DrawLines := ShowLines;
        KnobsWormPanelScratch.DrawLines := ShowLines;
        KnobsWormPatch       .DrawLines := ShowLines;
      end;
    end;


    procedure   TFormRandomizer.SetShowSoapy( aValue: Boolean);
    begin
      if aValue <> FShowSoapy
      then begin
        FShowSoapy                      := aValue;
        CheckBoxShowSoapy    .Checked   := aValue;
        KnobsWormPanelBreeder.DrawSoapy := ShowSoapy;
        KnobsWormPanelScratch.DrawSoapy := ShowSoapy;
        KnobsWormPatch       .DrawSoapy := ShowSoapy;
      end;
    end;


    procedure   TFormRandomizer.SetPatchEditor( const aValue: TKnobsWirePanel);
    begin
      if Assigned( aValue)
      then begin
        if aValue <> FPatchEditor
        then begin
          FPatchEditor := aValue;
          FInitialized := True;
        end
      end
      else begin
        FPatchEditor := nil;
        KnobsWormPatch       .Clear;
        KnobsWormPanelBreeder.Clear;
        KnobsWormPanelScratch.Clear
      end;
    end;


//  private

    procedure   TFormRandomizer.DeselectAllControlTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedControlTypes.Count - 1
      do CheckListBoxAllowedControlTypes.Checked[ i] := False;

      UpdateAllControlTypes;
    end;


    procedure   TFormRandomizer.DeselectAllModuleypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedModuleTypes.Count - 1
      do CheckListBoxAllowedModuleTypes.Checked[ i] := False;

      UpdateAllModuleTypes;
    end;


    procedure   TFormRandomizer.SelectAllControlTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedControlTypes.Count - 1
      do CheckListBoxAllowedControlTypes.Checked[ i] := True;

      UpdateAllControlTypes;
    end;


    procedure   TFormRandomizer.SelectAllModuleTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedModuleTypes.Count - 1
      do CheckListBoxAllowedModuleTypes.Checked[ i] := True;

      UpdateAllModuleTypes;
    end;


    procedure   TFormRandomizer.ToggleAllControlTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedControlTypes.Count - 1
      do CheckListBoxAllowedControlTypes.Checked[ i] := not CheckListBoxAllowedControlTypes.Checked[ i];

      UpdateAllControlTypes;
    end;


    procedure   TFormRandomizer.ToggleAllModuleTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedModuleTypes.Count - 1
      do CheckListBoxAllowedModuleTypes.Checked[ i] := not CheckListBoxAllowedModuleTypes.Checked[ i];

      UpdateAllModuleTypes;
    end;


    procedure   TFormRandomizer.SyncControlTypes;
    var
      i : Integer;
      C : TKnobsNullConverter;
    begin
      if Assigned( FControlTypes)
      then begin
        CheckListBoxAllowedControlTypes.Clear;
        CheckListBoxAllowedControlTypes.Sorted := False;

        for i := 0 to FControlTypes.Count - 1
        do begin
          C := FControlTypes.ConverterObject[ i];

          if Assigned( C)
          then begin
            CheckListBoxAllowedControlTypes.AddItem( C.Name, C);
            CheckListBoxAllowedControlTypes.Checked[ CheckListBoxAllowedControlTypes.Count - 1] := C.IsRandomizable;
          end;
        end;

        CheckListBoxAllowedControlTypes.Sorted := True;
      end;
    end;


    procedure   TFormRandomizer.CollectControlTypes;
    begin
      FControlTypes := Converters;
      SyncControlTypes;
    end;


    procedure   TFormRandomizer.SyncModuleTypes;
    var
      i : Integer;
      M : TKnobsCustomModule;
    begin
      if Assigned( FModuleTypes)
      then begin
        CheckListBoxAllowedModuleTypes.Clear;
        CheckListBoxAllowedModuleTypes.Sorted := False;

        for i := 0 to FModuleTypes.Count - 1
        do begin
          M := FModuleTypes[ i];

          if Assigned( M) and ( M.Title <> '') and ( M.ModuleType >= 0) and ( M.HasRandomizableControls)
          then begin
            CheckListBoxAllowedModuleTypes.AddItem( M.Title, M);
            CheckListBoxAllowedModuleTypes.Checked[ CheckListBoxAllowedModuleTypes.Count - 1] := M.AllowRandomization;
          end;
        end;

        CheckListBoxAllowedModuleTypes.Sorted := True;
      end;
    end;


    procedure   TFormRandomizer.CollectModuleTypes;
    begin
      if Assigned( FormStore)
      then begin
        FModuleTypes := FormStore.Modules;
        SyncModuleTypes;
      end;
    end;


    procedure   TFormRandomizer.CollectModuleCategories;
    var
      aList : TStringList;
      i     : Integer;
    begin
      if Assigned( FormStore)
      then begin
        aList := FormStore.CreatePageNames;

        try
          CheckListBoxAllowedModuleCategories.Clear;

          for i := 0 to aList.Count - 1
          do CheckListBoxAllowedModuleCategories.AddItem( aList[ i], nil);
        finally
          aList.DisposeOf;
        end;
      end;
    end;


    procedure   TFormRandomizer.UpdateControlType( anIndex: Integer);
    var
      S : string;
      R : Boolean;
    begin
      if Assigned( FControlTypes) and Assigned( FPatchEditor)
      then begin
        S := CheckListBoxAllowedControlTypes.Items  [ anIndex];
        R := CheckListBoxAllowedControlTypes.Checked[ anIndex];
        FControlTypes.SetRandomizable( S, R);
        FPatchEditor.SyncControlRandomization( S, R);
      end;
    end;


    procedure   TFormRandomizer.UpdateModuleType( anIndex: Integer);
    var
      S : string;
      R : Boolean;
      M : TKnobsCustomModule;
      T : TKnobsModuleType;
    begin
      if Assigned( FModuleTypes) and Assigned( FPatchEditor)
      then begin
        S := CheckListBoxAllowedModuleTypes.Items  [ anIndex];
        R := CheckListBoxAllowedModuleTypes.Checked[ anIndex];
        M := FModuleTypes.ByName[ S];

        if Assigned( M)
        then begin
          M.AllowRandomization := R;
          T := M.ModuleType;
          FPatchEditor.SyncModuleRandomization( T, R);
        end;
      end;
    end;


    procedure   TFormRandomizer.UpdateModuleCategory( anIndex: Integer);
    var
      i            : Integer;
      j            : Integer;
      R            : Boolean;
      CategoryName : string;
      ModuleName   : string;
    begin
      if Assigned( FModuleTypes) and Assigned( FPatchEditor)
      then begin
        CategoryName := CheckListBoxAllowedModuleCategories.Items  [ anIndex];
        R            := CheckListBoxAllowedModuleCategories.Checked[ anIndex];

        for i := 0 to FModuleTypes.Count - 1
        do begin
          if SameText( FModuleTypes.Module[ i].PageName, CategoryName)
          then begin
            ModuleName := FModuleTypes.Module[ i].Title;

            for j := 0 to CheckListBoxAllowedModuleTypes.Count - 1
            do begin
              if SameText( CheckListBoxAllowedModuleTypes.Items[ j], ModuleName)
              then begin
                CheckListBoxAllowedModuleTypes.Checked[ j] := R;
                Break;
              end;
            end;
          end;
        end;

        UpdateAllModuleTypes;
      end;
    end;


    procedure   TFormRandomizer.UpdateAllControlTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedControlTypes.Count - 1
      do UpdateControlType( i);
    end;


    procedure   TFormRandomizer.UpdateAllModuleTypes;
    var
      i : Integer;
    begin
      for i := 0 to CheckListBoxAllowedModuleTypes.Count - 1
      do UpdateModuleType( i);
    end;


    procedure   TFormRandomizer.CloseView;
    begin
      if Assigned( FPatchEditor)
      then FPatchEditor.ShowAllowRandomization := FOrigShowAllowRandomization;

      KnobsWormPatch       .OnLog := nil;
      KnobsWormPanelBreeder.OnLog := nil;
      KnobsWormPanelBreeder.OnLog := nil;
    end;


    procedure   TFormRandomizer.DoLog( const aSender: TObject; const aMsg: string);
    begin
      Log( aMsg);
    end;


    procedure   TFormRandomizer.Log( const aMsg: string);
    begin
      if Assigned( FLogger)
      then FLogger( Self, Format( 'Randomizer: %s', [ aMsg], AppLocale));
    end;


    procedure   TFormRandomizer.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


//  private


    procedure   TFormRandomizer.SendToEditor( aSelection: Integer);
    begin
      if Assigned( FOnSendToEditor)
      then begin
        case aSelection of
          0 : FOnSendToEditor( KnobsWormPanelBreeder, [  0 ..  7]);
          1 : FOnSendToEditor( KnobsWormPanelScratch, [  0 ..  7]);
          2 : FOnSendToEditor( KnobsWormPanelScratch, [  8 .. 15]);
          3 : FOnSendToEditor( KnobsWormPanelScratch, [ 16 .. 23]);
        end;
      end;
    end;


    procedure   TFormRandomizer.LoadFromEditor( aSelection: Integer);
    begin
      if Assigned( FOnLoadFromEditor)
      then begin
        case aSelection of
          0 : FOnLoadFromEditor( KnobsWormPanelBreeder, [  0 ..  7]);
          1 : FOnLoadFromEditor( KnobsWormPanelScratch, [  0 ..  7]);
          2 : FOnLoadFromEditor( KnobsWormPanelScratch, [  8 .. 15]);
          3 : FOnLoadFromEditor( KnobsWormPanelScratch, [ 16 .. 23]);
        end;
      end;
    end;


    procedure   TFormRandomizer.ClearWorms( aSelection: Integer);
    var
      i : Integer;
    begin
      case aSelection of
        1 : for i :=  0 to  7 do KnobsWormPanelScratch.Worm[ i].Clear;
        2 : for i :=  8 to 15 do KnobsWormPanelScratch.Worm[ i].Clear;
        3 : for i := 16 to 23 do KnobsWormPanelScratch.Worm[ i].Clear;
      end;
    end;


//  private

    procedure   TFormRandomizer.FixGeneSizes;
    begin
      if KnobsWormPanelBreeder.WormSize <> KnobsWormPanelScratch.WormSize
      then begin
        {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.FixGeneSizes.Enter'); {$endif}

        KnobsWormPanelScratch.WormSize := KnobsWormPanelBreeder.WormSize;
        KnobsWormPanelScratch.Clear;

        {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.FixGeneSizes.Leave'); {$endif}
      end;
    end;


    procedure   TFormRandomizer.BreederRandomize;
    begin
      if Assigned( FOnPatchRandomize)
      then FOnPatchRandomize( Self);
    end;


    procedure   TFormRandomizer.BreederMutate;
    begin
      if Assigned( FOnPatchMutate)
      then FOnPatchMutate( Self);
    end;


    procedure   TFormRandomizer.BreederMate;
    begin
      if Assigned( FOnPatchMate)
      then FOnPatchMate( Self);
    end;


    procedure   TFormRandomizer.BreederMorph;
    begin
      if Assigned( FOnPatchMorph)
      then FOnPatchMorph( Self);
    end;


    procedure   TFormRandomizer.BreederDropped;
    begin
      if Assigned( FOnWormDropped)
      then FOnWormDropped( Self, KnobsWormPanelBreeder.SelectedWorm);
    end;


    procedure   TFormRandomizer.BreederSelected;
    begin
      if Assigned( FOnWormSelected)
      then FOnWormSelected( Self, KnobsWormPanelBreeder.SelectedWorm);
    end;


    procedure   TFormRandomizer.ScratchSelected;
    begin
      if Assigned( FOnWormSelected)
      then FOnWormSelected( Self, KnobsWormPanelScratch.SelectedWorm);
    end;


//  private

    procedure   TFormRandomizer.SaveControlTypes( const anIniFile: TMemIniFile; const aSection: string);
    var
      i : Integer;
    begin
      with anIniFile
      do begin
        EraseSection( aSection);

        if Assigned( FControlTypes)
        then begin
          for i := 0 to FControlTypes.Count - 1
          do WriteBool( aSection, FControlTypes.ConverterObject[ i].ClassName, FControlTypes.ConverterObject[ i].IsRandomizable);
        end;
      end;
    end;


    procedure   TFormRandomizer.SaveModuleTypes( const anIniFile: TMemIniFile; const aSection: string);
    var
      i : Integer;
    begin
      with anIniFile
      do begin
        EraseSection( aSection);

        if Assigned( FModuleTypes)
        then begin
          for i := 0 to FModuleTypes.Count - 1
          do WriteBool( aSection, FModuleTypes[ i].Name, FModuleTypes[ i].AllowRandomization);
        end;
      end;
    end;


    procedure   TFormRandomizer.LoadControlTypes( const anIniFile: TMemIniFile; const aSection: string);
    var
      aList : TStringList;
      i     : Integer;
    begin
      with anIniFile
      do begin
        if Assigned( FControlTypes)
        then begin
          aList := TStringList.Create;

          try
            ReadSection( aSection, aList);

            for i := 0 to aList.Count - 1
            do begin
              FControlTypes.SetRandomizableByClass(
                aList[ i],
                ReadBool(
                  aSection,
                  aList[ i],
                  FControlTypes.GetRandomizableByClass( aList[ i])
                )
              );
            end;
          finally
            aList.DisposeOf;
          end;

         SyncControlTypes;
        end;
      end;
    end;


    procedure   TFormRandomizer.LoadModuleTypes( const anIniFile: TMemIniFile; const aSection: string);
    var
      aList : TStringList;
      i     : Integer;
      M     : TKnobsCustomModule;
    begin
      with anIniFile
      do begin
        if Assigned( FModuleTypes)
        then begin
          if FModuleTypes.Count = 0
          then Nop;

          aList := TStringList.Create;

          try
            ReadSection( aSection, aList);

            for i := 0 to aList.Count - 1
            do begin
              M := FModuleTypes.ByName[ aList[ i]];

              if Assigned( M)
              then begin
                M.AllowRandomization :=
                  ReadBool(
                    aSection,
                    aList[ i],
                    M.AllowRandomization
                  )
              end;
            end;
          finally
            aList.DisposeOf;
          end;

          SyncModuleTypes;
        end;
      end;
    end;


//  public


    procedure   TFormRandomizer.SaveSettings( const anIniFile: TMemIniFile; const aSection: string);
    begin
      if Assigned( anIniFile)
      then begin
        LogFmt( 'saving settings to: ''%s''', [ anIniFile.FileName]);

        with anIniFile
        do begin
          EraseSection( aSection);
          WriteInteger( aSection, 'left'                    , Left                                      );
          WriteInteger( aSection, 'top'                     , Top                                       );
          WriteInteger( aSection, 'width'                   , Width                                     );
          WriteInteger( aSection, 'height'                  , Height                                    );
          WriteInteger( aSection, 'rndknobposition'         , RndKnobPosition                           );
          WriteInteger( aSection, 'knobcrossoverprobability', KnobsKnobCrossoverProbability.KnobPosition);
          WriteInteger( aSection, 'knobmutationprobability' , KnobsKnobMutationProbability .KnobPosition);
          WriteInteger( aSection, 'knobmutationrange'       , KnobsKnobMutationRange       .KnobPosition);
          WriteBool   ( aSection, 'showlines'               , ShowLines                                 );
          WriteBool   ( aSection, 'showsoapy'               , ShowSoapy                                 );
        end;

        SaveControlTypes( anIniFile, aSection + sdfs_controltypes);
        SaveModuleTypes ( anIniFile, aSection + sdfs_moduletypes );

        Log( 'settings saved');
      end;
    end;


    procedure   TFormRandomizer.LoadSettings( const anIniFile: TMemIniFile; const aSection: string);
    var
      L, T, W, H : Integer;
    begin
      if Assigned( anIniFile)
      then begin
        LogFmt( 'loading settings from: ''%s''', [ anIniFile.FileName]);

        with anIniFile
        do begin
          L                                          := ReadInteger( aSection, 'left'                    , Left                                      );
          T                                          := ReadInteger( aSection, 'top'                     , Top                                       );
          W                                          := ReadInteger( aSection, 'width'                   , Width                                     );
          H                                          := ReadInteger( aSection, 'height'                  , Height                                    );
          RndKnobPosition                            := ReadInteger( aSection, 'rndknobposition'         , RndKnobPosition                           );
          KnobsKnobCrossoverProbability.KnobPosition := ReadInteger( aSection, 'knobcrossoverprobability', KnobsKnobCrossoverProbability.KnobPosition);
          KnobsKnobMutationProbability .KnobPosition := ReadInteger( aSection, 'knobmutationprobability' , KnobsKnobMutationProbability .KnobPosition);
          KnobsKnobMutationRange       .KnobPosition := ReadInteger( aSection, 'knobmutationrange'       , KnobsKnobMutationRange       .KnobPosition);
          ShowLines                                  := ReadBool   ( aSection, 'showlines'               , ShowLines                                 );
          ShowSoapy                                  := ReadBool   ( aSection, 'showsoapy'               , ShowSoapy                                 );

          SetBounds( L, T, W, H);
        end;

        LoadControlTypes( anIniFile, aSection + sdfs_controltypes);
        LoadModuleTypes ( anIniFile, aSection + sdfs_moduletypes );

        Log( 'settings loaded');
      end;
    end;


    procedure   TFormRandomizer.View;
    begin
      KnobsWormPatch       .OnLog := DoLog;
      KnobsWormPanelBreeder.OnLog := DoLog;
      KnobsWormPanelBreeder.OnLog := DoLog;

      PatchRecompiled;

      if Assigned( FPatchEditor)
      then begin
        FOrigShowAllowRandomization         := FPatchEditor.ShowAllowRandomization;
        FPatchEditor.ShowAllowRandomization := True;
      end;

      Show;
    end;


    procedure   TFormRandomizer.PatchChanged( aVariation: Integer);
    begin
      if Assigned( FPatchEditor)
      then begin
        if Lock( FUpdateLock)
        then begin
          try
            {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.PatchChanged.Enter'); {$endif}

            FPatchEditor.FillWorm( KnobsWormPatch, aVariation);

            if aVariation <> -1
            then FPatchEditor.FillWorm( KnobsWormPanelBreeder.Worm[ aVariation], aVariation);

            {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.PatchChanged.Leave'); {$endif}
          finally
            Unlock( FUpdateLock);
          end;
        end;
      end;
    end;


    procedure   TFormRandomizer.PatchRecompiled;
    begin
      if Assigned( FPatchEditor)
      then begin
        if Lock( FUpdateLock)
        then begin
          try
            {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.PatchRecompiled.Enter'); {$endif}

            FPatchEditor.FillWormPanel( KnobsWormPanelBreeder, [ 0 .. 7]);
            FPatchEditor.FillWorm     ( KnobsWormPatch       ,        -1);
            FixGeneSizes;

            {$ifdef DEBUG_GENE} Log( 'TFormRandomizer.PatchRecompiled.Leave'); {$endif}
          finally
            Unlock( FUpdateLock);
          end;
        end;
      end;
    end;


// Delphi area

{$R *.dfm}

procedure TFormRandomizer.ClearWormsClick(Sender: TObject);
begin
  ClearWorms(( Sender as TBitBtn).Tag);
end;


procedure TFormRandomizer.LoadFromEditorClick(Sender: TObject);
begin
  LoadFromEditor(( Sender as TBitBtn).Tag);
end;

procedure TFormRandomizer.RndAmount_ValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  RndKnobPosition := ( aSender as TKnobsKnob).KnobPosition;
  FRandomAmount   := aValue;
end;

procedure TFormRandomizer.BitBtnDeselectAllControlsClick(Sender: TObject);
begin
  DeselectAllControlTypes;
end;

procedure TFormRandomizer.BitBtnDeselectAllModulesClick(Sender: TObject);
begin
  DeselectAllModuleypes;
end;

procedure TFormRandomizer.SendToEditorClick(Sender: TObject);
begin
  SendToEditor(( Sender as TBitBtn).Tag);
end;

procedure TFormRandomizer.BitBtnMateClick(Sender: TObject);
begin
  BreederMate;
end;

procedure TFormRandomizer.BitBtnMorphClick(Sender: TObject);
begin
  BreederMorph;
end;

procedure TFormRandomizer.BitBtnMutateClick(Sender: TObject);
begin
  BreederMutate;
end;

procedure TFormRandomizer.BitBtnRandomizeClick(Sender: TObject);
begin
  BreederRandomize;
end;

procedure TFormRandomizer.BitBtnSelectAllControlsClick(Sender: TObject);
begin
  SelectAllControlTypes;
end;

procedure TFormRandomizer.BitBtnSelectAllModulesClick(Sender: TObject);
begin
  SelectAllModuleTypes;
end;

procedure TFormRandomizer.BitBtnToggleAllControlsClick(Sender: TObject);
begin
  ToggleAllControlTypes;
end;

procedure TFormRandomizer.BitBtnToggleAllModulesClick(Sender: TObject);
begin
  ToggleAllModuleTypes;
end;

procedure TFormRandomizer.CheckBoxShowLinesClick(Sender: TObject);
begin
  ShowLines := CheckBoxShowLines.Checked;
end;

procedure TFormRandomizer.CheckBoxShowSoapyClick(Sender: TObject);
begin
  ShowSoapy := CheckBoxShowSoapy.Checked;
end;

procedure TFormRandomizer.CheckListBoxAllowedControlTypesClickCheck(Sender: TObject);
begin
  UpdateControlType( CheckListBoxAllowedControlTypes.ItemIndex);
end;

procedure TFormRandomizer.CheckListBoxAllowedModuleCategoriesClickCheck(Sender: TObject);
begin
  UpdateModuleCategory( CheckListBoxAllowedModuleCategories.ItemIndex);
end;

procedure TFormRandomizer.CheckListBoxAllowedModuleTypesClickCheck(Sender: TObject);
begin
  UpdateModuleType( CheckListBoxAllowedModuleTypes.ItemIndex);
end;

procedure TFormRandomizer.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  CloseView;
end;

procedure TFormRandomizer.FormCreate(Sender: TObject);
begin
  CollectControlTypes;
  CollectModuleTypes;
  CollectModuleCategories;
end;

procedure TFormRandomizer.KnobsKnobCrossoverProbabilityValueChanged(const aSender: TObject; const aPath,
  aControlType: string; aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  CrossoverProbability := aValue;
end;

procedure TFormRandomizer.KnobsKnobLiveMorphValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  LiveMorph := aValue;
end;

procedure TFormRandomizer.KnobsKnobMutationProbabilityValueChanged(const aSender: TObject; const aPath,
  aControlType: string; aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  MutationProbability := aValue;
end;

procedure TFormRandomizer.KnobsKnobMutationRangeValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  MutationRange := aValue;
end;

procedure TFormRandomizer.KnobsWormPanelBreederSelected(Sender: TObject);
begin
  BreederSelected;
end;

procedure TFormRandomizer.KnobsWormPanelBreederWormDropped(const aSender: TObject; const aGene: TKnobsGene);
begin
  BreederDropped;
end;

procedure TFormRandomizer.KnobsWormPanelScratchSelected(Sender: TObject);
begin
  ScratchSelected;
end;

end.

