unit Globals;

{

   COPYRIGHT 2013 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}



// General purpose stuff


interface

uses

  System.SysUtils, System.StrUtils, WinApi.Windows, System.Classes, WinApi.MMSystem, Vcl.Forms, System.IniFiles,
  System.Win.Registry, Winapi.ShellAPI;


const

  DefaultUserTheme   = 'Windows';                // Use no theme for now at first run, but
  DefaultUseThemes   = True;                     // allow user to pick one
  slooks             = 'looks';                  // make a 'looks' ini section for the user interface stuff and theming
  BaseUrl            = 'https://www.bluehell.nl/wren/';
  EXT_WREN           = '.wren';
  EXT_ABSTRACT       = '.abstract';
  EXT_ABSTRACT_SAVED = '.abstract_saved';
  EXT_AUTO           = '.automation';
  EXT_SCALE          = '.scale';
  EXT_PRESET         = '.preset';


var

  UserTheme   : string;
  UseThemes   : Boolean;
  WrenMessage : DWord;                         // Mesage ID for system wide Wren messaging, see wren.dpr and FrmMain.pas


  function  ApplicationPath    : string;
  function  DocsPath           : string;
  function  LooksPath          : string;
  function  ApplicationFileName: string;
  function  IniFileName        : string;
  function  WrenIniFileName    : string;
  function  LogFileName        : string;
  function  ScaleFileName      : string;
  function  SetPriorityLevel( aLevel: Integer): Boolean;
  procedure StringTofile( const S, aFileName: string);
  function  CreateFileNames( const aFolder, aPattern: string; anAttributes: Integer; Clean: Boolean): TStrings;
  procedure ClearDebugTiming;
  function  GetDebugTimeStr: string;
  function  BrowseURL( const anURL: string) : boolean;
  function  CleanFileName( const aFileName : string): string;
  function  BytesFromByteArray( const aSrc: array of Byte; aCount: Integer): TBytes;
  procedure ByteArrayFromBytes( const aSrc: TBytes; out aDst: array of Byte; aCount: Integer);
  function  SafeWindowsName( const aPrefix: string): string;
  function  LongToBase( aValue: Cardinal; aBase: Cardinal; aFieldWidth: Integer = 1): string;
  function  StrToUIntBase( const aValue: string; aBase: Byte): Cardinal;


type


  TRecentStrings = class( TStringList)
  private
    FMaxDepth : Integer;
  private
    procedure   SetMaxDepth( aValue: Integer);
  public
    constructor Create( aMaxDepth : Integer);
    procedure   SetMostRecent( const aName: string);
    procedure   SaveToIni  ( anIniFile: TCustomIniFile; const aSection: string);
    procedure   LoadFromIni( anIniFile: TCustomIniFile; const aSection: string);
  public
    property    MaxDepth: Integer read FMaxDepth write SetMaxDepth;
  end;


  function  GetCompanyName      : string;
  function  GetFileDescription  : string;
  function  GetFileVersion      : string;
  function  GetInternalName     : string;
  function  GetLegalCopyRight   : string;
  function  GetLegalTradeMarks  : string;
  function  GetOriginalFileName : string;
  function  GetProductName      : string;
  function  GetProductVersion   : string;
  function  GetComments         : string;

  procedure FixTheme;



implementation


// Our name and path stuff

var

  strApplicationPath     : string;
  strDocsPath            : string;
  strLooksPath           : string;
  strApplicationFileName : string;
  strIniFileName         : string;
  strWrenIniFileName     : string;
  strLogFileName         : string;
  strScaleFileName       : string;


  function ApplicationPath: string;
  begin
    Result := strApplicationPath;
  end;


  function DocsPath: string;
  begin
    Result := strDocsPath;
  end;


  function LooksPath: string;
  begin
    Result := strLooksPath;
  end;


  function ApplicationFileName: string;
  begin
    Result := strApplicationFileName;
  end;


  function IniFileName: string;
  begin
    Result := strIniFileName;
  end;


  function WrenIniFileName: string;
  begin
    Result := strWrenIniFileName;
  end;


  function LogFileName: string;
  begin
    Result := strLogFileName;
  end;


  function ScaleFileName: string;
  begin
    Result := strScaleFileName;
  end;

// Date time stuff


  function  SetPriorityLevel( aLevel: Integer): Boolean;
  begin
    Result := False;

    case aLevel of
      -1 : Result := Setpriorityclass( GetCurrentProcess(), IDLE_PRIORITY_CLASS    );
       0 : Result := Setpriorityclass( GetCurrentProcess(), NORMAL_PRIORITY_CLASS  );
       1 : Result := Setpriorityclass( GetCurrentProcess(), HIGH_PRIORITY_CLASS    );
       2 : Result := Setpriorityclass( GetCurrentProcess(), REALTIME_PRIORITY_CLASS);
    end;
  end;


  procedure StringTofile( const S, aFileName: string);
  var
    aFile: TextFile;
  begin
    AssignFile( aFile, afileName);
    try
      Rewrite( aFile);
      Write( aFile, S);
    finally
      CloseFile( aFile);
    end;
  end;


  function  CreateFileNames( const aFolder, aPattern: string; anAttributes: Integer; Clean: Boolean): TStrings;
  var
    sr  : TSearchRec;
    res : Integer;
  begin
    Result := nil;
    res    := FindFirst( IncludeTrailingPathDelimiter( aFolder) + aPattern, anAttributes, sr);

    try
      if res = 0
      then begin
        Result := TStringList.Create;
        try
          while res = 0
          do begin
            if Clean
            then Result.Add( sr.FindData.cFileName)
            else Result.Add( IncludeTrailingPathDelimiter( aFolder) + sr.FindData.cFileName);
            res := FindNext( sr);
          end;
        except
          FreeAndNil( Result);
          raise;
        end;
      end;
    finally
      System.SysUtils.FindClose( sr);
    end;
  end;


  function  BytesFromByteArray( const aSrc: array of Byte; aCount: Integer): TBytes;
  begin
    if aCount > Length( aSrc)
    then aCount := Length( aSrc);

    SetLength( Result, aCount);
    Move( aSrc[ Low( aSrc)], Result[ 0], aCount); // Move checks size to be > 0
  end;


  procedure ByteArrayFromBytes( const aSrc: TBytes; out aDst: array of Byte; aCount: Integer);
  begin
    if aCount > Length( aDst)
    then aCount := Length( aDst);

    if aCount > Length( aSrc)
    then aCount := Length( aSrc);
    Move( aSrc[ 0], aDst[ Low( aDst)], aCount);   // Move checks size to be > 0
  end;


  function  SafeWindowsName( const aPrefix: string): string;
  begin
    Result := aPrefix + '_' + ReplaceText( IncludeTrailingPathDelimiter( Application.ExeName), '\', '_');
    Result := ReplaceText( Result, ':', '_');
    Result := ReplaceText( Result, '.', '_');
  end;


  function  LongToBase( aValue: Cardinal; aBase: Cardinal; aFieldWidth: Integer = 1): string;

    function ToDigit( aValue, aBase: Cardinal): string;
    begin
      Assert( aValue < aBase, 'ToDigit: Digit conversion error, aValue >= aBase');
      if aBase <= 62
      then
        if aValue > 9
        then begin
          if aValue > 35
          then Result := Char( aValue + Ord( '0') + 13)
          else Result := Char( aValue + Ord( '0') +  7);
        end
        else Result := Char( aValue + Ord( '0'))
      else Result := Format( '<&%d>', [ aValue]);
    end;

  begin
    Result := '';
    while aValue <> 0
    do begin
      Result := ToDigit( aValue mod aBase, aBase) + Result;
      aValue := aValue div aBase;
    end;
    if aFieldWidth > 0
    then begin
      while Length( Result) < aFieldWidth
      do Result := '0' + Result;
    end;
  end;


  function  StrToUIntBase( const aValue: string; aBase: Byte): Cardinal;
  const
    Digits : array[ 0 .. 15] of char =
    (
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
    );
    function LookUp( aDig: char): Byte;
    var
      n : Integer;
    begin
      Result := $ff;
      if ( aDig >= '0') and ( aDig <= 'F')
      then begin
        n := Ord( aDig) - Ord( '0');
        if n > 9
        then n := 10 + n - ( Ord( 'A') - Ord( '0'));
        Result := n;
      end;
    end;
  var
    C : Char;
    N : Byte;
  begin
    Result := 0;
    if ( aBase >= 2) and ( aBase <= 16)
    then begin
      for C in aValue
      do begin
        N := Lookup( UpCase( C));
        if ( N < aBase) and ( N <> $ff)
        then Result := Result * aBase + N
        else Break;
      end;
    end;
  end;


// ///////////// Debug timestamp support ///////////

var

  DebugName : string;
  DebugTime : LongWord;


  procedure ClearDebugTiming;
  begin
    DebugTime := TimeGetTime;
  end;

  function FormatDebugTime( aTime: Integer): string;
  var
    min, sec, ms : Integer;
  begin
    ms    := aTime mod 1000;
    aTime := aTime div 1000;
    sec   := aTime mod   60;
    aTime := aTime div   60;
    min   := aTime;
    Result := Format( '%.2d:%.2d %.3d', [ min, sec, ms]);
  end;

  function  GetDebugtime: Integer;
  begin
    Result := TimeGetTime - DebugTime;
  end;

  function  GetDebugTimeStr: string;
  begin
    Result := FormatDebugTime( GetDebugTime);
  end;

  // ///////////////////////////////////////////////


  function  BrowseURL( const anURL: string) : boolean;
  var
    Browser: string;
  begin
     Result := False;
     Browser := '';

     with TRegistry.Create
     do begin
       try
         RootKey := HKEY_CLASSES_ROOT;
         Access  := KEY_QUERY_VALUE;
         if OpenKey( '\http\shell\open\command', False)
         then Browser := ReadString( '');
         CloseKey;
       finally
         DisposeOf;
       end;
     end;

     if Browser <> ''
     then begin
       Browser := Copy( Browser, Pos( '"', Browser) + 1, Length( Browser));
       Browser := Copy( Browser, 1, Pos( '"', Browser) - 1);
       Result  := ShellExecute( 0, 'open', PChar( Browser), PChar( anURL), nil, SW_SHOW) >= 32;
     end;

     if not Result
     then Result := ShellExecute( 0, 'open', PChar( anURL), nil, nil, SW_SHOW) >= 32
  end;


const

  InfoNum  = 10;

var

  FileInfo : Array[ 1 .. InfoNum] Of String = (
    '',
    '',
    '',
    '',
    '',
    '',
    '',
    '',
    '',
    ''
  );


  procedure InitModule;
  const
    InfoStr : Array[ 1 .. InfoNum] Of String = (
      'CompanyName',
      'FileDescription',
      'FileVersion',
      'InternalName',
      'LegalCopyRight',
      'LegalTradeMarks',
      'OriginalFileName',
      'ProductName',
      'ProductVersion',
      'Comments'
    );
  Var
    S      : string;
    n      : Cardinal;
    Len    : Cardinal;
    Buf    : PChar;
    aValue : PChar;
    i      : Integer;
  begin
    S                      := Application.ExeName;
    strApplicationFileName := ExtractFileName( S);
    strApplicationPath     := ExtractFilePath( S);
    strDocsPath            := strApplicationPath + 'docs';
    strLooksPath           := strApplicationPath + 'looks';
    DebugName              := strApplicationPath + 'debug.txt';
    strIniFileName         := ChangeFileExt( S, '.ini'  );
    strWrenIniFileName     := strApplicationPath + 'wren.ini';
    strLogFileName         := ChangeFileExt( S, '.log'  );
    strScaleFileName       := ChangeFileExt( S, EXT_SCALE);

    n := GetFileVersionInfoSize( PChar( S), n);

    if n > 0
    then begin
      Buf := AllocMem( n);

      try
        GetFileVersionInfo( PChar( S), 0, n, Buf);

        for i := 1 to InfoNum
        do begin
          if
            VerQueryValue(
              Buf,
              PChar( '\StringFileInfo\040904E4\' + InfoStr[ i]),
              Pointer( aValue),
              Len
            )
          then FileInfo[ i] := aValue;
        end;
      finally
        FreeMem( Buf, n);
      end;
    end;
  end;


// System stuff

  function  CleanFileName( const aFileName : string): string;
  begin
    Result := ChangeFileExt( ExtractFileName( aFileName), '');
  end;


{
  TRecentStrings = class( TStringList)
  private
    FMaxDepth : Integer;
  public
    property    MaxDepth: Integer read FMaxDepth write SetMaxDepth;
  private
}


    procedure   TRecentStrings.SetMaxDepth( aValue: Integer);
    begin
      if aValue <> FMaxDepth
      then begin
        FMaxDepth := aValue;

        while Count > MaxDepth
        do Delete( Count - 1);
      end;
    end;


//  public

    constructor TRecentStrings.Create( aMaxDepth : Integer);
    begin
      inherited Create;
      MaxDepth      := aMaxDepth;
      CaseSensitive := False;
      Duplicates    := dupIgnore;
      Sorted        := False;
    end;


    procedure   TRecentStrings.SetMostRecent( const aName: string);
    var
      anIndex : Integer;
    begin
      if aName <> ''
      then begin
        anIndex := indexOf( aName);

        if anIndex >= 0
        then Delete( anIndex);

        Insert( 0, aName);

        while Count > MaxDepth
        do Delete( Count - 1);
      end;
    end;


    procedure   TRecentStrings.SaveToIni( anIniFile: TCustomIniFile; const aSection: string);
    var
      i : Integer;
    begin
      with anIniFile
      do begin
        EraseSection( aSection);
        WriteInteger( aSection, 'MaxDepth', MaxDepth);
        WriteInteger( aSection, 'Count'   , Count   );

        for i := 0 to Count - 1
        do WriteString( aSection, Format( 'item%d', [ i]), Strings[ i]);
      end;
    end;


    procedure   TRecentStrings.LoadFromIni( anIniFile: TCustomIniFile; const aSection: string);
    var
      n : Integer;
      i : Integer;
    begin
      with anIniFile
      do begin
        MaxDepth := ReadInteger( aSection, 'MaxDepth', MaxDepth);
        n        := ReadInteger( aSection, 'Count', 0);

        for i := 0 to n - 1
        do Add( ReadString( aSection, Format( 'item%d', [ i]), ''));
      end;
    end;


  function  GetCompanyName      : string; begin Result := FileInfo[  1]; end;
  function  GetFileDescription  : string; begin Result := FileInfo[  2]; end;
  function  GetFileVersion      : String; begin Result := FileInfo[  3]; end;
  function  GetInternalName     : String; begin Result := FileInfo[  4]; end;
  function  GetLegalCopyRight   : String; begin Result := FileInfo[  5]; end;
  function  GetLegalTradeMarks  : String; begin Result := FileInfo[  6]; end;
  function  GetOriginalFileName : String; begin Result := FileInfo[  7]; end;
  function  GetProductName      : String; begin Result := FileInfo[  8]; end;
  function  GetProductVersion   : String; begin Result := FileInfo[  9]; end;
  function  GetComments         : string; begin Result := FileInfo[ 10]; end;


  procedure FixTheme;
  var
    anIniFile : TMemIniFile;
  begin
    UserTheme := DefaultUserTheme;
    UseThemes := DefaultUseThemes;

    try
      if FileExists( IniFileName)
      then anIniFile := TMemIniFile.Create( IniFileName)
      else anIniFile := TMemIniFile.Create( WrenIniFileName);

      try
        UserTheme := anIniFile.ReadString( slooks, 'Theme'    , UserTheme);
        UseThemes := anIniFile.ReadBool  ( slooks, 'UseThemes', UseThemes);

        if UserTheme = ''
        then UserTheme := DefaultUserTheme;
      finally
        anIniFile.DisposeOf;
      end;
    except
      // Ignore theme loading eerrors
    end;
  end;




Initialization

  InitModule;

Finalization

end.


