unit Maze;

{

   COPYRIGHT 2017 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    Oogstplein 6
    7545 HP Enschede
    the Netherlands
    http://www.iaf.nl/Users/BlueHell/
    http://bluehell.electro-music.com/
    j_dot.punter2@t2iaf_dot.nl

  All rights attributed to Blue Hell are owned by Jan Punter.
}

interface

uses

  System.SysUtils, System.Types, System.Math,

  Vcl.Graphics,

  KnobsUtils, KnobsConversions;

type

  EMazeError = class( Exception);

  TMazeOnError = procedure( aSender: TObject; const aMsg: string) of object;

  TMazePoint = class;
  TMazeGraph = class;


  TMazeExtent = record
  public
    FLeft   : TSignal;
    FTop    : TSignal;
    FRight  : TSignal;
    FBottom : TSignal;
  public
    procedure   IncludePoint( anX, anY: TSignal);
  end;


  TMazeLink = record
  private
    FPoint   : TMazePoint;
    FLength  : TSignal;
    FAngle   : TSignal;
    FVisited : Boolean;
  public
    procedure   Init( const aPoint: TMazePoint; aLength, anAngle: TSignal);
  public
    property    Point   : TMazePoint read FPoint;
    property    Length  : TSignal    read FLength;
    property    Angle   : TSignal    read FAngle;
    property    Visited : Boolean    read FVisited write FVisited;
  end;


  TMazeLinks  = array  of TMazeLink;


  TMazePoint = class
  private
    FGraph : TMazeGraph;
    FIndex : Integer;
    FX     : TSignal;
    FY     : TSignal;
    FLinks : TMazeLinks;
  private
    function    GetCount: Integer;
    function    GetLink( anIndex: Integer): TMazeLink;
    procedure   SetLink( anIndex: Integer; const aValue: TMazeLink);
    procedure   FlagError( const aMsg: string);
    procedure   FlagErrorFmt( const aFmt: string; const anArgs: array of const);
  public
    constructor Create( const aMazeGraph: TMazeGraph; anIndex: Integer);
    destructor  Destroy;                                                                                       override;
    function    FindLink( const aPoint: TMazePoint): Integer;
    procedure   AddLink( const aPoint: TMazePoint; aLength, anAngle: TSignal);
    function    DistanceTo( anX, anY: TSignal): TSignal;
    procedure   UnvisitAll;
    procedure   VisitLink( anIndex: Integer);
    procedure   Paint( const aCanvas : TCanvas; aScale, anOffset: TSignal);
  public
    property    Count                    : Integer   read GetCount;
    property    Index                    : Integer   read FIndex;
    property    Link[ anIndex : Integer] : TMazeLink read GetLink  write SetLink;
    property    X                        : TSignal   read FX       write FX;
    property    Y                        : TSignal   read FY       write FY;
  end;


  TMazePoints = array of TMazePoint;


  TMazeGraph = class
  private
    FNodes   : TMazePoints;
    FExtent  : TMazeExtent;
    FOnError : TMazeOnError;
  private
    function    GetCount: Integer;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   FlagError( const aMsg: string);
    procedure   Clear;
    function    AddPoint( const aPoint: TMazePoint): Integer;
    function    CreatePoint( anX, anY: TSignal): Integer;
    procedure   Connect( aSource, aDestination: Integer; aLength, anAngle: TSignal);
    function    MoveTowards( aSource, aPrevious: Integer; anX, anY: TSignal): Integer;
    procedure   UnvisitAll;
    procedure   Paint( const aCanvas : TCanvas; aScale, anOffset: TSignal);
  public
    property    Count   : Integer      read GetCount;
    property    Extent  : TMazeExtent  read FExtent;
    property    OnError : TMazeOnError read FOnError write FOnError;
  end;


  TMazeHunter = class
  private
    FMazeGraph    : TMazeGraph;
    FPrevLocation : Integer;
    FLocation     : Integer;
  private
    procedure   SetPosition( anIndex: Integer);
  public
    constructor Create( const aMazeGraph: TMazeGraph);
    procedure   ChaseSwan( anX, anY: TSignal; var aDistance, anAngle: TSignal);
  end;


  TMazeVariable = record
  private
    FName  : string;
    FValue : TSignal;
  public
    procedure   Define( const aName: string; aValue: TSignal);
  public
    property    Name  : string  read FName;
    property    Value : TSignal read FValue write FValue;
  end;


  TMazeVariables = array of TMazeVariable;
  TMazeMaker     = class;


  TMazeStack = class
  private
    FDrawer : TMazeMaker;
    FValues : TSignalArray;
    FSize   : Integer;
    FSP     : Integer;
  private
    procedure   FlagError( const aMsg: string);
  public
    constructor Create( const aDrawer: TMazeMaker; aSize: Integer);
    procedure   Push( aValue: TSignal);
    function    Pop : TSignal;
    function    Peek: TSignal;
    procedure   Dup;
    procedure   Swap;
  end;


  TMazeMaker = class
  private
    FMazeGraph  : TMazeGraph;
    FAngle      : TSignal;
    FX          : TSignal;
    FY          : TSignal;
    FScale      : TSignal;
    FOrigin     : Integer;
    FVariables  : TMazeVariables;
    FValueStack : TMazeStack;
  private
    function    GetVarCount: Integer;
    function    FindVar    ( const aName: string): Integer;
    function    GetVarValue( const aName: string): TSignal;
    procedure   SetVarValue( const aName: string; aValue: TSignal);
    procedure   FlagError  ( const aMsg: string);
    procedure   FlagErrorFmt( const aFmt: string; const anArgs: array of const);
  public
    constructor Create       ( const aMazeGraph: TMazeGraph);
    destructor  Destroy;                                                                                       override;
    procedure   RotateDegrees( anAngle  : TSignal);
    procedure   RotateRadians( anAngle  : TSignal);
    procedure   Scale        ( anAmount : TSignal);
    procedure   Walk         ( aDistance: TSignal);
    procedure   PushValue    ( aValue   : TSignal);
    function    PopValue     : TSignal;
    function    PeekValue    : TSignal;
    procedure   DupValue;
    procedure   DropValue;
    procedure   SwapValue;
  public
    property    VarCount                       : Integer read GetVarCount;
    property    VarValue[ const aName: string] : TSignal read GetVarValue write SetVarValue;
  end;


implementation


  function ScalePoint( anX, anY, aScale, anOffset: TSignal): TPoint;
  begin
    Result.X := Round( anX * aScale + anOffset);
    Result.Y := Round( anY * aScale + anOffset);
  end;


{ ========
  TMazeExtent = record
  public
    FLeft   : TSignal;
    FTop    : TSignal;
    FRight  : TSignal;
    FBottom : TSignal;
  public
}

    procedure   TMazeExtent.IncludePoint( anX, anY: TSignal);
    begin
      if anX < FLeft
      then FLeft := anX
      else if anX > FRight
      then FRight := anX;

      if anY < FTop
      then FTop := anY
      else if anY > FBottom
      then FBottom := anY;
    end;


{ ========
  TMazeLink = record
  private
    FPoint   : TMazePoint;
    FLength  : TSignal;
    FAngle   : TSignal;
    FVisited : Boolean;
  public
    property    Point   : TMazePoint read FPoint;
    property    Length  : TSignal    read FLength;
    property    Angle   : TSignal    read FAngle;
    property    Visited : Boolean    read FVisited write FVisited;
  public
}

    procedure   TMazeLink.Init( const aPoint: TMazePoint; aLength, anAngle: TSignal);
    begin
      FPoint  := aPoint;
      FLength := aLength;
      FAngle  := anAngle;
    end;


{ ========
  TMazePoint = class
  private
    FGraph : TMazeGraph;
    FIndex : Integer;
    FX     : TSignal;
    FY     : TSignal;
    FLinks : TMazeLinks;
  public
    property    Count                    : Integer   read GetCount;
    property    Index                    : Integer   read FIndex;
    property    Link[ anIndex : Integer] : TMazeLink read GetLink  write SetLink;
    property    X                        : TSignal   read FX       write FX;
    property    Y                        : TSignal   read FY       write FY;
  private
}

    function    TMazePoint.GetCount: Integer;
    begin
      Result := Length( FLinks);
    end;


    function    TMazePoint.GetLink( anIndex: Integer): TMazeLink;
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then Result := FLinks[ anIndex]
      else begin
        Result.FPoint := nil;
        FlagErrorFmt( 'Link index %d out of bounds [0..%d>', [ anIndex, Count])
      end;
    end;


    procedure   TMazePoint.SetLink( anIndex: Integer; const aValue: TMazeLink);
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then FLinks[ anIndex] := aValue
      else FlagErrorFmt( 'Link index %d out of bounds [0..%d>', [ anIndex, Count])
    end;


    procedure   TMazePoint.FlagError( const aMsg: string);
    begin
      FGraph.FlagError( aMsg);
    end;


    procedure   TMazePoint.FlagErrorFmt( const aFmt: string; const anArgs: array of const);
    begin
      FlagError( Format( aFmt, anArgs));
    end;


//  public

    constructor TMazePoint.Create( const aMazeGraph: TMazeGraph; anIndex: Integer);
    begin
      Assert( anIndex >= 0);
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FGraph := aMazeGraph;
      FIndex := anIndex;
    end;


    destructor  TMazePoint.Destroy;  // override;
    begin
      SetLength( FLinks, 0);
    end;


    function    TMazePoint.FindLink( const aPoint: TMazePoint): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to Count - 1
      do begin
        if FLinks[ i].Point = aPoint
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TMazePoint.AddLink( const aPoint: TMazePoint; aLength, anAngle: TSignal);
    var
      aLink : TMazeLink;
    begin
      if   Assigned( aPoint)
      then begin
        if FindLink( aPoint) < 0
        then begin
          aLink.Init( aPoint, aLength, anAngle);
          SetLength( FLinks, Count + 1);
          FLinks[ Count - 1] := aLink;
        end;

        aPoint.AddLink( Self, aLength, MathFloatMod( anAngle + Pi, TWO_PI));
      end;
    end;


    function    TMazePoint.DistanceTo( anX, anY: TSignal): TSignal;
    var
      dx : TSignal;
      dy : TSignal;
    begin
      dx     := anX - X;
      dy     := anY - Y;
      Result := Sqrt( dx * dx + dy * dy);
    end;


    procedure   TMazePoint.UnvisitAll;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FLinks[ i].Visited := False;
    end;


    procedure   TMazePoint.VisitLink( anIndex: Integer);
    var
      Dst   : TMazePoint;
      Index : Integer;
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then begin
        FLinks[ anIndex].Visited := True;
        Dst := FLinks[ anIndex].Point;

        if Assigned( Dst)
        then begin
           Index := Dst.FindLink( Self);

           if Index > 0
           then Dst.VisitLink( Index)
           else FlagErrorFmt( 'Link with index %d has no backlink', [ anIndex]);
        end;
      end
      else FlagErrorFmt( 'Invalid link index %d', [ anIndex]);
    end;


    procedure   TMazePoint.Paint( const aCanvas : TCanvas; aScale, anOffset: TSignal);
    var
      Origin      : TPoint;
      Destination : TPoint;
      Dst         : TMazePoint;
      i           : Integer;
    begin
      Origin := ScalePoint( X, Y, aScale, anOffset);

      for i := 0 to Count - 1
      do begin
        if not Link[ i].Visited
        then begin
          Dst := Link[ i].FPoint;

          if Assigned( Dst)
          then begin
            Destination := ScalePoint( Dst.X, Dst.Y, aScale, anOffset);
            aCanvas.MoveTo( Origin.X     , Origin.Y     );
            aCanvas.LineTo( Destination.X, Destination.Y);
            VisitLink( i);
          end
          else FlagErrorFmt( 'Link with index %d has no valid point', [ i]);
        end;
      end;
    end;


{ ========
  TMaze = class
  private
    FNodes   : TMazePoints;
    FExtent  : TMazeExtent;
    FOnError : TMazeOnError;
  public
    property    Count   : Integer      read GetCount;
    property    Extent  : TMazeExtent  read FExtent;
    property    OnError : TMazeOnError read FOnError write FOnError;
  private
}

    function    TMazeGraph.GetCount: Integer;
    begin
      Result := Length( FNodes);
    end;


//  public

    constructor TMazeGraph.Create;
    begin
      FExtent.IncludePoint( 0.0, 0.0);
    end;


    destructor  TMazeGraph.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TMazeGraph.FlagError( const aMsg: string);
    begin
      if Assigned( FOnError)
      then FOnError( Self, aMsg)
      else raise EMazeError.Create( aMsg);
    end;


    procedure   TMazeGraph.Clear;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FreeAndNil( FNodes[ i]);

      SetLength( FNodes, 0);
    end;


    function    TMazeGraph.AddPoint( const aPoint: TMazePoint): Integer;
    begin
      Result := -1;

      if   Assigned( aPoint)
      then begin
        SetLength( FNodes, Count + 1);
        FNodes[ Count - 1] := aPoint;
        Result := Count - 1;
        FExtent.IncludePoint( aPoint.FX, aPoint.FY);
      end;
    end;


    function    TMazeGraph.CreatePoint( anX, anY: TSignal): Integer;
    var
      aPoint : TMazePoint;
    begin
      aPoint   := TMazePoint.Create( Self, Count);
      aPoint.X := anX;
      aPoint.Y := anY;
      Result   := AddPoint( aPoint);
    end;


    procedure   TMazeGraph.Connect( aSource, aDestination: Integer; aLength, anAngle: TSignal);
    var
      Src : TMazePoint;
      Dst : TMazePoint;
    begin
      Src := FNodes[ aSource     ];
      Dst := FNodes[ aDestination];

      if   Assigned( Src)
      and  Assigned( Dst)
      then Src.AddLink( Dst, aLength, anAngle);
    end;


    function    TMazeGraph.MoveTowards( aSource, aPrevious: Integer; anX, anY: TSignal): Integer;
    // Find a way in the maze that gets one step closer to a target X, Y position
    // The result is the LinkIndex of the link that moves us closer
    // aPrevious is not considered a valid path ... so thats skipped inthe search
    var
      Src         : TMazePoint;
      Dst         : TMazePoint;
      i           : Integer;
      D           : TSignal;
      MinD        : TSignal;
      HasDistance : Boolean;
    begin
      Result      := -1;
      Src         := FNodes[ aSource];
      HasDistance := False;
      MinD        := 1e6;

      if   Assigned( Src)
      then begin
        for i := 0 to Src.Count - 1
        do begin
          if i <> aPrevious
          then begin
            Dst := Src.Link[ i].FPoint;

            if Assigned( Dst)
            then begin
              D := Dst.DistanceTo( anX, anY);

              if not HasDistance
              then begin
                MinD        := D;
                HasDistance := True;
                Result      := i;
              end
              else begin
                if D < MinD
                then begin
                  MinD   := D;
                  Result := i;
                end;
              end;
            end;
          end
        end;
      end;
    end;


    procedure   TMazeGraph.UnvisitAll;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FNodes[ i].UnvisitAll;
    end;


    procedure   TMazeGraph.Paint( const aCanvas : TCanvas; aScale, anOffset: TSignal);
    var
      i : Integer;
    begin
      UnvisitAll;

      for i := 0 to Count - 1
      do FNodes[ i].Paint( aCanvas, aScale, anOffset);
    end;



{ ========
  TMazeHunter = class
  private
    FMazeGraph    : TMazeGraph;
    FPrevLocation : Integer;
    FLocation     : Integer;
  private
}

    procedure   TMazeHunter.SetPosition( anIndex: Integer);
    begin
      FPrevLocation := FLocation;
      FLocation     := anIndex;
    end;


//  public

    constructor TMazeHunter.Create( const aMazeGraph: TMazeGraph);
    begin
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FMazeGraph    := aMazeGraph;
      FPrevLocation := -1;
      FLocation     := -1;
    end;


    procedure   TMazeHunter.ChaseSwan( anX, anY: TSignal; var aDistance, anAngle: TSignal);
    var
      Src       : TMazePoint;
      DstIndex  : Integer;
      LinkIndex : Integer;
      Link      : TMazeLink;
    begin
      aDistance := -1;
      anAngle   := -1;

      if Assigned( FMazeGraph)
      then begin
        if   FLocation < 0                             // when we are nowhere ...
        then FLocation := Random( FMazeGraph.Count);   // jump into the maze at a random location

        // So now  we are somewhere ...

        LinkIndex := FMazeGraph.MoveTowards( FLocation, FPrevLocation, anX, anY);

        if   LinkIndex >= 0
        then begin
          Src := FMazeGraph.FNodes[ FLocation];

          if Assigned( Src)
          then begin
            Link := Src.Link[ LinkIndex];

            if Assigned( Link.Point)
            then begin
              DstIndex  := Link.Point.Index;
              aDistance := Link.Length;
              anAngle   := Link.Angle;
              SetPosition( DstIndex);
            end;
          end;
        end
      end;
    end;


{ ========
  TMazeVariable = record
  private
    FName  : string;
    FValue : TSignal;
  public
    property    Name  : string  read FName;
    property    Value : TSignal read FValue write FValue;
  public
}

    procedure   TMazeVariable.Define( const aName: string; aValue: TSignal);
    begin
      FName  := aName;
      FValue := aValue;
    end;


{ ========
  TMazeStack = class
  private
    FDrawer : TMazeMaker;
    FValues : TSignalArray;
    FSize   : Integer;
    FSP     : Integer;
  private
}

    procedure   TMazeStack.FlagError( const aMsg: string);
    begin
      FDrawer.FlagError( aMsg);
    end;


//  public

    constructor TMazeStack.Create( const aDrawer: TMazeMaker; aSize: Integer);
    begin
      Assert( aSize > 0);
      Assert( Assigned( aDrawer));
      inherited Create;
      FDrawer := aDrawer;
      FSize   := aSize;
      SetLength( FValues, FSize);
      FSP := -1;
    end;


    procedure   TMazeStack.Push( aValue: TSignal);
    begin
      if FSP < FSize
      then begin
        Inc( FSP);
        FValues[ FSP] := aValue;
      end
      else FlagError( 'Stack overflow');
    end;


    function    TMazeStack.Pop: TSignal;
    begin
      Result := 0.0;

      if FSP >= 0
      then begin
        Result := FValues[ FSP];
        Dec( FSP);
      end
      else FlagError( 'Stack underflow');
    end;


    function    TMazeStack.Peek: TSignal;
    begin
      Result := 0.0;

      if FSP >= 0
      then Result := FValues[ FSP]
      else FlagError( 'Stack underflow');
    end;


    procedure   TMazeStack.Dup;
    begin
      Push( Peek);
    end;


    procedure   TMazeStack.Swap;
    var
      Tmp : TSignal;
    begin
      if FSP >= 1
      then begin
        Tmp               := Peek;
        FValues[ FSP]     := FValues[ FSP - 1];
        FValues[ FSP - 1] := Tmp;
      end
      else FlagError( 'Stack overflow');
    end;


{ ========
  TMazeMaker = class
  private
    FMazeGraph  : TMazeGraph;
    FAngle      : TSignal;
    FX          : TSignal;
    FY          : TSignal;
    FScale      : TSignal;
    FOrigin     : Integer;
    FVariables  : TMazeVariables;
    FValueStack : TMazeStack;
  public
    property    VarCount                       : Integer read GetVarCount;
    property    VarValue[ const aName: string] : TSignal read GetVarValue write SetVarValue;
  private
}

    function    TMazeMaker.GetVarCount: Integer;
    begin
      Result := Length( FVariables);
    end;


    function    TMazeMaker.FindVar( const aName: string): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to VarCount - 1
      do begin
        if SameText( FVariables[ i].Name, aName)
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TMazeMaker.GetVarValue( const aName: string): TSignal;
    var
      anIndex : Integer;
    begin
      Result  := 0.0;
      anIndex := FindVar( aName);

      if anIndex >= 0
      then Result := FVariables[ anIndex].Value
      else FlagErrorFmt( 'Reading undefined variable: ''%s''', [ aName]);
    end;


    procedure   TMazeMaker.SetVarValue( const aName: string; aValue: TSignal);
    var
      anIndex : Integer;
    begin
      anIndex := FindVar( aName);

      if anIndex < 0
      then begin
        SetLength( FVariables, VarCount + 1);
        FVariables[ VarCount - 1].Define( aName, aValue);
      end
      else FVariables[ anIndex].Value := aValue;
    end;


    procedure   TMazeMaker.FlagError( const aMsg: string);
    begin
      FMazeGraph.FlagError( aMsg);
    end;


    procedure   TMazeMaker.FlagErrorFmt( const aFmt: string; const anArgs: array of const);
    begin
      FlagError( Format( aFmt, anArgs));
    end;


//  public

    constructor TMazeMaker.Create( const aMazeGraph: TMazeGraph);
    begin
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FValueStack := TMazeStack.Create( Self, 1024);
      FMazeGraph  := aMazeGraph;
      FAngle := 0.0;
      FX     := 0.0;
      FY     := 0.0;
      FScale := 1.0;
      FMazeGraph.Clear;
      FOrigin := FMazeGraph.CreatePoint( FX, FY);
    end;


    destructor  TMazeMaker.Destroy; // override;
    begin
      FreeAndNil( FValueStack);
      inherited;
    end;


    procedure   TMazeMaker.RotateDegrees( anAngle: TSignal);
    begin
      RotateRadians( DegToRad( anAngle));
    end;


    procedure   TMazeMaker.RotateRadians( anAngle: TSignal);
    begin
      FAngle := MathFloatMod( FAngle + anAngle, TWO_PI);
    end;


    procedure   TMazeMaker.Scale( anAmount : TSignal);
    begin
      FScale := FScale * anAmount;
    end;


    procedure   TMazeMaker.Walk( aDistance: TSignal);
    var
      aNewX     : TSignal;
      aNewY     : TSignal;
      aNewIndex : Integer;
      aStride   : TSignal;
    begin
      aStride   := FScale  * aDistance;
      aNewX     := aStride * Cos( FAngle);
      aNewY     := aStride * Sin( FAngle);
      aNewIndex := FMazeGraph.CreatePoint( aNewX, aNewY);
      FMazeGraph.Connect( FOrigin, aNewIndex, aStride, FAngle);
      FOrigin := aNewIndex;
      FX      := aNewX;
      FY      := aNewY;
    end;


    procedure   TMazeMaker.PushValue( aValue: TSignal);
    begin
      FValueStack.Push( aValue);
    end;


    function    TMazeMaker.PopValue: TSignal;
    begin
      Result := FValueStack.Pop;
    end;


    function    TMazeMaker.PeekValue: TSignal;
    begin
      Result := FValueStack.Peek;
    end;


    procedure   TMazeMaker.DupValue;
    begin
      FValueStack.Dup;
    end;


    procedure   TMazeMaker.DropValue;
    begin
      PopValue;
    end;


    procedure   TMazeMaker.SwapValue;
    begin
      FValueStack.Swap;
    end;


end.

