unit Moog;

(*

  Source : https://github.com/ddiakopoulos/MoogLadders

  --

  Copyright (c) 2015, Dimitri Diakopoulos All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  -- --

  Huovilainen developed an improved and physically correct model of the Moog
  Ladder filter that builds upon the work done by Smith and Stilson. This model
  inserts nonlinearities inside each of the 4 one-pole sections on account of the
  smoothly saturating function of analog transistors. The base-emitter voltages of
  the transistors are considered with an experimental value of 1.22070313 which
  maintains the characteristic sound of the analog Moog. This model also permits
  self-oscillation for resonances greater than 1. The model depends on five
  hyperbolic tangent functions (tanh) for each sample, and an oversampling factor
  of two (preferably higher, if possible). Although a more faithful
  representation of the Moog ladder, these dependencies increase the processing
  time of the filter significantly. Lastly, a half-sample delay is introduced for
  phase compensation at the final stage of the filter.

  References: Huovilainen (2004), Huovilainen (2010), DAFX - Zolzer (ed) (2nd ed)
  Original implementation: Victor Lazzarini for CSound5

  Considerations for oversampling:
  http://music.columbia.edu/pipermail/music-dsp/2005-February/062778.html
  http://www.synthmaker.co.uk/dokuwiki/doku.php?id=tutorials:oversampling

  --

  2017-02-16 : Adapted for use in Wren by Blue Hell

*)


interface

uses

  System.Math, KnobsUtils, KnobsConversions;


type

  TMoogFilter = class
  private
    FCutoff     : TSignal;
    FResonance  : TSignal;
    FSampleRate : TSignal;
  protected
    procedure   SetResonance( aValue: TSignal);                                                                 virtual;
    procedure   SetCutoff   ( aValue: TSignal);                                                                 virtual;
  public
    constructor Create( aSampleRate: TSignal);                                                                  virtual;
    procedure   Process( InSample: TSignal; var OutSample: TSignal);                                            virtual;
  public
    property    Resonance : TSignal read FResonance write SetResonance;
    property    CutOff    : TSignal read FCutoff    write SetCutoff;
  end;

  TMoogFilterClass = class of TMoogFilter;


  THuovilainenMoog = class( TMoogFilter)
  strict private
  const
    FThermal = 0.000025;
    FComp    = 0.5;
  private
    FStage0     : TSignal;
    FStage1     : TSignal;
    FStage2     : TSignal;
    FStage3     : TSignal;
    FStageTanh0 : TSignal;
    FStageTanh1 : TSignal;
    FStageTanh2 : TSignal;
    FDelay0     : TSignal;
    FDelay1     : TSignal;
    FDelay2     : TSignal;
    FDelay3     : TSignal;
    FDelay4     : TSignal;
    FDelay5     : TSignal;
    FTune       : TSignal;
    FAcr        : TSignal;
    FResQuad    : TSignal;
  protected
    procedure   SetResonance( aValue: TSignal);                                                                override;
    procedure   SetCutoff   ( aValue: TSignal);                                                                override;
  public
    constructor Create( aSampleRate: TSignal);                                                                 override;
    procedure   Process( InSample: TSignal; var OutSample: TSignal);                                           override;
  end;


implementation


{ ========
  TMoogFilter = class
  private
    FCutoff     : TSignal;
    FResonance  : TSignal;
    FSampleRate : TSignal;
  public
    property    Resonance : TSignal read FResonance write SetResonance;
    property    CutOff    : TSignal read FCutoff    write SetCutoff;
  protected
}

    procedure   TMoogFilter.SetResonance( aValue: TSignal); // virtual;
    begin
      FResonance := aValue;
    end;


    procedure   TMoogFilter.SetCutoff( aValue: TSignal); // virtual;
    begin
      FCutoff := aValue;
    end;


//  public

    constructor TMoogFilter.Create( aSampleRate: TSignal); // virtual;
    begin
      FSampleRate := aSampleRate;
    end;


    procedure   TMoogFilter.Process( InSample: TSignal; var OutSample: TSignal); // virtual;
    begin
      OutSample := InSample; // Wire bridge .. this is an abstract filter doing nothing.
    end;



{ ========
   THuovilainenMoog = class( TMoogFilter)
  private
    FStage0     : TSignal;
    FStage1     : TSignal;
    FStage2     : TSignal;
    FStage3     : TSignal;
    FStageTanh0 : TSignal;
    FStageTanh1 : TSignal;
    FStageTanh2 : TSignal;
    FDelay0     : TSignal;
    FDelay1     : TSignal;
    FDelay2     : TSignal;
    FDelay3     : TSignal;
    FDelay4     : TSignal;
    FDelay5     : TSignal;
    FTune       : TSignal;
    FAcr        : TSignal;
    FResQuad    : TSignal;
  protected
}

    procedure   THuovilainenMoog.SetResonance( aValue: TSignal); // virtual;]begin
    begin
      FResonance := aValue;
      FResQuad   := FResonance * FAcr;
    end;


    procedure   THuovilainenMoog.SetCutoff( aValue: TSignal); // virtual;
    var
      fc  : TSignal;
      f   : TSignal;
      fc2 : TSignal;
      fc3 : TSignal;
      fcr : TSignal;
    begin
      FCutoff  := aValue;
      fc       := FCutoff / FSampleRate;
      f        := fc * 0.5;                                   // oversampled twice
      fc2      := fc * fc;
      fc3      := fc2 * fc;
      fcr      := 1.8730 * fc3 + 0.4955 * fc2 - 0.6490 * fc + 0.9988;
      FAcr     := -3.9364 * fc2 + 1.8409 * fc + 0.9968;
      FTune    := ( 1.0 - exp( - (( 2 * PI) * f * fcr))) / FThermal;
      FResQuad := FResonance * FAcr;
    end;


//  public

    constructor THuovilainenMoog.Create( aSampleRate: TSignal); // override;
    begin
      FSampleRate := aSampleRate;
      CutOff      := 329.6;
      Resonance   := 0.10;
    end;


    procedure   THuovilainenMoog.Process( InSample: TSignal; var OutSample: TSignal); // virtual;
    var
      InVal  : TSignal;
      PreVal : TSignal;
    begin
      PreVal      := Normalize( InSample);

      InVal       := Normalize( PreVal - FResQuad * ( FDelay5 - FComp * InSample));
      FStage0     := Normalize( FDelay0 + FTune * LookupTanh( InVal * FThermal) - FStageTanh0);
      FDelay0     := FStage0;
      FStageTanh0 := Normalize( LookupTanh( FStage0 * FThermal));
      FStage1     := Normalize( FDelay1 + FTune * ( FStageTanh0 - FStageTanh1));
      FDelay1     := FStage1;
      FStageTanh1 := Normalize( LookupTanh( FStage1 * FThermal));
      FStage2     := Normalize( FDelay2 + FTune * ( FStageTanh1 - FStageTanh2));
      FDelay2     := FStage2;
      FStageTanh2 := Normalize( LookupTanh( FStage2 * FThermal));
      FStage3     := Normalize( FDelay3 + FTune * ( FStageTanh2 - FDelay3 * FThermal));
      FDelay3     := FStage3;
      FDelay5     := 0.5 * ( FStage3 + FDelay4);
      FDelay4     := FStage3;

      InVal       := Normalize( PreVal - FResQuad * ( FDelay5 - FComp * InSample));
      FStage0     := Normalize( FDelay0 + FTune * LookupTanh( InVal * FThermal) - FStageTanh0);
      FDelay0     := FStage0;
      FStageTanh0 := Normalize( LookupTanh( FStage0 * FThermal));
      FStage1     := Normalize( FDelay1 + FTune * ( FStageTanh0 - FStageTanh1));
      FDelay1     := FStage1;
      FStageTanh1 := Normalize( LookupTanh( FStage1 * FThermal));
      FStage2     := Normalize( FDelay2 + FTune * ( FStageTanh1 - FStageTanh2));
      FDelay2     := FStage2;
      FStageTanh2 := Normalize( LookupTanh( FStage2 * FThermal));
      FStage3     := Normalize( FDelay3 + FTune * ( FStageTanh2 - FDelay3 * FThermal));
      FDelay3     := FStage3;
      FDelay5     := 0.5 * ( FStage3 + FDelay4);
      FDelay4     := FStage3;

      OutSample   := LookupTanh( FDelay5);
    end;


end.

