unit SVFUnit;

{

   COPYRIGHT 2016 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl


  original code :

    Original source : http://www.musicdsp.org/showone.php?id=142

    no copyright notice was found for that code.
}

interface

uses

  System.Math,

  KnobsUtils, KnobsConversions, FormantControl;


type

  TSvfMode = (
    svfmLowPass,
    svfmBandPass,
    svfmBandReject,
    svfmHighPass
  );


  TSvfChamberlin = class
  private
    FInternalQ    : TSignal;
    FQ            : TSignal;
    FInternalFreq : TSignal;
    FFrequency    : TSignal;
    FSampleRate   : TSignal;
    FReciprocalSR : TSignal;
    FDelay1       : TSignal;
    FDelay2       : TSignal;
  private
    procedure   FixFrequency;
    procedure   FixQ;
  private
    procedure   SetFrequency   ( const aValue : TSignal);
    procedure   SetSampleRate  ( const aValue : TSignal);
    procedure   SetQ           ( const aValue : TSignal);
  public
    constructor Create;
    procedure   Process( const anInput : TSignal; var aLowPass, aBandPass, aBandReject, aHighPass : TSignal);
  public
    property    Frequency    : TSignal read FFrequency    write SetFrequency;
    property    SampleRate   : TSignal read FSampleRate   write SetSampleRate;
    property    Q            : TSignal read FQ            write SetQ;
  end;


  TSvfChamberlinRelative = class
  private
    FInternalQ    : TSignal;
    FQ            : TSignal;
    FInternalFreq : TSignal;
    FFrequency    : TSignal;    // in Units
    FLFM          : TSignal;
    FSampleRate   : TSignal;
    FReciprocalSR : TSignal;
    FDelay1       : TSignal;
    FDelay2       : TSignal;
    FRelFrequency : TSignal;    // in Units
    FRelQ         : TSignal;
    FMainQ        : TSignal;
  private
    procedure   FixFrequency;
    procedure   FixQ;
  private
    procedure   SetFrequency   ( const aValue : TSignal);
    procedure   SetLFM         ( const aValue : TSignal);
    procedure   SetSampleRate  ( const aValue : TSignal);
    procedure   SetQ           ( const aValue : TSignal);
    procedure   SetRelFrequency( const aValue : TSignal);
    procedure   SetRelQ        ( const aValue : TSignal);
    procedure   SetMainQ       ( const aValue : TSignal);
  public
    constructor Create;
    procedure   Process( const anInput : TSignal; var aLowPass, aBandPass, aBandReject, aHighPass : TSignal);
  public
    property    Frequency    : TSignal read FFrequency    write SetFrequency;
    property    LFM          : TSignal read FLFM          write SetLFM;
    property    SampleRate   : TSignal read FSampleRate   write SetSampleRate;
    property    Q            : TSignal read FQ            write SetQ;
    property    RelFrequency : TSignal read FRelFrequency write SetRelFrequency;
    property    RelQ         : TSignal read FRelQ         write SetRelQ;
    property    MainQ        : TSignal read FMainQ        write SetMainQ;
  end;


  TSvfChamberlinChain = class
  private
    FFilters    : array of TSvfChamberlin;
    FLevel      : TSignal;
    FFrequency  : TSignal;
    FSampleRate : TSignal;
    FQ          : TSignal;
  private
    function    GetCount: Integer;
    procedure   SetFrequency ( const aValue : TSignal);
    procedure   SetSampleRate( const aValue : TSignal);
    procedure   SetQ         ( const aValue : TSignal);
  public
    constructor Create( aCount: Integer);
    destructor  Destroy;                                                                                       override;
    function    Process( const anInput : TSignal): TSignal;                                           virtual; abstract;
  public
    property    Count      : Integer read GetCount;
    property    Level      : TSignal read FLevel      write FLevel;
    property    Frequency  : TSignal read FFrequency  write SetFrequency;
    property    SampleRate : TSignal read FSampleRate write SetSampleRate;
    property    Q          : TSignal read FQ          write SetQ;
  end;


  TSvfChamberlinChainLowPass = class( TSvfChamberlinChain)
  public
    function    Process( const anInput : TSignal): TSignal;                                                    override;
  end;


  TSvfChamberlinChainBandPass = class( TSvfChamberlinChain)
  public
    function    Process( const anInput : TSignal): TSignal;                                                    override;
  end;


  TSvfChamberlinChainBandReject = class( TSvfChamberlinChain)
  public
    function    Process( const anInput : TSignal): TSignal;                                                    override;
  end;


  TSvfChamberlinChainHighPass = class( TSvfChamberlinChain)
  public
    function    Process( const anInput : TSignal): TSignal;                                                    override;
  end;


  TSvfChamberlinBank = class
  strict private
  const
    FILTER_COUNT = 17;
  private
    FLocked   : Boolean;
    FFilters  : array of TSvfChamberlinChain;
    FFis      : Boolean;
  private
    function    GetCount: Integer;
    procedure   SetLevel    ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetFrequency( const anIndex: Integer; const aValue: TSignal);
    procedure   SetQ        ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetLevels( const aValue: TSignal);
    procedure   SetQs( const aValue: TSignal);
  public
    constructor Create( aSampleRate: TSignal; aLength, aSteepness: Integer; aModeLow, aModeMid, aModeHigh: TSvfMode);
    destructor  Destroy;                                                                                       override;
    procedure   Process( const anInput : TSignal; out aValue : TSignal);
    procedure   ProcessSplit  ( const anInput : TSignal; var out1, out2, out3, out4, out5, out6, out7, out8, out9, out10, out11, out12, out13, out14, out15, out16, out17 : TSignal);
    procedure   ProcessCombine( const anIn1, anIn2, anIn3, anIn4, anIn5, anIn6, anIn7, anIn8, anIn9, anIn10, anIn11, anIn12, anIn13, anIn14, anIn15, anIn16, anIn17: TSignal; var anOut: TSignal);
  public
    property    Count                              : Integer read GetCount;
    property    Level    [ const anIndex: Integer] : TSignal                write SetLevel;
    property    Frequency[ const anIndex: Integer] : TSignal                write SetFrequency;
    property    Q        [ const anIndex: Integer] : TSignal                write SetQ;
    property    Levels                             : TSignal                write SetLevels;
    property    Qs                                 : TSignal                write SetQs;
    property    FIs                                : Boolean read FFis      write FFIs;
  end;


  TSvfFormantFilter = class
  private
    FFormantCollection : TFormantCollection;
    FLocked            : Boolean;
    FSampleRate        : TSignal;
    FFilters           : array of TSvfChamberlin;
    FLevels            : array of TSignal;
    FCurrentSet        : Integer;
    FFormantCount      : Integer;
    FVowelIndex        : TSignal;
    FVowel             : TVowel;
    FFis               : Boolean;
  private
    function    GetSetCount : Integer;
    procedure   SetCurrentSet( anIndex: Integer);
    procedure   SetVowelIndex( anIndex: TSignal);
    procedure   SetSampleRate( aValue: TSignal);
  private
    procedure   RecreateFilter;
    procedure   UpdateFilter;
  public
    constructor Create( aFormantCollection: TFormantCollection; aSampleRate: TSignal);
    destructor  Destroy;                                                                                       override;
    procedure   Process( const anInput, aVowelIndex: TSignal; out aValue: TSignal);
  public
    property    SetCount   : Integer read GetSetCount;
    property    CurrentSet : Integer read FCurrentSet write SetCurrentSet;
    property    SampleRate : TSignal read FSampleRate write SetSampleRate;
    property    FIs        : Boolean read FFis        write FFIs;
  end;


  TSvfModalFilter = class
  private
    FModalCollection : TModalCollection;
    FMaxVoices       : Integer;
    FLocked          : Boolean;
    FSampleRate      : TSignal;
    FFilters         : array of TSvfChamberlinRelative;
    FLevels          : array of TSignal;
    FCurrent         : Integer;
    FFis             : Boolean;
  private
    function    GetCount       : Integer;
    function    GetFilterCount : Integer;
    function    GetName        : string;
    procedure   SetLevel    ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetFrequency( const anIndex: Integer; const aValue: TSignal);
    procedure   SetLFM      ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetQ        ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetMainQ    ( const aValue: TSignal);
    procedure   SetLevels( const aValue: TSignal);
    procedure   SetQs( const aValue: TSignal);
    procedure   SetCurrent( anIndex: Integer);
    procedure   SetSampleRate( aValue: TSignal);
  private
    procedure   RecreateFilter;
  public
    constructor Create( aModalCollection: TModalCollection; aSampleRate: TSignal; aMaxVoices : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Process( const anInput: TSignal; out aValue: TSignal);
  public
    property    Count                              : Integer read GetCount;
    property    Current                            : Integer read FCurrent       write SetCurrent;
    property    SampleRate                         : TSignal read FSampleRate    write SetSampleRate;
    property    FilterCount                        : Integer read GetFilterCount;
    property    Name                               : string  read GetName;
    property    Level    [ const anIndex: Integer] : TSignal                     write SetLevel;
    property    Frequency[ const anIndex: Integer] : TSignal                     write SetFrequency; // in Units
    property    LFM      [ const anIndex: Integer] : TSignal                     write SetLFM;
    property    Q        [ const anIndex: Integer] : TSignal                     write SetQ;
    property    Levels                             : TSignal                     write SetLevels;
    property    Qs                                 : TSignal                     write SetQs;
    property    MainQ                              : TSignal                     write SetMainQ;
    property    FIs                                : Boolean read FFis           write FFIs;
  end;


  TSvfChladniFilter = class
  private
    FMaxVoices  : Integer;
    FLocked     : Boolean;
    FSampleRate : TSignal;
    FFilters    : array of TSvfChamberlinRelative;
    FLevels     : array of TSignal;
    FFis        : Boolean;
  private
    function    GetFilterCount : Integer;
    procedure   SetLevel        ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetRelFrequency ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetQ            ( const anIndex: Integer; const aValue: TSignal);
    procedure   SetFrequency    ( const aValue: TSignal);
    procedure   SetLFM          ( const aValue: TSignal);
    procedure   SetMainQ        ( const aValue: TSignal);
    procedure   SetLevels       ( const aValue: TSignal);
    procedure   SetQs           ( const aValue: TSignal);
    procedure   SetSampleRate   ( const aValue: TSignal);
  private
    procedure   RecreateFilter;
  public
    constructor Create( aSampleRate: TSignal; aMaxVoices : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Process( const anInput: TSignal; out aValue: TSignal);
  public
    property    SampleRate                            : TSignal read FSampleRate    write SetSampleRate;
    property    FilterCount                           : Integer read GetFilterCount;
    property    Level       [ const anIndex: Integer] : TSignal                     write SetLevel;
    property    RelFrequency[ const anIndex: Integer] : TSignal                     write SetRelFrequency; // in Units
    property    Q           [ const anIndex: Integer] : TSignal                     write SetQ;
    property    Frequency                             : TSignal                     write SetFrequency;    // in Units
    property    LFM                                   : TSignal                     write SetLFM;
    property    Levels                                : TSignal                     write SetLevels;
    property    Qs                                    : TSignal                     write SetQs;
    property    MainQ                                 : TSignal                     write SetMainQ;
    property    FIs                                   : Boolean read FFis           write FFIs;
  end;


implementation


uses

  sysutils;

const

  MIN_SAMPRATE = 44100.0;                 // Minimal sample rate, in Hz
  NORMAL_LIMIT = 1e-20;                   // Value below which the Normalize functions will truncate the value to zero {  -400 dB}


  function  Normalize( const aValue: TSignal): TSignal; inline;
  // This actually checks if aValue is zero, a denormal or it's absulote value being < 1e-20
  // and it treats all those cases as if they were zero.
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
    if (( PWordArray( @ aValue)^[ 3] and $7ff0) = 0) or ( Abs( aValue) < NORMAL_LIMIT)
    then Result := 0
    else Result := aValue;
  end;


{ ========
  TSvfChamberlin = class
  private
    FInternalQ    : TSignal;
    FQ            : TSignal;
    FInternalFreq : TSignal;
    FFrequency    : TSignal;
    FSampleRate   : TSignal;
    FReciprocalSR : TSignal;
    FDelay1       : TSignal;
    FDelay2       : TSignal;
    FRelFrequency : TSignal;
    FRelQ         : TSignal;
    FMainQ        : TSignal;
  public
    property    Frequency    : TSignal read FFrequency    write SetFrequency;
    property    SampleRate   : TSignal read FSampleRate   write SetSampleRate;
    property    Q            : TSignal read FQ            write SetQ;
    property    RelFrequency : TSignal read FRelFrequency write SetRelFrequency;
    property    RelQ         : TSignal read FRelQ         write SetRelQ;
    property    MainQ        : TSignal read FMainQ        write SetMainQ;
  private
}

    procedure   TSvfChamberlin.FixFrequency;
    var
      aValue: TSignal;
    begin
      aValue := FFrequency;

      if aValue > FSampleRate / 6
      then aValue := FSampleRate / 6
      else if aValue < 0
      then aValue := 0;

      FInternalFreq := 2.0 * LookupSine( 0.5 * aValue * FReciprocalSR);
    end;


    procedure   TSvfChamberlin.FixQ;
    var
      aValue: TSignal;
    begin
      aValue := FQ;

      if aValue < 0.707
      then aValue := 0.707;

      FInternalQ := 1 / aValue;
    end;


//  private

    procedure   TSvfChamberlin.SetFrequency( const aValue : TSignal);
    begin
      if aValue <> FFrequency
      then begin
        FFrequency := aValue;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlin.SetSampleRate( const aValue : TSignal);
    var
      aRate : TSignal;
    begin
      if aValue <= MIN_SAMPRATE
      then aRate := MIN_SAMPRATE
      else aRate := aValue;

      if aRate <> FSampleRate
      then begin
        FSampleRate   := aValue;
        FReciprocalSR := 1 / FSampleRate;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlin.SetQ( const aValue : TSignal);
    begin
      if aValue <> FQ
      then begin
        FQ := aValue;
        FixQ;
      end;
    end;


//  public

    constructor TSvfChamberlin.Create;
    begin
      inherited;
      SampleRate    := MIN_SAMPRATE;         // First set sample rate, then Frequency
      Frequency     := 329.6;                // in Hz
      Q             :=   1.0;                // Q can be set at arbitrary time
      FixFrequency;
      FixQ;
    end;


    procedure   TSvfChamberlin.Process( const anInput : TSignal; var aLowPass, aBandPass, aBandReject, aHighPass : TSignal);
    begin
      aLowPass    := Normalize( FDelay2 + FInternalFreq * FDelay1);
      aHighPass   := Normalize( anInput - aLowPass - FInternalQ * FDelay1);
      aBandPass   := Normalize( FInternalFreq * aHighPass + FDelay1);
      aBandReject := Normalize( aHighPass + aLowPass);
      FDelay1     := aBandPass;
      FDelay2     := aLowPass;
    end;


{ ========
  TSvfChamberlinRelative = class
  private
    FInternalQ    : TSignal;
    FQ            : TSignal;
    FInternalFreq : TSignal;
    FFrequency    : TSignal;    // in Units
    FLFM          : TSignal;
    FSampleRate   : TSignal;
    FReciprocalSR : TSignal;
    FDelay1       : TSignal;
    FDelay2       : TSignal;
    FRelFrequency : TSignal;    // in Units
    FRelQ         : TSignal;
    FMainQ        : TSignal;
  public
    property    Frequency    : TSignal read FFrequency    write SetFrequency;
    property    LFM          : TSignal read FLFM          write SetLFM;
    property    SampleRate   : TSignal read FSampleRate   write SetSampleRate;
    property    Q            : TSignal read FQ            write SetQ;
    property    RelFrequency : TSignal read FRelFrequency write SetRelFrequency;
    property    RelQ         : TSignal read FRelQ         write SetRelQ;
    property    MainQ        : TSignal read FMainQ        write SetMainQ;
  private
}

    procedure   TSvfChamberlinRelative.FixFrequency;
    var
      aValue: TSignal;
    begin
      aValue := ( 1 + 0.5 * LFM) * LookupFrequency( FFrequency + FRelFrequency - NoteNumberToUnits( MiddleNote));

      if aValue > FSampleRate / 6
      then aValue := FSampleRate / 6
      else if aValue < 0
      then aValue := 0;

      FInternalFreq := 2.0 * LookupSine( 0.5 * aValue * FReciprocalSR);
    end;


    procedure   TSvfChamberlinRelative.FixQ;
    var
      aValue: TSignal;
    begin
      aValue := FQ * FRelQ * FmainQ;

      if aValue < 0.707
      then aValue := 0.707;

      FInternalQ := 1 / aValue;
    end;


//  private

    procedure   TSvfChamberlinRelative.SetFrequency( const aValue : TSignal);
    begin
      if aValue <> FFrequency
      then begin
        FFrequency := aValue;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetLFM( const aValue : TSignal);
    begin
      if aValue <> FLFM
      then begin
        FLFM := aValue;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetSampleRate( const aValue : TSignal);
    var
      aRate : TSignal;
    begin
      if aValue <= MIN_SAMPRATE
      then aRate := MIN_SAMPRATE
      else aRate := aValue;

      if aRate <> FSampleRate
      then begin
        FSampleRate   := aValue;
        FReciprocalSR := 1 / FSampleRate;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetQ( const aValue : TSignal);
    begin
      if aValue <> FQ
      then begin
        FQ := aValue;
        FixQ;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetRelFrequency( const aValue : TSignal);
    begin
      if aValue <> FRelFrequency
      then begin
        FRelFrequency := aValue;
        FixFrequency;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetRelQ( const aValue : TSignal);
    begin
      if aValue <> FRelQ
      then begin
        FRelQ := aValue;
        FixQ;
      end;
    end;


    procedure   TSvfChamberlinRelative.SetMainQ( const aValue : TSignal);
    begin
      if aValue <> FMainQ
      then begin
        FMainQ := aValue;
        FixQ;
      end;
    end;


//  public

    constructor TSvfChamberlinRelative.Create;
    begin
      inherited;
      FRelFrequency := 1.0;
      FRelQ         := 1.0;
      FMainQ        := 1.0;
      SampleRate    := MIN_SAMPRATE;         // First set sample rate, then Frequency
      Frequency     := 329.6;                // in Hz
      Q             :=   1.0;                // Q can be set at arbitrary time
      FixFrequency;
      FixQ;
    end;


    procedure   TSvfChamberlinRelative.Process( const anInput : TSignal; var aLowPass, aBandPass, aBandReject, aHighPass : TSignal);
    begin
      aLowPass    := Normalize( FDelay2 + FInternalFreq * FDelay1);
      aHighPass   := Normalize( anInput - aLowPass - FInternalQ * FDelay1);
      aBandPass   := Normalize( FInternalFreq * aHighPass + FDelay1);
      aBandReject := Normalize( aHighPass + aLowPass);
      FDelay1     := aBandPass;
      FDelay2     := aLowPass;
    end;


{ ========
  TSvfChamberlinChain = class
  private
    FFilters    : array of TSvfChamberlin;
    FLevel      : TSignal;
    FFrequency  : TSignal;
    FSampleRate : TSignal;
    FQ          : TSignal;
  public
    property    Count      : Integer read GetCount;
    property    Level      : TSignal read FLevel      write FLevel;
    property    Frequency  : TSignal read FFrequency  write SetFrequency;
    property    SampleRate : TSignal read FSampleRate write SetSampleRate;
    property    Q          : TSignal read FQ          write SetQ;
  private
}

    function    TSvfChamberlinChain.GetCount: Integer;
    begin
      Result := Length( FFilters);
    end;


    procedure   TSvfChamberlinChain.SetFrequency( const aValue : TSignal);
    var
      i : Integer;
    begin
      if aValue <> FFrequency
      then begin
        FFrequency := aValue;

        for i:= 0 to Count - 1
        do FFilters[ i].Frequency := FFrequency;
      end;
    end;


    procedure   TSvfChamberlinChain.SetSampleRate( const aValue : TSignal);
    var
      i : Integer;
    begin
      if aValue <> FSampleRate
      then begin
        FSampleRate := aValue;

        for i:= 0 to Count - 1
        do FFilters[ i].SampleRate := FSampleRate;
      end;
    end;


    procedure   TSvfChamberlinChain.SetQ( const aValue : TSignal);
    var
      i : Integer;
    begin
      if aValue <> FQ
      then begin
        FQ := aValue;

        for i:= 0 to Count - 1
        do FFilters[ i].Q := FQ / Count;
      end;
    end;


//  public

    constructor TSvfChamberlinChain.Create( aCount: Integer);
    var
      i : Integer;
    begin
      inherited Create;
      SetLength( FFilters, aCount);

      for i := 0 to Count - 1
      do FFilters[ i] := TSvfChamberlin.Create;
    end;


    destructor  TSvfChamberlinChain.Destroy; // override;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FreeAndNil( FFilters[ i]);

      inherited;
    end;


{ ========
  TSvfChamberlinChainLowPass = class( TSvfChamberlinChain)
  public
}

    function    TSvfChamberlinChainLowPass.Process( const anInput : TSignal): TSignal; // override;
    var
      aBandPass   : TSignal;
      aBandReject : TSignal;
      aHighPass   : TSignal;
      i           : Integer;
    begin
      FFilters[ 0].Process( Normalize( FLevel * anInput), Result, aBandPass, aBandReject, aHighPass);
      for i := 1 to Count - 1

      do FFilters[ i].Process( Result, Result, aBandPass, aBandReject, aHighPass);
    end;


{ ========
  TSvfChamberlinChainBandPass = class( TSvfChamberlinChain)
  public
}

    function    TSvfChamberlinChainBandPass.Process( const anInput : TSignal): TSignal; // override;
    var
      aLowPass    : TSignal;
      aBandReject : TSignal;
      aHighPass   : TSignal;
      i           : Integer;
    begin
      FFilters[ 0].Process( Normalize( FLevel * anInput), aLowPass, Result, aBandReject, aHighPass);

      for i := 1 to Count - 1
      do FFilters[ i].Process( Result, aLowPass, Result, aBandReject, aHighPass);
    end;


{ ========
  TSvfChamberlinChainBandReject = class( TSvfChamberlinChain)
  public
}

    function    TSvfChamberlinChainBandReject.Process( const anInput : TSignal): TSignal; // override;
    var
      aLowPass  : TSignal;
      aBandPass : TSignal;
      aHighPass : TSignal;
      i         : Integer;
    begin
      FFilters[ 0].Process( Normalize( FLevel * anInput), aLowPass, aBandPass, Result, aHighPass);

      for i := 1 to Count - 1
      do FFilters[ i].Process( Result, aLowPass, aBandPass, Result, aHighPass);
    end;


{ ========
  TSvfChamberlinChainHighPass = class( TSvfChamberlinChain)
  public
}

    function    TSvfChamberlinChainHighPass.Process( const anInput : TSignal): TSignal; // override;
    var
      aLowPass    : TSignal;
      aBandPass   : TSignal;
      aBandReject : TSignal;
      i           : Integer;
    begin
      FFilters[ 0].Process( Normalize( FLevel * anInput), aLowPass, aBandPass, aBandReject, Result);

      for i := 1 to Count - 1
      do FFilters[ i].Process( Result, aLowPass, aBandPass, aBandReject, Result);
    end;


{ ========
  TSvfChamberlinBank = class
  private
    FLocked   : Boolean;
    FFilters  : array of TSvfChamberlinChain;
    FFis      : Boolean;
  public
    property    Count                              : Integer read GetCount;
    property    Level    [ const anIndex: Integer] : TSignal                write SetLevel;
    property    Frequency[ const anIndex: Integer] : TSignal                write SetFrequency;
    property    Q        [ const anIndex: Integer] : TSignal                write SetQ;
    property    Levels                             : TSignal                write SetLevels;
    property    Qs                                 : TSignal                write SetQs;
    property    FIs                                : Boolean                write FFIs;
  private
}

    function    TSvfChamberlinBank.GetCount: Integer;
    begin
      Result := Length( FFilters);
    end;


    procedure   TSvfChamberlinBank.SetLevel( const anIndex: Integer; const aValue: TSignal);
    begin
      FFilters[ anIndex].Level := aValue;
    end;


    procedure   TSvfChamberlinBank.SetFrequency( const anIndex: Integer; const aValue: TSignal);
    begin
      FFilters[ anIndex].Frequency := aValue;
    end;


    procedure   TSvfChamberlinBank.SetQ( const anIndex: Integer; const aValue: TSignal);
    begin
      FFilters[ anIndex].Q := aValue;
    end;


    procedure   TSvfChamberlinBank.SetLevels( const aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FFilters[ i].Level := aValue;
    end;


    procedure   TSvfChamberlinBank.SetQs( const aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FFilters[ i].Q := aValue;
    end;


//  public

    constructor TSvfChamberlinBank.Create( aSampleRate: TSignal; aLength, aSteepness: Integer; aModeLow, aModeMid, aModeHigh: TSvfMode);

      function MakeFilter( aMode: TSvfMode; aSteepness: Integer): TSvfChamberlinChain;
      begin
        case aMode of
          svfmLowPass  : Result := TSvfChamberlinChainLowPass   .Create( aSteepness);
          svfmBandPass : Result := TSvfChamberlinChainBandPass  .Create( aSteepness);
          svfmHighPass : Result := TSvfChamberlinChainHighPass  .Create( aSteepness);
          else           Result := TSvfChamberlinChainBandReject.Create( aSteepness);
        end;
      end;

    var
      i : Integer;
    begin
      Assert( aSteepness > 0);
      FLocked := true;
      inherited Create;
      SetLength( FFilters, aLength);;

      for i := 0 to aLength - 1
      do begin
        if i = 0
        then FFilters[ i] := MakeFilter( aModeLow , aSteepness)
        else if i = aLength - 1
        then FFilters[ i] := MakeFilter( aModeHigh, aSteepness)
        else FFilters[ i] := MakeFilter( aModeMid , aSteepness);

        Level    [ i] := 0;
        Frequency[ i] := 329.6;
        FFilters[ i].SampleRate := aSampleRate;
      end;
      Qs := 0.5;

      FLocked := False;
    end;


    destructor  TSvfChamberlinBank.Destroy; // override;
    var
      i: Integer;
    begin
      FLocked := True;

      for i := 0 to Count - 1
      do FreeAndNil( FFilters[ i]);

      inherited;
    end;


    procedure   TSvfChamberlinBank.Process( const anInput : TSignal; out aValue : TSignal);
    var
      i : Integer;
    begin
      aValue := 0.0;

      if not FLocked
      then begin
        for i := 0 to Count - 1
        do begin
          if FIs
          then begin
            if Odd( i)
            then aValue := Normalize( aValue - FFilters[ i].Process( anInput))
            else aValue := Normalize( aValue + FFilters[ i].Process( anInput));
          end
          else aValue := Normalize( aValue + FFilters[ i].Process( anInput));
        end;
      end;
    end;


    procedure   TSvfChamberlinBank.ProcessSplit( const anInput : TSignal; var out1, out2, out3, out4, out5, out6, out7, out8, out9, out10, out11, out12, out13, out14, out15, out16, out17 : TSignal);
    begin
      if FLocked
      then begin
        out1  := 0.0;
        out2  := 0.0;
        out3  := 0.0;
        out4  := 0.0;
        out5  := 0.0;
        out7  := 0.0;
        out7  := 0.0;
        out8  := 0.0;
        out9  := 0.0;
        out10 := 0.0;
        out11 := 0.0;
        out12 := 0.0;
        out13 := 0.0;
        out14 := 0.0;
        out15 := 0.0;
        out16 := 0.0;
        out17 := 0.0;
      end
      else begin
        if FIs
        then begin
          out1  :=   Normalize( FFilters[  0].Process( anInput));
          out2  := - Normalize( FFilters[  1].Process( anInput));
          out3  :=   Normalize( FFilters[  2].Process( anInput));
          out4  := - Normalize( FFilters[  3].Process( anInput));
          out5  :=   Normalize( FFilters[  4].Process( anInput));
          out7  := - Normalize( FFilters[  5].Process( anInput));
          out7  :=   Normalize( FFilters[  6].Process( anInput));
          out8  := - Normalize( FFilters[  7].Process( anInput));
          out9  :=   Normalize( FFilters[  8].Process( anInput));
          out10 := - Normalize( FFilters[  9].Process( anInput));
          out11 :=   Normalize( FFilters[ 10].Process( anInput));
          out12 := - Normalize( FFilters[ 11].Process( anInput));
          out13 :=   Normalize( FFilters[ 12].Process( anInput));
          out14 := - Normalize( FFilters[ 13].Process( anInput));
          out15 :=   Normalize( FFilters[ 14].Process( anInput));
          out16 := - Normalize( FFilters[ 15].Process( anInput));
          out17 :=   Normalize( FFilters[ 16].Process( anInput));
        end
        else begin
          out1  :=   Normalize( FFilters[  0].Process( anInput));
          out2  :=   Normalize( FFilters[  1].Process( anInput));
          out3  :=   Normalize( FFilters[  2].Process( anInput));
          out4  :=   Normalize( FFilters[  3].Process( anInput));
          out5  :=   Normalize( FFilters[  4].Process( anInput));
          out7  :=   Normalize( FFilters[  5].Process( anInput));
          out7  :=   Normalize( FFilters[  6].Process( anInput));
          out8  :=   Normalize( FFilters[  7].Process( anInput));
          out9  :=   Normalize( FFilters[  8].Process( anInput));
          out10 :=   Normalize( FFilters[  9].Process( anInput));
          out11 :=   Normalize( FFilters[ 10].Process( anInput));
          out12 :=   Normalize( FFilters[ 11].Process( anInput));
          out13 :=   Normalize( FFilters[ 12].Process( anInput));
          out14 :=   Normalize( FFilters[ 13].Process( anInput));
          out15 :=   Normalize( FFilters[ 14].Process( anInput));
          out16 :=   Normalize( FFilters[ 15].Process( anInput));
          out17 :=   Normalize( FFilters[ 16].Process( anInput));
        end;
      end;
    end;


    procedure   TSvfChamberlinBank.ProcessCombine( const anIn1, anIn2, anIn3, anIn4, anIn5, anIn6, anIn7, anIn8, anIn9, anIn10, anIn11, anIn12, anIn13, anIn14, anIn15, anIn16, anIn17: TSignal; var anOut: TSignal);
    begin
      if FLocked
      then anOut  := 0.0
      else begin
        if FIs
        then
          anOut :=
            + Normalize( FFilters[  0].Process( anIn1 ))
            - Normalize( FFilters[  1].Process( anIn2 ))
            + Normalize( FFilters[  2].Process( anIn3 ))
            - Normalize( FFilters[  3].Process( anIn4 ))
            + Normalize( FFilters[  4].Process( anIn5 ))
            - Normalize( FFilters[  5].Process( anIn6 ))
            + Normalize( FFilters[  6].Process( anIn7 ))
            - Normalize( FFilters[  7].Process( anIn8 ))
            + Normalize( FFilters[  8].Process( anIn9 ))
            - Normalize( FFilters[  9].Process( anIn10))
            + Normalize( FFilters[ 10].Process( anIn11))
            - Normalize( FFilters[ 11].Process( anIn12))
            + Normalize( FFilters[ 12].Process( anIn13))
            - Normalize( FFilters[ 13].Process( anIn14))
            + Normalize( FFilters[ 14].Process( anIn15))
            - Normalize( FFilters[ 15].Process( anIn16))
            + Normalize( FFilters[ 16].Process( anIn17))
        else
          anOut :=
            + Normalize( FFilters[  0].Process( anIn1 ))
            + Normalize( FFilters[  1].Process( anIn2 ))
            + Normalize( FFilters[  2].Process( anIn3 ))
            + Normalize( FFilters[  3].Process( anIn4 ))
            + Normalize( FFilters[  4].Process( anIn5 ))
            + Normalize( FFilters[  5].Process( anIn6 ))
            + Normalize( FFilters[  6].Process( anIn7 ))
            + Normalize( FFilters[  7].Process( anIn8 ))
            + Normalize( FFilters[  8].Process( anIn9 ))
            + Normalize( FFilters[  9].Process( anIn10))
            + Normalize( FFilters[ 10].Process( anIn11))
            + Normalize( FFilters[ 11].Process( anIn12))
            + Normalize( FFilters[ 12].Process( anIn13))
            + Normalize( FFilters[ 13].Process( anIn14))
            + Normalize( FFilters[ 14].Process( anIn15))
            + Normalize( FFilters[ 15].Process( anIn16))
            + Normalize( FFilters[ 16].Process( anIn17));
        end;
    end;


{ ========
  TSvfFormantFilter = class
  private
    FFormantCollection : TFormantCollection;
    FLocked            : Boolean;
    FSampleRate        : TSignal;
    FFilters           : array of TSvfChamberlin;
    FLevels            : array of TSignal;
    FCurrentSet        : Integer;
    FFormantCount      : Integer;
    FVowelIndex        : TSignal;
    FVowel             : TVowel;
    FFis               : Boolean;
  public
    property    SetCount   : Integer read GetSetCount;
    property    CurrentSet : Integer read FCurrentSet write SetCurrentSet;
    property    SampleRate : TSignal read FSampleRate write SetSampleRate;
    property    FIs        : Boolean read FFis        write FFIs;
  private
}

    function    TSvfFormantFilter.GetSetCount: Integer;
    begin
      if Assigned( FFormantCollection)
      then Result := FFormantCollection.SetCount
      else Result := 0;
    end;


    procedure   TSvfFormantFilter.SetCurrentSet( anIndex: Integer);
    begin
      if SetCount > 0
      then begin
        anIndex := Clip( anIndex, 0, SetCount - 1);

        if anIndex <> FCurrentSet
        then begin
          FCurrentSet := anIndex;
          RecreateFilter;
        end;
      end;
    end;


    procedure   TSvfFormantFilter.SetVowelIndex( anIndex: TSignal);
    begin
      if Assigned( FFormantCollection) and ( SetCount > 0)
      then begin
        if anIndex <> FVowelIndex
        then begin
          FVowelIndex := anIndex;
          FFormantCollection.Interpolate( FCurrentSet, FVowelIndex, FVowel);
          UpdateFilter;
        end;
      end;
    end;


    procedure   TSvfFormantFilter.SetSampleRate( aValue: TSignal);
    var
      i : Integer;
    begin
      if aValue <> FSampleRate
      then begin
        FSampleRate := aValue;

        for i := 0 to Length( FFilters) - 1
        do FFilters[ i].SampleRate := FSampleRate;
      end;
    end;


//  private

    procedure   TSvfFormantFilter.RecreateFilter;
    var
      i : Integer;
    begin
      FLocked := True;

      try
        for i := 0 to Length( FFilters) - 1
        do FreeAndNil( FFilters[ i]);

        if SetCount > 0
        then FFormantCount := FFormantCollection.FormantCount[ FCurrentSet, 0]
        else FFormantCount := 0;

        SetLength( FFilters       , FFormantCount);
        SetLength( FLevels        , FFormantCount);
        SetLength( FVowel.Formants, FFormantCount);

        for i := 0 to Length( FFilters) - 1
        do FFilters[ i] := TSvfChamberlin.Create;

        FVowelIndex := -1;
        SetVowelIndex( 0);
      finally
        FLocked := False;
      end;
    end;


    procedure   TSvfFormantFilter.UpdateFilter;
    var
      i : Integer;
    begin
      for i := 0 to FFormantCount - 1
      do begin
        FFilters[ i].Frequency := FVowel.Formants[ i].Freg;
        FFilters[ i].Q         := FVowel.Formants[ i].Q;
        FLevels [ i]           := FVowel.Formants[ i].LinLevel;
      end;
    end;


//  public

    constructor TSvfFormantFilter.Create( aFormantCollection: TFormantCollection; aSampleRate: TSignal);
    begin
      inherited Create;
      FLocked            := True;
      FVowel.Name1       := '';
      FVowel.Name2       := '';
      FFormantCount      :=  0;
      FCurrentSet        := -1;
      FVowelIndex        := -1;
      FFormantCollection := aFormantCollection;
      CurrentSet         := 0;                     // Will clear FLocked and will (re)create the filter
      SampleRate         := aSampleRate;
      SetVowelIndex( 0);                           // Will update the filter
    end;


    destructor  TSvfFormantFilter.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;

      for i := 0 to Length( FFilters) - 1
      do FreeAndNil( FFilters[ i]);

      inherited;
    end;


    procedure   TSvfFormantFilter.Process( const anInput, aVowelIndex: TSignal; out aValue: TSignal);
    var
      i           : Integer;
      aLowPass    : TSignal; // note : need to use distinct variables for all params
      aBandPass   : TSignal; //        passed to FFilters[ i].Process, or odd stuff
      aBandReject : TSignal; //        will happen (even when the values are not being
      aHighPass   : TSignal; //        used).
    begin
      aValue := 0.0;

      if not FLocked
      then begin
        SetVowelIndex( aVowelIndex);

        for i := 0 to Length( FFilters) - 1
        do begin
          FFilters[ i].Process( anInput, aLowPass, aBandPass, aBandReject, aHighPass);
          aValue := Normalize( aValue + FLevels[ i] * aBandPass);
        end;
      end;
    end;


{ ========
  TSvfModalFilter = class
  private
    FModalCollection : TModalCollection;
    FMaxVoices       : Integer;
    FLocked          : Boolean;
    FSampleRate      : TSignal;
    FFilters         : array of TSvfChamberlinRelative;
    FLevels          : array of TSignal;
    FCurrent         : Integer;
    FFis             : Boolean;
  public
    property    Count                              : Integer read GetCount;
    property    Current                            : Integer read FCurrent       write SetCurrent;
    property    SampleRate                         : TSignal read FSampleRate    write SetSampleRate;
    property    FilterCount                        : Integer read GetFilterCount;
    property    Name                               : string  read GetName;
    property    Level    [ const anIndex: Integer] : TSignal                     write SetLevel;
    property    Frequency[ const anIndex: Integer] : TSignal                     write SetFrequency; // in Units
    property    LFM      [ const anIndex: Integer] : TSignal                     write SetLFM;
    property    Q        [ const anIndex: Integer] : TSignal                     write SetQ;
    property    Levels                             : TSignal                     write SetLevels;
    property    Qs                                 : TSignal                     write SetQs;
    property    MainQ                              : TSignal                     write SetMainQ;
    property    FIs                                : Boolean read FFis           write FFIs;
  private
}

    function    TSvfModalFilter.GetCount: Integer;
    begin
      if Assigned( FModalCollection)
      then Result := FModalCollection.SetCount
      else Result := 0;
    end;


    function    TSvfModalFilter.GetFilterCount: Integer;
    begin
      Result := Length( FFilters);
    end;


    function    TSvfModalFilter.GetName: string;
    begin
      if Assigned( FModalCollection)
      then Result := FModalCollection.Name[ FCurrent]
      else Result := '<no modals>';
    end;


    procedure   TSvfModalFilter.SetLevel( const anIndex: Integer; const aValue: TSignal);
    begin
      // Abs
      FLevels[ anIndex] := aValue;
    end;


    procedure   TSvfModalFilter.SetFrequency( const anIndex: Integer; const aValue: TSignal);
    var
      i : Integer;
    begin
      // Rel, exp
      for i := 0 to FilterCount - 1
      do FFilters[ i].RelFrequency := aValue;
    end;


    procedure   TSvfModalFilter.SetLFM( const anIndex: Integer; const aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to FilterCount - 1
      do FFilters[ i].LFM := aValue;
    end;


    procedure   TSvfModalFilter.SetQ( const anIndex: Integer; const aValue: TSignal);
    begin
      // Rel, lin
      FFilters[ anIndex].RelQ := aValue;
    end;


    procedure   TSvfModalFilter.SetMainQ( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Rel, lin
      for i := 0 to FilterCount - 1
      do FFilters[ i].MainQ := aValue;
    end;


    procedure   TSvfModalFilter.SetLevels( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Abs
      for i := 0 to FilterCount - 1
      do FLevels[ i] := aValue;
    end;


    procedure   TSvfModalFilter.SetQs( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Rel
      for i := 0 to FilterCount - 1
      do FFilters[ i].RelQ := aValue;
    end;


    procedure   TSvfModalFilter.SetCurrent( anIndex: Integer);
    begin
      if Count > 0
      then begin
        anIndex := Clip( anIndex, 0, Count - 1);

        if anIndex <> FCurrent
        then begin
          FCurrent := anIndex;
          RecreateFilter;
        end;
      end;
    end;


    procedure   TSvfModalFilter.SetSampleRate( aValue: TSignal);
    var
      i : Integer;
    begin
      if aValue <> FSampleRate
      then begin
        FSampleRate := aValue;

        for i := 0 to FilterCount - 1
        do FFilters[ i].SampleRate := FSampleRate;
      end;
    end;


//  private

    procedure   TSvfModalFilter.RecreateFilter;
    var
      i      : Integer;
      aCount : Integer;
    begin
      FLocked := True;

      try
        for i := 0 to FilterCount - 1
        do FreeAndNil( FFilters[ i]);

        if Count > 0
        then aCount := Min( FModalCollection.FormantCount[ FCurrent], FMaxVoices)
        else aCount := 0;

        SetLength( FFilters, aCount);
        SetLength( FLevels , aCount);

        for i := 0 to FilterCount - 1
        do begin
          FFilters[ i]            := TSvfChamberlinRelative.Create;
          FFilters[ i].SampleRate := SampleRate;
          FFilters[ i].Frequency  := FrequencyToUnits( FModalCollection.Frequency[ FCurrent, i]);
          FFilters[ i].Q          := FModalCollection.Q        [ FCurrent, i];
          FLevels [ i]            := 1.0;
        end;
      finally
        FLocked := False;
      end;
    end;


//  public

    constructor TSvfModalFilter.Create( aModalCollection: TModalCollection; aSampleRate: TSignal; aMaxVoices : Integer);
    begin
      inherited Create;
      FLocked          := True;
      FCurrent         := -1;
      FModalCollection := aModalCollection;
      FMaxVoices       := aMaxVoices;
      SampleRate       := aSampleRate;
      Current          := 0; // Will clear FLocked and will (re)create the filter, needs sample rate to be set before
    end;


    destructor  TSvfModalFilter.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;

      for i := 0 to Length( FFilters) - 1
      do FreeAndNil( FFilters[ i]);

      inherited;
    end;


    procedure   TSvfModalFilter.Process( const anInput: TSignal; out aValue: TSignal);
    var
      i           : Integer;
      aLowPass    : TSignal; // note : need to use distinct variables for all params
      aBandPass   : TSignal; //        passed to FFilters[ i].Process, or odd stuff
      aBandReject : TSignal; //        will happen (even when the values are not being
      aHighPass   : TSignal; //        used).
    begin
      aValue := 0.0;

      if not FLocked
      then begin
        for i := 0 to Length( FFilters) - 1
        do begin
          FFilters[ i].Process( anInput, aLowPass, aBandPass, aBandReject, aHighPass);

          if FIs
          then begin
            if Odd( i)
            then aValue := Normalize( aValue - FLevels[ i] * aBandPass)
            else aValue := Normalize( aValue + FLevels[ i] * aBandPass);
          end
          else aValue := Normalize( aValue + FLevels[ i] * aBandPass);
        end;
      end;
    end;


{ ========
  TSvfChladniFilter = class
  private
    FMaxVoices  : Integer;
    FLocked     : Boolean;
    FSampleRate : TSignal;
    FFilters    : array of TSvfChamberlinRelative;
    FLevels     : array of TSignal;
    FCurrent    : Integer;
    FFis        : Boolean;
  public
    property    SampleRate                            : TSignal read FSampleRate    write SetSampleRate;
    property    FilterCount                           : Integer read GetFilterCount;
    property    Level       [ const anIndex: Integer] : TSignal                     write SetLevel;
    property    RelFrequency[ const anIndex: Integer] : TSignal                     write SetRelFrequency; // in Units
    property    Q           [ const anIndex: Integer] : TSignal                     write SetQ;
    property    Frequency                             : TSignal                     write SetFrequency;    // in Units
    property    LFM                                   : TSignal                     write SetLFM;
    property    Levels                                : TSignal                     write SetLevels;
    property    Qs                                    : TSignal                     write SetQs;
    property    MainQ                                 : TSignal                     write SetMainQ;
    property    FIs                                   : Boolean read FFis           write FFIs;
  private
}

    function    TSvfChladniFilter.GetFilterCount: Integer;
    begin
      Result := Length( FFilters);
    end;


    procedure   TSvfChladniFilter.SetLevel( const anIndex: Integer; const aValue: TSignal);
    begin
      // Abs
      FLevels[ anIndex] := aValue;
    end;


    procedure   TSvfChladniFilter.SetRelFrequency( const anIndex: Integer; const aValue: TSignal);
    begin
      // Rel, exp
      FFilters[ anIndex].RelFrequency := aValue;
    end;


    procedure   TSvfChladniFilter.SetQ( const anIndex: Integer; const aValue: TSignal);
    begin
      // Rel, lin
      FFilters[ anIndex].RelQ := aValue;
    end;


    procedure   TSvfChladniFilter.SetFrequency( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Abs, exp
      for i := 0 to FilterCount - 1
      do FFilters[ i].Frequency := aValue;
    end;


    procedure   TSvfChladniFilter.SetLFM( const aValue: TSignal);
    var
      i : Integer;
    begin
      for i := 0 to FilterCount - 1
      do FFilters[ i].LFM := aValue;
    end;


    procedure   TSvfChladniFilter.SetMainQ( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Rel, lin
      for i := 0 to FilterCount - 1
      do FFilters[ i].MainQ := aValue;
    end;


    procedure   TSvfChladniFilter.SetLevels( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Abs
      for i := 0 to FilterCount - 1
      do FLevels[ i] := aValue;
    end;


    procedure   TSvfChladniFilter.SetQs( const aValue: TSignal);
    var
      i : Integer;
    begin
      // Rel
      for i := 0 to FilterCount - 1
      do FFilters[ i].RelQ := aValue;
    end;


    procedure   TSvfChladniFilter.SetSampleRate( const aValue: TSignal);
    var
      i : Integer;
    begin
      if aValue <> FSampleRate
      then begin
        FSampleRate := aValue;

        for i := 0 to FilterCount - 1
        do FFilters[ i].SampleRate := FSampleRate;
      end;
    end;


//  private

    procedure   TSvfChladniFilter.RecreateFilter;
    var
      i      : Integer;
      aCount : Integer;
    begin
      FLocked := True;

      try
        for i := 0 to FilterCount - 1
        do FreeAndNil( FFilters[ i]);

        aCount := FMaxVoices;

        SetLength( FFilters, aCount);
        SetLength( FLevels , aCount);

        for i := 0 to FilterCount - 1
        do begin
          FFilters[ i]            := TSvfChamberlinRelative.Create;
          FFilters[ i].SampleRate := SampleRate;
          FFilters[ i].Frequency  := FrequencyToUnits( 440.0);
          FFilters[ i].Q          := 1.0;
          FLevels [ i]            := 1.0;
        end;
      finally
        FLocked := False;
      end;
    end;


//  public

    constructor TSvfChladniFilter.Create( aSampleRate: TSignal; aMaxVoices : Integer);
    begin
      inherited Create;
      FLocked    := True;
      FMaxVoices := aMaxVoices;
      SampleRate := aSampleRate;
      RecreateFilter;
      FLocked    := False;
    end;


    destructor  TSvfChladniFilter.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;

      for i := 0 to Length( FFilters) - 1
      do FreeAndNil( FFilters[ i]);

      inherited;
    end;


    procedure   TSvfChladniFilter.Process( const anInput: TSignal; out aValue: TSignal);
    var
      i           : Integer;
      aLowPass    : TSignal; // note : need to use distinct variables for all params
      aBandPass   : TSignal; //        passed to FFilters[ i].Process, or odd stuff
      aBandReject : TSignal; //        will happen (even when the values are not being
      aHighPass   : TSignal; //        used).
    begin
      aValue := 0.0;

      if not FLocked
      then begin
        for i := 0 to Length( FFilters) - 1
        do begin
          FFilters[ i].Process( anInput, aLowPass, aBandPass, aBandReject, aHighPass);

          if FIs
          then begin
            if Odd( i)
            then aValue := Normalize( aValue - FLevels[ i] * aBandPass)
            else aValue := Normalize( aValue + FLevels[ i] * aBandPass);
          end
          else aValue := Normalize( aValue + FLevels[ i] * aBandPass);
        end;
      end;
    end;


end.

