unit Speech;

{
  This code is using parts from the 'Tiny Speech Synth'.

  Original copyright message as found at http://www.pouet.net/prod.php?which=50530 is :

    +-------------------------------------------------------------------------------------------------------------+
    | Type       : C++ subroutine                                                                                 |
    | Created by : Stepanov Andrey, 2008 ( ICQ: 129179794, e-mail: andrewstepanov@mail.ru )                       |
    |                                                                                                             |
    |                                                                                                             |
    | Description                                                                                                 |
    | ===========                                                                                                 |
    | This is a simple speech synth subroutine, based on formant synthesis theory.                                |
    | Speech is synthesized by passing source excitation signal through set of formant one-pole filters.          |
    | Excitation signal is a sawtooth or noise (depending on sound type), although you can try other signals.     |
    |                                                                                                             |
    |                                                                                                             |
    | Some explanation on design                                                                                  |
    | ==========================                                                                                  |
    | It was surprisingly hard to find any information on formant synthesis. Of course, there is a lot of         |
    | information on the Internet, but the biggest part of it is just a general words of theory,                  |
    | nothing practically usable. The biggest problem was to find concrete formant frequencies.                   |
    | After reading a lot of articles I found that I knew nothing new, compared to my knowledge before reading... |
    | So I did frequency analysis by myself. I recorded my own voice and analyzed it's FFT. Analysis is VERY      |
    | approximate and incomplete, so it causes bad speech quality.                                                |
    |                                                                                                             |
    |                                                                                                             |
    | Terms of use                                                                                                |
    | ============                                                                                                |
    | This program is free for use, and you can use it for any purposes you want, as long as you specify          |
    | my name in you program source code and description.                                                         |
    +-------------------------------------------------------------------------------------------------------------+

  2016-05-01: Modified for Delphi for real time synthesis by Blue Hell.

  Changed some stuff from the original code:

  - Turned things inside-out to allow for on-the-fly speech generation, phonemes are turned into audio demand based.
  - Added a tunable oscillator
  - made whisper to be controllable from zero to one.
  - made duration modulation for phonemes.
  - changed the specification for phonemes to be in proper units, Hz and s.

  The idea behind the changes being that with these some basic song synthesis would be possible for Wren.

  This version has modified terms of use :

     COPYRIGHT 2016 .. 2019 Blue Hell / Jan Punter

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as
    published by the Free Software Foundation;

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For all listed email addresses :

      _dot. to be substituted by a dot      '.'
      2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}


interface

uses

  System.Math, System.SysUtils, System.Classes,

  KnobsUtils, KnobsConversions;


type

  TFormant = record
  private
    // input stuff
    Frequency    : TSignal;                               // in Hz
    Width        : TSignal;                               // in Hz
    // calculated stuff
    InternalFreq : TSignal;                               // Calculated from Frequency
    Q            : TSignal;                               // calculated from Width
    Buf1Res      : TSignal;                               // Filter internal state 1
    Buf2Res      : TSignal;                               // Filter internal state 2
  public
    procedure Assign( const aFormant: TFormant);
    procedure Init;
    procedure Reset;
    function  Tick( anInput: TSignal): TSignal;
  end;


  TPhoneme = record
  private
    // input stuff
    Phoneme        : char;                                // The phoneme identification character
    Formants       : array[ 0 .. 2] of TFormant;          // Three formants max in a phoneme
    Duration       : TSignal;                             // in s
    Level          : TSignal;                             // [0, 1]
    Noise          : Boolean;                             // When true use noise, not an OSC
    Plosive        : Boolean;                             // When tru use a plosive
    // calculated stuff and state
    SampleCount    : Integer;                             // Duration in samples
    PlayPosition   : Integer;                             // Current play pointer in samples
    LastSample     : TSignal;                             // Last sample value calculated
    MixPointSent   : Boolean;                             // Flag to signal that mix point reached signal had been sent
    Phase          : TSignal;                             // Current oscillator phase
  public
    procedure Assign( const aPhoneme: TPhoneme);
    procedure Init;
    procedure Reset;
    function  Tick(
      var EndPointReached : Boolean;
      var MixPointReached : Boolean;
      aFreq               : TSignal;
      aPM                 : TSignal;
      aDuration           : TSignal;
      aWhisper            : TSignal;
      aFilterInversion    : Boolean;
      aShape              : Integer
    ): TSignal;
    procedure Dump( const aList: TStringList);
  end;


  TPhonemes          = array of TPhoneme;
  TOnTextStarted     = procedure( const aSender: TObject) of object;
  TOnTextEnded       = procedure( const aSender: TObject) of object;
  TOnPhonemeStarted  = procedure( const aSender: TObject; const aPhoneme: char) of object;


  TPhonemeSeries = class
  private
    FPhonemes         : TPhonemes;            // The 'sentence' of phonemes to speak
    FOnTextStarted    : TOnTextStarted;       // Callback for text    start
    FOnTextEnded      : TOnTextEnded;         // Callback for text    end
    FOnPhonemeStarted : TOnPhonemeStarted;    // Callback for phoneme start
    FFirstPhoneme     : Integer;              // Index of first currently active phoneme
    FLastPhoneme      : Integer;              // Index of phoneme just past the last active one
    FRunning          : Boolean;              // The phoneme sequence being active
    FLooped           : Boolean;              // The phoneme sequence being looped
    FFreq             : TSignal;              // Base frequency
    FPM               : TSignal;              // Current Phase Modulation
    FWhisper          : TSignal;              // Whisper amount
    FDuration         : TSignal;              // Relative duration [1/10, 10]
    FFilterInversion  : Boolean;              // Unwanted filter zero cancellation
    FShape            : Integer;              // 0 = saw, 1 = pulse
  private
    procedure   AddPhoneme( const aPhoneme: TPhoneme);
    procedure   TextStarted;
    procedure   TextEnded;
    procedure   PhonemeStarted( anIndex: Integer);
  public
    constructor Create(
      const aData              : string;
      const anOnTextStarted    : TOnTextStarted;
      const anOnTextEnded      : TOnTextEnded;
      const anOnPhonemeStarted : TOnPhonemeStarted
    );
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Reset;
    function    Tick: TSignal;
    procedure   SetFreq           ( const aFreq   , anFmMod    , anFmModAmt    , aPM, aPMAmt: TSignal);
    procedure   SetWhisper        ( const aWhisper, aWhisperMod, aWhisperModAmt             : TSignal);
    procedure   SetSpeed          ( const aSpeed  , aSpeedMod  , aSpeedModAmt               : TSignal);
    procedure   SetLooped         ( const aValue: Boolean);
    procedure   SetFilterInversion( const aValue: Boolean);
    procedure   SetShape          ( const aValue: Integer);
  public
    property    Runs : Boolean read FRunning;
  end;


  function  IsSilence( const aPhoneme: char): Boolean;
  function  IsVowel  ( const aPhoneme: char): Boolean;
  procedure InitModule;                                    // <-- Must be called when the sample rate changes
  procedure DumpPhonemes( const aFileName: string);


implementation


var

  Phonemes : TPhonemes = nil;


  function SearchPhoneme( aPhonemeChar: Char; var aPhonemeData: TPhoneme): Boolean;
  var
    i : Integer;
  begin
    FillChar( aPhonemeData, SizeOf( aPhonemeData), 0);
    Result := False;

    for i := Low( Phonemes) to High( Phonemes)
    do begin
      if aPhonemeChar = Phonemes[ i].Phoneme
      then begin
        aPhonemeData := Phonemes[ i];
        Result       := True;
        Break;
      end;
    end;
  end;


  function CutLevel( x, lvl: TSignal): TSignal;
  begin
    if x > lvl
    then Result := lvl
    else if x < - lvl
    then Result := - lvl
    else Result := x;
  end;


{ ========
  TFormant = record
  public
    // input stuff
    Frequency    : TSignal;                               // in Hz
    Width        : TSignal;                               // in Hz
    // calculated stuff
    InternalFreq : TSignal;                               // Calculated from Frequency
    Q            : TSignal;                               // calculated from Width
    Buf1Res      : TSignal;                               // Filter internal state 1
    Buf2Res      : TSignal;                               // Filter internal state 2
  public
}

  procedure TFormant.Assign( const aFormant: TFormant);
  begin
    Frequency := aFormant.Frequency;
    Width     := aFormant.Width;
    Init;
    Reset;
  end;


  procedure TFormant.Init;
  begin
    InternalFreq := 2.0 * Cos( TWO_PI * Frequency * System_Rate_Rec);
    Q            := 1.0 - Width * Pi * System_Rate_Rec;
  end;


  procedure TFormant.Reset;
  begin
    Buf1Res := 0;
    Buf2Res := 0;
  end;


  function  TFormant.Tick( anInput: TSignal): TSignal;
  begin
    Result  := anInput + InternalFreq * Buf1Res * Q - Buf2Res * Q * Q;
    Buf2Res := Buf1Res;
    Buf1Res := Result;
  end;


{ ========
  TPhoneme = record
  public
    // input stuff
    Phoneme        : char;                                // The phoneme identification character
    Formants       : array[ 0 .. 2] of TFormant;          // Three formants max in a phoneme
    Duration       : TSignal;                             // in s
    Level          : TSignal;                             // [0, 1]
    Noise          : Boolean;                             // When true use noise, not an OSC
    Plosive        : Boolean;                             // When tru use a plosive
    // calculated stuff and state
    SampleCount    : Integer;                             // Duration in samples
    PlayPosition   : Integer;                             // Current play pointer in samples
    LastSample     : TSignal;                             // Last sample value calculated
    MixPointSent   : Boolean;                             // Flag to signal that mix point reached signal had been sent
    Phase          : TSignal;                             // Current oscillator phase
  public
}

  procedure TPhoneme.Assign( const aPhoneme: TPhoneme);
  var
    i : Integer;
  begin
    Phoneme := aPhoneme.Phoneme;

    for i := Low( Formants) to High( Formants)
    do Formants[ i].Assign( aPhoneme.Formants[ i]);

    Duration := aPhoneme.Duration;
    Level    := aPhoneme.Level;
    Noise    := aPhoneme.Noise;
    Plosive  := aPhoneme.Plosive;

    Init;
    Reset;
  end;


  procedure TPhoneme.Init;
  var
    i : Integer;
  begin
    for i := Low( Formants) to High( Formants)
    do Formants[ i].Init;

    SampleCount  := Round( TimeToSampleCount( Duration));
    Reset;
  end;


  procedure TPhoneme.Reset;
  var
    i : Integer;
  begin
    for i := Low( Formants) to High( Formants)
    do Formants[ i].Reset;

    PlayPosition := 0;
    LastSample   := 0;
    Phase        := 0;
    MixPointSent := False;
  end;


  function  TPhoneme.Tick(
    var EndPointReached : Boolean;
    var MixPointReached : Boolean;
    aFreq               : TSignal;
    aPM                 : TSignal;
    aDuration           : TSignal;
    aWhisper            : TSignal;
    aFilterInversion    : Boolean;
    aShape              : Integer
  ): TSignal;
  const
    AmpFact = 1.0 / 1024.0;         // Output amplification factor
    PWM     = 0.1;                  // PWM for pulse mode generator
    WHREC   = 1.0 / 1.7;            // Whisper stability factor
  var
    aSample  : TSignal;
    i        : Integer;
    MixPoint : Integer;
    EndPoint : Integer;
    Signal   : TSignal;
    Whisp    : TSignal;
    Trunced  : Integer;
    SineVal  : TSignal;
    DPhase   : TSignal;
    Position : TSignal;
  begin
    // To allow for silence generation
    Result := 0.0;

    // On first sample of current phoneme calculate the SampleCount based on currently set duration
    if PlayPosition = 0
    then SampleCount := Round( TimeToSampleCount( Duration) * aDuration);

    // The point where the next phoneme must be started
    MixPoint := ( SampleCount * 3 ) div 4;

    // A little silence added after plosives
    if Plosive
    then EndPoint := 6 * SampleCount div 5
    else EndPoint :=     SampleCount;

    // If sound is actually needed - end point may be beyond SampleCount for added silence
    if PlayPosition < SampleCount
    then begin
      // Calculate excitation
      if Noise
      then Whisp := 1.0
      else Whisp := aWhisper;

      DPhase  := LookupPhaseDelta( aFreq);
      Phase   := Phase + DPhase;
      Trunced := Trunc( Phase);
      Phase   := Phase - Trunced;

      if Phase < 0
      then Phase := Phase + 1;

      Position := Phase + Clip( aPM, -1, 1);
      Trunced  := Trunc( Position);
      Position := Position - Trunced;

      if Position < 0
      then Position := Position + 1;

      case aShape of
        1 :                                 // Pulse, Saw otherwise.
          begin
            if Position > PWM
            then Signal :=       1.5 * PWM
            else Signal := 1.5 - 1.5 * PWM;
            Signal := Signal + PolyBLEP( Position, 2.0, DPhase) - PolyBLEP( FloatMod( Position + PWM, 1), 2.0, DPhase);
          end;
        else Signal := ( 2.0 * Position - 1) - PolyBLEP( Position, 2.0, DPhase); // Signal := 2.0 * Position - 1.0;
      end;

      aSample    := Signal - Whisp * ( Signal + Random - 0.5);
      LastSample := LastSample * Whisp * WHREC;

      // Calculate filters
      for i := Low( Formants) to High( Formants)
      do begin
        if Formants[ i].Frequency > 0
        then begin
          Result     := Result + Normalize( Formants[ i].Tick( aSample));
          Result     := Normalize( 0.75 * LastSample + Result * Level);
          LastSample := Result;

          // Filter cancellation?

          if aFilterInversion and Odd( i)
          then Result := - Result;
        end
        else Break;
      end;

      // Anti click and scale to reasonable level
      SineVal := Sin( Pi * PlayPosition / SampleCount);
      Result  := Normalize( Result * CutLevel( 5.0 * SineVal * SineVal, 1.0) * AmpFact);
    end;

    // Set ready for next sample and extract special events
    Inc( PlayPosition);
    MixPointReached := not MixPointSent and ( PlayPosition >= MixPoint);
    EndPointReached :=                        PlayPosition >= EndPoint;

    if MixPointReached
    then MixPointSent := True;
  end;


    procedure TPhoneme.Dump( const aList: TStringList);
    var
      S : string;
    begin
      if Assigned( aList)
      then begin
        S := Format(
          'RegisterPhoneme( ''%s'', %4.2f, %3.2f, %4.2f, %3.2f, %4.2f, %3.2f, %.3f, %.3f, %s, %s);',
          [
            Phoneme               ,
            Formants[ 0].Frequency,
            Formants[ 0].Width    ,
            Formants[ 1].Frequency,
            Formants[ 1].Width    ,
            Formants[ 2].Frequency,
            Formants[ 2].Width    ,
            Duration              ,
            Level                 ,
            BoolToStr( Noise  )   ,
            BoolToStr( Plosive)
          ]
        );
        aList.Add( S);
      end;
    end;


{ ========
  TPhonemeSeries = class
  private
    FPhonemes         : TPhonemes;            // The 'sentence' of phonemes to speak
    FOnTextStarted    : TOnTextStarted;       // Callback for text    start
    FOnTextEnded      : TOnTextEnded;         // Callback for text    end
    FOnPhonemeStarted : TOnPhonemeStarted;    // Callback for phoneme start
    FFirstPhoneme     : Integer;              // Index of first currently active phoneme
    FLastPhoneme      : Integer;              // Index of phoneme just past the last active one
    FRunning          : Boolean;              // The phoneme sequence being active
    FLooped           : Boolean;              // The phoneme sequence being looped
    FFreq             : TSignal;              // Base frequency
    FPM               : TSignal;              // Current Phase Modulation
    FWhisper          : TSignal;              // Whisper amount
    FDuration         : TSignal;              // Relative duration [1/10, 10]
    FFilterInversion  : Boolean;              // Unwanted filter zero cancellation
    FShape            : Integer;              // 0 = saw, 1 = pulse
  private
}

    procedure   TPhonemeSeries.AddPhoneme( const aPhoneme: TPhoneme);
    begin
      SetLength( FPhonemes, Length( FPhonemes) + 1);
      FPhonemes[ Length( FPhonemes) - 1].Assign( aPhoneme);
    end;


    procedure   TPhonemeSeries.TextStarted;
    begin
      if Assigned( FOnTextStarted)
      then FOnTextStarted( Self);
    end;


    procedure   TPhonemeSeries.TextEnded;
    begin
      if Assigned( FOnTextEnded)
      then FOnTextEnded( Self);
    end;


    procedure   TPhonemeSeries.PhonemeStarted( anIndex: Integer);
    begin
      if ( anIndex < Length( FPhonemes)) and Assigned( FOnPhonemeStarted)
      then FOnPhonemeStarted( Self, FPhonemes[ anIndex].Phoneme);
    end;


//  public

    constructor TPhonemeSeries.Create(
      const aData              : string;
      const anOnTextStarted    : TOnTextStarted;
      const anOnTextEnded      : TOnTextEnded;
      const anOnPhonemeStarted : TOnPhonemeStarted
    );
    var
      i        : Integer;
      aPhoneme : TPhoneme;
    begin
      inherited Create;
      FRunning := False;
      Clear;
      FOnTextStarted    := anOnTextStarted;
      FOnTextEnded      := anOnTextEnded;
      FOnPhonemeStarted := anOnPhonemeStarted;

      for i := Low( aData) to High( aData)
      do begin
        if SearchPhoneme( aData[ i], aPhoneme)
        then AddPhoneme( aPhoneme);
      end;

      FFirstPhoneme := -1;
      FLastPhoneme  :=  0;

      for i := Low( FPhonemes) to High( FPhonemes)
      do FPhonemes[ i].Reset;
    end;


    destructor  TPhonemeSeries.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TPhonemeSeries.Clear;
    begin
      SetLength( FPhonemes, 0);
    end;


    procedure   TPhonemeSeries.Reset;
    var
      i : Integer;
    begin
      FRunning      := False;
      FFirstPhoneme := -1;
      FLastPhoneme  :=  0;

      for i := Low( FPhonemes) to High( FPhonemes)
      do FPhonemes[ i].Reset;

      FRunning := True;
    end;


    function    TPhonemeSeries.Tick: TSignal;
    var
      Ended           : Boolean;
      MixPointReached : Boolean;
      i               : Integer;
      j               : Integer;
      Len             : Integer;
    begin
      Result := 0.0;
      Len    := Length( FPhonemes);

      // When active and text available

      if FRunning and ( Len > 0)
      then begin

        // if before the start of the text

        if FFirstPhoneme = -1
        then begin
          FFirstPhoneme := 0;
          FLastPhoneme  := 1;
          TextStarted;
          PhonemeStarted( 0);
        end;

        // Iterate over currently active phonemes, that is the one ones in the interval [FirstPhoneme,LastPhoneme>
        // This assumes that a phoneme that was started later than some other one will not be ended befor that other
        // one ends. Which is not true always when the speech speed gets modulated .. I'll leave it like this for
        // now though, as the situation would always give some odd or unexpected results. Anyways .. this will cause
        // todo: some clicky stuff on some talk speed changes.
        i := FFirstPhoneme;

        while i < FLastPhoneme
        do begin
          if i < Len
          then begin
            Ended           := False;
            MixPointReached := False;
            Result          := Result + FPhonemes[ i].Tick( Ended, MixPointReached, FFreq, FPM, FDuration, FWhisper, FFilterInversion, FShape);

            if MixPointReached
            then begin
              // Mixpoint reached for current phoneme, a new on is needed
              Inc( FLastPhoneme);

              if FLastPhoneme < Len
              then PhonemeStarted( i);
            end;

            if Ended
            then begin
              // Endpoint reached, one less phoneme to process
              Inc( FFirstPhoneme);

              if FFirstPhoneme >= Len
              then begin
                // All phonemes were processed
                FFirstPhoneme := -1;
                FRunning      := False;

                // Restart if looping
                if FLooped
                then Reset
                else begin
                  FFirstPhoneme := -1;
                  FLastPhoneme  :=  0;

                  for j := Low( FPhonemes) to High( FPhonemes)
                  do FPhonemes[ j].Reset;
                end;

                TextEnded;
                Break;
              end;
            end;

            Inc( i);
          end
          else Break; // i is not pointing in the list anymore, so ... done here.
        end;
      end;
    end;


    procedure   TPhonemeSeries.SetFreq( const aFreq, anFmMod, anFmModAmt, aPM, aPMAmt: TSignal);
    begin
      FFreq := aFreq + anFmMod * anFmModAmt;
      FPM   := aPM * aPMAmt;
    end;


    procedure   TPhonemeSeries.SetWhisper( const aWhisper, aWhisperMod, aWhisperModAmt: TSignal);
    begin
      FWhisper := Clip( aWhisper + aWhisperMod * aWhisperModAmt, 0.0, 1.0);
    end;


    procedure   TPhonemeSeries.SetSpeed( const aSpeed, aSpeedMod, aSpeedModAmt: TSignal);
    var
      aValue: TSignal;
    begin
      aValue := aSpeed * ( 1.0 + 3.0 * aSpeedMod * aSpeedModAmt);

      if aValue <= 0
      then FDuration := 10.0
      else FDuration := Clip( 1 / aValue, 0.1, 10.0);
    end;


    procedure   TPhonemeSeries.SetLooped( const aValue: Boolean);
    begin
      FLooped := aValue;

      if FLooped and not FRunning
      then Reset;
    end;


    procedure   TPhonemeSeries.SetFilterInversion( const aValue: Boolean);
    begin
      FFilterInversion := aValue;
    end;


    procedure   TPhonemeSeries.SetShape( const aValue: Integer);
    begin
      FShape := aValue;
    end;


  // ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  function  IsSilence( const aPhoneme: char): Boolean;
  begin
    Result := CharInSet( aPhoneme, [ ' ', ^I]);
  end;


  function  IsVowel( const aPhoneme: char): Boolean;
  begin
    Result := CharInSet( aPhoneme, [ 'o', 'i', 'j', 'u', 'a', 'e', 'E', 'w', 'v']);
  end;


  procedure RegisterPhoneme(
    aPhoneme    : Char;     // The phoneme identification character
    aFrequency1 : TSignal;  // in Hz
    aWidth1     : TSignal;  // in Hz
    aFrequency2 : TSignal;  // in Hz
    aWidth2     : TSignal;  // in Hz
    aFrequency3 : TSignal;  // in Hz
    aWidth3     : TSignal;  // in Hz
    aDuration   : TSignal;  // in s
    aLevel      : TSignal;  // linear
    aNoise      : TSignal;    // 0 = tone  , noise   otherwise
    aPlosive    : TSignal     // 0 = normal, plosive otherwise
  );
  begin
    SetLength( Phonemes, Length( Phonemes) + 1);

    Phonemes[ Length( Phonemes) - 1].Phoneme                := aPhoneme;
    Phonemes[ Length( Phonemes) - 1].Formants[ 0].Frequency := aFrequency1;
    Phonemes[ Length( Phonemes) - 1].Formants[ 1].Frequency := aFrequency2;
    Phonemes[ Length( Phonemes) - 1].Formants[ 2].Frequency := aFrequency3;
    Phonemes[ Length( Phonemes) - 1].Formants[ 0].Width     := aWidth1;
    Phonemes[ Length( Phonemes) - 1].Formants[ 1].Width     := aWidth2;
    Phonemes[ Length( Phonemes) - 1].Formants[ 2].Width     := aWidth3;
    Phonemes[ Length( Phonemes) - 1].Duration               := aDuration;
    Phonemes[ Length( Phonemes) - 1].Level                  := aLevel;
    Phonemes[ Length( Phonemes) - 1].Noise                  := aNoise   <> 0;
    Phonemes[ Length( Phonemes) - 1].Plosive                := aPlosive <> 0;

    Phonemes[ Length( Phonemes) - 1].Init;
  end;


  procedure InitModule;
  begin
    SetLength( Phonemes, 0);
    //
    //                aPhoneme ------------------------------------------------------------------------- Char;   // The phoneme ID character
    //                |         aFrequency1 ------------------------------------------------------------ TSignal // in Hz
    //                |         |        aWidth1 ------------------------------------------------------- TSignal // in Hz
    //                |         |        |        aFrequency2 ------------------------------------------ TSignal // in Hz
    //                |         |        |        |        aWidth2 ------------------------------------- TSignal // in Hz
    //                |         |        |        |        |        aFrequency3 ------------------------ TSignal // in Hz
    //                |         |        |        |        |        |       aWidth3 -------------------- TSignal // in Hz
    //                |         |        |        |        |        |       |      aDuration ----------- TSignal // in s
    //                |         |        |        |        |        |       |      |      aLevel ------  TSignal // linear 0.0 - 1.0
    //                |         |        |        |        |        |       |      |      |  aNoise ---- TSignal // 0 = tone  , 1 = noise
    //                |         |        |        |        |        |       |      |      |  |  aPlosive TSignal // 0 = normal, 1 = plosive
    //                |         |        |        |        |        |       |      |      |  |  |
    //                |         |        |        |        |        |       |      |      |  |  |
    RegisterPhoneme( ' ',    0.00,    0.00,    0.00,    0.00,    0.00,   0.00, 0.500, 0.000, 0, 0);
    RegisterPhoneme( ^I ,    0.00,    0.00,    0.00,    0.00,    0.00,   0.00, 1.500, 0.000, 0, 0);
    //                |         |        |        |        |        |       |      |      |  |  |
    //                |         |        |        |        |        |       |      |      |  |  |
    RegisterPhoneme( 'o',  600.00,  100.00,  750.00,  100.00,    0.00,   0.00, 1.500, 0.400, 0, 0);
    RegisterPhoneme( 'i',  250.00,  100.00, 2800.00,  100.00,    0.00,   0.00, 1.500, 0.200, 0, 0);
    RegisterPhoneme( 'j',  250.00,  100.00, 2800.00,  100.00,    0.00,   0.00, 0.500, 0.200, 0, 0);
    RegisterPhoneme( 'u',  250.00,  100.00,  700.00,  100.00,    0.00,   0.00, 1.500, 0.200, 0, 0);
    RegisterPhoneme( 'a',  900.00,  100.00, 1500.00,  100.00,    0.00,   0.00, 1.500, 1.000, 0, 0);
    RegisterPhoneme( 'e',  700.00,  100.00, 2500.00,  100.00,    0.00,   0.00, 1.500, 1.000, 0, 0);
    RegisterPhoneme( 'E', 1000.00,  100.00, 2000.00,  100.00,    0.00,   0.00, 1.500, 0.800, 0, 0);
    RegisterPhoneme( 'w',  150.00,  100.00,  700.00,  100.00,    0.00,   0.00, 1.500, 0.067, 0, 0);
    RegisterPhoneme( 'v',  100.00,  200.00, 1000.00,  100.00,    0.00,   0.00, 1.500, 0.200, 0, 0);
    //                |         |        |        |        |        |       |      |      |  |  |
    //                |         |        |        |        |        |       |      |      |  |  |
    RegisterPhoneme( 'T',  100.00,  400.00, 1000.00,   10.00,    0.00,   0.00, 1.500, 0.333, 0, 0);
    RegisterPhoneme( 'z',  250.00,  100.00, 1400.00,   50.00, 4000.00, 100.00, 1.500, 0.200, 0, 0);
    RegisterPhoneme( 'Z',  200.00,  500.00, 1500.00,   10.00, 3000.00,  50.00, 1.500, 0.333, 0, 0);
    RegisterPhoneme( 'b',  200.00,  100.00,    0.00,    0.00,    0.00,   0.00, 0.500, 0.133, 0, 0);
    RegisterPhoneme( 'd',  200.00,  100.00, 2000.00,  100.00, 4000.00, 100.00, 0.500, 0.133, 0, 0);
    RegisterPhoneme( 'm',  200.00,  100.00, 1000.00,  100.00,    0.00,   0.00, 1.500, 0.133, 0, 0);
    RegisterPhoneme( 'n',  200.00,  100.00, 2000.00,  100.00,    0.00,   0.00, 1.500, 0.133, 0, 0);
    RegisterPhoneme( 'r',  150.00,  300.00,  500.00,   80.00, 1000.00,  10.00, 1.500, 0.200, 0, 0);
    RegisterPhoneme( 'l',  400.00,  100.00, 1000.00,  100.00,    0.00,   0.00, 1.500, 0.333, 0, 0);
    RegisterPhoneme( 'g',  100.00,  150.00,  500.00,   50.00, 1300.00,  20.00, 1.000, 0.067, 0, 0);
    //                |         |        |        |        |        |       |      |      |  |  |
    //                |         |        |        |        |        |       |      |      |  |  |
    RegisterPhoneme( 'f',  400.00,  100.00, 1000.00,  100.00, 1700.00, 100.00, 1.500, 0.267, 1, 0);
    RegisterPhoneme( 'h', 1100.00,  300.00, 1300.00,  100.00, 1600.00, 300.00, 0.500, 0.667, 1, 0);
    RegisterPhoneme( 's', 4000.00,  800.00, 5500.00,  400.00,    0.00,   0.00, 1.500, 0.333, 1, 0);
    RegisterPhoneme( 'S', 1000.00, 1000.00, 1500.00, 1000.00,    0.00,   0.00, 1.500, 0.667, 1, 0);
    //                |         |        |        |        |        |       |      |      |  |  |
    //                |         |        |        |        |        |       |      |      |  |  |
    RegisterPhoneme( 'p',  200.00,   50.00,  500.00,  100.00, 1000.00, 100.00, 0.500, 0.133, 1, 1);
    RegisterPhoneme( 't',  200.00,  100.00, 1000.00,  200.00, 2000.00,  50.00, 0.500, 0.200, 1, 1);
    RegisterPhoneme( 'k', 1000.00,  100.00, 4000.00,  100.00,    0.00,   0.00, 0.500, 0.200, 1, 1);
  end;


  procedure  DumpPhList( const aList: TStringList);
  var
    aPhoneme : TPhoneme;
  begin
    if Assigned( Phonemes)
    then begin
      for aPhoneme in Phonemes
      do aPhoneme.Dump( aList);
    end;
  end;


  procedure DumpPhonemes( const aFileName: string);
  var
    aList : TStringList;
  begin
    aList := TStringList.Create;
    try
      DumpPhList( aList);
      aList.SaveToFile( aFileName);
    finally
      aList.DisposeOf;
    end;
  end;


initialization

  InitModule;

end.

