unit loro_sc;

  // C++ version (c) 2004 Russell Borogove / www.tinygod.com
  // Delphi Pascal Version 2005, Thaddy de Koning / www.thaddy.com
  //
  // Lorenz/Rossler iterative function systems as LFOs
  //

  //
  // This module defines the classes TLorenzOsc and TRosslerOsc - low frequency
  // oscillators suitable for modeling 'analog drift' or other random-but-smooth
  // processes. Both classes have identical APIs - you could unify the interface
  // with virtual functions easily.
  //
  // SetSampleRate:
  // Sets the sample rate at which the Iterate function will be called. Only
  // meaningful in conjunction with the SetFreq function.
  //
  // SetFreq:
  // Sets the fundamental frequency of the oscillator. The Rossler oscillator
  // should exhibit harmonic peaks at multiples of that frequency; the Lorenz
  // oscillator has a linear frequency-amplitude relation, so SetFreq will
  // only control the scale of waveform features in a general way.
  //
  // Iterate:
  // Advances the clock by one sample period and returns the value of the
  // function at the current clock; it should be called once per sample-tick.
  //
  // GetCurrent:
  // Returns the same value returned by the latest call to Iterate. Useful
  // in cases where one generator modulates multiple destinations, for example.
  //
  // GetAlternate:
  // Returns a value separate from the current value but correlated with it;
  // these are the X and Y values used for the well-known "butterfly" plots
  // of the Lorenz and Rossler functions. You can use GetAlternate if you
  // want two separate LFOs which are related in mysterious ways at a low
  // cost - for example, you can fine-tune one audio oscillator with the return
  // from Iterate and another oscillator with the return from GetAlternate.
  //
  // Both the primary and alternate returns are calibrated to a -1.0 to +1.0
  // range in normal usage. The implementation is discrete, though, so if the
  // sample rate is low or the frequency high, it may occasionally jump outside
  // that range -- the user is responsible for clamping if the range is
  // critical.
  //

  // 2015-06-09 : Simplified for use in Wren by Blue Hell
  //
  //   - Iterate replaced by Tick().
  //   - Tick() will produce all three output values at the same time.
  //   - GetCurrent was removed.
  //   - GetAlternate was removed.
  //   - The Z value is returned as the 3rd output.
  //   - Added oversampling to make the algorithms stable for higher frequencies.
  //     should be able to generate 12k5 Hz at a sample rate of 44k1 s/s now.
  //     For Lorenz FDT should get not larger than about 0.02, for Rssler the
  //     limit is about 0.08. Added a couple of debug checks for those bounds.
  //   - Factored out a common ancestor for both.
  //   - Unified outputscaling to be the same for all three outputs.
  //   - Replaced some variables used as constants by constants.
  //   - Renamed stuff.
  //   - Added some code conditionally compiled for debug versions only to make
  //     it easier to track issues with the code.


interface

uses

  SysUtils, Math,

  KnobsUtils, KnobsConversions;


type

  TAttractor = class
  // Base class for attractor type classes.
  protected
    FFreq : Double;
    FRate : Double;
    FDT   : Double;
    FDX   : Double;
    FDY   : Double;
    FDZ   : Double;
    FX    : Double;
    FY    : Double;
    FZ    : Double;
  protected
    procedure   Initialize;                                                                                     virtual;
  public
    procedure   SetSampleRate( aRate: Double);                                                        virtual; abstract;
    procedure   SetFreq      ( aFreq: Double);                                                        virtual; abstract;
    constructor Create;
    procedure   Reset;                                                                                          virtual;
  end;


  TLorenzOsc = class( TAttractor)
  // Lorenz system - very broad spectrum noise function with amplitude
  // decreasing with increasing frequency, but tight short-term correlation.
  //
  // The scale of waveform features will change somewhat with the set frequency
  // and sample rate, but not drastically - it's fairly fractal. In particular,
  // there will not be substantial spectral peaks at multiples of the frequency
  // selected by SetFreq.
  private
    procedure   FixDT;
  public
    procedure   SetSampleRate( aRate: Double);                                                                 override;
    procedure   SetFreq      ( aFreq: Double);                                                                 override;
    procedure   Tick( const Drive: Double; out Out1, Out2, Out3: Double);
  end;


  TRosslerOsc = class( TAttractor)
  // Rssler system - broad spectrum noise function with amplitude
  // decreasing with increasing frequency, and distinct harmonic peaks. The
  // peaks should occur at harmonics of the frequency set by SetFreq.
  private
    procedure   FixDT;
  public
    procedure   SetSampleRate( aRate: Double);                                                                 override;
    procedure   SetFreq      ( aFreq: Double);                                                                 override;
    procedure   Tick( const Drive: Double; out Out1, Out2, Out3: Double);
  end;


  TVanDerPolOsc = class( TAttractor)
  private
    FOverSampleRate : Integer;
  private
    procedure   FixDT;
  public
    constructor Create( OversampleRate: Integer);
    procedure   SetSampleRate( aRate: Double);                                                                 override;
    procedure   SetFreq      ( aFreq: Double);                                                                 override;
    procedure   Tick( const drive, mu: Double; out Out1, Out2: Double);
  end;



implementation



const

  LORENZ_OVERSAMPLE  = 16;
  LORENZ_SCALE       =  1.0 / 15.0;
  LORENZ_A           = 10.0;
  LORENZ_B           = 28.0;
  LORENZ_C           =  8.0 / 3.0;

  ROSSLER_OVERSAMPLE = 16;
  ROSSLER_SCALE      =  1.0 / 15.0;
  ROSSLER_A          =  0.15;
  ROSSLER_B          =  0.20;
  ROSSLER_C          = 10.0;

{$IFDEF DEBUG}
  procedure Breakpoint;
  begin
  end;
{$ENDIF}


{ ========
  TAttractor = class
  protected
    FFreq : Double;
    FRate : Double;
    FDT   : Double;
    FDX   : Double;
    FDY   : Double;
    FDZ   : Double;
    FX    : Double;
    FY    : Double;
    FZ    : Double;
  protected
}

    procedure   TAttractor.Initialize; // virtual;
    begin
      FDX := 0;
      FDY := 0;
      FDZ := 0;
      FX  := 1;
      FY  := 1;
      FZ  := 1;
    end;

//  public

    constructor TAttractor.Create;
    begin
      FFreq := 440;
      SetSampleRate( 44100.0);
      SetFreq      (   440.0);
      Initialize;
    end;

    procedure   TAttractor.Reset; // virtual;
    begin
      Initialize;
    end;


{ ========
  TLorenzOsc = class( TAttractor)
  private
}

    procedure   TLorenzOsc.FixDT;
    begin
      FDT := FFreq / ( FRate * LORENZ_OVERSAMPLE);
    {$IFDEF DEBUG}
      if Abs( FDT) > 0.02
      then Breakpoint
    {$ENDIF}
    end;

//  public

    procedure   TLorenzOsc.SetSampleRate( aRate: Double); // override;
    begin
      FRate := aRate;
      FixDT;
    end;

    procedure   TLorenzOsc.SetFreq( aFreq: Double); // override;
    begin
      FFreq := aFreq;
      FixDT;
    end;

    procedure   TLorenzOsc.Tick( const Drive: Double; out Out1, Out2, Out3: Double);
    var
      i : Integer;
    begin
      for i := 1 to LORENZ_OVERSAMPLE
      do begin
        FDX := LORENZ_A * ( FY - FX);
        FDY := FX * ( LORENZ_B - FZ) - FY + Drive;
        FDZ := FX * FY - LORENZ_C * FZ;

        FX := FX + FDX * FDT;
        FY := FY + FDY * FDT;
        FZ := FZ + FDZ * FDT;
      end;

      Out1 := FX * LORENZ_SCALE;
      Out2 := FY * LORENZ_SCALE;
      Out3 := FZ * LORENZ_SCALE;
    end;


{ ========
  TRosslerOsc = class( TAttractor)
  private
}

    procedure   TRosslerOsc.FixDT;
    begin
      FDT  := 2.91 * FFreq / ( FRate * ROSSLER_OVERSAMPLE);
    {$IFDEF DEBUG}
      if Abs( FDT) > 0.08
      then Breakpoint
    {$ENDIF}
    end;

//  public

    procedure   TRosslerOsc.SetSampleRate( aRate: Double); // override;
    begin
      FRate := aRate;
      FixDT;
    end;

    procedure   TRosslerOsc.SetFreq( aFreq: Double); // override;
    begin
      FFreq := aFreq;
      FixDT;
    end;

    procedure   TRosslerOsc.Tick( const Drive: Double; out Out1, Out2, Out3: Double);
    var
      i : Integer;
    begin
      for i := 1 to ROSSLER_OVERSAMPLE
      do begin
        FDX := - FY- FZ;
        FDY := FX + ROSSLER_A * FY;
        FDZ := ROSSLER_B + FZ * ( FX - ROSSLER_C + Drive / 4);

        FX := FX + FDX * FDT;
        FY := FY + FDY * FDT;
        FZ := FZ + FDZ * FDT;
      end;

      Out1 := FX * ROSSLER_SCALE;
      Out2 := FY * ROSSLER_SCALE;
      Out3 := FZ * ROSSLER_SCALE;
    end;


{ ========
  TVanDerPolOsc = class( TAttractor)
  private
    FOverSample : Integer;
  private
}
    procedure   TVanDerPolOsc.FixDT;
    begin
      FDT  := TWO_PI * FFreq / ( FRate * FOverSampleRate);
    {$IFDEF DEBUG}
      if Abs( FDT) > 0.08
      then Breakpoint
    {$ENDIF}
    end;


//  public

    constructor TVanDerPolOsc.Create( OversampleRate: Integer);
    begin
      FOverSampleRate := OversampleRate;
      inherited Create;
    end;


    procedure   TVanDerPolOsc.SetSampleRate( aRate: Double); // override;
    begin
      FRate := aRate;
      FixDT;
    end;


    procedure   TVanDerPolOsc.SetFreq( aFreq: Double); // override;
    begin
      FFreq := aFreq;
      FixDT;
    end;


    procedure   TVanDerPolOsc.Tick( const drive, mu: Double; out Out1, Out2: Double);
    var
      i : Integer;
      m : Double;
    begin
      m := Clip( mu, 0.01, 10.0);

      for i := 1 to FOverSampleRate
      do begin
        FDX := Clip( Normalize( FX + FY * FDT), -250, 250);
        FDY := Clip( Normalize( FY + ( drive + m * ( 1.0 - FX * FX) * FY - FX) * FDT), -250, 250);
        FX  := FDX;
        FY  := FDY;
      end;

      Out1 := FX / ( 4 + m);
      Out2 := FY / ( 4 + m);
    end;


end.
