unit makold;

{
   COPYRIGHT 2016 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    http://bluehell.electro-music.com/
    j_dot.punter2@t2iaf_dot.nl
}

interface

uses

  System.Math, System.Classes, System.SysUtils,

  KnobsUtils;


const

  MARKOV_VERSION_1  = 'markov 1';
  MARKOV_VERSION    = MARKOV_VERSION_1;

  MAX_MARKOV_LENGTH = 128;


type

  EMarkov         = class( Exception);
  EMarkovReadFail = class( EMarkov);

  TMarkovChain = class
  private
    FLocked   : Boolean;
    FOverflow : Boolean;
    FCyclic   : Boolean;
    FMaxSize  : Integer;
  private
    function    SearchPosition( aValue: TSignal): Integer;                                            virtual; abstract;
    procedure   Forget;                                                                               virtual; abstract;
  protected
    procedure   FailRead;                                                                             virtual; abstract;
  public
    procedure   Clear;                                                                                virtual; abstract;
    procedure   Learn( aNewValue, aStrength: TSignal);                                                virtual; abstract;
    function    NextValue: TSignal;                                                                   virtual; abstract;
    procedure   SaveToStrings  ( const aStrings : TStringList);                                       virtual; abstract;
    procedure   LoadFromStrings( const aStrings : TStringList);                                       virtual; abstract;
    procedure   SaveToFile     ( const aFilename: string     );                                       virtual; abstract;
    procedure   LoadFromFile   ( const aFileName: string     );                                       virtual; abstract;
  public
    property    Overflow : Boolean read FOverflow;
    property    MaxSize  : Integer read FMaxSize;
    property    Cyclic   : Boolean read FCyclic   write FCyclic;
  end;


  TMarkovChain1 = class( TMarkovChain)
  private
    FInPtr       : Integer;
    FValues      : TSignalArray;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FLastInValue : Integer;
    FState       : Integer;
  private
    function    SearchPosition( aValue: TSignal): Integer;                                                     override;
    procedure   Forget;                                                                                        override;
    function    FindValue( aValue: TSignal): Integer;
  protected
    procedure   FailRead;                                                                                      override;
  public
    constructor Create( DoCyclic: Boolean; aMaxSize : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;                                                                                         override;
    procedure   Learn( aNewValue, aStrength: TSignal);                                                         override;
    function    NextValue: TSignal;                                                                            override;
    procedure   SaveToStrings  ( const aStrings : TStringList);                                                override;
    procedure   LoadFromStrings( const aStrings : TStringList);                                                override;
    procedure   SaveToFile     ( const aFilename: string     );                                                override;
    procedure   LoadFromFile   ( const aFileName: string     );                                                override;
  end;


  PMarkovSignalSet2 = ^TMarkovSignalSet2;
  TMarkovSignalSet2 = record
  public
    Next   : PMarkovSignalSet2;
    Index  : Integer;
    First  : TSignal;
    Second : TSignal;
  public
    function    Hash: Byte;
    procedure   Append( aSignalPair: PMarkovSignalSet2);
    function    Equals( aValue: PMarkovSignalSet2): Boolean;
    function    Find  ( aValue: PMarkovSignalSet2): Integer;
    procedure   Forget( aValue: PMarkovSignalSet2);
  end;


  TMarkovSignalSets2 = array of TMarkovSignalSet2;
  TMarkovSignalHash2 = array[ Byte] of PMarkovSignalSet2;


  TMarkovData2 = class
  private
    FSize      : Integer;
    FSignals   : TMarkovSignalSets2;
    FHashIndex : TMarkovSignalHash2;
    FInPtr     : Integer;
    FOutPtr    : Integer;
    FFreeCount : Integer;
    FFillCount : Integer;
  public
    constructor Create( aSize: Integer);
    procedure   Clear;
    function    FindSignalPair( aValue: PMarkovSignalSet2): Integer;
    function    FindSignals( aFirst, aSecond: TSignal): Integer;
    function    AddSignalPair( aValue: PMarkovSignalSet2): Integer;
    function    AddSignals( aFirst, aSecond: TSignal): Integer;
    procedure   RemoveSignalPair;
  end;


  TMarkovChain2 = class( TMarkovChain)
  private
    FFillCount   : Integer;
    FValues      : TMarkovData2;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FLastInValue : Integer;
    FPrevValue   : TSignal;
    FState       : Integer;
  private
    function    SearchPosition( aValue: TSignal): Integer;                                                     override;
    procedure   Forget;                                                                                        override;
  protected
    procedure   FailRead;                                                                                      override;
  public
    constructor Create( DoCyclic: Boolean; aMaxSize : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;                                                                                         override;
    procedure   Learn( aNewValue, aStrength: TSignal);                                                         override;
    function    NextValue: TSignal;                                                                            override;
    procedure   SaveToStrings  ( const aStrings : TStringList);                                                override;
    procedure   LoadFromStrings( const aStrings : TStringList);                                                override;
    procedure   SaveToFile     ( const aFilename: string     );                                                override;
    procedure   LoadFromFile   ( const aFileName: string     );                                                override;
  end;


  PMarkovSignalSetn = ^TMarkovSignalSetn;
  TMarkovSignalSetn = record
  public
    Next     : PMarkovSignalSetn;
    Index    : Integer;
    DataSize : Integer;
    Data     : TSignalArray;
  public
    procedure   Initialize( aSize: Integer);
    function    Hash: Byte;
    procedure   Append( aSignalSet: PMarkovSignalSetn);
    function    Equals( aValue: PMarkovSignalSetn): Boolean;
    function    Find  ( aValue: PMarkovSignalSetn): Integer;
    procedure   Forget( aValue: PMarkovSignalSetn);
    function    HasNaNs: Boolean;
  end;


  TMarkovSignalSetsn = array of TMarkovSignalSetn;
  TMarkovSignalHashn = array[ Byte] of PMarkovSignalSetn;


  TMarkovDatan = class
  private
    FSize        : Integer;
    FOrder       : Integer;
    FSignals     : TMarkovSignalSetsn;
    FHashIndex   : TMarkovSignalHashn;
    FScratchData : TMarkovSignalSetn;
    FInPtr       : Integer;
    FOutPtr      : Integer;
    FFreeCount   : Integer;
    FFillCount   : Integer;
  public
    constructor Create( aSize, anOrder: Integer);
    procedure   Clear;
    function    FindSignalSet( aValue: PMarkovSignalSetn): Integer;
    function    FindSignals( aData: TSignalArray): Integer;
    function    AddSignalSet( aValue: PMarkovSignalSetn): Integer;
    function    AddSignals( aData: TSignalArray): Integer;
    procedure   RemoveSignalSet;
  end;


  TMarkovHistoryn = record
  public
    DataSize  : Integer;
    Data      : TSignalArray;
    InPtr     : Integer;
    OutPtr    : Integer;
    FreeCount : Integer;
    FillCount : Integer;
  public
    procedure   Initialize( aSize: Integer);
    procedure   AddValue( aValue: TSignal);
    procedure   Forget;
    procedure   FillData( aData: TSignalArray);
  end;


  TMarkovChainn = class( TMarkovChain)
  private
    FOrder       : Integer;
    FFillCount   : Integer;
    FValues      : TMarkovDatan;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FHistory     : TMarkovHistoryn;
    FScratch     : TSignalArray;
    FLastInValue : Integer;
    FState       : Integer;
  private
    function    SearchPosition( aValue: TSignal): Integer;                                                     override;
    procedure   Forget;                                                                                        override;
  protected
    procedure   FailRead;                                                                                      override;
  public
    constructor Create( DoCyclic: Boolean; aMaxSize, anOrder : Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;                                                                                         override;
    procedure   Learn( aNewValue, aStrength: TSignal);                                                         override;
    function    NextValue: TSignal;                                                                            override;
    procedure   SaveToStrings  ( const aStrings : TStringList);                                                override;
    procedure   LoadFromStrings( const aStrings : TStringList);                                                override;
    procedure   SaveToFile     ( const aFilename: string     );                                                override;
    procedure   LoadFromFile   ( const aFileName: string     );                                                override;
  end;


implementation


{ ========
  TMarkovChain = class
  private
    FLocked      : Boolean;
    FOverflow    : Boolean;
    FCyclic      : Boolean;
    FMaxSize     : Integer;
    FInPtr       : Integer;
    FValues      : TSignalArray;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FLastInValue : Integer;
    FState       : Integer;
  public
    property    Overflow : Boolean read FOverflow;
    property    MaxSize  : Integer read FMaxSize;
    property    Cyclic   : Boolean read FCyclic   write FCyclic;
  private
}

    function    TMarkovChain1.SearchPosition( aValue: TSignal): Integer;
    var
      P : Integer;
      L : Integer;
      H : Integer;
    begin
      L      := 0;
      H      := FInPtr - 1;
      Result := 0;

      while L <= H
      do begin
        P := ( L + H) div 2;

        if aValue < FLookups[ P]
        then H := P - 1
        else if ( P = H) or ( aValue < FLookups[ P + 1])
        then begin
          Result := P + 1;
          Break;
        end
        else L := P + 1;
      end;
    end;


    procedure   TMarkovChain1.Forget;
    var
      i : Integer;
    begin
      if FMaxSize > 0
      then begin
        Move( FValues[ 1], FValues[ 0], ( FMaxSize - 1) * SizeOf( FValues[ 0]));
        FValues[ FMaxSize - 1] := 0;

        for i := 0 to FMaxSize - 1
        do begin
          Move( FChances[ i, 1], FChances[ i, 0], ( FMaxSize - 1) * SizeOf( FChances[ 0, 0]));
          FChances[ i, FMaxSize - 1] := 0;
        end;

        if FLastInValue >= 0
        then Dec( FLastInValue);
      end;
    end;


    function    TMarkovChain1.FindValue( aValue: TSignal): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to FInPtr - 1
      do begin
        if FValues[ i] = aValue
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


//  protected

    procedure   TMarkovChain1.FailRead; // virtual;
    begin
      if not FLocked
      then FLocked := True;

      Clear;
      FLocked := False;

      raise EMarkovReadFail.Create( 'file read error');
    end;


//  public

    constructor TMarkovChain1.Create( DoCyclic: Boolean; aMaxSize : Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FLocked  := True;
      FCyclic  := DoCyclic;
      FMaxSize := Min( aMaxSize, MAX_MARKOV_LENGTH);
      SetLength( FValues , FMaxSize);
      SetLength( FLookups, FMaxSize);
      SetLength( FChances, FMaxSize);

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], FMaxSize);

      Clear;
      FLocked := False;
    end;


    destructor  TMarkovChain1.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], 0);

      SetLength( FChances, 0);
      inherited;
    end;


    procedure   TMarkovChain1.Clear;
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to FMaxSize - 1
      do begin
        FValues[ i] := 0;

        for j := 0 to FMaxSize - 1
        do FChances[ i, j] := 0;
      end;

      FInPtr       := 0;
      FLastInValue := -1;
      FOverflow    := False;
    end;


    procedure   TMarkovChain1.Learn( aNewValue, aStrength: TSignal);
    var
      p : Integer;
    begin
      if FLocked
      then Exit;

      p := FindValue( aNewValue);

      if p < 0
      then begin
        if FInPtr >= FMaxSize
        then begin
          if Cyclic
          then begin
            Forget;
            p           := FMaxSize - 1;
            FValues[ p] := aNewValue;
            FInPtr      := FMaxSize;
          end
          else FOverflow := True;
        end
        else begin
          p           := FInPtr;
          FValues[ p] := aNewValue;
          Inc( FInPtr);
        end
      end;

      if p >= 0
      then begin
        if FLastInValue >= 0
        then FChances[ FLastInValue, p] := Max( 0.0, FChances[ FLastInValue, p] + Clip( aStrength, -1.0, 1.0));

        FLastInValue := p;
      end;
    end;


    function    TMarkovChain1.NextValue: TSignal; // overload;
    var
      i   : Integer;
      Sum : TSignal;
    begin
      if FInPtr > 0
      then begin
        if ( FState >= 0) and ( FState < FInPtr)
        then begin
          Sum := 0;

          for i := 0 to FInPtr - 1
          do begin
            FLookups[ i] := Sum;
            Sum          := Sum + FChances[ FState, i];
          end;

          FState := SearchPosition( Sum * Random);
        end
        else FState := Random( FInPtr + 1);

        Result := FValues[ FState];
      end
      else Result := 0;
    end;


    procedure   TMarkovChain1.SaveToStrings( const aStrings : TStringList);
    var
      i : Integer;
      j : Integer;
      S : string;
      T : string;
    begin
      if Assigned( aStrings)
      then begin
        aStrings.Clear;

        if FInPtr > FMaxSize
        then FInPtr := FMaxSize;

        aStrings.Add( MARKOV_VERSION);
        aStrings.Add( IntToStr( FMaxSize));
        aStrings.Add( IntToStr( FInPtr  ));

        if Cyclic
        then aStrings.Add( '1')
        else aStrings.Add( '0');

        for i := 0 to FInPtr - 1
        do begin
          S := '';

          for j := 0 to FInPtr - 1
          do begin
            if S = ''
            then S :=     Format( '%g' , [ FChances[ i, j]], AppLocale)
            else S := S + Format( ':%g', [ FChances[ i, j]], AppLocale);
          end;

          T := Format( '%g;%s', [ FValues[ i], S], AppLocale);
          aStrings.Add( T);
        end;
      end;
    end;


    procedure   TMarkovChain1.LoadFromStrings( const aStrings : TStringList);
    const
      FIXED_LINES = 4;
    var
      aMaxSize : Integer;
      anInPtr  : Integer;
      aCyclic  : Integer;
      i        : Integer;
      j        : Integer;
      p        : Integer;
      S        : string;
      aVersion : string;
      aParts   : TStringList;
      bParts   : TStringList;
      aValue   : TSignal;
    begin
      if aStrings.Count >= FIXED_LINES
      then begin
        FLocked := True;

        try
          aVersion :=              aStrings[ 0];
          aMaxSize := StrToIntDef( aStrings[ 1], -1);
          anInPtr  := StrToIntDef( aStrings[ 2], -1);
          aCyclic  := StrToIntDef( aStrings[ 3], -1);

          if
            SameText( aVersion, MARKOV_VERSION)         and
            ( aMaxSize >  0                           ) and
            ( aMaxSize <= MAX_MARKOV_LENGTH           ) and
            ( anInPtr  >  0                           ) and
            ( anInPtr  =  aStrings.Count - FIXED_LINES) and
            ( aCyclic  >= 0                           )
          then begin
            Clear;

            if aCyclic = 0
            then FCyclic := False
            else FCyclic := True;

            if anInPtr > aMaxSize
            then anInPtr := aMaxSize;

            FMaxSize := aMaxSize;
            SetLength( FValues , FMaxSize);
            SetLength( FLookups, FMaxSize);
            SetLength( FChances, FMaxSize);

            for i := 0 to FMaxSize - 1
            do SetLength( FChances[ i], FMaxSize);

            Clear;
            FInPtr := anInPtr;
            p      := 0;

            for i := FIXED_LINES to aStrings.Count - 1
            do begin
              S      := aStrings[ i];
              aParts := Explode( S, ';');

              try
                if aParts.Count = 2
                then begin
                  aValue := StrToFloatDef( aParts[ 0], NaN);

                  if IsNan( aValue)
                  then FailRead
                  else begin
                    FValues[ p] := aValue;
                    bParts      := Explode( aParts[ 1], ':');

                    try
                      if bParts.Count = FInPtr
                      then begin
                        for j := 0 to FInPtr - 1
                        do begin
                          aValue := StrToFloatDef( aParts[ 0], NaN);

                          if IsNan( aValue)
                          then FailRead
                          else FChances[ p, j] := aValue;
                        end;
                      end
                      else FailRead;
                    finally
                      bParts.DisposeOf;
                    end;
                  end;
                end
                else FailRead;
              finally
                aParts.DisposeOf;
              end;

              Inc( p);
            end;
          end;
        finally
          FLocked := False;
        end;
      end
      else FailRead;
    end;


    procedure   TMarkovChain1.SaveToFile( const aFilename: string);
    var
      aStrings: TStringList;
    begin
      aStrings := TStringList.Create;

      try
        SaveToStrings( aStrings);
        aStrings.SaveToFile( aFilename);
      finally
        aStrings.DisposeOf;
      end;
    end;


    procedure   TMarkovChain1.LoadFromFile( const aFileName: string);
    var
      aStrings: TStringList;
    begin
      aStrings := TStringList.Create;

      try
        aStrings.LoadFromFile( aFilename);
        LoadFromStrings( aStrings);
      finally
        aStrings.DisposeOf;
      end;
    end;



{ ========
  TSignalPair = record
  public
    First  : TSignal;
    Second : TSignal;
    Next   : PSignalPair;
    Index  : Integer;
  public
}

    function TMarkovSignalSet2.Hash: Byte;
    type
      TByteArray = array[ 0 .. SizeOf( TSignal) - 1] of Byte;
      PByteArray = ^TByteArray;
    var
      i : Integer;
    begin
      Result := $55;

      for i := 0 to SizeOf( TSignal) - 1
      do Result := Result xor PByteArray( @ First)^[ i];

      for i := 0 to SizeOf( TSignal) - 1
      do Result := Result xor PByteArray( @ Second)^[ i];
    end;


    procedure   TMarkovSignalSet2.Append( aSignalPair: PMarkovSignalSet2);
    var
      aCurrent : PMarkovSignalSet2;
      aNext    : PMarkovSignalSet2;
      aCount   : Integer;
    begin
      if Assigned( aSignalPair)
      then begin
        aCurrent := @ Self;
        aNext    := Next;
        aCount   := 0;

        while Assigned( aNext)
        do begin
          aCurrent := aNext;
          aNext    := aNext^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;

        aCurrent^   .Next := aSignalPair;
        aSignalPair^.Next := nil;
      end;
    end;


    function    TMarkovSignalSet2.Equals( aValue: PMarkovSignalSet2): Boolean;
    begin
      Result := Assigned( aValue) and ( aValue^.First = First) and ( aValue^.Second = Second);
    end;


    function    TMarkovSignalSet2.Find( aValue: PMarkovSignalSet2): Integer;
    var
      aCurrent : PMarkovSignalSet2;
      aCount   : Integer;
    begin
      Result   := -1;
      aCurrent := @ Self;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if aCurrent^.Equals( aValue)
        then begin
          Result := aCurrent^.Index;
          Break;
        end
        else begin
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;
    end;


    procedure   TMarkovSignalSet2.Forget( aValue: PMarkovSignalSet2);
    var
      aCurrent : PMarkovSignalSet2;
      aPrev    : PMarkovSignalSet2;
      aCount   : Integer;
    begin
      aCurrent := @ Self;
      aPrev    := nil;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if Equals( aValue)
        then Break
        else begin
          aPrev    := aCurrent;
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;

      if Assigned( aPrev)
      then begin
        if Assigned( aCurrent)
        then aPrev.Next := aCurrent.Next
        else aPrev.Next := nil;
      end;
    end;



{ ========
  TMarkovSignalSets2 = array of TMarkovSignalSet2;
  TMArkovSignalHash2 = array[ Byte] of PMarkovSignalSet2;

  TMarkovData = class
  private
    FSize      : Integer;
    FSignals   : TMarkovSignalSets2;
    FHashIndex : TMarkovSignalHash2;
    FInPtr     : Integer;
    FOutPtr    : Integer;
    FFreeCount : Integer;
    FFillCount : Integer;
  public
}

    constructor TMarkovData2.Create( aSize: Integer);
    begin
      inherited Create;
      FSize := aSize;
      SetLength( FSignals, FSize);
      Clear;
    end;


    procedure   TMarkovData2.Clear;
    var
      i : Integer;
    begin
      FFreeCount := 0;
      FFillCount := 0;
      FInPtr     := 0;
      FOutPtr    := 0;

      for i := Low( FHashIndex) to High( FHashIndex)
      do FHashIndex[ i] := nil;

      FFreeCount := FSize;
    end;


    function    TMarkovData2.FindSignalPair( aValue: PMarkovSignalSet2): Integer;
    var
      aHash: Byte;
    begin
      Result := -1;
      aHash  := aValue.Hash;

      if Assigned( FHashIndex[ aHash])
      then Result := FHashIndex[ aHash]^.Find( aValue);
    end;


    function    TMarkovData2.FindSignals( aFirst, aSecond: TSignal): Integer;
    var
      aPair : TMarkovSignalSet2;
    begin
      aPair.First  := aFirst;
      aPair.Second := aSecond;
      Result := FindSignalPair( @ aPair);
    end;


    function    TMarkovData2.AddSignalPair( aValue: PMarkovSignalSet2): Integer;
    var
      aHash : Byte;
    begin
      Result := - 1;

      if Assigned( aValue) and ( FFreeCount > 0)
      then begin
        if IsNan( aValue^.First) or IsNan( aValue^.Second)
        then Nop;

        Result := FInPtr;
        FSignals[ FInPtr].First  := aValue^.First;
        FSignals[ FInPtr].Second := aValue^.Second;
        FSignals[ FInPtr].Index  := FInPtr;
        aHash := aValue^.Hash;

        if Assigned( FHashIndex[ aHash])
        then FHashIndex[ aHash]^.Append( @ FSignals[ FInPtr])
        else FHashIndex[ aHash] := @ FSignals[ FInPtr];

        FInPtr := ( FInPtr + 1) mod FSize;
        Inc( FFillCount);
        Dec( FFreeCount)
      end;
    end;


    function    TMarkovData2.AddSignals( aFirst, aSecond: TSignal): Integer;
    var
      aPair : TMarkovSignalSet2;
    begin
      aPair.First  := aFirst;
      aPair.Second := aSecond;
      Result := AddSignalPair( @ aPair);
    end;


    procedure   TMarkovData2.RemoveSignalPair;// overload;
    var
      anItem : PMarkovSignalSet2;
      aHash  : Byte;
    begin
      if FFillCount > 0
      then begin
        anItem := @ FSignals[ FOutPtr];
        aHash  := anItem^.Hash;

        if Assigned( FHashIndex[ aHash])
        then begin
          if anItem.Equals( FHashIndex[ aHash])
          then FHashIndex[ aHash] := FHashIndex[ aHash].Next
          else FHashIndex[ aHash]^.Forget( anItem);
        end;

        anItem^.Next := nil;
        FOutPtr := FOutPtr mod FSize;
        Dec( FFillCount);
        Inc( FFreeCount);
      end;
    end;


{ ========
  TMarkovChain2 = class
  private
    FLocked      : Boolean;
    FOverflow    : Boolean;
    FCyclic      : Boolean;
    FMaxSize     : Integer;
    FInPtr       : Integer;
    FValues      : TMarkovData2;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FLastInValue : Integer;
    FPrevValue   : TSignal;
    FState       : Integer;
  public
    property    Overflow : Boolean read FOverflow;
    property    MaxSize  : Integer read FMaxSize;
    property    Cyclic   : Boolean read FCyclic   write FCyclic;
  private
}

    function    TMarkovChain2.SearchPosition( aValue: TSignal): Integer;
    var
      P : Integer;
      L : Integer;
      H : Integer;
    begin
      L      := 0;
      H      := FFillCount - 1;
      Result := 0;

      while L <= H
      do begin
        P := ( L + H) div 2;

        if aValue < FLookups[ P]
        then H := P - 1
        else if ( P = H) or ( aValue < FLookups[ P + 1])
        then begin
          Result := P + 1;
          Break;
        end
        else L := P + 1;
      end;
    end;


    procedure   TMarkovChain2.Forget;
    begin
      if FMaxSize > 0
      then begin
        FValues.RemoveSignalPair;

        if FLastInValue >= 0
        then Dec( FLastInValue);

        if FLastInValue < 0
        then FPrevValue := NaN;
      end;
    end;


//  protected

    procedure   TMarkovChain2.FailRead; // virtual;
    begin
      if not FLocked
      then FLocked := True;

      Clear;
      FLocked := False;

      raise EMarkovReadFail.Create( 'file read error');
    end;


//  public

    constructor TMarkovChain2.Create( DoCyclic: Boolean; aMaxSize : Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FLocked  := True;
      FMaxSize := aMaxSize;
      FCyclic  := DoCyclic;
      FValues  := TMarkovData2.Create( FMaxSize);
      SetLength( FLookups, FMaxSize);
      SetLength( FChances, FMaxSize);

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], FMaxSize);

      Clear;
      FLocked := False;
    end;


    destructor  TMarkovChain2.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;
      FValues.DisposeOf;

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], 0);

      SetLength( FChances, 0);
      inherited;
    end;


    procedure   TMarkovChain2.Clear;
    var
      i : Integer;
      j : Integer;
    begin
      FValues.Clear;

      for i := 0 to FMaxSize - 1
      do begin
        for j := 0 to FMaxSize - 1
        do FChances[ i, j] := 0;
      end;

      FFillCount   := 0;
      FLastInValue := -1;
      FOverflow    := False;
      FPrevValue   := NaN;
    end;


    procedure   TMarkovChain2.Learn( aNewValue, aStrength: TSignal);
    var
      p : Integer;
    begin
      if FLocked
      then Exit;

      if not IsNan( FPrevValue)
      then begin
        p := FValues.FindSignals( FPrevValue, aNewValue);

        if p < 0
        then begin
          if FFillCount >= FMaxSize
          then begin
            if Cyclic
            then begin
              Forget;

              if not IsNan( FPrevValue)
              then begin
                p := FValues.AddSignals( FPrevValue, aNewValue);
                FFillCount := FMaxSize;
              end;
            end
            else FOverflow := True;
          end
          else begin
            p := FValues.AddSignals( FPrevValue, aNewValue);
            Inc( FFillCount);
          end;
        end;

        if p >= 0
        then begin
          if FLastInValue >= 0
          then FChances[ FLastInValue, p] := Max( 0.0, FChances[ FLastInValue, p] + Clip( aStrength, -1.0, 1.0));

          FLastInValue := p;
        end;
      end;

      FPrevValue := aNewValue;
    end;


    function    TMarkovChain2.NextValue: TSignal; // overload;
    var
      i   : Integer;
      Sum : TSignal;
    begin
      Result := 0;

      try
        if FFillCount > 0
        then begin
          if ( FState >= 0) and ( FState < FFillCount)
          then begin
            Sum := 0;

            for i := 0 to FFillCount - 1
            do begin
              FLookups[ i] := Sum;
              Sum          := Sum + FChances[ FState, i];
            end;

            if Sum > 0
            then FState := SearchPosition( Sum * Random)
            else FState := Random( FFillCount);
          end
          else FState := Random( FFillCount);

          Result := FValues.FSignals[ FState].First;
        end;
      except
        Nop;
      end;
    end;


    procedure   TMarkovChain2.SaveToStrings( const aStrings: TStringList);
    begin
    end;


    procedure   TMarkovChain2.LoadFromStrings( const aStrings: TStringList);
    begin
    end;


    procedure   TMarkovChain2.SaveToFile( const aFilename: string);
    begin
    end;


    procedure   TMarkovChain2.LoadFromFile( const aFileName: string);
    begin
    end;

{ ========
  PMarkovSignalSetn = ^TMarkovSignalSetn;
  TMarkovSignalSetn = record
  public
    Next     : PMarkovSignalSetn;
    Index    : Integer;
    DataSize : Integer;
    Data     : TSignalArray;
  public
}

    procedure   TMarkovSignalSetn.Initialize( aSize: Integer);
    begin
      DataSize := aSize;
      SetLength( Data, DataSize);
    end;


    function    TMarkovSignalSetn.Hash: Byte;
    type
      TByteArray = array[ 0 .. SizeOf( TSignal) - 1] of Byte;
      PByteArray = ^TByteArray;
    var
      i : Integer;
      j : Integer;
    begin
      Result := $55;

      for i := 0 to DataSize - 1
      do begin
        for j := 0 to SizeOf( Data[ 0]) - 1
        do Result := Result xor PByteArray( @ Data[ i])^[ j];
      end;
    end;


    procedure   TMarkovSignalSetn.Append( aSignalSet: PMarkovSignalSetn);
    var
      aCurrent : PMarkovSignalSetn;
      aNext    : PMarkovSignalSetn;
      aCount   : Integer;
    begin
      if Assigned( aSignalSet)
      then begin
        aCurrent := @ Self;
        aNext    := Next;
        aCount   := 0;

        while Assigned( aNext)
        do begin
          aCurrent := aNext;
          aNext    := aNext^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;

        aCurrent^  .Next := aSignalSet;
        aSignalSet^.Next := nil;
      end;
    end;


    function    TMarkovSignalSetn.Equals( aValue: PMarkovSignalSetn): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if Assigned( aValue)
      then begin
        Result := True;

        for i := 0 to DataSize - 1
        do begin
          if aValue^.Data[ i] <> Data[ i]
          then begin
            Result := False;
            Break;
          end;
        end;
      end;
    end;


    function    TMarkovSignalSetn.Find( aValue: PMarkovSignalSetn): Integer;
    var
      aCurrent : PMarkovSignalSetn;
      aCount   : Integer;
    begin
      Result   := -1;
      aCurrent := @ Self;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if aCurrent^.Equals( aValue)
        then begin
          Result := aCurrent^.Index;
          Break;
        end
        else begin
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;
    end;


    procedure   TMarkovSignalSetn.Forget( aValue: PMarkovSignalSetn);
    var
      aCurrent : PMarkovSignalSetn;
      aPrev    : PMarkovSignalSetn;
      aCount   : Integer;
    begin
      aCurrent := @ Self;
      aPrev    := nil;
      aCount   := 0;

      while Assigned( aCurrent)
      do begin
        if Equals( aValue)
        then Break
        else begin
          aPrev    := aCurrent;
          aCurrent := aCurrent^.Next;
          Inc( aCount);

          if aCount > 100
          then begin
            Nop;
            Break;
          end;
        end;
      end;

      if Assigned( aPrev)
      then begin
        if Assigned( aCurrent)
        then aPrev.Next := aCurrent.Next
        else aPrev.Next := nil;
      end;
    end;


    function    TMarkovSignalSetn.HasNaNs: Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to Length( Data) - 1
      do begin
        if IsNan( Data[ i])
        then begin
          Result := True;
          Break;
        end;
      end;
    end;



{ ========
  TMarkovSignalSetsn = array of TMarkovSignalSetn;
  TMarkovSignalHashn = array[ Byte] of PMarkovSignalSetn;


  TMarkovDatan = class
  private
    FSize        : Integer;
    FOrder       : Integer;
    FSignals     : TMarkovSignalSetsn;
    FHashIndex   : TMarkovSignalHashn;
    FScratchData : TMarkovSignalSetn;
    FInPtr       : Integer;
    FOutPtr      : Integer;
    FFreeCount   : Integer;
    FFillCount   : Integer;
  public
}

    constructor TMarkovDatan.Create( aSize, anOrder: Integer);
    begin
      inherited Create;
      FOrder := anOrder;
      FSize  := aSize;
      SetLength( FSignals, FSize);
      FScratchData.Initialize( FOrder);
      Clear;
    end;


    procedure   TMarkovDatan.Clear;
    var
      i : Integer;
    begin
      FFreeCount := 0;
      FFillCount := 0;
      FInPtr     := 0;
      FOutPtr    := 0;

      for i := Low( FHashIndex) to High( FHashIndex)
      do FHashIndex[ i] := nil;

      FFreeCount := FSize;
    end;


    function    TMarkovDatan.FindSignalSet( aValue: PMarkovSignalSetn): Integer;
    var
      aHash: Byte;
    begin
      Result := -1;
      aHash  := aValue.Hash;

      if Assigned( FHashIndex[ aHash])
      then Result := FHashIndex[ aHash]^.Find( aValue);
    end;


    function    TMarkovDatan.FindSignals( aData: TSignalArray): Integer;
    var
      i : Integer;
    begin
      for i := 0 to FScratchData.DataSize - 1
      do FScratchData.Data[ i] := aData[ i];

      Result := FindSignalSet( @ FScratchData);
    end;


    function    TMarkovDatan.AddSignalSet( aValue: PMarkovSignalSetn): Integer;
    var
      aHash : Byte;
      i     : Integer;
    begin
      Result := - 1;

      if Assigned( aValue) and ( FFreeCount > 0)
      then begin
        if aValue.HasNaNs
        then Nop;

        Result := FInPtr;

        for i := 0 to aValue^.DataSize - 1
        do FSignals[ FInPtr].Data[ i] := aValue^.Data[ i];

        FSignals[ FInPtr].Index  := FInPtr;
        aHash := aValue^.Hash;

        if Assigned( FHashIndex[ aHash])
        then FHashIndex[ aHash]^.Append( @ FSignals[ FInPtr])
        else FHashIndex[ aHash] := @ FSignals[ FInPtr];

        FInPtr := ( FInPtr + 1) mod FSize;
        Inc( FFillCount);
        Dec( FFreeCount)
      end;
    end;


    function    TMarkovDatan.AddSignals( aData: TSignalArray): Integer;
    var
      i : Integer;
    begin
      for i := 0 to FScratchData.DataSize - 1
      do FScratchData.Data[ i] := aData[ i];

      Result := AddSignalSet( @ FScratchData);
    end;


    procedure   TMarkovDatan.RemoveSignalSet;
    var
      anItem : PMarkovSignalSetn;
      aHash  : Byte;
    begin
      if FFillCount > 0
      then begin
        anItem := @ FSignals[ FOutPtr];
        aHash  := anItem^.Hash;

        if Assigned( FHashIndex[ aHash])
        then begin
          if anItem.Equals( FHashIndex[ aHash])
          then FHashIndex[ aHash] := FHashIndex[ aHash].Next
          else FHashIndex[ aHash]^.Forget( anItem);
        end;

        anItem^.Next := nil;
        FOutPtr := FOutPtr mod FSize;
        Dec( FFillCount);
        Inc( FFreeCount);
      end;
    end;


{ ========
  TMarkovHistoryn = record
  public
    DataSize  : Integer;
    Data      : TSignalArray;
    InPtr     : Integer;
    OutPtr    : Integer;
    FreeCount : Integer;
    FillCount : Integer;
  public
}

    procedure   TMarkovHistoryn.Initialize( aSize: Integer);
    begin
      DataSize := aSize;
      SetLength( Data, DataSize);
      InPtr     := 0;
      OutPtr    := 0;
      FillCount := 0;
      FreeCount := DataSize;
    end;


    procedure   TMarkovHistoryn.AddValue( aValue: TSignal);
    begin
      Data[ InPtr] := aValue;
      InPtr := ( InPtr + 1) mod DataSize;

      if FreeCount = 0
      then OutPtr := OutPtr + 1 mod DataSize
      else begin
        FillCount := FillCount + 1;
        FreeCount := FreeCount - 1;
      end;
    end;


    procedure   TMarkovHistoryn.Forget;
    begin
      Dec( FillCount);
      OutPtr := ( OutPtr + 1) mod DataSize;
      Inc( FreeCount);
    end;


    procedure   TMarkovHistoryn.FillData( aData: TSignalArray);
    var
      p : Integer;
      i : Integer;
    begin
      p := OutPtr;

      for i := 0 to DataSize - 1
      do begin
        aData[ i] := Data[ p];
        p := ( p + 1) mod DataSize;
      end;
    end;


{ ========
  TMarkovChainn = class( TMarkovChain)
  private
    FOrder       : Integer;
    FFillCount   : Integer;
    FValues      : TMarkovDatan;
    FLookups     : TSignalArray;
    FChances     : array of TSignalArray;
    FHistory     : TMarkovHistoryn;
    FScratch     : TSignalArray;
    FLastInValue : Integer;
    FState       : Integer;
  private
}

    function    TMarkovChainn.SearchPosition( aValue: TSignal): Integer; // override;
    var
      P : Integer;
      L : Integer;
      H : Integer;
    begin
      L      := 0;
      H      := FFillCount - 1;
      Result := 0;

      while L <= H
      do begin
        P := ( L + H) div 2;

        if aValue < FLookups[ P]
        then H := P - 1
        else if ( P = H) or ( aValue < FLookups[ P + 1])
        then begin
          Result := P + 1;
          Break;
        end
        else L := P + 1;
      end;
    end;


    procedure   TMarkovChainn.Forget; // override;
    begin
      if FMaxSize > 0
      then begin
        FValues.RemoveSignalSet;

        if FLastInValue >= 0
        then Dec( FLastInValue);

        if FLastInValue < 0
        then FHistory.Forget;
      end;
    end;


//  protected

    procedure   TMarkovChainn.FailRead; // override;
    begin
      if not FLocked
      then FLocked := True;

      Clear;
      FLocked := False;

      raise EMarkovReadFail.Create( 'file read error');
    end;


//  public

    constructor TMarkovChainn.Create( DoCyclic: Boolean; aMaxSize, anOrder : Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FLocked  := True;
      FMaxSize := aMaxSize;
      FOrder   := anOrder;
      FCyclic  := DoCyclic;
      FValues  := TMarkovDatan.Create( FMaxSize, FOrder);
      SetLength( FLookups, FMaxSize);
      SetLength( FChances, FMaxSize);
      SetLength( FScratch, FOrder);
      FHistory.Initialize( FOrder);

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], FMaxSize);

      Clear;
      FLocked := False;
    end;


    destructor  TMarkovChainn.Destroy; // override;
    var
      i : Integer;
    begin
      FLocked := True;
      FValues.DisposeOf;

      for i := 0 to FMaxSize - 1
      do SetLength( FChances[ i], 0);

      SetLength( FChances, 0);
      inherited;
    end;


    procedure   TMarkovChainn.Clear; // override;
    var
      i : Integer;
      j : Integer;
    begin
      FValues.Clear;

      for i := 0 to FMaxSize - 1
      do begin
        for j := 0 to FMaxSize - 1
        do FChances[ i, j] := 0;
      end;

      FFillCount   := 0;
      FLastInValue := -1;
      FOverflow    := False;

      FHistory.Initialize( FOrder);
    end;


    procedure   TMarkovChainn.Learn( aNewValue, aStrength: TSignal); // override;
    var
      p : Integer;
    begin
      if FLocked
      then Exit;

      if FHistory.FreeCount = 0
      then begin
        FHistory.AddValue( aNewValue);
        FHistory.FillData( FScratch);
        p := FValues.FindSignals( FScratch);

        if p < 0
        then begin
          if FFillCount >= FMaxSize
          then begin
            if Cyclic
            then begin
              Forget;

              if FHistory.FreeCount = 0
              then begin
                p := FValues.AddSignals( FScratch);
                FFillCount := FMaxSize;
              end;
            end
            else FOverflow := True;
          end
          else begin
            p := FValues.AddSignals( FScratch);
            Inc( FFillCount);
          end;
        end;

        if p >= 0
        then begin
          if FLastInValue >= 0
          then FChances[ FLastInValue, p] := Max( 0.0, FChances[ FLastInValue, p] + Clip( aStrength, -1.0, 1.0));

          FLastInValue := p;
        end;
      end
      else FHistory.AddValue( aNewValue);
    end;


    function    TMarkovChainn.NextValue: TSignal; // override;
    var
      i   : Integer;
      Sum : TSignal;
    begin
      Result := 0;

      try
        if FFillCount > 0
        then begin
          if ( FState >= 0) and ( FState < FFillCount)
          then begin
            Sum := 0;

            for i := 0 to FFillCount - 1
            do begin
              FLookups[ i] := Sum;
              Sum          := Sum + FChances[ FState, i];
            end;

            if Sum > 0
            then FState := SearchPosition( Sum * Random)
            else FState := Random( FFillCount);
          end
          else FState := Random( FFillCount);

          Result := FValues.FSignals[ FState].Data[ FOrder - 1];
        end;
      except
        Nop;
      end;
    end;


    procedure   TMarkovChainn.SaveToStrings( const aStrings : TStringList); // override;
    begin
    end;


    procedure   TMarkovChainn.LoadFromStrings( const aStrings : TStringList); // override;
    begin
    end;


    procedure   TMarkovChainn.SaveToFile( const aFilename: string); // override;
    begin
    end;


    procedure   TMarkovChainn.LoadFromFile( const aFileName: string); // override;
    begin
    end;


end.

