unit midi_reader;

{

   COPYRIGHT 2014 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl


  ------------------------------------------------------------

  MIDI file reader based on:

  * http://cs.fit.edu/~ryan/cse4051/projects/midi/midi.html
  * http://www.midi.org/techspecs/midimessages.php

  ------------------------------------------------------------

}

interface

uses

  System.SysUtils, System.Math, System.Classes,

  knobsutils;


type

  EWrenMidiReader = class( Exception);
  EWrenMidiParse  = class( EWrenMidiReader);


  TWrenMidiData = class
  private
    FData : TBytes;
  private
    procedure   AppendByte( aByte: Byte);
  public
    constructor Create;                                                                          virtual;
    constructor CreateFromData ( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer); virtual;
    constructor CreateFromBytes( const aData: TBytes);                                           virtual;
    constructor CreateFromFile ( const aFilename: string);                                       virtual;
  private
    function    GetCount: Integer;
    function    GetDataByte    ( anIndex: Integer): Byte;
    function    GetDataWord    ( anIndex: Integer): Word;
    function    GetDataInteger ( anIndex: Integer): Integer;
    function    GetDataCardinal( anIndex: Integer): Cardinal;
    function    GetDataLongInt ( anIndex: Integer): LongInt;
    function    GetDataFourCC  ( anIndex: Integer): string;
  public
    function    Remaining( anOffset: Integer): Integer;
  public
    property    Count                          : Integer  read GetCount;
    property    DataByte    [ anIndex: Integer]: Byte     read GetDataByte; default;
    property    DataWord    [ anIndex: Integer]: Word     read GetDataWord;
    property    DataInteger [ anIndex: Integer]: Integer  read GetDataInteger;
    property    DataCardinal[ anIndex: Integer]: Cardinal read GetDataCardinal;
    property    DataLongInt [ anIndex: Integer]: LongInt  read GetDataLongInt;
    property    DataFourCC  [ anIndex: Integer]: string   read GetDataFourCC;
  end;


  TWrenMidiHeader = class( TWrenMidiData)
  private
    FTrackFormat : Word;
    FTracks      : Word;
    FDivision    : Word;
  public
    constructor CreateFromData( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer);  override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
  public
    property    TrackFormat : Word read FTrackFormat;
    property    Tracks      : Word read FTracks;
    property    Division    : Word read FDivision;
  end;


  TWrenMidiDeltaTime = class
  private
    FDeltaTime: Integer;
  public
    constructor CreateParseFromData( const aData: TBytes; var anOffset: Integer);
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
  public
    property    DeltaTime: Integer read FDeltaTime;
  end;


  TWrenMidiEvent = class
  public
    constructor     CreateFromBytes( const aData: TBytes);                                        virtual; abstract;
    procedure       Dump( anInd: Integer; const aStringList: TStringList);                        virtual;
    class function  CreateParseFromData( const aData: TBytes; var anOffset: Integer; var RunningStatus: Byte): TWrenMidiEvent;
  end;


  TWrenMidiNoEvent = class( TWrenMidiEvent);


  TWrenMidiChannelEvent = class( TWrenMidiEvent)
  private
    FChannel : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Channel : Byte read FChannel;
  end;


  TWrenMidiChannelVoiceEvent = class( TWrenMidiChannelEvent)
  private
    FEvent : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Event : Byte read FEvent;
  end;


  TWrenMidiChannelModeEvent = class( TWrenMidiChannelEvent);


  TWrenMidiMetaEvent = class( TWrenMidiEvent)
  var
    FData : Tbytes;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
  public
    property    Data : TBytes read FData;
  end;


  TWrenMidiNoteEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FKey      : Byte;
    FVelocity : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Key      : Byte read FKey;
    property    Velocity : Byte read FVelocity;
  end;


  TWrenMidiNoteOffEvent         = class( TWrenMidiNoteEvent);
  TWrenMidiNoteOnEvent          = class( TWrenMidiNoteEvent);
  TWrenMidiPolyKeyPressureEvent = class( TWrenMidiNoteEvent);


  TWrenMidiControlChangeEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FController : Byte;
    FValue      : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Controller : Byte read FController;
    property    Value      : Byte read FValue;
  end;


  TWrenMidiProgramChangeEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FProgramNr : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    ProgramNr : Byte read FProgramNr;
  end;


  TWrenMidiChannelPressureEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FVelocity : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Velocity : Byte read FVelocity;
  end;


  TWrenMidiPitchBendEvent = class( TWrenMidiChannelVoiceEvent)
  var
    FBend : Word;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Bend : Word read FBend;
  end;


  TWrenMidiAllSoundOffEvent         = class( TWrenMidiChannelModeEvent);
  TWrenMidiResetAllControllersEvent = class( TWrenMidiChannelModeEvent);


  TWrenMidiLocalControlEvent = class( TWrenMidiChannelModeEvent)
  private
    FActive : Boolean;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Active : Boolean read FActive;
  end;


  TWrenMidiAllNotesOffEvent = class( TWrenMidiChannelModeEvent);
  TWrenMidiOmniModeOff      = class( TWrenMidiChannelModeEvent);
  TWrenMidiOmniModeOn       = class( TWrenMidiChannelModeEvent);


  TWrenMidiMonoModeOn = class( TWrenMidiChannelModeEvent)
  private
    FChannelCount : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    ChannelCount : Byte read FChannelCount;
  end;


  TWrenMidiPolyModeOn = class( TWrenMidiChannelModeEvent);


  TWrenMidiSystemExclusiveEvent = class( TWrenMidiEvent)
  private
    FData : TBytes;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
  public
    property    Data: TBytes read FData;
  end;


  TWrenMidiSequenceNumberEvent = class( TWrenMidiMetaEvent)
  private
    FSequenceNumber : Word;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    SequenceNumber : Word read FSequenceNumber;
  end;


  TWrenMidiTextEventEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiCopyRightNoticeEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiSequenceTrackNameEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiInstrumentNameEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiLyricEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiMarkerEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiCuePointEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Text : string read FText;
  end;


  TWrenMidiMidiChannelPrefixEvent = class( TWrenMidiMetaEvent)
  private
    FChannel : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Channel: Byte read FChannel;
  end;


  TWrenMidiEndOfTrackEvent = class( TWrenMidiMetaEvent);


  TWrenMidiSetTempoEvent = class( TWrenMidiMetaEvent)
  private
    FTempo : Cardinal;         // s / quarter note
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Tempo : Cardinal read FTempo;
  end;


  TWrenMidiSMTPOffsetEvent = class( TWrenMidiMetaEvent)
  private
    FHour            : Byte;
    FMinute          : Byte;
    FSecond          : Byte;
    FFrame           : Byte;
    FFractionalFrame : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Hour            : Byte read FHour;
    property    Minute          : Byte read FMinute;
    property    Second          : Byte read FSecond;
    property    Frame           : Byte read FFrame;
    property    FractionalFrame : Byte read FFractionalFrame;
  end;


  TWrenMidiTimeSignatureEvent = class( TWrenMidiMetaEvent)
  private
    FNumerator                  : Byte;
    FDenominator                : Cardinal;
    FClocksPerTick              : Byte;
    FSemisemiquaversPer24Clocks : Byte;                   // 1/32th notes per 24 MIDI clocks
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    Numerator                  : Byte     read FNumerator;
    property    Denominator                : Cardinal read FDenominator;
    property    ClocksPerTick              : Byte     read FClocksPerTick;
    property    SemisemiquaversPer24Clocks : Byte     read FSemisemiquaversPer24Clocks;
  end;


  TWrenMidiKeySignatureEvent = class( TWrenMidiMetaEvent)
  private
    FNumberOfSharps : Byte;
    FMajorMinor     : Byte;
  public
    constructor CreateFromBytes( const aData: TBytes);                                           override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);                           override;
  public
    property    NumberOfSharps : Byte read FNumberOfSharps;
    property    MajorMinor     : Byte read FMajorMinor;
  end;


  TWrenMidiSequencerSpecificEvent = class( TWrenMidiMetaEvent);


  TWrenMidiTimedEvent = class
  private
    FDeltaTime : TWrenMidiDeltaTime;
    FEvent     : TWrenMidiEvent;
  public
    constructor Create( const aDeltaTime: TWrenMidiDeltaTime; anEvent: TWrenMidiEvent);
    destructor  Destroy;                                                                         override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
  public
    function    IsNoEvent                 : Boolean;
  public
    function    IsChannelVoiceEvent       : Boolean;
    function    IsNoteEvent               : Boolean;
    function    IsNoteOffEvent            : Boolean;
    function    IsNoteOnEvent             : Boolean;
    function    IsPolyKeyPressureEveent   : Boolean;
    function    IsControlChange           : Boolean;
    function    IsProgramChange           : Boolean;
    function    IsChannelPressure         : Boolean;
    function    IsPitchBendEvent          : Boolean;
  public
    function    IsChannelModeEvent        : Boolean;
    function    IsAllSoundOffEvent        : Boolean;
    function    IsResetAllControllersEvent: Boolean;
    function    IsLocalControlEvent       : Boolean;
    function    IsAllNotesOffEvent        : Boolean;
  public
    function    IsSystemExclusiveEvent    : Boolean;
  public
    function    IsMetaEvent               : Boolean;
    function    IsSequenceNumberEvent     : Boolean;
    function    IsTextEventEvent          : Boolean;
    function    IsCopyRightNoticeEvent    : Boolean;
    function    IsSequenceTrackNameEvent  : Boolean;
    function    IsInstrumentNameEvent     : Boolean;
    function    IsLyricEvent              : Boolean;
    function    IsMarkerEvent             : Boolean;
    function    IsCuePointEvent           : Boolean;
    function    IsMidiChannelPrefixEvent  : Boolean;
    function    IsEndOfTrackEvent         : Boolean;
    function    IsSetTempoEvent           : Boolean;
    function    IsSMTPOffsetEvent         : Boolean;
    function    IsTimeSignatureEvent      : Boolean;
    function    IsKeySignatureEvent       : Boolean;
    function    IsSequencerSpecificEvent  : Boolean;
  public
    property    DeltaTime : TWrenMidiDeltaTime read FDeltaTime;
    property    Event     : TWrenMidiEvent     read FEvent;
  end;


  TWrenMidiTrack = class( TWrenMidiData)
  private
    FTimedEvents   : array of TWrenMidiTimedEvent;
    FOffset        : Integer;
    FRunningStatus : Byte;
    FCurrent       : Integer;
  private
    function    GetTimedEventCount: Integer;
    function    GetTimedEvent( anIndex: Integer): TWrenMidiTimedEvent;
    procedure   AddTimedEvent( const aTimedEvent: TWrenMidiTimedEvent);
    procedure   ParseTimedEvent;
    procedure   Parse;
  public
    constructor CreateFromData( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer);  override;
    destructor  Destroy;                                                                         override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
    procedure   Reset;
    function    NextTimedEvent: TWrenMidiTimedEvent;
  public
    property    TimedEventCount               : Integer             read GetTimedEventCount;
    property    TimedEvent[ anIndex: Integer] : TWrenMidiTimedEvent read GetTimedEvent;
  end;


  TWrenMiditracks = class
  private
    FTracks : array of TWrenMidiTrack;
  private
    function    GetTrackCount: Integer;
    function    GetTrack( anIndex: Integer): TWrenMidiTrack;
  public
    destructor  Destroy;                                                                   override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
    procedure   AddTrack( const aTrack: TWrenMidiTrack);
    procedure   Reset;
    function    NextTimedEvent( aTrack: Integer): TWrenMidiTimedEvent;
  public
    property    TrackCount                : Integer read GetTrackCount;
    property    Tracks[ anIndex: Integer] : TWrenMidiTrack read GetTrack;
  end;


  TWrenMidiReader = class
  private
    FFileName : string;
    FData     : TWrenMidiData;
    FOffset   : Integer;
    FHeader   : TWrenMidiHeader;
    FTracks   : TWrenMiditracks;
  private
    function    GetFormat  : Word;
    function    GetTracks  : Word;
    function    GetDivision: Word;
  private
    function    GetTrackCount: Integer;
    procedure   parseHeader;
    procedure   ParseTrack;
    procedure   ParseTracks;
    procedure   Parse;
  public
    constructor Create;
    destructor  Destroy;                                                                   override;
    procedure   Dump( anInd: Integer; const aStringList: TStringList);
    procedure   ReadFile( const aFileName: string);
    procedure   Reset;
    function    NextTimedEvent( aTrack: Integer): TWrenMidiTimedEvent;
  public
    property    TrackCount: Integer read GetTrackCount;
    property    Format      : Word  read GetFormat;
    property    Tracks      : Word  read GetTracks;
    property    Division    : Word  read GetDivision;
  end;



implementation

uses

  midi_defs;

const

  // When aCmd And $f0 = CSysEx low nibble of 1st byte is :

  SSysEx0               = $00; // Length = <n+1>
  SSysex7               = $07; // Length = <n>;
  SMeta                 = $0f; // Length = <n>;


{ ========
  EWrenMidiReader = class( Exception);
  EWrenMidiParse  = class( EWrenMidiReader);
}

{ ========
  TWrenMidiData = class
  private
    FData : TBytes;
  public
    property    Count                          : Integer  read GetCount;
    property    DataByte    [ anIndex: Integer]: Byte     read GetDataByte; default;
    property    DataWord    [ anIndex: Integer]: Word     read GetDataWord;
    property    DataInteger [ anIndex: Integer]: Integer  read GetDataInteger;
    property    DataCardinal[ anIndex: Integer]: Cardinal read GetDataCardinal;
    property    DataLongInt [ anIndex: Integer]: LongInt  read GetDataLongInt;
    property    DataFourCC  [ anIndex: Integer]: string   read GetDataFourCC;
  private
}

    procedure   TWrenMidiData.AppendByte( aByte: Byte);
    begin
      SetLength( FData, Count + 1);
      FData[ Count - 1] := aByte;
    end;


//  public

    constructor TWrenMidiData.Create; // virtual;
    begin
      inherited;
    end;


    constructor TWrenMidiData.CreateFromData( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer); // virtual;
    var
      i : Integer;
    begin
      Create;
      SetLength( FData, aSize);

      for i := 0 to Count - 1
      do FData[ i] := aData[ i + anOffset];
    end;


    constructor TWrenMidiData.CreateFromBytes( const aData: TBytes); // virtual;
    var
      i : Integer;
    begin
      Create;
      SetLength( FData, Length( aData));

      for i := 0 to Count - 1
      do FData[ i] := aData[ i];
    end;


    constructor TWrenMidiData.CreateFromFile( const aFilename: string); // virtual;
    var
      aFile : file of Byte;
      aByte : Byte;
    begin
      AssignFile( aFile, aFilename);

      try
        Reset( aFile);

        while not Eof( aFile)
        do begin
          Read( aFile, aByte);
          AppendByte( aByte);
        end;
      finally
        CloseFile( aFile);
      end;
    end;


//  private

    function    TWrenMidiData.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


    function    TWrenMidiData.GetDataByte( anIndex: Integer): Byte;
    begin
      Result := FData[ anIndex];
    end;


    function    TWrenMidiData.GetDataWord( anIndex: Integer): Word;
    begin
      Result := 256 * Word( DataByte[ anIndex]) + Word( DataByte[ anIndex + 1]);
    end;


    function    TWrenMidiData.GetDataInteger( anIndex: Integer): Integer;
    begin
      Result := Integer( DataWord[ anIndex]);
    end;


    function    TWrenMidiData.GetDataCardinal( anIndex: Integer): Cardinal;
    begin
      Result := 256 * 256 * Cardinal( DataWord[ anIndex]) + Cardinal( DataWord[ anIndex + 2]);
    end;


    function    TWrenMidiData.GetDataLongInt( anIndex: Integer): LongInt;
    begin
      Result := LongInt( DataCardinal[ anIndex]);
    end;


    function    TWrenMidiData.GetDataFourCC( anIndex: Integer): string;
    var
      i : Integer;
      C : Char;
    begin
      Result := '';

      for i := 0 to 3
      do begin
        C := Char( DataByte[ i + anIndex]);
        Result := Result + C;
      end;
    end;


//  public

    function    TWrenMidiData.Remaining( anOffset: Integer): Integer;
    begin
      Result := Count - anOffset;
    end;


{ ========
  TWrenMidiHeader = class( TWrenMidiData)
  private
    FTrackFormat : Word;
    FTracks      : Word;
    FDivision    : Word;
  public
    property    TrackFormat : Word read FTrackFormat;
    property    Tracks      : Word read FTracks;
    property    Division    : Word read FDivision;
  public
}

    constructor TWrenMidiHeader.CreateFromData( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer); // override;
    begin
      Assert( aSize = 6);
      FTrackFormat := aData.DataWord[ anOffset + 0];
      FTracks      := aData.DataWord[ anOffset + 2];
      FDivision    := aData.DataWord[ anOffset + 4];
    end;


    procedure   TWrenMidiHeader.Dump( anInd: Integer; const aStringList: TStringList);
    begin
      with aStringList
      do begin
        Add( Indent( anInd, 'header('));
        Add( Indent( anInd + 1, 'TrackFormat : %d', [ TrackFormat]));
        Add( Indent( anInd + 1, 'Tracks      : %d', [ Tracks     ]));
        Add( Indent( anInd + 1, 'Division    : %d', [ Division   ]));
        Add( Indent( anInd, ')'));
      end;
    end;


{ ========
  TWrenMidiDeltaTime = class
  private
    FDeltaTime: Integer;
  public
    property    DeltaTime: Integer read FDeltaTime;
  public
}

    constructor TWrenMidiDeltaTime.CreateParseFromData( const aData: TBytes; var anOffset: Integer);
    var
      aByte  : Byte;
      aCount : Integer;
    begin
      inherited Create;
      FDeltaTime := 0;
      aCount     := 0;

      repeat
        aByte := aData[ anOffset];
        Inc( aCount);

        if aCount > 4
        then raise EWrenMidiParse.Create( 'Variable sized data too long');

        Inc( anOffset);
        FDeltaTime := 128 * DeltaTime + Integer( aByte and $7f);
      until aByte and $80 = 0;
    end;


    procedure   TWrenMidiDeltaTime.Dump( anInd: Integer; const aStringList: TStringList);
    begin
      aStringList.Add( Indent( anInd, 'DeltaTime : %d', [ DeltaTime]));
    end;


{ ========
  TWrenMidiEvent = class
  public
}

    procedure    TWrenMidiEvent.Dump( anInd: Integer; const aStringList: TStringList); // virtual;
    begin
      aStringList.Add( Indent( anInd, 'MidiEvent : %s', [ ClassName]));
    end;


    class function  TWrenMidiEvent.CreateParseFromData( const aData: TBytes; var anOffset: Integer; var RunningStatus: Byte): TWrenMidiEvent;
    var
      aBytes : TBytes;

      function Remaining: Integer;
      // Local function
      begin
        Result := Length( aData) - anOffset;
      end;

      function ByteCount: Integer;
      // Local function
      begin
        Result := Length( aBytes);
      end;

      function PeekByte: Byte;
      // Local function
      begin
        Result := aData[ anOffset];
      end;

      function GetByte: Byte;
      // Local function
      begin
        if Remaining > 0
        then begin
          Result := PeekByte;
          Inc( anOffset);
        end
        else raise EWrenMidiParse.Create( 'not enough data left for expected midi event');
      end;

      procedure AddByte( aByte: Byte);
      // Local function
      begin
        SetLength( aBytes, ByteCount + 1);
        aBytes[ ByteCount - 1] := aByte;
      end;

      procedure CopyBytes( aCount: Integer);
      // Local function
      var
        i : Integer;
      begin
        for i := 1 to aCount
        do AddByte( GetByte);
      end;

      function ParseNoteOff: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 2);
        Result := TWrenMidiNoteOffEvent.CreateFromBytes( aBytes);
      end;

      function ParseNoteOn: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 2);
        Result := TWrenMidiNoteOnEvent.CreateFromBytes( aBytes);
      end;

      function ParseKetPressure: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 2);
        Result := TWrenMidiPolyKeyPressureEvent.CreateFromBytes( aBytes);
      end;

      function ParseControlChange: TWrenMidiEvent;
      // Local function
      var
        aCC    : Byte;
        aValue : Byte;
      begin
        AddByte( RunningStatus);
        Result := nil;                   // To stop compiler from complaining.
        aCC    := GetByte and $7f;
        aValue := GetByte and $7f;
        AddByte( aCC);                   // Add the CC
        AddByte( aValue);                // Add the Value

        if aCC < CCAllSoundOff
        then Result := TWrenMidiControlChangeEvent.CreateFromBytes( aBytes)
        else begin
          case aCC of

            CCAllSoundOff :

              begin
                if aValue = 0
                then Result := TWrenMidiAllSoundOffEvent.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for AllSoundOff event');
              end;

            CCResetAllControllers :

              begin
                if aValue = 0
                then Result := TWrenMidiResetAllControllersEvent.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for ResetAllControllers event');
              end;

            CCLoccalControl :

              begin
                if aValue in [ 0, 127]
                then Result := TWrenMidiLocalControlEvent.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for LocalControl event');
              end;

            CCAllNotesOff :

              begin
                if aValue = 0
                then Result := TWrenMidiAllNotesOffEvent.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for AllNotesOff event');
              end;

            CCOmniModeOff :

              begin
                if aValue = 0
                then Result := TWrenMidiOmniModeOff.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for OmniModeOff event');
              end;

            CCOmniModeOn :

              begin
                if aValue = 0
                then Result := TWrenMidiOmniModeOn.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for OmniModeOn event');
              end;

            CCMonoModeOn :

              begin
                Result := TWrenMidiMonoModeOn.CreateFromBytes( aBytes);
              end;

            CCPolyModeOn :

              begin
                if aValue = 0
                then Result := TWrenMidiPolyModeOn.CreateFromBytes( aBytes)
                else raise EWrenMidiParse.Create( 'invalid value for PolyModeOn event');
              end;

            else EWrenMidiParse.Create( 'unhandled channel mode message type, internal coding error');
          end;
        end;
      end;

      function ParseProgramChange: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 1);
        Result := TWrenMidiProgramChangeEvent.CreateFromBytes( aBytes);
      end;

      function ParseChannelPressure: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 1);
        Result := TWrenMidiChannelPressureEvent.CreateFromBytes( aBytes);
      end;

      function ParsePitchBend: TWrenMidiEvent;
      // Local function
      begin
        AddByte( RunningStatus);
        CopyBytes( 2);
        Result := TWrenMidiPitchBendEvent.CreateFromBytes( aBytes);
      end;

      function ParseSysExData: TWrenMidiEvent;
      // Local function
      var
        aCount: Byte;
      begin
        aCount := GetByte;
        CopyBytes( aCount);
        Result := TWrenMidiSystemExclusiveEvent.CreateFromBytes( aBytes);
      end;

      function ParseMeta: TWrenMidiEvent;
      // Local function
      var
        aType  : Byte;
        aCount : Byte;
      begin
        aType := GetByte;

        if aType and $80 = 0
        then begin
          aCount := GetByte;
          CopyBytes( aCount);

          case aType of
            CMSequenceNumber    : Result := TWrenMidiSequenceNumberEvent   .CreateFromBytes( aBytes);
            CMTextEvent         : Result := TWrenMidiTextEventEvent        .CreateFromBytes( aBytes);
            CMCopyRightNotice   : Result := TWrenMidiCopyRightNoticeEvent  .CreateFromBytes( aBytes);
            CMSequenceTrackName : Result := TWrenMidiSequenceTrackNameEvent.CreateFromBytes( aBytes);
            CMInstrumentName    : Result := TWrenMidiInstrumentNameEvent   .CreateFromBytes( aBytes);
            CMLyric             : Result := TWrenMidiLyricEvent            .CreateFromBytes( aBytes);
            CMMarker            : Result := TWrenMidiMarkerEvent           .CreateFromBytes( aBytes);
            CMCuePoint          : Result := TWrenMidiCuePointEvent         .CreateFromBytes( aBytes);
            CMMidiChannelPrefix : Result := TWrenMidiMidiChannelPrefixEvent.CreateFromBytes( aBytes);
            CMEndOfTrack        : Result := TWrenMidiEndOfTrackEvent       .CreateFromBytes( aBytes);
            CMSetTempo          : Result := TWrenMidiSetTempoEvent         .CreateFromBytes( aBytes);
            CMSMTPOffset        : Result := TWrenMidiSMTPOffsetEvent       .CreateFromBytes( aBytes);
            CMTimeSignature     : Result := TWrenMidiTimeSignatureEvent    .CreateFromBytes( aBytes);
            CMKeySignature      : Result := TWrenMidiKeySignatureEvent     .CreateFromBytes( aBytes);
            CMSequencerSpecific : Result := TWrenMidiSequencerSpecificEvent.CreateFromBytes( aBytes);
            else                  Result := TWrenMidiMetaEvent             .CreateFromBytes( aBytes);
          end;
        end
        else raise EWrenMidiParse.Create( 'invalid meta type seen, can not parse the data');
      end;


      function ParseSysEx: TWrenMidiEvent;
      // Local function
      begin
        case RunningStatus and $0f of

          SSysEx0 :

            begin
              AddByte( RunningStatus);
              Result := ParseSysexData;
            end;

          SSysex7 : Result := ParseSysexData;

          SMeta   : Result := ParseMeta;

          else raise EWrenMidiParse.Create( 'unexpected sysex/meta value, can not parse the data');
        end;

        RunningStatus := 0; // These all break running status
      end;

      function ParseEvent: TWrenMidiEvent;
      // Local function
      begin
        case RunningStatus and $f0 of
          CNoteOff        : Result := ParseNoteOff;
          CNoteOn         : Result := ParseNoteOn;
          CKeyPressure    : Result := ParseKetPressure;
          CControlChange  : Result := ParseControlChange;
          CProgramChange  : Result := ParseProgramChange;
          CChannelPressure: Result := ParseChannelPressure;
          CPitchBend      : Result := ParsePitchBend;
          CSysEx          : Result := ParseSysEx;
          else raise EWrenMidiParse.Create( 'unexpected running status/event value, can not parse the data');
        end;
      end;

    // class function  TWrenMidiEvent.CreateParseFromData( const aData: TBytes; var anOffset: Integer; var RunningStatus: Byte): TWrenMidiEvent;
    var
      aByte : Byte;
    begin
      SetLength( aBytes, 0);

      if ( PeekByte and $80 <> 0) or ( RunningStatus and $80 = 0)
      then begin
        aByte := GetByte;

        if aByte and $80 <> 0
        then RunningStatus := aByte
        else raise EWrenMidiParse.Create( 'no running status available, a midi event was expected but saw midi data');
      end;

      Result := ParseEvent;
    end;


{ ========
  TWrenMidiChannelEvent = class( TWrenMidiEvent)
  private
    FChannel: Byte;
  public
    property    Channel: Byte read FChannel;
  public
}

    constructor TWrenMidiChannelEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FChannel := aData[ 0] and $0f;
    end;


    procedure   TWrenMidiChannelEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Channel : %d', [ Channel]));
    end;


{ ========
  TWrenMidiChannelVoiceEvent = class( TWrenMidiChannelEvent)
  private
    FEvent : Byte;
  public
    property    Event : Byte read FEvent;
  public
}

    constructor TWrenMidiChannelVoiceEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FEvent := aData[ 0] and $f0;
    end;


    procedure   TWrenMidiChannelVoiceEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Event : %d', [ Event]));
    end;


{ ========
  TWrenMidiMetaEvent = class( TWrenMidiEvent)
  var
    FData : Tbytes;
  public
    property    Data : TBytes read FData;
  public
}

    constructor TWrenMidiMetaEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      SetLength( FData, Length( aData));

      for i := 0 to Length( FData) - 1
      do FData[ i] := aData[ i];
    end;


{ ========
  TWrenMidiNoteEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FNote     : Byte;
    FVelocity : Byte;
  public
    property    Note     : Byte read FNote;
    property    Velocity : Byte read FVelocity;
  public
}

    constructor TWrenMidiNoteEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FKey      := aData[ 1] and $7f;
      FVelocity := aData[ 2] and $7f;
    end;


    procedure   TWrenMidiNoteEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Key      : %d', [ Key     ]));
      aStringList.Add( Indent( anInd, 'Velocity : %d', [ Velocity]));
    end;


{ ========
  TWrenMidiControlChangeEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FController : Byte;
    FValue      : Byte;
  public
    property    Controller : Byte read FController;
    property    Value      : Byte read FValue;
  public
}

    constructor TWrenMidiControlChangeEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FController := aData[ 1] and $7f;
      FValue      := aData[ 2] and $7f;
    end;


    procedure   TWrenMidiControlChangeEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Controller : %d', [ Controller]));
      aStringList.Add( Indent( anInd, 'Value      : %d', [ Value     ]));
    end;


{ ========
  TWrenMidiProgramChangeEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FProgramNr : Byte;
  public
    property    ProgramNr : Byte read FProgramNr;
  public
}

    constructor TWrenMidiProgramChangeEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FProgramNr := aData[ 1] and $7f;
    end;


    procedure   TWrenMidiProgramChangeEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'ProgramNr : %d', [ ProgramNr]));
    end;


{ ========
  TWrenMidiChannelPressureEvent = class( TWrenMidiChannelVoiceEvent)
  private
    FVelocity : Byte;
  public
    property    Velocity : Byte read FVelocity;
  public
}

    constructor TWrenMidiChannelPressureEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FVelocity := aData[ 1] and $7f;
    end;


    procedure   TWrenMidiChannelPressureEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Velocity : %d', [ Velocity]));
    end;


{ ========
  TWrenMidiPitchBendEvent = class( TWrenMidiChannelVoiceEvent)
  var
    FBend : Word;
  public
    property    Bend : Word read FBend;
  public
}

    constructor TWrenMidiPitchBendEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FBend := Word( aData[ 1] and $7f) + 128 * Word( aData[ 2] and $7f);
    end;


    procedure   TWrenMidiPitchBendEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Bend : %d', [ Bend]));
    end;


{ ========
  TWrenMidiLocalControlEvent = class( TWrenMidiChannelModeEvent)
  private
    FActive : Boolean;
  public
    property    Active : Boolean read FActive;
  public
}

    constructor TWrenMidiLocalControlEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FActive := aData[ 2] <> 0;
    end;


    procedure   TWrenMidiLocalControlEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Active : %s', [ BoolToStr( Active)]));
    end;


{ ========
  TWrenMidiMonoModeOn = class( TWrenMidiChannelModeEvent)
  private
    FChannelCount : Byte;
  public
    property    ChannelCount : Byte read FChannelCount;
  public
}

    constructor TWrenMidiMonoModeOn.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FChannelCount := aData[ 2];
    end;


    procedure   TWrenMidiMonoModeOn.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'ChannelCount : %s', [ ChannelCount]));
    end;


{ ========
  TWrenMidiSystemExclusiveEvent = class( TWrenMidiEvent)
  private
    FData : TBytes;
  public
    property    Data: TBytes read FData;
  public
}

    constructor TWrenMidiSystemExclusiveEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      SetLength( FData, Length( aData));

      for i := 0 to Length( FData) - 1
      do FData[ i] := aData[ i];
    end;


{ ========
  TWrenMidiSequenceNumberEvent = class( TWrenMidiMetaEvent)
  private
    FSequenceNumber : Word;
  public
    property    SequenceNumber : Word read FSequenceNumber;
  public
}

    constructor TWrenMidiSequenceNumberEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FSequenceNumber := 256 * Word( aData[ 0]) + Word( aData[ 1]);
    end;


    procedure   TWrenMidiSequenceNumberEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'SequenceNumber : %d', [ SequenceNumber]));
    end;


{ ========
  TWrenMidiTextEventEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiTextEventEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiTextEventEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiCopyRightNoticeEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiCopyRightNoticeEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiCopyRightNoticeEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiSequenceTrackNameEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiSequenceTrackNameEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiSequenceTrackNameEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiInstrumentNameEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiInstrumentNameEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiInstrumentNameEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiLyricEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiLyricEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiLyricEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiMarkerEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiMarkerEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiMarkerEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiCuePointEvent = class( TWrenMidiMetaEvent)
  private
    FText : string;
  public
    property    Text : string read FText;
  public
}

    constructor TWrenMidiCuePointEvent.CreateFromBytes( const aData: TBytes); // override;
    var
      i : Integer;
    begin
      inherited Create;
      FText := '';

      for i := 0 to Length( aData) - 1
      do FText := FText + Char( aData[ i]);
    end;


    procedure   TWrenMidiCuePointEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Text : %s', [ Text]));
    end;


{ ========
  TWrenMidiMidiChannelPrefixEvent = class( TWrenMidiMetaEvent)
  private
    FChannel : Byte;
  public
    property    Channel: Byte read FChannel;
  public
}
    constructor TWrenMidiMidiChannelPrefixEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FChannel := aData[ 0];
    end;


    procedure   TWrenMidiMidiChannelPrefixEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Channel : %d', [ Channel]));
    end;


{ ========
  TWrenMidiSetTempoEvent = class( TWrenMidiMetaEvent)
  private
    FTempo : Cardinal;         // s / quarter note
  public
    property    Tempo : Cardinal read FTempo;
  public
}

    constructor TWrenMidiSetTempoEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FTempo := 256 * 256 * Cardinal( aData[ 0]) + 256 * Cardinal( aData[ 1]) + Cardinal( aData[ 2]);
    end;


    procedure   TWrenMidiSetTempoEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Tempo : %d', [ Tempo]));
    end;


{ ========
  TWrenMidiSMTPOffsetEvent = class( TWrenMidiMetaEvent)
  private
    FHour            : Byte;
    FMinute          : Byte;
    FSecond          : Byte;
    FFrame           : Byte;
    FFractionalFrame : Byte;
  public
    property    Hour            : Byte read FHour;
    property    Minute          : Byte read FMinute;
    property    Second          : Byte read FSecond;
    property    Frame           : Byte read FFrame;
    property    FractionalFrame : Byte read FFractionalFrame;
  public
}

    constructor TWrenMidiSMTPOffsetEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FHour            := aData[ 0];
      FMinute          := aData[ 1];
      FSecond          := aData[ 2];
      FFrame           := aData[ 3];
      FFractionalFrame := aData[ 4];
    end;


    procedure   TWrenMidiSMTPOffsetEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Hour            : %d', [ Hour           ]));
      aStringList.Add( Indent( anInd, 'Minute          : %d', [ Minute         ]));
      aStringList.Add( Indent( anInd, 'Second          : %d', [ Second         ]));
      aStringList.Add( Indent( anInd, 'Frame           : %d', [ Frame          ]));
      aStringList.Add( Indent( anInd, 'FractionalFrame : %d', [ FractionalFrame]));
    end;


{ ========
  TWrenMidiTimeSignatureEvent = class( TWrenMidiMetaEvent)
  private
    FNumerator                  : Byte;
    FDenominator                : Cardinal;
    FClocksPerTick              : Byte;
    FSemisemiquaversPer24Clocks : Byte;                   // 1/32th notes per 24 MIDI clocks
  public
    property    Numerator                  : Byte     read FNumerator;
    property    Denominator                : Cardinal read FDenominator;
    property    ClocksPerTick              : Byte     read FClocksPerTick;
    property    SemisemiquaversPer24Clocks : Byte     read FSemisemiquaversPer24Clocks;
  public
}

    constructor TWrenMidiTimeSignatureEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FNumerator                  := aData[ 0];
      FDenominator                := Trunc( Power( 2, aData[ 1]));
      FClocksPerTick              := aData[ 2];
      FSemisemiquaversPer24Clocks := aData[ 3];
    end;


    procedure   TWrenMidiTimeSignatureEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'Numerator                  : %d', [ Numerator ]));
      aStringList.Add( Indent( anInd, 'Denominator                : %d', [ Denominator]));
      aStringList.Add( Indent( anInd, 'ClocksPerTick              : %d', [ ClocksPerTick]));
      aStringList.Add( Indent( anInd, 'SemisemiquaversPer24Clocks : %d', [ SemisemiquaversPer24Clocks]));
    end;


{ ========
  TWrenMidiKeySignatureEvent = class( TWrenMidiMetaEvent)
  private
    FNumberOfSharps : Byte;
    FMajorMinor     : Byte;
  public
    property    NumberOfSharps : Byte read FNumberOfSharps;
    property    MajorMinor     : Byte read FMajorMinor;
  public
}

    constructor TWrenMidiKeySignatureEvent.CreateFromBytes( const aData: TBytes); // override;
    begin
      inherited Create;
      FNumberOfSharps := aData[ 0];
      FMajorMinor     := aData[ 1];
    end;


    procedure   TWrenMidiKeySignatureEvent.Dump( anInd: Integer; const aStringList: TStringList); // override;
    begin
      inherited;
      aStringList.Add( Indent( anInd, 'NumberOfSharps : %d', [ NumberOfSharps]));
      aStringList.Add( Indent( anInd, 'MajorMinor     : %d', [ MajorMinor    ]));
    end;


{ ========
  TWrenMidiTimedEvent = class
  private
    FDeltaTime : TWrenMidiDeltaTime;
    FEvent     : TWrenMidiEvent;
  public
    property    DeltaTime : TWrenMidiDeltaTime read FDeltaTime;
    property    Event     : TWrenMidiEvent     read FEvent;
  public
}

    constructor TWrenMidiTimedEvent.Create( const aDeltaTime: TWrenMidiDeltaTime; anEvent: TWrenMidiEvent);
    begin
      inherited Create;
      FDeltaTime := aDeltaTime;
      FEvent     := anEvent;
    end;


    destructor  TWrenMidiTimedEvent.Destroy; // override;
    begin
      FreeAndNil( FDeltaTime);
      FreeAndNil( FEvent    );
      inherited;
    end;


    procedure   TWrenMidiTimedEvent.Dump( anInd: Integer; const aStringList: TStringList);
    begin
      aStringList.Add( Indent( anInd, 'timedEvent('));
      DeltaTime.Dump( anInd + 1, aStringList);
      Event    .Dump( anInd + 1, aStringList);
      aStringList.Add( Indent( anInd, ')'));
    end;


//  public

    function    TWrenMidiTimedEvent.IsNoEvent: Boolean;
    begin
      Result := Event is TWrenMidiNoEvent;
    end;


//  public

    function    TWrenMidiTimedEvent.IsChannelVoiceEvent: Boolean;
    begin
      Result := Event is TWrenMidiChannelVoiceEvent;
    end;


    function    TWrenMidiTimedEvent.IsNoteEvent: Boolean;
    begin
      Result := Event is TWrenMidiNoteEvent;
    end;


    function    TWrenMidiTimedEvent.IsNoteOffEvent: Boolean;
    begin
      Result := Event is TWrenMidiNoteOffEvent;
    end;


    function    TWrenMidiTimedEvent.IsNoteOnEvent: Boolean;
    begin
      Result := Event is TWrenMidiNoteOnEvent;
    end;


    function    TWrenMidiTimedEvent.IsPolyKeyPressureEveent: Boolean;
    begin
      Result := Event is TWrenMidiPolyKeyPressureEvent;
    end;


    function    TWrenMidiTimedEvent.IsControlChange: Boolean;
    begin
      Result := Event is TWrenMidiControlChangeEvent;
    end;


    function    TWrenMidiTimedEvent.IsProgramChange: Boolean;
    begin
      Result := Event is TWrenMidiProgramChangeEvent;
    end;


    function    TWrenMidiTimedEvent.IsChannelPressure: Boolean;
    begin
      Result := Event is TWrenMidiChannelPressureEvent;
    end;


    function    TWrenMidiTimedEvent.IsPitchBendEvent: Boolean;
    begin
      Result := Event is TWrenMidiPitchBendEvent;
    end;


//  public

    function    TWrenMidiTimedEvent.IsChannelModeEvent: Boolean;
    begin
      Result := Event is TWrenMidiChannelModeEvent;
    end;


    function    TWrenMidiTimedEvent.IsAllSoundOffEvent: Boolean;
    begin
      Result := Event is TWrenMidiAllSoundOffEvent;
    end;


    function    TWrenMidiTimedEvent.IsResetAllControllersEvent: Boolean;
    begin
      Result := Event is TWrenMidiResetAllControllersEvent;
    end;


    function    TWrenMidiTimedEvent.IsLocalControlEvent: Boolean;
    begin
      Result := Event is TWrenMidiLocalControlEvent;
    end;


    function    TWrenMidiTimedEvent.IsAllNotesOffEvent: Boolean;
    begin
      Result := Event is TWrenMidiAllNotesOffEvent;
    end;


//  public

    function    TWrenMidiTimedEvent.IsSystemExclusiveEvent: Boolean;
    begin
      Result := Event is TWrenMidiSystemExclusiveEvent;
    end;


//  public

    function    TWrenMidiTimedEvent.IsMetaEvent: Boolean;
    begin
      Result := Event is TWrenMidiMetaEvent;
    end;


    function    TWrenMidiTimedEvent.IsSequenceNumberEvent: Boolean;
    begin
      Result := Event is TWrenMidiSequenceNumberEvent;
    end;


    function    TWrenMidiTimedEvent.IsTextEventEvent: Boolean;
    begin
      Result := Event is TWrenMidiTextEventEvent;
    end;


    function    TWrenMidiTimedEvent.IsCopyRightNoticeEvent: Boolean;
    begin
      Result := Event is TWrenMidiCopyRightNoticeEvent;
    end;


    function    TWrenMidiTimedEvent.IsSequenceTrackNameEvent: Boolean;
    begin
      Result := Event is TWrenMidiSequenceTrackNameEvent;
    end;


    function    TWrenMidiTimedEvent.IsInstrumentNameEvent: Boolean;
    begin
      Result := Event is TWrenMidiInstrumentNameEvent;
    end;


    function    TWrenMidiTimedEvent.IsLyricEvent: Boolean;
    begin
      Result := Event is TWrenMidiLyricEvent;
    end;


    function    TWrenMidiTimedEvent.IsMarkerEvent: Boolean;
    begin
      Result := Event is TWrenMidiMarkerEvent;
    end;


    function    TWrenMidiTimedEvent.IsCuePointEvent: Boolean;
    begin
      Result := Event is TWrenMidiCuePointEvent;
    end;


    function    TWrenMidiTimedEvent.IsMidiChannelPrefixEvent: Boolean;
    begin
      Result := Event is TWrenMidiMidiChannelPrefixEvent;
    end;


    function    TWrenMidiTimedEvent.IsEndOfTrackEvent: Boolean;
    begin
      Result := Event is TWrenMidiEndOfTrackEvent;
    end;


    function    TWrenMidiTimedEvent.IsSetTempoEvent: Boolean;
    begin
      Result := Event is TWrenMidiSetTempoEvent;
    end;


    function    TWrenMidiTimedEvent.IsSMTPOffsetEvent: Boolean;
    begin
      Result := Event is TWrenMidiSMTPOffsetEvent;
    end;


    function    TWrenMidiTimedEvent.IsTimeSignatureEvent: Boolean;
    begin
      Result := Event is TWrenMidiTimeSignatureEvent;
    end;


    function    TWrenMidiTimedEvent.IsKeySignatureEvent: Boolean;
    begin
      Result := Event is TWrenMidiKeySignatureEvent;
    end;


    function    TWrenMidiTimedEvent.IsSequencerSpecificEvent: Boolean;
    begin
      Result := Event is TWrenMidiSequencerSpecificEvent;
    end;


{ ========
  TWrenMidiTimedEvents = class( TWrenMidiData)
  private
    FTimedEvents   : array of TWrenMidiTimedEvent;
    FOffset        : Integer;
    FRunningStatus : Byte;
    FCurrent       : Integer;
  public
    property    TimedEventCount               : Integer             read GetTimedEventCount;
    property    TimedEvent[ anIndex: Integer] : TWrenMidiTimedEvent read GetTimedEvent;
  private
}

    function    TWrenMidiTrack.GetTimedEventCount: Integer;
    begin
      Result := Length( FTimedEvents);
    end;


    function    TWrenMidiTrack.GetTimedEvent( anIndex: Integer): TWrenMidiTimedEvent;
    begin
      Result := FTimedEvents[ anIndex];
    end;


    procedure   TWrenMidiTrack.AddTimedEvent( const aTimedEvent: TWrenMidiTimedEvent);
    begin
      SetLength( FTimedEvents, TimedEventCount + 1);
      FTimedEvents[ TimedEventCount - 1] := aTimedEvent;
    end;


    procedure   TWrenMidiTrack.ParseTimedEvent;
    var
      aDeltaTime  : TWrenMidiDeltaTime;
      anEvent     : TWrenMidiEvent;
      aTimedEvent : TWrenMidiTimedEvent;
    begin
      aDeltaTime  := TWrenMidiDeltaTime .CreateParseFromData( FData, FOffset);
      anEvent     := TWrenMidiEvent     .CreateParseFromData( FData, FOffset, FRunningStatus);
      aTimedEvent := TWrenMidiTimedEvent.Create( aDeltaTime, anEvent);
      AddTimedEvent( aTimedEvent);
    end;


    procedure   TWrenMidiTrack.Parse;
    begin
      FOffset        := 0;
      FRunningStatus := 0;

      while Remaining( FOffset) > 0
      do ParseTimedEvent;
    end;


//  public

    constructor TWrenMidiTrack.CreateFromData( const aData: TWrenMidiData; anOffset: Integer; aSize: Integer); // override;
    begin
      inherited;
      Parse;
    end;


    destructor  TWrenMidiTrack.Destroy; // override;
    begin
      while TimedEventCount > 0
      do begin
        FreeAndNil( FTimedEvents[ TimedEventCount - 1]);
        SetLength(  FTimedEvents, TimedEventCount - 1 );
      end;

      inherited;
    end;


    procedure   TWrenMidiTrack.Dump( anInd: Integer; const aStringList: TStringList);
    var
      i : Integer;
    begin
      aStringList.Add( Indent( anInd, 'track('));

      for i := 0 to TimedEventCount - 1
      do TimedEvent[ i].Dump( anInd + 1, aStringList);

      aStringList.Add( Indent( anInd, ')'));
    end;


    procedure   TWrenMidiTrack.Reset;
    begin
      FCurrent := 0;
    end;


    function    TWrenMidiTrack.NextTimedEvent: TWrenMidiTimedEvent;
    begin
      if FCurrent < TimedEventCount
      then begin
        Result := TimedEvent[ FCurrent];
        Inc( FCurrent);
      end
      else Result := nil;
    end;


{ ========
  TWrenMiditracks = class
  private
    FTracks : array of TWrenMidiTrack;
  public
    property    TrackCount                : Integer read GetTrackCount;
    property    Tracks[ anIndex: Integer] : TWrenMidiTrack read GetTrack;
  private
}

    function    TWrenMiditracks.GetTrackCount: Integer;
    begin
      Result := Length( FTracks);
    end;


    function    TWrenMiditracks.GetTrack( anIndex: Integer): TWrenMidiTrack;
    begin
      Result := FTracks[ anIndex];
    end;


//  public

    destructor  TWrenMiditracks.Destroy; // override;
    begin
      while TrackCount > 0
      do begin
        FreeAndNil( FTracks[ TrackCount - 1]);
        SetLength ( FTracks, TrackCount - 1 );
      end;

      inherited;
    end;


    procedure   TWrenMiditracks.Dump( anInd: Integer; const aStringList: TStringList);
    var
      i : Integer;
    begin
      aStringList.Add( Indent( anInd, 'tracks('));

      for i := 0 to TrackCount - 1

      do FTracks[ i].Dump( anInd + 1, aStringList);
      aStringList.Add( Indent( anInd, ')'));
    end;


    procedure   TWrenMiditracks.AddTrack( const aTrack: TWrenMidiTrack);
    begin
      SetLength( FTracks, TrackCount + 1);
      FTracks[ TrackCount - 1] := aTrack;
    end;


    procedure   TWrenMiditracks.Reset;
    var
      i : Integer;
    begin
      for i := 0 to TrackCount - 1
      do FTracks[ i].Reset;
    end;


    function    TWrenMiditracks.NextTimedEvent( aTrack: Integer): TWrenMidiTimedEvent;
    begin
      if ( aTrack < 0) and ( TrackCount > 0)
      then Result := FTracks[ 0].NextTimedEvent
      else if ( aTrack >= 0) and ( aTrack < TrackCount)
      then Result := FTracks[ aTrack].NextTimedEvent
      else Result := nil;
    end;


{ ========
  TWrenMidiReader = class
  private
    FFileName : string;
    FData     : TWrenMidiData;
    FOffset   : Integer;
    FHeader   : TWrenMidiHeader;
    FTracks   : TWrenMiditracks;
  public
    property    TrackCount: Integer read GetTrackCount;
    property    Format      : Word  read GetFormat;
    property    Tracks      : Word  read GetTracks;
    property    Division    : Word  read GetDivision;
  private
}

    function    TWrenMidiReader.GetFormat: Word;
    begin
      Result := FHeader.TrackFormat;
    end;


    function    TWrenMidiReader.GetTracks: Word;
    begin
      Result := FHeader.Tracks;
    end;


    function    TWrenMidiReader.GetDivision: Word;
    begin
      Result := FHeader.Division;
    end;


//  private

    function    TWrenMidiReader.GetTrackCount: Integer;
    begin
      Result := FTracks.TrackCount;
    end;


    procedure   TWrenMidiReader.parseHeader;
    var
      aFourCC : string;
      aCount  : Cardinal;
    begin
      if Assigned( FHeader)
      then FreeAndNil( FHeader);

      if FData.Remaining( FOffset) < 14
      then raise EWrenMidiParse.Create( 'remaining size < 14 bytes, no proper header present');

      aFourCC := FData.DataFourCC[ FOffset];

      if not SameText( aFourCC, 'MThd')
      then raise EWrenMidiParse.Create( 'no header valid marker found');

      Inc( FOffset, 4);

      aCount  := FData.DataCardinal[ FOffset];

      if aCount <> 6
      then raise EWrenMidiParse.Create( 'header data size reported <> 6, no proper header present');

      Inc( FOffset, 4);
      FHeader := TWrenMidiHeader.CreateFromData( FData, FOffset, 6);
      Inc( FOffset, 6);
    end;


    procedure   TWrenMidiReader.ParseTrack;
    var
      aFourCC : string;
      aSize   : Cardinal;
      aTrack  : TWrenMidiTrack;
    begin
      if FData.Remaining( FOffset) < 8
      then raise EWrenMidiParse.Create( 'less than 8 bytes left, no proper track can be present');

      aFourCC := FData.DataFourCC[ FOffset];

      if not SameText( aFourCC, 'MTrk')
      then raise EWrenMidiParse.Create( 'No proper track marker found');

      Inc( FOffset, 4);

      aSize := FData.DataCardinal[ FOffset];
      Inc( FOffset, 4);

      aTrack := TWrenMidiTrack.CreateFromData( FData, FOffset, aSize);
      FTracks.AddTrack( aTrack);
      Inc( FOffset, aSize);
    end;


    procedure   TWrenMidiReader.ParseTracks;
    var
      i : Integer;
    begin
      if Assigned( FTracks)
      then FreeAndNil( FTracks);

      FTracks := TWrenMiditracks.Create;

      for i := 0 to FHeader.Tracks - 1
      do ParseTrack;
    end;


    procedure   TWrenMidiReader.Parse;
    begin
      FOffset := 0;
      ParseHeader;
      ParseTracks;
    end;


//  public

    constructor TWrenMidiReader.Create;
    begin
      inherited;
      FData   := TWrenMidiData  .Create;
      FHeader := TWrenMidiHeader.Create;
      FTracks := TWrenMiditracks.Create;
    end;


    destructor  TWrenMidiReader.Destroy; // override;
    begin
      FreeAndNil( FTracks);
      FreeAndNil( FHeader);
      FreeAndNil( FData  );
    end;


    procedure   TWrenMidiReader.Dump( anInd: Integer; const aStringList: TStringList);
    begin
      aStringList.Add( Indent( anInd, 'reader('));
      FHeader.Dump( anInd + 1, aStringList);
      FTracks.Dump( anInd + 1, aStringList);
      aStringList.Add( Indent( anInd, ')'));
    end;


    procedure   TWrenMidiReader.ReadFile( const aFileName: string);
    begin
      if FileExists( aFileName)
      then begin
        if Assigned( FData)
        then FreeAndNil( FData);
        FFileName := aFileName;
        FData     := TWrenMidiData.CreateFromFile( aFileName);
        Parse;
      end
      else raise EWrenMidiParse.Create( 'MIDI file not found');
    end;


    procedure   TWrenMidiReader.Reset;
    begin
      FTracks.Reset;
    end;


    function    TWrenMidiReader.NextTimedEvent( aTrack: Integer): TWrenMidiTimedEvent;
    begin
      if Assigned( FTracks)
      then Result := FTracks.NextTimedEvent( aTrack)
      else Result := nil;
    end;


end.

