unit mod_external;

{

   COPYRIGHT 2017 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl




  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
  = = DOCS  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =
  = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =


  ** Record field alignment for passed data structures ( is / should be) set to quad-word (8 bytes).

    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

  ** The PMEIdentifier type currently is a pointer to a zero terminated 16 bit UTF-16 character array (PChar)
  ** The TMEValue type currently is a double

    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

  ** The functions external code (sh|c)ould implement:

    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GetCaps( var IsInput, IsOutput, IsSlow, IsMidi, IsOSC: Integer); cdecl;
    void __cdecl GetCaps( int * IsInput, int * IsOutput, int * IsSlow, int * IsFast, int * IsMidi, int * IsOSC);

      Called to get information about what the external code implements. Returning a zero means not implemented
      returning a 1 means the feature is implemented. This function must be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure Create( const aName: PMEIdentifier); cdecl;
    void __cdecl Create( PMEIdentifier aName);

      Called during creation of a module, during this call no DoTick or DoSlowTick calls will b made to the external
      code. The external code should use this to set up it's internal data sructures. This function needs not be
      implemented


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure Destroy; cdecl;
    void __cdecl Destroy( void);

      Called during destruction of a module, during or after this call no DoTick or DoSlowTick calls will be made to the
      external code. The external code should use this to clean up it's internal data structures. This dunction needs
      not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure Reset; cdecl;
    void __cdecl Reset( void);

      Called on automatic or manual patch resets, external code may use this to perfome reset actions. Reset is being
      called from the tick method, and it may be either be called for one specific tick or for a couple of times in
      succession. This function needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure SetDefaults; cdecl;
    void __cdecl SetDefaults( void);

      Called during module setup, external code may use this to set up it's default values. This function needs not be
      implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure SampleRateChanged( aSystemRate, aControlRate: TMEValue); cdecl;
    void __cdecl SampleRateChanged( TMEValue aSystemRate, TMEValue aControlRate);

      Called when the samplerate was changed, during this call no calls will be made to DoTick or DoSlowTick. The
      external code can use this to recalculate sample rate dependent data. This function needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure TuningChanged( const aReferenceA, aNotesPerOctave, aMiddleNote: TMEValue); cdecl;
    void __cdecl( TMEValue aReferenceA, TMEValue aNotesPerOctave, TMEValue aMiddleNote);

      Called when the system tuning was changed, during this call no calls will be made to DoTick or DoSlowTick. The
      external code can use this to recalculate sample tuning dependent data. This functions needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure SetStringValue( const aName: PMEIdentifier; const aValue: PMEIdentifier); cdecl;
    void __cdecl SetStringValue( PMEIdentifier aName, PMEIdentifier aValue);

      Called when a string value changed in the user interface, or during patch initiaiztion. This function needs not be
      implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetStringValue( const aName: PMEIdentifier): PMEIdentifier; cdecl;
    AnsiChar * __cdecl GetStringValue( PMEIdentifier aName);

      Called when the user interface wants to know about a string value. Is this ever? hmm ... anyway .. this function
      needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure Pulse( anIndex: Integer); cdecl;
    void __cdecl Pulse( int anIndex);

      Called when a pulse event occurs in the user interface. A pulse event is generated for a single valued control
      when it gets clicked but it did not change value, for instance a reset button, This function needs not be
      implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure SetInternal( const aName: PMEIdentifier; const aValue: TMEValue); cdecl;
    void __cdecl SetInternal( PMEIdentifier aName, TMEValue aValue);

      Called when an internal value changed in the user interface, this can be a knob, a fader, a selector etc. This
      function needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetInternal( const aName: PMEIdentifier): TMEValue; cdecl;
    TMEValue __cdecl GetInternal( PMEIdentifier aName);

      Called when th user interface wants to know a parameter value for a knob, a fader, a selector etc. Is this ever?
      hmm ... anyway ... this function needs not be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure DoTick( const anInputs: PExtData; const anOutputs: PExtData); cdecl;
    void __cdecl DoTick( TExtData * anInputs, TExtData * anOutputs);

      Called from the audio thread at the system rate when the synth engine wants the module to compute new output
      values. This will only be called when IsFast is true. IsFast should not be set for sped up modules. Either fast
      tick or slow tick or both must be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure DoSlowTick( const anInputs: PExtData; const anOutputs: PExtData); cdecl;
    void __cdecl DoTick( TExtData * anInputs, TExtData * anOutputs);

      Called from the audio thread at the audio rate or the control rate when the synth engine wants the module to
      compute new output values. This will only be called when IsSlow is true. When the module is sped up it will be
      called at the audio rate, otherwise at the control rate. IsFast should not be set for sped up modules. Either fast
      tick or slow tick or both must be implemented.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function GetInputName( const anIdex: Integer): PMEIdentifier; cdecl;
    AnsiChar * __cdecl GetInputName( int anIndex);

      Called by the system to make a connection between an input index (as used in the tick methods) and the name of
      that input. This function must be implemented when the module has any inputs.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetOutputName( const anIdex: Integer): PMEIdentifier; cdecl;
    AnsiChar * __cdecl GetOutputName( int anIndex);

      Called by the system to make a connection between an output index (as used in the tick methods) and the name of
      that output. Thos function must be implemented when the module has any outputs.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetSignalCount: Integer; cdecl;
    int __cdecl GetSignalCount( void);

      Called by the system to get the number of Signals in the user interface that the external code wishes to update.
      Signals are TMEValue values that get printed out as text in the user interface.Kb This function needs not be
      implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetLightsCount: Integer; cdecl;
    int __cdecl GetLightsCount( void);

      Called by the system to get the number of Lights in the user interface that the external code wishes to update.
      Lights are binary items that can be either on or off typically LED indicators on modules. This function needs not
      be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetDataCount: Integer; cdecl;
    int __cdecl GetDataCount( void);

      Called by the system to get the number of Data elements (for instance for a graph) in the user interface that the
      external code wishes to update. This function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetXYDataCount: Integer; cdecl;
    int __cdecl GetXYDataCount( void);

      Called by the system to get the number of X Y Data elements (currently only used for the XYScope module) in the
      user interface that the external code wishes to update. This function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetInfoCount: Integer; cdecl;
    int __cdecl GetInfoCount( void);

      Called by the system to get the number of Info elements in the user interface that the external code wishes to
      update. Info elementsa have a string value. This function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    function  GetCursorCount: Integer; cdecl;
    int __cdecl GetCursorCount( void);

      Called by the system to get the number of Cursor elements in the user interface that the external code wishes to
      update. A cursoe element consists of an X, Y pair. Currently this is used for the DataGraph like modules. This
      function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherSignals( const anIndex: Integer; out aData: PExtData); cdecl;
    void __cdecl GatherSignals( int anIndex, TExtData * aData);

      Called by the system GetSignalCount times to actually obtain the values. The external code should fill up a
      TExtData structure for passing the data. After the call the external code may free that structure. This function
      needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherLights( const anIndex: Integer; out aData: PExtData); cdecl;
    void __cdecl GatherLights( int anIndex, TExtData * aData);

      Called by the system GetLightsCount times to actually obtain the values. The external code should fill up a
      TExtData structure for passing the data. After the call the external code may free that structure. This function
      needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherData( const anIndex: Integer; out aData: PExtData); cdecl;
    void __cdecl GatherData( int anIndex, TExtData * aData);

      Called by the system GetDataCount times to actually obtain the values.The external code should fill up a
      TExtData structure for passing the data. After the call the external code may free that structure. This function
      needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherXYData( const anIndex: Integer; out aData: PExtPairData); cdecl;
    void __cdecl GatherXYData( int anIndex, TExtPairData * aData);

      Called by the system GetXYDataCount times to actually obtain the values. The external code should fill up a
      TExtPairData structure for passing the data. After the call the external code may free that structure. This
      function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherInfo( const anIndex: Integer; out aData: PExtStringData); cdecl;
    void __cdecl GatherInfo( int anIndex, TExtStringData * aData);

      Called by the system GetInfoCount times to actually obtain the values. The external code should fill up a
      TExtStringData structure for passing the data. After the call the external code may free that structure.  This
      function needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

    procedure GatherCursorData( const anIndex: Integer; out aData: PExtData); cdecl;
    void __cdecl GatherCursorData( int anIndex, TExtData * aData);

      Called by the system GetCursorCount times to actually obtain the values. The external code should fill up a
      TExtData structure for passing the data. After the call the external code may free that structure.  This function
      needs not be implemeted.


    . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .

}


interface

uses

  WinApi.Windows,

  System.SysUtils,

  KnobsUtils, KnobsConversions, ComplexMath, module_defs;


type

  PMEIdentifier = PChar;
  TMEValue      = Double;


  TMEValuePair = record
    X: TMEValue;
    Y: TMEValue;
  end;


  TExtData = record
    Name  : PMEIdentifier;
    Count : Integer;
    Data  : array[ 0 .. 0 { + Count - 1}] of TMEValue;
  end;
  PExtData = ^ TExtData;


  TExtPairData = record
    Name  : PMEIdentifier;
    Count : Integer;
    Data  : array[ 0 .. 0 { + Count - 1}] of TMEValuePair;
  end;
  PExtPairData = ^ TExtPairData;


  TExtStringData = record
    Name  : PMEIdentifier;
    Data  : PMEIdentifier;
  end;
  PExtStringData = ^ TExtStringData;


  TExtGetCaps           = procedure( var IsInput, IsOutput, IsSlow, IsFast, IsMidi, IsOSC: Integer);              cdecl;
  TExtCreate            = procedure( const aName: PMEIdentifier);                                                 cdecl;
  TExtProc              = procedure;                                                                              cdecl;
  TExtSampleRateChanged = procedure( aSystemRate, aControlRate: TMEValue);                                        cdecl;
  TExtTuningChanged     = procedure( const aReferenceA, aNotesPerOctave, aMiddleNote: TMEValue);                  cdecl;
  TExtSetStringValue    = procedure( const aName: PMEIdentifier; const aValue: PMEIdentifier);                    cdecl;
  TExtGetStringValue    = function ( const aName: PMEIdentifier): PMEIdentifier;                                  cdecl;
  TExtSetValue          = procedure( const aName: PMEIdentifier; const aValue: TMEValue);                         cdecl;
  TExtGetValue          = function ( const aName: PMEIdentifier): TMEValue;                                       cdecl;
  TExtPulse             = procedure( anIndex: Integer);                                                           cdecl;
  TExtTick              = procedure( const anInputs: PExtData; const anOutputs: PExtData);                        cdecl;
  TExtGetName           = function ( const anIdex: Integer): PMEIdentifier;                                       cdecl;
  TExtGetSignalCount    = function: Integer;                                                                      cdecl;
  TExtGetSignalData     = procedure( const anIndex: Integer; out aData: PExtData);                                cdecl;
  TExtGetData           = procedure( const anIndex: Integer; out aData: PExtData);                                cdecl;
  TExtGetPairData       = procedure( const anIndex: Integer; out aData: PExtPairData);                            cdecl;
  TExtGetStringData     = procedure( const anIndex: Integer; out aData: PExtStringData);                          cdecl;


  TModExternal = class( TMod)                     // Base class for external modules, in a DLL for instance
  strict protected
    FExternalName : string;                       // Could be a filename for instance, but not actually used here
    FSignalCount  : Integer;                      // The number of signal elements as reported by the external code
    FLightsCount  : Integer;                      // The number of light  elements as reported by the external code
    FDataCount    : Integer;                      // The number of data   elements as reported by the external code
    FXYDataCount  : Integer;                      // The number of xydata elements as reported by the external code
    FInfoCount    : Integer;                      // The number of info   elements as reported by the external code
    FCursorCount  : Integer;                      // The number of cursor elements as reported by the external code
  strict protected
    // The functions the external code (sh|c)ould implement
    FOnGetCaps           : TExtGetCaps;           // must be externally implemented
    FOnCreate            : TExtCreate;            // may  be externally implemented
    FOnDestroy           : TExtProc;              // may  be externally implemented
    FOnReset             : TExtProc;              // may  be externally implemented
    FOnSetDefaults       : TExtProc;              // may  be externally implemented
    FOnSampleRateChanged : TExtSampleRateChanged; // may  be externally implemented
    FOnTuningChanged     : TExtTuningChanged;     // may  be externally implemented
    FOnSetStringValue    : TExtSetStringValue;    // may  be externally implemented
    FOnGetStringValue    : TExtGetStringValue;    // may  be externally implemented
    FOnPulse             : TExtPulse;             // may  be externally implemented
    FOnSetInternal       : TExtSetValue;          // may  be externally implemented
    FOnGetInternal       : TExtGetValue;          // may  be externally implemented
    FOnDoTick            : TExtTick;              // either slow or fast tick or both must be externally implemented
    FOnDoSlowTick        : TExtTick;              // either slow or fast tick or both must be externally implemented
    FOnGetInputName      : TExtGetName;           // must be externally implemented
    FOnGetOutputName     : TExtGetName;           // must be externally implemented
    FOnGetSignalCount    : TExtGetSignalCount;    // may  be externally implemented
    FOnGetLightsCount    : TExtGetSignalCount;    // may  be externally implemented
    FOnGetDataCount      : TExtGetSignalCount;    // may  be externally implemented
    FOnGetXYDataCount    : TExtGetSignalCount;    // may  be externally implemented
    FOnGetInfoCount      : TExtGetSignalCount;    // may  be externally implemented
    FOnGetCursorCount    : TExtGetSignalCount;    // may  be externally implemented
    FOnGatherSignals     : TExtGetSignalData;     // may  be externally implemented
    FOnGatherLights      : TExtGetSignalData;     // may  be externally implemented
    FOnGatherData        : TExtGetData;           // may  be externally implemented
    FOnGatherXYData      : TExtGetPairData;       // may  be externally implemented
    FOnGatherStringData  : TExtGetStringData;     // may  be externally implemented
    FOnGatherCursorData  : TExtGetData;           // may  be externally implemented
    // End of external functions list
  strict protected
    procedure   BindFunctions;                                                                        virtual; abstract;
    procedure   UnbindFunctions;                                                                      virtual; abstract;
  strict protected
    procedure   SetExternalName( const aValue: string);                                                         virtual;
    function    GetInputName ( anIndex: Integer): string;                                                       virtual;
    function    GetOutputName( anIndex: Integer): string;                                                       virtual;
    procedure   DoReset;
  public
    procedure   Initialize;                                                                                    override;
    destructor  Destroy;                                                                                       override;
    procedure   SetDefaults;                                                                                   override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   TuningChanged;                                                                                 override;
    procedure   SetStringValue( const aName, aValue: string);                                                  override;
    function    GetStringValue( const aName: string): string;                                                  override;
    procedure   Pulse         ( anIndex: Integer);                                                             override;
    procedure   SetInternal   ( const aName: string; aValue: TSignal);                                         override;
    function    GetInternal   ( const aName: string): TSignal;                                                 override;
    procedure   DoTick;                                                                                        override;
    procedure   DoSlowTick;                                                                                    override;
    procedure   GatherSignals   ( const aPrefix: string; const aCallback: TSignalHandler    );                 override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                 override;
    procedure   GatherData      ( const aPrefix: string; const aCallback: TDataHandler      );                 override;
    procedure   GatherXYData    ( const aPrefix: string; const aCallback: TXYDataHandler    );                 override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
  public
    property    ExternalName : string  read FExternalName write SetExternalName;
    property    SignalCount  : Integer read FSignalCount;
    property    LightsCount  : Integer read FLightsCount;
    property    DataCount    : Integer read FDataCount  ;
    property    XYDataCount  : Integer read FXYDataCount;
    property    InfoCount    : Integer read FInfoCount  ;
    property    CursorCount  : Integer read FCursorCount;
  end;


  TModDll = class( TModExternal)
  strict private
    FDllHandle : Cardinal;
  strict protected
    procedure   BindFunctions;                                                                                 override;
    procedure   UnbindFunctions;                                                                               override;
  end;


implementation

type

  TLongData     = array[ 0 .. MaxInt div SizeOf( TMEValue    ) - 1] of TMEValue;
  PLongData     = ^ TLongData;
  TLongPairData = array[ 0 .. MaxInt div SizeOf( TMEValuePair) - 1] of TMEValuePair;
  PLongPairData = ^ TLongPairData;


{ ========
  TModExternal = class( TMod) // Base class for external modules, in a DLL for instance
  strict private
    FExternalName : string; // Could be a filename for instance
    FSignalCount  : Integer;
    FLightsCount  : Integer;
    FDataCount    : Integer;
    FXYDataCount  : Integer;
    FInfoCount    : Integer;
    FCursorCount  : Integer;
  public
    property    ExternalName : string read FExternalName write SetExternalName;
    property    SignalCount  : Integer read FSignalCount;
    property    LightsCount  : Integer read FLightsCount;
    property    DataCount    : Integer read FDataCount  ;
    property    XYDataCount  : Integer read FXYDataCount;
    property    InfoCount    : Integer read FInfoCount  ;
    property    CursorCount  : Integer read FCursorCount;
  strict protected
}

    procedure   TModExternal.SetExternalName( const aValue: string); // virtual;
    begin
      if aValue <> FExternalName
      then FExternalName := aValue;
    end;


    function    TModExternal.GetInputName( anIndex: Integer): string; // virtual;
    begin
      if Assigned( @ FOnGetInputName)
      then begin
        Result := string( FOnGetInputName( anIndex));

        if Result = ''
        then Result := NO_INPUT;
      end
      else Result := NO_INPUT;
    end;


    function    TModExternal.GetOutputName( anIndex: Integer): string; // virtual;
    begin
      if Assigned( @ FOnGetOutputName)
      then begin
        Result := string( FOnGetOutputName( anIndex));

        if Result = ''
        then Result := NO_OUTPUT;
      end
      else Result := NO_OUTPUT;
    end;


    procedure   TModExternal.DoReset;
    begin
      if Assigned( @ FOnReset)
      then FOnReset;
    end;


//  public

    procedure   TModExternal.Initialize; // override;
    var
      IIsInput  : Integer;
      IIsOutput : Integer;
      IIsSlow   : Integer;
      IIsFast   : Integer;
      IIsMidi   : Integer;
      IIsOSC    : Integer;
      S         : string;
      p         : Integer;
    begin
      Locked := True;
      BindFunctions; // abstract :: do not make instances of TModExternal, use a derrived type

      if Assigned( @ FOnGetCaps)
      then begin
        FOnGetCaps( IIsInput, IIsOutput, IIsSlow, IIsFast, IIsMidi, IIsOSC);
        FIsInput  := IIsInput  <> 0;
        FIsOutput := IIsOutput <> 0;
        FIsFast   := IIsFast   <> 0;
        FIsSlow   := IIsSlow   <> 0;
        FIsMidi   := IIsMidi   <> 0;
        FIsOSC    := IIsOSC    <> 0;
      end
      else begin
        LogFmt( 'Module "%s" implements no functionality', [ ExternalName]);
        FIsInput  := False;
        FIsOutput := False;
        FIsFast   := False;
        FIsSlow   := False;
        FIsMidi   := False;
        FIsOSC    := False;
      end;

      if Assigned( @ FOnCreate)
      then FOnCreate( PMEIdentifier( Name));

      if Assigned( @ FOnGetSignalCount)
      then FSignalCount := FOnGetSignalCount
      else FSignalCount := 0;

      if Assigned( @ FOnGetLightsCount)
      then FLightsCount := FOnGetLightsCount
      else FLightsCount := 0;

      if Assigned( @ FOnGetDataCount)
      then FDataCount := FOnGetDataCount
      else FDataCount := 0;

      if Assigned( @ FOnGetXYDataCount)
      then FXYDataCount := FOnGetXYDataCount
      else FXYDataCount := 0;

      if Assigned( @ FOnGetInfoCount)
      then FInfoCount := FOnGetInfoCount
      else FInfoCount := 0;

      if Assigned( @ FOnGetCursorCount)
      then FCursorCount := FOnGetCursorCount
      else FCursorCount := 0;

      // Collect input names

      p := 0;
      S := GetInputName( p);

      while not SameText( S, NO_INPUT)
      do begin
        AddInput( p, S);
        Inc( p);
        S := GetInputName( p);
      end;

      // Collect output names

      p := 0;
      S := GetOutputName( p);

      while not SameText( S, NO_OUTPUT)
      do begin
        AddOutput( p, S);
        Inc( p);
        S := GetOutputName( p);
      end;

      FixIO;
      SetDefaults;
      Locked := False;
    end;


    destructor  TModExternal.Destroy; // override;
    begin
      Locked := True;

      if Assigned( @ FOnDestroy)
      then FOnDestroy;

      UnbindFunctions; // virtual; abstract; do not make instances of TModExternal, use a derrived type
      inherited;
    end;


    procedure   TModExternal.SetDefaults; // override;
    begin
      if Assigned( @ FOnSetDefaults)
      then FOnSetDefaults;
    end;


    procedure   TModExternal.SampleRateChanged; // override;
    begin
      if Assigned( @ FOnSampleRateChanged)
      then begin
        Locked := True;

        try
          FOnSampleRateChanged( System_Rate, Control_Rate);
        finally
          Locked := False;
        end;
      end;
    end;


    procedure   TModExternal.TuningChanged; // override;
    begin
      if Assigned( @ FOnTuningChanged)
      then begin
        Locked := True;

        try
          FOnTuningChanged( ReferenceA, NotesPerOctave, MiddleNote);
        finally
          Locked := False;
        end;
      end;
    end;


    procedure   TModExternal.SetStringValue( const aName, aValue: string); // override;
    begin
      if Assigned( @ FOnSetStringValue)
      then FOnSetStringValue( PMEIdentifier( aName), PMEIdentifier( aValue));
    end;


    function    TModExternal.GetStringValue( const aName: string): string; // override;
    begin
      if Assigned( @ FOnGetStringValue)
      then Result := string( FOnGetStringValue( PMEIdentifier( aName)))
      else Result := '';
    end;


    procedure   TModExternal.Pulse( anIndex: Integer); // override;
    begin
      if Assigned( FOnPulse)
      then FOnPulse( anIndex);
    end;


    procedure   TModExternal.SetInternal( const aName: string; aValue: TSignal); // override;
    begin
      if Assigned( @ FOnSetInternal)
      then FOnSetInternal( PMEIdentifier( aName), aValue);
    end;


    function    TModExternal.GetInternal( const aName: string): TSignal; // override;
    begin
      if Assigned( @ FOnGetInternal)
      then Result := FOnGetInternal( PMEIdentifier( aName))
      else Result := UNDEFINED;
    end;


    procedure   TModExternal.DoTick; // override;
    begin
      if ResetFlag
      then begin
        DoReset;
        ResetFlag := False;
      end;

      if Assigned( @ FOnDoTick)
      then FOnDoTick( @ FInputs[ 0], @ FOutputs[ 0]);
    end;


    procedure   TModExternal.DoSlowTick; // override;
    begin
      if Assigned( @ FOnDoSlowTick)
      then FOnDoSlowTick( @ FInputs[ 0], @ FOutputs[ 0]);
    end;


    procedure   TModExternal.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    var
      i     : Integer;
      aData : PExtData;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( SignalCount > 0) and Assigned( @ FOnGatherSignals)
      then begin
        for i := 0 to SignalCount - 1
        do begin
          aData := nil;
          FOnGatherSignals( i, aData);

          if Assigned( aData) and Assigned( aData^.Name) and ( aData^.Count > 0)
          then aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), aData^.Data[ 0]));
        end;
      end;
    end;


    procedure   TModExternal.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      i     : Integer;
      aData : PExtData;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( LightsCount > 0) and Assigned( @ FOnGatherLights)
      then begin
        for i := 0 to LightsCount - 1
        do begin
          aData := nil;
          FOnGatherLights( i, aData);

          if Assigned( aData) and Assigned( aData^.Name) and ( aData.Count > 0)
          then aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), aData^.Data[ 0]));
        end;
      end;
    end;


    procedure   TModExternal.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    var
      i       : Integer;
      aData   : PExtData;
      aValues : TSignalArray;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( DataCount > 0) and Assigned( @ FOnGatherData)
      then begin
        for i := 0 to DataCount - 1
        do begin
          aData := nil;
          FOnGatherData( 1, aData);

          if Assigned( aData) and Assigned( aData^.Name) and ( aData^.Count > 0)
          then begin
            SetLength( aValues, aData^.Count);
            Move( aData^.Data, aValues[ 0], Length( aValues) * SizeOf( TMEValue));
            aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), aValues));
          end;
        end;
      end;
    end;


    procedure   TModExternal.GatherXYData( const aPrefix: string; const aCallback: TXYDataHandler); // override;
    var
      i       : Integer;
      j       : Integer;
      aData   : PExtPairData;
      aValues : TSignalPairFifo;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( XYDataCount > 0) and Assigned( @ FOnGatherXYData)
      then begin
        for i := 0 to XYDataCount - 1
        do begin
          aData := nil;
          FOnGatherXYData( i, aData);

          if Assigned( aData) and Assigned( aData^.Name) and ( aData^.Count > 0)
          then begin
            aValues := TSignalPairFifo.Create( aData.Count);

            try
              for j := 0 to aValues.Size - 1
              do aValues.Append( SignalPair( PLongPairData( @ aData^.Data)^[ j].X, PLongPairData( @ aData^.Data)^[ j].Y));

              aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), aValues));
            finally
              aValues.DisposeOf;
            end;
          end;
        end;
      end;
    end;


    procedure   TModExternal.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      i     : Integer;
      aData : PExtStringData;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( InfoCount > 0) and Assigned( @ FOnGatherStringData)
      then begin
        for i := 0 to InfoCount - 1
        do begin
          FOnGatherStringData( i, aData);

          if( Assigned( aData) and Assigned( aData^.Name) and Assigned( aData^.Data))
          then aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), string( aData^.Data)));
        end;
      end;
    end;


    procedure   TModExternal.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    var
      i      : Integer;
      aData  : PExtData;
      aValue : TSignalPair;
    begin
      Assert( Assigned( aCallback), 'design error, dont call this when callback not assigned');

      if ( CursorCount > 0) and Assigned( @ FOnGatherCursorData)
      then begin
        for i := 0 to CursorCount - 1
        do begin
          aData := nil;
          FOnGatherCursorData( i, aData);

          if Assigned( aData) and Assigned( aData^.Name) and ( aData^.Count > 1)
          then begin
            aValue.X := PLongData( @ ( aData^.Data))[ 0];
            aValue.Y := PLongData( @ ( aData^.Data))[ 1];
            aCallback( Self, MakeInfo( aPrefix, Name, string( aData^.Name), aValue));
          end;
        end;
      end;
    end;


{ ========
  TModDll = class( TModExternal)
  protected
}

    procedure   TModDll.BindFunctions; // override;
    begin
      if FileExists( ExternalName)
      then begin
        FDllHandle := LoadLibrary( PWideChar( ExternalName));

        if FDllHandle <> 0
        then begin
          @ FOnGetCaps           := GetProcAddress( FDllHandle, 'GetCaps'          );
          @ FOnCreate            := GetProcAddress( FDllHandle, 'Create'           );
          @ FOnDestroy           := GetProcAddress( FDllHandle, 'Destroy'          );
          @ FOnReset             := GetProcAddress( FDllHandle, 'Reset'            );
          @ FOnSetDefaults       := GetProcAddress( FDllHandle, 'SetDefaults'      );
          @ FOnSampleRateChanged := GetProcAddress( FDllHandle, 'SampleRateChanged');
          @ FOnTuningChanged     := GetProcAddress( FDllHandle, 'TuningChanged'    );
          @ FOnSetStringValue    := GetProcAddress( FDllHandle, 'SetStringValue'   );
          @ FOnGetStringValue    := GetProcAddress( FDllHandle, 'GetStringValue'   );
          @ FOnPulse             := GetProcAddress( FDllHandle, 'Pulse'            );
          @ FOnSetInternal       := GetProcAddress( FDllHandle, 'SetInternal'      );
          @ FOnGetInternal       := GetProcAddress( FDllHandle, 'GetInternal'      );
          @ FOnDoTick            := GetProcAddress( FDllHandle, 'DoTick'           );
          @ FOnDoSlowTick        := GetProcAddress( FDllHandle, 'DoSlowTick'       );
          @ FOnGetInputName      := GetProcAddress( FDllHandle, 'GetInputName'     );
          @ FOnGetOutputName     := GetProcAddress( FDllHandle, 'GetOutputName'    );
          @ FOnGetSignalCount    := GetProcAddress( FDllHandle, 'GetSignalCount'   );
          @ FOnGetLightsCount    := GetProcAddress( FDllHandle, 'GetLightsCount'   );
          @ FOnGetDataCount      := GetProcAddress( FDllHandle, 'GetDataCount'     );
          @ FOnGetXYDataCount    := GetProcAddress( FDllHandle, 'GetXYDataCount'   );
          @ FOnGetInfoCount      := GetProcAddress( FDllHandle, 'GetInfoCount'     );
          @ FOnGetCursorCount    := GetProcAddress( FDllHandle, 'GetCursoeCount'   );
          @ FOnGatherSignals     := GetProcAddress( FDllHandle, 'GatherSignals'    );
          @ FOnGatherLights      := GetProcAddress( FDllHandle, 'GatherLights'     );
          @ FOnGatherData        := GetProcAddress( FDllHandle, 'GatherData'       );
          @ FOnGatherXYData      := GetProcAddress( FDllHandle, 'GatherXYData'     );
          @ FOnGatherStringData  := GetProcAddress( FDllHandle, 'GatherStringData' );
          @ FOnGatherCursorData  := GetProcAddress( FDllHandle, 'GatherCursorData' );
        end;
      end;
    end;


    procedure   TModDll.UnbindFunctions; // override;
    begin
      FOnGetCaps           := nil;
      FOnCreate            := nil;
      FOnDestroy           := nil;
      FOnReset             := nil;
      FOnSetDefaults       := nil;
      FOnSampleRateChanged := nil;
      FOnTuningChanged     := nil;
      FOnSetStringValue    := nil;
      FOnGetStringValue    := nil;
      FOnSetInternal       := nil;
      FOnGetInternal       := nil;
      FOnDoTick            := nil;
      FOnDoSlowTick        := nil;
      FOnGetInputName      := nil;
      FOnGetOutputName     := nil;
      FOnGetSignalCount    := nil;
      FOnGetLightsCount    := nil;
      FOnGetDataCount      := nil;
      FOnGetXYDataCount    := nil;
      FOnGetInfoCount      := nil;
      FOnGetCursorCount    := nil;
      FOnGatherSignals     := nil;
      FOnGatherLights      := nil;
      FOnGatherData        := nil;
      FOnGatherXYData      := nil;
      FOnGatherStringData  := nil;
      FOnGatherCursorData  := nil;

      if FDllHandle <> 0
      then FreeLibrary( FDllHandle);
    end;


end.

