unit module_defs;

{

   COPYRIGHT 2013 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

{.$DEFINE PATCH_PROFILER :: is defined from build configuration 'ProfiledRelease'}

interface

uses

  System.SysUtils, System.Math, System.Classes,

  KnobsUtils, KnobsConversions, ComplexMath, midi_defs, OSCif;


const

  NO_INPUT               = 'noinput';
  NO_OUTPUT              = 'nooutput';
  NO_STRING              = 'nostring';
  NO_MODULE              = 'nomodule';
  NO_ID                  = -1;                  // Index for just about anything that was not found
  MAX_SIGNALS            =  512;
  MAX_INPUTS             = MAX_SIGNALS;
  MAX_OUTPUTS            = MAX_SIGNALS;
  DO_DEZIPPER            = True;
  NO_DEZIPPER            = False;
  HAS_PATCH_PROFILER     = {$IFDEF PATCH_PROFILER} True {$ELSE} False {$ENDIF} ;
  IdentifierStarters     = [ 'A' .. 'Z', 'a' .. 'z', '%', '#', '@', '_'];

  inputtype_normal       = 0;
  inputtype_inverted     = 1;
  inputtype_positive     = 2;
  inputtype_pos_inv      = 3;
  inputtype_negative     = 4;
  inputtype_neg_inv      = 5;

  outputtype_normal      = 0;
  outputtype_inverted    = 1;
  outputtype_positive    = 2;
  outputtype_pos_inv     = 3;
  outputtype_negative    = 4;
  outputtype_neg_inv     = 5;

  envoutputtype_normal   = 0;
  envoutputtype_inverted = 1;
  envoutputtype_negative = 2;
  envoutputtype_neg_inv  = 3;


type

  EModule           = class( Exception);
  EModuleNotFound   = class( EModule  );
  EModuleDuplicate  = class( EModule  );
  EInputNotFound    = class( EModule  );
  EOutputNotFound   = class( EModule  );
  EStringNotFound   = class( EModule  );
  EUnitsUndefined   = class( EModule  );
  ENoPinLookup      = class( EModule  );

  TStringArray      = array of string;
  TMod              = class;
  TModInput         = class;
  TModOutput        = class;
  TSynthPatch       = class;
  TModuleArray      = array of TMod;
  TModuleClass      = class of TMod;


  TModuleIdTypeMap = record
    ModuleId    : Integer;
    ModuleClass : TModuleClass;
  end;
  TModuleIdTypeMaps = array of TModuleIdTypeMap;


  TConnectorIDNameMap = record
    ID   : Integer;
    Name : string;
  end;
  TConnectorIDNameMaps = array of TConnectorIDNameMap;


  TPinMap = record
    ModuleIndex : Integer;
    Ins         : TByteSet;
    Outs        : TByteSet;
  end;
  TPinMaps = array of TPinMap;


  TLightsInfo = record
  public
    Prefix : string;
    Name   : string;
    Signal : string;
    Data   : TSignal;
  end;


  TDataInfo = record
  public
    Prefix : string;
    Name   : string;
    Signal : string;
    Data   : TSignalArray;
  end;


  TXYDataInfo = record
  public
    Prefix : string;
    Name   : string;
    Signal : string;
    Data   : TSignalPairFifo;
  end;


  TStringInfo = record
  public
    Prefix : string;
    Name   : string;
    Signal : string;
    Data   : string;
  end;


  TCursorInfo = record
  public
    Prefix : string;
    Name   : string;
    Signal : string;
    Data   : TSignalPair;
  end;


  TAutoFlag = (
    afRandomize   ,
    afMutate      ,
    afMate        ,
    afMorph       ,
    afPatchReset  ,
    afPatchCompile
  );

  TAutoFlags = set of TAutoFlag;


  TSignalHandler     = procedure( const aSender: TMod   ; const anInfo: TLightsInfo) of object;
  TLightsHandler     = procedure( const aSender: TMod   ; const anInfo: TLightsInfo) of object;
  TDataHandler       = procedure( const aSender: TMod   ; const anInfo: TDataInfo  ) of object;
  TXYDataHandler     = procedure( const aSender: TMod   ; const anInfo: TXYDataInfo) of object;
  TStringDataHandler = procedure( const aSender: TMod   ; const anInfo: TStringInfo) of object;
  TCursorDataHandler = procedure( const aSender: TMod   ; const anInfo: TCursorInfo) of object;
  TOnOSCString       = procedure( const aSender: TObject; const aMsg: TBytes)        of object;
  TOnMidiBytes       = procedure( const aSender: TObject; const aMsg: TBytes)        of object;


  TConnection = class
  public
    Peer      : TSynthPatch;
    Index     : Integer;
    SrcModule : Integer;
    SrcPin    : Integer;
    DstModule : Integer;
    DstPin    : Integer;
  protected
    procedure   Connect( const aPeer: TSynthPatch; anIndex, aSrcModule, aSrcPin, aDstModule, aDstPin: Integer);
  public
    procedure   Dump( anIndent: Integer; var S: string);
    function    IsSameConnection( aConnection: TConnection): Boolean;
  end;
  TConnectionArray  = array of TConnection;


  TConnections = class
  private
    FPeer       : TSynthPatch;
    FConnections: TConnectionArray;
  private
    function    GetCount: Integer;
    function    GetConnection( anIndex: Integer): TConnection;
    procedure   SetConnection( anIndex: Integer; const aValue: TConnection);
  public
    constructor Create( aPeer: TSynthPatch);
    destructor  Destroy;                                                                                       override;
    procedure   Dump( anIndent: Integer; const aName: string; var S: string);
    procedure   Clear;
    procedure   AddConnection( const aValue: TConnection);                                                     overload;
    procedure   AddConnection( aSrcModule, aSrcPin, aDstModule, aDstPin: Integer);                             overload;
    procedure   RemoveConnection( anIndex: Integer);
    function    IsSameConnections( const aConnections: TConnections): Boolean;
  public
    property    Count: Integer                             read GetCount;
    property    Connection[ anIndex: Integer]: TConnection read GetConnection write SetConnection;
  end;


  TSignals = array[ 0 .. MAX_INPUTS - 1] of PSignal;
  PSignals = ^TSignals;


  TLinks = record
  private
    FCount  : Integer;
    FSource : PSignal;
    FLinks  : PSignals;
  private
    function    FindLink( aDestination: PSignal): Integer;
  public
    procedure   Initialize( aSource: PSignal);
    procedure   Uninitialize;
    procedure   AddLink( aDestination: PSignal);
    procedure   RemoveLink( aDestination: PSignal);
  end;
  TLinksArray = array[ 0 .. MAX_OUTPUTS - 1] of TLinks;
  PLinksArray = ^TLinksArray;


  TOutputLinks = class
  private
    FCount    : Integer;
    FOutLinks : PLinksArray;
  private
    function    FindSource( aSource: PSignal): Integer;
  public
    destructor  Destroy;                                                                                       override;
    procedure   ClearLinks;
    procedure   AddLink( aSource, aDestination: PSignal);
    procedure   RemoveLink( aSource, aDestination: PSignal);
    procedure   Compile( const aModule: TMod; const aPatch: TSynthPatch);
  end;


  TMod = class
  // Synth module prototype
  private
    FParent          : TSynthPatch;          // The patch this module is part off
    FLocked          : Boolean;              // When true module should not be executed
    FExecuting       : Boolean;              // True when the module is currently being executed
    FName            : string;               // A unique name for this module - unique within it's parent
    FResetFlag       : Integer;              // Set when a module reset was requested
  {$IFDEF PATCH_PROFILER}
  private
    FSamplesFast     : Int64;                // Sample counter, incremented each time a new Fast sample is calcultaed
    FSamplesSlow     : Int64;                // Sample counter, incremented each time a new Slow sample is calcultaed
    FClocksFast      : Int64;                // CPU Fast clocks for current cycle
    FClocksSlow      : Int64;                // CPU Slow clocks for current cycle
    FTotalClocksFast : Int64;                // CPU Fast clocks accumulated over the module life time
    FTotalClocksSlow : Int64;                // CPU Slow clocks accumulated over the module life time
    FZips            : Int64;                // CPU clocks spent on dezippering for the current cycle
    FTotalZips       : Int64;                // CPU clocks acumulated FZips over the module life time
  {$ENDIF}
  protected
    FIsInput         : Boolean;              // True for a TModInput  module only
    FIsOutput        : Boolean;              // True for a TModOutput module only
    FIsFast          : Boolean;              // True when fast mode processing must be performed
    FIsSpedUp        : Boolean;              // True when module is normally slow but sped up due to fast signals being present on an input.
    FIsSlow          : Boolean;              // True when slow mode processing must be performed
    FIsMidi          : Boolean;              // True when MIDI reception must be enabled - for MIDI modules only
    FIsOSC           : Boolean;              // True when OSC  reception must be enabled - for OSC  modules only
    FInputMaps       : TConnectorIDNameMaps; // Maps input  pin names to IDs
    FOutputMaps      : TConnectorIDNameMaps; // Maps output pin names to IDs
    FInputs          : TSignalArray;         // The input signals, these include both patch signals and user controls, FInputs and FDezips have the same length
    FDezips          : TSignalArray;         // An extra set of inputs for user controls, these are copied to FInputs with some filtering when the signal happens to be be in the deziper map
    FOutputs         : TSignalArray;         // The ouput signals.
    FDezipperMap     : TByteSet;             // Determines  what signals should be dezippered, most, but not all, knobs should be dezippered
  [weak] FOutputLinks: TOutputLinks;         // For each output a set of signals that should be updated from it, compiled connections
                                             // This is a weak reference to make reference counting work properly.
  private
    function    AssertUniqueName( const aName: string): string;                                                 virtual;
  private
    procedure   SetLocked   ( aValue: Boolean);
    procedure   SetName     ( const aValue: string);
    function    GetResetFlag: Boolean;
    procedure   SetResetFlag( aValue: Boolean);                                                                 virtual;
    function    GetOutput( anIndex: Integer): TSignal;
    procedure   SetCommonValue        ( anIndex : Integer; aValue: TSignal; Dezipper: Boolean);                 virtual;
    procedure   SetInternalValue      ( const aName: string; aValue: TSignal);                                  virtual;
    procedure   SetInternalStringValue( const aType, aName, aValue: string);
    procedure   SetAutomationValue    ( const aName: string; aValue: TSignal);                                  virtual;
  protected
    procedure   SetStringValue        ( const aName, aValue: string);                                           virtual;
    function    GetStringValue        ( const aName: string): string;                                           virtual;
    procedure   PerformMorph          ( aChannel: Integer; aValue: TSignal);
  protected
    function    GetInputCount         : Integer;
    function    GetOutputCount        : Integer;
    function    FindInputID           ( anID: Integer): Integer;
    function    FindOutputID          ( anID: Integer): Integer;
    function    FindInputName         ( anID: Integer): string;
    function    FindOutputName        ( anID: Integer): string;
    procedure   AddInput              ( anID: Integer; const aName: string);
    procedure   AddOutput             ( anID: Integer; const aName: string);
    procedure   FixIO;
    function    HandleParam          ( const aControl: string; aValue: TSignal): Boolean;                       virtual;
    function    HandleAutomationParam( const aControl: string; aValue: TSignal): Boolean;                       virtual;
    procedure   Log( const aMsg: string);                                                                       virtual;
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    {$IFDEF PATCH_PROFILER}
  protected
    procedure   PreTickFast;                                                                                     inline;
    procedure   PostTickFast;                                                                                    inline;
    procedure   PreTickSlow;                                                                                     inline;
    procedure   PostTickSlow;                                                                                    inline;
    procedure   PreZipper;                                                                                       inline;
    procedure   PostZipper;                                                                                      inline;
    procedure   ClearProfile;
    {$ENDIF}
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                             virtual;
    procedure   CreateIO;                                                                                       virtual;
    procedure   Initialize;                                                                                     virtual;
    destructor  Destroy;                                                                                       override;
    procedure   CompileLinks;                                                                                   virtual;
    procedure   PropagateSignals;
    procedure   SetDefaults;                                                                                    virtual;
  public
    procedure   DumpKeyValue    ( anIndent: Integer; const aName: string; aValue: TSignal; var S: string);
    procedure   DumpInputValues ( anIndent: Integer; var S: string);
    procedure   DumpOutputValues( anIndent: Integer; var S: string);
    procedure   DumpDetails     ( anIndent: Integer; var S: string);                                            virtual;
    procedure   Dump            ( anIndent: Integer; var S: string);
  public
    procedure   SetInternal     ( const aName: string; aValue: TSignal);                                        virtual;
    function    GetInternal     ( const aName: string): TSignal;                                                virtual;
    procedure   Reset;                                                                                          virtual;
    procedure   ResetSampleCounts;                                                                              virtual;
    procedure   Clear;                                                                                          virtual;
    procedure   Panic;                                                                                          virtual;
    procedure   Pulse( anInput: Integer);                                                                       virtual;
    function    FindInput ( const aPinName: string): Integer;
    function    FindOutput( const aPinName: string): Integer;
    procedure   MakeFast;
    procedure   Tick;
    procedure   SlowTick;
    procedure   DeZipper;
    procedure   DoTick;                                                                                         virtual;
    procedure   DoSlowTick;                                                                                     virtual;
    procedure   FixDeZippers;                                                                                   virtual;
    procedure   GatherSignals   ( const aPrefix: string; const aCallback: TSignalHandler    );                  virtual;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler    );                  virtual;
    procedure   GatherData      ( const aPrefix: string; const aCallback: TDataHandler      );                  virtual;
    procedure   GatherXYData    ( const aPrefix: string; const aCallback: TXYDataHandler    );                  virtual;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                  virtual;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                  virtual;
    function    AcceptParam          ( const aPath: string; aValue: TSignal): Boolean;
    function    AcceptAutomationParam( const aPath: string; aValue: TSignal): Boolean;
    function    GetProfileInfoHtml: string;
    function    IsSameModule( const aModule: TMod): Boolean;
    procedure   AcceptMidi( const aMsg: TMidiMessage);                                                          virtual;
    procedure   AcceptShortMidi( const aMsg: TShortMidiMessage);                                                virtual;
    procedure   AcceptPn( aCh: Byte; aController, aValue: Integer; IsRegistered: Boolean);                      virtual;
    procedure   SendShortMidi( const aMsg: TShortMidiMessage);                                                  virtual;
    function    OSCMatched( const aMsg: TOSCMessage): Boolean;                                                  virtual;
    procedure   AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket);                                   virtual;
    procedure   SendOSCBytes( const aMsg: TBytes);                                                              virtual;
    procedure   SendMidiBytes( const aMsg: TBytes);                                                             virtual;
    procedure   SampleRateChanged;                                                                              virtual;
    procedure   TuningChanged;                                                                                  virtual;
    procedure   CollectDenormals( var aMap: TPinMap);
  public
    property    Parent                                           : TSynthPatch read FParent;
    property    Locked                                           : Boolean     read FLocked          write SetLocked;
    property    Name                                             : string      read FName            write SetName;
    property    ResetFlag                                        : Boolean     read GetResetFlag     write SetResetFlag;
    property    InputCount                                       : Integer     read GetInputCount;
    property    OutputCount                                      : Integer     read GetOutputCount;
    property    Output[ anIndex: Integer]                        : TSignal     read GetOutput;
    property    IsFast                                           : Boolean     read FIsFast;
    property    IsSlow                                           : Boolean     read FIsSlow;
    property    IsMidi                                           : Boolean     read FIsMidi;
    property    IsOSC                                            : Boolean     read FIsOSC;
    property    IsSpedUp                                         : Boolean     read FIsSpedUp;
  public
    property    InternalValue      [ const aName: string       ] : TSignal                           write SetInternalValue;
    property    AutomationValue    [ const aName: string       ] : TSignal                           write SetAutomationValue;
    property    InternalStringValue[ const aType, aName: string] : string                            write SetInternalStringValue;
    property    StringValue        [ const aName: string       ] : string      read GetStringValue   write SetStringValue;
  {$IFDEF PATCH_PROFILER}
  public
    property    SamplesFast                                      : Int64       read FSamplesFast;
    property    SamplesSlow                                      : Int64       read FSamplesSlow;
    property    TotalClocksFast                                  : Int64       read FTotalClocksFast;
    property    TotalClocksSlow                                  : Int64       read FTotalClocksSlow;
    property    TotalZips                                        : Int64       read FTotalZips;
  {$ENDIF}
  end;


  TModInput = class( TMod)
  // Standard input module
  public
    procedure   CreateIO;                                                                                      override;
  end;


  TModOutput = class( TMod)
  // Standard output module
  public
    procedure   CreateIO;                                                                                      override;
  end;


  TSynthPatch = class( TMod)
  // A complete synth patch, which is a module too ...
  private
    FVersion         : Integer;
    FMidiChannel     : Byte;
    FModules         : TModuleArray;
    FConnections     : TConnections;
    FDecimatorCount  : Integer;
    FInternalMidiMsg : TMidiMessage;
    FMorphs          : TSignalArray;
    FAutoFlags       : TAutoFlags;
    FLockSkips       : Cardinal;
    FOnSendShortMidi : TOnShortMidiMessage;
    FOnSendOscString : TOnOSCString;
    FOnSendMidiBytes : TOnMidiBytes;
  private
    function    AssertUniqueName( const aName: string): string;                                                override;
  private
    function    CheckModuleIndex( anIndex: Integer): Boolean;
    function    GetModuleCount: Integer;
    function    GetModule( anIndex: Integer): TMod;
    function    GetConnectionCount: Integer;
    procedure   SetResetFlag( aValue: Boolean);                                                                override;
    function    FixPrefix( const aPrefix: string): string;
    function    GetMorph( anIndex: Integer): TSignal;
    procedure   SetMorph( anIndex: Integer; aValue: TSignal);
    function    GetAutoFlag ( anIndex: TAutoFlag): Boolean;
    procedure   SetAutoFlag ( anIndex: TAutoFlag; aValue: Boolean);
  protected
    function    HandleParam          ( const aPath: string; aValue: TSignal): Boolean;                         override;
    function    HandleAutomationParam( const aPath: string; aValue: TSignal): Boolean;                         override;
  public
    constructor Create( aParent: TSynthPatch; const aName: string);                                            override;
    procedure   CreateIO;                                                                                      override;
    destructor  Destroy;                                                                                       override;
    procedure   CompileLinks;                                                                                  override;
    procedure   SetDefaults;                                                                                   override;
  public
    procedure   DumpConnections( anIndent: Integer; var S: string);
    procedure   DumpModules    ( anIndent: Integer; var S: string);
    procedure   DumpDetails    ( anIndent: Integer; var S: string);                                            override;
  public
    procedure   Clear;                                                                                         override;
    procedure   Reset;                                                                                         override;
    procedure   ResetSampleCounts;                                                                             override;
    procedure   ClearModules;
    procedure   Panic;                                                                                         override;
    function    FindModule( const aModuleName: string): Integer;
    function    ModuleName( anIndex: Integer): string;
    function    FindInputPin ( const aModuleIndex: Integer; const aPinName: string): Integer;
    function    FindOutputPin( const aModuleIndex: Integer; const aPinName: string): Integer;
    function    InputPinName ( aModuleIndex, aPinIndex: Integer): string;
    function    OutputPinName( aModuleIndex, aPinIndex: Integer): string;
    procedure   AddModule( const aModule: TMod);
    procedure   AddConnection( aSrcModule, aSrcPin, aDstModule, aDstPin: Integer);                             overload;
    procedure   AddConnection( const aSrcModule, aSrcPin, aDstModule, aDstPin: string);                        overload;
    procedure   DoTick;                                                                                        override;
    procedure   FixDeZippers;                                                                                  override;
    procedure   GatherSignals   ( const aPrefix: string; const aCallback: TSignalHandler);                     override;
    procedure   GatherLights    ( const aPrefix: string; const aCallback: TLightsHandler);                     override;
    procedure   GatherData      ( const aPrefix: string; const aCallback: TDataHandler);                       override;
    procedure   GatherXYData    ( const aPrefix: string; const aCallback: TXYDataHandler);                     override;
    procedure   GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler);                 override;
    procedure   GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler);                 override;
    procedure   TickWithData( aData: TSignalArray);
    procedure   ClearProfile;
    function    CreateProfileHtml: TStringList;
    function    IsSamePatch    ( const aPatch: TSynthPatch): Boolean;
    function    SameModules    ( const aPatch: TSynthPatch): Boolean;
    function    SameConnections( const aPatch: TSynthPatch): Boolean;
    procedure   AcceptMidi( const aMsg: TMidiMessage);                                                         override;
    procedure   AcceptShortMidi( const aMsg: TShortMidiMessage);                                               override;
    procedure   AcceptPn( aCh: Byte; aController, aValue: Integer; IsRegistered: Boolean);                     override;
    procedure   SendShortMidi( const aMsg: TShortMidiMessage);                                                 override;
    procedure   AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket);                                  override;
    procedure   SendOSCBytes( const aMsg: TBytes);                                                             override;
    procedure   SendMidiBytes( const aMsg: TBytes);                                                            override;
    procedure   SampleRateChanged;                                                                             override;
    procedure   TuningChanged;                                                                                 override;
    procedure   CollectDenormals( var aMap: TPinMaps);
  public
    property    Version                       : Integer             read FVersion           write FVersion;
    property    ModuleCount                   : Integer             read GetModuleCount;
    property    Module[ anIndex: Integer]     : TMod                read GetModule;
    property    ConnectionCount               : Integer             read GetConnectionCount;
    property    MidiChannel                   : Byte                read FMidiChannel       write FMidiChannel;
    property    Morph   [ anIndex: Integer  ] : TSignal             read GetMorph           write SetMorph;
    property    AutoFlag[ anIndex: TAutoFlag] : Boolean             read GetAutoFlag        write SetAutoFlag;
    property    LockSkips                     : Cardinal            read FLockSkips;
    property    OnSendShortMidi               : TOnShortMidiMessage read FOnSendShortMidi   write FOnSendShortMidi;
    property    OnSendOSCString               : TOnOSCString        read FOnSendOscString   write FOnSendOscString;
    property    OnSendMidiBytes               : TOnMidiBytes        read FOnSendMidiBytes   write FOnSendMidiBytes;
  end;


  procedure RegisterModuleType      ( const aModuleId: Integer; aModuleClass: TModuleClass);
  function  LookupModuleClass       ( const aModuleId: Integer): TModuleClass;
  function  LookupModuleTypeName    ( const aModuleId: Integer): string;
  function  LookupModuleGeneralClass( const aModuleId: Integer): TClass;
  procedure ParsePrefix ( const aPath: string; var aPrefix, aRest: string);
  function  ParseTextSequencerValue( const aValue: string): TSignalArray;
  function  SignalForInputType( const aSignal: TSignal; anInputType: Integer): TSignal;                          inline;
  function  SignalForOutputType( const aSignal: TSignal; anOutputType: Integer): TSignal;                        inline;
  function  SignalForEnvOutputType( const aSignal: TSignal; anOutputType: Integer): TSignal;                     inline;
  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignal)        : TLightsInfo;       overload;
  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalPair)    : TCursorInfo;       overload;
  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalArray)   : TDataInfo  ;       overload;
  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalPairFifo): TXYDataInfo;       overload;
  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: string)         : TStringInfo;       overload;



implementation


uses

  Globals;

var

  ModuleIdTypeMaps  : TModuleIdTypeMaps;


  procedure RegisterModuleType( const aModuleId: Integer; aModuleClass: TModuleClass);
  var
    aClass : TClass;
  begin
    Assert( Assigned( aModuleClass), 'Trying to register a NIL module type');
    aClass := LookupModuleClass( aModuleId);

    if   not Assigned( aClass)
    then begin
      SetLength( ModuleIdTypeMaps, Length( ModuleIdTypeMaps) + 1);
      ModuleIdTypeMaps[ Length( ModuleIdTypeMaps) - 1].ModuleId    := aModuleId;
      ModuleIdTypeMaps[ Length( ModuleIdTypeMaps) - 1].ModuleClass := aModuleClass;
    end
    else raise EModuleDuplicate.CreateFmt( 'Duplicate module ID %d for already present with type %s while trying to register type %s', [ aModuleId, aClass.ClassName, aModuleClass.ClassName]);
  end;


  function  LookupModuleClass( const aModuleId: Integer): TModuleClass;
  var
    M : TModuleIdTypeMap;
  begin
    Result := nil;

    for M in ModuleIdTypeMaps
    do begin
      if   M.ModuleId = aModuleId
      then begin
        Result := M.ModuleClass;
        Break;
      end;
    end;
  end;


  function  LookupModuleTypeName( const aModuleId: Integer): string;
  var
    aClass : TModuleClass;
  begin
    Result := '';
    aClass := LookupModuleClass( aModuleId);

    if   Assigned( aClass)
    then Result := LowerCase( Copy( aClass.ClassName, 5, Length( aClass.ClassName) - 4));
  end;


  function  LookupModuleGeneralClass( const aModuleId: Integer): TClass;
  var
    M : TModuleIdTypeMap;
  begin
    Result := nil;

    for M in ModuleIdTypeMaps
    do begin
      if   M.ModuleId = aModuleId
      then begin
        Result := M.ModuleClass;
        Break;
      end;
    end;
  end;


  procedure ParsePrefix( const aPath: string; var aPrefix, aRest: string);
  var
    i  : Integer;
  begin
    i       := 1;
    aPrefix := '';
    aRest   := '';

    while i <= Length( aPath)
    do begin
      if   aPath[ i] = '/'
      then begin
        aPrefix := Copy( aPath, 1, i - 1);
        aRest   := Copy( aPath, i + 1, Length( aPath));
        Break;
      end
      else Inc( i);
    end;
  end;



  function  ParseTextSequencerValue( const aValue: string): TSignalArray;

    procedure AddSignal( aValue: TSignal; var aResult: TSignalArray);
    var
      aLength : Integer;
    begin
      aLength := Length( aResult);
      SetLength( aResult, aLength + 1);
      aResult[ aLength] := aValue;
    end;

  var
    aStrings : TStringList;
    aString  : string;
    i        : Integer;
    j        : Integer;
    aSignal  : TSignal;
    aParts   : TStringList;
    aRepeats : Integer;
  begin
    Result := nil;
    aStrings := Explode( Trim( aValue), ',');

    try
      for i := 0 to aStrings.Count - 1
      do begin
        aString := Trim( aStrings[ i]);
        aParts  := Explode( aString, ':');

        try
          if   aParts.Count > 0
          then begin
            if   aParts.Count > 1
            then aRepeats := StrToIntDef( Trim( aParts[ 1]), 1)
            else aRepeats := 1;

            if   ParseTextValue( 'NoteName', Trim( aParts[ 0]), aSignal) // todo: is the NoteName type the right one here?
            then begin
              if   IsNan( aSignal)
              then aSignal := NO_NOTE;

              AddSignal( aSignal, Result);

              for j := 1 to aRepeats - 1
              do  AddSignal( NO_NOTE, Result);
            end
            else AddSignal( NO_NOTE, Result);
          end
          else AddSignal( NO_NOTE, Result);
        finally
          aParts.DisposeOf;
        end;
      end;
    finally
      aStrings.DisposeOf;
    end;
  end;


  function  SignalForInputType( const aSignal: TSignal; anInputType: Integer): TSignal; inline;
  // Translates input range to a [-1..1] output range
  begin
    case anInputType of
      inputtype_inverted : Result :=     - aSignal    ;         // [ 1..-1] -> [-1..1]
      inputtype_positive : Result :=   2 * aSignal - 1;         // [ 0.. 1] -> [-1..1]
      inputtype_pos_inv  : Result := - 2 * aSignal + 1;         // [ 1.. 0] -> [-1..1]
      inputtype_negative : Result :=   2 * aSignal + 1;         // [-1.. 0] -> [-1..1]
      inputtype_neg_inv  : Result := - 2 * aSignal - 1;         // [ 0..-1] -> [-1..1]
      else                 Result :=       aSignal    ;         // [-1.. 1] -> [-1..1]
    end;
  end;


  function  SignalForOutputType( const aSignal: TSignal; anOutputType: Integer): TSignal; inline;
  // Translates a [-1..1] range to an output range
  begin
    case anOutputType of
      outputtype_inverted : Result :=       - aSignal       ;   // [-1..1] -> [ 1..-1]
      outputtype_positive : Result := (   1 + aSignal) * 0.5;   // [-1..1] -> [ 0.. 1]
      outputtype_pos_inv  : Result := (   1 - aSignal) * 0.5;   // [-1..1] -> [ 1.. 0]
      outputtype_negative : Result := ( - 1 + aSignal) * 0.5;   // [-1..1] -> [-1.. 0]
      outputtype_neg_inv  : Result := ( - 1 - aSignal) * 0.5;   // [-1..1] -> [ 0..-1]
      else                  Result :=         aSignal       ;   // [-1..1] -> [-1.. 1]
    end;
  end;


  function  SignalForEnvOutputType( const aSignal: TSignal; anOutputType: Integer): TSignal; inline;
  // Translates a [0..1] range to an output range
  begin
    case anOutputType of
      envoutputtype_inverted : Result :=   - aSignal    ;       // [0..1] -> [ 0..-1]
      envoutputtype_negative : Result :=     aSignal - 1;       // [0..1] -> [-1.. 0]
      envoutputtype_neg_inv  : Result := 1 - aSignal    ;       // [0..1] -> [ 1.. 0]
      else                     Result :=     aSignal    ;       // [0..1] -> [ 0.. 1]
    end;
  end;


  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignal): TLightsInfo; overload;
  begin
    Result.Prefix := aPrefix;
    Result.Name   := aName;
    Result.Signal := aSignal;
    Result.Data   := aData;
  end;


  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalPair): TCursorInfo; overload;
  begin
    Result.Prefix := aPrefix;
    Result.Name   := aName;
    Result.Signal := aSignal;
    Result.Data   := aData;
  end;


  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalArray): TDataInfo  ; overload;
  begin
    Result.Prefix := aPrefix;
    Result.Name   := aName;
    Result.Signal := aSignal;
    Result.Data   := aData;
  end;


  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: TSignalPairFifo): TXYDataInfo; overload;
  begin
    Result.Prefix := aPrefix;
    Result.Name   := aName;
    Result.Signal := aSignal;
    Result.Data   := aData;
  end;


  function  MakeInfo( const aPrefix, aName, aSignal: string; const aData: string): TStringInfo; overload;
  begin
    Result.Prefix := aPrefix;
    Result.Name   := aName;
    Result.Signal := aSignal;
    Result.Data   := aData;
  end;


{ ========
  TConnection = class
  private
    Peer      : TSynthPatch;
    Index     : Integer;
    SrcModule : Integer;
    SrcPin    : Integer;
    DstModule : Integer;
    DstPin    : Integer;
  private
}

    procedure   TConnection.Connect( const aPeer: TSynthPatch; anIndex, aSrcModule, aSrcPin, aDstModule, aDstPin: Integer);
    begin
      Peer      := aPeer;
      Index     := anIndex;
      SrcModule := aSrcModule;
      SrcPin    := aSrcPin;
      DstModule := aDstModule;
      DstPin    := aDstPin;
    end;


//  public

    procedure   TConnection.Dump( anIndent: Integer; var S: string);
    var
      aSrcModName : string;
      aSrcPinName : string;
      aDstModName : string;
      aDstPinName : string;
      T           : string;
    begin
      if   Assigned( Peer)
      then begin
        aSrcModName := Peer.ModuleName   ( SrcModule);
        aSrcPinName := Peer.OutputPinName( SrcModule, SrcPin);
        aDstModName := Peer.ModuleName   ( DstModule);
        aDstPinName := Peer.InputPinName ( DstModule, DstPin);

        T := Format( '%sconnect( %s %s %s %s)' + sLineBreak, [ Indent( anIndent), aSrcModName, aSrcPinName, aDstModName, aDstPinName], AppLocale);
        S := S + T
      end;
    end;


    function    TConnection.IsSameConnection( aConnection: TConnection): Boolean;
    begin
      Result := False;

      if   Assigned( aConnection)
      then
        Result :=
          ( SrcModule = aConnection.SrcModule) and
          ( SrcPin    = aConnection.SrcPin   ) and
          ( DstModule = aConnection.DstModule) and
          ( DstPin    = aConnection.DstPin   );
    end;


{ ========
  TConnections = class
  private
    FPeer       : TSynthPatch;
    FConnections: TConnectionArray;
  public
    property    Count: Integer                             read GetCount;
    property    Connection[ anIndex: Integer]: TConnection read GetConnection write SetConnection;
  private
}

    function    TConnections.GetCount: Integer;
    begin
      Result := Length( FConnections);
    end;


    function    TConnections.GetConnection( anIndex: Integer): TConnection;
    begin
      Result := FConnections[ anIndex]
    end;


    procedure   TConnections.SetConnection( anIndex: Integer; const aValue: TConnection);
    begin
      FConnections[ anIndex] := aValue;
    end;


//  public

    constructor TConnections.Create( aPeer: TSynthPatch);
    begin
      inherited Create;
      FPeer := aPeer;
    end;


    destructor  TConnections.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TConnections.Dump( anIndent: Integer; const aName: string; var S: string);
    var
      i : Integer;
    begin
      S := S + Indent( anIndent) + aName + '(' + sLineBreak;

      for i := 0 to Count - 1
      do  Connection[ i].Dump( anIndent + 1, S);

      S := S + Indent( anIndent) + ')' + sLineBreak;
    end;


    procedure   TConnections.Clear;
    begin
      while Count > 0
      do    RemoveConnection( Count - 1);
    end;


    procedure   TConnections.AddConnection( const aValue: TConnection); // overload;
    begin
      SetLength( FConnections, Count + 1);
      FConnections[ Count - 1] := aValue;
    end;


    procedure   TConnections.AddConnection( aSrcModule, aSrcPin, aDstModule, aDstPin: Integer);  // overload;
    var
      aConnection: TConnection;
    begin
      aConnection := TConnection.Create;
      aConnection.Connect( FPeer, Count, aSrcModule, aSrcPin, aDstModule, aDstPin);
      AddConnection( aConnection);
    end;


    procedure   TConnections.RemoveConnection( anIndex: Integer); // overload;
    var
      aConnection: TConnection;
      i          : Integer;
    begin
      aConnection := Connection[ anIndex];

      if   Assigned( aConnection)
      then begin
        for i := anIndex to Count - 2
        do  Connection[ i] := Connection[ i + 1];

        SetLength( FConnections, Count - 1);
        aConnection.DisposeOf;
      end;
    end;


    function    TConnections.IsSameConnections( const aConnections: TConnections): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if   Assigned( aConnections)
      and  ( Count = aConnections.Count)
      then begin
        Result := True;

        for i := 0 to Count - 1
        do begin
          if   not Connection[ i].IsSameConnection( aConnections.Connection[ i])
          then begin
            Result := False;
            Break;
          end;
        end;
      end;
    end;


{ ========
  TLinks = record
  private
    FCount  : Integer;
    FSource : PSignal;
    FLinks  : PSignals;
  private
    function    FindLink( aDestination: PSignal): Integer;
  public
}

    function    TLinks.FindLink( aDestination: PSignal): Integer;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to FCount - 1
      do begin
        if   FLinks[ i] = aDestination
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


//  public

    procedure   TLinks.Initialize( aSource: PSignal);
    begin
      FSource := aSource;
      FLinks  := nil;
      FCount  := 0;
    end;


    procedure   TLinks.Uninitialize;
    begin
      if   Assigned( FLinks)
      then FreeMem( FLinks);

      FLinks := nil;
      FCount := 0;
    end;


    procedure   TLinks.AddLink( aDestination: PSignal);
    var
      anOldLinks : PSignals;
      aLinkIndex : Integer;
    begin
      aLinkIndex := FindLink( aDestination);

      if   aLinkIndex < 0
      then begin
        anOldLinks := FLinks;
        GetMem( FLinks, ( FCount + 1) * SizeOf( PSignal));

        if   Assigned( anOldLinks)
        then begin
          Move( anOldLinks^, FLinks^, FCount * SizeOf( PSignal));
          FreeMem( anOldLinks);
        end;

        FLinks^[ FCount] := aDestination;
        Inc( FCount);
      end;
    end;


    procedure   TLinks.RemoveLink( aDestination: PSignal);
    var
      anOldLinks : PSignals;
      aLinkIndex : Integer;
    begin
      aLinkIndex := FindLink( aDestination);

      if   aLinkIndex >= 0
      then begin
        anOldLinks := FLinks;
        GetMem( FLinks, ( FCount + 1) * SizeOf( PSignal));

        if   Assigned( anOldLinks)
        then begin
          Move( anOldLinks^             , FLinks^,              ( aLinkIndex              - 1) * SizeOf( PSignal));
          Move( anOldLinks^[ aLinkIndex], FLinks^[ aLinkIndex], ( FCount     - aLinkIndex - 1) * SizeOf( PSignal));
          FreeMem( anOldLinks);
        end;

        Dec( FCount);
      end;
    end;


{ ========
  TOutputLinks = class
  private
    FCount    : Integer;
    FOutLinks : PLinksArray;
  private
}


    function    TOutputLinks.FindSource( aSource: PSignal): Integer;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to FCount - 1
      do begin
        if   FOutLinks^[ i].FSource = aSource
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


//  public

    destructor  TOutputLinks.Destroy; // override;
    begin
      ClearLinks;
      inherited;
    end;


    procedure   TOutputLinks.ClearLinks;
    var
      i : Integer;
    begin
      if   Assigned( FOutLinks)
      then begin
        for i := 0 to FCount - 1
        do FOutLinks^[ i].Uninitialize;

        FreeMem( FOutLinks);
        FOutLinks := nil;
        FCount    := 0;
      end;
    end;


    procedure   TOutputLinks.AddLink( aSource, aDestination: PSignal);
    var
      anIndex    : Integer;
      anOldLinks : PLinksArray;
    begin
      anIndex := FindSource( aSource);

      if   anIndex < 0
      then begin
        anOldLinks := FOutLinks;
        GetMem( FOutLinks, ( FCount + 1) * SizeOf( TLinks));

        if   Assigned( anOldLinks)
        then begin
          Move( anOldLinks^, FOutLinks^, FCount * SizeOf( TLinks));
          FreeMem( anOldLinks);
        end;

        FOutLinks^[ FCount].Initialize( aSource);
        anIndex := FCount;
        Inc( FCount);
      end;

      FOutLinks^[ anIndex].AddLink( aDestination);
    end;


    procedure   TOutputLinks.RemoveLink( aSource, aDestination: PSignal);
    var
      anIndex    : Integer;
      anOldLinks : PLinksArray;
    begin
      anIndex := FindSource( aSource);

      if   anIndex >= 0
      then begin
        FOutLinks^[ anIndex].RemoveLink( aDestination);

        if   FOutLinks^[ anIndex].FCount <= 0
        then begin
          anOldLinks := FOutLinks;
          GetMem( FOutLinks, ( FCount - 1) * SizeOf( TLinks));

          if   Assigned( anOldLinks)
          then begin
            Move( anOldLinks^          , FOutLinks^          , ( anIndex           - 1) * SizeOf( TLinks));
            Move( anOldLinks^[ anIndex], FOutLinks^[ anIndex], ( FCount  - anIndex - 1) * SizeOf( TLinks));
            FreeMem( anOldLinks);
          end;

          Dec( FCount);
        end;
      end;
    end;


    procedure   TOutputLinks.Compile( const aModule: TMod; const aPatch: TSynthPatch);
    var
      aModuleIndex : Integer;
      aConnections : TConnections;
      aConnection  : TConnection;
      i            : Integer;
      aDstModule   : TMod;
      aSource      : PSignal;
      aDestination : PSignal;
    begin
      ClearLinks;

      if   Assigned( aModule)
      and  Assigned( aPatch)
      then begin
        aConnections := aPatch.FConnections;

        if   Assigned( aConnections)
        then begin
          aModuleIndex := aPatch.FindModule( aModule.Name);

          if   aModuleIndex >= 0
          then begin
            for i := 0 to aConnections.Count - 1
            do begin
              aConnection := aConnections.Connection[ i];

              if   aConnection.SrcModule = aModuleIndex
              then begin
                aDstModule   := aPatch.Module[ aConnection.DstModule];
                aSource      := @ aModule   .FOutputs[ aConnection.SrcPin];
                aDestination := @ aDstModule.FInputs [ aConnection.DstPin];
                AddLink( aSource, aDestination);
              end;
            end;
          end;
        end;
      end;
    end;


(* ========
  TMod = class
  private
    FParent          : TSynthPatch;       // The patch this module is part off
    FLocked          : Boolean;           // When true module should not be executed
    FExecuting       : Boolean;           // True when the module is currently being executed
    FName            : string;            // A unique name for this module - unique within it's parent
    FResetFlag       : Integer;           // Set when a module reset was requested
  {$IFDEF PATCH_PROFILER}
  private
    FSamplesFast     : Int64;             // Sample counter, incremented each time a new Fast sample is calcultaed
    FSamplesSlow     : Int64;             // Sample counter, incremented each time a new Slow sample is calcultaed
    FClocksFast      : Int64;             // CPU Fast clocks for current cycle
    FClocksSlow      : Int64;             // CPU Slow clocks for current cycle
    FTotalClocksFast : Int64;             // CPU Fast clocks accumulated over the module life time
    FTotalClocksSlow : Int64;             // CPU Slow clocks accumulated over the module life time
    FZips            : Int64;             // CPU clocks spent on dezippering for the current cycle
    FTotalZips       : Int64;             // CPU clocks acumulated FZips over the module life time
  {$ENDIF}

  protected
    FIsInput         : Boolean;           // True for a TModInput  module only
    FIsOutput        : Boolean;           // True for a TModOutput module only
    FIsFast          : Boolean;           // True when fast mode processing must be performed
    FIsSpedUp        : Boolean;           // True when module is normally slow but sped up due to fast signals being present on an input.
    FIsSlow          : Boolean;           // True when slow mode processing must be performed
    FIsMidi          : Boolean;           // True when MIDI reception must be enabled - for MIDI modules only
    FIsOSC           : Boolean;           // True when OSC  reception must be enabled - for OSC  modules only
    FInputCount      : Integer;           // Amount of input and amount of dezipper signals, input and dezipper arrays have the same length
    FOutputCount     : Integer;           // Amount of output signals
    FInputs          : TSignalArray;      // The input signals, these include both patch signals and user controls, FInputs and FDezips have the same length
    FDezips          : TSignalArray;      // An extra set of inputs for user controls, these are copied to FInputs with some filtering when the signal happens to be be in the deziper map
    FOutputs         : TSignalArray;      // The ouput signals.
    FDezipperMap     : TByteSet;          // Determines  what signals should be dezippered, most, but not all, knobs should be dezippered
  [weak] FOutputLinks: TOutputLinks;      // For each output a set of signals that should be updated from it, compiled connections
                                          // This is a weak reference to make reference counting work properly.
  public
    property    Parent                                           : TSynthPatch read FParent;
    property    Locked                                           : Boolean     read FLocked          write SetLocked;
    property    Name                                             : string      read FName            write SetName;
    property    ResetFlag                                        : Boolean     read GetResetFlag     write SetResetFlag;
    property    InputCount                                       : Integer     read FInputCount;
    property    OutputCount                                      : Integer     read FOutputCount;
    property    Output[ anIndex: Integer]                        : TSignal     read GetOutput;
    property    IsFast                                           : Boolean     read FIsFast;
    property    IsSlow                                           : Boolean     read FIsSlow;
    property    IsMidi                                           : Boolean     read FIsMidi;
    property    IsOSC                                            : Boolean     read FIsOSC;
    property    IsSpedUp                                         : Boolean     read FIsSpedUp;
  public
    property    InternalValue      [ const aName: string       ] : TSignal                           write SetInternalValue;
    property    AutomationValue    [ const aName: string       ] : TSignal                           write SetAutomationValue;
    property    InternalStringValue[ const aType, aName: string] : string                            write SetInternalStringValue;
    property    StringValue        [ const aName: string       ] : string      read GetStringValue   write SetStringValue;
  {$IFDEF PATCH_PROFILER}
  public
    property    SamplesFast                                      : Int64       read FSamplesFast;
    property    SamplesSlow                                      : Int64       read FSamplesSlow;
    property    TotalClocksFast                                  : Int64       read FTotalClocksFast;
    property    TotalClocksSlow                                  : Int64       read FTotalClocksSlow;
    property    TotalZips                                        : Int64       read FTotalZips;
  {$ENDIF}
  private
*)

    function    TMod.AssertUniqueName( const aName: string): string; // virtual;
    begin
      if   Assigned( FParent)
      then Result := FParent.AssertUniqueName( aName)
      else Result := aName;
    end;


//  private

    procedure   TMod.SetLocked( aValue: Boolean);
    // Note: Do not use a monitor construct here .. it is too slow ...
    begin
      if   aValue
      then begin
        FLocked := True;                         // Just claim the lock ...

        while FExecuting                         // ... but then wait for any execution to finish, which will be quick,
        do ;                                     // and as long as Locked is True execution will not be re-entered.
      end
      else FLocked := False;                     // The lock can be released at all times
    end;


    procedure   TMod.SetName( const aValue: string);
    begin
      if   aValue <> FName
      then FName := AssertUniqueName( aValue);
    end;


    function    TMod.GetResetFlag: Boolean; // virtual;
    begin
      Result := FResetFlag > 0;
    end;


    procedure   TMod.SetResetFlag( aValue: Boolean); // virtual;
    begin
      if   aValue
      then FResetFlag := 1
      else DecToZero( FResetFlag);
    end;


    function    TMod.GetOutput( anIndex: Integer): TSignal;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < OutputCount)
      then Result := FOutputs[ anIndex]
      else Result := 0.0;
    end;


    procedure   TMod.SetCommonValue( anIndex: Integer; aValue: TSignal; Dezipper: Boolean); // virtual;
    begin
      if   anIndex >= 0
      then begin
        if   Dezipper
        and  ( anIndex in FDezipperMap)
        then FDezips[ anIndex] := aValue
        else begin
          if   FInputs[ anIndex] = aValue
          then Pulse( anIndex)
          else begin
            FInputs[ anIndex] := aValue;
            FDezips[ anIndex] := aValue;
          end;
        end;
      end;
    end;


    procedure   TMod.SetInternalValue( const aName: string; aValue: TSignal); // virtual;
    var
      anIndex: Integer;
    begin
      anIndex := FindInput( aName);

      if   anIndex >= 0
      then SetCommonValue( anIndex, aValue, DO_DEZIPPER)
      else SetInternal( aName, aValue);
    end;


    procedure   TMod.SetInternalStringValue( const aType, aName, aValue: string);
    var
      aSignal : TSignal;
    begin
      if   ParseTextValue( aType, aValue, aSignal)
      then InternalValue[ aName] := aSignal
      else raise EUnitsUndefined.CreateFmt( 'Unparsable value ''%s''', [ aName])
    end;


    procedure   TMod.SetAutomationValue( const aName: string; aValue: TSignal); // virtual;
    var
      anIndex: Integer;
    begin
      anIndex := FindInput( aName);

      if   anIndex >= 0
      then SetCommonValue( anIndex, aValue, NO_DEZIPPER)
      else SetInternal( aName, aValue);
    end;


//  protected

    procedure   TMod.SetStringValue( const aName, aValue: string); // virtual;
    begin
      // Nothing here.
    end;


    function    TMod.GetStringValue( const aName: string): string; // virtual;
    begin
      Result := '';
    end;


    procedure   TMod.PerformMorph( aChannel: Integer; aValue: TSignal);
    begin
      if   Self is TSynthPatch
      then TSynthPatch( Self).Morph[ aChannel] := aValue
      else begin
        if   Assigned( FParent)
        then FParent.Morph[ aChannel] := aValue;
      end;
    end;


//  protected

    function    TMod.GetInputCount: Integer;
    begin
      Result := Length( FInputMaps);
    end;


    function    TMod.GetOutputCount: Integer;
    begin
      Result := Length( FOutputMaps);
    end;


    function    TMod.FindInputID( anID: Integer): Integer;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to InputCount - 1
      do begin
        if   FInputMaps[ i].ID = anID
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TMod.FindOutputID( anID: Integer): Integer;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to OutputCount - 1
      do begin
        if   FOutputMaps[ i].ID = anID
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TMod.FindInputName( anID: Integer): string;
    var
      p : Integer;
    begin
      p := FindInputID( anId);

      if   p >= 0
      then Result := FInputMaps[ p].Name
      else Result := NO_INPUT;
    end;


    function    TMod.FindOutputName( anID: Integer): string;
    var
      p : Integer;
    begin
      p := FindOutputID( anId);

      if   p >= 0
      then Result := FOutputMaps[ p].Name
      else Result := NO_OUTPUT;
    end;


    procedure   TMod.AddInput( anID: Integer; const aName: string);
    var
      p : Integer;
      N : string;
    begin
      N := Trim( aName);
      p := FindInputID( anID);

      if   p >= 0
      then FInputMaps[ p].Name := N
      else begin
        SetLength( FInputMaps, InputCount + 1);
        FInputMaps[ InputCount - 1].ID   := anID;
        FInputMaps[ InputCount - 1].Name := N;
      end;
    end;


    procedure   TMod.AddOutput( anID: Integer; const aName: string);
    var
      p : Integer;
      N : string;
    begin
      N := Trim( aName);
      p := FindOutputID( anID);

      if   p >= 0
      then FOutputMaps[ p].Name := N
      else begin
        SetLength( FOutputMaps, OutputCount + 1);
        FOutputMaps[ OutputCount - 1].ID   := anID;
        FOutputMaps[ OutputCount - 1].Name := N;
      end;
    end;


    procedure   TMod.FixIO;
    begin
      SetLength( FOutputs, OutputCount);
      SetLength( FInputs , InputCount );
      SetLength( FDezips , InputCount );
    end;


    function    TMod.HandleParam( const aControl: string; aValue: TSignal): Boolean; // virtual;
    // Handle a parameter set from the editor patch
    begin
      InternalValue[ aControl] := aValue;
      Result := True;
    end;


    function    TMod.HandleAutomationParam( const aControl: string; aValue: TSignal): Boolean; // virtual;
    begin
      AutomationValue[ aControl] := aValue;
      Result := True;
    end;


    procedure   TMod.Log( const aMsg: string); // virtual;
    begin
      if   Assigned( FParent)
      then FParent.Log( aMsg);
    end;


    procedure   TMod.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


    {$IFDEF PATCH_PROFILER}

//  protected

    procedure   TMod.PreTickFast;
    begin
      {$Q-R-}
      FClocksFast := GetCpuClockCycleCount;
      {$Q+R+}
    end;


    procedure   TMod.PostTickFast;
    begin
      {$Q-R-}
      FTotalClocksFast := FTotalClocksFast + GetCpuClockCycleCount - FClocksFast;
      {$Q+R+}
    end;


    procedure   TMod.PreTickSlow;
    begin
      {$Q-R-}
      FClocksSlow := GetCpuClockCycleCount;
      {$Q+R+}
    end;


    procedure   TMod.PostTickSlow;
    begin
      {$Q-R-}
      FTotalClocksSlow := FTotalClocksSlow + GetCpuClockCycleCount - FClocksSlow;
      {$Q+R+}
    end;


    procedure   TMod.PreZipper;
    begin
      {$Q-R-}
      FZips := GetCpuClockCycleCount;
      {$Q+R+}
    end;


    procedure   TMod.PostZipper;
    begin
      {$Q-R-}
      FTotalZips := FTotalZips + GetCpuClockCycleCount - FZips;
      {$Q+R+}
    end;


    procedure   TMod.ClearProfile;
    begin
      FTotalClocksFast := 0;
      FTotalClocksSlow := 0;
      FSamplesFast     := 0;
      FSamplesSlow     := 0;
      FTotalZips       := 0;
    end;

    {$ENDIF}


//  public

    constructor TMod.Create( aParent: TSynthPatch; const aName: string); // virtual;
    begin
      inherited Create;
      {$IFDEF PATCH_PROFILER}
      FTotalClocksSlow := 0;
      FTotalClocksFast := 0;
      FTotalZips       := 0;
      {$ENDIF}
      FISInput         := False;
      FIsOutput        := False;
      FIsFast          := False;
      FIsSlow          := False;
      FIsMidi          := False;
      FIsOSC           := False;
      FParent          := aParent;
      Name             := aName;
      FOutputLinks     := TOutputLinks.Create;
    end;


    procedure   TMod.CreateIO; // virtual;
    begin
    end;


    procedure   TMod.Initialize; // virtual;
    begin
      CreateIO;
      FixIO;
      SetDefaults;
    end;


    destructor  TMod.Destroy; // override;
    begin
      Locked  := True;
      FParent := nil;
      FreeAndNil( FOutputLinks);
      inherited;
    end;


    procedure   TMod.CompileLinks; // virtual;
    begin
      Locked := True;

      try
        FOutputLinks.Compile( Self, FParent);
      finally
        Locked := False;
      end;
    end;


//    procedure   TMod.PropagateSignals;
//    // This seems to be marginally slower than the code below
//    var
//      i : Integer;
//      j : Integer;
//      L : PLinksArray;
//      S : PSignal;
//      D : PSignals;
//      C : Integer;
//    begin
//      for i := 0 to FOutputLinks.FCount - 1
//      do begin
//        L := FOutputLinks.FOutLinks;
//        S := L^[ i].FSource;
//        D := L^[ i].FLinks;
//        C := L^[ i].FCount - 1;
//
//        for j := 0 to C
//        do  D^[ j]^ := S^;
//      end;
//    end;

    procedure   TMod.PropagateSignals;
    // This seems to be marginally faster than the code above
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to FOutputLinks.FCount - 1
      do begin
        for j := 0 to FOutputLinks.FOutLinks^[ i].FCount - 1
        do  FOutputLinks.FOutLinks^[ i].FLinks^[ j]^ := FOutputLinks.FOutLinks^[ i].FSource^;
      end;
    end;


    procedure   TMod.SetDefaults; // virtual;
    begin
    end;


//  public

    procedure   TMod.DumpKeyValue( anIndent: Integer; const aName: string; aValue: TSignal; var S: string);
    var
      aBaseName : string;
      i         : Integer;
      Output    : string;
    begin
      aBaseName := '';

      for i := 1 to Length( aName)
      do begin
        if   CharInSet( aName[ i], IdentifierStarters)
        then aBaseName := abaseName + aName[ i]
        else Break;
      end;

      Output := Format( '%s', [ SignalToStr( aValue)], AppLocale);
      S      := S + Format( Indent( anIndent + 1) + '%s %s', [ aName, Output], AppLocale);
    end;


    procedure   TMod.DumpInputValues( anIndent: Integer; var S: string);
    var
      i : Integer;
    begin
      S := S + Indent( anIndent) + 'inputs(' + sLineBreak;

      for i := 0 to InputCount - 1
      do begin
        DumpKeyValue( anIndent, FInputMaps[ i].Name, FInputs[ FInputMaps[ i].ID], S);
        S := S + sLineBreak;
      end;

      S := S + Indent( anIndent) + ')' + sLineBreak;
    end;


    procedure   TMod.DumpOutputValues( anIndent: Integer; var S: string);
    var
      i : Integer;
    begin
      S := S + Indent( anIndent) + 'outputs(' + sLineBreak;

      for i := 0 to OutputCount - 1
      do begin
        DumpKeyValue( anIndent, FOutputMaps[ i].Name, FOutputs[ FOutputMaps[ i].ID], S);
        S := S + sLineBreak;
      end;

      S := S + Indent( anIndent) + ')' + sLineBreak;
    end;


    procedure   TMod.DumpDetails( anIndent: Integer; var S: string); // virtual;
    begin
      S := S + Format ( Indent( anIndent) + '%s' + sLineBreak, [ ClassName ], AppLocale);
      S := S + Format ( Indent( anIndent) + '%s' + sLineBreak, [ Name      ], AppLocale);
      DumpInputValues ( anIndent, S);
      DumpOutputValues( anIndent, S);
    end;


    procedure   TMod.Dump( anIndent: Integer; var S: string);
    begin
      S := S + Indent( anIndent) + 'module(' + sLineBreak;
      DumpDetails( anIndent + 1, S);
      S := S + Indent( anIndent) + ')' + sLineBreak;
    end;


//  public

    procedure   TMod.SetInternal( const aName: string; aValue: TSignal); // virtual;
    begin
      // Internal is an option not present here.
    end;


    function    TMod.GetInternal( const aName: string): TSignal; // virtual;
    begin
      // Internal is an option not present here.
      Result := UNDEFINED;
    end;


    procedure   TMod.Reset; // virtual;
    begin
      ResetFlag := True;
    end;


    procedure   TMod.ResetSampleCounts;
    begin
    {$IFDEF PATCH_PROFILER}
      FSamplesFast := 0;
      FSamplesSlow := 0;
    {$ENDIF}
    end;


    procedure   TMod.Clear; // virtual;
    begin
    end;


    procedure   TMod.Panic; // virtual;
    begin
      // No panic here ...
    end;


    procedure   TMod.Pulse( anInput: Integer); // virtual;
    begin
      // No pulsing here ...
    end;


    function    TMod.FindInput( const aPinName: string): Integer; // virtual;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to InputCount - 1
      do begin
        if   SameText( aPinName, FInputMaps[ i].Name)
        then begin
          Result := FInputMaps[ i].ID;
          Break;
        end;
      end;
    end;


    function    TMod.FindOutput( const aPinName: string): Integer; // virtual;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to OutputCount - 1
      do begin
        if   SameText( aPinName, FOutputMaps[ i].Name)
        then begin
          Result := FOutputMaps[ i].ID;
          Break;
        end;
      end;
    end;


    procedure   TMod.MakeFast;
    begin
      FIsSpedUp := True;
    end;


    procedure   TMod.Tick;
    begin
      {$IFDEF PATCH_PROFILER}
      {$Q-R-}
      Inc( FSamplesFast);
      {$Q+R+}
      PreTickFast;
      {$ENDIF}
      DoTick;
      {$IFDEF PATCH_PROFILER}PostTickFast;{$ENDIF}
    end;


    procedure   TMod.SlowTick;
    begin
      {$IFDEF PATCH_PROFILER}
      {$Q-R-}
      Inc( FSamplesSlow);
      {$Q+R+}
      PreTickSlow;
      {$ENDIF}
      DoSlowTick;
      {$IFDEF PATCH_PROFILER}PostTickSlow;{$ENDIF}
    end;


    procedure   TMod.DeZipper;
    const
      Speed = 1.0 / 64.0;
    var
      i : Integer;
    begin
      {$IFDEF PATCH_PROFILER} PreZipper; {$ENDIF}
      for i := 0 to Length( FInputs) - 1
      do begin
        if   i in FDezipperMap
        then FInPuts[ i] := NormalizeMute( Speed * ( FDezips[ i] - FInputs[ i]) + FInputs[ i]);
      end;
      {$IFDEF PATCH_PROFILER} PostZipper; {$ENDIF}
    end;


    procedure   TMod.DoTick; // virtual;
    begin
      // audio rate
    end;


    procedure   TMod.DoSlowTick; // virtual;
    begin
      // control rate
    end;


    procedure   TMod.FixDeZippers; // virtual;
    var
      i : Integer;
    begin
      for i := 0 to Length( FInputs) - 1
      do begin
        if   i in FDezipperMap
        then FInputs[ i] := FDezips[ i];
      end;
    end;


    procedure   TMod.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // virtual;
    begin
      // Collect info to be sent to module controls - this module has no controls
    end;


    procedure   TMod.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // virtual;
    begin
      // Collect info to be sent to module lights - this module has no lights
    end;


    procedure   TMod.GatherData( const aPrefix: string; const aCallback: TDataHandler); // virtual;
    begin
      // Collect numerical info to be sent to module viewers - this module has no viewers
    end;


    procedure   TMod.GatherXYData( const aPrefix: string; const aCallback: TXYDataHandler); // virtual;
    begin
      // Collect numerical info to be sent to module XY viewers - this module has no XY viewers
    end;


    procedure   TMod.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // virtual;
    begin
      // Collect string info to be sent to module viewers - this module has no viewers
    end;


    procedure   TMod.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // virtual;
    begin
      // Collect cursor data info to be sent to module data makers - this module has no data makers
    end;


    function    TMod.AcceptParam( const aPath: string; aValue: TSignal): Boolean;
    // Accept knob data sent from visual modules
    var
      aPrefix : string;
      aParam  : string;
    begin
      Result := False;

      if ( ClassType = TSynthPatch)
      then Result := HandleParam( aPath, aValue)
      else begin
        ParsePrefix( aPath, aPrefix, aParam);

        if   SameText( aPrefix, Name)                      // is the message for us?
        then begin
          if   Pos( '/', aParam) = 0                       // When no more slashes left
          then begin                                       // we must be at the final bit ...
            aParam := ParsePostfix( aParam);
            Result := HandleParam( aParam, aValue);
          end;
        end;
      end;
    end;


    function   TMod.AcceptAutomationParam( const aPath: string; aValue: TSignal): Boolean;
    // Accept knob data sent from visual modules
    var
      aPrefix : string;
      aParam  : string;
    begin
      Result := False;

      if   ( ClassType = TSynthPatch)
      then Result := HandleAutomationParam( aPath, aValue)
      else begin

        ParsePrefix( aPath, aPrefix, aParam);

        if   SameText( aPrefix, Name)                      // is the message for us?
        then begin
          if   Pos( '/', aParam) = 0                       // When no more slashes left
          then begin                                       // we must be at the final bit ...
            aParam := ParsePostfix( aParam);
            Result := HandleAutomationParam( aParam, aValue);
          end;
        end;
      end;
    end;


    function    TMod.GetProfileInfoHtml: string;
    {$IFDEF PATCH_PROFILER}
    var
      FR          : string;
      SR          : string;
      FastSamples : Int64;
      FastClocks  : Int64;
      SlowSamples : Int64;
      SlowClocks  : Int64;
    {$ENDIF}
    begin
    {$IFDEF PATCH_PROFILER}
      if   IsSpedUp
      then begin
        SlowSamples := 0;
        SlowClocks  := 0;
        FastSamples := SamplesSlow;
        FastClocks  := TotalClocksSlow;
      end
      else begin
        SlowSamples := SamplesSlow;
        SlowClocks  := TotalClocksSlow;
        FastSamples := SamplesFast;
        FastClocks  := TotalClocksFast;
      end;

      if   FastSamples = 0
      then FR := ''
      else FR := Format( '%.1f', [ FastClocks / FastSamples], AppLocale);

      if   SlowSamples = 0
      then begin
        if IsSpedUp
        then SR := '*'
        else SR := '';
      end
      else SR := Format( '%.1f', [ SlowClocks / SlowSamples / Control_Decimation], AppLocale);

      Result :=
        Format(
          '<tr><td>%s</td> <td>%s</td> <td align="right">%s</td> <td align="right">%s</td></tr>',
          [
            Name,
            ClassName,
            FR,
            SR
          ],
          AppLocale
        );
    {$ELSE}
      Result := '';
    {$ENDIF}
    end;


    function    TMod.IsSameModule( const aModule: TMod): Boolean;
    begin
      Result := False;

      if   Assigned( aModule)
      then Result := ClassType = aModule.ClassType;
    end;


    procedure   TMod.AcceptMidi( const aMsg: TMidiMessage); // virtual;
    begin
      // Nothing here ... for inheritance only
    end;


    procedure   TMod.AcceptPn( aCh: Byte; aController, aValue: Integer; IsRegistered: Boolean); // virtual;
    begin
      // Nothing here ... for inheritance only
    end;


    procedure   TMod.AcceptShortMidi( const aMsg: TShortMidiMessage); // virtual;
    begin
      // Nothing here ... for inheritance only
    end;


    procedure   TMod.SendShortMidi( const aMsg: TShortMidiMessage);
    begin
      if   Assigned( FParent)
      then FParent.SendShortMidi( aMsg);
    end;


    function    TMod.OSCMatched( const aMsg: TOSCMessage): Boolean; // virtual;
    begin
      Result := False;
    end;


    procedure   TMod.AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket); // virtual;
    begin
      // Nothing here, if is for children, just building daydreams ...
    end;


    procedure   TMod.SendOSCBytes( const aMsg: TBytes); // virtual;
    begin
      if   Assigned( FParent)
      then FParent.SendOSCBytes( aMsg);
    end;


    procedure   TMod.SendMidiBytes( const aMsg: TBytes); // virtual;
    begin
      if   Assigned( FParent)
      then FParent.SendMidiBytes( aMsg);
    end;


    procedure   TMod.SampleRateChanged; // virtual;
    // Notification for sample rate change, the actual values are in global variables
    // System_Rate for the audio rate and Control_Rate for the control rate.
    // Modules caching sample rate dependent data should recalculate such data in an
    // overriden version of this procedure.
    begin
      // Nothing for abstract modules.
    end;


    procedure   TMod.TuningChanged; // virtual;
    // Notification for tuning change, the actual values are in global variables
    // ReferenceA, NotesPerOctave and MiddleNote
    begin
      // Nothing for abstract modules.
    end;


    procedure   TMod.CollectDenormals( var aMap: TPinMap);
    var
      i : Integer;
    begin
      aMap.Ins  := [];
      aMap.Outs := [];

      for i := 0 to InputCount - 1
      do begin
        if   RegardAsDenormal( FInputs[ i])
        then aMap.Ins := aMap.Ins + [ i];
      end;

      for i := 0 to OutputCount - 1
      do begin
        if   RegardAsDenormal( FOutputs[ i])
        then aMap.Outs := aMap.Outs + [ i];
      end;
    end;


{ ========
  TModInput = class( TMod)
  public
}

    procedure   TModInput.CreateIO; // override;
    var
      i : Integer;
    begin
      FIsInput := True;
      FIsFast  := True;             // Input module must be set fast or propagation will not work

      for i := 0 to IN_CHANNEL_COUNT - 1
      do  AddOutput( i, Format( 'out%d', [ i + 1], AppLocale));
    end;


{ ========
  TModOutput = class( TMod)
  public
}

    procedure   TModOutput.CreateIO; // override;
    var
      i : Integer;
    begin
      FIsOutput := True; // Output modules need not be made fast, it's implicit

      for i := 0 to OUT_CHANNEL_COUNT - 1
      do  AddInput( i, Format( 'in%d', [ i + 1], AppLocale));
    end;


(* ========
  TSynthPatch = class
  private
    FVersion         : Integer;
    FMidiChannel     : Byte;
    FModules         : TModuleArray;
    FConnections     : TConnections;
    FDecimatorCount  : Integer;
    FInternalMidiMsg : TMidiMessage;
    FLiveMorph       : TSignal;
    FAutoFlags       : TAutoFlags;
    FLockSkips       : Cardinal;
    FOnSendShortMidi : TOnShortMidiMessage;
    FOnSendOscString : TOnOSCString;
    FOnSendMidiBytes : TOnMidiBytes;
  public
    property    Version                      : Integer             read FVersion           write FVersion;
    property    ModuleCount                  : Integer             read GetModuleCount;
    property    Module[ anIndex: Integer]    : TMod                read GetModule;
    property    ConnectionCount              : Integer             read GetConnectionCount;
    property    MidiChannel                  : Byte                read FMidiChannel       write FMidiChannel;
    property    LiveMorph                    : TSignal             read FLiveMorph;
    property    AutoFlag[ anIndex: TAutoFlag]: Boolean             read GetAutoFlag        write SetAutoFlag;
    property    LockSkips                    : Cardinal            read FLockSkips;
    property    OnSendShortMidi              : TOnShortMidiMessage read FOnSendShortMidi   write FOnSendShortMidi;
    property    OnSendOSCString              : TOnOSCString        read FOnSendOscString   write FOnSendOscString;
    property    OnSendMidiBytes              : TOnMidiBytes        read FOnSendMidiBytes   write FOnSendMidiBytes;
  private
*)

    function    TSynthPatch.AssertUniqueName( const aName: string): string; // override;
    var
      aModule : TMod;
    begin
      // calling inherited here would go recursive
      Result := aName;

      for aModule in FModules
      do begin
        if   SameText( aModule.Name, aName)
        then raise EModule.CreateFmt( 'module name "%s" is not unique.', [ aName]);
      end;
    end;


//  private

    function    TSynthPatch.CheckModuleIndex( anIndex: Integer): Boolean;
    begin
      Result := ( anIndex >= 0) and ( anIndex < ModuleCount);
    end;


    function    TSynthPatch.GetModuleCount: Integer;
    begin
      Result := Length( FModules);
    end;


    function    TSynthPatch.GetModule( anIndex: Integer): TMod;
    begin
      Result := nil;

      if   CheckModuleIndex( anIndex)
      then Result := FModules[ anIndex];
    end;


    function    TSynthPatch.GetConnectionCount: Integer;
    begin
      Result := 0;

      if   Assigned( FConnections)
      then Result := FConnections.Count;
    end;


    procedure   TSynthPatch.SetResetFlag( aValue: Boolean); // override;
    begin
      if   aValue
      then FResetFlag := Control_Dec_2_p1    // Give control rate modules a chance to see the reset request
      else DecToZero( FResetFlag);
    end;


    function    TSynthPatch.FixPrefix( const aPrefix: string): string;
    begin
      if   aPrefix = ''
      then Result := Name
      else Result := aPrefix + '/' + Name;
    end;


    function    TSynthPatch.GetMorph( anIndex: Integer): TSignal;
    begin
      Result := FMorphs[ anIndex];
    end;


    procedure   TSynthPatch.SetMorph( anIndex: Integer; aValue: TSignal);
    begin
      FMorphs[ anIndex] := aValue;
    end;


    function    TSynthPatch.GetAutoFlag( anIndex: TAutoFlag): Boolean;
    begin
      Result := anIndex in FAutoFlags;
    end;


    procedure   TSynthPatch.SetAutoFlag( anIndex: TAutoFlag; aValue: Boolean);
    begin
      if   aValue
      then FAutoFlags := FAutoFlags + [ anIndex]
      else FAutoFlags := FAutoFlags - [ anIndex];
    end;


//  protected

    function    TSynthPatch.HandleParam( const aPath: string; aValue: TSignal): Boolean; // override;
    var
      aModule : TMod;
    begin
      Result := False;

      for aModule in FModules
      do begin
        Result := aModule.AcceptParam( aPath, aValue);

        if   Result
        then Break;
      end;
    end;


    function  TSynthPatch.HandleAutomationParam( const aPath: string; aValue: TSignal): Boolean; // override;
    var
      aModule : TMod;
    begin
      Result := False;

      for aModule in FModules
      do begin
        Result := aModule.AcceptAutomationParam( aPath, aValue);

        if   Result
        then Break;
      end;
    end;


//  public

    constructor TSynthPatch.Create( aParent: TSynthPatch; const aName: string); // override;
    begin
      inherited;
      FIsFast        := True;
      FIsSlow        := False;
      FIsMidi        := True;
      FIsOSC         := True;
      FVersion       := 0;
      FConnections   := TConnections.Create( Self);
      SetLength( FInternalMidiMsg.Data, 2);
      SetLength( FMorphs, 128);
    end;


    procedure   TSynthPatch.CreateIO; // override;
    var
      i : Integer;
    begin
      for i := 0 to IN_CHANNEL_COUNT - 1
      do  AddInput( i, Format( 'in%d', [ i + 1], AppLocale));

      for i := 0 to OUT_CHANNEL_COUNT - 1
      do  AddOutput( i, Format( 'in%d', [ i + 1], AppLocale));
    end;


    destructor  TSynthPatch.Destroy; // override;
    begin
      Locked := True;
      FreeAndNil( FConnections);  // Do this first for a faster clear
      Clear;                      // Clears the modules - that is it removes and frees them
      inherited;
    end;


    procedure   TSynthPatch.CompileLinks; // override;
    var
      aModule : TMod;
      i       : Integer;
    begin
      Locked := True;

      try
        FOutputLinks.ClearLinks;          // Clear any existing links, then build the new ones

        for aModule in FModules
        do begin
          if   aModule.FIsInput           // Manually handle input modules, output modules handled in CompileLinks
          then begin
            // For input modules couple their FInputs to all input modules' FOutputss
            // such that they later on can simply be propagated (and will get a
            // (useless) DoTick) in Self.DoTick.

            for i := 0 to InputCount - 1
            do  FOutputLinks.AddLink( @ FInputs[ i], @ aModule.FOutputs[ i]);

            aModule.CompileLinks;
          end
          else aModule.CompileLinks;
        end;
      finally
        Locked := False;
      end;
    end;


    procedure   TSynthPatch.SetDefaults; // override;
    var
      aModule : TMod;
    begin
      for aModule in FModules
      do  aModule.SetDefaults;
    end;


//  public

    procedure   TSynthPatch.DumpConnections( anIndent: Integer; var S: string);
    begin
      if   Assigned( FConnections)
      then FConnections.Dump( anIndent, 'connections', S);
    end;


    procedure   TSynthPatch.DumpModules( anIndent: Integer; var S: string);
    var
      aModule : TMod;
    begin
      S := S + Indent( anIndent) + 'modules(' + sLineBreak;

      for aModule in FModules
      do  aModule.Dump( anIndent + 1, S);

      S := S + Indent( anIndent) + ')' + sLineBreak;
    end;


    procedure   TSynthPatch.DumpDetails( anIndent: Integer; var S: string); // override;
    begin
      inherited;
      S := S + Indent( anIndent) + 'version ' + IntToStr( Version) + sLineBreak;
      DumpModules    ( anIndent, S);
      DumpConnections( anIndent, S);
    end;


//  public

    procedure   TSynthPatch.Clear;   // Clear moduleas and connections
    begin
      ClearModules;
      inherited Clear;
    end;


    procedure   TSynthPatch.Reset; // override;
    begin
      inherited;
      FLockSkips := 0;
    end;


    procedure   TSynthPatch.ResetSampleCounts; // override;
    var
      aModule : TMod;
    begin
      for aModule in FModules
      do  aModule.ResetSampleCounts;

      inherited;
    end;


    procedure   TSynthPatch.ClearModules;
    var
      aModule : TMod;
    begin
      while ModuleCount > 0
      do begin
        aModule :=  FModules[ ModuleCount - 1];
        SetLength(  FModules, ModuleCount - 1);
        aModule.DisposeOf;
      end;
    end;


    procedure   TSynthPatch.Panic; // override;
    var
      aModule : TMod;
    begin
      for aModule in FModules
      do  aModule.Panic;
    end;


    function    TSynthPatch.FindModule( const aModuleName: string): Integer;
    var
      i : Integer;
    begin
      Result := NO_ID;

      for i := 0 to ModuleCount - 1
      do begin
        if   SameText( aModuleName, Module[ i].Name)
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TSynthPatch.ModuleName( anIndex: Integer): string;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < ModuleCount)
      then Result := Module[ anIndex].Name
      else Result := NO_MODULE;
    end;


    function    TSynthPatch.FindInputPin( const aModuleIndex: Integer; const aPinName: string): Integer;
    begin
      Result := NO_ID;

      if   aModuleIndex >= 0
      then Result := Module[ aModuleIndex].FindInput( aPinName);
    end;


    function    TSynthPatch.FindOutputPin( const aModuleIndex: Integer; const aPinName: string): Integer;
    begin
      Result := NO_ID;

      if   aModuleIndex >= 0
      then Result := Module[ aModuleIndex].FindOutput( aPinName);
    end;


    function    TSynthPatch.InputPinName( aModuleIndex, aPinIndex: Integer): string;
    begin
      Result := NO_INPUT;

      if   ( aModuleIndex >= 0)
      and  ( aModuleIndex < ModuleCount)
      then Result := Module[ aModuleIndex].FindInputName( aPinIndex);
    end;


    function    TSynthPatch.OutputPinName( aModuleIndex, aPinIndex: Integer): string;
    begin
      Result := NO_OUTPUT;

      if   ( aModuleIndex >= 0)
      and  ( aModuleIndex < ModuleCount)
      then Result := Module[ aModuleIndex].FindOutputName( aPinIndex);
    end;


    procedure   TSynthPatch.AddModule( const aModule: TMod); // overload;
    begin
      SetLength( FModules, ModuleCount + 1);
      FModules[ ModuleCount - 1] := aModule;
    end;


    procedure   TSynthPatch.AddConnection( aSrcModule, aSrcPin, aDstModule, aDstPin: Integer); // overload;
    begin
      if   Assigned( FConnections)
      then FConnections.AddConnection( aSrcModule, aSrcPin, aDstModule, aDstPin);
    end;


    procedure   TSynthPatch.AddConnection( const aSrcModule, aSrcPin, aDstModule, aDstPin: string); // overload;
    var
      aSrcM: Integer;
      aSrcP: Integer;
      aDstM: Integer;
      aDstP: Integer;
    begin
      aSrcM := FindModule( aSrcModule);

      if   aSrcM >= 0
      then begin
        aSrcP := FindOutputPin( aSrcM, aSrcPin);

        if   aSrcP >= 0
        then begin
          aDstM := FindModule( aDstModule);

          if   aDstM >= 0
          then begin
            aDstP := FindInputPin( aDstM, aDstPin);

            if   aDstP >= 0
            then AddConnection( aSrcM, aSrcP, aDstM, aDstP)
            else raise EInputNotFound.CreateFmt( 'input "%s" not found on destination module "%s"', [ aDstPin, aDstModule]);
          end
          else raise EModuleNotFound.CreateFmt( 'destination module "%s" not found', [ aDstModule]);
        end
        else raise EOutputNotFound.CreateFmt( 'output "%s" not found on source module "%s"', [ aSrcPin, aSrcModule]);
      end;
   // else raise EModuleNotFound.CreateFmt( 'source module "%s" not found', [ aSrcModule]);
    end;


    procedure   TSynthPatch.DoTick; // Override;
    var
      m             : Integer;
      i             : Integer;
      CurrDecimator : Integer;
      aModCount     : Integer;
      aMod          : TMod;
      OldEM         : TArithmeticExceptionMask;
    begin
      try
        for i := 0 to OutputCount - 1                                     // Initialze outputs to zero
        do  FOutputs[ i] := 0.0;

        CurrDecimator   := 0;                                             // Initialize decimator
        FDecimatorCount := ( FDecimatorCount + 1) and Control_Dec_Min_One;
        aModCount       := ModuleCount;                                   // Module count into a local

        {$IFDEF DEBUG_DENORMS}
        OldEM := SetExceptionMask( GetExceptionMask - [ exDenormalized]); // For debugging trap denormalized exceptions in addition to the usual ones
        {$ELSE}
        OldEM := SetExceptionMask( exAllArithmeticExceptions);            // For non-debug code disable all floating point exceptions
        {$ENDIF}

        try
          FExecuting := True;                                             // Mark the patch as a whole to be executing

          try
            for m := 0 to aModCount - 1                                   // Tick the modules in the patch
            do begin
              {$IFDEF DEBUG} aMod := Module[ m]; {$ELSE} aMod := FModules[ m]; {$ENDIF}

              if   FResetFlag > 0                                         // Propagate reset requests to all modules
              then aMod.FResetFlag := 1;                                  // Modules must clear their own reset flag

              aMod.FExecuting := True;                                    // Start with flagging being in execution, later locking operations will
                                                                          // wait then till FExecuting is false again
              try
                if   not aMod.Locked                                      // Do not execute locked modules, this may get set even after FEcecuting got
                then begin                                                // True, but as long as FExecuting is True a lock can not be obtained and
                                                                          // the locking code will spin till FExecuting goes False (spin lock).
                                                                          // So effectively, Locked modules will skip a sample
                  if   CurrDecimator = FDecimatorCount                    // Process slow modules, or their slow parts, at control rate
                  then begin
                    if   aMod.FIsSlow                                     // Exclude sped up RateSmart modules
                    and  not aMod.FIsSpedUp
                    then begin
                      // Skip a call level and sample count inc when not profiled
                      {$IFDEF PATCH_PROFILER} aMod.SlowTick; {$ELSE} aMod.DoSlowTick; {$ENDIF}
                      aMod.PropagateSignals;
                    end;

                    aMod.DeZipper;                                        // Dezipper all modules at control rate
                  end;

                  if   aMod.FIsFast                                       // Normal fast mode, uses (Do)Tick
                  then begin
                    // Skip a call level and sample count inc when not profiled
                    {$IFDEF PATCH_PROFILER} aMod.Tick; {$ELSE} aMod.DoTick; {$ENDIF}
                    aMod.PropagateSignals;
                  end
                  else if aMod.FIsSpedUp                                  // Sped up slow mode for RateSmart modules, uses (Do)SlowTick
                  then begin
                    // Skip a call level and sample count inc when not profiled
                    {$IFDEF PATCH_PROFILER} aMod.SlowTick; {$ELSE} aMod.DoSlowTick; {$ENDIF}
                    aMod.PropagateSignals;
                  end
                  else if aMod.FIsOutput                                  // When module is an output module add its inputs to our local OutCh values
                  then begin
                    for i := 0 to OutputCount - 1
                    do  FOutputs[ i] := FOutputs[ i] + Normalize( aMod.FInputs[ i]);
                    // No propagation for output modules,
                    // in fact, the above is a 'partial normalized summing propagation'.
                  end;

                end
                else {$Q-$R-} Inc( FLockSkips) {$Q+R+} ;
              finally
                aMod.FExecuting := False;                                 // Allow module locking operations to preceed
              end;

              CurrDecimator := ( CurrDecimator + 1) and Control_Dec_Min_One;  // Next decimator value
            end;

            PropagateSignals;                                             // Propagate own signals FInputs -> TModInput modules
            ResetFlag := False;                                           // All modules got to see the reset request, so clear it
          finally
            FExecuting := False;                                          // Flag patch execution to be done, modules can be locked again
          end;
        finally
          ClearFPUExceptions( False);                                     // Clear any FPU exceptions without (re)raising
          SetExceptionMask( OldEM);                                       // Restore FPU exception mask to original value
        end;
      except
        on E: Exception                                                   // Always kill all exceptions here, calling into
        do KilledException( E);                                           // the killed exception handler for debugging
                                                                          // (the handler increments a counter).
      end;
    end;


    procedure   TSynthPatch.FixDeZippers; // override;
    // Unconditionally make all dezipped values jump to the associated input values.
    var
      i  : Integer;
    begin
      inherited;

      for i := 0 to ModuleCount - 1
      do  Module[ i].FixDeZippers;
    end;


    procedure   TSynthPatch.GatherSignals( const aPrefix: string; const aCallback: TSignalHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherSignals( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.GatherLights( const aPrefix: string; const aCallback: TLightsHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherLights( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.GatherData( const aPrefix: string; const aCallback: TDataHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherData( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.GatherXYData( const aPrefix: string; const aCallback: TXYDataHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherXYData( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.GatherStringData( const aPrefix: string; const aCallback: TStringDataHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherStringData( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.GatherCursorData( const aPrefix: string; const aCallback: TCursorDataHandler); // override;
    var
      aModule : TMod;
      S       : string;
    begin
      try
        S := FixPrefix( aPrefix);

        for aModule in FModules
        do  aModule.GatherCursorData( S, aCallback);
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TSynthPatch.TickWithData( aData: TSignalArray);
    var
      i : Integer;
    begin
      for i := Low( FInputs) to High( FInputs)
      do  FInputs[ i] := aData[ i];

      Tick;
    end;


    procedure   TSynthPatch.ClearProfile;
    {$IFDEF PATCH_PROFILER}
    var
      aModule : TMod;
    {$ENDIF}
    begin
    {$IFDEF PATCH_PROFILER}
      for aModule in FModules
      do  aModule.ClearProfile;
    {$ENDIF}
    end;


    function    TSynthPatch.CreateProfileHtml: TStringList;

  {$IFDEF PATCH_PROFILER}

    var
      aModule        : TMod;
      SumClocksFast  : Int64;
      SumClocksSlow  : Int64;
      SumSamplesFast : Int64;
      SumSamplesSlow : Int64;
      SumZips        : Int64;

      function Calc( Clocks, Samples: Int64): string;
      begin
        if   Samples = 0
        then Result := ''
        else Result := Format( '%.1f', [ Clocks / Samples], AppLocale);
      end;

  {$ENDIF}

    begin
      Result := TStringList.Create;

  {$IFDEF PATCH_PROFILER}

      SumClocksFast  := 0;
      SumClocksSlow  := 0;
      SumSamplesFast := 0;
      SumSamplesSlow := 0;
      SumZips        := 0;

      Result.Add( '<!doctype html>');
      Result.Add( '<html>');
      Result.Add( '<head><style type="text/css">');
      Result.Add( 'table, td, th { border-style: solid; border-width: 1px; padding: 5px;}');
      Result.Add( '</style></head>');
      Result.Add( '<body><table><tbody>');
      Result.Add( '<tr><td>name</td><td>type</td><td>fast clocks / sample</td><td>slow clocks / sample (/decimation)</td></tr>');

      for aModule in FModules
      do begin
        Result.Add( aModule.GetProfileInfoHtml);

        if   aModule.IsSpedUp
        then begin
          Inc( SumClocksFast , aModule.TotalClocksSlow);
          Inc( SumSamplesFast, aModule.SamplesSlow    );
        end
        else begin
          Inc( SumClocksFast , aModule.TotalClocksFast);
          Inc( SumClocksSlow , aModule.TotalClocksSlow);
          Inc( SumSamplesFast, aModule.SamplesFast    );
          Inc( SumSamplesSlow, aModule.SamplesSlow    );
        end;

        Inc( SumZips, aModule.TotalZips);
      end;

      Result.Add( GetProfileInfoHtml);
      Result.Add(
        Format(
          '<tr><td>%s</td><td>%s</td><td align="right">%s</td><td align="right">%s</td></tr>',
          [
            'patch',
            'module average',
            Calc( SumClocksFast, SumSamplesFast),
            Calc( SumClocksSlow, SumSamplesSlow * Control_Decimation)
          ],
          AppLocale
        )
      );
      Result.Add(
        Format(
          '<tr><td>%s</td><td>%s</td><td align="right">%s</td><td></td></tr>',
          [
            'dezippers',
            'total',
            Calc( SumZips, SamplesFast)
          ],
          AppLocale
        )
      );

      Result.Add(
        Format(
          '<tr><td>%s</td><td>%s</td><td align="right">%s</td><td></td></tr>',
          [
            'dezippers',
            'per module',
            Calc( SumZips, SamplesFast * ModuleCount)
          ],
          AppLocale
        )
      );

      Result.Add( '<tr><td>name</td><td>type</td><td>fast clocks / sample</td><td>slow clocks / sample</td></tr>');
      Result.Add( '</tbody></table>* : this is a sped-up rate smart module</body></html>');
      ClearProfile;

  {$ENDIF}

    end;


    function    TSynthPatch.IsSamePatch( const aPatch: TSynthPatch): Boolean;
    begin
      Result := False;

      if   Assigned( aPatch)
      then begin
        Result :=
          SameModules    ( aPatch) and
          SameConnections( aPatch)
      end;
    end;


    function    TSynthPatch.SameModules( const aPatch: TSynthPatch): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if   Assigned( aPatch)
      then begin
        if   ModuleCount = aPatch.ModuleCount
        then begin
          Result := True;

          for i := 0 to ModuleCount - 1
          do begin
            if   not Module[ i].IsSameModule( aPatch.Module[ i])
            then begin
              Result := False;
              Break;
            end;
          end;
        end;
      end;
    end;


    function    TSynthPatch.SameConnections( const aPatch: TSynthPatch): Boolean;
    begin
      Result := False;

      if   Assigned( aPatch)
      then Result := FConnections.IsSameConnections( aPatch.FConnections);
    end;


    procedure   TSynthPatch.AcceptMidi( const aMsg: TMidiMessage); // overide;
    var
      aModule : TMod;
    begin
      if   FIsMidi
      then begin
        for aModule in FModules
        do begin
          if   aModule.FIsMidi
          then aModule.AcceptMidi( aMsg)
        end;
      end;
    end;


    procedure   TSynthPatch.AcceptShortMidi( const aMsg: TShortMidiMessage); // override;
    var
      aModule : TMod;
    begin
      if FIsMidi
      then begin
        FInternalMidiMsg.Command  := aMsg.Command;
        FInternalMidiMsg.Channel  := aMsg.Channel;
        FInternalMidiMsg.Data[ 0] := aMsg.Data1;
        FInternalMidiMsg.Data[ 1] := aMsg.Data2;

        for aModule in FModules
        do begin
          if   aModule.FIsMidi
          then aModule.AcceptMidi( FInternalMidiMsg);
        end;
      end;
    end;


    procedure   TSynthPatch.AcceptPn( aCh: Byte; aController, aValue: Integer; IsRegistered: Boolean); // override;
    var
      aModule : TMod;
    begin
      if FIsMidi
      then begin
        for aModule in FModules
        do begin
          if   aModule.FIsMidi
          then aModule.AcceptPn( aCh, aController, aValue, IsRegistered)
        end;
      end;
    end;


    procedure   TSynthPatch.SendShortMidi( const aMsg: TShortMidiMessage); // override;
    var
      bMsg : TShortMidiMessage;
    begin
      if aMsg.Channel >= 16
      then begin
        bMsg.DataLen := aMsg.DataLen;
        bMsg.Command := aMSg.Command;
        bMsg.Channel := aMsg.Channel - 16;
        bMsg.Data1   := aMsg.Data1;
        bMsg.Data2   := aMsg.Data2;
        AcceptShortMidi( aMsg);
      end
      else begin
        if   Assigned( FOnSendShortMidi)
        then FOnSendShortMidi( Self, aMsg);
      end;
    end;


    procedure   TSynthPatch.AcceptOSC( const aSynthName: string; const aMsg: TOSCPacket); // override;
    var
      aModule : TMod;
    begin
      if   IsOSC
      and  Assigned( aMsg)
      then begin
        for aModule in FModules
        do begin
          if   aModule.IsOSC
          then begin
            aModule.AcceptOSC( aSynthName, aMsg);

            if   aMsg.Matched
            then Break;
          end;
        end;
      end;
    end;


    procedure   TSynthPatch.SendOSCBytes( const aMsg: TBytes); // override;
    begin
      if   Assigned( FOnSendOscString)
      then FOnSendOscString( Self, aMsg);
    end;


    procedure   TSynthPatch.SendMidiBytes( const aMsg: TBytes); // override;
    begin
      if   Assigned( FOnSendMidiBytes)
      then FOnSendMidiBytes( Self, aMsg);
    end;


    procedure   TSynthPatch.SampleRateChanged; // override;
    // Notification for sample rate change, the actual values are in global variables
    // System_Rate for the audio rate and Control_Rate for the control rate.
    // Modules caching sample rate dependent data should recalculate such data in an
    // overriden version of this procedure.
    var
      aModule : TMod;
    begin
      // Notify all sub-modules.
      for aModule in FModules
      do  aModule.SampleRateChanged;
    end;


    procedure   TSynthPatch.TuningChanged; // override;
    var
      aModule : TMod;
    begin
      // Notify all sub-modules.
      for aModule in FModules
      do  aModule.TuningChanged;
    end;


    procedure   TSynthPatch.CollectDenormals( var aMap: TPinMaps);
    var
      i : Integer;
    begin
      SetLength( aMap, ModuleCount);

      for i := 0 to ModuleCount - 1
      do begin
        Module[ i].CollectDenormals( aMap[ i]);
        aMap[ i].ModuleIndex := i;
      end;
    end;


end.
