unit module_scales;


{

   COPYRIGHT 2015 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.Classes, System.SysUtils,

  Globals, KnobsUtils;


type

  TScaleNote    = 0 .. 255;                      // Possible note values
  TScaleSize    = 0 .. 256;                      // The size of a scale
  TScaleNoteSet = set of TScaleNote;             // A set of notes defining a scale, implied order is lowest to highest
  TTetrad       = array[ 0 .. 3] of Integer;     // An array of four notes, where notes can be negative numbers, implied order is first note first



const


  (*                        variation, scale  , inversion                                        *)

  ScalesAndInversions : array[ 0 .. 2, 0 .. 19, 0 .. 3] of TTetrad =
  ((
      (* variation 0 - normal                                                                    *)
      (*               root                1st inversion       2nd inversion       3rd inversion *)

      (* maj     *) ((  0,  4,  7, 12),  (  4,  7, 12, 19),  (  7, 12, 19, 16),  ( 12, 19, 16, 24)), // Doubled triad
      (* min     *) ((  0,  3,  7, 12),  (  3,  7, 12, 19),  (  7, 12, 19, 15),  ( 12, 19, 15, 24)), // Doubled triad
      (* dim     *) ((  0,  3,  6, 12),  (  3,  6, 12, 18),  (  6, 12, 18, 15),  ( 12, 18, 15, 18)), // Doubled triad
      (* aug     *) ((  0,  4,  8, 12),  (  4,  8, 12, 20),  (  8, 12, 20, 16),  ( 12, 20, 16, 20)), // Doubled triad
      (* sus     *) ((  0,  5,  7, 12),  (  5,  7, 12, 17),  (  7, 12, 17, 19),  ( 12, 17, 19, 24)), // Doubled triad
      (* maj6    *) ((  0,  4,  7,  9),  (  4,  7,  9, 12),  (  7,  9, 12, 16),  (  9, 12, 16, 19)),
      (* min6    *) ((  0,  3,  7,  9),  (  3,  7,  9, 12),  (  7,  9, 12, 15),  (  9, 12, 15, 19)),
      (* maj7    *) ((  0,  4,  7, 11),  (  4,  7, 11, 12),  (  7, 11, 12, 16),  ( 11, 12, 16, 19)),
      (* maj9    *) ((  4,  7, 11, 14),  (  7, 11, 14, 16),  ( 11, 14, 16, 19),  ( 14, 16, 19, 25)),
      (* dom7    *) ((  0,  4,  7, 10),  (  4,  7, 10, 12),  (  7, 10, 12, 16),  ( 10, 12, 16, 19)),
      (* min9    *) ((  3,  7, 10, 14),  (  7, 10, 14, 15),  ( 10, 14, 15, 19),  ( 14, 15, 19, 22)),
      (* dim7    *) ((  0,  3,  6,  9),  (  3,  6,  9, 12),  (  6,  9, 12, 15),  (  9, 12, 15, 18)),
      (* m7b5    *) ((  0,  3,  6, 10),  (  3,  6, 10, 12),  (  6, 10, 12, 15),  ( 10, 12, 15, 18)),
      (* 7sus    *) ((  0,  5,  7, 10),  (  5,  7, 10, 12),  (  7, 10, 12, 17),  ( 10, 12, 17, 19)),
      (* maj9b7  *) ((  4,  7, 10, 14),  (  7, 10, 14, 16),  ( 10, 14, 16, 19),  ( 14, 16, 19, 22)),
      (* maj13_1 *) (( 10, 12, 16, 21),  ( 12, 16, 21, 22),  ( 16, 21, 22, 24),  ( 21, 22, 24, 28)),
      (* maj13_2 *) (( 10, 14, 17, 21),  ( 14, 17, 21, 22),  ( 17, 21, 22, 26),  ( 21, 22, 26, 29)),
      (* maj7b9  *) ((  4,  7, 10, 13),  (  7, 10, 13, 16),  ( 10, 13, 16, 19),  ( 13, 16, 19, 22)),
      (* maj13b9 *) (( 10, 13, 16, 21),  ( 13, 16, 21, 22),  ( 16, 21, 22, 25),  ( 21, 22, 25, 28)),
      (* maj9b5  *) ((  4,  6, 10, 14),  (  6, 10, 14, 16),  ( 10, 14, 16, 18),  ( 14, 16, 18, 22))

    ),(

      (* variation 1 - reduced                                                                   *)
      (*               root                1st inversion       2nd inversion       3rd inversion *)

      (* maj     *) ((  0,  4,  7, 12),  (  4,  7,  0, 16),  (  7,  0, 12,  4),  (  0,  7,  4, 12)), // Doubled triad
      (* min     *) ((  0,  3,  7, 12),  (  3,  7,  0, 15),  (  7,  0, 12,  3),  (  0,  7,  3, 12)), // Doubled triad
      (* dim     *) ((  0,  3,  6, 12),  (  3,  6,  0, 15),  (  6,  0, 12,  3),  (  0,  6,  3, 15)), // Doubled triad
      (* aug     *) ((  0,  4,  8, 12),  (  4,  8,  0, 16),  (  8,  0, 12,  4),  (  0,  8,  4, 16)), // Doubled triad
      (* sus     *) ((  0,  5,  7, 12),  (  5,  7,  0, 17),  (  7,  0,  5, 12),  (  0,  5,  7, 12)), // Doubled triad
      (* maj6    *) ((  0,  4,  7,  9),  (  4,  7,  9,  0),  (  7,  9,  0,  4),  (  9,  0,  4,  7)),
      (* min6    *) ((  0,  3,  7,  9),  (  3,  7,  9,  0),  (  7,  9,  0,  3),  (  9,  0,  3,  7)),
      (* maj7    *) ((  0,  4,  7, 11),  (  4,  7, 11,  0),  (  7, 11,  0,  4),  ( 11,  0,  4,  7)),
      (* maj9    *) ((  4,  7, 11,  2),  (  7, 11,  2,  4),  ( 11,  2,  4,  7),  (  2,  4,  7,  1)),
      (* dom7    *) ((  0,  4,  7, 10),  (  4,  7, 10,  0),  (  7, 10,  0,  4),  ( 10,  0,  4,  7)),
      (* min9    *) ((  3,  7, 10,  2),  (  7, 10,  2,  3),  ( 10,  2,  3,  7),  (  2,  3,  7, 10)),
      (* dim7    *) ((  0,  3,  6,  9),  (  3,  6,  9,  0),  (  6,  9,  0,  3),  (  9,  0,  3,  6)),
      (* m7b5    *) ((  0,  3,  6, 10),  (  3,  6, 10,  0),  (  6, 10,  0,  3),  ( 10,  0,  3,  6)),
      (* 7sus    *) ((  0,  5,  7, 10),  (  5,  7, 10,  0),  (  7, 10,  0,  5),  ( 10,  0,  5,  7)),
      (* maj9b7  *) ((  4,  7, 10,  2),  (  7, 10,  2,  4),  ( 10,  2,  4,  7),  (  2,  4,  7, 10)),
      (* maj13_1 *) (( 10,  0,  2,  9),  (  0,  4,  9, 10),  (  4,  9, 10,  0),  (  9, 10,  0,  4)),
      (* maj13_2 *) (( 10,  2,  5,  9),  (  2,  5,  9, 10),  (  5,  9, 10,  2),  (  9, 10,  2,  5)),
      (* maj7b9  *) ((  4,  7, 10,  1),  (  7, 10,  1,  4),  ( 10,  1,  4,  7),  (  1,  4,  7, 10)),
      (* maj13b9 *) (( 10,  1,  4,  9),  (  1,  4,  9, 10),  (  4,  9, 10,  1),  (  9, 10,  1,  4)),
      (* maj9b5  *) ((  4,  6, 10,  2),  (  6, 10,  2,  4),  ( 10,  2,  4,  6),  (  2,  4,  6, 10))

    ),(

      (* variation 2 - lowered                                                                   *)
      (*               root                1st inversion       2nd inversion       3rd inversion *)

      (* maj     *) ((  0,  4,  7, 12),  (  4,  7,  0, -5),  (  7,  0, -5,  4),  (  0, -5,  4, 12)), // Doubled triad
      (* min     *) ((  0,  3,  7, 12),  (  3,  7,  0, -5),  (  7,  0, -5,  3),  (  0, -5,  3, 12)), // Doubled triad
      (* dim     *) ((  0,  3,  6, 12),  (  3,  6,  0, -6),  (  6,  0, -6,  3),  (  0, -6,  3,  6)), // Doubled triad
      (* aug     *) ((  0,  4,  8, 12),  (  4,  8,  0, -4),  (  8,  0, -4,  4),  (  0, -4,  4,  8)), // Doubled triad
      (* sus     *) ((  0,  5,  7, 12),  (  5,  7,  0, -7),  (  7,  0, -7, -5),  (  0, -7,  7, 12)), // Doubled triad
      (* maj6    *) ((  0,  4,  7,  9),  (  4,  7,  9,  0),  (  7,  9,  0,  4),  (  9,  0,  4,  7)),
      (* min6    *) ((  0,  3,  7,  9),  (  3,  7,  9,  0),  (  7,  9,  0,  3),  (  9,  0,  3,  7)),
      (* maj7    *) ((  0,  4,  7, 11),  (  4,  7, 11,  0),  (  7, 11,  0,  4),  ( 11,  0,  4,  7)),
      (* maj9    *) ((  4,  7, 11,  2),  (  7, 11,  2,  4),  ( 11,  2,  4,  7),  (  2,  4,  7,  1)),
      (* dom7    *) ((  0,  4,  7, 10),  (  4,  7, 10,  0),  (  7, 10,  0,  4),  ( 10,  0,  4,  7)),
      (* min9    *) ((  3,  7, 10,  2),  (  7, 10,  2,  3),  ( 10,  2,  3,  7),  (  2,  3,  7, 10)),
      (* dim7    *) ((  0,  3,  6,  9),  (  3,  6,  9,  0),  (  6,  9,  0,  3),  (  9,  0,  3,  6)),
      (* m7b5    *) ((  0,  3,  6, 10),  (  3,  6, 10,  0),  (  6, 10,  0,  3),  ( 10,  0,  3,  6)),
      (* 7sus    *) ((  0,  5,  7, 10),  (  5,  7, 10,  0),  (  7, 10, 12,  5),  ( 10, 12,  5,  7)),
      (* maj9b7  *) ((  4,  7, 10,  2),  (  7, 10,  2,  4),  ( 10,  2,  4,  7),  (  2,  4,  7, 10)),
      (* maj13_1 *) (( 10,  0,  4,  9),  (  0,  4,  9, 10),  (  4,  9, 10,  0),  (  9, 10,  0,  4)),
      (* maj13_2 *) (( 10,  2,  5,  9),  (  2,  5,  9, 10),  (  5,  9, 10,  2),  (  9, 10,  2,  5)),
      (* maj7b9  *) ((  4,  7, 10,  1),  (  7, 10,  1,  4),  ( 10,  1,  4,  7),  (  1,  4,  7, 10)),
      (* maj13b9 *) (( 10,  1,  4,  9),  (  1,  4,  9, 10),  (  4,  9, 10,  1),  (  9, 10,  1,  4)),
      (* maj9b5  *) ((  4,  6, 10,  2),  (  6, 10,  2,  4),  ( 10,  2,  4,  6),  (  2,  4,  6, 10))
  ));


type

  TScaleNoteNameMap = record
  public
    Src   : string;                           // Raw source format
    Name  : string;                           // The scale name
    Notes : TScaleNoteSet;                    // The notes in the scale
  end;

  TScaleNoteNameMaps = array of TScaleNoteNameMap;

  TScaleData = class
  private
    FScaleSize : TScaleSize;
    FData      : TScaleNoteNameMaps;
    FCountRec  : TSignal;
  private
    function    GetCount: Integer;
    function    GetSource         (       anIndex: Integer): string;
    function    GetScaleName      (       anIndex: Integer): string;
    function    GetScaleNotes     (       anIndex: Integer): TScaleNoteSet;
    function    GetScaleDataByName( const anIndex: string ): TScaleNoteSet;
  private
    procedure   AddScaleData( const aData: TScaleNoteNameMap);                                                 overload;
    procedure   AddScaleData( const aSrc: string; const aName: string; const aData: TScaleNoteSet);            overload;
    procedure   AddScaleData( const aSrc: string; const aName, aData: string);                                 overload;
    procedure   AddScaleData( const aLine: string);                                                            overload;
    procedure   Clear;
  public
    constructor Create( aScaleSize: TScaleSize);                                                // Create with a defined 'octave' size
    procedure   LoadFromFile     ( const aFileName: string  );                                  // Parse scale data drom a file
    procedure   LoadFromStrings  ( const aStrings : TStrings);                                  // Parse scale data from a TStrings
    procedure   SaveToFile       ( const aFileName: string  );                                  // Save scale to  file
    procedure   SaveToStrings    ( const aStrings : TStrings);                                  // Save scale to a TStrings
    function    FindScale        ( const aName: string): Integer;                               // Find a scale by name, -1 if not found
    procedure   MakeDefaultScales;                                                              // Make a set of default scales, look here for syntax of scale files
    function    CreateScaleNames: TStringList;                                                  // Creates a list of names that can be used in a menu or as the captions for a selector
  public
    property    ScaleSize                              : TScaleSize    read FScaleSize;         // The number of notes in the 'octave'
    property    Count                                  : Integer       read GetCount;           // # scales present
    property    CountRec                               : TSignal       read FCountRec;          // reciprocal of # scales present
    property    Source         [ anIndex: Integer]     : string        read GetSource;          // Raw source code for a scale
    property    ScaleName      [ anIndex: Integer]     : string        read GetScaleName;       // Name for a scale
    property    ScaleNotes     [ anIndex: Integer]     : TScaleNoteSet read GetScaleNotes;      // The notes for a scale
    property    ScaleDataByName[ const anIndex: string]: TScaleNoteSet read GetScaleDataByName; // The notes for a scale found by name, empty set if not found
  end;


  function  ScaleToStr( aScale: TScaleNoteSet): string;
  function  CompleteScale( aScale: TScaleNoteSet; aLowNote, aHighNote: TScaleNote; aScaleSize: TScaleSize): TScaleNoteSet;
  function  TransposeScale( aScale: TScaleNoteSet; anAmount: Integer; aScaleSize: TScaleSize): TScaleNoteSet;
  function  TransPoseAndCompleteScale( aScale: TScaleNoteSet; aLowNote, aHighNote: TScaleNote; aTranspose: Integer; aScaleSize: TScaleSize): TScaleNoteSet;


var

  SystemScales : TScaleData; // The system scale, read from <applicationname>.scale if present, a default scale otherwise.



implementation


const

  PossibleNotes = [ Low( TScaleNote) .. High( TScaleNote)];


  function  ScaleToStr( aScale: TScaleNoteSet): string;
  var
    n : TScaleNote;
  begin
    Result := '';

    for n in aScale
    do begin
      if Result = ''
      then Result := IntToStr( n)
      else Result := Result + ' ' + IntToStr( n);
    end;

    if Result = ''
    then Result := '[]'
    else Result := '[ ' + Result + ']';
  end;


  function  CompleteScale( aScale: TScaleNoteSet; aLowNote, aHighNote: TScaleNote; aScaleSize: TScaleSize): TScaleNoteSet;
  var
    i : Integer;
    j : Integer;
  begin
    Result := [];
    for i := 0 to aScaleSize - 1
    do begin
      if i in aScale
      then begin
        j := i;

        while j <= High( TScaleNote)
        do begin
          if ( j >= aLowNote) and ( j <= aHighNote)
          then Result := Result + [ j];

          Inc( j, aScaleSize);
        end;
      end;
    end;
  end;


  function  TransposeScale( aScale: TScaleNoteSet; anAmount: Integer; aScaleSize: TScaleSize): TScaleNoteSet;
  var
    i : Integer;
  begin
    Result := [];

    for i := 0 to aScaleSize - 1
    do begin
      if i in aScale
      then Result := Result + [ ( i + aScaleSize + anAmount) mod aScaleSize];
    end;
  end;


  function  TransPoseAndCompleteScale( aScale: TScaleNoteSet; aLowNote, aHighNote: TScaleNote; aTranspose: Integer; aScaleSize: TScaleSize): TScaleNoteSet;
  begin
    Result := TransposeScale( aScale, aTranspose, aScaleSize);
    Result := CompleteScale( Result, aLowNote, aHighNote, aScaleSize);
  end;



{ ========
  TScaleData = class
  private
    FScaleSize : TScaleSize;
    FData      : TScaleNoteNameMaps;
  public
    property    ScaleSize                              : TScaleSize    read FScaleSize;
    property    Count                                  : Integer       read GetCount;
    property    Source         [ anIndex: Integer]     : string        read GetSource;
    property    ScaleName      [ anIndex: Integer]     : string        read GetScaleName;
    property    ScaleNotes     [ anIndex: Integer]     : TScaleNoteSet read GetScaleNotes;
    property    ScaleDataByName[ const anIndex: string]: TScaleNoteSet read GetScaleDataByName;
  private
}


    function    TScaleData.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


    function    TScaleData.GetSource( anIndex: Integer): string;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Count)
      then Result := FData[ anIndex].Src
      else Result := '';
    end;


    function    TScaleData.GetScaleName( anIndex: Integer): string;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Count)
      then Result := FData[ anIndex].Name
      else Result := '';
    end;


    function    TScaleData.GetScaleNotes( anIndex: Integer): TScaleNoteSet;
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Count)
      then Result := FData[ anIndex].Notes
      else Result := [];
    end;


    function    TScaleData.GetScaleDataByName( const anIndex: string): TScaleNoteSet;
    var
      p : Integer;
    begin
      p := FindScale( anIndex);

      if p >= 0
      then Result := ScaleNotes[ p]
      else Result := [];
    end;


//  private

    procedure   TScaleData.AddScaleData( const aData: TScaleNoteNameMap); // overload;
    begin
      SetLength( FData, Count + 1);
      FData[ Count - 1] := aData;
      FCountRec         := 1 / Count;
    end;


    procedure   TScaleData.AddScaleData( const aSrc: string; const aName: string; const aData: TScaleNoteSet); // overload;
    var
      aRec : TScaleNoteNameMap;
    begin
      aRec.Src   := aSrc;
      aRec.Name  := aName;
      aRec.Notes := aData * [ 0 .. FScaleSize - 1];
      AddScaleData( aRec);
    end;


    procedure   TScaleData.AddScaleData( const aSrc: string; const aName, aData: string); // overload;
    var
      aParts : TStringList;
      S      : string;
      aSet   : TScaleNoteSet;
      aVal   : Integer;
      i      : Integer;
    begin
      S      := Trim( StripQuotes( Trim( aData), '[', ']'));
      aSet   := [];
      aParts := Explode( S, ' ');

      try
        for i := 0 to aParts.Count - 1
        do begin
          aVal := StrToIntDef( Trim( aParts[ i]), -1);

          if aVal in PossibleNotes
          then aSet := aSet + [ aVal];
        end;
      finally
        aParts.DisposeOf;
      end;

      AddScaleData( aSrc, Trim( StripQuotes( Trim( aName), '"', '"')), aSet);
    end;


    procedure   TScaleData.AddScaleData( const aLine: string); // overload;
    var
      aParts : TStringList;
    begin
      aParts := Explode( Trim( aLine), ',');

      try
        if aParts.Count = 2
        then AddScaleData( aLine, aParts[ 0], aParts[ 1]);
      finally
        aParts.DisposeOf;
      end;
    end;


    procedure   TScaleData.Clear;
    begin
      SetLength( FData, 0);
    end;

//  public

    constructor TScaleData.Create( aScaleSize: TScaleSize);
    begin
      inherited Create;
      FScaleSize := aScaleSize;
      FCountRec  := 0;             // Yeah wrong ... should be infinity ...
    end;


    procedure   TScaleData.LoadFromFile( const aFileName: string);
    var
      aStringList: TStringList;
    begin
      aStringList := TStringList.Create;

      try
        aStringList.LoadFromFile( aFileName);
        LoadFromStrings( aStringList);
      finally
        aStringList.DisposeOf;
      end;
    end;


    procedure   TScaleData.LoadFromStrings( const aStrings: TStrings);
    var
      i : Integer;
    begin
      if Assigned( aStrings)
      then begin
        Clear;

        for i := 0 to aStrings.Count - 1
        do AddScaleData( aStrings[ i]);
      end;
    end;


    procedure   TScaleData.SaveToFile( const aFileName: string);
    var
      aStringList : TStringList;
    begin
      aStringList := TStringList.Create;

      try
        SaveToStrings( aStringList);
        aStringList.SaveToFile( aFileName);
      finally
        aStringList.DisposeOf;
      end;
    end;


    procedure   TScaleData.SaveToStrings( const aStrings: TStrings);
    var
      i : Integer;
    begin
      if Assigned( aStrings)
      then begin
        aStrings.Clear;

        for i := 0 to Count - 1
        do aStrings.Add( Source[ i])
      end;
    end;


    function    TScaleData.FindScale( const aName: string): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to Count - 1
      do begin
        if SameText( ScaleName[ i], aName)
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TScaleData.MakeDefaultScales;
    begin
      AddScaleData( '"Dyad minor second" , [ 0 1]'                       );  // Scale name and the set of notes in the base octave
      AddScaleData( '"Dyad second"       , [ 0 2]'                       );
      AddScaleData( '"Dyad minor third"  , [ 0 3]'                       );
      AddScaleData( '"Dyad major third"  , [ 0 4]'                       );
      AddScaleData( '"Dyad fourth"       , [ 0 5]'                       );
      AddScaleData( '"Dyad tritone"      , [ 0 6]'                       );
      AddScaleData( '"Dyad fifth"        , [ 0 7]'                       );
      AddScaleData( '"Dyad minor sixth"  , [ 0 8]'                       );
      AddScaleData( '"Dyad major sixth"  , [ 0 9]'                       );
      AddScaleData( '"Dyad minor seventh", [ 0 10]'                      );
      AddScaleData( '"Dyad major seventh", [ 0 11]'                      );

      AddScaleData( '"-"                 , []'                           );  // A separator item

      AddScaleData( '"Triad major"       , [ 0 4 7]'                     );
      AddScaleData( '"Triad minor"       , [ 0 3 7]'                     );
      AddScaleData( '"Triad diminished"  , [ 0 3 6]'                     );
      AddScaleData( '"Triad augmented"   , [ 0 4 8]'                     );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Blues major"       , [ 0 2 3 4 7 9]'               );
      AddScaleData( '"Harmonic major"    , [ 0 2 4 5 7 8 11]'            );
      AddScaleData( '"Major"             , [ 0 2 4 5 7 9 11]'            );
      AddScaleData( '"Major bebop"       , [ 0 2 4 5 7 8 9 11]'          );
      AddScaleData( '"Major Locrian"     , [ 0 2 4 5 6 8 10]'            );
      AddScaleData( '"Neapolitan major"  , [ 0 1 3 5 7 9 11]'            );
      AddScaleData( '"Pentatonic Major"  , [ 0 2 5 7 9]'                 );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Blues minor"       , [ 0 3 5 6 7 10]'              );
      AddScaleData( '"Harmonic minor 1"  , [ 0 2 3 5 7 8]'               );
      AddScaleData( '"Harmonic minor 2"  , [ 0 2 3 6 7 8 10]'            );
      AddScaleData( '"Melodic minor"     , [ 0 2 3 5 7 10]'              );
      AddScaleData( '"Minor"             , [ 0 2 3 5 7 8 10]'            );
      AddScaleData( '"Neapolitan minor"  , [ 0 1 3 5 7 8 11]'            );
      AddScaleData( '"Pentatonic minor"  , [ 0 3 5 7 10]'                );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Dorian"            , [ 0 2 3 5 7 9 10]'            );
      AddScaleData( '"Locrian"           , [ 0 1 3 5 6 8 10]'            );
      AddScaleData( '"Istrian"           , [ 0 1 3 4 6 7]'               );
      AddScaleData( '"Lydian"            , [ 0 2 4 6 7 9 11]'            );
      AddScaleData( '"Lydian augmented"  , [ 0 2 4 6 8 9 11]'            );
      AddScaleData( '"Mixolydian"        , [ 0 2 4 5 7 9 10]'            );
      AddScaleData( '"Phrygian"          , [ 0 1 3 5 7 8 10]'            );
      AddScaleData( '"Phrygian dominant" , [ 0 1 4 5 7 8 10]'            );
      AddScaleData( '"Ukrainian Dorian"  , [ 0 2 3 6 7 9 10]'            );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Gypsy"             , [ 0 1 4 5 6 9 10]'            );
      AddScaleData( '"Gypsy Hung. 1"     , [ 0 2 3 6 7 8 11]'            );
      AddScaleData( '"Gypsy Hung. 2"     , [ 0 2 3 6 7]'                 );
      AddScaleData( '"Gypsy Hung. aug"   , [ 0 2 3 6 8]'                 );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Hirajoshi"         , [ 0 2 3 7 8]'                 );
      AddScaleData( '"Insen"             , [ 0 1 5 7 10]'                );
      AddScaleData( '"Iwato"             , [ 0 1 5 6 10]'                );
      AddScaleData( '"Japanese"          , [ 0 4 5 9 11]'                );
      AddScaleData( '"Yo"                , [ 0 3 5 7 11]'                );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Acoustic"          , [ 0 2 4 6 7 9 10]'            );
      AddScaleData( '"Adonai malakh"     , [ 0 2 4 5 7 8 10]'            );
      AddScaleData( '"Algerian"          , [ 0 2 3 6 7 8]'               );
      AddScaleData( '"Altered"           , [ 0 1 3 4 6 8 10]'            );
      AddScaleData( '"Bebop dominant"    , [ 0 2 4 5 7 9 10 11]'         );
      AddScaleData( '"Enigmatic"         , [ 0 1 4 6 8 10 11]'           );
      AddScaleData( '"Flamenco"          , [ 0 1 4 5 7 8 11]'            );
      AddScaleData( '"Half diminished"   , [ 0 2 3 5 6 8 10]'            );
      AddScaleData( '"Augmented"         , [ 0 3 4 7 8 11]'              );
      AddScaleData( '"Persian"           , [ 0 1 4 5 6 8 11]'            );
      AddScaleData( '"Prometheus"        , [ 0 2 4 6 9 10]'              );

      AddScaleData( '"-"                 , []'                           );

      AddScaleData( '"Chromatic"         , [ 0 1 2 3 4 5 6 7 8 9 10 11]' );
      AddScaleData( '"Octatonic 1"       , [ 0 1 3 4 6 7 9 10]'          );
      AddScaleData( '"Octatonic 2"       , [ 0 2 3 5 6 8 9 10]'          );
      AddScaleData( '"Tritone"           , [ 0 1 4 6 7 10]'              );
      AddScaleData( '"Whole tone"        , [ 0 2 4 6 8 10]'              );
    end;


    function    TScaleData.CreateScaleNames: TStringList;
    var
      i : Integer;
    begin
      Result := TStringList.Create;
      Result.Duplicates := dupAccept;

      for i := 0 to Count - 1
      do begin
        if ScaleName[ i] = '-'
        then Result.Add( '-')
        else Result.Add( Format( '%s'^I'%s', [ ScaleName[ i], ScaleToStr( ScaleNotes[ i])], AppLocale));
      end;
    end;


initialization

  SystemScales  := TScaleData.Create( 12);

  if FileExists( ScaleFileName)
  then SystemScales.LoadFromFile( ScaleFileName)
  else begin
    SystemScales.MakeDefaultScales;
    SystemScales.SaveToFile( ScaleFileName);
  end;

finalization

  FreeAndNil( SystemScales);

end.

