unit noise;

interface

uses

  KnobsUtils, KnobsConversions;

const

  MAX_NOISE_LAYER_LEN = 4096;
  MAX_TONE_LAYER_LEN  =  256;



type

  TNoiseLayer  = array of TSignal;
  TNoiseLayers = array of TNoiseLayer;

  TPerlinNoise = class
  private
    FLayerCount  : Integer;
    FMaxLayerLen : Integer;
    FNoiseLayers : TNoiseLayers;
    FIndex       : Integer;
  private
    function    RandomValue: TSignal;                                                                            inline;
    function    Interpolate( aLayer: Integer; q: Integer): TSignal;
    procedure   CreateNoise;
    procedure   ChangeNoise( anIndex: Integer);
  public
    constructor Create( aLayerCount, aFMaxLayerLen: Integer);
    function    GetValue( anIndex: Integer; aRoughness: TSignal): TSignal;
    procedure   TickL ( var aValue: TSignal; aRoughness, aLayerLen         : TSignal; MakeNewValue: Boolean);
    procedure   TickP ( var aValue: TSignal; aRoughness, aPhase            : TSignal; MakeNewValue: Boolean);
    procedure   TickLP( var aValue: TSignal; aRoughness, aLayerLen, aPhase : TSignal; MakeNewValue: Boolean);
  end;



implementation


{ ========
  TPerlinNoise = class
  private
    FLayerCount  : Integer;
    FMaxLayerLen : Integer;
    FNoiseLayers : TNoiseLayers;
    FIndex       : Integer;
  private
}

    function    TPerlinNoise.RandomValue: TSignal; // inline;
    begin
      Result := 2 * Random - 1;
    end;


    function    TPerlinNoise.Interpolate( aLayer: Integer; q: Integer): TSignal;
    var
      p : Integer;
      r : Integer;
    begin
      p := ( q + FMaxLayerLen - 1) mod FMaxLayerLen;
      r := ( q                + 1) mod FMaxLayerLen;
      Result :=
        0.33 * FNoiseLayers[ aLayer, p] +
        0.41 * FNoiseLayers[ aLayer, q] +
        0.33 * FNoiseLayers[ aLayer, r];
    end;


    procedure   TPerlinNoise.CreateNoise;
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to FMaxLayerLen - 1
      do FNoiseLayers[ FLayerCount - 1, i] := RandomValue;

      for j := FLayerCount - 2 downto 0
      do begin
        for i := 0 to FMaxLayerLen - 1
        do FNoiseLayers[ j, i] := Interpolate( j + 1, i);
      end;
    end;


    procedure   TPerlinNoise.ChangeNoise( anIndex: Integer);
    var
      j : Integer;
    begin
      FNoiseLayers[ FLayerCount - 1, anIndex] := RandomValue;

      for j := FLayerCount - 2 downto 0
      do FNoiseLayers[ j, anIndex] := Interpolate( j + 1, anIndex);
    end;


//  public

    constructor TPerlinNoise.Create( aLayerCount, aFMaxLayerLen: Integer);
    var
      i : Integer;
    begin
      inherited Create;
      FLayerCount  := aLayerCount;
      FMaxLayerLen := aFMaxLayerLen;
      SetLength( FNoiseLayers, FLayerCount);

      for i := 0 to FLayerCount - 1
      do SetLength( FNoiseLayers[ i], FMaxLayerLen);

      CreateNoise;
    end;


    function    TPerlinNoise.GetValue( anIndex: Integer; aRoughness: TSignal): TSignal;
    var
      i        : Integer;
      LayerVal : TSignal;
    begin
      Result := FNoiseLayers[ 0, anIndex];                // Smoothest layer

      for i := 1 to FLayerCount - 1
      do begin                                            // Move up to roughest layer
        LayerVal := FNoiseLayers[ i, anIndex];
        Result   := aRoughness * ( LayerVal - Result) + Result;
      end;
    end;


    procedure   TPerlinNoise.TickL( var aValue: TSignal; aRoughness, aLayerLen: TSignal; MakeNewValue: Boolean); // overload;
    // L for length driven
    begin
      aValue := GetValue( FIndex, aRoughness);

      if MakeNewValue
      then ChangeNoise( FIndex);

      FIndex := ( FIndex + 1) mod Clip( Round( aLayerLen * FMaxLayerLen), 3, MAX_NOISE_LAYER_LEN);
    end;


    procedure   TPerlinNoise.TickP( var aValue: TSignal; aRoughness, aPhase: TSignal; MakeNewValue: Boolean); // overload;
    // P for phase driven, or interpolated
    var
      fp : TSignal;
      p  : Integer;
      q  : Integer;
      v1 : TSignal;
      v2 : TSignal;
    begin
      fp     := Clip( aPhase, 0.0, 1.0) * ( FMaxLayerLen - 1);
      p      := Trunc( fp);
      q      := ( p + 1) mod FMaxLayerLen;
      v1     := GetValue( p, aRoughness);
      v2     := GetValue( q, aRoughness);
      aValue := v1 - ( fp - p) * ( v1 + v2);

      if MakeNewValue
      then ChangeNoise( FIndex);

      FIndex := ( FIndex + 1) mod MAX_NOISE_LAYER_LEN;
    end;


    procedure   TPerlinNoise.TickLP( var aValue: TSignal; aRoughness, aLayerLen, aPhase: TSignal; MakeNewValue: Boolean);
    // L for length driven, P for Phase
    var
      fp     : TSignal;
      p      : Integer;
      q      : Integer;
      v1     : TSignal;
      v2     : TSignal;
      aLimit : Integer;
    begin
      aLimit := Clip( Round( aLayerLen * FMaxLayerLen), 3, MAX_NOISE_LAYER_LEN);
      fp     := Clip( aPhase, 0.0, 1.0) * ( aLimit - 1);
      p      := Trunc( fp);
      q      := ( p + 1) mod aLimit;
      v1     := GetValue( p, aRoughness);
      v2     := GetValue( q, aRoughness);
      aValue := v1 - ( fp - p) * ( v1 + v2);

      if MakeNewValue
      then ChangeNoise( FIndex);

      FIndex := ( FIndex + 1) mod aLimit;
    end;


end.

