unit ComplexMath;

{

   COPYRIGHT 2015 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.Sysutils, System.Math, KnobsUtils;


type

  EComplex    = class( Exception);
  ERangeError = class( EComplex );

  TSignalPair = record
  public
    Left  : TSignal;
    Right : TSignal;
  public
    property  X : TSignal read Left  write Left;
    property  Y : TSignal read Right write Right;
  end;


  TPolar = record
  private
    FNorm : TSignal;
    FPhi  : TSignal;
  public
    property  Norm : TSignal read FNorm write FNorm;
    property  Phi  : TSignal read FPhi  write FPhi;
  end;


  TComplex = record
  private
    FReal : TSignal;
    FImag : TSignal;
  private
    function  GetNorm: TSignal;
    procedure SetNorm( aValue : TSignal);
    function  GetPhi: TSignal;
    procedure SetPhi( aValue : TSignal);
  public
    class operator Implicit( a: TSignal): TComplex;                        // Implicit typecast
    class operator Implicit( a: TPolar): TComplex;                         // Implicit typecast
    class operator Implicit( a: TSignalPair): TComplex;                    // Implicit typecast
    class operator Explicit( a: TSignal): TComplex;                        // Explicit typecast
    class operator Explicit( a: TPolar): TComplex;                         // Implicit typecast
    class operator Explicit( a: TSignalPair): TComplex;                    // Implicit typecast
    class operator Negative( a: TComplex): TComplex;                       // unary minus
    class operator Positive( a: TComplex): TComplex;                       // unary plus
    class operator Equal( a, b: TComplex): Boolean;                        // =  operator
    class operator NotEqual( a, b: TComplex): Boolean;                     // <> operator
    class operator GreaterThan( a, b: TComplex): Boolean;                  // >  operator
    class operator GreaterThanOrEqual( a, b: TComplex): Boolean;           // >= operator
    class operator LessThan( a, b: TComplex): Boolean;                     // <  operator
    class operator LessThanOrEqual( a, b: TComplex): Boolean;              // >= operator
    class operator Add( a, b: TComplex): TComplex;                         // +  operator
    class operator Add( a: TSignal; b: TComplex): TComplex;                // +  operator
    class operator Add( a: TComplex; b: TSignal): TComplex;                // +  operator
    class operator Subtract( a, b: TComplex): TComplex;                    // -  operator
    class operator Subtract( a: TSignal; b: TComplex): TComplex;           // -  operator
    class operator Subtract( a: TComplex; b: TSignal): TComplex;           // -  operator
    class operator Multiply( a, b: TComplex): TComplex;                    // *  operator
    class operator Multiply( a: TSignal; b: TComplex): TComplex;           // *  operator
    class operator Multiply( a: TComplex; b: TSignal): TComplex;           // *  operator
    class operator Divide( a, b: TComplex): TComplex;                      // /  operator
    class operator Divide( a: TSignal; b: TComplex): TComplex;             // /  operator
    class operator Divide( a: TComplex; b: TSignal): TComplex;             // /  operator
  public
    procedure   FromPolar( aNorm, aPhi : TSignal);    overload;            // Set value from polar coordinates
    procedure   FromPolar( const aValue: TPolar);     overload;            // Set value from polar coordinates
    function    ToPolar: TPolar;
    procedure   Normalize;                                                 // Set Norm to 1.0
    procedure   Rotate( aPhi: TSignal);                                    // Rotate ove aPhi radians
  public
    property    Real : TSignal read FReal   write FReal;                   // Manipulate real part
    property    Imag : TSignal read FImag   write FImag;                   // Manipulate imaginary part
    property    Norm : TSignal read GetNorm write SetNorm;                 // Vector length
    property    Phi  : TSignal read GetPhi  write SetPhi;                  // Vector angle
  end;


  TPolarHelper = record helper for TPolar
  private
    function    GetReal: TSignal;
    procedure   SetReal( aValue : TSignal);
    function    GetImag: TSignal;
    procedure   SetImag( aValue : TSignal);
  public
    procedure   FromComplex( aReal, anImag : TSignal);  overload;          // Set value from complex value
    procedure   FromComplex( aValue: TComplex);         overload;          // Set value from complex value
    function    ToComplex: TComplex;
    procedure   Normalize;                                                 // Set Norm to 1.0
    procedure   NormalizePhi( aLower, anUpper: TSignal);                   // Set aLower <= Phi < anUpper
    procedure   Rotate( aPhi: TSignal);                                    // Rotate ove aPhi radians
  public
    property    Real : TSignal read GetReal   write SetReal;               // Manipulate real part
    property    Imag : TSignal read GetImag   write SetImag;               // Manipulate imaginary part
  end;


  TComplexArray     = array of TComplex;
  TPolarArray       = array of TPolar;
  TSignalPairArray  = TArray<TSignalPair>;

  function SignalPair( anX, anY: TSignal): TSignalPair; inline;


implementation



  function SignalPair( anX, anY: TSignal): TSignalPair; inline;
  begin
    Result.X := anX;
    Result.Y := anY;
  end;


{ ========
  TComplex = record
  private
    FReal : TSignal;
    FImag : TSignal;
  public
    property    Real : TSignal read FReal   write FReal;
    property    Imag : TSignal read FImag   write FImag;
    property    Norm : TSignal read GetNorm write SetNorm;
    property    Phi  : TSignal read GetPhi  write SetPhi;
  private
}

    function  TComplex.GetNorm: TSignal;
    begin
      Result := Sqrt( FReal * FReal + FImag * FImag);
    end;


    procedure TComplex.SetNorm( aValue : TSignal);
    var
      aNorm : TSignal;
    begin
      aNorm := Norm;
      FReal := FReal * aValue / aNorm;
      FImag := FImag * aValue / aNorm;
    end;


    function  TComplex.GetPhi: TSignal;
    begin
      Result := ArcTan2( FImag, FReal);
    end;


    procedure TComplex.SetPhi( aValue : TSignal);
    begin
      FromPolar( Norm, aValue);
    end;


//  public

    class operator TComplex.Implicit( a: TSignal): TComplex;
    begin
      Result.FReal := a;
      Result.FImag := 0;
    end;


    class operator TComplex.Implicit( a: TPolar): TComplex;
    begin
      Result.Norm := a.FNorm;
      Result.Phi  := a.FPhi;
    end;


    class operator TComplex.Explicit( a: TSignal): TComplex;
    begin
      Result.FReal := a;
      Result.FImag := 0;
    end;


    class operator TComplex.Implicit( a: TSignalPair): TComplex;
    begin
      Result.Imag := a.Left;
      Result.Real := a.Right;
    end;


    class operator TComplex.Explicit( a: TPolar): TComplex;
    begin
      Result.Norm := a.FNorm;
      Result.Phi  := a.FPhi;
    end;


    class operator TComplex.Explicit( a: TSignalPair): TComplex;
    begin
      Result.Imag := a.Left;
      Result.Real := a.Right;
    end;


    class operator TComplex.Negative( a: TComplex): TComplex;
    begin
      Result.FReal := -a.FImag;
      Result.FImag := -a.FImag;
    end;


    class operator TComplex.Positive( a: TComplex): TComplex;
    begin
      Result.FReal := a.FImag;
      Result.FImag := a.FImag;
    end;


    class operator TComplex.Equal( a, b: TComplex): Boolean;
    begin
      Result := ( a.FReal = b.FReal) and ( a.FImag = b.FImag);
    end;


    class operator TComplex.NotEqual( a, b: TComplex): Boolean;
    begin
      Result := ( a.FReal <> b.FReal) or ( a.FImag <> b.FImag);
    end;


    class operator TComplex.GreaterThan( a, b: TComplex): Boolean;
    begin
      Result := a.Norm > b.Norm;
    end;


    class operator TComplex.GreaterThanOrEqual( a, b: TComplex): Boolean;
    begin
      Result := a.Norm >= b.Norm;
    end;


    class operator TComplex.LessThan( a, b: TComplex): Boolean;
    begin
      Result := a.Norm < b.Norm;
    end;


    class operator TComplex.LessThanOrEqual( a, b: TComplex): Boolean;
    begin
      Result := a.Norm <= b.Norm;
    end;


    class operator TComplex.Add( a, b: TComplex): TComplex;
    begin
      Result.FReal := a.FReal + b.FReal;
      Result.FImag := a.FImag + b.FImag;
    end;


    class operator TComplex.Add( a: TSignal; b: TComplex): TComplex;
    begin
      Result.FReal := a + b.FReal;
      Result.FImag := b.FImag;
    end;


    class operator TComplex.Add( a: TComplex; b: TSignal): TComplex;
    begin
      Result.FReal := a.FReal + b;
      Result.FImag := a.FImag;
    end;


    class operator TComplex.Subtract( a, b: TComplex): TComplex;
    begin
      Result.FReal := a.FReal - b.FReal;
      Result.FImag := a.FImag - b.FImag;
    end;


    class operator TComplex.Subtract( a: TSignal; b: TComplex): TComplex;
    begin
      Result.FReal := a - b.FReal;
      Result.FImag :=   - b.FImag;
    end;


    class operator TComplex.Subtract( a: TComplex; b: TSignal): TComplex;
    begin
      Result.FReal := a.FReal - b;
      Result.FImag := a.FImag;
    end;


    class operator TComplex.Multiply( a, b: TComplex): TComplex;
    begin
      Result.FReal := a.FReal * b.FReal - a.FImag * b.FImag;
      Result.FImag := a.FReal * b.FImag + a.FImag * b.FReal;
    end;


    class operator TComplex.Multiply( a: TSignal; b: TComplex): TComplex;
    begin
      Result.FReal := a * b.FReal;
      Result.FImag := a * b.FImag;
    end;


    class operator TComplex.Multiply( a: TComplex; b: TSignal): TComplex;
    begin
      Result.FReal := a.FReal * b;
      Result.FImag := a.FImag * b;
    end;


    class operator TComplex.Divide( a, b: TComplex): TComplex;
    var
      QNorm : TSignal;
    begin
      QNorm        := b.FReal * b.FReal + b.FImag * b.FImag;
      Result.FReal := ( a.FImag * b.FImag + a.FReal * b.FReal) / QNorm;
      Result.FImag := ( a.FImag * b.FReal - a.FReal * b.FImag) / QNorm;
    end;


    class operator TComplex.Divide( a: TSignal; b: TComplex): TComplex;
    var
      QNorm : TSignal;
    begin
      QNorm        := b.FReal * b.FReal + b.FImag * b.FImag;
      Result.FReal := ( b.FImag + a * b.FReal) / QNorm;
      Result.FImag := ( b.FReal - a * b.FImag) / QNorm;
    end;


    class operator TComplex.Divide( a: TComplex; b: TSignal): TComplex;
    begin
      Result.FReal := a.FReal / b;
      Result.FImag := a.FImag / b;
    end;


//  public

    procedure   TComplex.FromPolar( aNorm, aPhi : TSignal); // overload;
    begin
      SinCos( aPhi, FImag, FReal);
      FReal := aNorm * FReal;
      FImag := aNorm * FImag;
    end;


    procedure   TComplex.FromPolar( const aValue: TPolar); // overload;
    begin
      FromPolar( aValue.Norm, aValue.Phi);
    end;


    function    TComplex.ToPolar: TPolar;
    begin
      Result.Phi  := Phi;
      Result.Norm := Norm;
    end;


    procedure   TComplex.Normalize;
    begin
      Norm := 1.0;
    end;


    procedure   TComplex.Rotate( aPhi: TSignal);
    begin
      Phi := Phi + aPhi;
    end;


{ ========
  TPolarHelper = record helper for TPolar
  public
    property    Real : TSignal read GetReal   write SetReal;                // Manipulate real part
    property    Imag : TSignal read GetImag   write SetImag;                // Manipulate imaginary part
  private
}

    function    TPolarHelper.GetReal: TSignal;
    begin
      Result := FNorm * Cos( FPhi);
    end;


    procedure   TPolarHelper.SetReal( aValue : TSignal);
    var
      aComplex : TComplex;
    begin
      aComplex := ToComplex;
      aComplex.Real := aValue;
      FromComplex( aComplex);
    end;


    function    TPolarHelper.GetImag: TSignal;
    begin
      Result := FNorm * Sin( FPhi);
    end;


    procedure   TPolarHelper.SetImag( aValue : TSignal);
    var
      aComplex : TComplex;
    begin
      aComplex := ToComplex;
      aComplex.Imag := aValue;
      FromComplex( aComplex);
    end;


//  public

    procedure   TPolarHelper.FromComplex( aReal, anImag : TSignal); // overload;
    begin
      Real := aReal;
      Imag := anImag;
    end;


    procedure   TPolarHelper.FromComplex( aValue: TComplex); // overload;
    begin
      FromComplex( aValue.Real, aValue.Imag);
    end;


    function    TPolarHelper.ToComplex: TComplex;
    begin
      SinCos( FPhi, Result.FImag, Result.FReal);
      Result.Real := FNorm * Result.Real;
      Result.Imag := FNorm * Result.Imag;
    end;


    procedure   TPolarHelper.Normalize;
    begin
      FNorm := 1.0;
    end;


    procedure   TPolarHelper.NormalizePhi( aLower, anUpper: TSignal);
    var
      aDelta : TSignal;
    begin
      aDelta := anUpper - aLower;

      if aDelta > 0
      then begin
        while Phi < aLower
        do Phi := Phi + aDelta;

        while Phi >= anUpper
        do Phi := Phi - aDelta;
      end
      else raise ERangeError.Create( 'Can not normalize when upper <= lower');
    end;


    procedure   TPolarHelper.Rotate( aPhi: TSignal);
    begin
      FPhi := FPhi + aPhi;
    end;


end.

