unit KnobsConversions;

{

  (C) COPYRIGHT 2014 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.SysUtils, System.Classes, System.Math,

  Vcl.Graphics,

  ComplexMath, KnobsUtils, Rationals;


const

  IN_CHANNEL_COUNT  = 8;                                          // Number of input  channels for a patch
  OUT_CHANNEL_COUNT = 8;                                          // Number of output channels for a patch
  SMALL_VALUE       = 1e-50;                                      // Lowest 'tolerable' signal level in patches                                ( -1000 dB)
  NORMAL_LIMIT      = 1e-20;                                      // Value below which the Normalize functions will truncate the value to zero {  -400 dB}
  MUTE_LIMIT        = 1e-7;                                       // Value below which mute signals are assumed to be off                      (  -140 dB)
  EULER             = 2.718281828459045235360287471;              // e .. as in Euler's constant
  EULER_REC         = 1 / EULER;                                  // 1 / e
  PI_REC            = 1.0 / PI;                                   // 1 / Pi
  TWO_PI            = PI * 2;                                     // Two times Pi, for angle conversions
  TWO_PI_REC        = 1.0 / TWO_PI;                               // 1 divided by two times PI
  HALF_PI           = PI / 2;                                     // Pi / 2      ; for fade tables and for S-envelopes
  QUARTER_PI        = PI / 4;                                     // Pi / 4
  SQRT2             = 1.4142135623730950488016887242097;          // Sqrt( 2)
  SQRT2_REC         = 1 / SQRT2;                                  // Reciprocal of SQRT_2
  SQRT3             = 1.7320508075688772935274463415059;          // Sqrt( 3)
  SQRT3_REC         = 1 / SQRT3;                                  // Reciprocal of SQRT_3
  SQRT5             = 2.2360679774997896964091736687313;          // Sqrt( 5)
  SQRT5_REC         = 1 / SQRT5;                                  // Reciprocal of SQRT_5
  TWO_PTWELVE       = 1.0594630943592952645618252949463;          // 2^(1/12)
  TWO_PTWELVE_REC   = 1 / TWO_PTWELVE;                            // 1/2^(1/12)
  THIRD             = 1.25992104960377;                           // 2 ^ (1/3)
  THIRD_REC         = 1 / THIRD;                                  // reciprocal of THIRD :: 0.7937005
  ONE_THIRD         = 1.0 / 3.0;                                  // just that 1/3 - 0.3333
  TWO_THIRDS        = 2.0 / 3.0;                                  // just that 2/3 - 0.6667
  LOG_TWO           = 0.30102999566;                              // 10log(2)
  LN0001            = -6.9077552789821370520539743640531;         // Ln( 0.001)
  NOTE_SCALING      = 128.0;                                      // Value used to scale note numbers to/from internal values
  NOTE_SCALING_REC  = 1 / NOTE_SCALING;                           // Reciprocal of NOTE_SCALING
  MIDI_SCALING      = 128.0;                                      // Value to scale MIDI values to/from internal signals
  MIDI_SCALING_REC  = 1 / MIDI_SCALING;                           // Reciprocal of MIDI_SCALING
  MORPH_RATE        = 10.0;                                       // Rate for live morphing, in Hz, determines MorphDecimation
  AUTOAMTION_RATE   = 1000.0;                                     // Automation rate in Hz, determines AutomationDecimation
  ONE_TWELFTH       = 1 / 12.0;                                   // Reciprocal of 12
  ONE_SIXTEENTH     = 1 / 16.0;                                   // Reciprocal of 16
  MAX_DB_VAL        = 140.0;                                      // Table scaling for dB lookup, range is [- DB_SCALE, + DB_SCALE]
  MAX_DB_VAL_REC    = 1.0/ MAX_DB_VAL;                            // Reciprocal of MAX_DB_VAL
  MIN_DB_SIGNAL     = 1e-7;                                       // Minimum signal value for which deciBels can be calculated
  MAX_DB_SIGNAL     = 1e7;                                        // Maximum signal value for which deciBels can be calculated
  MIN_DB_SIGNAL_REC = 1.0 / MIN_DB_SIGNAL;                        // Reciprocal of MIN_DB_SIGNAL;
  MAX_DB_SIGNAL_REC = 1.0 / MAX_DB_SIGNAL;                        // Reciprocal of MAX_DB_SIGNAL;
  CENTS             = 100.0;                                      // Scaling for cents lookup
  CENTS_REC         = 1.0 / CENTS;                                // reciprocal of CENTS
  FRACT_SCALING     = 1 shl 26;                                   // To scale from TSignal to fractional values - needs to match SmallForth setting in fixpoint.s4
  FRACT_SCALING_REC = 1 / FRACT_SCALING;                          // To scale from fractional values to TSignal
  KIBI              = 1024;                                       // Binary multiplication factors
  MEBI              = KIBI * KIBI;
  GIBI              = KIBI * MEBI;
  MIN_IN_SECS       = 1.0 / 60.0;                                 // Minutes in second
  UNITS_IN_CENT     = 1.0 / 256.0;                                // How many units aare needed for on cent
  UNITS_IN_DEGREE   = 1.0 / 360.0;                                // How many units are needed for one rotational degree
  NO_UNITS          = 'no_units';                                 // For conveters not accepting any units
  NO_NOTE           = 1.23456e7;                                  // ** MAGIC NUMBER ** for sequencers to play no note
  NOT_CONNECTED     = -1e6;                                       // Magic number to signal an input pin to not be connected
  FLASH_TIME        = 125e-3;                                     // LED flash time            (in s)
  TRIG_TIME         = 1e-3;                                       // Length of a trigger pulse (in s)
  NEURON_TIME       = 10e-3;                                      // Length of a neuron  pulse (in s)
  UNDEFINED         = 0.0;                                        // Undefined values are 0.0
  S_UNDEFINED       = 'undefined';                                // Undefined string value is 'undefined'
  MAX_CONV_ENUMS    = 1024;                                       // Max items in a converter enumeration


  ShortestPossibleEnvelopeTime =      1;                          // In samples
  EnvelopeSmoothing            =     99;                          // envout := ( EnvelopeSmooting * OldValue + NewValue) / ( 1 + EnvelopeSmoothing); - was 49
  EnvelopeSmoothingSlow        =     19;                          // envout := ( EnvelopeSmooting * OldValue + NewValue) / For control rate envelope processing
  EnvelopeSmoothingSlowFaster  =      9;                          // envout := ( EnvelopeSmooting * OldValue + NewValue) / For control rate envelope processing
  MinEnvTimeFast               =      0.0005;                     // Minimum time for fast   rate envelopes in seconds
  MinEnvTimeMedium             =      0.05;                       // Minimum time for medium rate envelopes in seconds
  MinEnvTimeSlow               =      5;                          // Minimum time for slow   rate envelopes in seconds
  MaxEnvTimeFast               =     12;                          // Maximum time for fast   rate envelopes in seconds
  MaxEnvTimeMedium             =   1200;                          // Maximum time for medium rate envelopes in seconds
  MaxEnvTimeSlow               = 120000;                          // Maximum time for slow   rate envelopes in seconds
  CompressorSmoothing          = 1.0 / 20.0;                      // envelope smoothing for compressor - was 9 then was 19, now is a float with value 1/20

  {$IFDEF DEBUG}
    {$INLINE OFF}                                                 // No inlining for debug code
  {$ELSE}
    {$INLINE ON}                                                  // Try to use inlining on production code
  {$ENDIF}


var

  // -------------------------------------------------------------------------------------------------------------------
  // NOTE : the tuning params below are user configurable. These are not used for LFO tuning

  ReferenceA         : TSignal;  // The value of the reference A to be used - default 440 Hz
  NotesPerOctave     : TSignal;  // The number of notes wanted in an octave - there only is notation for 12 tho ... A .. G#
  NotesPerOctaveRec  : TSignal;  // Reciprocal of NotesPerOctave
  MiddleNote         : TSignal;  // The middle note to be used - default 69 - this is a note number
  OctaveSpan         : TSignal;  // The ocatve span, default is 2.0

  // END NOTE.
  // -------------------------------------------------------------------------------------------------------------------


  // -------------------------------------------------------------------------------------------------------------------
  // NOTE : the following 15 CAN NOT BE CHANGED at will, some relations need to be maintained, for this see **1 and
  // please use the SetSystemRate function to change the System_Rate and all dependent rates will be changed ccorrectly.
  // To change the Control_Decimation use the SetControlDecimation function.
  // -------------------------------------------------------------------------------------------------------------------
  // These can not be moved to the implementation section as that will prevent inlining of some functions. Also after
  // changing these, audio processors need to be re-initialized and lookuptables for modules must be regenerated.

  System_Rate          : Integer;  // 44100.0;                               // samples / s  ( audio   rate)
  Control_Decimation   : Integer;  // 8                                      // Units, must be a power of two (((2^n)    ))
  Control_Dec_2        : Integer;  // 16                                     // Units, must be a power of two (((2^n)*2  ))
  Control_Dec_2_p1     : Integer;  // 17                                     // Units                         (((2^n)*2)+1)
  Control_Dec_Min_One  : Integer;  // 7                                      // Control_Decimation minus one  (((2^n)-1  ))
  System_Rate_2        : TSignal;  // System_Rate * 2;                       // samples / s
  Control_Rate         : TSignal;  // System_Rate / CONTROL_DEC;             // samples / s
  System_Rate_Rec      : TSignal;  // reciprocal of system rate              // s / sample
  System_Rate_2_Rec    : TSignal;  // reciprocal of system rate_2            // s / sample
  Control_Rate_Rec     : TSignal;  // reciprocal oc control rate             // samples / s
  MorphDecimation      : TSignal;  // Set such that the morph rate will be   // MORPH_RATE      Hz (   10 Hz) currently
  AutomationDecimation : TSignal;  // Set such that automation rate will be  // AUTOMATION_RATE Hz ( 1000 Hz) currently
  FlashCount           : Integer;  // # cycles for a LED flash               // FLASH_TIME      s  (  125 ms) currently
  FlashCountContr      : Integer;  // # control cycles for a LED flash       // FLASH_TIME      s  (  125 ms) currently
  PulseCount           : Integer;  // # cycles for an impulse signal         // TRIG_TIME       s  (    1 ms) currently
  PulseCountContr      : Integer;  // # control cycles for an impulse signal // TRIG_TIME       s  (    1 ms) currently
  NeuronCount          : Integer;  // # cycles for a neuron pulse            // NEURON_TIME     s  (  100 ms) currently
  NeuronCountContr     : Integer;  // # control cycles for a neuron pulse    // NEURON_TIME     s  (  100 ms) currently

  // END NOTE.
  // -------------------------------------------------------------------------------------------------------------------

  DefaultReferenceA    : TSignal = 440.0; // As read from INI file, just common storage here
  DefaultNotesPerOctave: TSignal =  12.0; // As read from INI file, just common storage here
  DefaultMiddleNote    : TSignal =  69.0; // As read from INI file, just common storage here
  DefaultOctaveSpan    : TSignal =   2.0; // As read from INI file, just common storage here

type

  EKnobsConversions   = class( Exception);
  ENoDefaultConverter = class( EKnobsConversions);
  EConvTooManyEnums   = class( EKnobsConversions);


  TSignalRate = (
    srControl,
    srAudio
  );


  TMidiSeptet = 0 .. 127;
  TMidiFilter = set of TMidiSeptet;
  TEnvRange   = ( erSlow, erMedium, erFast);


type

  TSignalFunction = function( const aValue: TSignal): TSignal;


  TSignalArrayWrapper = class
  public
    FSignals    : TSignalArray;
    FlastOutput : TSignal;
  private
    function    GetCount : Integer;
    function    GetSignal( anIndex: Integer): TSignal;
    procedure   SetSignal( anIndex: Integer; const aValue: TSignal);
  public
    constructor Create( aCount: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Erase;
    procedure   AddSignal( const aValue: TSignal);
    procedure   SetSignals( aStartIndex: Integer; const aValue: TSignalArrayWrapper);
    function    InterpolateL( const anIndex: TSignal): TSignal;   // index [0, Count-1] - linear   interpolation
    function    InterpolateA( const anIndex: TSignal): TSignal;   // index [0, Count-1] - all-pass interpolation
  public
    property    Count                    : Integer read GetCount;
    property    Signal[ anIndex: Integer]: TSignal read GetSignal write SetSignal; default;
  end;


  TSignalPairFifo = class
  private
    FSignals        : TSignalPairArray; // The data buffer
    FInp            : Integer;          // Current buffer insertion  point
    FOutp           : Integer;          // Current buffer extraction point
    FFreeCount      : Integer;          // Number of free data slots left
    FFillCount      : Integer;          // Number of filled data slots available
    FAllowOverflows : Boolean;          // True when overflows are allowed, pld data is sicarded then
    FOverflows      : Integer;          // Number of times Append  was called while the buffer had no free space
    FUnderflows     : Integer;          // Number of tiems GetData was called while the buffer had no items in it
    FLastValue      : TSignalPair;      // To repeatedly return the last pair when no samples are available
    FCountOverflows : Boolean;          // Set to true to keep count of overflows
  private
    function    GetOverflows : Integer;
    function    GetUnderflows: Integer;
    function    GetSize      : Integer;
  public
    constructor Create( aSize: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Append( const aLeft, aRight: TSignal);                                                         overload;
    procedure   Append( const aPair: TSignalPair);                                                             overload;
    function    GetData: TSignalPair;
    procedure   ClearOverUnderFlows;
  public
    property    Overflows      : Integer read GetOverflows;
    property    Underflows     : Integer read GetUnderflows;
    property    FillCount      : Integer read FFillCount;
    property    FreeCount      : Integer read FFreeCount;
    property    Size           : Integer read GetSize;
    property    AllowOverflows : Boolean read FAllowOverflows write FAllowOverflows;
    property    CountOverflows : Boolean read FCountOverflows write FCountOverflows  default True;
  end;


  TSignalOctet = record
    Ch1 : TSignal;
    Ch2 : TSignal;
    Ch3 : TSignal;
    Ch4 : TSignal;
    Ch5 : TSignal;
    Ch6 : TSignal;
    Ch7 : TSignal;
    Ch8 : TSignal;
  end;


  TBytesArray = array of TBytes;
  TBytesFifo  = class
  private
    FBuffer         : TBytesArray;
    FInPointer      : Integer;
    FOutPointer     : Integer;
    FFreeCount      : Integer;
    FFillCount      : Integer;
    FOverflows      : Integer;
    FUnderflows     : Integer;
    FCountOverflows : Boolean;
  private
    function    GetOverflows : Integer;
    function    GetUnderflows: Integer;
    function    GetSize      : Integer;
  public
    constructor Create( aSize: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Append( const aValue: TBytes);
    function    GetData: TBytes;
    procedure   ClearOverUnderFlows;
  public
    property    Overflows     : Integer read GetOverflows;
    property    Underflows    : Integer read GetUnderflows;
    property    FillCount     : Integer read FFillCount;
    property    FreeCount     : Integer read FFreeCount;
    property    Size          : Integer read GetSize;
    property    CountOverflows: Boolean read FCountOverflows write FCountOverflows  default True;
  end;


  TMedianCalculator = class
  private
    FSignals    : TSignalArray;  // The data buffer
    FSorted     : TSignalArray;  // The sorted buffer
    FInp        : Integer;       // Current buffer insertion  point
    FOutp       : Integer;       // Current buffer extraction point
    FIsSorted   : Boolean;
    FSum        : TSignal;       // Cached sum value
  private
    procedure   CheckSorted;
    function    GetCount   : Integer;
    function    GetLowest  : TSignal;
    function    GetMedian  : TSignal;
    function    GetHighest : TSignal;
    function    GetAverage : TSignal;
  private
    procedure   QuickSort;                                                                                     overload;
    procedure   QuickSort( const aValues: TSignalArray; L, R: Integer);                                        overload;
  public
    constructor Create( aSize: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Append( const aValue: TSignal);
    function    GetData: TSignal;
  public
    property    Count   : Integer read GetCount;
    property    Lowest  : TSignal read GetLowest;
    property    Median  : TSignal read GetMedian;
    property    Highest : TSignal read GetHighest;
    property    Average : TSignal read GetAverage;
  end;


  TRMSCalculator = class
  private
    FAvgTime       : TSignal;
    FIsFast        : Boolean;
    FMean          : TSignal;
    FAvgCoef       : TSignal;
  private
    procedure   Recalculate;
    procedure   SetAvgTime( aValue: TSignal);
  public
    constructor Create;
    procedure   Reset;
    procedure   Tick( anInValue: TSignal             ; var anOutValue: TSignal);                               overload;
    procedure   Tick( anInValueL, anInValueR: TSignal; var anOutValue: TSignal);                               overload;
    procedure   SampleRateChanged;
  public
    property    AvgTime : TSignal read FAvgTime write SetAvgTime;
    property    IsFast  : Boolean read FIsFast  write FIsFast;
  end;


  TDbMinMax = class
  private
    FCallRate   : TSignal;        // Tick calls / sample-set
    FLowValue   : TSignal;
    FHighValue  : TSignal;
    FValeyDecay : TSignal;
    FPeakDecay  : TSignal;
    FValue      : TSignal;
    FPeak       : TSignal;
    FValey      : TSignal;
    FAbsMin     : TSignal;
    FAbsMax     : TSignal;
  public
    constructor Create( const aLowValue, aHighValue, aValeyDecay, aPeakDecay: TSignal);
    procedure   Tick( const aLinearValue: TSignal);
    procedure   Reset;
    procedure   Clear;
    function    ValueRanged( aMin, aMax: Integer): Integer;
    function    PeakRanged ( aMin, aMax: Integer): Integer;
    function    ValeyRanged( aMin, aMax: Integer): Integer;
  public
    property    CallRate   : TSignal read FCallRate write FCallRate;
    property    LowValue   : TSignal read FLowValue  ;
    property    HighValue  : TSignal read FHighValue ;
    property    ValeyDecay : TSignal read FValeyDecay;
    property    PeakDecay  : TSignal read FPeakDecay ;
    property    Value      : TSignal read FValue     ;
    property    Peak       : TSignal read FPeak      ;
    property    Valey      : TSignal read FValey     ;
    property    AbsMin     : TSignal read FAbsMin    ;
    property    AbsMax     : TSignal read FAbsMax    ;
  end;


  TKnobsNullConverter = class
  private
    FSearched       : string;
    FDefaultUnits   : string;
    FIsRandomizable : Boolean;
    FIsEnumerable   : Boolean;
  protected
    FName           : string;
  private
    function    ApplyUnits   ( const aUnits      : string; var aValue: TSignal): Boolean;                       virtual;
    function    StringToValue( const aStringValue: string; var aValue: TSignal): Boolean;                       virtual;
    function    PostProcess  ( const aValue: TSignal): TSignal;                                                 virtual;
  public
    constructor Create;                                                                                         virtual;
    function    PosToHint     ( aPos, aSteps: Integer): string;                                                 virtual;
    function    PosToDisplay  ( aPos, aSteps: Integer): string;                                                 virtual;
    function    PosToValue    ( aPos, aSteps: Integer): TSignal;                                                virtual;
    function    ValueToPos    ( aVal: TSignal; aSteps: Integer): Integer;                                       virtual;
    function    DisplayToValue( const aDisplay: string; var aValue: TSignal): Boolean;                          virtual;
    function    ValueToDisplay( const aValue: TSignal; aSteps: Integer): string;
    function    CreateEnumeration: TStringList;                                                                 virtual;
  public
    property    Name           : string  read FName;
    property    DefaultUnits   : string  read FDefaultUnits;
    property    Searched       : string  read FSearched       write FSearched;
    property    IsRandomizable : Boolean read FIsRandomizable write FIsRandomizable;
    property    IsEnumerable   : Boolean read FIsEnumerable;
  end;


  TKnobsConverterClass = class of TKnobsNullConverter;


  TKnobsDynamicConverter = class( TKnobsNullConverter)
  private
    FCaptions : TStringList;
  private
    procedure   SetCaptions( const aValue: TStringList);
  public
    constructor Create;                                                                                        override;
    destructor  Destroy;                                                                                       override;
    function    PosToHint     ( aPos, aSteps: Integer): string;                                                override;
    function    PosToDisplay  ( aPos, aSteps: Integer): string;                                                override;
    function    DisplayToValue( const aDisplay: string; var aValue: TSignal): Boolean;                         override;
  public
    property    Captions : TStringList read FCaptions write SetCaptions;
  end;


  TKnobsConverters = class( TStringList)
  private
    function    GetConverterObject( anIndex: Integer): TKnobsNullConverter;
    function    Converter( const anIndex: string): TKnobsNullConverter;
  public
    constructor Create;
    procedure   RegisterConverter( aClass: TKnobsConverterClass);
    procedure   RegisterConverters( const aClasses: array of TKnobsConverterClass);
    function    PosToHint             ( const aName: string; aPos, aSteps: Integer): string;
    function    PosToDisplay          ( const aName: string; aPos, aSteps: Integer): string;
    function    PosToValue            ( const aName: string; aPos, aSteps: Integer): TSignal;
    function    ValueToPos            ( const aName: string; aVal: TSignal; aSteps: Integer): Integer;
    function    DisplayToValue        ( const aName, aDisplay: string; var aValue: TSignal): Boolean;
    function    ValueToDisplay        ( const aName: string; const aValue: TSignal; aSteps: Integer): string;
    function    IsEnumerable          ( const aName: string): Boolean;
    function    IsRandomizable        ( const aName: string): Boolean;
    procedure   SetRandomizable       ( const aName: string; aValue: Boolean);
    procedure   SetRandomizableByClass( const aName: string; aValue: Boolean);
    function    GetRandomizableByClass( const aName: string): Boolean;
    function    GetDynamicConverter   ( const aName: string): TKnobsDynamicConverter;
    function    CreateEnumeration     ( const aName: string): TStringList;
  public
    property    ConverterObject[ anIndex: Integer]: TKnobsNullConverter read GetConverterObject;
  end;


  function  ParseTextValue( const aType, aStringValue: string; var aSignal: TSignal): Boolean;

  function  ExpAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  LinAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  LogAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  SAttack  ( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  ExpDecay ( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  LinDecay ( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  LogDecay ( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  function  SDecay   ( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;

  function  RangeMap     ( const aValue, anInLow, anInHigh, anOutLow, anOutHigh: TSignal): TSignal;              inline;
  function  Normalize    ( const aValue : TSignal): TSignal;                                                     inline;
  function  NormalizeMute( const aValue : TSignal): TSignal;                                                     inline;
  procedure PNormalize   ( var aValue : TSignal);                                                                inline;
  function  RegardAsDenormal( const aValue: TSignal): Boolean;                                                   inline;
  function  CubeInterp ( const fr, inm1, inp, inp1, inp2: TSignal): TSignal;                                     inline;

  function  SignalTypeToColor( aValue: TSignalType): TColor;

  function  LogicToSignal              ( const aValue : Boolean): TSignal;                                       inline;
  function  SignalToLogic              ( const aValue : TSignal): Boolean;                                       inline;
  function  BinaryToSignal             ( const aValue : TBinary; aBitCount: TBitCount): TSignal;                 inline;
  function  SignalToBinary             ( const aValue : TSignal; aBitCount: TBitCount): TBinary;                 inline;
  function  SignalToMute               ( const aValue : TSignal): TSignal;                                       inline;

  function  UnitsTodB                  ( const aValue : TSignal): TSignal;                                       inline;
  function  dBToUnits                  ( const aValue : TSignal): TSignal;                                       inline;
  function  CentsToMultiplier          ( const aValue : TSignal): TSignal;                                       inline;

  function  NoteNumberToUnits          ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToNoteNumber          ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToFrequency           ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToFrequencyLin        ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToFrequencyCubed      ( const aValue : TSignal): TSignal;                                       inline;
  function  FrequencyToUnits           ( const aValue : TSignal): TSignal;                                       inline;

  function  FrequencyToPhaseDelta      ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToPhaseDelta          ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToPhaseDeltaLin       ( const aValue : TSignal): TSignal;                                       inline;

  function  SampleCountToTime          ( const aValue : TSignal): TSignal;                                       inline;
  function  TimeToSampleCount          ( const aValue : TSignal): TSignal;                                       inline;

  function  UnitsToLfoFastFrequency    ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoFastFrequencyToUnits    ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoMediumFrequency  ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoMediumFrequencyToUnits  ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoSlowFrequency    ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoSlowFrequencyToUnits    ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoFast2Frequency   ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoFast2FrequencyToUnits   ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoMedium2Frequency ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoMedium2FrequencyToUnits ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoSlow2Frequency   ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoSlow2FrequencyToUnits   ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoBPMFrequency     ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoBPMFrequencyToUnits     ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoLinFrequency     ( const aValue : TSignal): TSignal;                                       inline;
  function  LfoLinFrequencyToUnits     ( const aValue : TSignal): TSignal;                                       inline;

  function  FrequencyToLfoPhaseDelta   ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoFastPhaseDelta   ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoMediumPhaseDelta ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoSlowPhaseDelta   ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoFast2PhaseDelta  ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoMedium2PhaseDelta( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoSlow2PhaseDelta  ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoBPMPhaseDelta    ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToLfoLinPhaseDelta    ( const aValue : TSignal): TSignal;                                       inline;

  function  SlowTimeToSampleCount      ( const aValue : TSignal): TSignal;                                       inline;
  function  UnitsToSvfFrequency        ( const aValue : TSignal): TSignal;                                       inline;
  function  SignalToStr                ( const aValue: TSignal): string;
  function  TimeToStr                  ( const aValue: TSignal): string;
  function  FreqToStr                  ( const aValue: TSignal): string;
  function  FreqToBPMStr               ( const aValue: TSignal): string;
  function  RateMultiplierToStr        ( const aValue: TSignal): string;
  function  RateMultiplierValue        ( const aValue: TSignal): TSignal;
  function  SlewToCurve                ( const aValue: TSignal): TSignal;

  procedure RegisterConverters( const aClasses: array of TKnobsConverterClass);
  function  Converters: TKnobsConverters;


  function  LookupFrequency            ( const aValue: TSignal): TSignal;
  function  LookupFrequencyLin         ( const aValue: TSignal): TSignal;
  function  LookupFrequencyCubed       ( const aValue: TSignal): TSignal;
  function  LookupPhaseDelta           ( const aValue: TSignal): TSignal;
  function  LookupPhaseDeltaLin        ( const aValue: TSignal): TSignal;
  function  LookupSvfFrequency         ( const aValue: TSignal): TSignal;

  function  LookupLfoSlowFrequency     ( const aValue: TSignal): TSignal;
  function  LookupLfoSlowPhaseDelta    ( const aValue: TSignal): TSignal;
  function  LookupLfoMediumFrequency   ( const aValue: TSignal): TSignal;
  function  LookupLfoMediumPhaseDelta  ( const aValue: TSignal): TSignal;
  function  LookupLfoFastFrequency     ( const aValue: TSignal): TSignal;
  function  LookupLfoFastPhaseDelta    ( const aValue: TSignal): TSignal;
  function  LookupLfoSlow2Frequency    ( const aValue: TSignal): TSignal;
  function  LookupLfoSlow2PhaseDelta   ( const aValue: TSignal): TSignal;
  function  LookupLfoMedium2Frequency  ( const aValue: TSignal): TSignal;
  function  LookupLfoMedium2PhaseDelta ( const aValue: TSignal): TSignal;
  function  LookupLfoFast2Frequency    ( const aValue: TSignal): TSignal;
  function  LookupLfoFast2PhaseDelta   ( const aValue: TSignal): TSignal;
  function  LookupLfoBPMFrequency      ( const aValue: TSignal): TSignal;
  function  LookupLfoBPMPhaseDelta     ( const aValue: TSignal): TSignal;
  function  LookupLfoLinFrequency      ( const aValue: TSignal): TSignal;
  function  LookupLfoLinPhaseDelta     ( const aValue: TSignal): TSignal;

  function  LookupSine                 ( const aValue: TSignal): TSignal;
  function  LookupCosine               ( const aValue: TSignal): TSignal;
  function  LookupScaledEnvTime        ( const aValue: TSignal): TSignal;
  function  LookupScaledSEnvTime       ( const aValue: TSignal): TSignal;
  function  LookupEnvTime              ( aRange: TEnvRange; aValue: TSignal): TSignal;
  function  LookupEnvTimeAr            ( aRange: TEnvRange; aValue: TSignal): TSignal;
  function  LookupCentsMultiplier      ( const aValue: TSignal): TSignal;
  function  LookupSigmoid              ( const aValue: TSignal): TSignal;
  function  LookupTanh                 ( const aValue: TSignal): TSignal;

  function  MidiFilterToStr( const aValue: TMidiFilter): string;

  procedure ComputeTables;
  procedure SetSystemRate( aValue: Integer);
  procedure SetSystemControlDecimation( aValue: Integer);

  function  SignalToFractional( aValue: TSignal): Integer; inline;
  function  FractionalToSignal( aValue: Integer): TSignal; inline;

  function  SignalPairArrayToStr( const aValue: TSignalPairArray): string;
  function  StrToSignalPairArray( const aValue: string; DoFixup: Boolean; aLowX, aHiX: TSignal): TSignalPairArray;
  function  InterpolateSignalPairArray( const aData: TSignalPairArray; anX: TSignal): TSignal;

  function  RandomDiv( aRange, aDivision: Integer): Integer;
  function  BytesToStr( const aValue : TBytes): string;
  function  StrToBytes( const aValue: string): TBytes;
  function  SignalArrayToStr( const aValue : TSignalArray): string;
  function  StrToSignalArray( const aValue: string): TSignalArray;

  function  LedFlashTime ( IsFast: Boolean): Integer;                                                            inline;
  function  TrigPulseTime( IsFast: Boolean): Integer;                                                            inline;
  function  NeuronTime   ( IsFast: Boolean): Integer;                                                            inline;


var

  ColorControl    : TColor = clBlue;
  ColorAudio      : TColor = clRed;
  ColorLogic      : TColor = clYellow;
  ColorContrLogic : TColor = clLime;
  ColorUnknown    : TColor = clAqua;
  ColorHighlight  : TColor = clWhite;
  Shifts          : array[ TBitCount] of Integer;
  RecShifts       : array[ TBitCount] of TSignal;


implementation


var

  LocConverters : TKnobsConverters;



{ ========
  TSignals = class
  public
    FSignals : TSignalArray;
  public
    property    Count                    : Integer read GetCount;
    property    Signal[ anIndex: Integer]: TSignal read GetSignal write SetSignal; default;
  private
}

    function    TSignalArrayWrapper.GetCount: Integer;
    begin
      Result := Length( FSignals);
    end;


    function    TSignalArrayWrapper.GetSignal( anIndex: Integer): TSignal;
    begin
      Result := FSignals[ anIndex];
    end;


    procedure   TSignalArrayWrapper.SetSignal( anIndex: Integer; const aValue: TSignal);
    begin
      FSignals[ anIndex] := aValue;
    end;


//  public

    constructor TSignalArrayWrapper.Create( aCount: Integer); // overload;
    begin
      inherited Create;
      SetLength( FSignals, aCount);
    end;


    destructor  TSignalArrayWrapper.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TSignalArrayWrapper.Clear;
    begin
      SetLength( FSignals, 0);
    end;


    procedure   TSignalArrayWrapper.Erase;
//    var
//      i : Integer;
    begin
      FillChar( FSignals[ 0], SizeOf( FSignals[ 0]) * Length( FSignals), 0);
//      for i := 0 to Length( FSignals) - 1
//      do FSignals[ i] := 0.0;
    end;


    procedure   TSignalArrayWrapper.AddSignal( const aValue: TSignal);
    begin
      SetLength( FSignals, Count + 1);
      FSignals[ Count - 1] := aValue;
    end;


    procedure   TSignalArrayWrapper.SetSignals( aStartIndex: Integer; const aValue: TSignalArrayWrapper);
    var
      i : Integer;
    begin
      if Assigned( aValue)
      then begin
        SetLength( FSignals, aStartIndex + aValue.Count);

        for i := aStartIndex to Count - 1
        do Signal[ i] := aValue.Signal[ i - aStartIndex];
      end;
    end;


    function    TSignalArrayWrapper.InterpolateL( const anIndex: TSignal): TSignal;
    var
      p : Integer;
      q : Integer;
      d  : TSignal;
    begin
      p := Trunc( anIndex);
      d := anIndex - p;
      q := ( p + 1);

      if q > Count - 1
      then q := 0;

      Result := Normalize( FSignals[ p] + d * ( FSignals[ q] - FSignals[ p]));
    end;


    function    TSignalArrayWrapper.InterpolateA( const anIndex: TSignal): TSignal;
    var
      p : Integer;
      q : Integer;
      d  : TSignal;
      a  : TSignal;
    begin
      p := Trunc( anIndex);
      d := anIndex - p;

      if d < 0.5                 // best 0.5 <= d < 1.5
      then begin
        Inc( p);
        d := d + 1;

        if p > Count - 1
        then p := p - Count;
      end;

      a := ( 1 - d) / ( 1 + d);
      q := ( p + 1);

      if q > Count - 1
      then q := q - Count;

      FlastOutput := Normalize( FSignals[ p] + a * ( FSignals[ q] - FLastOutput));
      Result      := FlastOutput;
    end;



{ ========
  TSignalpairFifo = class
  private
    FSignals        : TSignalPairArray; // The data buffer
    FInp            : Integer;          // Current buffer insertion  point
    FOutp           : Integer;          // Current buffer extraction point
    FFreeCount      : Integer;          // Number of free data slots left
    FFillCount      : Integer;          // Number of filled data slots available
    FAllowOverflows : Boolean;          // True when overflows are allowed, pld data is sicarded then
    FOverflows      : Integer;          // Number of times Append  was called while the buffer had no free space
    FUnderflows     : Integer;          // Number of tiems GetData was called while the buffer had no items in it
    FLastValue      : TSignalPair;      // To repeatedly return the last pair when no samples are available
    FCountOverflows : Boolean;          // Set to true to keep count of overflows
  public
    property    Overflows      : Integer read GetOverflows;
    property    Underflows     : Integer read GetUnderflows;
    property    FillCount      : Integer read FFillCount;
    property    FreeCount      : Integer read FFreeCount;
    property    Size           : Integer read GetSize;
    property    AllowOverflows : Boolean read FAllowOverflows write FAllowOverflows;
    property    CountOverflows : Boolean read FCountOverflows write FCountOverflows  default True;
  private
}


    function    TSignalPairFifo.GetOverflows: Integer;
    begin
      Result     := FOverflows;
      FOverflows := FOverflows - Result;
    end;


    function    TSignalPairFifo.GetUnderflows: Integer;
    begin
      Result      := FUnderflows;
      FUnderflows := FUnderflows - Result;
    end;


    function    TSignalPairFifo.GetSize: Integer;
    begin
      Result := Length( FSignals);
    end;


//  public

    constructor TSignalPairFifo.Create( aSize: Integer);
    begin
      inherited Create;
      CountOverflows := True;
      SetLength( FSignals, aSize);
      Clear;
    end;


    destructor  TSignalPairFifo.Destroy; // override;
    begin
      Clear;
      FFreeCount := 0;
      inherited;
    end;


    procedure   TSignalPairFifo.Clear;
    begin
      FFillCount       := 0;
      FFreeCount       := 0;
      FInp             := 0;
      FOutp            := 0;
      FUnderflows      := 0;
      FOverflows       := 0;
      FLastValue.Left  := 0;
      FLastValue.Right := 0;
      FFreeCount       := Size;
    end;


    procedure   TSignalPairFifo.Append( const aLeft, aRight: TSignal); // overload;
    begin
      Append( SignalPair( aLeft, aRight));
    end;


    procedure   TSignalPairFifo.Append( const aPair: TSignalPair); // overload;
    // When the queue is full discard sample pair, and incremeent the oveflow counteer
    begin
      if FFreeCount > 0
      then begin
        Dec( FFreeCount);
        FSignals[ FInp] := aPair;
        FInp := ( FInp + 1) mod Size;
        Inc( FFillCount);
      end
      else if AllowOverflows
      then begin
        FSignals[ FInp] := aPair;
        FInp  := ( FInp  + 1) mod Size;
        FOutp := ( FOutp + 1) mod Size;
      end
      else begin
        if CountOverflows
        then {$Q-R-} Inc( FOverflows) {$Q+R+};
      end;
    end;


    function    TSignalPairFifo.GetData: TSignalPair;
    // When no sample pairs are available repeatedly return the last pair, and increment the underflow counter.
    begin
      if FFillCount > 0
      then begin
        Dec( FFillCount);
        FLastValue := FSignals[ FOutp];
        FOutp      := ( FOutp + 1) mod Size;
        Inc( FFreeCount);
      end
      else begin
        if CountOverflows
        then {$Q-R-} Inc( FUnderflows) {$Q+R+};
      end;

      Result := FLastValue;
    end;


    procedure   TSignalPairFifo.ClearOverUnderFlows;
    begin
      FUnderflows := 0;
      FOverflows  := 0;
    end;


{ ========
  TBytesArray = array of Bytes;
  TBytesFifo = class
  private
    FBuffer         : TBytesArray;
    FInPointer      : Integer;
    FOutPounter     : Integer;
    FFreeCount      : Integer;
    FFillCount      : Integer;
    FOverflows      : Integer;
    FUnderflows     : Integer;
    FCountOverflows : Boolean;
  public
    property    Overflows     : Integer read GetOverflows;
    property    Underflows    : Integer read GetUnderflows;
    property    FillCount     : Integer read FFillCount;
    property    FreeCount     : Integer read FFreeCount;
    property    Size          : Integer read GetSize;
    property    CountOverflows: Boolean read FCountOverflows write FCountOverflows  default True;
  private
}

    function    TBytesFifo.GetOverflows: Integer;
    begin
      Result     := FOverflows;
      FOverflows := FOverflows - Result;
    end;


    function    TBytesFifo.GetUnderflows: Integer;
    begin
      Result      := FUnderflows;
      FUnderflows := FUnderflows - Result;
    end;


    function    TBytesFifo.GetSize: Integer;
    begin
      Result := Length( FBuffer);
    end;


//  public

    constructor TBytesFifo.Create( aSize: Integer);
    begin
      inherited Create;
      CountOverflows := True;
      SetLength( FBuffer, aSize);
      Clear;
    end;


    destructor  TBytesFifo.Destroy; // override;
    begin
      Clear;
      FFreeCount := 0;
      inherited;
    end;


    procedure   TBytesFifo.Clear;
    begin
      FFreeCount  := 0;
      FFillCount  := 0;
      FFreeCount  := 0;
      FInPointer  := 0;
      FOutPointer := 0;
      FUnderflows := 0;
      FOverflows  := 0;
      FFreeCount  := Size;
    end;


    procedure   TBytesFifo.Append( const aValue: TBytes);
    begin
      if FFreeCount > 0
      then begin
        Dec( FFreeCount);
        FBuffer[ FInPointer] := aValue;
        FInPointer           := ( FInPointer + 1) mod Size;
        Inc( FFillCount);
      end
      else begin
        if CountOverflows
        then {$Q-R-} Inc( FOverflows) {$Q+R+};
      end;
    end;


    function    TBytesFifo.GetData: TBytes;
    begin
      if FFillCount > 0
      then begin
        Dec( FFillCount);
        Result      := FBuffer[ FOutPointer];
        FOutPointer := ( FOutPointer + 1) mod Size;
        Inc( FFreeCount);
      end
      else begin
        if CountOverflows
        then {$Q-R-} Inc( FUnderflows) {$Q+R+};
      end;
    end;


    procedure   TBytesFifo.ClearOverUnderFlows;
    begin
      FOverflows  := 0;
      FUnderflows := 0;
    end;


{ ========
  TMedianCalculator = class
  private
    FSignals    : TSignalArray;  // The data buffer
    FSorted     : TSignalArray;  // The sorted buffer
    FInp        : Integer;       // Current buffer insertion  point
    FOutp       : Integer;       // Current buffer extraction point
    FIsSorted   : Boolean;
    FSum        : TSignal;       // Cached sum value
  public
    property    Count   : Integer read GetCount;
    property    Lowest  : TSignal read GetLowest;
    property    Median  : TSignal read GetMedian;
    property    Highest : TSignal read GetHighest;
    property    Average : TSignal read GetAverage;
  private
}

    procedure   TMedianCalculator.CheckSorted;
    var
      i : Integer;
    begin
      if ( Count > 0) and not FIsSorted
      then begin
        for i := 0 to Count - 1
        do FSorted[ i] := FSignals[ i];

        QuickSort;
        FIsSorted := True;
      end;
    end;


    function    TMedianCalculator.GetCount: Integer;
    begin
      if Assigned( FSignals)
      then Result := Length( FSignals)
      else Result := 0;
    end;


    function    TMedianCalculator.GetLowest: TSignal;
    begin
      if Count > 0
      then begin
        CheckSorted;
        Result := FSorted[ 0];
      end
      else Result := 0;
    end;


    function    TMedianCalculator.GetMedian: TSignal;
    begin
      if Count > 0
      then begin
        CheckSorted;

        if Odd( Count)
        then Result := FSorted[ Count shr 1]
        else Result := ( FSorted[( Count shr 1) - 1] + FSorted[ Count shr 1]) * 0.5;
      end
      else Result := 0;
    end;


    function    TMedianCalculator.GetHighest: TSignal;
    begin
      if Count > 0
      then begin
        CheckSorted;
        Result := FSorted[ Count - 1];
      end
      else Result := 0;
    end;


    function    TMedianCalculator.GetAverage : TSignal;
    begin
      if Count > 0
      then Result := FSum / Count
      else Result := 0;
    end;


//  private

    procedure   TMedianCalculator.QuickSort; // overlaod;
    begin
      QuickSort( FSorted, 0, Count - 1);
    end;


    procedure   TMedianCalculator.QuickSort( const aValues: TSignalArray; L, R: Integer); // overlaod;
    var
      I, J, P: Integer;
      aTmp : TSignal;
    begin
      repeat
        I := L;
        J := R;
        P := ( L + R) shr 1;

        repeat
          while aValues[ I] < aValues[ P] do Inc( I);
          while aValues[ J] > aValues[ P] do Dec( J);
          if I <= J
          then begin
            if I <> J
            then begin
              aTmp        := aValues[ I];
              aValues[ I] := aValues[ J];
              aValues[ J] := aTmp;
            end;

            if P = I
            then P := J
            else if P = J
            then P := I;

            Inc( I);
            Dec( J);
          end;
        until I > J;

        if L < J then QuickSort( aValues, L, J);
        L := I;
      until I >= R;
    end;


//  public

    constructor TMedianCalculator.Create( aSize: Integer);
    begin
      inherited Create;
      SetLength( FSignals, aSize);
      SetLength( FSorted , aSize);
    end;


    destructor  TMedianCalculator.Destroy; // override;
    begin
      inherited;
    end;


    procedure   TMedianCalculator.Append( const aValue: TSignal);
    begin
      FSum            := FSum + aValue - FSignals[ FInp];
      FSignals[ FInp] := aValue;
      FInp            := ( FInp + 1) mod Count;
      FIsSorted       := False;
    end;


    function    TMedianCalculator.GetData: TSignal;
    begin
      Result := FSignals[ FOutp];
      FOutp  := ( FOutp + 1) mod Count;
    end;



{ ========
  TRMSCalculator = class
  private
    FAvgTime       : TSignal;
    FIsFast        : Boolean;
    FMean          : TSignal;
    FAvgCoef       : TSignal;
  public
    property    AvgTime : TSignal read FAvgTime write SetAvgTime;
    property    IsFast  : Boolean read FIsFast  write FIsFast;
  orivate
}

    procedure   TRMSCalculator.Recalculate;
    begin;
      if FIsFast
      then FAvgCoef := 1.0 - Exp( - 1.0 / TimeToSampleCount    ( FAvgTime))
      else FAvgCoef := 1.0 - Exp( - 1.0 / SlowTimeToSampleCount( FAvgTime));
    end;


    procedure   TRMSCalculator.SetAvgTime( aValue: TSignal);
    begin
      FAvgTime := aValue;
      Recalculate;
    end;


//  public

    constructor TRMSCalculator.Create;
    begin
      inherited;
      IsFast := False;
      Reset;
      AvgTime := 0.1;              // 100 ms
    end;


    procedure   TRMSCalculator.Reset;
    begin
      FMean := 0.0;
    end;


    procedure   TRMSCalculator.Tick( anInValue: TSignal; var anOutValue: TSignal); // overload;
    begin
      FMean      := FMean + FAvgCoef * ( anInValue * anInValue - FMean);
      anOutValue := Sqrt( FMean);
    end;


    procedure   TRMSCalculator.Tick( anInValueL, anInValueR: TSignal; var anOutValue: TSignal); // overload;
    begin
      FMean      := FMean + FAvgCoef * ( anInValueL * anInValueL + anInValueR * anInValueR - FMean);
      anOutValue := Sqrt( FMean);
    end;


    procedure   TRMSCalculator.SampleRateChanged;
    // Notification for sample rate change, the actual values are in global variables
    // System_Rate for the audio rate and Control_Rate for the control rate
    // Modules caching sample rate dependent data should recalculate such data
    // in an overriden version of SampleRateChanged.
    begin
      Recalculate;
    end;


{ ========
  TDbMinMaxCalculator = class
  private
    FCallRate   : TSignal;        // Tick calls / sample
    FLowValue   : TSignal;
    FHighValue  : TSignal;
    FValeyDecay : TSignal;
    FPeakDecay  : TSignal;
    FValue      : TSignal;
    FPeak       : TSignal;
    FValey      : TSignal;
    FAbsMin     : TSignal;
    FAbsMax     : TSignal;
  public
    property    CallRate   : TSignal read FCallRate write FCallRate;
    property    LowValue   : TSignal read FLowValue  ;
    property    HighValue  : TSignal read FHighValue ;
    property    ValeyDecay : TSignal read FValeyDecay;
    property    PeakDecay  : TSignal read FPeakDecay ;
    property    Value      : TSignal read FValue     ;
    property    Peak       : TSignal read FPeak      ;
    property    Valey      : TSignal read FValey     ;
    property    AbsMin     : TSignal read FAbsMin    ;
    property    AbsMax     : TSignal read FAbsMax    ;
  public
}

    constructor TDbMinMax.Create( const aLowValue, aHighValue, aValeyDecay, aPeakDecay: TSignal);
    const
      OFT = 1.0 / 512.0;
    begin
      inherited Create;
      FCallRate   := OFT;
      FLowValue   := aLowValue;
      FHighValue  := aHighValue;
      FValeyDecay := aValeyDecay;
      FPeakDecay  := aPeakDecay;
      Clear;
    end;


    procedure   TDbMinMax.Tick( const aLinearValue: TSignal);
    begin
      FValue := Normalize( UnitsTodB( aLinearValue));

      if FValue > FPeak
      then FPeak := FValue
      else FPeak := ( FPeakDecay / FCallRate) * ( FValue - FPeak) + FPeak;

      if FValue < FValey
      then FValey := FValue
      else FValey := ( FValeyDecay / FCallRate) * ( FValue - FValey) + FValey;

      if FValue > FAbsMax
      then FAbsMax := FValue;

      if FValue < FAbsMin
      then FAbsMin := FValue;
    end;


    procedure   TDbMinMax.Reset;
    begin
      FPeak   := FValue;
      FValey  := FValue;
      FAbsMin := FValue;
      FAbsMax := FValue;
    end;


    procedure   TDbMinMax.Clear;
    begin
      FAbsMin := FHighValue;
      FAbsMax := FLowValue;
      FValue  := FLowValue;
      FPeak   := FValue;
      FValey  := FValue;
    end;


    function    TDbMinMax.ValueRanged( aMin, aMax: Integer): Integer;
    begin
      Result := Round( RangeMap( Clip( FValue, FLowValue, FHighValue), FLowValue, FHighValue, aMin, aMax));
    end;


    function    TDbMinMax.PeakRanged( aMin, aMax: Integer): Integer;
    begin
      Result := Ceil( RangeMap( Clip( FPeak, FLowValue, FHighValue), FLowValue, FHighValue, aMin, aMax));
    end;


    function    TDbMinMax.ValeyRanged( aMin, aMax: Integer): Integer;
    begin
      Result := Floor( RangeMap( Clip( FValey, FLowValue, FHighValue), FLowValue, FHighValue, aMin, aMax));
    end;



  function  IsTimeValue( const aValue: string): Boolean;
  begin
    Result := Pos( ':', aValue) > 0;
  end;


  function  ParseTextValue( const aType, aStringValue: string; var aSignal: TSignal): Boolean;
  begin
    if Converters <> nil
    then Result := Converters.DisplayToValue( aType, aStringValue, aSignal)
    else begin
      aSignal := UNDEFINED;
      Result  := False;
    end;
  end;


  function  ExpAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 2 ^ x - 1  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := LookupScaledEnvTime( aTim);
    Result   := Normalize( aStartValue + Unscaled * ( anEndValue - aStartValue));
  end;


  function  LinAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = x  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := aTim;
    Result   := Normalize( aStartValue + Unscaled * ( anEndValue - aStartValue));
  end;


  function  LogAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 2 - 2 ^ ( 1 - x)  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := 1 - LookupScaledEnvTime( 1 - aTim);
    Result   := Normalize( aStartValue + Unscaled * ( anEndValue - aStartValue));
  end;


  function  SAttack( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = sqr( sin( x * pi / 2))  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := LookupScaledSEnvTime( aTim);
    Result   := Normalize( aStartValue + Unscaled * ( anEndValue - aStartValue));
  end;


  function  ExpDecay( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 2 ^ ( 1 - x) - 1  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := LookupScaledEnvTime( 1 - aTim);
    Result   := Normalize( anEndValue + Unscaled * ( aStartValue - anEndValue));
  end;


  function  LinDecay( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 1 - x  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := 1 - aTim;
    Result   := Normalize( anEndValue + Unscaled * ( aStartValue - anEndValue));
  end;


  function  LogDecay( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 2 - 2 ^ x  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := 1 - LookupScaledEnvTime( aTim);
    Result   := Normalize( anEndValue + Unscaled * ( aStartValue - anEndValue));
  end;


  function  SDecay( const aTime, aDuration, aStartValue, anEndValue: TSignal): TSignal;
  // y = 1 - sqr( sin( x * pi / 2))  [0,1] -> [0,1]
  var
    Unscaled : TSignal;
    aDur     : TSignal;
    aTim     : TSignal;
  begin
    aDur     := Max( 1, aDuration);
    aTim     := Clip( aTime, 0, aDur) / aDur;
    Unscaled := LookupScaledSEnvTime( 1 - aTim);
    Result   := Normalize( anEndValue + Unscaled * ( aStartValue - anEndValue));
  end;

  function  Normalize( const aValue: TSignal): TSignal; inline;
  //
  // This actually checks if aValue is zero, a denormal or it's absulote value being < NORMAL_LIMIT
  // [or the value not being a number (NaN)] and it treats all those cases as if they were zero.
  // hmm .. Nan check adds like 5% to computtional load ... commented it out. Maybe later I can
  // come up with a bit pattern check that finds both Nans and Denormals. Ok I did - read on ...
  //
  // ok, so what do we actually have ... IEEE 754 double precission:
  //
  //   +-+-----------+-------------------------------------------+
  //   |1|11         |52                                         |
  //   +-+-----------+-------------------------------------------+
  //   | |$7ff       |$fffffffffffff                             |
  //   +-+-----------+-------------------------------------------+
  //   |s|e          |m                                          |
  //   +-+-----------+-------------------------------------------+
  //
  //   normal double       0 < e < $7ff          v = (-1)^s * 2^(e-1023) * 1.m
  //   "denormal"          e = 0    and m <> 0   v = (-1)^s * 2^-1022    * 0.m
  //   Zero                e = 0    and m  = 0   v = (-1)^s * 2^-1022    * 0.m
  //   Infinity (INF)      e = $7ff and m  = 0   v = (-1)^s * INF
  //   Not a Number (NaN)  e = $7ff and m <> 0   NaN
  //
  // Original code here was :
  //
  //   if (( PWordArray( @ aValue)^[ 3] and $7ff0) = 0) or ( Abs( aValue) < NORMAL_LIMIT)
  //   then Result := 0
  //   else Result := aValue;
  //
  // Then tried :
  //
  //   if (( PWordArray( @ aValue)^[ 3] and $7ff0) = 0) or IsNan( aValue) or ( Abs( aValue) < NORMAL_LIMIT)
  //
  //   Which adds about 5% runtime overhead on a 90% patch.
  //
  // sooo ... but ... the solution may be to just sub one from the exponent ... and then checking it to
  // be >= $7fe something like in:
  //
  //   if ((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < NORMAL_LIMIT)
  //
  // Which then should lead to a conversion of all non-normal floats into zero. This adds less than 1% of
  // extra overhead for the same patch.
  //
  // This can be optimized a bit further though, noting that the NORMAL_LIMIT is 1e-20 which is about 2^-43,
  // we can just compare the exponent against -43 instead of calculating an abs function, provided we
  // beforehand made sure that we do no deal with a denormal.
  //
  // Which leads to :
  //
  //   const Limit = ( Word( -43) shl 4);
  //   var   anExp : Word;
  //
  //   anExp := PWordArray( @ aValue)^[ 3] and $7ff0;
  //   if ( anExp = $7ff0) or ( anExp > Limit)
  //
  // Which for some reason is incorrect ... or at least gives bad audio
  //
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
{$Q-R-}
 // if PWordArray( @ aValue)^[ 3] and $7ff0 > Limit
 // anExp := PWordArray( @ aValue)^[ 3] and $7ff0;
 // if ( anExp = $7ff0) or ( anExp > Limit)
 // if ((( anExp and $7ff0) - $10) >= $7fe0) or ( anExp >= Limit)

    if ((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < NORMAL_LIMIT)
    then Result := 0
    else Result := aValue;
{$Q+R+}
  end;


  function  NormalizeMute( const aValue: TSignal): TSignal; inline;
  //
  // Normalization for mute signals
  //
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
{$Q-R-}
    if ((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < MUTE_LIMIT)
    then Result := 0
    else Result := aValue;
{$Q+R+}
  end;


  procedure PNormalize( var aValue : TSignal); inline;
  // See remarks for 'function Normalize();' above.
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
{$Q-R-}
    if ((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < NORMAL_LIMIT)
    then aValue := 0;
{$Q+R+}
  end;


  function RegardAsDenormal( const aValue: TSignal): Boolean; inline;
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
{$Q-R-}
    Result := (((( PWordArray( @ aValue)^[ 3] and $7ff0) - $10) >= $7fe0) or ( Abs( aValue) < SMALL_VALUE)) and ( aValue <> 0);
{$Q+R+}
  end;


  function  RangeMap( const aValue, anInLow, anInHigh, anOutLow, anOutHigh: TSignal): TSignal; inline;
  begin
    Result := anOutLow + (( aValue - anInLow) / ( anInHigh - anInLow)) * ( anOutHigh - anOutLow);
  end;


  function  CubeInterp( const fr, inm1, inp, inp1, inp2: TSignal): TSignal; inline;
  begin
    Result :=
      inp + 0.5 * fr * ( inp1 - inm1 + fr * ( 4.0 * inp1 + 2.0 * inm1 - 5.0 * inp - inp2 + fr * ( 3.0 * ( inp - inp1) - inm1 + inp2)));
  end;


  function  SignalTypeToColor( aValue: TSignalType): TColor;
  begin
    case aValue of
      stControl    : Result := ColorControl;
      stAudio      : Result := ColorAudio;
      stLogic      : Result := ColorLogic;
      stContrLogic : Result := ColorContrLogic;
      else           Result := ColorUnknown;
    end;
  end;


  function  LogicToSignal( const aValue: Boolean): TSignal; inline;
  begin
    if aValue
    then Result := 1
    else Result := -1;
  end;


  function  SignalToLogic( const aValue: TSignal): Boolean; inline;
  begin
    Result := aValue > 0;
  end;


  function  BinaryToSignal( const aValue : TBinary; aBitCount: TBitCount): TSignal; inline;
  begin
    Result := aValue * RecShifts[ aBitCount];
  end;

  function  SignalToBinary( const aValue : TSignal; aBitCount: TBitCount): TBinary; inline;
  begin
    Result := Round( Clip( aValue, -1, 1) * Shifts[ aBitCount]);
  end;


  function  SignalToMute( const aValue  : TSignal): TSignal; inline;
  type
    TWordArray = array[ 0 .. 3] of Word;
    PWordArray = ^TWordArray;
  begin
    if (( PWordArray( @ aValue)^[ 3] and $7ff0) = 0) or ( Abs( aValue) < MUTE_LIMIT)
    then Result := 1.0
    else Result := Normalize( 1 - aValue);
  end;


  function  UnitsTodB( const aValue : TSignal): TSignal; inline;
  // todo : make a LUT
  begin
    if aValue <= MIN_DB_SIGNAL
    then Result := - MAX_DB_VAL
    else Result := 20 * Log10( aValue);
  end;


  function  dBToUnits( const aValue : TSignal): TSignal; inline;
  // todo : make a LUT
  const
    OT = 1.0 / 20.0;
  begin
    Result := Power( 10, aValue * OT);
  end;


  function  CentsToMultiplier( const aValue : TSignal): TSignal; inline;
  begin
    Result := Power( 2, aValue / 1200.0);   // todo: how do cents change when the octave changes .. from say 2.0 to 3.0?
  end;


  function  CentsToMultiplierScaled( const aValue : TSignal): TSignal; inline;
  begin
    Result := CentsToMultiplier( aValue * CENTS);
  end;


  function  Sigmoid( const aValue : TSignal): TSignal; inline;
  begin
    Result := ( 2.0 / ( 1.0 + Exp( - 8.0 * aValue))) - 1.0;
  end;


  function  UnitsToTanh( const aValue : TSignal): TSignal; inline;
  begin
    Result := Tanh( aValue * 8);
  end;


  function  UnitsToNoteNumber( const aValue: TSignal): TSignal; inline;
  begin
    Result := NOTE_SCALING * aValue;
  end;


  function  NoteNumberToUnits( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * NOTE_SCALING_REC;
  end;


  function  UnitsToFrequency( const aValue: TSignal): TSignal; inline;
  // such that note Nr 69 maps to A 440 Hz
  // see: http://www.phys.unsw.edu.au/jw/notes.html
  // freq = ( 2 ^ ( notenr - 69) / 12) * 440
  // freq = Power( 2, ( notenr - 69) / 12) * ReferenceA;
  // freq = Power( OctaveSpan, ( notenr - MiddleNote) / NotesPerOctave) * ReferenceA;
  begin
    Result := Power( OctaveSpan, ( UnitsToNoteNumber( aValue) - MiddleNote) * NotesPerOctaveRec) * ReferenceA;
  end;


  function  UnitsToFrequencyLin( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * 10000;
  end;


  function  UnitsToFrequencyCubed( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * aValue * aValue * 10000;
  end;


  function  FrequencyToUnits( const aValue: TSignal): TSignal; inline;
  // see: http://www.phys.unsw.edu.au/jw/notes.html
  // n  = 12 * log2( f / 440) + 69
  //  Result := Round( 12 * Log2( aValue / ReferenceA)) + 69;
  begin
    Result := NoteNumberToUnits( NotesPerOctave * LogN( OctaveSpan, aValue / ReferenceA) + MiddleNote);
  end;


  function  FrequencyToPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * System_Rate_Rec;
  end;


  function  UnitsToPhaseDelta( const aValue : TSignal): TSignal; inline;
  begin
    Result := FrequencyToPhaseDelta( UnitsToFrequency( aValue));
  end;


  function  UnitsToPhaseDeltaLin( const aValue : TSignal): TSignal; inline;
  begin
    Result := FrequencyToPhaseDelta( UnitsToFrequencyLin( aValue));
  end;


  function  SampleCountToTime( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * System_Rate_Rec;
  end;


  function  TimeToSampleCount( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * System_Rate;
  end;


  // ------------------------------------

  function  UnitsToLfoFastFrequency( const aValue: TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 43) * ONE_TWELFTH) * 3.05);
  end;


  function  LfoFastFrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 12 * Log2( aValue / 3.05) + 43);
  end;


  function  UnitsToLfoMediumFrequency( const aValue: TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 103) * ONE_TWELFTH) * 3.05);
  end;


  function  LfoMediumFrequencyToUnits( const aValue: TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 12 * Log2( aValue / 3.05) + 103);
  end;


  function  UnitsToLfoSlowFrequency( const aValue: TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 163) * ONE_TWELFTH) * 3.05);
  end;


  function  LfoSlowFrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 12 * Log2( aValue / 3.05) + 163);
  end;


  function  UnitsToLfoFast2Frequency( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 32) * ONE_SIXTEENTH) * 2.0);
  end;


  function  LfoFast2FrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 16 * Log2( aValue / 2.0) + 32);
  end;


  function  UnitsToLfoMedium2Frequency( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 128) * ONE_SIXTEENTH) * 2.0);
  end;


  function  LfoMedium2FrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 16 * Log2( aValue / 2.0) + 128);
  end;


  function  UnitsToLfoSlow2Frequency( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := Normalize( Power( 2, ( UnitsToNoteNumber( aValue) - 224) * ONE_SIXTEENTH) * 2.0);
  end;


  function  LfoSlow2FrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    // For LFOs the octave is fixed to 2.0 and there are always 12 notes per octave
    // UnitsToNoteNr is just a multiplication by a constant
    Result := NoteNumberToUnits( 16 * Log2( aValue / 2.0) + 224);
  end;


  function  UnitsToLfoBPMFrequency( const aValue: TSignal): TSignal; // inline;
  const
    OT = 1.0 / 30.0;
  begin
    Result := Normalize( UnitsToNoteNumber( aValue) * OT);
  end;


  function  LfoBPMFrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  begin
    Result := NoteNumberToUnits( aValue * 30.0);
  end;


  function  UnitsToLfoLinFrequency( const aValue: TSignal): TSignal; // inline;
  begin
    Result := aValue * 15;
  end;


  function  LfoLinFrequencyToUnits( const aValue : TSignal): TSignal; // inline;
  const
    OT = 1.0 / 15.0;
  begin
    Result := aValue * OT;
  end;


  function  FrequencyToLfoPhaseDelta( const aValue: TSignal): TSignal; // inline;
  begin
    Result := Normalize( Control_Decimation * aValue * System_Rate_Rec);
  end;


  function  UnitsToLfoFastPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoFastFrequency( aValue));
  end;


  function  UnitsToLfoMediumPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoMediumFrequency( aValue));
  end;


  function  UnitsToLfoSlowPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoSlowFrequency( aValue));
  end;


  function  UnitsToLfoFast2PhaseDelta( const aValue : TSignal): TSignal; // inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoFast2Frequency( aValue));
  end;


  function  UnitsToLfoMedium2PhaseDelta( const aValue : TSignal): TSignal; // inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoMedium2Frequency( aValue));
  end;


  function  UnitsToLfoSlow2PhaseDelta( const aValue : TSignal): TSignal; // inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoSlow2Frequency( aValue));
  end;


  function  UnitsToLfoBPMPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoBPMFrequency( aValue));
  end;


  function  UnitsToLfoLinPhaseDelta( const aValue: TSignal): TSignal; inline;
  begin
    Result := FrequencyToLfoPhaseDelta( UnitsToLfoLinFrequency( aValue))
  end;


  function  SlowTimeToSampleCount( const aValue: TSignal): TSignal; inline;
  begin
    Result := aValue * Control_Rate;
  end;


  function  UnitsToSvfFrequency( const aValue: TSignal): TSignal; inline;
  // NOTE: this is based on an SVF sampled at twice the system rate
  begin
    Result := Normalize( 2 * Sin( PI * Min( 0.25, ( UnitsToFrequency( aValue) * System_Rate_2_Rec))));
  end;


  function  UnitsToSine( const aValue: TSignal): TSignal;
  begin
    Result := Normalize( Sin( TWO_PI * aValue));
  end;


  function  UnitsToCosine( const aValue: TSignal): TSignal;
  begin
    Result := Normalize( Cos( TWO_PI * aValue));
  end;


  function  UnitsToScaledEnvTime( const aValue: TSignal): TSignal;
  const
    OTTT = 1.0 / 1023.0;
  begin
    Result := Normalize(( Power( 2, 10 * aValue) - 1 ) * OTTT);
  end;


  function  ScaledEnvTimeToUnits( const aValue: TSignal): TSignal;
  var
    p : TSignal;
  const
    OTTTI = 1023.0;
  begin
    p := aValue * OTTTI + 1;

    if p <= 0
    then Result := 0
    else Result := Log2( p) / 10;
  end;


  function  UnitsToScaledSEnvTime( const aValue: TSignal): TSignal;
  begin
    Result := Normalize( Sqr( Sin( HALF_PI * aValue)));
  end;


  function  TimeToSeconds( const aValue: string; const aDefault: TSignal): TSignal;
  var
    aParts   : TStringList;
    anHours  : Integer;
    aMinutes : Integer;
    aSeconds : Integer;
  begin
    Result    := aDefault;
    anHours   := -1;
    aMinutes  := -1;
    aSeconds  := -1;
    aParts := Explode( aValue, ':');

    try
      if aParts.Count = 2
      then begin
        aSeconds := StrToIntDef( aParts[ 1], -1);
        aMinutes := StrToIntDef( aParts[ 0], -1);
        anHours  := 0;
      end
      else if aParts.Count = 3
      then begin
        aSeconds := StrToIntDef( aParts[ 2], -1);
        aMinutes := StrToIntDef( aParts[ 1], -1);
        anHours  := StrToIntDef( aParts[ 0], -1);
      end;

      if ( aSeconds >= 0) and ( aMinutes >= 0) and ( anHours >= 0)
      then Result := aSeconds + 60 * ( aMinutes + 60 * anHours);
    finally
      aParts.DisposeOf;
    end;
  end;


  function  NoteNameToUnits( const aNoteName: string; const aDefault: TSignal): TSignal;
  // C[#][-]0
  var
    Base  : Integer;
    Sharp : Integer;
    Sign  : Integer;
    N     : Char;
    Shift : Integer;
    p     : Integer;
    T     : string;
    NN    : Integer;
  begin
    Shift :=  0;
    Sharp :=  0;
    Sign  :=  1;
    Base  := -1;

    if Length( aNoteName) > 0
    then begin
      N := UpCase( aNoteName[ 1]);

      case N of
        'C' : Base := 12     ;
        'D' : Base := 12 +  2;
        'E' : Base := 12 +  4;
        'F' : Base := 12 +  5;
        'G' : Base := 12 +  7;
        'A' : Base := 12 +  9;
        'B' : Base := 12 + 11;
        else  Base := -1;
      end;

      if ( Length( aNoteName) > 1) and ( Base > 0)
      then begin
        if aNoteName[ 2] = '#'
        then begin
          Sharp := 1;
          p     := 2;
        end
        else p := 1;
      end
      else p := 0;

      if ( Length( aNoteName) > p) and ( base > 0)
      then begin
        if aNoteName[ p + 1] = '-'
        then begin
          Sign := -1;
          p    := p + 1;
        end;
      end;

      T := Copy( aNoteName, p + 1, Length( aNoteName));

      if T[ Length( T)] = '#'
      then begin
        Sharp := 1;
        T     := Copy( T, 1, Length( T) - 1);
      end;

      Shift := StrToIntDef( T, 0);
    end;

    if Base > 0
    then begin
      NN     := Base + Sharp + Sign * Shift * 12;
      Result := NoteNumberToUnits( NN);
    end
    else Result := aDefault;
  end;


  function  UnitsToNoteName( const aValue: TSignal; Compact: Boolean = True): string;
  const
    Notes  : array[ 0 .. 11] Of Char = 'CCDDEFFGGAAB';
    Sharps : array[ 0 .. 11] Of Char = ' # #  # # # ';
  var
    i          : Integer;
    S          : string;
    aNoteValue : Integer;
  begin
    if IsNan( aValue)
    then aNoteValue := Round( MiddleNote)
    else aNoteValue := Round( UnitsToNoteNumber( aValue));

    i := MathIntMod( aNoteValue, 12);
    S := Sharps[ i];

    if   Compact
    and ( S = ' ')
    then S := '';

    Result := Format( '%s%s%d', [ Notes[ i], S, ( MathIntDiv( aNoteValue, 12)) - 1], AppLocale);
  end;


  function  UnitsToNoteName12( const aValue: TSignal): string;
  const
    Names : array[ 0 .. 11] of string = (
      'C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B'
    );
  var
    aNote : Integer;
  begin
    aNote := Round( 12 * aValue);

    if aNote in [ 0 .. 11]
    then Result := Names[ aNote]
    else Result := 'inv';
  end;


  function  IsNoteName( const aNoteName: string): Boolean;
  // C[#][-]0[#]
  var
    p : Integer;
    i : Integer;
  begin
    Result := False;

    if Length( aNoteName) > 0
    then begin
      Result := CharInSet( aNoteName[ 1], ['a' .. 'g', 'A' .. 'G']);

      if Result and ( Length( aNoteName) > 1)
      then begin
        if aNoteName[ 2] = '#'
        then p := 2
        else p := 1;
      end
      else p := 0;

      if Result and ( Length( aNoteName) > p) and ( aNoteName[ p + 1] = '-')
      then p := p + 1;

      if Result
      then begin
        for i := p + 1 to Length( aNoteName)
        do begin
          if not CharInset( aNoteName[ i], [ '0' .. '9'])
          then begin
            if ( aNoteName[ i] = '#') and ( i = Length( aNoteName))
            then Break
            else begin
              Result := False;
              Break;
            end;
          end;
        end;
      end;
    end;
  end;


  function  SignalToStr( const aValue: TSignal): string;
  begin
    Result := FloatToStrF( aValue, ffGeneral, 15, 0, AppLocale);
  end;


  function  TimeToStr( const aValue: TSignal): string;
  const
    DAY     = 24.0 * 3600.0;
    DAY_REC = 1.0 / DAY;
  var
    Days  : TSignal;
    Value : TSignal;
  begin
    if aValue = 0
    then Result := '0 s'
    else if aValue < 1e-4
    then Result := Format( '%.1f s'        , [ aValue * 1e6] , AppLocale)
    else if aValue < 1e-3
    then Result := Format( '%.0f s'        , [ aValue * 1e6] , AppLocale)
    else if aValue < 1e-2
    then Result := Format( '%.2f ms'        , [ aValue * 1e3] , AppLocale)
    else if aValue < 1e-1
    then Result := Format( '%.1f ms'        , [ aValue * 1e3] , AppLocale)
    else if aValue < 1e0
    then Result := Format( '%.0f ms'        , [ aValue * 1e3] , AppLocale)
    else if aValue < 1e1
    then Result := Format( '%.2f s'         , [ aValue      ] , AppLocale)
    else if aValue < 60
    then Result := Format( '%.1f s'         , [ aValue      ] , AppLocale)
    else if aValue < 3600
    then Result := FormatDateTime( 'nn:ss'  , aValue * DAY_REC, AppLocale)
    else if aValue < DAY
    then Result := FormatDateTime( 'h:nn:ss', aValue * DAY_REC, AppLocale)
    else begin
      Value  := FloatMod( aValue, DAY);
      Days   := aValue / DAY;
      Result := Format( '%.0f-%s', [ Days, FormatDateTime( 'hh:nn:ss', Value * DAY_REC, AppLocale)], AppLocale);
    end;
  end;


  function  FreqToStr( const aValue: TSignal): string;
  begin
    if aValue = 0
    then Result := '0 Hz'
    else if Abs( aValue) < 1e-1
    then Result := TimeToStr( 1 / aValue)
    else if Abs( aValue) < 1e0
    then Result := Format( '%.3f Hz', [ aValue], AppLocale)
    else if Abs( aValue) <1e1
    then Result := Format( '%.3f Hz', [ aValue], AppLocale)
    else if Abs( aValue) < 1e2
    then Result := Format( '%.2f Hz', [ aValue], AppLocale)
    else if Abs( aValue) < 1e3
    then Result := Format( '%.1f Hz', [ aValue], AppLocale)
    else if Abs( aValue) < 1e4
    then Result := Format( '%.2f kHz', [ aValue * 1e-3], AppLocale)
    else Result := Format( '%.1f kHz', [ aValue * 1e-3], AppLocale);
  end;


  function  MultiplierToStr( const aValue: TSignal): string;
  begin
    if aValue = 0
    then Result := '0'
    else if Abs( aValue) < 1e0
    then Result := Format( '%.4f', [ aValue], AppLocale)
    else if Abs( aValue) < 1e1
    then Result := Format( '%.3f', [ aValue], AppLocale)
    else if Abs( aValue) < 1e2
    then Result := Format( '%.2f', [ aValue], AppLocale)
    else if Abs( aValue) < 1e3
    then Result := Format( '%.1f', [ aValue], AppLocale)
    else if Abs( aValue) < 1e4
    then Result := Format( '%.2f', [ aValue], AppLocale)
    else Result := Format( '%.1f', [ aValue], AppLocale);
  end;


  function  FreqToBPMStr( const aValue: TSignal): string;
  begin
    Result := Format( '%.0f', [ aValue * 60], AppLocale);
  end;


  function  RateMultiplierToStr( const aValue: TSignal): string;
  var
    aVal : Integer;
  begin
    aVal := Round( aValue);

    if aVal < 0
    then Result := Format( '1 / %d', [ 1 - aVal])
    else Result := Format( '%d'    , [ 1 + aVal]);
  end;


  function  RateMultiplierValue( const aValue: TSignal): TSignal;
  var
    aVal : Integer;
  begin
    aVal := Round( aValue);
    if aVal < 0
    then Result := 1.0 / ( 1 - aVal)
    else Result :=         1 + aVal;
  end;


  function  SlewToCurve( const aValue: TSignal): TSignal;
  begin
    if aValue > 0.00001
    then Result := LogAttack( 0.6 + 0.4 * aValue, 1, 0, 1)
    else Result := aValue;
  end;


  procedure RegisterConverters( const aClasses: array of TKnobsConverterClass);
  begin
    LocConverters.RegisterConverters( aClasses);
  end;


  function  Converters: TKnobsConverters;
  begin
    Result := LocConverters;
  end;



const

  LutRange = 8192;          // [ 0, 1] TSignal range maps to [0        , LutRange] table range
  LutSize  = 2 * LutRange;  // [-1, 1] TSignal range maps to [-LutRange, LutRange] table range

type

  TLut = array[ - LutRange - 1 .. LutRange + 1] of TSignal;  // -/+ 1 to allow for interpolation without checking bounds on first/last table element

var

  UnitsToFrequencyTable            : TLut;
  UnitsToFrequencyLinTable         : TLut;
  UnitsToFrequencyCubedTable       : TLut;
  UnitsToPhaseDeltaTable           : TLut;
  UnitsToPhaseDeltaLinTable        : TLut;
  UnitsToSvfFreqTable              : TLut;
  UnitsToLfoSlowFrequencyTable     : TLut;
  UnitsToLfoSlowPhaseDeltaTable    : TLut;
  UnitsToLfoMediumFrequencyTable   : TLut;
  UnitsToLfoMediumPhaseDeltaTable  : TLut;
  UnitsToLfoFastFrequencyTable     : TLut;
  UnitsToLfoFastPhaseDeltaTable    : TLut;
  UnitsToLfoSlow2FrequencyTable    : TLut;
  UnitsToLfoSlow2PhaseDeltaTable   : TLut;
  UnitsToLfoMedium2FrequencyTable  : TLut;
  UnitsToLfoMedium2PhaseDeltaTable : TLut;
  UnitsToLfoFast2FrequencyTable    : TLut;
  UnitsToLfoFast2PhaseDeltaTable   : TLut;
  UnitsToLfoBPMFrequencyTable      : TLut;
  UnitsToLfoBPMPhaseDeltaTable     : TLut;
  UnitsToLfoLinFrequencyTable      : TLut;
  UnitsToLfoLinPhaseDeltaTable     : TLut;
  UnitsToSineTable                 : TLut;
  UnitsToCosineTable               : TLut;
  UnitsToScaledEnvTimeTable        : TLut;
  UnitsToScaledEnvTimeTableAr      : TLut;
  ScaledEnvTimeToUnitsTable        : TLut;
  UnitsToScaledSEnvTimeTable       : TLut;
  UnitsToScaledSEnvTimeTableAr     : TLut;
  CentsToMultiplierTable           : TLut;
  SigmoidTable                     : TLut;
  TanhTable                        : TLut;


  procedure ComputeTable( var aTable: TLut; const aFunc: TSignalFunction);
  var
    i : Integer;
  begin
    for i := Low( aTable) to High( aTable)
    do aTable[ i] := Normalize( aFunc( i / LutRange));
  end;


  function TableLookup( const aValue: TSignal; const aTable : TLut): TSignal; inline;
  var
    aLow   : Integer;
    aVal   : TSignal;
    aDelta : TSignal;
  begin
    aVal   := Clip( aValue * LutRange, Low( aTable), High( aTable) - 1);
    aLow   := Trunc( aVal);
    aDelta := aVal - aLow;
    Result := aTable[ aLow];
    Result := Result + aDelta * ( aTable[ aLow + 1] - Result);
  end;


  procedure ComputeTables;
  begin
    ComputeTable( UnitsToFrequencyTable           , UnitsToFrequency           );
    ComputeTable( UnitsToFrequencyLinTable        , UnitsToFrequencyLin        );
    ComputeTable( UnitsToFrequencyCubedTable      , UnitsToFrequencyCubed      );
    ComputeTable( UnitsToPhaseDeltaTable          , UnitsToPhaseDelta          );
    ComputeTable( UnitsToPhaseDeltaLinTable       , UnitsToPhaseDeltaLin       );
    ComputeTable( UnitsToSvfFreqTable             , UnitsToSvfFrequency        );
    ComputeTable( UnitsToLfoSlowFrequencyTable    , UnitsToLfoSlowFrequency    );
    ComputeTable( UnitsToLfoSlowPhaseDeltaTable   , UnitsToLfoSlowPhaseDelta   );
    ComputeTable( UnitsToLfoMediumFrequencyTable  , UnitsToLfoMediumFrequency  );
    ComputeTable( UnitsToLfoMediumPhaseDeltaTable , UnitsToLfoMediumPhaseDelta );
    ComputeTable( UnitsToLfoFastFrequencyTable    , UnitsToLfoFastFrequency    );
    ComputeTable( UnitsToLfoFastPhaseDeltaTable   , UnitsToLfoFastPhaseDelta   );
    ComputeTable( UnitsToLfoSlow2FrequencyTable   , UnitsToLfoSlow2Frequency   );
    ComputeTable( UnitsToLfoSlow2PhaseDeltaTable  , UnitsToLfoSlow2PhaseDelta  );
    ComputeTable( UnitsToLfoMedium2FrequencyTable , UnitsToLfoMedium2Frequency );
    ComputeTable( UnitsToLfoMedium2PhaseDeltaTable, UnitsToLfoMedium2PhaseDelta);
    ComputeTable( UnitsToLfoFast2FrequencyTable   , UnitsToLfoFast2Frequency   );
    ComputeTable( UnitsToLfoFast2PhaseDeltaTable  , UnitsToLfoFast2PhaseDelta  );
    ComputeTable( UnitsToLfoBPMFrequencyTable     , UnitsToLfoBPMFrequency     );
    ComputeTable( UnitsToLfoBPMPhaseDeltaTable    , UnitsToLfoBPMPhaseDelta    );
    ComputeTable( UnitsToLfoLinFrequencyTable     , UnitsToLfoLinFrequency     );
    ComputeTable( UnitsToLfoLinPhaseDeltaTable    , UnitsToLfoLinPhaseDelta    );
    ComputeTable( UnitsToSineTable                , UnitsToSine                );
    ComputeTable( UnitsToCosineTable              , UnitsToCosine              );
    ComputeTable( UnitsToScaledEnvTimeTable       , UnitsToScaledEnvTime       );
    ComputeTable( ScaledEnvTimeToUnitsTable       , ScaledEnvTimeToUnits       );
    ComputeTable( UnitsToScaledSEnvTimeTable      , UnitsToScaledSEnvTime      );
    ComputeTable( CentsToMultiplierTable          , CentsToMultiplierScaled    );
    ComputeTable( SigmoidTable                    , Sigmoid                    );
    ComputeTable( TanhTable                       , UnitsToTanh                );
  end;


  function  LookupFrequency           ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToFrequencyTable           ); end;
  function  LookupFrequencyLin        ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToFrequencyLinTable        ); end;
  function  LookupFrequencyCubed      ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToFrequencyCubedTable      ); end;
  function  LookupPhaseDelta          ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToPhaseDeltaTable          ); end;
  function  LookupPhaseDeltaLin       ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToPhaseDeltaLinTable       ); end;
  function  LookupSvfFrequency        ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToSvfFreqTable             ); end;
  function  LookupLfoSlowFrequency    ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoSlowFrequencyTable    ); end;
  function  LookupLfoSlowPhaseDelta   ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoSlowPhaseDeltaTable   ); end;
  function  LookupLfoMediumFrequency  ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoMediumFrequencyTable  ); end;
  function  LookupLfoMediumPhaseDelta ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoMediumPhaseDeltaTable ); end;
  function  LookupLfoFastFrequency    ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoFastFrequencyTable    ); end;
  function  LookupLfoFastPhaseDelta   ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoFastPhaseDeltaTable   ); end;
  function  LookupLfoSlow2Frequency   ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoSlow2FrequencyTable   ); end;
  function  LookupLfoSlow2PhaseDelta  ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoSlow2PhaseDeltaTable  ); end;
  function  LookupLfoMedium2Frequency ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoMedium2FrequencyTable ); end;
  function  LookupLfoMedium2PhaseDelta( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoMedium2PhaseDeltaTable); end;
  function  LookupLfoFast2Frequency   ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoFast2FrequencyTable   ); end;
  function  LookupLfoFast2PhaseDelta  ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoFast2PhaseDeltaTable  ); end;
  function  LookupLfoBPMFrequency     ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoBPMFrequencyTable     ); end;
  function  LookupLfoBPMPhaseDelta    ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoBPMPhaseDeltaTable    ); end;
  function  LookupLfoLinFrequency     ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoLinFrequencyTable     ); end;
  function  LookupLfoLinPhaseDelta    ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToLfoLinPhaseDeltaTable    ); end;
  function  LookupSine                ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToSineTable                ); end;
  function  LookupCosine              ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToCosineTable              ); end;
  function  LookupScaledEnvTime       ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToScaledEnvTimeTable       ); end;
  function  LookupScaledEnvTimeUnits  ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, ScaledEnvTimeToUnitsTable       ); end;
  function  LookupScaledSEnvTime      ( const aValue: TSignal): TSignal; begin Result := TableLookup( aValue, UnitsToScaledSEnvTimeTable      ); end;


  function  LookupEnvTime( aRange: TEnvRange; aValue: TSignal): TSignal;
  var
    p         : TSignal;
    anOutLow  : TSignal;
    anOutHigh : TSignal;
  begin
    case aRange of
      erSlow   : begin anOutLow := MinEnvTimeSlow  ; anOutHigh := MaxEnvTimeSlow  ; end;
      erMedium : begin anOutLow := MinEnvTimeMedium; anOutHigh := MaxEnvTimeMedium; end;
      else       begin anOutLow := MinEnvTimeFast  ; anOutHigh := MaxEnvTimeFast  ; end;
    end;
    p      := LookupScaledEnvTime( Clip( aValue, 0, 1));
    Result := anOutLow + p * ( anOutHigh - anOutLow);
  end;


  function  LookupEnvTimeAr( aRange: TEnvRange; aValue: TSignal): TSignal;
  begin
    Result := LookupEnvTime( aRange, aValue) / Control_Decimation;
  end;


  function  LookupCentsMultiplier( const aValue: TSignal): TSignal;
  begin
    Result := TableLookup( Clip( aValue, - CENTS, CENTS) * CENTS_REC, CentsToMultiplierTable);
  end;


  function  LookupSigmoid( const aValue: TSignal): TSignal;
  begin
    Result := TableLookup( Clip( aValue, -1, 1), SigmoidTable);
  end;


  function  LookupTanh( const aValue: TSignal): TSignal;
  begin
    Result := TableLookup( Clip( aValue / 8, -1, 1), TanhTable);
  end;


  function  MidiFilterToStr( const aValue: TMidiFilter): string;
  var
    i : TMidiSeptet;
  begin
    Result := '{ ';

    for i := Low( TMidiSeptet) to High( TMidiSeptet)
    do begin
      if i in aValue
      then Result := Result + Format( '%d ', [ i], AppLocale);
    end;

    Result := Result + '}';
  end;


  procedure SetSystemRate( aValue: Integer);
  begin
    System_Rate          := aValue;                           // samples / s (Hz)
    System_Rate_2        := System_Rate * 2;                  // samples / s (Hz)
    Control_Rate         := System_Rate / Control_Decimation; // samples / s (Hz)
    Control_Dec_Min_One  := Control_Decimation - 1;           // a factor
    Control_Dec_2        := Control_Decimation * 2;           // a factor
    Control_Dec_2_p1     := Control_Dec_2 + 1;                // a factor
    System_Rate_Rec      := 1 / System_Rate;                  // s / sample (  s)
    System_Rate_2_Rec    := 1 / System_Rate_2;                // s / sample (  s)
    Control_Rate_Rec     := 1 / Control_Rate;                 // s / sample (  s)
    MorphDecimation      := Control_Rate / MORPH_RATE;        // a factor
    AutomationDecimation := Control_Rate / AUTOAMTION_RATE;   // a factor
    FlashCount           := Ceil( System_Rate  * FLASH_TIME );
    FlashCountContr      := Ceil( Control_Rate * FLASH_TIME );
    PulseCount           := Ceil( System_Rate  * TRIG_TIME  );
    PulseCountContr      := Ceil( Control_Rate * TRIG_TIME  );
    NeuronCount          := Ceil( System_Rate  * NEURON_TIME);
    NeuronCountContr     := Ceil( Control_Rate * NEURON_TIME);
  end;


  procedure SetSystemControlDecimation( aValue: Integer);
  begin
    Control_Decimation := aValue;
    SetSystemRate( System_Rate);                             // Recalculate dependencies
  end;


  function  SignalToFractional( aValue: TSignal): Integer; inline;
  begin
    Result := Round( FRACT_SCALING * aValue);
  end;


  function  FractionalToSignal( aValue: Integer): TSignal; inline;
  begin
    Result := FRACT_SCALING_REC * aValue;
  end;


  function  SignalPairArrayToStr( const aValue: TSignalPairArray): string;
  var
    i : Integer;
  begin
    Result := '';

    for i := 0 to Length( aValue) - 1
    do begin
      if Result = ''
      then Result := Format(     '%g, %g', [         aValue[ i].Left, aValue[ i].Right], AppLocale)
      else Result := Format( '%s; %g, %g', [ Result, aValue[ i].Left, aValue[ i].Right], AppLocale);
    end;
  end;


  function  StrToSignalPairArray( const aValue: string; DoFixup: Boolean; aLowX, aHiX: TSignal): TSignalPairArray;

    procedure AddPair( const aData: TSignalPair);
    var
      i : Integer;
      p : Integer;
    begin
      if DoFixup
      then begin
        p := Length( Result);

        for i := 0 to Length( Result) - 2
        do begin
          if ( aData.Left > Result[ i].Left) and ( aData.Left < Result[ i + 1].Left)
          then begin
            p := i + 1;
            Break;
          end;
        end;
        SetLength( Result, Length( Result) + 1);

        for i := Length( Result) - 2 downto p
        do Result[ i + 1] := Result[ i];
      end
      else begin
        SetLength( Result, Length( Result) + 1);
        p := Length( Result) - 1;
      end;

      Result[ p].Left  := aData.Left;
      Result[ p].Right := aData.Right;
    end;

    procedure AddData( anX, anY: TSignal);
    begin
      AddPair( SignalPair( anX, anY));
    end;

    procedure AddStr( const aValue: string);
    var
      aPair : TStringList;
      anX   : TSignal;
      anY   : TSignal;
    begin
      aPair := Explode( aValue, ',');

      try
        if aPair.Count = 2
        then begin
          anX := StrToFloatDef( Trim( aPair[ 0]), Nan);
          anY := StrToFloatDef( Trim( aPair[ 1]), Nan);

          if ( not IsNan( anX)) and ( not IsNan( anY))
          then AddData( anX, anY);
        end;
      finally
        aPair.DisposeOf;
      end;
    end;

  var
    aPairs : TStringList;
    aPair  : string;
  begin
    SetLength( Result, 0);
    aPairs := Explode( aValue, ';');

    if DoFixup
    then begin
      if aPairs.Count = 0
      then aPairs.Add( Format( '%g, 0,0', [ aLowX], AppLocale));

      if aPairs.Count = 1
      then aPairs.Add( Format( '%g, 0,0', [ aHiX], AppLocale));
    end;

    try
      for aPair in aPairs
      do AddStr( Trim( aPair));

      if DoFixup
      then begin
        Result[ 0                  ].X := aLowX;
        Result[ Length( Result) - 1].X := aHiX;
      end;
    finally
      aPairs.DisposeOf;
    end;
  end;


  function  InterpolateSignalPairArray( const aData: TSignalPairArray; anX: TSignal): TSignal;
  //
  // This is linear interpolation only, for splined stuff see the BSplines unit.
  //
  // aData must be sorted such that:
  //
  //   for all i in [ 0, Length( aData) - 2] :: aData[ i].Left <= aData[ i + 1].Left
  //
  // or otherwise dX may become <= 0 which can not be handled with the current ccode.

    function FindLarger: Integer;
    var
      i : Integer;
    begin
      Result := Length( aData);

      for i := 0 to Length( aData) - 1
      do begin
        if aData[ i].X >= anX
        then begin
          Result := i;
          Break;
        end;
      end;
    end;

  var
    p  : Integer;
    dY : TSignal;
    dX : TSignal;
  begin
    p := FindLarger;

    if p < 0                                         // No data in aData
    then Result := 0.0
    else if p = 0                                    // anX < leftmost value in aDaata, use leftmost
    then Result := aData[ 0].Y
    else if p >= Length( aData)                      // anX >= rightmost value in aDat, use rightmost
    then Result := aData[ p - 1].Y
    else begin                                       // anX somewhere beteen points p - 1 and p,
                                                     // use linear interpolation in that segment
      dX := aData[ p].X - aData[ p - 1].X;           // Can never be <= 0 ::: !! assuming aData is sorted !!
      dY := aData[ p].Y - aData[ p - 1].Y;
      Result := aData[ p - 1].Y + ( anX - aData[ p - 1].X ) * ( dY / dX);
    end;
  end;


  function  RandomDiv( aRange, aDivision: Integer): Integer;
  var
    Range : Integer;
    i     : Integer;
  begin
    if aDivision > 0
    then begin
      Range := Ceil( LogN( aDivision, aRange + 1));

      for i := 0 to Random( Range)
      do aRange := aRange div aDivision;

      Result := aRange;
    end
    else Result := Random( aRange + 1);
  end;


  function  BytesToStr( const aValue : TBytes): string;
  var
    i : Integer;
  begin
    Result := '';

    for i := 0 to Length( aValue) - 1
    do begin
      if i = 0
      then Result := Format(    '%d', [         aValue[ i]], AppLocale)
      else Result := Format( '%s,%d', [ Result, aValue[ i]], AppLocale);
    end;
  end;


  function  StrToBytes( const aValue: string): TBytes;
  var
    aParts : TStringList;
    i      : Integer;
  begin
    aParts := Explode( aValue, ',');

    try
      if aValue = ''
      then SetLength( Result, 0)
      else begin
        SetLength( Result, aParts.Count);

        for i := 0 to aParts.Count - 1
        do Result[ i] := StrToIntDef( Format( '$%s', [ aParts[ i]], AppLocale), 0);
      end;
    finally
      aParts.DisposeOf;
    end;
  end;


  function  SignalArrayToStr( const aValue : TSignalArray): string;
  var
    i : Integer;
  begin
    Result := '';

    for i := 0 to Length( aValue) - 1
    do begin
      if i = 0
      then Result := Format(    '%g', [         aValue[ i]], AppLocale)
      else Result := Format( '%s,%g', [ Result, aValue[ i]], AppLocale);
    end;
  end;


  function  StrToSignalArray( const aValue: string): TSignalArray;
  var
    aParts : TStringList;
    i      : Integer;
  begin
    aParts := Explode( aValue, ',');

    try
      if aValue = ''
      then SetLength( Result, 0)
      else begin
        SetLength( Result, aParts.Count);

        for i := 0 to aParts.Count - 1
        do Result[ i] := StrToFloatDef( aParts[ i], 0);
      end;
    finally
      aParts.DisposeOf;
    end;
  end;


  function  LedFlashTime( IsFast: Boolean): Integer; inline;
  begin
    if IsFast
    then Result := FlashCount
    else Result := FlashCountContr
  end;


  function  TrigPulseTime( IsFast: Boolean): Integer; inline;
  begin
    if IsFast
    then Result := PulseCount
    else Result := PulseCountContr;
  end;


  function  NeuronTime( IsFast: Boolean): Integer; inline;
  begin
    if IsFast
    then Result := NeuronCount
    else Result := NeuronCountContr;
  end;


  function  KnobRangeLin( const aPos, aStepCount: Integer; anOutLow, anOutHigh: TSignal): TSignal; inline;
  // NOTE : aStepCount needs to actually be StepCount - 1
  begin
    if aStepCount <= 0
    then Result := anOutLow
    else Result := anOutLow + ( aPos / aStepCount) * ( anOutHigh - anOutLow);
  end;


  function  KnobRangeLinInv( aVal: TSignal; const aStepCount: Integer; anInLow, anInHigh: TSignal): Integer; inline;
  // NOTE : aStepCount needs to actually be StepCount - 1
  begin
    if anInlow > anInHigh
    then begin
      if ( aStepCount <= 0) or ( aVal > anInLow)
      then Result := aStepCount
      else if aVal > anInLow
      then Result := 0
      else Result := Round( aStepCount * ( aVal / ( anInHigh - anInLow) - anInLow));
    end
    else begin
      if ( aStepCount <= 0) or ( aVal < anInLow) or ( anInLow = anInHigh)
      then Result := 0
      else if aVal > anInHigh
      then Result := aStepCount
      else Result := Round( aStepCount * ( aVal / ( anInHigh - anInLow) - anInLow));
    end;
  end;


  function  KnobRangeExp( const aPos, aStepCount: Integer; anOutLow, anOutHigh: TSignal): TSignal; inline;
  var
    p : TSignal;
  begin
    if aStepCount <= 0
    then Result := anOutLow
    else begin
      p      := LookupScaledEnvTime( aPos / aStepCount);
      Result := anOutLow + p * ( anOutHigh - anOutLow);
    end;

    if Abs( Result) < MUTE_LIMIT
    then Result := 0;
  end;


  function  KnobRangeExpInv( aVal: TSignal; const aStepCount: Integer; anInLow, anInHigh: TSignal): Integer; inline;
  var
    p : TSignal;
  begin
    if ( aStepCount <= 0) or ( aVal < anInLow) or ( anInLow = anInHigh)
    then Result := 0
    else if aVal > anInHigh
    then Result := aStepCount
    else begin
      p      := Round( aStepCount * ( aVal / ( anInHigh - anInLow) - anInLow));
      Result := Round( LookupScaledEnvTimeUnits( p));
    end;
  end;


type

  TConvUndefined = class( TKnobsNullConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToHint   ( aPos, aSteps: Integer): string;                                                  override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TKnobsBaseConverter = class( TKnobsNullConverter)
  public
    function    PosToHint   ( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLinearMin1_05 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinear0_1 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinear0_4 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinear0_16 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinear1_4 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLambda = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinearMin1_1 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinearMin1_0 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinearMin4_4 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinearMin8_8 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLinearMin16_16 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvAdd0 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvAdd1 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvAdd3 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvOdd = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvQSteps = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvVSteps = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvVStepsOff = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvFractions = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdB0_1 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdB0_4 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdB0_16 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdB0_32 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdB0_64 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdBKS = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvdBResonator = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvOscFreqCoarse = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvOscLinCoarse = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvOscCubedCoarse = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseFast = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseMedium = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseSlow = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseFast2 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseMedium2 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseSlow2 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseBPM = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvLfoFreqCoarseLin = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvNoiseColor = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeFast = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeMedium = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeSlow = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeFastAr = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeMediumAr = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnvTimeSlowAr = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvMidiChIn = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvMidiChOut = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvNoteName = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvNoteName12 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;

  TConvDelayTime = class( TKnobsBaseConverter)
  public
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDelayTimeShort = class( TConvDelayTime)
  public
    constructor Create;                                                                                        override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvDelayTimeMedium = class( TConvDelayTime)
  public
    constructor Create;                                                                                        override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvDelayTimeLong = class( TConvDelayTime)
  public
    constructor Create;                                                                                        override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvFilterFreq = class( TConvOscFreqCoarse)
  public
    constructor Create;                                                                                        override;
  end;


  TConvAvgFilterAlpha = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvCubicClipAlpha = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvBlockDCAlpha = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvSlew = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvCents = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvSemitones = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvSemitonesFull = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvTranspose = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvEnumerable = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvMute = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvActivation = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSolo = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEnvMute = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvVcpsMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvOutType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEnvOutType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEnvSlopeType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvWaveShape = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPulseMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPulseDelayMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TTalkieFrameRate = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TTalkieSampleRate = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TRoomSize = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TReverbTime = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TRateMultiplier = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TSwing = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TSongSpeed = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  THzMeasure = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TPhase = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TFreqMult = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvNormalHold = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMixerModeSimple = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMixerModeAll = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMixerModeExtended = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMixerModeExtendedAmp = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSequencerMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLfoRange = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEnvelopeRange = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDelayRange = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvNoteStealing = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAutoMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvHrastOscType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvNoiseType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAttractorType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvInterpolationType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;

  TConvOffOnType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;

  TConvTrigType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;

  TConvLinExpType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMirrorType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSquareMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvFaderType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSkipMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPeakRMS = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPerlinMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvFIMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLoopMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRPNType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRPNUSel = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMonoStereo = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSawPulse = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAsyncSync = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRandomWalkMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSteppedSmooth = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLFORetrig = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvGateMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRectifierMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPinkMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;

  TConv7kHzMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConv28HzMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConv5k6HzMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConv139HzMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDemuxHoldMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDigiDemuxHoldMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvChebgenMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvQuantizeMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvQuantizeType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvGateFunction = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPatternMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvChordDegree = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvChordType = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvOctaveShift = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvTransInv = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRSTrigMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvCounterMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvTSRandomMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRepOnceMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvNormInvertMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPostPreMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSeq16GateMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvChordMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvRunEditMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvArpMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvArpInMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvOnceContMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvWrappedPlainMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvWrappedPlainMode2 = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvNormalFrozenMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvCompressorBypassMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSideChainMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAcDcMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPhaseDistortMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvClockedFastMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvValueMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAbsPosMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvFastSlowMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvGraphMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvClipWrapMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvWaveShapeLFO = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLinear1_8 = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvMConst = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMonadicFunction = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDyadicFunction = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLifeSeqMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvLifeSeqSumMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvTerragenAlgorithm = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvConvoderSize = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvTapperMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPhaseShift = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvPWMPLM = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvZNZ = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvMuseSwitch = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvFreeClocked = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEvenOdd = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvDupsMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvScaledMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvFmDepth = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvSinCosSin = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvNPMode = class( TConvEnumerable)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvSampleRate = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvContrRateDecimation = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAutoReady = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAbsNorm = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvAllSkipMode = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEspeakModifiers = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TConvEspeakPunctuation = class( TConvEnumerable)
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
  end;


  TESpeakRate = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TESpeakPitch = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TESpeakRange = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TESpeakWordGap = class( TKnobsBaseConverter)
  public
    constructor Create;                                                                                        override;
    function    PosToDisplay( aPos, aSteps: Integer): string;                                                  override;
    function    PosToValue  ( aPos, aSteps: Integer): TSignal;                                                 override;
    function    ValueToPos  ( aVal: TSignal; aSteps: Integer): Integer;                                        override;
  end;


  TConvDynTalkieBank = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynFormant2Vowels = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynModal2 = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynScale = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynGrainEnvelopes = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynMorphSelect = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvDynMazeType = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvSapiSpeechVoices = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvSapiSpeechRates = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvESpeakSpeechVoices = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


  TConvESpeakSpeechLanguages = class( TKnobsDynamicConverter)
  public
    constructor Create;                                                                                        override;
  end;


{ ========
  TKnobsNullConverter = class
  private
    FSearched       : string;
    FDefaultUnits   : string;
    FIsRandomizable : Boolean;
    FIsEnumerable   : Boolean;
  protected
    FName           : string;
    FFRiendlyName   : string;
  public
    property    Name           : string  read FName;
    property    FriendlyName   : string  read FFriendlyName;
    property    DefaultUnits   : string  read FDefaultUnits;
    property    Searched       : string  read FSearched       write FSearched;
    property    IsRandomizable : Boolean read FIsRandomizable write FIsRandomizable;
    property    IsEnumerable   : Boolean read FIsEnumerable;
  private
}


    function    TKnobsNullConverter.ApplyUnits( const aUnits: string; var aValue: TSignal): Boolean; // virtual;
    const
      OS  = 1.0 / 60.0;
      OTF = 1.0 / 256.0;
      OTS = 1.0 / 360.0;
    var
      Units : string;
    begin
      Result := True;

      if not SameText( DefaultUnits, NO_UNITS)
      then begin
        if aUnits = ''
        then Units := DefaultUnits
        else Units := aUnits;

        if      SameText( Units, 'Hz')
        then aValue := FrequencyToUnits( aValue)
        else if SameText( Units, 'kHz')
        then aValue := FrequencyToUnits( 1000 * aValue)
        else if SameText( Units, 'mHz')
        then aValue := FrequencyToUnits( aValue * 1e-3)
        else if SameText( Units, 'bpm')
        then aValue := FrequencyToUnits( aValue * OS)
        else if SameText( Units, 'degrees') or SameText( aUnits, 'deg')
        then aValue := aValue * OTS
        else if SameText( Units, 'radians') or SameText( aUnits, 'rad')
        then aValue := aValue * TWO_PI_REC
        else if SameText( Units, 'percent') or SameText( aUnits, '%')
        then aValue := aValue * 1e-2
        else if SameText( Units, 'ms')
        then aValue := aValue * 1e-3
        else if SameText( Units, 's')
        then aValue := aValue * 1e-6
        else if SameText( Units, 's')
        then // nothing
        else if SameText( Units, 'nn')
        then aValue := NoteNumberToUnits( aValue)
        else if SameText( Units, 'qsteps') or SameText( aUnits, 'q')
        then aValue := aValue * 1e-3
        else if SameText( Units, 'ct')
        then aValue := aValue * OTF
        else Result := False;
      end;
    end;


    function    TKnobsNullConverter.StringToValue( const aStringValue: string; var aValue: TSignal): Boolean; // virtual;
    begin
      Result := True;

      if IsNoteName( aStringValue)
      then aValue := NoteNameToUnits( aStringValue, UNDEFINED)
      else if IsTimeValue( aStringValue)
      then aValue := 1 / TimeToSeconds( aStringValue, UNDEFINED)
      else if SameText( aStringValue, 'on')
      then aValue := 1.0
      else if SameText( aStringValue, 'off')
      then aValue := 0.0
      else Result := TextToFloat( aStringValue, aValue, AppLocale);

      if not Result
      then aValue := UNDEFINED;
    end;


    function    TKnobsNullConverter.PostProcess( const aValue: TSignal): TSignal; // virtual;
    begin
      Result := aValue;
    end;


//  public

    constructor TKnobsNullConverter.Create; // virtual;
    begin
      FName           := 'default';
      FSearched       := '';
      FDefaultUnits   := NO_UNITS;
      FIsRandomizable := True;
      FIsEnumerable   := False;
    end;


    function    TKnobsNullConverter.PosToHint( aPos, aSteps: Integer): string; // virtual;
    begin
      Result := IntToStr( aPos);
    end;


    function    TKnobsNullConverter.PosToDisplay( aPos, aSteps: Integer): string; // virtual;
    begin
      Result := IntToStr( aPos);
    end;


    function    TKnobsNullConverter.PosToValue( aPos, aSteps: Integer): TSignal; // virtual;
    begin
      Result := aPos;
    end;


    function    TKnobsNullConverter.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // virtual;
    begin
      Result := Round( aVal);
    end;


    function    TKnobsNullConverter.DisplayToValue( const aDisplay: string; var aValue: TSignal): Boolean;
    var
      aParts : TStringList;
    begin
      aValue := UNDEFINED;
      Result := False;
      aParts := Explode( aDisplay, ' ');

      try
        if aParts.Count > 0
        then Result := StringToValue( Trim( aParts[ 0]), aValue);

        if Result and ( aParts.Count > 1)
        then Result := ApplyUnits( Trim( aParts[ 1]), aValue);

        if Result
        then aValue := PostProcess( aValue);
      finally
        aParts.DisposeOf;
      end;
    end;


    function    TKnobsNullConverter.ValueToDisplay( const aValue: TSignal; aSteps: Integer): string;
    begin
      Result := PosToDisplay( ValueToPos( aValue, aSteps), aSteps);
    end;


    function    TKnobsNullConverter.CreateEnumeration: TStringList; // virtual;
    var
      S : string;
      p : Integer;
    begin
      Result := nil;

      if FIsEnumerable
      then begin
        Result               := TStringList.Create;
        Result.Sorted        := False;
        Result.Duplicates    := dupAccept;
        Result.CaseSensitive := false;

        p := 0;
        S := PosToDisplay( p, 0);

        while not SameText( S, S_UNDEFINED)
        do begin
          Result.Add( S);
          Inc( p);
          S := PosToDisplay( p, 0);

          if p > MAX_CONV_ENUMS
          then raise EConvTooManyEnums.Create( 'Too many items in converter enumeration');
        end;
      end;
    end;


{ ========
  TKnobsDynamicConverter = class( TKnobsNullConverter)
  private
    FCaptions : TStringList;
  public
    property    Captions : TStringList read FCaptions write SetCaptions;
  private
}

    procedure   TKnobsDynamicConverter.SetCaptions( const aValue: TStringList);
    begin
      if   Assigned( FCaptions)
      and  ( FCaptions.Count = 0)
      then FCaptions.Assign( aValue);
    end;


//  public

    constructor TKnobsDynamicConverter.Create; // override;
    begin
      inherited;
      FName         := 'dynamic converter';
      FIsEnumerable := True;
      FCaptions     := TStringList.Create;
    end;


    destructor  TKnobsDynamicConverter.Destroy; // override;
    begin
      FreeAndNil( FCaptions);
      inherited;
    end;


    function    TKnobsDynamicConverter.PosToHint( aPos, aSteps: Integer): string; // override;
    begin
      Result :=
        Format
        (
          '%s'^M' value: %s'^M'position: %d / %d',
          [
            Searched                   ,
            PosToDisplay( aPos, aSteps),
            aPos,
            aSteps - 1
          ],
          AppLocale
        );
    end;


    function    TKnobsDynamicConverter.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      if Assigned( FCaptions)
      then begin
        if   ( aPos >= 0)
        and  Assigned( FCaptions)
        and  ( aPos < FCaptions.Count)
        then Result := FCaptions[ aPos]
        else Result := S_UNDEFINED
      end
      else Result := S_UNDEFINED;
    end;


    function    TKnobsDynamicConverter.DisplayToValue( const aDisplay: string; var aValue: TSignal): Boolean; // override;
    var
      p : Integer;
    begin
      aValue  := 0;
      Result  := False;
      p       := -1;

      if Assigned( FCaptions)
      then p := FCaptions.IndexOf( aDisplay);

      if p >= 0
      then begin
        aValue := p;
        Result := True;
      end;
    end;


{ ========
  TKnobsConverterUndefined = class( TKnobsNullConverter)
  public
}

    constructor TConvUndefined.Create; // override;
    begin
      inherited;
      FName := 'undefined converter';
    end;


    function    TConvUndefined.PosToHint( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%s'^M'%s'^M'%f', [ Searched, inherited, PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvUndefined.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%s %s [%f]', [ Searched, inherited, PosToValue( aPos, aSteps)], AppLocale);
    end;


{ ========
  TKnobsBaseConverter = class( TKnobsNullConverter)
  public
}

    function    TKnobsBaseConverter.PosToHint( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( ' value: %s '^M' position: %d / %d ', [ PosToDisplay( aPos, aSteps), aPos, aSteps - 1], AppLocale);
    end;


{ ========
  TConvLinearMin1_05 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin1_05.Create; // override;
    begin
      inherited;
      FName := 'Linear[-1,0.5]';
    end;


    function    TConvLinearMin1_05.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin1_05.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -1, 0.5);
    end;


    function    TConvLinearMin1_05.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -1, 0.5);
    end;





{ ========
  TConvLinear0_1 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinear0_1.Create; // override;
    begin
      inherited;
      FName := 'Linear[0,1]';
    end;


    function    TConvLinear0_1.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinear0_1.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvLinear0_1.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvLinear0_4 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinear0_4.Create; // override;
    begin
      inherited;
      FName := 'Linear[0,4]';
    end;


    function    TConvLinear0_4.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinear0_4.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 4);
    end;


    function    TConvLinear0_4.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 4);
    end;


{ ========
  TConvLinear0_16 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinear0_16.Create; // override;
    begin
      inherited;
      FName := 'Linear[0,16]';
    end;


    function    TConvLinear0_16.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinear0_16.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 16);
    end;


    function    TConvLinear0_16.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 16);
    end;


{ ========
  TConvLinear1_4 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinear1_4.Create; // override;
    begin
      inherited;
      FName := 'Linear[1,4]';
    end;


    function    TConvLinear1_4.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinear1_4.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 1, 4);
    end;


    function    TConvLinear1_4.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 1, 4);
    end;


{ ========
  TConvLambda = class( TKnobsBaseConverter)
  public
}

    constructor TConvLambda.Create; // override;
    begin
      inherited;
      FName := 'Lambda';
    end;


    function    TConvLambda.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLambda.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 5, 1);
    end;


    function    TConvLambda.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 5, 1);
    end;


{ ========
  TConvLinearMin1_1 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin1_1.Create; // override;
    begin
      inherited;
      FName := 'Linear[-1,1]';
    end;


    function    TConvLinearMin1_1.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin1_1.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -1, 1);
    end;


    function    TConvLinearMin1_1.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -1, 1);
    end;



{ ========
  TConvLinearMin1_0 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin1_0.Create; // override;
    begin
      inherited;
      FName := 'Linear[-1,0]';
    end;


    function    TConvLinearMin1_0.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin1_0.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -1, 0);
    end;


    function    TConvLinearMin1_0.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -1, 0);
    end;


{ ========
  TConvLinearMin4_4 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin4_4.Create; // override;
    begin
      inherited;
      FName := 'Linear[-4,4]';
    end;


    function    TConvLinearMin4_4.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin4_4.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -4, 4);
    end;


    function    TConvLinearMin4_4.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -4, 4);
    end;


{ ========
  TConvLinearMin8_8 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin8_8.Create; // override;
    begin
      inherited;
      FName := 'Linear[-8,8]';
    end;


    function    TConvLinearMin8_8.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin8_8.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -8, 8);
    end;


    function    TConvLinearMin8_8.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -8, 8);
    end;


{ ========
  TConvLinearMin16_16 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinearMin16_16.Create; // override;
    begin
      inherited;
      FName := 'Linear[-16,16]';
    end;


    function    TConvLinearMin16_16.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinearMin16_16.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -16, 16);
    end;


    function    TConvLinearMin16_16.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -16, 16);
    end;


{ ========
  TConvAdd0 = class( TKnobsBaseConverter)
  public
}

    constructor TConvAdd0.Create; // override;
    begin
      inherited;
      FName := 'Add0';
    end;


    function    TConvAdd0.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvAdd0.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos;
    end;


    function    TConvAdd0.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal);
    end;


{ ========
  TConvAdd1 = class( TKnobsBaseConverter)
  public
}

    constructor TConvAdd1.Create; // override;
    begin
      inherited;
      FName := 'Add1';
    end;


    function    TConvAdd1.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvAdd1.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 1;
    end;


    function    TConvAdd1.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 1);
    end;


{ ========
  TConvAdd3 = class( TKnobsBaseConverter)
  public
}

    constructor TConvAdd3.Create; // override;
    begin
      inherited;
      FName := 'Add3';
    end;


    function    TConvAdd3.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvAdd3.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 3;
    end;


    function    TConvAdd3.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 3);
    end;


{ ========
  TConvOdd = class( TKnobsBaseConverter)
  public
}

    constructor TConvOdd.Create; // override;
    begin
      inherited;
      FName := 'Odd';
    end;


    function    TConvOdd.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := IntToStr( 2 * aPos + 1);
    end;


    function    TConvOdd.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := 2 * aPos + 1;
    end;


    function    TConvOdd.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round(( aVal - 1) / 2);
    end;


{ ========
  TConvQSteps = class( TKnobsBaseConverter)
  public
}

    constructor TConvQSteps.Create; // override;
    begin
      inherited;
      FName := 'QSteps';
    end;


    function    TConvQSteps.PosToDisplay( aPos, aSteps: Integer): string; // override;
    const
      OSF = 1.0 / 64.0;
    var
      R : TRational;
    begin
      R := PosToValue( aPos, aSteps) * OSF;
      R.Invert;
      Result := Format( '%s', [ R.AsString], AppLocale);
    end;


    function    TConvQSteps.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 1;
    end;


    function    TConvQSteps.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 1);
    end;


{ ========
  TConvVSteps = class( TKnobsBaseConverter)
  public
}

    constructor TConvVSteps.Create; // override;
    begin
      inherited;
      FName := 'VSteps';
    end;


    function    TConvVSteps.PosToDisplay( aPos, aSteps: Integer): string; // override;
    const
      OSF = 1.0 / 8.0;
    var
      R : TRational;
    begin
      R := PosToValue( aPos, aSteps) * OSF;
      R.Invert;
      Result := Format( '%s', [ R.AsString], AppLocale);
    end;


    function    TConvVSteps.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 1;
    end;


    function    TConvVSteps.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 1);
    end;


{ ========
  TConvVStepsOff = class( TKnobsBaseConverter)
  public
}

    constructor TConvVStepsOff.Create; // override;
    begin
      inherited;
      FName := 'VStepsOff';
    end;


    function    TConvVStepsOff.PosToDisplay( aPos, aSteps: Integer): string; // override;
    const
      OSF = 1.0 / 8.0;
    var
      R : TRational;
    begin
      if aPos = aSteps - 1
      then Result := 'off'
      else begin
        R := PosToValue( aPos, aSteps) * OSF;
        R.Invert;
        Result := Format( '%s', [ R.AsString], AppLocale);
      end;
    end;


    function    TConvVStepsOff.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 1;
    end;


    function    TConvVStepsOff.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 1);
    end;


{ ========
  TConvFractions = class( TKnobsBaseConverter)
  public
}

    constructor TConvFractions.Create; // override;
    begin
      inherited;
      FName := 'Fractions';
    end;


    function    TConvFractions.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      R : TRational;
    begin
      R := PosToValue( aPos, aSteps);
      R.Invert;
      Result := Format( '%s', [ R.AsString], AppLocale);
    end;


    function    TConvFractions.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos + 1;
    end;


    function    TConvFractions.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal - 1);
    end;


{ ========
  TConvdB0_1 = class( TKnobsBaseConverter)
  public
}

    constructor TConvdB0_1.Create; // override;
    begin
      inherited;
      FName         := 'dB[0,1]';
      FDefaultUnits := 'dB';
    end;


    function    TConvdB0_1.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.1f dB', [ aValue], AppLocale);
    end;


    function    TConvdB0_1.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvdB0_1.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvdB0_4 = class( TKnobsBaseConverter)
  public
}

    constructor TConvdB0_4.Create; // override;
    begin
      inherited;
      FName         := 'dB[0,4]';
      FDefaultUnits := 'dB';
    end;


    function    TConvdB0_4.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.1f dB', [ aValue], AppLocale);
    end;


    function    TConvdB0_4.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 4);
    end;


    function    TConvdB0_4.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 4);
    end;


{ ========
  TConvdB0_16 = class( TKnobsBaseConverter)
  public
}

    constructor TConvdB0_16.Create; // override;
    begin
      inherited;
      FName         := 'dB[0,16]';
      FDefaultUnits := 'dB';
    end;


    function    TConvdB0_16.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.1f dB', [ aValue], AppLocale);
    end;


    function    TConvdB0_16.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 16);
    end;


    function    TConvdB0_16.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 16);
    end;


{ ========
  TConvdB0_32 = class( TKnobsBaseConverter)
  public
}

    constructor TConvdB0_32.Create; // override;
    begin
      inherited;
      FName         := 'dB[0,32]';
      FDefaultUnits := 'dB';
    end;


    function    TConvdB0_32.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.1f dB', [ aValue], AppLocale);
    end;


    function    TConvdB0_32.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 32);
    end;


    function    TConvdB0_32.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 32);
    end;


{ ========
  TConvdB0_64 = class( TKnobsBaseConverter)
  public
}

    constructor TConvdB0_64.Create; // override;
    begin
      inherited;
      FName         := 'dB[0,64]';
      FDefaultUnits := 'dB';
    end;


    function    TConvdB0_64.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.1f dB', [ aValue], AppLocale);
    end;


    function    TConvdB0_64.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 1600);
    end;


    function    TConvdB0_64.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 64);
    end;


{ ========
  TConvdBKS = class( TKnobsBaseConverter)
  public
}

    constructor TConvdBKS.Create; // override;
    begin
      inherited;
      FName         := 'dB KS';
      FDefaultUnits := 'dB';
    end;


    function    TConvdBKS.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.2f dB', [ aValue], AppLocale);
    end;


    function    TConvdBKS.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0.8, 1);
    end;


    function    TConvdBKS.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0.8, 1);
    end;


{ ========
  TConvdbResonator = class( TKnobsBaseConverter)
  public
}

    constructor TConvdbResonator.Create; // override;
    begin
      inherited;
      FName         := 'dB Resonator';
      FDefaultUnits := 'dB';
    end;


    function    TConvdbResonator.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := UnitsTodB( aValue);

      if aValue <= - MAX_DB_VAL
      then Result := 'off'
      else Result := Format( '%.6f dB', [ aValue], AppLocale);
    end;


    function    TConvdbResonator.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0.999, 0.999999999999999);
    end;


    function    TConvdbResonator.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0.999, 0.999999999999999);
    end;


{ ========
  TConvOscFreqCoarse = class( TKnobsBaseConverter)
  public
}

    constructor TConvOscFreqCoarse.Create; // override;
    begin
      inherited;
      FName         := 'OscFreqCoarse';
      FDefaultUnits := 'Hz';
    end;


    function    TConvOscFreqCoarse.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
      aFreq  : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aFreq  := LookupFrequency( aValue);
      Result := Format( '%s ~~ %s', [ FreqToStr( aFreq), UnitsToNoteName( aValue)], AppLocale);
    end;


    function    TConvOscFreqCoarse.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvOscFreqCoarse.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvOscLinCoarse = class( TKnobsBaseConverter)
  public
}

    constructor TConvOscLinCoarse.Create; // override;
    begin
      inherited;
      FName         := 'OscLinCoarse';
      FDefaultUnits := 'Hz';
    end;


    function    TConvOscLinCoarse.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupFrequencyLin( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvOscLinCoarse.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvOscLinCoarse.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvOscCubedCoarse = class( TKnobsBaseConverter)
  public
}

    constructor TConvOscCubedCoarse.Create; // override;
    begin
      inherited;
      FName         := 'OscCubedCoarse';
      FDefaultUnits := 'Hz';
    end;


    function    TConvOscCubedCoarse.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupFrequencyCubed( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvOscCubedCoarse.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvOscCubedCoarse.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseFast = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseFast.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseFast';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseFast.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoFastFrequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseFast.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseFast.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseMedium = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseMedium.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseMedium';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseMedium.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoMediumFrequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseMedium.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseMedium.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseSlow = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseSlow.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseSlow';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseSlow.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoSlowFrequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseSlow.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseSlow.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseFast2 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseFast2.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseFast2';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseFast2.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoFast2Frequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseFast2.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseFast2.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseMedium2 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseMedium2.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseMedium2';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseMedium2.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoMedium2Frequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseMedium2.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseMedium2.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseSlow2 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseSlow2.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseSlow2';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseSlow2.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoSlow2Frequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseSlow2.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseSlow2.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseBPM = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseBPM.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseBPM';
      FDefaultUnits := 'bpm';
    end;


    function    TConvLfoFreqCoarseBPM.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoBPMFrequency( aValue);
      Result := FreqToBPMStr( aValue);
    end;


    function    TConvLfoFreqCoarseBPM.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseBPM.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvLfoFreqCoarseLin = class( TKnobsBaseConverter)
  public
}

    constructor TConvLfoFreqCoarseLin.Create; // override;
    begin
      inherited;
      FName         := 'LfoFreqCoarseLin';
      FDefaultUnits := 'Hz';
    end;


    function    TConvLfoFreqCoarseLin.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);
      aValue := LookupLfoLinFrequency( aValue);
      Result := FreqToStr( aValue);
    end;


    function    TConvLfoFreqCoarseLin.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvLfoFreqCoarseLin.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvNoiseColor = class( TKnobsBaseConverter)
  public
}

    constructor TConvNoiseColor.Create; // override;
    begin
      inherited;
      FName         := 'NoiseColor';
      FDefaultUnits := '';
    end;


    function    TConvNoiseColor.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      if aPos = 0
      then Result := 'off'
      else if aPos = aSteps
      then Result := 'white'
      else Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvNoiseColor.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvNoiseColor.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeFast = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeFast.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeFast';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeFast.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeFast
      else aValue := LookupEnvTime( erFast, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeFast.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeFast.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeMedium = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeMedium.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeMedium';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeMedium.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeMedium
      else aValue := LookupEnvTime( erMedium, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeMedium.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeMedium.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeSlow = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeSlow.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeSlow';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeSlow.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeSlow
      else aValue := LookupEnvTime( erSlow, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeSlow.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeSlow.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeFastAr = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeFastAr.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeFastAr';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeFastAr.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeFast / Control_Decimation
      else aValue := LookupEnvTimeAr( erFast, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeFastAr.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeFastAr.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeMediumAr = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeMediumAr.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeMediumAr';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeMediumAr.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeMedium / Control_Decimation
      else aValue := LookupEnvTimeAr( erMedium, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeMediumAr.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeMediumAr.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvEnvTimeSlowAr = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnvTimeSlowAr.Create; // override;
    begin
      inherited;
      FName         := 'EnvTimeSlowAr';
      FDefaultUnits := 's';
    end;


    function    TConvEnvTimeSlowAr.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      if aSteps <= 1
      then aValue := MinEnvTimeSlow / Control_Decimation
      else aValue := LookupEnvTimeAr( erSlow, aPos / ( aSteps - 1));

      Result := TimeToStr( aValue);
    end;


    function    TConvEnvTimeSlowAr.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvEnvTimeSlowAr.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvMidiCCIn = class( TKnobsBaseConverter)
  public
}

    constructor TConvMidiChIn.Create; // override;
    begin
      inherited;
      FName         := 'MidiChIn';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMidiChIn.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      if  aPos = aSteps - 1
      then Result := 'omni'
      else Result := Format( 'ch %.0f', [ PosToValue( aPos, aSteps) + 1], AppLocale);
    end;


    function    TConvMidiChIn.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos;
    end;


    function    TConvMidiChIn.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal);
    end;


{ ========
  TConvMidiCCOut = class( TKnobsbaseConverter)
  public
}

    constructor TConvMidiChOut.Create; // override;
    begin
      inherited;
      FName         := 'MidiChOut';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMidiChOut.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := PosToValue( aPos, aSteps);

      if aValue > 15
      then Result := Format( 'ic %.0f', [ aValue - 15], AppLocale)
      else Result := Format( 'ch %.0f', [ aValue +  1], AppLocale);
    end;


    function    TConvMidiChOut.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos;
    end;


    function    TConvMidiChOut.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal);
    end;


{ ========
  TConvNoteName = class( TKnobsBaseConverter)
  public
}

    constructor TConvNoteName.Create; // override;
    begin
      inherited;
      FName         := 'NoteName';
      FDefaultUnits := '';
    end;


    function    TConvNoteName.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := UnitsToNoteName( PosToValue( aPos, aSteps));
    end;


    function    TConvNoteName.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvNoteName.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvNoteName12 = class( TKnobsBaseConverter)
  public
}

    constructor TConvNoteName12.Create; // override;
    begin
      inherited;
      FName         := 'NoteName12';
      FDefaultUnits := '';
    end;


    function    TConvNoteName12.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := UnitsToNoteName12( PosToValue( aPos, aSteps));
    end;


    function    TConvNoteName12.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0, 1);
    end;


    function    TConvNoteName12.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, 0, 1);
    end;


{ ========
  TConvDelayTime = class( TKnobsBaseConverter)
  public
}

    function    TConvDelayTime.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := TimeToStr( PosToValue( aPos, aSteps));
    end;


{ ========
  TConvDelayTimeShort = class( TConvDelayTime)
  public
}

    constructor TConvDelayTimeShort.Create; // override;
    begin
      inherited;
      FName         := 'DelayTimeShort';
      FDefaultUnits := 's';
    end;


    function    TConvDelayTimeShort.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 100e-6, 3e-2);
    end;


    function    TConvDelayTimeShort.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 100e-6, 3e-2);
    end;


{ ========
  TConvDelayTimeMedium = class( TConvDelayTime)
  public
}

    constructor TConvDelayTimeMedium.Create; // override;
    begin
      inherited;
      FName         := 'DelayTimeMedium';
      FDefaultUnits := 's';
    end;


    function    TConvDelayTimeMedium.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 10e-3, 3);
    end;


    function    TConvDelayTimeMedium.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 10e-3, 3);
    end;


{ ========
  TConvDelayTimeLong = class( TConvDelayTime)
  public
}

    constructor TConvDelayTimeLong.Create; // override;
    begin
      inherited;
      FName         := 'DelayTimeLong';
      FDefaultUnits := 's';
    end;


    function    TConvDelayTimeLong.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 1, 300);
    end;


    function    TConvDelayTimeLong.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 1, 300);
    end;


{ ========
  TConvFilterFreq = class( TConvOscFreqCoarse)
  public
}

    constructor TConvFilterFreq.Create; // override;
    begin
      inherited;
      FName         := 'FilterFreq';
      FDefaultUnits := 'Hz';
    end;


{ ========
  TConvAvgFilterAlpha = class( TKnobsBaseConverter)
  public
}

    constructor TConvAvgFilterAlpha.Create; // override;
    begin
      inherited;
      FName         := 'FilterAvgAlpha';
      FDefaultUnits := '';
    end;


    function    TConvAvgFilterAlpha.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.5f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvAvgFilterAlpha.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvAvgFilterAlpha.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvCubicClipAlpha = class( TKnobsBaseConverter)
  public
}

    constructor TConvCubicClipAlpha.Create; // override;
    begin
      inherited;
      FName         := 'CubicClipAlpha';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvCubicClipAlpha.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvCubicClipAlpha.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps, 0.1, 10);
    end;


    function    TConvCubicClipAlpha.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps, 0.1, 10);
    end;


{ ========
  TConvBlockDCAlpha = class( TKnobsBaseConverter)
  public
}

    constructor TConvBlockDCAlpha.Create; // override;
    begin
      inherited;
      FName         := 'DCBlockAlpha';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvBlockDCAlpha.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvBlockDCAlpha.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, 0.9995, 0.9000);
    end;


    function    TConvBlockDCAlpha.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.9995, 0.9000);
    end;


{ ========
  TConvSlew = class( TKnobsBaseConverter)
  public
}

    constructor TConvSlew.Create; // override;
    begin
      inherited;
      FName         := 'Slew';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSlew.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.5f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvSlew.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvSlew.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvCents = class( TKnobsBaseConverter)
  public
}

    constructor TConvCents.Create; // override;
    begin
      inherited;
      FName         := 'Cents';
      FDefaultUnits := 'ct';
    end;


    function    TConvCents.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f ct', [ 50 * 256 * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvCents.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    const
      OTSF  = 1.0 / 256.0;             // Quarter note positive
      OTSFN = - OTSF;                  // Quarter note negative
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, OTSFN, OTSF);
    end;


    function    TConvCents.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    const
      OTSF  = 1.0 / 256.0;             // Quarter note positive
      OTSFN = - OTSF;                  // Quarter note negative
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, OTSFN, OTSF);
    end;


{ ========
  TConvSemitones = class( TKnobsBaseConverter)
  public
}

    constructor TConvSemitones.Create; // override;
    begin
      inherited;
      FName         := 'Semitones';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSemitones.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ NOTE_SCALING * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvSemitones.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    const
      SCALE = 24.0 * NOTE_SCALING_REC;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, - SCALE, SCALE);
    end;


    function    TConvSemitones.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    const
      SCALE = 24.0 * NOTE_SCALING_REC;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, - SCALE, SCALE);
    end;


{ ========
  TConvSemitonesFull = class( TKnobsBaseConverter)
  public
}

    constructor TConvSemitonesFull.Create; // override;
    begin
      inherited;
      FName         := 'SemitonesFull';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSemitonesFull.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue   : TSignal;
      aSign    : Integer;
      anIntVal : Integer;
      aRepr    : string;
    begin
      aValue   := NOTE_SCALING * PosToValue( aPos, aSteps);
      aSign    := Sign( aValue);
      anIntVal := Round( Abs( aValue));
      aRepr    := Format( '%d semitones', [ anIntVal], AppLocale);

      if aSign < 0
      then Result := Format( '-%s', [ aRepr], AppLocale)
      else Result := aRepr;
    end;


    function    TConvSemitonesFull.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, -1, 1);
    end;


    function    TConvSemitonesFull.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, -1, 1);
    end;


{ ========
  TConvTranspose = class( TKnobsBaseConverter)
  public
}

    constructor TConvTranspose.Create; // override;
    begin
      inherited;
      FName         := 'Transpose';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvTranspose.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ NOTE_SCALING * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvTranspose.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    const
      SCALE = 24.0 * NOTE_SCALING_REC;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, - SCALE, SCALE);
    end;


    function    TConvTranspose.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    const
      SCALE = 24.0 * NOTE_SCALING_REC;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, - SCALE, SCALE);
    end;


{ ========
  TConvEnumerable = class( TKnobsBaseConverter)
  public
}

    constructor TConvEnumerable.Create; // override;
    begin
      inherited;
      FIsEnumerable := True;
    end;


    function    TConvEnumerable.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos;
    end;


    function    TConvEnumerable.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal);
    end;


{ ========
  TConvMute = class( TConvEnumerable)
  public
}

    constructor TConvMute.Create; // override;
    begin
      inherited;
      FName           := 'mute';
      FDefaultUnits   := NO_UNITS;
      FIsRandomizable := False;
    end;


    function    TConvMute.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'active';
        1 :  Result := 'muted';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvActivation = class( TConvEnumerable)
  public
}

    constructor TConvActivation.Create; // override;
    begin
      inherited;
      FName         := 'activation';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvActivation.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'off';
        1 :  Result := 'active';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvSolo = class( TConvEnumerable)
  public
}

    constructor TConvSolo.Create; // override;
    begin
      inherited;
      FName         := 'solo';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSolo.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'solo';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvEnvMute = class( TConvEnumerable)
  public
}

    constructor TConvEnvMute.Create; // override;
    begin
      inherited;
      FName           := 'envmute';
      FDefaultUnits   := NO_UNITS;
      FIsRandomizable := False;
    end;


    function    TConvEnvMute.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'active';
        1 :  Result := 'muted';
        2 :  Result := 'pass through';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvVcpsMode = class( TConvEnumerable)
  public
}

    constructor TConvVcpsMode.Create; // override;
    begin
      inherited;
      FName           := 'vcpsmode';
      FDefaultUnits   := NO_UNITS;
      FIsRandomizable := False;
    end;


    function    TConvVcpsMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'adc';
        1 :  Result := 'off';
        2 :  Result := 'on';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvOutType = class( TConvEnumerable)
  public
}

    constructor TConvOutType.Create; // override;
    begin
      inherited;
      FName         := 'outtype';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvOutType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Normal';
        1 :  Result := 'Inverted';
        2 :  Result := 'Positive';
        3 :  Result := 'Positve Inverted';
        4 :  Result := 'Negative';
        5 :  Result := 'Negative Inverted';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvEnvOutType = class( TConvEnumerable)
  public
}

    constructor TConvEnvOutType.Create; // override;
    begin
      inherited;
      FName         := 'envouttype';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvEnvOutType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Positive';
        1 :  Result := 'Positve Inverted';
        2 :  Result := 'Negative';
        3 :  Result := 'Negative Inverted';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvEnvSlopeType = class( TConvEnumerable)
  public
}

    constructor TConvEnvSlopeType.Create; // override;
    begin
      inherited;
      FName         := 'envslopetype';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvEnvSlopeType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'log';
        1 :  Result := 'lin';
        2 :  Result := 'exp';
        3 :  Result := 's';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvWaveShape = class( TConvEnumerable)
  public
}

    constructor TConvWaveShape.Create; // override;
    begin
      inherited;
      FName         := 'waveshape';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvWaveShape.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Sine';
        1 :  Result := 'Tri';
        2 :  Result := 'Saw';
        3 :  Result := 'Square';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvPulseMode = class( TConvEnumerable)
  public
}

    constructor TConvPulseMode.Create; // override;
    begin
      inherited;
      FName         := 'pulsemode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPulseMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Square';
        1 :  Result := 'Pulse';
        2 :  Result := 'Input';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvPulseDelayMode = class( TConvEnumerable)
  public
}

    constructor TConvPulseDelayMode.Create; // override;
    begin
      inherited;
      FName         := 'pulse delay mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPulseDelayMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Delay';
        1 :  Result := 'Stretch';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TTalkieFrameRate = class( TKnobsBaseConverter)
  public
}

    constructor TTalkieFrameRate.Create; // override;
    begin
      inherited;
      FName         := 'talkie frame rate';
      FDefaultUnits := 'Hz';
    end;


    function    TTalkieFrameRate.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f Hz', [ 80.0 * PosToValue( aPos, aSteps)]);
    end;


    function    TTalkieFrameRate.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TTalkieFrameRate.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TTalkieSampleRate = class( TKnobsBaseConverter)
  public
}

    constructor TTalkieSampleRate.Create; // override;
    begin
      inherited;
      FName         := 'talkie sample rate';
      FDefaultUnits := 'Hz';
    end;


    function    TTalkieSampleRate.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.1f Hz', [ 16000.0 * PosToValue( aPos, aSteps)]);
    end;


    function    TTalkieSampleRate.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TTalkieSampleRate.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TRoomSize = class( TKnobsBaseConverter)
  public
}

    constructor TRoomSize.Create; // override;
    begin
      inherited;
      FName         := 'room size';
      FDefaultUnits := NO_UNITS;
    end;


    function    TRoomSize.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f m', [ PosToValue( aPos, aSteps)]);
    end;


    function    TRoomSize.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 1, 6400);
    end;


    function    TRoomSize.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 1, 6400);
    end;


{ ========
  TReverbTime = class( TKnobsBaseConverter)
  public
}

    constructor TReverbTime.Create; // override;
    begin
      inherited;
      FName         := 'reverb time';
      FDefaultUnits := 's';
    end;


    function    TReverbTime.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := TimeToStr( PosToValue( aPos, aSteps));
    end;


    function    TReverbTime.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeExp( aPos, aSteps - 1, 0.050, 1800);
    end;


    function    TReverbTime.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeExpInv( aVal, aSteps - 1, 0.050, 1800);
    end;


{ ========
  TRateMultiplier = class( TKnobsBaseConverter)
  public
}

    constructor TRateMultiplier.Create; // override;
    begin
      inherited;
      FName         := 'rate multiplier';
      FDefaultUnits := NO_UNITS;
    end;


    function    TRateMultiplier.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := RateMultiplierToStr( PosToValue( aPos, aSteps));
    end;


    function    TRateMultiplier.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps, - 23, 24);
    end;


    function    TRateMultiplier.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps, - 23, 24);
    end;


{ ========
  TSwing = class( TKnobsBaseConverter)
  public
}

    constructor TSwing.Create; // override;
    begin
      inherited;
      FName         := 'swing';
      FDefaultUnits := NO_UNITS;
    end;


    function    TSwing.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.1f %%', [ 100.0 * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TSwing.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0.50, 0.96875);
    end;


    function    TSwing.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.50, 0.96875);
    end;


{ ========
  TSongSpeed = class( TKnobsBaseConverter)
  public
}

    constructor TSongSpeed.Create; // override;
    begin
      inherited;
      FName         := 'song speed';
      FDefaultUnits := NO_UNITS;
    end;


    function    TSongSpeed.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.2f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TSongSpeed.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0.1, 5.0);
    end;


    function    TSongSpeed.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.1, 5.0);
    end;


{ ========
  THzMeasure = class( TKnobsBaseConverter)
  public
}

    constructor THzMeasure.Create; // override;
    begin
      inherited;
      FName         := 'HzMeasure';
      FIsEnumerable := True;
      FDefaultUnits := 's';
    end;


    function    THzMeasure.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '1 kHz';
        1 :  Result := '100 Hz';
        2 :  Result := '10 Hz';
        3 :  Result := '1 Hz';
        4 :  Result := '10 s';
        else Result := S_UNDEFINED;
      end;
    end;


    function    THzMeasure.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := aPos;
    end;


    function    THzMeasure.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round( aVal);
    end;


{ ========
  TPhase = class( TKnobsBaseConverter)
  public
}

    constructor TPhase.Create; // override;
    begin
      inherited;
      FName         := 'phase';
      FDefaultUnits := NO_UNITS;
    end;


    function    TPhase.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ 180 * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TPhase.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := ( aPos - 180.0) / 180;
    end;


    function    TPhase.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := Round(( aVal + 180.0) / 180.0);
    end;


{ ========
  TFreqMult = class( TKnobsBaseConverter)
  public
}

    constructor TFreqMult.Create; // override;
    begin
      inherited;
      FName         := 'Frequency Multiplier';
      FDefaultUnits := NO_UNITS;
    end;


    function    TFreqMult.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aValue : TSignal;
    begin
      aValue := LookupFrequency( PosToValue( aPos, aSteps)) / ReferenceA;
      Result := Format( '%s', [ MultiplierToStr( aValue)], AppLocale);
    end;


    function    TFreqMult.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TFreqMult.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvNormalHold = class( TConvEnumerable)
  public
}

    constructor TConvNormalHold.Create; // override;
    begin
      inherited;
      FName         := 'normal/hold';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvNormalHold.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Normal';
        1 :  Result := 'Hold';
        else Result := S_UNDEFINED;
      end;
    end;

{ ========
  TConvMixerModeSimple = class( TConvEnumerable)
  public
}

    constructor TConvMixerModeSimple.Create; // override;
    begin
      inherited;
      FName         := 'Mixer mode simple';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMixerModeSimple.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'linear';
        1 :  Result := '_dB_';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========

  TConvMixerModeAll = class( TConvEnumerable)
  public
}

    constructor TConvMixerModeAll.Create; // override;
    begin
      inherited;
      FName         := 'Mixer mode all';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMixerModeAll.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'lin -4,4';
        1 :  Result := 'db 12';
        2 :  Result := 'lin -1,1';
        3 :  Result := 'db 0';
        4 :  Result := 'lin 0,4';
        5 :  Result := 'lin 0,1';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========

  TConvMixerModeExtended = class( TConvEnumerable)
  public
}

    constructor TConvMixerModeExtended.Create; // override;
    begin
      inherited;
      FName         := 'Mixer mode extended';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMixerModeExtended.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'lin -1,1';
        1 :  Result := 'db 0';
        2 :  Result := 'lin 0,1';
        3 :  Result := 'lin 0,4';
        4 :  Result := 'lin -4,4';
        5 :  Result := 'db 12';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========

  TConvMixerModeExtendedAmp = class( TConvEnumerable)
  public
}

    constructor TConvMixerModeExtendedAmp.Create; // override;
    begin
      inherited;
      FName         := 'Mixer mode extended amp';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMixerModeExtendedAmp.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'lin -4,4';
        1 :  Result := 'db 12';
        2 :  Result := 'lin -1,1';
        3 :  Result := 'db 0';
        4 :  Result := 'lin 0,1';
        5 :  Result := 'lin 0,4';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========

  TConvSequencerMode = class( TConvEnumerable)
  public
}

    constructor TConvSequencerMode.Create; // override;
    begin
      inherited;
      FName         := 'Sequencer mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSequencerMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'lin -4,4';
        1 :  Result := 'db 12';
        2 :  Result := 'lin -1,1';
        3 :  Result := 'db 0';
        4 :  Result := 'lin 0,4';
        5 :  Result := 'lin 0,1';
        6 :  Result := 'note';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvLfoRange = class( TConvEnumerable)
  public
}

    constructor TConvLfoRange.Create; // override;
    begin
      inherited;
      FName         := 'LFO range';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvLfoRange.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'fast';
        1 :  Result := 'med';
        2 :  Result := 'slow';
        3 :  Result := 'BPM';
        4 :  Result := 'Lin';
        5 :  Result := 'fast2';
        6 :  Result := 'med2';
        7 :  Result := 'slow2';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvEnvelopeRange = class( TConvEnumerable)
  public
}

    constructor TConvEnvelopeRange.Create; // override;
    begin
      inherited;
      FName         := 'Envelope range';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvEnvelopeRange.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'fast';
        1 :  Result := 'med';
        2 :  Result := 'slow';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvDelayRange = class( TConvEnumerable)
  public
}

    constructor TConvDelayRange.Create; // override;
    begin
      inherited;
      FName         := 'Delay range';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvDelayRange.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'short';
        1 :  Result := 'medium';
        2 :  Result := 'long';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvNoteStealing = class( TConvEnumerable)
  public
}

    constructor TConvNoteStealing.Create; // override;
    begin
      inherited;
      FName         := 'Note stealing';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvNoteStealing.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'none';
        1 :  Result := 'cyclic';
        2 :  Result := 'softest';
        3 :  Result := 'oldest';
        4 :  Result := 'lowest';
        5 :  Result := 'highest';
        6 :  Result := 'newest';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvAutoMode = class( TConvEnumerable)
  public
}

    constructor TConvAutoMode.Create; // override;
    begin
      inherited;
      FName         := 'Auto mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvAutoMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'randomize';
        1 :  Result := 'mutate';
        2 :  Result := 'mate';
        3 :  Result := 'morph';
        4 :  Result := 'patch reset';
        5 :  Result := 'patch compile';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvHrastOscType = class( TConvEnumerable)
  public
}

    constructor TConvHrastOscType.Create; // override;
    begin
      inherited;
      FName         := 'Hrast Osc type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvHrastOscType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'norm.';
        1 :  Result := 'vint.';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvNoiseType = class( TConvEnumerable)
  public
}

    constructor TConvNoiseType.Create; // override;
    begin
      inherited;
      FName         := 'Noise type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvNoiseType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Lin';
        1 :  Result := 'Gau';
        2 :  Result := 'Exp';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvAttractorType = class( TConvEnumerable)
  public
}

    constructor TConvAttractorType.Create; // override;
    begin
      inherited;
      FName         := 'Attractor type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvAttractorType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Lorenz';
        1 :  Result := 'Rssler';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvInterpolationType = class( TConvEnumerable)
  public
}

    constructor TConvInterpolationType.Create; // override;
    begin
      inherited;
      FName         := 'Interpolation type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvInterpolationType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'linear';
        1 :  Result := 'all pass';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvOffOnType = class( TConvEnumerable)
  public
}

    constructor TConvOffOnType.Create; // override;
    begin
      inherited;
      FName         := 'Off on type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvOffOnType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'off';
        1 :  Result := 'on';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvTrigType = class( TConvEnumerable)
  public
}

    constructor TConvTrigType.Create; // override;
    begin
      inherited;
      FName         := 'Trig type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvTrigType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'off';
        1 :  Result := 'trig';
        2 :  Result := 'gate';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvLinExpType = class( TConvEnumerable)
  public
}

    constructor TConvLinExpType.Create; // override;
    begin
      inherited;
      FName         := 'Lin exp type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvLinExpType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'lin';
        1 :  Result := 'exp';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvMirrorType = class( TConvEnumerable)
  public
}

    constructor TConvMirrorType.Create; // override;
    begin
      inherited;
      FName         := 'Mirror type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMirrorType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'off';
        1 :  Result := 'lock';
        2 :  Result := 'mirror';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSquareMode = class( TConvEnumerable)
  public
}

    constructor TConvSquareMode.Create; // override;
    begin
      inherited;
      FName         := 'Square mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSquareMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'pulse';
        1 :  Result := 'square';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvFaderType = class( TConvEnumerable)
  public
}

    constructor TConvFaderType.Create; // override;
    begin
      inherited;
      FName         := 'Fader type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvFaderType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'linear';
        1 :  Result := 'square root';
        2 :  Result := 'sine';
        3 :  Result := 'sine squared';
        4 :  Result := 'squared';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSkipMode = class( TConvEnumerable)
  public
}

    constructor TConvSkipMode.Create; // override;
    begin
      inherited;
      FName         := 'Skip mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSkipMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'inactive';
        1 :  Result := 'active';
        2 :  Result := 'skip';
        3 :  Result := 'reset';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPeakRMS = class( TConvEnumerable)
  public
}

    constructor TConvPeakRMS.Create; // override;
    begin
      inherited;
      FName         := 'Peak RMS';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPeakRMS.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'peak';
        1 :  Result := 'RMS';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPerlinMode = class( TConvEnumerable)
  public
}

    constructor TConvPerlinMode.Create; // override;
    begin
      inherited;
      FName         := 'Perlin mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPerlinMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'n';
        1 :  Result := 'i';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvFIMode = class( TConvEnumerable)
  public
}

    constructor TConvFIMode.Create; // override;
    begin
      inherited;
      FName         := 'FI mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvFIMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '.';
        1 :  Result := 'FI';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvLoopMode = class( TConvEnumerable)
  public
}

    constructor TConvLoopMode.Create; // override;
    begin
      inherited;
      FName         := 'Loop mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvLoopMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'once';
        1 :  Result := 'loop';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRPNType = class( TConvEnumerable)
  public
}

    constructor TConvRPNType.Create; // override;
    begin
      inherited;
      FName         := 'RPN Type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRPNType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'NRPN';
        1 :  Result := 'RPN';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRPNUSel = class( TConvEnumerable)
  public
}

    constructor TConvRPNUSel.Create; // override;
    begin
      inherited;
      FName         := 'RPN USel';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRPNUSel.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'M';
        1 :  Result := 'B';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvMonoStereo = class( TConvEnumerable)
  public
}

    constructor TConvMonoStereo.Create; // override;
    begin
      inherited;
      FName         := 'Mono Stereo';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvMonoStereo.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'mono';
        1 :  Result := 'stereo';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSawPulse = class( TConvEnumerable)
  public
}

    constructor TConvSawPulse.Create; // override;
    begin
      inherited;
      FName         := 'Saw Pulse';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSawPulse.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'saw';
        1 :  Result := 'pls';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvAsyncSync = class( TConvEnumerable)
  public
}

    constructor TConvAsyncSync.Create; // override;
    begin
      inherited;
      FName         := 'Async Sync';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvAsyncSync.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'async';
        1 :  Result := 'sync';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRandomWalkMode = class( TConvEnumerable)
  public
}

    constructor TConvRandomWalkMode.Create; // override;
    begin
      inherited;
      FName         := 'Randomwalk mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRandomWalkMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'E';
        1 :  Result := 'L';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSteppedSmooth = class( TConvEnumerable)
  public
}

    constructor TConvSteppedSmooth.Create; // override;
    begin
      inherited;
      FName         := 'Stepped Smooth';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSteppedSmooth.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'stepped';
        1 :  Result := 'smooth';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvLFORetrig = class( TConvEnumerable)
  public
}

    constructor TConvLFORetrig.Create; // override;
    begin
      inherited;
      FName         := 'LFO Retrig';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvLFORetrig.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'retrig';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvGateMode = class( TConvEnumerable)
  public
}

    constructor TConvGateMode.Create; // override;
    begin
      inherited;
      FName         := 'Gate mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvGateMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'gate';
        1 :  Result := 'trig';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRectifierMode = class( TConvEnumerable)
  public
}

    constructor TConvRectifierMode.Create; // override;
    begin
      inherited;
      FName         := 'Rectifier mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRectifierMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'half wave';
        1 :  Result := 'full wave';
        2 :  Result := 'transparent';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPinkMode = class( TConvEnumerable)
  public
}

    constructor TConvPinkMode.Create; // override;
    begin
      inherited;
      FName         := 'Pink mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPinkMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'economy';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConv7kHzMode = class( TConvEnumerable)
  public
}

    constructor TConv7kHzMode.Create; // override;
    begin
      inherited;
      FName         := '7k Hz mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConv7kHzMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '7k HP';
        1 :  Result := '7k BP';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConv28HzMode = class( TConvEnumerable)
  public
}

    constructor TConv28HzMode.Create; // override;
    begin
      inherited;
      FName         := '28 Hz mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConv28HzMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '28 LP';
        1 :  Result := '28 BP';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConv5k6HzMode = class( TConvEnumerable)
  public
}

    constructor TConv5k6HzMode.Create; // override;
    begin
      inherited;
      FName         := '5k6 Hz mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConv5k6HzMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '5k6 HP';
        1 :  Result := '5k6 BP';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConv139HzMode = class( TConvEnumerable)
  public
}

    constructor TConv139HzMode.Create; // override;
    begin
      inherited;
      FName         := '139 Hz mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConv139HzMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '139 LP';
        1 :  Result := '139 BP';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvDemuxHoldMode = class( TConvEnumerable)
  public
}

    constructor TConvDemuxHoldMode.Create; // override;
    begin
      inherited;
      FName         := 'Demux hold mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvDemuxHoldMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'xfade';
        1 :  Result := 'hold';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvDigiDemuxHoldMode = class( TConvEnumerable)
  public
}

    constructor TConvDigiDemuxHoldMode.Create; // override;
    begin
      inherited;
      FName         := 'Digi demux hold mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvDigiDemuxHoldMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'zero';
        1 :  Result := 'hold';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvChebgenMode = class( TConvEnumerable)
  public
}

    constructor TConvChebgenMode.Create; // override;
    begin
      inherited;
      FName         := 'Chebgen mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvChebgenMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'sine';
        1 :  Result := 'tri';
        2 :  Result := 'saw';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvQuantizeMode = class( TConvEnumerable)
  public
}

    constructor TConvQuantizeMode.Create; // override;
    begin
      inherited;
      FName         := 'Quantize mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvQuantizeMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'notes';
        1 :  Result := 'fractions';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvQuantizeType = class( TConvEnumerable)
  public
}

    constructor TConvQuantizeType.Create; // override;
    begin
      inherited;
      FName         := 'Quantize type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvQuantizeType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'floor';
        1 :  Result := 'round';
        2 :  Result := 'ceil';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvGateFunction = class( TConvEnumerable)
  public
}

    constructor TConvGateFunction.Create; // override;
    begin
      inherited;
      FName         := 'Gate function';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvGateFunction.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'and';
        1 :  Result := 'nand';
        2 :  Result := 'or';
        3 :  Result := 'nor';
        4 :  Result := 'xor';
        5 :  Result := 'xnor';
        6 :  Result := 'one';
        7 :  Result := 'none';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPatternMode = class( TConvEnumerable)
  public
}

    constructor TConvPatternMode.Create; // override;
    begin
      inherited;
      FName         := 'Pattern mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPatternMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'nrm';
        1 :  Result := 'rn1';
        2 :  Result := 'rn2';
        3 :  Result := 'rn3';
        4 :  Result := 'skip';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvChordDegree = class( TConvEnumerable)
  public
}

    constructor TConvChordDegree.Create; // override;
    begin
      inherited;
      FName         := 'Chord degree';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvChordDegree.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'I';
        1 :  Result := 'II';
        2 :  Result := 'III';
        3 :  Result := 'IV';
        4 :  Result := 'V';
        5 :  Result := 'VI';
        6 :  Result := 'VII';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvChordType = class( TConvEnumerable)
  public
}

    constructor TConvChordType.Create; // override;
    begin
      inherited;
      FName         := 'Chord type';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvChordType.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 :  Result := 'Maj'  ;
         1 :  Result := 'min'  ;
         2 :  Result := 'dim'  ;
         3 :  Result := 'aug'  ;
         4 :  Result := 'sus'  ;
         5 :  Result := 'M6'   ;
         6 :  Result := 'm6'   ;
         7 :  Result := 'M7'   ;
         8 :  Result := 'M9'   ;
         9 :  Result := 'dom7' ;
        10 :  Result := 'm9'   ;
        11 :  Result := 'dim7' ;
        12 :  Result := 'm7b5' ;
        13 :  Result := '7sus' ;
        14 :  Result := 'M9b7' ;
        15 :  Result := 'M13_1';
        16 :  Result := 'M13_2';
        17 :  Result := 'M7b9' ;
        18 :  Result := 'M13b9';
        19 :  Result := 'M9b5' ;
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvOctaveShift = class( TConvEnumerable)
  public
}

    constructor TConvOctaveShift.Create; // override;
    begin
      inherited;
      FName         := 'Octave shift';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvOctaveShift.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '-3';
        1 :  Result := '-2';
        2 :  Result := '-1';
        3 :  Result := '0';
        4 :  Result := '1';
        5 :  Result := '2';
        6 :  Result := '3';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvTransInv = class( TConvEnumerable)
  public
}

    constructor TConvTransInv.Create; // override;
    begin
      inherited;
      FName         := 'Transpose invert';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvTransInv.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'transpose';
        1 :  Result := 'invert';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRSTrigMode = class( TConvEnumerable)
  public
}

    constructor TConvRSTrigMode.Create; // override;
    begin
      inherited;
      FName         := 'RS Trig mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRSTrigMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'edge';
        1 :  Result := 'level';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvCounterMode = class( TConvEnumerable)
  public
}

    constructor TConvCounterMode.Create; // override;
    begin
      inherited;
      FName         := 'Counter mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvCounterMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'binary';
        1 :  Result := 'seq.';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvTSRandomMode = class( TConvEnumerable)
  public
}

    constructor TConvTSRandomMode.Create; // override;
    begin
      inherited;
      FName         := 'TS random mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvTSRandomMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'random';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRepOnceMode = class( TConvEnumerable)
  public
}

    constructor TConvRepOnceMode.Create; // override;
    begin
      inherited;
      FName         := 'Rep once mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRepOnceMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'rep';
        1 :  Result := 'once';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvNormInvertMode = class( TConvEnumerable)
  public
}

    constructor TConvNormInvertMode.Create; // override;
    begin
      inherited;
      FName         := 'Norm invert mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvNormInvertMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'norm';
        1 :  Result := 'inv';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPostPreMode = class( TConvEnumerable)
  public
}

    constructor TConvPostPreMode.Create; // override;
    begin
      inherited;
      FName         := 'Post pre mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPostPreMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'post';
        1 :  Result := 'pre';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSeq16GateMode = class( TConvEnumerable)
  public
}

    constructor TConvSeq16GateMode.Create; // override;
    begin
      inherited;
      FName         := 'Seq16 gate mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSeq16GateMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'T';
        1 :  Result := 'G';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvChordMode = class( TConvEnumerable)
  public
}

    constructor TConvChordMode.Create; // override;
    begin
      inherited;
      FName         := 'Chord mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvChordMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'reduce';
        2 :  Result := 'lower';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvRunEditMode = class( TConvEnumerable)
  public
}

    constructor TConvRunEditMode.Create; // override;
    begin
      inherited;
      FName         := 'Run edit mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvRunEditMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'run';
        1 :  Result := 'edit';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvArpMode = class( TConvEnumerable)
  public
}

    constructor TConvArpMode.Create; // override;
    begin
      inherited;
      FName         := 'Arp mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvArpMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'unsorted';
        1 :  Result := 'low to high';
        2 :  Result := 'high to low';
        3 :  Result := 'random';
        4 :  Result := 'reversed';
        5 :  Result := 'pingpong';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvArpInMode = class( TConvEnumerable)
  public
}

    constructor TConvArpInMode.Create; // override;
    begin
      inherited;
      FName         := 'Arp inmode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvArpInMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'all';
        1 :  Result := 'changed';
        2 :  Result := 'unique';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvOnceContMode = class( TConvEnumerable)
  public
}

    constructor TConvOnceContMode.Create; // override;
    begin
      inherited;
      FName         := 'Once cont mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvOnceContMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'once';
        1 :  Result := 'cont';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvWrappedPlainMode = class( TConvEnumerable)
  public
}

    constructor TConvWrappedPlainMode.Create; // override;
    begin
      inherited;
      FName         := 'Wrapped plain mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvWrappedPlainMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'wrapped';
        1 :  Result := 'plain';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvWrappedPlainMode2 = class( TConvEnumerable)
  public
}

    constructor TConvWrappedPlainMode2.Create; // override;
    begin
      inherited;
      FName         := 'Wrapped plain mode2';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvWrappedPlainMode2.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'no wrap';
        1 :  Result := 'wrap xy';
        2 :  Result := 'wrap x';
        3 :  Result := 'wrap y';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvNormalFrozenMode = class( TConvEnumerable)
  public
}

    constructor TConvNormalFrozenMode.Create; // override;
    begin
      inherited;
      FName         := 'Normal frozen mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvNormalFrozenMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'frozen';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvCompressorBypassMode = class( TConvEnumerable)
  public
}

    constructor TConvCompressorBypassMode.Create; // override;
    begin
      inherited;
      FName         := 'Compressor bypass mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvCompressorBypassMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'on';
        1 :  Result := 'byp';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvSideChainMode = class( TConvEnumerable)
  public
}

    constructor TConvSideChainMode.Create; // override;
    begin
      inherited;
      FName         := 'Sidechain mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvSideChainMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'on';
        1 :  Result := 'byp';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvAcDcMode = class( TConvEnumerable)
  public
}

    constructor TConvAcDcMode.Create; // override;
    begin
      inherited;
      FName         := 'AC DC mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvAcDcMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'DC';
        1 :  Result := 'AC';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvPhaseDistortMode = class( TConvEnumerable)
  public
}

    constructor TConvPhaseDistortMode.Create; // override;
    begin
      inherited;
      FName         := 'Phase distort mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvPhaseDistortMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'normal';
        1 :  Result := 'phase rev.';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvClockedFastMode = class( TConvEnumerable)
  public
}

    constructor TConvClockedFastMode.Create; // override;
    begin
      inherited;
      FName         := 'Clocked fast mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvClockedFastMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'clocked';
        1 :  Result := 'fast';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvValueMode = class( TConvEnumerable)
  public
}

    constructor TConvValueMode.Create; // override;
    begin
      inherited;
      FName         := 'Value mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvValueMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'units';
        1 :  Result := 'dB';
        2 :  Result := 'Freq';
        3 :  Result := 'Note';
        4 :  Result := 'Lfo fast';
        5 :  Result := 'Lfo medium';
        6 :  Result := 'Lfo slow';
        7 :  Result := 'Lfo BPM';
        8 :  Result := 'Lfo Lin';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvAbsPosMode = class( TConvEnumerable)
  public
}

    constructor TConvAbsPosMode.Create; // override;
    begin
      inherited;
      FName         := 'Abs pos mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvAbsPosMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'abs';
        1 :  Result := 'pos';
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvFastSlowMode = class( TConvEnumerable)
  public
}

    constructor TConvFastSlowMode.Create; // override;
    begin
      inherited;
      FName         := 'Fast slow mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvFastSlowMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'fast'   ;
        1 :  Result := 'slow'   ;
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvGraphMode = class( TConvEnumerable)
  public
}

    constructor TConvGraphMode.Create; // override;
    begin
      inherited;
      FName         := 'Graph mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvGraphMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'linear' ;
        1 :  Result := 'splines';
        2 :  Result := 'steps'  ;
        else Result := S_UNDEFINED;
      end;
    end;


{ =======
  TConvClipWrapMode = class( TConvEnumerable)
  public
}

    constructor TConvClipWrapMode.Create; // override;
    begin
      inherited;
      FName         := 'Clip wrap mode';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvClipWrapMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'clip';
        1 :  Result := 'wrap';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvWaveShapeLFO = class( TConvEnumerable)
  public
}

    constructor TConvWaveShapeLFO.Create; // override;
    begin
      inherited;
      FName         := 'waveshape LFO';
      FDefaultUnits := NO_UNITS;
    end;


    function    TConvWaveShapeLFO.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'Sine';
        1 :  Result := 'Tri';
        2 :  Result := 'Saw';
        3 :  Result := 'Square';
        4 :  Result := 'Random Square';
        5 :  Result := 'Random';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvLinear1_8 = class( TKnobsBaseConverter)
  public
}

    constructor TConvLinear1_8.Create; // override;
    begin
      inherited;
      FName := 'Linear[1,8]';
    end;


    function    TConvLinear1_8.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.3f', [ 1.0 + 7.0 * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TConvLinear1_8.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvLinear1_8.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvMConst = class( TConvEnumerable)
  public
}

    constructor TConvMConst.Create; // override;
    begin
      inherited;
      FName := 'MConst';
    end;


    function    TConvMConst.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 : Result := '1';
         1 : Result := '2';
         2 : Result := 'e';
         3 : Result := 'Pi';
         4 : Result := 'sqrt(2)';
         5 : Result := 'sqrt(3)';
         6 : Result := 'sqrt(5)';
         7 : Result := '2^(1/12)';
         8 : Result := 'semitone';
         9 : Result := '1/2';
        10 : Result := '1/e';
        11 : Result := '1/Pi';
        12 : Result := '1/sqrt(2)';
        13 : Result := '1/sqrt(3)';
        14 : Result := '1/sqrt(5)';
        15 : Result := '2^(-1/12)';
        16 : Result := '1 mi 2nd';
        17 : Result := '2 2nd';
        18 : Result := '3 mi 3rd';
        19 : Result := '4 ma 3rd';
        20 : Result := '5 4th';
        21 : Result := '6 dim 5th';
        22 : Result := '7 5th';
        23 : Result := '8 mi 6th';
        24 : Result := '9 ma 6th';
        25 : Result := '10 mi 7th';
        26 : Result := '11 ma 7th';
        27 : Result := '12 octave';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvMonadicFunction = class( TConvEnumerable)
  public
}

    constructor TConvMonadicFunction.Create; // override;
    begin
      inherited;
      FName := 'Monadic function';
    end;


    function    TConvMonadicFunction.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 : Result := 'x ^ 2';
         1 : Result := 'x ^ 3';
         2 : Result := 'sqrt( x)';
         3 : Result := 'x ^ ( 1/3)';
         4 : Result := 'sin( x)';
         5 : Result := 'cos( x)';
         6 : Result := 'tan( x)';
         7 : Result := 'asin( x)';
         8 : Result := 'acos( x)';
         9 : Result := 'atan( x)';
        10 : Result := '2 ^ x';
        11 : Result := 'e ^ x';
        12 : Result := '10 ^ x';
        13 : Result := '2log( x)';
        14 : Result := 'ln( x)';
        15 : Result := '10log( x)';
        16 : Result := '2 ^ ( x / 12)';
        17 : Result := '12 * 2log( x)';
        18 : Result := 'x * Pi';
        19 : Result := 'x * 2 * Pi';
        20 : Result := 'x / Pi';
        21 : Result := 'x / ( 2 * Pi)';
        22 : Result := 'floor( x)';
        23 : Result := 'round( x)';
        24 : Result := 'ceil( x)';
        25 : Result := 'abs( x)';
        26 : Result := '- x';
        27 : Result := '1 - x';
        28 : Result := 'tanh( x)';
        29 : Result := '1 - abs( x)';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvDyadicFunction = class( TConvEnumerable)
  public
}

    constructor TConvDyadicFunction.Create; // override;
    begin
      inherited;
      FName := 'Dyadic function';
    end;


    function    TConvDyadicFunction.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'x mod y';
        1 :  Result := 'x ^ y';
        2 :  Result := 'xlog( y)';
        3 :  Result := 'atan( y / x)';
        4 :  Result := 'x - y';
        5 :  Result := 'y - x';
        6 :  Result := 'x + y';
        7 :  Result := 'x ^ ( 1 / y)';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvLifeSeqMode = class( TConvEnumerable)
  public
}

    constructor TConvLifeSeqMode.Create; // override;
    begin
      inherited;
      FName := 'Life seq mode';
    end;


    function    TConvLifeSeqMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 : Result := 'Life';
         1 : Result := 'HighLife';
         2 : Result := 'DayNight';
         3 : Result := '2x2';
         4 : Result := '34Life';
         5 : Result := 'Amoeba';
         6 : Result := 'Gnarl';
         7 : Result := 'Seeds';
         8 : Result := 'Serviettes';
         9 : Result := 'Stains';
        10 : Result := 'Pseudo';
        11 : Result := 'Move';
        12 : Result := 'StarWars';
        13 : Result := 'Inverse';
        14 : Result := 'Freeze';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvLifeSeqSumMode = class( TConvEnumerable)
  public
}

    constructor TConvLifeSeqSumMode.Create; // override;
    begin
      inherited;
      FName := 'Life seq sum mode';
    end;


    function    TConvLifeSeqSumMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'bin';
        1 :  Result := 'lin';
        2 :  Result := 'max';
        3 :  Result := 'min';
        4 :  Result := 'avg';
        5 :  Result := 'note';
        6 :  Result := 'maxn';
        7 :  Result := 'minn';
        8 :  Result := 'avgn';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvTerragenAlgorithm = class( TConvEnumerable)
  public
}

    constructor TConvTerragenAlgorithm.Create; // override;
    begin
      inherited;
      FName := 'Terragen algorithm';
    end;


    function    TConvTerragenAlgorithm.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 : Result := 'off';
         1 : Result := 'x';
         2 : Result := 'y';
         3 : Result := 'x * y';
         4 : Result := 'curtis roads';
         5 : Result := 'bass';
         6 : Result := 'lead1';
         7 : Result := 'hollow lead';
         8 : Result := 'pulsar';
         9 : Result := 'flat growl';
        10 : Result := 'atan';
        11 : Result := 'softclip';
        12 : Result := 'third';
        13 : Result := 'chebyshev 2';
        14 : Result := 'chebyshev 3';
        15 : Result := 'chebyshev 4';
        16 : Result := 'chebyshev 5';
        17 : Result := 'chebyshev 6';
        18 : Result := 'chebyshev 7';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvConvoderSize = class( TConvEnumerable)
  public
}

    constructor TConvConvoderSize.Create; // override;
    begin
      inherited;
      FName := 'Convoder size';
    end;


    function    TConvConvoderSize.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 : Result := '64';
         1 : Result := '128';
         2 : Result := '256';
         3 : Result := '512';
         4 : Result := '1024';
         5 : Result := '2048';
         6 : Result := '4096';
         7 : Result := '8192';
         8 : Result := '16384';
         9 : Result := '32768';
        10 : Result := '65536';
        11 : Result := '131072';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvTapperMode = class( TConvEnumerable)
  public
}

    constructor TConvTapperMode.Create; // override;
    begin
      inherited;
      FName := 'Tapper mode';
    end;


    function    TConvTapperMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'LFO med 2:05';
        1 :  Result := 'DELAY med 3.0 s';
        2 :  Result := 'DELAY long 5:00';
        3 :  Result := 'BPM ClockGen';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvPhaseShift = class( TConvEnumerable)
  public
}

    constructor TConvPhaseShift.Create; // override;
    begin
      inherited;
      FName := 'Phase shift';
    end;


    function    TConvPhaseShift.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '90';
        1 :  Result := '120';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvPWMPLM = class( TConvEnumerable)
  public
}

    constructor TConvPWMPLM.Create; // override;
    begin
      inherited;
      FName := 'PWM PLM';
    end;


    function    TConvPWMPLM.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'PWM';
        1 :  Result := 'PLM';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvZNZ = class( TConvEnumerable)
  public
}

    constructor TConvZNZ.Create; // override;
    begin
      inherited;
      FName := 'ZNZ';
    end;


    function    TConvZNZ.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'z';
        1 :  Result := 'nz';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvMuseSwitch = class( TConvEnumerable)
  public
}

    constructor TConvMuseSwitch.Create; // override;
    begin
      inherited;
      FName := 'MuseSwitch';
    end;


    function    TConvMuseSwitch.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 :  Result := 'off';
         1 :  Result := 'on';
         2 :  Result := 'C 0';
         3 :  Result := 'C 1';
         4 :  Result := 'C 2';
         5 :  Result := 'C 3';
         6 :  Result := 'C 4';
         7 :  Result := 'C 5';
         8 :  Result := 'C 6';
         9 :  Result := 'C 7';
        10 :  Result := 'C 8';
        11 :  Result := 'B 1';
        12 :  Result := 'B 2';
        13 :  Result := 'B 3';
        14 :  Result := 'B 4';
        15 :  Result := 'B 5';
        16 :  Result := 'B 6';
        17 :  Result := 'B 7';
        18 :  Result := 'B 8';
        19 :  Result := 'B 9';
        20 :  Result := 'B 10';
        21 :  Result := 'B 11';
        22 :  Result := 'B 12';
        23 :  Result := 'B 13';
        24 :  Result := 'B 14';
        25 :  Result := 'B 15';
        26 :  Result := 'B 16';
        27 :  Result := 'B 17';
        28 :  Result := 'B 18';
        29 :  Result := 'B 19';
        30 :  Result := 'B 20';
        31 :  Result := 'B 21';
        32 :  Result := 'B 22';
        33 :  Result := 'B 23';
        34 :  Result := 'B 24';
        35 :  Result := 'B 25';
        36 :  Result := 'B 26';
        37 :  Result := 'B 27';
        38 :  Result := 'B 28';
        39 :  Result := 'B 29';
        40 :  Result := 'B 30';
        41 :  Result := 'B 31';
        42 :  Result := 'ext 1';
        43 :  Result := 'ext 2';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvFreeClocked = class( TConvEnumerable)
  public
}

    constructor TConvFreeClocked.Create; // override;
    begin
      inherited;
      FName := 'FreeClocked';
    end;


    function    TConvFreeClocked.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'free';
        1 :  Result := 'clocked';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvEvenOdd = class( TConvEnumerable)
  public
}

    constructor TConvEvenOdd.Create; // override;
    begin
      inherited;
      FName := 'EvenOdd';
    end;


    function    TConvEvenOdd.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'even';
        1 :  Result := 'odd';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvDupsMode = class( TConvEnumerable)
  public
}

    constructor TConvDupsMode.Create; // override;
    begin
      inherited;
      FName := 'DupsMode';
    end;


    function    TConvDupsMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'dups ok';
        1 :  Result := 'no dups';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvScaledMode = class( TConvEnumerable)
  public
}

    constructor TConvScaledMode.Create; // override;
    begin
      inherited;
      FName := 'ScaledMode';
    end;


    function    TConvScaledMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'unscaled';
        1 :  Result := 'octave';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvFmDepth = class( TKnobsBaseConverter)
  public
}

    constructor TConvFmDepth.Create; // override;
    begin
      inherited;
      FName := 'FM depth';
    end;


    function    TConvFmDepth.PosToDisplay( aPos, aSteps: Integer): string; // override;
    var
      aSemis : TSignal;
    begin
      aSemis := NOTE_SCALING * PosToValue( aPos, aSteps);
      Result := Format( '%.1f semis, %.0f ct', [ aSemis, 100.0 * aSemis], AppLocale);
    end;


    function    TConvFmDepth.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0, 1);
    end;


    function    TConvFmDepth.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0, 1);
    end;


{ ========
  TConvSinCosSin = class( TConvEnumerable)
  public
}

    constructor TConvSinCosSin.Create; // override;
    begin
      inherited;
      FName := 'SinCosSin';
    end;


    function    TConvSinCosSin.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'sin cos';
        1 :  Result := 'sin -sin';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvNpMode = class( TConvEnumerable)
  public
}

    constructor TConvNpMode.Create; // override;
    begin
      inherited;
      FName := 'NP Mode';
    end;


    function    TConvNpMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '0 !';
        1 :  Result := '1 +';
        2 :  Result := '2 *';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvSampleRate = class( TConvEnumerable)
  public
}

    constructor TConvSampleRate.Create; // override;
    begin
      inherited;
      FName := 'Sample Rate';
    end;


    function    TConvSampleRate.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '44k1';
        1 :  Result := '48k';
        2 :  Result := '88k2';
        3 :  Result := '96k';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvContrRateDecimation = class( TConvEnumerable)
  public
}

    constructor TConvContrRateDecimation.Create; // override;
    begin
      inherited;
      FName := 'Control Rate Decimation';
    end;


    function    TConvContrRateDecimation.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := '4';
        1 :  Result := '8';
        2 :  Result := '16';
        3 :  Result := '32';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvAutoReady = class( TConvEnumerable)
  public
}

    constructor TConvAutoReady.Create; // override;
    begin
      inherited;
      FName := 'Auto ready';
    end;


    function    TConvAutoReady.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'rdy wait';
        1 :  Result := 'auto rdy';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvAbsNorm = class( TConvEnumerable)
  public
}

    constructor TConvAbsNorm.Create; // override;
    begin
      inherited;
      FName := 'AbsNorm mode';
    end;


    function    TConvAbsNorm.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'norm';
        1 :  Result := 'abs';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvAllSkipMode = class( TConvEnumerable)
  public
}

    constructor TConvAllSkipMode.Create; // override;
    begin
      inherited;
      FName := 'All Skip mode';
    end;


    function    TConvAllSkipMode.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
        0 :  Result := 'all';
        1 :  Result := 'skip dead';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvESpeakModifiers = class( TConvEnumerable)
  public
}

    constructor TConvESpeakModifiers.Create; // override;
    begin
      inherited;
      FName := 'eSpeak modifiers';
    end;


    function    TConvESpeakModifiers.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 :  Result := '';
         1 :  Result := 'm1';
         2 :  Result := 'm2';
         3 :  Result := 'm3';
         4 :  Result := 'm4';
         5 :  Result := 'm5';
         6 :  Result := 'm6';
         7 :  Result := 'm7';
         8 :  Result := 'f1';
         9 :  Result := 'f2';
        10 :  Result := 'f3';
        11 :  Result := 'f4';
        12 :  Result := 'f5';
        13 :  Result := 'croak';
        14 :  Result := 'whisperm';
        15 :  Result := 'whisperf';
        16 :  Result := 'klatt1';
        17 :  Result := 'klatt2';
        18 :  Result := 'klatt3';
        19 :  Result := 'klatt4';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TConvESpeakPunctuation = class( TConvEnumerable)
  public
}

    constructor TConvESpeakPunctuation.Create; // override;
    begin
      inherited;
      FName := 'eSpeak puctuation';
    end;


    function    TConvESpeakPunctuation.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      case aPos of
         0 :  Result := 'none';
         1 :  Result := 'some';
         2 :  Result := 'all';
        else Result := S_UNDEFINED;
      end;
    end;


{ ========
  TESpeakRate = class( TKnobsBaseConverter)
  public
}
    // todo: put in some espeak stuff

    constructor TESpeakRate.Create; // override;
    begin
      inherited;
      FName := 'eSpeak rate';
    end;


    function    TESpeakRate.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TESpeakRate.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 80.0, 450.0);
    end;


    function    TESpeakRate.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 80.0, 450.0);
    end;


{ ========
  TESpeakPitch = class( TKnobsBaseConverter)
  public
}
    // todo: put in some espeak stuff

    constructor TESpeakPitch.Create; // override;
    begin
      inherited;
      FName := 'eSpeak pitch';
    end;


    function    TESpeakPitch.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TESpeakPitch.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0.0, 100.0);
    end;


    function    TESpeakPitch.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.0, 100.0);
    end;


{ ========
  TESpeakRange = class( TKnobsBaseConverter)
  public
}
    // todo: put in some espeak stuff

    constructor TESpeakRange.Create; // override;
    begin
      inherited;
      FName := 'eSpeak range';
    end;


    function    TESpeakRange.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f', [ PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TESpeakRange.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0.0, 100.0);
    end;


    function    TESpeakRange.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.0, 100.0);
    end;


{ ========
  TESpeakWordGap = class( TKnobsBaseConverter)
  public
}
    // todo: put in some espeak stuff

    constructor TESpeakWordGap.Create; // override;
    begin
      inherited;
      FName := 'eSpeak word gap';
    end;


    function    TESpeakWordGap.PosToDisplay( aPos, aSteps: Integer): string; // override;
    begin
      Result := Format( '%.0f ms', [ 10 * PosToValue( aPos, aSteps)], AppLocale);
    end;


    function    TESpeakWordGap.PosToValue( aPos, aSteps: Integer): TSignal; // override;
    begin
      Result := KnobRangeLin( aPos, aSteps - 1, 0.0, 100.0);
    end;


    function    TESpeakWordGap.ValueToPos( aVal: TSignal; aSteps: Integer): Integer; // override;
    begin
      Result := KnobRangeLinInv( aVal, aSteps - 1, 0.0, 100.0);
    end;


{ ========
  TConvDynTalkieBank = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynTalkieBank.Create; // override;
    begin
      inherited;
      FName := 'Dyn Talkie bank';
    end;


{ ========
  TConvDynFormant2Vowels = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynFormant2Vowels.Create; // override;
    begin
      inherited;
      FName := 'Dyn Formant2 vowels';
    end;


{ ========
  TConvDynModal2V = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynModal2.Create; // override;
    begin
      inherited;
      FName := 'Dyn Modal2';
    end;


{ ========
  TConvDynScale = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynScale.Create; // override;
    begin
      inherited;
      FName := 'Dyn Scale';
    end;


{ ========
  TConvDynGrainEnvelopes = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynGrainEnvelopes.Create; // override;
    begin
      inherited;
      FName := 'Dyn Grain envelopes';
    end;


{ ========
  TConvDynMorphSelect = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynMorphSelect.Create; // override;
    begin
      inherited;
      FName := 'Dyn Morph select';
    end;


{ ========
  TConvDynMazeType = class( TKnobsDynamicConverter)
  public
}


    constructor TConvDynMazeType.Create; // override;
    begin
      inherited;
      FName := 'Dyn maze type';
    end;


{ ========
  TConvSapiSpeechVoices = class( TKnobsDynamicConverter)
  public
}


    constructor TConvSapiSpeechVoices.Create; // override;
    begin
      inherited;
      FName := 'Dyn sapi speech voices';
    end;


{ ========
  TConvSapiSpeechRates = class( TKnobsDynamicConverter)
  public
}


    constructor TConvSapiSpeechRates.Create; // override;
    begin
      inherited;
      FName := 'Dyn sapi speech rates';
    end;


{ ========
  TConvESpeakSpeechVoices = class( TKnobsDynamicConverter)
  public
}
    constructor TConvESpeakSpeechVoices.Create; // override;
    begin
      FName := 'Dyn eSpeak speech voices';
    end;


{ ========
  TConvESpeakSpeechLanguages = class( TKnobsDynamicConverter)
  public
}
    constructor TConvESpeakSpeechLanguages.Create; // override;
    begin
      FName := 'Dyn eSpeak speech languages';
    end;



{ ========
  TKnobsConverters = class( TStringList)
  private
 }

    function    TKnobsConverters.GetConverterObject( anIndex: Integer): TKnobsNullConverter;
    begin
      Result := nil;

      if ( anIndex >= 0) and ( anIndex < Count)
      then Result := Objects[ anIndex] as TKnobsNullConverter;
    end;


    function    TKnobsConverters.Converter( const anIndex: string): TKnobsNullConverter;
    var
      p : Integer;
    begin
      p  := IndexOf( anIndex);

      if p < 0
      then begin
        p  := IndexOf( 'undefined converter');

        if p >= 0
        then ( Objects[ p] as TKnobsNullConverter).Searched := anIndex
        else raise ENoDefaultConverter.Create( 'TKnobsConverters should at the very least have a default converter ("undefined converter", TKnobsNullConverter).');
      end;

      Result := Objects[ p] as TKnobsNullConverter;
    end;


 // public

    constructor TKnobsConverters.Create;
    begin
      inherited;
      OwnsObjects   := True;
      Duplicates    := dupError;
      Sorted        := True;
      CaseSensitive := False;

      RegisterConverters
      ([
        // Static converters

        TKnobsNullConverter      , TConvUndefined           , TConvLinearMin1_05       , TConvLinear0_1           ,
        TConvLinear0_4           , TConvLinear0_16          , TConvLinear1_4           , TConvLambda              ,
        TConvLinearMin1_1        , TConvLinearMin1_0        , TConvLinearMin4_4        , TConvLinearMin8_8        ,
        TConvLinearMin16_16      , TConvdB0_1               , TConvdB0_4               , TConvdB0_16              ,
        TConvdB0_32              , TConvdB0_64              , TConvdBKS                , TConvdBResonator         ,
        TConvOscFreqCoarse       , TConvNoiseColor          , TConvOscLinCoarse        , TConvOscCubedCoarse      ,
        TConvLfoFreqCoarseFast   , TConvLfoFreqCoarseMedium , TConvLfoFreqCoarseSlow   , TConvLfoFreqCoarseFast2  ,
        TConvLfoFreqCoarseMedium2, TConvLfoFreqCoarseSlow2  , TConvLfoFreqCoarseBPM    , TConvLfoFreqCoarseLin    ,
        TConvEnvTimeFast         , TConvEnvTimeMedium       , TConvEnvTimeSlow         , TConvEnvTimeFastAr       ,
        TConvEnvTimeMediumAr     , TConvEnvTimeSlowAr       , TConvAdd0                , TConvAdd1                ,
        TConvAdd3                , TConvOdd                 , TConvQSteps              , TConvFractions           ,
        TConvMidiChIn            , TConvMidiChOut           , TConvNoteName            , TConvNoteName12          ,
        TConvDelayTimeShort      , TConvDelayTimeMedium     , TConvDelayTimeLong       , TConvFilterFreq          ,
        TConvAvgFilterAlpha      , TConvCubicClipAlpha      , TConvBlockDCAlpha        , TConvSlew                ,
        TConvCents               , TConvSemitones           , TConvSemitonesFull       , TConvTranspose           ,
        TConvMute                , TConvActivation          , TConvSolo                , TConvEnvMute             ,
        TConvVcpsMode            , TConvOutType             , TConvEnvOutType          , TConvEnvSlopeType        ,
        TConvWaveShape           , TConvPulseMode           , TConvPulseDelayMode      , TTalkieFrameRate         ,
        TTalkieSampleRate        , TRoomSize                , TReverbTime              , TRateMultiplier          ,
        TSwing                   , TSongSpeed               , THzMeasure               , TPhase                   ,
        TFreqMult                , TConvNormalHold          , TConvMixerModeSimple     , TConvMixerModeAll        ,
        TConvMixerModeExtended   , TConvMixerModeExtendedAmp, TConvSequencerMode       , TConvLfoRange            ,
        TConvEnvelopeRange       , TConvDelayRange          , TConvNoteStealing        , TConvAutoMode            ,
        TConvHrastOscType        , TConvNoiseType           , TConvAttractorType       , TConvInterpolationType   ,
        TConvOffOnType           , TConvTrigType            , TConvLinExpType          , TConvMirrorType          ,
        TConvSquareMode          , TConvFaderType           , TConvSkipMode            , TConvPeakRMS             ,
        TConvPerlinMode          , TConvFIMode              , TConvLoopMode            , TConvRPNType             ,
        TConvRPNUSel             , TConvMonoStereo          , TConvSawPulse            , TConvAsyncSync           ,
        TConvRandomWalkMode      , TConvSteppedSmooth       , TConvLFORetrig           , TConvGateMode            ,
        TConvRectifierMode       , TConvPinkMode            , TConv7kHzMode            , TConv28HzMode            ,
        TConv5k6HzMode           , TConv139HzMode           , TConvDemuxHoldMode       , TConvDigiDemuxHoldMode   ,
        TConvChebgenMode         , TConvQuantizeMode        , TConvQuantizeType        , TConvGateFunction        ,
        TConvPatternMode         , TConvChordDegree         , TConvChordType           , TConvOctaveShift         ,
        TConvTransInv            , TConvRSTrigMode          , TConvCounterMode         , TConvTSRandomMode        ,
        TConvRepOnceMode         , TConvNormInvertMode      , TConvPostPreMode         , TConvSeq16GateMode       ,
        TConvChordMode           , TConvRunEditMode         , TConvArpMode             , TConvArpInMode           ,
        TConvOnceContMode        , TConvWrappedPlainMode    , TConvWrappedPlainMode2   , TConvNormalFrozenMode    ,
        TConvCompressorBypassMode, TConvSideChainMode       , TConvAcDcMode            , TConvPhaseDistortMode    ,
        TConvClockedFastMode     , TConvValueMode           , TConvAbsPosMode          , TConvFastSlowMode        ,
        TConvGraphMode           , TConvClipWrapMode        , TConvWaveShapeLFO        , TConvLinear1_8           ,
        TConvMConst              , TConvMonadicFunction     , TConvDyadicFunction      , TConvLifeSeqMode         ,
        TConvLifeSeqSumMode      , TConvTerragenAlgorithm   , TConvConvoderSize        , TConvTapperMode          ,
        TConvVSteps              , TConvVStepsOff           , TConvPhaseShift          , TConvPWMPLM              ,
        TConvZNZ                 , TConvMuseSwitch          , TConvFreeClocked         , TConvEvenOdd             ,
        TConvDupsMode            , TConvScaledMode          , TConvFmDepth             , TConvSinCosSin           ,
        TConvNpMode              , TConvSampleRate          , TConvContrRateDecimation , TConvAutoReady           ,
        TConvAbsNorm             , TConvAllSkipMode         , TConvESpeakModifiers     , TConvESpeakPunctuation   ,
        TESpeakRate              , TESpeakPitch             , TESpeakRange             , TESpeakWordGap           ,

        // Dynamic converters

        TConvDynTalkieBank       , TConvDynFormant2Vowels   , TConvDynModal2           , TConvDynScale            ,
        TConvDynGrainEnvelopes   , TConvDynMorphSelect      , TConvDynMazeType         , TConvSapiSpeechVoices    ,
        TConvSapiSpeechRates     , TConvESpeakSpeechVoices  , TConvESpeakSpeechLanguages
      ]);

    end;


    procedure   TKnobsConverters.RegisterConverter( aClass: TKnobsConverterClass);
    var
      C: TKnobsNullConverter;
    begin
      C := aClass.Create;
      AddObject( C.Name, C);
    end;


    procedure   TKnobsConverters.RegisterConverters( const aClasses: array of TKnobsConverterClass);
    var
      C: TKnobsConverterClass;
    begin
      for C in aClasses
      do RegisterConverter( C);
    end;


    function    TKnobsConverters.PosToHint( const aName: string; aPos, aSteps: Integer): string;
    begin
      Result := Converter( aName).PosToHint( aPos, aSteps);
    end;


    function    TKnobsConverters.PosToDisplay( const aName: string; aPos, aSteps: Integer): string;
    begin
      Result := Converter( aName).PosToDisplay( aPos, aSteps);
    end;


    function    TKnobsConverters.PosToValue( const aName: string; aPos, aSteps: Integer): TSignal;
    begin
      Result := Converter( aName).PosToValue( aPos, aSteps);
    end;


    function    TKnobsConverters.ValueToPos( const aName: string; aVal: TSignal; aSteps: Integer): Integer;
    begin
      Result := Converter( aName).ValueToPos( aVal, aSteps);
    end;


    function    TKnobsConverters.DisplayToValue( const aName, aDisplay: string; var aValue: TSignal): Boolean;
    begin
      Result := Converter( aName).DisplayToValue( aDisplay, aValue);
    end;


    function    TKnobsConverters.ValueToDisplay( const aName: string; const aValue: TSignal; aSteps: Integer): string;
    begin
      Result := Converter( aName).ValueToDisplay( aValue, aSteps);
    end;


    function    TKnobsConverters.IsEnumerable( const aName: string): Boolean;
    begin
      Result := Converter( aName).IsEnumerable;
    end;


    function    TKnobsConverters.IsRandomizable( const aName: string): Boolean;
    begin
      Result := Converter( aName).IsRandomizable;
    end;


    procedure   TKnobsConverters.SetRandomizable( const aName: string; aValue: Boolean);
    begin
      Converter( aName).IsRandomizable := aValue;
    end;


    procedure   TKnobsConverters.SetRandomizableByClass( const aName: string; aValue: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do begin
        if SameText( ConverterObject[ i].ClassName, aName)
        then begin
          ConverterObject[ i].IsRandomizable := aValue;
          Break;
        end;
      end;
    end;


    function    TKnobsConverters.GetRandomizableByClass( const aName: string): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      for i := 0 to Count - 1
      do begin
        if SameText( ConverterObject[ i].ClassName, aName)
        then begin
          Result := ConverterObject[ i].IsRandomizable;
          Break;
        end;
      end;
    end;


    function    TKnobsConverters.GetDynamicConverter( const aName: string): TKnobsDynamicConverter;
    var
      aConverter : TKnobsNullConverter;
    begin
      Result     := nil;
      aConverter := Converter( aName);

      if   Assigned( aConverter)
      and  ( aConverter is TKnobsDynamicConverter)
      then Result := TKnobsDynamicConverter( aConverter);
    end;


    function    TKnobsConverters.CreateEnumeration( const aName: string): TStringList;
    begin
      Result := Converter( aName).CreateEnumeration;
    end;


  // ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  procedure InitializeUnit;
  begin
    ReferenceA         := 440.0;
    NotesPerOctave     := 12.0;
    NotesPerOctaveRec  := 1.0 / NotesPerOctave;
    MiddleNote         := 69;
    OctaveSpan         := 2.0;
    System_Rate        := 22050;        // To be overriden later
    Control_Decimation := 4;            // Must be a power of two
    SetSystemRate             ( 44100); // Overide system rate and force change callbacks to be called into
    SetSystemControlDecimation(     8); // Must be a power of two
    ComputeTables;
  end;


  procedure CreateConverters;
  var
    i : Integer;
    M : Int64;
  begin
    LocConverters := TKnobsConverters.Create;
    M             := 2;

    for i := Low( TBitCount) to High( TBitCount)
    do begin
      Shifts   [ i] := M;
      RecShifts[ i] := 1 / M;
      M             := M shl 1;
    end;
  end;


  procedure FreeConverters;
  begin
    FreeAndNil( LocConverters);
  end;


initialization

  CreateConverters;
  InitializeUnit;

finalization

  FreeConverters;

end.

