unit KnobsForth;

{

   COPYRIGHT 2017 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  WinApi.Windows, WinApi.ShellApi, System.Types, System.SysUtils, System.DateUtils, System.Classes, System.Contnrs,
  System.Variants, System.Math,

  KnobsUtils;

const

  endoffile    = '#EOF'; // special token denoting end of file
  NoName       = '';
  Nothing      = '<nothing-50E13785-E2DF-4ADB-BE9F-2931DB2C4809>'; // magic cookie
  Marker       = '<marker--C8A47D06-69B5-4D3A-8077-B550111F3F51>'; // magic cookie

  CBeginUntilWhile =  1;
  CWhileRepeat     =  2;
  CDoLoop          =  3;
  CIfElseThen      =  4;
  CColonSemi       =  5;
  CCaseEndCase     =  6;
  COfEndOf         =  7;
  CInEndIn         =  8;
  CStructField     =  9;
  CTryExcept       = 10;
  CExceptEndTry    = 11;
  CPastLast        = 12;

  ControlNames : array[ CBeginUntilWhile .. CPastLast - 1] of string = (
    '''until'' or ''while''',
    '''repeat''',
    '''loop'' or ''+loop''',
    '''else'', ''then'' or ''endif''',
    ''';''',
    '''endCase''',
    '''endOf''',
    '''endIn''',
    '''endStruct''',
    '''except'', ''catch'' or ''finally''',
    '''endTry'''
  );
  ControlIntros : array[ CBeginUntilWhile .. CPastlast - 1] of string = (
    '''begin''',
    '(''begin'' ..) ''while''',
    '''do'' or ''?do''',
    '''if'' or ''else''',
    ''':'' (colon)',
    '''case''',
    '''of''',
    '''in''',
    '''struct'', ''field'' or ''2field''',
    '''try''',
    '''except'', ''catch'' or ''finally'''
  );


type

  TCardinals = array of Cardinal;
  TStack     = class;

  TOnForthResponse  = procedure( aSender: TObject; const aMsg: string)                                        of object;
  TOnStackUnderflow = procedure( aSender: TObject; const aMsg: string)                                        of object;
  TOnExternalFunc   = procedure( aSender: TObject; anId: Integer; const aStack: TStack)                       of object;


  EForth = class( Exception);
  TWord  = Class;
  TForth = class;

  TStack = class
  private
    FSystem      : TForth;
    FName        : string;
    FData        : array of Variant;
    FOnUnderflow : TOnStackUnderflow;
  private
    function    GetDepth: Integer;
  public
    constructor Create( aSystem: TForth; const aName: string);
    destructor  Destroy;                                                                                       override;
    procedure   Push( aValue: Variant);
    procedure   PushWord( const aValue: TWord);
    function    Pop: Variant;
    function    PopWord: TWord;
    function    Index( anIndex: Integer): Variant;
    function    Fetch: Variant;
    procedure   Swap;
    procedure   Clear;
    function    Dump( aBase: Integer): string;
  public
    property    Depth       : Integer           read GetDepth;
    property    OnUnderflow : TOnStackUnderflow read FOnUnderflow write FOnUnderflow;
  end;

  TWord = class
  public
    class var
      WordCounter      : Integer;
      AnonymousCounter : Integer;
  private
    FSystem    : TForth;
    FName      : string;       // Name of the word (can be '')
    FComment   : string;       // Comment for word, probably stack image
    FCFA       : Integer;
    FImmediate : Boolean;      // True when executing in compilation mode
    FVisible   : Boolean;      // True when word can be found in dictionary
    FExported  : Boolean;      // True when the word is exportable
  public
    constructor Create( aSystem: TForth; const aName: string);                                                  virtual;
    destructor  Destroy;                                                                                       override;
    procedure   EndCreate;                                                                                      virtual;
    procedure   Interpret;                                                                                      virtual;
    procedure   Compile( const aStringList: TStringList);                                                       virtual;
    procedure   Hide;                                                                                           virtual;
    procedure   Reveal;                                                                                         virtual;
    function    See: string;                                                                                    virtual;
    function    ShowWordInfo: string;                                                                           virtual;
    procedure   HandleForget;                                                                                   virtual;
    function    AppendFlags( const S: string): string;                                                          virtual;
  public
    class function TypeName: string;
  public
    property    Name      : string  read FName;
    property    Comment   : string  read FComment   write FComment;
    property    CFA       : Integer read FCFA       write FCFA;
    property    Immediate : Boolean read FImmediate write FImmediate;
    property    Exported  : Boolean read FExported  write FExported;
    property    Visible   : Boolean read FVisible   write FVisible;
  end;
  TWordClass = class of TWord;


  TWords = class( TList)
  private
    FIndex  : TStringList;
    FSystem : TForth;
  private
    function    GetWord( anIndex: Integer): TWord;
  public
    constructor Create( aSystem: TForth);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;                                                                                         override;
    function    AddWord( const aName: string; aWordClass: TWordClass; aCFA: Integer): TWord;                   overload;
    function    AddWord( aWord: TWord; aCFA: Integer): TWord;                                                  overload;
    function    RemoveWord( const aName: string): TWord;
    function    FindToken( const aToken: string): TWord;
    function    CfaToWord( aCFA: Integer): TWord;
    function    Forget( const aWord: TWord): Boolean;
    function    Words: string;
    function    CreateExportList: TStringList;
  public
    property    Word[ anIndex: Integer]: TWord read GetWord; default;
  end;

  TCode = class( TWord)
  strict private
    FValue : TList;
  strict private
    function    GetCount: Integer;
    procedure   SetCount( aValue: Integer);
  public
    constructor Create( aSystem: TForth; const aName: string);                                                 override;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
    procedure   Append( aValue: TWord);
    function    GetWord( anIndex: Integer): TWord;
    procedure   AcceptToken( const aToken: string);
    procedure   CompileLiteral( const aValue: Variant);
    procedure   CompileWord( const aWord: TWord);
    function    Here: Integer;
    procedure   SetHere( aValue: Integer);
    procedure   Reverse;
    function    First: TWord;
    procedure   Rest;
    procedure   Concat( const aValue: TCode);
  public
    property    Count: Integer read GetCount write SetCount;
  end;

  TData = class( TWord)
  private
    FValue : Variant;
  public
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  public
    property    Value: Variant read FValue write FValue;
  end;

  TConstant = class( TData)
  public
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  end;

  TVariable = class( TConstant)
  public
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  end;

  TAlias = class( TWord)
  private
    FAliased : TWord;
  public
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  public
    property    Aliased: TWord read FAliased write FAliased;
  end;

  TDefer = class( TAlias)
  public
    procedure   Resolve( aWord: TWord);
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  end;

  TExecutable = class( TWord)
  end;

  TInternalProc = procedure( aSystem: TForth);

  TInternal = class( TExecutable)
  private
    FInternalProc: TInternalProc;
  public
    procedure   Interpret;                                                                                     override;
  public
    property    InternalProc: TInternalProc read FInternalProc write FInternalProc;
  end;

  TColon = class( TExecutable)
  public
    procedure   Interpret;                                                                                     override;
    function    See: string;                                                                                   override;
  end;

  TCreate = class( TColon)
  public
    procedure   Interpret;                                                                                     override;
  end;

  TForth = class
  private
    FOnResponse        : TOnForthResponse;
    FOnSave            : TNotifyEvent;
    FOnExternalFunc    : TOnExternalFunc;
  private
    FDataStack         : TStack;
    FLoopStack         : TStack;
    FReturnStack       : TStack;
    FControlStack      : TStack;
    FCode              : TCode;                // Compiled forth code
    FWords             : TWords;               // The dictionary
    FAnonymous         : TObjectList;          // List of all anonymous words (so they can be freed)
    FInputBuffer       : string;
    FCurrToken         : string;
    FLastToken         : string;
    FOffset            : Integer;
    FBase              : Integer;
    FInvoker           : Integer;
    FCurrentFileName   : string;
  private
    FCompiling         : Boolean;
    FWreg              : Integer;              // Forth W register - needed for create .. does> - index into FCode,
                                               // a CFA
    FIP                : Integer;              // Forth instruction pointer
    FCW                : TWord;                // Forth code word
    FLastDefined       : TWord;                // The last defined word
    FCurrentColon      : TColon;               // The colon definition under construction
  public
    procedure   RaiseError( const aMsg: string);
    procedure   RaiseErrorFmt( const aFmt: string; const anArgs: array of const);
    procedure   ShowWarning( const aMsg: string);
    procedure   ShowWarningFmt( const aFmt: string; const anArgs: array of const);
    procedure   Respond( const S: string);
    procedure   RespondFmt( const aFmt: string; const anArgs: array of const);
    procedure   DoStackUnderflow( aSender: TObject; const aMsg: string);
    function    GetString( aTerminator: Char): string;
    function    GetToken( aSeparator: Char; Extended: Boolean = False): string;
    procedure   PutToken( aSeparator: Char; const aToken: string);
    function    Number                 ( const aToken: string; out aNumber: Variant): Boolean;
    function    Float                  ( const aToken: string; out aNumber: Variant): Boolean;
    function    MustFindToken          ( const aToken: string; aWordClass: TWordClass): TWord;
    function    MustFindWord           ( const aToken: string): TWord;
    procedure   MustBeCompilingFor     ( const aToken: string);
    procedure   MustNotBeCompilingFor  ( const aToken: string);
    function    FindToken( const aToken: string): TWord;
    procedure   AcceptData( const aHandler: string; const aData: TCardinals);
    procedure   AcceptToken( const aToken: string);
    procedure   Explain( aWord: TWord);
    procedure   See;
    procedure   ShowWordInfo;
    procedure   Exec( aWord: TWord);
    procedure   Interpret( const aWord: TWord);
    procedure   AddDefer( const aName: string);
    procedure   AddAlias( const anOldName, aNewName: string);
    procedure   CompileLiteral( const aValue: Variant);
    procedure   Compile( const aWord: TWord);
    function    Here: Integer;
    procedure   SetHere( aValue: Integer);
    function    DataDump: string;
    function    DataDepth: Integer;
    procedure   DataPush( aValue: Variant);
    procedure   DataPushWord( const aValue: TWord);
    function    DataPop: Variant;
    function    DataFetch: Variant;
    function    DataPopWord: TWord;
    procedure   DataSwap;
    function    LoopDump : string;
    procedure   LoopPush ( aValue: Variant);
    function    LoopPop  : Variant;
    function    ReturnDump : string;
    procedure   ReturnPush ( aValue: Variant);
    function    ReturnPop  : Variant;
    function    ReturnFetch: Variant;
    function    ControlDump: string;
    procedure   ControlPush( aValue: Variant);
    function    ControlPop: Variant;
    function    StacksDump: string;
    function    AddWord( const aName: string; aWordClass: TWordClass): TWord;
    procedure   InternalWord( const aName: string; aProc: TInternalProc; const aComments: string);
    procedure   SetupInternalWords;
    procedure   Immediate;
    procedure   Exported;
    procedure   Hide;
    procedure   Reveal;
    procedure   ExternalWord;
    procedure   CleanupAnonymousWords;
    procedure   Forget;
    procedure   Unload;
    procedure   CheckControlStructure( aSeen: Integer);
    procedure   CheckDataStack   ( const aMsg: string);
    procedure   CheckControlStack( const aMsg: string);
    procedure   CheckLoopStack   ( const aMsg: string);
    procedure   CheckReturnStack ( const aMsg: string);
    procedure   CheckStacks( const aMsg: string);
    function    Words: string;
    function    CreateExportList: TStringList;
    function    CfaToWord( aCFA: Integer): TWord;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   StackReset;
    procedure   Reset;
    procedure   Save( const aFileName: string);
    procedure   AcceptText( const aText: string; anInvoker: Integer);
    procedure   SaveWords( const aStringList: TStringList);                                                    overload;
    procedure   SaveWords( const aFileName: string);                                                           overload;
    procedure   LoadWords( const aStringList: TStringList; anInvoker: Integer);                                overload;
    procedure   LoadWords( const aFileName: string       ; anInvoker: Integer);                                overload;
  public
    property    CurrentFileName : string           read FCurrentFileName;
    property    OnResponse      : TOnForthResponse read FOnResponse      write FOnResponse;
    property    OnSave          : TNotifyEvent     read FOnSave          write FOnSave;
    property    OnExternalFunc  : TOnExternalFunc  read FOnExternalFunc  write FOnExternalFunc;
  end;




implementation



    function WordToVariant( const anObject: TWord): Variant;
    begin
      Result := Integer( anObject);
      TVarData( Result).VType := varByRef;
    end;


    function VariantIsString( V: Variant): Boolean;
    begin
      Result := VarIsStr( V);
    end;


    function VariantIsTWord( V: Variant): Boolean;
    var
      W : TWord;
    begin
      if VarIsType( V, varByRef)
      then begin
        TVarData( V).VType := varInteger;
        W := TWord( Integer( V));
        Result := W is TWord;
      end
      else Result := False;
    end;


    function VariantIsTCode( V: Variant): Boolean;
    var
      W : TWord;
    begin
      if VarIsType( V, varByRef)
      then begin
        TVarData( V).VType := varInteger;
        W := TWord( Integer( V));
        Result := W is TCode;
      end
      else Result := False;
    end;


    function VariantToWord( V: Variant): TWord;
    begin
      TVarData( V).VType := varInteger;
      Result := TWord( Integer( V));
    end;


    function VariantToCode( V: Variant): TCode;
    begin
      TVarData( V).VType := varInteger;
      Result := TCode( Integer( V));
    end;


    function  LongToBase( aValue: Cardinal; aBase: Cardinal; aFieldWidth: Integer = 1): string;

      function ToDigit( aValue, aBase: Cardinal): string;
      begin
        Assert( aValue < aBase, 'ToDigit: Digit conversion error, aValue >= aBase');
        if aBase <= 62
        then
          if aValue > 9
          then begin
            if aValue > 35
            then Result := Char( aValue + Ord( '0') + 13)
            else Result := Char( aValue + Ord( '0') +  7);
          end
          else Result := Char( aValue + Ord( '0'))
        else Result := Format( '<&%d>', [ aValue]);
      end;

    begin
      Result := '';

      while aValue <> 0
      do begin
        Result := ToDigit( aValue mod aBase, aBase) + Result;
        aValue := aValue div aBase;
      end;

      if aFieldWidth > 0
      then begin
        while Length( Result) < aFieldWidth
        do Result := '0' + Result;
      end;
    end;


    function  StrToUIntBase( const aValue: string; aBase: Byte): Cardinal;

    const
      Digits : array[ 0 .. 15] of char =
      (
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
      );
      function LookUp( aDig: char): Byte;
      var
        n : Integer;
      begin
        Result := $ff;
        if ( aDig >= '0') and ( aDig <= 'F')
        then begin
          n := Ord( aDig) - Ord( '0');
          if n > 9
          then n := 10 + n - ( Ord( 'A') - Ord( '0'));
          Result := n;
        end;
      end;

    var
      C : Char;
      N : Byte;
    begin
      Result := 0;

      if ( aBase >= 2) and ( aBase <= 16)
      then begin
        for C in aValue
        do begin
          N := Lookup( UpCase( C));
          if ( N < aBase) and ( N <> $ff)

          then Result := Result * aBase + N
          else Break;
        end;
      end;
    end;


    function DataToString( aValue: Variant; aBase: Integer): string;
    var
      aWord : TWord;
    begin
      if VariantIsTWord( aValue)
      then begin
        aWord  := VariantToWord( aValue);
        Result := aWord.See;
      end
      else if VarIsOrdinal( aValue)
      then begin
        if aValue < 0
        then Result := '-' + LongToBase( - aValue, aBase)
        else Result :=       LongToBase(   aValue, aBase);
      end
      else Result := aValue;
    end;


    function DataToEngineering( aValue: Variant; aBase, aDigits: Integer): string;

      function FixUnits( aValue, aMultiplier: Double; const aUnits: string): string;
      var
        aFmt : string;
      begin
        aFmt   := Format( '%%.%df %%s', [ aDigits]);
        Result := Format( aFmt, [ aValue * aMultiplier, aUnits]);
      end;

    var
      aWord  : TWord;
      aFloat : Double;
      aStr   : string;
    begin
      if VariantIsTWord( aValue)
      then begin
        aWord  := VariantToWord( aValue);
        Result := aWord.See;
      end
      else if VarIsOrdinal( aValue)
      then begin
        if aValue < 0
        then Result := '-' + LongToBase( - aValue, aBase)
        else Result :=       LongToBase(   aValue, aBase);
      end
      else if VarIsFloat( aValue)
      then begin
        aStr   := aValue;
        aFloat := aValue;

        if      ( aFloat >= 1e24 ) and ( aFloat < 1e27 ) then Result := FixUnits( aFloat, 1e-24, 'Y (1e24)' )
        else if ( aFloat >= 1e21 ) and ( aFloat < 1e24 ) then Result := FixUnits( aFloat, 1e-21, 'Z (1e21)' )
        else if ( aFloat >= 1e18 ) and ( aFloat < 1e21 ) then Result := FixUnits( aFloat, 1e-18, 'E (1e18)' )
        else if ( aFloat >= 1e15 ) and ( aFloat < 1e18 ) then Result := FixUnits( aFloat, 1e-15, 'P (1e15)' )
        else if ( aFloat >= 1e12 ) and ( aFloat < 1e15 ) then Result := FixUnits( aFloat, 1e-12, 'T (1e12)' )
        else if ( aFloat >= 1e9  ) and ( aFloat < 1e12 ) then Result := FixUnits( aFloat, 1e-9 , 'G (1e9)'  )
        else if ( aFloat >= 1e6  ) and ( aFloat < 1e9  ) then Result := FixUnits( aFloat, 1e-6 , 'M (1e6)'  )
        else if ( aFloat >= 1e3  ) and ( aFloat < 1e6  ) then Result := FixUnits( aFloat, 1e-3 , 'k (1e3)'  )
        else if ( aFloat >= 1    ) and ( aFloat < 1e3  ) then Result := FixUnits( aFloat, 1    , ''         )
        else if ( aFloat >= 1e-3 ) and ( aFloat < 1    ) then Result := FixUnits( aFloat, 1e3  , 'm (1e-3)' )
        else if ( aFloat >= 1e-6 ) and ( aFloat < 1e-3 ) then Result := FixUnits( aFloat, 1e6  , ' (1e-6)' )
        else if ( aFloat >= 1e-9 ) and ( aFloat < 1e-6 ) then Result := FixUnits( aFloat, 1e9  , 'p (1e-9)' )
        else if ( aFloat >= 1e-12) and ( aFloat < 1e-9 ) then Result := FixUnits( aFloat, 1e12 , 'n (1e-12)')
        else if ( aFloat >= 1e-15) and ( aFloat < 1e-12) then Result := FixUnits( aFloat, 1e15 , 'f (1e-15)')
        else if ( aFloat >= 1e-18) and ( aFloat < 1e-15) then Result := FixUnits( aFloat, 1e18 , 'a (1e-18)')
        else if ( aFloat >= 1e-21) and ( aFloat < 1e-18) then Result := FixUnits( aFloat, 1e21 , 'z (1e-21)')
        else if ( aFloat >= 1e-24) and ( aFloat < 1e-21) then Result := FixUnits( aFloat, 1e24 , 'y (1e-21)');
      end
      else Result := aValue;
    end;


    procedure _reset( aSystem: TForth);
    // reset ( - )
    begin
      aSystem.Respond( 'ok');
      aSystem.Reset;
    end;


    procedure _forget( aSystem: TForth);
    // forget ( - )
    begin
      with aSystem
      do Forget;
    end;


    procedure _unload( aSystem: TForth);
    // unload ( - )
    begin
      with aSystem
      do Unload;
    end;


    procedure _external( aSystem: TForth);
    // external ( ... func_id -- ) // Call into an externally defined function');
    begin
      with aSystem
      do ExternalWord;
    end;


    procedure _words( aSystem: TForth);
    begin
      with aSystem
      do Respond( Words);
    end;


    procedure _tostring( aSystem: TForth);
    begin
      with aSystem
      do DataPush( DataToString( DataPop, FBase));
    end;


    procedure _format( aSystem: TForth);
    // format ( n fmt -- )
    var
      aFmt : string;
    begin
      with aSystem
      do begin
        aFmt := DataPop;
        DataPush( Format( aFmt, [ Double( DataPop)]));
      end;
    end;


    procedure _dot( aSystem: TForth);
    // . ( n -- )
    begin
      with aSystem
      do Respond( DataToString( DataPop, FBase));
    end;


    procedure _pdote( aSystem: TForth);
    // (.e) ( n digits -- )
    var
      aDigits: Integer;
    begin
      with aSystem
      do begin
        aDigits := DataPop;
        Respond( DataToEngineering( DataPop, FBase, aDigits));
      end;
    end;


    procedure _dotbase( aSystem: TForth);
    // .base ( n base -- )
    var
      aBase  : Integer;
      aValue : Int64;
    begin
      with aSystem
      do begin
        aBase  := DataPop;
        aValue := DataPop;
        Respond( LongToBase( aValue, aBase));
      end;
    end;


    procedure _dots( aSystem: TForth);
    // .s ( n - )
    begin
      with aSystem
      do Respond( FDataStack.Dump( FBase));
    end;


    procedure _dotstacks( aSystem: TForth);
    // .s ( n - )
    begin
      with aSystem
      do Respond( StacksDump);
    end;


    procedure _decimal( aSystem: TForth); // Base := D'10'
    // Decimal ( -- )
    // Set base to &10
    begin
      aSystem.FBase := 10;
    end;


    procedure _hex( aSystem: TForth); // Base := D'16'
    // Hex ( -- )
    // Set base to &16
    begin
      aSystem.FBase := 16;
    end;


    procedure _binary( aSystem: TForth); // Base := D'2'
    // Binary ( -- )
    // Set base to &2
    begin
      aSystem.FBase := 2;
    end;


    procedure _baseFetch( aSystem: TForth);
    // Base@ ( -- n )
    // Get current number base
    begin
      with aSystem
      do DataPush( FBase);
    end;


    procedure _baseStore( aSystem: TForth);
    // Base! ( n -- )
    // Set current number base
    begin
      with aSystem
      do FBase := DataPop;
    end;


    procedure _lparen( aSystem: TForth);
    // ( ( -- )
    // Comment up to )
    var
      S : string;
    begin
      with aSystem
      do begin
        S := GetToken( ')');

        if SameText( S, endoffile)
        then
          RaiseError(
            'unexpected end of text after ('
          );

        if   ( FLastDefined is TColon)
        and  ( FLastDefined.Comment = '')
        then FLastDefined.Comment := S;
      end;
    end;


    procedure _slashslash( aSystem: TForth);
    // // ( -- )
    // Comment up to end of input buffer
    begin
      with aSystem
      do FInputBuffer := '';
    end;


    procedure _swap( aSystem: TForth);
    // Swap ( n1 n2 -- n2 n1 )
    var
      A : Variant;
      B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;
        DataPush( A);
        DataPush( B);
      end;
    end;


    procedure _plus( aSystem: TForth);
    // + ( n n - n )
    var
      a1, a2 : Variant;
    begin
      with aSystem
      do begin
          // Weird, but makes string addition to be performed left to right
          // and for integer addition it does not matter ...
        a1 := DataPop;
        a2 := DataPop;
        DataPush( a2 + a1);
      end;
    end;


    procedure _minus( aSystem: TForth);
    // - ( n n - n )
    begin
      with aSystem
      do DataPush( - DataPop + DataPop);
    end;


    procedure _times( aSystem: TForth);
    // * ( n n - n )
    begin
      with aSystem
      do DataPush( DataPop * DataPop);
    end;


    procedure _div( aSystem: TForth);
    // / ( n n - n )
    var
      A : Variant;
      B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;

        if VarIsFloat( A) or VarIsFloat( B)
        then DataPush( B / A)
        else DataPush( B div A);
      end;
    end;


    procedure _divMod( aSystem: TForth);
    // /mod ( n n - n n )
    var
      A : Variant;
      B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;

        if VarIsFloat( A) or VarIsFloat( B)
        then DataPush( B / A)
        else DataPush( B div A);

        DataPush( B mod A);
      end;
    end;


    procedure _int( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Int( DataPop));
    end;


    procedure _frac( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do begin
        DataPush( Frac( DataPop));
      end;
    end;


    procedure _power( aSystem: TForth);
    // / ( n n - n )
    var
      A : Variant;
      B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;
        DataPush( Power( A, B));
      end;
    end;


    procedure _exp( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Exp( DataPop));
    end;


    procedure _ln( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Ln( DataPop));
    end;


    procedure _rad2deg( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( RadToDeg( DataPop));
    end;


    procedure _deg2rad( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( DegToRad( DataPop));
    end;


    procedure _sin( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Sin( DataPop));
    end;


    procedure _asin( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcSin( DataPop));
    end;


    procedure _cos( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Cos( DataPop));
    end;


    procedure _acos( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcCos( DataPop));
    end;


    procedure _tan( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Tan( DataPop));
    end;


    procedure _atan( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcTan( DataPop));
    end;


    procedure _atan2( aSystem: TForth);
    // / ( n1 n2 - n )
    var
      A : Variant;
      B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;
        DataPush( ArcTan2( B, A));
      end;
    end;


    procedure _cosh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Cosh( DataPop));
    end;


    procedure _sinh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Sinh( DataPop));
    end;


    procedure _tanh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( Tanh( DataPop));
    end;


    procedure _atanh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcTanh( DataPop));
    end;


    procedure _sech( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( SecH( DataPop));
    end;


    procedure _csch( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( CscH( DataPop));
    end;


    procedure _acot( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcCot( DataPop));
    end;


    procedure _asec( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcSec( DataPop));
    end;


    procedure _acsc( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcCsc( DataPop));
    end;


    procedure _acosh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcCosh( DataPop));
    end;


    procedure _asinh( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcSinh( DataPop));
    end;


    procedure _acoth( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcCotH( DataPop));
    end;


    procedure _asech( aSystem: TForth);
    // / ( n - n )
    begin
      with aSystem
      do DataPush( ArcSecH( DataPop));
    end;


    procedure _acsch( aSystem: TForth);
    // / ( n n - n )
    begin
      with aSystem
      do DataPush( ArcCscH( DataPop));
    end;


    procedure _sqrt( aSystem: TForth);
    // / ( n n - n )
    begin
      with aSystem
      do DataPush( Sqrt( DataPop));
    end;


    procedure _and( aSystem: TForth);
    // And ( n n - n )
    begin
      with aSystem
      do DataPush( DataPop and DataPop);
    end;


    procedure _or( aSystem: TForth);
    // Or ( n n - n )
    begin
      with aSystem
      do DataPush( DataPop or DataPop);
    end;


    procedure _xor( aSystem: TForth);
    // Xor ( n n - n )
    begin
      with aSystem
      do DataPush( DataPop xor DataPop);
    end;


    procedure _constant( aSystem: TForth);
    // Constant XXXX ( n -- )
    // define a constant
    var
      aConstant: TConstant;
    begin
      with aSystem
      do begin
        aConstant := AddWord( GetToken( ' '), TConstant) as TConstant;
        aConstant.Value := DataPop;
        FLastDefined := aConstant;

        with FLastDefined
        do begin
          Reveal;
          EndCreate;
        end;
      end;
    end;


    procedure _variable( aSystem: TForth);
    // Variable XXXX ( -- )
    // Define a variable
    var
      aVariable : TVariable;
    begin
      with aSystem
      do begin
        aVariable := TVariable( AddWord( GetToken( ' '), TVariable));
        FLastDefined := aVariable;

        with FLastDefined
        do begin
          Reveal;
          EndCreate;
        end;
      end;
    end;


    procedure _fetch( aSystem: TForth);
    // @ ( var-token -- val )
    begin
      with aSystem
      do DataPush(( DataPopWord as TData).Value);
    end;


    procedure _store( aSystem: TForth);
    // ! ( val var-token -- )
    var
      aVar : TData;
    begin
      with aSystem
      do begin
        aVar := DataPopWord as TData;
        aVar.Value := DataPop;
      end;
    end;


    procedure _plusStore( aSystem: TForth);
    // +! ( val var-token -- )
    var
      aVar : TVariable;
    begin
      with aSystem
      do begin
        aVar := DataPopWord as TVariable;
        aVar.Value := aVar.Value + DataPop;
      end;
    end;


    procedure _minStore( aSystem: TForth);
    // -! ( val var-token -- )
    var
      aVar : TVariable;
    begin
      with aSystem
      do begin
        aVar := DataPopWord as TVariable;
        aVar.Value := aVar.Value - DataPop;
      end;
    end;


    procedure _drop( aSystem: TForth);
    // Drop ( n -- )
    begin
      with aSystem
      do DataPop;
    end;


    procedure _dup( aSystem: TForth);
    // Drop ( n -- )
    var
      A : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        DataPush( A);
        DataPush( A);
      end;
    end;


    procedure _true( aSystem: TForth);
    // True ( -- f )
    begin
      with aSystem
      do DataPush( True);
    end;


    procedure _false( aSystem: TForth);
    // False ( -- f )
    begin
      with aSystem
      do DataPush( False);
    end;


    procedure _e( aSystem: TForth);
    // e ( -- n )
    begin
      with aSystem
      do DataPush( Exp( 1));
    end;


    procedure _pi( aSystem: TForth);
    // pi ( -- n )
    begin
      with aSystem
      do DataPush( pi);
    end;


    procedure _space( aSystem: TForth);
    // Space ( -- n )
    begin
      with aSystem
      do DataPush( ' ');
    end;


    procedure _negate( aSystem: TForth);
    // Negate ( n -- -n )
    begin
      with aSystem
      do DataPush( - DataPop);
    end;


    procedure _not( aSystem: TForth);
    // Not ( n -- !n )
    begin
      with aSystem
      do DataPush( not DataPop);
    end;


    procedure _equal( aSystem: TForth);
    // = ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop = DataPop);
    end;


    procedure _unequal( aSystem: TForth);
    // <> ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop <> DataPop);
    end;


    procedure _smaller( aSystem: TForth);
    // < ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop > DataPop);
    end;


    procedure _greater( aSystem: TForth);
    // > ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop < DataPop);
    end;


    procedure _smallerEqual( aSystem: TForth);
    // <= ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop >= DataPop);
    end;


    procedure _greaterEqual( aSystem: TForth);
    // >= ( n1 n2 -- f )
    begin
      with aSystem
      do DataPush( DataPop <= DataPop);
    end;


    procedure _random( aSystem: TForth);
    // >= ( n1 n2 -- n3 )
    var
      A : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        DataPush( DataPop + Floor( $ffffff * Random) mod A);
      end;
    end;


    procedure _format_time( aSystem: TForth);
    // format_time ( nt nf -- s )
    var
      F : string;
    begin
      with aSystem
      do begin
        F := DataPop;
        DataPush( FormatDateTime( F, DataPop));
      end;
    end;


    procedure _questionDup( aSystem: TForth);
    // ?dup ( false -- false | true -- true true )
    var
      A : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        DataPush( A);

        if A <> 0
        then DataPush( A);
      end;
    end;


    procedure _2drop( aSystem: TForth);
    // 2Drop ( n n -- )
    begin
      with aSystem
      do begin
        DataPop;
        DataPop;
      end;
    end;


    procedure _over( aSystem: TForth);
    // Over ( n1 n2 -- n1 n2 n1 )
    var
      A, B : Variant;
    begin
      with aSystem
      do begin
        A := DataPop;
        B := DataPop;
        DataPush( B);
        DataPush( A);
        DataPush( B);
      end;
    end;


    procedure _2dup( aSystem: TForth);
    // 2Dup ( n1 n2 -- n1 n2 n1 n2 )
    begin
      with aSystem
      do begin
        _over( aSystem);
        _over( aSystem);
      end;
    end;


    procedure _2swap( aSystem: TForth);
    // 2Swap ( n1 n2 n3 n4 -- n3 n4 n1 n2 )
    var
      t1, t2, t3, t4 : Variant;
    begin
      with aSystem
      do begin
        t4 := DataPop;
        t3 := DataPop;
        t2 := DataPop;
        t1 := DataPop;

        DataPush( t3);
        DataPush( t4);
        DataPush( t2);
        DataPush( t1);
      end;
    end;


    procedure _lmshift( aSystem: TForth);
    // << ( n1 n2 -- n3 )
    // Shift n1 left n2 bit
    var
      aCnt : Variant;
    begin
      with aSystem
      do begin
        aCnt := DataPop;
        DataPush( DataPop shl aCnt);
      end;
    end;


    procedure _rmshift( aSystem: TForth);
    // >> ( n1 n2 -- n3 )
    // Shift n1 right n2 bit
    var
      aCnt : Variant;
    begin
      with aSystem
      do begin
        aCnt := DataPop;
        DataPush( DataPop Shr aCnt);
      end;
    end;


    procedure _2over( aSystem: TForth);
    // 2Over ( n1 n2 n3 n4 -- n1 n2 n3 n4 n1 n2 )
    var
      A, B, C, D : Variant;
    begin
      with aSystem
      do begin
        D := DataPop;
        C := DataPop;
        B := DataPop;
        A := DataPop;
        DataPush( A);
        DataPush( B);
        DataPush( C);
        DataPush( D);
        DataPush( A);
        DataPush( B);
      end;
    end;


    procedure _rot( aSystem: TForth);
    // Rot ( n1 n2 n3 -- n2 n3 n1 )
    var
      A, B, C : Variant;
    begin
      with aSystem
      do begin
        C := DataPop;
        B := DataPop;
        A := DataPop;
        DataPush( B);
        DataPush( C);
        DataPush( A);
      end;
    end;


    procedure _plusrot( aSystem: TForth);
    // +Rot ( n1 n2 n3 -- n2 n3 n1 )
    begin
      with aSystem
      do _rot( aSystem);
    end;


    procedure _minusrot( aSystem: TForth);
    // -Rot ( n1 n2 n3 -- n3 n1 n2 )
    var
      A, B, C : Variant;
    begin
      with aSystem
      do begin
        C := DataPop;
        B := DataPop;
        A := DataPop;
        DataPush( C);
        DataPush( A);
        DataPush( B);
      end;
    end;


    procedure _abs( aSystem: TForth);
    // Abs ( n1 -- n2 )
    begin
      with aSystem
      do DataPush( Abs( DataPop));
    end;


    procedure _max( aSystem: TForth);
    // Max ( n1 n2 -- max(n1,n2) )
    begin
      with aSystem
      do DataPush( Max( DataPop, DataPop));
    end;


    procedure _min( aSystem: TForth);
    // Min ( n1 n2 -- min(n1,n2) )
    begin
      with aSystem
      do DataPush( Min( DataPop, DataPop));
    end;


    procedure _doubleQuote( aSystem: TForth);
    // " a string" ( -- <string> ) Immediate
    // Scan next word quote delimited from the input stream
    // and push it onto the stack as a string.
    var
      S     : string;
      aData : TData;
    begin
      with aSystem
      do begin
        S := GetToken( '"');
        if FCompiling
        then begin
          aData := TData.Create( aSystem, NoName);
          aData.Value := S;
          Compile( aData);
        end
        else DataPush( S);
      end;
    end;


    procedure _singleQuote( aSystem: TForth);
    // ' a string' ( -- <string> ) Immediate
    // Scan next word quote delimited from the input stream
    // and push it onto the stack as a string.
    // Same as " except the closure is ' here.
    var
      S     : string;
      aData : TData;
    begin
      with aSystem
      do begin
        S := GetToken( '''');

        if FCompiling
        then begin
          aData := TData.Create( aSystem, NoName);
          aData.Value := S;
          Compile( aData);
        end
        else DataPush( S);
      end;
    end;


    procedure _colon( aSystem: TForth);
    // Colon ( -- n )
    // start a colon definition
    var
      aToken : string;
      aColon : TColon;
    begin
      with aSystem
      do begin
        MustNotBeCompilingFor  ( ':');
        FCompiling := True;
        aToken     := GetToken( ' ');
        aColon     := TColon( AddWord( aToken, TColon));
        FCurrentColon := aColon;
        FLastDefined  := aColon; // Delay Reveal and EndCreate until Semi or SemiNoReturn
        ControlPush( CColonSemi);
      end;
    end;


    procedure _recurse( aSystem: TForth);
    // Recurse ( ? -- ? ) Immediate
    // Call into self at run time;
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'recurse');
        Compile( FLastDefined);
      end;
    end;


    procedure _semi( aSystem: TForth);
    // ; ( x -- ) Immediate
    // Terminate a colon definition
    begin
      with aSystem
      do begin
        if FCompiling
        then begin
          CheckControlStructure( CColonSemi);
          Compile( MustFindWord( ';'));

          with FCurrentColon
          do begin
            Reveal;
            EndCreate;
          end;

          FCurrentColon := nil;
          FCompiling    := False;
        end
        else FIP := -1;
      end;
    end;


    procedure _token( aSystem: TForth);
    // ( bSep -- <string> )
    // Read bSep separated token from input stream
    var
      S : string;
    begin
      with aSystem
      do begin
        S := DataPop;

        if Length( S) > 0
        then DataPush( GetToken( S[ 1], True)) // Uses extended token scan i.e. < ....> stuff
        else DataPush( '');
      end;
    end;


    procedure _untoken( aSystem: TForth);
    // ( <string> bSep -- )
    // Write bSep separated token to input stream
    var
      S : Char;
      T : string;
    begin
      with aSystem
      do begin
        S := string( DataPop)[ 1];
        T := DataPop;
        PutToken( S, T);
      end;
    end;


    procedure _flush_input( aSystem: TForth);
    // ( -- )
    // Flush rest of input stream
    begin
      with aSystem
      do FInputBuffer := '';
    end;


    procedure _forwardMark( aSystem: TForth);
    // >Mark ( -- n )
    // set up a mark to be resolved by _ForwardResolve
    // Forward jump
    begin
      with aSystem
      do begin
        DataPush( Here);
        Compile( TData.Create( aSystem, NoName)); // To be resolved by >resolve
      end;
    end;


    procedure _forwardResolve( aSystem: TForth);
    // >Resolve ( n -- )
    // Resolve a _ForwardMark
    // Forward jump
    begin
      with aSystem
      do ( FCode.GetWord( DataPop) as TData).Value := Here   // Resolve >mark label
    end;


    procedure _backwardMark( aSystem: TForth);
    // <Mark ( -- n )
    // set up a mark to be resolved by _BackwardResolve
    // Backward jump
    begin
      with aSystem
      do DataPush( Here);
    end;


    procedure _backwardResolve( aSystem: TForth);
    // <Resolve ( n -- )
    // Resolve a _BackwardMark
    // Backward jump
    var
      aData : TData;
    begin
      with aSystem
      do begin
        aData := TData.Create( aSystem, NoName);
        aData.Value := DataPop;
        Compile( aData);
      end;
    end;


    procedure _comma( aSystem: TForth);
    // , ( n -- )
    // Add n to the end of the FCode space
    var
      aData : TData;
    begin
      with aSystem
      do begin
        aData := TData.Create( aSystem, NoName);
        aData.Value := DataPop;
        Compile( aData);
      end;
    end;


    procedure _allot( aSystem: TForth);
    // Allot ( n -- )
    // Claim n cells of FCode space
    var
      i     : Integer;
      aData : TData;
    begin
      with aSystem
      do begin
        for i := 0 to DataPop - 1
        do begin
          aData := TData.Create( aSystem, NoName);
          aData.Value := 0;
          Compile( aData);
        end;
      end;
    end;


    procedure _here( aSystem: TForth);
    // Here ( -- n )
    // Returns the index of the first free FCode space location
    begin
      with aSystem
      do DataPush( Here);
    end;


    procedure _create( aSystem: TForth);
    // Create XXXX ( -- )
    // Define XXXX as a new word
    // When XXXX is later executed it pushes it's CFA onto the stack.
    var
      aWord : TCreate;
    begin
      with aSystem
      do begin
        aWord := TCreate( AddWord( GetToken( ' '), TCreate));
        aWord.CFA := Here;
        Compile( TData.Create( aSystem, NoName)); // Pointer to be resolved by 'Does>'
        FLastDefined := aWord;

        with FLastDefined
        do begin
          Reveal;
          EndCreate;
        end;
      end;
    end;


    procedure _pdoes( aSystem: TForth);
    // (Does>) ( -- <pfa> )
    // Changes the last word defined with CREATE to make it
    // execute the FCode that follows inline, it then exits
    // from the current thread.
    // note: Probably the same as (;FCode) - hmm prolly not ...
    var
      aWord : TCreate;
      aCfa  : Integer;
      aData : TData;
    begin
      with aSystem
      do begin
        if FLastDefined is TCreate                      // Pick up word defined by 'Create'
        then begin
          aWord := TCreate( FLastDefined);

          with aWord
          do begin
            aCfa  := Cfa;                               // Get its CFA reference
            aData := TData( FCode.GetWord( aCfa));      // And make the data elt at CFA
            aData.Value := FIP;                         // point to the 'does' part
          end;
        end
        else RaiseError( 'LastDefined is not of class ''Create'', can''t use ''(does>)'' to modify it''s semantics.');

        FIP := ReturnPop;                               // Perform Semi semantics
      end;
    end;


    procedure _does( aSystem: TForth);
    // Does> ( -- ) Immediate
    begin
      with aSystem
      do begin
        MustBeCompilingFor    (  'does>');
        Compile( MustFindToken( '(does>)', TInternal));
      end;
    end;


    procedure _wreg( aSystem: TForth);
    // w ( -- n ) // Push the current Wreg value
    begin
      with aSystem
      do DataPush( FWreg);
    end;


    procedure _ip( aSystem: TForth);
    // FIP ( -- n ) // Push the current FIP value
    begin
      with aSystem
      do DataPush( FIP);
    end;


    procedure _pfaFetch( aSystem: TForth);
    // pfa@ ( n<pfa> -- n<data> ) // Get a TWord object from FCode[ n<pfa>]
    begin
      with aSystem
      do DataPush( Cardinal( FCode.GetWord( DataPop)));
    end;


    procedure _branch( aSystem: TForth);
    // ( -- )
    // Executes a branch
    begin
      with aSystem
      do FIP := ( FCode.GetWord( FIP) as TData).Value;
    end;


    procedure _fbranch( aSystem: TForth);
    // ( -- )
    // Executes a conditional branch
    begin
      with aSystem
      do begin
        if Boolean( DataPop)
        then FIP := FIP + 1
        else FIP := ( FCode.GetWord( FIP) as TData).Value;
      end;
    end;


    procedure _immediate( aSystem: TForth);
    // ( -- )
    begin
      with aSystem
      do Immediate;
    end;


    procedure _export( aSystem: TForth);
    // ( -- )
    begin
      with aSystem
      do Exported;
    end;


    procedure _hide( aSystem: TForth);
    // ( -- )
    begin
      with aSystem
      do Hide;
    end;


    procedure _reveal( aSystem: TForth);
    // ( -- )
    begin
      with aSystem
      do Reveal;
    end;


    // ////////////////////////////////////////////////////////////////////////

    procedure __begin( aSystem: TForth);
    // ( -- )
    begin
      // Just a marker for debugging
    end;


    procedure __until( aSystem: TForth);
    // ( n -- ), alias for ?Branch
    begin
      _fbranch( aSystem);
    end;


    procedure __while( aSystem: TForth);
    // ( n -- ), alias for FBranch
    begin
      _fbranch( aSystem);
    end;


    procedure __again( aSystem: TForth);
    // ( -- ) alias for Branch
    begin
      _branch( aSystem);
    end;


    procedure __repeat( aSystem: TForth);
    // ( -- ) alias for Branch
    begin
      _branch( aSystem);
    end;


    procedure __do( aSystem: TForth);
    // ( ns ne -- ) runtime handler for Do
    begin
      with aSystem
      do begin
        LoopPush( DataPop);    // start value
        LoopPush( DataPop);    // end   value
        FIP := FIP + 1;        // Skip forward branch
      end;
    end;


    procedure __qdo( aSystem: TForth);
    // ( ns ne -- ) runtime handler for ?Do
    var
      S, E : Integer;
    begin
      with aSystem
      do begin
        S := DataPop;
        E := DataPop;
        if E = S
        then _branch( aSystem)
        else begin
          LoopPush( S);    // start value
          LoopPush( E);    // end   value
          FIP := FIP + 1;  // Skip forward branch
        end;
      end;
    end;


    procedure __loop( aSystem: TForth);
    // ( -- ) runtime handler for Loop
    var
      S, E : Integer;
    begin
      with aSystem
      do begin
        E := LoopPop;
        S := LoopPop + 1;

        if S >= E
        then FIP := FIP + 1       // Skip branch
        else begin
          LoopPush( S);
          LoopPush( E);
          _branch( aSystem);      // Backward branch
        end;
      end;
    end;


    procedure __ploop( aSystem: TForth);
    // ( n -- ) runtime handler for +Loop
    var
      S, E : Integer;
    begin
      with aSystem
      do begin
        E := LoopPop;
        S := LoopPop + DataPop;

        if S < E
        then begin
          LoopPush( S);
          LoopPush( E);
          _branch( aSystem);      // Backward branch
        end
        else FIP := FIP + 1;      // Skip branch
      end;
    end;


    procedure __unloop( aSystem: TForth);
    // ( -- ) runtime handler for UnLoop
    begin
      with aSystem
      do begin
        LoopPop;
        LoopPop;
      end;
    end;


    procedure __leave( aSystem: TForth);
    // ( -- ) runtime handler for Leave, non ANSI
    var
      E : Integer;
    begin
      with aSystem
      do begin
        E := LoopPop;           // Make loop terminate
             LoopPop;
        LoopPush( E - 1);
        LoopPush( E);
      end;
    end;


    procedure __i( aSystem: TForth);
    // ( -- n ) runtime handler for I
    begin
      with aSystem
      do DataPush( FLoopStack.Index( 1));
    end;


    procedure __j( aSystem: TForth);
    // ( -- n ) runtime handler for J
    begin
      with aSystem
      do DataPush( FLoopStack.Index( 3));
    end;


    procedure __k( aSystem: TForth);
    // ( -- n ) runtime handler for K
    begin
      with aSystem
      do DataPush( FLoopStack.Index( 5));
    end;


    procedure __tor( aSystem: TForth);
    // ( n -- ) runtime handler for >r
    begin
      with aSystem
      do ReturnPush( DataPop);
    end;


    procedure __rto( aSystem: TForth);
    // ( -- n ) runtime handler for r>
    begin
      with aSystem
      do DataPush( ReturnPop);
    end;


    procedure __rfetch( aSystem: TForth);
    // ( -- n ) runtime handler for r@
    begin
      with aSystem
      do DataPush( ReturnFetch);
    end;


    procedure __if( aSystem: TForth);
    // ( n -- ) runtime handler for If
    begin
      with aSystem
      do _fbranch( aSystem);
    end;


    procedure __else( aSystem: TForth);
    // ( -- ) runtime handler for Else
    begin
      with aSystem
      do _branch( aSystem);
    end;


    procedure __then( aSystem: TForth);
    // ( -- ) runtime handler for Then
    begin
      // Just a marker for debugging
    end;


    procedure __endif( aSystem: TForth);
    // ( -- ) runtime handler for EndIf
    begin
      // Just a marker for debugging
    end;


    procedure __case( aSystem: TForth);
    // ( -- ) runtime handler for Case
    begin
      // Just a marker for debugging
    end;


    procedure __default( aSystem: TForth);
    // ( -- ) runtime handler for Default
    begin
      // Just a marker for debugging
    end;


    procedure __endcase( aSystem: TForth);
    // ( -- ) runtime handler for EndCase
    begin
      // Just a marker for debugging
    end;


    procedure __of( aSystem: TForth);
    // ( n1 n2 -- n1 | ) runtime handler for of
    begin
      with aSystem
      do begin
        if DataPop = DataFetch
        then begin
          DataPop;
          FIP := FIP + 1;
        end
        else _branch( aSystem);
      end;
    end;


    procedure __endof( aSystem: TForth);
    // ( -- ) runtime handler for EndOf
    begin
      with aSystem
      do _branch( aSystem); // Jump to after EndCase
    end;


    procedure __in( aSystem: TForth);
    // ( n1 n2 -- n1 | ) runtime handler for in
    type
      TSmallSet = set of 0 .. 31;
      TSetConvertor = record
        case Boolean of
          False : ( int     : Integer   );
          True  : ( smallset: TSmallSet );
      end;
    var
      aSet : TSetConvertor;
    begin
      with aSystem
      do begin
        aSet.int := DataPop;
        if Integer( DataFetch) in aSet.smallset
        then begin
          DataPop;
          FIP := FIP + 1;
        end
        else _branch( aSystem);
      end;
    end;


    procedure __endin( aSystem: TForth);
    // ( -- ) runtime handler for EndIn
    begin
      with aSystem
      do _branch( aSystem); // Jump to after EndCase
    end;


    procedure __exit( aSystem: TForth);
    // ( -- ) runtime handler for Exit
    begin
      with aSystem
      do FIP := ReturnPop;
    end;


    procedure _begin( aSystem: TForth);
    // Begin ( -- n n )
    // Begin .. Again
    // Begin .. Until
    // Begin .. While .. Repeat
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Begin');
        Compile( MustFindWord( '_begin'));
        _backwardMark( aSystem);
        ControlPush( CBeginUntilWhile);
      end;
    end;


    procedure _until( aSystem: TForth);
    // Until ( n n -- )
    // End a Begin .. until loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Until');
        CheckControlStructure( CBeginUntilWhile);
        Compile( MustFindWord( '_until'));
        _backwardResolve( aSystem);
      end;
    end;


    procedure _again( aSystem: TForth);
    // Again ( n -- )
    // End a begin .. again loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Again');
        CheckControlStructure( CBeginUntilWhile);
        Compile( MustFindWord( '_again'));
        _backwardResolve( aSystem);
      end;
    end;


    procedure _while( aSystem: TForth);
    // While ( n -- )
    // Begin .. While .. Repeat
    var
      aData  : TData;
      aLabel : Variant;
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'While');
        CheckControlStructure( CBeginUntilWhile);
        Compile( MustFindWord( '_while'));

        aLabel := DataPop;
        DataPush( Here);

        DataPush( aLabel);
        aData := TData.Create( aSystem, NoName);
        Compile( aData);

        ControlPush( CWhileRepeat);
      end;
    end;


    procedure _repeat( aSystem: TForth);
    // Repeat ( n n -- )
    // Begin ... While ... Repeat
    var
      aData  : TData;
      aLabel : Variant;
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Repeat');
        CheckControlStructure( CWhileRepeat);
        Compile( MustFindWord( '_repeat'));

        aData  := TData.Create( aSystem, NoName);
        aData.Value := DataPop;
        Compile( aData);

        aLabel := Here;

        ( FCode.GetWord( DataPop) as TData).Value := aLabel; // Resolve While
      end;
    end;


    procedure _do( aSystem: TForth);
    // Do ( -- n )
    // Start a do .. loop or a do .. +loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'do');
        Compile( MustFindWord( '_do'));
        _forwardMark ( aSystem);
        _backwardMark( aSystem);
        ControlPush( CDoLoop);
      end;
    end;


    procedure _qdo( aSystem: TForth);
    // ?Do ( -- n )
    // Start a ?do .. loop or a ?do .. +loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( '?do');
        Compile( MustFindWord( '_?do'));
        _forwardMark ( aSystem);
        _backwardMark( aSystem);
        ControlPush( CDoLoop);
      end;
    end;


    procedure _loop( aSystem: TForth);
    // Loop ( n -- )
    // End a do .. loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'loop');
        CheckControlStructure( CDoLoop);
        Compile( MustFindWord( '_loop'));
        _backwardResolve( aSystem);
        _forwardResolve ( aSystem);
      end;
    end;


    procedure _ploop( aSystem: TForth);
    // +Loop ( n -- )
    // End a do .. +loop
    begin
      with aSystem
      do begin
        MustBeCompilingFor( '+loop');
        CheckControlStructure( CDoLoop);
        Compile( MustFindWord( '_ploop'));
        _backwardResolve( aSystem);
        _forwardResolve ( aSystem);
      end;
    end;


    procedure _unloop( aSystem: TForth);
    // UnLoop ( -- )
    // Drop loop stack entries
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'unloop');
        Compile( MustFindWord( '_unloop'));
      end;
    end;


    procedure _leave( aSystem: TForth);
    // ( -- )
    // Leave the enclosing (?)Do .. (+)Loop when the next
    // Loop or +Loop is reached (non ANSI)
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Leave');
        Compile( MustFindWord( '_Leave'));
      end;
    end;


    procedure _i( aSystem: TForth);
    // ( -- n )
    // Get innermost (?)do .. (+)loop index
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'I');
        Compile( MustFindWord( '_I'));
      end;
    end;


    procedure _j( aSystem: TForth);
    // ( -- n )
    // Get 2nd (?)do .. (+)loop index
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'J');
        Compile( MustFindWord( '_J'));
      end;
    end;


    procedure _k( aSystem: TForth);
    // ( -- n )
    // Get 3rd (?)do .. (+)loop index
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'K');
        Compile( MustFindWord( '_K'));
      end;
    end;


    procedure _tor( aSystem: TForth);
    // ( n -- )
    // Transfer n to return stack, or to loop stack for target
    begin
      with aSystem
      do begin
        MustBeCompilingFor( '>R');
        Compile( MustFindWord( '_>R'));
      end;
    end;


    procedure _rto( aSystem: TForth);
    // ( -- n )
    // Get n from return stack (or from loop stack for target)
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'R>');
        Compile( MustFindWord( '_R>'));
      end;
    end;


    procedure _rfetch( aSystem: TForth);
    // ( -- n )
    // Get n from return stack (or from loop stack for target)
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'R@');
        Compile( MustFindWord( '_R@'));
      end;
    end;


    procedure _if( aSystem: TForth);
    // If ( -- n )
    // Start an if [else] then conditional
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'If');
        Compile( MustFindWord( '_if'));
        _forwardMark( aSystem);
        ControlPush( CIfElseThen);
      end;
    end;


    procedure _else( aSystem: TForth);
    // Else ( n -- n )
    // Start an else part in a if else then conditional
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Else');
        CheckControlStructure( CIfElseThen);
        Compile( MustFindWord( '_else'));
        _forwardMark( aSystem);
        DataSwap;
        _forwardResolve( aSystem);
        ControlPush( CIfElseThen);
      end;
    end;


    procedure _then( aSystem: TForth);
    // Then ( n -- )
    // End an if [else] then conditional
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Then');
        CheckControlStructure( CIfElseThen);
        Compile( MustFindWord( '_then'));
        _forwardResolve( aSystem);
      end;
    end;


    procedure _endif( aSystem: TForth);
    // Then ( n -- )
    // End an if [else] endIf conditional
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'endif');
        CheckControlStructure( CIfElseThen);
        Compile( MustFindWord( '_endif'));
        _forwardResolve( aSystem);
      end;
    end;


    procedure _case( aSystem: TForth);
    // Case ( -- 4 0 )
    // Start a CASE statement
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Case');
        ControlPush( CCaseEndCase);
        Compile( MustFindWord( '_case'));
        DataPush( 0);  // End of resolve list marker - for ENDCASE
      end;
    end;


    procedure _default( aSystem: TForth);
    // Default ( -- )
    // Dummy word in Case ... EndCase
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Default');
        Compile( MustFindWord( '_Default'));
      end;
    end;


    procedure _endcase( aSystem: TForth);
    // EndCase ( n0 n1 .. nn-1 0 4 -- )
    // End a CASE statement
    var
      aLabel : Variant;
      Loc    : Integer;
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'EndCase');
        Compile( MustFindWord( '_endcase'));
        aLabel := Here;

        Loc := DataPop;
        while Loc <> 0
        do begin
          ( FCode.GetWord( Loc) as TData).Value := aLabel; // Resolve EndOf labels.
          Loc := DataPop;
        end;
        CheckControlStructure( CCaseEndCase);
      end;
    end;


    procedure _of( aSystem: TForth);
    // of ( -- n 5 )
    // Start an OF section with a CASE .. ENCASE construct
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Of');
        Compile( MustFindWord( '_of'));
        _forwardMark( aSystem);
        ControlPush( COfEndOf);
      end;
    end;


    procedure _endof( aSystem: TForth);
    // EndOf ( n1 5 -- n1 n2 )
    // End an OF section
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'EndOf');
        CheckControlStructure( COfEndOf);
        Compile( MustFindWord( '_endof'));
        _forwardMark( aSystem);
        DataSwap;
        _forwardResolve( aSystem);
      end;
    end;


    procedure _in( aSystem: TForth);
    // in ( -- n 5 )
    // Start an IN section within a CASE .. ENDCASE construct
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'in');
        Compile( MustFindWord( '_in'));
        _forwardMark( aSystem);
        ControlPush( CInEndIn);
      end;
    end;


    procedure _endin( aSystem: TForth);
    // EndIn ( n1 5 -- n1 n2 )
    // End an IN section
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'endIn');
        CheckControlStructure( CInEndIn);
        Compile( MustFindWord( '_endin'));
        _forwardMark( aSystem);
        DataSwap;
        _forwardResolve( aSystem);
      end;
    end;


    procedure _exit( aSystem: TForth);
    // ( -- )
    // Immediately leave the current colon word returning to the caller
    begin
      with aSystem
      do begin
        MustBeCompilingFor( 'Exit');
        Compile( MustFindWord( '_Exit'));
      end;
    end;



    procedure _defer( aSystem: TForth);
    // Defer <XXXX> ( -- )
    // Make XXXX a word with out semantics, to be reolved by Is or Implements
    begin
      with aSystem
      do begin
        MustNotBeCompilingFor( 'Defer');
        AddDefer( GetToken( ' '));
      end;
    end;


    procedure _tick( aSystem: TForth);
    //  `  <XXXX> ( -- exec-token )
    // [`] <XXXX> ( -- exec-token ) Immediate
    // Get next word from input stream, search it and leave it's execution
    // token on the stack. Error if not found
    var
      aWord : TWord;
    begin
      with aSystem
      do begin
        aWord := MustFindWord( GetToken( ' '));
        DataPushWord( aWord);
      end;
    end;


    procedure _alias( aSystem: TForth);
    // Alias ( -- )
    // Define a new name for an existing word
    // Used as 'Alias XXXX YYYY' to make XXXX also known as YYYY
    // or  alias old_name new_name
    var
      aNewName  : string;
      anOldName : string;
    begin
      with aSystem
      do begin
        MustNotBeCompilingFor( 'Alias');
        anOldName := GetToken( ' ');
        aNewName  := GetToken( ' ');
        AddAlias( anOldName, aNewName);
      end;
    end;


    procedure _is( aSystem: TForth);
    // Is <XXXX>( exec-token -- ) Immediate
    // Used as ' YYYY Is XXXX, used to resolve a deferred word
    // to exec-token.
    // When XXXX is not a deferred word an error is generated
    var
      aName  : string;
      aDefer : TDefer;
    begin
      with aSystem
      do begin
        // MustNotBeCompilingFor( 'Is');
        aName  := GetToken( ' ');
        aDefer := TDefer( MustFindToken( aName, TDefer));

        try
          aDefer.Resolve( DataPopWord as TWord);
        except
          on E: Exception
          do
            aSystem.
              RaiseErrorFmt(
                'Resolve error for ''%s'' with message: %s',
                [
                  aName,
                  E.Message
                ]
              );
        end;
      end;
    end;


    procedure _lastDefined( aSystem: TForth);
    // lastDefined ( -- exec-token )
    // Returns the executionj token of the word that was last defined
    begin
      with aSystem
      do begin
        MustNotBeCompilingFor( 'LastDefined');
        DataPushWord( FLastDefined);
      end;
    end;


    procedure _implements( aSystem: TForth);
    // Implements XXXX ( -- )
    // Binds the last defined word to the deferred word XXXX
    // When XXXX is not a deferred word an error is generated.
    begin
      aSystem.MustNotBeCompilingFor( 'Implements');
      _LastDefined( aSystem);
      _Is( aSystem);
    end;


    procedure _execute( aSystem: TForth);
    // Interpret ( exec-token -- ?? )
    // Executes (interprets) exec-token.
    begin
      with aSystem
      do ( DataPopWord as TWord).Interpret;
    end;


    procedure _evaluate( aSystem: TForth);
    // Interpret ( $ -- ?? )
    // Interprets a string
    begin
      with aSystem
      do AcceptText( DataPop, FInvoker);
    end;


    procedure _explain( aSystem: TForth);
    // Explain ( exec-token -- )
    // Gives word info exec-token
    begin
      with aSystem
      do Explain( DataPopWord as TWord);
    end;


    procedure _see( aSystem: TForth);
    // See <aword>
    // Gives word info on the next inline word
    begin
      with aSystem
      do See;
    end;


    procedure _info( aSystem: TForth);
    // show_word_info ( exec-token -- )
    // Gives help info exec-token
    begin
      with aSystem
      do ShowWordInfo;
    end;


{ ========
  TStack = class
  private
    FSystem      : TForth;
    FName        : string;
    FData        : array of Variant;
    FOnUnderflow : TOnStackUnderflow;
  public
    property    Depth       : Integer           read GetDepth;
    property    OnUnderflow : TOnStackUnderflow read FOnUnderflow write FOnUnderflow;
  private
}

    function    TStack.GetDepth: Integer;
    begin
      Result := Length( FData);
    end;


//  public

    constructor TStack.Create( aSystem: TForth; const aName: string);
    begin
      inherited Create;
      FSystem := aSystem;
      FName   := aName;
    end;


    destructor  TStack.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TStack.Push( aValue: Variant);
    begin
      if aValue <> Null
      then begin
        SetLength( FData, Length( FData) + 1);
        FData[ Length( FData) - 1] := aValue;
      end;
    end;


    procedure   TStack.PushWord( const aValue: TWord);
    begin
      Push( WordToVariant( aValue));
    end;


    function    TStack.Pop: Variant;
    begin
      if Depth > 0
      then begin
        Result := FData[ Length( FData) - 1];
        SetLength( FData, Length( FData) - 1);
      end
      else begin
        if Assigned( FOnUnderflow)
        then FOnUnderflow( Self, FName);

        Result := Null;
      end;
    end;


    function    TStack.PopWord: TWord;
    var
      V : Variant;
    begin
      Result := nil;
      V      := Pop;

      if VarIsType( V, varByRef)
      then begin
        TVarData( V).VType := varInteger;
        Result := TWord( Integer( V));
      end
      else FSystem.RaiseError( 'Object expected');
    end;


    procedure   TStack.Swap;
    var
      A, B: Variant;
    begin
      A := Pop;
      B := Pop;
      Push( A);
      Push( B);
    end;


    function    TStack.Index( anIndex: Integer): Variant;
    begin
      if ( anIndex >= 0) and ( anIndex < Depth)
      then Result := FData[ Depth - anIndex - 1]
    end;


    function    TStack.Fetch: Variant;
    begin
      Result := Index( 0);
    end;


    procedure   TStack.Clear;
    begin
      while Depth > 0
      do Pop;
    end;


    function    TStack.Dump( aBase: Integer): string;
    var
      i : Integer;
      S : string;
    begin
      Result := '[stack empty]';

      for i := Depth - 1 downto 0
      do begin
        S := DataToString( FData[ i], aBase);

        if i = Depth - 1
        then Result := S
        else Result := Result + ' ' + S;
      end;
    end;


{ ========
  TWord = class
  public
    class var
      WordCounter      : Integer;
      AnonymousCounter : Integer;
  private
    FSystem    : TForth;
    FName      : string;       // Name of the word (can be '')
    FComment   : string;       // Comment for word, probably stack image
    FCFA       : Integer;
    FImmediate : Boolean;      // True when executing in compilation mode
    FExported  : Boolean;      // True when the word is exportable
    FVisible   : Boolean;      // True when word can be found in dictionary
  public
    property    Name      : string  read FName;
    property    Comment   : string  read FComment   write FComment;
    property    CFA       : Integer read FCFA       write FCFA;
    property    Immediate : Boolean read FImmediate write FImmediate;
    property    Exported  : Boolean read FExported  write FExported;
    property    Visible   : Boolean read FVisible   write FVisible;
  public
}

    constructor TWord.Create( aSystem: TForth; const aName: string); // virtual;
    begin
      inherited Create;
      FSystem  := aSystem;
      FName    := aName;
      FComment := '';
      CFA      := FSystem.Here;
      Inc( WordCounter);

      if SameText( aName, NoName) // Keep track of anonymous words
      then begin
        FSystem.FAnonymous.Add( Self);
        Inc( AnonymousCounter);
      end;
    end;


    destructor  TWord.Destroy; // override;
    begin
      Dec( WordCounter);

      if SameText( Name, NoName) // Keep track of anonymous words
      then Dec( AnonymousCounter);

      inherited;
    end;


    procedure   TWord.EndCreate; // virtual;
    begin
      // Flags end of creation, not used here, it's for children
    end;


    procedure   TWord.Interpret; // virtual;
    begin
      FSystem.
        RaiseErrorFmt(
          'can''t interpret ''%s'' (type=%s), it has no interpretation semantics.',
          [ Name, TypeName]
        );
    end;


    procedure   TWord.Compile( const aStringList: TStringList); // virtual;
    begin
      FSystem.
        RaiseErrorFmt(
          'can''t compile ''%s'' (type=%s), it has no compilation semantics.',
          [ Name, TypeName]
        );
    end;


    procedure   TWord.Hide; // virtual;
    begin
      Visible := False;
    end;


    procedure   TWord.Reveal; // virtual;
    begin
      Visible := True;
    end;


    function    TWord.See: string; // virtual;
    begin
      Result := Format( '%s %s ( %s )', [ LowerCase( TypeName), Name, Comment]);
    end;


    function    TWord.ShowWordInfo: string; // virtual;
    begin
      Result := Format( '%s ( %s )', [ See, Comment]);
    end;


    procedure   TWord.HandleForget; // virtual;
    begin
      // Nothing special here ... some word classes will need it tho
    end;


    function    TWord.AppendFlags( const S: string): string; // virtual;
    begin
      Result := S;

      if Immediate
      then Result := Result + ' immediate';

      if not Visible
      then Result := Result + ' hide';

      if Immediate
      then Result := Result + ' exported';
    end;


//  public

    class function TWord.TypeName   : string;
    begin
      Result := Copy( ClassName, 2, Length( ClassName) - 1);
    end;


{ ========
  TWords = class( TList)
  private
    FIndex  : TStringList;
    FSystem : TSystem;
  public
    property    Word[ anIndex: Integer]: TWord read GetWord; default;
  private
}

    function    TWords.GetWord( anIndex: Integer): TWord;
    begin
      Result := TWord( Items[ anIndex]);
    end;


//  public

    constructor TWords.Create( aSystem: TForth);
    begin
      inherited Create;
      FSystem := aSystem;
      FIndex  := TStringList.Create;

      with FIndex
      do begin
        Sorted        := True;
        CaseSensitive := True;      // For faster compares
        Duplicates    := dupAccept;
      end;
    end;


    destructor  TWords.Destroy; // override;
    begin
      FreeAndNil( FIndex); // The index is owned and must be disposed off, the
                           //   objects in the index are not owned though
      inherited Destroy;   // This will call Clear whilst FIndex doesn't exist
                           //   anymore. Also see [ TWords.Clear ]
    end;


    procedure   TWords.Clear; //                                                 override;
    var
      R : TWord;
    begin
      if Assigned( FIndex) // Destroy calls clear, but kills FIndex first.
      then FIndex.Clear;   // Remocves all strings from the index, frees no
                           // objects. All the words are owned and should be
                           // disposed off. Also see [ TWords.Destroy ]
      while Count > 0
      do begin
        R := Word[ Count - 1];
        Delete( Count - 1);
        R.Free;
      end;

      inherited;
    end;


    function    TWords.AddWord( const aName: string; aWordClass: TWordClass; aCFA: Integer): TWord; // overload;
    begin
      if SameText( aName, NoName)
      then FSystem.RaiseError( 'Not allowwed to use AddWord for anonymous words');

      Result := AddWord( aWordClass.Create( FSystem, aName), aCFA);
    end;


    function    TWords.AddWord( aWord: TWord; aCFA: Integer): TWord; // overload;
    begin
      aWord.CFA := aCFA;

      if Assigned( aWord)
      then begin
        Add( aWord);
        FIndex.AddObject( UpperCase( aWord.Name), aWord);
      end
      else FSystem.RaiseError( 'Compiler error, can''t add <nil> to a word list.');

      Result := aWord;
    end;


    function    TWords.RemoveWord( const aName: string): TWord;
    var
      i : Integer;
    begin
      i := FIndex.IndexOf( UpperCase( aName));

      if i >= 0
      then begin
        Result := TWord( FIndex.Objects[ i]); // Remove word from index
        FIndex.Delete( i);
        Remove( Result);                      // Remove word from self
      end
      else Result := nil;
    end;


    function    TWords.FindToken( const aToken: string): TWord;
    // Make sure to find the latest definition
    var
      i : Integer;
      p : Integer;
      U : string;
      W : TWord;
    begin
      U := UpperCase( aToken);
      i := FIndex.IndexOf( U);

      if i >= 0
      then begin
        Result := TWord( FIndex.Objects[ i]);

        if not Result.Visible
        then begin
          Result := nil;

          for p := i - 1 downto 0
          do begin
            W := TWord( FIndex.Objects[ p]);

            if W.Visible
            then begin
              if SameText( W.Name, aToken)
              then begin
                Result := W;
                Break;
              end;
            end;
          end;
        end;
      end
      else Result := nil;
    end;


    function    TWords.CfaToWord( aCFA: Integer): TWord;
    var
      i     : Integer;
      aWord : TWord;
    begin
      Result := nil;

      for i := 0 to Count - 1
      do begin
        aWord := Word[ i];

        if ( aWord is TColon) and ( TColon( aWord).CFA = aCFA)
        then begin
          Result := aWord;
          Break;
        end;
      end;
    end;


    function    TWords.Forget( const aWord: TWord): Boolean;
    var
      WordPos   : Integer;
      FencePos  : Integer;
      WordIndex : Integer;
      IndexPos  : Integer;
      TheWord   : TWord;
      TheFence  : TWord;
    begin
      Result   := False;
      TheFence := FindToken( 'fence');
      FencePos := Count - 1;

      while FencePos >= 0
      do begin
        if Word[ FencePos] = TheFence
        then Break
        else Dec( FencePos);
      end;

      if FencePos > 0
      then begin
        WordPos := Count - 1;

        while WordPos >= 0
        do begin
          if Word[ WordPos] = aWord
          then Break
          else Dec( WordPos);
        end;

        if WordPos > FencePos
        then begin
          if WordPos < Count
          then begin
            for WordIndex := Count - 1 downto WordPos
            do begin
              TheWord  := Word[ WordIndex];
              IndexPos := FIndex.IndexOf( UpperCase( TheWord.Name));
              TheWord.HandleForget;

              if IndexPos >= 0
              then FIndex.Delete( IndexPos);

              Delete( WordIndex);
              TheWord.Free;
            end;
          end;

          Result := True;
        end
        else FSystem.RaiseError( 'Cannot forget fence or words defined before');
      end
      else FSystem.RaiseError( 'fence not defined, can not forget');
    end;


    function    TWords.Words: string;
    var
      i : Integer;
    begin
      Result := '';
      for i := 0 to Count - 1
      do begin
        if Result = ''
        then Result := Word[ i].Name
        else Result := Result + ' ' + Word[ i].Name;
      end;
    end;


    function    TWords.CreateExportList: TStringList;
    var
      i : Integer;
    begin
      Result := TStringList.Create;
      for i := 0 to Count - 1
      do begin
        if Word[ i].Exported
        then Result.Add( Word[ i].Name);
      end;
    end;


{ ========
  TCode = class( TWord)
  strict private
    FValue : TList;
  public
    property    Count: Integer read GetCount write SetCount;
  strict private
}

    function    TCode.GetCount: Integer;
    begin
      Result := FValue.Count;
    end;


    procedure   TCode.SetCount( aValue: Integer);
    begin
      FValue.Count := aValue;
    end;


//  public

    constructor TCode.Create( aSystem: TForth; const aName: string); // override;
    begin
      inherited;
      FValue := TList.Create;
    end;


    destructor  TCode.Destroy; // override;
    begin
      FreeAndNil( FValue);
      inherited;
    end;


    procedure   TCode.Clear;
    begin
      FValue.Clear;
    end;


    procedure   TCode.Interpret; // override;
    begin
      FSystem.DataPushWord( Self);
    end;


    function    TCode.See: string; // override;
    var
      i : Integer;
    begin
      Result := '{';

      for i := 0 to Count - 1
      do Result := Result + ' ' + GetWord( i).See;

      Result := Result + ' }';
    end;


    procedure   TCode.Append( aValue: TWord);
    begin
      FValue.Add( aValue);
    end;


    function    TCode.GetWord( anIndex: Integer): TWord;
    begin
      Result := FValue[ anIndex];
    end;


    procedure   TCode.AcceptToken( const aToken: string);
    var
      aWord   : TWord;
      aNumber : Variant;
    begin
      try
        aWord := nil;
        // Search for global definitions
        if not Assigned( aWord)
        then aWord := FSystem.FindToken( aToken);
        // When found let the word be interpreted or compiled depending
        // on the word's immediateness and the current compile state.
        if Assigned( aWord)
        then CompileWord( aWord)
        // Otherwise see if the token is a number
        else if FSystem.Number( aToken, aNumber)
        then CompileLiteral( aNumber)                // Compile literal number
        // None worked, it's an error then.
        else FSystem.RaiseErrorFmt( '%s ?', [ aToken]);
      except
        // All errors are trapped here.
        on E: Exception
        do FSystem.Respond( E.Message);
      end;
    end;


    procedure   TCode.CompileLiteral( const aValue: Variant);
    var
      aData : TData;
    begin
      aData := TData.Create( FSystem, NoName);
      aData.Value := aValue;
      CompileWord( aData);
    end;


    procedure   TCode.CompileWord( const aWord: TWord);
    begin
      Append( aWord);
    end;


    function    TCode.Here: Integer;
    begin
      Result := Count;
    end;


    procedure   TCode.SetHere( aValue: Integer);
    begin
      Count := aValue;
    end;


    procedure   TCode.Reverse;
    var
      Cnt : Integer;
      i   : Integer;
      T   : TWord;
    begin
      Cnt := Count div 2;

      for i := 0 to Cnt - 1
      do begin
        T                      := FValue[ i];
        FValue[ i]             := FValue[ Count - i - 1];
        FValue[ Count - i - 1] := T
      end;
    end;


    function    TCode.First: TWord;
    begin
      Result := nil;

      if Count > 0
      then Result := GetWord( 0)
      else FSystem.RaiseError( 'can''t get first on empty list');
    end;


    procedure   TCode.Rest;
    begin
      FValue.Delete( 0);
    end;


    procedure   TCode.Concat( const aValue: TCode);
    var
      i : Integer;
    begin
      for i := 0 to aValue.Count
      do CompileWord( aValue.GetWord( i));
    end;


{ ========
  TData = class( TWord)
  private
    FValue : Variant;
  public
    property    Value: Variant read FValue write FValue;
  public
}

    procedure   TData.Interpret; // override;
    begin
      FSystem.DataPush( FValue);
    end;


    function    TData.See: string; // override;
    begin
      Result := Format( '%s', [ DataToString( FValue, FSystem.FBase)]);
    end;


{ ========
  TConstant = class( TData)
  public
}

    procedure   TConstant.Interpret; // override;
    begin
      with FSystem
      do DataPush( Value);
    end;


    function    TConstant.See: string; // override;
    begin
      Result := Format( '%s constant %s', [ FValue, Name]);
      Result := AppendFlags( Result);
    end;


{ ========
  TVariable = class( TConstant)
  public
}

    procedure   TVariable.Interpret; // override;
    begin
      FSystem.DataPushWord( Self);
    end;


    function    TVariable.See: string; // override;
    begin
      Result := Format( 'variable %s', [ Name]);
      Result := AppendFlags( Result);
    end;


{ ========
  TAlias = class( TWord)
  private
    FAliased : TWord;
  public
    property    Aliased: TWord read FAliased write FAliased;
  public
}


    procedure   TAlias.Interpret; // override;
    begin
      if Assigned( Aliased)
      then Aliased.Interpret
      else FSystem.RaiseErrorFmt( 'Aliased ''%s'' Is %s, can''t interpret it.', [ Name, Nothing]);
    end;


    function    TAlias.See: string; // override;
    begin
      Result := Format( 'alias %s %s', [ Aliased.Name, Name]);
      Result := AppendFlags( Result);
    end;



{ ========
  TDefer = class( TAlias)
  end;
}

    procedure   TDefer.Resolve( aWord: TWord);
    begin
      Aliased := aWord;
    end;


    procedure   TDefer.Interpret; // override;
    begin
      if not Assigned( Aliased)
      then
        FSystem.
          RaiseErrorFmt(
            'Deferred ''%s'' is %s, can''t interpret it.', [ Name, Nothing]
          )
      else
        try
          Aliased.Interpret;
        except
          on E: Exception
          do
            FSystem.
              RaiseErrorFmt(
                '''%s'', deferred to ''%s'', generated error ''%s'' during interpretation',
                [
                  Name,
                  Aliased.Name,
                  E.Message
                ]
              );
        end;
    end;


    function    TDefer.See: string; // override;
    begin
      Result := Format( 'defer %s '' %s is %s', [ Name, Aliased.Name, Name]);
      Result := AppendFlags( Result);
    end;


{ ========
  TExecutable = class( TWord)
  public
}

{
  TInternalProc = procedure( aSystem: TSystem);

  TInternal = class( TExecutable)
  private
    FInternalProc: TInternalProc;
  public
    property    InternalProc: TInternalProc read FInternalProc write FInternalProc;
  public
}

    procedure   TInternal.Interpret; // override;
    begin
      InternalProc( FSystem);
    end;


{ ========
  TColon = class( TExecutable)
  public
}

    procedure   TColon.Interpret; // override;
    begin
      with FSystem
      do begin
        ReturnPush( FIP);
        FIP := CFA;

        while FIP >= 0
        do Exec( FCode.GetWord( FIP));

        FIP := ReturnPop;
      end;
    end;


    function    TColon.See: string; // override;
    var
      IP   : Integer;
      CW   : TWord;
      Val  : Variant;
      Skip : Integer;
    begin
      Result := ': ' + Name + ' ';

      if Comment <> ''
      then Result := Result + Format( '( %s ) ', [ Comment]);

      with FSystem
      do begin
        IP := CFA;
        CW := FCode.GetWord( IP);

        while Assigned( CW) and not SameText( CW.Name, ';')
        do begin
          Skip := 0;

          if CW = Self
          then Result := Result + ' recurse '
          else if CW is TData and not ( CW is TConstant)
          then begin
            Val := ( CW as TData).Value;
            if VarIsStr( Val)
            then Result := Result + Format( '" %s"', [ string( Val)])
            else begin
              if VarIsFloat( Val)
              then Result := Result + Format( '%f', [ Extended( Val)])
              else Result := Result + string( Val)
            end;
          end
          else if ( CW is TInternal) and ( Length( CW.Name) > 0) and ( CW.Name[ 1] = '_')
          then begin
            if SameText( CW.Name, '_Begin')
            then begin
              Result := Result + 'begin';   Skip := 0;
            end
            else if SameText( CW.Name,'_Until')
            then begin
              Result := Result + 'until';   Skip := 1;
            end
            else if SameText( CW.Name,'_While')
            then begin
              Result := Result + 'while';   Skip := 1;
            end
            else if SameText( CW.Name,'_Again')
            then begin
              Result := Result + 'again';   Skip := 1;
            end
            else if SameText( CW.Name,'_Repeat')
            then begin
              Result := Result + 'repeat';  Skip := 1;
            end
            else if SameText( CW.Name,'_Do')
            then begin
              Result := Result + 'do';      Skip := 1;
            end
            else if SameText( CW.Name,'_?Do')
            then begin
              Result := Result + '?do';     Skip := 1;
            end
            else if SameText( CW.Name,'_Loop')
            then begin
              Result := Result + 'loop';    Skip := 1;
            end
            else if SameText( CW.Name,'_UnLoop')
            then begin
              Result := Result + 'unloop';  Skip := 0;
            end
            else if SameText( CW.Name,'_PLoop')
            then begin
              Result := Result + '+loop';   Skip := 1;
            end
            else if SameText( CW.Name,'_Leave')
            then begin
              Result := Result + 'leave';   Skip := 0;
            end
            else if SameText( CW.Name,'_I')
            then begin
              Result := Result + 'i';       Skip := 0;
            end
            else if SameText( CW.Name,'_J')
            then begin
              Result := Result + 'j';       Skip := 0;
            end
            else if SameText( CW.Name,'_K')
            then begin
              Result := Result + 'k';       Skip := 0;
            end
            else if SameText( CW.Name,'_>R')
            then begin
              Result := Result + '>r';      Skip := 0;
            end
            else if SameText( CW.Name,'_R>')
            then begin
              Result := Result + 'r>';      Skip := 0;
            end
            else if SameText( CW.Name,'_If')
            then begin
              Result := Result + 'if';      Skip := 1;
            end
            else if SameText( CW.Name,'_Else')
            then begin
              Result := Result + 'else';    Skip := 1;
            end
            else if SameText( CW.Name,'_Then')
            then begin
              Result := Result + 'then';    Skip := 0;
            end
            else if SameText( CW.Name,'_EndIf')
            then begin
              Result := Result + 'endif';   Skip := 0;
            end
            else if SameText( CW.Name,'_Case')
            then begin
              Result := Result + 'case';    Skip := 0;
            end
            else if SameText( CW.Name,'_default')
            then begin
              Result := Result + 'default';  Skip := 0;
            end
            else if SameText( CW.Name,'_EndCase')
            then begin
              Result := Result + 'endCase';  Skip := 0;
            end
            else if SameText( CW.Name,'_Of')
            then begin
              Result := Result + 'of';       Skip := 1;
            end
            else if SameText( CW.Name,'_EndOf')
            then begin
              Result := Result + 'endOf';    Skip := 1;
            end
            else if SameText( CW.Name,'_In')
            then begin
              Result := Result + 'in';       Skip := 1;
            end
            else if SameText( CW.Name,'_EndIn')
            then begin
              Result := Result + 'endIn';    Skip := 1;
            end
            else if SameText( CW.Name,'_Exit')
            then begin
              Result := Result + 'exit';     Skip := 0;
            end
            else Result := Result + CW.Name;
          end
          else Result := Result + CW.Name;

          Result := Result + ' ';
          IP := IP + 1 + Skip;
          CW := FCode.GetWord( IP);
        end;

        Result := Result + ';';
      end;

      Result := AppendFlags( Result);
    end;


{ ========
  TCreate = class( TColon)
  public
}

    procedure   TCreate.Interpret; // override;
    begin
      with FSystem, FSystem.FCode
      do begin
        ReturnPush( FIP);
        FWReg := CFA + 1;                        // Setup Wreg to point to the data
        DataPush( FWReg);                        // Push data index - code after does> needs data@
        FIP := ( GetWord( CFA) as TData).Value;  // Set up IP
      end;
    end;


{ ========
  TSystem = class
  private
    FOnResponse        : TOnForthResponse;
    FOnSave            : TNotifyEvent;
    FOnExternalFunc    : TOnExternalFunc;
  private
    FDataStack         : TStack;
    FLoopStack         : TStack;
    FReturnStack       : TStack;
    FControlStack      : TStack;
    FCode              : TCode;                // Compiled forth code
    FWords             : TWords;               // The dictionary
    FAnonymous         : TObjectList;          // List of all anonymous words (so they can be freed)
    FInputBuffer       : string;
    FCurrToken         : string;
    FLastToken         : string;
    FOffset            : Integer;
    FBase              : Integer;
    FInvoker           : Integer;
    FCurrentFileName   : string;
  private
    FCompiling         : Boolean;
    FWreg              : Integer;              // Forth W register - needed for create .. does> - index into FCode,
                                               // a CFA
    FIP                : Integer;              // Forth instruction pointer
    FCW                : TWord;                // Forth code word
    FLastDefined       : TWord;                // The last defined word
    FCurrentColon      : TColon;               // The colon definition under construction
  public
    property    CurrentFileName : string           read FCurrentFileName;
    property    OnResponse      : TOnForthResponse read FOnResponse      write FOnResponse;
    property    OnSave          : TNotifyEvent     read FOnSave          write FOnSave;
    property    OnExternalFunc  : TOnExternalFunc  read FOnExternalFunc  write FOnExternalFunc;
  public
}


    procedure   TForth.Respond( const S: string);
    begin
      if Assigned( FOnResponse)
      then FOnResponse( Self, S)
      else raise EForth.CreateFmt( 'No forth response handler, message was: %s', [ S]);
    end;


    procedure   TForth.RespondFmt( const aFmt: string; const anArgs: array of const);
    begin
      Respond( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TForth.RaiseError( const aMsg: string);
    begin
      FInputBuffer := '';
      FCompiling   := False;
      FDataStack   .Clear;
      FLoopStack   .Clear;
      FReturnStack .Clear;
      FControlStack.Clear;
      raise EForth.CreateFmt( '%s', [ aMsg]);
    end;


    procedure   TForth.RaiseErrorFmt( const aFmt: string; const anArgs: array of const);
    begin
      RaiseError( Format( aFmt, anArgs));
    end;


    procedure   TForth.ShowWarning( const aMsg: string);
    begin
      RespondFmt( 'warning: %s', [ aMsg]);
    end;


    procedure   TForth.ShowWarningFmt( const aFmt: string; const anArgs: array of const);
    begin
      ShowWarning( Format( aFmt, anArgs));
    end;


    procedure   TForth.DoStackUnderflow( aSender: TObject; const aMsg: string);
    begin
      RaiseErrorFmt( '%s stack underflow', [ aMsg]);
    end;


    function    TForth.GetString( aTerminator: Char): string;

    // done: bug in string parser, " \" does not terminate scanning.

    //
    //   Uses escape characters in tokens
    //
    //   Process '\' characters in a C like manner :
    //
    //     \&nnn      - treat as a decimal       - 3 digits must be present
    //     \$xx       - treat as a hex number    - 2 digits must be present
    //     \%bbbbbbbb - treat as a binary number - 8 digits must be present
    //
    //     \0         - NUL
    //     \a         - BEL
    //     \B         - BS
    //     \F         - FF
    //     \N         - LF
    //     \R         - CR
    //     \T         - TAB
    //     \V         - VT
    //     \X         - same as $ - note C uses three digits, we use 2 here.
    //
    //   Specifically :
    //
    //     \\         - treat as a single \ character
    //     \"         - treat as a single " character
    //     \'         - treat as a single ' character

    var
      Done    : Boolean;
      Escaped : Boolean;
      i       : Integer;
      NumBase : Integer;
      N       : string;
    begin
      Result  := '';
      Done    := False;
      Escaped := False;
      i       := 1;
      NumBase := 0;      // Not scanning a number
      N       := '';

      if FInputBuffer = ''
      then begin
        Result := endoffile;
        Done   := True;
      end;

      while not Done and ( i <= Length( FInputBuffer))
      do begin
        if Escaped
        then begin
          case UpCase( FInputBuffer[ i]) of

            '0' .. '9', 'A' .. 'F' :

              begin
                if NumBase = 0
                then begin
                  case UpCase( FInputBuffer[ i]) of
                    '0' : Result := Result + #$00;  // NUL
                    'A' : Result := Result + #$07;  // BEL
                    'B' : Result := Result + #$08;  // BS
                    'F' : Result := Result + #$0c;  // FF
                    else  Result := Result + FInputBuffer[ i];
                  end;

                  Escaped := False;
                end
                else begin
                  N := N + FInputBuffer[ i];
                  if
                    ((NumBase = 16) and ( Length( N) = 2)) or
                    ((NumBase = 10) and ( Length( N) = 3)) or
                    ((NumBase =  2) and ( Length( N) = 8))
                  then begin
                    Result  := Result + Char( StrToUIntBase( N, NumBase) and $ff);
                    Escaped := False;
                    N       := '';
                    NumBase := 0;
                  end;
                end
              end;

            'N' :                                // line feed

              begin
                Result  := Result + #$0a;
                Escaped := False;
              end;

            'R' :                                // carriage return

              begin
                Result  := Result + #$0d;
                Escaped := False;
              end;

            'T' :                                // tab

              begin
                Result  := Result + #$09;
                Escaped := False;
              end;

            'V' :                                // vertical tab

              begin
                Result  := Result + #$0b;
                Escaped := False;
              end;

            'X', '$' :                           // Hex format

              begin
                Numbase := 16;
                N       := '';
              end;

            '&' :                                // decimal format

              begin
                Numbase := 10;
                N       := '';
              end;

            '%' :                                // binary format

              begin
                Numbase := 2;
                N       := '';
              end;

            else begin
              Result  := Result + FInputBuffer[ i];
              Escaped := False;
            end;
          end;
        end
        else begin
          case FInputBuffer[ i] of
            '\' : Escaped := True;
            '''', '"' :
              begin
                if FInputBuffer[ i] = aTerminator
                then Done := True
                else Result := Result + FInputBuffer[ i];
              end;
            else Result := Result + FInputBuffer[ i];
          end;
        end;
        Inc( i);
        Inc( FOffset);
      end;
      if i > 1
      then FInputBuffer := Copy( FInputBuffer, i, Length( FInputBuffer));
    end;


    function    TForth.GetToken( aSeparator: Char; Extended: Boolean = False): string;
    //
    // if extended is True then :
    //
    //   if aSeparator = ' ' and the token to scan starts with '<'
    //   then use [ '>'] as the separator and treat everything between '<' and
    //   '>' as a single token - even when spaces are contained within this
    //   string.
    //
    //   escape characters in tokens
    //
    //   if aSeparator = '"' (the double quote character)
    //   then scan for a string in the following way :
    //
    //   Process '\' characters in a C like manner :
    //
    //     \\         - treat as a single '\' character
    //     \"         - treat as a single '"' character
    //     \nnn       - treat as a decimal       - 3 digits must be present
    //     \$xx       - treat as a hex number    - 2 digits must be present
    //     \%bbbbbbbb - treat as a binary number - 8 difits must be present
    //
    //
    const
      WhiteSpace : set of AnsiChar = [ #$00 .. #$20];
    var
      i       : Integer;
      Done    : Boolean;
      StopSet : set of AnsiChar;

      procedure LTrimBuffer;
      var
        i : Integer;
      begin
        if aSeparator = ' '
        then begin
          i := 1;

          while ( i <= Length( FInputBuffer)) and CharInSet( FInputBuffer[ i], StopSet)
          do begin
            Inc( FOffset);
            Inc( i);
          end;

          if i > 1
          then FInputBuffer := Copy( FInputBuffer, i, Length( FInputBuffer));
        end;
      end;

    begin
      FCurrToken := endoffile;
      if aSeparator = '"'

      then FCurrToken := GetString( '"')
      else if aSeparator = ''''
      then FCurrToken := GetString( '''')
      else begin
        Done := False;

        if aSeparator = ' '
        then StopSet := WhiteSpace
        else StopSet := [ aSeparator];

        LTrimBuffer;

        while not Done
        do begin
          if FInputBuffer = ''
          then begin
            FCurrToken := endoffile;
            Done       := True;
          end
          else begin
            i := 1;

            while not Done and ( i <= Length( FInputBuffer))
            do begin
              if
                Extended                  and
                ( i = 1)                  and
                ( aSeparator       = ' ') and
                ( FInputBuffer[ i] = '<')
              then begin
                StopSet       := [ '>'];
                FInputBuffer  := Copy( FInputBuffer, 2, Length( FInputBuffer));
                i             := 1;
                Extended      := False; // Don't do this twice !!
              end
              else Inc( i);

              Inc( FOffset);
              Done :=
                (( i <= Length( FInputBuffer)) and CharInSet( FInputBuffer[ i], StopSet)) or
                ( aSeparator = ' ') and ( i > Length( FInputBuffer))
                ;
            end;

            if i > 1
            then begin
              FCurrToken   := Copy( FInputBuffer, 1, i - 1);
              FInputBuffer := Copy( FInputBuffer, i + 1, Length( FInputBuffer));
            end
            else FInputBuffer := '';
          end;
        end;
      end;

      Result     := FCurrToken;
      FCurrToken := '';
      FLastToken := Result;
    end;


    procedure   TForth.PutToken( aSeparator: Char; const aToken: string);
    begin
      FInputBuffer := aToken + aSeparator + FInputBuffer;
    end;


    function    TForth.Number( const aToken: string; out aNumber: Variant): Boolean;
    const
      DigitValues : array[ AnsiChar] of Byte = (

      {          0    1    2    3     4    5    6    7      8    9    a    b     c    d    e    f}

      { 00 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255, // Ctrl
      { 10 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { 20 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255, // Specials
      { 30 }     0,   1,   2,   3,    4,   5,   6,   7,     8,   9, 255, 255,  255, 255, 255, 255, // Nums
      { 40 }   255,  10,  11,  12,   13,  14,  15,  16,    17,  18,  19,  20,   21,  22,  23,  24, // UpCase
      { 50 }    25,  26,  27,  28,   29,  30,  31,  32,    33,  34,  35, 255,  255, 255, 255, 255,
      { 60 }   255,  36,  37,  38,   39,  40,  41,  42,    43,  44,  45,  46,   47,  48,  49,  50, // LoCase
      { 70 }    51,  52,  53,  54,   55,  56,  57,  58,    59,  60,  61, 255,  255, 255, 255, 255,
      { 80 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255, // Extended ascii
      { 90 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { a0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { b0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { c0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { d0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { e0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255,
      { f0 }   255, 255, 255, 255,  255, 255, 255, 255,   255, 255, 255, 255,  255, 255, 255, 255

      );

    var
      i        : Integer;                  // Loop index
      d        : Integer;                  // Current digit
      n        : Int64;                    // Number accumulator
      c        : AnsiChar;                 // aToken[ i] when within the for loop
      Neg      : Boolean;                  // True -> leading minus seen
      Asc      : Boolean;                  // True -> Ascii modifier seen
      Base     : Integer;                  // The current number base in which the string is to be interpreted
      DotCount : Integer;                  // Number of DOTs in number
      DotPos   : Integer;                  // Position of dot
    begin
      Result   := Length( aToken) > 0;     // Numbers must have length > 0
      n        := 0;                       // Initially the number is zero
      Neg      := False;                   //   and not negative
      Asc      := False;                   //   and not anAsc
      Base     := FBase;                   // The default number base is the current system number base
      DotCount := 0;                       // We saw no dots yet
      DotPos   := 0;                       // So it's at the 0th position currently

      for i := 1 to Length( aToken)
      do begin
        c := AnsiChar( aToken[ i]);        // Setup c as part of the loop invariant

        if Asc                             // In Asc mode, special handling needed
        then begin
          {$Q-R-}                          // Allow for overflows and range violations
          n := 256 * n + Ord( c);          // Asc numbers are considerd to be in base &256
          {$Q+R+}
        end
        else begin                         // Not in Asc mode now
          case c of                        // Check for special chars

            '.' :
              begin
                Inc( DotCount);            // Float   modifier
                DotPos := i;
              end;

            '%' : Base :=  2;              // Binary  modifier

            '&' : Base := 10;              // Decimal modifier

            '$' : Base := 16;              // Hex     modifier

            '''' : begin                   // Asc     modifier
                if                         // Asc     modifier - must be 1st char or 1st after '-'
                  (  i = 1) or
                  (( i = 2) and Neg)
                then Asc := True           // Switch to Asc mode
                else begin
                  Result := False;         // Not a valid Asc modifier
                  Break;                   // Not a number
                end;
              end;

            '-' : begin                    // Negative number modifier, unary minus
                if i = 1                   // but only if '-' is the first char
                then Neg := True           // Indicate negative number
                else begin
                  Result := False;         // Not a valid unary minus
                  Break;                   // Not a number
                end;
              end;

            else begin                     // No special char, should be a digit then
              if Base <= 36
              then d := DigitValues[ UpCase( c)]
              else d := DigitValues[ c];   // Invalid digits return 255

              if d >= Base                 // Check for invalid digit
              then begin
                Result := False;           // Invalid digit
                Break;                     // Not a number
              end
              else begin                   // Valid digit within number base
                {$Q-R-}                    // Allow for overflows and range violations
                n := Base * n + d;         // Accumulate digit into number
                {$Q+R+}
              end;
            end;
          end; // Case
        end
      end; // For

      if DotCount > 1                      // no more than two dots in a number
      then Result := False;

      if Result                            // Most likely it's a number then ... sift out some possible mistakes
      then begin
      {$Q-R-}                              // Allow for overflows and range violations
        if Neg                             // Check if an unary minus was seen
        then aNumber := - Int64( n)        // and if so return -n
        else aNumber :=   Int64( n);       // otherwise just    n
      {$Q+R+}

        if DotCount = 1
        then aNumber := aNumber / Power( 10, Length( aToken) - DotPos);
      end;

      if not Result                        // if not a number grReturn $deadbeef
      then aNumber := Integer( $deadbeef); // as a probable invalid number indicator.
    end;


    function    TForth.Float( const aToken: string; out aNumber: Variant): Boolean;
    //
    // Only supports floats in decimal notation.
    //
    var
      aFloat: Extended;
    begin
      Result  := False;
      aNumber := $deadbeef;

      try
        if TextToFloat( PChar( aToken), aFloat, fvExtended)
        then begin
          aNumber := aFloat;
          Result  := True;
        end;
      except
        // Just ignorre any exceptions and return False instead.
      end;
    end;


    function    TForth.MustFindToken( const aToken: string; aWordClass: TWordClass): TWord;
    var
      W: TWord;
    begin
      Result := FindToken( aToken);

      if ( not Assigned( Result)) or not ( Result is aWordClass)
      then begin
        Result := nil;
        if aWordClass = TWord                     // The generic case should find all words
        then
          RaiseErrorFmt(
            '%s ?',
            [
              aToken
            ]
          )
        else begin                                // and otherwise a specialised class was wanted
          W := FindToken( aToken);
          if Assigned( W)
          then
            RaiseErrorFmt(
              '%s is of type %s and %s was requested.',
              [
                aToken,
                W.TypeName,
                aWordClass.TypeName
              ]
            )
          else
            RaiseErrorFmt(
              '%s ?',
              [
                aToken
              ]
            );
        end;
      end;
    end;


    function    TForth.MustFindWord ( const aToken: string): TWord;
    begin
      Result := MustFindToken( aToken, TWord);
    end;


    procedure   TForth.MustBeCompilingFor( const aToken: string);
    begin
      if not FCompiling
      then
        RaiseErrorFmt(
          'The word ''%s'' can''t be used outside a colon definition. ',
          [ aToken]
        );
    end;


    procedure   TForth.MustNotBeCompilingFor( const aToken: string);
    begin
      if FCompiling
      then
        RaiseErrorFmt(
          'The word ''%s'' can''t be used inside a colon definition. ',
          [ aToken]
        );
    end;


    function    TForth.FindToken( const aToken: string): TWord;
    begin
      Result := FWords.FindToken( aToken);
    end;


    procedure   TForth.AcceptData( const aHandler: string; const aData: TCardinals);
    var
      aWord    : TWord;
      i        : Integer;
      aLen     : Integer;
      SWreg    : Integer;
      SIP      : Integer;
      SCW      : TWord;
    begin
      aWord := FindToken( aHandler);

      if Assigned( aWord)
      then begin
        aLen := Length( aData);
        SWreg := FWReg;
        SIP   := FIP;
        SCW   := FCW;
        FWreg := -1;
        FIP   := -1;
        FCW   := nil;

        try
          for i := aLen - 1 downto 0
          do DataPush( aData[ i]);
          DataPush( aLen);
          Exec( aWord);
        finally
          FWreg := SWReg;
          FIP   := SIP;
          FCW   := SCW;
        end;
      end;
    end;


    procedure   TForth.AcceptToken( const aToken: string);
    var
      aWord   : TWord;
      aNumber : Variant;
    begin
      try
        aWord := nil;
        // Search for global definitions
        if not Assigned( aWord)
        then aWord := FindToken( aToken);
        // When found let the word be interpreted or compiled depending
        // on the word's immediateness and the current compile state.
        if Assigned( aWord)
        then begin
          if aWord.Immediate or not FCompiling
          then Interpret( aWord)
          else Compile  ( aWord);
        end
        // Otherwise see if the token is a floating point number
        else if Float( aToken, aNumber)
        then begin
          if FCompiling
          then CompileLiteral( aNumber)                // Compile literal number
          else DataPush( aNumber);                     // At compile time all numbers are equal (32 bits, signed)
        end
        // Otherwise see if the token is a number
        else if Number( aToken, aNumber)
        then begin
          if FCompiling
          then CompileLiteral( aNumber)                // Compile literal number
          else DataPush( aNumber);                     // At compile time all numbers are equal (32 bits, signed)
        end
        // None worked, it's an error then.
        else RaiseErrorFmt( '%s ?', [ aToken]);
      except
        // All errors are trapped here.
        on E: Exception
        do Respond( E.Message);
      end;
    end;


    procedure   TForth.Explain( aWord: TWord);
    begin
      Respond( aWord.See);
    end;


    procedure   TForth.See;
    var
      aToken : string;
      aWord  : TWord;
    begin
      aToken := GetToken( ' ', False);
      aWord  := FindToken( aToken);

      if Assigned( aWord)
      then Respond( aWord.See)
      else RespondFmt( '%s ?', [ aToken])
    end;


    procedure   TForth.ShowWordInfo;
    var
      aToken : string;
      aWord  : TWord;
    begin
      aToken := GetToken( ' ', False);
      aWord  := FindToken( aToken);

      if Assigned( aWord)
      then Respond( aWord.ShowWordInfo)
      else RespondFmt( '%s ?', [ aToken])
    end;

    procedure   TForth.Exec( aWord: TWord);
    begin
      FCW := aWord;

      if FIP >= 0
      then FIP := FIP + 1;

      FCW.Interpret;
    end;


    procedure   TForth.Interpret( const aWord: TWord);
    begin
      FIP := -1;
      Exec( aWord);
      // Some code moved to TColon.Interpret
    end;


    procedure   TForth.AddDefer( const aName: string);
    var
      aWord : TWord;
    begin
      aWord := AddWord( aName, TDefer);

      with aWord
      do begin
        Reveal;
        EndCreate;
      end;

      FLastDefined := aWord;
    end;


    procedure   TForth.AddAlias( const anOldName, aNewName: string);
    var
      aWord   : TWord;
      anAlias : TAlias;
    begin
      aWord   := MustFindWord( anOldName);
      anAlias := AddWord( aNewName, TAlias) as TAlias;

      with anAlias
      do begin
        Aliased := aWord;
        Reveal;
        EndCreate;
      end;

      FLastDefined := anAlias;
    end;


    procedure   TForth.CompileLiteral( const aValue: Variant);
    begin
      FCode.CompileLiteral( aValue);
    end;


    procedure   TForth.Compile( const aWord: TWord);
    begin
      FCode.CompileWord( aWord);
    end;


    function    TForth.Here: Integer;
    begin
      if Assigned( Fcode)
      then Result := FCode.Here
      else Result := 0;
    end;


    procedure   TForth.SetHere( aValue: Integer);
    begin
      FCode.SetHere( aValue);
    end;


    function    TForth.DataDump: string;
    begin
      Result := FDataStack.Dump( 16);
    end;


    function    TForth.DataDepth: Integer;
    begin
      Result := FDataStack.Depth;
    end;


    procedure   TForth.DataPush( aValue: Variant);
    begin
      FDataStack.Push( aValue);
    end;


    procedure   TForth.DataPushWord( const aValue: TWord);
    begin
      FDataStack.PushWord( aValue);
    end;


    function    TForth.DataPop: Variant;
    begin
      Result := FDataStack.Pop;
    end;


    function    TForth.DataFetch: Variant;
    begin
      Result := FDataStack.Fetch;
    end;


    function    TForth.DataPopWord: TWord;
    begin
      Result := FDataStack.PopWord;
    end;


    procedure   TForth.DataSwap;
    begin
      FDataStack.Swap;
    end;


    function    TForth.LoopDump: string;
    begin
      Result := FLoopStack.Dump( 16);
    end;


    procedure   TForth.LoopPush( aValue: Variant);
    begin
      FLoopStack.Push( aValue);
    end;


    function    TForth.LoopPop: Variant;
    begin
      Result := FLoopStack.Pop;
    end;


    function    TForth.ReturnDump: string;
    begin
      Result := FReturnStack.Dump( 16);
    end;


    procedure   TForth.ReturnPush( aValue: Variant);
    begin
      FReturnStack.Push( aValue);
    end;


    function    TForth.ReturnPop: Variant;
    begin
      Result := FReturnStack.Pop;
    end;


    function    TForth.ReturnFetch: Variant;
    begin
      Result := FReturnStack.Fetch;
    end;


    function    TForth.ControlDump: string;
    begin
      Result := FControlStack.Dump( 16);
    end;


    procedure   TForth.ControlPush( aValue: Variant);
    begin
      FControlStack.Push( aValue);
    end;


    function    TForth.ControlPop: Variant;
    begin
      Result := FControlStack.Pop;
    end;


    function    TForth.StacksDump: string;
    begin
      Result := Format( 'data(%s) loop(%s) return(%s) control(%s)', [ DataDump, LoopDump, ReturnDump, ControlDump]);
    end;


    function    TForth.AddWord( const aName: string; aWordClass: TWordClass): TWord;
    begin
      if Assigned( FindToken( aName))
      then begin
        ShowWarningFmt( 'Redefinition of ''%s''', [ aName]);
        // Remove the word from the word list, it will no longer be found but its
        // code will still be in the code space such that old definitionswill still work.
        // FWords.RemoveWord( aName);
        // Bad idea ... the latest one should be found ...
      end;

      Result := FWords.AddWord( aName, aWordClass, Here);
    end;


    procedure   TForth.InternalWord( const aName: string; aProc: TInternalProc; const aComments: string);
    var
      anInternal : TInternal;
    begin
      anInternal := TInternal( AddWord( aName, TInternal));

      with anInternal
      do begin
        InternalProc := aProc;
        Comment      := aComments;
      end;

      FLastDefined := anInternal;

      with FLastDefined
      do begin
        Reveal;
        EndCreate;
      end;
    end;

    procedure   TForth.SetupInternalWords;
    begin
      InternalWord( 'reset'               , _reset               , '( ? -- ) // clear all');
      InternalWord( 'forget'              , _forget              , '( -- ) // Forget next inline token - and all that was defined later than it');
      Internalword( 'unload'              , _unload              , '( -- ) // Forget next inline token but do not complain about undefinedness');
      InternalWord( 'external'            , _external            , '( ... func id -- ) // Call into an externally defined function');
      InternalWord( 'words'               , _words               , '( -- ) // show the word list');
      InternalWord( '.s'                  , _dots                , '( -- ) // print stack dump');
      InternalWord( '.stacks'             , _dotstacks           , '( -- ) // print stack dump for all stacks');
      InternalWord( 'decimal'             , _decimal             , '( -- ) // &10 [[Base]] [[!]]');
      InternalWord( 'hex'                 , _hex                 , '( -- ) // &16 [[Base]] [[!]]');
      InternalWord( 'binary'              , _binary              , '( -- ) // &2  [[Base]] [[!]]');
      InternalWord( 'base@'               , _baseFetch           , '( -- n ) // [[Base]] [[@]]');
      InternalWord( 'base!'               , _baseStore           , '( n -- ) // [[Base]] [[!]]');
      InternalWord( '('                   , _lparen              , '( -- ) // skip input till ), use this for stack comments');
        Immediate;
      InternalWord( '//'                  , _slashslash          , '( -- ) // skip the rest of the input buffer');
        Immediate;

      InternalWord( '"'                   , _doubleQuote         , '( -- <string> ) // " a string" [[String]] or [[Data]] or [[RawData]] ...');
        Immediate;
      InternalWord( ''''                  , _singleQuote         , '( -- <string> ) // '' a string'' [[String]] or [[Data]] or [[RawData]] ...');
        Immediate;

      InternalWord( 'true'                , _true                , '( -- f ) // Push boolean true');
      InternalWord( 'false'               , _false               , '( -- f ) // Push boolean false');
      InternalWord( 'e'                   , _e                   , '( -- n ) // Push e');
      InternalWord( 'pi'                  , _pi                  , '( -- n ) // Push pi');
      InternalWord( 'bl'                  , _space               , '( -- n ) // Push a space character');
      InternalWord( '.'                   , _dot                 , '( n -- ) // print tos');
      InternalWord( '(.e)'                , _pdote               , '( n digits -- ) // print n in engineering format with specified digit count after the dot');
      InternalWord( '.base'               , _dotbase             , '( n base -- ) // print n in number base base');
      InternalWord( '+'                   , _plus                , '( n1 n2 -- n3 ) // adition');
      InternalWord( '-'                   , _minus               , '( n1 n2 -- n3 ) // subtraction of n2 from n1');
      InternalWord( '*'                   , _times               , '( n1 n2 -- n3 ) // multiplication');
      InternalWord( '/'                   , _div                 , '( n1 n2 -- n3 ) // division of n1 by n2');
      InternalWord( '/mod'                , _divMod              , '( n1 n2 -- n3 n4 ) // division of n1 by n2 with result n3 and remainder n4');
      InternalWord( 'int'                 , _int                 , '( n -- n ) // integer part of n');
      InternalWord( 'frac'                , _frac                , '( n -- n ) // fractional part of n');
      InternalWord( '^'                   , _power               , '( n1 n2 -- n3 ) // n1 to the power n2');
      InternalWord( 'exp'                 , _exp                 , '( n -- n ) // e to power of n');
      InternalWord( 'ln'                  , _ln                  , '( n -- n ) // natural log of n');
      InternalWord( 'rad2deg'             , _rad2deg             , '( n -- n ) // radians to degrees');
      InternalWord( 'deg2rad'             , _deg2rad             , '( n -- n ) // radians to degrees');
      InternalWord( 'sin'                 , _sin                 , '( n -- n ) // sine of n');
      InternalWord( 'asin'                , _asin                , '( n -- n ) // arc sine of n');
      InternalWord( 'cos'                 , _cos                 , '( n -- n ) // cosine of n');
      InternalWord( 'acos'                , _acos                , '( n -- n ) // arc cosine of n');
      InternalWord( 'tan'                 , _tan                 , '( n -- n ) // tangent of n');
      InternalWord( 'atan'                , _atan                , '( n -- n ) // arc tangent of n');
      InternalWord( 'atan2'               , _atan2               , '( n1 n2 -- ) // arc tangent of n1/n2');
      InternalWord( 'cosh'                , _cosh                , '( n -- n ) //');
      InternalWord( 'sinh'                , _sinh                , '( n -- n ) //');
      InternalWord( 'tanh'                , _tanh                , '( n -- n ) //');
      InternalWord( 'atanh'               , _atanh               , '( n -- n ) //');
      InternalWord( 'sech'                , _sech                , '( n -- n ) //');
      InternalWord( 'csh'                 , _csch                , '( n -- n ) //');
      InternalWord( 'acot'                , _acot                , '( n -- n ) //');
      InternalWord( 'asec'                , _asec                , '( n -- n ) //');
      InternalWord( 'acsc'                , _acsc                , '( n -- n ) //');
      InternalWord( 'acosh'               , _acosh               , '( n -- n ) //');
      InternalWord( 'asinh'               , _asinh               , '( n -- n ) //');
      InternalWord( 'acoth'               , _acoth               , '( n -- n ) //');
      InternalWord( 'asech'               , _asech               , '( n -- n ) //');
      InternalWord( 'acsch'               , _acsch               , '( n -- n ) //');
      InternalWord( 'sqrt'                , _sqrt                , '( n -- ) // arc tangent of n');
      InternalWord( 'abs'                 , _abs                 , '( n1 -- n2 ) // Get the absolute value of the top stack element');
      InternalWord( 'max'                 , _max                 , '( n1 n2 -- n3 ) // Get the maximum of the top two stack elements');
      InternalWord( 'min'                 , _min                 , '( n1 n2 -- n3 ) // Get the minimum of the top two stack elements');
      InternalWord( 'and'                 , _and                 , '( n1 n2 -- n3 ) // bitwise and');
      InternalWord( 'or'                  , _or                  , '( n1 n2 -- n3 ) // bitwise or');
      InternalWord( 'xor'                 , _xor                 , '( n1 n2 -- n3 ) // bitwise xor');
      InternalWord( 'negate'              , _negate              , '( n1 -- n2 ) // Unary minus performed on top stack element');
      InternalWord( 'not'                 , _not                 , '( n1 -- n2 ) // Logic inversion performed on top stack element');
      InternalWord( '<<'                  , _lmshift             , '( n1 n2 -- n3 ) // Shift the second stack element left over the number of bits indicated by the top stack element');
      InternalWord( '>>'                  , _rmshift             , '( n1 n2 -- n3 ) // Shift the second stack element right over the number of bits indicated by the top stack element');
      InternalWord( '='                   , _equal               , '( n1 n2 -- f ) // Push true when the top two stack elements are equal, false otherwise');
      InternalWord( '<>'                  , _unequal             , '( n1 n2 -- f ) // Push true when the top two stack elements are un-equal, false otherwise');
      InternalWord( '<'                   , _smaller             , '( n1 n2 -- f ) // Push true when u1 < u2, false otherwise');
      InternalWord( '>'                   , _greater             , '( n1 n2 -- f ) // Push true when u1 > u2, false otherwise');
      InternalWord( '<='                  , _smallerEqual        , '( n1 n2 -- f ) // Push true when u1 <= u2, false otherwise');
      InternalWord( '>='                  , _greaterEqual        , '( n1 n2 -- f ) // Push true when u1 >= u2, false otherwise');
      InternalWord( 'random'              , _random              , '( n1 n2 -- n3 ) // produce a random value [n1 .. n2]');

      InternalWord( '>$'                  , _tostring            , '( n -- n ) // make sure n is a string (typecast)');
      InternalWord( 'format'              , _format              , '( n fmt - ) // Format a float ... decimal only');
      InternalWord( 'format_time'         , _format_time         , '( nt nf -- nr ) // format date_time nt according to nf');

      InternalWord( 'constant'            , _constant            , '( n -- ) // Define a constant :: 12 Constant Joop');
      InternalWord( 'variable'            , _variable            , '( -- ) // Define a variable :: Variable Truus');

      InternalWord( '@'                   , _fetch               , '( n1 -- n2 ) // Fetch a value from an address');
      InternalWord( '!'                   , _store               , '( n1 n2 -- ) // Store value n1 at address n2');
      InternalWord( '+!'                  , _plusStore           , '( n1 n2 -- ) // Increment value at address n1 with n2');
      InternalWord( '-!'                  , _minStore            , '( n1 n2 -- ) // Decrement value at address n1 with n2');

      InternalWord( 'drop'                , _drop                , '( n1 -- ) // drop top stack element');
      InternalWord( 'dup'                 , _dup                 , '( n1 -- n1 n1 ) // duplicate top stack element');
      InternalWord( 'swap'                , _swap                , '( n1 n2 -- n2 n1 ) // swap top and sub top stack elements');
      InternalWord( '?dup'                , _questionDup         , '( n1 -- n1 n1 | 0 -- 0 ) // Dup the top stack element when it is not equal to zero');
      InternalWord( '2drop'               , _2drop               , '( n1 n2 -- ) // Drop the top two stack elements');
      InternalWord( 'over'                , _over                , '( n1 n2 -- n1 n2 n1 ) // Duplicate the second stack element and make it the top item');
      InternalWord( '2dup'                , _2dup                , '( n1 n2 -- n1 n2 n1 n2 ) // Duplicate the top two stack elements');
      InternalWord( '2swap'               , _2swap               , '( n1 n2 n3 n4 -- n3 n4 n1 n2 ) // Exchange the top two sets of two stack items');
      InternalWord( '2over'               , _2over               , '( n1 n2 n3 n4 -- n1 n2 n3 n4 n1 n2 ) // Duplicate the second set of two stack elements');
      InternalWord( 'rot'                 , _rot                 , '( n1 n2 n3 -- n2 n3 n1 ) // rotate the 3rd stack element to the top');
      InternalWord( '+rot'                , _plusrot             , '( n1 n2 n3 -- n2 n3 n1 ) // rotate the 3rd stack element to the top');
      InternalWord( '-rot'                , _minusrot            , '( n1 n2 n3 -- n3 n1 n2 ) // rotate the top stack element to the 3rd position');

      InternalWord( 'token'               , _token               , '( bSep -- <string> ) // Read bSep separated token from input stream');
      InternalWord( 'untoken'             , _untoken             , '( <string> bSep -- ) // prepend token and separator to input stream');
      InternalWord( 'flush_input'         , _flush_input         , '( -- ) // flush rest of input stream');
      InternalWord( '>mark'               , _forwardMark         , '( -- n ) // Forward jump (for if .. endif etc), also see [[<Mark]], [[>Resolve]] and [[<Resolve]]');
      InternalWord( '>resolve'            , _forwardResolve      , '( n -- ) // Resolve [[>Mark]], also see [[<Mark]] and [[<Resolve]]');
      InternalWord( '<mark'               , _backwardMark        , '( -- <string> | n ) // Backward jump (for begin ... again etc, also see [[>Mark]]');
      InternalWord( '<resolve'            , _backwardResolve     , '( <string> } n -- ), resilve [[<Mark]], also see [[>Mark]] and [[>Resolve]]');
      InternalWord( 'branch'              , _branch              , '( -- ), unconditional branch');
      InternalWord( '?branch'             , _fbranch             , '( f -- ), branch if TOS is zero');
      InternalWord( ':'                   , _colon               , '( -- ) // Start a colon definition : name ... [[;]]');
      InternalWord( 'recurse'             , _recurse             , '( -- ) // Call into self at run time, explicit recursion');
        Immediate;
      InternalWord( ';'                   , _semi                , '( -- ) // End a colon definition [[:]] name ... ;');
        Immediate;
      InternalWord( 'create'              , _create              , '( -- ): joop create ... [[does>]] ... ;');
      InternalWord( '(does>)'             , _pdoes               , '( -- <pfa> ), helper for [[Does>]]');
      InternalWord( 'does>'               , _does                , '( -- ), see [[Create]]');
        Immediate;
      InternalWord( 'w'                   , _wreg                , '( -- n ) // Push the current Wreg value');
      InternalWord( 'ip'                  , _ip                  , '( -- n ) // Push the current IP value');
      InternalWord( 'pfa@'                , _pfaFetch            , '( n<pfa> -- n<data> ) // Get a TData object from FCode[ n<pfa>]');
      InternalWord( ','                   , _comma               , '( n -- ) // Allots in code space');
      InternalWord( 'allot'               , _allot               , '( n -- ) // Allots in code space');
      InternalWord( 'here'                , _here                , '( -- n ) // Location in code space');
      InternalWord( 'immediate'           , _immediate           , '( -- ) // make latest word toggle its immediate bit');
      InternalWord( 'export'              , _export              , '( -- ) // make latest word toggle its exported bit');
      InternalWord( 'hide'                , _hide                , '( -- ) // make latest word set its visible bit');
      InternalWord( 'reveal'              , _reveal              , '( -- ) // make latest word clear its visible bit');

      // These are for the compiler
      InternalWord( '_begin'              , __begin              , '( -- ) // compiled begin');
      InternalWord( '_until'              , __until              , '( n -- ) // compiled until');
      InternalWord( '_while'              , __while              , '( n -- ) // compiled while');
      InternalWord( '_again'              , __again              , '( -- ) // compiled again');
      InternalWord( '_repeat'             , __repeat             , '( -- ) // compiled repeat');
      InternalWord( '_do'                 , __do                 , '( n1 n2 -- ) // compiled do');
      InternalWord( '_?do'                , __qdo                , '( n1 n2 -- ) // compiled ?do');
      InternalWord( '_loop'               , __loop               , '( -- ) // compiled loop');
      InternalWord( '_ploop'              , __ploop              , '( n -- ) // compiled +loop');
      InternalWord( '_unloop'             , __unloop             , '( -- ) // compiled unloop');
      InternalWord( '_leave'              , __leave              , '( -- ) // compliled leave');
      InternalWord( '_i'                  , __i                  , '( -- n ) // compiled i');
      InternalWord( '_j'                  , __j                  , '( -- n ) // compiled j');
      InternalWord( '_k'                  , __k                  , '( -- n ) // compiled k');
      InternalWord( '_>r'                 , __tor                , '( n -- ) // compiled >r');
      InternalWord( '_r>'                 , __rto                , '( -- n ) // compiled r>');
      InternalWord( '_r@'                 , __rfetch             , '( -- n ) // compiled r@');
      InternalWord( '_if'                 , __if                 , '( f -- ) // compiled if');
      InternalWord( '_else'               , __else               , '( -- ) // compiled else');
      InternalWord( '_then'               , __then               , '( -- ) // compiled then');
      InternalWord( '_endIf'              , __endif              , '( -- ) // compiled endif');
      InternalWord( '_case'               , __case               , '( -- ) // compiled case');
      InternalWord( '_default'            , __default            , '( -- ) // compiled default');
      InternalWord( '_endCase'            , __endcase            , '( -- ) // compiled endcase');
      InternalWord( '_of'                 , __of                 , '( n1 n2 -- n1 | ) // compiled of');
      InternalWord( '_endOf'              , __endof              , '( -- ) // compiled endof');
      InternalWord( '_in'                 , __in                 , '( n1 n2 -- n1 | ) // compiled in');
      InternalWord( '_endIn'              , __endin              , '( -- ) // compiled endin');
      InternalWord( '_exit'               , __exit               , '( -- ) // compiled exit');

      InternalWord( 'begin'               , _begin               , '( -- n n ) // Begin .. [[Again]], Begin .. 0 [[Until]], Begin .. (f) [[While]] .. [[Repeat]].');
        Immediate;
      InternalWord( 'again'               , _again               , '( n n -- ) // [[Begin]] .. Again');
        Immediate;
      InternalWord( 'until'               , _until               , '( n n -- ) // [[Begin]] .. ( f ) Until');
        Immediate;
      InternalWord( 'while'               , _while               , '( n n -- ) // [[Begin]] .. While .. [[Repeat]]');
        Immediate;
      InternalWord( 'repeat'              , _repeat              , '( n n -- ) // [[Begin]] .. [[While]] .. Repeat');
        Immediate;
      InternalWord( 'do'                  , _do                  , '( -- n n ) // n1 n2 Do .. [[Leave]] .. [[Loop]] or n [[+Loop]]');
        Immediate;
      InternalWord( '?do'                 , _qdo                 , '( -- n n ) // n1 n2 ?Do .. [[Leave]] .. [[Loop]] or n [[+Loop]]');
        Immediate;
      InternalWord( 'loop'                , _loop                , '( n n -- ) // n1 n2 [[Do]] .. [[Leave]] .. Loop');
        Immediate;
      InternalWord( '+loop'               , _ploop               , '( n n -- ) // n1 n2 [[Do]] .. n +Loop');
        Immediate;
      InternalWord( 'unloop'              , _unloop              , '( -- ) // n1 n2 [[Do]] .. [[Leave]] .. UnLoop .. [[Loop]]');
        Immediate;
      InternalWord( 'leave'               , _leave               , '( -- ) // Leave a [[do]] .. Leave .. [[loop]]');
        Immediate;
      InternalWord( 'i'                   , _i                   , '( -- n ) // Get inner most [[loop]] index');
        Immediate;
      InternalWord( 'j'                   , _j                   , '( -- n ) // Get 2nd level [[loop]] index');
        Immediate;
      InternalWord( 'k'                   , _k                   , '( -- n ) // Get 3rd level [[loop]] index');
        Immediate;
      InternalWord( '>r'                  , _tor                 , '( n -- ) // Move a value from the data to the loop stack');
        Immediate;
      InternalWord( 'r>'                  , _rto                 , '( -- n ) // Move a value from the loop to the data stack');
        Immediate;
      InternalWord( 'r@'                  , _rfetch              , '( -- n ) // Copy a value from the loop to the data stack');
        Immediate;
      InternalWord( 'if'                  , _if                  , '( -- n ) // n If .. [[Else]] .. [[Then]] or [[endIf]]');
        Immediate;
      InternalWord( 'else'                , _else                , '( n -- n ) // n [[If]] .. Else .. [[Then]] or [[endIf]]');
        Immediate;
      InternalWord( 'then'                , _then                , '( n -- ) // n [[If]] .. [[Else]] .. Then or [[endIf]]');
        Immediate;
      InternalWord( 'endIf'               , _endif               , '( n -- ) // n [[If]] .. [[Else]] .. [[Then]] or endIf');
        Immediate;
      InternalWord( 'case'                , _case                , '( -- 4 0 ) // ( n ) Case .. [[of]] .. [[endOf]] .. [[default]] .. [[endCase]]');
        Immediate;
      InternalWord( 'default'             , _default             , '( -- ) // ( n ) [[Case]] .. [[of]] .. [[endOf]] .. default .. [[endCase]]');
        Immediate;
      InternalWord( 'endCase'             , _endcase             , '( n0 n1 .. nn-1 0 4 -- ) // ( n ) [[Case]] .. [[of]] .. [[endOf]] .. [[default]] .. endCase');
        Immediate;
      InternalWord( 'of'                  , _of                  , '( -- n 5 ) // ( n ) [[Case]] .. of .. [[endOf]] .. [[default]] .. [[endCase]]');
        Immediate;
      InternalWord( 'endOf'               , _endof               , '( n1 5 -- n1 n2 ) // ( n ) [[Case]] .. [[of]] .. endOf .. [[default]] .. [[endCase]]');
        Immediate;
      InternalWord( 'in'                  , _in                  , '( -- n 5 ) // ( n ) [[Case]] .. in .. [[endIn]] .. [[default]] .. [[endCase]]');
        Immediate;
      InternalWord( 'endIn'               , _endin               , '( n1 5 -- n1 n2 ) // ( n ) [[Case]] .. [[in]] .. endIn .. [[default]] .. [[endCase]]');
        Immediate;
      InternalWord( 'exit'                , _exit                , '( -- ) // Immediately leave the current colon word returning to the caller');
        Immediate;

      InternalWord( 'defer'               , _defer               , '( -- ) // Defer XXXX');
      InternalWord( '`'                   , _tick                , '( -- <exec-token> ) // Search next word in input stream and return it''s execution token. Also see [[[`]]]');
      InternalWord( '[`]'                 , _tick                , '( -- <exec-token> ) // Search next word in input stream and return it''s execution token - immediate, see `');
        Immediate;
      InternalWord( 'alias'               , _alias               , '( -- ) // Defines an alias for a 4th word');
      InternalWord( 'is'                  , _is                  , '( <exec-token> -- ) // Resolve a [[defer]]red XXXX');
      InternalWord( 'implements'          , _implements          , '( -- ) // [[LastDefind]] [[Is]] XXXX');
      InternalWord( 'execute'             , _execute             , '( exec-token -- ?? ) // execute exec-token');
      InternalWord( 'evaluate'            , _evaluate            , '( $ -- ?? ) // evaluate a string as forth');
      InternalWord( 'explain'             , _explain             , '( exec-token -- ) // explain exec-token');
      InternalWord( 'see'                 , _see                 , '( -- ) // explain the next inline token');
      InternalWord( 'info'                , _info                , '( -- ) // show help for the next inline token');

      AcceptText( ': fence ( -- ) ;', 0);
    end;


    procedure   TForth.Immediate;
    begin
      if Assigned( FLastDefined)
      then
        with FLastDefined
        do Immediate := not Immediate;
    end;


    procedure   TForth.Exported;
    begin
      if Assigned( FLastDefined)
      then
        with FLastDefined
        do Exported := not Exported;
    end;


    procedure   TForth.Hide;
    begin
      if Assigned( FLastDefined)
      then FLastDefined.Hide;
    end;


    procedure   TForth.Reveal;
    begin
      if Assigned( FLastDefined)
      then FLastDefined.Reveal;
    end;


    procedure   TForth.ExternalWord;
    begin
      if Assigned( FOnExternalFunc)
      then FOnExternalFunc( Self, FInvoker, FDataStack)
      else ShowWarning( 'Calling external function without an external function handler being installed');
    end;


    procedure   TForth.CleanupAnonymousWords;
    var
      aWord : TWord;
      i     : Integer;
    begin
      for i := FAnonymous.Count - 1 downto 0
      do begin
        aWord := TWord( FAnonymous[ i]);

        if aWord.CFA >= Here
        then FAnonymous.Delete( i);
      end;
    end;


    procedure   TForth.Forget;
    var
      aToken  : string;
      aWord   : TWord;
    begin
      aToken := GetToken( ' ');
      aWord  := MustFindWord( aToken);

      if FWords.Forget( aWord)
      then begin
        SetHere( aWord.CFA);
        CleanupAnonymousWords;
        RespondFmt( 'forgot %s and all that came after', [ aToken]);
      end;
    end;


    procedure   TForth.Unload;
    var
      aToken  : string;
      aWord   : TWord;
    begin
      aToken := GetToken( ' ');

      if aToken <> endoffile
      then begin
        aWord  := FindToken( aToken);

        if Assigned( aWord)
        then begin
          if FWords.Forget( aWord)
          then begin
            SetHere( aWord.CFA);
            CleanupAnonymousWords;
            RespondFmt( 'unloaded %s', [ aToken]);
          end;
        end;

        FLastDefined := AddWord( aToken, TWord);
        Reveal;
      end;
    end;


    procedure   TForth.CheckControlStructure( aSeen: Integer);
    var
      anExpected : Integer;
      SExpected  : string;
      SSeen      : string;
      SIntro     : string;
    begin
      anExpected := ControlPop;

      if aSeen <> anExpected
      then begin
        if anExpected in [ Low( ControlNames) .. High( ControlNames)]
        then begin
          SExpected := ControlNames [ anExpected];
          SIntro    := ControlIntros[ anExpected];
        end
        else begin
          SExpected := Format( '<unkown %d>', [ anExpected]);
          SIntro    := '<unknown>';
        end;

        if aSeen in [ Low( ControlNames) .. High( ControlNames)]
        then SSeen := ControlNames[ aSeen]
        else SSeen := Format( '<unknown %d>', [ aSeen]);

        RaiseErrorFmt(
          'expected <%s>, seen <%s>, introduced by <%s>',
          [
            SExpected,
            SSeen,
            SIntro
          ]
        );
      end;
    end;


    procedure   TForth.CheckDataStack( const aMsg: string);
    begin
      if FDataStack.Depth <> 0
      then RaiseErrorFmt( 'Data stack not empty %s', [ aMsg]);
    end;


    procedure   TForth.CheckControlStack( const aMsg: string);
    begin
      if FControlStack.Depth <> 0
      then RaiseErrorFmt( 'Control stack not empty %s', [ aMsg]);
    end;


    procedure   TForth.CheckLoopStack( const aMsg: string);
    begin
      if FLoopStack.Depth <> 0
      then RaiseErrorFmt( 'Loop stack not empty %s', [ aMsg]);
    end;


    procedure   TForth.CheckReturnStack( const aMsg: string);
    begin
      if FReturnStack.Depth <> 0
      then RaiseErrorFmt( 'Return stack not empty %s', [ aMsg]);
    end;


    procedure   TForth.CheckStacks( const aMsg: string);
    begin
      CheckLoopStack   ( aMsg);
      CheckReturnStack ( aMsg);
      CheckControlStack( aMsg);
      CheckDataStack   ( aMsg);
    end;


    function    TForth.Words: string;
    begin
      Result := FWords.Words;
    end;


    function    TForth.CreateExportList: TStringList;
    begin
      Result := FWords.CreateExportList;
    end;


    function    TForth.CfaToWord( aCFA: Integer): TWord;
    begin
      if Assigned( FWords)
      then Result := FWords.CfaToWord( aCFA)
      else Result := nil;
    end;


//  public

    constructor TForth.Create;
    begin
      inherited;
      FDataStack     := TStack.Create( Self, 'data'   ); FDataStack   .OnUnderflow := DoStackUnderflow;
      FLoopStack     := TStack.Create( Self, 'loop'   ); FLoopStack   .OnUnderflow := DoStackUnderflow;
      FReturnStack   := TStack.Create( Self, 'return' ); FReturnStack .OnUnderflow := DoStackUnderflow;
      FControlStack  := TStack.Create( Self, 'control'); FControlStack.OnUnderflow := DoStackUnderflow;
      FCode          := TCode .Create( Self, 'code');    // Adding FCode to the word list -> disaster on saving ... don't!!
      FWords         := TWords.Create( Self);
      FAnonymous     := TObjectList.Create;
      Reset;
    end;


    destructor  TForth.Destroy; // override;
    begin
      FreeAndNil( FControlStack);
      FreeAndNil( FReturnStack );
      FreeAndNil( FLoopStack   );
      FreeAndNil( FDataStack   );
      FreeAndNil( FCode        );
      FreeAndNil( FWords       );
      FreeAndNil( FAnonymous   );
      inherited;
    end;


    procedure   TForth.StackReset;
    begin
      FDataStack   .Clear;
      FLoopStack   .Clear;
      FReturnStack .Clear;
      FControlStack.Clear;
      FBase         := 10;
    end;


    procedure   TForth.Reset;
    begin
      StackReset;
      FCode     .Clear;
      FWords    .Clear;
      FAnonymous.Clear;
      SetupInternalWords;
    end;


    procedure   TForth.Save( const aFileName: string);
    begin
      SaveWords( aFileName);

      if Assigned( FOnSave)
      then FonSave( Self);

      RespondFmt( 'saved %s', [ aFileName]);
    end;


    procedure   TForth.AcceptText( const aText: string; anInvoker: Integer);
    var
      aToken : string;
    begin
      FInvoker     := anInvoker;
      FInputBuffer := aText;
      aToken       := GetToken( ' ');

      while aToken <> endoffile
      do begin
        AcceptToken( aToken);
        aToken := GetToken( ' ');
      end;
    end;


    procedure   TForth.SaveWords( const aStringList: TStringList); // overload;
    var
      i     : Integer;
      aWord : TWord;
    begin
      if Assigned( aStringList)
      then begin
        with FWords
        do begin
          for i := 0 to Count - 1
          do begin
            aWord := Word[ i];

            if not ( aWord is TInternal)
            then aStringList.Add( aWord.See);
          end;
        end;
      end;
    end;


    procedure   TForth.SaveWords( const aFileName: string); // overload;
    var
      aStringList : TStringList;
    begin
        aStringList := TStringList.Create;
        try
          SaveWords( aStringList);
          aStringList.SaveToFile( aFileName);
        finally
          aStringList.Free;
        end;
    end;


    procedure   TForth.LoadWords( const aStringList: TStringList; anInvoker: Integer); // overload;
    var
      S : string;
    begin
      if Assigned( aStringList)
      then begin
        for S in aStringList
        do AcceptText( S, anInvoker);
      end;

      RespondFmt( 'loaded %s', [ FCurrentFileName]);
    end;


    procedure   TForth.LoadWords( const aFileName: string; anInvoker: Integer); // overload;
    var
      aStringList : TStringList;
    begin
      if FileExists( aFileName)
      then begin
        FCurrentFileName := aFileName;
        aStringList      := TStringList.Create;
        try
          aStringList.LoadFromFile( aFileName);
          LoadWords( aStringList, anInvoker);
        finally
          aStringList.Free;
        end;
      end;
    end;


var

  Tmp : Integer;

initialization


finalization

  Tmp := TWord.WordCounter;
  Tmp := TWord.AnonymousCounter;

end.
