unit KnobsLightning;

{
  Found this on the webs as Java code :

    https://www.cc.gatech.edu/grads/m/mluffel/2009/fall/cs7001/dendrites/
    https://www.cc.gatech.edu/grads/m/mluffel/2009/fall/cs7001/dendrites/applet/fast_dla.pde

  Original author notes :

    Mark Luffel, Georgia Tech, 2009

    These branching patterns are known as dendrites and they appear in nature in various instances: a patch of lichen,
    mineral deposits, fungus mycelium, lightning, etc.

  // ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  Ported to Delph by Blue Hell :


   COPYRIGHT 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.SysUtils, System.Classes, System.Types, System.Math, System.Generics.Collections, System.Generics.Defaults,
  System.JSON, System.JSON.Types, System.JSON.Builders, System.JSON.Writers, System.JSON.Readers, System.RTTI,

  KnobsUtils;

type

  ELightning = class( Exception);

  TCellMatrix<T> = class
  private
    FData   : array of T;
    FWidth  : Integer;
    FHeight : Integer;
    FCount  : Integer;
  private
    procedure   SetWidth ( aValue: Integer);
    procedure   SetHeight( aValue: Integer);
    function    GetData( anX, anY: Integer): T;
    procedure   SetData( anX, anY: Integer; aValue: T);
  public
    constructor Create( aWidth, aHeight: Integer);
    procedure   AssignFrom( const aValue: TCellMatrix<T>);
  public
    property    Count                    : Integer read FCount;
    property    Width                    : Integer read FWidth  write SetWidth;
    property    Height                   : Integer read FHeight write SetHeight;
    property    Data[ anX, anY: Integer] : T       read GetData write SetData;                                  default;
  end;


  TCellLocation = record
    X : Integer;
    Y : Integer;
  end;

  TCandidate = record
    X : Integer;
    Y : Integer;
    P : TSignal;
  end;


  TRawPotential = TCellMatrix<TSignal>;
  TRawIsShape   = TCellMatrix<Boolean>;


  TLightningEngine = class
  private
    FWidth         : Integer;
    FHeight        : Integer;
    FStartRandom   : Boolean;
    FEta           : TSignal;
    FRandomNew     : TSignal;
    FPotential     : TRawPotential;
    FIsShape       : TRawIsShape;
    FIsCandidate   : TCellMatrix<Boolean>;
    FCandidates    : TList<TCandidate>;
    FNeighborhood  : array[ 0 .. 7] of TCellLocation;
    FShowField     : Boolean;
    FMinValue      : TSignal;
    FMaxValue      : TSignal;
    FComparison    : TComparison<TCandidate>;
    FLocked        : Boolean;
  private
    function    GetPotential( anX, anY: Integer): TSignal;
    function    GetIsShape  ( anX, anY: Integer): Boolean;
    function    GetPotentialAsString : string;
    function    GetIsShapeAsString   : string;
    function    GetAsString          : string;
    procedure   SetAsString( const aValue: string);
    procedure   SetRawPotential( const aValue: TRawPotential);
    procedure   SetRawIsShape  ( const aValue : TRawIsShape );
  private
    procedure   DataInitialize;
    procedure   DataUnintialize;
    procedure   AddPixel( aCandidateIndex, anX, anY: Integer);
    function    FullPotential( anX, anY: Integer): TSignal;
    function    IncrementPotential( anX, anY, anSX, anSY: Integer): TSignal;
 // procedure   ResetCandidates;
  public
    constructor Create( aWidth, aHeight: Integer; anEta, aRandomNew: TSignal; aStartRandom: Boolean);
    destructor  Destroy;                                                                                       override;
    procedure   Reset;
    procedure   Step;
    procedure   AddRandomPoint;
    procedure   SetSize( aWidth, aHeight: Integer);
  private
    property    PotentialAsString : string  read GetPotentialAsString;
    property    IsShapeAsString   : string  read GetIsShapeAsString;
  public
    property    Width             : Integer       read FWidth;
    property    Height            : Integer       read FHeight;
    property    ShowField         : Boolean       read FShowField   write FShowField;
    property    StartRandom       : Boolean       read FStartRandom write FStartRandom;
    property    Eta               : TSignal       read FEta         write FEta;
    property    RandomNew         : TSignal       read FRandomNew   write FRandomNew;
    property    MaxValue          : TSignal       read FMaxValue;
    property    MinValue          : TSignal       read FMinValue;
    property    AsString          : string        read GetAsString  write SetAsString;
    property    RawPotential      : TRawPotential read FPotential   write SetRawPotential;
    property    RawIsShape        : TRawIsShape   read FIsShape     write SetRawIsShape;
  public
    property    Potential[ anX, anY: Integer] : TSignal read GetPotential;
    property    IsShape  [ anX, anY: Integer] : Boolean read GetIsShape;
  end;


implementation


{ ========
  TCellMatrix<T> = class
  private
    FData   : array of T;
    FWidth  : Integer;
    FHeight : Integer;
    FCount  : Integer;
  public
    property    Count                    : Integer read FCount;
    property    Width                    : Integer read FWidth  write SetWidth;
    property    Height                   : Integer read FHeight write SetHeight;
    property    Cell[ anX, anY: Integer] : T       read GetCell write SetCell;
  private
}

    procedure   TCellMatrix<T>.SetWidth( aValue: Integer);
    begin
      SetLength( FData, aValue* FHeight);
      FCount  := aValue * FHeight;
      FWidth  := aValue;
    end;


    procedure   TCellMatrix<T>.SetHeight( aValue: Integer);
    begin
      SetLength( FData, FWidth * aValue);
      FCount  := FWidth * aValue;
      FHeight := aValue;
    end;


    function    TCellMatrix<T>.GetData( anX, anY: Integer): T;
    begin
      Result := FData[ anx + Width * anY];
    end;


    procedure   TCellMatrix<T>.SetData( anX, anY: Integer; aValue: T);
    begin
      FData[ anx + Width * anY] := aValue;
    end;


//  public

    constructor TCellMatrix<T>.Create( aWidth, aHeight: Integer);
    begin
      SetLength( FData, aWidth * aHeight);
      FCount  := aWidth * aHeight;
      FWidth  := aWidth;
      FHeight := aHeight;
    end;


    procedure   TCellMatrix<T>.AssignFrom( const aValue: TCellMatrix<T>);
    begin
      if   Assigned( aValue)
      and  ( aValue.Count = Count)
      then Move( aValue.FData[ 0], FData[ 0], Count * SizeOf( T));
    end;



{ ========
  TCellLocation = record
    X : Integer;
    Y : Integer;
  end;

  TCandidate = record
    X : Integer;
    Y : Integer;
    P : TSignal;
  end;


  TLightningEngine = class
  private
    FWidth         : Integer;
    FHeight        : Integer;
    FStartRandom   : Boolean;
    FEta           : TSignal;
    FRandomNew     : TSignal;
    FPotential     : TCellMatrix<TSignal>;
    FIsCandidate   : TCellMatrix<Boolean>;
    FIsShape       : TCellMatrix<Boolean>;
    FCandidates    : TList<TCandidate>;
    FNeighborhood  : array[ 0 .. 7] of TCellLocation;
    FShowField     : Boolean;
    FMinValue      : TSignal;
    FMaxValue      : TSignal;
    FComparison    : TComparison<TCandidate>;
  private
    property    PotentialAsString : string  read GetPotentialAsString;
    property    IsShapeAsString   : string  read GetIsShapeAsString;
  public
    property    Width             : Integer read FWidth;
    property    Height            : Integer read FHeight;
    property    ShowField         : Boolean read FShowField           write FShowField;
    property    StartRandom       : Boolean read FStartRandom         write FStartRandom;
    property    Eta               : TSignal read FEta                 write FEta;
    property    RandomNew         : TSignal read FRandomNew           write FRandomNew;
    property    MaxValue          : TSignal read FMaxValue;
    property    MinValue          : TSignal read FMinValue;
    property    AsString          : string  read GetAsString          write SetAsString;
  public
    property    Potential[ anX, anY: Integer] : TSignal read GetPotential;
    property    IsShape  [ anX, anY: Integer] : Boolean read GetIsShape;
  private
}

    function    TLightningEngine.GetPotential( anX, anY: Integer): TSignal;
    begin
      Result := FPotential[ anX, anY];
    end;


    function    TLightningEngine.GetIsShape( anX, anY: Integer): Boolean;
    begin
      Result := FIsShape[ anX, anY];
    end;


    function    TLightningEngine.GetPotentialAsString: string;
    var
      i : Integer;
      j : Integer;
      S : string;
    begin
      Result := '[';

      for i := 0 to Width - 1
      do begin
        for j := 0 to Height - 1
        do begin
          S := FloatToStr( Potential[ i, j]);

          if   Result = '['
          then Result := Result + Format(  '%s', [ S])
          else Result := Result + Format( ',%s', [ S]);
        end
      end;

      Result := Result + ']';
    end;


    function    TLightningEngine.GetIsShapeAsString: string;
    var
      i : Integer;
      j : Integer;
      S : string;
    begin
      Result := '[';

      for i := 0 to Width - 1
      do begin
        for j := 0 to Height - 1
        do begin
          S := BoolToStr( IsShape[ i, j], False);

          if   Result = '['
          then Result := Result + Format(  '%s', [ S])
          else Result := Result + Format( ',%s', [ S]);
        end
      end;

      Result := Result + ']';
    end;


    function    TLightningEngine.GetAsString: string;
    var
      aBuilder       : TJSONObjectBuilder;
      aWriter        : TJsonTextWriter;
      aStringWriter  : TStringWriter;
      aStringBuilder : TStringBuilder;
    begin
      aStringBuilder     := TStringBuilder.Create;
      aStringWriter      := TStringWriter.Create( aStringBuilder);
      aWriter            := TJsonTextWriter.Create( aStringWriter);
      aWriter.Formatting := TJsonFormatting.None;

      try
        aBuilder := TJSONObjectBuilder.Create( aWriter);

        try
          aBuilder
            .BeginObject
              .Add( 'type'       , ClassName)
              .Add( 'width'      , Width    )
              .Add( 'height'     , Height   )
              .Add( 'showfield'  , ShowField)
              .BeginArray( 'potential')
                .AddElements( PotentialAsString)
              .EndArray
              .BeginArray( 'isshape')
                .AddElements( IsShapeAsString)
              .EndArray
            .EndObject;

           Result := aStringBuilder.ToString;
        finally
          aBuilder.DisposeOf;
        end;
      finally
        aWriter       .DisposeOf;
        aStringWriter .DisposeOf;
        aStringBuilder.DisposeOf;
      end;
    end;


    procedure   TLightningEngine.SetAsString( const aValue: string);
    type
      TLocState = (
        ltNone       ,
        ltScanning   ,
        ltType       ,
        ltWidth      ,
        ltHeight     ,
        ltShowField  ,
        ltPotential  ,
        ltIsShape    ,
        ltInPotential,
        ltInIsShape  ,
        ltError      ,
        ltSuccess
      );
    var
      anError : string;
      aState  : TLocState;

      procedure SetError( const aMsg: string);
      begin
        aState  := ltError;
        anError := aMsg;
      end;

      procedure SetErrorFmt( const aFmt: string; const anArgs: array of const);
      begin
        SetError( Format( aFmt, anArgs, AppLocale));
      end;

    var
      aStringReader : TStringReader;
      aReader       : TJsonTextReader;
      aType         : string;
      aWidth        : Integer;
      aHeight       : Integer;
      aShowField    : Boolean;
      aPotential    : TCellMatrix<TSignal>;
      aIsShape      : TCellMatrix<Boolean>;
      i             : Integer;
      j             : Integer;
    begin
      aState     := ltNone;
      aType      := '';
      aWidth     := 0;
      aHeight    := 0;
      aShowField := False;
      aPotential := nil;
      aIsShape   := nil;
      i          := 0;
      j          := 0;
      anError    := '';

      try
        aStringReader := TStringReader.Create( aValue);

        try
          aReader := TJsonTextReader.Create( aStringReader);

          try
            while aReader.Read
            and   not ( aState in [ ltError, ltSuccess])
            do begin
              case aReader.TokenType of

                TJsonToken.StartObject :

                  if aState = ltNone
                  then aState := ltScanning
                  else SetError( 'State ltNone expected for StartObject');

                TJsonToken.EndObject :

                  if aState = ltScanning
                  then aState := ltSuccess
                  else SetError( 'State ltScanning expected for EndObject');

                TJsonToken.PropertyName :

                    if aState = ltScanning
                    then begin
                      if      SameText( aReader.Value.AsString, 'type'     )
                      then aState := ltType
                      else if SameText( aReader.Value.AsString, 'width'    )
                      then aState := ltWidth
                      else if SameText( aReader.Value.AsString, 'height'   )
                      then aState := ltHeight
                      else if SameText( aReader.Value.AsString, 'showfield')
                      then aState := ltShowField
                      else if SameText( aReader.Value.AsString, 'potential')
                      then aState := ltPotential
                      else if SameText( aReader.Value.AsString, 'isshape'  )
                      then aState := ltIsShape
                      else SetError( 'type, width, height, showfiled, potential or isshape expected');
                    end
                    else SetError( 'state ltScanning expected for PropertyName');

                TJsonToken.String :

                  if aState = ltType
                  then begin
                    aType := aReader.Value.AsString;

                    if SameText( aType, ClassName)
                    then aState := ltScanning
                    else SetErrorFmt( 'Wrong typename: '' %s '', expected: ''%s''', [ aType, ClassName]);
                  end
                  else if aState = ltInPotential
                  then begin
                    if   ( i >= 0     )
                    and  ( i < aWidth )
                    and  ( j >=0      )
                    and  ( j < aHeight)
                    then begin
                      aPotential[ i, j] := aReader.Value.AsExtended;
                      Inc( j);

                      if( j > Height)
                      then begin
                        j := 0;
                        Inc( i);
                      end;
                    end
                    else SetError( 'Width or Height out of bounds error');
                  end
                  else SetError( 'State ltType or ltInPotential expected for String token type');

                TJsonToken.Integer :

                  if aState = ltWidth
                  then begin
                    aWidth := aReader.Value.AsInteger;

                    if aWidth = Width
                    then aState := ltScanning
                    else SetErrorFmt( 'Width error, seen %d expected %d', [ aWidth, Width]);
                  end
                  else if aState = ltHeight
                  then begin
                    aHeight := aReader.Value.AsInteger;

                    if aHeight = Height
                    then aState := ltScanning
                    else SetErrorFmt( 'Height error, seen %d expected %d', [ aHeight, Height]);
                  end
                  else if aState = ltInPotential
                  then begin
                    if   ( i >= 0     )
                    and  ( i < aWidth )
                    and  ( j >=0      )
                    and  ( j < aHeight)
                    then begin
                      aPotential[ i, j] := aReader.Value.AsInteger;
                      Inc( j);

                      if( j >= Height)
                      then begin
                        j := 0;
                        Inc( i);
                      end;
                    end
                    else SetError( 'Width or Height out of bounds error');
                  end
                  else if aState = ltInIsShape
                  then begin
                    if   ( i >= 0     )
                    and  ( i < aWidth )
                    and  ( j >=0      )
                    and  ( j < aHeight)
                    then begin
                      aIsShape[ i, j] := aReader.Value.AsInteger <> 0;
                      Inc( j);

                      if( j >= Height)
                      then begin
                        j := 0;
                        Inc( i);
                      end;
                    end
                    else SetError( 'Width or Height out of bounds error');
                  end
                  else SetError( 'State ltWidth, ltHeight, ltInPotential, ltInIsHape expected for Integer token type');

                TJsonToken.Float :

                  if aState = ltInPotential
                  then begin
                    if   ( i >= 0     )
                    and  ( i < aWidth )
                    and  ( j >=0      )
                    and  ( j < aHeight)
                    then begin
                      aPotential[ i, j] := aReader.Value.AsExtended;
                      Inc( j);

                      if( j >= Height)
                      then begin
                        j := 0;
                        Inc( i);
                      end;
                    end
                    else SetError( 'Width or Height out of bounds error');
                  end
                  else SetError( 'State ltInPotential expected for Float token type');

                TJsonToken.Boolean :

                  if aState = ltShowField
                  then begin
                    aShowField := aReader.Value.AsBoolean;
                    aState     := ltScanning;
                  end
                  else SetError( 'State ltShowField expected for Boolean token type');

                TJsonToken.StartArray :

                  case aState of
                    ltPotential : begin aState := ltInPotential; aPotential := TCellMatrix<TSignal>.Create( aWidth, aHeight); i := 0; j := 0; end;
                    ltIsShape   : begin aState := ltInIsShape;   aIsShape   := TCellMatrix<Boolean>.Create( aWidth, aHeight); i := 0; j := 0; end;
                    else  SetError( 'State ltPotentia or ltIsShape expected for StartArray token type');
                  end;

                TJSonToken.EndArray :

                  if aState in [ ltInPotential, ltInIsShape]
                  then aState := ltScanning
                  else SetError( 'State ltInPotential or ltInIsShape expected for EndArray toke type');

                else SetError( 'Unexpected token type');

              end;
            end;
          finally
            aReader.DisposeOf;
          end;
        finally
          aStringReader.DisposeOf;
        end;

        if aState = ltError
        then raise ELightning.CreateFmt( 'ReadString: error parsing JSON value [%s]', [ anError])
        else if not ( aState in [ ltNone, ltSuccess])
        then raise ELightning.CreateFmt( 'ReadString: not successfully completed [%s]', [ anError]);

        if   ( aState = ltSuccess)
        and  Assigned( aPotential)
        and  Assigned( aIsShape  )
        then begin
          ShowField := aShowField;
          FPotential.AssignFrom( aPotential);
          FIsShape  .AssignFrom( aIsShape  );
       // ResetCandidates;
        end;
      finally
        FreeAndNil( aPotential);
        FreeAndNil( aIsShape  );
      end;
    end;


    procedure   TLightningEngine.SetRawPotential( const aValue: TRawPotential);
    begin
      if   Assigned( aValue)
      and  ( aValue.Count = FPotential.Count)
      then Move( aValue.FData[ 0], FPotential.FData[ 0], FPotential.Count * SizeOf( TSignal));
    end;


    procedure   TLightningEngine.SetRawIsShape( const aValue : TRawIsShape );
    begin
      if   Assigned( aValue)
      and  ( aValue.Count = FIsShape.Count)
      then Move( aValue.FData[ 0], FIsShape.FData[ 0], FPotential.Count * SizeOf( Boolean));
    end;


//  private

    procedure   TLightningEngine.DataInitialize;
    begin
      FComparison :=
        function( const aLeft, aRight: TCandidate): Integer
        begin
          Result := TComparer<TSignal>.Default.Compare( aLeft.P, aRight.P);
        end;

      FPotential   := TCellMatrix<TSignal>.Create( FWidth, FHeight);
      FIsCandidate := TCellMatrix<Boolean>.Create( FWidth, FHeight);
      FIsShape     := TCellMatrix<Boolean>.Create( FWidth, FHeight);
      FCandidates  := TList<TCandidate>.Create( TComparer<TCandidate>.Construct( FComparison));
      FNeighborhood[ 0].X := -1; FNeighborhood[ 0].Y := -1;
      FNeighborhood[ 1].X :=  0; FNeighborhood[ 1].Y := -1;
      FNeighborhood[ 2].X :=  1; FNeighborhood[ 2].Y := -1;
      FNeighborhood[ 3].X := -1; FNeighborhood[ 3].Y :=  0;
      FNeighborhood[ 4].X :=  1; FNeighborhood[ 4].Y :=  0;
      FNeighborhood[ 5].X := -1; FNeighborhood[ 5].Y :=  1;
      FNeighborhood[ 6].X :=  0; FNeighborhood[ 6].Y :=  1;
      FNeighborhood[ 7].X :=  1; FNeighborhood[ 7].Y :=  1;

      if StartRandom
      then AddRandomPoint
      else begin
        FIsCandidate[ Width div 2, Height div 2] := True;
        AddPixel    ( -1, Width div 2, Height div 2);
      end;
    end;


    procedure   TLightningEngine.DataUnintialize;
    begin
      FreeAndNil( FCandidates);
      FreeAndNil( FIsShape);
      FreeAndNil( FIsCandidate);
      FreeAndNil( FPotential);
    end;


    procedure   TLightningEngine.AddPixel( aCandidateIndex, anX, anY: Integer);
    var
      i  : Integer;
      j  : Integer;
      C  : TCandidate;
      NX : Integer;
      NY : Integer;
      NP : TSignal;
    begin
      FIsShape[ anX, anY] := True;

      // Remove this location from candidates

      if ( aCandidateIndex >= 0) and ( aCandidateIndex < FCandidates.Count)
      then FCandidates.Delete( aCandidateIndex);

      if ShowField     // Slow .. it says ... show the potential field .. this is a nice view
      then begin
        for i := 0 to Width - 1
        do begin
          for j := 0 to Height - 1
          do FPotential[ i, j] := FPotential[ i, j] + IncrementPotential( i, j, anX, anY);
        end;
      end
      else begin        // Faster .. it says .. so less nice I guess
        for i := 0 to FCandidates.Count - 1
        do begin
          C := FCandidates[ i];
          FPotential[ C.X, C.Y] := FPotential[ C.X, C.Y] + IncrementPotential( C.X, C.Y, anX, anY);
        end;
      end;

      // Make any neighbors into candidates if they ar not already

      for i := 0 to Length( FNeighborhood) - 1
      do begin
        NX := anX + FNeighborhood[ i].X;
        NY := anY + FNeighborhood[ i].Y;

        if   ( NX >= 0    )                    // Skip out of bound neighbors
        and  ( NX < Width )
        and  ( NY >= 0    )
        and  ( NY < Height)
        and  not FIsCandidate[ NX, NY]
        then begin
          FIsCandidate[ NX, NY] := True;     // become a candidate
          NP                    := FullPotential( NX, NY);
          FPotential[ NX, NY]   := NP;
          C.X := NX;
          C.Y := NY;
          C.P := NP;
          FCandidates.Add( C);
        end;
      end;
    end;


    function    TLightningEngine.FullPotential( anX, anY: Integer): TSignal;
    // Calculate the potential for a point from scratch
    // used when adding a new candidate whose potential has not been tracked
    var
      i : Integer;
      j : Integer;
    begin
      Result := 0;

      for i := 0 to Width - 1
      do begin
        for j := 0 to Height - 1
        do begin
          if( IsShape[ i, j])                                          // if a pixel is part of the dendrite then compute
          then Result := Result + IncrementPotential( i, j, anX, anY); // its contribution to the potential at this point
        end;
      end;
    end;


    function    TLightningEngine.IncrementPotential( anX, anY, anSX, anSY: Integer): TSignal;
    // Calculate the cchange in potential, this is symmetric and isotropic.
    // Changes to this function can result im more complex shapes
    var
      D : TSignal;
    begin
      D := Distance( Point( anX - anSX, anY - anSY));

      if D <> 0
      then Result := 1 - 0.5 / D
      else Result := 1;
    end;


//    function    TLightningEngine.IncrementPotential( anX, anY, anSX, anSY: Integer): TSignal;
//    // Calculate the cchange in potential, this is symmetric and isotropic.
//    // Changes to this function can result im more complex shapes
//    var
//      D  : TSignal;
//      DX : TSignal;
//      DY : TSignal;
//    begin
//      DX := anX - anSX;
//      DY := anY - anSY;
//      D  := Sqrt( DX * DX + DY * DY);
//
//      if D <> 0
//      then Result := 1 - 0.5 / D
//      else Result := 1;
//    end;


//    procedure   TLightningEngine.ResetCandidates;
//    var
//      i : Integer;
//      j : Integer;
//    begin
//      FCandidates.Clear;
//
//      for i := 0 to Width - 1
//      do begin
//        for j := 0 to Height - 1
//        do begin
//          if IsShape[ i, j]
//          then AddPixel( -1, i, j);
//        end;
//      end;
//    end;


//  public

    constructor TLightningEngine.Create( aWidth, aHeight: Integer; anEta, aRandomNew: TSignal; aStartRandom: Boolean);
    begin
      inherited Create;
      FLocked      := True;
      FWidth       := aWidth;
      FHeight      := aHeight;
      FEta         := anEta;
      FrandomNew   := aRandomNew;
      FStartRandom := aStartRandom;
      DataInitialize;
      FLocked := False;
    end;


    destructor  TLightningEngine.Destroy; // override;
    begin
      DataUnintialize;
      inherited;
    end;


    procedure   TLightningEngine.Reset;
    begin
      DataUnintialize;
      DataInitialize;
    end;


    procedure   TLightningEngine.Step;
    var
      Scaling      : TSignal;
      Value        : TSignal;
      Prev         : TSignal;
      PotentialSum : TSignal;
      i            : Integer;
      V            : TCandidate;
    begin
      if FLocked
      then Exit;

      // Slurp the potential back into the candidate list

      if Random < RandomNew
      then AddRandomPoint;

      for i := 0 to FCandidates.Count - 1
      do begin
        V   := FCandidates[ i];
        V.P := FPotential[ V.X, V.Y];
        FCandidates.Items[ i] := V;
      end;

      FCandidates.Sort;

      // We need the max and the min value to scale our potentials into [0,1]
      // so that our N parameter (Eta) works

      if FCandidates.Count > 0
      then begin
        FMinValue := FCandidates[ 0                    ].P;
        FMaxValue := FCandidates[ FCandidates.Count - 1].P;
      end
      else begin
        FMinValue := 0;
        FMaxValue := 0;
      end;

      if FMinValue = FMaxValue
      then Scaling := 1
      else Scaling := 1 / ( FMaxValue - FMinValue);

      // Calculate the partial sum, this enables us to pick a uniformly random sample from 0 .. PotentialSum
      // and then binary search to find the candidate whose partial sum is is closest.
      // Because the partial sum is sorted by definition we do not need to sort the list beforehand .. the
      // sorting above is just to get the min and max.

      // Not sure here .. about the above ... the later binary search needs the loist to be sorted ...
      // but it is changed before ... in:
      //
      //   V.P := Prev;
      //   FCandidates[ i] := V;
      //
      // So maybe I'll need to resort? or use an Insert .. instead of an assignement? or something ..
      //
      // Hmm .. actually same issue in Java - so probaly this is OK.

      Prev := 0;

      for i := 0 to FCandidates.Count - 1
      do begin
        V     := FCandidates[ i];
        Value := V.P;
        Value := ( Value - MinValue) * Scaling;
        Value := Power( Value, Eta);
        Prev  := Prev + Value;
        V.P   := Prev;
        FCandidates[ i] := V;
      end;

      PotentialSum := Prev;

      // Pick a candidate

      V.X := -1;
      V.Y := -1;
      V.P := Random * PotentialSum;

      FCandidates.BinarySearch( V, i);

      if   ( i >= 0)
      and  ( i < FCandidates.Count)
      then AddPixel( i, FCandidates[ i].X, FCandidates[ i].Y) // Needed to pass i to AddPixel for it to work ...
      else Nop;
    end;


    procedure   TLightningEngine.AddRandomPoint;
    var
      X : Integer;
      Y : Integer;
    begin
      X := Random( Width );
      Y := Random( Height);
      FIsCandidate[ X, Y] := True;
      AddPixel( -1, X, Y);
    end;


    procedure   TLightningEngine.SetSize( aWidth, aHeight: Integer);
    begin
      if   ( aWidth  <> Width )
      or   ( aHeight <> Height)
      then begin
        FLocked := True;

        try
          FWidth  := aWidth;
          FHeight := aHeight;
          Reset;
        finally
          FLocked := False;
        end;
      end;
    end;


end.
