unit KnobsMaze;

{

   COPYRIGHT 2017 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.SysUtils, System.Types, System.Math, System.Classes,

  Vcl.Graphics,

  KnobsUtils, KnobsConversions, KnobsForth;


type

  TMazeErrorType =
  (
    mzError      ,
    mzWarning    ,
    mzInfo       ,
    mzInterpreter
  );

  TmazeHunterRule =
  (
    hrAllowDups,
    hrNoDups
  );


  EMazeError         = class( Exception);
  TMazeOnError       = procedure( const aSender: TObject; aType: TMazeErrorType; const aMsg: string)                       of object;
  TMazeOnHunterMoved = procedure( const aSender: TObject; anId, aLocation: Integer; aDistance, anAngle, anX, anY: TSignal) of object;
  TMazeVertex        = class;
  TMazeGraph         = class;
  TMazeMaker         = class;
  TMazeVertices      = array of TMazeVertex;


  TMazePoint = record
  public
    X : TSignal;
    Y : TSignal;
  public
    procedure   MakePoint( anX, anY: TSignal);
  end;


  TMazeRect = record
  public
    Left   : TSignal;
    Top    : TSignal;
    Right  : TSignal;
    Bottom : TSignal;
  public
    procedure   IncludePoint( anX, anY: TSignal);                                                              overload;
    procedure   IncludePoint( const aPoint: TMazePoint);                                                       overload;
    procedure   Clear;
    procedure   MakeRect( aLeft, aTop, aRight, aBottom: TSignal);
    function    Width  : TSignal;
    function    Height : TSignal;
    function    IsNull : Boolean;
  end;


  TMazeEdge = record
  private
    FVertex  : TMazeVertex;
    FLength  : TSignal;
    FAngle   : TSignal;
    FColor   : TColor;
    FPen     : Integer;
    FVisible : Boolean;
    FVisited : Boolean;
  public
    procedure   Init( const aVertex: TMazeVertex; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    procedure   Dump( const aStringList: TStringList; anIndent: Integer);
  public
    property    Vertex  : TMazeVertex read FVertex;
    property    Length  : TSignal     read FLength;
    property    Angle   : TSignal     read FAngle;
    property    Color   : TColor      read FColor;
    property    Pen     : Integer     read FPen;
    property    Visible : Boolean     read FVisible;
    property    Visited : Boolean     read FVisited write FVisited;
  end;

  TMazeEdges  = array of TMazeEdge;


  TMazeVertex = class
  private
    FGraph   : TMazeGraph;
    FIndex   : Integer;
    FX       : TSignal;
    FY       : TSignal;
    FDotSize : Integer;
    FEdges   : TMazeEdges;
    FVisited : Boolean;
  private
    function    GetCount: Integer;
    function    GetEdge( anIndex: Integer): TMazeEdge;
    procedure   SetEdge( anIndex: Integer; const aValue: TMazeEdge);
    procedure   FlagError( aType: TMazeErrorType; const aMsg: string);
    procedure   FlagErrorFmt( aType: TMazeErrorType; const aFmt: string; const anArgs: array of const);
  public
    constructor Create( const aMazeGraph: TMazeGraph; anIndex: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Dump( const aStringList: TStringList; anIndent: Integer);
    function    FindEdgeTo( const aVertex: TMazeVertex): Integer;
    procedure   AddEdgeTo( const aVertex: TMazeVertex; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    function    DistanceTo( anX, anY: TSignal): TSignal;
    function    IsNearTo( anX, anY: TSignal): Boolean;
    procedure   UnvisitAllEdges;
    procedure   VisitEdge( anIndex: Integer);
    procedure   Paint( const aCanvas : TCanvas; aScale: TSignal; anOffset: TMazePoint);
  public
    property    Count                    : Integer   read GetCount;
    property    Index                    : Integer   read FIndex;
    property    Visited                  : Boolean   read FVisited write FVisited;
    property    Edge[ anIndex : Integer] : TMazeEdge read GetEdge  write SetEdge;
    property    X                        : TSignal   read FX       write FX;
    property    Y                        : TSignal   read FY       write FY;
    property    DotSize                  : Integer   read FDotSize write FDotSize;
  end;


  TMazeHunter = class
  private
    FMazeGraph    : TMazeGraph;
    FId           : Integer;
    FPrevLocation : Integer;
    FLocation     : Integer;
  private
    procedure   SetLocation( anIndex: Integer);
  public
    constructor Create( const aMazeGraph: TMazeGraph; anId: Integer);
    procedure   Clear;
    procedure   ChaseSwan( anX, anY: TSignal; var aLocation: Integer; var aDistance, anAngle, anXOut, anYOut: TSignal; atRandom : Boolean);
  public
    property    Location: Integer read FLocation write SetLocation;
  end;

  TMazeHunters = array of TMazeHunter;


  TMazeGraph = class
  private
    FVertices      : TMazeVertices;
    FExtent        : TMazeRect;
    FSwanPosition  : TMazePoint;
    FHunters       : TMazeHunters;
    FAutoScale     : Boolean;
    FSwanSize      : Integer;
    FHunterSize    : Integer;
    FSwanColor     : TColor;
    FHunterColor   : TColor;
    FHunterRule    : TmazeHunterRule;
    FOnError       : TMazeOnError;
    FOnHunterMoved : TMazeOnHunterMoved;
  private
    function    GetCount: Integer;
    function    GetHunterCount: Integer;
    procedure   SetHunterCount( aValue: Integer);
    function    GetHunter( anIndex: Integer): TMazeHunter;
    function    GetHunterPosition( anIndex: Integer): TMazePoint;
  public
    constructor Create( aHunterCount: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   FlagError( aType: TMazeErrorType; const aMsg: string);
    procedure   HunterMoved( anId, aLocation: Integer; aDistance, anAngle, anX, anY: TSignal);
    procedure   Clear;
    procedure   Reset;
    procedure   Dump( const aStringList: TStringList; anIndent: Integer);
    function    AddVertex( const aVertex: TMazeVertex): Integer;
    function    FindVertexNearTo( anX, anY: TSignal): Integer;
    function    CreateVertex( anX, anY: TSignal; aDotSize: Integer): Integer;
    procedure   Connect( aSource, aDestination: Integer; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    function    IsHunterLocation( aSrc, aDst: Integer): Boolean;
    function    MoveTowards( aSource, aPrevious: Integer; anX, anY: TSignal; atRandom: Boolean): Integer;
    procedure   UnvisitAllEdges;
    procedure   MoveHunter ( anIndex: Integer; atRandom: Boolean);
    procedure   MoveHunters( atRandom: Boolean);
    procedure   PaintSwan   ( const aCanvas: TCanvas; aScale: TSignal; anOffset: TMazePoint);
    procedure   PaintHunters( const aCanvas: TCanvas; aScale: TSignal; anOffset: TMazePoint);
    procedure   Paint       ( const aCanvas: TCanvas; aScale: TSignal; anOffset: TMazePoint);
  public
    property    Count                             : Integer            read GetCount;
    property    Extent                            : TMazeRect          read FExtent;
    property    SwanPosition                      : TMazePoint         read FSwanPosition     write FSwanPosition;
    property    AutoScale                         : Boolean            read FAutoScale        write FAutoScale;
    property    SwanSize                          : Integer            read FSwanSize         write FSwanSize;
    property    HunterSize                        : Integer            read FHunterSize       write FHunterSize;
    property    SwanColor                         : TColor             read FSwanColor        write FSwanColor;
    property    HunterColor                       : TColor             read FHunterColor      write FHunterColor;
    property    HunterRule                        : TMazeHunterRule    read FHunterRule       write FHunterRule;
    property    HunterCount                       : Integer            read GetHunterCount    write SetHunterCount;
    property    Hunter        [ anIndex: Integer] : TMazeHunter        read GetHunter;
    property    HunterPosition[ anIndex: Integer] : TMazePoint         read GetHunterPosition;
    property    OnError                           : TMazeOnError       read FOnError          write FOnError;
    property    OnHunterMoved                     : TMazeOnHunterMoved read FOnHunterMoved    write FOnHunterMoved;
  end;


  TMazeStack<T> = class
  private
    FDrawer : TMazeMaker;
    FValues : array of T;
    FSize   : Integer;
    FSP     : Integer;
  private
    procedure   FlagError( aType: TMazeErrorType; const aMsg: string);
  public
    constructor Create( const aDrawer: TMazeMaker; aSize: Integer);
    procedure   Clear;
    procedure   Push( aValue: T);
    function    Pop : T;
    function    Peek: T;
    procedure   Drop;
    procedure   Dup;
    procedure   Swap;
  end;


  TMazeMakerState = record
  public
    Angle      : TSignal;
    X          : TSignal;
    Y          : TSignal;
    Scale      : TSignal;
    Origin     : Integer;
    Color      : TColor;
    Visible    : Boolean;
    PenSize    : Integer;
    DotSize    : Integer;
  end;


  TMazeMaker = class
  private
    FState      : TMazeMakerState;
    FMazeGraph  : TMazeGraph;
    FStateStack : TMazeStack<TMazeMakerState>;
  private
    function    GetAngle  : TSignal;
    procedure   SetAngle  ( aValue: TSignal);
    function    GetX      : TSignal;
    procedure   SetX      ( aValue: TSignal);
    function    GetY      : TSignal;
    procedure   SetY      ( aValue: TSignal);
    function    GetScale  : TSignal;
    procedure   SetScale  ( aValue: TSignal);
    function    GetOrigin : Integer;
    procedure   SetOrigin ( aValue: Integer);
    function    GetColor  : TColor;
    procedure   SetColor  ( aValue: TColor);
    function    GetVisible: Boolean;
    procedure   SetVisible( aValue: Boolean);
    function    GetPenSize: Integer;
    procedure   SetPenSize( aValue: Integer);
    function    GetDotSize: Integer;
    procedure   SetDotSize( aValue: Integer);
    function    GetExtent: TMazeRect;
    procedure   FlagError  ( aType: TMazeErrorType; const aMsg: string);
    procedure   FlagErrorFmt( aType: TMazeErrorType; const aFmt: string; const anArgs: array of const);
  public
    constructor Create( const aMazeGraph: TMazeGraph);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Dump( const aStringList: TStringList);
    function    CreateDump: TStringList;
  public
    procedure   RotateDegrees( anAngle  : TSignal);
    procedure   RotateRadians( anAngle  : TSignal);
    procedure   Inflate      ( anAmount : TSignal);
    procedure   Walk         ( aDistance: TSignal);
    procedure   Move         ( aDistance: TSignal);
    procedure   Pen          ( aSize    : Integer);
    procedure   Dot          ( aSize    : Integer);
    procedure   PenUp;
    procedure   PenDown;
    procedure   PushState;
    procedure   PopState;
    procedure   DupState;
    procedure   DropState;
    procedure   SwapState;
  public
    property    Angle       : TSignal   read GetAngle    write SetAngle;
    property    X           : TSignal   read GetX        write SetX;
    property    Y           : TSignal   read GetY        write SetY;
    property    Scale       : TSignal   read GetScale    write SetScale;
    property    Origin      : Integer   read GetOrigin   write SetOrigin;
    property    Color       : TColor    read GetColor    write SetColor;
    property    Visible     : Boolean   read GetVisible  write SetVisible;
    property    PenSize     : Integer   read GetPenSize  write SetPenSize;
    property    DotSize     : Integer   read GetDotSize  write SetDotSize;
    property    Extent      : TMazeRect read GetExtent;
  end;


  TMazeLSystemCommand = record
    FCommand   : Char;
    FExpansion : string;
  end;


  TMazeLSystemCommands = array of TMazeLSystemCommand;


  TMazeLSystem = class
  private
    FRules    : TMazeLSystemCommands;
    FMeanings : TMazeLSystemCommands;
  private
    function    GetRulesCount    : Integer;
    function    GetMeaningsCount : Integer;
    function    FindRuleFor   ( aCommand: char): string;
    function    FindMeaningFor( aCommand: char): string;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   AddRuleFor   ( aCommand: Char; const anExpansion: string);
    procedure   AddMeaningFor( aCommand: Char; const anExpansion: string);
    function    Expand( const aName, anAxiom: string; aDepth: Integer): string;
  public
    property    RulesCount    : Integer read GetRulesCount;
    property    MeaningsCount : Integer read GetMEaningsCount;
  end;


  TMazeForth = class( TMazeMaker)
  private
    FForth   : TForth;
    FLSystem : TMazeLSystem;
    FId      : Integer;
  private
    procedure   DoForthResponse( aSender: TObject; const aMsg: string);
    procedure   DoExternalFunc ( aSender: TObject; anId: Integer; const aStack: TStack);
    procedure   LClear;
    procedure   LRule   ( const aCommand, anExpansion: string);
    procedure   LMeaning( const aCommand, anExpansion: string);
    procedure   LAxiom  ( const aName, anAxiom: string; aDepth: Integer);
    procedure   PushExtent( aStack: TStack);
  public
    constructor Create( const aMazeGraph: TMazeGraph; anId: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Initialize;
    procedure   AcceptText( const S: string);
    procedure   LoadFile( const aFileName: string);
    function    CreateExportList: TStringList;
  end;


  function  MazeErrorTypeToStr( aType: TMazeErrorType): string;
  function  MazePoint( anX, anY: TSignal): TMazePoint;
  function  MazeRect( aLeft, aTop, aRight, aBottom: TSignal): TMazeRect;


implementation


  function MazeErrorTypeToStr( aType: TMazeErrorType): string;
  begin
    case aType of
      mzError       : Result := 'Error';
      mzWarning     : Result := 'Warning';
      mzInfo        : Result := 'Info';
      mzInterpreter : Result := 'Interpreter';
    end;
  end;


  function MazePoint( anX, anY: TSignal): TMazePoint;
  begin
    Result.MakePoint( anX, anY);
  end;


  function MazeRect( aLeft, aTop, aRight, aBottom: TSignal): TMazeRect;
  begin
    Result.MakeRect( aLeft, aTop, aRight, aBottom);
  end;



  function ScalePoint( anX, anY, aScale: TSignal; anOffset: TMazePoint): TPoint;
  begin
    Result.X := Round( anX * aScale + anOffset.X);
    Result.Y := Round( anY * aScale + anOffset.Y);
  end;


{ ========
  TMazePoint = record
  public
    X : TSignal;
    Y : TSignal;
  public
}

    procedure   TMazePoint.MakePoint( anX, anY: TSignal);
    begin
      X := anX;
      Y := anY;
    end;


{ ========
  TMazeExtent = record
  public
    FLeft   : TSignal;
    FTop    : TSignal;
    FRight  : TSignal;
    FBottom : TSignal;
  public
}

    procedure   TMazeRect.IncludePoint( anX, anY: TSignal); // overload;
    begin
      if anX < Left
      then Left := anX
      else if anX > Right
      then Right := anX;

      if anY < Top
      then Top := anY
      else if anY > Bottom
      then Bottom := anY;
    end;


    procedure   TMazeRect.IncludePoint( const aPoint: TMazePoint); // overload;
    begin
      IncludePoint( aPoint.X, aPoint.Y);
    end;


    procedure   TMazeRect.Clear;
    begin
      Left   := 0.0;
      Top    := 0.0;
      Right  := 0.0;
      Bottom := 0.0;
    end;


    procedure   TMazeRect.MakeRect( aLeft, aTop, aRight, aBottom: TSignal);
    begin
      Left   := aLeft;
      Top    := aTop;
      Right  := aRight;
      Bottom := aBottom;
    end;


    function    TMazeRect.Width: TSignal;
    begin
      Result := Right - Left;
    end;


    function    TMazeRect.Height: TSignal;
    begin
      Result := Bottom - Top;
    end;


    function    TMazeRect.IsNull : Boolean;
    begin
      Result := ( Width = 0) or ( Height = 0);
    end;



{ ========
  TMazeEdge = record
  private
    FVertex  : TMazeVertex;
    FLength  : TSignal;
    FAngle   : TSignal;
    FColor   : TColor;
    FPen     : Integer;
    FVisible : Boolean;
    FVisited : Boolean;
  public
    property    Vertex  : TMazeVertex read FVertex;
    property    Length  : TSignal     read FLength;
    property    Angle   : TSignal     read FAngle;
    property    Color   : TColor      read FColor;
    property    Pen     : Integer     read FPen;
    property    Visible : Boolean     read FVisible;
    property    Visited : Boolean     read FVisited write FVisited;
  public
}

    procedure   TMazeEdge.Init( const aVertex: TMazeVertex; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    begin
      FVertex  := aVertex;
      FLength  := aLength;
      FAngle   := anAngle;
      FColor   := aColor;
      FPen     := aPen;
      FVisible := aVisible;
    end;


    procedure   TMazeEdge.Dump( const aStringList: TStringList; anIndent: Integer);
    begin
      if Assigned( aStringList)
      then begin
        aStringList.Add(      Format( '%sEdge('             , [ Indent( anIndent    )                           ], AppLocale));
        aStringList.Add(      Format(   '%sLength   : %g'   , [ Indent( anIndent + 1), Length                   ], AppLocale));
        aStringList.Add(      Format(   '%sAngle    : %g'   , [ Indent( anIndent + 1), Angle                    ], AppLocale));
        aStringList.Add(      Format(   '%sColor    : %.8x' , [ Indent( anIndent + 1), Color                    ], AppLocale));
        aStringList.Add(      Format(   '%sVisible  : %s'   , [ Indent( anIndent + 1), BoolToStr( Visible, True)], AppLocale));
        aStringList.Add(      Format(   '%sVisited  : %s'   , [ Indent( anIndent + 1), BoolToStr( Visited, True)], AppLocale));

        if Assigned( Vertex)
        then aStringList.Add( Format(   '%sVertex   : %d'   , [ Indent( anIndent + 1), Vertex.Index             ], AppLocale))
        else aStringList.Add( Format(   '%sVertex   : <nil>', [ Indent( anIndent + 1)                           ], AppLocale));

        aStringList.Add(      Format( '%s)'                 , [ Indent( anIndent    )                           ], AppLocale));
      end;
    end;


{ ========
  TMazeVertex = class
  private
    FGraph   : TMazeGraph;
    FIndex   : Integer;
    FX       : TSignal;
    FY       : TSignal;
    FDotSize : Integer;
    FEdges   : TMazeEdges;
    FVisited : Boolean;
  public
    property    Count                    : Integer   read GetCount;
    property    Index                    : Integer   read FIndex;
    property    Visited                  : Boolean   read FVisited write FVisited;
    property    Edge[ anIndex : Integer] : TMazeEdge read GetEdge  write SetEdge;
    property    X                        : TSignal   read FX       write FX;
    property    Y                        : TSignal   read FY       write FY;
    property    DotSize                  : Integer   read FDotSize write FDotSize;
  private
}

    function    TMazeVertex.GetCount: Integer;
    begin
      Result := Length( FEdges);
    end;


    function    TMazeVertex.GetEdge( anIndex: Integer): TMazeEdge;
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then Result := FEdges[ anIndex]
      else begin
        Result.FVertex := nil;
        FlagErrorFmt( mzError, 'Vertex %d: Edge index %d out of bounds [0..%d>', [ Index, anIndex, Count])
      end;
    end;


    procedure   TMazeVertex.SetEdge( anIndex: Integer; const aValue: TMazeEdge);
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then FEdges[ anIndex] := aValue
      else FlagErrorFmt( mzError, 'Vertex %d: Edge index %d out of bounds [0..%d>', [ Index, anIndex, Count])
    end;


    procedure   TMazeVertex.FlagError( aType: TMazeErrorType; const aMsg: string);
    begin
      FGraph.FlagError( aType, aMsg);
    end;


    procedure   TMazeVertex.FlagErrorFmt( aType: TMazeErrorType; const aFmt: string; const anArgs: array of const);
    begin
      FlagError( aType, Format( aFmt, anArgs, AppLocale));
    end;


//  public

    constructor TMazeVertex.Create( const aMazeGraph: TMazeGraph; anIndex: Integer);
    begin
      Assert( anIndex >= 0);
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FGraph := aMazeGraph;
      FIndex := anIndex;
    end;


    destructor  TMazeVertex.Destroy;  // override;
    begin
      SetLength( FEdges, 0);
    end;


    procedure   TMazeVertex.Dump( const aStringList: TStringList; anIndent: Integer);
    var
      i : Integer;
    begin
      if Assigned( aStringList)
      then begin
        aStringList.Add( Format( '%sVertex('     , [ Indent( anIndent    )       ], AppLocale));
        aStringList.Add( Format(   '%sIndex : %d', [ Indent( anIndent + 1), Index], AppLocale));

        for i := 0 to Count - 1
        do FEdges[ i].Dump( aStringlist, anIndent + 1);

        aStringList.Add( Format( '%s)'            , [ Indent( anIndent    )      ], AppLocale));
      end;
    end;


    function    TMazeVertex.FindEdgeTo( const aVertex: TMazeVertex): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to Count - 1
      do begin
        if FEdges[ i].Vertex = aVertex
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TMazeVertex.AddEdgeTo( const aVertex: TMazeVertex; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    var
      aEdge : TMazeEdge;
    begin
      if   Assigned( aVertex) and not Visited
      then begin
        Visited := True;

        try
          if FindEdgeTo( aVertex) < 0
          then begin
            aEdge.Init( aVertex, aLength, anAngle, aColor, aPen, aVisible);
            SetLength( FEdges, Count + 1);
            FEdges[ Count - 1] := aEdge;
          end;

          aVertex.AddEdgeTo( Self, aLength, MathFloatMod( anAngle + Pi, TWO_PI), aColor, aPen, aVisible);
        finally
          Visited := False;
        end;
      end;
    end;


    function    TMazeVertex.DistanceTo( anX, anY: TSignal): TSignal;
    var
      dx : TSignal;
      dy : TSignal;
    begin
      dx     := Normalize( anX - X);
      dy     := Normalize( anY - Y);
      Result := Sqrt( dx * dx + dy * dy);
    end;


    function    TMazeVertex.IsNearTo( anX, anY: TSignal): Boolean;
    begin
      Result := DistanceTo( anX, anY) < 1e-6;
    end;


    procedure   TMazeVertex.UnvisitAllEdges;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FEdges[ i].Visited := False;
    end;


    procedure   TMazeVertex.VisitEdge( anIndex: Integer);
    var
      Dst   : TMazeVertex;
      Index : Integer;
    begin
      if ( anIndex >= 0) and ( anIndex < Count)
      then begin
        if not FEdges[ anIndex].Visited
        then begin
          FEdges[ anIndex].Visited := True;
          Dst := FEdges[ anIndex].Vertex;

          if Assigned( Dst)
          then begin
             Index := Dst.FindEdgeTo( Self);

             if Index >= 0
             then Dst.VisitEdge( Index)
             else FlagErrorFmt( mzWarning, 'Vertex %d: Edge with index %d to Vertex %d has no back Edge (to %d)', [ Self.Index, anIndex, Index, Self.Index]);
          end
          else FlagErrorFmt( mzWarning, 'Vertex %d: Edge with index %d has a NIL Destination Vertex', [ Self.Index, anIndex]);
        end;
      end
      else FlagErrorFmt( mzError, 'Vertex %d: Invalid Edge index %d', [ Self.Index, anIndex]);
    end;


    procedure   TMazeVertex.Paint( const aCanvas : TCanvas; aScale: TSignal; anOffset: TMazePoint);
    var
      Origin      : TPoint;
      Destination : TPoint;
      Dst         : TMazeVertex;
      i           : Integer;
    begin
      Origin := ScalePoint( X, Y, aScale, anOffset);

      for i := 0 to Count - 1
      do begin
        if not Edge[ i].Visited
        then begin
          Dst := Edge[ i].Vertex;

          if Assigned( Dst)
          then begin
            Destination := ScalePoint( Dst.X, Dst.Y, aScale, anOffset);

            if Edge[ i].Visible
            then begin
              aCanvas.Pen.Color := Edge[ i].Color;
              aCanvas.Pen.Width := Edge[ i].Pen;
              aCanvas.MoveTo( Origin.X, Origin.Y);

              if DotSize > 0
              then aCanvas.Ellipse( Origin.X - DotSize, Origin.Y - DotSize, Origin.X + DotSize, Origin.Y + DotSize);

              aCanvas.LineTo( Destination.X, Destination.Y);

              if DotSize > 0
              then aCanvas.Ellipse( Destination.X - DotSize, Destination.Y - DotSize, Destination.X + DotSize, Destination.Y + DotSize);
            end
            else aCanvas.MoveTo( Destination.X, Destination.Y);

            VisitEdge( i);
          end
          else FlagErrorFmt( mzWarning, 'Vertex %d: Edge with index %d has no valid Vertex', [ Index, i]);
        end;
      end;
    end;


{ ========
  TMazeHunter = class
  private
    FMazeGraph    : TMazeGraph;
    FId           : Integer;
    FPrevLocation : Integer;
    FLocation     : Integer;
  public
    property    Location: Integer read FLocation write SetLocation;
  private
}

    procedure   TMazeHunter.SetLocation( anIndex: Integer);
    begin
      FPrevLocation := FLocation;
      FLocation     := anIndex;
    end;


//  public

    constructor TMazeHunter.Create( const aMazeGraph: TMazeGraph; anId: Integer);
    begin
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FMazeGraph := aMazeGraph;
      FId        := anId;
      Clear;
    end;


    procedure   TMazeHunter.Clear;
    begin
      FPrevLocation := -1;
      FLocation     := -1;
    end;


    procedure   TMazeHunter.ChaseSwan( anX, anY: TSignal; var aLocation: Integer; var aDistance, anAngle, anXOut, anYOut: TSignal; atRandom : Boolean);
    var
      Src          : TMazeVertex;
      NewEdgeIndex : Integer;
      Edge         : TMazeEdge;
    begin
      aDistance := -1;
      anAngle   := -1;
      aLocation := -1;
      anXOut    := -1;
      anYOut    := -1;

      if Assigned( FMazeGraph)
      then begin
        if   FLocation < 0                             // when we are nowhere ...
        then FLocation := Random( FMazeGraph.Count);   // jump into the maze at a random location

        // So now  we are somewhere ...

        NewEdgeIndex := FMazeGraph.MoveTowards( FLocation, FPrevLocation, anX, anY, atRandom);

        if   NewEdgeIndex >= 0
        then begin
          Src := FMazeGraph.FVertices[ FLocation];

          if Assigned( Src)
          then begin
            Edge := Src.Edge[ NewEdgeIndex];

            if Assigned( Edge.Vertex)
            then begin
              Location  := Edge.Vertex.Index;
              aLocation := Location;
              aDistance := Edge.Length;
              anAngle   := Edge.Angle;
              anXOut    := Edge.Vertex.FX;
              anYOut    := Edge.Vertex.FY;
            end;
          end;
        end
        else Clear;
      end;
    end;


{ ========
  TMazeGraph = class
  private
    FVertices      : TMazeVertices;
    FExtent        : TMazeRect;
    FSwanPosition  : TMazePoint;
    FHunters       : TMazeHunters;
    FAutoScale     : Boolean;
    FSwanSize      : Integer;
    FHunterSize    : Integer;
    FSwanColor     : TColor;
    FHunterColor   : TColor;
    FHunterRule    : TmazeHunterRule;
    FOnError       : TMazeOnError;
    FOnHunterMoved : TMazeOnHunterMoved;
  public
    property    Count                             : Integer            read GetCount;
    property    Extent                            : TMazeRect          read FExtent;
    property    SwanPosition                      : TMazePoint         read FSwanPosition     write FSwanPosition;
    property    AutoScale                         : Boolean            read FAutoScale        write FAutoScale;
    property    SwanSize                          : Integer            read FSwanSize         write FSwanSize;
    property    HunterSize                        : Integer            read FHunterSize       write FHunterSize;
    property    SwanColor                         : TColor             read FSwanColor        write FSwanColor;
    property    HunterColor                       : TColor             read FHunterColor      write FHunterColor;
    property    HunterRule                        : TMazeHunterRule    read FHunterRule       write FHunterRule;
    property    HunterCount                       : Integer            read GetHunterCount    write SetHunterCount;
    property    Hunter        [ anIndex: Integer] : TMazeHunter        read GetHunter;
    property    HunterPosition[ anIndex: Integer] : TMazePoint         read GetHunterPosition;
    property    OnError                           : TMazeOnError       read FOnError          write FOnError;
    property    OnHunterMoved                     : TMazeOnHunterMoved read FOnHunterMoved    write FOnHunterMoved;
  private
}

    function    TMazeGraph.GetCount: Integer;
    begin
      Result := Length( FVertices);
    end;


    function    TMazeGraph.GetHunterCount: Integer;
    begin
      Result := Length( FHunters);
    end;


    procedure   TMazeGraph.SetHunterCount( aValue: Integer);
    var
      i : Integer;
    begin
      for i := 0 to HunterCount - 1
      do FreeAndNil( FHunters[ i]);

      SetLength( FHunters, aValue);

      for i := 0 to HunterCount - 1
      do FHunters[ i] := TMazeHunter.Create( Self, i);
    end;


    function    TMazeGraph.GetHunter( anIndex: Integer): TMazeHunter;
    begin
      if ( anIndex >= 0) or ( anIndex < HunterCount)
      then Result := FHunters[ anIndex]
      else begin
        Result := nil;
        FlagError( mzError, 'Hunter index out of bounds');
      end;
    end;


    function    TMazeGraph.GetHunterPosition( anIndex: Integer): TMazePoint;
    begin
      if ( anIndex >= 0) or ( anIndex < HunterCount)
      then begin
      end
      else begin
        Result.X := 0;
        Result.Y := 0;
      end;
    end;


//  public

    constructor TMazeGraph.Create( aHunterCount: Integer);
    begin
      FExtent.Clear;
      FSwanSize    := 4;
      FHunterSize  := 3;
      FSwanColor   := clBlack;
      FHunterColor := clBlue;
      FHunterRule  := hrNoDups;
      HunterCount  := aHunterCount;
    end;


    destructor  TMazeGraph.Destroy; // override;
    var
      i : Integer;
    begin
      Clear;

      for i := 0 to HunterCount - 1
      do FreeAndNil( FHunters[ i]);

      SetLength( FHunters, 0);
      inherited;
    end;


    procedure   TMazeGraph.FlagError( aType: TMazeErrorType; const aMsg: string);
    begin
      if Assigned( FOnError)
      then FOnError( Self, aType, aMsg)
      else begin
        if aType in [ mzError, mzInterpreter]
        then raise EMazeError.Create( aMsg);
      end
    end;


    procedure   TMazeGraph.HunterMoved( anId, aLocation: Integer; aDistance, anAngle, anX, anY: TSignal);
    begin
      if Assigned( FOnHunterMoved)
      then FOnHunterMoved( Self, anId, aLocation, aDistance, anAngle, anX, anY);
    end;


    procedure   TMazeGraph.Clear;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FreeAndNil( FVertices[ i]);

      SetLength( FVertices, 0);
      Reset;
    end;


    procedure   TMazeGraph.Reset;
    var
      i : Integer;
    begin
      for i := 0 to HunterCount - 1
      do Hunter[ i].Clear;
    end;


    procedure   TMazeGraph.Dump( const aStringList: TStringList; anIndent: Integer);
    var
      i : Integer;
    begin
      if   Assigned( aStringList)
      then begin
        aStringList.Clear;
        aStringList.Add( Format( '%sMazeGraph('                  , [ Indent( anIndent)], AppLocale));
        aStringList.Add( Format(   '%sExtent : ( %g, %g, %g, %g)', [ Indent( anIndent + 1), Extent.Left, Extent.Top, Extent.Right, Extent.Bottom], AppLocale));

        for i := 0 to Count - 1
        do FVertices[ i].Dump( aStringList, anIndent + 1);

        aStringList.Add( Format( '%s)', [ Indent( anIndent)], AppLocale));
      end;
    end;


    function    TMazeGraph.AddVertex( const aVertex: TMazeVertex): Integer;
    begin
      Result := -1;

      if   Assigned( aVertex)
      then begin
        SetLength( FVertices, Count + 1);
        FVertices[ Count - 1] := aVertex;
        Result := Count - 1;

        If Result = 0
        then FExtent.Clear
        else FExtent.IncludePoint( aVertex.FX, aVertex.FY);
      end;
    end;


    function    TMazeGraph.FindVertexNearTo( anX, anY: TSignal): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to Count - 1
      do begin
        if FVertices[ i].IsNearTo( anX, anY)
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    function    TMazeGraph.CreateVertex( anX, anY: TSignal; aDotSize: Integer): Integer;
    var
      aVertex : TMazeVertex;
    begin
      Result := FindVertexNearTo( anX, anY);

      if Result < 0
      then begin
        aVertex         := TMazeVertex.Create( Self, Count);
        aVertex.X       := anX;
        aVertex.Y       := anY;
        aVertex.DotSize := aDotSize;
        Result          := AddVertex( aVertex);
      end;
    end;


    procedure   TMazeGraph.Connect( aSource, aDestination: Integer; aLength, anAngle: TSignal; aColor: TColor; aPen: Integer; aVisible: Boolean);
    var
      Src : TMazeVertex;
      Dst : TMazeVertex;
    begin
      Src := FVertices[ aSource     ];
      Dst := FVertices[ aDestination];

      if   Assigned( Src)
      and  Assigned( Dst)
      then Src.AddEdgeTo( Dst, aLength, anAngle, aColor, aPen, aVisible);
    end;


    function    TMazeGraph.IsHunterLocation( aSrc, aDst: Integer): Boolean;
    var
      i : Integer;
    begin
      Result := False;

      if HunterRule = hrNoDups
      then begin
        for i := 0 to HunterCount - 1
        do begin
          if i <> aSrc
          then begin
            if aDst = Hunter[ i].FLocation
            then begin
              Result := True;
              Break;
            end;
          end;
        end;
      end;
    end;


    function    TMazeGraph.MoveTowards( aSource, aPrevious: Integer; anX, anY: TSignal; atRandom: Boolean): Integer;
    // Find a way in the maze that gets one step closer to a target X, Y position
    // The result is the EdgeIndex of the Edge that moves us closer
    // aPrevious is not considered a valid path ... so thats skipped inthe search
    var
      Src         : TMazeVertex;
      Dst         : TMazeVertex;
      i           : Integer;
      p           : Integer;
      D           : TSignal;
      MinD        : TSignal;
      HasDistance : Boolean;
    begin
      Result      := -1;
      Src         := FVertices[ aSource];
      HasDistance := False;
      MinD        := 1e6;

      if   Assigned( Src)
      then begin
        if atRandom
        then p := Random( Src.Count)
        else p := 0;

        for i := 0 to Src.Count - 1
        do begin
          Dst := Src.Edge[ p].FVertex;

          if Assigned( Dst)
          then begin
            if   ( Dst.Index <> aPrevious)
            and  not IsHunterLocation( aSource, Dst.Index)
            then begin

              D := Dst.DistanceTo( anX, anY);

              if not HasDistance
              then begin
                MinD        := D;
                HasDistance := True;
                Result      := p;
              end
              else begin
                if D < MinD
                then begin
                  MinD   := D;
                  Result := p;
                end;
              end;
            end;
          end;

          p := ( p + 1) mod Src.Count;
        end;
      end;
    end;


    procedure   TMazeGraph.UnvisitAllEdges;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do FVertices[ i].UnvisitAllEdges;
    end;


    procedure   TMazeGraph.MoveHunter( anIndex: Integer; atRandom: Boolean);
    var
      Location : Integer;
      Distance : TSignal;
      Angle    : TSignal;
      X        : TSignal;
      Y        : TSignal;
    begin
      if ( anIndex >= 0) and ( anIndex < HunterCount)
      then begin
        Hunter[ anIndex].ChaseSwan( SwanPosition.X, SwanPosition.Y, Location, Distance, Angle, X, Y, atRandom);

        if ( Angle >= 0) and ( Distance >= 0)
        then HunterMoved( anIndex, Location, Distance, Angle, X, Y);
      end;
    end;


    procedure   TMazeGraph.MoveHunters( atRandom: Boolean);
    var
      i : Integer;
    begin
      for i := 0 to HunterCount - 1
      do MoveHunter( i, atRandom);
    end;


    procedure   TMazeGraph.PaintSwan( const aCanvas: TCanvas; aScale: TSignal; anOffset: TMazePoint);
    var
      P : TPoint;
    begin
      P := ScalePoint( SwanPosition.X, SwanPosition.Y, aScale, anOffset);
      aCanvas.Brush.Style := bsSolid;
      aCanvas.Brush.Color := SwanColor;
      aCanvas.Pen.Width   := 1;
      aCanvas.Pen.Color   := SwanColor;
      aCanvas.Ellipse( P.X - SwanSize, P.Y - SwanSize, P.X + SwanSize, P.Y + SwanSize);
    end;


    procedure   TMazeGraph.PaintHunters( const aCanvas: TCanvas; aScale: TSignal; anOffset: TMazePoint);
    var
      i         : Integer;
      P         : TPoint;
      aLocation : Integer;
    begin
      aCanvas.Brush.Style := bsSolid;
      aCanvas.Brush.Color := HunterColor;
      aCanvas.Pen.Width   := 1;
      aCanvas.Pen.Color   := HunterColor;

      for i := 0 to HunterCount - 1
      do begin
        aLocation := Hunter[ i].FLocation;

        if ( aLocation >= 0) and ( aLocation < Count)
        then begin
          P := ScalePoint( FVertices[ aLocation].FX, FVertices[ aLocation].FY, aScale, anOffset);
          aCanvas.Ellipse( P.X - HunterSize, P.Y - HunterSize, P.X + HunterSize, P.Y + HunterSize);
        end;
      end;
    end;


    procedure   TMazeGraph.Paint( const aCanvas : TCanvas; aScale: TSignal; anOffset: TMazePoint);
    var
      i : Integer;
    begin
      UnvisitAllEdges;
      aCanvas.Brush.Style := bsClear;

      for i := 0 to Count - 1
      do FVertices[ i].Paint( aCanvas, aScale, anOffset);

      PaintHunters( aCanvas, aScale, anOffset);
      PaintSwan   ( aCanvas, aScale, anOffset);
    end;


{ ========
  TMazeStack = class
  private
    FDrawer : TMazeMaker;
    FValues : TSignalArray;
    FSize   : Integer;
    FSP     : Integer;
  private
}

    procedure   TMazeStack<T>.FlagError( aType: TMazeErrorType; const aMsg: string);
    begin
      FDrawer.FlagError( aType, aMsg);
    end;


//  public

    constructor TMazeStack<T>.Create( const aDrawer: TMazeMaker; aSize: Integer);
    begin
      Assert( aSize > 0);
      Assert( Assigned( aDrawer));
      inherited Create;
      FDrawer := aDrawer;
      FSize   := aSize;
      SetLength( FValues, FSize);
      Clear;
    end;


    procedure   TMazeStack<T>.Clear;
    begin
      FSP := -1;
    end;


    procedure   TMazeStack<T>.Push( aValue: T);
    begin
      if FSP < FSize
      then begin
        Inc( FSP);
        FValues[ FSP] := aValue;
      end
      else FlagError( mzError, 'Stack overflow');
    end;


    function    TMazeStack<T>.Pop: T;
    begin
      Result := Default( T);

      if FSP >= 0
      then begin
        Result := FValues[ FSP];
        Dec( FSP);
      end
      else FlagError( mzError, 'Stack underflow');
    end;


    function    TMazeStack<T>.Peek: T;
    begin
      Result := Default( T);

      if FSP >= 0
      then Result := FValues[ FSP]
      else FlagError( mzError, 'Stack underflow');
    end;


    procedure   TMazeStack<T>.Drop;
    begin
      if FSP >= 0
      then Dec( FSP)
      else FlagError( mzError, 'Stack underflow');
    end;


    procedure   TMazeStack<T>.Dup;
    begin
      Push( Peek);
    end;


    procedure   TMazeStack<T>.Swap;
    var
      Tmp : T;
    begin
      if FSP >= 1
      then begin
        Tmp               := Peek;
        FValues[ FSP]     := FValues[ FSP - 1];
        FValues[ FSP - 1] := Tmp;
      end
      else FlagError( mzError, 'Stack underflow');
    end;


{ ========
  TMazeMaker = class
  private
    FState      : TMazeMakerState;
    FMazeGraph  : TMazeGraph;
    FValueStack : TMazeStack<TSignal>;
    FStateStack : TMazeStack<TMazeMakerState>;
  public
    property    Angle       : TSignal read GetAngle    write SetAngle;
    property    X           : TSignal read GetX        write SetX;
    property    Y           : TSignal read GetY        write SetY;
    property    Scale       : TSignal read GetScale    write SetScale;
    property    Origin      : Integer read GetOrigin   write SetOrigin;
    property    Color       : TColor  read GetColor    write SetColor;
    property    Visible     : Boolean read GetVisible  write SetVisible;
    property    PenSize     : Integer read GetPenSize  write SetPenSize;
    property    DotSize     : Integer read GetDotSize  write SetDotSize;
    property    HunterCount : Integer read GetHunterCount;
  private
}

    function    TMazeMaker.GetAngle: TSignal;
    begin
      Result := FState.Angle;
    end;


    procedure   TMazeMaker.SetAngle( aValue: TSignal);
    begin
      FState.Angle := aValue;
    end;


    function    TMazeMaker.GetX: TSignal;
    begin
      Result := FState.X;
    end;


    procedure   TMazeMaker.SetX( aValue: TSignal);
    begin
      FState.X := aValue;
    end;


    function    TMazeMaker.GetY: TSignal;
    begin
      Result := FState.Y;
    end;


    procedure   TMazeMaker.SetY( aValue: TSignal);
    begin
      FState.Y := aValue;
    end;


    function    TMazeMaker.GetScale: TSignal;
    begin
      Result := FState.Scale;
    end;


    procedure   TMazeMaker.SetScale( aValue: TSignal);
    begin
      FState.Scale := aValue;
    end;


    function    TMazeMaker.GetOrigin: Integer;
    begin
      Result := FState.Origin;
    end;


    procedure   TMazeMaker.SetOrigin( aValue: Integer);
    begin
      FState.Origin := aValue;
    end;


    function    TMazeMaker.GetColor: TColor;
    begin
      Result := FState.Color;
    end;


    procedure   TMazeMaker.SetColor( aValue: TColor);
    begin
      FState.Color := aValue;
    end;


    function    TMazeMaker.GetVisible: Boolean;
    begin
      Result := FState.Visible;
    end;


    procedure   TMazeMaker.SetVisible( aValue: Boolean);
    begin
      FState.Visible := aValue;
    end;


    function    TMazeMaker.GetPenSize: Integer;
    begin
      Result := FState.PenSize;
    end;


    procedure   TMazeMaker.SetPenSize( aValue: Integer);
    begin
      FState.PenSize := aValue;
    end;


    function    TMazeMaker.GetDotSize: Integer;
    begin
      Result := FState.DotSize;
    end;


    procedure   TMazeMaker.SetDotSize( aValue: Integer);
    begin
      FState.DotSize := aValue;
    end;


    function    TMazeMaker.GetExtent: TMazeRect;
    begin
      Result.Left   := 0;
      Result.Top    := 0;
      Result.Right  := 0;
      Result.Bottom := 0;

      if Assigned( FMazeGraph)
      then Result := FMazeGraph.Extent;
    end;


    procedure   TMazeMaker.FlagError( aType: TMazeErrorType; const aMsg: string);
    begin
      FMazeGraph.FlagError( aType, aMsg);
    end;


    procedure   TMazeMaker.FlagErrorFmt( aType: TMazeErrorType; const aFmt: string; const anArgs: array of const);
    begin
      FlagError( aType, Format( aFmt, anArgs));
    end;


//  public

    constructor TMazeMaker.Create( const aMazeGraph: TMazeGraph);
    begin
      Assert( Assigned( aMazeGraph));
      inherited Create;
      FStateStack := TMazeStack<TMazeMakerState>.Create( Self, 1024);
      FMazeGraph  := aMazeGraph;
      Clear;
    end;


    destructor  TMazeMaker.Destroy; // override;
    begin
      FreeAndNil( FStateStack);
      inherited;
    end;


    procedure   TMazeMaker.Clear;
    begin
      Angle   := 0.0;
      X       := 0.0;
      Y       := 0.0;
      Scale   := 1.0;
      PenSize := 1;
      DotSize := 0;
      Color   := clBlack;
      Visible := True;
      FMazeGraph .Clear;
      Origin := FMazeGraph.CreateVertex( X, Y, DotSize);
    end;


    procedure   TMazeMaker.Dump( const aStringList: TStringList);
    begin
      if   Assigned( aStringList)
      then FMazeGraph.Dump( aStringList, 0);
    end;


    function    TMazeMaker.CreateDump: TStringList;
    begin
      Result := TStringList.Create;
      Dump( Result);
    end;


//  public

    procedure   TMazeMaker.RotateDegrees( anAngle: TSignal);
    begin
      RotateRadians( DegToRad( anAngle));
    end;


    procedure   TMazeMaker.RotateRadians( anAngle: TSignal);
    begin
      Angle := MathFloatMod( Angle + anAngle, TWO_PI);
    end;


    procedure   TMazeMaker.Inflate( anAmount : TSignal);
    begin
      Scale := Scale * anAmount;
    end;


    procedure   TMazeMaker.Walk( aDistance: TSignal);
    var
      aNewX     : TSignal;
      aNewY     : TSignal;
      aNewIndex : Integer;
      aStride   : TSignal;
    begin
      aStride   := Scale  * aDistance;
      aNewX     := X + aStride * Cos( Angle);
      aNewY     := Y + aStride * Sin( Angle);
      aNewIndex := FMazeGraph.CreateVertex( aNewX, aNewY, DotSize);
      FMazeGraph.Connect( Origin, aNewIndex, Abs( aStride), Angle, Color, PenSize, Visible);
      Origin := aNewIndex;
      X      := aNewX;
      Y      := aNewY;
    end;


    procedure   TMazeMaker.Move( aDistance: TSignal);
    var
      aStride : TSignal;
    begin
      aStride := Scale  * aDistance;
      X       := X + aStride * Cos( Angle);
      Y       := Y + aStride * Sin( Angle);
      Origin  := FMazeGraph.CreateVertex( X, Y, DotSize);
    end;


    procedure   TMazeMaker.Pen( aSize: Integer);
    begin
      PenSize := aSize;
    end;


    procedure   TMazeMaker.Dot( aSize: Integer);
    begin
      DotSize := aSize;
    end;


    procedure   TMazeMaker.PenUp;
    begin
      Visible := False;
    end;


    procedure   TMazeMaker.PenDown;
    begin
      Visible := True;
    end;


    procedure   TMazeMaker.PushState;
    begin
      FStateStack.Push( FState);
    end;


    procedure   TMazeMaker.PopState;
    begin
      FState := FStateStack.Pop;
    end;


    procedure   TMazeMaker.DupState;
    begin
      FStateStack.Dup;
    end;


    procedure   TMazeMaker.DropState;
    begin
      FStateStack.Drop;
    end;


    procedure   TMazeMaker.SwapState;
    begin
      FStateStack.Swap;
    end;


{ ========
  TMazeLSystemMeaning = record
    FCommand : Char;
    FAction  : string;
  end;


  TMazeLSystemMeanings = array of TMazeLSystemMeaning;


  TMazeLSystem = class
  private
    FRules    : TMazeLSystemCommands;
    FMeanings : TMazeLSystemCommands;
  public
    property    RulesCount    : Integer read GetRulesCount;
    property    MeaningsCount : Integer read GetMEaningsCount;
  private
}

    function    TMazeLSystem.GetRulesCount: Integer;
    begin
      Result := Length( FRules);
    end;


    function    TMazeLSystem.GetMeaningsCount: Integer;
    begin
      Result := Length( FMeanings);
    end;


    function    TMazeLSystem.FindRuleFor( aCommand: char): string;
    var
      i : Integer;
    begin
      Result := aCommand;

      for i := 0 to RulesCount - 1
      do begin
        if FRules[ i].FCommand = aCommand
        then begin
          Result := FRules[ i].FExpansion;
          Break;
        end;
      end;
    end;


    function    TMazeLSystem.FindMeaningFor( aCommand: char): string;
    var
      i : Integer;
    begin
      Result := '';

      for i := 0 to MeaningsCount - 1
      do begin
        if FMeanings[ i].FCommand = aCommand
        then begin
          Result := FMeanings[ i].FExpansion;
          Break;
        end;
      end;
    end;


//  public

    constructor TMazeLSystem.Create;
    begin
      inherited;
      Clear;
    end;


    destructor  TMazeLSystem.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TMazeLSystem.Clear;
    begin
      SetLength( FRules   , 0);
      SetLength( FMeanings, 0);
    end;


    procedure   TMazeLSystem.AddRuleFor( aCommand: Char; const anExpansion: string);
    begin
      SetLength( FRules, RulesCount + 1);
      FRules[ RulesCount - 1].FCommand   := aCommand;
      FRules[ RulesCount - 1].FExpansion := anExpansion;
    end;


    procedure   TMazeLSystem.AddMeaningFor( aCommand: Char; const anExpansion: string);
    begin
      SetLength( FMeanings, MeaningsCount + 1);
      FMeanings[ MeaningsCount - 1].FCommand   := aCommand;
      FMeanings[ MeaningsCount - 1].FExpansion := anExpansion;
    end;


    function    TMazeLSystem.Expand( const aName, anAxiom: string; aDepth: Integer): string;
    var
      i    : Integer;
      j    : Integer;
      Prev : string;
      Next : string;
      S    : string;
    begin
      Result := Format( ': %s ( -- ) ', [ aName], AppLocale);
      Prev   := anAxiom;
      Next   := '';

      for i := 1 to aDepth
      do begin
        for j := Low( Prev) to High( Prev)
        do Next := Next + FindRuleFor( Prev[ j]);

        Prev := Next;
        Next := '';
      end;

      for i := Low( Prev) to High( Prev)
      do begin
        S := FindMeaningFor( Prev[ i]);

        if S <> ''
        then Result := Result + S + ' ';
      end;

      Result := Result + '; export';
    end;


{ ========
  TMazeForth = class( TMazeMaker)
  private
    FForth   : TForth;
    FLSystem : TMazeLSystem;
    FId      : Integer;
  private
}

    procedure   TMazeForth.DoForthResponse( aSender: TObject; const aMsg: string);
    begin
      FlagError( mzInfo, aMsg);
    end;


    procedure   TMazeForth.DoExternalFunc( aSender: TObject; anId: Integer; const aStack: TStack);
    var
      anLCommand : string;
      anLAction  : string;
      anLName    : string;
      anLDepth   : Integer;
    const
      FuncNames : array[ 0 .. 20] of string = (
          ' 0 : Clear'        ,
          ' 1 : RotateDegrees',
          ' 2 : RotateRadians',
          ' 3 : Walk'         ,
          ' 4 : Move'         ,
          ' 5 : PenUp'        ,
          ' 6 : PenDown'      ,
          ' 7 : Color'        ,
          ' 8 : Inflate'      ,
          ' 9 : PushState'    ,
          '10 : PopState'     ,
          '11 : DupState'     ,
          '12 : DropState'    ,
          '13 : SwapState'    ,
          '14 : PenSize'      ,
          '15 : DotSize'      ,
          '16 : LClear'       ,
          '17 : LRule'        ,
          '18 : LMeaning'     ,
          '19 : LAxiom'       ,
          '20 : Extent'
      );
    var
      aFunc : Integer;

      function FuncName : string;
      begin
        if ( aFunc >= Low( FuncNames)) and ( aFunc < High( FuncNames))
        then Result := FuncNames[ aFunc]
        else Result := Format( 'Unknown function %d', [ aFunc], AppLocale);
      end;

      function CheckStack( anAmount: Integer): Boolean;
      begin
        if aStack.Depth >= anAmount
        then Result := True
        else begin
          Result := False;
          FlagErrorFmt( mzError, 'Interpreter stack holds not enough data for function %s, it needs %d arguments', [ FuncName, anAmount]);
        end;
      end;

    begin
      if anId = FId
      then begin
        if aStack.Depth > 0
        then begin
          aFunc := aStack.Pop;

          case aFunc of
             0 : Clear;
             1 : if CheckStack( 1) then RotateDegrees( aStack.Pop);
             2 : if CheckStack( 1) then RotateRadians( aStack.Pop);
             3 : if CheckStack( 1) then Walk( aStack.Pop);
             4 : if CheckStack( 1) then Move( aStack.Pop);
             5 : PenUp;
             6 : PenDown;
             7 : if CheckStack( 1) then Color := aStack.Pop;
             8 : if CheckStack( 1) then Inflate( aStack.Pop);
             9 : PushState;
            10 : PopState;
            11 : DupState;
            12 : DropState;
            13 : SwapState;
            14 : if CheckStack( 1) then Pen( aStack.Pop);
            15 : if CheckStack( 1) then Dot( aStack.Pop);
            16 : LClear;
            17 : if CheckStack( 2) then begin anLAction := aStack.Pop; anLCommand := aStack.Pop; LRule   ( anLCommand, anLAction) end;
            18 : if CheckStack( 2) then begin anLAction := aStack.Pop; anLCommand := aStack.Pop; LMeaning( anLCommand, anLAction) end;
            19 : if CheckStack( 3) then begin anLDepth  := aStack.Pop; anLCommand := aStack.Pop; anLName := aStack.Pop; LAxiom( anLName, anLCommand, anLDepth) end;
            20 : PushExtent( aStack);
          end;
        end
        else FlagError( mzError, 'Interpreter stack holds not enough items for any external function');
      end;
    end;


    procedure   TMazeForth.LClear;
    begin
      if Assigned( FLSystem)
      then FLSystem.Clear;
    end;


    procedure   TMazeForth.LRule( const aCommand, anExpansion: string);
    begin
      if Assigned( FLSystem) and ( Length( aCommand) = 1)
      then FLSystem.AddRuleFor( aCommand[ Low( aCommand)], anExpansion);
    end;


    procedure   TMazeForth.LMeaning( const aCommand, anExpansion: string);
    begin
      if Assigned( FLSystem) and ( Length( aCommand) = 1)
      then FLSystem.AddMeaningFor( aCommand[ Low( aCommand)], anExpansion);
    end;


    procedure   TMazeForth.LAxiom( const aName, anAxiom: string; aDepth: Integer);
    begin
      if Assigned( FLSystem)
      then AcceptText( FLSystem.Expand( aName, anAxiom, aDepth));
    end;


    procedure   TMazeForth.PushExtent( aStack: TStack);
    var
      E : TMazeRect;
    begin
      E := Extent;
      aStack.Push( E.Left  );
      aStack.Push( E.Top   );
      aStack.Push( E.Right );
      aStack.Push( E.Bottom);
    end;


//  public

    constructor TMazeForth.Create( const aMazeGraph: TMazeGraph; anId: Integer);
    begin
      inherited Create( aMazeGraph);
      FId                   := anId;
      FForth                := TForth.Create;
      FForth.OnResponse     := DoForthResponse;
      FForth.OnExternalFunc := DoExternalFunc;
      FLSystem              := TMazeLSystem.Create;
      LoadFile( ApplicationPath + 'mazes.4th');
    end;


    destructor  TMazeForth.Destroy; // override;
    begin
      FreeAndNil( FLSystem);
      FreeAndNil( FForth  );
      inherited;
    end;


    procedure   TMazeForth.Initialize;
    begin
      FForth.StackReset;
    end;


    procedure   TMazeForth.AcceptText( const S: string);
    begin
      FForth.AcceptText( S, FId);
    end;


    procedure   TMazeForth.LoadFile( const aFileName: string);
    begin
      FForth.LoadWords( aFileName, FId);
    end;


    function    TMazeForth.CreateExportList: TStringList;
    begin
      Result := FForth.CreateExportList;
    end;


end.

