unit KnobsUtils;

{

  (C) COPYRIGHT 1999 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  WinApi.Windows, WinApi.ShLwApi,

  System.SysUtils, System.Types, System.Classes, System.DateUtils, System.Math, System.TypInfo,

  Vcl.Themes, Vcl.Controls, Vcl.Forms, Vcl.Graphics;


type

  TSignal          = Double;
  PSignal          = ^ TSignal;
  TKnobsTimeStamp  = Int64;
  TBinary          = Int32;
  TBitCount        = 1 .. 32;
  TSignalArray     = array of TSignal;
  PSignalArray     = ^ TSignalArray;
  TKnobsGridData   = Boolean;
  TKnobsDataLine   = array of TKnobsGridData;
  TKnobsDataPlane  = array of TKnobsDataLine;
  TOscTime         = UInt64;
  TKnobsModuleType = Integer;
  TByteSet         = set of Byte;
  TLogClass        = integer;

  TDistanceMode = (
    dmCircular,
    dmHorizontal,
    dmVertical
  );


  TSignalType = (
    stUnknown,        // treat as stAudio
    stControl,        // control rate
    stAudio,          // audio rate
    stLogic,          // audio rate logic levels
    stContrLogic      // control rate logic signals
  );

  TKnobsOnLog        = procedure( const aSender: Tobject; aLogClass: TLogClass; const aMsg: string) of object;
  TKnobsValueCompare = function( aVal1, aVal2: TSignal): Integer                                    of object;

const

  // LOG classes

  LC_NO_CLASS     = -1;
  LC_GENERAL      =  0;
  LC_STARTUP      =  1;
  LC_COMPILER     =  2;
  LC_PORTAUDIO    =  3;
  LC_RANDOMIZER   =  4;
  LC_RENDERER     =  5;
  LC_MIDI         =  6;
  LC_MIDI_MSG     =  7;
  LC_OSC          =  8;
  LC_OSC_MSG      =  9;
  LC_PATCH_DUMP   = 10;
  LC_PROFILER     = 11;
  LC_AUDIO        = 12;
  LC_DESIGNER     = 13;
  LC_SYNTH        = 14;
  LC_SYNTH_ERROR  = 15;
  LC_TERMINATE    = 16;
  LC_FORMANT      = 17;

  // Predefined module types

  MTYPE_AUDIO_INPUT       = 101;             // preset type for input  module
  MTYPE_AUDIO_OUTPUT      = 102;             // preset type for output module
  DELTA_LIMIT             =   1e-6;          // Small value for equality comparisons {  -120 dB}}


var

  AppLocale : TFormatSettings;

  procedure Nop;
  function  PreDec ( var aValue: Integer): Integer; inline;
  function  PostDec( var aValue: Integer): Integer; inline;
  procedure KilledException( const anException: Exception);                                                      inline;
  function  GetCpuClockCycleCount: Int64;                                                           assembler; register;
  function  Clip  ( const aValue: Integer; const aLow, aHigh: Integer): Integer;                       inline; overload;
  function  Clip  ( const aValue: TSignal; const aLow, aHigh: TSignal): TSignal;                       inline; overload;
  procedure PClip( var aValue: TSignal; const aLow, aHigh: TSignal);                                             inline;
  function  UnClip( const aValue: TSignal; const aLow, aHigh: TSignal): TSignal;
  function  FloatMod( const X, Y: TSignal): TSignal;                                                assembler; register;
  function  MathFloatMod( const X, Y: TSignal): TSignal;                                                         inline;
  function  MathFloatDiv( const X, Y: TSignal): TSignal;                                                         inline;
  function  MathIntMod( X, Y: Integer): Integer;                                                       overload; inline;
  function  MathIntDiv( X, Y: Integer): Integer;                                                       overload; inline;
  function  MathIntMod( X, Y: Int64  ): Int64;                                                         overload; inline;
  function  MathIntDiv( X, Y: Int64  ): Int64;                                                         overload; inline;
  function  RandomGaus( aSigma : TSignal; AbsVal: Boolean = False): TSignal;
  function  RandomExpo( aLambda: TSignal; AbsVal: Boolean = False): TSignal;
  function  AlmostEqual( const aValue1, aValue2: TSignal; const aDelta: TSignal = DELTA_LIMIT): Boolean;         inline;
  function  AlmostZero( const aValue : TSignal; const aDelta: TSignal = DELTA_LIMIT): Boolean;                   inline;
  function  PolyBLEP( aPhase, anAmount, aDeltaPhase: TSignal): TSignal; inline;
  function  ReverseNumber( aNumber: Int64): Int64;

  function  ApplicationPath: String;
  procedure Exchange( var A, B: Integer);
  function  RectanglesOverlap( A, B: TRect): Boolean;
  function  Distance( aPoint: TPoint): Integer;
  function  MultPoint( A: TPoint; M: Integer): TPoint;
  function  DivPoint( A: TPoint; D: Integer): TPoint;
  function  ValuesToAngle( MinValue, Value, MaxValue: TSignal): TSignal;
  function  PolarToPoint( aCenter: TPoint; anAngle, aLength: TSignal): TPoint;
  function  DistanceToValue( MinValue, MaxValue: TSignal; aPoint: TPoint; aDistanceMode: TDistanceMode): TSignal;
  function  Rotate( A: TPoint; anAngle: TSignal): TPoint;

  function  Indent( anInd: Integer): string;                                                                   overload;
  function  Indent( anInd: Integer; const S: string): string;                                                  overload;
  function  Indent( anInd: Integer; const aFmt: string; const anArgs: array of const): string;                 overload;

  procedure WriteInd  ( var aFile: TextFile; anInd: Integer; S: string);
  procedure WriteLnInd( var aFile: TextFile; anInd: Integer; S: string);

  function  TrimStr  ( const S: string; aLength: Integer): string;
  function  ExtendStr( const S: string; aLength: Integer): string;

  function  MakePath( const aParts: array of string): string;
  function  ParsePostfix( const aName: string): string;
  function  Explode( const aString: string; const aChar: Char): TStringList;                                   overload;
  function  Explode( aString: string; aSeparator: string): TStringList;                                        overload;
  function  Implode( const aStrings: TStringList    ; const aSeparator: string): string;                       overload;
  function  Implode( const aStrings: array of string; const aSeparator: string): string;                       overload;
  function  Implode( const aStrings: TStringList    ): string;                                                 overload;
  function  Implode( const aStrings: array of string): string;                                                 overload;
  procedure TrimStrings( const aStrings: TStrings);
  function  StripQuotes( const aValue, aLeftQuote, aRightQuote: string): string;
  function  UnQuote    ( const aValue, aLeftQuote, aRightQuote: string): string;
  function  Quote      ( const aValue, aLeftQuote, aRightQuote: string): string;
  function  TabsToSpaces( const aValue: string): string;

  function  OscTimeToDateTime( aTime: TOscTime): TDateTime;
  function  DateTimeToOscTime( aTime: TDateTime): TOscTime;
  function  OscNow: TOsctime;
  function  GetTimeZoneBias: Integer;
  function  IsDaylightSavingOn: Boolean;
  function  UTC: TDateTime;
  function  Now64: TKnobsTimeStamp;
  function  FormatTime64  ( aTime: TKnobsTimeStamp): string;
  function  FormatTime64ms( aTime: TKnobsTimeStamp): string;

  function  InterpolateColors( aColor, bColor: TColor; aMix: Byte): TColor;
  function  GetThemedPanelColor : TColor;
  function  IsPrime( aValue: Cardinal): Boolean;
  function  NextPrime( aValue: Cardinal): Cardinal;
  function  NextPowerOfTwo( aValue: Cardinal): Cardinal;                                                       overload;
  function  NextPowerOfTwo( aValue: UInt64): UInt64;                                                           overload;
  function  GrayCode( aValue: Byte    ): Byte;                                                                 overload;
  function  GrayCode( aValue: Word    ): Word;                                                                 overload;
  function  GrayCode( aValue: Cardinal): Cardinal;                                                             overload;
  function  GrayCode( aValue: UInt64  ): UInt64;                                                                overload;

  function  SignalTypeToStr( const aValue: TSignalType): string;
  function  SignalIsFast( const aValue: TSignalType): Boolean;
  function  ByteSetToStr( const aValue: TByteSet): string;
  function  DataPlaneToStr( const aValue: TKnobsDataPlane): string;
  function  StrToDataPlane( const aValue: string): TKnobsDataPlane;                                            overload;
  procedure StrToDataPlane( const aValue: string; const aPlane: TKnobsDataPlane);                              overload;
  function  FindControlUnderMouse: TControl;

  function  Lock  ( var aLocker: Integer): Boolean;                                                              inline;
  function  Unlock( var aLocker: Integer): Boolean;                                                              inline;
  function  Locked(     aLocker: Integer): Boolean;                                                              inline;
  procedure DecToZero( var aValue: TSignal);                                                           inline; overload;
  procedure DecToZero( var aValue: Integer);                                                           inline; overload;
  function  DecToZeroWithResetValue( var aValue : Integer; aResetValue: Integer): Boolean;                       inline;
  function  DecToZeroWithResetValueToggle( var aValue : Integer; var Memory: Boolean; aResetValue: Integer): Boolean; inline;

  function  LogClassToStr( aLogClass: TLogClass): string;
  function  RelativeToAbsolutePath( const aRelPath , aBasePath: string): string;
  function  AbsoluteToRelativePath( const anAbsPath, aBasePath: string): string;
  function  ComponentToStream( const aComponent : TComponent): TMemoryStream;
  function  ComponentToString( const aComponent : TComponent): string;
  procedure ComponentToFile  ( const aComponent : TComponent; const aFileName: string);
  function  StreamToComponent( const aStream   : TStream; const anOwner: TComponent): TComponent;
  function  StringToComponent( const aString   : string ; const anOwner: TComponent): TComponent;
  function  FileToComponent  ( const aFileName : string ; const anOwner: TComponent): TComponent;
  function  MakeValidFileName( const aValue: string; aReplacement: Char; AllowSlashes: Boolean = False): string;
  function  UniqueFilenameFromTemplate( const aPath, aTemplate, anExtension, aGuid: string): string;

  procedure PointerDebug( aPointer: Pointer; aSize: Integer = 0);


var

  ExceptionCounter : Int64;


implementation

uses

  Winapi.MMSystem,

  System.SyncObjs;


var

  strIndent   : string;
  GNowLock    : TCriticalSection;
  GNowHigh    : Cardinal;
  GNowLastLow : Cardinal;


  procedure Nop;
  begin
    // a Nop ... allows for setting breakpoints in places
  end;


  function  PreDec( var aValue: Integer): Integer; inline;
  begin
    Dec( aValue);
    Result := aValue;
  end;


  function  PostDec( var aValue: Integer): Integer; inline;
  begin
    Result := aValue;
    Dec( aValue);
  end;


  procedure KilledException( const anException: Exception); inline;
  begin
    {$Q-R-} Inc( ExceptionCounter); {$Q+R+}
  end;


  function GetCpuClockCycleCount: Int64;
  asm
    rdtsc
  end;


  function KnobClip( const aValue: TSignal; const aLow, aHigh: TSignal): TSignal;
  begin
    Result := aValue;

    if aLow < aHigh
    then begin
      if aValue < aLow
      then Result := aLow
      else If aValue > aHigh
      then Result := aHigh;
    end
    else begin
      if aValue < aHigh
      then Result := aHigh
      else If aValue > aLow
      then Result := aLow;
    end;
  end;


  function  Clip( const aValue: Integer; const aLow, aHigh: Integer): Integer; inline; overload;
  begin
    if aValue < aLow
    then Result := aLow
    else if aValue > aHigh
    then Result := aHigh
    else Result := aValue;
  end;


  function Clip( const aValue: TSignal; const aLow, aHigh: TSignal): TSignal; inline; overload;
  begin
    if aValue < aLow
    then Result := aLow
    else if aValue > aHigh
    then Result := aHigh
    else Result := aValue;
  end;


  procedure PClip( var aValue: TSignal; const aLow, aHigh: TSignal); inline;
  begin
    if aValue < aLow
    then aValue := aLow
    else if aValue > aHigh
    then aValue := aHigh;
  end;


  function  UnClip( const aValue: TSignal; const aLow, aHigh: TSignal): TSignal;
  begin
    if ( aValue < aLow) or ( aValue > aHigh)
    then Result := aValue
    else begin
      if aValue > ( aLow + aHigh) / 2
      then Result := aHigh
      else Result := aLow;
    end;
  end;


  function  FLoatMod( const X, Y: TSignal): TSignal; assembler; register;
  asm
    fld qword ptr[ Y]
    fld qword ptr[ X]
@r:
    fprem
    fstsw ax
    sahf
    jp    @r
    fstp  st( 1)
  end;


//  function  FloatMod( X, Y: Double): Double; inline;
//  begin
//    Result := Frac( X / Y) * Y;
//  end;


  function  MathFloatMod( const X, Y: TSignal): TSignal; inline;
  begin
    Result := FloatMod( FloatMod( X, Y) + Y, Y);
  end;


  function  MathFloatDiv( const X, Y: TSignal): TSignal; // inline;
  begin
    Result := Floor( X / Y);
  end;


  function  MathIntMod( X, Y: Integer): Integer; inline; // overload;
  begin
    Result := ( X mod Y + Y) mod Y;
  end;


  function  MathIntDiv( X, Y: Integer): Integer; inline; // overload;
  begin
    Result := Floor( X / Y);
  end;


  function  MathIntMod( X, Y: Int64): Int64; inline; // overload;
  begin
    Result := ( X mod Y + Y) mod Y;
  end;


  function  MathIntDiv( X, Y: Int64): Int64; inline; // overload;
  begin
    Result := Floor( X / Y);
  end;


  function  RandomGaus( aSigma: TSignal; AbsVal: Boolean = False): TSignal;
  begin
    if AbsVal
    then Result := Abs( RandG( 0, aSigma))
    else Result := RandG( 0, aSigma);
  end;


  function  RandomExpo( aLambda: TSignal; AbsVal: Boolean = False): TSignal;
  begin
    Result := - Ln( 1.0 - Random) / aLambda;

    if not AbsVal
    then begin
      if Random( 2) = 0
      then Result := - Result;
    end;
  end;


  function  AlmostEqual( const aValue1, aValue2: TSignal; const aDelta: TSignal = DELTA_LIMIT): Boolean;  inline;
  begin
    Result := Abs( aValue1 - aValue2) < aDelta;
  end;


  function  AlmostZero( const aValue: TSignal; const aDelta: TSignal = DELTA_LIMIT): Boolean; inline;
  begin
    Result := Abs( aValue) < aDelta;
  end;


  function  PolyBLEP( aPhase, anAmount, aDeltaPhase: TSignal): TSignal; inline;
  // Moved out of HrastUnit.pas for usage by other oscillators than HrastOsc as well.
  var
    dt: TSignal;
  begin
    Result := 0;

    if AlmostZero( anAmount)
    then Exit;

    dt := anAmount * aDeltaPhase;

    if aPhase < dt
    then begin
      aPhase := aPhase / dt - 1;
      Result := - aPhase * aPhase;
    end
    else if aPhase > 1 - dt
    then begin
      aPhase := ( aPhase - 1) / dt + 1;
      Result := aPhase * aPhase;
    end;
  end;


  function  ReverseNumber( aNumber: Int64): Int64;
  begin
    Result := 0;

    while aNumber <> 0
    do begin
      Result  := Result * 10;
      Result  := Result + MathIntMod( aNumber, 10);
      aNumber := MathIntDiv( aNumber, 10);
    end;
  end;


  function  ApplicationPath: string;
  begin
    Result := ExtractFilePath( Application.ExeName);
  end;


  procedure Exchange( var A, B: Integer);
  var
    T : Integer;
  begin
    T := A;
    A := B;
    B := T;
  end;


  function  RectanglesOverlap( A, B: TRect): Boolean;
  var
    R : TRect;
  begin
    Result := IntersectRect( R, A, B);
  end;


  function  Distance( aPoint: TPoint): Integer;
  begin
    with aPoint
    do Result := Round( Sqrt( 1.0 * x * x + 1.0 * y * y));
  end;


  function   MultPoint( A: TPoint; M: Integer): TPoint;
  begin
    Result := Point( A.x * M, A.y * M);
  end;


  function   DivPoint( A: TPoint; D: Integer): TPoint;
  begin
    Result := Point( A.x div D, A.y div D);
  end;


  function  ValuesToAngle( MinValue, Value, MaxValue: TSignal): TSignal;
  begin
    if MinValue <> MaxValue
    then Result := 270 * ( Value - MinValue) / ( MinValue - MaxValue) + 225
    else Result := 0; // arbitrary ...
  end;


  function  PolarToPoint( aCenter: TPoint; anAngle, aLength: TSignal): TPoint;
  begin
    with Result
    do begin
      x := Round( aCenter.x + aLength * Cos( DegToRad( anAngle)));
      y := Round( aCenter.y - aLength * Sin( DegToRad( anAngle)));
    end;
  end;


  function  DistanceToValue( MinValue, MaxValue: TSignal; aPoint: TPoint; aDistanceMode: TDistanceMode): TSignal;
  var
    anAngle : Extended;
    anAlpha : Extended;
  begin
    case aDistanceMode of
      dmCircular :
        begin
          anAngle := RadToDeg( ArcTan2( - aPoint.y, aPoint.x));

          while anAngle < 0
          do anAngle := anAngle + 360;

          while anAngle > 270
          do anAngle := anAngle - 360;

          Result := KnobClip( MinValue + (( MinValue - MaxValue) * ( anAngle - 225) / 270), MinValue, MaxValue);
        end;

      dmHorizontal :
        begin
          anAlpha := 0.5 + aPoint.X / 200;
          Result  := KnobClip( MinValue + anAlpha * ( MaxValue - MinValue), MinValue, MaxValue);
        end;

      dmVertical :
        begin
          anAlpha := 0.5 - aPoint.Y / 200;
          Result  := KnobClip( MinValue + anAlpha * ( MaxValue - MinValue), MinValue, MaxValue);
        end;

      else begin
        Result := 0;
        Assert( False, 'unhandled knob control mode');
      end;
    end;
  end;


  function   Rotate( A: TPoint; anAngle: TSignal): TPoint;
  begin
    anAngle := DegToRad( anAngle);

    with Result
    do begin
      x := Round( Cos( anAngle) * A.x - Sin( anAngle) * A.y);
      y := Round( Sin( anAngle) * A.x + Cos( anAngle) * A.y);
    end;
  end;


  function  Indent( anInd: Integer): string; // overload;
  begin
    Result := Copy( strIndent, 1, 2 * anInd);
  end;


  function  Indent( anInd: Integer; const S: string): string; // overload;
  begin
    Result := Format( '%s%s', [ Indent( anInd), S]);
  end;


  function  Indent( anInd: Integer; const aFmt: string; const anArgs: array of const): string; // overload;
  begin
    Result := Indent( anInd, Format( aFmt, anArgs));
  end;


  procedure WriteInd( var aFile: TextFile; anInd: Integer; S: string);
  begin
    Write( aFile, Indent( anInd), S);
  end;


  procedure WriteLnInd( var aFile: TextFile; anInd: Integer; S: string);
  begin
    WriteLn( aFile, Indent( anInd), S);
  end;


  function  TrimStr( const S: string; aLength: Integer): string;
  begin
    Result := Copy( ExtendStr( S, aLength), 1, aLength); // Procrustes trimming.
  end;


  function  ExtendStr( const S: string; aLength: Integer): string;
  begin
    Result := S;

    while Length( Result) < aLength
    do Result := Result + ' ';
  end;


  function  MakePath( const aParts: array of string): string;
  begin
    Result := Implode( aParts, '/');
  end;


  function  ParsePostfix( const aName: string): string;
  var
    i : Integer;
    p : Integer;
  begin
    p := 0;

    for i := Length( aName) downto 1
    do begin
      if aName[ i] = '_'
      then begin
        p := i;
        Break;
      end;
    end;

    if p > 0
    then Result := Copy( aName, p + 1, Length( aName))
    else Result := aName;
  end;


  function  Explode( const aString: string; const aChar: Char): TStringList; // overload;
  // Note : when aSring = '' this one eturns stringist with one item
  //        containong the empty string.
  var
    p : Integer;
    i : integer;
    S : string;
  begin
    Result := TStringList.Create;
    p      := 1;
    i      := 1;

    while i <= Length( aString)
    do begin
      if aString[ i] = aChar
      then begin
        S := Copy( aString, p, i - p);
        Result.Add( S);

        if CharInSet( aChar, [ ' ', ^I])
        then begin
          while ( i <= Length( aString)) and ( aString[ i] = aChar)
          do Inc( i);
        end
        else Inc( i);

        p := i;
      end
      else Inc( i);
    end;

    if p <= i
    then begin
      S := Copy( aString, p, Length( aString));
      Result.Add( S);
    end;
  end;


  function Explode( aString: string; aSeparator: string): TStringList; // overload;
  // Note : when aSring = '' this one eturns a stringist with no
  //        items in it.
  var
    c : word;
    p : Integer;
  begin
    Result := TStringList.Create;
    c := 0;

    while aString <> ''
    do begin
      p := Pos( aSeparator, aString);

      if p > 0
      then begin
        Result.Add( Copy( aString, 1, p - 1));
        Delete( aString, 1, Length( Result[ c]) + Length( aSeparator));
      end
      else begin
        Result.Add( aString);
        aString := '';
      end;

      inc( c);
    end;
  end;


  function  Implode( const aStrings: TStringList; const aSeparator: string): string; // overload;
  var
    i : Integer;
  begin
    Result := '';

    if Assigned( aStrings)
    then begin
      for i := 0 to aStrings.Count - 1
      do begin
        if i = aStrings.Count - 1
        then Result := Result + aStrings[ i]
        else Result := Result + aStrings[ i] + aSeparator;
      end;
    end;
  end;


  function  Implode( const aStrings: array of string; const aSeparator: string): string; // overload;
  var
    i : Integer;
  begin
    Result := '';
    for i := Low( aStrings) to High( aStrings)
    do begin
      if i = High( aStrings)
      then Result := Result + aStrings[ i]
      else Result := Result + aStrings[ i] + aSeparator;
    end;
  end;


  function  Implode( const aStrings: TStringList): string; // overload;
  begin
    Result := Implode( aStrings, ' ');
  end;


  function  Implode( const aStrings: array of string): string; // overload;
  begin
    Result := Implode( aStrings, ' ');
  end;


  procedure TrimStrings( const aStrings: TStrings);
  var
    i : Integer;
  begin
    if Assigned( aStrings)
    then begin
      for i := 0 to aStrings.Count - 1
      do aStrings[ i] := Trim( aStrings[ i]);
    end;
  end;


  function  StripQuotes( const aValue, aLeftQuote, aRightQuote: string): string;
  begin
    if ( Pos( aLeftQuote, aValue) = 1) and ( Pos( aRightQuote, aValue, Length( aValue) - Length( aRightQuote) + 1) = Length( aValue) - Length( aRightQuote) + 1)
    then Result := Copy( aValue, Length( aLeftQuote) + 1, Length( aValue) - Length( aLeftQuote) - Length( aRightQuote))
    else Result := aValue;
  end;


  function  UnQuote    ( const aValue, aLeftQuote, aRightQuote: string): string;
  begin
    Result := StripQuotes( aValue, aLeftQuote, aRightQuote);
  end;


  function  Quote( const aValue, aLeftQuote, aRightQuote: string): string;
  begin
    Result := Format( '%s%s%s', [ aLeftQuote, aValue, aRightQuote]);
  end;


  function  TabsToSpaces( const aValue: string): string;
  var
    C : Char;
  begin
    Result := '';
    for C in aValue
    do begin
      if C = ^H
      then Result := Result + ' '
      else Result := Result + C;
    end;
  end;



const

  SecsPerDay = 86400.0;          // seconds per day
  C1shl32    = 4294967296.0;     // 1 << 32 as a float
  DayShift   = 2.0;              // to move from dec 30 1899 to jan 1 1900

  // OSC time is a 64 bit integer, where the leftmost 32 bits specify seconds
  // and the right most 32 bits specify fractional seconds, the time is relative
  // to 1900-01-01 00:00:00 UTC - this is the same as an NTP timestamp.
  // TDateTime is a floating point time, the integer part holds days, the
  // fractional bit holds the fraction of a day. This time is relative
  // to December 30, 1899; 12:00 A.M, in the local time zone ...

  procedure DateTimeToNTP( aTime: TDateTime; var aSecs, aFracs: Cardinal);
  var
    Value1 : Double;
    Value2 : Double;
  begin
    Value1 := ( aTime - DayShift) * SecsPerDay;
    Value2 := Value1;

    if Value2 > C1shl32
    then Value2 := Value2 - C1shl32;

    aSecs := Cardinal( Trunc( Value2));
    Value2 := (( Frac( Value1) * 1000) / 1000) * C1shl32;

    if Value2 > C1shl32
    then Value2 := Value2 - C1shl32;

    aFracs := Cardinal( Trunc( Value2));
  end;


  function NTPToDateTime( aSecs, aFrac: Cardinal): TDateTime;
  var
    Value1 : Double;
    Value2 : Double;
  begin
    Value1 := aSecs;

    if Value1 < 0
    then Value1 := C1shl32 + Value1 - 1;

    Value2 := aFrac;

    if Value2 < 0
    then Value2 := C1shl32 + Value2 - 1;

    Value2 := Trunc(( Value2 / C1shl32) * 1000) / 1000;
    Result := (( Value1 + Value2) / SecsPerDay) + DayShift;
  end ;


  function  OscTimeToDateTime( aTime: TOscTime): TDateTime;
  var
    aSecs : Cardinal;
    aFrac : Cardinal;
  begin
    aSecs  := ( aTime shr 32) and $ffffffff;
    aFrac  := ( aTime shr  0) and $ffffffff;
    Result := NTPToDateTime( aSecs, aFrac);
  end;


  function  DateTimeToOscTime( aTime: TDateTime): TOscTime;
  var
    aSecs : Cardinal;
    aFrac : Cardinal;
  begin
    DateTimeToNTP( aTime, aSecs, aFrac);
    Result := ( UInt64( aSecs) shl 32) + UInt64( aFrac);
  end;


  function  OscNow: TOsctime;
  begin
    Result := DateTimeToOscTime( UTC);
  end;


  function  GetTimeZoneBias: Integer;
  var
    tz: TTimeZoneInformation;
  begin
    case GetTimeZoneInformation( tz) of
      TIME_ZONE_ID_STANDARD: Result := - ( tz.StandardBias + tz.Bias) div 60;
      TIME_ZONE_ID_DAYLIGHT: Result := - ( tz.DaylightBias + tz.Bias) div 60;
    else
      Result := 0;
    end;
  end;


  function  IsDaylightSavingOn: Boolean;
  var
    tz: TTimeZoneInformation;
  begin
    Result := GetTimeZoneInformation( tz) = TIME_ZONE_ID_DAYLIGHT;
  end;


  function  UTC: TDateTime;
  begin
    Result := Now - GetTimeZoneBias * OneHour;
  end;


  function  Now64: Int64;
  begin
    GNowLock.Acquire;

    try
      Int64Rec( Result).Lo := timeGetTime;

      if Int64Rec( Result).Lo < GNowLastLow
      then Inc( Int64Rec( Result).Hi);

      GNowLastLow          := Int64Rec( Result).Lo;
      Int64Rec( Result).Hi := GNowHigh;
    finally
      GNowLock.Release;
    end
  end;


  function  FormatTime64( aTime: TKnobsTimeStamp): string;
  var
    aTTime : TTime;
  begin
    if aTime < 0
    then Result := '--:--:--:--'
    else begin
      aTTime := aTime / MSecsPerDay;
      Result := Format( FormatDateTime( '"%.2d":hh:mm:ss', aTTime), [ Floor( aTTime)], AppLocale);
    end;
  end;


  function  FormatTime64ms( aTime: TKnobsTimeStamp): string;
  var
    aTTime : TTime;
  begin
    if aTime < 0
    then Result := '--:--:--:--:---'
    else begin
      aTTime := aTime / MSecsPerDay;
      Result := Format( FormatDateTime( '"%.2d":hh:mm:ss:zzz', aTTime), [ Floor( aTTime)], AppLocale);
    end;
  end;


  function  InterpolateColors( aColor, bColor: TColor; aMix: Byte): TColor;
  // aMix from 0 - 255, when 0 aColor is returned, when 255 bColor
  begin
    Result := RGB(
      (( Word( 255 - aMix) * Word( GetRValue( aColor)) + Word( aMix) * Word( GetRValue( bColor))) div 256) and $ff,
      (( Word( 255 - aMix) * Word( GetGValue( aColor)) + Word( aMix) * Word( GetGValue( bColor))) div 256) and $ff,
      (( Word( 255 - aMix) * Word( GetBValue( aColor)) + Word( aMix) * Word( GetBValue( bColor))) div 256) and $ff
    );
  end;


  function  GetThemedPanelColor : TColor;
  var
    LDetails : TThemedElementDetails;
    aColor   : TColor;
  begin
    Result := clNone;

    if StyleServices.Available
    then begin
      LDetails := StyleServices.GetElementDetails( tpPanelBackground);

      if StyleServices.GetElementColor( LDetails, ecFillColor, aColor) and ( aColor <> clNone)
      then Result := aColor;
    end;
  end;


  function  IsPrime( aValue: Cardinal): Boolean;
  var
    X       : Cardinal;
    Largest : Cardinal;
  begin
    if aValue < 4                                  // 1, 2 and 3 are primes
    then Result := True
    else begin
      if ( aValue mod 2) = 0                       // skip all even values to make it a tad faster
      then Result := False
      else begin
        Largest := Trunc( Sqrt( 1.0 * aValue));    // Only check up to Sqrt( aValue)
        Result  := True;                           // Assume prime
        X       := 3;                              // First divisor to try is 3

        while ( X <= Largest) and Result
        do begin
          Inc( X, 2);                              // Go to next odd value
          Result := ( aValue mod X) <> 0;          // Get false when a divisor found, so not prime then.
        end;
      end;
    end;
  end;


  function  NextPrime( aValue: Cardinal): Cardinal;
  begin
    Result := aValue;

    if not IsPrime( Result)                        // when not a prime yet
    then begin
      Result := Result or 1;                       // make sure its not a multiple of two ..

      while not IsPrime( Result)                   // then repeatedly add two till its a prime
      do Inc( Result, 2);
    end;
  end;


  function  NextPowerOfTwo( aValue: Cardinal): Cardinal; // overload;
  var
    i : Integer;
    s : Integer;
  begin
    aValue := aValue - 1;
    s      := 1;

    for i := 0 to 4
    do begin
      aValue := aValue or ( aValue shr s);
      s      := s shl 1;
    end;

    Result := aValue + 1;
  end;


  function  NextPowerOfTwo( aValue: UInt64): UInt64; // overload;
  var
    i : Integer;
    s : Integer;
  begin
    aValue := aValue - 1;
    s      := 1;

    for i := 0 to 5
    do begin
      aValue := aValue or ( aValue shr s);
      s      := s shl 1;
    end;

    Result := aValue + 1;
  end;


  function  GrayCode( aValue: Byte): Byte; // overload;
  begin
    Result := aValue xor ( aValue shr 1);
  end;


  function  GrayCode( aValue: Word): Word; // overload;
  begin
    Result := aValue xor ( aValue shr 1);
  end;


  function  GrayCode( aValue: Cardinal): Cardinal; // overload;
  begin
    Result := aValue xor ( aValue shr 1);
  end;


  function  GrayCode( aValue: UInt64): UInt64; // overload;
  begin
    Result := aValue xor ( aValue shr 1);
  end;


  function  SignalTypeToStr( const aValue: TSignalType): string;
  begin
    Result := GetEnumName( TypeInfo( TSignalType), Ord( aValue));
  end;


  function  SignalIsFast( const aValue: TSignalType): Boolean;
  begin
    Result := aValue in [ stAudio, stLogic];
  end;


  function  ByteSetToStr( const aValue: TByteSet): string;
  var
    Curr    : Byte;
    Prev    : Integer;
    InRange : Boolean;
  begin
    Result  := '';
    InRange := False;
    Prev    := -2;                    // To make '[ 0' not result in '[ .. 0'

    for Curr in aValue
    do begin
      if Curr = Prev + 1
      then begin
        if not InRange
        then begin
          InRange := True;
          Result  := Result + ' .. ';
        end;
      end
      else begin
        if InRange
        then begin
          InRange := False;
          Result  := Result + IntToStr( Prev) + ', ' + IntToStr( Curr);
        end
        else begin
          if Result = ''
          then Result := Result +        IntToStr( Curr)
          else Result := Result + ', ' + IntToStr( Curr);
        end;
      end;

      Prev := Curr;
    end;

    if InRange
    then Result := Result + IntToStr( Prev);

    if Result = ''
    then Result := '[]'
    else Result := '[ ' + Result + ']';
  end;


  function  DataPlaneToStr( const aValue: TKnobsDataPlane): string;
  var
    i : Integer;
    j : Integer;
    S : string;
  begin
    Result := '';

    for i := 0 to Length( aValue) - 1
    do begin
      S := '';

      for j := 0 to Length( aValue[ i]) - 1
      do begin
        if aValue[ i, j]
        then S := S + 'X'              // X is arbitrary, any non-space non-'|' should work
        else S := S + ' ';
      end;

      if Result = ''
      then Result := S
      else Result := Result + '|' + S;
    end;
  end;


  function  StrToDataPlane( const aValue: string): TKnobsDataPlane;
  var
    aParts : TStringList;
    S      : string;
    i      : Integer;
    j      : Integer;
  begin
    aParts := Explode( aValue, '|');

    try
      SetLength( Result, aParts.Count);

      for i := 0 to aParts.Count - 1
      do begin
        S := aParts[ i];
        SetLength( Result[ i], Length( S));

        for j := 0 to Length( S) - 1
        do Result[ i, j] := S[ Low( S) + j] <> ' ';
      end;
    finally
      aParts.DisposeOf;
    end;
  end;


  procedure StrToDataPlane( const aValue: string; const aPlane: TKnobsDataPlane); // overload;
  var
    aParts : TStringList;
    S      : string;
    i      : Integer;
    j      : Integer;
  begin
    aParts := Explode( aValue, '|');

    try
      if Length( aPlane) = aParts.Count
      then begin
        for i := 0 to aParts.Count - 1
        do begin
          S := aParts[ i];

          if Length( aPlane[ i]) = Length( S)
          then begin
            for j := 0 to Length( S) - 1
            do aPlane[ i, j] := S[ Low( S) + j] <> ' ';
          end;
        end;
      end;
    finally
      aParts.DisposeOf;
    end;
  end;


  function  FindControlUnderMouse: TControl;
  begin
    Result := FindDragTarget( Mouse.CursorPos, False);
  end;


  function  Lock( var aLocker: Integer): Boolean; inline;
  begin
    Result := not Locked( aLocker);

    if Result
    then Inc( aLocker);
  end;


  function  Unlock( var aLocker: Integer): Boolean; inline;
  begin
    if aLocker > 0
    then DecToZero( aLocker);

    Result := not Locked( aLocker);
  end;


  function  Locked( aLocker: Integer): Boolean; inline;
  begin
    Result := aLocker > 0;
  end;


  procedure DecToZero( var aValue: TSignal); inline; // overload;
  begin
    if aValue > 0
    then aValue := aValue - 1
    else aValue := 0;
  end;


  procedure DecToZero( var aValue: Integer); inline; // overload;
  begin
    if aValue > 0
    then Dec( aValue);
  end;

  function DecToZeroWithResetValue( var aValue: Integer; aResetValue: Integer): Boolean; inline;
  begin
    Result := False;

    if aValue > 1
    then Dec( aValue)
    else begin
      aValue := aResetValue;
      Result := True;
    end;
  end;


  function  DecToZeroWithResetValueToggle( var aValue : Integer; var Memory: Boolean; aResetValue: Integer): Boolean; inline;
  begin
    if DecToZeroWithResetValue( aValue, aResetValue)
    then Memory := not Memory;

    Result := Memory;
  end;


  function  LogClassToStr( aLogClass: TLogClass): string;
  begin
    case aLogClass of
      LC_NO_CLASS    : Result :=         'NO_CLASS     ';
      LC_GENERAL     : Result :=         'GENERAL      ';
      LC_STARTUP     : Result :=         'STARTUP      ';
      LC_COMPILER    : Result :=         'COMPILER     ';
      LC_PORTAUDIO   : Result :=         'PORTAUDIO    ';
      LC_RANDOMIZER  : Result :=         'RANDOMIZER   ';
      LC_RENDERER    : Result :=         'RENDERER     ';
      LC_MIDI        : Result :=         'MIDI         ';
      LC_MIDI_MSG    : Result :=         'MIDI_MSG     ';
      LC_OSC         : Result :=         'OSC          ';
      LC_OSC_MSG     : Result :=         'OSC_MSG      ';
      LC_PATCH_DUMP  : Result :=         'PATCH_DUMP   ';
      LC_PROFILER    : Result :=         'PROFILER     ';
      LC_AUDIO       : Result :=         'AUDIO        ';
      LC_DESIGNER    : Result :=         'DESIGNER     ';
      LC_SYNTH       : Result :=         'SYNTH        ';
      LC_SYNTH_ERROR : Result :=         'SYNTH_ERROR  ';
      LC_TERMINATE   : Result :=         'TERMINATE    ';
      LC_FORMANT     : Result :=         'FORMANT      ';
      else             Result := Format( 'LC_UNKNOWN_%.2d', [ aLogClass], AppLocale);
    end;
  end;


  function   RelativeToAbsolutePath( const aRelPath, aBasePath: string): string;
  begin
    SetLength( Result, MAX_PATH);

    if Assigned( PathCombine( @ Result[ Low( Result)], PChar( IncludeTrailingPathDelimiter( aBasePath)), PChar( aRelPath)))
    then SetLength( Result, Length( PChar( @ Result[ 1])))
    else Result := '';
  end;


  function  AbsoluteToRelativePath( const anAbsPath, aBasePath: string): string;
  var
    aPath : array[ 0 .. MAX_PATH - 1] of Char;
  begin
    PathRelativePathTo( @ aPath[ 0], PChar( aBasePath), FILE_ATTRIBUTE_DIRECTORY, PChar( anAbsPath), 0);
    Result := aPath;
  end;


  function  ComponentToStream( const aComponent: TComponent): TMemoryStream;
  // Result is in readable format
  var
    aBinStream : TMemoryStream;
  begin
    Result := nil;

    if Assigned( aComponent)
    then begin
      aBinStream := TMemoryStream.Create;

      try
        aBinStream.WriteComponent( aComponent);
        aBinStream.Position := 0;
        Result := TMemoryStream.Create;
        ObjectBinaryToText( aBinStream, Result);
        Result.Position := 0;
      finally
        aBinStream.DisposeOf;
      end;
    end;
  end;


  function  ComponentToString( const aComponent: TComponent): string;
  // Result is in readable format
  var
    aStream    : TMemoryStream;
    aStrStream : TStringStream;
    S          : string;
  begin
    aStream := ComponentToStream( aComponent);

    try
      aStrStream := TStringStream.Create( S);

      try
        aStrStream.CopyFrom( aStream, 0);
        aStrStream.Seek( 0, soFromBeginning);
        Result := aStrStream.DataString;
      finally
        aStrStream.Free;
      end;
    finally
      aStream.DisposeOf;
    end;
  end;


  procedure ComponentToFile( const aComponent: TComponent; const aFileName: string);
  // File contents is in readable format
  var
    aStream     : TMemoryStream;
    aFileStream : TFileStream;
  begin
    aStream := ComponentToStream( aComponent);

    try
      aFileStream := TFileStream.Create( aFileName, fmCreate);

      try
        aFileStream.CopyFrom( aStream, 0);
      finally
        aFileStream.DisposeOf;
      end;
    finally
      aStream.DisposeOf;
    end;
  end;


  function  StreamToComponent( const aStream: TStream; const anOwner: TComponent): TComponent;
  // aValue is in readable format
  var
    aBinStream : TMemoryStream;
  begin
    Result := nil;

    if Assigned( aStream)
    then begin
      aBinStream := TMemoryStream.Create;

      try
        ObjectTextToBinary( aStream, aBinStream);
        aBinStream.Position := 0;
        Result := aBinStream.ReadComponent( anOwner);
      finally
        aBinStream.DisposeOf;
      end;
    end;
  end;


  function  StringToComponent( const aString: string; const anOwner: TComponent): TComponent;
  // aValue is in readable format
  var
    aStrStream : TStringStream;
  begin
    aStrStream := TStringStream.Create( aString);

    try
      Result := StreamToComponent( aStrStream, anOwner);
    finally
      aStrStream.DisposeOf;
    end;
  end;


  function  FileToComponent( const aFileName: string; const anOwner: TComponent): TComponent;
  // File contensts in readable format
  var
    aFileStream : TFileStream;
  begin
    Result := nil;

    if FileExists( aFileName)
    then begin
      aFileStream := TFileStream.Create( aFileName, fmOpenRead);

      try
        aFileStream.Position := 0;
        Result := StreamToComponent( aFileStream, anOwner);
      finally
        aFileStream.DisposeOf;
      end;
    end;
  end;


  function  MakeValidFileName( const aValue: string; aReplacement: Char; AllowSlashes: Boolean = False): string;
  // Replace invalid filename characters with '_', but optionally do allow '\' and '/'
  const
    InvalidChars = [ '<', '>', '|', '"', ':', '*', '?', '$'];
    Slashes      = [ '/', '\'];
  var
    i : Integer;
  begin
    Result   := '';

    for i := Low( aValue) to High( aValue)
    do begin
      if CharInSet( aValue[ i], InvalidChars)
      or
        (
              ( not AllowSlashes)
          and CharInSet( aValue[ i], Slashes)
        )
      then Result := Result + aReplacement
      else Result := Result + aValue[ i];
    end;
  end;


  function  UniqueFilenameFromTemplate( const aPath, aTemplate, anExtension, aGuid: string): string;
  const
    NumFormat = '%.2d';
  var
    aGui  : string;
    aPref : string;
    aTemp : string;
    aNow  : TDateTime;
    aDate : string;
    aTime : string;
    anExt : string;
    N     : Integer;
    aSeq  : string;
    S     : string;
  begin
    if aTemplate = ''
    then aTemp := '$D-$N'
    else aTemp := aTemplate;

    if aGuid.IsEmpty
    then aGui := aGuid
    else aGui := Trim( StripQuotes( aGuid, '{', '}'));

    aNow   := Now;
    aDate  := FormatDateTime( AppLocale.ShortDateFormat, aNow);
    aTime  := FormatDateTime( AppLocale.LongTimeFormat , aNow);
    Result := StringReplace( aTemp , '$D', aDate,[ rfReplaceAll]);
    Result := StringReplace( Result, '$T', aTime,[ rfReplaceAll]);
    Result := StringReplace( Result, '$G', aGui ,[ rfReplaceAll]);
    N      := 1;
    aSeq   := Format( NumFormat, [ N], AppLocale);
    S      := MakeValidFileName( StringReplace( Result, '$N', aSeq ,[ rfReplaceAll]), '-', False);

    while FileExists( S)
    do begin
      Inc( N);
      aSeq := Format( NumFormat, [ N], AppLocale);
      S    := MakeValidFileName( StringReplace( Result, '$N', aSeq ,[ rfReplaceAll]), '-', False);
    end;

    if anExtension.IsEmpty
    or ( anExtension[ 1] = '.')
    then anExt := anExtension
    else anExt := '.' + anExtension;

    if aPath = ''
    then aPref := ''
    else aPref := IncludeTrailingPathDelimiter( aPath);

    Result := aPref + S + anExt;
  end;


{$HINTS OFF} // DebugMem not used, but used ... for inspection .. when optimization is off
  procedure PointerDebug( aPointer: Pointer; aSize: Integer = 0);
  const
    COUNT = 1024;
  type
    TDebugMem = record
      case Integer of
         0 : ( AsBools       : array[ 0 .. COUNT div SizeOf( Boolean   )] of Boolean   );
         1 : ( AsBytes       : array[ 0 .. COUNT                        ] of Byte      );
         2 : ( AsChars       : array[ 0 .. COUNT div SizeOf( Char      )] of Char      );
         3 : ( AsAnsiChars   : array[ 0 .. COUNT div SizeOf( AnsiChar  )] of AnsiChar  );
         4 : ( AsShortInts8  : array[ 0 .. COUNT div SizeOf( ShortInt  )] of ShortInt  );
         5 : ( AsWords16     : array[ 0 .. COUNT div SizeOf( Word      )] of Word      );
         6 : ( AsSmallInts16 : array[ 0 .. COUNT div SizeOf( SmallInt  )] of SmallInt  );
         7 : ( AsInts32      : array[ 0 .. COUNT div SizeOf( Integer   )] of Integer   );
         8 : ( AsCardinals32 : array[ 0 .. COUNT div SizeOf( Cardinal  )] of Cardinal  );
         9 : ( AsUInt64s     : array[ 0 .. COUNT div SizeOf( UInt64    )] of UInt64    );
        10 : ( AsInt64s      : array[ 0 .. COUNT div SizeOf( Int64     )] of Int64     );
        11 : ( AsNativeUInts : array[ 0 .. COUNT div SizeOf( NativeUInt)] of NativeUInt);
        12 : ( AsNativeInts  : array[ 0 .. COUNT div SizeOf( NativeInt )] of NativeInt );
        13 : ( AsFSingles    : array[ 0 .. COUNT div SizeOf( Single    )] of Single    );
        14 : ( AsFDoubles    : array[ 0 .. COUNT div SizeOf( Double    )] of Double    );
        15 : ( AsFExtendeds  : array[ 0 .. COUNT div SizeOf( Extended  )] of Extended  );
        16 : ( AsFCurrencies : array[ 0 .. COUNT div SizeOf( Currency  )] of Currency  );
        17 : ( AsFComps      : array[ 0 .. COUNT div SizeOf( Comp      )] of Comp      );
        18 : ( AsPointers    : array[ 0 .. COUNT div SizeOf( Pointer   )] of Pointer   );
    end;
    PDebugMem = ^TDebugMem;
  var
    DebugMem : PDebugMem;
  begin
    DebugMem := PDebugMem( aPointer);    //      set a watch for DebugMem^
    Nop;                                 // <--- and set a breakpoint here to view DebugMem^
  end;
{$HINTS ON}


  // ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////


  procedure InitializeUnit;
  var
    i : Integer;
  begin
    strIndent := ' ';

    while Length( strIndent) < 256
    do strIndent := strIndent + strIndent;

    AppLocale.CurrencyFormat    := FormatSettings.CurrencyFormat;
    AppLocale.NegCurrFormat     := FormatSettings.NegCurrFormat;
    AppLocale.ThousandSeparator := ',';
    AppLocale.DecimalSeparator  := '.';
    AppLocale.CurrencyDecimals  := FormatSettings.CurrencyDecimals;
    AppLocale.DateSeparator     := '-';
    AppLocale.TimeSeparator     := ':';
    AppLocale.ListSeparator     := ',';
    AppLocale.CurrencyString    := FormatSettings.CurrencyString;
    AppLocale.ShortDateFormat   := 'yyyy-MM-dd';
    AppLocale.LongDateFormat    := 'yyyy-MM-dd';
    AppLocale.TimeAMString      := FormatSettings.TimeAMString;
    AppLocale.TimePMString      := FormatSettings.TimePMString;
    AppLocale.ShortTimeFormat   := 'hh:mm';
    AppLocale.LongTimeFormat    := 'hh:mm:ss';

    for i := Low( FormatSettings.ShortMonthNames) to High( FormatSettings.ShortMonthNames) do AppLocale.ShortMonthNames[ i] := FormatSettings.ShortMonthNames[ i];
    for i := Low( FormatSettings.LongMonthNames ) to High( FormatSettings.LongMonthNames ) do AppLocale.LongMonthNames [ i] := FormatSettings.LongMonthNames [ i];
    for i := Low( FormatSettings.ShortDayNames  ) to High( FormatSettings.ShortDayNames  ) do AppLocale.ShortDayNames  [ i] := FormatSettings.ShortDayNames  [ i];
    for i := Low( FormatSettings.LongDayNames   ) to High( FormatSettings.LongDayNames   ) do AppLocale.LongDayNames   [ i] := FormatSettings.LongDayNames   [ i];

    AppLocale.TwoDigitYearCenturyWindow := FormatSettings.TwoDigitYearCenturyWindow ;

    FormatSettings.ThousandSeparator := AppLocale.ThousandSeparator;
    FormatSettings.DecimalSeparator  := AppLocale.DecimalSeparator;
    FormatSettings.DateSeparator     := AppLocale.DateSeparator;
    FormatSettings.TimeSeparator     := AppLocale.TimeSeparator;
    FormatSettings.ListSeparator     := AppLocale.ListSeparator;
    FormatSettings.ShortDateFormat   := AppLocale.ShortDateFormat;
    FormatSettings.LongDateFormat    := AppLocale.LongDateFormat;
    FormatSettings.ShortTimeFormat   := AppLocale.ShortTimeFormat;
    FormatSettings.LongTimeFormat    := AppLocale.LongTimeFormat;
  end;


initialization

  InitializeUnit;
  GNowLock    := TCriticalSection.Create;
  GNowHigh    := 0;
  GNowLastLow := 0;
  timeBeginPeriod( 1);

finalization

  timeEndPeriod( 1);
  FreeAndNil( GNowLock);

end.
