unit KnobsWorms;

{

  (C) COPYRIGHT 2000 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  WinApi.Windows, System.Math, System.SysUtils, System.Classes, Vcl.Graphics, Vcl.Controls, Vcl.StdCtrls, Vcl.ExtCtrls,
  System.Types, WinApi.Messages, System.UITypes,

  KnobsUtils, KnobsConversions;

type


  EKnobsWorm        = class( Exception );
  EKnobsInvalidFile = class( EKnobsWorm);
  EKnobsBounds      = class( EKnobsWorm);

  TKnobsWormData   = TSignalArray;
  TKnobsPercentage = 0 .. 100;

  TKnobsGene = class;

  TKnobsOnWormChanged = procedure( const aSender: TObject; const aGene: TKnobsGene)                    of object;
  TKnobsOnWormClicked = procedure( const aSender: TObject; const aGene: TKnobsGene)                    of object;
  TKnobsOnWormDropped = procedure( const aSender: TObject; const aGene: TKnobsGene)                    of object;
  TKnobsOnWormLog     = procedure( const aSender: TObject; const aMsg: string)                         of object;

  TKnobsWorm = class( TGraphicControl)
  private
    FUpdateLock           : Integer;
    FGene                 : TKnobsGene;
    FActive               : Boolean;
    FFrozen               : Boolean;
    FSelected             : Boolean;
    FClearance            : Byte;
    FBackColor            : TColor;
    FBorderColorNormal    : TColor;
    FBorderColorActive    : TColor;
    FBorderColorFrozen    : TColor;
    FBorderColorBoth      : TColor;
    FBorderColorSelected  : TColor;
    FPenColor             : TColor;
    FDrawLines            : Boolean;
    FDrawSoapy            : Boolean;
    FSoapSize             : Integer;
    FAllowDrop            : Boolean;
    FCanAcceptEmptyWorm   : Boolean;
    FOnActive             : TNotifyEvent;
    FOnFrozen             : TNotifyEvent;
    FOnSelected           : TNotifyEvent;
    FOnWormChanged        : TKnobsOnWormChanged;
    FOnWormClicked        : TKnobsOnWormClicked;
    FOnWormDropped        : TKnobsOnWormDropped;
    FOnLog                : TKnobsOnWormLog;
  private
    procedure   MutePeers;
    procedure   DeactivatePeers;
    procedure   DeselectPeers;
  private
    function    GetData( anIndex: Integer): TSignal;
    function    GetSize               : Integer;
    procedure   SetSize               ( aValue: Integer);
    procedure   SetActive             ( aValue: Boolean);
    procedure   SetFrozen             ( aValue: Boolean);
    procedure   SetSelected           ( aValue: Boolean);
    procedure   SetClearance          ( aValue: Byte   );
    procedure   SetBackColor          ( aValue: TColor );
    procedure   SetBorderColorNormal  ( aValue: TColor );
    procedure   SetBorderColorActive  ( aValue: TColor );
    procedure   SetBorderColorFrozen  ( aValue: TColor );
    procedure   SetBorderColorBoth    ( aValue: TColor );
    procedure   SetBorderColorSelected( aValue: TColor);
    procedure   SetPenColor           ( aValue: TColor );
    procedure   SetDrawLines          ( aValue: Boolean);
    procedure   SetDrawSoapy          ( aValue: Boolean);
    procedure   SetSoapSize           ( aValue: Integer);
  protected
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   DoLog( const aSender: TObject; const aMsg: string);
    procedure   PaintToCanvas( const aCanvas: TCanvas);
    procedure   Paint;                                                                                         override;
    procedure   Do_OnDragDrop ( Sender, Source: TObject; X, Y: Integer);
    procedure   Do_OnDragOver ( Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure   Do_OnEndDrag  ( Sender, Target: TObject; X, Y: Integer);
    procedure   Do_OnMouseDown( Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure   Changed;
  public
    constructor Create( anOwner: TComponent);                                                                  override;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   FillRandom;
    procedure   CopyFrom( const aParams: TKnobsWormData);                                                      overload;
    procedure   CopyFrom( const aGene  : TKnobsGene);                                                          overload;
    procedure   CopyFrom( const aWorm  : TKnobsWorm);                                                          overload;
    procedure   TriggerPaint;
  public
    property    Gene                   : TKnobsGene          read FGene;
    property    Data[ anIndex: Integer]: TSignal             read GetData; default;
  published
    property    Align;
    property    Size                   : Integer             read GetSize               write SetSize                default 120;
    property    Active                 : Boolean             read FActive               write SetActive              default False;
    property    Frozen                 : Boolean             read FFrozen               write SetFrozen              default False;
    property    Selected               : Boolean             read FSelected             write SetSelected            default False;
    property    Clearance              : Byte                read FClearance            write SetClearance           default 10;
    property    BackColor              : TColor              read FBackColor            write SetBackColor           default clWhite;
    property    BorderColorNormal      : TColor              read FBorderColorNormal    write SetBorderColorNormal   default clGray;
    property    BorderColorActive      : TColor              read FBorderColorActive    write SetBorderColorActive   default clYellow;
    property    BorderColorFrozen      : TColor              read FBorderColorFrozen    write SetBorderColorFrozen   default clBlue;
    property    BorderColorBoth        : TColor              read FBorderColorBoth      write SetBorderColorBoth     default clLime;
    property    BorderColorSelected    : TColor              read FBorderColorSelected  write SetBorderColorSelected default clWhite;
    property    PenColor               : TColor              read FPenColor             write SetPenColor            default clBlack;
    property    DrawLines              : Boolean             read FDrawLines            write SetDrawLines           default True;
    property    DrawSoapy              : Boolean             read FDrawSoapy            write SetDrawSoapy           default False;
    property    SoapSize               : Integer             read FSoapSize             write SetSoapSize            default 10;
    property    AllowDrop              : Boolean             read FAllowDrop            write FAllowDrop             default True;
    property    CanAcceptEmptyWorm     : Boolean             read FCanAcceptEmptyWorm   write FCanAcceptEmptyWorm    default False;
    property    Hint;
    property    ShowHint;
    property    ParentShowHint;
  published
    property    OnActive               : TNotifyEvent        read FOnActive             write FOnActive;
    property    OnFrozen               : TNotifyEvent        read FOnFrozen             write FOnFrozen;
    property    OnSelected             : TNotifyEvent        read FOnSelected           write FOnSelected;
    property    OnWormChanged          : TKnobsOnWormChanged read FOnWormChanged        write FOnWormChanged;
    property    OnWormClicked          : TKnobsOnWormClicked read FOnWormClicked        write FOnWormClicked;
    property    OnWormDropped          : TKnobsOnWormDropped read FOnWormDropped        write FOnWormDropped;
    property    OnLog                  : TKnobsOnWormLog     read FOnLog                write FOnLog;
  end;


  TKnobsWormPanel = class( TCustomPanel)
  private
    FWorms                : array of TKnobsWorm;
    FRowCount             : Integer;
    FColCount             : Integer;
    FSpacing              : Byte;
    FClearance            : Byte;
    FWormWidth            : Byte;
    FWormHeight           : Byte;
    FWormSize             : Integer;
    FBackColor            : TColor;
    FBorderColorNormal    : TColor;
    FBorderColorActive    : TColor;
    FBorderColorFrozen    : TColor;
    FBorderColorBoth      : TColor;
    FBorderColorSelected  : TColor;
    FPenColor             : TColor;
    FPosition             : TKnobsPercentage;
    FDrawLines            : Boolean;
    FDrawSoapy            : Boolean;
    FSoapSize             : Integer;
    FActiveWorm           : TKnobsWorm;
    FFrozenWorm           : TKnobsWorm;
    FSelectedWorm         : TKnobsWorm;
    FActive               : Integer;
    FFrozen               : Integer;
    FSelected             : Integer;
    FCanAcceptEmptyWorm   : Boolean;
    FOnActive             : TNotifyEvent;
    FOnFrozen             : TNotifyEvent;
    FOnSelected           : TNotifyEvent;
    FOnWormChanged        : TKnobsOnWormChanged;
    FOnWormClicked        : TKnobsOnWormClicked;
    FOnWormDropped        : TKnobsOnWormDropped;
    FOnLog                : TKnobsOnWormLog;
  private
    function    SubHint( anIndex: Integer): string;
    procedure   UpdateWorms;
    procedure   PlaceWorms;
    procedure   UpdateCount;
    procedure   AddWorm;
    procedure   DelWorm;
    procedure   DoActive  ( aSender: Tobject);
    procedure   DoFrozen  ( aSender: TObject);
    procedure   DoSelected( aSender: TObject);
    procedure   DoResize  ( aSender: TObject);
    procedure   DoWormChanged( const aSender: TObject; const aGene: TKnobsGene);
    procedure   DoWormClicked( const aSender: TObject; const aGene: TKnobsGene);
    procedure   DoWormDropped( const aSender: TObject; const aGene: TKnobsGene);
  private
    function    GetWorm( anIndex: Integer): TKnobsWorm;
    function    GetData( wIndex, dIndex: Integer): TSignal;
    function    GetHint: string;
    procedure   SetHint( const aValue: string);
    function    GetShowHint: Boolean;
    procedure   SetShowHint( aValue: Boolean);
    function    GetParentShowHint: Boolean;
    procedure   SetParentShowHint( aValue: Boolean);
    function    GetWormCount: Integer;
    procedure   SetRowCount            ( aValue : Integer         );
    procedure   SetColCount            ( aValue : Integer         );
    procedure   SetActive              ( aValue : Integer         );
    procedure   SetFrozen              ( aValue : Integer         );
    procedure   SetSelected            ( aValue : Integer         );
    procedure   SetSpacing             ( aValue : Byte            );
    procedure   SetClearance           ( aValue : Byte            );
    procedure   SetWormWidth           ( aValue : Byte            );
    procedure   SetWormHeight          ( aValue : Byte            );
    procedure   SetWormSize            ( aValue : Integer         );
    procedure   SetBackColor           ( aValue : TColor          );
    procedure   SetBorderColorNormal   ( aValue : TColor          );
    procedure   SetBorderColorActive   ( aValue : TColor          );
    procedure   SetBorderColorFrozen   ( aValue : TColor          );
    procedure   SetBorderColorBoth     ( aValue : TColor          );
    procedure   SetBorderColorSelected ( aValue : TColor          );
    procedure   SetPenColor            ( aValue : TColor          );
    procedure   SetPosition            ( aValue : TKnobsPercentage);
    procedure   SetDrawLines           ( aValue : Boolean         );
    procedure   SetDrawSoapy           ( aValue : Boolean         );
    procedure   SetSoapSize            ( aValue: Integer          );
    procedure   SetCanAcceptEmptyWorm  ( aValue : Boolean         );
  private
    procedure   RegisterWorm( const aWorm: TKnobsWorm);
    procedure   UnregisterWorm( const aWorm: TKnobsWorm);
  protected
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   DoLog( const aSender: TObject; const aMsg: string);
    procedure   Notification( aComponent: TComponent; anOperation: TOperation);                                    override;
    procedure   WMEraseBackground( var aMsg: TWMEraseBkgnd);                                          message WM_ERASEBKGND;
    procedure   DeactivateAll;
    procedure   UnfreezeAll;
    procedure   DeselectAll;
  public
    constructor Create( anOwner: TComponent);                                                                      override;
    procedure   Clear;
    procedure   FillRandom;
  public
    property    Worm[ anIndex       : Integer] : TKnobsWorm read GetWorm;                                           default;
    property    Data[ wIndex, dIndex: Integer] : TSignal    read GetData;
    property    ActiveWorm                     : TKnobsWorm read FActiveWorm;
    property    FrozenWorm                     : TKnobsWorm read FFrozenWorm;
    property    SelectedWorm                   : TKnobsWorm read FSelectedWorm;
    property    WormCount                      : Integer    read GetWormCount;
  published
    property    Align;
    property    BorderStyle;
    property    BorderWidth;
    property    BevelInner                                                                                          default bvLowered;
    property    BevelOuter                                                                                          default bvRaised;
    property    BevelWidth;
    property    Caption;
    property    Color;
    property    Enabled;
    property    Height                                                                                              default 167;
    property    ParentColor;
    property    PopupMenu;
    property    Hint                 : string              read GetHint               write SetHint;
    property    ShowHint             : Boolean             read GetShowHint           write SetShowHint;
    property    ParentShowHint       : Boolean             read GetParentShowHint     write SetParentShowHint;
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    DoubleBuffered;
    property    ParentDoubleBuffered;
    property    Width                                                                                               default 167;
    property    RowCount             : Integer             read FRowCount             write SetRowCount             default   2;
    property    ColCount             : Integer             read FColCount             write SetColCount             default   2;
    property    Active               : Integer             read FActive               write SetActive               default  -1;
    property    Frozen               : Integer             read FFrozen               write SetFrozen               default  -1;
    property    Selected             : Integer             read FSelected             write SetSelected             default  -1;
    property    Spacing              : Byte                read FSpacing              write SetSpacing              default   2;
    property    Clearance            : Byte                read FClearance            write SetClearance            default  10;
    property    WormWidth            : Byte                read FWormWidth            write SetWormWidth            default  80;
    property    WormHeight           : Byte                read FWormHeight           write SetWormHeight           default  80;
    property    WormSize             : Integer             read FWormSize             write SetWormSize             default 120;
    property    BackColor            : TColor              read FBackColor            write SetBackColor            default clWhite;
    property    BorderColorNormal    : TColor              read FBorderColorNormal    write SetBorderColorNormal    default clGray;
    property    BorderColorActive    : TColor              read FBorderColorActive    write SetBorderColorActive    default clYellow;
    property    BorderColorFrozen    : TColor              read FBorderColorFrozen    write SetBorderColorFrozen    default clBlue;
    property    BorderColorBoth      : TColor              read FBorderColorBoth      write SetBorderColorBoth      default clLime;
    property    BorderColorSelected  : TColor              read FBorderColorSelected  write SetBorderColorSelected  default clWhite;
    property    PenColor             : TColor              read FPenColor             write SetPenColor             default clBlack;
    property    Position             : TKnobsPercentage    read FPosition             write SetPosition             default   0;
    property    DrawLines            : Boolean             read FDrawLines            write SetDrawLines            default True;
    property    DrawSoapy            : Boolean             read FDrawSoapy            write SetDrawSoapy            default False;
    property    SoapSize             : Integer             read FSoapSize             write SetSoapSize             default 10;
    property    CanAcceptEmptyWorm   : Boolean             read FCanAcceptEmptyWorm   write SetCanAcceptEmptyWorm   default False;
  published
    property    OnActive             : TNotifyEvent        read FOnActive             write FOnActive;
    property    OnFrozen             : TNotifyEvent        read FOnFrozen             write FOnFrozen;
    property    OnSelected           : TNotifyEvent        read FOnSelected           write FOnSelected;
    property    OnWormChanged        : TKnobsOnWormChanged read FOnWormChanged        write FOnWormChanged;
    property    OnWormClicked        : TKnobsOnWormClicked read FOnWormClicked        write FOnWormClicked;
    property    OnWormDropped        : TKnobsOnWormDropped read FOnWormDropped        write FOnWormDropped;
    property    OnLog                : TKnobsOnWormLog     read FOnLog                write FOnLog;
  end;


  TKnobsGene = class
  private
    FData  : TKnobsWormData;
    FOnLog : TKnobsOnWormLog;
  private
    function    GetSize: Integer;
    procedure   SetSize( aValue: Integer);
    function    GetData( anIndex: Integer): TSignal;
    procedure   SetData( anIndex: Integer; aValue: TSignal);
  protected
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
  public
    constructor Create( aWorm: TKnobsWorm = nil);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
  public
    property    Size                    : Integer         read GetSize write SetSize;
    property    Data[ anIndex: Integer] : TSignal         read GetData write SetData;    default;
    property    OnLog                   : TKnobsOnWormLog read FOnLog  write FOnLog;
  end;



implementation


const

  SoapSize = 1 + 10;



{ ========
  TKnobsWorm = class( TGraphicControl)
  private
    FLockCount            : Integer;
    FGene                 : TKnobsGene;
    FActive               : Boolean;
    FFrozen               : Boolean;
    FSelected             : Boolean;
    FClearance            : Byte;
    FBackColor            : TColor;
    FBorderColorNormal    : TColor;
    FBorderColorActive    : TColor;
    FBorderColorFrozen    : TColor;
    FBorderColorBoth      : TColor;
    FBorderColorSelected  : TColor;
    FPenColor             : TColor;
    FDrawLines            : Boolean;
    FDrawSoapy            : Boolean;
    FSoapSize             : Integer;
    FAllowDrop            : Boolean;
    FCanAcceptEmptyWorm   : Boolean;
    FOnActive             : TNotifyEvent;
    FOnFrozen             : TNotifyEvent;
    FOnSelected           : TNotifyEvent;
    FOnWormChanged        : TKnobsOnWormChanged;
    FOnWormClicked        : TKnobsOnWormClicked;
    FOnWormDropped        : TKnobsOnWormDropped;
    FOnLog                : TKnobsOnWormLog;
  public
    property    Gene                   : TKnobsGene          read FGene;
    property    Data[ anIndex: Integer]: TSignal             read GetData; default;
  published
    property    Align;
    property    Size                   : Integer             read GetSize               write SetSize                default 120;
    property    Active                 : Boolean             read FActive               write SetActive              default False;
    property    Frozen                 : Boolean             read FFrozen               write SetFrozen              default False;
    property    Selected               : Boolean             read FSelected             write SetSelected            default False;
    property    Clearance              : Byte                read FClearance            write SetClearance           default 10;
    property    BackColor              : TColor              read FBackColor            write SetBackColor           default clWhite;
    property    BorderColorNormal      : TColor              read FBorderColorNormal    write SetBorderColorNormal   default clGray;
    property    BorderColorActive      : TColor              read FBorderColorActive    write SetBorderColorActive   default clYellow;
    property    BorderColorFrozen      : TColor              read FBorderColorFrozen    write SetBorderColorFrozen   default clBlue;
    property    BorderColorBoth        : TColor              read FBorderColorBoth      write SetBorderColorBoth     default clLime;
    property    BorderColorSelected    : TColor              read FBorderColorSelected  write SetBorderColorSelected default clWhite;
    property    PenColor               : TColor              read FPenColor             write SetPenColor            default clBlack;
    property    DrawLines              : Boolean             read FDrawLines            write SetDrawLines           default True;
    property    DrawSoapy              : Boolean             read FDrawSoapy            write SetDrawSoapy           default False;
    property    SoapSize               : Integer             read FSoapSize             write SetSoapSize            default 10;
    property    AllowDrop              : Boolean             read FAllowDrop            write FAllowDrop             default True;
    property    CanAcceptEmptyWorm     : Boolean             read FCanAcceptEmptyWorm   write FCanAcceptEmptyWorm    default False;
    property    Hint;
    property    ShowHint;
    property    ParentShowHint;
  published
    property    OnActive               : TNotifyEvent        read FOnActive             write FOnActive;
    property    OnFrozen               : TNotifyEvent        read FOnFrozen             write FOnFrozen;
    property    OnSelected             : TNotifyEvent        read FOnSelected           write FOnSelected;
    property    OnWormChanged          : TKnobsOnWormChanged read FOnWormChanged        write FOnWormChanged;
    property    OnWormClicked          : TKnobsOnWormClicked read FOnWormClicked        write FOnWormClicked;
    property    OnWormDropped          : TKnobsOnWormDropped read FOnWormDropped        write FOnWormDropped;
    property    OnLog                  : TKnobsOnWormLog     read FOnLog                write FOnLog;
  private
 }

    procedure   TKnobsWorm.MutePeers;
    var
      i : Integer;
    begin
      if Assigned( Parent) and ( Parent is TKnobsWormPanel)
      then begin
        with TKnobsWormPanel( Parent)
        do begin
          for i := 0 to WormCount - 1
          do Worm[ i].Active := False;
        end;
      end;
    end;


    procedure   TKnobsWorm.DeactivatePeers;
    var
      i : Integer;
    begin
      if Assigned( Parent) and ( Parent is TKnobsWormPanel)
      then begin
        With TKnobsWormPanel( Parent)
        do begin
          for i := 0 to WormCount - 1
          do Worm[ i].Frozen := False;
        end;
      end;
    end;


    procedure   TKnobsWorm.DeselectPeers;
    var
      i : Integer;
    begin
      if Assigned( Parent) and ( Parent is TKnobsWormPanel)
      then begin
        With TKnobsWormPanel( Parent)
        do begin
          for i := 0 to WormCount - 1
          do Worm[ i].Selected := False;
        end;
      end;
    end;


    function    TKnobsWorm.GetSize: Integer;
    begin
      Result := FGene.Size;
    end;


    procedure   TKnobsWorm.SetSize( aValue: Integer);
    begin
      if aValue <> FGene.Size
      then begin
        FGene.Size := aValue;

        if csDesigning in ComponentState
        then Changed;
      end;
    end;


    function    TKnobsWorm.GetData( anIndex: Integer): TSignal;
    begin
      Result := FGene[ anIndex];
    end;


    procedure   TKnobsWorm.SetActive( aValue: Boolean);
    begin
      if FActive <> aValue
      then begin
        if aValue
        then MutePeers;

        FActive := aValue;

        if FActive and Assigned( FOnActive)
        then FOnActive( Self);

        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetFrozen( aValue: Boolean);
    begin
      if FFrozen <> aValue
      then Begin
        if aValue
        then DeactivatePeers;

        FFrozen := aValue;

        if FFrozen and Assigned( FOnFrozen)
        then FOnFrozen( Self);

        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetSelected( aValue: Boolean);
    begin
      if FSelected <> aValue
      then Begin
        // LogFmt( 'SetSelected( %s)', [ BoolToStr( aValue, True)]);

        if aValue
        then DeselectPeers;

        FSelected := aValue;

        if FSelected and Assigned( FOnSelected)
        then FOnSelected( Self);

        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetClearance;
    begin
      if aValue <> FClearance
      then begin
        FClearance := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBackColor( aValue: TColor);
    begin
      if aValue <> FBackColor
      then begin
        FBackColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBorderColorNormal( aValue: TColor);
    begin
      if aValue <> FBorderColorNormal
      then begin
        FBorderColorNormal := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBorderColorActive( aValue: TColor);
    begin
      if aValue <> FBorderColorActive
      then begin
        FBorderColorActive := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBorderColorFrozen( aValue: TColor);
    begin
      if aValue <> FBorderColorFrozen
      then begin
        FBorderColorFrozen := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBorderColorBoth( aValue: TColor);
    begin
      if aValue <> FBorderColorBoth
      then begin
        FBorderColorBoth := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetBorderColorSelected( aValue: TColor);
    begin
      if aValue <> FBorderColorSelected
      then begin
        FBorderColorSelected := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetPenColor( aValue: TColor);
    begin
      if aValue <> FPenColor
      then begin
        FPenColor := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetDrawLines( aValue: Boolean);
    begin
      if aValue <> FDrawLines
      then begin
        FDrawLines := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetDrawSoapy( aValue: Boolean);
    begin
      if aValue <> FDrawSoapy
      then begin
        FDrawSoapy := aValue;
        Invalidate;
      end;
    end;


    procedure   TKnobsWorm.SetSoapSize( aValue: Integer);
    begin
      if aValue <> FSoapSize
      then begin
        FSoapSize := aValue;
        Invalidate;
      end;
    end;


//   Protected

    procedure   TKnobsWorm.Log( const aMsg: string);
    var
      S : string;
    begin
      S := Format( 'TKnobsWorm[ %d, %s]: %s', [ Tag, Name, aMsg], AppLocale);

      if Assigned( FOnLog)
      then FOnLog( Self, S)
      else begin
        if assigned( Parent) and ( Parent is TKnobsWormPanel)
        then TKnobsWormPanel( Parent).DoLog( Self, S);
      end;
    end;


    procedure   TKnobsWorm.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TKnobsWorm.DoLog( const aSender: TObject; const aMsg: string);
    begin
      Log( aMsg);
    end;


    procedure   TKnobsWorm.PaintToCanvas( const aCanvas: TCanvas);
    type
      TP = record
        X, Y : TSignal;
      end;

    var
      Pts  : array of TP;
      maxx : TSignal;
      maxy : TSignal;
      minx : TSignal;
      miny : TSignal;

      function P( anX, anY: TSignal): TP;
      begin
        Result.X := anX;
        Result.y := anY;
      end;

      procedure ScalePts;
      var
        dx : TSignal;
        dy : TSignal;
        i  : Integer;
        d  : Integer;
      begin
        dx := maxx - minx + 1;

        if dx < 1
        then dx := 1;

        dy := maxy - miny + 1;

        if dy < 1
        then dy := 1;

        if DrawLines
        then d := 1
        else if DrawSoapy
        then d := SoapSize - 1
        else d := 2;

        for i := 0 to Size
        do begin
          Pts[ i].X := d + FClearance + ( Width  - 2 * FClearance - 2 * d) * (( Pts[ i].X - minx) / dx);
          Pts[ i].Y := d + FClearance + ( Height - 2 * FClearance - 2 * d) * (( Pts[ i].Y - miny) / dy);
        end;
      end;

      procedure CalcPts;
      var
        i  : Integer;
        nx : TSignal;
        ny : TSignal;
      begin
        Pts[ 0] := P( 0, 0);
        maxx    := 0;
        maxy    := 0;
        minx    := 0;
        miny    := 0;

        for i := 0 to Size - 1
        do begin
          nx := Pts[ i].X + LookupCosine( FGene[ i]);
          ny := Pts[ i].Y + LookupSine  ( FGene[ i]);

          if nx > maxx
          then maxx := nx;

          if nx < minx
          then minx := nx;

          if ny > maxy
          then maxy := ny;

          if ny < miny
          then miny := ny;

          Pts[ i + 1] := P( nx, ny);
        end;
      end;

    var               // Act  // Frozen
      Colors : array[ Boolean, Boolean] of TColor;
      i      : Integer;
      d      : Integer;
    begin
      Colors[ false, false] := BorderColorNormal;
      Colors[ false, true ] := BorderColorFrozen;
      Colors[ true , false] := BorderColorActive;
      Colors[ true , true ] := BorderColorBoth;

      SetLength( Pts, Size + 1);

      CalcPts;
      ScalePts;

      with aCanvas
      do begin
        Brush.Color := BackColor;

        if Selected
        then Pen.Color := BorderColorSelected
        else Pen.Color := Colors[ Active, Frozen];

        Pen.Width := 2;

        Rectangle( 1, 1, Width, Height);
        Pen.Color := PenColor;
        Pen.Width := 1;

        if FDrawLines
        then Begin
          MoveTo( Round( Pts[ 0].X), Round( Pts[ 0].Y));

          for i := 1 to Size
          do LineTo( Round( Pts[ i].X), Round( Pts[ i].Y));
        end
        else begin
          for i := 1 to Size - 1
          do begin
            if DrawSoapy
            then d := ( i + 371) mod SoapSize
            else d := 2;

            Ellipse(
              Round( Pts[ i].X) - d,
              Round( Pts[ i].Y) - d,
              Round( Pts[ i].X) + d,
              Round( Pts[ i].Y) + d
            );
          end;
        end;
      end;
    end;


    procedure   TKnobsWorm.Paint; // Override;
    var
      aBitmap: TBitmap;
    begin
      aBitmap := TBitmap.Create;

      try
        aBitmap.Width  := Width;
        aBitmap.Height := Height;
        PaintToCanvas( aBitmap.Canvas);

        Canvas.Draw( 0, 0, aBitmap);
      finally
        aBitmap.DisposeOf;
      end;
    end;


    procedure   TKnobsWorm.Do_OnDragDrop( Sender, Source: TObject; X, Y: Integer);
    begin
      Selected := True;

      if Source is TKnobsWorm
      then begin
        CopyFrom( TKnobsWorm( Source));

        if Assigned( FOnWormDropped)
        then FOnWormDropped( Self, Gene)
      end;
    end;


    procedure   TKnobsWorm.Do_OnDragOver( Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    begin
      Accept :=
        AllowDrop and
        (
          (
            ( Source is TKnobsWorm) and
            (( TKnobsWorm( Source).Size <> 0) or CanAcceptEmptyWorm) and
            ( Source <> Self)
          )
        )
    end;


    procedure   TKnobsWorm.Do_OnEndDrag( Sender, Target: TObject; X, Y: Integer);
    begin
      try
        if not Assigned( Target) and ( Sender is TKnobsWorm) and assigned( FOnWormClicked)
        then FOnWormClicked( Self, Gene);
      finally
        Unlock( FUpdateLock);
      end;
    end;


    procedure   TKnobsWorm.Do_OnMouseDown( Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    begin
      if Button = mbLeft
      then begin
        Selected := True;

        Lock( FUpdateLock);
        BeginDrag( False, -1);
      end;
    end;


    procedure   TKnobsWorm.Changed;
    begin
      if Assigned( FOnWormChanged) and not ( csDestroying in ComponentState) and not Locked( FUpdateLock)
      then begin
        Lock( FUpdateLock);

        try
          FOnWormChanged( Self, Gene);
        finally
          Unlock( FUpdateLock);
        end;
      end;

      Invalidate;
    end;


//  Public

    constructor TKnobsWorm.Create( anOwner: TComponent); // Override;
    begin
      inherited;
      FGene                := TKnobsGene.Create;
      Size                 := 120;
      Active               := False;
      Frozen               := False;
      Selected             := False;
      Clearance            := 10;
      BackColor            := clWhite;
      BorderColorNormal    := clGray;
      BorderColorActive    := clYellow;
      BorderColorFrozen    := clBlue;
      BorderColorBoth      := clLime;
      BorderColorSelected  := clWhite;
      PenColor             := clBlack;
      Width                := 80;
      Height               := 80;
      OnDragDrop           := Do_OnDragDrop;
      OnDragOver           := Do_OnDragOver;
      OnEndDrag            := Do_OnEndDrag;
      OnMouseDown          := Do_OnMouseDown;
      DragMode             := dmManual;
      DrawLines            := True;
      DrawSoapy            := False;
      SoapSize             := 10;
      AllowDrop            := True;
      CanAcceptEmptyWorm   := False;

      if csDesigning in ComponentState
      then FillRandom;
    end;


    destructor  TKnobsWorm.Destroy; // Override;
    begin
      FOnActive      := nil;
      FOnFrozen      := nil;
      FOnSelected    := nil;
      FOnWormChanged := nil;
      FOnWormClicked := nil;
      FOnWormDropped := nil;
      Size           := 0;
      FGene.DisposeOf;
      inherited;
    end;


    procedure   TKnobsWorm.Clear;
    begin
      if Size <> 0
      then begin
        FGene.Clear;
        Changed;
      end;
    end;


    procedure   TKnobsWorm.FillRandom;
    var
      i : Integer;
    begin
      for i := 0 to Size - 1
      do FGene[ i] := Random;

      Changed;
    end;


    procedure   TKnobsWorm.CopyFrom( const aParams: TKnobsWormData); // overload;
    begin
      if not Locked( FUpdateLock)
      then begin
        Size := Length( aParams);
        Move( aParams[ 0], FGene.FData[ 0], Size * SizeOf( TSignal));
        Changed;
      end;
    end;


    procedure   TKnobsWorm.CopyFrom( const aGene: TKnobsGene);
    begin
      if Assigned( aGene)
      then CopyFrom( aGene.FData);
    end;


    procedure   TKnobsWorm.CopyFrom( const aWorm: TKnobsWorm);
    begin
      if Assigned( aWorm)
      then CopyFrom( aWorm.Gene);
    end;


    procedure   TKnobsWorm.TriggerPaint;
    begin
      Invalidate;
    end;


{ ========
  TWormPanel = class( TPanel)
  private
    FWorms                : array of TKnobsWorm;
    FRowCount             : Integer;
    FColCount             : Integer;
    FSpacing              : Byte;
    FClearance            : Byte;
    FWormWidth            : Byte;
    FWormHeight           : Byte;
    FWormSize             : Integer;
    FBackColor            : TColor;
    FBorderColorNormal    : TColor;
    FBorderColorActive    : TColor;
    FBorderColorFrozen    : TColor;
    FBorderColorBoth      : TColor;
    FBorderColorSelected  : TColor;
    FPenColor             : TColor;
    FPosition             : TKnobsPercentage;
    FDrawLines            : Boolean;
    FDrawSoapy            : Boolean;
    FSoapSize             : Integer;
    FActiveWorm           : TKnobsWorm;
    FFrozenWorm           : TKnobsWorm;
    FSelectedWorm         : TKnobsWorm;
    FActive               : Integer;
    FFrozen               : Integer;
    FSelected             : Integer;
    FCanAcceptEmptyWorm   : Boolean;
    FOnActive             : TNotifyEvent;
    FOnFrozen             : TNotifyEvent;
    FOnSelected           : TNotifyEvent;
    FOnWormChanged        : TKnobsOnWormChanged;
    FOnWormClicked        : TKnobsOnWormClicked;
    FOnWormDropped        : TKnobsOnWormDropped;
    FOnLog                : TKnobsOnWormLog;
  public
    property    Worm[ anIndex       : Integer] : TKnobsWorm read GetWorm;                                           default;
    property    Data[ wIndex, dIndex: Integer] : TSignal    read GetData;
    property    ActiveWorm                     : TKnobsWorm read FActiveWorm;
    property    FrozenWorm                     : TKnobsWorm read FFrozenWorm;
    property    SelectedWorm                   : TKnobsWorm read FSelectedWorm;
    property    WormCount                      : Integer    read GetWormCount;
  published
    property    Worm[ anIndex       : Integer] : TKnobsWorm read GetWorm;                                           default;
    property    Data[ wIndex, dIndex: Integer] : TSignal    read GetData;
    property    ActiveWorm                     : TKnobsWorm read FActiveWorm;
    property    FrozenWorm                     : TKnobsWorm read FFrozenWorm;
    property    SelectedWorm                   : TKnobsWorm read FSelectedWorm;
    property    WormCount                      : Integer    read GetWormCount;
  published
    property    Align;
    property    BorderStyle;
    property    BorderWidth;
    property    BevelInner                                                                                          default bvLowered;
    property    BevelOuter                                                                                          default bvRaised;
    property    BevelWidth;
    property    Caption;
    property    Color;
    property    Enabled;
    property    Height                                                                                              default 167;
    property    ParentColor;
    property    ParentShowHint;
    property    PopupMenu;
    property    ShowHint;
    property    TabOrder;
    property    TabStop;
    property    Visible;
    property    DoubleBuffered;
    property    ParentDoubleBuffered;
    property    Width                                                                                               default 167;
    property    RowCount             : Integer             read FRowCount             write SetRowCount             default   2;
    property    ColCount             : Integer             read FColCount             write SetColCount             default   2;
    property    Active               : Integer             read FActive               write SetActive               default  -1;
    property    Frozen               : Integer             read FFrozen               write SetFrozen               default  -1;
    property    Selected             : Integer             read FSelected             write SetSelected             default  -1;
    property    Spacing              : Byte                read FSpacing              write SetSpacing              default   2;
    property    Clearance            : Byte                read FClearance            write SetClearance            default  10;
    property    WormWidth            : Byte                read FWormWidth            write SetWormWidth            default  80;
    property    WormHeight           : Byte                read FWormHeight           write SetWormHeight           default  80;
    property    WormSize             : Integer             read FWormSize             write SetWormSize             default 120;
    property    BackColor            : TColor              read FBackColor            write SetBackColor            default clWhite;
    property    BorderColorNormal    : TColor              read FBorderColorNormal    write SetBorderColorNormal    default clGray;
    property    BorderColorActive    : TColor              read FBorderColorActive    write SetBorderColorActive    default clYellow;
    property    BorderColorFrozen    : TColor              read FBorderColorFrozen    write SetBorderColorFrozen    default clBlue;
    property    BorderColorBoth      : TColor              read FBorderColorBoth      write SetBorderColorBoth      default clLime;
    property    BorderColorSelected  : TColor              read FBorderColorSelected  write SetBorderColorSelected  default clWhite;
    property    PenColor             : TColor              read FPenColor             write SetPenColor             default clBlack;
    property    Position             : TKnobsPercentage    read FPosition             write SetPosition             default   0;
    property    DrawLines            : Boolean             read FDrawLines            write SetDrawLines            default True;
    property    DrawSoapy            : Boolean             read FDrawSoapy            write SetDrawSoapy            default False;
    property    SoapSize             : Integer             read FSoapSize             write SetSoapSize             default 10;
    property    CanAcceptEmptyWorm   : Boolean             read FCanAcceptEmptyWorm   write SetCanAcceptEmptyWorm   default False;
  published
    property    OnActive             : TNotifyEvent        read FOnActive             write FOnActive;
    property    OnFrozen             : TNotifyEvent        read FOnFrozen             write FOnFrozen;
    property    OnSelected           : TNotifyEvent        read FOnSelected           write FOnSelected;
    property    OnWormChanged        : TKnobsOnWormChanged read FOnWormChanged        write FOnWormChanged;
    property    OnWormClicked        : TKnobsOnWormClicked read FOnWormClicked        write FOnWormClicked;
    property    OnWormDropped        : TKnobsOnWormDropped read FOnWormDropped        write FOnWormDropped;
    property    OnLog                : TKnobsOnWormLog     read FOnLog                write FOnLog;
  private
}

    function    TKnobsWormPanel.SubHint( anIndex: Integer): string;
    begin
      Result := Format( '%s %d', [ Hint, anIndex + 1], AppLocale);
    end;


    procedure   TKnobsWormPanel.UpdateWorms;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do begin
        with Worm[ i]
        do begin
          Clearance            := Self.Clearance;
          BackColor            := Self.BackColor;
          BorderColorNormal    := Self.BorderColorNormal;
          BorderColorActive    := Self.BorderColorActive;
          BorderColorFrozen    := Self.BorderColorFrozen;
          BorderColorBoth      := Self.BorderColorBoth;
          BorderColorSelected  := Self.BorderColorSelected;
          PenColor             := Self.PenColor;
          DrawLines            := Self.DrawLines;
          DrawSoapy            := Self.DrawSoapy;
          SoapSize             := Self.SoapSize;
          CanAcceptEmptyWorm   := Self.CanAcceptEmptyWorm;
          Width                := WormWidth;
          Height               := WormHeight;
          Hint                 := SubHint( i);
          ShowHint             := Self.ShowHint;
          ParentShowHint       := Self.ParentShowHint;
        end;
      end;
    end;


    procedure   TKnobsWormPanel.PlaceWorms;
    var
      i      : Integer;
      CurCol : Integer;
      CurRow : Integer;
    begin
      Width  := ColCount * ( WormWidth  + Spacing) + Spacing;
      Height := RowCount * ( WormHeight + Spacing) + Spacing;
      CurCol := 0;
      CurRow := 0;

      for i := 0 to RowCount * ColCount - 1
      do begin
        with Worm[ i]
        do begin
          Left := Spacing + ( CurCol * ( Width  + Spacing));
          Top  := Spacing + ( CurRow * ( Height + Spacing));
          Inc( CurCol);

          if CurCol >= ColCount
          then begin
            CurCol := 0;
            Inc( CurRow);
          end;
        end;
      end;
    end;


    procedure   TKnobsWormPanel.AddWorm;
    begin
      TKnobsWorm.Create( Self).Parent := Self;
      UpdateWorms;
    end;


    procedure   TKnobsWormPanel.DelWorm;
    begin
      Worm[ WormCount - 1].DisposeOf;
    end;


    procedure   TKnobsWormPanel.UpdateCount;
    var
      aCount : Integer;
    begin
      aCount := FRowCount * FColCount;

      while WormCount < aCount
      do AddWorm;

      while WormCount > aCount
      do DelWorm;

      PlaceWorms;
    end;


    procedure   TKnobsWormPanel.DoActive( aSender: Tobject);
    begin
      if aSender is TKnobsWorm
      then begin
        FActiveWorm := TKnobsWorm( aSender);
        FActive     := TKnobsWorm( aSender).Tag;

        if Assigned( FOnActive)
        then FOnActive( Self);

        Changed;
      end;
    end;


    procedure   TKnobsWormPanel.DoFrozen( aSender: TObject);
    begin
      if aSender is TKnobsWorm
      then begin
        FFrozenWorm := TKnobsWorm( aSender);
        FFrozen     := TKnobsWorm( aSender).Tag;

        if Assigned( FOnFrozen)
        then FOnFrozen( Self);
      end;
    end;


    procedure   TKnobsWormPanel.DoSelected( aSender: TObject);
    begin
      if aSender is TKnobsWorm
      then begin
        FSelectedWorm := TKnobsWorm( aSender);
        FSelected     := TKnobsWorm( aSender).Tag;

        if Assigned( FOnSelected)
        then FOnSelected( Self);
      end;
    end;


    procedure   TKnobsWormPanel.DoResize( aSender: TObject);
    begin
      PlaceWorms;
    end;


    procedure   TKnobsWormPanel.DoWormChanged( const aSender: TObject; const aGene: TKnobsGene);
    begin
      if Assigned( FOnWormChanged)
      then FOnWormChanged( Self, aGene);
    end;


    procedure   TKnobsWormPanel.DoWormClicked( const aSender: TObject; const aGene: TKnobsGene);
    begin
      if Assigned( FOnWormClicked)
      then FOnWormClicked( Self, aGene);
    end;


    procedure   TKnobsWormPanel.DoWormDropped( const aSender: TObject; const aGene: TKnobsGene);
    begin
      if Assigned( FOnWormDropped)
      then FOnWormDropped( Self, aGene);
    end;


    function    TKnobsWormPanel.GetWorm( anIndex: Integer): TKnobsWorm;
    begin
      if ( anIndex >= 0) and ( anIndex < WormCount)
      then Result := TKnobsWorm( FWorms[ anIndex])
      else Result := nil;
    end;


    function    TKnobsWormPanel.GetData( wIndex, dIndex: Integer): TSignal;
    begin
      if ( wIndex >= 0) and ( wIndex < WormCount)
      then Result := Worm[ wIndex][ dIndex]
      else Result := 0;
    end;


    function    TKnobsWormPanel.GetHint: string;
    begin
      Result := inherited Hint;
    end;


    procedure   TKnobsWormPanel.SetHint( const aValue: string);
    var
      i : Integer;
    begin
      inherited Hint := aValue;

      for i := 0 to WormCount - 1
      do Worm[ i].Hint := SubHint( i);
    end;


    function    TKnobsWormPanel.GetShowHint: Boolean;
    begin
      result := inherited ShowHint;
    end;


    procedure   TKnobsWormPanel.SetShowHint( aValue: Boolean);
    var
      i : Integer;
    begin
      inherited ShowHint := aValue;

      for i := 0 to WormCount - 1
      do Worm[ i].ShowHint := aValue;
    end;


    function    TKnobsWormPanel.GetParentShowHint: Boolean;
    begin
      Result := inherited ParentShowHint;
    end;


    procedure   TKnobsWormPanel.SetParentShowHint( aValue: Boolean);
    var
      i : Integer;
    begin
      inherited ParentShowHint := aValue;

      for i := 0 to WormCount - 1
      do Worm[ i].ParentShowHint := aValue;
    end;


    function    TKnobsWormPanel.GetWormCount: Integer;
    begin
      Result := Length( FWorms);
    end;


    procedure   TKnobsWormPanel.SetRowCount( aValue : Integer);
    begin
      if aValue <> FRowCount
      then begin
        FRowCount := aValue;
        UpdateCount;
      end;
    end;


    procedure   TKnobsWormPanel.SetColCount( aValue : Integer);
    begin
      if aValue <> FColCount
      then begin
        FColCount := aValue;
        UpdateCount;
      end;
    end;


    procedure   TKnobsWormPanel.SetActive( aValue : Integer);
    begin
      if ( aValue >= 0) and ( aValue < WormCount) and ( aValue <> Active)
      then Worm[ aValue].Active := True
      else DeactivateAll;

      FActive := aValue;
    end;


    procedure   TKnobsWormPanel.SetFrozen( aValue : Integer);
    begin
      if ( aValue >= 0) and ( aValue < WormCount) and ( aValue <> Frozen)
      then Worm[ aValue].Frozen := True
      else UnfreezeAll;

      FFrozen := aValue;
    end;


    procedure   TKnobsWormPanel.SetSelected( aValue : Integer);
    begin
      if ( aValue >= 0) and ( aValue < WormCount) and ( aValue <> Selected)
      then begin
        FSelected := aValue;
        Worm[ aValue].Selected := True;

        if Assigned( FOnWormClicked)
        then FOnWormClicked( Self, Worm[ aValue].Gene);
      end
      else begin
        FSelected := aValue;

        if aValue < 0
        then DeselectAll;
      end;
    end;


    procedure   TKnobsWormPanel.SetSpacing( aValue : Byte);
    begin
      if aValue <> FSpacing
      then begin
        FSpacing := aValue;
        PlaceWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetClearance( aValue : Byte);
    begin
      if aValue <> FClearance
      then begin
        FClearance := aValue;
        UpdateWorms;
        PlaceWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetWormWidth( aValue : Byte);
    begin
      if aValue <> FWormWidth
      then begin
        FWormWidth := aValue;
        UpdateWorms;
        PlaceWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetWormHeight( aValue : Byte);
    begin
      if aValue <> FWormHeight
      then begin
        FWormHeight := aValue;
        UpdateWorms;
        PlaceWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetWormSize( aValue : Integer);
    var
      i : Integer;
    begin
      if aValue <> FWormSize
      then begin
        FWormSize := aValue;

        for i := 0 to WormCount - 1
        do Worm[ i].Size := FWormSize;
      end;
    end;


    procedure   TKnobsWormPanel.SetBackColor( aValue : TColor);
    begin
      if aValue <> FBackColor
      then begin
        FBackColor := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetBorderColorNormal( aValue : TColor);
    begin
      if aValue <> FBorderColorNormal
      then begin
        FBorderColorNormal := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetBorderColorActive( aValue : TColor);
    begin
      if aValue <> FBorderColorActive
      then begin
        FBorderColorActive := aValue;
        UpdateWorms;
      End;
    end;


    procedure   TKnobsWormPanel.SetBorderColorFrozen( aValue : TColor);
    begin
      If aValue <> FBorderColorFrozen
      then begin
        FBorderColorFrozen := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetBorderColorBoth( aValue : TColor);
    begin
      if aValue <> FBorderColorBoth
      then begin
        FBorderColorBoth := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetBorderColorSelected( aValue : TColor);
    begin
      if aValue <> FBorderColorSelected
      then begin
        FBorderColorSelected := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetPenColor( aValue : TColor);
    begin
      if aValue <> FPenColor
      then begin
        FPenColor := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetPosition( aValue : TKnobsPercentage);
    begin
      if aValue <> FPosition
      then begin
        FPosition := aValue;
        Selected  := Round(( WormCount - 1) * aValue / 100);
      end;
    end;


    procedure   TKnobsWormPanel.SetDrawLines( aValue : Boolean);
    begin
      if aValue <> FDrawLines
      then begin
        FDrawLines := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetDrawSoapy( aValue : Boolean);
    begin
      if aValue <> FDrawSoapy
      then begin
        FDrawSoapy := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetSoapSize( aValue : Integer);
    begin
      if aValue <> FSoapSize
      then begin
        FSoapSize := aValue;
        UpdateWorms;
      end;
    end;


    procedure   TKnobsWormPanel.SetCanAcceptEmptyWorm( aValue : Boolean);
    begin
      if aValue <> FCanAcceptEmptyWorm
      then begin
        FCanAcceptEmptyWorm := aValue;
        UpdateWorms;
      end;
    end;


//  private

    procedure   TKnobsWormPanel.RegisterWorm( const aWorm: TKnobsWorm);
    begin
      SetLength( FWorms, WormCount + 1);
      FWorms[ WormCount - 1] := aWorm;
    end;


    procedure   TKnobsWormPanel.UnregisterWorm( const aWorm: TKnobsWorm);
    var
      p : Integer;
      i : Integer;
    begin
      p := -1;

      for i := 0 to WormCount - 1
      do begin
        if FWorms[ i] = aWorm
        then begin
          p := i;
          Break;
        end;
      end;

      if p >= 0
      then begin
        for i := p to WormCount - 2
        do FWorms[ i] := FWorms[ i + 1];

        SetLength( FWorms, WormCount - 1);
      end;
    end;


// protected

    procedure   TKnobsWormPanel.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( Self, Format( 'TKnobsWormPanel[ %s]: %s', [ Name, aMsg], AppLocale))
    end;


    procedure   TKnobsWormPanel.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TKnobsWormPanel.DoLog( const aSender: TObject; const aMsg: string);
    begin
      Log( aMsg);
    end;


    procedure   TKnobsWormPanel.Notification( aComponent: TComponent; anOperation: TOperation); // Override;
    var
      i : Integer;
    begin
      if aComponent is TKnobsWorm
      then begin
        if anOperation = opInsert
        then begin
          RegisterWorm( TKnobsWorm( aComponent));

          for i := 0 to WormCount - 1
          do Worm[ i].Tag := i;

          with TKnobsWorm( aComponent)
          do begin
            OnActive      := DoActive;
            OnFrozen      := DoFrozen;
            OnSelected    := DoSelected;;
            OnWormChanged := DoWormChanged;
            OnWormClicked := DoWormClicked;
            OnWormDropped := DoWormDropped;
          end;
        end
        else if anOperation = opRemove
        then begin
          with TKnobsWorm( aComponent)
          do begin
            OnActive      := nil;
            OnFrozen      := nil;
            OnSelected    := nil;
            OnWormChanged := nil;
            OnWormClicked := nil;
            OnWormDropped := nil;
          end;

          UnregisterWorm( TKnobsWorm( aComponent));

          for i := 0 to WormCount - 1
          do Worm[ i].Tag := i;
        end;
      end;

      inherited;
    end;


    procedure   TKnobsWormPanel.WMEraseBackground( var aMsg: TWMEraseBkgnd); // message WM_ERASEBKGND;
    begin
      if aMsg.DC <> 0
      then Canvas.Handle := aMsg.DC;

      try
        Canvas.Pen.Color   := Color;
        Canvas.Brush.Color := Color;
        Canvas.Rectangle( 0, 0, Width, Height);
      finally
        if aMsg.DC <> 0
        then Canvas.Handle := 0;
      end;

      aMsg.Result := 0;
    end;


    procedure   TKnobsWormPanel.DeactivateAll;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do Worm[ i].Active := False;
    end;


    procedure   TKnobsWormPanel.UnfreezeAll;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do Worm[ i].Frozen := False;
    end;


    procedure   TKnobsWormPanel.DeselectAll;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do Worm[ i].Selected := False;
    end;


//  Public

    constructor TKnobsWormPanel.Create( anOwner: TComponent); // Override;
    begin
      inherited;
      OnResize             := DoResize;
      DoubleBuffered       := True;
      BevelInner           := bvLowered;
      BevelOuter           := bvRaised;
      Caption              := '.';
      Height               := 167;
      Width                := 167;
      Spacing              :=   2;
      Clearance            :=  10;
      WormWidth            :=  80;
      WormHeight           :=  80;
      WormSize             := 120;
      BackColor            := clWhite;
      BorderColorNormal    := clGray;
      BorderColorActive    := clYellow;
      BorderColorFrozen    := clBlue;
      BorderColorBoth      := clLime;
      BorderColorSelected  := clWhite;
      PenColor             := clBlack;
      Position             :=   0;
      DrawLines            := True;
      DrawSoapy            := False;
      SoapSize             := 10;
      Active               :=  -1;
      Frozen               :=  -1;
      Selected             :=  -1;
      RowCount             :=   2;
      ColCount             :=   2;
    end;


    procedure   TKnobsWormPanel.Clear;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do Worm[ i].Clear;
    end;


    procedure   TKnobsWormPanel.FillRandom;
    var
      i : Integer;
    begin
      for i := 0 to WormCount - 1
      do begin
        if ( i <> Active) and ( i <> Frozen)
        then Worm[ i].FillRandom;
      end;
    end;


{========
  TGene = class( TObject)
  private
    FData  : TKnobsWormData;
    FOnLog : TKnobsOnWormLog;
  public
    property    Size                    : Integer         read GetSize write SetSize;
    property    Data[ anIndex: Integer] : TSignal         read GetData write SetData;    default;
    property    OnLog                   : TKnobsOnWormLog read FOnLog  write FOnLog;
  private
}

    function    TKnobsGene.GetSize: Integer;
    begin
      Result := Length( FData)
    end;


    procedure   TKnobsGene.SetSize( aValue: Integer);
    begin
      if aValue <> Length( FData)
      then SetLength( FData, aValue);
    end;


    function    TKnobsGene.GetData( anIndex: Integer): TSignal;
    begin
      Result := FData[ anIndex];
    end;


    procedure   TKnobsGene.SetData( anIndex: Integer; aValue: TSignal);
    begin
      FData[ anIndex] := aValue;
    end;


//  protected

    procedure   TKnobsGene.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( Self, Format( 'TKnobsGene: %s', [ aMsg], AppLocale));
    end;


    procedure   TKnobsGene.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


//  Public

    constructor TKnobsGene.Create( aWorm: TKnobsWorm);
    begin
      inherited Create;

      if Assigned( aWorm)
      then begin
        OnLog := aWorm.DoLog;
        Size := aWorm.Size;
        Move( aWorm.FGene.FData[ 0], FData[ 0], Size * SizeOf( TSignal));
      end;
    end;


    destructor  TKnobsGene.Destroy;
    begin
      Clear;
      inherited;
    end;


    procedure   TKnobsGene.Clear;
    begin
      Size := 0;
    end;


initialization

  // Regiser classes with the component streaming system

  RegisterClasses(
    [
      TKnobsWorm,
      TKnobsWormPanel
    ]
  );

finalization

  UnregisterClasses(
    [
      TKnobsWorm,
      TKnobsWormPanel
    ]
  );


end.
