// Obtained from : http://w-shadow.com/blog/2006/08/27/how-to-get-the-cpu-usage-of-a-process/


unit CpuUsage;

interface


const

  wsMinMeasurementInterval = 250;     // minimum amount of time that must have elapsed to calculate CPU usage,
                                      // miliseconds. If time elapsed is less than this, previous result is
                                      // returned, or zero, if there is no previous result.

type

  TCPUUsageData = record
  public
    PID            : Cardinal;
    Handle         : Cardinal;
    OldUser        : Int64;
    OldKernel      : Int64;
    LastUpdateTime : Cardinal;
    LastUsage      : Single;          // Last result of wsGetCpuUsage is saved here
    Tag            : Cardinal;        // Use it for anythin you like, not modified by this unit
  end;

  PCPUUsageData = ^TCPUUsageData;

  function  CreateCpuUsageCounter ( aPID    : Cardinal     ): PCPUUsageData;
  function  GetCpuUsage           ( aCounter: PCPUUsageData): Single;
  procedure DestroyCpuUsageCounter( aCounter: PCPUUsageData);


implementation


uses

  WinApi.Windows;

  function  CreateCpuUsageCounter( aPID: Cardinal): PCPUUsageData;
  var
    UsageData     : PCPUUsageData;
    CreationTime  : _FILETIME;
    ExitTime      : _FILETIME;
    KernelTime    : _FILETIME;
    UserTime      : _FILETIME;
    ProcessHandle : Cardinal;
  begin
    Result := nil;
    // We need a handle with PROCESS_QUERY_INFORMATION privileges
    ProcessHandle := OpenProcess( PROCESS_QUERY_INFORMATION, False, aPID);

    if ProcessHandle = 0
    then Exit;

    New( UsageData);
    UsageData.PID            := aPID;
    UsageData.Handle         := ProcessHandle;
    UsageData.LastUpdateTime := GetTickCount;
    UsageData.LastUsage      := 0;

    if GetProcessTimes( UsageData.Handle, CreationTime, ExitTime, KernelTime, UserTime)
    then begin
      // convert _FILETIME to Int64
      UsageData.OldKernel := int64( KernelTime.dwLowDateTime or ( KernelTime.dwHighDateTime shl 32));
      UsageData.OldUser   := int64( UserTime  .dwLowDateTime or ( UserTime  .dwHighDateTime shl 32));
      Result              := UsageData;
    end
    else Dispose( UsageData);
  end;

  procedure DestroyCpuUsageCounter( aCounter: PCPUUsageData);
  begin
    CloseHandle( aCounter.Handle);
    Dispose( aCounter);
  end;

  function GetCpuUsage( aCounter: PCPUUsageData): Single;
  var
    CreationTime : _FILETIME;
    ExitTime     : _FILETIME;
    KernelTime   : _FILETIME;
    UserTime     : _FILETIME;
    DeltaMs      : Cardinal;
    Now          : Cardinal;
    Kernel       : Int64;
    User         : Int64;
    Delta        : Int64;
  begin
    Result := aCounter.LastUsage;
    Now    := GetTickCount;         // Get the time elapsed since last query

    DeltaMs := Now - aCounter.LastUpdateTime;

    if DeltaMs < wsMinMeasurementInterval
    then exit;

    aCounter.LastUpdateTime := Now;

    GetProcessTimes( aCounter.Handle, CreationTime, ExitTime, KernelTime, UserTime);    // convert _FILETIME to Int64.
    Kernel := int64( KernelTime.dwLowDateTime or ( KernelTime.dwHighDateTime shr 32));
    User   := int64( UserTime.dwLowDateTime   or ( UserTime.dwHighDateTime   shr 32));  // get the delta
    Delta  := User + Kernel - aCounter.OldUser - aCounter.OldKernel;

    aCounter.OldUser   := User;
    aCounter.OldKernel := Kernel;

    Result:= ( Delta / DeltaMs) / 100;            // mDelta is in units of 100 nanoseconds, so
    aCounter.LastUsage := Result;                 // just in case you want to use it later, too
  end;

end.
