unit FrmAbstraction;

{

   COPYRIGHT 2017 .. 2018 Blue Hell / Jan Punter

  Some parts are copyright :

     Author  : Neugls.
     Website : Http://www.neugls.info :: NOTE : as of 2017-06-11 website is all chinese now
     Email   : NeuglsWorkStudio@gmail.com

   these are marked with (*Neugls*)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  Winapi.Windows, Winapi.Messages,

  System.SysUtils, System.Variants, System.Classes, System.Rtti, System.UITypes, System.TypInfo, System.IniFiles,
  System.Types, System.Math,

  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.ComCtrls, Vcl.ToolWin, Vcl.ImgList, Vcl.ExtCtrls,
  Vcl.StdCtrls, Vcl.Buttons,

  Globals, KnobsUtils, KnobsConversions, KnobParser, Knobs2013, KnobsWorms, KnobsDesigner,

  JvDesignSurface, JvExControls, JvInspector, JvResources, JvComponentBase, System.ImageList;


type

  EAbstraction = class( Exception);

  // Need some extra info in the design panel, make a class imposer

  TJvDesignPanel = class( JvDesignSurface.TJvDesignPanel)
  private
    FPatchFile : string;
    FGUID      : TGUID;
    FSaveGuid  : Boolean;
  private
    function    GetGuid: string;
    procedure   SetGuid( const aValue: string);
    procedure   ClearGuid;
  public
    property    SaveGuid  : Boolean read FSaveGuid  write FSaveGuid;
  published
    property    PatchFile : string  read FPatchFile write FPatchFile;
    property    Guid      : string  read GetGUID    write SetGuid    stored FSaveGUID;
  end;


  TAbstractionList<T: TControl> = class
  private
    FData : TArray<T>;
  private
    function    GetCount: Integer;
    procedure   ValidateIndex( anIndex: Integer);
    function    GetItem( anIndex: Integer): T;
    function    GetName( anIndex: Integer): string;
  public
    procedure   Clear;
    function    FindItem  ( const anItem: T): Integer;
    procedure   AddItem   ( const anItem: T);
    procedure   RemoveItem( const anItem: T);
  public
    property    Count                  : Integer read GetCount;
    property    Item[ anIndex: Integer]: T       read GetItem;                                                  default;
    property    Name[ anIndex: Integer]: string  read GetName;
  end;


  TFormAbstraction = class( TForm, IKnobsDesigner)
    PanelTop: TPanel;
    PanelClient: TPanel;
    SplitterLeft: TSplitter;
    JvDesignPanel: TJvDesignPanel;
    ListBoxNames: TListBox;
    BitBtnClose: TBitBtn;
    BitBtnLoad: TBitBtn;
    BitBtnSave: TBitBtn;
    JvInspector: TJvInspector;
    SplitterRight: TSplitter;
    ImageList: TImageList;
    ToolBar: TToolBar;
    but_select: TToolButton;
    but_display: TToolButton;
    but_textlabel: TToolButton;
    but_box: TToolButton;
    but_image: TToolButton;
    BitBtnToggleNames: TBitBtn;
    procedure BitBtnCloseClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure JvDesignPanelSelectionChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure JvDesignPanelGetAddClass(Sender: TObject; var ioClass: string);
    procedure BitBtnLoadClick(Sender: TObject);
    procedure BitBtnSaveClick(Sender: TObject);
    procedure PaletteButtonClick(Sender: TObject);
    procedure ListboxSelectionChanged(Sender: TObject);
    procedure ListBoxNamesKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure JvInspectorDataValueChanged(Sender: TObject; Data: TJvCustomInspectorData);
    procedure FormDestroy(Sender: TObject);
    procedure JvInspectorBeforeItemCreate(Sender: TObject; Data: TJvCustomInspectorData;
      var ItemClass: TJvInspectorItemClass);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure BitBtnToggleNamesClick(Sender: TObject);
  private
    FOnLog        : TKnobsOnLog;
    FInitialized  : Boolean;
    FEditor       : TKnobsWirePanel;
    FModule       : TKnobsAbstractModule;
    FStickyMode   : Boolean;
    FDesignClass  : string;
    FAllowedProps : TStringList;
    FDisplayList  : TAbstractionList<TKnobsDisplay>;
  private
    function    GetUseAlternateTitles: Boolean;
    procedure   SetUseAlternateTitles( aValue: Boolean);
    procedure   Log( const aMsg: string);
    procedure   LogFmt( const aFmt: string; const anArgs: array of const);
    procedure   SetEditor( const aValue: TKnobsWirePanel);
    procedure   NoEditor( aWhere: Integer);
    procedure   AddDisplay( const aDisplay: TKnobsDisplay);
    procedure   ClearDisplayList;
    procedure   PopulateNames;
    procedure   RePopulate;
    procedure   CreateModule;
    procedure   PopulateEditor;
    procedure   HandleStructureChange;
    function    FindInDesigner( const aName: string): TControl;
    function    FindInListbox( const aControl: TControl): Integer;
    procedure   UpdatePatchFromDesigner;
    procedure   UpdatePatchPublics( const aRemovedItems: TStringList);
    procedure   DataValueChanged( const aData: TJvCustomInspectorData);
    procedure   RegisterAllowedProps;
    function    IsValidDesignItem( const aData: TJvCustomInspectorData): Boolean;
    function    PatchFileName: string;
    function    AbstractionFileName: string;
    procedure   ResyncFModule;
    procedure   LoadAbstraction;
    procedure   SaveAbstraction;
    function    LongToFriendlyName( const aValue: string): string;
    function    FriendlyToLongName( const aValue: string): string;
    procedure   FixGuid;
    function    CheckGuid: Boolean;
    procedure   ClearGuid;
  public
    procedure   NotifyAbout( const aTarget: IKnobsDesignable; const aReasons: TNotifyReasons; const aValue: TValue);
    procedure   UpdateModuleNames( const anOldNames, aNewNames: TStringArray);
  public
    procedure   Save;
    procedure   SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    procedure   LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
    function    CreateUserModule( anOwner: TComponent; const aFileName: string): TKnobsAbstractModule;
  private
    property    UseAlternateTitles : Boolean read GetUseAlternateTitles write SetUseAlternateTitles;
  public
    property    OnLog  : TKnobsOnLog     read FOnLog  write FOnLog;
    property    Editor : TKnobsWirePanel read FEditor write SetEditor;
  end;

var

  FormAbstraction: TFormAbstraction = nil;


implementation

{$R *.dfm}

uses

  JvDesignImp, FrmStore;


// User area

const

  GAddableTypes : array[ 1 .. 4] of string = (
    'TKnobsDisplay'  ,
    'TKnobsTextLabel',
    'TKnobsBox'      ,
    'TImage'
  );


{ ========
  TJvDesignPanel = class( JvDesignSurface.TJvDesignPanel)
  private
    FPatchFile : string;
    FGUID      : TGUID;
  published
    property    PatchFile : string  read FPatchFile write FPatchFile;
    property    Guid      : string  read GetGUID    write SetGuid    stored FSaveGUID;
  private
}

    function    TJvDesignPanel.GetGuid: string;
    begin
      try
        if IsEqualGUID( FGUID, TGUID.Empty)
        then CreateGUID( FGUID);

        Result := GUIDToString( FGUID);
      except
        Result := GUIDToString( FGUID);
      end;
    end;


    procedure   TJvDesignPanel.SetGuid( const aValue: string);
    begin
      if not SameText( aValue, Guid)
      then begin
        try
          FGUID := StringToGUID( aValue);
        except
          CreateGUID( FGUID);
        end;
      end;
    end;


    procedure   TJvDesignPanel.ClearGUID;
    begin
      FGUID := TGUID.Empty;
    end;


{ ========
  TAbstractionList<T> = class
  private
    FData : TArray<T>;
  public
    property    Item[ anIndex: Integer]: T read GetItem;
  private
}

    function    TAbstractionList<T>.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


    procedure   TAbstractionList<T>.ValidateIndex( anIndex: Integer);
    begin
      if not(( anIndex >= Low( FData)) and ( anIndex <= High( FData)))
      then raise EAbstraction.Create( 'Index error in abstraction list');
    end;


    function    TAbstractionList<T>.GetItem( anIndex: Integer): T;
    begin
      ValidateIndex( anIndex);;
      Result := FData[ anIndex]
    end;


    function    TAbstractionList<T>.GetName( anIndex: Integer): string;
    begin
      Result := Item[ anIndex].Name;
    end;


//  public

    procedure   TAbstractionList<T>.Clear;
    begin
      SetLength( FData, 0);
    end;


    function    TAbstractionList<T>.FindItem( const anItem: T): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := Low( FData) to High( FData)
      do begin
        if FData[ i] = anItem
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TAbstractionList<T>.AddItem( const anItem: T);
    begin
      SetLength( FData, Count + 1);
      FData[ Count - 1] := anItem;
    end;


    procedure   TAbstractionList<T>.RemoveItem( const anItem: T);
    var
      p : Integer;
    begin
      p := FindItem( anItem);

      if p >= Low( FData)
      then begin
        Move( FData[ p + 1], FData[ p], ( Count - p - 1) * SizeOf( T));
        SetLength( FData, Count - 1);
      end;
    end;




{ ========
  TFormAbstraction = class( TForm, IKnobsDesigner)
}

//  private

    function    TFormAbstraction.GetUseAlternateTitles: Boolean;
    begin
      if Assigned( FEditor)
      then Result := FEditor.UseAlternateTitles
      else Result := False;
    end;


    procedure   TFormAbstraction.SetUseAlternateTitles( aValue: Boolean);
    begin
      if Assigned( FEditor)
      then FEditor.UseAlternateTitles := aValue;
    end;


    procedure   TFormAbstraction.Log( const aMsg: string);
    begin
      if Assigned( FOnLog)
      then FOnLog( self, LC_ABSTRACTION, aMsg);
    end;


    procedure   TFormAbstraction.LogFmt( const aFmt: string; const anArgs: array of const);
    begin
      Log( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TFormAbstraction.SetEditor( const aValue: TKnobsWirePanel);
    begin
      if Assigned( aValue) and ( aValue <> FEditor)
      then begin
        FEditor := aValue;
        FixGuid;
      end;
    end;


    procedure   TFormAbstraction.NoEditor( aWhere: Integer);
    begin
      LogFmt( 'oops, the editor is not assigned (%d)', [ aWhere]);
    end;


    procedure   TFormAbstraction.AddDisplay( const aDisplay: TKnobsDisplay);
    begin
      FDisplayList.AddItem( aDisplay);
    end;


    procedure   TFormAbstraction.ClearDisplayList;
    begin
      FDisplayList.Clear;
    end;


    procedure   TFormAbstraction.PopulateNames;
    var
      i : Integer;
      p : Integer;
      S : string;
      aRemovedItems : TStringList;
    begin
      aRemovedItems := TStringList.Create;

      try
        aRemovedItems.Assign( ListBoxNames.Items);
        ListBoxNames .Clear;

        if Assigned( FModule)
        then begin
          ListboxNames.Items.Add( LongToFriendlyName( FModule.Name));

          p := aRemovedItems.IndexOf( LongToFriendlyName( FModule.Name));

          if p >= 0
          then aRemovedItems.Delete( p);

          p := aRemovedItems.IndexOf( LongToFriendlyName( FModule.TitleLabel.Name));

          if p >= 0
          then aRemovedItems.Delete( p);

          ClearDisplayList;

          for i := 0 to FModule.ControlCount - 1
          do begin
            S := LongToFriendlyName( FModule.Controls[ i].Name);
            p := aRemovedItems.IndexOf( S);

            if p >= 0
            then aRemovedItems.Delete( p);

            ListBoxNames.Items.Add( S);

            if FModule.Controls[ i] is TKnobsDisplay
            then AddDisplay( TKnobsDisplay( FModule.Controls[ i]));
          end;

          UpdatePatchPublics( aRemovedItems);
        end;
      finally
        aRemovedItems.DisposeOf;
      end;
    end;


    procedure   TFormAbstraction.RePopulate;
    var
      S : TMemoryStream;
    begin
      // todo : this must persist links .. like module to titlelable, knob to display ... seems to get lost now .. or does it ???
      S := TMemoryStream.Create;
      try
        JvDesignPanel.SaveToStream( S);
        S.Position := 0;
        JvDesignPanel.LoadFromStream( S);
      finally
        S.DisposeOf;
      end;

      PopulateNames;
    end;


    procedure   TFormAbstraction.CreateModule;
    begin
      if Assigned( FEditor)
      then begin
        JvDesignPanel.Active := False;
        JvDesignPanel.Clear;
        FModule := FEditor.CreateAbstract( JvDesignPanel);

        if Assigned( FModule)
        then begin
          if Assigned( FormStore)
          then FModule.Color := FormStore.GroupAbstractColor
          else FModule.Color := clBlack;;

          JvDesignPanel.InsertControl( FModule);
          JvDesignPanel.Surface.SetSelected( [ FModule]);
        end;
      end;
    end;


    procedure   TFormAbstraction.PopulateEditor;
    begin
      if Assigned( FEditor)
      then begin
        CreateModule;
        LoadAbstraction;
        RePopulate;
        JvDesignPanel.Active := True;
        JvDesignPanel.Invalidate;
        FEditor.SetActiveInDesigner( FModule.Name, False);
      end
      else NoEditor( 1);
    end;


    procedure   TFormAbstraction.HandleStructureChange;
    var
      i        : Integer;
      aParts   : TStringList;
      aModule  : TKnobsCustomModule;
      aControl : TControl;
    begin

      // Note: this will be called - after - a module was inserted and renaming took place
      // Note: this will be called - after - a module was deleted, which does not result in renamed components

      if assigned( FModule) and Assigned( FEditor)
      then begin
        FixGuid;

        // Remove all local controls that have no counter part in the editor

        for i := FModule.ControlCount - 1 downto 0
        do begin
          aParts := Explode( FModule.Controls[ i].Name, '_');

          try
            if aParts.Count > 2                              // 'local' names should have less than two parts ...
            then begin
              aModule := FEditor.FindModule( aParts[ 0]);

              if not Assigned( aModule)
              then FModule.Controls[ i].DisposeOf;
            end
          finally
            FreeAndNil( aParts);
          end;
        end;

        // Then the other way around ... add editor stuff that is not present locally ...

        aModule := FEditor.CreateAbstract( nil);

        try
          if Assigned( aModule)
          then begin
            for i := aModule.ControlCount - 1 downto 0
            do begin
              if not Assigned( FindInDesigner( aModule.Controls[ i].Name))
              then begin
                aControl := aModule.Controls[ i];
                aModule.RemoveControl( aControl);
                FModule.InsertControl( aControl);
                aControl.Parent := FModule;
              end;
            end;
          end;
        finally
          aModule.DisposeOf;
        end;
      end;
    end;


    function    TFormAbstraction.FindInDesigner( const aName: string): TControl;
    var
      i : Integer;
    begin
      Result := nil;

      if Sametext( FModule.Name, aName)
      then Result := FModule
      else begin
        for i := 0 to FModule.ControlCount - 1
        do begin
          if SameText( FModule.Controls[ i].Name, aName)
          then begin
            Result := FModule.Controls[ i];
            Break;
          end;
        end;
      end;
    end;


    function    TFormAbstraction.FindInListbox( const aControl: TControl): Integer;
    begin
      Result := -1;

      if Assigned( aControl)
      then Result := ListBoxNames.Items.IndexOf( LongToFriendlyName( aControl.Name));
    end;


    procedure   TFormAbstraction.UpdatePatchFromDesigner;
    begin
      if Assigned( FEditor)
      then begin
        if ListBoxNames.ItemIndex >= 0
        then FEditor.SetActiveInDesigner( FriendlyToLongName( ListBoxNames.Items[ ListBoxNames.ItemIndex]), False)
        else begin
          if Assigned( FModule)
          then FEditor.SetActiveInDesigner( FriendlyToLongName( FModule.Name), False)
          else FEditor.SetActiveInDesigner( '', False);
        end;
      end
      else NoEditor( 2);
    end;


    procedure   TFormAbstraction.UpdatePatchPublics( const aRemovedItems: TStringList);
    var
      i : Integer;
    begin
      if Assigned( FEditor)
      then begin
        if Assigned( aRemovedItems) and ( aRemovedItems.Count > 0)
        then begin
          for i := 0 to aRemovedItems.Count - 1
          do FEditor.RemovePublic( aRemovedItems[ i]);
        end;
      end
      else NoEditor( 3);
    end;


    procedure   TFormAbstraction.DataValueChanged( const aData: TJvCustomInspectorData);
    var
      anItemName        : string;
      aControl          : TObject;
      anObjectName      : string;
      aDesValuedControl : TKnobsValuedControl;     // des - for abstraction designer
      aDesTextControl   : TKnobsTextControl;
      aPatModule        : TKnobsCustomModule;      // pat - for editor patch
      aPatValuedControl : TKnobsValuedControl;
      aPatTextControl   : TKnobsTextControl;
      aModuleName       : string;
      aControlName      : string;
    begin
      if Assigned( FEditor)
      then begin
        anItemName := aData.Name;
        aControl   := JvInspector.InspectObject;

        if Assigned( aControl)
        then begin
          if ( aControl is TKnobsValuedControl) and SameText( anItemName, 'KnobPosition')
          then begin
            aDesValuedControl := TKnobsValuedControl( aControl);
            anObjectName      := aDesValuedControl.Name;
            aModuleName       := GetCtrlNamePrefix( anObjectName);
            aControlName      := GetCtrlNameSuffix( anObjectName);
            aPatModule        := FEditor.FindModule( aModuleName);

            if Assigned( aPatModule)
            then begin
              aPatModule.FixControlInsertions;
              aPatValuedControl := aPatModule.FindValuedControl( aControlName);

              if Assigned( aPatValuedControl)
              then aPatValuedControl.KnobPosition := aDesValuedControl.KnobPosition;
            end;
          end
          else if ( aControl is TKnobsTextControl) and SameText( anItemName, 'TextValue')
          then begin
            aDesTextControl := TKnobsTextControl( aControl);
            anObjectName    := aDesTextControl.Name;
            aModuleName     := GetCtrlNamePrefix( anObjectName);
            aControlName    := GetCtrlNameSuffix( anObjectName);
            aPatModule      := FEditor.FindModule( aModuleName);

            if Assigned( aPatModule)
            then begin
              aPatModule.FixControlInsertions;
              aPatTextControl := aPatModule.FindTextControl( aControlName);

              if Assigned( aPatTextControl)
              then aPatTextControl.TextValue := aDesTextControl.TextValue;
            end;
          end;
        end;
      end
      else NoEditor( 4);
    end;


    procedure   TFormAbstraction.RegisterAllowedProps;
    begin
      FAllowedProps.Clear;
      FAllowedProps.Add( 'AllowAutomation'   );
      FAllowedProps.Add( 'AllowRandomization');
      FAllowedProps.Add( 'AssignedMidiCC'    );
      FAllowedProps.Add( 'ControlType'       );
      FAllowedProps.Add( 'Dezippered'        );
      FAllowedProps.Add( 'Display'           );
      FAllowedProps.Add( 'FriendlyName'      );
      FAllowedProps.Add( 'Height'            );
      FAllowedProps.Add( 'KnobPosition'      );
      FAllowedProps.Add( 'Left'              );
      FAllowedProps.Add( 'Locked'            );
      FAllowedProps.Add( 'TabOrder'          );
      FAllowedProps.Add( 'TabStop'           );
      FAllowedProps.Add( 'Title'             );
      FAllowedProps.Add( 'Top'               );
      FAllowedProps.Add( 'Tag'               );
      FAllowedProps.Add( 'Visible'           );
      FAllowedProps.Add( 'Width'             );

      FAllowedProps.Add( 'Alignment'         );
      FAllowedProps.Add( 'Enabled'           );
      FAllowedProps.Add( 'HasEditor'         );
      FAllowedProps.Add( 'HasParentEditor'   );
      FAllowedProps.Add( 'HasScrollbar'      );
      FAllowedProps.Add( 'MultiLine'         );
      FAllowedProps.Add( 'MustSave'          );
      FAllowedProps.Add( 'TextValue'         );

      FAllowedProps.Add( 'AutoSize'          );
      FAllowedProps.Add( 'Center'            );
      FAllowedProps.Add( 'Picture'           );
      FAllowedProps.Add( 'Proportional'      );
      FAllowedProps.Add( 'Stretch'           );
      FAllowedProps.Add( 'Transparent'       );

      FAllowedProps.Add( 'Caption'           );
      FAllowedProps.Add( 'Comment'           );

// todo: the auto generated controls should no have an editable name
      FAllowedProps.Add( 'Name'              );
    end;


    function    TFormAbstraction.IsValidDesignItem( const aData: TJvCustomInspectorData): Boolean;
    begin
      Result := FAllowedProps.IndexOf( aData.Name) >= 0;
    end;


    function    TFormAbstraction.PatchFileName: string;
    begin
      if Assigned( FEditor)
      then Result := AbsoluteToRelativePath( FEditor.Filename, ExtractFilePath( FEditor.Filename))
      else Result := '';
    end;


    function    TFormAbstraction.AbstractionFileName: string;
    begin
      if Assigned( FEditor)
      then Result := RelativeToAbsolutePath( ChangeFileExt( PatchFileName, EXT_ABSTRACT), ExtractFilePath( FEditor.Filename))
      else NoEditor( 5);
    end;


    procedure   TFormAbstraction.ResyncFModule;
    var
      i : Integer;
    begin
      FModule := nil;

      for i := 0 to JvDesignPanel.ControlCount - 1
      do begin
        if JvDesignPanel.Controls[ i] is TKnobsAbstractModule
        then begin
          FModule := TKnobsAbstractModule( JvDesignPanel.Controls[ i]);
          Break;
        end;
      end;
    end;


    procedure   TFormAbstraction.LoadAbstraction;
    var
      Failed     : Boolean;
      WasDeleted : Boolean;
    begin
      ClearGuid;
      Failed     := True;
      WasDeleted := False;

      try
        if FileExists( AbstractionFileName)
        then begin
          JvDesignPanel.LoadFromFile( AbstractionFileName);
          ResyncFModule;

          if not Assigned( FModule)
          then raise EAbstraction.Create( 'Resync error, FModule not assigned after loading abstraction');

          if not CheckGuid
          then raise EAbstraction.Create( 'Guids of patch and abstraction differ');

          Failed := False;
        end;
      except
        on E: Exception
        do begin
          LogFmt( 'Error in abstraction file [%s]: %s', [ AbstractionFileName, E.ToString]);

          if MessageDlg(
            Format(
              'The abstraction file "%s" could not be read' + ^M   +
              'for reason : %s'                             + ^M^M +
              'Shall I delete it and make a new one?'
              ,
              [ AbstractionFileName, E.ToString],
              AppLocale
            ),
            mtError,
            [ mbYes, mbNo],
            0
          ) = mrYes
          then begin
            WasDeleted := True;

            try
              if FileExists( AbstractionFileName)
              then begin
                RenameFile( AbstractionFileName, ChangeFileExt( AbstractionFileName, EXT_ABSTRACT_SAVED));
                DeleteFile( AbstractionFileName);
              end;
            except
              on E: Exception
              do LogFmt( 'File "%s" not deleted for reason : %s', [ AbstractionFileName, E.ToString]);
            end;
          end;
        end;
      end;

      if Failed     // Try to make 'something' on failure, when no abstraction file was ever created Failed will be set too
      then begin
        CreateModule;
        RePopulate;
        JvDesignPanel.Active    := True;
        JvDesignPanel.PatchFile := PatchFileName;
        JvDesignPanel.Invalidate;
        FEditor.SetActiveInDesigner( FModule.Name, False);
        FixGuid;

        if WasDeleted and Assigned( FEditor)
        then begin
          FInitialized  := True;
          SaveAbstraction;
        end;
      end;
    end;


    procedure   TFormAbstraction.SaveAbstraction;
    begin
      if ( PatchFileName <> '') and FInitialized
      then begin
        try
          FixGuid;
          JvDesignPanel.SaveGuid := True;

          try
            FModule.PatchFile := JvDesignPanel.PatchFile;
            FModule.UniqueId  := JvDesignPanel.Guid;
            JvDesignPanel.SaveToFile( AbstractionFileName);
          finally
            JvDesignPanel.SaveGuid := False;
          end;
        except
          on E: Exception
          do LogFmt( 'Could not save file "%s" for reason: %s', [ AbstractionFileName, E.ToString]);
        end;
      end;
    end;


    function    TFormAbstraction.LongToFriendlyName( const aValue: string): string;
    begin
      Result := StringReplace( aValue, '_', ' ', [ rfReplaceAll, rfIgnoreCase]);
      Result := ModuleLongToFriendlyName( Result);
    end;


    function    TFormAbstraction.FriendlyToLongName( const aValue: string): string;
    begin
      Result := ModuleFriendlyToLongName( aValue);
      Result := StringReplace( Result, ' ', '_', [ rfReplaceAll, rfIgnoreCase]);
    end;


    procedure   TFormAbstraction.FixGuid;
    begin
      if   Assigned( JvDesignPanel)
      and  Assigned( FEditor)
      then JvDesignPanel.Guid := FEditor.Guid;
    end;


    function    TFormAbstraction.CheckGuid: Boolean;
    begin
      Result := Assigned( FEditor) and Assigned( JvDesignPanel) and ( FEditor.Guid = JvDesignPanel.Guid);
    end;


    procedure   TFormAbstraction.ClearGuid;
    begin
      if Assigned( JvDesignPanel)
      then JvDesignPanel.ClearGuid;
    end;


//  public

    procedure   TFormAbstraction.NotifyAbout( const aTarget: IKnobsDesignable; const aReasons: TNotifyReasons; const aValue: TValue);
    var
      C : TControl;
      V : TKnobsValuedControl;
      T : TKnobsTextControl;
    begin
      if Assigned( aTarget)
      then LogFmt( 'TFormAbstraction.NotifyAbout: name: ''%s'' reasons: %s value: %s', [ aTarget.CtrlName, NotifyReasonsToStr( aReasons), aValue.ToString])
      else LogFmt( 'TFormAbstraction.NotifyAbout: name: ''nil'' reasons: %s value: %s', [ NotifyReasonsToStr( aReasons), aValue.ToString]);

      if Assigned( FModule) and Assigned( FEditor)
      then begin
        if nrStructureChanged in aReasons
        then HandleStructureChange;

        if Assigned( aTarget) and ( [ nrValueChanged, nrTextChanged] * aReasons <> [])
        then begin
          C := FindInDesigner( aTarget.CtrlName);

          if Assigned( C)
          then begin
            if C is TKnobsValuedControl
            then begin
              V := TKnobsValuedControl( C);
              V.KnobPosition := Converters.ValueToPos( V.ControlType, aValue.AsExtended, V.StepCount);

              if JvInspector.InspectObject = V
              then JvInspector.RefreshValues;
            end
            else if C is TKnobsTextControl
            then begin
              T := TKnobsTextControl( C);
              T.TextValue := aValue.AsString;
            end;
          end;
        end;

        if Assigned( aTarget) and ( nrPublicChanged in aReasons) and ( aTarget is TControl)
        then begin
          if aTarget.IsPublic
          then begin
            C := Clone( TControl( aTarget), JvDesignPanel, FModule);
            C.Name := aTarget.CtrlName;
            RePopulate;
          end
          else begin
            C := FindInDesigner( aTarget.CtrlName);

            if Assigned( C)
            then begin
              JvInspector.InspectObject := FModule;
              JvDesignPanel.Surface.Select( FModule);
              C.DisposeOf;
              FEditor.DesignerChanged( Self);
              PopulateNames;
            end;
          end;
        end;

        if Assigned( aTarget) and ( nrNameChanged in aReasons)
        then begin
          // todo: Hmm .. does nrNameChanged ever happen? .. UpdateModuleNames will do the trick it seems
          // nrNameChanged would likely have lost the old names already anyway
        end;
      end;
    end;


    procedure   TFormAbstraction.UpdateModuleNames( const anOldNames, aNewNames: TStringArray);
    var
      i       : Integer;
      j       : Integer;
      Changed : Boolean;
      O       : string;
      N       : string;
      M       : string;
      aList   : TStringList;
    begin
      Changed := False;

      if Assigned( FModule) and ( Length( anOldNames) > 0)
      then begin
        aList := TStringList.Create;

        try
          for i := 0 to FModule.ControlCount - 1
          do begin
            M := FModule.Controls[ i].Name + '_';

            for j := 0 to Min( Length( anOldNames), Length( aNewNames)) - 1
            do begin
              O := anOldNames[ j] + '_';

              if M.Contains( O)
              then begin
                N       := aNewNames[ j] + '_';
                M       := StringReplace( FModule.Controls[ i].Name, O, N, [ rfReplaceAll, rfIgnoreCase]);
                aList.Add( M);
                Changed := True;
                Break;
              end;
            end;

            if not Changed
            then aList.Add( FModule.Controls[ i].Name);
          end;

          if Changed
          then begin
            for i := 0 to FModule.ControlCount - 1
            do FModule.Controls[ i].Name := Format( 'a%d', [ i], AppLocale);

            ListBoxNames.Clear;
            ListBoxNames.Items.Add( FModule.Name);

            for i := 0 to FModule.ControlCount - 1
            do begin
              FModule.Controls[ i].Name := aList[ i];
              N := LongToFriendlyName( aList[ i]);
              ListBoxNames.Items.Add( N);
            end;
          end;
        finally
          aList.DisposeOf;
        end;
      end;
    end;


//  public

    procedure   TFormAbstraction.Save;
    begin
      try
        if ( PatchFileName <> '') and FInitialized
        then begin
          JvDesignPanel.SaveToFile( AbstractionFileName);
          LogFmt( 'TFormAbstraction.Save: Saved [%s]', [ AbstractionFileName]);
        end
      except
        on E: Exception
        do KilledException( E);
      end;
    end;


    procedure   TFormAbstraction.SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    begin
      if Assigned( anIniFile)
      then begin
        LogFmt( 'TFormAbstraction.SaveIni: "%s"', [ anIniFile.FileName]);

        with anIniFile
        do begin
          EraseSection( aSectionName);
          WriteInteger( aSectionName, 'Left'               , Left                 );
          WriteInteger( aSectionName, 'Top'                , Top                  );
          WriteInteger( aSectionName, 'Width'              , Width                );
          WriteInteger( aSectionName, 'Height'             , Height               );
          WriteInteger( aSectionName, 'WindowState'        , Integer( WindowState));
          WriteInteger( aSectionName, 'JvInspector.Width'  , JvInspector.Width    );
          WriteInteger( aSectionName, 'JvInspector.Divider', JvInspector.Divider  );
          WriteInteger( aSectionName, 'ListBoxNames.Width' , ListBoxNames.Width   );
        end;
      end;
    end;


    procedure   TFormAbstraction.LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
    var
      L, T, W, H : Integer;
      WinState   : TWindowState;
    begin
      if Assigned( anIniFile)
      then begin
        LogFmt( 'TFormAbstraction.LoadIni: "%s"', [ anIniFile.FileName]);

        with anIniFile
        do begin
          L        :=                ReadInteger( aSectionName, 'Left'       ,          Left        );
          T        :=                ReadInteger( aSectionName, 'Top'        ,          Top         );
          W        :=                ReadInteger( aSectionName, 'Width'      ,          Width       );
          H        :=                ReadInteger( aSectionName, 'Height'     ,          Height      );
          WinState := TWindowState ( ReadInteger( aSectionName, 'WindowState', Integer( WindowState)));

          if ( L < 0) or ( L > Screen.Width  - 20) then L := 0;
          if ( T < 0) or ( T > Screen.Height - 20) then T := 0;
          if W > Screen.Width                      then W := Screen.Width  - 20;
          if H > Screen.Height                     then H := Screen.Height - 20;

          SetBounds( L, T, W, H);

          if WinState = wsMaximized
          then WindowState := wsMaximized;

          JvInspector .Width   := ReadInteger( aSectionName, 'JvInspector.Width'  , JvInspector.Width  );
          JvInspector .Divider := ReadInteger( aSectionName, 'JvInspector.Divider', JvInspector.Divider);
          ListBoxNames.Width   := ReadInteger( aSectionName, 'ListBoxNames.Width' , ListBoxNames.Width );
        end;
      end;
    end;


    function    TFormAbstraction.CreateUserModule( anOwner: TComponent; const aFileName: string): TKnobsAbstractModule;
    var
      D : TJvDesignPanel;
      C : TControl;
      i : Integer;
      F : string;
    begin
      Result := nil;

      if FileExists( aFileName)
      then begin
        D := TJvDesignPanel.Create( nil);
        try
          D.Parent := Self;
          D.LoadFromFile( aFileName);
          F := RelativeToAbsolutePath( D.PatchFile, ExtractFilePath( aFileName));

          if FileExists( F)
          then begin
            for i := 0 to D.ControlCount
            do begin
              C := D.Controls[ i];

              if C is TKnobsAbstractModule
              then begin
                Result := TKnobsAbstractModule( C).Clone( anOwner, nil, Point( 0, 0), False, False) as TKnobsAbstractModule;
                Result.Selected := False;
                Break;
              end;
            end;
          end
          else LogFmt( 'Can not create user module, the associated patch file does not exist [%s]', [ F]);
        finally
          D.DisposeOf;
        end;
      end
      else LogFmt( 'Can not create user module, file does not exist [%s]', [ aFileName]);
    end;


// Delphi area

procedure TFormAbstraction.ListBoxNamesKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  ListboxSelectionChanged( Sender);
end;

procedure TFormAbstraction.ListboxSelectionChanged(Sender: TObject);
var
  aControl: TControl;
begin
  if FInitialized and ( ListBoxNames.ItemIndex >= 0)
  then begin
    aControl := FindInDesigner( FriendlyToLongName( ListBoxNames.Items[ ListBoxNames.ItemIndex]));

    if Assigned( aControl)
    then begin
      JvInspector.InspectObject := aControl;
      JvDesignPanel.Surface.Select( aControl);
      UpdatePatchFromDesigner;
    end;
  end;
end;

procedure TFormAbstraction.PaletteButtonClick(Sender: TObject);
var
  aTag : Integer;
begin
  if Sender is TControl
  then begin
    aTag := TControl( Sender).Tag;

    if ( aTag >= 1) and ( aTag <= 4)
    then begin
      FStickyMode  := GetKeyState( VK_SHIFT) < 0;
      FDesignClass := GAddableTypes[ TControl( Sender).Tag]
    end
    else begin
      FStickyMode  := False;
      FDesignClass := '';
    end;
  end
  else Log( 'oops, sender is not a TControl');
end;

procedure TFormAbstraction.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if Assigned( FEditor)
  then begin
    FEditor.DesignerChanged( nil);
    FEditor.SetActiveInDesigner( '', True);
  end;

  Log( 'TFormAbstraction.FormClose');
end;

procedure TFormAbstraction.FormCreate(Sender: TObject);
begin
  FDisplayList                := TAbstractionList<TKnobsDisplay>.Create;
  FAllowedProps               := TStringList.Create;
  FAllowedProps.Sorted        := True;
  FAllowedProps.Duplicates    := dupError;
  FAllowedProps.CaseSensitive := False;
  ListBoxNames .Sorted        := True;
  RegisterAllowedProps;
  // todo : experiment which MessengerClass to use - both seem to work OK ... stick with the 2nd one then
  // JvDesignPanel.Surface.MessengerClass := TJvDesignDesignerMessenger;
  JvDesignPanel.Surface.MessengerClass := TJvDesignWinControlHookMessenger;
end;

procedure TFormAbstraction.FormDestroy(Sender: TObject);
begin
  JvDesignPanel.Clear;

  if Assigned( FEditor)
  then FEditor.DesignerChanged( nil);

  FreeAndNil( FAllowedProps);
  FreeAndNil( FDisplayList );
end;

procedure TFormAbstraction.FormShow(Sender: TObject);
begin
  Log( 'TFormAbstraction.FormShow');

  if Assigned( FEditor)
  then begin
    if FileExists( FEditor.Filename)
    then begin
      PopulateEditor;
      JvDesignPanel.PatchFile := PatchFileName;
      FEditor.DesignerChanged( Self);
      FInitialized := True;
    end
    else begin
      Log( 'TFormAbstraction.FormShow: no disk patch available, informing user and closing form');
      MessageDlg( 'Please save the current patch to disk before starting the abstraction editor', mtError, [ mbOk], 0);
      PostMessage( Handle, WM_CLOSE, 0, 0);
    end;
  end
  else Log( 'TFormAbstraction.FormShow: Editor was not assigned');
end;

procedure TFormAbstraction.JvDesignPanelGetAddClass(Sender: TObject; var ioClass: string);
begin
  ioClass := FDesignClass;

  if not FStickyMode
  then begin
    FDesignClass := '';
    but_select.Down := True;
  end;
end;

procedure TFormAbstraction.JvDesignPanelSelectionChange(Sender: TObject);
begin
  if Length( JvDesignPanel.Surface.Selected) > 0
  then begin
    PopulateNames;
    JvInspector.InspectObject := JvDesignPanel.Surface.Selected[ 0];

    if JvInspector.InspectObject is TControl
    then ListBoxNames.ItemIndex := FindInListbox( TControl( JvInspector.InspectObject))
    else ListBoxNames.ItemIndex := - 1;
  end
  else begin
    JvInspector .InspectObject := nil;
    ListBoxNames.ItemIndex     := - 1;

    if Assigned( FModule)
    then PopulateNames;
  end;

  UpdatePatchFromDesigner;
end;

procedure TFormAbstraction.JvInspectorBeforeItemCreate(Sender: TObject; Data: TJvCustomInspectorData;
  var ItemClass: TJvInspectorItemClass);
begin
  if not IsValidDesignItem( Data)
  then ItemClass := nil;
end;

procedure TFormAbstraction.JvInspectorDataValueChanged(Sender: TObject; Data: TJvCustomInspectorData);
begin
  DataValueChanged( Data);
end;

procedure TFormAbstraction.BitBtnLoadClick(Sender: TObject);
begin
  LoadAbstraction;
end;

procedure TFormAbstraction.BitBtnSaveClick(Sender: TObject);
begin
  SaveAbstraction;
end;

procedure TFormAbstraction.BitBtnToggleNamesClick(Sender: TObject);
begin
  UseAlternateTitles := not UseAlternateTitles;
end;

procedure TFormAbstraction.BitBtnCloseClick(Sender: TObject);
begin
  Close;
end;


initialization

  RegisterClass( TKnobsModule        );
  RegisterClass( TKnobsAbstractModule);
  RegisterClass( TKnobsClicker       );
  RegisterClass( TKnobsLed           );
  RegisterClass( TKnobsSelector      );
  RegisterClass( TKnobsKnob          );
  RegisterClass( TKnobsSmallKnob     );
  RegisterClass( TKnobsNoKnob        );
  RegisterClass( TKnobsDisplay       );
  RegisterClass( TKnobsFileSelector  );
  RegisterClass( TKnobsIndicator     );
  RegisterClass( TKnobsIndicatorBar  );
  RegisterClass( TKnobsIndicatorText );
  RegisterClass( TKnobsDataViewer    );
  RegisterClass( TKnobsInput         );
  RegisterClass( TKnobsOutput        );
  RegisterClass( TKnobsTextLabel     );
  RegisterClass( TKnobsEditLabel     );
  RegisterClass( TKnobsBox           );
  RegisterClass( TKnobsData          );
  RegisterClass( TKnobsSlider        );
  RegisterClass( TKnobsHSlider       );
  RegisterClass( TKnobsPad           );
  RegisterClass( TKnobsValuedButton  );
  RegisterClass( TKnobsValuedButtons );
  RegisterClass( TKnobsModule        );
  RegisterClass( TKnobsWorm          );
  RegisterClass( TKnobsWormPanel     );
  RegisterClass( TKnobsDataMaker     );
  RegisterClass( TKnobsGridControl   );
  RegisterClass( TImage              );

end.

