unit FrmAutomation;

{

   COPYRIGHT 2017 .. 2018 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  Winapi.Windows, Winapi.Messages,

  System.SysUtils, System.Variants, System.Classes, System.Math, System.IniFiles, System.Types,
  System.Generics.Defaults, System.Generics.Collections,

  Vcl.Graphics, Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.ExtCtrls, Vcl.StdCtrls, Vcl.Buttons,

  Globals, knobs2013, KnobsUtils, Vcl.Samples.Spin, Vcl.ComCtrls, KnobsConversions, ComplexMath, BSplines,

  FrmViewer;


type

  TScrollBox = class( Vcl.Forms.TScrollBox)
  protected
    procedure   CreateParams( var Params: TCreateParams);                                                      override;
  end;

  TCheckBox = class( Vcl.StdCtrls.TCheckBox)
  private
    FOriginalCaption : string;
    FSetCap          : Boolean;
  protected
    procedure   WMPaint( var aMsg: TWMPaint);                                                          message WM_PAINT;
    procedure   CMTextChanged( var aMsg: TMessage);                                              message CM_TEXTCHANGED;
  end;


  TKnobsViewerItem = record
    Index      : Integer;
    Viewer     : TFrameViewer;
    Control    : TKnobsValuedControl;
    FData      : TSignalPairArray;
    FSpline    : TBSpline;
    FPosition  : TSignal;
  private
    function    GetCount     : Integer;
    function    GetVisible   : Boolean;
    procedure   SetVisible   ( aValue : Boolean);
    function    GetGraphMode : TKnobsDMGraphMode;
    procedure   SetGraphMode ( aValue : TKnobsDMGraphMode);
    function    GetDuration  : TSignal;
    procedure   SetPosition  ( aValue: TSignal);
  private
    function    GetValue( aTime: TSignal): TSignal;
  private
    procedure   SyncGraphToData;
  public
    procedure   PointChanged( anIndex: Integer; anX, anY: TSignal);
    procedure   PointAdded  ( anIndex: Integer; anX, anY: TSignal);
    procedure   PointRemoved( anIndex: Integer; anX, anY: TSignal);
    procedure   ClearGraph;
  public
    property    Count                  : Integer           read GetCount;
    property    Visible                : Boolean           read GetVisible   write SetVisible;
    property    GraphMode              : TKnobsDMGraphMode read GetGraphMode write SetGraphMode;
    property    Duration               : TSignal           read GetDuration;
    property    Position               : TSignal           read FPosition    write SetPosition;
  public
    property    Value[ aTime: TSignal] : TSignal           read GetValue;
  end;


  TKnobsViewers = class
  private
    FParent       : TWinControl;
    FHistory      : TKnobsEditHistory;
    FItems        : TArray<TKnobsViewerItem>;
    FXZoom        : TSignal;
    FYZoom        : TSignal;
    FCursorX      : Integer;
    FCursorY      : Integer;
    FGraphMode    : TKnobsDMGraphMode;
    FAllowXMoves  : Boolean;
    FAllowYMoves  : Boolean;
    FViewTimed    : Boolean;
    FLoopable     : Boolean;
  private
    FOnFrameClose : TOnFrameClose;
  private
    function    GetCount      : Integer;
    procedure   SetXZoom      ( aValue: TSignal);
    procedure   SetYZoom      ( aValue: TSignal);
    procedure   SetCursorX    ( aValue: Integer);
    procedure   SetCursorY    ( aValue: Integer);
    procedure   SetGraphMode  ( aValue: TKnobsDMGraphMode);
    procedure   SetAllowXMoves( aValue: Boolean);
    procedure   SetAllowYMoves( aValue: Boolean);
    procedure   SetLoopable   ( aValue: Boolean);
    procedure   SetViewTimed  ( aValue: Boolean);
  private
    procedure   ArrangeItems;
    procedure   Show( anIndex: Integer);
    procedure   Hide( anIndex: Integer);
    procedure   DoPointChanged( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    procedure   DoPointAdded  ( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    procedure   DoPointRemoved( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    procedure   DoFrameClose  ( const aSender: TObject; anId: Integer);
  public
    constructor Create( const aParent: TWinControl; const aHistory: TKnobsEditHistory);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   AddItem( anIndex: Integer; const aControl: TKnobsValuedControl; const aParamName: string);
    procedure   SetFromHistory;
    procedure   ClearGraphs;
  public
    property    Count       : Integer           read GetCount;
    property    XZoom       : TSignal           read FXZoom       write SetXZoom;
    property    YZoom       : TSignal           read FYZoom       write SetYZoom;
    property    CursorX     : Integer           read FCursorX     write SetCursorX;
    property    CursorY     : Integer           read FCursorY     write SetCursorY;
    property    GraphMode   : TKnobsDMGraphMode read FGraphMode   write SetGraphMode;
    property    AllowXMoves : Boolean           read FAllowXMoves write SetAllowXMoves;
    property    AllowYMoves : Boolean           read FAllowYMoves write SetAllowYMoves;
    property    Loopable    : Boolean           read FLoopable    write SetLoopable;
    property    ViewTimed   : Boolean           read FViewTimed   write SetViewTimed;
  public
    property    OnFrameClose : TOnFrameClose read FOnFrameClose write FOnFrameClose;
  end;


  TFormAutomation = class(TForm)
    PanelTop: TPanel;
    PanelMain: TPanel;
    KnobsSelectorSnap: TKnobsSelector;
    TimerStatusUpdate: TTimer;
    KnobsSelectorClear: TKnobsSelector;
    LabelAutomationInfo: TLabel;
    EditDummy: TEdit;
    PanelDetails: TPanel;
    SplitterDetails: TSplitter;
    PanelDetailsTop: TPanel;
    LabelTime: TLabel;
    ListView: TListView;
    PanelDetailsEditor: TPanel;
    BitBtnToggleNames: TBitBtn;
    Panel2: TPanel;
    BitBtnSaveRecording: TBitBtn;
    BitBtnLoadRecording: TBitBtn;
    TimerBlink: TTimer;
    CheckBoxRecordAll: TCheckBox;
    ScrollBoxDataMaker: TScrollBox;
    PanelGridControl: TPanel;
    datagraph_mode: TKnobsSelector;
    CheckBoxLockValue: TCheckBox;
    CheckBoxLockTime: TCheckBox;
    CheckBoxViewTimed: TCheckBox;
    SpinButtonXZoom: TSpinButton;
    SpinButtonYZoom: TSpinButton;
    Label1: TLabel;
    Label2: TLabel;
    CheckBoxLoopable: TCheckBox;
    SpeedButtonPlay: TKnobsTimerSpeedButton;
    SpeedButtonPause: TKnobsTimerSpeedButton;
    SpeedButtonLeftStop: TKnobsTimerSpeedButton;
    SpeedButtonRightStop: TKnobsTimerSpeedButton;
    SpeedButtonLeftFast: TKnobsTimerSpeedButton;
    SpeedButtonRightFast: TKnobsTimerSpeedButton;
    SpeedButtonRecord: TKnobsTimerSpeedButton;
    SpeedButtonLoop: TKnobsTimerSpeedButton;
    KnobsLedAutomationActive: TKnobsLed;
    PanelRuler: TPanel;
    procedure SpeedButtonRecordClick(Sender: TObject);
    procedure SpeedButtonLoopClick(Sender: TObject);
    procedure SpeedButtonStopClick(Sender: TObject);
    procedure SpeedButtonPlayClick(Sender: TObject);
    procedure SpeedButtonPauseClick(Sender: TObject);
    procedure TimerStatusUpdateTimer(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure KnobsSelectorSnapValueChanged(const aSender: TObject; const aPath, aControlType: string;
      aValue: Double; IsFinal, IsAutomation: Boolean);
    procedure KnobsSelectorClearValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
      IsFinal, IsAutomation: Boolean);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint;
      var Handled: Boolean);
    procedure KnobsLedAutomationActiveClick(Sender: TObject);
    procedure ControlUnfocus(const aSender: TObject);
    procedure FormResize(Sender: TObject);
    procedure EditDummyChange(Sender: TObject);
    procedure ListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);
    procedure KnobsKnobKnobPositionChanged(const aSender: TKnobsValuedControl; const aPath, aControlType: string;
      aPosition: Integer);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure BitBtnToggleNamesClick(Sender: TObject);
    procedure datagraph_modeValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
      IsFinal, IsAutomation: Boolean);
    procedure BitBtnSaveRecordingClick(Sender: TObject);
    procedure BitBtnLoadRecordingClick(Sender: TObject);
    procedure KnobsDataMakerPointChanged(const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    procedure CheckBoxLockValueClick(Sender: TObject);
    procedure CheckBoxLockTimeClick(Sender: TObject);
    procedure TimerBlinkTimer(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure CheckBoxViewTimedClick(Sender: TObject);
    procedure CheckBoxRecordAllClick(Sender: TObject);
    procedure SpinButtonXZoomUpClick(Sender: TObject);
    procedure SpinButtonXZoomDownClick(Sender: TObject);
    procedure SpinButtonYZoomUpClick(Sender: TObject);
    procedure SpinButtonYZoomDownClick(Sender: TObject);
    procedure CheckBoxLoopableClick(Sender: TObject);
  private
    FOnPlayingStarted     : TNotifyEvent;
    FOnPlayingStopped     : TNotifyEvent;
    FOnRecordingStarted   : TNotifyEvent;
    FOnRecordingStopped   : TNotifyEvent;
    FOnPauseStarted       : TNotifyEvent;
    FOnPauseStopped       : TNotifyEvent;
    FOnLoopingStarted     : TNotifyEvent;
    FOnLoopingStopped     : TNotifyEvent;
  private
    FShown                : Boolean;
    FColWidths            : array[ 0 .. 1] of Integer;
    FGraphMode            : TKnobsDMGraphMode;
    FXZoom                : TSignal;
    FYZoom                : TSignal;
  private
    FEditor               : TKnobsWirePanel;
    FAutomationData       : TKnobsEditHistory;
    FViewers              : TKnobsViewers;
    FLockTime             : Boolean;
    FLockValue            : Boolean;
    FLoopable             : Boolean;
    FViewTimed            : Boolean;
    FCurrentTime          : TKnobsTimeStamp;
    FCurrentIndex         : Integer;
    FModuleName           : string;
    FCursorPosition       : TPoint;
    FRecording            : Boolean;
    FPlaying              : Boolean;
    FLooping              : Boolean;
    FReversed             : Boolean;
    FPaused               : Boolean;
    FStructureChanged     : Boolean;
    FUsedWidth            : Integer;
    FAssociate            : TKnobsValuedControl;
    FPlayPointer          : Integer;
  private
    FLastTickTime         : TKnobsTimeStamp;
    FTickTimeDelta        : TKnobsTimeStamp;
    FlargestTickTimeDelta : TKnobsTimeStamp;
  private
    function    GetDetailsWidth   : Integer;
    procedure   SetDetailsWidth   ( aValue: Integer);
    procedure   SetCursorPosition ( const aValue: TPoint);
    procedure   FixButtonEnables;
    procedure   SetRecording      ( aValue: Boolean);
    procedure   SetPlaying        ( aValue: Boolean);
    procedure   SetLooping        ( aValue: Boolean);
    procedure   SetPaused         ( aValue: Boolean);
    function    GetFreeCount      : Integer;
    function    GetFillCount      : Integer;
    function    GetSize           : Integer;
    procedure   SetPlayPointer    ( aValue: Integer);
    function    GetUseAlternateTitles: Boolean;
    procedure   SetUseAlternateTitles( aValue: Boolean);
    procedure   SetGraphMode      ( aValue: TKnobsDMGraphMode);
    procedure   SetXZoom          ( aValue: TSignal);
    procedure   SetYZoom          ( aValue: TSignal);
    procedure   FixTimeNameLabel;
    procedure   SetCurrentTime    ( const aValue: TKnobsTimeStamp);
    procedure   SetCurrentIndex   ( const aValue: Integer);
    procedure   SetModuleName     ( const aValue: string);
    procedure   SetAssociate      ( const aValue: TKnobsValuedControl);
    procedure   SetLockTime       ( aValue: Boolean);
    procedure   SetLockValue      ( aValue: Boolean);
    procedure   SetLoopable       ( aValue: Boolean);
    procedure   SetViewTimed      ( aValue: Boolean);
    function    GetRecordAll      : Boolean;
    procedure   SetRecordAll      ( aValue: Boolean);
  private
    procedure   ClearGraphs;
    procedure   FillListView;
    procedure   ListViewClicked( const anItem: TListItem; Selected: Boolean);
    procedure   AddEditData( const aSender: TKnobsValuedControl; const aData: PKnobsEditData);
    procedure   StatusUpdate;
    procedure   BlinkUpdate;
    procedure   HandleGridPointChanged( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    procedure   IncXZoom;
    procedure   DecXZoom;
    procedure   IncYZoom;
    procedure   DecYZoom;
    procedure   DoFrameClose( const aSender: TObject; anId: Integer);
  public
    procedure   Initialize( anEditor: TKnobsWirePanel);
    procedure   UpdateModuleNames( const anOldNames, aNewNames: TStringArray);
    procedure   ClearAutomation;
    procedure   ResetAutomation;
    procedure   TakeSnapShot;
    procedure   Stop;
    procedure   RegisterCallbacks(
      anOnPlayingStarted   : TNotifyEvent;
      anOnPlayingStopped   : TNotifyEvent;
      anOnRecordingStarted : TNotifyEvent;
      anOnRecordingStopped : TNotifyEvent;
      anOnPauseStarted     : TNotifyEvent;
      anOnPauseStopped     : TNotifyEvent;
      anOnLoopingStarted   : TNotifyEvent;
      anOnLoopingStopped   : TNotifyEvent
    );
    procedure   UnregisterCallbacks;
    procedure   SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    procedure   LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
    procedure   FillStatusFeedback( const aLabel: TLabel);
    procedure   FillLedFeedback( const aLed: TKnobsLed);
    procedure   FlagStructureChanged;
    procedure   RemoveValuedControl( aControl: TKnobsValuedControl);
    procedure   FixTransportButtons( aRecordButton, aPlayButton, aPauseButton, aLoopButton: TSpeedButton);
    procedure   SaveAutomation;
    procedure   LoadAutomation;
    procedure   FixAutomation;
    procedure   Tick( CurrentTime: TKnobsTimeStamp);
    procedure   Clock;
  private
    property    DetailsWidth       : Integer             read GetDetailsWidth       write SetDetailsWidth;
    property    Associate          : TKnobsValuedControl read FAssociate            write SetAssociate;
    property    UseAlternateTitles : Boolean             read GetUseAlternateTitles write SetUseAlternateTitles;
    property    GraphMode          : TKnobsDMGraphMode   read FGraphMode            write SetGraphMode;
    property    XZoom              : TSignal             read FXZoom                write SetXZoom;
    property    YZoom              : TSignal             read FYZoom                write SetYZoom;
    property    LockTime           : Boolean             read FLockTime             write SetLockTime;
    property    LockValue          : Boolean             read FLockValue            write SetLockValue;
    property    Loopable           : Boolean             read FLoopable             write SetLoopable;
    property    ViewTimed          : Boolean             read FViewTimed            write SetViewTimed;
    property    RecordAll          : Boolean             read GetRecordAll          write SetRecordAll;
  public
    property    Recording          : Boolean             read FRecording            write SetRecording;
    property    Playing            : Boolean             read FPlaying              write SetPlaying;
    property    Looping            : Boolean             read FLooping              write SetLooping;
    property    Reversed           : Boolean             read FReversed             write FReversed;
    property    Paused             : Boolean             read FPaused               write SetPaused;
    property    CursorPosition     : TPoint              read FCursorPosition       write SetCursorPosition;
    property    CurrentTime        : TKnobsTimeStamp     read FCurrentTime          write SetCurrentTime;
    property    CurrentIndex       : Integer             read FCurrentIndex         write SetCurrentIndex;
    property    ModuleName         : string              read FModuleName           write SetModuleName;
    property    FreeCount          : Integer             read GetFreeCount;
    property    FillCount          : Integer             read GetFillCount;
    property    PlayPointer        : Integer             read FPlayPointer          write SetPlayPointer;
    property    Size               : Integer             read GetSize;
  end;


var

  FormAutomation: TFormAutomation;


implementation


// User area

const

  DIGITS = [ '0' .. '9'];


{ ========
  TScrollBox = class( Vcl.Forms.TScrollBox)
  protected
}
    procedure   TScrollBox.CreateParams( var Params: TCreateParams); // override;
    begin
      inherited;
      Params.ExStyle := Params.ExStyle or WS_EX_RIGHTSCROLLBAR;
    end;

{ ========
  TCheckBox = class( Vcl.StdCtrls.TCheckBox)
  private
    FOriginalCaption : string;
    FSetCap          : Boolean;
  protected
}

    procedure   TCheckBox.WMPaint( var aMsg: TWMPaint); // message WM_PAINT;
    var
      BtnWidth : Integer;
      aCanvas  : TControlCanvas;
    begin
      BtnWidth := GetSystemMetrics( SM_CXMENUCHECK);
      FSetCap  := True;

      if   not ( csDesigning in ComponentState)
      then Caption := '';

      FSetCap := False;
      inherited;

      aCanvas := TControlCanvas.Create;

      try
        aCanvas.Control := Self;
        aCanvas.Font    := Font;
        SetBkMode( aCanvas.Handle, Ord( TRANSPARENT));
        aCanvas.TextOut( BtnWidth + 1, 2, FOriginalCaption);
      finally
        aCanvas.DisposeOf;
      end;
    end;


    procedure   TCheckBox.CMTextChanged( var aMsg: TMessage); // message CM_TEXTCHANGED:
    begin
      inherited;

      if   FSetCap
      then Exit;

      FOriginalCaption := Caption;
    end;


{ ========
  TKnobsViewerItem = record
    Index      : Integer;
    Viewer     : TFrameViewer;
    Control    : TKnobsValuedControl;
    FData      : TSignalPairArray;
    FPosition  : TSignal;
  public
    property    Count                  : Integer           read GetCount;
    property    Visible                : Boolean           read GetVisible   write SetVisible;
    property    GraphMode              : TKnobsDMGraphMode read GetGraphMode write SetGraphMode;
    property    Duration               : TSignal           read GetDuration;
    property    Position               : TSignal           read FPosition    write SetPosition;
  public
    property    Value[ aTime: TSignal] : TSignal           read GetValue;
  private
}

    function    TKnobsViewerItem.GetCount: Integer;
    begin
      Result := Length( FData);
    end;


//  public

    function    TKnobsViewerItem.GetVisible: Boolean;
    begin
      if   Assigned( Viewer)
      then Result := Viewer.Visible
      else Result := False;
    end;


    procedure   TKnobsViewerItem.SetVisible( aValue : Boolean);
    begin
      if   aValue <> Visible
      then begin
        if   Assigned( Viewer)
        then Viewer.Visible := aValue;

        if   Visible
        then SyncGraphToData;
      end;
    end;


    function    TKnobsViewerItem.GetGraphMode: TKnobsDMGraphMode;
    begin
      if   Assigned( Viewer)
      then Result := Viewer.GraphMode
      else Result := dmgLinear;
    end;


    procedure   TKnobsViewerItem.SetGraphMode( aValue : TKnobsDMGraphMode);
    begin
      if   Assigned( Viewer)
      then Viewer.GraphMode := aValue;
    end;


    function    TKnobsViewerItem.GetDuration: TSignal;
    begin
      if   Length( FData) >= 2
      then Result := FData[ Count - 1].X - FData[ 0].X
      else Result := 0;
    end;


    procedure   TKnobsViewerItem.SetPosition( aValue: TSignal);
    begin
      if   aValue <> FPosition
      then FPosition := aValue;
    end;


    function    TKnobsViewerItem.GetValue( aTime: TSignal): TSignal;
    begin
      Result := 0;

      if   Duration > 0
      then begin
        Position := aTime / Duration;

        case GraphMode of
          dmgLinear  : Result := InterpolateSignalPairArray( FData, Position)  ;
          dmgSplines : Result := FSpline.Value(                     Position).Y;
          dmgSteps   : Result := FData[ Trunc( FPosition)                   ].Y;
        end;
      end
      else Position := 0;
    end;


//  private

    procedure   TKnobsViewerItem.SyncGraphToData;
    begin
      Viewer.SetData( FData);
    end;


//  public

    procedure   TKnobsViewerItem.PointChanged( anIndex: Integer; anX, anY: TSignal);
    begin
      if   ( anIndex >= 0)
      and  ( anIndex < Count)
      and  (( FData[ anIndex].X <> anX) or ( FData[ anIndex].Y <> anY))
      then begin
        FData[ anIndex].X := anX;
        FData[ anIndex].Y := anY;
      end;
    end;


    procedure   TKnobsViewerItem.PointAdded( anIndex: Integer; anX, anY: TSignal);
    begin
      if   ( anIndex >= 0    )
      and  ( anIndex <= Count)
      then begin
        SetLength( FData, Count + 1);
        Move( FData[ anIndex], FData[ anIndex + 1], ( Count - anIndex) * SizeOf( TSignalPair));
        FData[ anIndex].X := anX;
        FData[ anIndex].Y := anY;
      end;
    end;


    procedure   TKnobsViewerItem.PointRemoved( anIndex: Integer; anX, anY: TSignal);
    begin
      if   ( anIndex >= 0    )
      and  ( anIndex <  Count)
      then begin
        Move( FData[ anIndex + 1], FData[ anIndex], ( Count - anIndex - 1) * SizeOf( TSignalPair));
        SetLength( FData, Count - 1);
      end;
    end;


    procedure   TKnobsViewerItem.ClearGraph;
    begin
      SetLength( FData, 2);
      FData[ 0].X := 0.0;
      FData[ 0].Y := 0.0;
      FData[ 1].X := 1.0;
      FData[ 1].Y := 0.0;
      SyncGraphToData;
    end;



{ ========
  TKnobsViewers = class
  private
    FParent       : TWinControl;
    FControlIndex : TKnobsControlIndex;
    FItems        : TArray<TKnobsViewerItem>;
    FXZoom        : TSignal;
    FYZoom        : TSignal;
    FCursorX      : Integer;
    FCursorY      : Integer;
    FGraphMode    : TKnobsDMGraphMode;
    FAllowXMoves  : Boolean;
    FAllowYMoves  : Boolean;
    FViewTimed    : Boolean;
    FLoopable     : Boolean;
  public
    property    Count       : Integer           read GetCount;
    property    XZoom       : TSignal           read FXZoom       write SetXZoom;
    property    YZoom       : TSignal           read FYZoom       write SetYZoom;
    property    CursorX     : Integer           read FCursorX     write SetCursorX;
    property    CursorY     : Integer           read FCursorY     write SetCursorY;
    property    GraphMode   : TKnobsDMGraphMode read FGraphMode   write SetGraphMode;
    property    AllowXMoves : Boolean           read FAllowXMoves write SetAllowXMoves;
    property    AllowYMoves : Boolean           read FAllowYMoves write SetAllowYMoves;
    property    Loopable    : Boolean           read FLoopable    write SetLoopable;
    property    ViewTimed   : Boolean           read FViewTimed   write SetViewTimed;
  private
}

    function    TKnobsViewers.GetCount: Integer;
    begin
      Result := Length( FItems);
    end;


    procedure   TKnobsViewers.SetXZoom( aValue: TSignal);
    begin
      if   aValue <> FXZoom
      then begin
        FXZoom := aValue;
        ArrangeItems;
      end;
    end;


    procedure   TKnobsViewers.SetYZoom( aValue: TSignal);
    begin
      if   aValue <> FYZoom
      then begin
        FYZoom := aValue;
        ArrangeItems;
      end;
    end;


    procedure   TKnobsViewers.SetCursorX( aValue: Integer);
    var
      i : Integer;
    begin
      if   aValue <> FCursorX
      then begin
        FCursorX := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.CursorX := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetCursorY( aValue: Integer);
    var
      i : Integer;
    begin
      if   aValue <> FCursorY
      then begin
        FCursorY := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.CursorY := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetGraphMode( aValue : TKnobsDMGraphMode);
    var
      i : Integer;
    begin
      if   aValue <> FGraphMode
      then begin
        FGraphMode := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].GraphMode := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetAllowXMoves( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FAllowXMoves
      then begin
        FAllowXMoves := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.AllowXMoves := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetAllowYMoves( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FAllowYMoves
      then begin
        FAllowYMoves := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.AllowYMoves := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetLoopable( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FLoopable
      then begin
        FLoopable := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.Loopable := aValue;
      end;
    end;


    procedure   TKnobsViewers.SetViewTimed( aValue: Boolean);
    var
      i : Integer;
    begin
      if   aValue <> FViewTimed
      then begin
        FViewTimed := aValue;

        for i := 0 to Count - 1
        do  FItems[ i].Viewer.ViewTimed := aValue;
      end;
    end;


// private

    procedure   TKnobsViewers.ArrangeItems;
    var
      i : Integer;
      t : Integer;
      h : Integer;
      w : Integer;
    begin
      t := 0;
      w := Round( 250 * XZoom);
      h := Round( 250 * YZoom);

      for i := 0 to Count - 1
      do begin
        if   FItems[ i].Visible
        then begin
          FItems[ i].Viewer.SetBounds( 0, t, w, h);
          t := t + h;
        end;
      end;
    end;


    procedure   TKnobsViewers.Show( anIndex: Integer);
    begin
      if   ( anIndex >= 0   )
      and  ( anIndex < Count)
      then begin
        FItems[ anIndex].Visible := True;
        ArrangeItems;
      end;
    end;


    procedure   TKnobsViewers.Hide( anIndex: Integer);
    begin
      if   ( anIndex >= 0   )
      and  ( anIndex < Count)
      then begin
        FItems[ anIndex].Visible := False;
        ArrangeItems;
      end;
    end;


    procedure   TKnobsViewers.DoPointChanged(const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    var
      V : TFrameViewer;
    begin
      if   Assigned( aSender)
      and  ( aSender is TFrameViewer)
      then V := TFrameViewer( aSender);
    end;


    procedure   TKnobsViewers.DoPointAdded( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    var
      V : TFrameViewer;
    begin
      if   Assigned( aSender)
      and  ( aSender is TFrameViewer)
      then V := TFrameViewer( aSender);
    end;


    procedure   TKnobsViewers.DoPointRemoved( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    var
      V : TFrameViewer;
    begin
      if   Assigned( aSender)
      and  ( aSender is TFrameViewer)
      then V := TFrameViewer( aSender);
    end;


    procedure   TKnobsViewers.DoFrameClose( const aSender: TObject; anId: Integer);
    begin
      if   Assigned( FOnFrameClose)
      then FOnFrameClose( aSender, anId);
    end;


//  public

    constructor TKnobsViewers.Create( const aParent: TWinControl; const aHistory: TKnobsEditHistory);
    begin
      Assert( Assigned( aParent ));
      Assert( Assigned( aHistory));
      inherited Create;
      FParent  := aParent;
      FHistory := aHistory;
      Clear;
      SetFromHistory;
    end;


    destructor  TKnobsViewers.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    procedure   TKnobsViewers.Clear;
    var
      i : Integer;
    begin
      for i := Count - 1 downto 0
      do FItems[ i].Viewer.DisposeOf;

      SetLength( FItems, 0);
    end;


    procedure   TKnobsViewers.AddItem( anIndex: Integer; const aControl: TKnobsValuedControl; const aParamName: string);
    var
      Item   : TKnobsViewerItem;
      Viewer : TFrameViewer;
    begin
      if   Assigned( aControl)
      and  ( anIndex >= 0    )
      and  ( anIndex <  Count)
      then begin
        Viewer                := TFrameViewer.Create( nil);
        Item.Index            := anIndex;
        Item.Control          := aControl;
        Item.Viewer           := Viewer;
        Viewer.Parent         := FParent;
        Viewer.Initialize;
        Viewer.Id             := anIndex;
        Viewer.Control        := aControl;
        Viewer.GraphMode      := GraphMode;
        Viewer.AllowXMoves    := AllowXMoves;
        Viewer.AllowYMoves    := AllowYMoves;
        Viewer.ViewTimed      := ViewTimed;
        Viewer.OnPointChanged := DoPointChanged;
        Viewer.OnPointAdded   := DoPointAdded;
        Viewer.OnPointRemoved := DoPointRemoved;
        Viewer.OnFrameClose   := DoFrameClose;
        Viewer.ParamName      := aParamName;
        Viewer.Visible        := False;
        FItems[ anIndex]      := Item;
      end;
    end;


    procedure   TKnobsViewers.SetFromHistory;
    begin
    end;


    procedure   TKnobsViewers.ClearGraphs;
    var
      i : Integer;
    begin
      for i := 0 to Count - 1
      do  FItems[ i].ClearGraph;
    end;


{ ========
  TFormAutomation = class(TForm)
  private
}

    function    TFormAutomation.GetDetailsWidth: Integer;
    begin
      Result := PanelDetails.Width;
    end;


    procedure   TFormAutomation.SetDetailsWidth( aValue: Integer);
    begin
      PanelDetails.Width := aValue;
    end;


    procedure   TFormAutomation.SetCursorPosition( const aValue: TPoint);
    begin
    end;


    procedure   TFormAutomation.FixButtonEnables;
    begin
      FixTransportButtons( SpeedButtonRecord, SpeedButtonPlay, SpeedButtonPause, SpeedButtonLoop);
    end;


    procedure   TFormAutomation.SetRecording( aValue: Boolean);
    begin
      if   aValue <> FRecording
      then begin
        FRecording             := aValue;
        SpeedButtonRecord.Down := FRecording;

        if   FRecording
        then begin
          Playing := False;

          if   Assigned( FOnRecordingStarted)
          then FOnRecordingStarted( Self);
        end
        else begin
          if   Assigned( FOnRecordingStopped)
          then FOnRecordingStopped( Self);
        end;

        if   Assigned( FEditor)
        and  FRecording
        then begin
          if   Assigned( FAutomationData)
          and  ( FAutomationData.FillCount = 0)
          then FEditor.CreateValueSnapShot( FAutomationData);
        end;

        FixButtonEnables;
      end
    end;


    procedure   TFormAutomation.SetPlaying( aValue: Boolean);
    begin
      if   Assigned( FAutomationData)
      then begin
        if   aValue <> FPlaying
        then begin
          FPlaying             := aValue;
          SpeedButtonPlay.Down := FPlaying;

          if   FPlaying
          then begin
            Recording := False;
            FAutomationData.FixTimes;

            if   Assigned( FOnPlayingStarted)
            then FOnPlayingStarted( Self);
          end
          else begin
            if   Assigned( FOnPlayingStopped)
            then FOnPlayingStopped( Self);

            if   Reversed
            then PlayPointer := FUsedWidth
            else PlayPointer := -1;
          end;

          FixButtonEnables;
        end;
      end;
    end;


    procedure   TFormAutomation.SetLooping ( aValue: Boolean);
    begin
      if   aValue <> FLooping
      then begin
        FLooping := aValue;
        SpeedButtonLoop.Down := FLooping;

        if   FLooping
        then begin
          if   Assigned( FOnLoopingStarted)
          then FOnLoopingStarted( Self);
        end
        else begin
          if   Assigned( FOnLoopingStopped)
          then FOnLoopingStopped( Self);
        end;

        FixButtonEnables;
      end;
    end;


    procedure   TFormAutomation.SetPaused( aValue: Boolean);
    begin
      if   aValue <> FPaused
      then begin
        FPaused := aValue;
        SpeedButtonPause.Down := FPaused;

        if   FPaused
        then begin
          if   Assigned( FOnPauseStarted)
          then FOnPauseStarted( Self);
        end
        else begin
          if   Assigned( FOnPauseStopped)
          then FOnPauseStopped( Self);
        end;

        FixButtonEnables;
      end;
    end;


    function    TFormAutomation.GetFreeCount: Integer;
    begin
      if   Assigned( FAutomationData)
      then Result := FAutomationData.FreeCount
      else Result := 0;
    end;


    function    TFormAutomation.GetFillCount: Integer;
    begin
      if   Assigned( FAutomationData)
      then Result := FAutomationData.FillCount
      else Result := 0;
    end;


    function    TFormAutomation.GetSize: Integer;
    begin
      if   Assigned( FAutomationData)
      then Result := FAutomationData.Size
      else Result := 0;
    end;


    procedure   TFormAutomation.SetPlayPointer( aValue: Integer);
    begin
      if   aValue <> FPlayPointer
      then begin
        FPlayPointer   := aValue;
        CursorPosition := Point( aValue, CursorPosition.Y);
      end;
    end;


    function    TFormAutomation.GetUseAlternateTitles: Boolean;
    begin
      if   Assigned( FEditor)
      then Result := FEditor.UseAlternateTitles
      else Result := False;
    end;


    procedure   TFormAutomation.SetUseAlternateTitles( aValue: Boolean);
    begin
      if   Assigned( FEditor)
      then FEditor.UseAlternateTitles := aValue;
    end;


    procedure   TFormAutomation.SetGraphMode( aValue: TKnobsDMGraphMode);
    begin
      if   aValue <> FGraphMode
      then begin
        FGraphMode := aValue;

        if   Assigned( FViewers)
        then FViewers.GraphMode := aValue;

        if   Assigned( datagraph_mode)
        then datagraph_mode.KnobPosition := Ord( aValue);
      end;
    end;


    procedure   TFormAutomation.SetXZoom( aValue: TSignal);
    begin
      if   aValue <> XZoom
      then begin
        FXZoom         := aValue;
        FViewers.XZoom := aValue;
      end;
    end;


    procedure   TFormAutomation.SetYZoom( aValue: TSignal);
    begin
      if   aValue <> YZoom
      then begin
        FYZoom         := aValue;
        FViewers.YZoom := aValue;
      end;
    end;


    procedure   TFormAutomation.FixTimeNameLabel;
    begin
      if   ViewTimed
      then LabelTime.Caption := Format( '%s - %s'      , [ FormatTime64ms( CurrentTime), ModuleName], AppLocale)
      else LabelTime.Caption := Format( 'index %d - %s', [ CurrentIndex                , ModuleName], AppLocale);
    end;


    procedure   TFormAutomation.SetCurrentTime( const aValue: TKnobsTimeStamp);
    begin
      if   aValue <> FCurrentTime
      then begin
        FCurrentTime := aValue;

        if   ViewTimed
        then FixTimeNameLabel;
      end;
    end;


    procedure   TFormAutomation.SetCurrentIndex( const aValue: Integer);
    begin
      if   aValue <> FCurrentIndex
      then begin
        FCurrentIndex := aValue;

        if   not ViewTimed
        then FixTimeNameLabel;
      end;
    end;


    procedure   TFormAutomation.SetModuleName( const aValue: string);
    begin
      if   aValue <> FModuleName
      then begin
        FModuleName := aValue;
        FixTimeNameLabel;
      end;
    end;


    procedure   TFormAutomation.SetAssociate( const aValue: TKnobsValuedControl);
    begin
      if   aValue <> FAssociate
      then FAssociate := aValue;
    end;



    procedure   TFormAutomation.SetLockTime( aValue: Boolean);
    begin
      if   aValue <> FLockTime
      then begin
        FLockTime                :=     aValue;
        CheckBoxLockTime.Checked :=     aValue;
        FViewers.AllowXMoves     := not aValue;

        if   aValue
        then LockValue := False;
      end;
    end;


    procedure    TFormAutomation.SetLockValue( aValue: Boolean);
    begin
      if   aValue <> FLockValue
      then begin
        FLockValue                :=     aValue;
        CheckBoxLockValue.Checked :=     aValue;
        FViewers  .AllowYMoves    := not aValue;

        if   aValue
        then LockTime := False;
      end;
    end;


    procedure   TFormAutomation.SetLoopable( aValue: Boolean);
    begin
      if   aValue <> FLoopable
      then begin
        FLoopable                 := aValue;
        CheckBoxLoopable.Checked  := aValue;
        FViewers        .Loopable := aValue;
      end;
    end;


    procedure   TFormAutomation.SetViewTimed( aValue: Boolean);
    const
      Caps : array[ Boolean] of string = (
        'Lock index',
        'Lock time'
      );
    begin
      if   aValue <> FViewTimed
      then begin
        FViewTimed                  := aValue;
        CheckBoxViewTimed.Checked   := aValue;
        CheckBoxLockTime .Caption   := Caps[ aValue];
        FViewers         .ViewTimed := aValue;
      end;
    end;


    function    TFormAutomation.GetRecordAll: Boolean;
    begin
      if   Assigned( FAutomationData)
      then Result := FAutomationData.RecordAll
      else Result := False;
    end;


    procedure   TFormAutomation.SetRecordAll( aValue: Boolean);
    begin
      if   aValue <> RecordAll
      then begin
        if   Assigned( FAutomationData)
        then FAutomationData.RecordAll := aValue;

        CheckBoxRecordAll.Checked := RecordAll;
        FViewers.Clear;
        FillListView;
      end;
    end;


// private

    procedure   TFormAutomation.ClearGraphs;
    begin
      if   Assigned( FViewers)
      then FViewers.ClearGraphs;
    end;


    procedure   TFormAutomation.FillListView;
    var
      i          : Integer;
      j          : Integer;
      M          : TKnobsCustomModule;
      C          : TKnobsValuedControl;
      ModuleName : string;
      Item       : TListItem;
      OldIndex   : Integer;
    begin
      OldIndex := ListView.ItemIndex;
      ListView.Clear;

      if   Assigned( FEditor        )
      and  Assigned( FAutomationData)
      then begin
        for i := 0 to FEditor.ModuleCount - 1
        do begin
          M          := FEditor.Module[ i];
          ModuleName := Format( 'mod %d  %s', [ i, M.Title], AppLocale);

          for j := 0 to M.ControlCount - 1
          do begin
            if   M.Controls[ j] is TKnobsValuedControl
            then begin
              C := TKnobsValuedControl( M.Controls[ j]);

              if   FAutomationData.CanRecord( C)
              then begin
                Item := ListView.Items.Add;
                Item.Data    := M.Controls[ j];
                Item.Caption := ModuleName;
                Item.SubItems.Add( C.ShortName);
              end;
            end;
          end;
        end;

        ListView.ItemIndex := -1;
        ClearGraphs;

        if   ( OldIndex >= 0)
        then begin
          if   OldIndex < ListView.Items.Count
          then ListView.ItemIndex := OldIndex
          else ListView.ItemIndex := ListView.Items.Count - 1;
        end
        else begin
          if   ListView.Items.Count > 0
          then ListView.ItemIndex := 0;
        end;
      end;
    end;


    procedure   TFormAutomation.ListViewClicked( const anItem: TListItem; Selected: Boolean);
    begin
      if   Assigned( FEditor        )
      and  Assigned( FAutomationData)
      and  Assigned( FViewers       )
      then begin
        if   Assigned( anItem)
        then begin
          if   Selected
          then begin
            Associate := TObject( anItem.Data) as TKnobsValuedControl;

            if   Assigned( Associate)
            then FEditor.SetActiveInDesigner( Format( '%s_%s', [ Associate.ModuleName, Associate.Name], AppLocale), True);

            FViewers.Show( anItem.Index);;
          end
          else FViewers.Hide( anItem.Index);
        end;
      end;
    end;


    procedure   TFormAutomation.AddEditData( const aSender: TKnobsValuedControl; const aData: PKnobsEditData);
    begin
      if   Assigned( FAutomationData)
      and  Assigned( aSender        )
      and  Recording
      and  not Paused
      and  FAutomationData.CanRecord( aSender)
      then FAutomationData.AddData( aData^);
    end;


    procedure   TFormAutomation.StatusUpdate;
    begin
      if  Visible
      then begin
        if   FStructureChanged
        then begin
          FViewers.Clear;
          FillListView;
          FStructureChanged := False;
        end;
      end;
    end;


    procedure   TFormAutomation.BlinkUpdate;
    begin
      if   Visible
      then FillLedFeedback( KnobsLedAutomationActive);
    end;


    procedure   TFormAutomation.HandleGridPointChanged( const aSender: TObject; const aPath: string; anIndex: Integer; anX, anY: Double);
    begin
      if   Assigned( Associate)
      and  ( FUsedWidth > 0)
      then begin
      end;
    end;


    procedure   TFormAutomation.IncXZoom;
    begin
      if   XZoom < 1024.0
      then XZoom := XZoom * SQRT2;
    end;


    procedure   TFormAutomation.DecXZoom;
    begin
      if   XZoom > 1
      then XZoom := XZoom * SQRT2_REC;
    end;


    procedure   TFormAutomation.IncYZoom;
    begin
      if   YZoom < 1024.0
      then YZoom := YZoom * SQRT2;
    end;


    procedure   TFormAutomation.DecYZoom;
    begin
      if   YZoom > 0.25
      then YZoom := YZoom * SQRT2_REC;
    end;


    procedure   TFormAutomation.DoFrameClose( const aSender: TObject; anId: Integer);
    begin
      ListView.Items[ anId].Selected := False;
    end;


// public

    procedure   TFormAutomation.Initialize( anEditor: TKnobsWirePanel);
    begin
      if   Assigned( FEditor)
      then FEditor.OnAddEditData := nil;

      FEditor := anEditor;

      if   Assigned( FEditor)
      then FEditor.OnAddEditData := AddEditData;

      FViewers.Clear;
      FillListView;
    end;


    procedure   TFormAutomation.UpdateModuleNames( const anOldNames, aNewNames: TStringArray);
    var
      OldCursor : TPoint;
    begin
      if   Assigned( FAutomationData)
      then begin
        OldCursor := CursorPosition;
        FAutomationData.UpdateNames( anOldNames, aNewNames);
        FStructureChanged := True;
        CursorPosition := OldCursor;
      end;
    end;


    procedure   TFormAutomation.ClearAutomation;
    begin
      if   Assigned( FAutomationData)
      and  not Playing
      then begin
        FAutomationData.Clear;

        if   FRecording
        then TakeSnapShot
        else FStructureChanged := True;
      end;
    end;


    procedure   TFormAutomation.ResetAutomation;
    begin
      if   Assigned( FAutomationData)
      and  not Recording
      then PlayPointer := 0;
    end;


    procedure   TFormAutomation.TakeSnapShot;
    begin
      if   Assigned( FAutomationData)
      and  Assigned( FEditor)
      and  Assigned( FViewers)
      and  not Playing
      then begin
        FEditor.CreateValueSnapShot( FAutomationData);
        PlayPointer := PlayPointer + 1;
      end;
    end;


    procedure   TFormAutomation.Stop;
    begin
      Recording := False;
      Playing   := False;
      Paused    := False;
    end;


    procedure   TFormAutomation.RegisterCallbacks(
      anOnPlayingStarted   : TNotifyEvent;
      anOnPlayingStopped   : TNotifyEvent;
      anOnRecordingStarted : TNotifyEvent;
      anOnRecordingStopped : TNotifyEvent;
      anOnPauseStarted     : TNotifyEvent;
      anOnPauseStopped     : TNotifyEvent;
      anOnLoopingStarted   : TNotifyEvent;
      anOnLoopingStopped   : TNotifyEvent
    );
    begin
      FOnPlayingStarted   := anOnPlayingStarted  ;
      FOnPlayingStopped   := anOnPlayingStopped  ;
      FOnRecordingStarted := anOnRecordingStarted;
      FOnRecordingStopped := anOnRecordingStopped;
      FOnPauseStarted     := anOnPauseStarted    ;
      FOnPauseStopped     := anOnPauseStopped    ;
      FOnLoopingStarted   := anOnLoopingStarted  ;
      FOnLoopingStopped   := anOnLoopingStopped  ;
    end;


    procedure   TFormAutomation.UnregisterCallbacks;
    begin
      FOnPlayingStarted   := nil;
      FOnPlayingStopped   := nil;
      FOnRecordingStarted := nil;
      FOnRecordingStopped := nil;
      FOnPauseStarted     := nil;
      FOnPauseStopped     := nil;
      FOnLoopingStarted   := nil;
      FOnLoopingStopped   := nil;
    end;


    procedure   TFormAutomation.SaveIni( const aSectionName: string; const anIniFile: TMemIniFile);
    var
      i : Integer;
    begin
      if   Assigned( anIniFile)
      then begin
        with anIniFile
        do begin
          EraseSection( aSectionName);
          WriteInteger( aSectionName, 'Left'          , Left                  );
          WriteInteger( aSectionName, 'Top'           , Top                   );
          WriteInteger( aSectionName, 'Width'         , Width                 );
          WriteInteger( aSectionName, 'Height'        , Height                );
          WriteInteger( aSectionName, 'WindowState'   , Integer( WindowState) );
          WriteInteger( aSectionName, 'DetailsWidth'  , DetailsWidth          );
          WriteInteger( aSectionName, 'GraphMode'     , Ord( GraphMode)       );
          WriteFloat  ( aSectionName, 'XZoom'         , XZoom                 );
          WriteFloat  ( aSectionName, 'YZoom'         , YZoom                 );
          WriteBool   ( aSectionName, 'LockTime'      , LockTime              );
          WriteBool   ( aSectionName, 'LockValue'     , LockValue             );
          WriteBool   ( aSectionName, 'ViewTimed'     , ViewTimed             );
          WriteBool   ( aSectionName, 'RecordAll'     , RecordAll             );

          if   FShown
          then begin
            for i := Low( FColWidths) to High( FColWidths)
            do  WriteInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), ListView.Columns[ i].Width);
          end
          else begin
            for i := Low( FColWidths) to High( FColWidths)
            do  WriteInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), FColWidths[ i]);
          end;
        end;
      end;
    end;


    procedure   TFormAutomation.LoadIni( const aSectionName: string; const anIniFile: TMemIniFile);
    var
      L, T, W, H : Integer;
      WinState   : TWindowState;
      i          : Integer;
    begin
      if   Assigned( anIniFile)
      then begin
        with anIniFile
        do begin
          L              :=                    ReadInteger( aSectionName, 'Left'          , Left                  );
          T              :=                    ReadInteger( aSectionName, 'Top'           , Top                   );
          W              :=                    ReadInteger( aSectionName, 'Width'         , Width                 );
          H              :=                    ReadInteger( aSectionName, 'Height'        , Height                );
          WinState       := TWindowState     ( ReadInteger( aSectionName, 'WindowState'   , Integer( WindowState)));
          DetailsWidth   :=                    ReadInteger( aSectionName, 'DetailsWidth'  , DetailsWidth          );
          GraphMode      := TKnobsDMGraphMode( ReadInteger( aSectionName, 'GraphMode'     , Ord( GraphMode      )));
          XZoom          :=                    ReadFloat  ( aSectionName, 'XZoom'         , XZoom                 );
          YZoom          :=                    ReadFloat  ( aSectionName, 'YZoom'         , YZoom                 );
          LockTime       :=                    ReadBool   ( aSectionName, 'LockTime'      , LockTime              );
          LockValue      :=                    ReadBool   ( aSectionName, 'LockValue'     , LockValue             );
          ViewTimed      :=                    ReadBool   ( aSectionName, 'ViewTimed'     , ViewTimed             );
          RecordAll      :=                    ReadBool   ( aSectionName, 'RecordAll'     , RecordAll             );

          if ( L < 0) or ( L > Screen.Width  - 20) then L := 0;
          if ( T < 0) or ( T > Screen.Height - 20) then T := 0;
          if W > Screen.Width                      then W := Screen.Width  - 20;
          if H > Screen.Height                     then H := Screen.Height - 20;

          SetBounds( L, T, W, H);

          if   WinState = wsMaximized
          then WindowState := wsMaximized;

          for i := Low( FColWidths) to High( FColWidths)
          do  FColWidths[ i] := ReadInteger( aSectionName, Format( 'ColWidth%d', [ i], AppLocale), ListView.Columns[ i].Width)
        end;
      end;
    end;


    procedure   TFormAutomation.FillStatusFeedback( const aLabel: TLabel);
    begin
      if   Assigned( aLabel)
      then begin
      end;
    end;


    procedure   TFormAutomation.FillLedFeedback( const aLed: TKnobsLed);
    begin
      if   Assigned( aLed      )
      then begin
        if   Playing
        then begin
          if   Paused
          then begin
            aLed.Active  := True;
            aLed.ColorOn := clGreen;
            aLed.Hint    := 'Play pause';
          end
          else begin
            aLed.Active  := True;
            aLed.ColorOn := clLime;
            aLed.Hint    := 'Playing';
          end;
        end
        else if Recording
        then begin
          if   Paused
          then begin
            aLed.Active  := True;
            aLed.ColorOn := clMaroon;
            aLed.Hint    := 'Rec. pause';
          end
          else begin
            aLed.Active  := True;
            aLed.ColorOn := clRed;
            aLed.Hint    := 'Recording';
          end;
        end
        else begin
          aLed.Active := False;
          aLed.Hint   := 'Automation off';
        end;
      end;
    end;


    procedure   TFormAutomation.FlagStructureChanged;
    begin
      FStructureChanged := True;
    end;


    procedure   TFormAutomation.RemoveValuedControl( aControl: TKnobsValuedControl);
    begin
      if   Assigned( aControl)
      and  Assigned( FAutomationData)
      then begin
        FAutomationData.RemoveControl( aControl);
        FlagStructureChanged;
      end;
    end;


    procedure   TFormAutomation.FixTransportButtons( aRecordButton, aPlayButton, aPauseButton, aLoopButton: TSpeedButton);
    begin
      if   Assigned( aRecordButton)
      and  Assigned( aPlayButton  )
      and  Assigned( aPauseButton )
      and  Assigned( aLoopButton  )
      then begin
        aLoopButton  .Enabled := True;
        aPauseButton .Enabled := True;
        aPlayButton  .Enabled := True;
        aRecordButton.Enabled := True;
      end;
    end;


    procedure   TFormAutomation.SaveAutomation;
    var
      aStringList : TStringList;
    begin
      if   Assigned( FEditor)
      and  Assigned( FAutomationData)
      then begin
        FAutomationData.FixTimes;
        aStringList := TStringList.Create;

        try
          FAutomationData.SaveToStringList( aStringList, FEditor.Guid);
          aStringList.SaveToFile( ChangeFileExt( FEditor.Filename, EXT_AUTO));
        finally
          aStringList.DisposeOf;
        end;
      end;
    end;


    procedure   TFormAutomation.LoadAutomation;
    var
      aStringList : TStringList;
    begin
      if   Assigned( FEditor)
      and  Assigned( FAutomationData)
      then begin
        aStringList := TStringList.Create;

        try
          aStringList.LoadFromFile( ChangeFileExt( FEditor.Filename, EXT_AUTO));
          FAutomationData.LoadFromStringList( FEditor, aStringList, FEditor.Guid);
          FixAutomation;
          FViewers.Clear;
          FillListView;
        finally
          aStringList.DisposeOf;
        end;
      end;
    end;


    procedure   TFormAutomation.FixAutomation;
    begin
      // todo : there may be controls in the automation system that have no value set for the first and last time
      //        this makes KnobsDataMaker unhappy - look into this ...
    end;


    procedure   TFormAutomation.Tick( CurrentTime: TKnobsTimeStamp);
    begin
      // todo: Play timed, should be called once every ms and then call Clock() on time

      if   Assigned( FAutomationData)
      and  Playing
      and  not Paused
      then begin
        if   CurrentTime < FLastTickTime
        then begin
          if   CurrentTime = 0
          then FLastTickTime := CurrentTime
          else FLastTickTime := CurrentTime - 1;
        end;

        FTickTimeDelta := CurrentTime - FLastTickTime;
        FLastTickTime  := CurrentTime;

        if   FTickTimeDelta > FlargestTickTimeDelta
        then FlargestTickTimeDelta := FTickTimeDelta;
      end;
    end;


    procedure   TFormAutomation.Clock;
    // Play all items having the same timestamp as the 'current one'
    begin
    end;


// Delphi area

{$R *.dfm}

procedure TFormAutomation.CheckBoxLoopableClick(Sender: TObject);
begin
  Loopable := CheckBoxLoopable.Checked;
end;

procedure TFormAutomation.CheckBoxLockTimeClick(Sender: TObject);
begin
  LockTime := CheckBoxLockTime.Checked;
end;

procedure TFormAutomation.CheckBoxLockValueClick(Sender: TObject);
begin
  LockValue := CheckBoxLockValue.Checked;
end;

procedure TFormAutomation.CheckBoxRecordAllClick(Sender: TObject);
begin
  RecordAll := CheckBoxRecordAll.Checked;
end;

procedure TFormAutomation.CheckBoxViewTimedClick(Sender: TObject);
begin
  ViewTimed := CheckBoxViewTimed.Checked;
end;

procedure TFormAutomation.ListViewSelectItem(Sender: TObject; Item: TListItem; Selected: Boolean);
begin
  ListviewClicked( Item, Selected);
end;

procedure TFormAutomation.EditDummyChange(Sender: TObject);
begin
  if   EditDummy.Text <> ''
  then EditDummy.Text := '';
end;

procedure TFormAutomation.ControlUnfocus(const aSender: TObject);
begin
  EditDummy.SetFocus;
end;

procedure TFormAutomation.datagraph_modeValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  GraphMode := TKnobsDMGraphMode( Round( aValue));
end;

procedure TFormAutomation.BitBtnLoadRecordingClick(Sender: TObject);
begin
  LoadAutomation;
end;

procedure TFormAutomation.BitBtnSaveRecordingClick(Sender: TObject);
begin
  SaveAutomation;
end;

procedure TFormAutomation.BitBtnToggleNamesClick(Sender: TObject);
begin
  UseAlternateTitles := not UseAlternateTitles;
end;

procedure TFormAutomation.KnobsDataMakerPointChanged(const aSender: TObject; const aPath: string; anIndex: Integer; anX,
  anY: Double);
begin
  HandleGridPointChanged( aSender, aPath, anIndex, anX, anY);
end;

procedure TFormAutomation.KnobsKnobKnobPositionChanged(const aSender: TKnobsValuedControl; const aPath,
  aControlType: string; aPosition: Integer);
begin
  if   Assigned( Associate)
  then Associate.KnobPosition := aPosition;
end;

procedure TFormAutomation.KnobsLedAutomationActiveClick(Sender: TObject);
begin
  Stop;
end;

procedure TFormAutomation.KnobsSelectorClearValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  ClearAutomation;
end;

procedure TFormAutomation.KnobsSelectorSnapValueChanged(const aSender: TObject; const aPath,
  aControlType: string; aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  TakeSnapShot;
end;

procedure TFormAutomation.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if   Assigned( FEditor)
  then FEditor.SetActiveInDesigner( '', True);
end;

procedure TFormAutomation.FormCreate(Sender: TObject);
begin
  FAutomationData                := TKnobsEditHistory.Create( nil, 1, HISTORY_STR_LEN, True);
  FAutomationData.AllowNonAuto   := False;               // do not allow non automatable controls to end up in the data
  FAutomationData.AllowUnchanged := True;                // do     allow unchanged       controls to end up in the data
  FViewers                       := TKnobsViewers.Create( ScrollBoxDataMaker, FAutomationData);
  FViewers.OnFrameClose          := DoFrameClose;
  FCursorPosition                := Point(  0,  0);
  CursorPosition                 := Point( -1, -1);
  FStructureChanged              := True;
  FLockTime                      := False;
  LockTime                       := True;
  FLockValue                     := True;
  LockValue                      := False;
  FViewTimed                     := False;
  ViewTimed                      := True;
  RecordAll                      := False;
  RecordAll                      := True;
  FXZoom                         := 0.5;
  FYZoom                         := 0.5;
  XZoom                          := 1.0;
  YZoom                          := 1.0;
end;

procedure TFormAutomation.FormDestroy(Sender: TObject);
begin
  UnregisterCallbacks;
  FreeAndNil( FViewers       );
  FreeAndNil( FAutomationData);
end;

procedure TFormAutomation.FormHide(Sender: TObject);
var
  i : Integer;
begin
  for i := Low( FColWidths) to High( FColWidths)
  do  FColWidths[ i] := ListView.Columns[ i].Width;
end;

procedure TFormAutomation.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if   WindowFromPoint( Mouse.CursorPos) = ScrollBoxDataMaker.Handle
  then begin
  end;
end;

procedure TFormAutomation.FormMouseWheel(Sender: TObject; Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint;
  var Handled: Boolean);
var
  aMsg  : Cardinal;
  aCode : Cardinal;
  i     : Integer;
  n     : Integer;
begin
//  if ScrollBoxDataMaker.BoundsRect.Contains( ScreenToClient( MousePos))
  if   WindowFromPoint( Mouse.CursorPos) = ScrollBoxDataMaker.Handle
  then begin
    Handled := True;

    if   ssShift in Shift
    then begin
      aMsg := WM_HSCROLL;

      if   WheelDelta < 0
      then aCode := SB_LINEUP
      else aCode := SB_LINEDOWN;
    end
    else begin
      aMsg := WM_VSCROLL;

      if   WheelDelta > 0
      then aCode := SB_LINEUP
      else aCode := SB_LINEDOWN;
    end;

    n := Mouse.WheelScrollLines;

    for i := 0 to n
    do  ScrollBoxDataMaker.Perform( aMsg, aCode, 0);

    ScrollBoxDataMaker.Perform( aMsg, SB_ENDSCROLL, 0);
  end;
end;

procedure TFormAutomation.FormResize(Sender: TObject);
begin
  EditDummy.Left := Width + 10;
end;

procedure TFormAutomation.FormShow(Sender: TObject);
var
  i : Integer;
begin
  FStructureChanged := True;
  FShown            := True;
  EditDummy.SetFocus;
  FixButtonEnables;
  FViewers.AllowXMoves := not LockTime;
  FViewers.AllowYMoves := not LockValue;

  if   Reversed
  then PlayPointer := FUsedWidth
  else PlayPointer := -1;

  for i := Low( FColWidths) to High( FColWidths)
  do  ListView.Columns[ i].Width := FColWidths[ i];
end;

procedure TFormAutomation.TimerBlinkTimer(Sender: TObject);
begin
  BlinkUpdate;
end;

procedure TFormAutomation.TimerStatusUpdateTimer(Sender: TObject);
begin
  StatusUpdate;
end;

procedure TFormAutomation.SpeedButtonLoopClick(Sender: TObject);
begin
  Looping := SpeedButtonLoop.Down;
end;

procedure TFormAutomation.SpeedButtonPauseClick(Sender: TObject);
begin
  Paused := SpeedButtonPause.Down;
end;

procedure TFormAutomation.SpeedButtonPlayClick(Sender: TObject);
begin
  Playing := SpeedButtonPlay.Down;
end;

procedure TFormAutomation.SpeedButtonRecordClick(Sender: TObject);
begin
  Recording := SpeedButtonRecord.Down;
end;

procedure TFormAutomation.SpeedButtonStopClick(Sender: TObject);
begin
  Stop;
end;

procedure TFormAutomation.SpinButtonXZoomDownClick(Sender: TObject);
begin
  DecXZoom;
end;

procedure TFormAutomation.SpinButtonXZoomUpClick(Sender: TObject);
begin
  IncXZoom;
end;

procedure TFormAutomation.SpinButtonYZoomDownClick(Sender: TObject);
begin
  DecYZoom;
end;

procedure TFormAutomation.SpinButtonYZoomUpClick(Sender: TObject);
begin
  IncYZoom;
end;

end.

