unit FrmDesign;

{

   COPYRIGHT 2014 .. 2019 Blue Hell / Jan Punter

  Some parts are copyright :

     Author  : Neugls.
     Website : Http://www.neugls.info
     Email   : NeuglsWorkStudio@gmail.com

   these are marked with (*Neugls*)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms, Dialogs, ToolWin, StdCtrls, ComCtrls,
  ExtCtrls, Menus, ImgList, Vcl.Buttons, IniFiles, System.UITypes, System.Math,

  SpeechLib_TLB, ESpeakInterface, TaggedAudio,

  JvDesignSurface, JvDesignUtils, JvExControls, JvInspector, JvResources, Vcl.Samples.Spin,

  KnobsUtils, Knobs2013, KnobsConversions, Rationals, IntSequences, KnobsLightning,

  Globals, System.ImageList;

type

  TPixelChanger = function( aColor: TColor): TColor of object;


  TWeatherRec = record
    Direction     : Integer;  // Wind direction 0 .. 359 -> 0 .. 1
    Speed         : Integer;  // Wind speed
    Temperature   : Integer;
    Sunshine      : Integer;  // 0 - 100
    Precipitation : Integer;
    BaroPressure  : Integer;
    Sight         : Integer;
    Humidity      : Integer;  // 0 - 100
  end;


  TWeatherData = class
  private
    FReadFilename     : string;
    FWriteFilename    : string;
    FRawData          : TStringList;
    FData             : array of TWeatherRec;
    FMinSpeed         : Integer;
    FMaxSpeed         : Integer;
    FMinTemperature   : Integer;
    FMaxTemperature   : Integer;
    FMinPrecipitation : Integer;
    FMaxPrecipitation : Integer;
    FMinBaroPressure  : Integer;
    FMaxBaroPressure  : Integer;
    FMinSight         : Integer;
    FMaxSight         : Integer;
    FMinHumidity      : Integer;
    FMaxHumidity      : Integer;
  public
    constructor Create( const aFilename: string);
    destructor  Destroy;                                   override;
    procedure   Calculate;
    procedure   Write( const aFilename: string);
  end;

  TMainForm = class( TForm)
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    ImageList: TImageList;
    MainMenu: TMainMenu;
    MenuItemFile: TMenuItem;
    MenuItemOpen: TMenuItem;
    Save1: TMenuItem;
    MenuItemNew: TMenuItem;
    MenuItemSep1: TMenuItem;
    MenuItemExit: TMenuItem;
    N1: TMenuItem;
    MenuItemDesigenActive: TMenuItem;
    PageControlMain: TPageControl;
    TabSheetModuleDesign: TTabSheet;
    ToolBar: TToolBar;
    but_select: TToolButton;
    but_module: TToolButton;
    but_led: TToolButton;
    but_selector: TToolButton;
    but_knob: TToolButton;
    but_smallknob: TToolButton;
    but_noknob: TToolButton;
    but_display: TToolButton;
    but_fileselector: TToolButton;
    but_indicator: TToolButton;
    but_indicatorbar: TToolButton;
    but_indicatortext: TToolButton;
    but_dataviewer: TToolButton;
    but_input: TToolButton;
    but_output: TToolButton;
    but_textlabel: TToolButton;
    but_editlabel: TToolButton;
    but_box: TToolButton;
    but_image: TToolButton;
    JvDesignScrollBox: TJvDesignScrollBox;
    JvDesignPanel: TJvDesignPanel;
    JvInspector: TJvInspector;
    Splitter: TSplitter;
    TabSheetModuleContents: TTabSheet;
    TabSheetFonts: TTabSheet;
    FontDialog: TFontDialog;
    PanelFont: TPanel;
    Label5pt: TLabel;
    Label6pt: TLabel;
    Label7pt: TLabel;
    Label8pt: TLabel;
    Label9pt: TLabel;
    ListBoxFonts: TListBox;
    TabSheetImages: TTabSheet;
    PanelImages: TPanel;
    OpenDialogBitmap: TOpenDialog;
    ScrollBoxImages: TScrollBox;
    PanelImageControl: TPanel;
    Image: TImage;
    BitBtnSwapRB: TBitBtn;
    BitBtnLoadBitmap: TBitBtn;
    BitBtnSwapRG: TBitBtn;
    BitBtnSwapBG: TBitBtn;
    BitBtnInvertColors: TBitBtn;
    BitBtnMakeTransparent: TBitBtn;
    TabSheetFractions: TTabSheet;
    PanelFractions: TPanel;
    MemoFractions: TMemo;
    BitBtnMakeFractions: TBitBtn;
    LabelUpTo: TLabel;
    SpinEditMaxDenominator: TSpinEdit;
    BitBtnDataGridTest: TBitBtn;
    PanelDataGridTest: TPanel;
    KnobsGridControl: TKnobsGridControl;
    BitBtnStepGameOfLife: TBitBtn;
    BitBtnRunGame: TBitBtn;
    TimerGame: TTimer;
    CheckBoxFastGame: TCheckBox;
    BitBtnGridLoadSaveTest: TBitBtn;
    CheckBoxWrap: TCheckBox;
    BitBtnClearGame: TBitBtn;
    ComboBoxGameMode: TComboBox;
    PaintBoxSines: TPaintBox;
    SpinEditSineCount: TSpinEdit;
    BitBtnShowSineSum: TBitBtn;
    ComboBoxShape: TComboBox;
    Label1: TLabel;
    KnobsDisplayEta: TKnobsDisplay;
    BitBtnLightningStep: TBitBtn;
    BitBtnLightningRun: TBitBtn;
    BitBtnLightningReset: TBitBtn;
    CheckBoxShowField: TCheckBox;
    Label2: TLabel;
    KnobsDisplayRndDots: TKnobsDisplay;
    KnobsKnobRandomDots: TKnobsKnob;
    KnobsKnobEta: TKnobsKnob;
    PanelLightningTest: TPanel;
    KnobsLightningViewer: TKnobsLightningViewer;
    TabSheetSapiSpeech: TTabSheet;
    BitBtnSpeakText: TBitBtn;
    EditSpeechText: TEdit;
    MemoVoiceNames: TMemo;
    PaintBoxVoice: TPaintBox;
    TabSheetESpeak: TTabSheet;
    PaintBoxEVoice: TPaintBox;
    BitBtnESpeakText: TBitBtn;
    EditESpeechText: TEdit;
    MemoEVoiceFeedback: TMemo;
    LabelESpeakVersion: TLabel;
    ListBoxEVoices: TListBox;
    BitBtnInitializeESpeak: TBitBtn;
    LabelEVoice: TLabel;
    ListBoxELanguages: TListBox;
    LabelELanguage: TLabel;
    CheckBoxEDebugEnabled: TCheckBox;
    LabelTime: TLabel;
    Label3: TLabel;
    EditPhonemes: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure OpenButtonClick(Sender: TObject);
    procedure SaveButtonClick(Sender: TObject);
    procedure PaletteButtonClick(Sender: TObject);
    procedure MenuItemNewClick(Sender: TObject);
    procedure JvDesignPanelGetAddClass(Sender: TObject; var ioClass: String);
    procedure MenuItemExitClick(Sender: TObject);
    procedure JvDesignPanelSelectionChange(Sender: TObject);
    procedure MenuItemDesigenActiveClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure ListBoxFontsClick(Sender: TObject);
    procedure BitBtnLoadBitmapClick(Sender: TObject);
    procedure BitBtnSwapRBClick(Sender: TObject);
    procedure BitBtnSwapRGClick(Sender: TObject);
    procedure BitBtnSwapBGClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure BitBtnInvertColorsClick(Sender: TObject);
    procedure BitBtnMakeTransparentClick(Sender: TObject);
    procedure BitBtnMakeFractionsClick(Sender: TObject);
    procedure SpinEditMaxDenominatorChange(Sender: TObject);
    procedure BitBtnDataGridTestClick(Sender: TObject);
    procedure BitBtnStepGameOfLifeClick(Sender: TObject);
    procedure BitBtnRunGameClick(Sender: TObject);
    procedure TimerGameTimer(Sender: TObject);
    procedure CheckBoxFastGameClick(Sender: TObject);
    procedure BitBtnGridLoadSaveTestClick(Sender: TObject);
    procedure CheckBoxWrapClick(Sender: TObject);
    procedure BitBtnClearGameClick(Sender: TObject);
    procedure ComboBoxGameModeChange(Sender: TObject);
    procedure PaintBoxSinesPaint(Sender: TObject);
    procedure BitBtnShowSineSumClick(Sender: TObject);
    procedure SpinEditSineCountChange(Sender: TObject);
    procedure ComboBoxShapeChange(Sender: TObject);
    procedure BitBtnLightningStepClick(Sender: TObject);
    procedure BitBtnLightningRunClick(Sender: TObject);
    procedure BitBtnLightningResetClick(Sender: TObject);
    procedure KnobsKnobEtaValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
      IsFinal, IsAutomation: Boolean);
    procedure CheckBoxShowFieldClick(Sender: TObject);
    procedure KnobsKnobRandomDotsValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
      IsFinal, IsAutomation: Boolean);
    procedure BitBtnSpeakTextClick(Sender: TObject);
    procedure PaintBoxVoicePaint(Sender: TObject);
    procedure BitBtnESpeakTextClick(Sender: TObject);
    procedure PaintBoxEVoicePaint(Sender: TObject);
    procedure BitBtnInitializeESpeakClick(Sender: TObject);
    procedure ListBoxEVoicesClick(Sender: TObject);
    procedure ListBoxELanguagesClick(Sender: TObject);
    procedure CheckBoxEDebugEnabledClick(Sender: TObject);
  private
    FDesignClass : string;
    FStickyMode  : Boolean;
    FBitmap      : TBitmap;
  private
    procedure   LoadIni;
    procedure   SaveIni;
    procedure   CollectFonts;
    procedure   SelectFont;
  private
    procedure   LoadBitmap;
    function    SwapRB( aColor: TColor): TColor;
    function    SwapRG( aColor: TColor): TColor;
    function    SwapBG( aColor: TColor): TColor;
    function    InvertColors( aColor: TColor): TColor;
    function    MakeBlackTransparent( aColor: TColor): TColor;
    procedure   ChangePixels( aPixelChanger: TPixelChanger);
  private
    FFractions      : array of TRational;
    FMaxDenominator : Integer;
  private
    function    GetFractionCount: Integer;
    procedure   SetFractionCount( aValue: Integer);
    function    GetFraction( anIndex: Integer): TRational;
    procedure   SetFraction( anIndex: Integer; aValue: TRational);
    procedure   SetMaxDenominator( aValue: Integer);
    procedure   InitFractions;
    function    FindFraction( aValue: TRational): Integer;
    procedure   AddFraction( aValue: TRational);
    function    CompareFractions( V1, V2: TRational): Integer;
    procedure   SwapFractions( I1, I2: Integer);
    procedure   SortFractions;
    procedure   PrintFraction( aValue: TRational);
    procedure   PrintFractions;
    procedure   MakeFractions;
  private
    property    FractionCount               : Integer   read GetFractionCount write SetFractionCount;
    property    Fraction[ anIndex: Integer] : TRational read GetFraction      write SetFraction;
    property    MaxDenominator              : Integer   read FMaxDenominator  write SetMaxDenominator;
  private
    FGameOfLife  : TGameOfLife;
    FGameRuns    : Boolean;
    FFastGame    : Boolean;
    FWrappedGame : Boolean;
    FMode        : TLifeMode;
  private
    procedure   SetGameRuns   ( aValue: Boolean);
    procedure   SetFastGame   ( aValue: Boolean);
    procedure   SetWrappedGame( aValue: Boolean);
    procedure   SetMode       ( aValue: TLifeMode);
  private
    procedure   GameTimerFired;
    procedure   StepGame;
    procedure   DataGridTest;
    procedure   GridLoadSavetest;
    procedure   ClearGame;
  private
    property    GameRuns    : Boolean   read FGameRuns    write SetGameRuns;
    property    FastGame    : Boolean   read FFastGame    write SetFastGame;
    property    WrappedGame : Boolean   read FWrappedGame write SetWrappedGame;
    property    Mode        : TLifeMode read FMode        write SetMode;
  private
    FPoints : TSignalArray;
  private
    procedure   InitPoints;
    procedure   CalcSine( aFreq: Integer; anAmpl, aPhase: TSignal; var aPoints: TSignalArray);
    procedure   ClearPoints;
    procedure   AddPoints( const aPoints: TSignalArray);
    procedure   CalcSineSum( aPartialCount: Integer);
    procedure   PaintSinePoints( const aCanvas: TCanvas);
  private
    FLightningEngine : TLightningEngine;
    FLightningRuns   : Boolean;
    FLightningTimer  : TTimer;
    FEta             : TSignal;
    FShowField       : Boolean;
    FRandomDots      : TSignal;
  private
    procedure   SetEta       ( aValue: TSignal);
    procedure   SetShowField ( aValue: Boolean);
    procedure   SetRandomDots( aValue: TSignal);
    procedure   LightningInit;
    procedure   LightningDone;
    procedure   StartLightningTimer;
    procedure   StopLightningTimer;
    procedure   LightningTimerFired( aSender: TObject);
    procedure   ResetLightning;
    procedure   StepLightning;
    procedure   StartLightning;
    procedure   StopLightning;
  private
    property    Eta        : TSignal read FEta        write SetEta;
    property    ShowField  : Boolean read FShowField  write SetShowField;
    property    RandomDots : TSignal read FRandomDots write SetRandomDots;
  private
    FVoiceData        : TSignalArray;
  private
    procedure   TestSpeakText;
    procedure   PaintVoiceWave( aCanvas: TCanvas; aWidth, aHeight: Integer);
  private
    FSelectedVoice    : Integer;
    FSelectedLanguage : Integer;
    FSpeaker          : TESpeakSpeaker;
    FTaggedAudio      : TTaggedAudio;
    FEDebugEnabled    : Boolean;
    FTime             : TDateTime;
    FPhonemes         : string;
  private
    procedure   SetEDebugEnabled( aValue: Boolean);
  private
    procedure   StartTimer;
    procedure   StopTimer;
    procedure   HandleRenderingDone( const aSender: TObject);
    procedure   DoTextStart( const aSender: TObject                                );
    procedure   DoSentence ( const aSender: TObject; anId: Integer                 );
    procedure   DoWord     ( const aSender: TObject; anId: Integer                 );
    procedure   DoPhoneme  ( const aSender: TObject; const aPhoneme: string        );
    procedure   DoSilence  ( const aSender: TObject                                );
    procedure   DoMark     ( const aSender: TObject; const aMark   : string        );
    procedure   DoUserMark ( const aSender: TObject; anId: Integer; aValue: TSignal);
    procedure   DoTextDone ( const aSender: TObject                                );
    procedure   SelectEVoice;
    procedure   SelectELanguage;
    procedure   Debug( const aMsg: string);
    procedure   DebugFmt( const aFmt: string; const anArgs: array of const);
    procedure   EDebugHandler( const aSender: TObject; const aMsg: string);
    procedure   InitializeESpeak;
    procedure   TestESpeakText;
    procedure   PaintEVoiceWave( aCanvas: TCanvas; aWidth, aHeight: Integer);
  private
    property    EDebugEnabled : Boolean read FEDebugEnabled write SetEDebugEnabled;
  end;

var

  MainForm: TMainForm;



implementation


uses

  JvDesignImp;

// User area


  function EnumFontsProc( var LogFont: TLogFont; var TextMetric: TTextMetric; FontType: Integer; Data: Pointer): Integer; stdcall;
  var
    S    : TStrings;
    Temp : string;
  begin
    S := TStrings( Data);
    Temp := LogFont.lfFaceName;

    if (S.Count = 0) or (AnsiCompareText(S[S.Count-1], Temp) <> 0)
    then S.Add( Temp);

    Result := 1;
  end;

  function CreateFontList: TStringList;
  var
    DC    : HDC;
    LFont : TLogFont;
  begin
    Result := TStringList.Create;
    DC := GetDC( 0);
    FillChar( LFont, sizeof( LFont), 0);
    LFont.lfCharset := DEFAULT_CHARSET;
    EnumFontFamiliesEx( DC, LFont, @ EnumFontsProc, LPARAM( Result), 0);
    ReleaseDC(0, DC);
  end;



{ ========
  TWeatherRec = record
    Direction     : Integer;  // Wind direction 0 .. 359 -> 0 .. 1
    Speed         : Integer;  // Wind speed
    Temperature   : Integer;
    Sunshine      : Integer;  // 0 - 100
    Precipitation : Integer;
    BaroPressure  : Integer;
    Sight         : Integer;
    Humidity      : Integer;  // 0 - 100
  end;

  TWeatherData = class
  private
    FReadFilename     : string;
    FWriteFilename    : string;
    FRawData          : TStringList;
    FData             : array of TWeatherRec;
    FMinSpeed         : Integer;
    FMaxSpeed         : Integer;
    FMinTemperature   : Integer;
    FMaxTemperature   : Integer;
    FMinPrecipitation : Integer;
    FMaxPrecipitation : Integer;
    FMinBarePressure  : Integer;
    FMaxBaroPresssure : Integer;
    FMinSight         : Integer;
    FMaxSight         : Integer;
    FMinHumidity      : Integer;
    FMaxHumidity      : Integer;
  public
}

    constructor TWeatherData.Create( const aFilename: string);
    var
      i : Integer;
    begin
      inherited Create;
      FReadFilename := aFilename;
      FRawData := TStringList.Create;
      if FileExists( FReadFilename)
      then begin
        FRawData.LoadFromFile( FReadFilename);
        i := 0;
        while i < FRawData.Count
        do begin
          if Pos( '#', FRawData[ i]) > 0
          then FRawData.Delete( i)
          else Inc( i);
        end;
      end;
    end;

    destructor  TWeatherData.Destroy; // override;
    begin
      FreeAndNil( FRawData);
      inherited;
    end;

    procedure   TWeatherData.Calculate;
    var
      aLine  : string;
      aParts : TStringList;
      i      : Integer;
      p      : Integer;
    begin
      FMinSpeed         :=   MaxInt;
      FMaxSpeed         := - MaxInt;
      FMinTemperature   :=   MaxInt;
      FMaxTemperature   := - MaxInt;
      FMinPrecipitation :=   MaxInt;
      FMaxPrecipitation := - MaxInt;
      FMinBaroPressure  :=   MaxInt;
      FMaxBaroPressure  := - MaxInt;
      FMinSight         :=   MaxInt;
      FMaxSight         := - MaxInt;
      FMinHumidity      :=   MaxInt;
      FMaxHumidity      := - MaxInt;
      SetLength( FData, FRawData.Count);

      p := 0;
      for aLine in FRawData
      do begin
        aParts := Explode( aLine, ',');
        try
          for i := 0 to aParts.Count - 1
          do aParts[ i] := Trim( aParts[ i]);

          with FData[ p]
          do begin
            Direction := StrToIntDef( aParts[ 2], 0) mod 360;

            Speed := StrToIntDef( aParts[ 3], 0);
            if Speed < FMinSpeed then FMinSpeed := Speed;
            if Speed > FMaxSpeed then FMaxSpeed := Speed;

            Temperature := StrToIntDef( aParts[ 4], 0);
            if Temperature < FMinTemperature then FMinTemperature := Temperature;
            if Temperature > FMaxTemperature then FMaxTemperature := Temperature;

            Sunshine := StrToIntDef( aParts[ 5], 0);

            Precipitation := StrToIntDef( aParts[ 6], 0);
            if Precipitation < 0
            then Precipitation := 0;
            if Precipitation < FMinPrecipitation then FMinPrecipitation := Precipitation;
            if Precipitation > FMaxPrecipitation then FMaxPrecipitation := Precipitation;

            BaroPressure := StrToIntDef( aParts[ 7], 0);
            if BaroPressure < FMinBaroPressure then FMinBaroPressure := BaroPressure;
            if BaroPressure > FMaxBaroPressure then FMaxBaroPressure := BaroPressure;

            Sight := StrToIntDef( aParts[ 8], 0);
            if Sight < FMinSight then FMinSight := Sight;
            if Sight > FMaxSight then FMaxSight := Sight;

            Humidity := StrToIntDef( aParts[ 9], 0);
            if Humidity < FMinHumidity then FMinHumidity := Humidity;
            if Humidity > FMaxHumidity then FMaxHumidity := Humidity;
          end;
        finally
          aParts.Free;
        end;
        Inc( p);
      end;
    end;

    procedure   TWeatherData.Write( const aFilename: string);

      function Scale( const aName: string; aValue, aMin, aMax: Integer): Double;
      begin
        Result := ( aValue - aMin) / ( aMax - aMin);
      end;

    var
      i     : Integer;
      aData : TStringList;
    begin
      FWriteFilename := aFilename;
      aData := TStringList.Create;
      try
        aData.Add( Format( '# MinSpeed         = %d', [ FMinSpeed         ]));
        aData.Add( Format( '# MaxSpeed         = %d', [ FMaxSpeed         ]));
        aData.Add( Format( '# MinTemperature   = %d', [ FMinTemperature   ]));
        aData.Add( Format( '# MaxTemperature   = %d', [ FMaxTemperature   ]));
        aData.Add( Format( '# MinPrecipitation = %d', [ FMinPrecipitation ]));
        aData.Add( Format( '# MaxPrecipitation = %d', [ FMaxPrecipitation ]));
        aData.Add( Format( '# MinBaroPressure  = %d', [ FMinBaroPressure  ]));
        aData.Add( Format( '# MaxBaroPressure  = %d', [ FMaxBaroPressure  ]));
        aData.Add( Format( '# MinSight         = %d', [ FMinSight         ]));
        aData.Add( Format( '# MaxSight         = %d', [ FMaxSight         ]));
        aData.Add( Format( '# MinHumidity      = %d', [ FMinHumidity      ]));
        aData.Add( Format( '# MaxHumidity      = %d', [ FMaxHumidity      ]));

        for i := 0 to Length( FData) - 1
        do begin
          with FData[ i]
          do begin
            aData.Add(
              Format( '%.4f, %.4f, %.4f, %.4f, %.4f, %.4f, %.4f, %.4f',
                [
                  Direction / 360,
                  Scale( 'speed', Speed, FMinSpeed, FMaxSpeed),
                  Scale( 'temperature', Temperature, FMinTemperature, FMaxTemperature),
                  Sunshine / 100,
                  Scale( 'precipitation', Precipitation, FMinPrecipitation, FMaxPrecipitation),
                  Scale( 'baropressure', BaroPressure, FMinBaroPressure, FMaxBAroPressure),
                  Scale( 'sight', Sight, FMinSight, FMaxSight),
                  Scale( 'humidity', Humidity, FMinHumidity, FMaxHumidity)
                ]
              )
            );
          end;
        end;
        aData.SaveToFile( FWriteFilename);
      finally
        aData.Free;
      end;
    end;


  function Color2RGB( aColor: TColor):Integer;
  (*Neugls*)
  begin
    Result := 0;
    Result := Result or ((( aColor shr  0) and $ff) shl 16);
    Result := Result or ((( aColor shr  8) and $ff) shl  8);
    Result := Result or ((( aColor shr 16) and $ff) shl  0);
  end;


type

  TJvInspectorColorItem = class( TJvCustomInspectorItem)
  (*Neugls*)
  protected
    function    GetColorName( const Color: TColor): string;
    procedure   PaintValue( const Color: TColor; const ColorName: string;const ACanvas: TCanvas; const ARect: TRect);
  protected
    procedure   SetFlags( const Value: TInspectorItemFlags);                                                   override;
    procedure   SetRects( const RectKind: TInspectorPaintRect; Value: TRect);                                  override;
    procedure   ButtonClick( Sender: TObject);                                                                 override;
    function    GetDisplayValue: string;                                                                       override;
  public
    constructor Create( const AParent: TJvCustomInspectorItem; const AData: TJvCustomInspectorData);           override;
    procedure   DrawValue( const ACanvas: TCanvas);                                                            override;
  public
    class procedure RegisterAsDefaultItem;
    class procedure UnregisterAsDefaultItem;
  end;

  TJvInspectorControlTypeItem = class( TJvCustomInspectorItem)
  protected
    procedure   SetFlags( const Value: TInspectorItemFlags);                                                   override;
  public
    constructor Create( const AParent: TJvCustomInspectorItem; const AData: TJvCustomInspectorData);           override;
  public
    class procedure RegisterAsDefaultItem;
    class procedure UnregisterAsDefaultItem;
  end;


{ ========
  TJvInspectorColorItem = class( TJvCustomInspectorItem)
  protected
}

    function    TJvInspectorColorItem.GetColorName( const Color: TColor): string;
    (*Neugls*)
    var
      RGBColor:Integer;
    begin
      RGBColor := Color2RGB( Color);
      Result   := '$'+ IntToHex( RGBColor, 6);
    end;

    procedure   TJvInspectorColorItem.PaintValue( const Color: TColor; const ColorName: string;const ACanvas: TCanvas; const ARect: TRect);
    (*Neugls*)
    var
      TH      : Integer;
      BoxRect : TRect;
      bc      : TColor;
      pc      : TColor;
      txtRect : TRect;
    begin
      TH             := Rects[ iprValue].Bottom - Rects[ iprValue].Top - 2;
      BoxRect.Left   := ARect.Left + ( ARect.Bottom - ARect.Top - TH) div 2;
      BoxRect.Top    := ARect.Top  + BoxRect.Left - ARect.Left;
      BoxRect.Right  := BoxRect.Left + TH;
      BoxRect.Bottom := BoxRect.Top  + TH;
      with ACanvas do
      begin
        if Color <> clNone then
        begin
          bc := Brush.Color;
          pc := Pen.Color;
          try
            Brush.Color := Color;
            Pen.Color   := clBlack;
            Rectangle(BoxRect);
          finally
            Pen.Color := pc;
            Brush.Color := bc;
          end;
        end;
       txtRect := ARect;
       txtRect.Left := txtRect.Left + (txtRect.Bottom-txtRect.Top)+ 1;
       TextRect(txtRect, txtRect.Left, BoxRect.Top, ColorName);
      end;
    end;

//  protected

    procedure   TJvInspectorColorItem.SetFlags( const Value: TInspectorItemFlags); // override;
    (*Neugls*)
    begin
      inherited SetFlags( Value + [ iifEditButton, iifEditFixed]);
    end;

    procedure   TJvInspectorColorItem.SetRects( const RectKind: TInspectorPaintRect; Value: TRect); // override;
    (*Neugls*)
    begin
      if RectKind = iprValue
      then Value.Left := Value.Left + (Value.Bottom - Value.Top) + 2;
      inherited SetRects( RectKind, Value);
    end;

    procedure   TJvInspectorColorItem.ButtonClick( Sender: TObject); // override;
    (*Neugls*)
    var
      ColorDlg: TColorDialog;
    begin
      ColorDlg := TColorDialog.Create( nil);
      try
        if ColorDlg.Execute
        then begin
       // DisplayValue := NameForColor( FColor);
          Data.AsOrdinal := ColorDlg.Color;
        end;
      finally
        ColorDlg.Free;
      end;
      inherited;
    end;

    function    TJvInspectorColorItem.GetDisplayValue: string; // override;
    (*Neugls*)
    begin
      Result := GetColorName( Data.AsOrdinal);
    end;

//  public

    constructor TJvInspectorColorItem.Create( const AParent: TJvCustomInspectorItem; const AData: TJvCustomInspectorData); // override;
    (*Neugls*)
    begin
      inherited Create( AParent, AData);
      Flags := [ iifVisible, iifEditButton, iifEditFixed];
    end;

    procedure   TJvInspectorColorItem.DrawValue( const ACanvas: TCanvas); // override;
    (*Neugls*)
    var
      Color     : TColor;
      S         : string;
      ARect     : TRect;
      SafeColor : TColor;
    begin
      Color := clNone;

      if Data = nil
      then S := RsJvInspItemUnInitialized
      else begin
        try
          if not Data.IsInitialized
          then S := RsJvInspItemUnInitialized
          else if not Data.HasValue
          then S := RsJvInspItemNoValue
          else if not Data.IsAssigned
          then S := RsJvInspItemUnassigned
          else begin
            S     := DisplayValue;
            Color := Data.AsOrdinal;
          end;
        except
          on E: Exception
          do S := RsJvInspItemValueException + E.ClassName + ': ' + E.Message;
        end;

        ARect     := Rects[ iprValueArea];
        SafeColor := ACanvas.Brush.Color;

        if Editing
        then ACanvas.Brush.Color := clWindow;

        try
          ACanvas.FillRect( ARect);
          PaintValue( Color, S, ACanvas, ARect);
          if Editing
          then DrawEditor( ACanvas);
        finally
          if Editing
          then ACanvas.Brush.Color := SafeColor;
        end;
      end;
    end;

//  public

    class procedure TJvInspectorColorItem.RegisterAsDefaultItem;
    (*Neugls*)
    begin
      with TJvCustomInspectorData.ItemRegister
      do begin
        if IndexOf( Self) = -1
        then Add( TJvInspectorTypeInfoRegItem.Create( Self, TypeInfo( TColor)));
      end;
    end;

    class procedure TJvInspectorColorItem.UnregisterAsDefaultItem;
    (*Neugls*)
    begin
      TJvCustomInspectorData.ItemRegister.Delete( Self);
    end;


{ ========
  TJvInspectorControlTypeItem = class( TJvCustomInspectorItem)
  protected
}

    procedure   TJvInspectorControlTypeItem.SetFlags( const Value: TInspectorItemFlags); // override;
    begin
      inherited SetFlags( Value + [ iifEditButton, iifEditFixed, iifValueList]);
    end;

//  public

    constructor TJvInspectorControlTypeItem.Create( const AParent: TJvCustomInspectorItem; const AData: TJvCustomInspectorData); // override;
    begin
      inherited Create( AParent, AData);
      Flags := [ iifVisible, iifEditButton, iifEditFixed, iifValueList];
    end;

//  public

    class procedure TJvInspectorControlTypeItem.RegisterAsDefaultItem;
    begin
      with TJvCustomInspectorData.ItemRegister
      do begin
        if IndexOf( Self) = -1
        then Add( TJvInspectorTypeInfoRegItem.Create( Self, TypeInfo( TColor)));
      end;
    end;

    class procedure TJvInspectorControlTypeItem.UnregisterAsDefaultItem;
    begin
      TJvCustomInspectorData.ItemRegister.Delete( Self);
    end;


//  private

const

  sdfs = 'Defaults';
  sdws = 'window';
  hist = 'CmdHistory';

    procedure   TMainForm.LoadIni;
    var
      anIniFile    : TMemIniFile;
      L, T, W, H   : Integer;
      WinState     : TWindowState;
    begin
      anIniFile := TMemIniFile.Create( IniFileName);
      with anIniFile
      do begin
        try
          L                        :=                ReadInteger( sdws, 'Left'          ,          Left           );
          T                        :=                ReadInteger( sdws, 'Top'           ,          Top            );
          W                        :=                ReadInteger( sdws, 'Width'         ,          Width          );
          H                        :=                ReadInteger( sdws, 'Height'        ,          Height         );
          WinState                 := TWindowState(  ReadInteger( sdws, 'WindowState'   , Integer( WindowState )) );
          PageControlMain.TabIndex :=                ReadInteger( sdws, 'TabIndex'      , PageControlMain.TabIndex);
          Eta                      :=                ReadFloat  ( sdws, 'Eta'           , Eta                     );
          ShowField                :=                ReadBool   ( sdws, 'ShowField'     , ShowField               );
          RandomDots               :=                ReadFloat  ( sdws, 'RandomDots'    , RandomDots              );

          if ( L < 0) or ( L > Screen.Width  - 20) then L := 0;
          if ( T < 0) or ( T > Screen.Height - 20) then T := 0;
          if W > Screen.Width                      then W := Screen.Width  - 20;
          if H > Screen.Height                     then H := Screen.Height - 20;

          SetBounds( L, T, W, H);
          if WinState = wsMaximized
          then WindowState := wsMaximized;

        finally
          Free;
        end;
      end;
    end;

    procedure   TMainForm.SaveIni;
    var
      anIniFile : TMemIniFile;
    begin
      anIniFile := TMemIniFile.Create( IniFileName);
      with anIniFile
      do begin
        try
          EraseSection( sdfs);
          if WindowState = wsNormal
          then begin
            WriteInteger( sdws, 'Left'       , Left       );
            WriteInteger( sdws, 'Top'        , Top        );
            WriteInteger( sdws, 'Width'      , Width      );
            WriteInteger( sdws, 'Height'     , Height     );
          end;
          WriteInteger( sdws, 'WindowState'  , Integer( WindowState)   );
          WriteInteger( sdws, 'TabIndex'     , PageControlMain.TabIndex);
          WriteFloat  ( sdws, 'Eta'          , Eta                     );
          WriteBool   ( sdws, 'ShowField'    , ShowField               );
          WriteFloat  ( sdws, 'RandomDots'   , RandomDots              );
        finally
          UpdateFile;
          Free;
        end;
      end;
    end;

    procedure   TMainForm.CollectFonts;
    var
      aStringList : TStringList;
    begin
      aStringList := CreateFontList;
      if Assigned( aStringList)
      then begin
        try
          ListBoxFonts.Items := aStringList;
        finally
          aStringList.DisposeOf;
        end;
      end;
    end;

    procedure   TMainForm.SelectFont;
    var
      aFont : TFont;
    begin
      if ListBoxFonts.ItemIndex >= 0
      then begin
        aFont := TFont.Create;
        try
          aFont.Name := ListBoxFonts.Items[ ListBoxFonts.ItemIndex];
          aFont.Size := 5;
          Label5pt.Font := aFont;
          aFont.Size := 6;
          Label6pt.Font := aFont;
          aFont.Size := 7;
          Label7pt.Font := aFont;
          aFont.Size := 8;
          Label8pt.Font := aFont;
          aFont.Size := 9;
          Label9pt.Font := aFont;
        finally
          aFont.DisposeOf;
        end;
      end;
    end;


//  private

    procedure   TMainForm.LoadBitmap;
    begin
      with OpenDialogBitmap
      do begin
        if Execute
        then begin
          FBitmap.LoadFromFile( FileName);
          FBitmap.PixelFormat := pf32bit;
          Image.Picture.Bitmap.Assign( FBitmap);
        end;
      end;
    end;


    function    TMainForm.SwapRB( aColor: TColor): TColor;
    begin
      Result := RGB( GetBValue( aColor), GetGValue( aColor), GetRValue( aColor));
    end;


    function    TMainForm.SwapRG( aColor: TColor): TColor;
    begin
      Result := RGB( GetGValue( aColor), GetRValue( aColor), GetBValue( aColor));
    end;


    function    TMainForm.SwapBG( aColor: TColor): TColor;
    begin
      Result := RGB( GetRValue( aColor), GetBValue( aColor), GetGValue( aColor));
    end;


    function    TMainForm.InvertColors( aColor: TColor): TColor;
    begin
      Result := RGB( 255 - GetRValue( aColor), 255 - GetGValue( aColor), 255 - GetBValue( aColor));
    end;


    function    TMainForm.MakeBlackTransparent( aColor: TColor): TColor;
    begin
      Result := TColor((255 - GetRValue( aColor)) shl 24) + aColor;
    end;


    procedure   TMainForm.ChangePixels( aPixelChanger: TPixelChanger);
    var
      i      : Integer;
      j      : Integer;
    begin
      Assert( FBitmap.PixelFormat = pf32bit);

      for i := 0 to FBitmap.Width - 1
      do begin
        for j := 0 to FBitmap.Height - 1
        do FBitmap.Canvas.Pixels[ i, j] := aPixelChanger( FBitmap.Canvas.Pixels[ i, j])
      end;
      Image.Picture.Bitmap.Assign( FBitmap);
    end;

{ ========
  private
    FFractions      : array of TRational;
    FMaxDenominator : Integer;
  private
    property    FractionCount               : Integer   read GetFractionCount write SetFractionCount;
    property    Fraction[ anIndex: Integer] : TRational read GetFraction      write SetFraction;
    property    MaxDenominator              : Integer   read FMaxDenominator  write SetMaxDenominator;
  private
}

    function    TMainForm.GetFractionCount: Integer;
    begin
      Result := Length( FFractions);
    end;


    procedure   TMainForm.SetFractionCount( aValue: Integer);
    begin
      SetLength( FFractions, aValue);
    end;


    function    TMainForm.GetFraction( anIndex: Integer): TRational;
    begin
      Result := FFractions[ anIndex];
    end;


    procedure   TMainForm.SetFraction( anIndex: Integer; aValue: TRational);
    begin
      FFractions[ anIndex] := aValue;
    end;


    procedure   TMainForm.SetMaxDenominator( aValue: Integer);
    begin
      if aValue <> FMaxDenominator
      then begin
        FMaxDenominator              := aValue;
        SpinEditMaxDenominator.Value := aValue;
      end;
    end;


    procedure   TMainForm.InitFractions;
    begin
      MaxDenominator := 32;
    end;


    function    TMainForm.FindFraction( aValue: TRational): Integer;
    var
      i : Integer;
    begin
      Result := -1;

      for i := 0 to FractionCount - 1
      do begin
        if Fraction[ i].AsDouble = aValue.AsDouble
        then begin
          Result := i;
          Break;
        end;
      end;
    end;


    procedure   TMainForm.AddFraction( aValue: TRational);
    begin
      if FindFraction( aValue) < 0
      then begin
        FractionCount := FractionCount + 1;
        Fraction[ FractionCount - 1] := aValue;
      end;
    end;


    function    TMainForm.CompareFractions( V1, V2: TRational): Integer;
    begin
      if V1 < V2
      then Result := -1
      else if V1 = V2
      then Result := 0
      else Result := 1;
    end;


    procedure   TMainForm.SwapFractions( I1, I2: Integer);
    var
      Tmp : TRational;
    begin
      Tmp           := Fraction[ I1];
      Fraction[ I1] := Fraction[ I2];
      Fraction[ I2] := Tmp;
    end;


    procedure   TMainForm.SortFractions;

      procedure DoSort( LB, UB: Integer);
      var
        i : Integer;
        j : Integer;
        p : Integer;
      begin
        repeat
          i := LB;
          j := UB;
          p := ( LB + UB) shr 1;

          repeat
            while CompareFractions( Fraction[ i], FRaction[ p]) < 0
            do Inc( i);

            while CompareFractions( Fraction[ j], Fraction[ p]) > 0
            do Dec( j);

            if i <= j
            then begin
              if i <> j
              then SwapFractions( i, j);

              if p = i
              then p := j
              else if p = j
              then p := i;

              Inc( i);
              Dec( j);
            end;
          until i > j;

          if LB < j
          then DoSort( LB, j);

          LB := i;
        until i >= UB;
      end;

    begin
      DoSort( 0, FractionCount - 1);
    end;


    procedure   TMainForm.PrintFraction( aValue: TRational);
    begin
      MemoFractions.Lines.Add( Format( '%9s -> %g', [ aValue.AsString, aValue.AsDouble], AppLocale));
    end;


    procedure   TMainForm.PrintFractions;
    var
      i : Integer;
    begin
      for i := 0 to FractionCount - 1
      do PrintFraction( Fraction[ i]);
    end;


    procedure   TMainForm.MakeFractions;
    var
      i : Integer;
      j : Integer;
      R : TRational;
    begin
      MemoFractions.Clear;

      for i := 1 to MaxDenominator
      do begin
        if
          ( i in [ 1, 2, 3, 4, 5, 6, 8, 12, 16, 24, 32, 48, 64, 96, 128, 192]) or
          ( i = 256) or ( i = 384) or ( i = 512) or ( i = 768) or ( i = 1024)
        then begin
          for j := 0 to i
          do begin
            R.Numerator   := j;
            R.Denominator := i;
            AddFraction( R);
          end;
        end;
      end;

      SortFractions;
      PrintFractions;
    end;


//  private

    procedure   TMainForm.SetGameRuns( aValue: Boolean);
    begin
      if aValue <> FGameRuns
      then begin
        FGameRuns := aValue;

        if FGameRuns
        then begin
          TimerGame.Enabled := True;
          BitBtnRunGame.Caption := 'Stop';
        end
        else begin
          TimerGame.Enabled := False;
          BitBtnRunGame.Caption := 'Run';
        end;
      end;
    end;


    procedure   TMainForm.SetFastGame( aValue: Boolean);
    begin
      if aValue <> FFastGame
      then begin
        FFastGame                := aValue;
        CheckBoxFastGame.Checked := aValue;

        if FastGame
        then begin
          KnobsGridControl.ShowCursorX := False;
          KnobsGridControl.ShowCursorY := False;
        end
        else begin
          KnobsGridControl.ShowCursorX := True;
//          KnobsGridControl.ShowCursorY := True;
        end;
      end;
    end;


    procedure   TMainForm.SetWrappedGame( aValue: Boolean);
    begin
      if aValue <> FWrappedGame
      then begin
        FWrappedGame         := aValue;
        CheckBoxWrap.Checked := aValue;
        FGameOfLife .Wrapped := lwXY;

        if WrappedGame
        then FGameOfLife.FaultRate := 0.0005
        else FGameOfLife.FaultRate := 0.01;
      end;
    end;


    procedure   TMainForm.SetMode( aValue: TLifeMode);
    begin
      if aValue <> FMode
      then begin
        FMode                      := aValue;
        ComboBoxGameMode.ItemIndex := Ord( aValue);
        FGameOfLife     .Mode      := aValue;
      end;
    end;


//  private

    procedure   TMainForm.GameTimerFired;
    begin
      if FastGame
      then StepGame
      else begin
        KnobsGridControl.CursorX := KnobsGridControl.CursorX + 1 / KnobsGridControl.DataWidth;

        if KnobsGridControl.CursorX > 1
        then begin
          KnobsGridControl.CursorX := KnobsGridControl.CursorX - 1;
          StepGame;
        end;
      end;
    end;


    procedure   TMainForm.StepGame;
    begin
      FGameOfLife.Execute;

      if FGameOfLife.FillCount < 5
      then begin
        KnobsGridControl.FillRandom( 0.05);
        KnobsGridControl.CursorX := Random( KnobsGridControl.DataWidth);
        FGameOfLife.Data         := KnobsGridControl.GetData;
      end;

      KnobsGridControl.SetData( FGameOfLife.Data);
    end;


    procedure   TMainForm.DataGridTest;
    begin
      KnobsGridControl.FillRandom( 0.2);
      KnobsGridControl.CursorX := Random( KnobsGridControl.DataWidth);
      FGameOfLife.Data         := KnobsGridControl.GetData;

      if FGameOfLife.Wrapped <> lwOff
      then FGameOfLife.FaultRate := 0.0005
      else FGameOfLife.FaultRate := 0.01;
    end;


    procedure   TMainForm.GridLoadSavetest;
    var
      S : string;
    begin
      S := KnobsGridControl.AsString;
      KnobsGridControl.AsString := S;
    end;


    procedure   TMainForm.ClearGame;
    begin
      KnobsGridControl.ClearData;
      KnobsGridControl.CursorX := Random( KnobsGridControl.DataWidth);
      FGameOfLife.Data         := KnobsGridControl.GetData;
    end;


  // private

    procedure   TMainForm.InitPoints;
    begin
      SetLength( FPoints, PaintBoxSines.Width - 2);
    end;


    procedure   TMainForm.CalcSine( aFreq: Integer; anAmpl, aPhase: TSignal; var aPoints: TSignalArray);
    var
      i : Integer;
      N : Integer;
    begin
      N := Length( aPoints);

      for i := 0 to N - 1
      do aPoints[ i] := anAmpl * Sin(( aFreq * 2 * ( i + ( N - 1) * aPhase) * Pi) / ( N - 1));
    end;


    procedure   TMainForm.ClearPoints;
    var
      i : Integer;
    begin
      for i := 0 to Length( FPoints) - 1
      do FPoints[ i] := 0;
    end;


    procedure   TMainForm.AddPoints( const aPoints: TSignalArray);
    var
      i : Integer;
    begin
      for i := 0 to Length( FPoints) - 1
      do FPoints[ i] := FPoints[ i] + aPoints[ i];
    end;


    procedure   TMainForm.CalcSineSum( aPartialCount: Integer);
    var
      f : Integer;
      P : TSignalArray;

      function OddSign( f: Integer): TSignal;
      begin
        if Odd( f)
        then Result :=   1.0
        else Result := - 1.0;
      end;

    begin
      SetLength( P, Length( FPoints));
      ClearPoints;

      case ComboBoxShape.ItemIndex of
        0 :                                                    // Saw
          begin
            for f := 1 to aPartialCount
            do begin
              CalcSine( f, OddSign( f) / ( 2 * f), 0, P);
              AddPoints( P);
            end;
          end;

        1 :                                                    // Square
          begin
            for f := 1 to aPartialCount
            do begin
              CalcSine( 2 * f - 1, 1 / ( 2 * f - 1), 0, P);
              AddPoints( P);
            end;
          end;

        2 :                                                    // Tri
          begin
            for f := 1 to aPartialCount
            do begin
              CalcSine( 2 * f - 1, OddSign( f) / Sqr( 2 * f - 1), 0, P);
              AddPoints( P);
            end;
          end;
      end;

      PaintBoxSines.Invalidate;
    end;


    procedure   TMainForm.PaintSinePoints( const aCanvas: TCanvas);
    var
      X : Integer;
      Y : Integer;
      S : TSignal;
    begin
      aCanvas.Pen.Color   := clBlue;
      aCanvas.Pen.Width   := 1;
      aCanvas.Pen.Style   := psSolid;
      aCanvas.Brush.Color := clWhite;
      aCanvas.Brush.Style := bsSolid;

      aCanvas.Rectangle( 0, 0, PaintBoxSines.Width, PaintBoxSines.Height);

      S := 0;

      for X := 0 to Length( FPoints) - 1
      do begin
        if Abs( FPoints[ X]) > S
        then S := Abs( FPoints[ X]);
      end;

      if S <> 0
      then begin
        S := 1 / S;

        for X := 0 to Length( FPoints) - 1
        do begin
          Y := Round( 0.5 * ( PaintBoxSines.Height - 3) * ( 1 - S * FPoints[ X]));

          if X = 0
          then aCanvas.MoveTo( X + 1, Y + 1)
          else aCanvas.LineTo( X + 1, Y + 1);
        end;
      end;

      aCanvas.MoveTo( 1                  , PaintBoxSines.Height div 2);
      aCanvas.LineTo( PaintBoxSines.Width, PaintBoxSines.Height div 2);
    end;


{ ========
  private
    FLightningEngine : TLightningEngine;
    FLightningRuns   : Boolean;
    FLightningTimer  : TTimer;
    FEta             : TSignal;
    FShowField       : Boolean;
    FRandomDots      : TSignal;
  private
    property    Eta        : TSignal read FEta        write SetEta;
    property    ShowField  : Boolean read FShowField  write SetShowField;
    property    RandomDots : TSignal read FRandomDots write SetRandomDots;
  private
}

    procedure   TMainForm.SetEta( aValue: TSignal);
    begin
      if aValue <> FEta
      then begin
        FEta                      := aValue;
        KnobsKnobEta.KnobPosition := Round( FEta - 1);

        if Assigned( FLightningEngine)
        then FLightningEngine.Eta := FEta;
      end;
    end;


    procedure   TMainForm.SetShowField( aValue: Boolean);
    begin
      if aValue <> FShowField
      then begin
        FShowField                := aValue;
        CheckboxShowField.Checked := aValue;

        if Assigned( FLightningEngine)
        then FLightningEngine.ShowField := FShowField;

        if Assigned( KnobsLightningViewer)
        then KnobsLightningViewer.ShowField := FShowField;
      end;
    end;


    procedure   TMainForm.SetRandomDots( aValue: TSignal);
    begin
      if aValue <> FRandomDots
      then begin
        FRandomDots                      := aValue;
        KnobsKnobRandomDots.KnobPosition := Round( FRandomDots);

        if Assigned( FLightningEngine)
        then FLightningEngine.RandomNew := FRandomDots / 10000;
      end;
    end;


    procedure   TMainForm.LightningInit;
    begin
      Eta            := 6;
      ShowField      := True;
      FLightningRuns := False;
      RandomDots     := 0;

      FLightningEngine := TLightningEngine.Create( KnobsLightningViewer.Width - 2, KnobsLightningViewer.Height - 2, Eta, RandomDots / 10000, True);
      FLightningEngine.ShowField := True;

      FLightningTimer := TTimer.Create( nil);
      FLightningTimer.Enabled  := False;
      FLightningTimer.OnTimer  := LightningTimerFired;
      FLightningTimer.Interval := 50;

      KnobsLightningViewer.RawPotential := FLightningEngine.RawPotential;
      KnobsLightningViewer.RawIsShape   := FLightningEngine.RawIsShape;
      KnobsLightningViewer.Invalidate;
//    KnobsLightningViewer.AsString := FLightningEngine.AsString;
    end;


    procedure   TMainForm.LightningDone;
    begin
      FLightningRuns := False;
      FLightningTimer.Enabled  := False;
      FLightningTimer.OnTimer  := nil;
      FreeAndNil( FLightningTimer );
      FreeAndNil( FLightningEngine);
    end;


    procedure   TMainForm.StartLightningTimer;
    begin
      FLightningTimer.Enabled := True;
    end;


    procedure   TMainForm.StopLightningTimer;
    begin
      FLightningTimer.Enabled := False;
    end;


    procedure   TMainForm.LightningTimerFired( aSender: TObject);
    begin
      if FLightningRuns
      then StepLightning
      else StopLightningTimer;
    end;


    procedure   TMainForm.ResetLightning;
    begin
      FLightningEngine.Reset;
      KnobsLightningViewer.RawPotential := FLightningEngine.RawPotential;
      KnobsLightningViewer.RawIsShape   := FLightningEngine.RawIsShape;
      KnobsLightningViewer.Invalidate;
    end;


    procedure   TMainForm.StepLightning;
    var
      i : Integer;
    begin
      for i := 1 to 20
      do FLightningEngine.Step;

      KnobsLightningViewer.RawPotential := FLightningEngine.RawPotential;
      KnobsLightningViewer.RawIsShape   := FLightningEngine.RawIsShape;
      KnobsLightningViewer.Invalidate;
   // KnobsLightningViewer.AsString := FLightningEngine.AsString;
    end;


    procedure   TMainForm.StartLightning;
    begin
      FLightningRuns := True;
      StartLightningTimer;
      BitBtnLightningRun.Caption := 'Stop';
    end;


    procedure   TMainForm.StopLightning;
    begin
      FLightningRuns := False;
      BitBtnLightningRun.Caption := 'Run';
    end;


//  private

    procedure   TMainForm.TestSpeakText;
    var
      aVoice    : TSpVoice;
      aStream   : TSpMemoryStream;
      aFormat   : TSpAudioFormat;
      aData     : OleVariant;
      aByteData : array of Byte;
      aLength   : Integer;
      aRead     : Integer;
      i         : Integer;
      aTokens   : ISpeechObjectTokens;
      aToken    : ISpeechObjectToken;
      aName     : string;
      aVendor   : string;
      anAge     : string;
      aGender   : string;
      aLanguage : string;
      aRegKey   : string;
      anIntVal  : SmallInt;
      aHiByte   : Byte;
      aLoByte   : Byte;
    begin
      MemoVoiceNames.Clear;
      aVoice  := TSpVoice.Create( nil);
      aVoice.EventInterests := SVEAllEvents;
      aTokens := aVoice.GetVoices( '', '');

      for i := 0 to aTokens.Count - 1
      do MemoVoiceNames.Lines.Add( aTokens.Item( i).GetDescription( 0));

      aToken    := aTokens.Item( aTokens.Count - 1);
      aName     := aToken.GetAttribute( 'Name'    );
      aVendor   := aToken.GetAttribute( 'Vendor'  );
      anAge     := aToken.GetAttribute( 'Age'     );
      aGender   := aToken.GetAttribute( 'Gender'  );
      aLanguage := aToken.GetAttribute( 'Language');
      aRegKey   := aToken.Id;

      aVoice.Voice  := aToken;
      aVoice.Rate   := 0;
      aVoice.Volume := 100;

      aStream := TSpMemoryStream.Create( nil);
      aFormat := TSpAudioFormat .Create( nil);
      aFormat.type_ := SAFT44kHz16BitMono;

      aVoice.AudioOutputStream := aStream.DefaultInterface;
      aStream.Format := aFormat.DefaultInterface;
      aVoice.Speak( EditSpeechText.Text, SVSFDefault or SVSFIsXML or SVSFPurgeBeforeSpeak);

      aLength := aStream.Seek( 0, SSSPTRelativeToEnd);
      aStream.Seek( 0, SSSPTRelativeToStart);

      aRead := aStream.Read( aData, aLength);
      SetLength( aByteData , aRead      );
      SetLength( FVoiceData, aRead div 2);

      for i := 0 to aRead - 1
      do aByteData[ i] := aData[ i];

      for i := 0 to ( aRead div 2) - 1
      do begin
        aLoByte := aByteData[ 2 * i    ];
        aHiByte := aByteData[ 2 * i + 1];
        anIntVal       := Integer( aLoByte) + Integer( 256 * aHiByte);
        FVoiceData[ i] := 0.5 + anIntVal / 32768.0;
      end;

      aVoice .DisposeOf;
      aStream.DisposeOf;
      aFormat.DisposeOf;
      PaintBoxVoice.Invalidate;
    end;


    procedure   TMainForm.PaintVoiceWave( aCanvas: TCanvas; aWidth, aHeight: Integer);

      function ScaleY( aValue: Double): Integer;
      begin
        Result := 1 + Round(( aHeight - 2) * ( 1 - aValue));
      end;

    var
      i          : Integer;
      Decimation : Integer;
    begin
      if   Assigned( aCanvas)
      then begin
        aCanvas.Pen.Color   := clBlack;
        aCanvas.Brush.Color := clWhite;
        aCanvas.Rectangle( 0, 0, aWidth - 1, aHeight - 1);
        Decimation := Length( FVoiceData) div aWidth;

        if Length( FVoiceData) > 1
        then begin
          aCanvas.Pen.Color := clBlue;
          aCanvas.MoveTo( 0, ScaleY( FVoiceData[ 0]));

          for i := 1 to Length( FVoiceData) - 1
          do begin
            if i mod Decimation = 0
            then aCanvas.LineTo( i div Decimation, ScaleY( FVoiceData[ i]));
          end;
        end;
      end;
    end;


//  private

    procedure   TMainForm.SetEDebugEnabled( aValue: Boolean);
    begin
      if aValue <> FEDebugEnabled
      then begin
        FEDebugEnabled                := aValue;
        CheckBoxEDebugEnabled.Checked := aValue;
      end;
    end;


//  private

    procedure   TMainForm.StartTimer;
    begin
      FTime := Now;
    end;


    procedure   TMainForm.StopTimer;
    var
      Elapsed : TDateTime;
      S       : string;
    begin
      Elapsed := Now - FTime;
      S := FormatDateTime( 'nn:ss.zzz', Elapsed);
      LabelTime.Caption := Format( 'Time: %s', [ S], AppLocale);
    end;


    procedure   TMainForm.HandleRenderingDone( const aSender: TObject);
    var
      i      : Integer;
      aValue : TSignal;
    begin
      StopTimer;
      FTaggedAudio.RenderToWave( FVoiceData);
      PaintBoxEVoice.Invalidate;
      Update;

      for i := 0 to FTaggedAudio.SampleCount - 1
      do aValue := FTaggedAudio.InterpolateL( i);

      EditPhonemes.Text := FPhonemes;
    end;


    procedure   TMainForm.DoTextStart( const aSender: TObject);
    begin
      Debug   ( '- textstart');
    end;


    procedure   TMainForm.DoSentence( const aSender: TObject; anId: Integer);
    begin
      DebugFmt( '- sentence %d', [ anId]);
    end;


    procedure   TMainForm.DoWord( const aSender: TObject; anId: Integer);
    begin
      DebugFmt( '- word     %d', [ anId]);
    end;


    procedure   TMainForm.DoPhoneme( const aSender: TObject; const aPhoneme: string);
    begin
      DebugFmt( '- phoneme  %s', [ aPhoneme]);
      FPhonemes := FPhonemes + aPhoneme;
    end;


    procedure   TMainForm.DoSilence( const aSender: TObject);
    begin
      Debug( '- silence');
      FPhonemes := FPhonemes + '_';
    end;


    procedure   TMainForm.DoMark( const aSender: TObject; const aMark : string);
    begin
      DebugFmt( '- mark  %s', [ aMark]);
    end;


    procedure   TMainForm.DoUserMark( const aSender: TObject; anId: Integer; aValue: TSignal);
    begin
      DebugFmt( '- usermark %d %g', [ anId, aValue]);
    end;


    procedure   TMainForm.DoTextDone( const aSender: TObject);
    begin
      Debug( '- textdone');
    end;


    procedure   TMainForm.SelectEVoice;
    var
      i : Integer;
    begin
      FSelectedVoice := ListBoxEVoices.ItemIndex;
      FSpeaker.SelectVoice( FSelectedVoice);
      LabelEVoice.Caption := Format( 'Voice: %s', [ FSpeaker.VoiceName[ FSelectedVoice]], AppLocale);

      ListBoxELanguages.Clear;

      for i := 0 to FSpeaker.LanguageCount - 1
      do  ListBoxELanguages.Items.Add( FSpeaker.Language[ i]);

      if   ListBoxELanguages.Items.Count >  0
      then ListBoxELanguages.ItemIndex :=  0
      else ListBoxELanguages.ItemIndex := -1;

      SelectELanguage;
    end;


    procedure   TMainForm.SelectELanguage;
    begin
      FSelectedLanguage := ListBoxELanguages.ItemIndex;
      FSpeaker.SelectLanguage( FSelectedLanguage);
      LabelELanguage.Caption := Format( 'Language: %s', [ FSpeaker.Language[ FSelectedLanguage]], AppLocale);
    end;


    procedure   TMainForm.Debug( const aMsg: string);
    begin
      if EDebugEnabled
      then MemoEVoiceFeedback.Lines.Add( aMsg);
    end;


    procedure   TMainForm.DebugFmt( const aFmt: string; const anArgs: array of const);
    begin
      Debug( Format( aFmt, anArgs, AppLocale));
    end;


    procedure   TMainForm.EDebugHandler( const aSender: TObject; const aMsg: string);
    begin
      Debug( aMsg);
    end;


    procedure   TMainForm.InitializeESpeak;
    var
      i : Integer;
    begin
      MemoEVoiceFeedback.Clear;
      ListBoxEVoices.Clear;

      if Assigned( FTaggedAudio)
      then begin
        FTaggedAudio.DisconnectFromSpeaker( FSpeaker);
        FreeAndNil( FTaggedAudio);
      end;

      if Assigned( FSpeaker)
      then FreeAndNil( FSpeaker);

      try
        FSpeaker         := TESpeakSpeaker.Create;
        FSpeaker.OnDebug := EDebugHandler;

        LabelESpeakVersion.Caption := Format( 'Version: %s', [ FSpeaker.ESpeakVersion], AppLocale);

        for i := 0 to FSpeaker.VoiceCount - 1
        do ListboxEVoices.Items.Add( FSpeaker.VoiceName[ i]);

        if   ListBoxEVoices.Items.Count > 0
        then ListboxEVoices.ItemIndex :=  0
        else ListboxEVoices.ItemIndex := -1;

        SelectEVoice;
      except
        on E: Exception
        do begin
          MemoEVoiceFeedback.Lines.Add( E.Message);
        end;
      end;
    end;


    procedure   TMainForm.TestESpeakText;
    begin
      MemoEVoiceFeedback.Clear;
      FPhonemes := '';
      SetLength( FVoiceData, 0);

      if   Assigned( FTaggedAudio)
      then FreeAndNil( FTaggedAudio);

      if Assigned( FSpeaker)
      then begin
        try
          FTaggedAudio := TTaggedAudio.Create( FSpeaker.SampleRate);
          FTaggedAudio.ConnectToSpeaker( FSpeaker, HandleRenderingDone);
          FTaggedAudio.OnTextStart := DoTextStart;
          FTaggedAudio.OnSentence  := DoSentence;
          FTaggedAudio.OnWord      := DoWord;
          FTaggedAudio.OnPhoneme   := DoPhoneme;
          FTaggedAudio.OnSilence   := DoSilence;
          FTaggedAudio.OnMark      := DoMark;
          FTaggedAudio.OnUserMark  := DoUserMark;
          FTaggedAudio.OnTextDone  := DoTextDone;
          FSpeaker.SelectVoice( FSelectedVoice);
          StartTimer;
          FSpeaker.Speak( EditESpeechText.Text);
        except
          on E: Exception
          do MemoEVoiceFeedback.Lines.Add( E.Message);
        end;
      end
      else MemoEVoiceFeedback.Lines.Add( 'Use Initialize first');
    end;


    procedure   TMainForm.PaintEVoiceWave( aCanvas: TCanvas; aWidth, aHeight: Integer);

      function ScaleY( aValue: Double): Integer;
      begin
        Result := 1 + Round(( aHeight - 2) * ( 0.5 - aValue));
      end;

    var
      i          : Integer;
      Decimation : Integer;
    begin
      if   Assigned( aCanvas)
      then begin
        aCanvas.Pen.Color   := clBlack;
        aCanvas.Brush.Color := clWhite;
        aCanvas.Rectangle( 0, 0, aWidth - 1, aHeight - 1);
        Decimation := Length( FVoiceData) div aWidth;

        if Length( FVoiceData) > 1
        then begin
          aCanvas.Pen.Color := clBlue;
          aCanvas.MoveTo( 0, ScaleY( FVoiceData[ 0]));

          for i := 1 to Length( FVoiceData) - 1
          do begin
            if i mod Decimation = 0
            then aCanvas.LineTo( i div Decimation, ScaleY( FVoiceData[ i]));
          end;
        end;
      end;
    end;


// Delphi area

{$R *.dfm}

procedure TMainForm.PaintBoxEVoicePaint(Sender: TObject);
begin
  PaintEVoiceWave( PaintboxEVoice.Canvas, PaintBoxEVoice.Width, PaintBoxEVoice.Height);
end;

procedure TMainForm.PaintBoxSinesPaint(Sender: TObject);
begin
  PaintSinePoints( PaintBoxSines.Canvas);
end;

procedure TMainForm.PaintBoxVoicePaint(Sender: TObject);
begin
  PaintVoiceWave( PaintboxVoice.Canvas, PaintBoxVoice.Width, PaintBoxVoice.Height);
end;

procedure TMainForm.ComboBoxGameModeChange(Sender: TObject);
begin
  Mode := TLifeMode( ComboBoxGameMode.ItemIndex);
end;

procedure TMainForm.ComboBoxShapeChange(Sender: TObject);
begin
  CalcSineSum( SpinEditSineCount.Value);
end;

procedure TMainForm.CheckBoxEDebugEnabledClick(Sender: TObject);
begin
  EDebugEnabled := CheckBoxEDebugEnabled.Checked;
end;

procedure TMainForm.CheckBoxFastGameClick(Sender: TObject);
begin
  FastGame := CheckBoxFastGame.Checked;
end;

procedure TMainForm.CheckBoxShowFieldClick(Sender: TObject);
begin
  ShowField := CheckboxShowField.Checked;
end;

procedure TMainForm.CheckBoxWrapClick(Sender: TObject);
begin
  WrappedGame := CheckBoxWrap.Checked;
end;

procedure TMainForm.TimerGameTimer(Sender: TObject);
begin
  GameTimerFired;
end;

procedure TMainForm.SpinEditMaxDenominatorChange(Sender: TObject);
begin
  if SpinEditMaxDenominator.Text <> ''
  then MaxDenominator := SpinEditMaxDenominator.Value;
end;

procedure TMainForm.SpinEditSineCountChange(Sender: TObject);
begin
  CalcSineSum( SpinEditSineCount.Value);
end;

procedure TMainForm.BitBtnInitializeESpeakClick(Sender: TObject);
begin
  InitializeESpeak;
end;

procedure TMainForm.BitBtnClearGameClick(Sender: TObject);
begin
  ClearGame;
end;

procedure TMainForm.BitBtnDataGridTestClick(Sender: TObject);
begin
  DataGridTest;
end;

procedure TMainForm.BitBtnESpeakTextClick(Sender: TObject);
begin
  TestESpeakText;
end;

procedure TMainForm.BitBtnGridLoadSaveTestClick(Sender: TObject);
begin
  GridLoadSavetest;
end;

procedure TMainForm.BitBtnInvertColorsClick(Sender: TObject);
begin
  ChangePixels( InvertColors);
end;

procedure TMainForm.BitBtnLightningResetClick(Sender: TObject);
begin
  ResetLightning;
end;

procedure TMainForm.BitBtnLightningRunClick(Sender: TObject);
begin
  if FLightningRuns
  then StopLightning
  else StartLightning;
end;

procedure TMainForm.BitBtnLightningStepClick(Sender: TObject);
begin
  FLightningRuns := False;
  StepLightning;
end;

procedure TMainForm.BitBtnLoadBitmapClick(Sender: TObject);
begin
  LoadBitmap;
end;

procedure TMainForm.BitBtnMakeFractionsClick(Sender: TObject);
begin
  MakeFractions;
end;

procedure TMainForm.BitBtnMakeTransparentClick(Sender: TObject);
begin
  ChangePixels( MakeBlackTransparent);
end;

procedure TMainForm.BitBtnRunGameClick(Sender: TObject);
begin
  GameRuns := not GameRuns;
end;

procedure TMainForm.BitBtnShowSineSumClick(Sender: TObject);
begin
  CalcSineSum( SpinEditSineCount.Value);
end;

procedure TMainForm.BitBtnSpeakTextClick(Sender: TObject);
begin
  TestSpeakText;
end;

procedure TMainForm.BitBtnStepGameOfLifeClick(Sender: TObject);
begin
  StepGame;
end;

procedure TMainForm.BitBtnSwapBGClick(Sender: TObject);
begin
  ChangePixels( SwapBG);
end;

procedure TMainForm.BitBtnSwapRBClick(Sender: TObject);
begin
  ChangePixels( SwapRB);
end;

procedure TMainForm.BitBtnSwapRGClick(Sender: TObject);
begin
  ChangePixels( SwapRG);
end;

procedure TMainForm.ListBoxEVoicesClick(Sender: TObject);
begin
  SelectEVoice;
end;

procedure TMainForm.ListBoxFontsClick(Sender: TObject);
begin
  SelectFont;
end;

procedure TMainForm.ListBoxELanguagesClick(Sender: TObject);
begin
  SelectELanguage;
end;

procedure TMainForm.MenuItemExitClick(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.MenuItemDesigenActiveClick(Sender: TObject);
begin
  JvDesignPanel.Active := not JvDesignPanel.Active;
  MenuItemDesigenActive.Checked := JvDesignPanel.Active;
end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  SaveIni;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  OpenDialog.InitialDir := ExtractFilePath(Application.ExeName);
  SaveDialog.InitialDir := OpenDialog.InitialDir;
  JvDesignPanel.Surface.MessengerClass := TJvDesignWinControlHookMessenger;
// JvDesignPanel.Surface.MessengerClass := TJvDesignDesignerMessenger;
  JvDesignPanel.Surface.Active  := True;
  MenuItemDesigenActive.Checked := JvDesignPanel.Surface.Active;
  FBitmap := TBitmap.Create;
  FBitmap.PixelFormat     := pf32bit;
  FBitmap.TransparentMode := tmFixed;
  FBitmap.Transparent     := False;
  InitFractions;
  LightningInit;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  FreeAndNil( FBitmap);
  FreeAndNil( FGameOfLife);
  LightningDone;
  FreeAndNil( FSpeaker);
  FreeAndNil( FTaggedAudio);
end;

procedure TMainForm.FormShow(Sender: TObject);
begin
  LoadIni;

  if not Assigned( FGameOfLife)
  then FGameOfLife := TGameOfLife.Create( KnobsGridControl.DataWidth, KnobsGridControl.DataHeight, nil);

  InitPoints;
  CollectFonts;
  DataGridTest;
  FastGame := True;
  SpinEditSineCount.Value := 5;
  MemoEVoiceFeedback.Clear;
  EdebugEnabled := False;
  EDebugEnabled := True;
  EditPhonemes.Text := '';
end;

procedure TMainForm.JvDesignPanelGetAddClass(Sender: TObject;
  var ioClass: String);
begin
  ioClass := FDesignClass;

  if not FStickyMode
  then begin
    FDesignClass := '';
    but_select.Down := True;
  end;
end;

procedure TMainForm.PaletteButtonClick(Sender: TObject);
const
  cClasses: array[ 0 .. 18] of string =
  (
    ''                   ,
    'TKnobsModule'       ,
    'TKnobsLed'          ,
    'TKnobsSelector'     ,
    'TKnobsKnob'         ,
    'TKnobsSmallKnob'    ,
    'TKnobsNoKnob'       ,
    'TKnobsDisplay'      ,
    'TKnobsFileSelector' ,
    'TKnobsIndicator'    ,
    'TKnobsIndicatorBar' ,
    'TKnobsIndicatorText',
    'TKnobsDataViewer'   ,
    'TKnobsInput'        ,
    'TKnobsOutput'       ,
    'TKnobsTextLabel'    ,
    'TKnobsEditLabel'    ,
    'TKnobsBox'          ,
    'TImage'
  );
begin
  FStickyMode  := GetKeyState( VK_SHIFT) < 0;
  FDesignClass := cClasses[TControl( Sender).Tag];
end;

procedure TMainForm.MenuItemNewClick(Sender: TObject);
begin
  JvDesignPanel.Clear;
end;

procedure TMainForm.OpenButtonClick(Sender: TObject);
begin
  if OpenDialog.Execute
  then JvDesignPanel.LoadFromFile( OpenDialog.Filename);
end;

procedure TMainForm.SaveButtonClick(Sender: TObject);
begin
  if SaveDialog.Execute
  then JvDesignPanel.SaveToFile( SaveDialog.Filename);
end;

procedure TMainForm.JvDesignPanelSelectionChange(Sender: TObject);
begin
  if Length( JvDesignPanel.Surface.Selected) > 0
  then begin
    // JvInspector.AddComponent( JvDesignPanel.Surface.Selected[ 0]);
    JvInspector.InspectObject := JvDesignPanel.Surface.Selected[ 0];
  end
  else JvInspector.InspectObject := nil;
end;


procedure TMainForm.KnobsKnobEtaValueChanged(const aSender: TObject; const aPath, aControlType: string; aValue: Double;
  IsFinal, IsAutomation: Boolean);
begin
  if IsFinal
  then Eta := aValue;
end;

procedure TMainForm.KnobsKnobRandomDotsValueChanged(const aSender: TObject; const aPath, aControlType: string;
  aValue: Double; IsFinal, IsAutomation: Boolean);
begin
  if IsFinal
  then RandomDots := aValue;
end;

initialization


  RegisterClass( TKnobsModule       );
  RegisterClass( TKnobsLed          );
  RegisterClass( TKnobsSelector     );
  RegisterClass( TKnobsTextControl  );
  RegisterClass( TKnobsKnob         );
  RegisterClass( TKnobsSmallKnob    );
  RegisterClass( TKnobsNoKnob       );
  RegisterClass( TKnobsDisplay      );
  RegisterClass( TKnobsFileSelector );
  RegisterClass( TKnobsIndicator    );
  RegisterClass( TKnobsIndicatorBar );
  RegisterClass( TKnobsIndicatorText);
  RegisterClass( TKnobsDataViewer   );
  RegisterClass( TKnobsInput        );
  RegisterClass( TKnobsOutput       );
  RegisterClass( TKnobsTextLabel    );
  RegisterClass( TKnobsEditLabel    );
  RegisterClass( TKnobsBox          );
  RegisterClass( TImage             );

  TJvInspectorColorItem      .RegisterAsDefaultItem;
  TJvInspectorControlTypeItem.RegisterAsDefaultItem;

end.
