unit FrmSwanTester;

interface

uses

  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.ExtCtrls, Vcl.StdCtrls, Vcl.Buttons, System.IniFiles,

  Globals, KnobsUtils, Knobs2013, KnobsMaze;

{

   COPYRIGHT 2017 .. 2019 Blue Hell / Jan Punter

  Some parts are copyright :

     Author  : Neugls.
     Website : Http://www.neugls.info
     Email   : NeuglsWorkStudio@gmail.com

   these are marked with (*Neugls*)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

type

  TFormSwanTester = class(TForm)
    Label1: TLabel;
    BitBtnReloadForth: TBitBtn;
    ComboBoxImports: TComboBox;
    BitBtnMoveSwan: TBitBtn;
    BitBtnMoveHunters: TBitBtn;
    CheckBoxMazeAutoHunt: TCheckBox;
    CheckBoxMazeAutoSwan: TCheckBox;
    PanelMazeGraphViewer: TPanel;
    MemoMazeMsgs: TMemo;
    BitBtnClearMsgs: TBitBtn;
    TimerHunter: TTimer;
    TimerSwan: TTimer;
    BitBtnShowWords: TBitBtn;
    BitBtnExecute: TBitBtn;
    EditForthCode: TEdit;
    BitBtnHelp: TBitBtn;
    Label2: TLabel;
    procedure ComboBoxImportsSelect(Sender: TObject);
    procedure BitBtnMoveSwanClick(Sender: TObject);
    procedure BitBtnMoveHuntersClick(Sender: TObject);
    procedure BitBtnReloadForthClick(Sender: TObject);
    procedure CheckBoxMazeAutoSwanClick(Sender: TObject);
    procedure CheckBoxMazeAutoHuntClick(Sender: TObject);
    procedure BitBtnClearMsgsClick(Sender: TObject);
    procedure TimerHunterTimer(Sender: TObject);
    procedure TimerSwanTimer(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure BitBtnShowWordsClick(Sender: TObject);
    procedure BitBtnExecuteClick(Sender: TObject);
    procedure BitBtnHelpClick(Sender: TObject);
    procedure EditForthCodeKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure EditForthCodeChange(Sender: TObject);
  private
    FMazeGraphViewer : TKnobsMazeGraphViewer;
    FMazeAutoHunt    : Boolean;
    FMazeAutoSwan    : Boolean;
    FForthText       : string;
  private
    procedure   SetMazeAutoHunt ( aValue: Boolean);
    procedure   SetMazeAutoSwan ( aValue: Boolean);
    procedure   SetForthText    ( const aValue: string);
  private
    procedure   LoadIni;
    procedure   SaveIni;
    procedure   ClearMazeMemo;
    procedure   DoMazeError( const aSender: TObject; aType: TMazeErrorType; const aMsg: string);
    procedure   InitMazeStuff;
    procedure   DoneMazeStuff;
    procedure   ImportForthExports;
    procedure   PerformForthImport;
    procedure   ReloadMazeForth;
    procedure   ExecuteForthText( const S: string);
    procedure   MoveSwan;
    procedure   MoveHunters;
    procedure   ShowWords;
    procedure   ShowHelp;
  private
    property    MazeAutoHunt  : Boolean read FMazeAutoHunt  write SetMazeAutoHunt;
    property    MazeAutoSwan  : Boolean read FMazeAutoSwan  write SetMazeAutoSwan;
    property    ForthText     : string  read FForthText     write SetForthText;
  end;

var

  FormSwanTester: TFormSwanTester;

implementation


// User area

//  private

    procedure   TFormSwanTester.SetMazeAutoHunt ( aValue: Boolean);
    begin
      if aValue <> FMazeAutoHunt
      then begin
        FMazeAutoHunt                := aValue;
        CheckBoxMazeAutoHunt.Checked := aValue;
        TimerHunter         .Enabled := MazeAutoHunt;
      end;
    end;


    procedure   TFormSwanTester.SetMazeAutoSwan ( aValue: Boolean);
    begin
      if aValue <> FMazeAutoSwan
      then begin
        FMazeAutoSwan                := aValue;
        CheckBoxMazeAutoSwan.Checked := aValue;
        TimerSwan           .Enabled := MazeAutoSwan;
      end;
    end;


    procedure   TFormSwanTester.SetForthText( const aValue: string);
    begin
      if aValue <> FForthText
      then begin
        FForthText         := aValue;
        EditForthCode.Text := aValue;
      end;
    end;


//  private

const

  sdfs = 'Defaults';
  sdws = 'window';
  hist = 'CmdHistory';

    procedure   TFormSwanTester.LoadIni;
    var
      anIniFile    : TMemIniFile;
      L, T, W, H   : Integer;
      WinState     : TWindowState;
    begin
      anIniFile := TMemIniFile.Create( IniFileName);
      with anIniFile
      do begin
        try
          L                        :=                ReadInteger( sdws, 'Left'          ,          Left           );
          T                        :=                ReadInteger( sdws, 'Top'           ,          Top            );
          W                        :=                ReadInteger( sdws, 'Width'         ,          Width          );
          H                        :=                ReadInteger( sdws, 'Height'        ,          Height         );
          WinState                 := TWindowState(  ReadInteger( sdws, 'WindowState'   , Integer( WindowState )) );

          if ( L < 0) or ( L > Screen.Width  - 20) then L := 0;
          if ( T < 0) or ( T > Screen.Height - 20) then T := 0;
          if W > Screen.Width                      then W := Screen.Width  - 20;
          if H > Screen.Height                     then H := Screen.Height - 20;

          SetBounds( L, T, W, H);
          if WinState = wsMaximized
          then WindowState := wsMaximized;

          MazeAutoHunt := ReadBool  ( sdfs, 'MazeAutoHunt', MazeAutoHunt);
          MazeAutoSwan := ReadBool  ( sdfs, 'MazeAutoSwan', MazeAutoSwan);
          ForthText    := ReadString( sdfs, 'ForthText'   , ForthText   );
        finally
          Free;
        end;
      end;
    end;


    procedure   TFormSwanTester.SaveIni;
    var
      anIniFile : TMemIniFile;
    begin
      anIniFile := TMemIniFile.Create( IniFileName);
      with anIniFile
      do begin
        try
          EraseSection( sdfs);
          if WindowState = wsNormal
          then begin
            WriteInteger( sdws, 'Left'       , Left       );
            WriteInteger( sdws, 'Top'        , Top        );
            WriteInteger( sdws, 'Width'      , Width      );
            WriteInteger( sdws, 'Height'     , Height     );
          end;
          WriteInteger( sdws, 'WindowState'  , Integer( WindowState));

          WriteBool  ( sdfs, 'MazeAutoHunt', MazeAutoHunt);
          WriteBool  ( sdfs, 'MazeAutoSwan', MazeAutoSwan);
          WriteString( sdfs, 'ForthText'   , ForthText   );
        finally
          UpdateFile;
          Free;
        end;
      end;
    end;


    procedure   TFormSwanTester.ClearMazeMemo;
    begin
      MemoMazeMsgs.Clear;
    end;


    procedure   TFormSwanTester.DoMazeError( const aSender: TObject; aType: TMazeErrorType; const aMsg: string);
    var
      S : string;
    begin
      S := Format( '%s: %s', [ MazeErrorTypeToStr( aType), aMsg], AppLocale);
      MemoMazeMsgs.Lines.Add( S);
    end;


    procedure   TFormSwanTester.InitMazeStuff;
    begin
      ClearMazeMemo;
      FMazeAutoHunt    := True;
      MazeAutoHunt     := False;
      FMazeAutoSwan    := True;
      MazeAutoSwan     := False;
      FMazeGraphViewer := TKnobsMazeGraphViewer.Create( nil);
      FMazeGraphViewer.Parent  := PanelMazeGraphViewer;
      FMazeGraphViewer.OnError := DoMazeError;
      FMazeGraphViewer.ForthFileName := 'mazes.4th';
      ImportForthExports;
    end;


    procedure   TFormSwanTester.DoneMazeStuff;
    begin
      FreeAndNil( FMazeGraphViewer);
    end;


    procedure   TFormSwanTester.ImportForthExports;
    var
      aList : TStringList;
    begin
      if Assigned( FMazeGraphViewer)
      then begin
        aList := FMazeGraphViewer.ImportList;

        if   Assigned( aList)
        and  ( aList.Count > 0)
        then begin
          ComboBoxImports.Items.Assign( aList);
          ComboBoxImports.ItemIndex := 0;
        end;
      end;
    end;


    procedure   TFormSwanTester.PerformForthImport;
    begin
      if ComboBoxImports.ItemIndex >= 0
      then ExecuteForthText( ComboBoxImports.Items[ ComboBoxImports.ItemIndex]);
    end;


    procedure   TFormSwanTester.ReloadMazeForth;
    begin
      FMazeGraphViewer.ReloadMazeForth;
      ImportForthExports;
    end;


    procedure   TFormSwanTester.ExecuteForthText( const S: string);
    begin
      FMazeGraphViewer.ExecuteForthText( S);
    end;


    procedure   TFormSwanTester.MoveSwan;
    const
      NoiseColor = 0.08;               // 0.0 <= NoiseColor <= 1.0
      Lambda     = 1.1;                // Lamda > 1.0
      Dist       = 0.7;                // Dist  > 0.0
    var
      Swan : TMazePoint;
    begin
      Swan.X := FMazeGraphViewer.SwanX;
      Swan.Y := FMazeGraphViewer.SwanY;
      Swan.X := ( 1 - NoiseColor) * Swan.X + NoiseColor * ( Swan.X + RandomExpo( Lambda, False) * Dist);
      Swan.Y := ( 1 - NoiseColor) * Swan.Y + NoiseColor * ( Swan.Y + RandomExpo( Lambda, False) * Dist);

      if Swan.X < -1 then Swan.X := - 2 - Swan.X else if Swan.X > 1 then Swan.X := 2 - Swan.X;
      if Swan.Y < -1 then Swan.Y := - 2 - Swan.Y else if Swan.Y > 1 then Swan.Y := 2 - Swan.Y;

      FMazeGraphViewer.SwanX := Swan.X;
      FMazeGraphViewer.SwanY := Swan.Y;
    end;


    procedure   TFormSwanTester.MoveHunters;
    begin
      FMazeGraphViewer.MoveHunters( True);
    end;


    procedure   TFormSwanTester.ShowWords;
    begin
      ExecuteForthText( 'words');
    end;


    procedure   TFormSwanTester.ShowHelp;
    begin
      MemoMazeMsgs.Lines.Add( 'Use the "Show words" button to get a list of all defined words');
      MemoMazeMsgs.Lines.Add( 'Use "See aWord" in the forth text Edit control, then hit enter or click "Execute"');
      MemoMazeMsgs.Lines.Add( 'Arbitrary forth text can be executed by entering it in the edit box and hitting return or clicking "Execute"');
      MemoMazeMsgs.Lines.Add( 'Non standard for this forth are: stack data can be almost anything (strings, floats, ints), create does> works "different"');
      MemoMazeMsgs.Lines.Add( 'The do .. loop family of words uses a separate loop stack, so >r r> r@ etc. can be used within loops');
    end;


// Delphi area

{$R *.dfm}

procedure TFormSwanTester.EditForthCodeChange(Sender: TObject);
begin
  ForthText := EditForthCode.Text;
end;

procedure TFormSwanTester.EditForthCodeKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if ( Shift = []) and ( Key = VK_RETURN)
  then ExecuteForthText( EditForthCode.Text);
end;

procedure TFormSwanTester.ComboBoxImportsSelect(Sender: TObject);
begin
  PerformForthImport;
end;

procedure TFormSwanTester.BitBtnClearMsgsClick(Sender: TObject);
begin
  ClearMazeMemo;
end;

procedure TFormSwanTester.BitBtnExecuteClick(Sender: TObject);
begin
  ExecuteForthText( EditForthCode.Text);
end;

procedure TFormSwanTester.BitBtnHelpClick(Sender: TObject);
begin
  ShowHelp;
end;

procedure TFormSwanTester.BitBtnMoveHuntersClick(Sender: TObject);
begin
  MoveHunters;
end;

procedure TFormSwanTester.BitBtnMoveSwanClick(Sender: TObject);
begin
  MoveSwan;
end;

procedure TFormSwanTester.BitBtnReloadForthClick(Sender: TObject);
begin
  ReloadMazeForth;
end;

procedure TFormSwanTester.BitBtnShowWordsClick(Sender: TObject);
begin
  ShowWords;
end;

procedure TFormSwanTester.CheckBoxMazeAutoHuntClick(Sender: TObject);
begin
  MazeAutoHunt := CheckBoxMazeAutoHunt.Checked;
end;

procedure TFormSwanTester.CheckBoxMazeAutoSwanClick(Sender: TObject);
begin
  MazeAutoSwan := CheckBoxMazeAutoSwan.Checked;
end;

procedure TFormSwanTester.TimerHunterTimer(Sender: TObject);
begin
  MoveHunters;
end;

procedure TFormSwanTester.TimerSwanTimer(Sender: TObject);
begin
  MoveSwan;
end;

procedure TFormSwanTester.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  SaveIni;
end;

procedure TFormSwanTester.FormCreate(Sender: TObject);
begin
  InitMazeStuff;
end;

procedure TFormSwanTester.FormDestroy(Sender: TObject);
begin
  DoneMazeStuff;
end;

procedure TFormSwanTester.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if   ( Char( Key) = 'X')
  and  ( Shift = [ ssAlt])
  then Close;
end;

procedure TFormSwanTester.FormShow(Sender: TObject);
begin
  ForthText := 'see see';
  LoadIni;
end;

end.

