unit HrastUnit;

// version: 17
// Copyright  2014 by HrastProgrammer. All rights reserved.
// This unit contains WREN synthesizer modules developed by HrastProgrammer.
// Can be distributed only as part of the complete WREN source package.
//
// version  8: Blue Hell : changed module registration procedure.
// version  9: Blue Hell : layout changes.
// version 10: Blue Hell : some code changes.
// version 11: Blue Hell : added constructors for the Fast/Slow selection.
// version 12: Blue Hell : removal of outcommented code and object renaming.
// version 13: Blue Hell : added un-attenuated FM to HrastSVF.
// version 14: Blue Hell : changed the lights gatherer.
// version 15: Blue Hell : added some normalization to the filter.
// version 16: Blue Hell : renamed some stuff to make things consistent with other modules - theis will break some patches
// version 17: Blue Hell : added unattenuated freq input to osc, added morph and shape modulation to osc.
// version 18: Blue Hell : modified for the new wy to couple pin names to memory locations

interface

uses

  KnobsUtils, KnobsConversions, Module_Defs, Module_Implementations;

type

  TModHrastOsc = class( TMod)
  strict private
  const
    i_frequency   =  0;
    i_morph       =  1;
    i_shape       =  2;
    i_bandlimit   =  3;
    i_pw          =  4;
    i_pwmlevel    =  5;
    i_pwm         =  6;
    i_fmlevel     =  7;
    i_fm          =  8;
    i_pmlevel     =  9;
    i_phase       = 10;
    i_vintage     = 11;
    i_sync        = 12;
    i_level       = 13;
    i_cents       = 14;
    i_shapemod    = 15;
    i_shapemodamt = 16;
    i_morphmod    = 17;
    i_morphmodamt = 18;
    i_freq        = 19;
  const
    o_out         =  0;
  private
    FZ        : TSignal;
    FPhase    : TSignal;
    FPrevSync : Boolean;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
    procedure   GatherLights( const aPrefix: string; const aCallback: TLightsHandler);                         override;
  end;


  TModHrastSVF = class( TMod)
  strict private
  const
    i_in         =  0;
    i_frequency  =  1;
    i_q          =  2;
    i_fm         =  3;
    i_fmlevel    =  4;
    i_qm         =  5;
    i_qmlevel    =  6;
    i_inlevel    =  7;
    i_distortion =  8;
    i_lowpass    =  9;
    i_bandpass   = 10;
    i_highpass   = 11;
    i_level12    = 12;
    i_level24    = 13;
    i_freq       = 14;
  const
    o_lp         =  0;
    o_bp         =  1;
    o_hp         =  2;
    o_br         =  3;
    o_12         =  4;
    o_24         =  5;
  private
    FBand12 : TSignal;
    FLow12  : TSignal;
    FBand24 : TSignal;
    FLow24  : TSignal;
  public
    procedure   CreateIO;                                                                                      override;
    procedure   SetDefaults;                                                                                   override;
    procedure   DoTick;                                                                                        override;
  end;



  procedure CreateTables; // BLue Hell : renamed initialize to CreateTables and made it public; table re-init is needed
                          //             after changing the system rate



implementation



uses

  Globals, Math, SysUtils;

const

  TableSize = 8192;

var

  Pow2Table : array [ 0 .. TableSize + 1] of TSignal;
  SineTable : array [ 0 .. TableSize + 1] of TSignal;


  function Pow2Freq( Freq: TSignal): TSignal;
  var
    n: Integer;
    f: TSignal;
  begin
    f := Abs( Freq) * TableSize;

    if f > TableSize - 1
    then f := TableSize - 1;

    n      := Trunc( f);
    f      := f - n;
    Result := Pow2Table[ n];
    Result := Result + f * ( Pow2Table[ n + 1] - Result);
  end;


  function Sine( Freq: TSignal): TSignal;
  var
    n: Integer;
    f: TSignal;
  begin
    f := Abs( Freq) * TableSize;

    if f > TableSize - 1
    then f := TableSize - 1;

    n      := Trunc( f);
    f      := f - n;
    Result := SineTable[ n];
    Result := Result + f * ( SineTable[ n + 1] - Result);
  end;


{ ========
  TModHrastOsc = class( TMod)
  private
    FZ        : TSignal;
    FPhase    : TSignal;
    FPrevSync : Boolean;
  public
}

    procedure   TModHrastOsc.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := False;
      AddInput ( i_frequency   , 'frequency'  );
      AddInput ( i_morph       , 'morph'      );
      AddInput ( i_shape       , 'shape'      );
      AddInput ( i_bandlimit   , 'bandlimit'  );
      AddInput ( i_pw          , 'pw'         );
      AddInput ( i_pwmlevel    , 'pwmlevel'   );
      AddInput ( i_pwm         , 'pwm'        );
      AddInput ( i_fmlevel     , 'fmlevel'    );
      AddInput ( i_fm          , 'fm'         );
      AddInput ( i_pmlevel     , 'pmlevel'    );
      AddInput ( i_phase       , 'phase'      );
      AddInput ( i_vintage     , 'vintage'    );
      AddInput ( i_sync        , 'sync'       );
      AddInput ( i_level       , 'level'      );
      AddInput ( i_cents       , 'cents'      );
      AddInput ( i_shapemod    , 'shapemod'   );
      AddInput ( i_shapemodamt , 'shapemodamt');
      AddInput ( i_morphmod    , 'morphmod'   );
      AddInput ( i_morphmodamt , 'morphmodamt');
      AddInput ( i_freq        , 'freq'       );
      AddOutput( o_out         , 'out'        );
    end;


    procedure   TModHrastOsc.SetDefaults;
    begin
      FInputs[ i_shapemod] := 1.0;
      FDezipperMap := [
        i_frequency  ,
        i_morph      ,
        i_shape      ,
        i_bandlimit  ,
        i_pw         ,
        i_pwmlevel   ,
        i_fmlevel    ,
        i_pmlevel    ,
        i_level      ,
        i_cents      ,
        i_shapemodamt,
        i_morphmodamt
      ];
    end;


    procedure TModHrastOsc.DoTick;
    var
      DeltaPhase: TSignal;
    var
      n   : Integer;
      t   : TSignal;
      y   : TSignal;
      ypw : TSignal;
    begin
      if not FPrevSync and SignalToLogic( FInputs[i_sync])
      then FPhase := 0;

      FPrevSync  := SignalToLogic( FInputs[ i_sync]);
      DeltaPhase := Pow2Freq( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_cents] + FInputs[ i_fmlevel] * FInputs[ i_fm]) * System_Rate_Rec;

      t := FPhase + FInputs[ i_pmlevel] * FInputs[ i_phase];
      n := Trunc( t);
      t := t - n;

      if t < 0
      then t := t + 1;

      FPhase := FPhase + DeltaPhase;
      n      := Trunc( FPhase);
      FPhase := FPhase - n;

      if FPhase < 0
      then FPhase := FPhase + 1;

      y := 2 * t - 1 - PolyBlep( t, FInputs[ i_bandlimit], DeltaPhase);
      t := t + FInputs[ i_pw] + FInputs[ i_pwmlevel] * FInputs[ i_pwm];
      n := Trunc( t);
      t := t - n;

      if t < 0
      then t := t + 1;

      ypw := 2 * t - 1 - PolyBlep( t, FInputs[ i_bandlimit], DeltaPhase);
      t   := Clip( FInputs[ i_shape] + FInputs[ i_shapemod] * FInputs[ i_shapemodamt], 0, 1);

      if FInputs[ i_vintage] <> 0
      then begin                            // Vintage
        if t > 0.999
        then t := 0.999;

        y   := ( y + 1) * 0.5;
        y   := y * y;
        y   := 2 * y - 1;
        ypw := ( ypw + 1) * 0.5;
        ypw := ypw * ypw;
        ypw := 2 * ypw - 1;
      end;

      FZ := y - ypw * Clip( FInputs[ i_morph] + FInputs[ i_morphmod] * FInputs[ i_morphmodamt], 0, 1) + t * FZ;

      if t > 0.99
      then t := 0.99;

      y := FZ * ( t - 1);
      FOutputs[ o_out] := y * FInputs[ i_level];
    end;


    procedure TModHrastOsc.GatherLights( const aPrefix: string; const aCallback: TLightsHandler);
    begin
      aCallback( Self, MakeInfo( aPrefix, Name, 'syncinput', FInputs[ i_sync]));
    end;


{ ========
  TModHrastSVF = class( TMod)
  private
    FBand12 : TSignal;
    FLow12  : TSignal;
    FBand24 : TSignal;
    FLow24  : TSignal;
  public
}

    procedure   TModHrastSVF.CreateIO; // override;
    begin
      FIsFast := True;
      FIsSlow := False;
      AddInput ( i_in        , 'in'        );
      AddInput ( i_frequency , 'frequency' );
      AddInput ( i_q         , 'q'         );
      AddInput ( i_fm        , 'fm'        );
      AddInput ( i_fmlevel   , 'fmlevel'   );
      AddInput ( i_qm        , 'qm'        );
      AddInput ( i_qmlevel   , 'qmlevel'   );
      AddInput ( i_inlevel   , 'inlevel'   );
      AddInput ( i_distortion, 'distortion');
      AddInput ( i_lowpass   , 'lowpass'   );
      AddInput ( i_bandpass  , 'bandpass'  );
      AddInput ( i_highpass  , 'highpass'  );
      AddInput ( i_level12   , 'level12'   );
      AddInput ( i_level24   , 'level24'   );
      AddInput ( i_freq      , 'freq'      );
      AddOutput( o_lp        , 'lp'        );
      AddOutput( o_bp        , 'bp'        );
      AddOutput( o_hp        , 'hp'        );
      AddOutput( o_br        , 'br'        );
      AddOutput( o_12        , '12'        );
      AddOutput( o_24        , '24'        );
    end;


    procedure   TModHrastSVF.SetDefaults;
    begin
      FInputs[ i_in  ] := 0;
      FInputs[ i_fm  ] := 0;
      FInputs[ i_qm  ] := 0;
      FInputs[ i_freq] := 0;
      FDezipperMap:=[
        i_frequency,
        i_q,
        i_fmlevel,
        i_qmlevel,
        i_inlevel,
        i_distortion,
        i_lowpass,
        i_bandpass,
        i_highpass,
        i_level12,
        i_level24
      ]
    end;


    procedure TModHrastSVF.DoTick;
    var
      Oversample : Integer;
      f          : TSignal;
      fb         : TSignal;
      y          : TSignal;
      High12     : TSignal;
      High24     : TSignal;
      Output12   : TSignal;
      Output24   : TSignal;
    begin
      f  := Pow2Freq( FInputs[ i_frequency] + FInputs[ i_freq] + FInputs[ i_fmlevel] * FInputs[ i_fm]) / System_Rate_2;
      f  := 2 * Sine( f);
      fb := Clip( 1 - ( FInputs[ i_q] + FInputs[ i_qmlevel] * FInputs[ i_qm]), 0, 1);
      y  := FInputs[ i_in] * FInputs[ i_inlevel];
      y  := y / ( Abs( y) + ( 1 - Clip( FInputs[ i_distortion], 0, 0.999)));

      for Oversample := 1 to 2
      do begin
        High12   := Normalize( y - fb * FBand12 - FLow12);
        FBand12  := Normalize( f * High12  + FBand12);
        FLow12   := Normalize( f * FBand12 + FLow12);
        FLow12   := Normalize( FLow12 * 1.0001 / ( 0.0001 * Abs( FLow12) + 1));
        Output12 := Normalize( FLow12 * FInputs[ i_lowpass] + FBand12 * FInputs[ i_bandpass] + High12 * FInputs[ i_highpass]);

        High24   := Normalize( Output12 - fb * FBand24 - FLow24);
        FBand24  := Normalize( f * High24  + FBand24);
        FLow24   := Normalize( f * FBand24 + FLow24);
        FLow24   := Normalize( FLow24 * 1.0001 / ( 0.01 * Abs( FLow24) + 1));
      end;

      Output24 := FLow24 * FInputs[ i_lowpass] + FBand24 * FInputs[ i_bandpass] + High24 * FInputs[ i_highpass];
      FOutputs[ o_lp] := FLow12;
      FOutputs[ o_bp] := FBand12;
      FOutputs[ o_hp] := High12;
      FOutputs[ o_br] := FLow12 + High12;
      FOutputs[ o_12] := Output12 * FInputs[ i_level12];
      FOutputs[ o_24] := Output24 * FInputs[ i_level24];
    end;




  procedure CreateTables;
  var
    i: Integer;
    x: TSignal;
  begin
    for i := 0 to TableSize
    do Pow2Table[ i] := UnitsToFrequency( i / TableSize);

    Pow2Table[ TableSize + 1] := Pow2Table[ TableSize];
    x := Pi / TableSize;

    for i := 0 to TableSize - 1
    do SineTable[ i] := Sin( x * i);

    SineTable[ TableSize div 2] := 0;
    SineTable[ TableSize      ] := SineTable[ 0];
    SineTable[ TableSize + 1  ] := SineTable[ 1];
  end;


initialization

  CreateTables;

end.
