unit IntSequences;

{
  After an idea by Robert Inventor : http://robertinventor.com/software/tunesmithy/music.htm as explained in
  http://www.science20.com/robert_inventor/music_and_mathematics_fractallike_sloth_canon_number_sequences-113689

   COPYRIGHT 2016 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}

interface

uses

  System.SysUtils, System.Classes, System.Math,

  KnobsUtils, KnobsConversions;



type

  TTuneSmithySeed = array of Integer;

  TTuneSmithy = class
  private
    FBase    : Integer;
    FSeed    : TTuneSmithySeed;
    FWeights : TSignalArray;
  private
    procedure   InitializeWeights;
    procedure   AddToSeed( aNumber: Integer);
  public
    constructor Create( const aSeed: string);
    function    Element( anIndex: Integer): Integer;
  end;


  // Cellular automatons

  TBitArray = class
  strict private
  const
    MASKS  : array[ 0 .. 7] of Byte = ( $01, $02, $04, $08, $10, $20, $40, $80);
    IMASKS : array[ 0 .. 7] of Byte = ( $FE, $FD, $FB, $F7, $EF, $DF, $BF, $7F);
  private
    FData     : array of Byte;
    FBitCount : Integer;
  private
    function    GetBit( anIndex: Integer): Integer;
    procedure   SetBit( anIndex: Integer; aValue: Integer);
    procedure   SetBitCount( aValue: Integer);
  public
    procedure   Clear;
    procedure   CopyFrom( const aValue: TBitArray);
    procedure   Difs( const aSrc: TBitArray; var aResult: TBitArray);
  public
    property    Bit[ anIndex: Integer] : Integer read GetBit    write SetBit;                                   default;
    property    BitCount               : Integer read FBitCount write SetBitCount;
  end;

  PBitArray = ^ TBitArray;


  TElementaryCA = class
  // Elementary Cellular Automaton, catalogued by FRule
  private
    FBits1         : TBitArray;
    FBits2         : TBitArray;
    FBitsO         : PBitArray;
    FBitsN         : PBitArray;
    FRule          : Byte;
    FHasRandomSeed : Integer;
    FFaultRate     : TSignal;
    FSumCount      : Integer;
  private
    function    GetBit( anIndex: Integer) : Integer;
    function    GetSum( anIndex: Integer) : Integer;
    function    GetBitCount               : Integer;
    procedure   SwapON;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   Execute;
    procedure   SetRandomSeed( aBitCount: Integer);
  public
    property    Bit[ anIndex: Integer]: Integer read GetBit;
    property    Sum[ anIndex: Integer]: Integer read GetSum;
    property    BitCount              : Integer read GetBitCount;
    property    Rule                  : Byte    read FRule       write FRule;
    property    FaultRate             : TSignal read FFaultRate  write FFaultRate;
    property    SumCount              : Integer read FSumCount   write FSumCount;
  end;


  TLimitedCA = class
  // A limited width elementary CA, with a finite run-length
  private
    FWidth     : Integer;
    FBits1     : TBitArray;
    FBits2     : TBitArray;
    FBitsO     : PBitArray;
    FBitsN     : PBitArray;
    FRule      : Byte;
    FFaultRate : TSignal;
    FMaxLen    : Integer;
    FContext   : Integer;
    FCount     : Integer;
  private
    function    GetInternalBit( anIndex: Integer): Integer;
    procedure   SetInternalBit( anIndex, aValue: Integer);
    function    GetBit        ( anIndex: Integer): Integer;
    function    GetBitCount   : Integer;
    procedure   SetMaxLen     ( aValue: Integer);
    procedure   SwapON;
  public
    constructor Create( aWidth, aMaxLen: Integer);
    destructor  Destroy;                                                                                       override;
    procedure   Clear;
    procedure   InitBits;
    procedure   Execute;
  private
    property    InternalBit[ anIndex: Integer] : Integer read GetInternalBit write SetInternalBit;
    property    BitCount                       : Integer read GetBitCount;
  public
    property    Bit[ anIndex: Integer]         : Integer read GetBit;
    property    Width                          : Integer read FWidth;
    property    MaxLen                         : Integer read FMaxLen;
    property    Rule                           : Byte    read FRule          write FRule;
    property    FaultRate                      : TSignal read FFaultRate     write FFaultRate;
    property    Count                          : Integer read FCount;
  end;


  TLifeMode = (
    lmLife       ,
    lmHighLife   ,
    lmDayAndNight,
    lm2x2        ,
    lm34Life     ,
    lmAmoeba     ,
    lmGnarl      ,
    lmSeeds      ,
    lmServiettes ,
    lmStains     ,
    lmPseudo     ,
    lmMove       ,
    lmStarWars   ,
    lmInverse    ,
    lmFreeze
  );

  TLifeSumMode = (
    smBin        ,
    smLin        ,
    smMax        ,
    smMin        ,
    smAverage    ,
    smNote       ,
    smMaxNote    ,
    smMinNote    ,
    smAverageNote
  );

  TLifeWrapMode = (
    lwOff,
    lwXY ,
    lwY  ,
    lwX
  );

  TLifeOnCellChange = procedure( aSender: TObject; anX, anY: Integer; IsAlive: Boolean) of object;

  TGameOfLife = class
  private
    FOnCellChange : TLifeOnCellChange;
    FData0        : TKnobsDataPlane;
    FData1        : TKnobsDataPlane;
    FActive       : Integer;
    FXSize        : Integer;
    FYSize        : Integer;
    FLock         : Integer;
    FFaultRate    : TSignal;
    FFillChance   : TSignal;
    FMode         : TLifeMode;
    FWrapped      : TLifeWrapMode;
    FColumn       : Integer;
    FSumMode      : TLifeSumMode;
    FColFunction  : TSignal;
    FFillCount    : Integer;
  private
    function    GetPixel( X, Y: Integer): Boolean;
    procedure   SetPixel( X, Y: Integer; aValue: Boolean);
    function    GetData: TKnobsDataPlane;
    procedure   SetData ( const aData: TKnobsDataPlane);
    function    GetLocked: Boolean;
    procedure   SetLocked( aValue: Boolean);
    procedure   SetColumn( aValue: Integer);
    function    GetAsString: string;
    procedure   SetAsString( const aValue: string);
    procedure   CalcColFunction;
    function    CountAround( X, Y: Integer): Integer;
  public
    constructor Create( anXSize, anYSize: Integer; anOnCellChange: TLifeOnCellChange);
    destructor  Destroy;                                                                                       override;
    function    CalcRule( IsAlive, Faulty: Boolean; aCount: Integer): Boolean;
    function    ExecuteCol( anX: Integer): Boolean;
    procedure   Execute;
    procedure   Clear;
    procedure   FillRandom;
  public
    property    Pixel[ X, Y: Integer] : Boolean         read GetPixel    write SetPixel;
    property    Data                  : TKnobsDataPlane read GetData     write SetData;
    property    Locked                : Boolean         read GetLocked   write SetLocked;
    property    FaultRate             : TSignal         read FFaultRate  write FFaultRate;
    property    FillChance            : TSignal         read FFillChance write FFillChance;
    property    Mode                  : TLifeMode       read FMode       write FMode;
    property    Wrapped               : TLifeWrapMode   read FWrapped    write FWrapped;
    property    Column                : Integer         read FColumn     write SetColumn;
    property    SumMode               : TLifeSumMode    read FSumMode    write FSumMode;
    property    AsString              : string          read GetAsString write SetAsString;
    property    XSize                 : Integer         read FXSize;
    property    YSize                 : Integer         read FYSize;
    property    ColFunction           : TSignal         read FColFunction;
    property    FillCount             : Integer         read FFillCount;
  end;


implementation


const

  CARDINAL_BITS = SizeOf( Cardinal) * 8;

{ ========
  TTuneSmithy = class
  private
    FBase    : Integer;
    FSeed    : TTuneSmithySeed;
    FWeights : TSignalArray;
  private
}

    procedure   TTuneSmithy.InitializeWeights;
    var
      i : Integer;
    begin
      SetLength( FWeights, Length( FSeed));
      FBase        := Length( FSeed);
      FWeights[ 0] := 1;

      for i := Low( FSeed) + 1 to High( FSeed)
      do FWeights[ i] := FSeed[ i] / i;
    end;


    procedure   TTuneSmithy.AddToSeed( aNumber: Integer);
    begin
      SetLength( FSeed, Length( FSeed) + 1);
      FSeed[ Length( FSeed) - 1] := aNumber;
    end;


//  public

    constructor TTuneSmithy.Create( const aSeed: string);
    var
      i      : Integer;
      aParts : TStringList;
    begin
      AddToSeed( 0);

      aParts := Explode( aSeed, ',');

      try
        for i := 0 to aParts.Count - 1
        do AddToSeed( StrToIntDef( Trim( aParts[ i]), 1));
      finally
        aParts.DisposeOf;
      end;

      InitializeWeights;
    end;


    function    TTuneSmithy.Element( anIndex: Integer): Integer;
    var
      aDigit : Integer;
    begin
      Result := 0;

      while anIndex > 0
      do begin
        aDigit  := anIndex mod FBase;
        Result  := Result + Round( aDigit * FWeights[ aDigit]);
        anIndex := anIndex div FBase;
      end;
    end;


{ ========
  TBitArray = class
  strict private
  const
    MASKS  : array[ 0 .. 7] of Byte = ( $01, $02, $04, $08, $10, $20, $40, $80);
    IMASKS : array[ 0 .. 7] of Byte = ( $FE, $FD, $FB, $F7, $EF, $DF, $BF, $7F);
  private
    FData     : array of Byte;
    FBitCount : Integer;
  public
    property    Bit[ anIndex: Integer] : Integer read GetBit    write SetBit;
    property    BitCount               : Integer read FBitCount;
  private
}

    function    TBitArray.GetBit( anIndex: Integer): Integer;
    var
      i : Integer;
      b : Integer;
    begin
      if ( anIndex >= 0) and ( anIndex < FBitCount)
      then begin
        i := anIndex div 8;
        b := anIndex mod 8;

        if FData[ i] and MASKS[ b] <> 0
        then Result := 1
        else Result := 0;
      end
      else Result := 0;
    end;


    procedure   TBitArray.SetBit( anIndex: Integer; aValue: Integer);
    var
      i : Integer;
      b : Integer;
    begin

      if anIndex >= 0
      then begin
        i := anIndex div 8;
        b := anIndex mod 8;

        if i >= Length( FData)
        then SetLength( FData, i + 1);

        if anIndex >= FBitCount
        then FBitCount := anIndex + 1;

        if aValue <> 0
        then FData[ i] := FData[ i] or   MASKS[ b]
        else FData[ i] := FData[ i] and IMASKS[ b];
      end;
    end;


    procedure   TBitArray.SetBitCount( aValue: Integer);
    begin
      if ( aValue <> FBitCount) and ( aValue >= 0)
      then begin
        SetLength( FData, ( aValue + 7) div 8);
        FBitCount := aValue;
      end;
    end;


//  public

    procedure   TBitArray.Clear;
    begin
      FBitCount := 0;
      SetLength( FData, 0);;
    end;


    procedure   TBitArray.CopyFrom( const aValue: TBitArray);
    begin
      if Assigned( aValue)
      then begin
        BitCount := aValue.BitCount;
        Move( aValue.FData[ 0], FData[ 0], Length( aValue.FData));
      end;
    end;


    procedure   TBitArray.Difs( const aSrc: TBitArray; var aResult: TBitArray);
    var
      i : Integer;
    begin
      if
        Assigned( aSrc)    and
        Assigned( aResult) and
        ( aSrc.BitCount = BitCount)
      then begin
        SetLength( aResult.FData, Length( FData));
        aResult.FBitCount := BitCount;

        for i := 0 to Length( FData) - 1
        do aResult.FData[ i] := aSrc.FData[ i] xor FData[ i];


      end;
    end;


{ ========
  TElementaryCA = class
  private
    FBits1         : TBitArray;
    FBits2         : TBitArray;
    FBitsO         : PBitArray;
    FBitsN         : PBitArray;
    FRule          : Byte;
    FHasRandomSeed : Integer;
    FFaultRate     : TSignal;
    FSumCount      : Integer;
  public
    property    Bit[ anIndex: Integer]: Integer read GetBit;
    property    Sum[ anIndex: Integer]: Integer read GetSum;
    property    BitCount              : Integer read GetBitCount;
    property    Rule                  : Byte    read FRule       write FRule;
    property    FaultRate             : TSignal read FFaultRate  write FFaultRate;
    property    SumCount              : Integer read FSumCount   write FSumCount;
  private
}

    function    TElementaryCA.GetBit( anIndex: Integer): Integer;
    begin
      if Assigned( FBitsO)
      then Result := FBitsO^[ anIndex]
      else Result := 0;
    end;


    function    TElementaryCA.GetSum( anIndex: Integer) : Integer;
    var
      i : Integer;
    begin
      Result := 0;

      for i := 0 to FSumCount - 1
      do Result := ( Result shl 1) + Bit[ anIndex - i];
    end;


    function    TElementaryCA.GetBitCount: Integer;
    begin
      if Assigned( FBitsO)
      then Result := FBitsO^.BitCount
      else Result := 0;
    end;


    procedure   TElementaryCA.SwapON;
    var
      aTmp: PBitArray;
    begin
      aTmp   := FBitsO;
      FBitsO := FBitsN;
      FBitsN := aTmp;
    end;


//  public

    constructor TElementaryCA.Create;
    var
      i : Integer;
    begin
      inherited Create;
      FRule     := 0;
      FBits1    := TBitArray.Create;
      FBits2    := TBitArray.Create;
      FBitsO    := @ FBits1;
      FBitsN    := @ FBits2;
      FSumCount := 3;

      if FHasRandomSeed > 0
      then begin
        for i := 0 to FHasRandomSeed - 1
        do FBits1[ i] := Random( 2);
      end
      else FBits1[ 0] := 1;
    end;


    destructor  TElementaryCA.Destroy; // override;
    begin
      FBitsO := nil;
      FBitsN := nil;
      FBits1.DisposeOf;
      FBits2.DisposeOf;
      inherited;
    end;


    procedure   TElementaryCA.Clear;
    var
      i : Integer;
    begin
      FBits1.Clear;
      FBits2.Clear;
      FBitsO := @ FBits1;
      FBitsN := @ FBits2;

      if FHasRandomSeed > 0
      then begin
        for i := 0 to FHasRandomSeed - 1
        do FBits1[ i] := Random( 2);
      end
      else FBits1[ 0] := 1;
    end;


    procedure   TElementaryCA.Execute;
    var
      P : Byte;
      N : Byte;
      i : Integer;
      F : Integer;
    begin
      if Assigned( FBitsN)
      then begin
        for i := 0 to BitCount + 1
        do begin
          N := Bit[ i] + ( Bit[ i - 1] shl 1) + ( Bit[ i - 2] shl 2);
          P := 1 shl N;

          if Random < FFaultRate
          then F := 1
          else F := 0;

          if ( FRule and P) <> 0
          then FBitsN^[ i] := 1 xor F
          else FBitsN^[ i] := 0 xor F;
        end;

        SwapON;
      end;
    end;


    procedure   TElementaryCA.SetRandomSeed( aBitCount: Integer);
    begin
      FHasRandomSeed := aBitCount;
      Clear;
    end;


{ ========
  TLimitedCA = class
  private
    FWidth     : Integer;
    FBits1     : TBitArray;
    FBits2     : TBitArray;
    FBitsO     : PBitArray;
    FBitsN     : PBitArray;
    FRule      : Byte;
    FFaultRate : TSignal;
    FMaxLen    : Integer;
    FContext   : Integer;
    FCount     : Integer;
  private
    property    InternalBit[ anIndex: Integer] : Integer read GetInternalBit write SetInternalBit;
    property    BitCount                       : Integer read GetBitCount;
  public
    property    Bit[ anIndex: Integer]         : Integer read GetBit;
    property    Width                          : Integer read FWidth;
    property    MaxLen                         : Integer read FMaxLen;
    property    Rule                           : Byte    read FRule          write FRule;
    property    FaultRate                      : TSignal read FFaultRate     write FFaultRate;
  private
}

    function    TLimitedCA.GetInternalBit( anIndex: Integer): Integer;
    begin
      if   Assigned( FBitsO)
      then Result := FBitsO^[ BitCount div 2 + anIndex]
      else Result := 0;
    end;


    procedure   TLimitedCA.SetInternalBit( anIndex, aValue: Integer);
    begin
      if   Assigned( FBitsN)
      and  ( anIndex >= - BitCount div 2)
      and  ( anIndex <=   BitCount div 2)
      then FBitsN^[ BitCount div 2 + anIndex] := aValue;
    end;


    function    TLimitedCA.GetBit( anIndex: Integer): Integer;
    begin
      if   Assigned( FBitsO)
      then Result := FBitsO^[ anIndex + FContext]
      else Result := 0;
    end;


    function    TLimitedCA.GetBitCount: Integer;
    begin
      Result := Width + 2 * FContext;
    end;


    procedure   TLimitedCA.SetMaxLen( aValue: Integer);
    begin
      if aValue <> FMaxLen
      then begin
        FMaxLen  := aValue;
        FContext := FMaxLen - ( WIDTH + 1);

        if FContext < 0
        then FContext := 0;
      end;
    end;


    procedure   TLimitedCA.SwapON;
    var
      aTmp: PBitArray;
    begin
      aTmp   := FBitsO;
      FBitsO := FBitsN;
      FBitsN := aTmp;
    end;


//  public

    constructor TLimitedCA.Create( aWidth, aMaxLen: Integer);
    begin
      inherited Create;
      FWidth := aWidth;
      SetMaxLen( aMaxLen);
      FRule  := 0;
      FBits1 := TBitArray.Create;
      FBits2 := TBitArray.Create;
      FBitsO := @ FBits1;
      FBitsN := @ FBits2;
      InitBits;
    end;


    destructor  TLimitedCA.Destroy; // override;
    begin
      FBitsO := nil;
      FBitsN := nil;
      FBits1.DisposeOf;
      FBits2.DisposeOf;
      inherited;
    end;


    procedure   TLimitedCA.Clear;
    begin
      FBitsO := @ FBits1;
      FBitsN := @ FBits2;
      InitBits;
    end;


    procedure   TLimitedCA.InitBits;
    var
      i : Integer;
    begin
      // Fix the width for the bit arrays, imposing the limitation

      if   Assigned( FBitsN)
      and  Assigned( FBitsO)
      then begin
        for i := 0 to BitCount - 1
        do begin
          FBits1.Bit[ i] := 0;
          FBits2.Bit[ i] := 0;
        end;

        InternalBit[ 0] := 1; // Initialize the middle bit to be a 1
        SwapON;
        FCount := 0;
      end;
    end;


    procedure   TLimitedCA.Execute;
    var
      P : Byte;
      N : Byte;
      i : Integer;
      F : Integer;
    begin
      if   Assigned( FBitsN)
      and  Assigned( FBitsO)
      then begin
        for i := - BitCount div 2 to ( BitCount - 1) div 2
        do begin
          N := InternalBit[ i - 1] shl 2 + ( InternalBit[ i] shl 1) + ( InternalBit[ i + 1]);
          P := 1 shl N;

          if Random < FFaultRate
          then F := 1
          else F := 0;

          if ( FRule and P) <> 0
          then InternalBit[ i] := 1 xor F
          else InternalBit[ i] := 0 xor F;
        end;

        SwapON;
        Inc( FCount);

        if FCount >= MaxLen
        then Clear;
      end;
    end;


{ ========
  TGameOfLife = class
  private
    FOnCellChange : TLifeOnCellChange;
    FData0        : TKnobsDataPlane;
    FData1        : TKnobsDataPlane;
    FActive       : Integer;
    FXSize        : Integer;
    FYSize        : Integer;
    FLock         : Integer;
    FFaultRate    : TSignal;
    FFillChance   : TSignal;
    FMode         : TLifeMode;
    FWrapped      : TLifeWrapMode;
    FColumn       : Integer;
    FSumMode      : TLifeSumMode;
    FColFunction  : TSignal;
    FFillCount    : Integer;
  public
    property    Pixel[ X, Y: Integer] : Boolean         read GetPixel    write SetPixel;
    property    Data                  : TKnobsDataPlane read GetData     write SetData;
    property    Locked                : Boolean         read GetLocked   write SetLocked;
    property    FaultRate             : TSignal         read FFaultRate  write FFaultRate;
    property    FillChance            : TSignal         read FFillChance write FFillChance;
    property    Mode                  : TLifeMode       read FMode       write FMode;
    property    Wrapped               : TLifeWrapMode   read FWrapped    write FWrapped;
    property    Column                : Integer         read FColumn     write SetColumn;
    property    SumMode               : TLifeSumMode    read FSumMode    write FSumMode;
    property    AsString              : string          read GetAsString write SetAsString;
    property    XSize                 : Integer         read FXSize;
    property    YSize                 : Integer         read FYSize;
    property    ColFunction           : TSignal         read FColFunction;
    property    FillCount             : Integer         read FFillCount;
  private
}

    function    TGameOfLife.GetPixel( X, Y: Integer): Boolean;
    begin
      if ( X >= 0) and ( X < FXSize) and ( Y >= 0) and ( Y < FYSize)
      then Result := Data[ Y, X]
      else Result := False;
    end;


    procedure   TGameOfLife.SetPixel( X, Y: Integer; aValue: Boolean);
    var
      OldValue : Boolean;
    begin
      if ( X >= 0) and ( X < FXSize) and ( Y >= 0) and ( Y < FYSize)
      then begin
        OldValue := Data[ Y, X];

        if OldValue <> aValue
        then begin
          Data[ Y, X] := aValue;

          if Assigned( FOnCellChange)
          then FOnCellChange( Self, X, Y, aValue);
        end;
      end;
    end;


    function    TGameOfLife.GetData: TKnobsDataPlane;
    begin
      if FActive = 0
      then Result := FData0
      else Result := FData1;
    end;


    procedure   TGameOfLife.SetData( const aData: TKnobsDataPlane);
    var
      i       : Integer;
      j       : Integer;
      Changed : Boolean;
    begin
      if Assigned( aData)
      then begin
        Locked := True;

        try
          if FActive = 0
          then SetLength( FData0, Length( aData))
          else SetLength( FData1, Length( aData));

          for i := 0 to Length( aData) - 1
          do begin
            if FActive = 0
            then SetLength( FData0[ i], Length( aData[ i]))
            else SetLength( FData1[ i], Length( aData[ i]));

            for j := 0 to Length( aData[ i]) - 1
            do begin
              Changed := Data[ i, j] <> aData[ i, j];
              Data[ i, j] := aData[ i, j];

              if Changed and Assigned( FOnCellChange)
              then FOnCellChange( self, j, i, aData[ i, j]);
            end;
          end;
        finally
          Locked := False;
        end;
      end;
    end;


    function    TGameOfLife.GetLocked: Boolean;
    begin
      Result := FLock > 0;
    end;


    procedure   TGameOfLife.SetLocked( aValue: Boolean);
    begin
      if aValue
      then Inc( FLock)
      else DecToZero( FLock);
    end;


    procedure   TGameOfLife.SetColumn( aValue: Integer);
    begin
      aValue := MathIntMod( aValue, FXSize);

      if aValue <> FColumn
      then begin
        FColumn := aValue;
        CalcColFunction;
      end;
    end;


    function    TGameOfLife.GetAsString: string;
    begin
      if FActive = 0
      then Result := DataPlaneToStr( FData0)
      else Result := DataPlaneToStr( FData1);
    end;


    procedure   TGameOfLife.SetAsString( const aValue: string);
    var
      i      : Integer;
      j      : Integer;
      FDataX : ^ TKnobsDataPlane;
    begin
      if FActive = 0
      then FDataX := @ FData0
      else FDataX := @ FData1;

      StrToDataPlane( aValue, FDataX^);

      if Assigned( FOnCellChange)
      then begin
        for i := 0 to YSize - 1
        do begin
          for j := 0 to XSize - 1
          do FOnCellChange( Self, j, i, FDataX^[ i, j]);
        end;
      end;
    end;


    procedure   TGameOfLife.CalcColFunction;
    var
      i          : Integer;
      FuncResult : TSignal;
      Count      : Integer;
    begin
      FuncResult := 0;

      case SumMode of

        smBin :

          begin
            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := FuncResult + ( UInt64( 1) shl i);
            end;

            FuncResult := FuncResult / ( UInt64( 1) shl YSize);
          end;

        smLin :

          begin
            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, i]
              then FuncResult := FuncResult + 1;
            end;

            FuncResult := FuncResult / YSize;
          end;

        smMax :

          begin
            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := Max( FuncResult, i);
            end;

            FuncResult := FuncResult / YSize;
          end;

        smMin :

          begin
            FuncResult := YSize + 1;

            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := Min( FuncResult, i);
            end;

            if FuncResult = YSize + 1
            then FuncResult := 0;

            FuncResult := FuncResult / YSize;
          end;

        smAverage :

          begin
            Count := 0;

            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then begin
                FuncResult := FuncResult + i;
                Inc( Count);
              end;
            end;

            if Count = 0
            then FuncResult := 0.5
            else FuncResult := FuncResult / ( YSize * Count);
          end;

        smNote :

          begin
            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := FuncResult + 1
            end;

            FuncResult := NoteNumberToUnits( FuncResult);
          end;

        smMaxNote :

          begin
            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := Max( FuncResult, i);
            end;

            FuncResult := NoteNumberToUnits( FuncResult);
          end;

        smMinNote :

          begin
            FuncResult := YSize + 1;

            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then FuncResult := Min( FuncResult, i);
            end;

            if FuncResult = YSize + 1
            then FuncResult := 0;

            FuncResult := NoteNumberToUnits( FuncResult);
          end;

        smAverageNote :

          begin
            Count := 0;

            for i := 0 to YSize - 1
            do begin
              if Pixel[ FColumn, YSize - i + 1]
              then begin
                FuncResult := FuncResult + i;
                Inc( Count);
              end;
            end;

            if Count = 0
            then FuncResult := NoteNumberToUnits( YSize div 2)
            else FuncResult := NoteNumberToUnits( Round( FuncResult / Count));
          end;

      end;

      FColFunction := FuncResult;
    end;


    function    TGameOfLife.CountAround( X, Y: Integer): Integer;
    var
      anX : Integer;
      anY : Integer;
      pX  : Integer;
      pY  : Integer;
    begin
      Result := 0;

      case Wrapped of

        lwOff :                             // No warpping

          begin
            for px := X - 1 to X + 1
            do begin
              for py := Y - 1 to Y + 1
              do begin
                if   ( not (( pX = X) and ( pY = Y))) and
                     ( pX >= 0)                       and
                     ( pY >= 0)                       and
                     ( pX < FXSize)                   and
                     ( pY < FYSize)                   and
                     Pixel[ pX, pY]
                then Inc( Result);
              end;
            end;
          end;

        lwXY :                              // Torus mode

          begin
            pY := MathIntMod( Y - 1, FYSize);

            for anX := X - 1 to X + 1
            do begin
              pX := MathIntMod( anX, FXSize);

              if Pixel[ pX, pY]
              then Inc( Result);
            end;

            px := MathIntMod( X - 1, FXSize);

            if Pixel[ pX, Y]
            then Inc( Result);

            px := MathIntMod( X + 1, FXSize);

            if Pixel[ pX, Y]
            then Inc( Result);

            pY := MathIntMod( Y + 1, FYSize);

            for anX := X - 1 to X + 1
            do begin
              pX := MathIntMod( anX, FXSize);

              if Pixel[ pX, pY]
              then Inc( Result);
            end;
          end;

        lwX :                               // Vertical cylinder

          begin
            for anY := Y - 1 to Y + 1
            do begin
              pY := MathIntMod( anY, FYSize);

              for px := X - 1 to X + 1
              do begin
                if   ( not (( pX = X) and ( pY = Y))) and
                     ( pX >= 0    )                   and
                     ( pX < FXSize)                   and
                     Pixel[ pX, pY]
                then Inc( Result);
              end;
            end;
          end;

        lwY :                               // Horizontal cylinder

          begin
            for anX := X - 1 to X + 1
            do begin
              pX := MathIntMod( anX, FXSize);

              for py := Y - 1 to Y + 1
              do begin
                if   ( not (( pX = X) and ( pY = Y))) and
                     ( pY >= 0    )                   and
                     ( pY < FYSize)                   and
                     Pixel[ pX, pY]
                then Inc( Result);
              end;
            end;
          end;

      end;
    end;


//  public

    constructor TGameOfLife.Create( anXSize, anYSize: Integer; anOnCellChange: TLifeOnCellChange);
    var
      i : Integer;
    begin
      inherited Create;
      Locked := True;
      FXSize := anXSize;
      FYSize := anYSize;

      SetLength( FData0, FYSize);
      SetLength( FData1, FYSize);

      for i := 0 to FYSize - 1
      do begin
        SetLength( FData0[ i], FXSize);
        SetLength( FData1[ i], FXSize);
      end;

      FOnCellChange := anOnCellChange;
      FColFunction  := 0;
      FFillCount    := 0;
      Locked        := False;
    end;


    destructor  TGameOfLife.Destroy; // override;
    begin
      Locked := True;
      FXSize := 0;
      FYSize := 0;
      inherited;
    end;


    function    TGameOfLife.CalcRule( IsAlive, Faulty: Boolean; aCount: Integer): Boolean;
    type
      N = set of 0 .. 8;

      function R( L, D: N): Boolean;
      begin
        if IsAlive
        then R := ( aCount in L) xor Faulty
        else R := ( aCount in D) xor Faulty
      end;

    begin
      case Mode of
        lmLife        : Result := R([       2, 3               ], [          3               ]);
        lmHighLife    : Result := R([       2, 3               ], [          3,       6      ]);
        lmDayAndNight : Result := R([          3, 4,    6, 7, 8], [          3,       6, 7, 8]);
        lm2x2         : Result := R([    1, 2,       5         ], [          3,       6      ]);
        lm34Life      : Result := R([          3, 4            ], [          3, 4            ]);
        lmAmoeba      : Result := R([    1,    3,    5,       8], [          3,    5,    7   ]);
        lmGnarl       : Result := R([    1                     ], [    1                     ]);
        lmSeeds       : Result := R([                          ], [       2                  ]);
        lmServiettes  : Result := R([                          ], [       2, 3, 4            ]);
        lmStains      : Result := R([       2, 3,    5, 6, 7, 8], [          3,       6, 7, 8]);
        lmPseudo      : Result := R([       2, 3,             8], [          3,    5,    7   ]);
        lmMove        : Result := R([       2,    4, 5         ], [          3,       6,    8]);
        lmStarWars    : Result := R([          3, 4, 5         ], [          3               ]);
        lmInverse     : Result := R([ 0,       3, 4,    6, 7, 8], [ 0, 1, 2, 3, 4,       7, 8]);
        else {Freeze}   Result := IsAlive;
      end;
    end;


    function    TGameOfLife.ExecuteCol( anX: Integer): Boolean;
    var
      i       : Integer;
      aCount  : Integer;
      L       : Boolean;
      Changed : Boolean;
    begin
      Result := False;

      if anX = 0
      then FFillCount := 0;

      if not Locked and ( FYSize > 0)
      then begin
        if FActive = 0
        then begin
          for i := 0 to FYSize - 1
          do begin
            aCount          := CountAround( anX, i);
            L               := CalcRule( Pixel[ anX, i], FaultRate > Random, aCount);
            Changed         := L <> FData1[ i, anX];
            FData1[ i, anX] := L;

            if L
            then Inc( FFillCount);

            if Assigned( FOnCellChange) and Changed
            then FOnCellChange( Self, anX, i, L);
          end;
        end
        else begin
          for i := 0 to FYSize - 1
          do begin
            aCount          := CountAround( anX, i);
            L               := CalcRule( Pixel[ anX, i], FaultRate > Random, aCount);
            Changed         := L <> FData0[ i, anX];
            FData0[ i, anX] := L;

            if L
            then Inc( FFillCount);

            if Assigned( FOnCellChange) and Changed
            then FOnCellChange( Self, anX, i, L);
          end;
        end;

        if anX = FXSize - 1
        then begin
          FActive := 1 - FActive;
          Result  := True;
        end;

        if anX = FColumn
        then CalcColFunction;
      end;
    end;


    procedure   TGameOfLife.Execute;
    var
      i : Integer;
    begin
      if not Locked and ( FXSize > 0)
      then begin
        for i := 0 to FXSize - 1
        do ExecuteCol( i);
      end;
    end;


    procedure   TGameOfLife.Clear;
    var
      i : Integer;
      j : Integer;
    begin
      for i := 0 to FXSize - 1
      do begin
        for j := 0 to FYSize - 1
        do Pixel[ i, j] := False;
      end;

      FColFunction := 0;
      FFillCount   := 0;
    end;


    procedure   TGameOfLife.FillRandom;
    var
      i : Integer;
      j : Integer;
    begin
      FFillCount := 0;

      for i := 0 to FXSize - 1
      do begin
        for j := 0 to FYSize - 1
        do begin
          if FillChance > Random
          then begin
            Pixel[ i, j] := True;
            Inc( FFillCount);
          end
          else Pixel[ i, j] := False;
        end;
      end;

      CalcColFunction;
    end;


end.

