unit KnobParser;

{

   COPYRIGHT 1999 .. 2019 Blue Hell / Jan Punter

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 as
  published by the Free Software Foundation;

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}


interface

uses

  WinApi.Windows,

  System.Types, System.Classes, System.SysUtils, System.Math,

  Vcl.Buttons, Vcl.Clipbrd, Vcl.Graphics,

  Globals, KnobsUtils, Knobs2013, KnobsConversions;


const

  KnobsPatchVersion = 8;                    // Patch file format version number

type

  TOnPatchReadError  = procedure( aSender: Tobject; aMsg: string; var Continue: Boolean) of object;
  TOnPatchWriteError = procedure( aSender: Tobject; aMsg: string; var Continue: Boolean) of object;


  TTokenType = (
    ttNothing,           // Token type not determined yet, start value
    ttError,             // Token did not start with a valid character
                         // [.()=;'] | (alpha) | (numeric) (any intro
                         // (white)* is always skipped).
    ttEof,               // Input exhausted without determining a token type.
    ttLParen,            // Left parenthesis  seen as the first non (white)
    ttRParen,            // Right parenthesis seen as the first non (white)
    ttEqual,             // Equial sign       seen as the first non (white)
    ttIdentifier,        // Identifier        seen : identifier = (alpha)(alpha-numeric)* .
    ttInteger,           // Integer           seen : integer = (numeric)*
    ttString             // String            seen : string = (quote)((any)*(quote)(quote)(any)*)*(quote) .
  );

  TReaderState = record
    Pos       : Integer;
    LineNr    : Integer;
    Offset    : Integer;
    Token     : string;
    TokenType : TTokenType;
  end;


  TPatchReader = class( TInterfacedObject, IKnobsPatchReader)
  var
    FFileName             : string;
    FReadMode             : TKnobsReadMode;
    FPos                  : Integer;
    FLineNr               : Integer;
    FOffset               : Integer;
    FData                 : string;
    FWirePanel            : TKnobsWirePanel;
    FModule               : TKnobsCustomModule;
    FToken                : string;
    FTokenType            : TTokenType;
    FOnReadError          : TOnPatchReadError;
    FPatchVersion         : Integer;
    FVariation            : Integer;
    FGuid                 : string;
    FWarnOnStructMismatch : Boolean;
    FReadTuning           : Boolean;
    FTuningChanged        : Boolean;
  strict private
    procedure   SetWirePanel( const aWirePanel: TKnobsWirePanel);
  protected
    procedure   SaveState   ( var   aState: TReaderState);
    procedure   RestoreState( const aState: TReaderState);
    procedure   Clear;                                                                                          virtual;
    procedure   ReadError( const aMsg: string);                                                                 virtual;
    procedure   ReadErrorFmt( const aFmt: string; const anArgs: array of const);
    procedure   Warning( const aMsg: string);                                                                   virtual;
    procedure   WarningFmt( const aFmt: string; const anArgs: array of const);
    procedure   ReadToken;
    procedure   ExpectedSingle( aChar: Char);
    procedure   ExpectedIdentifier( const anIdent: string);
    procedure   ExpectedString( const aString: string);
    procedure   ExpectedInteger;
    procedure   ExpectedStringOrNumber;
    procedure   ExpectedIdentifierOrSingle( const anIdent: string; aSingle: Char);
    function    ReadLParen: Boolean;
    function    ReadRParen: Boolean;
    function    ReadEqual : Boolean;
    function    ReadIdentifier( const anIdent: string): Boolean;
    procedure   RequireLParen;
    procedure   RequireRParen;
    procedure   ScanRParen;
    procedure   RequireEqual;
    procedure   RequireIdentifier  ( const anIdent: string);
    function    ReadKVString       ( const anIdent: string; var aValue: string ): Boolean;
    function    ReadKVStringOption ( const anIdent: string; var aValue: string ): Boolean;
    function    ReadKVInteger      ( const anIdent: string; var aValue: Integer): Boolean;
    function    ReadKVIntegerOption( const anIdent: string; var aValue: Integer): Boolean;
    function    ReadKVFloat        ( const anIdent: string; var aValue: Double ): Boolean;
    function    ReadClass          : string;
    function    ReadName           : string;
    function    ReadTitle          : string;
    function    ReadVersion        : Integer;
    function    ReadSrc            : string;
    function    ReadDst            : string;
    function    ReadType           : Integer;
    function    ReadLeft           : Integer;
    function    ReadTop            : Integer;
    function    ReadValue          ( var aValue : Integer): Boolean;
    function    ReadLockedOption   ( var aValue : Boolean): Boolean;
    function    ReadCCOption       ( var aValue : Byte   ): Boolean;
    function    ReadRndOption      ( var aValue : Byte   ): Boolean;
    procedure   ReadLowOption      ( var aValue : string);
    procedure   ReadHighOption     ( var aValue : string);
    function    ReadVarOption      ( var aValue : string ): Boolean;
    function    ReadPublicOption   ( var aValue : Boolean): Boolean;
    function    ReadStringValue    ( var aValue : string ): Boolean;
    function    ReadColorOption    ( var aValue : TColor ): Boolean;
    function    ReadVariationOption( var aValue : Integer): Boolean;
    function    ReadGuidOption     ( var aValue : string ): Boolean;
    procedure   ReadWire;                                                                                       virtual;
    procedure   ReadWires;                                                                                      virtual;
    procedure   ReadTuningOption;                                                                               virtual;
    procedure   ReadDetail;                                                                                     virtual;
    procedure   ReadDetails;                                                                                    virtual;
    procedure   ReadModule;                                                                                     virtual;
    procedure   ReadModules;                                                                                    virtual;
    procedure   Read;                                                                                           virtual;
    function    TryRead: Boolean;                                                                               virtual;
  public
    constructor Create;                                                                                         virtual;
    destructor  Destroy;                                                                                       override;
    function    GetWarnOnStructMismatch: Boolean;
    procedure   SetWarnOnStructMismatch( aValue: Boolean);
    function    GetTuningChanged: Boolean;


    procedure   ReadParams(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );                                                                                                          virtual;

    procedure   ReadString(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      aDistance        : TPoint;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode;
      const aFileName  : string
    );                                                                                                          virtual;

    procedure   ReadStrings(
      const aData      : TStrings;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode;
      const aFileName  : string
    );                                                                                                          virtual;

    procedure   ReadFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );                                                                                                          virtual;

    procedure   ImportFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );                                                                                                          virtual;

    procedure   ReadFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    );                                                                                                          virtual;

    procedure   ParamsFromFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );                                                                                                          virtual;

    procedure   ParamsFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    );                                                                                                          virtual;
  public
    property    FileName             : string  read FFileName;
    property    LineNr               : Integer read FLineNr;
    property    Offset               : Integer read FOffset;
    property    WarnOnStructMismatch : Boolean read GetWarnOnStructMismatch write SetWarnOnStructMismatch;
    property    TuningChanged        : Boolean read GetTuningChanged;
  public
    property    OnReadError : TOnPatchReadError read FOnReadError write FOnReadError;
  end;


  TPatchWriter = class( TInterfacedObject, IKnobsPatchWriter)
  strict private
  var
    FFileName      : string;
    FWriteMode     : TKnobsWriteMode;
    FWirePanel     : TKnobsWirePanel;
    FModule        : TKnobsCustomModule;
    FComponent     : TComponent;
    FWire          : TKnobsWire;
    FOnWriteError  : TOnPatchWriteError;
    FResult        : string;
    FLineBreak     : string;
  protected
    procedure   Clear;                                                                                          virtual;
    procedure   Write           ( anInd: Integer; const S: string);
    procedure   WriteLn         ( anInd: Integer; const S: string);
    procedure   WriteWire       ( anInd: Integer);                                                              virtual;
    procedure   WriteWires      ( anInd: Integer);                                                              virtual;
    procedure   WriteTuning     ( anInd: Integer);                                                              virtual;
    function    MakeLockedStr   ( aLocked: Boolean): string;
    function    MakeLowHighStr  ( aLow, aHigh: string): string;
    function    MakePublicStr   ( aPublic: Boolean): string;
    function    MakeVariationStr( const aVariations: string): string;
    procedure   WriteDetailData (
      anInd                      : Integer;
      const aValue, aCC, aRnd    : Integer;
      aPublic, aLocked           : Boolean;
      aLow, aHigh                : string;
      const aSuffix, aVariations : string
    );                                                                                                overload; virtual;
    procedure   WriteDetailData(
      anInd             : Integer;
      const aValue      : string;
      aPublic           : Boolean;
      const aVariations : string
    );                                                                                                overload; virtual;
    procedure   WriteDetail     ( anInd: Integer);                                                              virtual;
    procedure   WriteDetails    ( anInd: Integer);                                                              virtual;
    procedure   WriteModule     ( anInd, anXOffset, anYOffset: Integer);                                        virtual;
    procedure   WriteModules    ( anInd: Integer);                                                              virtual;
    procedure   WritePatch;                                                                                     virtual;
    function    TryWrite: Boolean;                                                                              virtual;
  public
    constructor Create;                                                                                         virtual;
    destructor  Destroy;                                                                                       override;

    function    WriteString(
      const aWirePanel: TKnobsWirePanel;
      const aFileName : string     = '';
      aWriteMode      : TKnobsWriteMode = wmAll
    ): string;                                                                                                  virtual;

    function    WriteStrings(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string     = '';
      aWriteMode       : TKnobsWriteMode = wmAll
    ): TStrings;                                                                                                virtual;

    procedure   WriteFile(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string;
      aWriteMode       : TKnobsWriteMode = wmAll
    );                                                                                                          virtual;

    procedure   WriteToClipboard(
      const aWirePanel : TKnobswirePanel;
      aWriteMode       : TKnobsWriteMode = wmAll
    );                                                                                                          virtual;

  public
    property    FileName  : string read FFileName;
    property    LineBreak : string read FLineBreak write FLineBreak;
  public
    property    OnWriteError : TOnPatchWriteError read FOnWriteError write FOnWriteError;
  end;



implementation

uses

 FrmStore;


const

  STRUCT_MISMATCH_NO_PASTE = 'Structures differ, can not paste parameters';


  function QuoteEscape( const aValue: string): string;
  var
    i : Integer;
  begin
    Result := '';
    for i := 1 to Length( aValue)
    do begin
      if aValue[ i] = ''''
      then Result := Result + ''''''
      else Result := Result + aValue[ i];
    end;
  end;


  function UnQuoteEscape( const aValue: string): string;
  type
    TState = (
      stNormal,
      stInQuote
    );
  var
    State : TState;
    i     : Integer;
  begin
    State  := stNormal;
    Result := '';

    for i := 0 to Length( aValue)
    do begin
      case State of

        stNormal :
          begin
            if aValue[ i] = ''''
            then State := stInQuote
            else Result := Result + aValue[ i]
          end;

        stInQuote :
          begin
            if aValue[ i] = ''''
            then Result := Result + ''''
            else Result := Result + '''' + aValue[ i];

            State := stNormal;
          end;
      end;
    end;
  end;


{ ========
  TPatchReader = class( TInterfacedObject, IKnobsPatchReader)
  strict private
  type
    TTokenType = (
      ttNothing,           // Token type not determined yet, start value
      ttError,             // Token did not start with a valid character
                           // [.()=;'] | (alpha) | (numeric) (any intro
                           // (white)* is always skipped).
      ttEof,               // Input exhausted without determining a token type.
      ttLParen,            // Left parenthesis  seen as the first non (white)
      ttRParen,            // Right parenthesis seen as the first non (white)
      ttEqual,             // Equial sign       seen as the first non (white)
      ttIdentifier,        // Identifier        seen : identifier = (alpha)(alpha-numeric)* .
      ttInteger,           // Integer           seen : integer = (numeric)*
      ttString             // String            seen : string = (quote)((any)*(quote)(quote)(any)*)*(quote) .
    );
  var
    FFileName             : string;
    FReadMode             : TKnobsReadMode;
    FPos                  : Integer;
    FLineNr               : Integer;
    FOffset               : Integer;
    FData                 : string;
    FWirePanel            : TKnobsWirePanel;
    FModule               : TKnobsCustomModule;
    FToken                : string;
    FTokenType            : TTokenType;
    FOnReadError          : TOnPatchReadError;
    FPatchVersion         : Integer;
    FVariation            : Integer;
    FGuid                 : string;
    FWarnOnStructMismatch : Boolean;
    FTuningChanged        : Boolean;
    FReadTuning           : Boolean;
  public
    property    FileName             : string  read FFileName;
    property    LineNr               : Integer read FLineNr;
    property    Offset               : Integer read FOffset;
    property    WarnOnStructMismatch : Boolean read FWarnOnStructMismatch write FWarnOnStructMismatch;
  public
    property    OnReadError : TOnPatchReadError read FOnReadError write FOnReadError;
  strict private
}

    procedure   TPatchReader.SetWirePanel( const aWirePanel: TKnobsWirePanel);
    begin
      Clear;
      FWirePanel := aWirePanel;
    end;


//  protected

    procedure   TPatchReader.SaveState( var aState: TReaderState);
    begin
      aState.Pos       := FPos;
      aState.LineNr    := LineNr;
      aState.Offset    := Offset;
      aState.Token     := FToken;
      aState.TokenType := FTokenType;
    end;


    procedure   TPatchReader.RestoreState( const aState: TReaderState);
    begin
      FPos       := aState.Pos;
      FLineNr    := aState.LineNr;
      FOffset    := aState.Offset;
      FToken     := aState.Token;
      FTokenType := aState.TokenType;
    end;


    procedure   TPatchReader.Clear; // virtual;
    begin
      if Assigned( FWirePanel)
      then FreeAndNil( FWirePanel);
    end;


    procedure   TPatchReader.ReadError( const aMsg: string); // virtual;
    var
      S : string;
    begin
      if FileName = ''
      then
        S :=
          Format(
            'parse error: '^M^M'line = %d '^M'offset = %d '^M^M'%s',
            [
              LineNr,
              Offset,
              aMsg
            ]
          )
      else
        S :=
          Format(
            'parse error: '^M^M'file = ''%s'''^M'line = %d '^M'offset = %d '^M^M'%s',
            [
              FileName,
              LineNr,
              Offset,
              aMsg
            ]
          );

      raise EKnobsCorruptPatch.Create( S);
    end;


    procedure   TPatchReader.ReadErrorFmt( const aFmt: string; const anArgs: array of const);
    begin
      ReadError( Format( aFmt, anArgs));
    end;


    procedure   TPatchReader.Warning( const aMsg: string); // virtual;
    begin
      raise EKnobsWarning.CreateFmt( 'Warning: %s', [ aMsg]);
    end;


    procedure   TPatchReader.WarningFmt( const aFmt: string; const anArgs: array of const);
    begin
      Warning( Format( aFmt, anArgs));
    end;


    procedure   TPatchReader.ReadToken;
    const
      quote         = '''';                               // Single quotes used for strings, escaped as ''
      lparen        = '(';                                // Left  parenthesis
      rparen        = ')';                                // Right parenthesis
      equal         = '=';                                // Equals sign, assignment operator
      cr            = ^M;                                 // Carriage return, line feeds are ignored
      lf            = ^J;                                 // line feeds to be ignored
      white         = [ #$00 .. #$20, #$7f .. #$ff];      // White space
      alpha         = [ 'A' .. 'Z', 'a' .. 'z'];          // Alpha characters
      numeric       = [ '0' .. '9', '-'];                 // Numeric characters
      alpha_numeric = alpha + numeric;                    // Alpha-numeric characters
    type
      TState = (
        stInitial,                                        // Initial parser state, not recognized anything yet
        stDone,                                           // Parsing completed, current token and token type are set
        stString,                                         // In string parsing, after seeing a string quote, escapes allowed
        stIdentifier,                                     // In identifier parsing, after seeing an alpha character
        stInteger                                         // In integer scanning, after seeing a numeric character
      );
    var
      i     : Integer;
      State : TState;
    begin
      Assert( FPos > 0, 'Can''t start token parsing upfront input :shock:');
      State      := stInitial;
      FTokenType := ttNothing;
      FToken     := '';
      i          := FPos;

      while ( i <= Length( FData)) and ( State <> stDone)
      do begin
        if FData[ i] = cr
        then begin
          inc( FLineNr);
          FOffset := 0;
        end
        else if FData[ i] <> lf
        then begin

          case State of

            stInitial :

              begin
                case FData[ i] of
                  lparen : begin State := stDone; FTokenType := ttLParen; FToken := lparen; end;
                  rparen : begin State := stDone; FTokenType := ttRParen; FToken := rparen; end;
                  equal  : begin State := stDone; FTokenType := ttEqual;  FToken := equal ; end;
                  quote  : State := stString;
                  else begin
                    if CharInSet( FData[ i], numeric)
                    then begin State := stInteger;    FToken := FData[ i]; end
                    else if CharInSet( Fdata[ i], alpha)
                    then begin State := stIdentifier; FToken := FData[ i]; end
                    else if not CharInSet( FData[ i], white)
                    then begin State := stDone; FToken := ''; FTokenType := ttError end;
                  end;
                end;
              end;

            stDone :

              begin
                Assert( False, 'We should not be here ...');
              end;

            stString :

              begin
                if FData[ i] = quote
                then begin
                  if ( i < Length( FData)) and ( FData[ i + 1] = quote)
                  then begin
                    FToken := FToken + quote;
                    inc( i      );
                    inc( FOffset);
                  end
                  else begin
                    State      := stDone;
                    FTokenType := ttString;
                  end;
                end
                else FToken := FToken + FData[ i];
              end;

            stIdentifier :

              begin
                if CharInSet( FData[ i], alpha_numeric)
                then FToken := FToken + FData[ i]
                else begin
                  dec( i      );
                  dec( FOffset);
                  State      := stDone;
                  FTokenType := ttIdentifier;
                end;
              end;

            stInteger :

              begin
                if CharInSet( FData[ i], numeric)
                then FToken := FToken + FData[ i]
                else begin
                  dec( i      );
                  dec( FOffset);
                  State      := stDone;
                  FTokenType := ttInteger;
                end;
              end;
          end;

          Inc( FOffset);
        end;

        inc( i      );
      end;

      if i > Length( FData)
      then FTokenType := ttEof;

      FPos := i;
    end;


//  protected

    procedure   TPatchReader.ExpectedSingle( aChar: Char);
    begin
      ReadErrorFmt( '''%s'' expected', [ aChar]);
    end;


    procedure   TPatchReader.ExpectedIdentifier( const anIdent: string);
    begin
      if anIdent = ''
      then ReadError   ( '<identifier> expected')
      else ReadErrorFmt( '''%s'' expected', [ anIdent]);
    end;


    procedure   TPatchReader.ExpectedString( const aString: string);
    begin
      if aString = ''
      then ReadError   ( '<string> expected')
      else ReadErrorFmt( '''%s'' expected', [ aString]);
    end;


    procedure   TPatchReader.ExpectedStringOrNumber;
    begin
      ReadError( '<string> or <integer> expected');
    end;


    procedure   TPatchReader.ExpectedIdentifierOrSingle( const anIdent: string; aSingle: Char);
    begin
      if anIdent = ''
      then ReadErrorFmt( '<identifier> or ''%s'' expected', [ aSingle])
      else ReadErrorFmt( '''%s'' or ''%s'' expected', [ anIdent, aSingle]);
    end;


    procedure   TPatchReader.ExpectedInteger;
    begin
      ReadError( 'integer value expected');
    end;


    function    TPatchReader.ReadLParen : Boolean;
    begin
      ReadToken;
      Result := FTokenType = ttLParen;
    end;


    function    TPatchReader.ReadRParen : Boolean;
    begin
      ReadToken;
      Result := FTokenType = ttRParen;
    end;


    function    TPatchReader.ReadEqual : Boolean;
    begin
      ReadToken;
      Result := FTokenType = ttEqual;
    end;


    function    TPatchReader.ReadIdentifier( const anIdent: string): Boolean;
    begin
      ReadToken;
      Result := ( FTokenType = ttIdentifier) and SameText( FToken, anIdent);
    end;


    procedure   TPatchReader.RequireLParen;
    begin
      if not ReadLParen
      then ExpectedSingle( '(');
    end;


    procedure   TPatchReader.RequireRParen;
    begin
      if not ReadRParen
      then ExpectedSingle( ')');
    end;


    procedure   TPatchReader.ScanRParen;
    // done : this should silently skip properly balanced '(' ')' pairs.
    var
      Level : Integer;
      Done  : Boolean;
    begin
      Level := 0;
      Done  := False;

      while not Done
      do begin
        ReadToken;

        case FTokenType of
          ttLParen       : Inc( Level);
          ttRParen       : begin Dec( Level); if Level <= 0 then Done := True; end;
          ttEof, ttError : Done := True;
        end;
      end;

      if FTokenType <> ttRParen
      then ExpectedSingle( ')');
    end;


    procedure   TPatchReader.RequireEqual;
    begin
      if not ReadEqual
      then ExpectedSingle( '=');
    end;


    procedure   TPatchReader.RequireIdentifier( const anIdent: string);
    begin
      if not ReadIdentifier( anIdent)
      then ExpectedIdentifier( anIdent);
    end;


    function    TPatchReader.ReadKVString( const anIdent: string; var aValue: string): Boolean;
    // Scan for 'anIdent = <stringval>;'
    begin
      Result := False;
      RequireIdentifier( anIdent);
      RequireEqual;
      ReadToken;
      aValue := FToken;

      if FTokenType = ttString
      then Result := True;
    end;


    function    TPatchReader.ReadKVStringOption( const anIdent: string; var aValue: string): Boolean;
    // Scan for 'anIdent = <stringval>;'
    var
      aState : TReaderState;
    begin
      aValue := '';
      Result := False;
      SaveState( aState);

      if ReadIdentifier( anIdent)
      then begin
        RequireEqual;
        ReadToken;

        if FTokenType = ttString
        then begin
          aValue := FToken;
          Result := True;
        end
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadKVInteger( const anIdent: string; var aValue: Integer): Boolean;
    // Scan for 'anIdent = <intval>;'
    var
      aVal : string;
    begin
      Result := False;
      RequireIdentifier( anIdent);
      RequireEqual;
      ReadToken;
      aVal := FToken;

      if FTokenType = ttInteger
      then begin
        aValue := StrToIntDef( aVal, - MaxInt);
        Result := aValue <> - MaxInt;
      end
      else aValue := 0;
    end;


    function    TPatchReader.ReadKVIntegerOption( const anIdent: string; var aValue: Integer): Boolean;
    // Scan for 'anIdent = <intval>;'
    var
      aVal   : string;
      aState : TReaderState;
    begin
      Result := False;
      SaveState( aState);

      if ReadIdentifier( anIdent)
      then begin
        RequireEqual;
        ReadToken;
        aVal := FToken;

        if FTokenType = ttInteger
        then begin
          aValue := StrToIntDef( aVal, - MaxInt);
          Result := aValue <> - MaxInt;
        end
        else aValue := 0;
      end
      else begin
        RestoreState( aState);
        aValue := 0;
      end;
    end;


    function    TPatchReader.ReadKVFloat( const anIdent: string; var aValue: Double): Boolean;
    // Scan for 'anIdent = <floatval>;'
    var
      aVal : string;
    begin
      Result := False;
      RequireIdentifier( anIdent);
      RequireEqual;
      ReadToken;
      aVal := FToken;

      if FTokenType = ttString
      then begin
        aValue := StrToFloatDef( aVal, NaN);
        Result := not IsNan( aValue);
      end;

      if not Result
      then aValue := 0.0;
    end;


    function    TPatchReader.ReadClass : string;
    // Scan for 'class = <stringval>;'
    begin
      if not ReadKVString( 'class', Result)
      then ExpectedString( '');
    end;


    function    TPatchReader.ReadName : string;
    // Scan for 'name = <stringval>;'
    begin
      if not ReadKVString( 'name', Result)
      then ExpectedString( '');
    end;


    function    TPatchReader.ReadTitle : string;
    // Scan for 'title = <stringval>;'
    begin
      if not ReadKVString( 'title', Result)
      then ExpectedString( '');
    end;


    function    TPatchReader.ReadVersion: Integer;
    var
      aValue : string;
      aState : TReaderState;
    begin
      Result := 0;
      SaveState( aState);

      if ReadIdentifier( 'version')
      then begin
        RequireEqual;
        ReadToken;
        aValue := FToken;

        if FTokenType = ttInteger
        then Result := StrToInt( aValue)
        else begin
          RestoreState( aState);
          ExpectedInteger;
        end;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadSrc : string;
    begin
      if not ReadKVString( 'src', Result)
      then ExpectedString( '');
    end;


    function    TPatchReader.ReadDst : string;
    begin
      if not ReadKVString( 'dst', Result)
      then ExpectedString( '');
    end;


    function    TPatchReader.ReadType : Integer;
    // Scan for 'type = <intval>;'
    begin
      if not ReadKVInteger( 'type', Result)
      then ExpectedInteger;
    end;


    function    TPatchReader.ReadLeft : Integer;
    // Scan for 'left = <intval>;'
    begin
      if not ReadKVInteger( 'left', Result)
      then ExpectedInteger;
    end;


    function    TPatchReader.ReadTop : Integer;
    // Scan for 'top = <intval>;'
    begin
      if not ReadKVInteger( 'top', Result)
      then ExpectedInteger;
    end;


    function    TPatchReader.ReadValue( var aValue : Integer): Boolean;
    // Scan for 'value = <Integer>;'
    begin
      Result := ReadKVInteger( 'value', aValue);
    end;


    function    TPatchReader.ReadLockedOption( var aValue: Boolean): Boolean;
    // Scan for 'locked'
    var
      aState : TReaderState;
    begin
      Result := False;
      aValue := False;
      SaveState( aState);

      if ReadIdentifier( 'locked')
      then begin
        aValue := True;
        Result := True;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadCCOption( var aValue : Byte): Boolean;
    // Scan for 'cc = <byte>'
    var
      aState : TReaderState;
      anInt  : Integer;
    begin
      Result := False;
      aValue := 0;
      SaveState( aState);

      if ReadKVIntegerOption( 'cc', anInt)
      then begin
        if ( anInt >= 0) and ( anInt < 256)
        then begin
          aValue := anInt;
          Result := True;
        end;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadRndOption( var aValue : Byte): Boolean;
    // Scan for 'rnd = <byte>' - randomization control
    var
      aState : TReaderState;
      anInt  : Integer;
    begin
      Result := False;
      aValue := 1;          // By default randomization will be on
      SaveState( aState);

      if ReadKVIntegerOption( 'rnd', anInt)
      then begin
        if ( anInt >= 0) and ( anInt < 256)
        then begin
          aValue := anInt;
          Result := True;
        end;
      end
      else RestoreState( aState);
    end;


    procedure   TPatchReader.ReadLowOption( var aValue: string);
    // Scan for low = '...'
    var
      aState : TReaderState;
    begin
      SaveState( aState);

      if not ReadKVStringOption( 'low', aValue)
      then RestoreState( aState);
    end;


    procedure   TPatchReader.ReadHighOption( var aValue: string);
    // Scan for high = '...'
    var
      aState : TReaderState;
    begin
      SaveState( aState);

      if not ReadKVStringOption( 'high', aValue)
      then RestoreState( aState);
    end;


    function    TPatchReader.ReadVarOption( var aValue : string): Boolean;
    // Scan for 'variations = '<data>'' - variations
    begin
      Result := ReadKVStringOption( 'variations', aValue);
    end;


    function    TPatchReader.ReadPublicOption( var aValue : Boolean): Boolean;
    // Scan for 'public' - public control
    var
      aState : TReaderState;
    begin
      Result := False;
      aValue := False;
      SaveState( aState);

      if ReadIdentifier( 'public')
      then begin
        aValue := True;
        Result := True;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadStringValue( var aValue : string): Boolean;
    // Scan for 'value = <string>;'
    begin
      Result := ReadKVString( 'value', aValue);
    end;

    function    TPatchReader.ReadColorOption( var aValue: TColor): Boolean;
    var
      aStrVal : string;
      aState  : TReaderState;
    begin
      Result := False;
      aValue := clYellow;
      SaveState( aState);

      if ReadIdentifier( 'color')
      then begin
        RequireEqual;
        ReadToken;
        aStrVal := FToken;

        if FTokenType = ttInteger
        then begin
          aValue := TColor( StrToInt( aStrVal));
          Result := True;
        end
        else begin
          RestoreState( aState);
          ExpectedInteger;
        end;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadVariationOption( var aValue: Integer ): Boolean;
    // Scan for 'variation = <int>' - active variation
    // False result is OK as long as variation 0 is selected then
    var
      aState : TReaderState;
      anInt  : Integer;
    begin
      Result := False;
      aValue := 0;          // By default variation 0 will be selected
      SaveState( aState);

      if ReadKVIntegerOption( 'variation', anInt)
      then begin
        if ( anInt >= 0) and ( anInt < 256)
        then begin
          aValue := anInt;
          Result := True;
        end;
      end
      else RestoreState( aState);
    end;


    function    TPatchReader.ReadGuidOption( var aValue: string  ): Boolean;
    // Scan for 'guid = <string>' - unique patch ID
    // False result is OK as long as a fresh guid is being created then
    var
      aState  : TReaderState;
    begin
      Result := False;
      SaveState( aState);

      if ReadKVStringOption( 'guid', aValue)
      then Result := True
      else RestoreState( aState);
    end;


    procedure   TPatchReader.ReadWire; // virtual;
    var
      aSrc : string;
      aDst : string;
    begin
      RequireIdentifier( 'wire');
      RequireLParen;
      ReadClass; // Result ignored
      aSrc := ReadSrc;
      aDst := ReadDst;
      RequireRParen;
      FWirePanel.AddWire( aSrc, aDst);
    end;


    procedure   TPatchReader.ReadWires; // virtual;
    var
      Done   : Boolean;
      aState : TReaderState;
    begin
      RequireIdentifier( 'wires');
      RequireLParen;
      Done := False;

      while not Done
      do begin
        SaveState( aState);
        ReadToken;

        if ( FTokenType = ttIdentifier) and SameText( FToken, 'wire')
        then begin
          RestoreState( aState);
          ReadWire;
        end
        else if ( FTokenType = ttRParen)
        then Done := True
        else ExpectedIdentifierOrSingle( 'wire', ')');
      end;
    end;


    procedure   TPatchReader.ReadTuningOption; // virtual;
    var
      aState          : TReaderState;
      aRead           : Boolean;
      aReadOk         : Boolean;
      aReferenceA     : Double;
      aNotesPerOctave: Double; 
      aMiddleNote     : Double; 
      anOctaveSpan    : Double; 
    begin
      aReferenceA      := KnobsConversions.ReferenceA    ;
      aNotesPerOctave := KnobsConversions.NotesPerOctave;
      aMiddleNote      := KnobsConversions.MiddleNote    ;
      anOctaveSpan     := KnobsConversions.OctaveSpan    ;
      FTuningChanged   := False;
      aRead            := False;
      SaveState( aState);
      ReadToken;

      if ( FTokenType = ttIdentifier) and SameText( FToken, 'tuning')
      then begin
        RequireLParen;

        aReadOk := True;
        aReadOk := aReadOk and ReadKVFloat( 'ReferenceA'    , aReferenceA    );
        aReadOk := aReadOk and ReadKVFloat( 'NotesPerOctave', aNotesPerOctave);
        aReadOk := aReadOk and ReadKVFloat( 'MiddleNote'    , aMiddleNote    );
        aReadOk := aReadOk and ReadKVFloat( 'OctaveSpan'    , anOctaveSpan   );

        RequireRParen;

        if FReadTuning
        then aRead := aReadOk;
      end
      else RestoreState( aState);

      if not aRead
      then begin
        aReferenceA     := KnobsConversions.DefaultReferenceA    ;
        aNotesPerOctave := KnobsConversions.DefaultNotesPerOctave;
        aMiddleNote     := KnobsConversions.DefaultMiddleNote    ;
        anOctaveSpan    := KnobsConversions.DefaultOctaveSpan    ;
      end;

      FTuningChanged :=
        ( aReferenceA     <> KnobsConversions.ReferenceA     ) or
        ( aNotesPerOctave <> KnobsConversions.NotesPerOctave ) or
        ( aMiddleNote     <> KnobsConversions.MiddleNote     ) or
        ( anOctaveSpan    <> KnobsConversions.OctaveSpan     );

      if FTuningChanged
      then begin
        KnobsConversions.ReferenceA        := aReferenceA    ;
        KnobsConversions.NotesPerOctave    := aNotesPerOctave;
        KnobsConversions.MiddleNote        := aMiddleNote    ;
        KnobsConversions.OctaveSpan        := anOctaveSpan   ;
        KnobsConversions.NotesPerOctaveRec := 1.0 / KnobsConversions.NotesPerOctave;
      end;
    end;


    procedure   TPatchReader.ReadDetail;

      function FixVar( const aVar: string; anAmount: TSignal; aVarCount: Integer): string;
      var
        aParts : TStringList;
        aCount : Integer;
        i      : Integer;
        aValue : TSignal;
      begin
        Result := '';
        aParts := Explode( aVar, ',');

        try
          if   aParts.Count > 0
          then begin
            aCount := StrToIntDef( aParts[ 0], -1);

            if   aCount = aVarCount
            then begin
              Result := IntToStr( acount);

              for i := 0 to aVarCount - 1
              do begin
                aValue := anAmount * StrToFloatDef( aParts[ i + 1], 0);
                Result := Format( '%s,%g', [ Result, aValue], AppLocale);
              end;
            end
            else Result := aVar;
          end
          else Result := aVar;
        finally
          aParts.DisposeOf;
        end;
      end;

    var
      aType     : string;
      aName     : string;
      aValue    : Integer;
      aString   : string;
      isString  : Boolean;
      aState    : TReaderState;
      aLocked   : Boolean;
      aCC       : Byte;
      aRnd      : Byte;
      HasRnd    : Boolean;
      aVar      : string;
      HasVar    : Boolean;
      aPublic   : Boolean;
      aLow      : string;
      aHigh     : string;
      aVarSet   : Boolean;
      aVarFixed : Boolean;
    begin
      isString := False;
      RequireIdentifier( 'detail');
      RequireLParen;
      aLocked := False;
      aCC     := 0;
      aType   := ReadClass;
      aName   := ReadName;
      SaveState( aState);

      if not ReadValue( aValue)
      then begin
        RestoreState( aState);
        isString := ReadStringValue( aString);

        if not isString
        then ExpectedStringOrNumber;
      end;

                 ReadLockedOption( aLocked);
                 ReadCCOption    ( aCC    );
      HasRnd  := ReadRndOption   ( aRnd   );
                 ReadLowOption   ( aLow   );
                 ReadHighOption  ( aHigh  );
                 ReadPublicOption( aPublic);
      HasVar  := ReadVarOption   ( aVar   );

      ScanRParen;                   // this was RequireRParen, but we allow for as of yet unforseen options to be added and be silently ignored

      if Assigned( FModule)
      then begin

      // @@123

        // First set a possible lock, so a later position setting for a knob (SetKnobValue) will not change a possibly
        // associated display (for a locked knob the display holds the value, if present).

        if aLocked
        then FModule.SetLockOn( aType, aName);

        aVarSet   := False;
        aVarFixed := False;

        if FPatchVersion <= 7                                 // Some stuff was changed for param ranges .. which needs be fixed here
        then begin                                            // which needs be fixed here
          if SameText( aName, 'lfo_range'          )
          or SameText( aName, 'squarelfo_range'    )
          or SameText( aName, 'lfotrig_range'      )
          or SameText( aName, 'squarelfotrig_range')
          or SameText( aName, 'randomwalklfo_range')
          or SameText( aName, 'randsig_range'      )
          or SameText( aName, 'pulses_range'       )
          or SameText( aName, 'attractorlfo_range' )
          or SameText( aName, 'squaresinelfo_range')
          or SameText( aName, 'lfomultiphase_range')
          or SameText( aName, 'randsigs_range'     )
          or SameText( aName, 'vanderpollfo_range' )
          then begin
            if HasVar
            then begin
              aVar      := FixVar( aVar, 5.0 / 8.0, 8);       // Value count went from 5 to 8, so scale var values by 5/8
              aVarFixed := True;
            end;
          end;
        end;

        if   ( FPatchVersion < 7)
        and  not aVarFixed
        then begin
          FModule.SetVarOn( aType, aName, aVar, HasVar);
          aVarSet := True;
        end;

        if isString
        then FModule.SetDetailValue( aType, aName, aString)
        else FModule.SetKnobValue  ( aType, aName, aValue );

        if not aVarSet                                        // When vars were not applied before do it here to get the
        then FModule.SetVarOn( aType, aName, aVar, HasVar);   // regular old time behaviour, to not loose variations.

        FModule.SetCCOn          ( aType, aName, aCC               );
        FModule.SetRndOn         ( aType, aName, aRnd   , HasRnd   );
        FModule.SetLowHighMarksOn( aType, aName, aLow   , aHigh    );
      end;
    end;


    procedure   TPatchReader.ReadDetails;
    var
      Done   : Boolean;
      aState : TReaderState;
    begin
      // Skip possible unknown attributes, but scan for 'details'

      ReadToken;

      while not ( FTokenType in [ ttIdentifier, ttRParen, ttEof, ttError]) and not SameText( FToken, 'details')
      do ReadToken;

      if ( FTokenType <> ttIdentifier) and not SameText( FToken, 'details')
      then ExpectedIdentifier( 'details');

      RequireLParen;
      Done := False;

      while not Done
      do begin
        SaveState( aState);
        ReadToken;

        if ( FTokenType = ttIdentifier) and SameText( FToken, 'detail')
        then begin
          RestoreState( aState);
          ReadDetail;
        end
        else if ( FTokenType = ttRParen)
        then Done := True
        else ExpectedIdentifierOrSingle( 'detail', ')');
      end;
    end;


    procedure   TPatchReader.ReadModule; // virtual;
    var
      aModuleClass   : string;
      aModuleName    : string;
      aModuleTitle   : string;
      aModuleType    : Integer;
      aModuleLeft    : Integer;
      aModuleTop     : Integer;
      aModuleColor   : TColor;
      HasCustomColor : Boolean;
      HasRndOption   : Boolean;
      anRndValue     : Byte;
    begin
      RequireIdentifier( 'module');
      RequireLParen;
      aModuleClass   := ReadClass;
      aModuleName    := ReadName;
      aModuleTitle   := ReadTitle;
      aModuleType    := ReadType;
      aModuleLeft    := ReadLeft;
      aModuleTop     := ReadTop;
      HasCustomColor := ReadColorOption( aModuleColor);
      HasRndOption   := ReadRndOption  ( anRndValue   );
      FModule        := FormStore.CreateModule( FWirePanel, aModuleType, NO_DRAG, False);

      if Assigned( FModule)
      then begin
        try
          with FModule
          do begin
            Name  := aModuleName;
            Title := aModuleTitle;
            Left  := aModuleLeft;
            Top   := aModuleTop;

            if HasCustomColor
            then Color := aModuleColor
            else Color := TColor( $ffffffff);

            if HasRndOption
            then AllowRandomization := anRndValue <> 0;

            FixControlInsertions;
          end;

          ReadDetails;
          RequireRParen;
        except
          FreeAndNil( FModule);
          raise;
        end;
      end
      else ReadErrorFmt( 'Could not create module "%s" of type %d with title "%s"', [ aModuleName, aModuleType, aModuleTitle]);
    end;


    procedure   TPatchReader.ReadModules; // virtual;
    var
      Done   : Boolean;
      aState : TReaderState;
    begin
      Done := False;

      repeat
        ReadToken;

        if ( FTokenType = ttIdentifier) and SameText( FToken, 'modules')
        then begin
          SaveState( aState);
          ReadToken;

          if FTokenType = ttLParen
          then Break
          else RestoreState( aState);
        end;

        Done := FTokenType = ttEof;
      until Done;

      if Done
      then ExpectedIdentifier( 'modules');


      while not Done
      do begin
        SaveState( aState);
        ReadToken;

        if ( FTokenType = ttIdentifier) and SameText( FToken, 'module')
        then begin
          RestoreState( aState);
          ReadModule;
        end
        else if ( FTokenType = ttRParen)
        then Done := True
        else ExpectedIdentifierOrSingle( 'module', ')');
      end;
    end;


    procedure   TPatchReader.Read; // virtual;
    var
      aPanelClass : string;
    begin
      FLineNr  := 1;
      FOffset  := 1;
      FPos     := 1;
      RequireIdentifier( 'panel');
      RequireLParen;
      aPanelClass := ReadClass;

      if SameText( aPanelClass, 'TKnobsWirePanel')
      then begin
        FWirePanel.BeginStateChange( True);

        try
          FTuningChanged     := False;
          FWirePanel.Name    := ReadName;
          FWirePanel.Title   := ReadTitle;
          FWirePanel.Version := ReadVersion;
          FPatchVersion      := FWirePanel.Version;

          if not ( FPatchVersion in [ 2 .. 8])
          then ReadErrorFmt( 'can''t handle patch version %d', [ FPatchVersion]);

          ReadVariationOption( FVariation);
          ReadGuidOption( FGuid);
          ReadModules;
          ReadWires;
          ReadTuningOption;
          ScanRParen;                   // this was RequireRParen, but we allow for as of yet unforseen options to be added and be silently ignored
          FWirePanel.ActiveVariation := FVariation;
          FWirePanel.Guid            := FGuid;
        finally
          FWirePanel.EndStateChange( True, True);
        end;
      end
      else ReadErrorFmt( 'can''t handle objects of class ''%s'' - ''TKnobsWirePanel'' expected', [ aPanelClass]);
    end;


    function    TPatchReader.TryRead: Boolean; // virtual;
    begin
      Result := False;

      try
        Read;
        Result := True;
      except
        on E: EKnobsCorruptPatch
        do begin
          if Assigned( FOnReadError)
          then FOnReadError( Self, E.ToString, Result)
          else raise;
        end;

        on E: Exception
        do KilledException( E);
      end;
    end;


//  public

    constructor TPatchReader.Create; // virtual;
    begin
      inherited;
    end;


    destructor  TPatchReader.Destroy; // override;
    begin
      Clear;
      inherited;
    end;


    function    TPatchReader.GetWarnOnStructMismatch: Boolean;
    begin
      Result := FWarnOnStructMismatch;
    end;


    procedure   TPatchReader.SetWarnOnStructMismatch( aValue: Boolean);
    begin
      FWarnOnStructMismatch := aValue;
    end;


    function    TPatchReader.GetTuningChanged: Boolean;
    begin
      Result := FTuningChanged;
    end;


    procedure   TPatchReader.ReadParams(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    ); // virtual;
    // This copies to aWirePanel.
    // The internal panel FWirePanel is read from the clipboard, and when the structure of FWirePanel
    // matches the structure of the selected modules in aWirePanel then copy over the parameters
    // from FWirePanel to aWirePanel.
    begin
      if assigned( aWirePanel)
      then begin
        FData := aData;
        SetWirePanel( TKnobsWirePanel.Create( nil));

        if TryRead
        then begin
          aWirePanel.BeginStateChange( False);

          try
            if not aWirePanel.CopyParamsFrom( FWirePanel, ReadStrict) and WarnOnStructMismatch and ReadStrict
            then Warning( STRUCT_MISMATCH_NO_PASTE);
          finally
            aWirePanel.EndStateChange( False, False);
          end;
        end;
      end;
    end;


    procedure   TPatchReader.ReadString(
      const aData      : string;
      const aWirePanel : TKnobsWirePanel;
      aDistance        : TPoint;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode;
      const aFileName  : string
    ); // virtual;
    begin
      if Assigned( aWirePanel)
      then begin
        FFileName      := aFileName;
        FData          := aData;
        FReadTuning    := aReadTuning;
        FTuningChanged := False;
        FReadMode      := aReadMode;
        SetWirePanel( TKnobsWirePanel.Create( nil));

        if TryRead
        then begin
          aWirePanel.BeginStateChange( True);
       // aWirePanel.Initialize;

          try
            if FReadMode = rmParams
            then begin
              if not aWirePanel.CopyParamsFrom( FWirePanel, ReadStrict) and WarnOnStructMismatch and ReadStrict
              then Warning( STRUCT_MISMATCH_NO_PASTE);
            end
            else begin
              if FReadMode = rmReplace
              then aWirePanel.FreeModules( nil);

              aWirePanel.AddModules(
                FWirePanel,
                aDistance,
                DO_RESTACK,
                DO_REDRAW_WIRES,
                MustDrag,
                CopyMidiCC
              );

              if aReadMode = rmReplace
              then begin
                aWirePanel.Title           := FWirePanel.Title;
                aWirePanel.Version         := FWirePanel.Version;
                aWirePanel.ActiveVariation := FWirePanel.ActiveVariation;
                aWirePanel.Guid            := FWirePanel.Guid;
              end;

       //     aWirePanel.BuildModuleIndex;
            end;
          finally
            aWirePanel.EndStateChange( FReadMode <> rmParams, FReadMode <> rmParams);
          end;
        end;
      end;
    end;


    procedure   TPatchReader.ReadStrings(
      const aData      : TStrings;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode;
      const aFileName  : string
    ); // virtual;
    begin
      ReadString( aData.Text, aWirePanel, Point( 0, 0), True, MustDrag, CopyMidiCC, aReadTuning, aReadMode, aFileName);
    end;


    procedure   TPatchReader.ReadFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    ); // virtual;
    var
      aData : TStrings;
    begin
      aData := TStringList.Create;

      try
        aData.LoadFromFile( aFileName);
        ReadStrings( aData, aWirePanel, ReadStrict, MustDrag, CopyMidiCC, aReadTuning, aReadMode, aFileName);
      finally
        aData.DisposeOf;
      end;
    end;


    procedure   TPatchReader.ImportFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    ); // virtual;
    var
      S     : string;
      aData : TStrings;
    begin
      aData := TStringList.Create;

      try
        aData.LoadFromFile( aFileName);
        S := aData.Text;

        if Pos( 'Panel(', S) = 1
        then begin
          try
            ReadString( S, aWirePanel, Point( 0, 0), ReadStrict, MustDrag, CopyMidiCC, aReadTuning, aReadMode, '<clip>');
          except
            // Ignore exceptions - prolly something else was being pasted ..
            on E: Exception
            do KilledException( E);
          end;
        end;
      finally
        aData.DisposeOf;
      end;
    end;


    procedure   TPatchReader.ReadFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean;
      MustDrag         : Boolean;
      CopyMidiCC       : Boolean;
      aReadTuning      : Boolean;
      aReadMode        : TKnobsReadMode
    ); // virtual;
    //* Reads a wirepanel from the clipboard and copies the modules etc.
    //* found in it to the aWirePanel parameter. Existing modules
    //* are deleted when aReadMode = rmReplace, otherwise the new modules
    //* are added to the wire panel. The modules pasted should be set into
    //* drag mode, see __2 for other cases (in Knobs2013).
    var
      S : string;
    begin
      S := Clipboard.AsText;

      if Pos( 'Panel(', S) = 1
      then begin
        try
          ReadString( S, aWirePanel, Point( 0, 0), ReadStrict, MustDrag, CopyMidiCC, aReadTuning, aReadMode, '<clip>');
        except
          // Ignore exceptions - prolly something else was being pasted ..
          on E: Exception
          do KilledException( E);
        end;
      end;
    end;


    procedure   TPatchReader.ParamsFromFile(
      const aFileName  : string;
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    ); // virtual;
    //* Reads a wirepanel from a file, and when that panel matches the structure
    //* of the current patch selection in aWirePanel copy the parameter values
    //* from the clipboard over to the selection of aWirePanel.
    begin
      ReadFile( aFileName, aWirePanel, ReadStrict, NO_DRAG, NO_COPY_MIDI, NO_COPY_TUNING, rmParams);
    end;


    procedure   TPatchReader.ParamsFromClipboard(
      const aWirePanel : TKnobsWirePanel;
      ReadStrict       : Boolean
    ); // virtual;
    //* Reads a wirepanel from the clipboard, and when that panel matches the structure
    //* of the current patch selection in aWirePanel copy the parameter values
    //* from the clipboard over to the selection of aWirePanel.
    var
      S : string;
    begin
      S := Clipboard.AsText;

      if Pos( 'Panel(', S) = 1
      then ReadParams( S, aWirePanel, ReadStrict);
    end;


{ ========
  TPatchWriter = class( TInterfacedObject, IKnobsPatchWriter)
  strict private
  var
    FFileName     : string;
    FWriteMode    : TKnobsWriteMode;
    FWirePanel    : TKnobsWirePanel;
    FModule       : TKnobsCustomModule;
    FComponent    : TComponent;
    FWire         : TKnobsWire;
    FOnWriteError : TOnPatchWriteError;
    FResult       : string;
    FLineBreak    : string;
  public
    property    FileName  : string read FFileName;
    property    LineBreak : string read FLineBreak write FLineBreak;
  public
    property    OnWriteError : TOnPatchWriteError read FOnWriteError write FOnWriteError;
  protected
}

    procedure   TPatchWriter.Clear; // virtual;
    begin
      FFileName     := '';
      FWirePanel    := nil;
      FResult       := '';
    end;


    procedure   TPatchWriter.Write( anInd: Integer; const S: string);
    begin
      FResult := FResult + Indent( anInd) + S;
    end;


    procedure   TPatchWriter.WriteLn( anInd: Integer; const S: string);
    begin
      Write( anInd, S);
      Write( 0, LineBreak);
    end;


    procedure   TPatchWriter.WriteWire( anInd: Integer); // virtual;
    begin
      with FWire
      do begin
        if ( FWriteMode = wmAll) or ( Source.Module.Selected and Destination.Module.Selected)
        then WriteLn( anInd, AsString);
      end;
    end;


    procedure   TPatchWriter.WriteWires( anInd: Integer); // virtual;
    var
      i : Integer;
    begin
      with FWirePanel
      do begin
        for i := 0 to WireCount - 1
        do begin
          FWire := Wire[ i];
          WriteWire( anInd);
        end;
      end;
    end;


    procedure   TPatchWriter.WriteTuning( anInd: Integer); // virtual;
    begin
      WriteLn( anInd, Format( 'ReferenceA     = ''%g''', [ KnobsConversions.ReferenceA    ]));
      WriteLn( anInd, Format( 'NotesPerOctave = ''%g''', [ KnobsConversions.NotesPerOctave]));
      WriteLn( anInd, Format( 'MiddleNote     = ''%g''', [ KnobsConversions.MiddleNote    ]));
      WriteLn( anInd, Format( 'OctaveSpan     = ''%g''', [ KnobsConversions.OctaveSpan    ]));
    end;


    function    TPatchWriter.MakeLockedStr( aLocked: Boolean): string;
    begin
      if aLocked
      then Result := 'locked '
      else Result := '';
    end;


    function    TPatchWriter.MakeLowHighStr( aLow, aHigh: string): string;
    begin
      if  ( aLow  = '')
      and ( aHigh = '')
      then Result := ''
      else Result := Format( 'low = ''%s'' high = ''%s'' ', [ aLow, aHigh], AppLocale);
    end;


    function    TPatchWriter.MakePublicStr( aPublic: Boolean): string;
    begin
      if aPublic
      then Result := 'public '
      else Result := '';
    end;


    function    TPatchWriter.MakeVariationStr( const aVariations: string): string;
    begin
      if aVariations = ''
      then Result := ''
      else Result := Format( 'variations = ''%s'' ', [ aVariations], AppLocale)
    end;


    procedure   TPatchWriter.WriteDetailData (
      anInd                      : Integer;
      const aValue, aCC, aRnd    : Integer;
      aPublic, aLocked           : Boolean;
      aLow, aHigh                : string;
      const aSuffix, aVariations : string
    ); // overload; virtual;
    begin
      WriteLn(
        anInd,
        Format(
          'detail( class = ''%s'' name = ''%s%s'' value = %s %scc = %d rnd = %d %s%s%s)',
          [
            FComponent.ClassName          ,
            FComponent.Name               ,
            aSuffix                       ,
            IntToStr( aValue)             ,
            MakeLockedStr( aLocked)       ,
            aCC                           ,
            aRnd                          ,
            MakeLowHighStr( aLow, aHigh)  ,
            MakePublicStr( aPublic)       ,
            MakeVariationStr( aVariations)
          ]
        )
      );
    end;


    procedure   TPatchWriter.WriteDetailData(
      anInd             : Integer;
      const aValue      : string;
      aPublic           : Boolean;
      const aVariations : string
    ); // overload; virtual;
    begin
      WriteLn(
        anInd,
        Format(
          'detail( class = ''%s'' name = ''%s'' value = ''%s'' %s%s)',
          [
            FComponent.ClassName          ,
            FComponent.Name               ,
            QuoteEscape( aValue)          ,
            MakePublicStr( aPublic)       ,
            MakeVariationStr( aVariations)
          ]
        )
      );
    end;


    procedure   TPatchWriter.WriteDetail( anInd: Integer); // virtual;

      function BoolToInt( aBool: Boolean): Integer;
      begin
        if aBool
        then Result := 1
        else Result := 0;
      end;

    var
      aDisplay       : TKnobsDisplay;
      aValuedControl : TKnobsValuedControl;
      aPublic        : Boolean;
    begin
      aPublic := False;

      if FComponent is TKnobsValuedControl
      then begin
        aValuedControl := TKnobsValuedControl( FComponent);

        if aValuedControl.Locked
        then begin

          // For a locked knob write both the knob position and the associated display's caption.

          WriteDetailData(
            anInd,
            aValuedControl.KnobPosition,
            aValuedControl.AssignedMIDICC,
            BoolToInt( aValuedControl.AllowRandomization),
            aPublic,
            True,
            aValuedControl.LowRangeAsStr,
            aValuedControl.HighRangeAsStr,
            '',
            aValuedControl.VariationsAsString
          );

          if ( FComponent is TKnobsKnob)
          then begin
            aDisplay := TKnobsKnob( FComponent).Display;

            if Assigned( aDisplay)
            then begin
              FComponent := aDisplay;
              WriteDetailData( anInd, aDisplay.Caption, aPublic, '');
            end;
          end;
        end
        else begin
          if not aValuedControl.IsIndicator
          then
            WriteDetailData(
              anInd,
              aValuedControl.KnobPosition,
              aValuedControl.AssignedMIDICC,
              BoolToInt( aValuedControl.AllowRandomization),
              aPublic,
              False,
              aValuedControl.LowRangeAsStr,
              aValuedControl.HighRangeAsStr,
              '',
              aValuedControl.VariationsAsString
            );
        end;
      end
      else if ( FComponent is TKnobsDisplay) and ( TKnobsDisplay( FComponent).MustSave)
      then WriteDetailData( anInd, TKnobsDisplay( FComponent).Caption, aPublic, '')
      else if ( FComponent is TKnobsFileSelector) and ( TKnobsFileSelector( FComponent).MustSave)
      then WriteDetailData( anInd, TKnobsFileSelector( FComponent).DataFileName, aPublic, '')
      else if ( FComponent is TKnobsXYControl)
      then begin
        WriteDetailData( anInd, TKnobsXYControl( FComponent).KnobPositionX, 0, 0, aPublic, False, '', '', 'x', TKnobsXYControl( FComponent).VariationsAsString);
        WriteDetailData( anInd, TKnobsXYControl( FComponent).KnobPositionY, 0, 0, aPublic, False, '', '', 'y', TKnobsXYControl( FComponent).VariationsAsString);
      end
      else if ( FComponent is TKnobsDataMaker)
      then WriteDetailData( anInd, TKnobsDataMaker( FComponent).AsString, aPublic, TKnobsDataMaker( FComponent).VariationsAsString)
      else if ( FComponent is TKnobsGridControl)
      then WriteDetailData( anInd, TKnobsGridControl( FComponent).AsString, aPublic, '')
      else if ( FComponent is TKnobsConnector) and ( TKnobsConnector( FComponent).WireColor <> TKnobsConnector( FComponent).DefaultWireColor)
      then WriteDetailData( anInd, IntToStr( Cardinal( TKnobsConnector( FComponent).WireColor)), aPublic, '') // Write Color as a string value, that's easier to parse on patch reads
      else if ( FComponent is TKnobsData)
      then WriteDetailData( anInd, TKnobsData( FComponent).AsString, aPublic, '');
    end;


    procedure   TPatchWriter.WriteDetails( anInd: Integer); // virtual;
    var
      i : Integer;
    begin
      with FModule
      do begin
        for i := 0 to ComponentCount - 1
        do begin
          FComponent := Components[ i];
          WriteDetail( anInd);
        end;
      end;
    end;


    procedure   TPatchWriter.WriteModule( anInd, anXOffset, anYOffset: Integer); // virtual;
    var
      anRndValue : Byte;
    begin
      with FModule
      do begin
        if ( FWriteMode = wmAll) or Selected
        then begin
          if AllowRandomization
          then anRndValue := 1
          else anRndValue := 0;

          WriteLn( anInd + 0,         'module('                               );
          WriteLn( anInd + 1, Format(   'class = ''%s''', [ ClassName       ]));
          WriteLn( anInd + 1, Format(   'name  = ''%s''', [ Name            ]));
          WriteLn( anInd + 1, Format(   'title = ''%s''', [ Title           ]));
          WriteLn( anInd + 1, Format(   'type  = %d'    , [ ModuleType      ]));
          WriteLn( anInd + 1, Format(   'left  = %d'    , [ Left + anXOffset]));
          WriteLn( anInd + 1, Format(   'top   = %d'    , [ Top  + anYOffset]));
          WriteLn( anInd + 1, Format(   'color = %d'    , [ Cardinal( Color)]));
          WriteLn( anInd + 1, Format(   'rnd   = %d'    , [ anRndValue      ]));
          WriteLn( anInd + 1,           'details('                            );

          WriteDetails( anInd + 2);

          WriteLn( anInd + 1,           ')'                                   );
          WriteLn( anInd + 0,         ')'                                     );
        end;
      end;
    end;


    procedure   TPatchWriter.WriteModules( anInd: Integer); // virtual;
    var
      i         : Integer;
      anXOffset : Integer;
      anYOffset : Integer;
    begin
      with FWirePanel
      do begin
        // Find largest Left and Top offsets needed to avoid writing a patch with negative Left or Top indices.
        // On write all the modules will be shifted by these offsets.
        anXOffset := 0;
        anYOffset := 0;

        for i := 0 to ModuleCount - 1
        do begin
          FModule := Module[ i];

          if - FModule.Left > anXOffset
          then anXOffset := - FModule.Left;
          if - FModule.Top > anYOffset
          then anYOffset := - FModule.Top;
        end;

        // Write the moddules shifted by the calculated offsets.
        for i := 0 to ModuleCount - 1
        do begin
          FModule := Module[ i];
          WriteModule( anInd, anXOffset, anYOffset);
        end;
      end;
    end;


    procedure   TPatchWriter.WritePatch; // virtual;
    var
      anInd : Integer;
    begin
      with FWirePanel
      do begin
        anInd := 0;
        WriteLn( anInd + 0,         'Panel('                                     );
        WriteLn( anInd + 1, Format(   'class     = ''%s''', [ ClassName        ]));
        WriteLn( anInd + 1, Format(   'name      = ''%s''', [ Name             ]));
        WriteLn( anInd + 1, Format(   'title     = ''%s''', [ Title            ]));
        WriteLn( anInd + 1, Format(   'version   = %d'    , [ KnobsPatchVersion]));
        WriteLn( anInd + 1, Format(   'variation = %d'    , [ ActiveVariation  ]));
        WriteLn( anInd + 1, Format(   'guid      = ''%s''', [ Guid             ]));
        WriteLn( anInd + 1,           'modules('                                 );
        WriteModules( anInd + 2);
        WriteLn( anInd + 1,           ')'                                        );
        WriteLn( anInd + 1,           'wires('                                   );
        WriteWires( anInd + 2);
        WriteLn( anInd + 1,           ')'                                        );
        WriteLn( anInd + 1,           'tuning('                                  );
        WriteTuning( anInd + 2);
        WriteLn( anInd + 1,           ')'                                        );
        WriteLn( anInd + 0,         ')'                                          );
      end;
    end;


    function    TPatchWriter.TryWrite: Boolean; // virtual;
    begin
      Result := False;

      try
        WritePatch;
        Result := True;
      except on E: Exception
      do
        if Assigned( FOnWriteError)
        then FOnWriteError( Self, E.ToString, Result)
        else raise;
      end;
    end;


//  public

    constructor TPatchWriter.Create; // virtual;
    begin
      inherited;

      LineBreak := #13#10;
    end;


    destructor  TPatchWriter.Destroy; // override;
    begin
      Clear;

      inherited;
    end;


    function    TPatchWriter.WriteString(
      const aWirePanel: TKnobsWirePanel;
      const aFileName : string     = '';
      aWriteMode      : TKnobsWriteMode = wmAll
    ): string; // virtual;
    begin
      FResult    := '';
      FFileName  := aFileName;
      FWirePanel := aWirePanel;
      FWriteMode := aWriteMode;

      if TryWrite
      then Result := FResult;
    end;


    function    TPatchWriter.WriteStrings(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string     = '';
      aWriteMode       : TKnobsWriteMode = wmAll
    ): TStrings; // virtual;
    begin
      Result := TStringList.Create;

      try
        Result.Text := WriteString( aWirePanel, aFileName, aWriteMode);
      except
        Result.DisposeOf;
        raise;
      end;
    end;


    procedure   TPatchWriter.WriteFile(
      const aWirePanel : TKnobsWirePanel;
      const aFileName  : string;
      aWriteMode       : TKnobsWriteMode = wmAll
    ); // virtual;
    var
      aStrings   : TStrings;
      anOldTitle : string;
    begin
      if Assigned( aWirePanel)
      then begin
        anOldTitle := aWirePanel.Title;

        if aWriteMode = wmParams
        then aWirePanel.Title := 'params';

        try
          aStrings := WriteStrings( aWirePanel, aFileName, aWriteMode);
        finally
          aWirePanel.Title := anOldTitle;
        end;

        try
          aStrings.SaveToFile( aFileName);
        finally
          aStrings.DisposeOf;
        end;
      end;
    end;


    procedure   TPatchWriter.WriteToClipboard(
      const aWirePanel : TKnobswirePanel;
      aWriteMode       : TKnobsWriteMode = wmAll
    ); // virtual;
    begin
      ClipBoard.AsText := WriteString( aWirePanel, '', aWriteMode);
    end;


end.
