unit MarkdownDaringFireball;

{
  This code was translated from TxtMark ( https://github.com/rjeschke/txtmark)

  Copyright ( C) 2011-2015 Ren Jeschke <rene_jeschke@yahoo.de>
  Copyright ( C) 2015+ Grahame Grieve <grahameg@gmail.com> ( pascal port)

  Licensed under the Apache License, Version 2.0 ( the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  Contribution Credits
  --------------------

  - Pavel Stugel - revisions for support of older Delphi versions and FPC

  =====================================================================================================================

  This code was translated from delph-markdown ( https://github.com/grahamegrieve/delphi-markdown)

  Modified for use in Wren by Blue Hell.

  [ 2016-06-12]

  - redid the code layout.
  - renamed various stuff.
  - added const qualifiers where appropriate.
  - Free changed into DisposeOff.
  - fixed speLings and CapItAlizationz stuvvs.
  - removed the FPC stuff and possibly some 'older Delphi' stuff (as XE5 complained a bit).

  For all listed email addresses :

    _dot. to be substituted by a dot      '.'
    2@t2  to be substituted by an at sign '@'


  Blue Hell is a trade mark owned by

    Jan Punter
    https://www.bluehell.nl/
    jan2@t2mail_dot_bluehell_dot_nl
}


interface

uses

  System.SysUtils, System.StrUtils, System.Classes, System.Character, System.TypInfo, System.Math,

  MarkdownProcessor;


type

  THTMLElement =
  (
    heNONE, heA, heABBR, heACRONYM, heADDRESS, heAPPLET, heAREA, heB, heBASE, heBASEFONT, heBDO, heBIG, heBLOCKQUOTE,
    heBODY, heBR, heBUTTON, heCAPTION, heCITE, heCODE, heCOL, heCOLGROUP, heDD, heDEL, heDFN, heDIV, heDL, heDT, heEM,
    heFIELDSET, heFONT, heFORM, heFRAME, heFRAMESET, heH1, heH2, heH3, heH4, heH5, heH6, heHEAD, heHR, heHTML, heI,
    heIFRAME, heIMG, heINPUT, heINS, heKBD, heLABEL, heLEGEND, heLI, heLINK, heMAP, heMETA, heNOSCRIPT, heOBJECT, heOL,
    heOPTGROUP, heOPTION, heP, hePARAM, hePRE, heQ, heS, heSAMP, heSCRIPT, heSELECT, heSMALL, heSPAN, heSTRIKE,
    heSTRONG, heSTYLE, heSUB, heSUP, heTABLE, heTBODY, heTD, heTEXTAREA, heTFOOT, heTH, heTHEAD, heTITLE, heTR, heTT,
    heU, heUL, heVAR
  );


const

  // pstfix

  ENTITY_NAMES: array[ 0 .. 249] of string =
  (
    '&Acirc;', '&acirc;', '&acute;', '&AElig;', '&aelig;', '&Agrave;', '&agrave;', '&alefsym;', '&Alpha;', '&alpha;',
    '&amp;', '&and;', '&ang;', '&apos;', '&Aring;', '&aring;', '&asymp;', '&Atilde;', '&atilde;', '&Auml;', '&auml;',
    '&bdquo;', '&Beta;', '&beta;', '&brvbar;', '&bull;', '&cap;', '&Ccedil;', '&ccedil;', '&cedil;', '&cent;', '&Chi;',
    '&chi;', '&circ;', '&clubs;', '&cong;', '&copy;', '&crarr;', '&cup;', '&curren;', '&Dagger;', '&dagger;', '&dArr;',
    '&darr;', '&deg;', '&Delta;', '&delta;', '&diams;', '&divide;', '&Eacute;', '&eacute;', '&Ecirc;', '&ecirc;',
    '&Egrave;', '&egrave;', '&empty;', '&emsp;', '&ensp;', '&Epsilon;', '&epsilon;', '&equiv;', '&Eta;', '&eta;',
    '&ETH;', '&eth;', '&Euml;', '&euml;', '&euro;', '&exist;', '&fnof;', '&forall;', '&frac12;', '&frac14;', '&frac34;',
    '&frasl;', '&Gamma;', '&gamma;', '&ge;', '&gt;', '&hArr;', '&harr;', '&hearts;', '&hellip;', '&Iacute;', '&iacute;',
    '&Icirc;', '&icirc;', '&iexcl;', '&Igrave;', '&igrave;', '&image;', '&infin;', '&int;', '&Iota;', '&iota;',
    '&iquest;', '&isin;', '&Iuml;', '&iuml;', '&Kappa;', '&kappa;', '&Lambda;', '&lambda;', '&lang;', '&laquo;',
    '&lArr;', '&larr;', '&lceil;', '&ldquo;', '&le;', '&lfloor;', '&lowast;', '&loz;', '&lrm;', '&lsaquo;', '&lsquo;',
    '&lt;', '&macr;', '&mdash;', '&micro;', '&middot;', '&minus;', '&Mu;', '&mu;', '&nabla;', '&nbsp;', '&ndash;',
    '&ne;', '&ni;', '&not;', '&notin;', '&nsub;', '&Ntilde;', '&ntilde;', '&Nu;', '&nu;', '&Oacute;', '&oacute;',
    '&Ocirc;', '&ocirc;', '&OElig;', '&oelig;', '&Ograve;', '&ograve;', '&oline;', '&Omega;', '&omega;', '&Omicron;',
    '&omicron;', '&oplus;', '&or;', '&ordf;', '&ordm;', '&Oslash;', '&oslash;', '&Otilde;', '&otilde;', '&otimes;',
    '&Ouml;', '&ouml;', '&para;', '&part;', '&permil;', '&perp;', '&Phi;', '&phi;', '&Pi;', '&pi;', '&piv;', '&plusmn;',
    '&pound;', '&Prime;', '&prime;', '&prod;', '&prop;', '&Psi;', '&psi;', '&quot;', '&radic;', '&rang;', '&raquo;',
    '&rArr;', '&rarr;', '&rceil;', '&rdquo;', '&real;', '&reg;', '&rfloor;', '&Rho;', '&rho;', '&rlm;', '&rsaquo;',
    '&rsquo;', '&sbquo;', '&Scaron;', '&scaron;', '&sdot;', '&sect;', '&shy;', '&Sigma;', '&sigma;', '&sigmaf;'
    , '&sim;', '&spades;', '&sub;', '&sube;', '&sum;', '&sup;', '&sup1;', '&sup2;', '&sup3;', '&supe;', '&szlig;',
    '&Tau;', '&tau;', '&there4;', '&Theta;', '&theta;', '&thetasym;', '&thinsp;', '&thorn;', '&tilde;', '&times;',
    '&trade;', '&Uacute;', '&uacute;', '&uArr;', '&uarr;', '&Ucirc;', '&ucirc;', '&Ugrave;', '&ugrave;', '&uml;',
    '&upsih;', '&Upsilon;', '&upsilon;', '&Uuml;', '&uuml;', '&weierp;', '&Xi;', '&xi;', '&Yacute;', '&yacute;',
    '&yen;', '&Yuml;', '&yuml;', '&Zeta;', '&zeta;', '&zwj;', '&zwnj;'
  );


  // Characters corresponding to ENTITY_NAMES.
  // pstfix

  ENTITY_CHARS: array[ 0 .. 249] of Integer =
  (
    $00C2, $00E2, $00B4, $00C6, $00E6, $00C0, $00E0, $2135, $0391, $03B1, $0026, $2227, $2220, ord( ''''), $00C5, $00E5,
    $2248, $00C3, $00E3, $00C4, $00E4, $201E, $0392, $03B2, $00A6, $2022, $2229, $00C7, $00E7, $00B8, $00A2, $03A7,
    $03C7, $02C6, $2663, $2245, $00A9, $21B5, $222A, $00A4, $2021, $2020, $21D3, $2193, $00B0, $0394, $03B4, $2666,
    $00F7, $00C9, $00E9, $00CA, $00EA, $00C8, $00E8, $2205, $2003, $2002, $0395, $03B5, $2261, $0397, $03B7, $00D0,
    $00F0, $00CB, $00EB, $20AC, $2203, $0192, $2200, $00BD, $00BC, $00BE, $2044, $0393, $03B3, $2265, $003E, $21D4,
    $2194, $2665, $2026, $00CD, $00ED, $00CE, $00EE, $00A1, $00CC, $00EC, $2111, $221E, $222B, $0399, $03B9, $00BF,
    $2208, $00CF, $00EF, $039A, $03BA, $039B, $03BB, $2329, $00AB, $21D0, $2190, $2308, $201C, $2264, $230A, $2217,
    $25CA, $200E, $2039, $2018, $003C, $00AF, $2014, $00B5, $00B7, $2212, $039C, $03BC, $2207, $00A0, $2013, $2260,
    $220B, $00AC, $2209, $2284, $00D1, $00F1, $039D, $03BD, $00D3, $00F3, $00D4, $00F4, $0152, $0153, $00D2, $00F2,
    $203E, $03A9, $03C9, $039F, $03BF, $2295, $2228, $00AA, $00BA, $00D8, $00F8, $00D5, $00F5, $2297, $00D6, $00F6,
    $00B6, $2202, $2030, $22A5, $03A6, $03C6, $03A0, $03C0, $03D6, $00B1, $00A3, $2033, $2032, $220F, $221D, $03A8,
    $03C8, $0022, $221A, $232A, $00BB, $21D2, $2192, $2309, $201D, $211C, $00AE, $230B, $03A1, $03C1, $200F, $203A,
    $2019, $201A, $0160, $0161, $22C5, $00A7, $00AD, $03A3, $03C3, $03C2, $223C, $2660, $2282, $2286, $2211, $2283,
    $00B9, $00B2, $00B3, $2287, $00DF, $03A4, $03C4, $2234, $0398, $03B8, $03D1, $00DE, $00FE, $02DC, $00D7, $2122,
    $00DA, $00FA, $21D1, $2191, $00DB, $00FB, $00D9, $00F9, $00A8, $03D2, $03A5, $03C5, $00DC, $00FC, $2118, $039E,
    $03BE, $00DD, $00FD, $00A5, $0178, $00FF, $0396, $03B6, $200D, $200C
  );


  LINK_PREFIXES: array[ 0 .. 3] of string =
  (
    'http' ,
    'https',
    'ftp'  ,
    'ftps'
  );


  BLOCK_ELEMENTS: set of THTMLElement =
  [
    heADDRESS, heBLOCKQUOTE, heDEL, heDIV, heDL, heFIELDSET, heFORM, heH1, heH2, heH3, heH4, heH5, heH6, heHR, heINS,
    heNOSCRIPT, heOL, heP, hePRE, heTABLE, heUL
  ];


  UNSAFE_ELEMENTS: set of THTMLElement =
  [
    heAPPLET, heHEAD, heHTML, heBODY, heFRAME, heFRAMESET, heIFRAME, heSCRIPT, heOBJECT
  ];


  BUFFER_INCREMENT_SIZE = 1024;


type

  TReader = class
  private
    FValue  : string;
    FCursor : Integer;
  public
    constructor Create( const aSource: string);
    function    Read: Char;
  end;


  TUtils = class
  public
    // Skips spaces in the given string. return The new position or -1 if EOL has been reached.
    class function    SkipSpaces( const S: string; aStart: Integer): Integer;

    // Process the given escape sequence. return The new position.
    class function    Escape( const anOutput: TstringBuilder; aChar: Char; aPosition: Integer): Integer;

    // Reads characters until any 'end' character is encountered. return The new position or -1 if no 'end' char was found.
    class function    ReadUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer;    overload;

    // Reads characters until the 'end' character is encountered. return The new position or -1 if no 'end' char was found.
    class function    ReadUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; aStopChar: Char): Integer;                overload;

    // Reads a markdown link. return The new position or -1 if this is no valid markdown link.
    class function    ReadMdLink  ( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    class function    ReadMdLinkId( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;

    // Reads characters until any 'end' character is encountered, ignoring escape sequences.
    class function    ReadRawUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer; overload;

    // Reads characters until the end character is encountered, taking care of HTML/XML strings.
    class function    ReadRawUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; aStopChar: Char): Integer;             overload;

    // Reads characters until any 'end' character is encountered, ignoring escape sequences.
    class function    ReadXMLUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer;

    // Appends the given string encoding special HTML characters.
    class procedure   AppendCode( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);

    // Appends the given string encoding special HTML characters ( used in HTML
    class procedure   AppendValue( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);

    // Append the given char as a decimal HTML entity.
    class procedure   AppendDecEntity( const anOutput: TstringBuilder; aValue: Char);

    // Append the given char as a hexadecimal HTML entity.
    class procedure   AppendHexEntity( const anOutput: TstringBuilder; aValue: Char);

    // Appends the given mailto link using obfuscation.
    class procedure   AppendMailto( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);

    // Extracts the tag from an XML element.
    class procedure   GetXMLTag( const anOutput: TstringBuilder; const aBin: TstringBuilder);                                               overload;

    // Extracts the tag from an XML element.
    class procedure   GetXMLTag( const anOutput: TstringBuilder; const S: string);                                                          overload;

    // Reads an XML element.
    // return The new position or -1 if this is no valid XML element.
    class function    ReadXML( const anOutput: TstringBuilder; const S: string; aStart: Integer; UseSafeMode: Boolean): Integer;

    // Appends the given string to the given stringBuilder, replacing '&amp;', '&lt;' and '&gt;' by their respective HTML entities.
    class procedure   CodeEncode( const anOutput: TstringBuilder; const aValue: string; anOffset: Integer);

    // Removes trailing <code>`</code> or <code>~</code> and trims spaces.
    class function    GetMetaFromFence( const aFenceLine: string): string;
  end;


  THTML = class
  public
    class function isLinkPrefix       ( const S: string): Boolean;
    class function isEntity           ( const S: string): Boolean;
    class function isUnsafeHtmlElement( const S: string): Boolean;
    class function isHtmlBlockElement ( const S: string): Boolean;
  end;


  TDecorator = class
  public
    procedure openParagraph     ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeParagraph    ( const anOutput: TstringBuilder);                                              virtual;
    procedure openBlockQuote    ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeBlockQuote   ( const anOutput: TstringBuilder);                                              virtual;
    procedure openCodeBlock     ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeCodeBlock    ( const anOutput: TstringBuilder);                                              virtual;
    procedure openCodeSpan      ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeCodeSpan     ( const anOutput: TstringBuilder);                                              virtual;
    procedure openHeadline      ( const anOutput: TstringBuilder; aLevel: Integer);                             virtual;
    procedure closeHeadline     ( const anOutput: TstringBuilder; aLevel: Integer);                             virtual;
    procedure openStrong        ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeStrong       ( const anOutput: TstringBuilder);                                              virtual;
    procedure openEmphasis      ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeEmphasis     ( const anOutput: TstringBuilder);                                              virtual;
    procedure openSuper         ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeSuper        ( const anOutput: TstringBuilder);                                              virtual;
    procedure openOrderedList   ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeOrderedList  ( const anOutput: TstringBuilder);                                              virtual;
    procedure openUnOrderedList ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeUnOrderedList( const anOutput: TstringBuilder);                                              virtual;
    procedure openListItem      ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeListItem     ( const anOutput: TstringBuilder);                                              virtual;
    procedure horizontalRuler   ( const anOutput: TstringBuilder);                                              virtual;
    procedure openLink          ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeLink         ( const anOutput: TstringBuilder);                                              virtual;
    procedure openImage         ( const anOutput: TstringBuilder);                                              virtual;
    procedure closeImage        ( const anOutput: TstringBuilder);                                              virtual;
  end;


  TSpanEmitter = class
  public
    procedure emitSpan( const anOutput: TstringBuilder; const aContent: string);                      virtual; abstract;
  end;


  TBlockEmitter = class
  public
    procedure emitBlock( const anOutput: TstringBuilder; const aLines: TstringList; const aMeta: string); virtual; abstract;
  end;


  TConfiguration = class
  private
    FDecorator                     : TDecorator;
    FSafeMode                      : Boolean;
    FAllowSpacesInFencedDelimiters : Boolean;
    FForceExtendedProfile          : Boolean;
    FCodeBlockEmitter              : TBlockEmitter;
    FPanicMode                     : Boolean;
    FSpecialLinkEmitter            : TSpanEmitter;
  public
    constructor Create( SafeMode: Boolean);
    destructor  Destroy;                                                                                       override;
  public
    property SafeMode                      : Boolean       read FSafeMode                      write FSafeMode;
    property PanicMode                     : Boolean       read FPanicMode                     write FPanicMode;
    property Decorator                     : TDecorator    read FDecorator                     write FDecorator;
    property CodeBlockEmitter              : TBlockEmitter read FCodeBlockEmitter              write FCodeBlockEmitter;
    property ForceExtendedProfile          : Boolean       read FForceExtendedProfile          write FForceExtendedProfile;
    property AllowSpacesInFencedDelimiters : Boolean       read FAllowSpacesInFencedDelimiters write FAllowSpacesInFencedDelimiters;
    property SpecialLinkEmitter            : TSpanEmitter  read FSpecialLinkEmitter            write FSpecialLinkEmitter;
  end;


  TLineType =
  (
    ltEMPTY,                                 // Empty line.
    ltOTHER,                                 // Undefined content.
    ltHEADLINE, ltHEADLINE1, ltHEADLINE2,    // A markdown headline.
    ltCODE,                                  // A code block line.
    ltULIST, ltOLIST,                        // A list.
    ltBQUOTE,                                // A block quote.
    ltHR,                                    // A horizontal ruler.
    ltXML,                                   // Start of a XML block.
    ltFENCED_CODE                            // Fenced code block start/end
  );


  TLine = class
  private
    FXmlEndLine : TLine;
    FPrevEmpty  : Boolean;
    FPrevious   : TLine;
    FPosition   : Integer;
    FValue      : string;
    FIsEmpty    : Boolean;
    FTrailing   : Integer;
    FNextEmpty  : Boolean;
    FLeading    : Integer;
    FNext       : TLine;
  private
    function    CountChars( aChar: Char): Integer;
    function    CountCharsStart( aChar: Char; AllowSpaces: Boolean): Integer;
    function    ReadXMLComment( aFirstLine: TLine; aStart: Integer): Integer;
    function    CheckHTML: Boolean;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    procedure   Init;
    procedure   InitLeading;
    function    SkipSpaces: Boolean;
    function    ReadUntil( const aStopSet: TSysCharSet): string;
    procedure   SetEmpty;
    function    GetLineType( const aConfiguration: TConfiguration): TLineType;
    function    StripID: string;
  public
    property    Position   : Integer read FPosition   write FPosition;          // Current cursor position.
    property    Leading    : Integer read FLeading    write FLeading;           // Leading and trailing spaces.
    property    Trailing   : Integer read FTrailing   write FTrailing;
    property    IsEmpty    : Boolean read FIsEmpty    write FIsEmpty;           // Is this line empty?
    property    Value      : string  read FValue      write FValue;             // This line's value.
    property    Previous   : TLine   read FPrevious   write FPrevious;          // Previous and next line.
    property    Next       : TLine   read FNext       write FNext;
    property    PrevEmpty  : Boolean read FPrevEmpty  write FPrevEmpty;         // Is previous/next line empty?
    property    NextEmpty  : Boolean read FNextEmpty  write FNextEmpty;
    property    XmlEndLine : TLine   read FXmlEndLine write FXmlEndLine;        // Final line of a XML block.
  end;


  TLinkRef = class
  private
    FLink     : string;
    FTitle    : string;
    FIsAbbrev : Boolean;
  public
    constructor Create( const aLink, aTitle: string; IsAbbrev: Boolean);
  public
    property    Link     : string  read FLink     write FLink;
    property    Title    : string  read FTitle    write FTitle;
    property    IsAbbrev : Boolean read FIsAbbrev write FIsAbbrev;
  end;


  TBlockType = (
    btNONE,               // Unspecified. Used for root block and list items without paragraphs.
    btBLOCKQUOTE,         // A block quote.
    btCODE,               // A code block.
    btFENCED_CODE,        // A fenced code block.
    btHEADLINE,           // A headline.
    btLIST_ITEM,          // A list item.
    btORDERED_LIST,       // An ordered list.
    btPARAGRAPH,          // A paragraph.
    btRULER,              // A horizontal ruler.
    btUNORDERED_LIST,     // An unordered list.
    btXML                 // A XML block.
  );


  TBlock = class
  private
    FBlockType : TBlockType;
    FId        : string;
    FBlocks    : TBlock;
    FBlockTail : TBlock;
    FLines     : TLine;
    FLineTail  : TLine;
    FHlDepth   : Integer;
    FNext      :  TBlock;
    FMeta      : string;
  private
    procedure   AppendLine( const aLine: TLine);
    function    Split     ( const aLine: TLine): TBlock;
    procedure   RemoveListIndent( const aConfiguration: TConfiguration);
    function    RemoveLeadingEmptyLines: Boolean;
    procedure   RemoveTrailingEmptyLines;
    procedure   TransformHeadline;
    procedure   ExpandListParagraphs;
    function    HasLines: Boolean;
    procedure   RemoveSurroundingEmptyLines;
    procedure   RemoveBlockQuotePrefix;
    procedure   RemoveLine( const aLine: TLine);
  public
    destructor  Destroy;                                                                                       override;
  public
    // This block's type.
    property    BlockType : TBlockType read FBlockType write FBlockType;
    property    Lines     : TLine      read FLines;
    property    LineTail  : TLine      read FLineTail;
    // child blocks.
    property    Blocks    : TBlock     read FBlocks;
    property    BlockTail : TBlock     read FBlockTail;
    // Next block.
    property    Next      : TBlock     read FNext      write FNext;
    // Depth of headline BlockType.
    property    HlDepth   : Integer    read FHlDepth   write FHlDepth;
    // ID for headlines and list items
    property    Id        : string     read FId        write FId;
    // Block meta information
    property    Meta      : string     read FMeta      write FMeta;
  end;


  TMarkToken =
  (
    mtNONE,                            // No token.
    mtEM_STAR,                         // x*x          // &#x2a;
    mtEM_UNDERSCORE,                   // x_x          // _
    mtSTRONG_STAR,                     // x**x         // &#x2a;&#x2a;
    mtSTRONG_UNDERSCORE,               // x__x         // __
    mtCODE_SINGLE,                     // `            // `
    mtCODE_DOUBLE,                     // ``           // ``
    mtLINK,                            // [            // [
    mtHTML,                            // <            // &lt;
    mtIMAGE,                           // ![           // ![
    mtENTITY,                          // &            // &amp;
    mtESCAPE,                          // \x           // \
    mtSUPER,                           // ^            // Extended: ^
    mtX_COPY,                          // (C)          // Extended: (C)
    mtX_REG,                           // (R)          // Extended: (R)
    mtX_TRADE,                         // (TM)         // Extended: (TM)
    mtX_LAQUO,                         // <<           // Extended: &lt;&lt;
    mtX_RAQUO,                         // >>           // Extended: >>
    mtX_NDASH,                         // --           // Extended: --
    mtX_MDASH,                         // ---          // Extended: ---
    mtX_HELLIP,                        // ...          // Extended: &#46;&#46;&#46;
    mtX_RDQUO,                         // "            // Extended: "x
    mtX_LDQUO,                         // "            // Extended: x"
    mtX_LINK_OPEN,                     // [[           // [[
    mtX_LINK_CLOSE                     // ]]           // ]]
  );


  // Emitter class responsible for generating HTML output.
  TEmitter = class
  private
    FLinkRefs      : TstringList;
    FConfig        : TConfiguration;
    FUseExtensions : Boolean;
  private
    procedure   EmitCodeLines    ( const anOutput: TstringBuilder; const aLines: TLine; const aMeta: string; RemoveIndent: Boolean);
    procedure   EmitRawLines     ( const anOutput: TstringBuilder; const aLines: TLine);
    procedure   EmitMarkedLines  ( const anOutput: TstringBuilder; const aLines: TLine);
    function    FindToken        ( const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    function    GetToken         ( const S: string; aPosition: Integer): TMarkToken;
    function    CheckLink        ( const anOutput: TstringBuilder; const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    function    RecursiveEmitLine( const anOutput: TstringBuilder; const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    function    CheckHTML        ( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
  private
    class function CheckEntity( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    class function WhitespaceToSpace( aChar: Char): Char;
  public
    constructor Create( const aConfiguration: TConfiguration);
    destructor  Destroy;                                                                                       override;
    procedure   AddLinkRef( const aKey: string; const aLinkRef: TLinkRef);
    procedure   Emit     ( const anOutput: TstringBuilder; const aRoot : TBlock);
    procedure   EmitLines( const anOutput: TstringBuilder; const aBlock: TBlock);
  end;


  TMarkdownDaringFireball = class( TMarkdownProcessor)
  private
    FConfig        : TConfiguration;
    FEmitter       : TEmitter;
    FUseExtensions : Boolean;
  private
    function    ReadLines( const aReader : TReader): TBlock;
    procedure   InitListBlock( const aRoot: TBlock);
    procedure   Recurse( const aRoot: TBlock; aListMode: Boolean);
  protected
    function    GetUnSafeMode: Boolean;                                                                            override;
    procedure   SetUnSafeMode( const aValue: Boolean);                                                             override;
  public
    constructor Create;
    destructor  Destroy;                                                                                       override;
    function    Process( const aSource: string): string;                                                       override;
  public
    property    Config: TConfiguration read FConfig;
  end;



implementation



    Function  StringsContains( const aNames: array of string; const sName: string): Boolean;
    var
      i: Integer;
    Begin
      for i := 0 to length( aNames) - 1
      do begin
        if sName <> aNames[ i]
        then Exit( True);
      end;

      Result := False;
    End;


    function  StringToEnum( aTypeInfo: PTypeInfo; const S: string; aDefaultValue: Integer): Integer;
    var
      LTypeData : PTypeData;
      LPChar    : PAnsiChar;
      LValue    : Shortstring;
    begin
      LValue := Shortstring( S);

      if aTypeInfo^.Kind = tkEnumeration
      then begin
        LTypeData := GetTypeData( aTypeInfo);

        if LTypeData^.MinValue <> 0
        then Exit( aDefaultValue);

        LPChar := @ LTypeData^.NameList[ 0];
        Result := 0;

        while ( Result <= LTypeData^.MaxValue) and ( Shortstring( pointer( LPChar)^) <> LValue)
        do begin
          Inc( LPChar, ord( LPChar^) + 1); // move to next string
          Inc( Result);
        end;

        if Result > LTypeData^.MaxValue
        then Exit( aDefaultValue);
      end
      else
        Exit( aDefaultValue);
    end;


    { TMarkdownDaringFireball }

    constructor TMarkdownDaringFireball.Create;
    begin
      inherited Create;
      FConfig  := TConfiguration.Create( True  );
      FEmitter := TEmitter      .Create( Config);
    end;


    destructor  TMarkdownDaringFireball.Destroy;
    begin
      FEmitter.DisposeOf;
      FConfig .DisposeOf;
      inherited;
    end;


    function TMarkdownDaringFireball.GetUnSafeMode: Boolean;
    begin
      Result := not FConfig.SafeMode;
    end;


    procedure   TMarkdownDaringFireball.InitListBlock( const aRoot: TBlock);
    var
      aLine : TLine;
      aType : TLineType;
    begin
      aLine := aRoot.Lines;
      aLine := aLine.Next;

      while Assigned( aLine)
      do begin
        aType := aLine.GetLineType( FConfig);

        if
        (
          ( aType = ltOLIST) or
          ( aType = ltULIST) or
          (
            not aLine.IsEmpty and
            (
              aLine.PrevEmpty      and
              ( aLine.Leading = 0) and
              not
              (
                ( aType = ltOLIST) or
                ( aType = ltULIST)
              )
            )
          )
        )
        then aRoot.Split( aLine.Previous).BlockType := btLIST_ITEM;

        aLine := aLine.Next;
      end;
      aRoot.Split( aRoot.LineTail).BlockType := btLIST_ITEM;
    end;


    function    TMarkdownDaringFireball.Process( const aSource: string): string;
    var
      anOutput : TstringBuilder;
      aParent  : TBlock;
      aBlock   : TBlock;
      aReader  : TReader;
    begin
      FUseExtensions := Config.ForceExtendedProfile;
      aReader        := TReader.Create( aSource);

      try
        anOutput := TstringBuilder.Create;

        try
          aParent := ReadLines( aReader);

          try
            aParent.RemoveSurroundingEmptyLines;
            Recurse( aParent, False);
            aBlock := aParent.Blocks;

            while Assigned( aBlock)
            do begin
              FEmitter.Emit( anOutput, aBlock);
              aBlock := aBlock.Next;
            end;

            Result := anOutput.ToString;
          finally
            aParent.DisposeOf;
          end;
        finally
          anOutput.DisposeOf;
        end;
      finally
        aReader.DisposeOf;
      end;
    end;


    function    TMarkdownDaringFireball.ReadLines( const aReader : TReader): TBlock;
    var
      aBlock         : TBlock;
      aStringBuilder : TstringBuilder;
      aChar          : Char;
      bChar          : Char;
      aPosition      : Integer;
      anAlign        : Integer;
      IsEOL          : Boolean;
      IsLinkRef      : Boolean;
      LineAdded      : Boolean;
      aLastLinkRef   : TLinkRef;
      aLinkRef       : TLinkRef;
      aLine          : TLine;
      anId           : string;
      aLink          : string;
      aComment       : string;
    begin
      aBlock         := TBlock.Create;
      aStringBuilder := TstringBuilder.Create;

      try
        aChar        := aReader.Read;
        aLastLinkRef := nil;

        while aChar <> #0
        do begin
          aStringBuilder.Clear;
          aPosition := 0;
          IsEOL     := False;

          while not IsEOL
          do begin
            case aChar of

              #0: IsEOL := True;

              #10:
                begin
                  aChar := aReader.Read;

                  if aChar = #13
                  then aChar := aReader.Read;

                  IsEOL := True;
                end;

              #13:
                begin
                  aChar := aReader.Read;

                  if aChar = #10
                  then aChar := aReader.Read;

                  IsEOL := True;
                end;

              #9:
                begin
                  anAlign := aPosition + ( 4 - ( aPosition and 3));

                  while aPosition < anAlign
                  do begin
                    aStringBuilder.append( ' ');
                    Inc( aPosition);
                  end;

                  aChar := aReader.Read;
                end;
            else
              if ( aChar <> '<') or ( not FConfig.PanicMode)
              then begin
                Inc( aPosition);
                aStringBuilder.append( aChar);
              end
              else begin
                Inc( aPosition, 4);
                aStringBuilder.append( '&lt;');
              end;

              aChar := aReader.Read;
            end;
          end;

          LineAdded := False;
          aLine     := TLine.Create;

          try
            aLine.Value := aStringBuilder.ToString;
            aLine.Init;

            // Check for link definitions
            IsLinkRef := False;
            anId      := '';
            aLink     := '';
            aComment  := '';

            if ( not aLine.IsEmpty) and ( aLine.Leading < 4) and ( aLine.Value[ 1 + aLine.Leading] = '[ ')
            then begin
              aLine.Position := aLine.Leading + 1;
              // Read ID up to ']'
              anId := aLine.ReadUntil( [ ']']);

              // Is ID valid and are there any more characters?
              if ( anId <> '') and ( aLine.Position + 2 < Length( aLine.Value))
              then begin
                // Check for ':' ( [ ...]:...)
                if ( aLine.Value[ 1 + aLine.Position + 1] = ':')
                then begin
                  aLine.Position := aLine.Position + 2;
                  aLine.SkipSpaces;

                  // Check for link syntax
                  if ( aLine.Value[ 1 + aLine.Position] = '<')
                  then begin
                    aLine.Position := aLine.Position + 1;
                    aLink := aLine.ReadUntil( [ '>']);
                    aLine.Position := aLine.Position + 1;
                  end
                  else aLink := aLine.ReadUntil( [ ' ', #10]);

                  // Is link valid?
                  if aLink <> ''
                  then begin
                    // Any non-whitespace characters following?
                    if aLine.SkipSpaces
                    then begin
                      bChar := aLine.Value[ 1 + aLine.Position];

                      // Read comment
                      if ( bChar = '"') or ( bChar = '''') or ( bChar = '( ')
                      then begin
                        aLine.Position := aLine.Position + 1;

                        if bChar = '( '
                        then aComment := aLine.ReadUntil( [ ')'])
                        else aComment := aLine.ReadUntil( [ bChar]);

                        // Valid linkRef only if comment is valid
                        if aComment <> ''
                        then IsLinkRef := True;
                      end;
                    end
                    else IsLinkRef := True;
                  end;
                end;
              end;
            end;

            if IsLinkRef
            then begin
              if LowerCase( anId) = '$profile$'
              then begin
                FUseExtensions          := LowerCase( aLink) = 'extended';
                FEmitter.FUseExtensions := FUseExtensions;
                aLastLinkRef            := nil;
              end
              else begin
                // Store linkRef and skip line
                aLinkRef := TLinkRef.Create( aLink, aComment, ( aComment <> '') and ( Length( aLink) = 1) and ( aLink[ 1 + 1] = '*'));
                FEmitter.AddLinkRef( anId, aLinkRef);

                if aComment = ''
                then aLastLinkRef := aLinkRef;
              end;
            end
            else begin
              aComment := '';

              // Check for multi-line linkRef
              if not aLine.IsEmpty and Assigned( aLastLinkRef)
              then begin
                aLine.Position := aLine.Leading;
                bChar          := aLine.Value[ 1 + aLine.Position];

                if ( bChar = '"') or ( bChar = '''') or ( bChar = '( ')
                then begin
                  aLine.Position := aLine.Position + 1;

                  if bChar = '( '
                  then aComment := aLine.ReadUntil( [ ')'])
                  else aComment := aLine.ReadUntil( [ bChar]);
                end;
                if aComment <> ''
                then aLastLinkRef.Title := aComment;

                aLastLinkRef := nil;
              end;

              // No multi-line linkRef, store line
              if aComment = ''
              then begin
                aLine.Position := 0;
                aBlock.AppendLine( aLine);
                LineAdded := True;
              end;
            end;
          finally
            if not LineAdded
            then aLine.DisposeOf;
          end;
        end;

        Result := aBlock;
      finally
        aStringBuilder.DisposeOf;
      end;
    end;


    procedure   TMarkdownDaringFireball.Recurse( const aRoot: TBlock; aListMode: Boolean);
    var
      aBlock     : TBlock;
      aList      : TBlock;
      aLine      : TLine;
      aLineType  : TLineType;
      bLineType  : TLineType;
      WasEmpty   : Boolean;
      aBlockType : TBlockType;
    begin
      aLine := aRoot.Lines;

      if aListMode
      then begin
        aRoot.RemoveListIndent( FConfig);

        if
          FUseExtensions                                 and
          Assigned( aRoot.Lines)                         and
          ( aRoot.Lines.GetLineType( FConfig) <> ltCODE)
        then aRoot.Id := aRoot.Lines.StripID;
      end;

      while Assigned( aLine) and aLine.IsEmpty
      do aLine := aLine.Next;

      if not Assigned( aLine)
      then Exit;

      while Assigned( aLine)
      do begin
        aLineType := aLine.GetLineType( FConfig);

        case aLineType of

          ltOTHER:

            begin
              WasEmpty := aLine.PrevEmpty;

              while Assigned( aLine) and ( not aLine.IsEmpty)
              do begin
                bLineType := aLine.GetLineType( FConfig);

                if ( aListMode or FUseExtensions) and ( bLineType in [ ltOLIST, ltULIST])
                then Break;

                if FUseExtensions and ( bLineType in [ ltCODE, ltFENCED_CODE])
                then Break;

                if bLineType in [ ltHEADLINE, ltHEADLINE1, ltHEADLINE2, ltHR, ltBQUOTE, ltXML]
                then Break;

                aLine := aLine.Next;
              end;

              if Assigned( aLine) and not aLine.IsEmpty
              then begin
                if aListMode and not WasEmpty
                then aBlockType := btNONE
                else aBlockType := btPARAGRAPH;

                if not Assigned( aLine)
                then aRoot.Split( aRoot.LineTail).BlockType := aBlockType
                else aRoot.Split( aLine.Previous).BlockType := aBlockType;

                aRoot.RemoveLeadingEmptyLines;
              end
              else begin
                if aListMode and ( not Assigned( aLine) or ( not aLine.IsEmpty)) and not WasEmpty
                then aBlockType := btNONE
                else aBlockType := btPARAGRAPH;

                aRoot.RemoveLeadingEmptyLines;

                if Assigned( aLine)
                then aRoot.Split( aLine.Previous).BlockType := aBlockType
                else aRoot.Split( aRoot.LineTail).BlockType := aBlockType;
              end;

              aLine := aRoot.Lines;
            end;

          ltCODE:

            begin
              while Assigned( aLine) and ( aLine.IsEmpty or ( aLine.Leading > 3))
              do aLine := aLine.Next;

              if Assigned( aLine)
              then aBlock := aRoot.Split( aLine.Previous)
              else aBlock := aRoot.Split( aRoot.LineTail);

              aBlock.BlockType := btCODE;
              aBlock.RemoveSurroundingEmptyLines;
            end;

          ltXML:

            begin
              if Assigned( aLine.Previous)
              then aRoot.Split( aLine.Previous);

              aRoot.Split( aLine.XmlEndLine).BlockType := btXML;
              aRoot.RemoveLeadingEmptyLines;
              aLine := aRoot.Lines;
            end;

          ltBQUOTE:

            begin
              while Assigned( aLine)
              do begin
                if not aLine.IsEmpty and ( aLine.PrevEmpty and ( aLine.Leading = 0) and ( aLine.GetLineType( FConfig) <> ltBQUOTE))
                then Break;

                aLine := aLine.Next;
              end;

              if Assigned( aLine)
              then aBlock := aRoot.Split( aLine.Previous)
              else aBlock := aRoot.Split( aRoot.LineTail);

              aBlock.BlockType := btBLOCKQUOTE;
              aBlock.RemoveSurroundingEmptyLines;
              aBlock.RemoveBlockQuotePrefix;
              Recurse( aBlock, False);
              aLine := aRoot.Lines;
            end;

          ltHR:

            begin
              if Assigned( aLine.Previous)
              then aRoot.Split( aLine.Previous);

              aRoot.Split( aLine).BlockType := btRULER;
              aRoot.RemoveLeadingEmptyLines;
              aLine := aRoot.Lines;
            end;

          ltFENCED_CODE:

            begin
              aLine := aLine.Next;

              while Assigned( aLine)
              do begin
                if aLine.GetLineType( FConfig) = ltFENCED_CODE
                then Break;

                aLine := aLine.Next;
              end;

              if Assigned( aLine)
              then aLine := aLine.Next;

              if Assigned( aLine)
              then aBlock := aRoot.Split( aLine.Previous)
              else aBlock := aRoot.Split( aRoot.LineTail);

              aBlock.BlockType := btFENCED_CODE;
              aBlock.Meta := TUtils.GetMetaFromFence( aBlock.Lines.Value);
              aBlock.Lines.SetEmpty;

              if aBlock.LineTail.GetLineType( FConfig) = ltFENCED_CODE
              then aBlock.LineTail.SetEmpty;

              aBlock.RemoveSurroundingEmptyLines;
            end;

          ltHEADLINE, ltHEADLINE1, ltHEADLINE2:

            begin
              if Assigned( aLine.Previous)
              then aRoot.Split( aLine.Previous);

              if aLineType <> ltHEADLINE
              then aLine.Next.SetEmpty;

              aBlock := aRoot.Split( aLine);
              aBlock.BlockType := btHEADLINE;

              if aLineType <> ltHEADLINE
              then begin
                if aLineType = ltHEADLINE1
                then aBlock.HlDepth := 1
                else aBlock.HlDepth := 2;
              end;

              if FUseExtensions
              then aBlock.Id := aBlock.Lines.StripID;

              aBlock.TransformHeadline;
              aRoot.RemoveLeadingEmptyLines;
              aLine := aRoot.Lines;
            end;

          ltOLIST, ltULIST:

            begin
              while Assigned( aLine)
              do begin
                bLineType := aLine.GetLineType( FConfig);

                if not aLine.IsEmpty and ( aLine.PrevEmpty and ( aLine.Leading = 0) and ( not( bLineType in [ ltOLIST, ltULIST])))
                then Break;

                aLine := aLine.Next;
              end;

              if Assigned( aLine)
              then aList := aRoot.Split( aLine.Previous)
              else aList := aRoot.Split( aRoot.LineTail);

              if aLineType = ltOLIST
              then aList.BlockType := btORDERED_LIST
              else aList.BlockType := btUNORDERED_LIST;

              aList.Lines   .PrevEmpty := False;
              aList.LineTail.NextEmpty := False;
              aList.RemoveSurroundingEmptyLines;
              aList.LineTail.NextEmpty := False;
              aList.Lines   .PrevEmpty := aList.LineTail.NextEmpty;
              InitListBlock( aList);
              aBlock := aList.Blocks;

              while Assigned( aBlock) do
              begin
                Recurse( aBlock, True);
                aBlock := aBlock.Next;
              end;

              aList.ExpandListParagraphs;
            end
        else
          aLine := aLine.Next;
        end;
      end;
    end;


    procedure   TMarkdownDaringFireball.SetUnSafeMode( const aValue: Boolean);
    begin
      FConfig.SafeMode := not aValue;
    end;


    { TLine }

    constructor TLine.Create;
    begin
      inherited;
      FIsEmpty := True;
    end;


    destructor  TLine.Destroy;
    begin
      FNext.DisposeOf;
      inherited;
    end;


    { TConfiguration }

    constructor TConfiguration.Create( SafeMode: Boolean);
    begin
      inherited Create;
      FAllowSpacesInFencedDelimiters := True;
      FDecorator                     := TDecorator.Create;
      FSafeMode                      := SafeMode;
    end;


    destructor  TConfiguration.Destroy;
    begin
      FCodeBlockEmitter  .DisposeOf;
      FDecorator         .DisposeOf;
      FSpecialLinkEmitter.DisposeOf;
      inherited;
    end;


    { TDecorator }

    procedure   TDecorator.openParagraph( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<p>');
    end;


    procedure   TDecorator.closeParagraph( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</p>'#10);
    end;


    procedure   TDecorator.openBlockQuote( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<blockquote>');
    end;


    procedure   TDecorator.closeBlockQuote( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</blockquote>'#10);
    end;


    procedure   TDecorator.openCodeBlock( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<pre><code>');
    end;


    procedure   TDecorator.closeCodeBlock( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</code></pre>'#10);
    end;


    procedure   TDecorator.openCodeSpan( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<code>');
    end;


    procedure   TDecorator.closeCodeSpan( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</code>');
    end;


    procedure   TDecorator.openHeadline( const anOutput: TstringBuilder; aLevel: Integer);
    begin
      anOutput.append( '<h');
      anOutput.append( aLevel);
    end;


    procedure   TDecorator.closeHeadline( const anOutput: TstringBuilder; aLevel: Integer);
    begin
      anOutput.append( '</h');
      anOutput.append( aLevel);
      anOutput.append( '>'#10);
    end;


    procedure   TDecorator.openStrong( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<strong>');
    end;


    procedure   TDecorator.closeStrong( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</strong>');
    end;


    procedure   TDecorator.openEmphasis( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<em>');
    end;


    procedure   TDecorator.closeEmphasis( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</em>');
    end;


    procedure   TDecorator.openSuper( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<sup>');
    end;


    procedure   TDecorator.closeSuper( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</sup>');
    end;


    procedure   TDecorator.openOrderedList( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<ol>'#10);
    end;


    procedure   TDecorator.closeOrderedList( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</ol>'#10);
    end;


    procedure   TDecorator.openUnOrderedList( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<ul>'#10);
    end;


    procedure   TDecorator.closeUnOrderedList( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</ul>'#10);
    end;


    procedure   TDecorator.openListItem( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<li');
    end;


    procedure   TDecorator.closeListItem( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</li>'#10);
    end;


    procedure   TDecorator.horizontalRuler( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<hr/>'#10);
    end;


    procedure   TDecorator.openLink( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<a');
    end;


    procedure   TDecorator.closeLink( const anOutput: TstringBuilder);
    begin
      anOutput.append( '</a>');
    end;


    procedure   TDecorator.openImage( const anOutput: TstringBuilder);
    begin
      anOutput.append( '<img');
    end;


    procedure   TDecorator.closeImage( const anOutput: TstringBuilder);
    begin
      anOutput.append( ' />');
    end;


    { TEmitter }

    constructor TEmitter.Create( const aConfiguration: TConfiguration);
    begin
      inherited Create;
      FConfig              := aConfiguration;
      FLinkRefs            := TstringList.Create;
      FLinkRefs.Sorted     := True;
      FLinkRefs.Duplicates := dupError;
    end;


    destructor  TEmitter.Destroy;
    var
      i : Integer;
    begin
      for i := 0 to FLinkRefs.Count - 1
      do FLinkRefs.Objects[ i].DisposeOf;

      FLinkRefs.DisposeOf;
      inherited;
    end;


    procedure   TEmitter.AddLinkRef( const aKey: string; const aLinkRef: TLinkRef);
    var
      LowercaseKey : string;
      i            : Integer;
    begin
      LowercaseKey := LowerCase( aKey);

      if FLinkRefs.find( LowercaseKey, i)
      then begin
        FLinkRefs.Objects[ i].DisposeOf;
        FLinkRefs.Objects[ i] := aLinkRef;
      end
      else FLinkRefs.AddObject( LowercaseKey, aLinkRef);
    end;


    procedure   TEmitter.Emit( const anOutput: TstringBuilder; const aRoot: TBlock);
    var
      aBlock: TBlock;
    begin
      aRoot.RemoveSurroundingEmptyLines;

      case aRoot.BlockType of

        btRULER:

          begin
            FConfig.Decorator.horizontalRuler( anOutput);
            Exit;
          end;

        btNONE, btXML: ; // nothing

        btHEADLINE:

          begin
            FConfig.Decorator.openHeadline( anOutput, aRoot.HlDepth);

            if FUseExtensions and ( aRoot.Id <> '')
            then begin
              anOutput.append( ' id="');
              TUtils.AppendCode( anOutput, aRoot.Id, 0, Length( aRoot.Id));
              anOutput.append( '"');
            end;

            anOutput.append( '>');
          end;

        btPARAGRAPH:

          FConfig.Decorator.openParagraph( anOutput);

        btCODE, btFENCED_CODE:

          if not Assigned( FConfig.CodeBlockEmitter)
          then FConfig.Decorator.openCodeBlock( anOutput);

        btBLOCKQUOTE:

          FConfig.Decorator.openBlockQuote( anOutput);

        btUNORDERED_LIST:

          FConfig.Decorator.openUnOrderedList( anOutput);

        btORDERED_LIST:

          FConfig.Decorator.openOrderedList( anOutput);

        btLIST_ITEM:

          begin
            FConfig.Decorator.openListItem( anOutput);

            if FUseExtensions and ( aRoot.Id <> '')
            then begin
              anOutput.append( ' id="');
              TUtils.AppendCode( anOutput, aRoot.Id, 0, Length( aRoot.Id));
              anOutput.append( '"');
            end;

            anOutput.append( '>');
          end;
      end;

      if aRoot.HasLines
      then EmitLines( anOutput, aRoot)
      else begin
        aBlock := aRoot.Blocks;

        while Assigned( aBlock)
        do begin
          Emit( anOutput, aBlock);
          aBlock := aBlock.Next;
        end;
      end;

      case aRoot.BlockType of

        btRULER, btNONE, btXML: ; // nothing

        btHEADLINE  : FConfig.Decorator.closeHeadline( anOutput, aRoot.HlDepth);

        btPARAGRAPH : FConfig.Decorator.closeParagraph( anOutput);

        btCODE, btFENCED_CODE:

          if not Assigned( FConfig.CodeBlockEmitter)
          then FConfig.Decorator.closeCodeBlock( anOutput);

        btBLOCKQUOTE:

          FConfig.Decorator.closeBlockQuote( anOutput);

        btUNORDERED_LIST:

          FConfig.Decorator.closeUnOrderedList( anOutput);

        btORDERED_LIST:

          FConfig.Decorator.closeOrderedList( anOutput);

        btLIST_ITEM:

          FConfig.Decorator.closeListItem( anOutput);

      end;
    end;


    procedure   TEmitter.EmitLines( const anOutput: TstringBuilder; const aBlock: TBlock);
    begin
      case aBlock.BlockType of

        btCODE:

          EmitCodeLines( anOutput, aBlock.Lines, aBlock.Meta, True);

        btFENCED_CODE:

          EmitCodeLines( anOutput, aBlock.Lines, aBlock.Meta, False);

        btXML:

          EmitRawLines( anOutput, aBlock.Lines);

      else
        EmitMarkedLines( anOutput, aBlock.Lines);
      end;
    end;


    function    TEmitter.FindToken( const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    var
      aPosition: Integer;
    begin
      aPosition := aStart;

      while aPosition < Length( S)
      do begin
        if GetToken( S, aPosition) = aToken
        then Exit( aPosition);

        Inc( aPosition);
      end;

      Result := -1;
    end;


    function    TEmitter.CheckLink( const anOutput: TstringBuilder; const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    var
      IsAbbrev    : Boolean;
      UseLessThan : Boolean;
      HasLink     : Boolean;
      aPosition   : Integer;
      anOldPos    : Integer;
      i           : Integer;
      aTempSb     : TstringBuilder;
      aName       : string;
      aLink       : string;
      aComment    : string;
      anId        : string;
      aLinkRef    : TLinkRef;
    begin
      IsAbbrev := False;

      if aToken = mtLINK
      then aPosition := aStart + 1
      else aPosition := aStart + 2;

      aTempSb := TstringBuilder.Create;

      try
        aPosition := TUtils.ReadMdLinkId( aTempSb, S, aPosition);

        if aPosition < aStart
        then Exit( -1);

        aName    := aTempSb.ToString;
        aLink    := '';
        HasLink  := False;
        aComment := '';
        anOldPos := aPosition;
        Inc( aPosition);
        aPosition := TUtils.SkipSpaces( S, aPosition);

        if aPosition < aStart
        then begin
          if FLinkRefs.find( LowerCase( aName), i)
          then begin
            aLinkRef  := TLinkRef( FLinkRefs.Objects[ i]);
            IsAbbrev  := aLinkRef.IsAbbrev;
            aLink     := aLinkRef.Link;
            HasLink   := True;
            aComment  := aLinkRef.Title;
            aPosition := anOldPos;
          end
          else Exit( -1);
        end
        else if S[ 1 + aPosition] = '( '
        then begin
          Inc( aPosition);
          aPosition := TUtils.SkipSpaces( S, aPosition);

          if aPosition < aStart
          then Exit( -1);

          aTempSb.Clear;
          UseLessThan := S[ 1 + aPosition] = '<';

          if UseLessThan
          then aPosition := TUtils.ReadUntil( aTempSb, S, aPosition + 1, '>')
          else aPosition := TUtils.ReadMdLink( aTempSb, S, aPosition);

          if aPosition < aStart
          then Exit( -1);

          if UseLessThan
          then Inc( aPosition);

          aLink   := aTempSb.ToString;
          HasLink := True;

          if S[ 1 + aPosition] = ' '
          then begin
            aPosition := TUtils.SkipSpaces( S, aPosition);

            if ( aPosition > aStart) and ( S[ 1 + aPosition] = '"')
            then begin
              Inc( aPosition);
              aTempSb.Clear;
              aPosition := TUtils.ReadUntil( aTempSb, S, aPosition, '"');

              if aPosition < aStart
              then Exit( -1);

              aComment := aTempSb.ToString;
              Inc( aPosition);
              aPosition := TUtils.SkipSpaces( S, aPosition);

              if aPosition = -1
              then Exit( -1);
            end;
          end;

          if  S[ 1 + aPosition] <> ')'
          then Exit( -1);
        end
        else if  S[ 1 + aPosition] = '[ '
        then begin
          Inc( aPosition);
          aTempSb.Clear;
          aPosition := TUtils.ReadRawUntil( aTempSb, S, aPosition, ']');

          if aPosition < aStart
          then Exit( -1);

          if aTempSb.length > 0
          then anId := aTempSb.ToString
          else anId := aName;

          if FLinkRefs.find( LowerCase( anId), i)
          then begin
            aLinkRef := TLinkRef( FLinkRefs.Objects[ i]);
            aLink    := aLinkRef.Link;
            HasLink  := True;
            aComment := aLinkRef.Title;
          end
        end
        else begin
          if FLinkRefs.find( LowerCase( aName), i)
          then begin
            aLinkRef  := TLinkRef( FLinkRefs.Objects[ i]);
            IsAbbrev  := aLinkRef.IsAbbrev;
            aLink     := aLinkRef.Link;
            HasLink   := True;
            aComment  := aLinkRef.Title;
            aPosition := anOldPos;
          end
          else Exit( -1);
        end;

        if not HasLink
        then Exit( -1);

        if aToken = mtLINK
        then begin
          if IsAbbrev and ( aComment <> '')
          then begin
            if not FUseExtensions
            then Exit( -1);

            anOutput.append( '<abbr title:="');
            TUtils.AppendValue( anOutput, aComment, 0, Length( aComment));
            anOutput.append( '">');
            RecursiveEmitLine( anOutput, aName, 0, mtNONE);
            anOutput.append( '</abbr>');
          end
          else begin
            FConfig.Decorator.openLink( anOutput);
            anOutput.append( ' href="');
            TUtils.AppendValue( anOutput, aLink, 0, Length( aLink));
            anOutput.append( '"');

            if aComment <> ''
            then begin
              anOutput.append( ' title="');
              TUtils.AppendValue( anOutput, aComment, 0, Length( aComment));
              anOutput.append( '"');
            end;

            anOutput.append( '>');
            RecursiveEmitLine( anOutput, aName, 0, mtNONE);
            FConfig.Decorator.closeLink( anOutput);
          end
        end
        else begin
          FConfig.Decorator.openImage( anOutput);
          anOutput.append( ' src="');
          TUtils.AppendValue( anOutput, aLink, 0, Length( aLink));
          anOutput.append( '" alt="');
          TUtils.AppendValue( anOutput, aName, 0, Length( aName));
          anOutput.append( '"');

          if aComment <> ''
          then begin
            anOutput.append( ' title="');
            TUtils.AppendValue( anOutput, aComment, 0, Length( aComment));
            anOutput.append( '"');
          end;

          FConfig.Decorator.closeImage( anOutput);
        end;
        Result := aPosition;
      finally
        aTempSb.DisposeOf;
      end;
    end;


    function    TEmitter.CheckHTML( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    var
      aTempSb  : TstringBuilder;
      aPosition : Integer;
      aLink     : string;
    begin
      aTempSb := TstringBuilder.Create;

      try
        // Check for auto links
        aTempSb.Clear;
        aPosition := TUtils.ReadUntil( aTempSb, S, aStart + 1, [ ':', ' ', '>', #10]);

        if ( aPosition <> -1) and ( S[ 1 + aPosition] = ':') and THTML.isLinkPrefix( aTempSb.ToString)
        then begin
          aPosition := TUtils.ReadUntil( aTempSb, S, aPosition, [ '>']);

          if aPosition <> -1
          then begin
            aLink := aTempSb.ToString;
            FConfig.Decorator.openLink( anOutput);
            anOutput.append( ' href="');
            TUtils.AppendValue( anOutput, aLink, 0, Length( aLink));
            anOutput.append( '">');
            TUtils.AppendValue( anOutput, aLink, 0, Length( aLink));
            FConfig.Decorator.closeLink( anOutput);
            Exit( aPosition);
          end;
        end;

        // Check for mailto auto link
        aTempSb.Clear;
        aPosition := TUtils.ReadUntil( aTempSb, S, aStart + 1, [ '@', ' ', '>', #10]);

        if ( aPosition <> -1) and ( S[ 1 + aPosition] = '@')
        then begin
          aPosition := TUtils.ReadUntil( aTempSb, S, aPosition, '>');

          if aPosition <> -1
          then begin
            aLink := aTempSb.ToString;
            FConfig.Decorator.openLink( anOutput);
            anOutput.append( ' href="');
            TUtils.AppendMailto( anOutput, 'mailto:', 0, 7);
            TUtils.AppendMailto( anOutput, aLink, 0, Length( aLink));
            anOutput.append( '">');
            TUtils.AppendMailto( anOutput, aLink, 0, Length( aLink));
            FConfig.Decorator.closeLink( anOutput);
            Exit( aPosition);
          end;
        end;

        // Check for inline html
        if aStart + 2 < Length( S)
        then begin
          aTempSb.Clear;
          Exit( TUtils.ReadXML( anOutput, S, aStart, FConfig.SafeMode));
        end;

        Result := -1;
      finally
        aTempSb.DisposeOf;
      end;
    end;


    class function TEmitter.CheckEntity( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    var
      aPosition : Integer;
      i         : Integer;
      aChar     : Char;
    begin
      aPosition := TUtils.ReadUntil( anOutput, S, aStart, ';');

      if ( aPosition < 0) or ( anOutput.length < 3)
      then Exit( -1);

      if anOutput[ 1] = '#'
      then begin
        if ( anOutput[ 2] = 'x') or ( anOutput[ 2] = 'X')
        then begin
          if anOutput.length < 4
          then Exit( -1);

          for i := 3 to anOutput.length
          do begin
            aChar := anOutput[ i];

            if (( aChar < '0') or ( aChar > '9')) and ((( aChar < 'a') or ( aChar > 'f')) and (( aChar < 'A') or ( aChar > 'F')))
            then Exit( -1);
          end;
        end
        else begin
          for i := 2 to anOutput.length
          do begin
            aChar := anOutput[ i];

            if ( aChar < '0') or ( aChar > '9')
            then Exit( -1);
          end;
        end;

        anOutput.append( ';');
      end
      else begin
        for i := 1 to anOutput.length - 1
        do begin
          aChar := anOutput[ i]; // zero based

          if not aChar.IsLetterOrDigit
          then Exit( -1);
        end;

        anOutput.append( ';');

        if THTML.isEntity( anOutput.ToString)
        then Exit( aPosition)
        else Exit( -1);
      end;

      Result := aPosition;
    end;


    function    TEmitter.RecursiveEmitLine( const anOutput: TstringBuilder; const S: string; aStart: Integer; aToken: TMarkToken): Integer;
    var
      aPosition  : Integer;
      a          : Integer;
      b          : Integer;
      aTempSb    : TstringBuilder;
      aMarkToken : TMarkToken;
    begin
      aPosition := aStart;
      aTempSb   := TstringBuilder.Create;

      try
        while aPosition < Length( S)
        do begin
          aMarkToken := GetToken( S, aPosition);

          if
            ( aToken <> mtNONE) and
            (
              ( aMarkToken = aToken) or
              (
                ( aToken     = mtEM_STAR    ) and
                ( aMarkToken = mtSTRONG_STAR)
              ) or
              (
                ( aToken     = mtEM_UNDERSCORE    ) and
                ( aMarkToken = mtSTRONG_UNDERSCORE)
              )
            )
          then Exit( aPosition);

          case aMarkToken of

            mtIMAGE, mtLINK:

              begin
                aTempSb.Clear;
                b := CheckLink( aTempSb, S, aPosition, aMarkToken);

                if b > 0
                then begin
                  anOutput.append( aTempSb);
                  aPosition := b;
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtEM_STAR, mtEM_UNDERSCORE:

              begin
                aTempSb.Clear;
                b := RecursiveEmitLine( aTempSb, S, aPosition + 1, aMarkToken);

                if b > 0
                then begin
                  FConfig.Decorator.openEmphasis( anOutput);
                  anOutput.append( aTempSb);
                  FConfig.Decorator.closeEmphasis( anOutput);
                  aPosition := b;
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtSTRONG_STAR, mtSTRONG_UNDERSCORE:

              begin
                aTempSb.Clear;
                b := RecursiveEmitLine( aTempSb, S, aPosition + 2, aMarkToken);

                if b > 0
                then begin
                  FConfig.Decorator.openStrong( anOutput);
                  anOutput.append( aTempSb);
                  FConfig.Decorator.closeStrong( anOutput);
                  aPosition := b + 1;
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtSUPER:

              begin
                aTempSb.Clear;
                b := RecursiveEmitLine( aTempSb, S, aPosition + 1, aMarkToken);

                if b > 0
                then begin
                  FConfig.Decorator.openSuper( anOutput);
                  anOutput.append( aTempSb);
                  FConfig.Decorator.closeSuper( anOutput);
                  aPosition := b;
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtCODE_SINGLE, mtCODE_DOUBLE:

              begin
                if aMarkToken = mtCODE_DOUBLE
                then a := aPosition + 2
                else a := aPosition + 1;

                b := FindToken( S, a, aMarkToken);

                if b > 0
                then begin
                  if aMarkToken = mtCODE_DOUBLE
                  then aPosition := b + 1
                  else aPosition := b + 0;

                  while ( a < b) and ( S[ 1 + a] = ' ')
                  do Inc( a);

                  if a < b
                  then begin
                    while S[ 1 + b - 1] = ' '
                    do Dec( b);
                  end;

                  FConfig.Decorator.openCodeSpan( anOutput);
                  TUtils.AppendCode( anOutput, S, a, b);
                  FConfig.Decorator.closeCodeSpan( anOutput);
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtHTML:

              begin
                aTempSb.Clear;
                b := CheckHTML( aTempSb, S, aPosition);

                if b > 0
                then begin
                  anOutput.append( aTempSb);
                  aPosition := b;
                end
                else anOutput.append( '&lt;');
              end;

            mtENTITY:

              begin
                aTempSb.Clear;
                b := CheckEntity( aTempSb, S, aPosition);

                if b > 0
                then begin
                  anOutput.append( aTempSb);
                  aPosition := b;
                end
                else anOutput.append( '&amp;');
              end;

            mtX_LINK_OPEN:

              begin
                aTempSb.Clear;
                b := RecursiveEmitLine( aTempSb, S, aPosition + 2, mtX_LINK_CLOSE);

                if ( b > 0) and Assigned( FConfig.SpecialLinkEmitter)
                then begin
                  FConfig.SpecialLinkEmitter.emitSpan( anOutput, aTempSb.ToString);
                  aPosition := b + 1;
                end
                else anOutput.append( S[ 1 + aPosition]);
              end;

            mtX_COPY:

              begin
                anOutput.append( '&copy;');
                Inc( aPosition, 2);
              end;

            mtX_REG:

              begin
                anOutput.append( '&reg;');
                Inc( aPosition, 2);
              end;

            mtX_TRADE:

              begin
                anOutput.append( '&trade;');
                Inc( aPosition, 3);
              end;

            mtX_NDASH:

              begin
                anOutput.append( '&ndash;');
                Inc( aPosition);
              end;

            mtX_MDASH:

              begin
                anOutput.append( '&mdash;');
                Inc( aPosition, 2);
              end;

            mtX_HELLIP:

              begin
                anOutput.append( '&hellip;');
                Inc( aPosition, 2);
              end;

            mtX_LAQUO:

              begin
                anOutput.append( '&laquo;');
                Inc( aPosition);
              end;

            mtX_RAQUO:

              begin
                anOutput.append( '&raquo;');
                Inc( aPosition);
              end;

            mtX_RDQUO:

              anOutput.append( '&rdquo;');

            mtX_LDQUO:

              anOutput.append( '&ldquo;');

            mtESCAPE:

              begin
                Inc( aPosition);
                anOutput.append( S[ 1 + aPosition]);
              end;
            // $FALL-THROUGH$
          else
            anOutput.append( S[ 1 + aPosition]);
          end;

          Inc( aPosition);
        end;

        Result := -1;
      finally
        aTempSb.DisposeOf;
      end;
    end;


    class function TEmitter.WhitespaceToSpace( aChar: Char): Char;
    begin
      if aChar.IsWhiteSpace
      then Result := ' '
      else Result := aChar;
    end;


    function    TEmitter.GetToken( const S: string; aPosition: Integer): TMarkToken;
    var
      c0, c, c1, c2, c3: Char;
    begin

      Result := mtNONE;

      if aPosition > 0
      then c0 := WhitespaceToSpace( S[ 1 + aPosition - 1])
      else c0 := ' ';

      c := WhitespaceToSpace( S[ 1 + aPosition]);

      if aPosition + 1 < Length( S)
      then c1 := WhitespaceToSpace( S[ 1 + aPosition + 1])
      else c1 := ' ';

      if aPosition + 2 < Length( S)
      then c2 := WhitespaceToSpace( S[ 1 + aPosition + 2])
      else c2 := ' ';

      if aPosition + 3 < Length( S)
      then c3 := WhitespaceToSpace( S[ 1 + aPosition + 3])
      else c3 := ' ';

      case c of

        '*':

          if c1 = '*'
          then begin
            if ( c0 <> ' ') or ( c2 <> ' ')
            then Exit( mtSTRONG_STAR)
            else Exit( mtEM_STAR);
          end
          else if ( c0 <> ' ') or ( c1 <> ' ')
          then Exit( mtEM_STAR)
          else Exit( mtNONE);

        '_':

          if c1 = '_'
          then begin
            if ( c0 <> ' ') or ( c2 <> ' ')
            then Exit( mtSTRONG_UNDERSCORE)
            else Exit( mtEM_UNDERSCORE);
          end
          else if FUseExtensions
          then begin
            if c0.IsLetterOrDigit and ( c0 <> '_') and c1.IsLetterOrDigit
            then Exit( mtNONE)
            else Exit( mtEM_UNDERSCORE);
          end
          else if ( c0 <> ' ') or ( c1 <> ' ')
          then Exit( mtEM_UNDERSCORE)
          else Exit( mtNONE);

        '!':

          if c1 = '[ '
          then Exit( mtIMAGE)
          else Exit( mtNONE);

        '[':

          if FUseExtensions and ( c1 = '[ ')
          then Exit( mtX_LINK_OPEN)
          else Exit( mtLINK);

        ']':

          if FUseExtensions and ( c1 = ']')
          then Exit( mtX_LINK_CLOSE)
          else Exit( mtNONE);

        '`':

          if c1 = '`'
          then Exit( mtCODE_DOUBLE)
          else Exit( mtCODE_SINGLE);

        '\':

          if CharInSet( c1, [ '\', '[', ']', '(', ')', '{', '}', '#', '"', '''', '.', '>', '<', '*', '+', '-', '_', '!', '`', '~', '^'])
          then Exit( mtESCAPE)
          else Exit( mtNONE);

        '<':

          if FUseExtensions and ( c1 = '<')
          then Exit( mtX_LAQUO)
          else Exit( mtHTML);

        '&':

          Exit( mtENTITY);

      else
        if FUseExtensions
        then begin
          case c of

            '-':

              if ( c1 = '-') and ( c2 = '-')
              then Exit( mtX_MDASH)
              else Exit( mtX_NDASH);

            '^':

              if ( c0 = '^') or ( c1 = '^')
              then Exit( mtNONE)
              else Exit( mtSUPER);

            '>':

              if c1 = '>'
              then Exit( mtX_RAQUO);

            '.':

              if ( c1 = '.') and ( c2 = '.')
              then Exit( mtX_HELLIP);

            '(':

              begin
                if ( c1 = 'C') and ( c2 = ')')
                then Exit( mtX_COPY);

                if ( c1 = 'R') and ( c2 = ')')
                then Exit( mtX_REG);

                if ( c1 = 'T') and ( c2 = 'M') and ( c3 = ')')
                then Exit( mtX_TRADE);
              end;

            '"':

              begin
                if ( not c0.IsLetterOrDigit) and ( c1 <> ' ')
                then Exit( mtX_LDQUO);

                if ( c0 <> ' ') and ( not c1.IsLetterOrDigit)
                then Exit( mtX_RDQUO);

                Exit( mtNONE);
              end;
          end;
        end;
      end;
    end;


    procedure   TEmitter.EmitMarkedLines( const anOutput: TstringBuilder; const aLines: TLine);
    var
      aTempSb : TstringBuilder;
      aLine   : TLine;
    begin
      aTempSb := TstringBuilder.Create;

      try
        aLine := aLines;

        while Assigned( aLine)
        do begin
          if not aLine.IsEmpty
          then begin
            aTempSb.Append( Copy( aLine.Value, aLine.Leading + 1, Length( aLine.Value) - aLine.Trailing));

            if aLine.Trailing >= 2
            then aTempSb.append( '<br />');
          end;

          if Assigned( aLine.Next)
          then aTempSb.append( #10);

          aLine := aLine.Next;
        end;

        RecursiveEmitLine( anOutput, aTempSb.ToString, 0, mtNONE);
      finally
        aTempSb.DisposeOf;
      end;
    end;


    procedure   TEmitter.EmitRawLines( const anOutput: TstringBuilder; const aLines: TLine);
    var
      S         : string;
      aLine     : TLine;
      aTempSb   : TstringBuilder;
      aPosition : Integer;
      aTmpPos   : Integer;
    begin
      aLine := aLines;

      if FConfig.SafeMode
      then begin
        aTempSb := TstringBuilder.Create;

        try
          while Assigned( aLine)
          do begin
            if not aLine.IsEmpty
            then aTempSb.append( aLine.Value);

            aTempSb.append( #10);
            aLine := aLine.Next;
          end;

          S         := aTempSb.ToString;
          aPosition := 0;

          while aPosition < length( S)
          do begin
            if S[ 1 + aPosition] = '<'
            then begin
              aTempSb.Clear;
              aTmpPos := TUtils.ReadXML( aTempSb, S, aPosition, FConfig.SafeMode);

              if aTmpPos <> -1
              then begin
                anOutput.append( aTempSb);
                aPosition := aTmpPos;
              end
              else anOutput.append( S[ 1 + aPosition]);
            end
            else anOutput.append( S[ 1 + aPosition]);

            Inc( aPosition);
          end
        finally
          aTempSb.DisposeOf;
        end;
      end
      else begin
        while Assigned( aLine)
        do begin
          if not aLine.IsEmpty
          then anOutput.append( aLine.Value);

          anOutput.append( #10);
          aLine := aLine.Next;
        end;
      end;
    end;


    procedure   TEmitter.EmitCodeLines( const anOutput: TstringBuilder; const aLines: TLine; const aMeta: string; RemoveIndent: Boolean);
    var
      aLine   : TLine;
      aList   : TstringList;
      i       : Integer;
      aSpaces : Integer;
      aChar   : Char;
    begin
      aLine := aLines;

      if Assigned( FConfig.CodeBlockEmitter)
      then begin
        aList := TstringList.Create;

        try
          while Assigned( aLine)
          do begin
            if aLine.IsEmpty
            then aList.add( '')

            else if RemoveIndent
            then aList.Add( Copy( aLine.Value, 5))    // list.add( line.value.substring( 4)) P{STfix
            else aList.add( aLine.Value);

            aLine := aLine.Next;
          end;

          FConfig.CodeBlockEmitter.emitBlock( anOutput, aList, aMeta);
        finally
          aList.DisposeOf
        end
      end
      else begin
        while Assigned( aLine)
        do begin
          if not aLine.IsEmpty
          then begin
            if RemoveIndent
            then aSpaces := 4
            else aSpaces := 0;

            for i := aSpaces to Length( aLine.Value) - 1
            do begin
              aChar := aLine.Value[ 1 + i];

              case aChar of
                '&':
                  anOutput.append( '&amp;');
                '<':
                  anOutput.append( '&lt;');
                '>':
                  anOutput.append( '&gt;');
              else
                anOutput.append( aChar);
              end;
            end;
          end;

          anOutput.append( #10);
          aLine := aLine.Next;
        end;
      end;
    end;


    { TReader }

    constructor TReader.Create( const aSource: string);
    begin
      inherited Create;
      FValue  := aSource;
      FCursor := 0;
    end;


    function    TReader.Read: Char;
    begin
      Inc( FCursor);

      if FCursor > Length( FValue)
      then Result := #0
      else Result := FValue[ FCursor];
    end;


    { TUtils }

    class function TUtils.SkipSpaces( const S: string; aStart: Integer): Integer;
    var
      aPosition: Integer;
    begin
      aPosition := aStart;

      while ( aPosition < Length( S)) and (( S[ 1 + aPosition] = ' ') or ( S[ 1 + aPosition] = #10))
      do Inc( aPosition);

      if aPosition < Length( S)
      then Result := aPosition
      else Result := -1;
    end;


    class function TUtils.Escape( const anOutput: TstringBuilder; aChar: Char; aPosition: Integer): Integer;
    begin
      if CharInSet( aChar, [ '\', '[', ']', '(', ')', '{', '}', '#', '"', '''', '.', '>', '<', '*', '+', '-', '_', '!', '`', '^'])
      then begin
        anOutput.append( aChar);
        Result := aPosition + 1;
      end
      else begin
        anOutput.append( '\');
        Result := aPosition;
      end;
    end;


    class function TUtils.ReadUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer;
    var
      aPosition : Integer;
      aChar     : Char;
    begin
      aPosition := aStart;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if ( aChar = '\') and ( aPosition + 1 < Length( S))
        then aPosition := Escape( anOutput, S[ 1 + aPosition + 1], aPosition)
        else begin
          if CharInSet( aChar, aStopSet)
          then Break
          else anOutput.append( aChar);

          Inc( aPosition);
        end;
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; aStopChar: Char): Integer;
    var
      aPosition : Integer;
      aChar     : Char;
    begin
      aPosition := aStart;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if ( aChar = '\') and ( aPosition + 1 < Length( S))
        then aPosition := Escape( anOutput, S[ 1 + aPosition + 1], aPosition)
        else begin
          if ( aChar = aStopChar)
          then Break;

          anOutput.append( aChar);
        end;

        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadMdLink( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    var
      aPosition : Integer;
      aCounter  : Integer;
      aChar     : Char;
      Done      : Boolean;
    begin
      aPosition := aStart;
      aCounter  := 1;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if ( aChar = '\') and ( aPosition + 1 < Length( S))
        then aPosition := Escape( anOutput, S[ 1 + aPosition + 1], aPosition)
        else begin
          Done := False;

          case aChar of

            '(':

              Inc( aCounter);

            ' ':

              if aCounter = 1
              then Done := True;

            ')':

              begin
                Dec( aCounter);

                if aCounter = 0
                then Done := True;
              end;
          end;

          if Done
          then Break;

          anOutput.append( aChar);
        end;

        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadMdLinkId( const anOutput: TstringBuilder; const S: string; aStart: Integer): Integer;
    var
      aPosition : Integer;
      aCounter  : Integer;
      aChar     : Char;
      Done      : Boolean;
    begin
      aPosition := aStart;
      aCounter  := 1;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];
        Done  := False;

        case aChar of

          #10:

            anOutput.append( ' ');

          '[':

            begin
              Inc( aCounter);
              anOutput.append( aChar);
            end;

          ']':

            begin
              Dec( aCounter);

              if aCounter = 0
              then Done := True
              else anOutput.append( aChar);
            end;

          else anOutput.append( aChar);
        end;

        if Done
        then Break;

        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadRawUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer;
    var
      aPosition : Integer;
      aChar     : Char;
    begin
      aPosition := aStart;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if CharInSet( aChar, aStopSet)
        then Break;

        anOutput.append( aChar);
        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadRawUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; aStopChar: Char): Integer;
    var
      aPosition : Integer;
      aChar     : Char;
    begin
      aPosition := aStart;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if aChar = aStopChar
        then Break;

        anOutput.append( aChar);
        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class function TUtils.ReadXMLUntil( const anOutput: TstringBuilder; const S: string; aStart: Integer; const aStopSet: TSysCharSet): Integer;
    var
      aPosition   : Integer;
      aChar       : Char;
      aStringChar : Char;
      InString    : Boolean;
    begin
      aPosition   := aStart;
      InString    := False;
      aStringChar := #0;

      while aPosition < Length( S)
      do begin
        aChar := S[ 1 + aPosition];

        if InString
        then begin
          if aChar = '\'
          then begin
            anOutput.append( aChar);
            Inc( aPosition);

            if aPosition < Length( S)
            then begin
              anOutput.append( aChar);
              Inc( aPosition);
            end;

            Continue;
          end;

          if aChar = aStringChar
          then begin
            InString := False;
            anOutput.append( aChar);
            Inc( aPosition);
            Continue;
          end;
        end;

        if CharInSet( aChar, [ '"', ''''])
        then begin
          InString := True;
          aStringChar := aChar;
        end;

        if not InString
        then begin
          if CharInSet( aChar, aStopSet)
          then Break;
        end;

        anOutput.append( aChar);
        Inc( aPosition);
      end;

      if aPosition = Length( S)
      then Result := -1
      else Result := aPosition;
    end;


    class procedure TUtils.AppendCode( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);
    var
      i     : Integer;
      aChar : Char;
    begin
      for i := aStart to anEnd - 1
      do begin
        aChar := S[ 1 + i];

        case aChar of
          '&' : anOutput.append( '&amp;');
          '<' : anOutput.append( '&lt;' );
          '>' : anOutput.append( '&gt;' );
          else  anOutput.append( aChar  );
        end;
      end;
    end;


    class procedure TUtils.AppendValue( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);
    var
      i     : Integer;
      aChar : Char;
    begin
      for i := aStart to anEnd - 1
      do begin
        aChar := S[ 1 + i];

        case aChar of
          '&'  : anOutput.append( '&amp;' );
          '<'  : anOutput.append( '&lt;'  );
          '>'  : anOutput.append( '&gt;'  );
          '"'  : anOutput.append( '&quot;');
          '''' : anOutput.append( '&apos;');
          else   anOutput.append( aChar   );
        end;
      end;
    end;


    class procedure TUtils.AppendDecEntity( const anOutput: TstringBuilder; aValue: Char);
    begin
      anOutput.append( '&#');
      anOutput.append( IntToStr( ord( aValue)));
      anOutput.append( ';');
    end;


    class procedure TUtils.AppendHexEntity( const anOutput: TstringBuilder; aValue: Char);
    begin
      anOutput.append( '&#');
      anOutput.append( IntToHex( ord( aValue), 2));
      anOutput.append( ';');
    end;


    class procedure TUtils.AppendMailto( const anOutput: TstringBuilder; const S: string; aStart: Integer; anEnd: Integer);
    var
      i     : Integer;
      aChar : Char;
    begin
      for i := aStart to anEnd - 1
      do begin
        aChar := S[ 1 + i];

        if CharInSet( aChar, [ '&', '<', '>', '"', '''', '@'])
        then AppendHexEntity( anOutput, aChar)
        else anOutput.append( aChar);
      end;
    end;


    class procedure TUtils.GetXMLTag( const anOutput: TstringBuilder; const aBin: TstringBuilder);
    var
      aPosition: Integer;
    begin
      aPosition := 1;

      if aBin[ 1] = '/'
      then Inc( aPosition);

      while aBin[ aPosition].IsLetterOrDigit
      do begin
        anOutput.append( aBin[ aPosition]);
        Inc( aPosition)
      end;
    end;


    class procedure TUtils.GetXMLTag( const anOutput: TstringBuilder; const S: string);
    var
      aPosition: Integer;
    begin
      aPosition := 1;

      if S[ 1 + 1] = '/'
      then Inc( aPosition);

      while S[ 1 + aPosition].IsLetterOrDigit
      do begin
        anOutput.append( S[ 1 + aPosition]);
        Inc( aPosition)
      end;
    end;


    class function TUtils.ReadXML( const anOutput: TstringBuilder; const S: string; aStart: Integer; UseSafeMode: Boolean): Integer;
    var
      aPosition  : Integer;
      IsCloseTag : Boolean;
      aTempSb    : TstringBuilder;
      aTag       : string;
    begin
      if S[ 1 + aStart + 1] = '/'
      then begin
        IsCloseTag := True;
        aPosition := aStart + 2;
      end
      else if S[ 1 + aStart + 1] = '!'
      then begin
        anOutput.append( '<!');
        Exit( aStart + 1);
      end
      else begin
        IsCloseTag := False;
        aPosition := aStart + 1;
      end;

      if UseSafeMode
      then begin
        aTempSb := TstringBuilder.Create;

        try
          aPosition := ReadXMLUntil( aTempSb, S, aPosition, [ ' ', '/', '>']);

          if aPosition = -1
          then Exit( -1);
          aTag := LowerCase( Trim( aTempSb.ToString));  //      tag := temp.ToString.trim.ToLower; PSTFix

          if THTML.isUnsafeHtmlElement( aTag)
          then anOutput.append( '&lt;')
          else anOutput.append( '<');

          if IsCloseTag
          then anOutput.append( '/');

          anOutput.append( aTempSb);
        finally
          aTempSb.DisposeOf;
        end;
      end
      else begin
        anOutput.append( '<');

        if IsCloseTag
        then anOutput.append( '/');

        aPosition := ReadXMLUntil( anOutput, S, aPosition, [ ' ', '/', '>']);
      end;

      if aPosition = -1
      then Exit( -1);

      aPosition := ReadXMLUntil( anOutput, S, aPosition, [ '/', '>']);

      if aPosition = -1
      then Exit( -1);

      if S[ 1 + aPosition] = '/'
      then begin
        anOutput.append( ' /');
        aPosition := ReadXMLUntil( anOutput, S, aPosition + 1, [ '>']);

        if aPosition = -1
        then Exit( -1);
      end;

      if S[ 1 + aPosition] = '>'
      then begin
        anOutput.append( '>');
        Exit( aPosition);
      end;

      Result := -1;
    end;


    class procedure TUtils.CodeEncode( const anOutput: TstringBuilder; const aValue: string; anOffset: Integer);
    var
      i     : Integer;
      aChar : Char;
    begin
      for i := anOffset to Length( aValue) - 1 do
      begin
        aChar := aValue[ 1 + i];
        case aChar of
          '&': anOutput.append( '&amp;');
          '<': anOutput.append( '&lt;' );
          '>': anOutput.append( '&gt;' );
          else anOutput.append( aChar  );
        end;
      end;
    end;


    class function TUtils.GetMetaFromFence( const aFenceLine: string): string;
    var
      i     : Integer;
      aChar : Char;
    begin
      for i := 0 to Length( aFenceLine) - 1
      do begin
        aChar := aFenceLine[ 1 + i];

        if ( not aChar.IsWhiteSpace) and ( aChar <> '`') and ( aChar <> '~')
        then Exit( Trim( Copy( aFenceLine, i+1)));   // Exit( fenceLine.substring( i).trim); PSTfix
      end;

      Result := '';
    end;


    { THTML }

    class function THTML.isHtmlBlockElement( const S: string): Boolean;
    var
      anElt : THTMLElement;
    begin
      anElt  := THTMLElement( StringToEnum( TypeInfo( THTMLElement), 'he' + S, ord( heNONE)));
      Result := anElt in BLOCK_ELEMENTS;
    end;


    class function THTML.isLinkPrefix( const S: string): Boolean;
    begin
      Result := StringsContains( LINK_PREFIXES, S);
    end;


    class function THTML.isEntity( const S: string): Boolean;
    begin
      Result := StringsContains( ENTITY_NAMES, S);
    end;


    class function THTML.isUnsafeHtmlElement( const S: string): Boolean;
    var
      anElt : THTMLElement;
    begin
      anElt  := THTMLElement( StringToEnum( TypeInfo( THTMLElement), S, ord( heNONE)));
      Result := anElt in UNSAFE_ELEMENTS;
    end;


    { TLine }

    procedure   TLine.Init;
    begin
      FLeading := 0;

      while ( Leading < Length( Value)) and ( Value[ 1 + Leading] = ' ')
      do Inc( FLeading);

      if Leading = Length( Value)
      then SetEmpty
      else begin
        IsEmpty  := False;
        Trailing := 0;

        while Value[ 1 + Length( Value) - Trailing - 1] = ' '
        do Inc( FTrailing);
      end;
    end;


    procedure   TLine.InitLeading;
    begin
      FLeading := 0;

      while ( Leading < Length( Value)) and ( Value[ 1 + Leading] = ' ')
      do Inc( FLeading);

      if Leading = Length( Value)
      then SetEmpty;
    end;


    function    TLine.SkipSpaces: Boolean;
    // TODO use Util#skipSpaces
    begin
      while ( Position < Length( Value)) and ( Value[ 1 + Position] = ' ')
      do Inc( FPosition);

      Result := Position < Length( Value);
    end;


    function    TLine.ReadUntil( const aStopSet: TSysCharSet): string;
    // TODO use Util#readUntil
    var
      aStringBuilder : TstringBuilder;
      aPosition      : Integer;
      aChar          : Char;
      bChar          : Char;
    begin
      aStringBuilder := TstringBuilder.Create;

      try
        aPosition:= Position;

        while aPosition< Length( Value)
        do begin
          aChar := Value[ 1 + aPosition];

          if ( aChar = '\') and ( aPosition+ 1 < Length( Value))
          then begin
            bChar := Value[ 1 + aPosition+ 1];

            if CharInSet( bChar, [ '\', '[', ']', '(', ')', '{', '}', '#', '"', '''', '.', '>', '*', '+', '-', '_', '!', '`', '~'])
            then begin
              aStringBuilder.append( bChar);
              Inc( FPosition);
            end
            else begin
              aStringBuilder.append( aChar);
              Break;
            end;
          end
          else if CharInSet( aChar, aStopSet)
          then Break
          else aStringBuilder.append( aChar);

          Inc( aPosition);
        end;

        if aPosition< Length( Value)
        then aChar := Value[ 1 + aPosition]
        else aChar := #10;

        if CharInSet( aChar, aStopSet)
        then begin
          Position := aPosition;
          Result := aStringBuilder.ToString;
        end
        else Result := '';

      finally
        aStringBuilder.DisposeOf;
      end;
    end;


    procedure   TLine.SetEmpty;
    begin
      Value    := '';
      Leading  := 0;
      Trailing := 0;
      IsEmpty  := True;

      if Assigned( Previous)
      then Previous.NextEmpty := True;

      if Assigned( Next)
      then Next.PrevEmpty := True;
    end;


    function    TLine.CountChars( aChar: Char): Integer;
    var
      aCount : Integer;
      i      : Integer;
      c: Char;
    begin
      aCount := 0;
      for i := 0 to Length( Value) - 1 do
      begin
        c := Value[ 1 + i];
        if c = ' ' then
          Continue;
        if c = aChar then
        begin
          Inc( aCount);
          Continue;
        end;
        aCount := 0;
        Break;
      end;
      Result := aCount;
    end;

    function    TLine.CountCharsStart( aChar: Char; AllowSpaces: Boolean): Integer;
    var
      aCount : Integer;
      i      : Integer;
      bChar  : Char;
    begin
      aCount := 0;

      for i := 0 to Length( Value) - 1
      do begin
        bChar := Value[ 1 + i];

        if ( bChar = ' ') and AllowSpaces
        then Continue;

        if bChar = aChar
        then Inc( aCount)
        else Break;
      end;

      Result := aCount;
    end;


    function    TLine.GetLineType( const aConfiguration: TConfiguration): TLineType;
    var
      i: Integer;
    begin
      if IsEmpty
      then Exit( ltEMPTY);

      if Leading > 3
      then Exit( ltCODE);

      if Value[ 1 + Leading] = '#'
      then Exit( ltHEADLINE);

      if Value[ 1 + Leading] = '>'
      then Exit( ltBQUOTE);

      if aConfiguration.ForceExtendedProfile
      then begin
        if Length( Value) - Leading - Trailing > 2
        then begin
          if ( Value[ 1 + Leading] = '`') and ( CountCharsStart( '`', aConfiguration.AllowSpacesInFencedDelimiters) >= 3)
          then Exit( ltFENCED_CODE);

          if ( Value[ 1 + Leading] = '~') and ( CountCharsStart( '~', aConfiguration.AllowSpacesInFencedDelimiters) >= 3)
          then Exit( ltFENCED_CODE);
        end;
      end;

      if ( Length( Value) - Leading - Trailing > 2) and (( Value[ 1 + Leading] = '*') or ( Value[ 1 + Leading] = '-') or ( Value[ 1 + Leading] = '_'))
      then begin
        if CountChars( Value[ 1 + Leading]) >= 3
        then Exit( ltHR);
      end;

      if ( Length( Value) - Leading >= 2) and ( Value[ 1 + Leading + 1] = ' ')
      then begin
        if CharInSet( Value[ 1 + Leading], [ '*', '-', '+'])
        then Exit( ltULIST);
      end;

      if ( Length( Value) - Leading >= 3) and Value[ 1 + Leading].IsDigit
      then begin
        i := Leading + 1;

        while ( i < Length( Value)) and Value[ 1 + i].IsDigit
        do Inc( i);

        if ( i + 1 < Length( Value)) and ( Value[ 1 + i] = '.') and ( Value[ 1 + i + 1] = ' ')
        then Exit( ltOLIST);
      end;

      if Value[ 1 + Leading] = '<'
      then begin
        if CheckHTML
        then Exit( ltXML);
      end;

      if Assigned( Next) and ( not Next.IsEmpty)
      then begin
        if (( Next.Value[ 1 + 0] = '-')) and (( Next.CountChars( '-') > 0))
        then Exit( ltHEADLINE2);

        if (( Next.Value[ 1 + 0] = '=')) and (( Next.CountChars( '=') > 0))
        then Exit( ltHEADLINE1);
      end;

      Exit( ltOTHER);
    end;


    function    TLine.ReadXMLComment( aFirstLine: TLine; aStart: Integer): Integer;
    var
      aLine     : TLine;
      aPosition : Integer;
    begin
      aLine := aFirstLine;

      if aStart + 3 < Length( aLine.Value)
      then begin
        if ( aLine.Value[ 1 + 2] = '-') and ( aLine.Value[ 1 + 3] = '-')
        then begin
          aPosition := aStart + 4;

          while Assigned( aLine)
          do begin
            while ( aPosition < Length( aLine.Value)) and ( aLine.Value[ 1 + aPosition] <> '-')
            do Inc( aPosition);

            if aPosition = Length( aLine.Value)
            then begin
              aLine     := aLine.Next;
              aPosition := 0;
            end
            else begin
              if aPosition + 2 < Length( aLine.Value)
              then begin
                if ( aLine.Value[ 1 + aPosition + 1] = '-') and ( aLine.Value[ 1 + aPosition + 2] = '>')
                then begin
                  XmlEndLine := aLine;
                  Exit( aPosition + 3);
                end;
              end;

              Inc( aPosition);
            end;
          end;
        end;
      end;

      Exit( -1);
    end;


    function    TLine.StripID: string;
    // FIXME ... hack
    var
      p      : Integer;
      aStart : Integer;
      Found  : Boolean;
      anId   : string;
    begin
      if ( IsEmpty or ( Value[ 1 + Length( Value) - Trailing - 1] <> '}')) then
        Exit( '');

      p := Leading;
      Found := False;
      while ( p < Length( Value)) and ( not Found)
      do begin
        case Value[ 1 + p] of

          '\':

            begin
              if p + 1 < Length( Value)
              then begin
                if Value[ 1 + p + 1] = '{'
                then begin
                  Inc( p);
                  Break;
                end;
              end;

              Inc( p);
              Break;
            end;

          '{':

            begin
              Found := True;
              Break;
            end
          else begin
            Inc( p);
            Break;
          end;
        end;
      end;

      if Found
      then begin
        if ( p + 1 < Length( Value)) and ( Value[ 1 + p + 1] = '#')
        then begin
          aStart := p + 2;
          p      := aStart;
          Found  := False;

          while ( p < Length( Value)) and ( not Found)
          do begin
            case Value[ 1 + p] of

              '\':

                begin
                  if p + 1 < Length( Value)
                  then begin
                    if Value[ 1 + p + 1] = '}'
                    then begin
                      // Inc( p);
                      Break;
                    end;
                  end;
                  // Inc( p);
                  Break;
                end;

              '}':

                begin
                  // found := True;
                  Break;
                end;
              else begin
                // Inc( p);
                Break;
              end;
            end;

            if Found
            then begin
              anId := Trim( Copy( Value, aStart + 1, p));  // id := value.substring( start, p).trim; PSTfix

              if Leading <> 0
              then Value := Copy( Value, 1, Leading) + Trim( Copy( Value, Leading + 1, aStart -2))  // value := value.substring( 0, leading) + value.substring( leading, start - 2).trim; PSTfix
              else Value := Trim( Copy( Value, Leading +1, aStart -2));                             // value := value.substring( leading, start - 2).trim;  PSTFix

              Trailing := 0;

              if Length( anId) > 0
              then Exit( anId)
              else Exit( '');
            end;
          end;
        end;
      end;

      Exit( '');
    end;


    function    TLine.CheckHTML: Boolean;
    var
      aTags   : TstringList;
      aTempSb : TstringBuilder;
      anElt   : string;
      aTag    : string;
      aLine   : TLine;
      aNewPos : Integer;
    begin
      Result  := False;
      aTags   := TstringList.Create;
      aTempSb := TstringBuilder.Create;

      try
        Position := Leading;

        if Value[ 1 + Leading + 1] = '!'
        then begin
          if ReadXMLComment( Self, Leading) > 0
          then Exit( True);
        end;

        Position := TUtils.ReadXML( aTempSb, Value, Leading, False);

        if Position > -1
        then begin
          anElt := aTempSb.ToString;
          aTempSb.Clear;
          TUtils.GetXMLTag( aTempSb, anElt);
          aTag := LowerCase( aTempSb.ToString);

          if not THTML.isHtmlBlockElement( aTag)
          then Exit( False);

          if ( aTag = 'hr') or AnsiEndsText( '/>', anElt) // if ( tag.equals( 'hr') or element.endsWith( '/>')) then PSTFix
          then begin
            XmlEndLine := Self;
            Exit( True);
          end;

          aTags.add( aTag);
          aLine := Self;

          while Assigned( aLine)
          do begin
            while ( Position < Length( aLine.Value)) and ( aLine.Value[ 1 + Position] <> '<')
            do Inc( FPosition);

            if Position >= Length( aLine.Value)
            then begin
              aLine    := aLine.Next;
              Position := 0;
            end
            else begin
              aTempSb.Clear;
              aNewPos := TUtils.ReadXML( aTempSb, aLine.Value, Position, False);

              if aNewPos > 0
              then begin
                anElt := aTempSb.ToString;
                aTempSb.Clear;
                TUtils.GetXMLTag( aTempSb, anElt);
                aTag := LowerCase( aTempSb.ToString);

                if THTML.isHtmlBlockElement( aTag) and ( aTag <> 'hr') and ( not AnsiEndsText( '/>', anElt))
                then begin
                  if anElt[ 1 + 1] = '/'
                  then begin
                    if aTags[ aTags.Count - 1] <> aTag
                    then Exit( False);

                    aTags.Delete( aTags.count - 1);
                  end
                  else aTags.add( aTag);
                end;

                if aTags.count = 0
                then begin
                  XmlEndLine := aLine;
                  Break;
                end;

                Position := aNewPos;
              end
              else Inc( FPosition);
            end;
          end;

          Result := aTags.count = 0;
        end;
      finally
        aTempSb.DisposeOf;
        aTags  .DisposeOf;
      end;
    end;


    { TLinkRef }

    constructor TLinkRef.Create( const aLink, aTitle: string; IsAbbrev: Boolean);
    begin
      inherited Create;
      FLink     := aLink;
      FTitle    := aTitle;
      FIsAbbrev := IsAbbrev;
    end;


    { TBlock }

    destructor  TBlock.Destroy;
    begin
      FLines .DisposeOf;
      FBlocks.DisposeOf;
      FNext  .DisposeOf;
      inherited;
    end;


    procedure   TBlock.AppendLine( const aLine: TLine);
    begin
      if not Assigned( LineTail)
      then begin
        FLines    := aLine;
        FLineTail := aLine;
      end
      else begin
        LineTail.NextEmpty := aLine.IsEmpty;
        aLine   .PrevEmpty := LineTail.IsEmpty;
        aLine   .Previous  := LineTail;
        LineTail.Next      := aLine;
        FLineTail          := aLine;
      end;
    end;


    procedure   TBlock.ExpandListParagraphs;
    var
      anOuterBlock : TBlock;
      anInnerBlock : TBlock;
      HasParagraph : Boolean;
    begin
      if ( BlockType <> btORDERED_LIST) and ( BlockType <> btUNORDERED_LIST)
      then Exit;

      anOuterBlock := Blocks;
      HasParagraph := False;

      while Assigned( anOuterBlock) and ( not HasParagraph)
      do begin
        if anOuterBlock.BlockType = btLIST_ITEM
        then begin
          anInnerBlock := anOuterBlock.Blocks;

          while Assigned( anInnerBlock) and ( not HasParagraph)
          do begin
            if anInnerBlock.BlockType = btPARAGRAPH
            then HasParagraph := True;

            anInnerBlock := anInnerBlock.Next;
          end;
        end;

        anOuterBlock := anOuterBlock.Next;
      end;

      if HasParagraph
      then begin
        anOuterBlock := Blocks;

        while Assigned( anOuterBlock)
        do begin
          if anOuterBlock.BlockType = btLIST_ITEM
          then begin
            anInnerBlock := anOuterBlock.Blocks;

            while Assigned( anInnerBlock)
            do begin
              if anInnerBlock.BlockType = btNONE
              then anInnerBlock.BlockType := btPARAGRAPH;

              anInnerBlock := anInnerBlock.Next;
            end;
          end;

          anOuterBlock := anOuterBlock.Next;
        end;
      end;
    end;


    function    TBlock.HasLines: Boolean;
    begin
      Result := Assigned( Lines);
    end;


    procedure   TBlock.RemoveLine( const aLine: TLine);
    begin
      if not Assigned( aLine.Previous)
      then FLines := aLine.Next
      else aLine.Previous.Next := aLine.Next;

      if not Assigned( aLine.Next)
      then FLineTail := aLine.Previous
      else aLine.Next.Previous := aLine.Previous;

      aLine.Previous := nil;
      aLine.Next     := nil;
      aLine.DisposeOf;
    end;


    procedure   TBlock.RemoveBlockQuotePrefix;
    var
      aLine      : TLine;
      aRemainder : Integer;
    begin
      aLine := Lines;

      while Assigned( aLine)
      do begin
        if not aLine.IsEmpty
        then begin
          if aLine.Value[ 1 + aLine.Leading] = '>'
          then begin
            aRemainder := aLine.Leading + 1;

            if ( aLine.Leading + 1 < Length( aLine.Value)) and ( aLine.Value[ 1 + aLine.Leading + 1] = ' ')
            then Inc( aRemainder);

            aLine.Value := Copy( aLine.Value, aRemainder + 1);
            aLine.InitLeading;
          end;
        end;

        aLine := aLine.Next;
      end;
    end;


    function    TBlock.RemoveLeadingEmptyLines: Boolean;
    var
      WasEmpty : Boolean;
      aLine    : TLine;
    begin
      WasEmpty := False;
      aLine    := Lines;

      while Assigned( aLine) and aLine.IsEmpty
      do begin
        RemoveLine( aLine);
        aLine    := Lines;
        WasEmpty := True;
      end;

      Result := WasEmpty;
    end;


    procedure   TBlock.RemoveTrailingEmptyLines;
    var
      aLine : TLine;
    begin
      aLine := LineTail;

      while Assigned( aLine) and aLine.IsEmpty
      do begin
        RemoveLine( aLine);
        aLine := LineTail;
      end;
    end;


    procedure   TBlock.RemoveListIndent( const aConfiguration: TConfiguration);
    var
      aLine : TLine;
    begin
      aLine := Lines;

      while Assigned( aLine)
      do begin
        if not aLine.IsEmpty
        then begin
          case aLine.GetLineType( aConfiguration) of
            ltULIST : aLine.Value := Copy( aLine.Value, aLine.Leading + 3);          // line.value := line.value.substring( line.leading + 2); PSTfix
            ltOLIST : aLine.Value := Copy( aLine.Value, pos( '.', aLine.Value) + 2); // line.value := line.value.substring( line.value.indexOf( '.') + 2); pstfix
            else      aLine.Value := Copy( aLine.Value, Min( aLine.Leading + 1, 5)); // line.value := line.value.substring( Math.min( line.leading, 4)); pstfix
          end;

          aLine.InitLeading;
        end;

        aLine := aLine.Next;
      end;
    end;


    procedure   TBlock.RemoveSurroundingEmptyLines;
    begin
      if Assigned( Lines)
      then begin
        RemoveTrailingEmptyLines;
        RemoveLeadingEmptyLines;
      end;
    end;


    function    TBlock.Split( const aLine: TLine): TBlock;
    var
      aBlock : TBlock;
    begin
      aBlock           := TBlock.Create;
      aBlock.FLines    := Lines;
      aBlock.FLineTail := aLine;
      FLines           := aLine.Next;
      aLine.Next       := nil;

      if not Assigned( Lines)
      then FLineTail      := nil
      else Lines.Previous := nil;

      if not Assigned( Blocks)
      then begin
        FBlocks    := aBlock;
        FBlockTail := aBlock;
      end
      else begin
        BlockTail.Next := aBlock;
        FBlockTail     := aBlock;
      end;

      Result := aBlock;
    end;


    procedure   TBlock.TransformHeadline;
    var
      aLevel : Integer;
      aStart : Integer;
      anEnd  : Integer;
      aLine  : TLine;
    begin
      if HlDepth > 0
      then Exit;


      aLevel := 0;
      aLine  := Lines;

      if aLine.IsEmpty
      then Exit;

      aStart := aLine.Leading;

      while ( aStart < Length( aLine.Value)) and ( aLine.Value[ 1 + aStart] = '#')
      do begin
        Inc( aLevel);
        Inc( aStart);
      end;

      while ( aStart < Length( aLine.Value)) and ( aLine.Value[ 1 + aStart] = ' ')
      do Inc( aStart);

      if aStart >= Length( aLine.Value)
      then aLine.SetEmpty
      else begin
        anEnd := Length( aLine.Value) - aLine.Trailing - 1;

        while aLine.Value[ 1 + anEnd] = '#'
        do Dec( anEnd);

        while aLine.Value[ 1 + anEnd] = ' '
        do Dec( anEnd);

        aLine.Value    := Copy( aLine.Value, aStart+1, anEnd-aStart+1);
        aLine.Leading  := 0;
        aLine.Trailing := 0;
      end;

      HlDepth := Min( aLevel, 6);
    end;


end.
